#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <ifaddrs.h>
#include <getopt.h>
#include <libgen.h>
#include <pthread.h>
#include <ifaddrs.h>
#include <netdb.h>
#include <linux/if_packet.h>
#include <linux/rtnetlink.h>

#if HOST_NAME_MAX < 128
  #define NAME_LEN	(128)	// max number of characters in name
#else
  #define NAME_LEN	HOST_NAME_MAX
#endif

#define PACKET_SIZE		(2200)
#define UDP_PORT		(64159)

#define UNUSED(x)		(void)(x)	// This macro silences unused parameter warnings

#ifndef MSG_CONFIRM
  #define MSG_CONFIRM (0)		// This is for OSX which does not define this
#endif

static int			Rx_Socket = -1 ;
static int			Tx_Socket = -1 ;
static int			Shutdown = 0 ;
struct sockaddr_in		Rx_Address, Tx_Address ;
static char			My_Name [NAME_LEN] ;
static char			My_IPv4_Address[NI_MAXHOST] ;
pthread_t			Rx_Thread_ID, Tx_Thread_ID ;

static const char		*Multicast_Group = "225.45.21.183" ;		// pure spurious random numbers in the multicast range
///////////////////////////////////////////////////////////////////////////////////////////
void *
Tx_Thread (void *arg)
{
  int			i, n ;
  char			send_data[PACKET_SIZE] ;

  UNUSED (arg) ; // silence the compiler warning

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  printf ("Commencing Tx Thread\n") ;

  for (i=0 ; Shutdown == 0 ; i++)
  {
    n = snprintf (send_data, PACKET_SIZE, "[%d] %s", i, My_IPv4_Address) ;

    // Don't send 'n' bytes, send the full packet length as a test to see we can send more than 1500 bytes UDP
    // with multicast. The string will be terminated by a null, and the remaining characters can there be ignored
    n = PACKET_SIZE ;
    if (sendto (Tx_Socket, send_data, n, MSG_CONFIRM, (struct sockaddr *)&Tx_Address, sizeof(Tx_Address)) != n)
    {
      fprintf (stderr, "sendto call failed to send %d byte message '%s': %s\n", n, send_data, strerror(errno)) ;
      Shutdown = 1 ;
    }
    else
      sleep (2) ; // send a message every 2 seconds
  }

  pthread_cancel (Rx_Thread_ID) ;
  return NULL ;
}
///////////////////////////////////////////////////////////////////////////////////////////
void *
Rx_Thread (void *arg)
{
  int			n ;
  socklen_t		src_addr_len ;
  struct sockaddr_in6	src_addr ;
  char			recv_data[PACKET_SIZE] ;

  UNUSED (arg) ; // silence the compiler warning

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  printf ("Commencing Rx Thread\n") ;
  src_addr_len = sizeof (src_addr) ;

  for (; Shutdown == 0 ;)
  {
    // Receive a message from any source address....
    if ((n = recvfrom (Rx_Socket, recv_data, PACKET_SIZE, MSG_WAITALL, (struct sockaddr *)&src_addr, &src_addr_len)) < 0)
    {
      fprintf (stderr, "recvfrom call failed: %s\n", strerror(errno)) ;
      break ;
    }

    recv_data[n]= '\0' ; // terminate with a null. The message over the wire does not have the null terminator
    printf("Received %d byte packet with string '%s'\n", n, recv_data) ;
  }

  pthread_cancel (Tx_Thread_ID) ;
  return NULL ;
}
///////////////////////////////////////////////////////////////////////////////////////////
int
main ()
{
  int				c ;
  struct ifaddrs		*ifaddr ;
  struct ifaddrs		*ifa ;
  struct ip_mreq		mreq ;

  //------------------------------------------------------------------------------------------------------------------
  // Determine the hostname and IP address of this system into My_Name and My_IPv4_Address
  if ((c = gethostname(My_Name, NAME_LEN)) < 0)
  {
    fprintf (stderr, "Cannot read own hostname: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  My_Name[NAME_LEN-1] = '\0' ;

  // Determine my own IP address
  if (getifaddrs(&ifaddr) < 0)
  {
    fprintf (stderr, "getifaddrs() failed: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // parse the ifaddr data to determine the active IPv4 address
  for (ifa=ifaddr ; ifa != NULL ; ifa=ifa->ifa_next)
  {
    // If there is no IP address, or this is not a IPv4 record, or this is the loop back interface, ignore it
    if ((ifa->ifa_addr == NULL) || (ifa->ifa_addr->sa_family != AF_INET) || (strcmp(ifa->ifa_name, "lo") == 0))
      continue ;

    if ((c = getnameinfo(ifa->ifa_addr, sizeof(struct sockaddr_in), My_IPv4_Address, NI_MAXHOST, NULL, 0, NI_NUMERICHOST)) != 0)
    {
      fprintf(stderr, "getnameinfo() failed: %s\n", gai_strerror(c)) ;
      return EXIT_FAILURE ;
    }

    break ;
  }

  freeifaddrs(ifaddr) ;

  printf ("My details are [%s / %s]\n", My_Name, My_IPv4_Address) ;
  printf ("Multicasting to group %s:%d\n", Multicast_Group, UDP_PORT) ;
  //------------------------------------------------------------------------------------------------------------------
  // We will use one socket for receive and another for transmit of the multicast UDP packets
  if ((Rx_Socket = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    fprintf (stderr, "Rx socket creation failed: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Enable SO_REUSEADDR to allow multiple instances of this programme to receive copies of the multicast datagrams
  c = 1 ;

  if (setsockopt(Rx_Socket, SOL_SOCKET, SO_REUSEADDR, (char *)&c, sizeof(c)) < 0)
  {
    perror ("Setting SO_REUSEADDR error") ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }

  // Define the IP address for receiving
  memset (&Rx_Address, 0x00, sizeof(Rx_Address)) ;
  Rx_Address.sin_family		= AF_INET ;
  Rx_Address.sin_addr.s_addr	= htonl(INADDR_ANY) ;
  Rx_Address.sin_port		= htons(UDP_PORT) ;

  // Bind the socket with the port address specified above
  if (bind (Rx_Socket, (struct sockaddr *)&Rx_Address, sizeof(Rx_Address)) < 0)
  {
    fprintf (stderr, "Socket bind failed: %s\n", strerror(errno)) ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }

  // use setsockopt() to request that the kernel join a multicast group
  mreq.imr_multiaddr.s_addr = inet_addr(Multicast_Group) ;
  mreq.imr_interface.s_addr = htonl(INADDR_ANY) ;

  if (setsockopt(Rx_Socket, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&mreq, sizeof(mreq)) < 0)
  {
    fprintf (stderr, "setsockopt IP_ADD_MEMBERSHIP: %s\n", strerror(errno)) ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------
  if ((Tx_Socket = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    fprintf (stderr, "Tx socket creation failed: %s\n", strerror(errno)) ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }

  // Define the IP address for transmitting
  memset (&Tx_Address, 0x00, sizeof(Tx_Address)) ;
  Tx_Address.sin_family		= AF_INET ;
  Tx_Address.sin_addr.s_addr	= inet_addr(Multicast_Group) ;
  Tx_Address.sin_port		= htons(UDP_PORT) ;
  //------------------------------------------------------------------------------------------------------------------
  // Start two threads: one for receiving packets. The other for sending packets.

  if (pthread_create (&Rx_Thread_ID, NULL, Rx_Thread, NULL) != 0)
  {
    fprintf (stderr, "Cannot create Rx Thread: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  if (pthread_create (&Tx_Thread_ID, NULL, Tx_Thread, NULL) != 0)
  {
    fprintf (stderr, "Cannot create Tx Thread: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  pthread_join (Rx_Thread_ID, NULL) ;
  Shutdown = 1 ;
  pthread_join (Tx_Thread_ID, NULL) ;

  // Release resources
  close (Rx_Socket) ;
  close (Tx_Socket) ;
}
