////////////////////////////////////////////////////////////////////////////////////////
// This is a programme that demonstrates very basic launching of linux threads
// and shows how you can have more threads running than actual CPU cores
///////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>

typedef struct timespec		TS ;
#define NS_PER_SECOND		(1000000000)
///////////////////////////////////////////////////////////////////////////////////////////
// Sleep_ns is a wrapper to make calls to nanosleep easier
// It is set up as a delay call with ns granularity
//
// The maximum sleep time is ~4 seconds, as the number of nanoseconds is a 32 bit unsigned int

void
Sleep_ns (const uint32_t ns)
{
  TS    ts ;

  if (ns == 0)
    return ;

  ts.tv_sec = ns / NS_PER_SECOND ;
  ts.tv_nsec = ((int)ns - (ts.tv_sec * NS_PER_SECOND)) ;        // nanoseconds (fractional seconds)

  // nanosleep should return 0 if it slept correctly for the full duration
  // However it can be interrupted by a signal, and if so, we need to process
  // the signal and reinitiate the remainder of the delay.
  for (; nanosleep(&ts, &ts) != 0 ;)
  {
    // if interrupted by a signal, errno will be EINTR, and we just try again
    if (errno != EINTR)
      fprintf (stderr, "Problem with Sleep_ns (0x%08x ==> %d.%d): %s\n", ns, (int)ts.tv_sec, (int)ts.tv_nsec, strerror(errno)) ;
  }
} // Sleep_ns
///////////////////////////////////////////////////////////////////////////////////////////
// This child subroutine is is written in a more OS friendly way than the example in
// test_pthread2.c. It is written so that the total delay will amount to around five
// seconds of elapsed time, but the child will reliquish control to the OS while it is
// delaying. Other children can then run in the time that is relinquished.
//
// You can run hundreds of these threads in parallel, and they will all wait for approximately
// five seconds each.

#define MAX_DEL	(5000000) // 5M microseconds

void *
child_subroutine (void *arg)
{
  int			*i ;
  uint32_t		del, microseconds ;

  // read the parameter that was passed
  i = (int *)arg ;

  for (microseconds=0 ; microseconds < MAX_DEL ; microseconds += del)
  {
    del = (rand() % 1000000) + 1 ; // Pick a random number between 1 and 1M

    if ((microseconds + del) > MAX_DEL)
      del = MAX_DEL - microseconds ;

    fprintf (stderr, "CHILD %4d: t=%.3fs, delaying %.3fs\n", *i, microseconds / 1000000.0, del / 1000000.0) ;
    Sleep_ns (del * 1000) ;
  }

  fprintf (stderr, "CHILD %4d: exited\n", *i) ;
  free (arg) ; // return the arg memory to the OS
  pthread_exit (NULL) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
int
main (int argc, char **argv)
{
  int		N, i, *p ;
  time_t	t ;
  struct timeval start_time, end_time ;
  double	duration ;

  if (argc != 2)
    goto show_usage ;

  // Read the value
  if ((N = atoi(argv[1])) <= 0)
  {
    printf ("The number of child threads must be a positive integer.\n\n") ;

  show_usage:
    printf ("Usage: %s [number of children]\n", argv[0]) ;
    printf ("Launch the specified number of efficient time-sharing threads. (Try 5000!)\n\n") ;
    printf ("Watch how the OS efficiently shares time. You will be able to launch many more threads\n") ;
    printf ("than there are CPU cores, and the threads will seemingly all run at full speed even\n") ;
    printf ("You dramatically exceed the number of CPU cores.\n") ;
    return EXIT_FAILURE ;
  }

  printf ("This demo programme will create %d different threads called CHILD, and each of those\n", N) ;
  printf ("will sleep for random amounts of time in an OS friendly way.\n") ;
  printf ("---------------------\n") ;

  pthread_t	thread [N];

  // Seed the random number generator using system time. We will also remember that time so we know
  // how long this programme ran
  srand ((unsigned) time (&t)) ;

  // Remember the exact time we started
  gettimeofday (&start_time, NULL) ;

  // Launch N threads, mallocing some memory so we can pass a parameter that gives each
  // child a unique number
  for (i=0 ; i < N ; i++)
  {
    p = malloc (sizeof(int)) ; // allocate memory for the parameter and initialise it
    *p = i ;

    // This is the Linux call to start a thread
    pthread_create (&thread[i], NULL, child_subroutine, (void *)p) ;
    fprintf (stderr, "MAIN   %4d: thread=%lx\n", i, thread[i]) ;
  }

  // Note: all the memory we allocated for the parameters is no longer ours! It belongs to the
  // threads, so we should not access it any more from here.

  // After the thread(s) have launched, we need to wait for them to finish.
  // The joining need not actually happen in the same order that the threads finish.
  // As implemented here, we wait for them to finish in a particular sequence, but the
  // actual ordering of the pthread_exit calls in real time might be different.
  for (i=0 ; i < N ; i++)
  {
    pthread_join (thread[i], NULL) ;
    fprintf (stderr, "MAIN   %4d thread returned\n", i) ;
  }

  // Remember the exact time we stopped
  gettimeofday (&end_time, NULL) ;

  // Determine the elapsed duration (in ms)
  duration = (end_time.tv_sec - start_time.tv_sec) * 1000.0 ;		// sec to ms
  duration += (end_time.tv_usec - start_time.tv_usec) / 1000.0 ;	// us to ms

  printf ("---------------------\nRan for %.3f seconds\n", duration / 1000.0) ;
}
