////////////////////////////////////////////////////////////////////////////////////////
// This is a programme that demonstrates some of the mpv interfaces in libmpv
///////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////////
// For additional information about using the libmpv API, try these links:
//  https://mpv.io/manual/master
//  https://github.com/mpv-player/mpv/blob/master/libmpv/client.h
//
// I found mpvlib to not have the best documentation.... hence this sandpit programme.
// I kind-of worked it out using the links above, and trial and error.
///////////////////////////////////////////////////////////////////////////////////////////
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <errno.h>
#include <mqueue.h>
#include <inttypes.h>

#include <mpv/client.h>
///////////////////////////////////////////////////////////////////////////////////////////
#define UNUSED(x)		(void)(x)	// This macro silences unused parameter warnings

mpv_handle			*Mpv_Handle ;
mqd_t				Media_Msgq ;
uint8_t				Threads_Active ;        // Keeps track of the multi-tasking state
uint8_t				Media_Playing ;

static const char Media_Msgq_name[] = "/Media" ;

enum _state
{
  _INITIALISING=0,
  _RUNNING,
  _SHUTTING_DOWN,
  _TERMINATED
} ;

#define MSG_LEN	(1000)
typedef char MSG_BUF[MSG_LEN] ;
///////////////////////////////////////////////////////////////////////////////////////////
// Open_Msg_Queue_Rx and Open_Msg_Queue_Tx
// Opens a message queue for reading or writing

mqd_t
Open_Msg_Queue_Rx (const char *name, long max_msg, long msg_size)
{
  struct mq_attr	attr, old_attr ;
  mqd_t			Q ;
  char			buffer [msg_size] ;
  unsigned int		priority ;

  // First we need to set up the attribute structure
  attr.mq_maxmsg = max_msg ;
  attr.mq_msgsize = msg_size ;
  attr.mq_flags = 0 ;

  if ((Q = mq_open (name, O_RDONLY | O_CREAT, 0600, &attr)) < 0)
  {
    fprintf (stderr, "Cannot create message queue %s (max_msg=%ld, msg_size=%ld): %s\n", name, max_msg, msg_size, strerror(errno)) ;
    _exit (EXIT_FAILURE);
  }

  // Are there any messages currently in this queue (hanging over from something else??)
  mq_getattr (Q, &attr) ;

  if (attr.mq_curmsgs != 0)
  {
    // There are some messages on this queue. Remove them so the queue starts empty

    // First set the queue to not block any calls
    attr.mq_flags = O_NONBLOCK ;
    mq_setattr (Q, &attr, &old_attr) ;

    // Now eat all of the messages
    while (mq_receive (Q, buffer, msg_size, &priority) >= 0) ;

    // The call failed.  Make sure errno is EAGAIN which is the only acceptable failure
    if (errno != EAGAIN)
    {
      fprintf (stderr, "Cannot empty message queue %s: %s\n", name, strerror(errno)) ;
      _exit (EXIT_FAILURE);
    }

    // Now restore the attributes
    mq_setattr (Q, &old_attr, 0);
  }

  return Q ;
} // Open_Msg_Queue_Rx

mqd_t
Open_Msg_Queue_Tx (const char *name, const int no_abort)
{
  mqd_t			Q ;

  // The call for opening to write is different than the call for opening to read!
  if ((Q = mq_open (name, O_WRONLY)) < 0)
  {
    fprintf (stderr, "Cannot open queue %s for writing: %s\n", name, strerror(errno)) ;

    // Return a negative number if we've been asked not to abort
    if (no_abort)
      return Q ;

    _exit (EXIT_FAILURE);
  }

  return Q ;
} // Open_Msg_Queue_Tx
///////////////////////////////////////////////////////////////////////////////////////////
static void
check_error (int status)
{
  if (status < 0)
  {
    printf("mpv API error: %s\n", mpv_error_string(status)) ;
    Threads_Active = _SHUTTING_DOWN ;
    exit (EXIT_FAILURE);
  }
}
///////////////////////////////////////////////////////////////////////////////////////////
// Wait_For_Next_Message
//
// Waits for the OS to send us a message, then does some basic checks, and returns
// The return value is either the number of bytes in the message, or 0 (if something went wrong)

int
Wait_For_Next_Message (mqd_t Q, void *msg, const int min_size, const int max_size, const char *name)
{
  int			n ;
  unsigned int		priority ;

  // No timeout - just wait as long as it takes for the next message to arrive
  if ((n = mq_receive (Q, (char *)msg, max_size, &priority)) < 0)
  {
    // Was the system call interrupted? Silently return failure if so
    if (errno == EAGAIN)
      return n ;

    fprintf (stderr, "%s: msg receive error: %s\n", name, strerror(errno)) ;
  }

  else if (n < min_size)
    fprintf (stderr, "%s: Short message received: %d bytes\n", name, n) ;

  // It shouldn't be possible for n to exceed max_size - the call to mq_send will fail instead
  return n ;
}
// Wait_For_Next_Message
///////////////////////////////////////////////////////////////////////////////////////////
void *
Media_Thread (void *xarg)
{
  MSG_BUF	msg ;
  int		i, n ;
  int64_t	val ;
  mpv_node	pl ;

  UNUSED (xarg) ; // silence the compiler warning
  fprintf (stderr, "Launching media thread\n") ;
  Media_Msgq = Open_Msg_Queue_Rx (Media_Msgq_name, 5, sizeof(msg)) ;
  Media_Playing = 0 ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    usleep (10000) ;

  fprintf (stderr, "Media thread running\n") ;

  // Messages to the media thread contain the linux command text for launching a new media stream
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // This is an infinte loop - wait for messages and then do what the message requests.
    // Different messages to (a) start a new stream, (b) pause a stream, (c) stop a stream, (d) check on the status of a stream
    if ((n = Wait_For_Next_Message (Media_Msgq, (void *)&msg, 1, sizeof(msg), "Media Thread")) < 1)
      continue ;

    msg[n] = '\0' ;
    fprintf (stderr, "Media thread received '%s'\n", msg) ;

    if (strcasecmp (msg, "quit") == 0)
    {
      // Quit mpv by issuing a quit command
      const char *cmd[] = {"quit", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      Threads_Active = _SHUTTING_DOWN ;
      continue ;
    }

    if (strcasecmp (msg, "stop") == 0)
    {
      // Stop whatever is playing and clear the playlist
      const char *cmd[] = {"stop", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "pause") == 0)
    {
      // Toggles the pause/play state
      const char *cmd[] = {"cycle", "pause", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "set_pause") == 0)
    {
      i = 1 ;

      if (mpv_set_property (Mpv_Handle, "pause", MPV_FORMAT_FLAG, &i) < 0)
      {
	fprintf (stderr, "mpv_set_property error\n") ;
	continue ;
      }

      printf ("Paused\n") ;
      continue ;
    }

    if (strcasecmp (msg, "reset_pause") == 0)
    {
      i = 0 ;

      if (mpv_set_property (Mpv_Handle, "pause", MPV_FORMAT_FLAG, &i) < 0)
      {
	fprintf (stderr, "mpv_set_property error\n") ;
	continue ;
      }

      printf ("Unpaused\n") ;
      continue ;
    }

    if (strcasecmp (msg, "next") == 0)
    {
      // Jumps to next track in playlist
      const char *cmd[] = {"playlist-next", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "prev") == 0)
    {
      // Jumps to prev track in playlist
      const char *cmd[] = {"playlist-prev", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "restart") == 0)
    {
      // Restarts the current track from the beginning
      const char *cmd[] = {"seek", "0", "absolute", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "seek-fwd") == 0)
    {
      // Seeks to a position 5 seconds after the current position in track
      const char *cmd[] = {"seek", "5", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "seek-back") == 0)
    {
      // Seeks to a position 5 seconds before the current position in track
      const char *cmd[] = {"seek", "-5", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "shuffle") == 0)
    {
      // Shuffles the current contents of the playlist
      const char *cmd[] = {"playlist-shuffle", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "clear") == 0)
    {
      // Clears playlist except the currently playing file
      const char *cmd[] = {"playlist-clear", NULL};
      mpv_command (Mpv_Handle, cmd) ;
      continue ;
    }

    if (strcasecmp (msg, "pos") == 0)
    {
      // Reads current position in the playlist
      if (mpv_get_property (Mpv_Handle, "playlist-pos-1", MPV_FORMAT_INT64, &val) < 0)
      {
	fprintf (stderr, "mpv_get_property error\n") ;
	continue ;
      }

      printf ("Playlist position: %" PRId64 "\n", val) ;
      continue ;
    }

    if (strcasecmp (msg, "count") == 0)
    {
      // Reads size of the current playlist
      if (mpv_get_property (Mpv_Handle, "playlist-count", MPV_FORMAT_INT64, &val) < 0)
      {
	fprintf (stderr, "mpv_get_property error\n") ;
	continue ;
      }

      printf ("Playlist count: %" PRId64 "\n", val) ;
      continue ;
    }

    if (strncasecmp (msg, "play ", 5) == 0)
    {
      // Jumps to the specified position in the playlist
      if (sscanf(&msg[5], "%" PRId64 "", &val) != 1)
      {
	fprintf (stderr, "play index number error: '%s' --> %" PRId64 "\n", msg, val) ;
	continue ;
      }

      if (mpv_set_property (Mpv_Handle, "playlist-pos-1", MPV_FORMAT_INT64, &val) < 0)
      {
	fprintf (stderr, "mpv_set_property error\n") ;
	continue ;
      }

      continue ;
    }

    if (strcasecmp (msg, "playlist") == 0)
    {
      // Reads the playlist
      if (mpv_get_property (Mpv_Handle, "playlist", MPV_FORMAT_NODE, &pl) < 0)
      {
	fprintf (stderr, "mpv_get_property error\n") ;
	continue ;
      }

      if (pl.format == MPV_FORMAT_NODE_ARRAY)
      {
	printf ("Playlist:\n") ;

	for (i=0 ; i < pl.u.list->num ; i++)
	{
	  // i iterates through the tracks in the playlist
	  printf ("%d: ", i) ;

	  // n iterates through the attributes for each track.
	  // Firstly find and display the filename
	  for (n=0 ; n < pl.u.list->values[i].u.list->num ; n++)
	    if (strcasecmp (pl.u.list->values[i].u.list->keys[n], "filename") == 0)
	    {
	      printf ("%s", pl.u.list->values[i].u.list->values[n].u.string) ;
	      break ;
	    }

	  // Secondly, identify if this is the current track (playing)
	  for (n=0 ; n < pl.u.list->values[i].u.list->num ; n++)
	    if ((strcasecmp (pl.u.list->values[i].u.list->keys[n], "current") == 0) ||
		(strcasecmp (pl.u.list->values[i].u.list->keys[n], "playing") == 0))
	    {
	      printf (" (playing)") ;
	      break ;
	    }

	  printf ("\n") ;
	}
      }

      mpv_free_node_contents (&pl) ;
      continue ;
    }

    // Append this file to the playlist
    const char *cmd[] = {"loadfile", msg, "append-play", NULL};
    mpv_command (Mpv_Handle, cmd) ;
  }

  fprintf (stderr, "Dropping out bottom of Media_Thread\n") ;

  // Tidy up before exiting
  mq_unlink (Media_Msgq_name) ;
  mq_close (Media_Msgq) ;
  pthread_exit (NULL) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
void *
Monitor_Thread (void *xarg)
{
  mpv_event	*event ;
  int64_t	val ;

  UNUSED (xarg) ; // silence the compiler warning
  fprintf (stderr, "Launching monitor thread\n") ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    usleep (10000) ;

  fprintf (stderr, "Monitor thread running\n") ;

  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    event = mpv_wait_event (Mpv_Handle, 10000) ;
    fprintf(stderr, "Event monitor received: %d='%s'\n", event->event_id, mpv_event_name(event->event_id));

    switch (event->event_id)
    {
      case MPV_EVENT_SHUTDOWN:
	fprintf (stderr, "  Shutdown\n") ;
	Threads_Active = _SHUTTING_DOWN ;
	break ;

      case MPV_EVENT_IDLE:
	fprintf (stderr, "  Idle\n") ;
	Media_Playing = 0 ;
	break ;

      case MPV_EVENT_PLAYBACK_RESTART:
	fprintf (stderr, "  Playback-Restart\n") ;
	Media_Playing = 1 ;

	// Read size of the current playlist
	if (mpv_get_property (Mpv_Handle, "playlist-count", MPV_FORMAT_INT64, &val) < 0)
	  fprintf (stderr, "mpv_get_property error\n") ;
	else
	  printf ("   ->Playlist length: %" PRId64 "\n", val) ;

	// Read current position in the playlist
	if (mpv_get_property (Mpv_Handle, "playlist-pos-1", MPV_FORMAT_INT64, &val) < 0)
	  fprintf (stderr, "mpv_get_property error\n") ;
	else
	  printf ("   ->New playlist position: %" PRId64 "\n", val) ;

	break ;

      case MPV_EVENT_QUEUE_OVERFLOW:
	fprintf (stderr, "  Queue overflow\n") ;
	break ;

      case MPV_EVENT_AUDIO_RECONFIG:
	fprintf (stderr, "  Audio reconfig - has it quit?\n") ;
	break ;

      default:
	fprintf (stderr, "Not processing this message. Looping to wait for next event\n") ;
	break ;
    }
  }

  fprintf (stderr, "Dropping out bottom of Monitor_Thread\n") ;
  pthread_exit (NULL) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
int
main (int argc, char *argv[])
{
  pthread_t	mp_id, mon_id ;
  int		n ;
  mqd_t		media_msgq ;
  MSG_BUF	msg ;

  if ((argc < 2) || (argc > 3))
  {
    printf ("Usage: %s -s\n", argv[0]) ;
    printf ("Start an mpv server. The server will wait for commands or media as specified below.\n\n") ;

    printf ("Usage: %s [command]\n", argv[0]) ;
    printf ("  quit\t\tquits the server by issuing a quit command and waiting for the server to exit with shutdown event\n") ;
    printf ("  stop\t\tstops currently playing media and clears playlist\n") ;
    printf ("  pause\t\ttoggles pause state\n") ;
    printf ("  set_pause\tpause ON\n") ;
    printf ("  reset_pause\tpause OFF\n") ;
    printf ("  next\t\tjumps to next track in playlist\n") ;
    printf ("  prev\t\tjumps to previous track in playlist\n") ;
    printf ("  restart\trestarts currently playing track\n") ;
    printf ("  seek-fwd\tjumps 5 seconds forwards\n") ;
    printf ("  seek-back\tjumps 5 seconds backwards\n") ;
    printf ("  clear\t\tclears current playlist (keeps playing current track)\n") ;
    printf ("  shuffle\tShuffles the current playlist\n") ;
    printf ("  playlist\tdisplays the accumulated playlist\n") ;
    printf ("  pos\t\tdisplays the position in the playlist of currently playing track\n") ;
    printf ("  count\t\tdisplays the count of entries in the playlist\n\n") ;
    printf ("  play N\t\tStarts playing track N from the playlist\n\n") ;

    printf ("Usage: %s [media URL or path to file or directory]\n", argv[0]) ;
    printf ("  Appends the specified URL or path to a file or directory to the current playlist\n") ;
    return EXIT_FAILURE ;
  }
  //-------------------------------------------------------
  if (strcasecmp(argv[1], "-s") == 0)
  {
    fprintf (stderr, "Starting server\n") ;

    // Create the media player handle
    if ((Mpv_Handle = mpv_create()) == NULL)
    {
      printf ("failed creating mpv context\n") ;
      return EXIT_FAILURE ;
    }

    // Launch mpv
    // The following calls specify what you'd normally specify when invoking mpv using the command line interface
    // each of the options are what would normally be specified with '--' then the option name
    check_error (mpv_set_option_string (Mpv_Handle, "input-default-bindings", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "input-vo-keyboard", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "input-terminal", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "vid", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "osc", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "term-osd", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "msg-level", "all=error")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "config", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "terminal", "no")) ;
    check_error (mpv_set_option_string (Mpv_Handle, "ytdl-raw-options", "no-check-certificate=yes")) ;

    // Done setting up options.
    check_error (mpv_initialize(Mpv_Handle)) ;

    Threads_Active = _INITIALISING ;

    // Create a player thread
    if ((n = pthread_create (&mp_id, NULL, Media_Thread, NULL)) != 0)
    {
      fprintf (stderr, "Can't create media player thread (%d): %s\n", n, strerror(errno)) ;
      return EXIT_FAILURE ;
    }

    // Create a monitor thread
    if ((n = pthread_create (&mon_id, NULL, Monitor_Thread, NULL)) != 0)
    {
      fprintf (stderr, "Can't create monitor thread (%d): %s\n", n, strerror(errno)) ;
      return EXIT_FAILURE ;
    }

    // Wait for the threads to initialise
    for (; Media_Msgq == 0 ; usleep (10000)) ;
    Threads_Active = _RUNNING ;

    // Wait for threads to exit
    pthread_join(mp_id, NULL) ;
    Threads_Active = _SHUTTING_DOWN ;
    pthread_join(mon_id, NULL);
    fprintf (stderr, "server exiting\n") ;
    mpv_terminate_destroy (Mpv_Handle);
    exit (EXIT_SUCCESS) ;
  }
  //-------------------------------------------------------
  // We are going to try to send messages to the server
  // join the message queue so we can communiate with the media thread
  media_msgq = Open_Msg_Queue_Tx (Media_Msgq_name, 0) ;

  if (argc == 2)
    n = sprintf (msg, "%s", argv[1]) ;
  else
    n = sprintf (msg, "%s %s", argv[1], argv[2]) ;

  // Send the media request to the media thread
  if (mq_send (media_msgq, msg, n+1, 0) < 0)
  {
    fprintf (stderr,"Failed to send media msg '%s': %s\n", msg, strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  printf ("Message sent to daemon. Refer to output of daemon thread\n") ;
  return EXIT_SUCCESS ;
}
