///////////////////////////////////////////////////////////////////////////////////////////
// Short programme to check the status of the alarm-clock message queues
// Whether the queues exist or not, and their state, will depend on whether its running
////////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
// message_queue_status.c
///////////////////////////////////////////////////////////////////////////////////////////
#include <mqueue.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
///////////////////////////////////////////////////////////////////////////////////////////
// These strings define the names that will be used for Linux message passing queues.
// Threads will communicate with each other via these message queues.
// Message queue names must all commence with a slash

static const char DD_Msgq_name[] = "/DD" ;		// Display driver thread message queue
static const char Dim_Msgq_name[] = "/Dim" ;		// LED dimmer thread message queue
static const char Setup_Msgq_name[] = "/S" ;		// Setup thread message queue
static const char Media_Msgq_name[] = "/M" ;		// Media thread message queue
static const char Amp_Msgq_name[] = "/A" ;		// Amp Manager thread message queue
static const char Button_Msgq_name[] = "/B" ;		// Button Manager thread message queue
///////////////////////////////////////////////////////////////////////////////////////////
int
check_queue (const char *Qname)
{
  mqd_t			mqd;
  struct mq_attr	attr;

  printf ("%s:\n", Qname) ;

  if ((mqd = mq_open (Qname, O_RDONLY, 0, NULL)) == (mqd_t) -1)
  {
    printf ("   mq_open failed: %s\n\n", strerror (errno)) ;
    return 1 ;
  }

  if (mq_getattr (mqd, &attr) == -1)
  {
    printf ("   mq_getattr failed: %s\n\n", strerror (errno)) ;
    return 1 ;
  }

  printf ("  Maximum # of messages in queue:   %ld\n", attr.mq_maxmsg) ;
  printf ("  Maximum message size:             %ld\n", attr.mq_msgsize) ;
  printf ("  Current number of messages:       %ld\n", attr.mq_curmsgs) ;

  if (mq_close (mqd) == -1)
  {
    printf ("   mq_close failed: %s\n\n", strerror (errno)) ;
    return 1 ;
  }

  printf ("\n") ;
  return 0 ;
}
///////////////////////////////////////////////////////////////////////////////////////////
int
main(int argc, char *argv[])
{
  int			v ;

  if (argc != 1)
  {
    fprintf (stderr, "Usage: %s\n", argv[0]) ;

    fprintf (stderr, "\nA programme to display the message queue status for the Linux message queues used by the clock.\n") ;
    exit (EXIT_FAILURE) ;
  }

  v = check_queue (DD_Msgq_name) ;
  v += check_queue (Dim_Msgq_name) ;
  v += check_queue (Setup_Msgq_name) ;
  v += check_queue (Media_Msgq_name) ;
  v += check_queue (Amp_Msgq_name) ;
  v += check_queue (Button_Msgq_name) ;

  exit (v) ;
}
