#!/bin/bash
#                           Digital Alarm Clock
#                Copyright (C) 2023,2024 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
####################################################################################################################
# Reads the wpa_supplicant.conf file, allows it to be edited, and rewrites it
# If there's a wpa_supplicant.conf in /boot, read and modify that one. Otherwise, read the one in /etc
THIS_SCRIPT="/cgi-bin/wifi.cgi"

# Load common functions and data
INCLUDE_COMMON="common.sh"
WPA_SUPPLICANT_CONF="/boot/wpa_supplicant.conf"

if test -f "${INCLUDE_COMMON}" ; then
  . ${INCLUDE_COMMON}
fi

Make_Temp "$0"
Read_HTML_Parameters
Process_Inputs
####################################################################################################################
# Is this a system being controlled by Network Manager? (Bullseye=NO, Bookworm=Yes)
# First verify nmcli command exists
command -v nmcli 2>&1 > /dev/null

if [[ $? -eq 0 ]] ; then
  # nmcli exists. Is network manager running?
  nmcli 2>&1 > /dev/null

  if [[ $? -eq 0 ]] ; then
    # NetworkManager is running - we will use this interface rather than /boot/wpa_supplicant
    NETMANAGER_RUNNING="Y"
  fi
fi
####################################################################################################################
# Read any text strings and remove html encoding
if [[ -n "${ssid}" ]] ; then
  SSID="$(Percent_Decode "${ssid}")"
  SSID_VALUE="${SSID}"
fi

if [[ -n "${password}" ]] ; then
  PASSWORD="$(Percent_Decode "${password}")"
  PASSWORD_VALUE="value='${PASSWORD}'"
fi

if [[ -n "${username}" ]] ; then
  USERNAME="$(Percent_Decode "${username}")"
  USERNAME_VALUE="username=${USERNAME}"
fi
####################################################################################################################
Emit_HTML_Headers
Do_Start_Body

echo '  <div class="background">'
####################################################################################################################
# This section cancels any changes that have been made already
if [[ "${command}" == "cancel" ]] && [[ -f "${WPA_SUPPLICANT_CONF}" ]] ; then
  rm -f "${WPA_SUPPLICANT_CONF}" 2> /dev/null

  echo '   <fieldset>'
  echo '    <legend><b>Note!</b></legend>'
  echo '    <p style="color:blue;font-size:20px;">Pending changes were cancelled!</p>'
  echo '    <p>Reverting to the configuration before the changes were made.</p>'
  echo '   </fieldset>'
  echo '   <br>'
fi
####################################################################################################################
# Subroutine to read the configured NetworkManager SSIDs and passwords
Read_SSID_and_Passwords()
{
  network_index=0

  for C in /etc/NetworkManager/system-connections/* NOTHING ; do
    [[ ! -f "${C}" ]] && continue
    [[ "${C}" == *.nmconnection ]] || continue
    sudo grep "type.*=.*wifi" "${C}" 2>&1 > /dev/null || continue
    SSID_ARRAY[${network_index}]=$(echo ${C} | sed 's/^.*\///g;s/.nmconnection//g')
    SECURITY=$(sudo nmcli -s -f 802-11-wireless-security.key-mgmt connection show "${SSID_ARRAY[${network_index}]}"| sed 's/^.*security.key-mgmt:[[:space:]]*//g')

    if [[ "${SECURITY}" == *"wpa-psk"* ]] ; then
      PSK_ARRAY[${network_index}]=$(sudo nmcli -s -f 802-11-wireless-security.psk connection show "${SSID_ARRAY[${network_index}]}" | sed 's/^.*security.psk:[[:space:]]*//g')
    elif [[ "${SECURITY}" == *"wpa-eap"* ]] ; then
      # This is enterprise wifi - do we want to show the user's password?
      #PSK_ARRAY[${network_index}]=$(sudo nmcli -s -f 802-1x.password connection show "${SSID_ARRAY[${network_index}]}" | sed 's/^.*802-1x.password:[[:space:]]*//g')
      PSK_ARRAY[${network_index}]="<i>Username and Password have been hidden</i>"
    else
      PSK_ARRAY[${network_index}]="${SECURITY}"
    fi

    ((network_index++))
  done
}
####################################################################################################################
# This section reads the wpa_supplicant.conf file into some BASH variables

network_index=0
outside_index=0
config_changed=''
[[ ! -f "${WPA_SUPPLICANT_CONF}" ]] && WPA_SUPPLICANT_CONF="/etc/wpa_supplicant/wpa_supplicant.conf"

if [[ -n "${NETMANAGER_RUNNING}" ]] ; then
  # Network Manager is running. Get wifi info using nmcli
  Read_SSID_and_Passwords

elif [[ ! -f "${WPA_SUPPLICANT_CONF}" ]] ; then
  echo '   <fieldset>'
  echo '    <legend><b>Error!</b></legend>'
  echo '    <p style="color:red;font-size:25px;">Cannot locate a wpa_supplicant.conf file!</p>'
  echo '    <p>ssh into the Pi and manually verify or repair the configuration.</p>'
  echo '   </fieldset>'

else
  # Network manager is not running - use wpa_supplicant info
  in_network_section=-1

  while read LINE ; do
    case "${LINE}" in
      "network={")
	# The wpa_passphrase command creates sections with a very specific format - this is the opening
	in_network_section=0
	;;

      "}")
	((network_index++))
	in_network_section=-1
	;;

      *)
	if [[ "${in_network_section}" -ge 0 ]] ; then
	  # We need to pick out some specific network section parameters, and just lump others together
	  IFS='=' read VARNAME VALUE <<< "${LINE}"

	  case "${VARNAME}" in
	    "ssid")
	      SSID_ARRAY[${network_index}]="${VALUE}"
	      ;;

	    "psk")
	      PSK_ARRAY[${network_index}]="${VALUE}"
	      ;;

	    "key_mgmt")
	      KEY_MGMT_ARRAY[${network_index}]="${VALUE}"
	      ;;

	    *)
	      if [[ ! -z "${LINE}" ]] ; then
		declare OTHER_ARRAY${network_index}[${in_network_section}]="${LINE}"
		((in_network_section++))
	      fi
	  esac

	else
	  if [[ ! -z "${LINE}" ]] ; then
	    # Not in network definition section
	    CONFIG_ARRAY[${outside_index}]="${LINE}"
	    ((outside_index++))
	  fi
	fi
    esac
  done < "${WPA_SUPPLICANT_CONF}"

  # Do some basic sanity checks
  if [[ ${outside_index} -eq 0 ]] ; then
    echo '   <fieldset>'
    echo '    <legend><b>Error!</b></legend>'
    echo '    <p style="color:red;font-size:25px;">The wpa_supplicant.conf file is missing common definitions!</p>'
    echo '    <p>ssh into the Pi and manually verify or repair the configuration.</p>'
    echo '   </fieldset>'
  fi

  if [[ ${network_index} -eq 0 ]] ; then
    echo '   <fieldset>'
    echo '    <legend><b>Error!</b></legend>'
    echo '    <p style="color:red;font-size:25px;">No networks defined in the wpa_supplicant.conf file!</p>'
    echo '    <p>The current wifi configuration will not attempt any connections.</p>'
    echo '   </fieldset>'
  fi
fi

# File is now read into a bunch of arrays - we can work on it
####################################################################################################################
# Has a new SSID and password been specified? We can add a new record
if [[ "${command}" == "new_ssid" ]] ; then
  if [[ -z "${ssid}" ]] ; then
    echo '   <fieldset>'
    echo '    <legend><b>Error!</b></legend>'
    echo '    <p style="color:red;font-size:25px;">No SSID specified!</p>'
    echo '    <p>Fill in the SSID text box.</p>'
    echo '   </fieldset>'

  else
    if [[ -n "${NETMANAGER_RUNNING}" ]] ; then
      # Verify that the new SSID is NOT already present
      for ((i=0 ; i < ${network_index} ; i++)) ; do
	[[ "${SSID^^}" != "${SSID_ARRAY[${i}]^^}" ]] && continue

	echo '   <fieldset>'
	echo '    <legend><b>Error!</b></legend>'
	echo "    <p style='color:red;font-size:25px;'>An SSID definition for <b>${SSID_ARRAY[${i}]}</b> already exists!</p>"
	echo "    <p>I'm refusing to define another SSID of the same name. Delete the existing one first.</p>"
	echo '   </fieldset>'
	SSID_ERR="Y"
      done

      if [[ -z "${SSID_ERR}" ]] ; then
	sudo nmcli connection add type wifi con-name "${SSID}" ifname wlan0 ssid "${SSID}" 2>&1 > /dev/null
	SSID_ARRAY[${network_index}]="${SSID}"

	# Is this enterprise wifi?
	if [[ -z "${USERNAME}" ]] ; then
	  if [[ -n "${password}" ]] ; then
	    # Only set password if non-blank. If blank, it means open network
	    sudo nmcli connection modify "${SSID}" wifi-sec.key-mgmt wpa-psk wifi-sec.psk "${PASSWORD}" 2>&1 > /dev/null
            PSK_ARRAY[${network_index}]="${PASSWORD}"
	  fi

	else
	  sudo nmcli connection modify "${SSID}" 802-11-wireless.ssid "${SSID}" 802-11-wireless-security.key-mgmt wpa-eap 802-1x.eap peap 802-1x.phase2-auth mschapv2 802-1x.identity "${USERNAME}" 802-1x.password "${PASSWORD}" 2>&1 > /dev/null
	  PSK_ARRAY[${network_index}]="<i>Username and Password have been hidden</i>"
	fi
      fi

      # Re-read the SSID and passwords because the order of the list may now have changed, and we need the correct indices!
      Read_SSID_and_Passwords

    else #Netmanager is not running
      # enclose the SSID and PSK in double quotes
      SSID_ARRAY[${network_index}]="\"${SSID}\""

      if [[ -n "${password}" ]] ; then
        # Only set password if non-blank. If blank, it means open network
        PSK_ARRAY[${network_index}]="\"${PASSWORD}\""
      fi

      config_changed='Y'
    fi

    SSID_VALUE=""
    PASSWORD_VALUE=""
    USERNAME_VALUE=""
    ((network_index++))
  fi
fi
###################################################################################################################
# Have any deletions been specified? If so, delete those records

if [[ "${command}" == "delete_network" ]] && [[ -n "${network_index}" ]] ; then
  for ((i=0 ; i < ${network_index} ; i++)) ; do
    SELNAME="Sel${i}"

    if [[ "${!SELNAME}" == 'Y' ]] ; then
      if [[ -n "${NETMANAGER_RUNNING}" ]] ; then
	# is this connection currently the active one? Refuse to delete if it is! Create an array where [0] is the SSID and [1] is UUID
	ACTIVE_CON=($(sudo nmcli con show --active 2> /dev/null | grep wifi))

	if [[ "${SSID_ARRAY[${i}]}" == "${ACTIVE_CON[0]}" ]] ; then
	  echo '   <fieldset>'
	  echo '    <legend><b>Error!</b></legend>'
	  echo '    <p style="color:red;font-size:25px;">I refuse to delete that wifi network!</p>'
	  echo "    <p>You are currently connected to <b>${SSID_ARRAY[${i}]}</b>!</p>"
	  echo '   </fieldset>'
	else
	  echo '   <fieldset>'
	  echo '    <legend><b>Deleting connection</b></legend>'
	  echo "    <p style=\"color:blue;font-size:20px;\">${SSID_ARRAY[${i}]}</p>"
	  echo -n '    <p>'

	  # Issue the nmcli command to remove the marked entry - but we need its UUID, so need to look it up
	  sudo nmcli -t --fields TYPE,NAME,UUID connection show | grep "802-11-wireless" | grep "${SSID_ARRAY[${i}]}" | sed "s/^.*${SSID_ARRAY[${i}]}://g" | while read line; do sudo nmcli connection delete uuid $line ; done
	  echo '   </p></fieldset>'
	  echo '   <br>'

	  SSID_ARRAY[${i}]=''
	  VAR="${SELNAME}=''"
	  eval ${VAR}
	  config_changed='Y'
	fi
      else

	SSID_ARRAY[${i}]=''
	VAR="${SELNAME}=''"
	eval ${VAR}
	config_changed='Y'
      fi
    fi
  done
fi
###################################################################################################################
# Has a connect command been issued? We will only try to connect to the FIRST selected network

if [[ "${command}" == "connect" ]] && [[ -n "${network_index}" ]] ; then
  for ((i=0 ; i < ${network_index} ; i++)) ; do
    SELNAME="Sel${i}"

    if [[ "${!SELNAME}" == 'Y' ]] ; then
      if [[ -n "${NETMANAGER_RUNNING}" ]] ; then
	# is this connection currently the active one? Refuse to connect to it if so!
	ACTIVE_CON=($(sudo nmcli con show --active | grep wifi))

	if [[ "${SSID_ARRAY[${i}]}" == "${ACTIVE_CON[0]}" ]] ; then
	  echo '   <fieldset>'
	  echo '    <legend><b>Error!</b></legend>'
	  echo '    <p style="color:red;font-size:25px;">Already connected!</p>'
	  echo "    <p>You are currently connected to <b>${SSID_ARRAY[${i}]}</b>!</p>"
	  echo '   </fieldset>'
	else
	  # Issue the nmcli command to connect to the specified network. This will fail if the connection barfs
	  sudo nmcli dev wifi connect "${SSID_ARRAY[${i}]}" 2>&1 > /dev/null

	  # Did it connect or did it fail??
	  if [[ $? -ne 0 ]] ; then
	    echo '   <fieldset>'
	    echo '    <legend><b>Error!</b></legend>'
	    echo '    <p style="color:red;font-size:25px;">Connection failed!</p>'
	    echo "    <p>Failed to connect to <b>${SSID_ARRAY[${i}]}</b>!</p>"
	    echo '   </fieldset>'
	  fi
	  # If the connection succeeded, there is no point in displaying anything because the browser will
	  # have lost its connectivity because of the change of SSID!
	fi
      else
	echo 'not yet supported'
	break
      fi
    fi
  done
fi
###################################################################################################################
if [[ "${command}" == "reboot" ]] && [[ "${reboot_confirm}" == 'Y' ]] ; then
  echo '    <fieldset>'
  echo '     <legend><b>Rebooting the Clock</b></legend>'
  echo "     <p>Please wait while the clock reboots</p>"
  echo '    </fieldset>'
  echo '  </div>'

  echo '  <div>'
  echo '    <p style="color:red;font-size:25px;">This page will reload in <span id="div_counter">30</span> seconds</p>'
  echo '  </div>'

  echo '  <script>'
  echo '    var counter = 30;'
  echo
  echo '    setInterval(function()'
  echo '    {'
  echo '      counter--;'

  echo '      if (counter >= 0)'
  echo '      {'
  echo '        var span;'
  echo '        span = document.getElementById("div_counter");'
  echo '        span.innerHTML = counter;'
  echo '      }'

  echo '      if (counter <= 0)'
  echo '      {'
  echo '        window.location.replace("/cgi-bin/wifi.cgi");'
  echo '      }'

  echo '    }, 1000);'

  echo '  </script>'

  # Issue the command to reboot the raspberry pi
  sudo reboot

else
  if [[ "${command}" == "reboot" ]] ; then
    echo '   <fieldset>'
    echo '    <legend><b>Not rebooting!</b></legend>'
    echo '    <p style="color:red;font-size:25px;">Box not selected!</p>'
    echo '    <p>Select the box before clicking the reboot button in order to reboot the Pi.</p>'
    echo '   </fieldset>'
  fi

  ####################################################################################################################
  # Display the existing wifi settings (if any exist)
  if [[ ${network_index} -gt 0 ]] ; then

    # Read the list of currently accessible wifi networks two different ways, then combine the two lists
    IFS_BAK="${IFS}"
    IFS="
"
    AN1=( $(sudo iwlist wlan0 scan 2> /dev/null | grep ESSID | sed 's/"//g;s/^.*://g' | sort | uniq) )
    IFS="${IFS_BAK}"

    AN2=$(nmcli device wifi list | tail -n +2 | sed 's/^[ *]*[A-F:0-9]*[ ]*//g;s/ .*//g')
    AN3=( $(sort <(printf '%s\n' "${AN1}") <(printf '%s\n' "${AN2}") | uniq) )

    # Remove the hidden SSIDs (which are called '--')
    for i in "${!AN3[@]}" ; do
      [[ "${AN3[$i]}" == "--" ]] && unset -v 'AN3[$i]'
    done

    ACCESSIBLE_NETWORKS=( "${AN3[@]}" )

    echo "   <form action='${THIS_SCRIPT}' method='get'>"
    echo '    <fieldset>'
    echo '     <legend><b>Wifi Settings</b></legend>'
    Include_Defaults

    echo "     <div class='tip'>"
    echo -n "      <span class='text'>"
    echo -n "If two or more SSIDs have been configured, when wifi connectivity is lost, the clock will attempt to reconnect with any of the configured SSIDs to reestablish a LAN connection.<br><br>If the clock cannot reestablish a link to any of the configured SSIDs"

    # Does this Pi have a wired ethernet port?
    ifconfig | grep eth0 > /dev/null

    if [[ ${?} -eq 0 ]] ; then
      echo -n " (and the wired LAN interface is not connected)"
    fi

    echo -n ", the clock's digits will blink intermittently to alert you.<br><br>"
    echo -n "Without a LAN connection, the clock will continue to function and alarms played from media stored on the clock will work as expected. Internet and LAN based media will not be accessible without LAN connectivity and these alarms will revert to <b>local</b> fallback media (if configured).<br><hr>"
    echo -n "If LAN connectivity remains down continuously for a week, the clock will reboot.<br><hr>"
    echo -n "Refer to the list of visible networks below. If you can't see your preferred SSID in the list and the SSID has not been hidden, the clock won't be able to connect to it!<hr>"
    echo -n "To get the clock to connect to a newly configured SSID, select it by ticking the box and clicking 'Connect to Selected Network'. If the connection is successful, this browser window will then hang.<br><br>When the clock's wifi connects to a different SSID, you will only be able to reconnect to the clock via a device connected to that different SSID.</span>"

    SSID_CMD=$(which iwgetid)

    if [[ -n "${SSID_CMD}" ]] ; then
      CMD_OUT=$(${SSID_CMD})
      CURRENT_SSID=$(echo ${CMD_OUT#*\"} | sed 's/\"$//g')
      echo "      Current wifi connection: <b>${CURRENT_SSID}</b><br><br>"
    fi

    echo "      <label>Select the box and click the 'connect' button to reconnect to a different wifi network, or the 'delete' button to remove associated Network/SSID:<br>"
    echo "       <table class='striped'>"
    echo '        <tr>'
    echo '         <th>Network / SSID</th>'
    echo '         <th>Password or Encrypted Password</th>'
    echo '        </tr>'

    # Loop through each network
    for ((i=0 ; i < ${network_index} ; i++)) ; do
      [[ -z "${SSID_ARRAY[${i}]}" ]] && continue

      SELNAME="Sel${i}"
      echo '        <tr>'
      CHECKED=""
      [[ "${!SELNAME}" == 'Y' ]] && CHECKED="checked"
      echo "         <td><input type='checkbox' name='${SELNAME}' value='Y' ${CHECKED}>${SSID_ARRAY[${i}]}</td>"

      # If the password is blank, assume its an open SSID (not encrypted)
      if [[ -z "${PSK_ARRAY[${i}]}" ]] ; then
	# Blank
	echo "         <td><i>No password. Open network.</i></td>"

      else
	echo "         <td>${PSK_ARRAY[${i}]}</td>"
      fi

      echo '        </tr>'
    done

    echo '       </table>'
    echo '      </label>'

    if [[ "${network_index}" -gt 1 ]] ; then
      echo "      <button type='submit' class='blue' name='command' value='connect'>Connect to Selected Network</button>"
    fi

    echo "      <button type='submit' class='red' name='command' value='delete_network'>Delete Selected Networks</button><br><br>"
    echo '     </div>'

    echo "     <div class='tip'>"
    echo '      <br><label>Add new network details:</label><br>'
    Do_Text_Input 'ssid' "${SSID_VALUE}" 'Network / SSID' "80"
    Do_Text_Input 'password' "${PASSWORD_VALUE}" 'Password - leave blank if network is open' "80"
    Do_Text_Input 'username' "${USERNAME_VALUE}" 'Username - only enter a username for networks using enterprise wifi (i.e. EAP), otherwise leave this blank' "80"
    echo "      <span class='text'>For almost all WiFi networks, you will only need to specify the Network / SSID and Password and then leave the Username field blank.<br><br>Complete all three fields (including the Username) to connect to an access point that uses the EAP enterprise WiFi protocol. You can recognise these access points by trying to connect a smart-phone or laptop and finding that you are asked for a Username/Password <i>BEFORE</i> the WiFi indicates that it has connected. Campus accommodation such as student accommodation often implements this kind of enterprise WiFi.<br><br>This web page release will not configure WiFi networks that request a Username/Password via a web browser (i.e. you receive the Username/Password prompt in a web browser <i>AFTER</i> the WiFi has connected.) or enterprise WiFi networks that require a certificate. You can however search the Internet to learn how to manually configure such networks via the clock's ssh command prompt.</span>"
    echo '     </div>'

    echo '     <br>'
    echo "     <div class='tip'>"
    echo "      <button type='submit' class='green' name='command' value='new_ssid'>Add Specified Network Details</button>"
    echo "      <span class='text'>Clicking here will cause changes to the SSID, password (if specified) and username (if specified) to be saved.<br><br>Only specify a non-blank username if the SSID is an enterprise network! If you specify a username and it's not an enterprise network, you'll be unable to connect!</span>"
    echo '     </div>'

    if [[ ${#ACCESSIBLE_NETWORKS[@]} -gt 0 ]] ; then
      echo '     <hr>'
      echo "     <div class='tip'>"
      echo "      <table class='striped'>"
      echo '       <tr>'
      echo '        <th>List of currently visible networks / SSIDs</th>'
      echo '       </tr>'

      for i in "${ACCESSIBLE_NETWORKS[@]}" ; do
	echo "        <tr><td>${i}</td></tr>"
      done

      echo '      </table>'
      echo "      <span class='text'>Networks with hidden SSIDs will not appear in the list of currently visible networks. The clock will nevertheless be able to connect to a hidden network if it is within reliable wifi range.<br><br>Detection of a network / SSID at the fringe of its reception area will be unreliable! When you reload this web page, fringe networks will randomly appear and disappear from the list. The clock will not work reliably if the network is not reliably reachable.</span>"
      echo '     </div>'
    fi

    echo '    </fieldset>'
    echo '   </form>'
  fi
  ####################################################################################################################
  # This section uses nmcli to modify connections, or rewrite any changes to the wpa_supplicant.conf file
  if [[ "${config_changed}" == 'Y' ]] && [[ -z "${NETMANAGER_RUNNING}" ]] ; then
    rm -f "${TMP}" > /dev/null 2>&1

    # Two sections to the file....
    # Write the header section with global variables
    for ((i=0 ; i < ${outside_index} ; i++)) ; do
      echo "${CONFIG_ARRAY[$i]}" >> "${TMP}"
    done

    # Write each individual bracketed network section
    N=0

    for ((i=0 ; i < ${network_index} ; i++)) ; do
      [[ -z "${SSID_ARRAY[${i}]}" ]] && continue

      echo >> "${TMP}"
      echo "network={" >> "${TMP}"
      printf "\tssid=%s\n" "${SSID_ARRAY[${i}]}" >> "${TMP}"

      # If the password is blank, assume its an open SSID (not encrypted)
      if [[ -z "${PSK_ARRAY[${i}]}" ]] ; then
	# Blank
        [[ -z "${KEY_MGMT_ARRAY[${i}]}" ]] && printf "\tkey_mgmt=NONE\n" >> "${TMP}"

      else
	printf "\tpsk=%s\n" "${PSK_ARRAY[${i}]}" >> "${TMP}"
      fi

      [[ ! -z "${KEY_MGMT_ARRAY[${i}]}" ]] && printf "\tkey_mgmt=%s\n" "${KEY_MGMT_ARRAY[${i}]}" >> "${TMP}"

      # OTHER_ARRAYn may or may not exist, so we need to check if the variable exists before writing contents
      # Because we are using a variable array name, we need to dance around expressing it.... we will use
      # bash's ability to use the contents of a variable to reference a variable (exclamation mark)
      VAR="OTHER_ARRAY${i}[0]"

      if [[ -n "${!VAR+x}" ]] ; then
	# The variable exists - now we need to work out the length of that variable-name array
	# Its tricky to find the length of an array that has a variable name. I will use eval.
        VAR="COUNT=\${#OTHER_ARRAY${i}[@]}"
	eval ${VAR}

	# The number of elements in the array is now in a variable called COUNT
	for ((j=0 ; j < ${COUNT} ; j++)) ; do
	  VAR="OTHER_ARRAY${i}[${j}]"
	  printf "\t%s\n" "${!VAR}" >> "${TMP}"
	done
      fi

      echo "}" >> "${TMP}"
      ((N++))
    done

    network_index=$N # (Re)Set to the number of networks that were written

    # If the file wrote OK, we can copy to the /boot partition where it will be activated upon next boot
    if [[ -f "${TMP}" ]] ; then
      WPA_SUPPLICANT_CONF="/boot/wpa_supplicant.conf"

      # The boot partition can be written by root. It can't be written by a normal user.
      rm -f "${WPA_SUPPLICANT_CONF}" > /dev/null 2>&1
      mv "${TMP}" "${WPA_SUPPLICANT_CONF}" > /dev/null 2>&1
    fi
  fi
  ####################################################################################################################
  # Button to abort the wifi configuration menu
  echo "   <form action='${THIS_SCRIPT}' method='get'>"
  echo '    <fieldset>'
  echo "     <input type='checkbox' name='reboot_confirm' value='Y'> Select this box before clicking the reboot button below to trigger a reboot.<br>"

  echo "     <div class='tip'>"
  echo "      <button type='submit' class='blue' name='command' value='reboot'>REBOOT THE CLOCK</button>"
  echo "      <span class='text'>Clicking here when the box is selected will cause any changes that have been made to be applied. Any unsaved changes to be discarded and lost so think before clicking! When the Pi boots, if a connection cannot be established to at least one network, the clock will not work properly and you may require a wired connection to recover.</span>"
  echo '     </div>'

  echo '    </fieldset>'
  echo '   </form>'

  echo "   <form action='/index.html'>"
  echo '    <fieldset>'
  echo "     <div class='tip'>"
  echo "      <button type='submit' class='red' >Return to main page</button>"
  echo "      <span class='text'>Clicking here will cause any unsaved changes in this form to be discarded and lost</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '   </form>'
  ####################################################################################################################
  if [[ "${WPA_SUPPLICANT_CONF}" == "/boot/wpa_supplicant.conf" ]] ; then
    echo "   <form action='${THIS_SCRIPT}' method='get'>"
    echo '    <fieldset>'

    if [[ ${network_index} -eq 0 ]] ; then
      echo '     <p style="color:red;font-size:25px;">WARNING! No wireless networks are defined!</p>'
      echo '     <p><b>If you reboot now, your clock will only work as expected if it has a wired Ethernet connection.</b></p>'
    fi

    echo "     <button type='submit' class='red' name='command' value='cancel'>Cancel pending wifi config changes</button>"
    echo '    </fieldset>'
    echo '   </form>'

    echo '   <fieldset>'
    echo '    <legend><b>Note!</b></legend>'
    echo '    <p style="color:blue;font-size:20px;">Original wifi settings still being used!</p>'
    echo '    <p>Reboot the pi for new wifi settings to take effect.</p>'
    echo '   </fieldset>'
  fi

  echo '  </div>'

  Insert_Debug_Section
####################################################################################################################
fi

echo ' </body>'
echo '</html>'

[[ -f "${TMP}" ]] && rm -f "${TMP}"
exit 0
