#!/bin/bash
#                           Digital Alarm Clock
#                Copyright (C) 2023,2024 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
####################################################################################################################
THIS_SCRIPT="/cgi-bin/index.cgi"

# Load common functions and data
INCLUDE_COMMON="common.sh"

if test -f "${INCLUDE_COMMON}" ; then
  . ${INCLUDE_COMMON}
fi

Make_Temp "$0"
Read_HTML_Parameters
Process_Inputs
Emit_HTML_Headers icons,noborder

if [[ "${skip_to_track}" =~ ${IsNumber} ]] ; then
  $EXE -j "${skip_to_track}" > /dev/null 2>&1
fi

echo ' <body>'
echo '  <div class="background">'
####################################################################################################################
VERSION=$($EXE -q)

echo '   <fieldset>'
echo "    <legend><b>Raspberry Pi Alarm Clock executable version ${VERSION} / Web interface version 1.1 (5th July 2024)</b></legend>"

echo "    <table class='noborder'>"
echo '     <tr>'
echo '      <td>'
echo "       <form action='/cgi-bin/alarms.cgi'>"
echo "        <button class='blue' type='submit'>Alarm Configuration<br><i class='fa fa-clock-o fa-2x' style='padding-top: 7px'></i></button>"
echo '       </form>'
echo '      </td>'

echo '      <td>'
echo "       <form action='/cgi-bin/setup.cgi'>"
echo "        <button class='blue' type='submit'>Clock Setup<br><i class='fa fa-cogs fa-2x' style='padding-top: 7px'></i></button>"
echo '       </form>'
echo '      </td>'

echo '      <td>'
echo "       <form action='/cgi-bin/wifi.cgi'>"
echo "        <button class='blue' type='submit'>WiFi Configuration<br><i class='fa fa-wifi fa-2x' style='padding-top: 7px'></i></button>"
echo '       </form>'
echo '      </td>'

echo '      <td>'
echo "       <form action='/cgi-bin/mediaplayer.cgi'>"
echo "        <button class='green' type='submit'>Media Player<br><i class='fa fa-music fa-2x' style='padding-top: 7px'></i></button>"
echo '       </form>'
echo '      </td>'
echo '     </tr>'
echo '    </table>'

echo "    This clock's name and type: <b>$(hostname)</b> (${CLOCK_TYPE_STRING})"
echo '   </fieldset>'

echo '   <br>'
Emit_HTML_Alarm_Table

echo '   <br>'
echo "   <small>Click <a href='/cgi-bin/buttons.cgi' target='_blank' rel='noopener noreferrer'>here</a> to open a tab with information about using the buttons.</small><br><br>"
echo "   <small>Click <a href='/schematics.html' target='_blank' rel='noopener noreferrer'>here</a> to open a tab with the schematic diagrams.</small><br><br>"

# Are there any other clocks that have been discovered?
readarray -t ClockList < <( $EXE -c clocks )

if [[ ${#ClockList[@]} -gt 0 ]] ; then
  echo '   <fieldset>'
  echo "    <legend><b>Other clocks on this network</b></legend>"

  # Display the clock list. It is in CSV format - display in a table
  echo "    <table class='striped'>"
  echo '     <tr>'
  echo '      <th>IP address</th>'
  echo '      <th>System Name</th>'
  echo '      <th>Clock type</th>'
  echo '     </tr>'

  for ((i=0 ; i < ${#ClockList[@]} ; i++)) ; do
    # There are four columns delimited by commas
    IFS=',' read -ra Array <<< "${ClockList[$i]}"
    echo '     <tr>'

    # If the first character in the first column is an asterisk, this is an error message
    if [[ "${Array[0]}" == ERROR* ]] ; then
      echo "      <td><b>${Array[0]}</b></td></tr>"
      continue
    fi

    # Not an error - display the IP address as a link
    echo "      <td><a href='http://${Array[0]}${THIS_SCRIPT}'>${Array[0]}</a></td>"

    # Display the system name in next column - make it a link if we know the domain name on this network
    if [[ -z "${DOMAIN_NAME}" ]] ; then
      echo "      <td>${Array[3]}</td>"
    else
      echo "      <td><a href='http://${Array[3]}${DOMAIN_NAME}${THIS_SCRIPT}'>${Array[3]}</a></td>"
    fi

    # Display the system type in the last column
    echo "      <td>${Array[1]}</td>"
    echo '     </tr>'
  done

  echo '    </table>'
  echo '   </fieldset>'
fi

# Is the media player currently playing?
CURRENT_READINGS=$($EXE -a)

# Set VOLUME and DEFAULT_MEDIA_DURATION based on the values that the clock is currently using
if [[ -n "${CURRENT_READINGS}" ]] ; then
  CURRENT_DURATION=$(echo $CURRENT_READINGS | cut -d, -f3)
  CURRENT_BLUETOOTH=$(echo $CURRENT_READINGS | cut -d, -f4)

  if [[ -n "${CURRENT_DURATION}" ]] && [[ "${CURRENT_DURATION}" -ge 0 ]] && [[ "${CURRENT_DURATION}" -le 180 ]] ; then
    echo '   <br>'
    Insert_Current_Playlist

  elif [[ -n "${CURRENT_BLUETOOTH}" ]] && [[ "${CURRENT_BLUETOOTH}" -gt 0 ]] ; then
    echo '   <br>'
    Insert_Current_Playlist
  fi
fi

echo '  </div>'
###################################################################################################################
####################################################################################################################
Insert_Debug_Section
echo ' </body>'
echo '</html>'
####################################################################################################################
# Clean out playlist files that are no longer required
# NOTHING is to avoid errors when playlist directory is empty
for i in ${PLAYLIST_DIR}/*.m3u NOTHING ; do
  [[ -f "${i}" ]] || continue
  [[ "${i}" == "${FALLBACK_PLAYLIST}" ]] && continue

  INDEX_FILENAME="${i%.m3u}.index"

  # Does the file $i exist in ${ALARM_FILE}
  grep -q "${i}" "${ALARM_FILE}"

  # Remove the playlist if it is not in the alarm file, and it was created more than 90 days ago
  if [[ $? -ne 0 ]] && [[ $(find "${i}" -mtime +90) ]] && [[ ! -f "${INDEX_FILENAME}" || $(find "${INDEX_FILENAME}" -mtime +90) ]] ; then
    rm -f "${i}" "${INDEX_FILENAME}" 2> /dev/null # Remove the unused playlist and index (if there)
  fi
done
####################################################################################################################
[[ -f "${TMP}" ]] && rm -f "${TMP}"
exit 0
