//////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023,2024 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock software.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////
#define _GNU_SOURCE // otherwise strcasestr won't be loaded

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <getopt.h>
#include <pthread.h>
#include <signal.h>
#include <stdint.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdnoreturn.h>
#include <time.h>
#include <utime.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/time.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <math.h>
#include <mqueue.h>
#include <mpv/client.h>
#include <inttypes.h>
#include <limits.h>
#include <ifaddrs.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <linux/if_packet.h>
#include <linux/rtnetlink.h>
///////////////////////////////////////////////////////////////////////////////////////////
// The GPIO library has two forms: a daemonised version, and a direct calling version
// Use the daemonised version if more than one programme (or more than one running instance of
// the same programme) is going to be calling the library. i.e. use the daemonised version
// when sharing access to the GPIO library from multiple programmes.
//
// In this case, there's only one programme and it is going to access all 28 GPIO pins. So
// you can choose either one. The daemonised version seems to work a little better, so we
// will stick with that one. But the code has been tested with the non daemonised version too.
//
// Note: when linking, the daemonised calls need to be linked with a different C library
// than the direct calls!
// The calls are slightly different for the daemonised and direct versions of the library

// use -DUSE_GPIO_DAEMON compiler switch in the Makefile

// Map the daemonised calls to the same names as the direct calls, to simplify the source code
#ifdef	USE_GPIO_DAEMON
  #include <pigpiod_if2.h>

  #define gpioInitialise()		(GPIOD_daemon_id = pigpio_start (NULL, NULL))
  #define gpioTerminate()		(pigpio_stop (GPIOD_daemon_id))

  #define gpioSetMode(gpio,mode)	(set_mode (GPIOD_daemon_id, gpio, mode))
  #define gpioSetPullUpDown(gpio,mode)	(set_pull_up_down (GPIOD_daemon_id, gpio, mode))

  #define gpioWrite(gpio,level)		(gpio_write (GPIOD_daemon_id, gpio, level))
  #define gpioRead(gpio)		(gpio_read (GPIOD_daemon_id, gpio))

  #define spiOpen(chan,baud,flags)	(spi_open (GPIOD_daemon_id, chan, baud, flags))
  #define spiClose(handle)		(spi_close (GPIOD_daemon_id, handle))
  #define spiRead(handle,buffer,count)	(spi_read (GPIOD_daemon_id, handle, buffer, count))
  #define spiWrite(handle,buffer,count)	(spi_write (GPIOD_daemon_id, handle, buffer, count))
  #define spiXfer(handle,txbuf,rxbuf,count) (spi_write (GPIOD_daemon_id, handle, txbuf, rxbuf, count))

  #define gpioPWM(gpio,duty)		(set_PWM_dutycycle (GPIOD_daemon_id, gpio, duty))
  #define gpioGetPWMdutycycle(gpio)	(get_PWM_dutycycle (GPIOD_daemon_id, gpio))
  #define gpioSetPWMrange(gpio,range)	(set_PWM_range (GPIOD_daemon_id, gpio, range))
  #define gpioGetPWMrange(gpio)		(get_PWM_range (GPIOD_daemon_id, gpio))
  #define gpioGetPWMrealRange(gpio)	(get_PWM_real_range (GPIOD_daemon_id, gpio))
  #define gpioSetPWMfrequency(gpio,freq) (set_PWM_frequency (GPIOD_daemon_id, gpio, freq))
  #define gpioGetPWMfrequency(gpio)	(get_PWM_frequency (GPIOD_daemon_id, gpio))

  #define gpioHardwareClock(gpio,freq)	(hardware_clock (GPIOD_daemon_id, gpio, freq)) // Works with GPIO 4, 5, 6, 20, 21
  #define gpioHardwarePWM(gpio,freq,duty) (hardware_PWM (GPIOD_daemon_id, gpio, freq, duty)) // Works with GPIO 12, 13, 18, 19

#else
  #include <pigpio.h>
#endif
///////////////////////////////////////////////////////////////////////////////////////////
typedef struct timespec		TS ;
typedef struct tm		TM ;
typedef struct timeval          TV ;			// System time in seconds and microseconds
typedef struct utimbuf		UT ;			// Structure for setting modification and creation times
typedef struct stat		ST ;			// structure containing information about a file
#//////////////////////////////////////////////////////////////////////////////////////////
#define VERSION			"1.4 (3rd July 2024)"	// What version of the C code is this?

#define BUS_WIDTH		((int)(sizeof(void*) * CHAR_BIT))	// Is this compiled for 32 bit or 64 bit???

#define NS_PER_SECOND		(1000000000)
#define THREAD_START_DELAY	(100000)		// THREAD_START_DELAY is expressed in nanoseconds (i.e. 100us)
#define HUNDRED_MS		(NS_PER_SECOND / 10)	// HUNDRED_MS is expressed in nanoseconds (i.e. 100ms)
#define QUARTER_SECOND		(NS_PER_SECOND / 4)	// QUARTER_SECOND is expressed in nanoseconds (i.e. 250ms)
#define HALF_SECOND		(NS_PER_SECOND / 2)	// HALF_SECOND is expressed in nanoseconds (i.e. 500ms)
#define PERIODIC_INTERRUPT	(25000)			// PERIODIC_INTERRUPT is expressed in microseconds (i.e. 25ms)

#define INTERRUPTS_PER_SEC	(1000000 / PERIODIC_INTERRUPT) // 40 if PERIODIC_INTERRUPT is 25ms

#define WRITE_BYTE_DELAY	(5000)			// 5000ns = 5 microseconds delay when writing bytes to the display latches

// The following divisions are integer divisons. Be careful that you know the results are also integer!
// Note that the calculated values need to be able to fit into 8 bits!
#define AMP_SCAN_FREQ		(INTERRUPTS_PER_SEC / 20)	// run 20 times per second
#define BUTTON_SCAN_FREQ	(INTERRUPTS_PER_SEC / 10)	// run 10 times per second
#define ALARM_SCAN_FREQ		(INTERRUPTS_PER_SEC / 5)	// run 5 times per second
#define LDR_SCAN_FREQ		(INTERRUPTS_PER_SEC / 4)	// run 4 times per second
#define MPV_SCAN_FREQ		(INTERRUPTS_PER_SEC / 3)	// run ~3 times per second (there's a small rounding error here)
#define MPV_CHECK_FREQ		(INTERRUPTS_PER_SEC)		// run once per second
#define NET_CHECK_FREQ		(INTERRUPTS_PER_SEC)		// run once per second
#define PEER_CHECK_FREQ		(0xff)				// run every six and a bit seconds
#define BLUETOOTH_CHECK_FREQ	(INTERRUPTS_PER_SEC / 4)	// run 4 times per second

// The following are not limited to 8 bits
#define ONE_HOUR		(3600)
#define EIGHT_HOURS		(ONE_HOUR * 8)
#define SIXTEEN_HOURS		(ONE_HOUR * 16)
#define ONE_DAY			(ONE_HOUR * 24)
#define TWO_DAYS		(ONE_DAY * 2)
#define ONE_WEEK		(ONE_DAY * 7)
#define MPV_HOLDON_SECONDS	(10 * 60)			// Hold MPV on for 10 min before stopping. This retains the current playlist
#define NET_FAILED_DOWNCOUNT	(ONE_WEEK)			// If the network has been down for ONE WEEK, reboot the clock just
								// in case there was a network lock-up. Don't worry, the clock will keep going!

#define SETUP_FILE_RELOAD_DELAY	(INTERRUPTS_PER_SEC * 90 * 60)	// 90 minutes
#define SETUP_FILE_TIMER_DELAY	(30 * BUTTON_SCAN_FREQ)		// 30 seconds
#define BUTTON_REPEAT_WAIT	(BUTTON_SCAN_FREQ * 1.0)	// 1.0 seconds
#define BUTTON_REPEAT_HALF_WAIT	(BUTTON_REPEAT_WAIT / 2.0)	// 0.5 seconds
#define SLOW_REPEAT		(BUTTON_SCAN_FREQ * 3)		// 3 seconds at 2 per second

#define FILENAME_SIZE		(64)		// bytes/characters: max length of the name of this programme
#define TIME_STRING_LEN		(26)		// bytes/characters: length of a time and date string generated by Display_Time
#define LOG_STRING_LEN		(2000)		// bytes/characters
#define SETUP_FIELD_LEN		(200)		// bytes/characters
#define LONGER_STR_LEN		(500)		// bytes/characters
#define MODSTR_LEN		(100)		// bytes/characters
#define MAX_NUM_FORK_PARAMS	(15)

#define MAX_SIGNED_MS		(0x7fff)	// This is a 16 bit signed value: max +ve size of calculated difference returned by Time_Difference_ms
#define MIN_SIGNED_MS		(0x8000)	// This is a 16 bit signed value: max -ve size of calculated difference returned by Time_Difference_ms

#define MIN_LDR_THRESH		(100)
#define MAX_LDR_THRESH		(4000)
#define MIN_PWM_RATIO		(0.1)
#define LOW_PWM_RATIO		(0.2)
#define MAX_PWM_RATIO		(1.0)
#define DEFAULT_VOLUME		(23)		// This is the defaut for the PAM8407 amp chip, on a scale of 0=min, 31=max
#define MAX_VOLUME		(31)		// This is the max volume level the PAM8407 amp chip will produce.
#define MAX_MINIMUM_VOLUME	(15)
#define MAX_VOL_OFFSET		(24)
#define DEFAULT_SNOOZE_DURATION	(10 * 60)	// 10 minutes expressed in seconds
#define DEFAULT_DURATION	(60)		// 60 minutes expressed in minutes
#define MAX_DURATION		(180)		// 180 minutes expressed in minutes
#define DOT_FLASH_COUNT		(6)		// ticks

// The following fractions are used for moving average calculations
// With a fraction (time constant) of 0.98, a step change will reach 50% in 35  iterations, 90% in 114 iterations, and 99% in 228 iterations
// With a fraction (time constant) of 0.97, a step change will reach 50% in 23  iterations, 90% in 76 iterations, and 99% in 152 iterations
// With a fraction (time constant) of 0.95, a step change will reach 50% in 14  iterations, 90% in 45 iterations, and 99% in 90 iterations
// With a fraction (time constant) of 0.93, a step change will reach 50% in 10  iterations, 90% in 32 iterations, and 99% in 64 iterations
// With a fraction (time constant) of 0.9,  a step change will reach 50% in 7   iterations, 90% in 22 iterations, and 99% in 44 iterations
// With a fraction (time constant) of 0.8,  a step change will reach 50% in 3   iterations, 90% in 10 iterations, and 99% in 21 iterations
// With a fraction (time constant) of 0.7,  a step change will reach 50% in 2   iterations, 90% in 6.5 iterations, and 99% in 13 iterations
// With a fraction (time constant) of 0.6,  a step change will reach 50% in 1.3 iterations, 90% in 4.5 iterations, and 99% in 9 iterations
// With a fraction (time constant) of 0.5,  a step change will reach 50% in 1.0 iterations, 90% in 3.25 iterations, and 99% in 6.5 iterations
// With a fraction (time constant) of 0.4,  a step change will reach 50% in 0.8 iterations, 90% in 2.5 iterations, and 99% in 5 iterations
#define MOV_AV_FRACT	(0.93)
///////////////////////////////////////////////////////////////////////////////////////////
// Define the GPIO pin functions here. Note that the GPIO names differ from the Raspberry Pi pin numbers!

//      Label name              GPIO number        40 pin expansion connector pin number
#define D0			(4)		// 7
#define D1			(18)		// 12
#define D2			(17)		// 11
#define D3			(27)		// 13
#define D4			(23)		// 16
#define D5			(22)		// 15
#define D6			(24)		// 18
#define D7			(25)		// 22

#define A0			(6)		// 31
#define A1			(5)		// 29

#define EE			(12)		// 32

#define DIM_PWM			(13)		// 33

#define AMP_ENABLE		(19)		// 35
#define RADIO_ENABLE		(10)		// 19
#define VOLUP			(14)		// 8
#define VOLDOWN			(15)		// 10

#define CS			(8)		// 24
#define MISO			(9)		// 21
#define SCLK			(11)		// 23

#define TOG1a			(21)		// 40    B_PLUS
#define TOG1b			(20)		// 38    B_MINUS
#define PB1			(26)		// 37    B_STOP_TOGGLE
#define PB2			(16)		// 36    B_DURATION
#define PB3			(0)		// 27    B_MEDIA
#define TOG2a			(1)		// 28    B_ALARM1
#define TOG2b			(7)		// 26    B_ALARM2
#define TOG3a			(3)		// 5     B_ALARM3
#define TOG3b			(2)		// 3     B_ALARM4

// The following assigns the button inputs to a bitmask to represent all buttons in one 16 bit variable
#define B_PLUS			(0x001)		// TOG1a up/+
#define B_MINUS			(0x002)		// TOG1b down/-
#define B_STOP_TOGGLE		(0x004)		// PB1 Snooze / Stop / Toggle
#define B_DURATION		(0x008)		// PB2 Duration / Sleep
#define B_MEDIA			(0x010)		// PB3 Media functions
#define B_ALARM1		(0x020)		// TOG2a Alarm 1
#define B_ALARM2		(0x040)		// TOG2b Alarm 2
#define B_ALARM3		(0x080)		// TOG3a Alarm 3
#define B_ALARM4		(0x100)		// TOG3b Alarm 4
#define B_REMOTE		(0x800)		// A dummy 'button' set if the button sequence originated from a remote clock

#define NUM_ALARM_BUTTONS	(4)		// B_ALARM1 .. B_ALARM4
///////////////////////////////////////////////////////////////////////////////////////////
#define ADDR_SECONDS		(0x00)		// address of the seconds LEDs
#define ADDR_MINUTES		(0x01)		// address of the minutes LEDs
#define ADDR_HOURS		(0x02)		// address of the hours LEDs
#define ADDR_LATCH		(0x03)		// address of the 8 bit latch

#define TOP_COLON		(0x40)		// top LED of the colon between hours and minutes
#define BOT_COLON		(0x80)		// bottom LED of the colon between hours and minutes

#define SPI_MAIN		(0)		// I cannot find this documented anywhere. It is SPI0 on the pi
#define SPI_BAUDRATE		(800000)	// 800 kHz gives 50ksps which is max at 2.7V. Could go higher at 3.3V, but this is OK.

// For PWM frequency options, refer to the options defined at http://abyz.me.uk/rpi/pigpio/pdif2.html#set_PWM_frequency
// The alarm clock is configured to run pigpiod at a 10us sample rate
// Potential SW frequency choices: 4k, 2k, 1k, 800, 500, 400, 250, 200, 160, 125, 100, 80, 50, 40, 25, 20, 10, 5
#define PWM_FREQ		(50)		// Hz (for HW PWM: Frequencies above 30 MHz unlikely to work)

// When the mpv media player is running, there can be jitter on the software PWM. I've found this behaves differently on the
// Pi4 compared with the Pi3 (haven't tested this situation on other models). Changing the PWM frequency will *reduce* the
// visibility of the jitter (but not eliminate the flicker). I needed to choose different alternate PWM rates for the Pi3 and 4.
#define LOW_PI4_PWM_FREQ	(20)		// Hz (Applied on Pi4 when mpv media player running & PWM ratio low to reduce flicker due to SW PWM inaccuracy)
#define LOW_PI3_PWM_FREQ	(40)		// Hz (Applied on Pi3 etc when mpv media player running & PWM ratio low to reduce flicker due to SW PWM inaccuracy)

#define SW_PWM_RANGE_MAX	(255)		// The fully off to fully on range will be 0..PWM_RANGE (library supports 0..40k)
#define HW_PWM_RANGE_MAX	(1000000)	// HW PWM supports duty from 0 .. 1M. This cannot be changed.
#define PWM_RANGE		SW_PWM_RANGE_MAX // Because we are using SW PWM
///////////////////////////////////////////////////////////////////////////////////////////
#if HOST_NAME_MAX < 128
  #define NAME_LEN	(128)			// max number of characters in name
#else
  #define NAME_LEN	HOST_NAME_MAX
#endif

// We want to transfer the alarm file in a single UDP frame, so we need to determine the maximum size of that file.
// On the wire, ethernet frames can have varying sizes, but are typically max 1518 bytes. Then when you add the IP
// and UDP overheads, you end up with less than 1500 bytes of payload. However the Linux operating system will
// automatically fragment larger frames and send them as a series of smaller frames, and then reassemble them.
// So we can
#define MAX_NUM_ALARMS		(20)		// A bit overkill - but we will nominally support 20 different alarms
#define ONE_LINE_ALARM_FILE	(100)		// nominal average length of an alarm definition in the alarms.csv file
#define PACKET_SIZE		((MAX_NUM_ALARMS+1) * ONE_LINE_ALARM_FILE)	// add one line for the column headings
#define UDP_PORT		(64159)

#define HELLO_PERIOD		(15)
#define PEER_TIMEOUT		((2 * HELLO_PERIOD) + 5) // enough for two lost hello messages
///////////////////////////////////////////////////////////////////////////////////////////
// The state enum is used to record initialisation status, so the threads know where they stand
enum _state
{
  _INITIALISING=0, // define the message queues
  _RUNNING,
  _SHUTTING_DOWN,
  _TERMINATED
} ;

enum _mpv_state
{
  MPV_STOPPED=0,
  MPV_WAITING_FOR_EXIT,
  MPV_PAUSED,
  MPV_PLAYING
} ;

// The verbose enum is used to record all the debugging modes
enum _verbose_types
{
  _V_DISPLAY	= 0x0001,
  _V_CLOCK	= 0x0002,
  _V_DIMMER	= 0x0004,
  _V_TIMER	= 0x0008,
  _V_SETUP	= 0x0010,
  _V_MEDIA	= 0x0020,
  _V_AMP	= 0x0040,
  _V_BUTTON	= 0x0080,
  _V_AMP_MEDIA	= 0x0100,
  _V_ALARM	= 0x0200,
  _V_MULTICAST	= 0x0400,
  _V_BLUETOOTH	= 0x0800
} ;
///////////////////////////////////////////////////////////////////////////////////////////
#define UNUSED(x)		(void)(x)	// This macro silences unused parameter warnings
///////////////////////////////////////////////////////////////////////////////////////////
// We now define a structure to hold an alarm definition. We will use this to create an array
struct _alarm_definition
{
  uint16_t		minute_of_day ;
  uint16_t		duration ;
  int16_t		init_vol_offset ;
  int16_t		targ_vol_offset ;
  time_t		day [7] ;		// 32 bits for each day of the week. The value held is an epoch time
  char			url_path [SETUP_FIELD_LEN+1] ; // a string that holds the URL that will play when the alarm goes
} ;

typedef struct _alarm_definition ALARMS ;
///////////////////////////////////////////////////////////////////////////////////////////
// Structure to record / remember activity from other multicast hosts
struct _peer_table
{
  struct in_addr	sin_addr ;		// holds IPv4 address from which packet came
  time_t		last_seen ;		// Holds the last time (in seconds) that packet was seen
  char			type ;			// Holds 'S', 'C', or 'Z' depending upone whether set as standalone, clustered or silent clustered
  char			name[SETUP_FIELD_LEN] ; // a string that holds the name of the system
} ;

typedef struct _peer_table PEERS ;

#define NUM_PEERS	(5)			// Maximum number of other clocks that are supported
///////////////////////////////////////////////////////////////////////////////////////////
int			Read_Ambient ;		// set if the command line -a flag specified
int			Daemonise ;		// set if the command line -d flag specified
double			Trial_PWM = -1.0 ;	// set if the command line -p flag specified
int			Reload ;		// set if the command line -r flag specified
int			Media_Time = -1 ;	// set if the command line -t flag specified
int16_t			Set_Volume = -1 ;	// set if the command line -v flag specified
int			Track_Num = -1 ;	// set if the command line -j flag specified
FILE			*Log_File ;		// set if the command line -L flag specified
uint16_t		Stop_Clock ;		// set if the command line -x flag specified
uint16_t		Brightness_Test ;	// set if the command line -y flag specified
int			Display_Test ;		// set if the command line -z flag specified
uint16_t		Display_IP ;		// set when button sequence for displaying IP address is active
char			Log_Filename[SETUP_FIELD_LEN+1] ;
uint16_t		Cmdline_Verbose ;	// set when any of the command line verbose flags are specified
uint16_t		Verbose ;		// An OR of the command line verbose flags, and what is read from the setup.conf file

uint8_t			Time_Format_1224 ;
uint8_t			Leading_Zero_Blanking ;
uint8_t			Second_Snooze_Ignore ;
uint16_t		Default_Snooze_Duration = DEFAULT_SNOOZE_DURATION ; // variable is in seconds
uint16_t		Default_Alarm_Hour ;
uint16_t		Default_Alarm_Minute ;
uint16_t		Default_Alarm_Duration = DEFAULT_DURATION ; // variable is in minutes
uint16_t		Default_Media_Duration = DEFAULT_DURATION; // variable is in minutes
uint16_t		Min_LDR_Threshold = MIN_LDR_THRESH ;
uint16_t		Max_LDR_Threshold = MAX_LDR_THRESH ;
double			Min_PWM_Ratio = MIN_PWM_RATIO ;
double			Max_PWM_Ratio = MAX_PWM_RATIO ;

ALARMS			Alarms[MAX_NUM_ALARMS] ;
int16_t			Volume ;
int16_t			Min_Volume ;
int16_t			Vol_Offset ;
int16_t			Targ_Vol_Offset ;
int			Num_Alarms ;
int			Next_Alarm_Index ;
int			Seconds_To_Next_Alarm = -1 ;

int			Seconds_Of_Snooze_Remaining ;	// downcount for snooze
int			Seconds_Of_Play_Remaining ;	// downcount for media or alarm playing
int			MPV_Kill_Timer ;		// downcount for an idle MPV session. Kills the handle / mpv when expires
int			Network_Failure_Downcount ;	// downcount for network failure such as wifi interface going down

uint8_t			Is_Alarm ;
uint8_t			MPV_State ;		// Used to track whether media player (mpv) is playing or not
uint8_t			Amp_State ;		// Used to flag whether the amplifier is turned on or off
uint8_t			Fallback_State ;	// Keeps track of whether we are playing an alarm, fallback or the radio

pthread_mutex_t		Alarms_Mutex = PTHREAD_MUTEX_INITIALIZER ; // Used to protect Alarms[] and related variables
uint8_t			Dot_Flash_Counter ;	// Used to synchronise updating of decimal points with seconds ticking over
int8_t			Audible_Clustered_Clock_Visible ;	// True if a clustered clock is visible (not timed out)

mpv_handle		*Mpv_Handle = NULL ;

time_t			Current_Seconds_Count ;		// Kept by timer thread to record the last epoch second. It will hold an increasing sequence
PEERS			Peers [NUM_PEERS] ;		// Keeps track of multicast packets

double			Moving_Average_Light_Level ;
uint16_t		Instantaneous_Light_Level ;
TV			Dimmer_Timeout ;

mqd_t			DD_Msgq ;
mqd_t			Dim_Msgq ;
mqd_t			Setup_Msgq ;
mqd_t			Media_Msgq ;
mqd_t			Amp_Msgq ;
mqd_t			Button_Msgq ;

mqd_t			Response_Msgq ;

int			Spi_Handle ;		// This stores an identifier for the opened SPI port

uint32_t		Button_Control_Token ;	// A magic random number chosen by the remote system when controlling the main
uint16_t		Remote_Buttons ;	// Used by a main clock to make it appear as if local buttons have been pressed
int16_t			Button_Downcount ;	// A counter that will get us out of remote control mode if the host disapparates before we know it

uint8_t			Threads_Active ;	// Keeps track of the multi-tasking state
pid_t			Pigpiod_PID ;		// the PID of the pigpio daemon
pid_t			Bluetoothctl_PID ;	// the PID of the bluetoothctl process

char			This_Clock_Type ;
char			Default_Stream_Or_File [SETUP_FIELD_LEN+1] ;
char			Fallback_Alarm_File [SETUP_FIELD_LEN+1] ; // Play this if the stream fails or alarm file cannot be found
char			Current_Alarm_Stream [SETUP_FIELD_LEN+1] ; // This is the stream that is currently playing

char			Index_Filename [SETUP_FIELD_LEN+1] ; // This holds filename of the index file for currently playing playlist
ST			Index_Filename_fileinfo ;

char			Model_String [MODSTR_LEN] ;
char			OS_String [MODSTR_LEN] ;
uint16_t		OS_Version ;

uint32_t		Alarm_File_Hash ;
time_t			Alarm_File_Time = -1 ;	// Negative number indicates that the file has not yet been read

#define TEMP_NAME_LEN	(32)
static char const	Temp_Name[TEMP_NAME_LEN] = "/tmp/alarm-clock_XXXXXX" ;	// A template used when calling mkstemp.

static char const	Setup_Thread_Str[] = "Setup Loader Thread" ;
static char const	Time_Format_Str[] = "%d/%m/%Y %H:%M:%S" ;
static char const	DayOfWeek[7][4] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"} ;
static char const	Alarm_Filename[] = "/etc/alarm-clock/alarms.csv" ;
static char const	Setup_Filename[] = "/etc/alarm-clock/setup.conf" ;
static char const	Model[] = "/sys/firmware/devicetree/base/model" ;
static char const	OS[] = "/etc/os-release" ;
static char const	Mem[] = "/proc/meminfo" ;

static char		Response_Msgq_name[TEMP_NAME_LEN] ;	// Special temporary queue established to receive responses - opened on demand

#ifdef	USE_GPIO_DAEMON
  int			GPIOD_daemon_id = -1 ;	// identifies the daemon registration
#endif

int			Rx_Socket = -1 ;
int			Tx_Socket = -1 ;
struct sockaddr_in	Rx_Address, Tx_Address ;
char			My_Name [NAME_LEN] ;
uint32_t		My_IPv4_sin_addr ;
char			My_IPv4_Address[NI_MAXHOST] ;

const char		*Multicast_Group = "225.45.21.183" ;		// pure spurious random numbers in the multicast range

#define MAX_LINES	(300)				// Number of lines in the circular read buffer
char			*Read_Lines [MAX_LINES] ;

#define MAX_LINE	(1000)				// Longest line that we will support when reading a pipe
char			This_Question [MAX_LINE] ;
char			This_Line [MAX_LINE] ;

volatile int16_t	RB_insert ;
volatile int16_t	RB_extract ;

char			*Bluetoothctl_Devices_Cmd ;
char			*Bluetoothctl_Devices_String ;
int16_t			Initiate_Bluetooth_Pairing ;
int16_t			Initiate_Bluetooth_UnPairing ;
int16_t			Disconnect_Bluetooth_Devices ;
int16_t			Check_Bluetooth_Devices ;
int16_t			Check_Bluetooth_Disconnect ;
int16_t			Bluetooth_Init_Completed ;
uint16_t		Num_Bluetooth_Clients ;
///////////////////////////////////////////////////////////////////////////////////////////
// The definitions below look like a mess. The mess is caused by the 'pipe' system call
// and the way that the parent and child processes use DIFFERENT identifiers for the same pipe

#define PARENT_READ_PIPE	(0)	// The parent will read from this pipe (and child will write)
#define PARENT_WRITE_PIPE	(1)	// The parent will write into this pipe (and child will read)
#define NUM_PIPES		(2)

#define READ_FD			(0)	// read file desriptor offset used by pipe call
#define WRITE_FD		(1)	// write file descriptor offset used by pipe call
#define NUM_FDS			(2)

int	Bluetooth_Pipes [NUM_PIPES][NUM_FDS] ;	// Will hold pipe identifiers for stdin and out comms to forked task

#define PARENT_READ_FD		(Bluetooth_Pipes[PARENT_READ_PIPE][READ_FD])
#define PARENT_WRITE_FD		(Bluetooth_Pipes[PARENT_WRITE_PIPE][WRITE_FD])
#define CHILD_READ_FD		(Bluetooth_Pipes[PARENT_WRITE_PIPE][READ_FD])
#define CHILD_WRITE_FD		(Bluetooth_Pipes[PARENT_READ_PIPE][WRITE_FD])
///////////////////////////////////////////////////////////////////////////////////////////
#define MAC_ADDRESS_LEN		(18) // This is a fixed length format - colon separated, null terminated
#define DEVICE_NAME_LEN		(32)
#define BLUETOOTH_DEV_RECS	(10) // Up to 10 connected devices

struct _bluetooth_device_record
{
  uint8_t	trusted ;
  char		mac_address [MAC_ADDRESS_LEN] ;
  char		device_name [DEVICE_NAME_LEN] ;
} ;

typedef struct _bluetooth_device_record BLUETOOTH_DEV_REC ;

BLUETOOTH_DEV_REC	Bluetooth_Devices [BLUETOOTH_DEV_RECS] ;
///////////////////////////////////////////////////////////////////////////////////////////
// Structure for specifying expected strings and responses
struct _expected_strings
{
  char	const 	*target_string ;
  uint8_t const	find_prompt ;
  char	const	*command ;
} ;

typedef struct _expected_strings EXPECT ;

EXPECT const Bluetooth_Initialisation[] =
{
  {"Pairable: yes",		1,		"power off"},
  {"Changing power off",	1,		"agent off"},
  {"Agent unregistered",	1,		"power on"},
  {"Changing power on",		1,		NULL},
  {NULL,			0,		NULL}
} ;

EXPECT const Bluetooth_Pairing[] =
{
  {NULL,			0,		"discoverable on"},
  {"Changing discoverable",	1,		"pairable on"},
  {"Changing pairable",		1,		"agent NoInputNoOutput"},
  {"registered",		1,		"default-agent"},
  {NULL,			0,		NULL}
} ;
///////////////////////////////////////////////////////////////////////////////////////////
// This section defines thread identities.... This is Linux. we are going to multi-task

extern void *			Display_Driver_Thread (void *) ;
extern void *			Clock_Thread (void *) ;
extern void *			LED_Dimmer_Thread (void *) ;
extern void *			Timer_Thread (void *) ;
extern void *			Setup_Thread (void *) ;
extern void *			Media_Player_Thread (void *) ;
extern void *			Media_Manager_Thread (void *) ;
extern void *			Amp_Manager_Thread (void *) ;
extern void *			Button_Manager_Thread (void *) ;
extern void *			Multicast_Tx_Thread (void *) ;
extern void *			Multicast_Rx_Thread (void *) ;
extern void *			Bluetooth_Rx_Thread (void *) ;
extern void *			Bluetooth_Manager_Thread (void *) ;
///////////////////////////////////////////////////////////////////////////////////////////
// This programme is multi-threaded. That is, there are several different tasks running
// pseudo-independently at the same time, each in a different thread.
//
// Each thread has access to the same data space as every other thread. Global variables are
// visible to all threads, so that changes made by one of the threads are immediately visible
// to the others.
//
// Depending on raspberry pi model, there are a different number of CPUs available for running
// these threads. The more capable models can run several threads at exactly the same time.
struct _subtask_struct
{
  void          *(*task) (void *arg) ;
  char const    * const name ;
  char		log_string [LOG_STRING_LEN] ;
  uint16_t	abort_on_exit ;
  uint16_t	n ;
  pthread_t     supervisor_id ;
  pthread_t     child_id ;
} ;

typedef struct _subtask_struct SUBTASK ;

// This array defines the threads that will be created
SUBTASK Subtasks[] =
{
  {&Display_Driver_Thread,	"LED display driver",	"", 1, 0, 0, 0},
  {&Clock_Thread,		"Clock",		"", 1, 0, 0, 0},
  {&LED_Dimmer_Thread,		"LED Dimmer",		"", 1, 0, 0, 0},
  {&Timer_Thread,		"Timer",		"", 1, 0, 0, 0},
  {&Setup_Thread,		"Setup Loader",		"", 1, 0, 0, 0},
  {&Media_Player_Thread,	"Media Player",		"", 1, 0, 0, 0},
  {&Media_Manager_Thread,	"Media Manager",	"", 1, 0, 0, 0},
  {&Amp_Manager_Thread,		"Amp Manager",		"", 1, 0, 0, 0},
  {&Button_Manager_Thread,	"Button Manager",	"", 1, 0, 0, 0},
  {&Multicast_Tx_Thread,	"Multicast Tx",		"", 1, 0, 0, 0},
  {&Multicast_Rx_Thread,	"Multicast Rx",		"", 1, 0, 0, 0},
  {&Bluetooth_Rx_Thread,	"Bluetooth Rx",		"", 0, 0, 0, 0},
  {&Bluetooth_Manager_Thread,	"Bluetooth Manager",	"", 0, 0, 0, 0},

  {NULL, NULL,	"", 0, 0, 0, 0} // This must be the last entry in this array
} ;

// The elements of this enum must be defined in the same order as the SUBTASK array above
enum _thread_order
{
  T_DISPLAY_DRIVER,
  T_CLOCK,
  T_LED_DIMMER,
  T_TIMER,
  T_SETUP,
  T_MEDIA_PLAYER,
  T_MEDIA_MANAGER,
  T_AMP,
  T_BUTTON,
  T_MULTICAST_TX,
  T_MULTICAST_RX,
  T_BLUETOOTH_RX,
  T_BLUETOOTH_MAN,

  T_NULL // This must be the last entry in this enum
} ;
///////////////////////////////////////////////////////////////////////////////////////////
// These strings define the names that will be used for Linux message passing queues.
// Threads will communicate with each other via these message queues.
// Message queue names must all commence with a slash
static const char DD_Msgq_name[] = "/DD" ;		// Display driver thread message queue
static const char Dim_Msgq_name[] = "/D" ;		// LED dimmer thread message queue
static const char Setup_Msgq_name[] = "/S" ;		// Setup thread message queue
static const char Media_Msgq_name[] = "/M" ;		// Media thread message queue
static const char Amp_Msgq_name[] = "/A" ;		// Amp Manager thread message queue
static const char Button_Msgq_name[] = "/B" ;		// Button Manager thread message queue
///////////////////////////////////////////////////////////////////////////////////////////
// Define message structures that will be used for our specific messages

struct _ddmsgbuf
{
  uint8_t	type ;		// indicates the kind of message this container is holding
  uint8_t	hours ;		// hours, minutes and seconds are in BINARY (not BCD) (Hours are in 24H time, not AM/PM)
  uint8_t	minutes ;
  uint8_t	seconds ;
  uint8_t	dots ;
} ;

typedef struct _ddmsgbuf DD_MSG_BUF ;

// The _display_messages enumeration holds the identity of the different display messages
enum _display_messages
{
  DISP_TIME,
  DISP_NUMBERS,
  DISP_RAW_NUMBERS,
  DISP_DOTS,
  DISP_LOCKED_COLON,
  DISP_LOCKED_NO_COLON,
  DISP_UNLOCKED_COLON,
} ;

enum _colon_format
{
  COL_OFF,
  COL_ON,
  COL_AM,
  COL_PM
} ;

struct _dimmsgbuf
{
  uint8_t	type ;		// indicates the kind of message this container is holding (timeout, manual setting, ...)
  uint16_t	seconds ;	// An integer indicating the time during which the given pwm_ration will override calculated dimmer setting
  double	pwm_ratio ;	// A number between 0.0 and 1.0
} ;

typedef struct _dimmsgbuf DIM_MSG_BUF ;

// The _dimmer_messages enumeration holds the identity of the different dimmer messages
enum _dimmer_messages
{
  DIM_UPDATE,
  DIM_SET,
  DIM_BRIGHTNESS_TEST,
  DIM_STOP_CLOCK,
  DIM_TIMEOUT_TO_BRIGHT,
  DIM_TIMEOUT_TO_NORMAL
} ;

enum _dimmer_state
{
  DIMMER_NORMAL,
  DIMMER_FORCED
} ;

// The _setup_messages enumeration holds the identity of the different setup messages
enum _setup_messages
{
  S_RELOAD,
  S_READ_LDR_AND_STATUS,
  S_DISPLAY_TEST,
  S_SET_VOL_DUR,
  S_PAIR,
  S_UNPAIR,
} ;

struct _setup_msgbuf
{
  uint8_t	type ;		// type of setup message
  uint16_t	ambient ;	// An ambient light level reading between 0 and 4095
  int16_t	volume ;	// An volume (value) between 0 and 31
  int16_t	duration ;	// A time duration in minutes
  uint16_t	num_blue ;	// The number of connected bluetooth clients
  pid_t		responseQ ;	// The PID of the calling process - to use as the response queue name
} ;

typedef struct _setup_msgbuf  SETUP_MSG_BUF ;

// The _media_messages enumeration holds the identity of the different media messages
enum _media_messages
{
  MEDIA_NOTHING=0,
  MEDIA_TICK,		// 1
  MEDIA_ADD,		// 2
  MEDIA_ALARM_START,	// 3
  MEDIA_PAUSE,		// 4
  MEDIA_STOP,		// 5
  MEDIA_NEXT,		// 6
  MEDIA_PREV,		// 7
  MEDIA_RESTART,	// 8
  MEDIA_SEEK_FWD,	// 9
  MEDIA_SEEK_BACK,	// 10
  MEDIA_PLAYLIST_CLEAR,	// 11
  MEDIA_PLAYLIST_REQUEST,// 12
  MEDIA_PLAYLIST_JUMP,	// 13
  MEDIA_SHUFFLE,	// 14
  MEDIA_PEERS_REQUEST,	// 15
} ;

struct _media_msgbuf
{
  int		seconds ;	// Indicates the number of seconds to play the media stream
  int		index ;		// If positive, indicates which track in the playlist to commence playing
  int16_t	volume ;	// If not negative and less than or equal to 31, this field represents volume level
  int16_t	init_vol_offset ;// If between -MAX_VOL_OFFSET and +MAX_VOL_OFFSET, this field represents initial volume offset
  int16_t	targ_vol_offset ;// If between -MAX_VOL_OFFSET and +MAX_VOL_OFFSET, this field represents target volume offset
  uint8_t	type ;		// type of media message
  pid_t		responseQ ;	// The PID of the calling process - to use as the response queue name
  char		url_path [SETUP_FIELD_LEN+1] ; // a string representing the path to the URL/file
} ;

typedef struct _media_msgbuf MEDIA_MSG_BUF ;

// The _amp_messages enumeration holds the identity of the different amp management thread messages
enum _amp_messages
{
  A_AMP_OFF,			// 0
  A_AMP_ON_BLUETOOTH,		// 1
  A_AMP_ON_MEDIA_OR_FILE,	// 2
  A_AMP_ON_RADIO,		// 3
  A_AMP_ON_RADIO_PAUSE,		// 4
  A_AMP_TICK,			// 5
  A_RESTART_LAST_MEDIA		// 6
} ;

struct _amp_msgbuf
{
  uint8_t	type ;		// type of amp message
  uint8_t	media_cmd ;	// accompanying media command (if any)
  int		index ;		// Index number of track in playlist to play
  char		url_path [SETUP_FIELD_LEN+1] ; // a string representing the path to the URL/file
} ;

typedef struct _amp_msgbuf AMP_MSG_BUF ;

#define MEDIA_MSG_LEN	(sizeof(MEDIA_MSG_BUF) - SETUP_FIELD_LEN - 1)
#define	AMP_MSG_LEN	(sizeof(AMP_MSG_BUF) - SETUP_FIELD_LEN - 1)

enum _fallback_states
{
  F_OFF,
  F_MEDIA_OR_FILE,
  F_RADIO
} ;

typedef  char	MEDIA_RESPONSE_BUF [LONGER_STR_LEN] ; // a string that holds an item from the current playlist
///////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////
// NOTE: For PiGPIOD error code explanations, refer to
// https://abyz.me.uk/rpi/pigpio/cif.html (search for Error Codes)
///////////////////////////////////////////////////////////////////////////////////////////
// The Release_Resources routine is called upon any exit from this programme. It cleans up any
// DMA handlers and memory, otherwise, the Raspberry Pi hardware will be left in a dodgy state
//
// Note: Release_Resources is a signal handler, and it is unsafe to use many kinds of call
// within a signal handler, including printf. write() however is safe. Its a bit more work
// than printf however.

static const char *Quit_Cmd[]		= {"quit", NULL};

noreturn void
Release_Resources (const int signum)
{
  struct itimerval	timer ;
  SUBTASK		*subtask ;
  char			*p ;

  // Unless we are catching a CTRL-C (SIGINT), then its an error
  if ((signum != SIGINT) && (signum != SIGKILL))
  {
    p = strsignal (signum) ;

    // Note - I have manually counted the length of the string to include in the write call
    write (STDERR_FILENO, "\nRelease Resources received unexpected signal: ", 47) ;

    if (p != NULL)
      write (STDERR_FILENO, p, strlen(p)) ;

    write (STDERR_FILENO, "\n", 1) ;
  }

  // Only run this code a single time - we may end up in Release_Resources multiple times
  if (Threads_Active != _TERMINATED)
  {
    // Note - I have manually counted the length of the string to include in the write call
    if (Log_File != (FILE *)0)
    {
      p = strsignal (signum) ;
      fprintf (Log_File, "\nReceived unexpected signal: %d (%s)\nDropping out\n", signum, p) ;
      fflush (Log_File) ;
      fclose (Log_File) ;
      Log_File = NULL ;
    }

    Threads_Active = _TERMINATED ;

    // Send kills to threads
    // No need to catch errors - we are aborting anyway
    // Generally there won't be errors. Rarely the shutdown timing allows some threads to interrupt
    // the shutdown. This generates errors that we are not interested in. So I have commented out.

    // Cancel any timer/alarms pending (setting the timeouts to zero will cause timer to cancel)
    timer.it_value.tv_sec = 0 ;
    timer.it_value.tv_usec = 0 ;
    timer.it_interval.tv_sec = 0 ;
    timer.it_interval.tv_usec = 0 ;
    setitimer (ITIMER_REAL, &timer, NULL);

    // Turn off the amplifier and radio, and Release the GPIO resources
    gpioWrite (AMP_ENABLE, 0) ;
    gpioWrite (RADIO_ENABLE, 0) ;

    // Kill any media that is currently playing
    if (Mpv_Handle != NULL)
    {
      // Issue a quit command to the MPV instance. It will than abort
      mpv_command (Mpv_Handle, Quit_Cmd) ;
      Mpv_Handle = NULL ;
    }

    spiClose (Spi_Handle) ;

    // indiscriminately kill both parent and child threads
    for (subtask=&Subtasks[0] ; subtask->task != NULL ; subtask++)
    {
      // Kill the parent, then child
      if (subtask->supervisor_id != 0)
	pthread_cancel (subtask->supervisor_id) ;

      if (subtask->child_id != 0)
	pthread_cancel (subtask->child_id) ;
    }

    gpioTerminate() ;  // unlink from the pigpio library

    // Release message queues
    mq_unlink (DD_Msgq_name) ;
    mq_unlink (Dim_Msgq_name) ;
    mq_unlink (Setup_Msgq_name) ;
    mq_unlink (Media_Msgq_name) ;
    mq_unlink (Amp_Msgq_name) ;
    mq_unlink (Button_Msgq_name) ;

    if (Response_Msgq_name[0] != '\0')
      mq_unlink (Response_Msgq_name) ;

    mq_close (DD_Msgq) ;
    mq_close (Dim_Msgq) ;
    mq_close (Setup_Msgq) ;
    mq_close (Media_Msgq) ;
    mq_close (Amp_Msgq) ;
    mq_close (Button_Msgq) ;

    if (Response_Msgq != 0)
      mq_close (Response_Msgq) ;

    write (STDERR_FILENO, "\nDropping out\n", 14) ;

    // Initiate a shutdown -r now - if this isn't a CTRL-C (SIGINT)
    if (signum != SIGINT)
      system ("sudo shutdown -r now") ; // may fail if not running as root

    // Kill the Pigpiod daemon by sending signal 9
    // This will trigger systemctl to restart both the pigpiod and alarm-clock programmes
    if (Stop_Clock == 0)
    {
      if (Pigpiod_PID > 0)
	kill (Pigpiod_PID, SIGKILL) ;

      if (Bluetoothctl_PID > 0)
	kill (Bluetoothctl_PID, SIGKILL) ;
    }
  }

  exit (signum != SIGINT) ;
} // Release_Resources
///////////////////////////////////////////////////////////////////////////////////////////
// Some of the stuff below may appear intimidating if you are not so familiar with Linux C
// coding. Don't worry too much if that's your reaction. Concentrate on the other source code.
///////////////////////////////////////////////////////////////////////////////////////////
// Goodbye is a general abort point.
// It will display an optional error message similarly to printf, close GPIO resources, and kill threads.

noreturn void
Goodbye (const char *fmt, ...)
{
  va_list               arglist ;

  // makes use of C's variable argument list macros - the dot dot dot above
  va_start (arglist, fmt) ;
  vfprintf (stderr, fmt, arglist) ; // a special version of fprintf using the variable argument list

  if (Log_File != NULL)
  {
    vfprintf (Log_File, fmt, arglist) ;
    fflush (Log_File) ;
  }

  va_end (arglist) ;

  if (Stop_Clock)
    Release_Resources (SIGKILL) ;

  Release_Resources (SIGINT) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Display_Time
//
// convert a timestamp into human readable date/time form and returns a string

char *
Display_Time (char *s, TV *t)
{
  time_t        now ;
  TM		*now_tm ;
  size_t        n ;

  now = t->tv_sec ;
  now_tm = localtime (&now) ;

  if (((n = strftime (s, TIME_STRING_LEN, Time_Format_Str, now_tm)) > 0) && (n < (TIME_STRING_LEN-5)))
    sprintf (&s[n], ".%03hu", (uint16_t)(t->tv_usec / 1000)) ;
  else
    s[0] = '\0' ;

  return s ;
} // Display_Time
///////////////////////////////////////////////////////////////////////////////////////////
// Print_Log
//
// Writes a log message for debugging purposes. I've defined a routine to do this so that
// I can ensure the format of the log messages is consistent

void
Print_Log (const int who, const char *fmt, ...)
{
  SUBTASK		*subtask ;
  va_list		arglist ;
  TV			Time ;
  uint16_t		n, close_default ;
  int			m ;
  char			time_string [TIME_STRING_LEN] ;
  char			*s, *o, *beginning, *out_str ;

  if (Threads_Active == _TERMINATED)
    return ;

  // This is a last ditch test for sanity
  if (who > T_NULL)
    Goodbye ("Print Log called with illegal task number %d\n", who) ;

  if ((out_str = malloc(LOG_STRING_LEN)) == NULL)
    Goodbye ("Print Log cannot malloc\n") ;

  close_default = 0 ;

  if ((who == T_NULL) && (Log_File == NULL))
  {
    // This is a last gasp - append the message to the log file
    // (it will open a fresh file if it doesn't already exist)
    Log_File = fopen ("/var/log/alarm-clock.log", "a") ;
    close_default = 1 ;
  }

  // prepare the time of day in case there's a new line to display
  gettimeofday (&Time, NULL) ;
  Display_Time (time_string, &Time) ;

  // makes use of C's variable argument list macros - the dot dot dot above
  subtask = &Subtasks[who] ;
  beginning = s = subtask->log_string ;

  if ((n = subtask->n) > (LOG_STRING_LEN-100))
    n = LOG_STRING_LEN - 100 ;

  va_start (arglist, fmt) ;
  vsnprintf (&s[n], LOG_STRING_LEN - n, fmt, arglist) ;
  va_end (arglist) ;
  s[LOG_STRING_LEN-1] = '\0' ;

  // s[] now contains what exists of the line(s) to be printed.
  // It may contain embedded linefeeds that are not in the last character position
  // Scan through the whole string finding linefeeds, then print one line at a time
  for (; *s != '\0' ;)
  {
    // Prepend the string with the time
    m = sprintf (out_str, "%s: ", time_string) ;
    o = &out_str[m] ;

    // copy the next line to end of out string but only up until the next newline (if one exists)
    for (; (*s != '\0') && (*s != '\n') ; m++)
      *o++ = *s++ ;

    // If we are pointing to a null - exit the loop - nothing to display this time
    if (*s == '\0')
      break ;

    // s must be pointing to '\n' - but this character has not yet been copied into out_str
    *o++ = '\n' ;
    *o = '\0' ; // Required for the fputs

    // The line to display is now in the out_str, terminated by a '\n'
    if (Log_File != NULL)
    {
      fputs (out_str, Log_File) ;
      fflush (Log_File) ;
    }

    write (STDOUT_FILENO, out_str, ++m) ; // Write out the number of characters in this line including newline
    beginning = ++s ; // point to next character of string - will be first character on next line
  }

  // s points to null termination in subtask->log_string
  // beginning points to first character yet-to-be-printed in subtask->log_string
  subtask->n = s - beginning ; // Number of residual characters that we still need to display (and remember)

  if (subtask->n > 0)
    memmove (subtask->log_string, beginning, subtask->n) ;

  free (out_str) ;

  if ((close_default != 0) && (Log_File != NULL))
  {
    fclose (Log_File) ;
    Log_File = NULL ;
  }
  // We exit with New_Log_Line being true if last character was a newline, and false if not
} // Print_Log
///////////////////////////////////////////////////////////////////////////////////////////
// Is_Playlist - first checks to see if this file is a playlist, and if so, whether a
// last played is recorded. If it exists, the value in that file is returned, otherwise,
// it returns -1

int
ends_with (const char *s, const char *suffix, char *index_file)
{
  size_t	string_len ;
  size_t	suffix_len ;

  string_len = strlen (s) ;
  suffix_len = strlen (suffix) ;

  // Initialise the index_file to an empty string
  if (index_file != NULL)
    index_file[0] = '\0' ;

  // Returns false if doesn't end with the suffix
  if ((suffix_len > string_len) || (strcasecmp(&s[string_len - suffix_len], suffix) != 0))
    return 0 ;

  // We have a match. Create a string with the count file suffix in the same location as the playlist extension
  if (index_file != NULL)
  {
    strncpy (index_file, s, SETUP_FIELD_LEN) ;
    strcpy (&index_file[string_len - suffix_len], ".index") ;
  }

  return 1 ;
} // ends_with

int
Is_Playlist (const char *path, char *index_file, ST *file_information)
{
  FILE		*f ;
  ST		file_info ;
  int		index ;
  char		s[SETUP_FIELD_LEN] ;

  index = -1 ;

  // Zero out the index filename to ensure we return blank if path doesn't point to a playlist
  if (index_file != NULL)
    *index_file = '\0' ;

  if (ends_with(path, ".m3u", s) ||
      ends_with(path, ".pl", s) ||
      ends_with(path, ".pls", s) ||
      ends_with(path, ".asx", s) ||
      ends_with(path, ".pla", s) ||
      ends_with(path, ".play", s) ||
      ends_with(path, ".playlist", s) ||
      ends_with(path, ".txt", s))
  {
    // The file appears to be a playlist.... does a record exist for the last track?
    if (index_file != NULL)
      strcpy (index_file, s) ;

    // Use the size field as a flag to indicate the fileinfo is valid. 0 means invalid
    if (file_information != NULL)
    {
      file_information->st_size = 0 ;

      // Try to grab the file ownership information. A return value of 0 means success
      if (stat (path, &file_info) == 0)
	memmove (file_information, &file_info, sizeof(file_info)) ;
    }

    // Can we open the index file?
    if ((f = fopen(s, "r")) != (FILE *)0)
    {
      // A count file exists. Does it contain a positive number?
      if ((fscanf (f, "%d", &index) != 1) || (index < 1))
	index = -1 ;

      fclose (f) ;
    }
  }

  return index ;
} // Is_Playlist
///////////////////////////////////////////////////////////////////////////////////////////
// Supervisor_Thread
//
// This is a reentrant (running multiple times in parallel) task that supervises child thread
// Its only purpose is for debugging to catch the child if it exits (crashes) and identify it.

void *
Supervisor_Thread (void *arg)
{
  SUBTASK              **subtask ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // this dancing is because pthreads are defined with an argument type of (void *)
  // (note that you can actually pass any type cast to a void pointer. But we have passed a
  // pointer to a SUBTASK structure element)
  subtask = (SUBTASK **)arg ;

  // Initialise the logging variables
  // If this is the beginning of a new line, log the time
  (*subtask)->log_string[0] = '\0' ;
  (*subtask)->n = 0 ;

  // Now launch the child
  if (pthread_create (&(*subtask)->child_id, NULL, (*subtask)->task, NULL) != 0)
    fprintf (stderr, "\nCan't create %s thread: %s\n", (*subtask)->name, strerror(errno)) ;

  // Now do our job of waiting patiently for the child to exit (which we hope doesn't happen)
  else if ((pthread_join ((*subtask)->child_id, NULL) != 0) && (Stop_Clock == 0))
    fprintf (stderr, "\n%s thread join problem: %s\n", (*subtask)->name, strerror(errno)) ;

  else if ((*subtask)->abort_on_exit)
  {
    fprintf (stderr, "\n%s thread has exited! Stop_Clock=%hu\n", (*subtask)->name, Stop_Clock) ;
    Threads_Active = _SHUTTING_DOWN ;
  }

  // Free the called parameter and exit
  free (arg) ;
  pthread_exit (NULL) ;
} // supervisor_thread
///////////////////////////////////////////////////////////////////////////////////////////
// Abort_If_Another_Instance_Already_Running()
//
// Linux is a multitasking operating system. We don't want to accidently start two versions of
// this programme which are each trying to access the GPIO pins. This routine checks to see if
// an instance is already running, and if so, it will abort.
//
// get_process_name()
//
// gets the process name associated with a PID.  Returns true if it fails to get the process name
// This is used to work out what name Linux knows PID by (usually the filename of the executable)
// We read the name from the pseudo filesystem under /proc

#define PROC_NAME_LEN	(255)

int
get_process_name (const int pid, char *name)
{
  FILE			*f ;
  char			*p, *d ;
  int			ret ;

  snprintf (name, PROC_NAME_LEN, "/proc/%d/status", pid) ;
  name[PROC_NAME_LEN] = '\0' ;
  ret = 1 ;

  // ignore fopen errors that occur (eg) if task has just terminated
  if ((f = fopen (name, "r")) == NULL)
    return ret ; // might fail because process has since terminated

  for (;;)
  {
    if (fgets (name, PROC_NAME_LEN, f) == NULL)
      break ;

    // The "Name:" line contains the name of the command that this process is running
    if (strncmp(name, "Name:", 5) == 0)
    {
      for (p=name+5 ; (*p != '\0') && (isspace((unsigned char) *p)) ;)
	p++ ;

      // p now points to the beginning of the process name.
      for (d=name ; (*p != '\0') && (*p != '\n') ; *d++=*p++) ;
      *d = '\0' ;

      ret = (name[0] == '\0') ;
      break ;
    }
  }

  fclose (f) ;
  return ret ;
} // get_process_name

int
Abort_If_Another_Instance_Already_Running (const int kill_daemon)
{
  DIR			*dirp ;
  struct dirent 	*dp ;
  pid_t			pid, this_pid, ret_pid ;
  char			pname[PROC_NAME_LEN] ;
  char			this_name[PROC_NAME_LEN] ;

  // Step 1. What is this process's PID and name???
  this_pid = getpid() ; // get this process's PID
  ret_pid = 0 ;

  if (get_process_name (this_pid, this_name))
  {
    Print_Log (T_NULL, "Cannot determine this process name PID=%d\n", this_pid) ;
    _exit (EXIT_FAILURE) ;
  }

  // Believe it or not, the only way to reliably get a list of running processes on a linux
  // system is to read the directory tree under '/proc'
  if ((dirp = opendir("/proc")) == NULL)
  {
    Print_Log (T_NULL, "Can't open /proc for reading (%s)\n", strerror(errno)) ;
    _exit (EXIT_FAILURE) ;
  }

  // Read the contents of the /proc directory one by one and see if there are any other
  // processes running that are named the same as this one. If so, we don't want to proceed
  for (;;)
  {
    errno = 0 ; // To distinguish other errors from end-of-directory
    pname[0] = '\0' ;

    if ((dp = readdir(dirp)) == NULL)
    {
      if (errno != 0)
      {
	Print_Log (T_NULL, "Can't read directory contents (%s)\n", strerror(errno)) ;
	_exit (EXIT_FAILURE) ;
      }

      break ; // we're done (end of directory)
    }

    // Since we are looking for /proc/PID directories, skip entries
    // that are not directories, or don't begin with a digit.
    // Then if this process has the same PID as the current process ID, ignore also
    if ((dp->d_type != DT_DIR) ||
	!(isdigit ((unsigned char) dp->d_name[0])) ||
	(sscanf (dp->d_name, "%d", &pid) != 1) ||
	(pid == this_pid))
      continue ;

    if (get_process_name (pid, pname))
      continue ; // didn't find the process name - just ignore. Maybe process terminated

    // If we found a matching process name, abort
    if (strcmp (this_name, pname) != 0)
      continue ; // didn't match

    // We have found a PID of a process that has the same name as this one
    if (kill_daemon)
    {
      // We have identified another process that has the same name as this one
      // Return the PID (which will be non-zero)
      ret_pid = pid ;
    }

    else
    {
      // We want to kill this process, not the other one
      printf ("Duplicate PID found for '%s'. This PID=%d, duplicate=%d\n", this_name, this_pid, pid) ;
      Goodbye ("You can kill the %s programme by running 'sudo pkill -9 %s', or\n"
	       "prevent systemd from relaunching by running 'sudo systemctl stop %s'\n",
	       this_name, this_name, this_name) ;
    }
  }

  closedir (dirp) ;
  return (int)ret_pid ;
} // Abort_If_Another_Instance_Already_Running
///////////////////////////////////////////////////////////////////////////////////////////
// Fork_Off_A_Daemon
//
// Forks the current process and closes stdin/out/err (so the new daemon does not have
// input/output via the regular channels) Returns 0 if this is the parent.
// Returns non-zero if this is the DAEMON child
//
// Forking is the process of creating an independent copy of the currently running process
// which then runs as a 'child'. If the parent dies, the child becomes a so-called zombie,
// and so there's a special trick that's employed: the first child forks a second time which
// creates a grandchild. The grandchild is detached from the grandparent, and is truly
// independent of the grandparent, but the grandchild has still inherited the same stdin,
// stdout and stderr of the grandparent.
//
// A daemon is just a name of a task that runs in the background independent of a terminal
// session. Daemons don't have parents. To become independent of a terminal session, the
// daemon must close stdin, stdout and stderr.

int
set_cloexec (int fd)
{
  int     val;

  // Sets the close on exec flag for the file descriptor
  if ((val = fcntl(fd, F_GETFD, 0)) < 0)
    return -1 ;

  val |= FD_CLOEXEC ; // enable close-on-exec

  return fcntl(fd, F_SETFD, val) ;
}

int
Fork_Off_A_Daemon (char *argv0)
{
  pid_t		pid ;
  int		i, fd0, fd1, fd2 ;
  char		command[100] ;

  // Fork off the parent process
  if ((pid = fork()) < 0)
  {
    Print_Log (T_NULL, "Fork Failed: %s\n", strerror(errno)) ;
    goto do_exit1 ;
  }

  // Success: The parent will now return with zero value
  if (pid > 0)
    return 0 ;

  // We are the child and still need to become the daemon
  // Let the child process become session leader
  if ((i = setsid()) < 0)
  {
    Print_Log (T_NULL, "setsid failed: %s\n", strerror(errno)) ;
    goto do_exit1 ;
  }

  // Change file mask
  umask (0) ;

  // Change the working directory to the root directory
  if ((i = chdir("/")) < 0)
  {
    Print_Log (T_NULL, "chdir failed: %s\n", strerror(errno)) ;
    goto do_exit1 ;
  }

  // Catch, ignore and handle signals
  signal (SIGCHLD, SIG_IGN);
  signal (SIGHUP, SIG_IGN);

  // Fork off for the second time
  if ((pid = fork()) < 0)
    exit (EXIT_FAILURE) ;

  // Second fork succeeded. Let the parent terminate
  if (pid > 0)
    exit (EXIT_SUCCESS) ;

  // Close all open file descriptors
  for (i=sysconf(_SC_OPEN_MAX) ; i>=0 ; i--)
    close (i) ;

  fd0 = fd1 = fd2 = -1 ;

  // Attach file descriptors 0, 1, and 2 to /dev/null.
  if ( ((fd0 = open("/dev/null", O_RDWR)) < 0) ||
       ((fd1 = dup(0)) < 0) ||
       ((fd2 = dup(0)) < 0) )
  {
    Print_Log (T_NULL, "file descriptor open failed: %s, fd0=%d, fd1=%d, fd2=%d\n", strerror(errno), fd0, fd1, fd2) ;

  do_exit1:
    sprintf (command, "pkill -9 %s", basename(argv0)) ;
    system (command) ;
    exit (EXIT_FAILURE) ;
  }

  set_cloexec (fd0) ;
  set_cloexec (fd1) ;
  set_cloexec (fd2) ;

  // syslog (LOG_MAKEPRI(LOG_DAEMON, LOG_NOTICE), "Daemon initialised") ;
  return 1 ; // Returns true to indicate this is the new daemon
} // Fork_Off_A_Daemon
///////////////////////////////////////////////////////////////////////////////////////////
// Sleep_ns is a wrapper to make calls to nanosleep easier
// It is set up as a delay call with ns granularity
//
// The maximum sleep time is ~4 seconds, as the number of nanoseconds is a 32 bit unsigned int

void
Sleep_ns (const uint32_t ns)
{
  TS	ts ;

  if (ns == 0)
    return ;

  ts.tv_sec = ns / NS_PER_SECOND ;
  ts.tv_nsec = ((int)ns - (ts.tv_sec * NS_PER_SECOND)) ;	// nanoseconds (fractional seconds)

  // nanosleep should return 0 if it slept correctly for the full duration
  // However it can be interrupted by a signal, and if so, we need to process
  // the signal and reinitiate the remainder of the delay.
  for (; nanosleep(&ts, &ts) != 0 ;)
  {
    // if interrupted by a signal, errno will be EINTR, and we just try again
    if (errno != EINTR)
      Goodbye ("Problem with Sleep_ns (0x%08x ==> %d.%d): %s\n", ns, ts.tv_sec, ts.tv_nsec, strerror(errno)) ;
  }
} // Sleep_ns
///////////////////////////////////////////////////////////////////////////////////////////
// Time_Difference_ms
//
// calculates the (signed) difference between two different times. Although called 'earlier'
// and 'later', these names are only to help remember what the sign of the returned
// time difference will be.
//
// If the time difference is greater than 32767ms, then the difference will be truncated to that

int16_t
Time_Difference_ms (const TV *earlier, const TV *later)
{
  int64_t	diff ;

  // Calculate the difference in microseconds
  diff = (((int64_t)later->tv_sec * (int64_t)1000000) + later->tv_usec) - (((int64_t)earlier->tv_sec * (int64_t)1000000) + (int64_t)earlier->tv_usec) ;

  // check that the result will fit into 16 bits and clip if not
  if (diff >= (MAX_SIGNED_MS * 1000))
    return (int16_t)MAX_SIGNED_MS ;

  if (diff <= (-MIN_SIGNED_MS * 1000))
    return (int16_t)MIN_SIGNED_MS ;

  // remember that diff is in microseconds...
  // return the time difference (signed) in ms
  return (int16_t)(diff / (int64_t)1000) ;
} // Time_Difference_ms
///////////////////////////////////////////////////////////////////////////////////////////
// Add_Time_ms
//
// Adds a constant to a time-of-day/date. The constant is UNSIGNED

void
Add_Time_ms (TV *t, uint32_t ms)
{
  t->tv_sec += (ms / 1000) ;
  t->tv_usec += (ms % 1000) * 1000 ; // multiply by 1000 to convert ms to us

  // account for overflow
  if (t->tv_usec >= 1000000)
  {
    t->tv_usec -= 1000000 ;
    t->tv_sec++ ;
  }
} // Add_Time_ms
///////////////////////////////////////////////////////////////////////////////////////////
// Number rounding routines

// Round_Nearest_Places
// Places=0 rounds to nearest integer
// Places=1 rounds to nearest single decimal point, and so on
double
Round_Nearest_Places (double number, int places)
{
  double		round_from ;
  long long int		rounded ;

  if (places < 0)
    places = 0 ;

  round_from = number * pow (10, places) ;
  rounded = (long long int)((round_from < 0) ? round_from - 0.5 : round_from + 0.5) ;

  // Precaution to stop spurious decimal places due to rounding error
  if (places == 0)
    return rounded ;

  return rounded / pow (10, places) ;
}

int
Round_Nearest_Int (const double number)
{
  int			x ;

  if (number > 0)
  {
    // Positive number
    x = number ;

    if ((number - x) >= 0.5)
      return x + 1 ;

    return x ;
  }

  // Negative number - we have trouble with rounding errors
  // Use the Round_Nearest_Places function to avoid this
  x = -number ; // x is positive equivalent integer of the negative number

  // The rounding issue happens with (-number-x)
  if (Round_Nearest_Places(-number - x, 1) >= 0.5)
    return -x - 1 ;

  return -x ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// PWM_Duty() is a function that determines a duty cycle constant for the pigpio calls
// Different ranges of duty cycle are required for the HW PWM and SW PWM functions.

uint32_t
PWM_Duty (double duty)
{
  // Enforce duty into the range 0 (fully off) to 1 (fully on)
  if (duty < 0.0)
    duty = 0.0 ;
  else if (duty > 1.0)
    duty = 1.0 ;

  return Round_Nearest_Int (duty * PWM_RANGE) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Set_PWM() is a generic function call that sets PWM to a specified duty
// The same call works for hardare or software PWM
void
Set_PWM (double duty)
{
  int	error ;

  if ((error = gpioPWM (DIM_PWM, PWM_Duty(duty))) != 0)
    Goodbye ("Error setting SW PWM Duty to %.2f (%d): %d\n", duty, PWM_Duty(duty), error) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Check_For_pigpiod
//
// Checks to see whether an instance of gpiod daemon is running or not
// Returns -1 if an error
// Returns pigpiod PID if it is running
// Returns 0 if pigpiod not running

int
Check_For_Specific_Process_Name (const char *check_name)
{
  DIR			*dirp ;
  struct dirent 	*dp ;
  pid_t			pid, this_pid, target_pid ;
  char			proc_name[PROC_NAME_LEN+1] ;

  // Believe it or not, the only way to reliably get a list of running processes on a linux
  // system is to read the directory tree under '/proc'
  if ((dirp = opendir("/proc")) == NULL)
  {
    Print_Log (T_NULL, "Can't open /proc for reading: %s\n", strerror(errno)) ;
    return -1 ;
  }

  this_pid = getpid() ; // get this process's PID
  target_pid = 0 ; // 0 means not found

  // Read the contents of the /proc directory one by one and see if an instance of pigpiod is running
  for (;;)
  {
    errno = 0 ; // To distinguish other errors from end-of-directory
    proc_name[0] = '\0' ;

    if ((dp = readdir(dirp)) == NULL)
    {
      if (errno != 0)
      {
	Print_Log (T_NULL, "Can't read directory contents: %s\n", strerror(errno)) ;
	return -1 ;
      }

      break ; // we're done (end of directory)
    }

    // Since we are looking for /proc/PID directories, skip entries
    // that are not directories, or don't begin with a digit. (ie not a process number)
    if ((dp->d_type != DT_DIR) ||
	!(isdigit ((unsigned char) dp->d_name[0])) ||
	(sscanf (dp->d_name, "%d", &pid) != 1) ||
	(pid == this_pid))
      continue ;

    // Convert the process number into a name
    if (get_process_name (pid, proc_name))
      continue ; // didn't find the process name - just ignore. Maybe process terminated

    // If we have found pigpiod, we can exit the loop
    if (strcmp (check_name, proc_name) == 0)
    {
      target_pid = pid ; // This will become the return value from this function
      break ;
    }
  }

  closedir (dirp) ;
  return target_pid ;
} // Check_For_Specific_Process_Name

#ifdef	USE_GPIO_DAEMON
int
Start_PiGpiod (void)
{
  pid_t		pid, pigpiod_pid ;

  // -t0 or 1 selects the clock peripheral that pigpiod will use for timing. Choosing the wrong one buggers Audio.
  // -l means disable socket interface for remote connections
  // -s10 means 10us timing resolution
  // -x 0xFFFFFFF means allow access to all GPIO pins (by default, GPIO0 and GPIO1 are not accessible)
  char		*e_argv[] = {"pigpiod", "-t0", "-l", "-s10", "-x0xFFFFFFF", NULL} ;

  pigpiod_pid = 0 ;

  // If it is not already running, launch the pigpiod daemon.
  //
  // Launching the daemon requires root privileges - so ensure that this programme is either run using sudo, or it
  // has its SUID mode bit set and is owned by root!
  if ((pigpiod_pid = Check_For_Specific_Process_Name("pigpiod")) < 0)
  {
    Print_Log (T_NULL, "Cannot determine if pigpiod is running or not\n") ;
    return EXIT_FAILURE ;
  }

  else if (pigpiod_pid == 0) // If pigpiod is not running
  {
    Print_Log (T_NULL, "pigpiod is not running. Attempting to start it now\n") ;

    // I am the child. This is where we are going to try and launch pigpiod.
    if (BUS_WIDTH == 32)
      e_argv[1] = "-t1" ; // Who decided to set up pigpiod defaults differently for 32 bit and 64 bit architectures? It was a dumb idea.

    else if (BUS_WIDTH != 64)
      Goodbye ("Cannot determine the correct bus width. Compiler says it is %d bits???\n", BUS_WIDTH) ;

    // The forking below is to create a new task which inherits suid privileges from this one
    if ((pid = fork()) < 0)
    {
      Print_Log (T_NULL, "Cannot fork: %s\n", strerror(errno)) ;
      return EXIT_FAILURE ;
    }

    // Success: The child returns with zero value, the parent returns with non-zero
    if (pid != 0)
    {
      // I am the parent. Wait for a short time for the daemon to start, or the connection to the daemon will fail
      Sleep_ns (0.5 * NS_PER_SECOND) ;
    }
    else // pid == 0
    {
      // I am the child. We can now launch the gpiod by calling execvp (whcih replaces the current process with the new one)
      if (execvp(e_argv[0], e_argv) < 0)
      {
	Print_Log (T_NULL, "Cannot execvp: %s\n", strerror(errno)) ;
	return EXIT_FAILURE ;
      }

      // The child has done its work and can just exit. The parent will keep running
    }

    // We still need to find the PID of the new pigpiod daemon we just started...
    if ((pigpiod_pid = Check_For_Specific_Process_Name("pigpiod")) <= 0)
    {
      Print_Log (T_NULL, "Cannot determine pigpiod PID after launching it!\n") ;
      return EXIT_FAILURE ;
    }
  }

  Pigpiod_PID = pigpiod_pid ;

  if (Verbose & _V_DIMMER)
    Print_Log (T_NULL, "pigpiod PID: %d\n", Pigpiod_PID) ;

  // If the daemon has initialised, or was already running, return OK
  return EXIT_SUCCESS ;
} // Start_PiGpiod
#endif
///////////////////////////////////////////////////////////////////////////////////////////
int
Start_Bluetooth (void)
{
  pid_t		pid ;
  ST		file_info ;
  FILE		*f ;
  char		*e_argv[] = {"bluetoothctl", NULL} ;

  // Does bluetoothctl exist??? If not, don't start this thread
  if (stat ("/usr/bin/bluetoothctl", &file_info) == 0)
  {
    // Before starting bluetoothctl, make sure bluetoothd is running.
    if ((pid = Check_For_Specific_Process_Name("bluetoothd")) < 0)
    {
      if (Model_String[0] == '\0')
      {
	// Bluetoothd is not running. determine which raspberry Pi model this is
	// if we are anything but a Pi2, then barf
	if ((f = fopen (Model, "r")) == NULL)
	  Goodbye ("Cannot open '%s': %s\n", Model, strerror(errno)) ;

	else if (fgets (Model_String, sizeof(Model_String), f) == NULL)
	  Goodbye ("Cannot read from '%s': %s\n", Model, strerror(errno)) ;

	fclose (f) ;
      }

      // If its a Pi2, it will say something like this "Raspberry Pi 2 Model B rev 1.1", or "Raspberry Pi Model B Plus Rev 1.2"
      if ((Model_String[13] != '2') && (Model_String[19] != 'A') && (Model_String[19] != 'B'))
	Goodbye ("bluetoothd is not running on '%s'\n", Model_String) ;

      // Silently return if this is one of the models of Pi that doesn't support bluetooth
      // because it doesn't matter whether bluetoothd is running or not!
      return EXIT_SUCCESS ;
    }

    if (Verbose & _V_BLUETOOTH)
      Print_Log (T_NULL, "Attempting to start the %s process\n", e_argv[0]) ;

    // The pipe function creates a pipe and puts the file descriptors for the reading and writing ends of the
    // pipe (respectively) into Bluetooth_Pipes[][0] and Bluetooth_Pipes[][1].
    if ((pipe(Bluetooth_Pipes[PARENT_READ_PIPE]) < 0) || (pipe(Bluetooth_Pipes[PARENT_WRITE_PIPE]) < 0))
    {
      Print_Log (T_NULL, "Cannot create pipes: %s\n", strerror(errno)) ;
      return EXIT_FAILURE ;
    }
    //------------------------------------------------------------
    // fork to create a child process
    if ((pid = fork()) < 0)
    {
      Print_Log (T_NULL, "Cannot fork: %s\n", strerror(errno)) ;
      return EXIT_FAILURE ;
    }

    // Success: The child returns with zero value, the parent returns with non-zero
    if (pid == 0)
    {
      // I am the child
      // duplicate the pipes created by the parent and use these for stdin and stdout
      dup2 (CHILD_READ_FD, STDIN_FILENO) ;
      dup2 (CHILD_WRITE_FD, STDOUT_FILENO) ;
      dup2 (CHILD_WRITE_FD, STDERR_FILENO) ;

      // close the file descriptors that are not required by the child
      close (CHILD_READ_FD) ;
      close (CHILD_WRITE_FD) ;
      close (PARENT_READ_FD) ;
      close (PARENT_WRITE_FD) ;

      // We are now ready to switch and become the bluetoothctl process
      if (execvp(e_argv[0], e_argv) < 0)
      {
	Print_Log (T_NULL, "Cannot execvp: %s\n", strerror(errno)) ;
	return EXIT_FAILURE ;
      }

      // Not Reached
    }
    //------------------------------------------------------------
    // I am the parent - close the file descriptors not required by the parent (ie the child ends of the pipes)
    close (CHILD_READ_FD) ;
    close (CHILD_WRITE_FD) ;

    Bluetoothctl_PID = pid ;

    if (Verbose & _V_BLUETOOTH)
      Print_Log (T_NULL, "Bluetoothctl PID: %d\n", Bluetoothctl_PID) ;
  }

  return EXIT_SUCCESS ;
} // Start_Bluetooth
///////////////////////////////////////////////////////////////////////////////////////////
// Set_Default_Volume
//
// This is a kludge to set the default system volume to something above 40%, which is the
// apparent default under the Bookworm OS. I tried many ways of trying to force pipewire
// to do something other than this, but failed. I even tried a startup systemd job, but that
// always fails too. But if I wait a while, then try and set the volume, then it works.
//
// One day, I may figure out what's really happening - but the audio subsystem with pipewire
// is really obscure and difficult!
//
// Note: The way this routine is coded below, it should only be called BEFORE alarm-clock
// does its thread-creating. That's because the fork will create a complete copy of the
// currently running image in RAM, including duplicates of all the threads! This won't matter
// for the duplicate copy which does an execvp (because execvp replaces the running image
// with the new target), but it WILL matter for the copies which just exit (i.e. first child
// in the 'double fork dance') - because the duplicate threads will then exit etc.
// To do this properly if you need to change volume with threads running, you must fork
// off a daemon BEFORE the threads are created, and then use signals to communicate between
// the forked daemon and the main alarm-clock programme.

int
Issue_Default_Vol_Command (void)
{
  pid_t		pid ;
  char		*e_argv[] = {"sudo", "systemctl", "restart", "set-clock-audio.service", NULL} ;

  // The forking below is to create a new task which inherits suid privileges from this one
  if ((pid = fork()) < 0)
  {
    fprintf (stderr, "Cannot fork: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Success: The child returns with zero value, the parent returns with non-zero
  if (pid == 0)
  {
    // I am the child. We can now launch the command by calling execvp (whcih replaces the current process with the new one)
    if (execvp(e_argv[0], e_argv) < 0)
    {
      fprintf (stderr, "Cannot execvp: %s\n", strerror(errno)) ;
      _exit (EXIT_FAILURE) ;
    }

    // This should not be reached because the execvp call doesn't return
  }

  // I am the parent. Set signals from the child to be ignored so we don't end up creating zombies!
  return EXIT_SUCCESS ;
}

int
Set_Default_Volume (void)
{
  pid_t		pid ;

  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
    Print_Log (T_NULL, "Forking to set the default volume\n") ;
  //------------------------------------------------------------
  // fork to create a child process
  if ((pid = fork()) < 0)
  {
    Print_Log (T_NULL, "Cannot fork: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Success: The child returns with zero value, the parent returns with non-zero
  if (pid == 0)
  {
    // I am the child - double fork, to prevent a zombie process
    if ((pid = fork()) < 0)
    {
      fprintf (stderr, "Cannot double fork: %s\n", strerror(errno)) ;
      return EXIT_FAILURE ;
    }

    // Success: The grandchild returns with zero value, the parent of the grandchild returns with non-zero
    if (pid == 0)
    {
      // I am the grandchild. I can now proceed to reset volume to default.
      // But wait ten seconds before changing volume because there's something else that is forcing it
      // to its dumb value.
      sleep (1) ;

      // We are now ready to reset the volume to the default
      Issue_Default_Vol_Command () ;

      // All the gumpf below is because there's some kind of audio issue with the Bookworm OS release.
      // It is probably just a transitory thing because Bookworm has just been released and it
      // is still carrying some bugs. Some have been fixed with updates, and I expect even more to
      // be fixed in the coming weeks. Hopefully by the time you read this, its stable and bug-free.
      //
      // This change may also help you if you want to use an audio interface that's non-standard.
      // Log into an ssh session, and work out the CLI commands to set your audio device up. Then
      // use 'mpv' from the command line to play an audio file to test it. The commands will probably
      // be of the form 'pactl ....' and you can google to work out how to use these commands.
      // Hint: try 'pactl list short sinks' to see what is available
      // also try 'pactl set-default-sink <device-name>'
      //
      // When you get your audio device configured as you want, edit the file
      // /usr/local/bin/set-clock-audio.sh to reflect the config commands you worked out for your
      // setup. i.e. add the set-default-sink command by "sudo nano /usr/local/bin/set-clock-audio.sh"
      //
      // NOTE: If you use an HDMI analogue audio adapter on a PiZero2W (or probably also a Pi5), then
      // the pi will auto configure that output so that no further action will be required. I've
      // tested this on the PiZero2W / Bookworm and it works OK.
      sleep (10) ;
      Issue_Default_Vol_Command () ;

      // Triple paranoid
      sleep (10) ;
      Issue_Default_Vol_Command () ;

      // Now the child is just going to die because we are now done.
      _exit (EXIT_SUCCESS) ;
    }

    // else I am the child. We want to exit immediately so the main thread's wait() call below will
    // be released (but exit after telling Linux we are not interested in child signals.)
    // This double-fork dance is so that Linux releases the resources of the child and grandchild
    // properly when they exit. Otherwise, the child will be linked to the parent and its
    // resources will not be released because the parent didn't call wait. And just to confuse
    // you, the double fork (as opposed to just one fork) is required because we want the parent
    // to be able to continue immediately without waiting much, and we want its child to exit
    // quickly (so the parent doesn't have to wait much) and we want the grandchild to continue
    // for a while. What a mess..... I guess this dance seemed like the thing to do in the early 1970s
    // when Kernigan and Ritchie were writing the first version of Unix?
    //
    // If I confused you, google 'double fork unix' or variations of that and see if somebody
    // else can explain this better. It is also closely related to detatching daemon processes.
    //
    // Set signals from the child to be ignored and continue. This is an alternate to waiting
    // without setting a signal mask (which would hold up the child until the grandchild exited,
    // and therefore hold up the parent which is waiting on this child.
    signal (SIGCHLD, SIG_IGN);
    _exit (EXIT_SUCCESS) ;
  }

  // I am the original parent.
  // Wait for the child to complete - the wait should be very quick. The wait is required
  // so Linux doesn't keep the child resources hanging around forever.
  // Note: calling wait as opposed to setting a signal mask here is desirable because setting
  // a signal mask would have an ongoing global effect whereas wait is ephemeral.
  wait (NULL) ;

  return EXIT_SUCCESS ;
} // Set_Default_Volume
///////////////////////////////////////////////////////////////////////////////////////////
// Open_Msg_Queue_Rx and Open_Msg_Queue_Tx
// Opens a message queue for reading or writing

mqd_t
Open_Msg_Queue_Rx (const int who, const char *name, long max_msg, long msg_size)
{
  struct mq_attr	attr, old_attr ;
  mqd_t			Q ;
  char			buffer [msg_size] ;
  unsigned int		priority ;
  mode_t		prev_mode ;

  // First we need to set up the attribute structure
  attr.mq_maxmsg = max_msg ;
  attr.mq_msgsize = msg_size ;
  attr.mq_flags = 0 ;

  // The mq_open call applies the task umask, so to assure we open with the intended permissions, we
  // must set umask. The value of umask is strange - its the INVERSE of the mask we want to create.
  prev_mode = umask (0) ; // mask of 0 means let us set any bit

  if ((Q = mq_open (name, O_RDONLY | O_CREAT, 0666, &attr)) < 0) // mask is in octal, hence leading zero
  {
    Print_Log (who, "Cannot create message queue %s (max_msg=%ld, msg_size=%ld): %s\n", name, max_msg, msg_size, strerror(errno)) ;
    _exit (EXIT_FAILURE);
  }

  umask (prev_mode) ; // return the mask to its original setting

  // Are there any messages currently in this queue (hanging over from something else??)
  mq_getattr (Q, &attr) ;

  if (attr.mq_curmsgs != 0)
  {
    // There are some messages on this queue. Remove them so the queue starts empty

    // First set the queue to not block any calls
    attr.mq_flags = O_NONBLOCK ;
    mq_setattr (Q, &attr, &old_attr) ;

    // Now eat all of the messages
    while (mq_receive (Q, buffer, msg_size, &priority) >= 0) ;

    // The call failed.  Make sure errno is EAGAIN which is the only acceptable failure
    if (errno != EAGAIN)
    {
      Print_Log (who, "Cannot empty message queue %s: %s\n", name, strerror(errno)) ;
      _exit (EXIT_FAILURE);
    }

    // Now restore the attributes
    mq_setattr (Q, &old_attr, 0);
  }

  return Q ;
} // Open_Msg_Queue_Rx
///////////////////////////////////////////////////////////////////////////////////////////
mqd_t
Open_Msg_Queue_Tx (const int who, const char *name, const int no_abort)
{
  mqd_t			Q ;

  // The call for opening to write is different than the call for opening to read!
  // Open the queue in blocking mode. i.e. when writing to the queue, if the queue is full,
  // the call will block until space becomes available.
  if ((Q = mq_open (name, O_WRONLY)) < 0)
  {
    Print_Log (who, "%d Cannot open queue %s for writing: %s\n", who, name, strerror(errno)) ;

    // Wait a moment, and then try again - this is to catch queued requests at the moment of system reboot
    Sleep_ns (NS_PER_SECOND * 1.5) ;

    // Return a negative number if we've been asked not to abort
    if (no_abort)
      return Q ;

    _exit (EXIT_FAILURE);
  }

  return Q ;
} // Open_Msg_Queue_Tx
///////////////////////////////////////////////////////////////////////////////////////////
// Wait_For_Next_Message
//
// Waits for the OS to send us a message, then does some basic checks, and returns
// The return value is either the number of bytes in the message, or 0 (if something went wrong)

int
Wait_For_Next_Message (const int who, mqd_t Q, void *msg, const int min_size, const int max_size, const char *name)
{
  int			n ;
  unsigned int		priority ;

  // No timeout - just wait as long as it takes for the next message to arrive
  if ((n = mq_receive (Q, (char *)msg, max_size, &priority)) < 0)
  {
    // Was the system call interrupted? Silently return failure if so
    if (errno == EAGAIN)
      return n ;

    Print_Log (who, "%s: msg receive error: %s\n", name, strerror(errno)) ;
  }

  else if (n < min_size)
    Print_Log (who, "%s: Short message received: %d bytes\n", name, n) ;

  // It shouldn't be possible for n to exceed max_size - the call to mq_send will fail instead
  return n ;
}
// Wait_For_Next_Message
///////////////////////////////////////////////////////////////////////////////////////////
// Write_Byte
//
// Valid addresses: 0..3
//  0 = Hours LEDs
//  1 = Minutes LEDs
//  2 = Seconds LEDs
//  3 = Additional 8 bit latch

void
Write_Byte (const uint8_t val, const uint8_t addr)
{
  int		error ;

  // Firstly, set the two address bits, then the eight data bits, then strobe E low and high
  if (((error = gpioWrite (A0, (addr & 0x01) != 0)) != 0) ||
      ((error = gpioWrite (A1, (addr & 0x02) != 0)) != 0) ||
      ((error = gpioWrite (D0,  (val & 0x01) != 0)) != 0) ||
      ((error = gpioWrite (D1,  (val & 0x02) != 0)) != 0) ||
      ((error = gpioWrite (D2,  (val & 0x04) != 0)) != 0) ||
      ((error = gpioWrite (D3,  (val & 0x08) != 0)) != 0) ||
      ((error = gpioWrite (D4,  (val & 0x10) != 0)) != 0) ||
      ((error = gpioWrite (D5,  (val & 0x20) != 0)) != 0) ||
      ((error = gpioWrite (D6,  (val & 0x40) != 0)) != 0) ||
      ((error = gpioWrite (D7,  (val & 0x80) != 0)) != 0) ||
      ((error = gpioWrite (EE,                  0)) != 0))
    Goodbye ("Error writing data %02x to addr %d: %d\n", val, addr, error) ;

  // Wait 5us before pulsing E high again
  Sleep_ns (WRITE_BYTE_DELAY) ;

  if ((error = gpioWrite (EE, 1)) != 0)
    Goodbye ("Error setting E high\n") ;
} // Write_Byte
///////////////////////////////////////////////////////////////////////////////////////////
// Initialise the pigpio interface/library,
// Configure I/O pins as required by the clock
// Set the outputs to their initial values as required by the clock
//
// Some terse help is available at the linux command line using 'man pigpio'

static uint8_t GPIO_inputs[]  = {TOG1a, TOG1b, PB1, PB2, PB3, TOG2a, TOG2b, TOG3a, TOG3b, 0xFF} ;
static uint8_t GPIO_outputs[] = {D0, D1, D2, D3, D4, D5, D6, D7, A0, A1, AMP_ENABLE, RADIO_ENABLE, DIM_PWM, VOLUP, VOLDOWN, 0xFF} ;

void
Initialise_GPIO (void)
{
  int		g, error ;

#ifndef	USE_GPIO_DAEMON
  // If we are not using a daemon, set the sample rate to 10us and use the PWM clocik for timing (leaving PCM clock for audio)
  if ((error = gpioCfgClock(10, 0, 0)) < 0)
    Goodbye ("gpioCfgClock failed: %d\n", error) ;
#endif

  if ((error = gpioInitialise()) < 0)
    Goodbye ("gpioInitialise failed: %d\n", error) ;

  if (Verbose)
    Print_Log (T_NULL, "Setting GPIO inputs:") ;

  for (g=0 ; GPIO_inputs[g] != 0xff ; g++)
  {
    if ((error = gpioSetMode (GPIO_inputs[g], PI_INPUT)) != 0)
      Goodbye ("Error defining GPIO %d as input: %d\n", GPIO_inputs[g], error) ;

    // Set the internal pull up resistors on... its minimum 50k maximum 65k Ohm
    // Valid settings: PI_PUD_OFF, PI_PUD_UP, PI_PUD_DOWN
    // Setting the pull up on will avoid random noise if the input is left floating
    if ((error = gpioSetPullUpDown (GPIO_inputs[g], PI_PUD_UP)) != 0)
      Goodbye ("Error setting pullup/down on GPIO %d input: %d\n", GPIO_inputs[g], error) ;

    if (Verbose)
      Print_Log (T_NULL, " %d", GPIO_inputs[g]) ;
  }

  if (Verbose)
    Print_Log (T_NULL, "\nSetting GPIO outputs:") ;

  for (g=0 ; GPIO_outputs[g] != 0xff ; g++)
  {
    if ((error = gpioSetMode (GPIO_outputs[g], PI_OUTPUT)) != 0)
      Goodbye ("Error defining GPIO %d as output: %d\n", GPIO_outputs[g], error) ;

    if ((error = gpioSetPullUpDown (GPIO_outputs[g], PI_PUD_OFF)) != 0)
      Goodbye ("Error setting pullup/down on GPIO %d output: %d\n", GPIO_outputs[g], error) ;

    if ((error = gpioWrite (GPIO_outputs[g], 1)) != 0)
      Goodbye ("Error setting output %d to 1: %d\n", GPIO_outputs[g], error) ;

    if (Verbose)
      Print_Log (T_NULL, " %d", GPIO_outputs[g]) ;
  }

  // The SPI pins are set into the correct mode when the SPI interface is opened
  if (Verbose)
    Print_Log (T_NULL, "\nInitialising SPI\n") ;

  // Initialise the main SPI, but leave channels 1 and 2 for GPIO (just use channel 0)
  // The SPI needs to be opened in mode 0 (refer to https://abyz.me.uk/rpi/pigpio/cif.html#spiOpen)
  if ((Spi_Handle = spiOpen (SPI_MAIN, SPI_BAUDRATE, 0x0C0)) < 0)
    Goodbye ("Error opening SPI: %d\n", Spi_Handle) ;

  // Now that the ports are initialised, write actual values to the LEDs and latch
  // Writing a non-BCD value to the LEDs will blank them
  Write_Byte (0xff, ADDR_HOURS) ;
  Write_Byte (0xff, ADDR_MINUTES) ;
  Write_Byte (0xff, ADDR_SECONDS) ;
  Write_Byte (0x00, ADDR_LATCH) ; // Turn off all individual LEDs + decimal points

  // Using SW PWM: initialise PWM
  // Hardware PWM only works on some pins. It requires little software intervention, but is not
  // compatible with simultaneous use of the analogue audio output (which uses one of the timers
  // also used by pigpio library)
  // Software PWM works on any gpio pin, but is not as accurate as HW. It jitters around
  // depending on the CPU load. Its only noticeable at the very lowest PWM ratios and particularly
  // when media is playing (ie the CPU is busy)
  if ((error = gpioSetPWMfrequency (DIM_PWM, PWM_FREQ)) < 0)
    Goodbye ("Error initialising SW PWM frequency: %d\n", error) ;

  if ((error = gpioSetPWMrange (DIM_PWM, PWM_RANGE)) < 0)
    Goodbye ("Error initialising SW PWM range: %d\n", error) ;

  if (Verbose)
    Print_Log (T_NULL, "GPIO initialised\n") ;
} // Initialise_GPIO
///////////////////////////////////////////////////////////////////////////////////////////
// Enforce_Volume_Limits
//
// Ensure that Volume, Min_Volume, Vol_Offset and Targ_Vol_Offset are in a sensible range
// when read together

void
Enforce_Volume_Limits (const int new_alarm)
{
  // Ensure Volume is pegged within a sensible range
  if (Volume > MAX_VOLUME)
    Volume = MAX_VOLUME ;
  else if (Volume < 0)
    Volume = 0 ;

  // Ensure Min_Volume is pegged within a sensible range
  if (Min_Volume > MAX_MINIMUM_VOLUME)
    Min_Volume = MAX_MINIMUM_VOLUME ;
  else if (Min_Volume < 0)
    Min_Volume = 0 ;

  // Ensure Vol_Offset is pegged within a sensible range
  if (Vol_Offset > MAX_VOL_OFFSET)
    Vol_Offset = MAX_VOL_OFFSET ;
  else if (Vol_Offset < -MAX_VOL_OFFSET)
    Vol_Offset = -MAX_VOL_OFFSET ;

  // Ensure Targ_Vol_Offset is pegged within a sensible range
  if (Targ_Vol_Offset > MAX_VOL_OFFSET)
    Targ_Vol_Offset = MAX_VOL_OFFSET ;
  else if (Targ_Vol_Offset < -MAX_VOL_OFFSET)
    Targ_Vol_Offset = -MAX_VOL_OFFSET ;

  if (new_alarm)
  {
    // Ensure the new alarm always starts with AT LEAST the minimum volume setting
    // (the offset is then added to the minimum volume setting so that alarms with
    // non-zero offset stay at the same level relative to each other)
    if (Volume < Min_Volume)
    {
      Vol_Offset += Min_Volume - Volume ;
      Targ_Vol_Offset += Min_Volume - Volume ;
    }

    // The target volume can be anywhere between 0 and max - ie the target can be BELOW the minimum starting volume
    if ((Volume + Targ_Vol_Offset) > MAX_VOLUME)
      Targ_Vol_Offset = MAX_VOLUME - Volume ;
    else if ((Volume + Targ_Vol_Offset) < 0)
      Targ_Vol_Offset = -Volume ;
  }

  // Now look at the combo of Volume and offset and ensure they are within the supported range
  if ((Volume + Vol_Offset) > MAX_VOLUME)
    Vol_Offset = MAX_VOLUME - Volume ;
  else if ((Volume + Vol_Offset) < 0)
    Vol_Offset = -Volume ;
} // Enforce_Volume_Limits
///////////////////////////////////////////////////////////////////////////////////////////
// Time_To_Next_Alarm
//
// Determine the number of seconds until the next alarm occurs and set the global next alarm timer (Seconds_To_Next_Alarm)
// If required, also determine if the previous instance of this alarm is still currently active
//
// Note: assumed that pthread_mutex_lock (&Alarms_Mutex) has been called before Time_To_Next_Alarm

time_t
calc_alarm_epoch (const int i, const int j, const int weekday, const uint16_t now_minute_of_day, TS *now)
{
  TS			tnext ;
  TM			tm ;
  int			days ;

  // Is this alarm set on this day of the week? Zero means not set on this day.
  if (Alarms[i].day[j] == 0)
    return -1 ; // alarm i is not set for day j

  // Is it a one-off alarm (>1) or an ongoing alarm (==1)
  if (Alarms[i].day[j] == 1)
  {
    // This alarm occurs every week. We need to determine the epoch when it will next trip
    memmove (&tnext, now, sizeof(tnext)) ;

    // If the alarm will occur later today. all the fields are now set correctly to calculate epoch
    // Does this alarm occur today and is the time in the future?
    if ((j == weekday) && (now_minute_of_day >= Alarms[i].minute_of_day))
    {
      // The next time this alarm occurs is 7 days in the future. Add 7 days in seconds
      tnext.tv_sec += ONE_WEEK ;
    }
    else if (j != weekday)
    {
      // The next time this alarm occurs is less than 7 days in the future. Determine how many days
      // until that date. This will be a number between 0 and 6
      days = (7 + j - weekday) % 7 ;

      // Add that number of days in seconds
      tnext.tv_sec += ONE_DAY * days ;
    }

    // get the date of the alarm into the tm structure
    if (localtime_r (&tnext.tv_sec, &tm) == NULL)
      Goodbye ("localtime_r failed\n") ;

    // Set the correct time of day for the future alarm (leaving date alone)
    tm.tm_hour = Alarms[i].minute_of_day / 60 ;
    tm.tm_min =  Alarms[i].minute_of_day - (tm.tm_hour * 60) ;
    tm.tm_sec = 0 ;

    // Return the epoch associated with that date and time
    return mktime (&tm) ;
  }

  // This alarm is a one-off
  // has the alarm already occurred???
  if (Alarms[i].day[j] <= now->tv_sec)
    return -1 ; // alarm has already tripped

  return Alarms[i].day[j] ;
} // calc_alarm_epoch

void
Time_To_Next_Alarm (const int who, const int check_if_active)
{
  int			i, j, next_alarm_index, prev_alarm_index, time_to_next_alarm, time_from_prev_alarm, time_in_seconds ;
  uint16_t		now_minute_of_day ;
  TS			ts_now, ts_last_week ;
  TM			tm_today ;
  TV			tv ;
  time_t		next_alarm_epoch, prev_alarm_epoch ;
  char			time_string [TIME_STRING_LEN] ;

  // Read the current time (in seconds and nanoseconds since 1st Jan 1970, or whenever the system starting epoch was)
  if (clock_gettime (CLOCK_REALTIME, &ts_now) < 0)
    Goodbye ("clock_gettime failed: %s\n", strerror(errno)) ;

  memmove (&ts_last_week, &ts_now, sizeof(ts_last_week)) ;
  ts_last_week.tv_sec -= ONE_WEEK ;

  // We need a different system call to get today's date and the hour and minute of the day
  // (which also manages the timezone calculation for us)
  if (localtime_r (&ts_now.tv_sec, &tm_today) == NULL)
    Goodbye ("localtime_r failed\n") ;

  // What is the minute of the day NOW?
  now_minute_of_day = (tm_today.tm_hour * 60) + tm_today.tm_min ;

  if (Verbose & (_V_TIMER | _V_SETUP | _V_ALARM))
  {
    // I'm not going to make another OS call to get a different format time. I can just copy the data across
    // from what I have already read
    tv.tv_sec = ts_now.tv_sec ;
    tv.tv_usec = ts_now.tv_nsec / 1000 ;
    Print_Log (who, "Calculating time to next alarm: Now=%s %s (minute of today=%d), Look in the past=%d\n",
	       DayOfWeek[tm_today.tm_wday], Display_Time (time_string, &tv), now_minute_of_day, check_if_active) ;
  }

  // We are only looking one week into the future for alarm times. So set 'infinity' to be beyond that
  time_to_next_alarm = time_from_prev_alarm = 4 * ONE_WEEK ;
  next_alarm_index = prev_alarm_index = -1 ;

  // The outer loop will check every alarm and identify the next one to trip, and the previous one (if prev still active)
  // Each alarm may be set on more than one day of the week, so we will need an inner loop too, for the different days
  for (i=0 ; i < Num_Alarms ; i++)
  {
    // The inner loop below will separately check each day in this alarm definition
    for (j=0 ; j < 7 ; j++)
    {
      // Identify when this alarm last tripped. It must be a time in the past, and not NOW
      // prev_alarm_epoch = time of the previous alarm, ts_now.tv_sec == NOW
      // Is the previous alarm currently active???
      if (check_if_active &&
	  ((prev_alarm_epoch = calc_alarm_epoch (i, j, tm_today.tm_wday, now_minute_of_day, &ts_last_week)) > 0) &&
	  ((time_in_seconds = (int)(ts_now.tv_sec - prev_alarm_epoch)) > 0))
      {
	if (time_in_seconds < time_from_prev_alarm)
	{
	  // This alarm is closer to now than the current previous alarm.... remember it
	  time_from_prev_alarm = time_in_seconds ;
	  prev_alarm_index = i ;
	}

	if (Verbose & (_V_TIMER | _V_SETUP | _V_ALARM))
	{
	  // Display the time at epoch
	  tv.tv_sec = prev_alarm_epoch ;
	  tv.tv_usec = 0 ;
	  Print_Log (who, "  Alarm %d: %s %s. Previously tripped=%ds\n", i+1, DayOfWeek[j], Display_Time (time_string, &tv), -time_in_seconds) ;
	}
      }

      // Identify when this alarm will next trip. It must be a time in the future, and not NOW
      // next_alarm_epoch = time of the next alarm, ts_now.tv_sec == NOW
      if (((next_alarm_epoch = calc_alarm_epoch (i, j, tm_today.tm_wday, now_minute_of_day, &ts_now)) > 0) &&
	  ((time_in_seconds = (int)(next_alarm_epoch - ts_now.tv_sec)) > 0))
      {
	if (time_in_seconds < time_to_next_alarm)
	{
	  // This alarm is sooner than the current soonest.... remember it
	  time_to_next_alarm = time_in_seconds ;
	  next_alarm_index = i ;
	}

	if (Verbose & (_V_TIMER | _V_SETUP | _V_ALARM))
	{
	  // Display the time at epoch
	  tv.tv_sec = next_alarm_epoch ;
	  tv.tv_usec = 0 ;
	  Print_Log (who, "  Alarm %d: %s %s. Time to next alarm=%ds\n", i+1, DayOfWeek[j], Display_Time (time_string, &tv), time_in_seconds) ;
	}
      }
    }
  }

  // Have we been asked to set the alarm on if it is currently active??
  // Need to determine if the most recent previous alarm is still active.
  if (check_if_active &&
      (prev_alarm_index >= 0) &&
      (time_from_prev_alarm < (Alarms[prev_alarm_index].duration * 60)) )
  {
    // There is a previous alarm and it is still active
    Seconds_To_Next_Alarm = 1 ; // flag that this alarm is going to trip in one second from now
    Next_Alarm_Index = prev_alarm_index ;

    if (Verbose & (_V_TIMER | _V_SETUP | _V_ALARM))
      Print_Log (who, "  Previously tripped alarm: (Alarm=%d)\n", Next_Alarm_Index+1) ;

    return ;
  }

  // If there is at least one alarm, next_alarm_index will be >= 0, and time_to_next_alarm will be #seconds to wait
  if (next_alarm_index >= 0)
    Seconds_To_Next_Alarm = time_to_next_alarm ;
  else
    Seconds_To_Next_Alarm = -1 ; // the alarm is not running

  Next_Alarm_Index = next_alarm_index ;

  if (Verbose & (_V_TIMER | _V_SETUP | _V_ALARM))
    Print_Log (who, "  Seconds-to-next-alarm: %ds (Alarm=%d)\n", Seconds_To_Next_Alarm, Next_Alarm_Index+1) ;
} // Time_To_Next_Alarm
///////////////////////////////////////////////////////////////////////////////////////////
time_t
read_day_or_epoch (time_t *t, char *s)
{
  uint32_t	T ;

  // Note: we are going to define sunday as day 0, to make the comparison with a (struct tm) day of week
  // We are going to define time=0 to be N, time=1 to be Y, and other numbers to be an EPOCH time
  if ((*s == 'N') || (*s == 'n'))
  {
    *t = 0 ;
    return 0 ;
  }

  if ((*s == 'Y') || (*s == 'y'))
  {
    *t = 1 ;
    return 0 ;
  }

  // The string can now only contain digits - we will read them as an epoch.
  if (sscanf(s, "%" SCNu32, &T) != 1)
  {
    Print_Log (T_SETUP, "Invalid epoch time '%s'\n", s) ;
    return 1 ;
  }

  *t = (time_t)T ; // time_t is defined differently for 32 and 64 bit systems
  return 0 ;
} // read_day_or_epoch

FILE *
Open_Setup_For_Reading (void)
{
  FILE		*f ;
  int		i ;
  char		buf [FILENAME_SIZE] ;

  // We will try a series of setup files. Normally, the file is called Setup_Filename, but we will try a series of numbered backups also
  // Can we open the setup file?
  if ((f = fopen (Setup_Filename, "r")) != NULL)
    return f ;

  // The default setup file did not work - try one of the backups
  for (i=0 ; i < 5 ; i++)
  {
    sprintf (buf, "%s.%d", Setup_Filename, i) ;

    if ((f = fopen (buf, "r")) != NULL)
      return f ;
  }

  // We have a problem - there is no setup file!
  return NULL ;
} // Open_Setup_For_Reading

// Rename_Backups - renames 'dest_filename' by appending a single digit, then renames
// 'source_filename' to 'dest_filename'
int
Rename_Backups (const int who, char const *source_filename, char const *dest_filename)
{
  int		i, do_move, got_perms, got_older_perms ;
  char		old_name [LOG_STRING_LEN] ;
  char		new_name [LOG_STRING_LEN] ;
  ST		dest_info, file_info, fi ;

  // We want to keep a series of backups of the dest file.
  // Rename the older backups before moving the original file.
  // Note that source_filename and dest_filename could be the same name.
  // The two are specified separately, because the 'source' may be a temp file under /tmp
  do_move = got_perms = got_older_perms = 0 ;

  // Start by finding the current age of the destination filename.
  if (stat (dest_filename, &dest_info) == 0)
  {
    // If the existing log file is larger than 100k bytes, rename it, otherwise keep it
    if (dest_info.st_size >= 100000)
      do_move = 1 ;

    got_perms = 1 ;
  }

  if (do_move)
  {
    // The modification time is older than 1 hour - rename the old versions of the file
    // Also work out the permissions and ownership of the most recent file
    for (i=9 ; i > 0 ; i--)
    {
      sprintf (old_name, "%s.%d", dest_filename, i-1) ;
      sprintf (new_name, "%s.%d", dest_filename, i) ;

      // Can we glean the owner and file permissions from this file?
      if ((rename (old_name, new_name) == 0) && (stat (new_name, &fi) == 0))
      {
	memmove (&file_info, &fi, sizeof(fi)) ; // store the intact information into file_info
	Print_Log (who, "Renamed '%s' to '%s'\n", old_name, new_name) ;
	got_older_perms = 1 ;
      }
    }

    // Move the current dest file to a filename with the backup numbering scheme
    rename (dest_filename, old_name) ;

    if ((got_perms == 0) && got_older_perms)
    {
      memmove (&dest_info, &file_info, sizeof(file_info)) ; // store the intact information into file_info
      got_perms = 1 ;
    }
  }

  // Move the fresh file (source) onto the dest filename (if the dest is different than the source)
  if (strcmp (source_filename, dest_filename) != 0)
  {
    if (rename (source_filename, dest_filename) != 0)
    {
      Print_Log (who, "Renaming '%s' to '%s' failed: %s\n", source_filename, dest_filename, strerror(errno)) ;
      return -1 ;
    }

    Print_Log (who, "Renamed '%s' to '%s'\n", source_filename, dest_filename) ;
  }

  // Last task is to apply the same permissions as the most recent moved file
  if (got_perms)
  {
    chown (dest_filename, dest_info.st_uid, dest_info.st_gid) ;
    chmod (dest_filename, dest_info.st_mode) ;
  }
  else
    chmod (dest_filename, 0666) ; // For some reason, stat failed. Force change permissions

  return 0 ;
} // Rename_Backups
///////////////////////////////////////////////////////////////////////////////////////////
int
Read_Alarm_File_Into_String (const int who, int *file_size, char **file_string, uint32_t *hash_return, time_t *time_return)
{
  FILE		*f ;
  uint32_t	hash ;
  char		*p, c ;
  ST		file_info ;

  // Can we open the alarm file, determine its size and read the file into memory?
  if (((f = fopen (Alarm_Filename, "r")) == NULL) ||
      (fseek (f, 0, SEEK_END) != 0) ||
      ((*file_size = ftell (f)) <= 0) ||
      (fseek (f, 0, SEEK_SET) != 0) )
  {
    Print_Log (who, "Cannot open alarm file and determine its size, or file '%s' is empty: %s\n", Alarm_Filename, strerror(errno)) ;
    goto abort_read_alarm_file_into_string ;
  }

  // We now have an opened file and know its size. Read the entire file into a malloced memory buffer.
  if ((*file_string = malloc ((*file_size)+1)) == NULL)
    Goodbye ("Cannot malloc '%d' bytes for holding '%s'\n", (*file_size)+1, Alarm_Filename) ;

  if ((fread (*file_string, 1, *file_size, f)) != (unsigned long)(*file_size))
  {
    Print_Log (who, "Cannot read '%d' bytes from '%s'\n", *file_size, Alarm_Filename) ;
    free (*file_string) ;

  abort_read_alarm_file_into_string:
    if (f != NULL)
      fclose (f) ;

    *file_string = NULL ;
    *file_size = 0 ;
    return -1 ;
  }

  // The file is now in the memory buffer, we can process its contents
  fclose (f) ;
  (*file_string)[*file_size] = '\0' ; // Terminate the long file string with a null

  // Are we asked to determine the hash of this file?
  if (hash_return != NULL)
  {
    // djb2 hash algorithm creditet to Dan Bernstein
    // 5381 and 33 are important magic numbers for this algorithm
    // 33 comes from shifting left 5 bits and adding - ie multiplying by 33
    for (hash=5381,p=*file_string ; (c = *p++) ;)
      hash = ((hash << 5) + hash) ^ c ;

    *hash_return = hash ;
  }

  // Are we asked to determine the file time?
  if (time_return != NULL)
  {
    if (stat (Alarm_Filename, &file_info) != 0)
    {
      Print_Log (who, "Cannot read file timestamp from '%s': %s\n", Alarm_Filename, strerror(errno)) ;
      *time_return = 0 ; // Return 1st Jan 1970 (a non-negative value that will cause remote file to be loaded)
    }
    else
      *time_return = file_info.st_mtime ;
  }

  return 0 ;
} // Read_Alarm_File_Into_String
///////////////////////////////////////////////////////////////////////////////////////////
// Next_Line
// Returns offset to the beginning of the next line of text.
// next_newline is set to point to the terminating newline after the offset

int
Next_Line (char *buf, int file_size, int current_pos, char **next_newline)
{
  int		i ;

  // Special case with current_pos=-1. This is used to return the beginning of the buffer
  if ((i = current_pos) < 0)
  {
    i = 0 ;
    goto already_know_start ;
  }

  // Find the end of the line commencing at offset=i (this will be a newline characer, or a null)
  for (; (i < file_size) && (buf[i] != '\0') && (buf[i] != '\n'); i++) ;
  i++ ; // point to first character of next line

  // Skip past any consecutive space/tab characters at the beginning of the line
  for (; (i < file_size) && ((buf[i] == ' ') || (buf[i] == '\t')) ; i++) ;

 already_know_start:
  // Remember the start of the next line - we will be returning this number
  current_pos = i ;

  // Find the end of this line
  if (next_newline != NULL)
  {
    for (; (i < file_size) && (buf[i] != '\0') && (buf[i] != '\n'); i++) ;
    *next_newline = &buf[i] ;
  }

  // Finally, return offset of the next line, or -1 if its the end of the file
  if ((current_pos >= file_size) || (buf[current_pos] == '\0'))
    return -1 ;

  return current_pos ;
} // Next_Line
///////////////////////////////////////////////////////////////////////////////////////////
// Read_Variable
//
// decodes a line of text assuming VARIABLE=VALUE format and tries to decode the value
// as an integer.
//
// Returns INT_MIN if it cannot decode anything (meaning ignore this line)

int
Read_Variable (const int who, const int line, const char *filename, char *line_buffer, char *variable, char *value, int *errors)
{
  int		n, var_len, val_len, quoted ;
  char		*s ;

  // We have the next line in line_buffer - the termination (newline) is also in the buffer
  // Eliminate any control characters and whitespace from the end of the line (if any there)
  for (n=strlen(line_buffer) ; (n > 0) && (line_buffer[n-1] <= ' ') ; n--)
    line_buffer[n-1] = '\0' ;

  // skip over leading whitespace, and continue if we find a comment character (# or ;)
  for (s=line_buffer ; (*s != '\0') && (isspace (*s)) ; s++) ;

  if ((*s == '\0') || (*s == '#') || (*s == ';'))
    return INT_MIN ;

  // s now points to the variable name. Read it into variable, converting to upper case
  for (var_len=0 ; var_len < SETUP_FIELD_LEN ; var_len++)
  {
    if ((! isalnum(s[var_len])) && (s[var_len] != '_'))
      break ;

    variable[var_len] = toupper(s[var_len]) ;
  }

  variable[var_len] = '\0' ;

  if (var_len == 0)
  {
    Print_Log (who, "Unrecognised configuration setting at line %d of '%s': '%s'\n", line, filename, line_buffer) ;

    if (errors != NULL)
      (*errors)++ ;

    return INT_MIN ;
  }

  // Skip over whitespace and point to equals sign
  s += var_len ;
  for (; (*s != '\0') && (isspace (*s)) ; s++) ;

  if (*s != '=')
  {
    Print_Log (who, "Expecting VAR=VALUE at line %d of '%s': '%s'\n", line, filename, line_buffer) ;

    if (errors != NULL)
      (*errors)++ ;

    return INT_MIN ;
  }

  // Skip over the equals sign and following whitespace
  for (; (*s != '\0') && (*s == '=') ; s++) ;
  for (; (*s != '\0') && (isspace (*s)) ; s++) ;

  // If quoted, skip over the quotes - we are copying from s
  for (val_len=quoted=0 ; (val_len < SETUP_FIELD_LEN) && (*s != '\0') && (*s != quoted) ; val_len++)
  {
    // Determine if the field starts with a quote
    if ((val_len == 0) && ((*s == '"') || (*s == '\'')))
    {
      quoted = *s++ ; // this skips over the first quote

      // Check for a zero-length string
      if ((*s == quoted) || (*s == '\0'))
	break ;
    }

    value[val_len] = *s++ ;
  }

  value[val_len] = '\0' ;

  if (val_len == 0)
    return INT_MIN ;

  // At this point, we have VAR = VALUE - we can save the data
  n = atoi (value) ; // Convert the value field to a number, (in case we need a numeric value below)

  // slight of hand to avoid returning INT_MIN. It shouldn't happen in all likelihood.
  if (n == INT_MIN)
    n = INT_MIN + 1 ;

  return n ;
}
// Read_Variable
///////////////////////////////////////////////////////////////////////////////////////////
// determine_hostname_and_ip_address() gets the hostname string into My_Name, and
// IP address string into My_IPv4_Address

int
determine_hostname_and_ip_address (const int who)
{
  struct ifaddrs		*ifaddr ;
  struct ifaddrs		*ifa ;
  int				i ;
  SUBTASK			*subtask ;
  char				name [NAME_LEN] ;
  char				address [NI_MAXHOST] ;

  // Determine the hostname and IP address of this system into My_Name and My_IPv4_Address
  if ((i = gethostname(name, NAME_LEN)) < 0)
  {
    Print_Log (who, "Cannot read own hostname: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Determine my own IP address
  if (getifaddrs(&ifaddr) < 0)
  {
    Print_Log (who, "getifaddrs() failed: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // parse the ifaddr data to determine the active IPv4 address
  for (ifa=ifaddr ; ifa != NULL ; ifa=ifa->ifa_next)
  {
    // If there is no IP address, or this is not a IPv4 record, or this is the loop back interface, ignore it
    if ((ifa->ifa_addr != NULL) && (ifa->ifa_addr->sa_family == AF_INET) && (strcmp(ifa->ifa_name, "lo") != 0))
      break ;
  }

  if ((i = getnameinfo (ifa->ifa_addr, sizeof(struct sockaddr_in), address, NI_MAXHOST, NULL, 0, NI_NUMERICHOST)) != 0)
  {
    fprintf(stderr, "getnameinfo() failed: %s\n", gai_strerror(i)) ;
    return EXIT_FAILURE ;
  }

  name [NAME_LEN-1] = '\0' ;
  memmove (My_Name, name, NAME_LEN) ;
  My_IPv4_sin_addr = ((struct sockaddr_in *)ifa->ifa_addr)->sin_addr.s_addr ; // which D***head defined this structure way back when?
  memmove (My_IPv4_Address, address, NI_MAXHOST) ;

  if ((who >= 0) && (who < T_NULL) && (Verbose & (_V_SETUP | _V_BUTTON)))
  {
    // Determine who has initiated the call and get the task name
    subtask = &Subtasks[who] ;
    Print_Log (who, "%s: Hostname='%s' IP='%s'\n", subtask->name, My_Name, My_IPv4_Address) ;
  }

  freeifaddrs (ifaddr) ;
  return EXIT_SUCCESS ;
} // determine_hostname_and_ip_address
///////////////////////////////////////////////////////////////////////////////////////////
// Read_Setup_And_Alarms
// routine that loads the two configuration files - setup file and alarm definition file
int
Read_Setup_And_Alarms (const int who, const int check_if_alarm_active)
{
  FILE		*f, *g ;
  int		val_len, quoted, line, line_os, n, m, file_size, offset, mem_size, setup_errors ;
  int16_t	init_vol_offset, targ_vol_offset ;
  uint16_t	hours, minutes, duration, verbose ;
  uint32_t	hash ;
  time_t	file_time ;
  TM		*time_info ;
  char		s_day[7][TIME_STRING_LEN] ;
  char		*s, *file_string, *end_of_line ;
  char		line_buffer [LOG_STRING_LEN] ;
  char		variable[SETUP_FIELD_LEN+1] ;
  char		value[SETUP_FIELD_LEN+1] ;
  char		path[SETUP_FIELD_LEN+1] ;
  char		temp_str[LONGER_STR_LEN+1] ;

  // Can we open the setup file?
  if ((f = Open_Setup_For_Reading()) == NULL)
  {
    // We have a problem - there is no setup file!
    Print_Log (who, "Cannot open setup file '%s': %s\n", Setup_Filename, strerror(errno)) ;
    return 0 ;
  }

  strcpy (Default_Stream_Or_File, "radio") ;
  strcpy (Fallback_Alarm_File, "radio") ;
  This_Clock_Type = 'S' ;
  temp_str[0] = '\0' ;
  setup_errors = 0 ;
  verbose = 0 ; // start with empty verbose bits

  // we now have an opened file. We can read and process the file line by line
  for (line=1 ; fgets (line_buffer, LOG_STRING_LEN, f) != NULL ; line++)
  {
    if ((n = Read_Variable (who, line, Setup_Filename, line_buffer, variable, value, &setup_errors)) < 0)
      continue ;

    // Check the options one by one and set the variables
    if (strcasecmp(variable, "DEFAULT_1224") == 0)
    {
      if (n == 12)
	Time_Format_1224 = 12 ;
      else if (n == 24)
	Time_Format_1224 = 24 ;
      else
      {
	Time_Format_1224 = 24 ;
	Print_Log (who, "Unrecognised DEFAULT_1224 value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_SNOOZE_DURATION") == 0)
    {
      // the DEFAULT_SNOOZE_DURATION is expressed in minutes, but here, its expressed in seconds
      if ((n > 0) && (n <= 60))
	Default_Snooze_Duration = (uint16_t)n * 60 ; // expressed in seconds
      else
      {
	Default_Snooze_Duration = DEFAULT_SNOOZE_DURATION ;
	Print_Log (who, "Unrecognised DEFAULT_SNOOZE_DURATION value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_HOUR_ZERO") == 0)
    {
      if (strcasecmp(value, "BLANK") == 0)
	Leading_Zero_Blanking = 1 ;
      else
	Leading_Zero_Blanking = 0 ;

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_SNOOZE_PRESS") == 0)
    {
      if (strcasecmp(value, "IGNORE") == 0)
	Second_Snooze_Ignore = 1 ;
      else
	Second_Snooze_Ignore = 0 ;

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_MIN_LED") == 0)
    {
      if ((n >= 0) && (n <= 0xff))
	Min_PWM_Ratio = (double)n / 255.0 ;
      else
      {
	Min_PWM_Ratio = MIN_PWM_RATIO ;
	Print_Log (who, "Unrecognised DEFAULT_MIN_LED value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
   }

    if (strcasecmp(variable, "DEFAULT_MAX_LED") == 0)
    {
      if ((n >= 0) && (n <= 0xff))
	Max_PWM_Ratio = (double)n / 255.0 ;
      else
      {
	Max_PWM_Ratio = MAX_PWM_RATIO ;
	Print_Log (who, "Unrecognised DEFAULT_MAX_LED value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_MIN_AMBIENT") == 0)
    {
      if ((n >= 0) && (n <= 4095))
	Min_LDR_Threshold = n ;
      else
      {
	Min_LDR_Threshold = MIN_LDR_THRESH ;
	Print_Log (who, "Unrecognised DEFAULT_MIN_AMBIENT value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_MAX_AMBIENT") == 0)
    {
      if ((n >= 0) && (n <= 4095))
	Max_LDR_Threshold = n ;
      else
      {
	Max_LDR_Threshold = MAX_LDR_THRESH ;
	Print_Log (who, "Unrecognised DEFAULT_MAX_AMBIENT value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_TIME") == 0)
    {
      // Read the hour and minute from the string
      if ((sscanf (value, " %02hu:%02hu", &Default_Alarm_Hour, &Default_Alarm_Minute) != 2) ||
	  (Default_Alarm_Hour > 23) || (Default_Alarm_Minute > 59))
      {
	Default_Alarm_Hour = Default_Alarm_Minute = 0 ;
	Print_Log (who, "Unrecognised DEFAULT_TIME value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_ALARM_DURATION") == 0)
    {
      if ((n > 0) && (n <= MAX_DURATION))
	Default_Alarm_Duration = (uint16_t)n ;
      else
      {
	Default_Alarm_Duration = DEFAULT_DURATION ;
	Print_Log (who, "Unrecognised DEFAULT_ALARM_DURATION value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_MEDIA_DURATION") == 0)
    {
      if ((n > 0) && (n <= MAX_DURATION))
	Default_Media_Duration = (uint16_t)n ;
      else
      {
	Default_Media_Duration = DEFAULT_DURATION ;
	Print_Log (who, "Unrecognised DEFAULT_MEDIA_DURATION value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "DEFAULT_STREAM_OR_FILE") == 0)
    {
      strncpy (Default_Stream_Or_File, value, SETUP_FIELD_LEN+1) ;

      // ALL paths to a file must commence with a '/'. If it does, verify that the file exists
      if (Default_Stream_Or_File[0] == '/')
      {
	if (access (Default_Stream_Or_File, R_OK))
	{
	  // File doesn't exist or is not readable
	  Print_Log (who, "Default Alarm File '%s' is not accessible\n", Default_Stream_Or_File) ;
	  strcpy (Default_Stream_Or_File, "radio") ;
	  setup_errors++ ;
	}
      }

      // Is it a URL or radio? If not, there's a problem
      else if ((strncasecmp(Default_Stream_Or_File, "http://", 7) != 0) && (strcasecmp(Default_Stream_Or_File, "radio") != 0))
      {
	// File doesn't exist or is not readable
	Print_Log (who, "Fallback Alarm File '%s' is not in URL format or 'radio'\n", Default_Stream_Or_File) ;
	strcpy (Default_Stream_Or_File, "radio") ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "FALLBACK_ALARM_FILE") == 0)
    {
      strncpy (Fallback_Alarm_File, value, SETUP_FIELD_LEN+1) ;

      // ALL paths to a file must commence with a '/'. If it does, verify that the file exists
      if (Fallback_Alarm_File[0] == '/')
      {
	if (access (Fallback_Alarm_File, R_OK))
	{
	  // File doesn't exist or is not readable
	  Print_Log (who, "Fallback Alarm File '%s' is not accessible\n", Fallback_Alarm_File) ;
	  strcpy (Fallback_Alarm_File, "radio") ;
	  setup_errors++ ;
	}
      }

      // Is it a URL or radio? If not, there's a problem
      else if ((strncasecmp(Fallback_Alarm_File, "http://", 7) != 0) && (strcasecmp(Fallback_Alarm_File, "radio") != 0))
      {
	// File doesn't exist or is not readable
	Print_Log (who, "Fallback Alarm File '%s' is not in URL format or 'radio'\n", Fallback_Alarm_File) ;
	strcpy (Fallback_Alarm_File, "radio") ;
	setup_errors++ ;
      }

      continue ;
    }

    // Clock types:
    // Stand Alone: Stand alone clocks operate independently of any other clocks. They do not synchronise
    //              their alarms with other clocks, are not controlled by other clocks, and do not control
    //              other clocks. The clock does not share alarm settings or button activity with other
    //              clocks. Stand alone clocks are audible.
    //
    // If you have more than one clock, you can optionally group two or more clocks into a cluster. Each
    // clustered clock will sync its alarms with the others so that alarm changes on any clock in the
    // cluster propagate shortly after to the other clocks in the cluster.
    //
    // Clustered clocks may either be 'silent' or 'audible'. Silent clocks will not normally sound any
    // alarms even if alarms are defined within the cluster. The exception to this is if and only if
    // a silent clustered clock cannot see any audible clustered clocks on the network. In that case,
    // the silent clock will sound its alarms.
    //
    // You would want to declare a clock to be silent if two or more clustered clocks are within hearing
    // distance of each (for example on either side of a bed) because digital media isn't perfectly
    // synchronised across devices and audio time lags don't sound good.
    //
    // Autonomous: The clock shares alarm settings with other clocks. It can control other clocks but
    //             cannot be controlled by other clocks. Autonomous clocks are audible.
    //
    // Clustered:  The clock shares alarm settings with other clocks. It can both control and be
    //             controlled by other clocks. Clustered clocks are audible.
    //
    // Silent clustered: The clock shares alarm settings with other clocks. It can control other clocks.
    //             Silent clustered clocks will not sound alarms UNLESS there are no other audible
    //             clustered clocks visible on the network!

    if (strcasecmp(variable, "CLOCK_TYPE") == 0)
    {
      if ((value[0] == 'A') || (value[0] == 'a'))
	This_Clock_Type = 'A' ;

      else if ((value[0] == 'C') || (value[0] == 'c'))
	This_Clock_Type = 'C' ;

      else if ((value[0] == 'Z') || (value[0] == 'z'))
	This_Clock_Type = 'Z' ;

      else if ((value[0] == 'S') || (value[0] == 's'))
	This_Clock_Type = 'S' ;

      else
      {
	Print_Log (who, "CLOCK_TYPE setting unexpected: '%s'\n", value) ;
	setup_errors++ ;
      }

      continue ;
    }

    if (strcasecmp(variable, "VOLUME") == 0)
    {
      if ((n >= 0) && (n <= MAX_VOLUME))
	Volume = n ;
      else
      {
	Volume = DEFAULT_VOLUME ;
	Print_Log (who, "Unrecognised VOLUME value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      Enforce_Volume_Limits (0) ;
      continue ;
    }

    if (strcasecmp(variable, "MIN_VOLUME") == 0)
    {
      if ((n >= 0) && (n <= MAX_MINIMUM_VOLUME))
	Min_Volume = n ;
      else
      {
	Min_Volume = 0 ;
	Print_Log (who, "Unrecognised MIN_VOLUME value at line %d of '%s': '%s'\n", line, Setup_Filename, line_buffer) ;
	setup_errors++ ;
      }

      Enforce_Volume_Limits (0) ;
      continue ;
    }

    // VERBOSE - this is a special variable. It is not defined or managed by the HTML code.
    // Previx 0x for hex, or no prefix means decimal
    if (strcasecmp(variable, "VERBOSE") == 0)
    {
      // Try reading the value as a hex number - format 0xHHHH or 0XHHHH.
      // If that fails, try reading the decimal value
      if (sscanf (value, "0%*[xX]%" SCNx16, &verbose) != 1)
	verbose = 0 ;

      continue ;
    }

    // LOG_FILE - also a special variable
    // If you want the clock daemon to dump a log, then manually edit the setup.conf file
    // adding the variable LOG_FILE=path/to/file, and this will trigger the opening of the
    // log file. If the log file is ALREADY OPENED, then don't open again!
    if ((strcasecmp(variable, "LOG_FILE") == 0) && (Log_File == NULL) && (value[0] != '\0'))
    {
      strcpy (Log_Filename, value) ;

      // Before we open the log file for writing, rename any previous log files
      Rename_Backups (who, Log_Filename, Log_Filename) ;

      // Open the log file for appending (it will open a fresh file if it doesn't already exist)
      if ((Log_File = fopen (Log_Filename, "a")) != NULL)
      {
	Print_Log (who, "*****************************************************************************\n"
		   "Digital Alarm Clock V" VERSION ". Setup verbosity=0x%03x, Command line verbosity=0x%03x\n", verbose, Cmdline_Verbose) ;

	if (Model_String[0] == '\0')
	{
	  // Determine details about the raspberry pi hardware
	  if ((g = fopen (Model, "r")) == NULL)
	    Print_Log (who, "Cannot open '%s': %s\n", Model, strerror(errno)) ;

	  else if (fgets (Model_String, sizeof(Model_String), g) == NULL)
	  {
	    Print_Log (who, "Cannot read from '%s': %s\n", Model, strerror(errno)) ;
	    Model_String[0] = '\0' ;
	  }

	  fclose (g) ;
	}

	if ((g = fopen (Mem, "r")) == NULL)
	  fprintf (stderr, "Cannot open '%s': %s\n", Mem, strerror(errno)) ;

	else if (fgets (temp_str, sizeof(temp_str), g) == NULL)
	  Print_Log (who, "Cannot read from %s: %s\n", Mem, strerror(errno)) ;

	else if (sscanf (temp_str, "MemTotal: %d", &mem_size) != 1)
	  Print_Log (who, "Cannot read memory size from '%s'\n", Mem) ;

	else
	  Print_Log (who, "Running on %s [%.3f GB RAM available to Linux]\n", Model_String, mem_size / 1024.0 / 1024.0) ;

	if (g != NULL)
	  fclose (g) ;

	// Determine the OS version and architecture using uname -r and uname -m
	if (OS_String[0] == '\0')
	{
	  // Determine details about the raspberry pi hardware
	  if ((g = fopen (OS, "r")) == NULL)
	    Print_Log (who, "Cannot open '%s': %s\n", OS, strerror(errno)) ;

	  else // Read a couple of values out of the file
	  {
	    for (line_os=1 ; fgets (line_buffer, LOG_STRING_LEN, g) != NULL ; line_os++)
	    {
	      if ((n = Read_Variable (who, line_os, OS, line_buffer, variable, value, NULL)) < 0)
		continue ;

	      // Check the options one by one and set the variables
	      if (strcasecmp(variable, "PRETTY_NAME") == 0)
	      {
		strncpy (OS_String, value, MODSTR_LEN) ;
		OS_String[MODSTR_LEN-1] = '\0' ;
		continue ;
	      }

	      if ((strcasecmp(variable, "VERSION_ID") == 0) && (n > 0))
		OS_Version = (uint16_t)n ;
	    }

	    if (OS_String[0] != '\0')
	      Print_Log (who, "Operating System='%s'\n", OS_String) ;

	    fclose (g) ;
	  }
	}

	// Determine the OS version and architecture using uname -r and uname -m
	if ((g = popen ("uname -m", "r")) == NULL)
	  Print_Log (who, "Cannot popen to determine architecture: %s\n", strerror(errno)) ;

	else if (fgets (temp_str, sizeof(temp_str), g) == NULL)
	  Print_Log (who, "Cannot read architecture: %s\n", strerror(errno)) ;

	else
	{
	  pclose (g) ;

	  if ((g = popen ("uname -r", "r")) == NULL)
	    Print_Log (who, "Cannot popen to determine kernel version: %s\n", strerror(errno)) ;

	  else if (fgets (value, sizeof(value), g) == NULL)
	    Print_Log (who, "Cannot read kernel version: %s\n", strerror(errno)) ;

	  else
	  {
	    // Eliminate newlines at the ends of the two strings.
	    for (s=temp_str ; (*s != '\0') && (*s != '\n') ; s++) ;
	    *s = '\0' ;
	    for (s=value ; (*s != '\0') && (*s != '\n') ; s++) ;
	    *s = '\0' ;
	    Print_Log (who, "Architecture=%s, kernel=%s\n", temp_str, value) ;
	  }
	}

	if (g != NULL)
	  fclose (g) ;

	// Check for presence of Y2038 bug. The 32 bit versions of the Pi Bullseye image contain this bug!
	// It has allegedly been fixed a long time ago in glibc and the Linux kernel, but the fix does not
	// yet seem to have been ported to the 32 bit Pi image!
	if (sizeof(time_t) <= 4)
	  Print_Log (who, "This operating system has the Y2038 time bug. Refer to comment at end of the Readme.txt file\nin the alarm_clock source code directory. Reinstall the clock using a compliant OS before 2038!\n") ;

	// Determine the version of mpv that is installed, and display
	if ((g = popen ("mpv --version", "r")) == NULL)
	{
	  Print_Log (who, "Cannot popen to determine mpv version: %s\n", strerror(errno)) ;
	  setup_errors++ ;
	}

	else if (fgets (temp_str, sizeof(temp_str), g) == NULL)
	  Print_Log (who, "Cannot read mpv version: %s\n", strerror(errno)) ;

	else // The mpv version has a trailing new line
	  Print_Log (who, "%s", temp_str) ;

	if (g != NULL)
	  pclose (g) ;

	// Determine the version of bluetoothctl that is installed, and display
	// The Bookworm and Bullseye versions of the OS have different bluetoothctl syntax for the paired devices command
	Bluetoothctl_Devices_Cmd = "paired-devices\n" ;
	Bluetoothctl_Devices_String = "paired-devices" ;

	if ((g = popen ("bluetoothctl --version", "r")) == NULL)
	{
	  Print_Log (who, "Cannot popen to determine bluetoothctl version: %s\n", strerror(errno)) ;
	  setup_errors++ ;
	}

	else if (fgets (temp_str, sizeof(temp_str), g) == NULL)
	  Print_Log (who, "Cannot read bluetoothctl version: %s\n", strerror(errno)) ;

	else // The bluetoothctl version has a trailing new line
	{
	  Print_Log (who, "%s", temp_str) ;

	  // find the character after the space and read the number... this is the version number
	  for (s=temp_str ; (*s != '\0') && (!isspace(*s)) ; s++) ;
	  for (; (*s != '\0') && (isspace (*s)) ; s++) ;

	  // Versions greater than 5.65 have slightly different syntax
	  if ((*s != '\0') && (atof(s) >= 5.65))
	  {
	    Bluetoothctl_Devices_Cmd = "devices Paired\n" ; // Note the capital P in Paired! It is case sensitive!!
	    Bluetoothctl_Devices_String = "devices Paired" ;
	  }
	}

	if (g != NULL)
	  pclose (g) ;

	Print_Log (who, "This PID: %5d\n", getpid()) ;
	Print_Log (who, "Real UID: %5d, Effective UID: %5d\n", getuid(), geteuid()) ;
	Print_Log (who, "Real GID: %5d, Effective GID: %5d\n", getgid(), getegid()) ;

	n = umask (0666) ; // Set file creation mask to 666
	m = umask (0666) ; // now check its value
	Print_Log (who, "File creation mask: %o at launch. Now set to %o.\n\n", ~n & 0x1ff, ~m & 0x1ff) ;
      }
      else
      {
	Print_Log (who, "Cannot open log file '%s': %s\n", Log_Filename, strerror(errno)) ;
	setup_errors++ ;
      }

      continue ;
    }

    // Ignore all other variable names
    // Some of the variables in the setup file are used by the cgi scripts, so we won't count the unknows as errors
  }

  // Close the setup file to release the file handle
  fclose (f) ;
  Verbose = verbose | Cmdline_Verbose ; // Set this in one location now as an atomic function, so other threads don't skip any logs

  // Do some basic sanity checks on ranges and make sensible
  if (Min_PWM_Ratio > Max_PWM_Ratio)
  {
    Print_Log (who, "Minimum PWM ratio (%.3f) exeeds max (%.3f)\n", Min_PWM_Ratio, Max_PWM_Ratio) ;
    Min_PWM_Ratio = Max_PWM_Ratio ;
    setup_errors++ ;
  }

  if (Min_LDR_Threshold > Max_LDR_Threshold)
  {
    Print_Log (who, "Minimum ambient light threshold (%d) exeeds max (%d)\n", Min_LDR_Threshold, Max_LDR_Threshold) ;
    Min_LDR_Threshold = Max_LDR_Threshold ;
    setup_errors++ ;
  }

  if (Verbose & _V_SETUP)
  {
    Print_Log (who, "Reloaded setup:\n") ;

    if (setup_errors > 0)
      Print_Log (who, "  *** %d ERRORS when loading setup file\n", setup_errors) ;

    Print_Log (who, "  Clock type: %c\n", This_Clock_Type) ;
    Print_Log (who, "  Default 12 or 24H display: %d\n", Time_Format_1224) ;
    Print_Log (who, "  Leading zero blanking: %c\n", Leading_Zero_Blanking ? 'Y' : 'N') ;
    Print_Log (who, "  Default alarm time: %02hu:%02hu\n", Default_Alarm_Hour, Default_Alarm_Minute) ;
    Print_Log (who, "  Default alarm duration: %hu (minutes)\n", Default_Alarm_Duration) ;
    Print_Log (who, "  Default media duration: %hu (minutes)\n", Default_Media_Duration) ;
    Print_Log (who, "  Default snooze duration: %hu (seconds)\n", Default_Snooze_Duration) ;
    Print_Log (who, "  Second snooze press ignore: %c\n", Second_Snooze_Ignore ? 'Y' : 'N') ;
    Print_Log (who, "  Volume: %d\n", Volume) ;
    Print_Log (who, "  Minimum Volume: %d\n", Min_Volume) ;
    Print_Log (who, "  Default stream or file: '%s'\n", Default_Stream_Or_File) ;
    Print_Log (who, "  Fallback alarm file: '%s'\n", Fallback_Alarm_File) ;
    Print_Log (who, "  Default minimum ambient light threshold: %hu\n", Min_LDR_Threshold) ;
    Print_Log (who, "  Default maximum ambient light threshold: %hu\n", Max_LDR_Threshold) ;
    Print_Log (who, "  Minimum PWM ratio: %.2f\n", Min_PWM_Ratio) ;
    Print_Log (who, "  Maximum PWM ratio: %.2f\n", Max_PWM_Ratio) ;

    if (Log_File != NULL)
    {
      Print_Log (who, "  Log File: '%s'\n", Log_Filename) ;
      Print_Log (who, "  Setup file verbose bitmask: 0x%03X\n", verbose) ;
    }

    Print_Log (who, "  Command line verbose bitmask: 0x%03X\n\n", Cmdline_Verbose) ;
  }
  //------------------------------------------------------------------------------------------------------------------
  if (Read_Alarm_File_Into_String (who, &file_size, &file_string, &hash, &file_time) < 0)
  {
    // Set a dummy hash, and set file time to EPOCH + 1 second - it is almost the first possible time
    // If this clock is clustered, it will multicast its presence and will learn about other clocks
    // and will request a download of an alarm file that has a more recent timestamp than EPOCH + 1s
    Alarm_File_Hash = 0x00000000 ;
    Alarm_File_Time = 1 ;
    return 0 ;
  }

  if ((Verbose & _V_SETUP) &&
      ((time_info = localtime (&file_time)) != NULL) &&
      (strftime (temp_str, sizeof(temp_str), Time_Format_Str, time_info) > 0) )
    Print_Log (who, "  Alarm config file hash: 0x%x\n  Alarm config file time: %s\n", hash, temp_str) ;

  // We have read the hash - save it so it can be muticast if required
  // The reason for using the temporary variable 'hash' is so as not to klobber the global variable if something failed
  Alarm_File_Hash = hash ;
  Alarm_File_Time = file_time ;

  // The reading of each alarm line is not quite trivial because the day columnes may contain either 'Y', 'N' or a number representing a time epoch.
  // This means we cannot use a one-liner sscanf for the whole line, as we don't know in advance what the column formats will be
  // We will need to read the day columns as strings, and then convert those strings afterwards.
  // Note: the dancing around with printing the format string to a variable in the line below is to convey the max length of the string
  // (The absence of this capability with sscanf is an oversight in the C language IMO)
  //
  // Format: HH:MM, active/suspended, duration, vol offset , Mon , Tue , Wed , Thu , Fri , Sat , Sun , URL/File Path/radio
  sprintf (temp_str, " %%02hu:%%02hu , %%%d[^ ,] , %%hu , %%%d[0-9- .] , %%%d[YyNn0-9] , %%%d[YyNn0-9] , %%%d[YyNn0-9] , %%%d[YyNn0-9] , %%%d[YyNn0-9] , %%%d[YyNn0-9] , %%%d[YyNn0-9] , %%%d[^ \n\t\r\f\v]",
	   SETUP_FIELD_LEN, TIME_STRING_LEN, TIME_STRING_LEN, TIME_STRING_LEN, TIME_STRING_LEN, TIME_STRING_LEN, TIME_STRING_LEN, TIME_STRING_LEN, TIME_STRING_LEN, SETUP_FIELD_LEN) ;

  // Ensure only one thread can access the alarm variables at any one time
  pthread_mutex_lock (&Alarms_Mutex) ;

  // Next_Line() returns offset to the beginning of the next line of text
  // end_of_line is pointer to the terminating newline of the next line
  end_of_line = NULL ; // to shut up the compiler warning
  Num_Alarms = 0 ;

  for (offset=-1,line=0 ; (offset = Next_Line (file_string, file_size, offset, &end_of_line)) >= 0 ; line++)
  {
    // Skip over the first line - it contains column headings which we are not interested in
    if (offset == 0)
      continue ;

    // Read the time and days and audio stream from their separate columns
    if ((n=sscanf(&file_string[offset], temp_str, &hours, &minutes, variable, &duration, value, s_day[1], s_day[2], s_day[3], s_day[4], s_day[5], s_day[6], s_day[0], path)) != 13)
    {
      Print_Log (who, "Read %d fields from alarm definition line %d of '%s'\n", n, line, Alarm_Filename) ;
      continue ;
    }

    // Extract the volume offsets from the volume offset string
    init_vol_offset = targ_vol_offset = 0 ;

    if ((n=sscanf(value, "%" SCNd16 " %*[.] %" SCNd16, &init_vol_offset, &targ_vol_offset)) == 0)
    {
      Print_Log (who, "Volume offset syntax problem at line %d of '%s': '%s'\n", line, Alarm_Filename, value) ;
      continue ;
    }

    if (n == 1)
      targ_vol_offset = init_vol_offset ;

    // Is this record suspended? Ignore if so
    if (strcasecmp(variable, "_SUSPENDED_") == 0)
      continue ;

    // We have read valid alarm data into our variables. Define the alarm
    if (Num_Alarms == MAX_NUM_ALARMS)
    {
      Print_Log (who,
		 "Support for a maximum of %d alarms exceeded at line %d of '%s'\n"
		 "Ignoring the remainder of the alarm file\n", MAX_NUM_ALARMS, line, Alarm_Filename) ;
      break ;
    }

    // The sscanf has already determined that the characters in the day fields are Y, N or a number
    if ((hours > 23) || (minutes > 59) || (duration == 0) || (duration > MAX_DURATION) ||
	(init_vol_offset < -MAX_VOL_OFFSET) || (init_vol_offset > MAX_VOL_OFFSET) ||
	(targ_vol_offset < -MAX_VOL_OFFSET) || (targ_vol_offset > MAX_VOL_OFFSET) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[0], s_day[0]) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[1], s_day[1]) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[2], s_day[2]) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[3], s_day[3]) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[4], s_day[4]) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[5], s_day[5]) ||
	read_day_or_epoch (&Alarms[Num_Alarms].day[6], s_day[6]))
    {
      Print_Log (who, "Cannot validate all alarm definition fields at line %d of '%s'\n", line, Alarm_Filename) ;
      continue ;
    }

    // The media file may be enclosed in quotes - we need to strip out the quotes if present
    for (s=path ; (*s != '\0') && (isspace (*s)) ; s++) ;

    // Strip quotes by moving the unquoted characters (at *s) to the beginning of value[]
    // Quotes may either be ' or "
    // If quoted, skip over the quotes
    for (val_len=quoted=0 ; (val_len < SETUP_FIELD_LEN) && (*s != '\0') && (*s != quoted) ; val_len++)
    {
      // Determine if the media field starts with a quote
      if ((val_len == 0) && ((*s == '"') || (*s == '\'')))
      {
	quoted = *s++ ; // this skips over the first quote

	// Check for a zero-length string
	if ((*s == quoted) || (*s == '\0'))
	  break ;
      }

      value[val_len] = *s++ ; // accumulate characters in 'value' string
    }

    value[val_len] = '\0' ; // terminate the media field string

    // If a file, verify that the file exists. ALL paths to a file must commence with a '/' character
    if (value[0] == '/')
    {
      if (access (value, R_OK))
      {
	// File doesn't exist or is not readable
	Print_Log (who, "Alarm %d: cannot access alarm file '%s'\n", Num_Alarms+1, value) ;
	goto replace_path ;
      }
    }

    // Is it a URL or radio? If not, there's a problem
    else if ((strncasecmp(value, "http://", 7) != 0) &&
	     (strncasecmp(value, "https://", 8) != 0) &&
	     (strcasecmp(value, "radio") != 0))
    {
      // File doesn't exist or is not readable
      Print_Log (who, "Alarm %d: '%s' is not in URL format or 'radio'\n", Num_Alarms+1, value) ;

    replace_path:
      if (Fallback_Alarm_File[0] == '\0')
	continue ;

      // Set the fallback alarm file as the alarm
      strncpy (value, Fallback_Alarm_File, SETUP_FIELD_LEN+1) ;
    }

    // Our pre-checks are OK. Lets record the new alarm details
    Alarms[Num_Alarms].minute_of_day = (hours * 60) + minutes ;
    Alarms[Num_Alarms].duration = duration ;
    Alarms[Num_Alarms].init_vol_offset = init_vol_offset ;
    Alarms[Num_Alarms].targ_vol_offset = targ_vol_offset ;
    strncpy (Alarms[Num_Alarms].url_path, value, SETUP_FIELD_LEN+1) ; // a string that holds the path to the file or media URL
    Num_Alarms++ ;

    if (Verbose & _V_SETUP)
    {
      if (init_vol_offset == targ_vol_offset)
	Print_Log (who, "  Alarm %d: %02hu:%02hu (%hu minutes) (vol offset=%" PRId16 ") '%s'\n",
		   Num_Alarms, hours, minutes, duration, init_vol_offset, value) ;
      else
	Print_Log (who, "  Alarm %d: %02hu:%02hu (%hu minutes) (vol offset=%" PRId16 "..%" PRId16 ") '%s'\n",
		   Num_Alarms, hours, minutes, duration, init_vol_offset, targ_vol_offset, value) ;
    }
  }

  // Now that the alarm table has been reset, determine when the next alarm will occur
  Time_To_Next_Alarm (who, check_if_alarm_active) ;
  pthread_mutex_unlock (&Alarms_Mutex) ;

  if (file_string != NULL)
    free (file_string) ;

  if (determine_hostname_and_ip_address (who) != EXIT_SUCCESS)
    return 0 ;

  return 1 ;
} // Read_Setup_And_Alarms
///////////////////////////////////////////////////////////////////////////////////////////
// Read_A_to_D
// Written for MCP3201 chip.
//
// LDR is connected between ground and Vin+ of A to D.
// Vin+ is pulled high by 10k resistor, forming a voltage divider with the LDR.
// Vref of the A to D chip is tied high. Vin- is tied to ground.
// A dark input reads close to 0xfff. A very bright input reads close to 0x000.
//
// Due to thermal variation and imprecision of the A to D, the last digit varies a little.
// For practical purposes, a moving average of recent readings is recommended to level out
// the noise.

#define NUM_SPI_BYTES	(2)

uint16_t
Read_A_to_D (void)
{
  uint8_t	buf[NUM_SPI_BYTES] ;
  uint32_t	n ;

  if ((n = (uint32_t)spiRead (Spi_Handle, (char *)buf, NUM_SPI_BYTES)) != NUM_SPI_BYTES) // read the A to D converter
    Goodbye ("Error reading A to D: %d\n", n) ;

  // The format of the SPI read needs to be bit-bashed
  // n will be a number between 0x000 and 0xfff (low meaning brighter light level)
  n = ((buf[1] >> 1) & 0x7f) | ((buf[0] & 0x1f) << 7) ;

  // Flip the result around so a lower number means darker input
  return (uint16_t)(0xfff - n) ;
} // Read_A_to_D
///////////////////////////////////////////////////////////////////////////////////////////
// Num_To_BCD - routine to convert a binary number into binary coded decimal
//
// Returns the 8 bit binary number to be written to one of the dual-digit displays.
// The 8 bit number represents two 4 bit Binary Coded Decimal values
// The 74HCT4511 display drivers will blank the digit if the BCD value exceeds 9

uint8_t
Num_To_BCD (uint8_t val, uint8_t blank_leading_zero)
{
  uint8_t	ret ;

  // The number which is being displayed MUST be in the range 0..99
  if (val > 99)
  {
    if (Verbose & _V_DISPLAY)
      Print_Log (T_DISPLAY_DRIVER, "Error with BCD conversion: number=%d\n", val) ;

    val = 99 ;
  }

  // Calculate the first 4 bit BCD digit (the most significant 4 bits, ie the tens column)
  ret = (val / 10) << 4 ;

  if ((blank_leading_zero != 0) && (ret == 0))
    ret = 0xf0 ; // A non-decimal value will cause the 74HCT4511 to blank the display

  // Now calculate the units BCD value
  ret += val % 10 ;
  return ret ;
} // Num_To_BCD
///////////////////////////////////////////////////////////////////////////////////////////
// Start_MPV and Quit_MPV
//
// Call these to launch or stop the mpv player

void
Start_MPV (const int who, const char *from)
{
  if (Mpv_Handle != NULL)
    return ;

  // Launch mpv
  // The following calls specify what you'd normally specify when invoking mpv using the command line interface
  // each of the options are what would normally be specified with '--' then the option name
  if (((Mpv_Handle = mpv_create()) == NULL) ||
      (mpv_set_option_string (Mpv_Handle, "input-default-bindings", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "input-vo-keyboard", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "input-terminal", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "vid", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "osc", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "term-osd", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "msg-level", "all=error") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "config", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "terminal", "no") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "volume-max", "130") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "volume", "100") < 0) ||
      (mpv_set_option_string (Mpv_Handle, "replaygain", "album") < 0) ||
      (mpv_initialize(Mpv_Handle)) )
  {
    Mpv_Handle = NULL ;
    Goodbye ("Failed initialising mpv\n") ;
  }

  // mpv is not playing anything yet. We just started it. Its state is stopped
  MPV_State = MPV_STOPPED ;
  MPV_Kill_Timer = MPV_HOLDON_SECONDS ;

  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
    Print_Log (who, "%s launched mpv\n", from) ;
}

void
Quit_MPV (const int who, const char *from)
{
  MPV_Kill_Timer = -1 ;

  if (Mpv_Handle != NULL)
  {
    // Issue a quit command to the MPV instance. It will than abort
    mpv_command (Mpv_Handle, Quit_Cmd) ;

    // This log message will only emit if the thread has not been terminated
    if ((Threads_Active != _TERMINATED) && (Verbose & (_V_MEDIA | _V_AMP_MEDIA)))
      Print_Log (who, "%s quit mpv\n", from) ;

    MPV_State = MPV_WAITING_FOR_EXIT ;
  }

  else
    MPV_State = MPV_STOPPED ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Display_Driver_Thread
//
// A thread to coordinate output to all the LEDs
//
// Time messages arrive with proposed numbers for the Hours, Minutes and Seconds digits
// Dots messages arrive with proposed settings for the seven segment display dots.
// Alt messages arrive with a numbers to display as an alternate to the time.
//
// If the network is currently down, flash the LEDs

static const char Display_Driver_Thread_Str[] = "Display Driver Thread" ;

void *
Display_Driver_Thread (void *xarg)
{
  DD_MSG_BUF		msg ;
  int8_t		local_settings_mode, set_colon ;
  uint8_t		dots, seconds, minutes, hours, colon_seconds ;
  uint8_t		last_dots, last_seconds, last_minutes, last_hours ;
  uint8_t		set_dots, set_seconds, set_minutes, set_hours ;
  static uint8_t	doing_brightness = 0 ;
  static uint8_t	doing_numbers = 0 ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_DISPLAY)
    Print_Log (T_DISPLAY_DRIVER, "Entering %s\n", Display_Driver_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // We are the recipient of DD_Msgq. Open it now for reading before any other threads try to open it for writing.
  DD_Msgq = Open_Msg_Queue_Rx (T_DISPLAY_DRIVER, DD_Msgq_name, 5, sizeof(msg)) ;

  // Wait a short time for all of the threads to have been launched (and all message queues to have been defined)
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  if (Verbose & _V_DISPLAY)
    Print_Log (T_DISPLAY_DRIVER, "%s initialised\n", Display_Driver_Thread_Str) ;

  local_settings_mode = 0 ; // normal
  set_colon = COL_OFF ;
  last_dots = last_seconds = last_minutes = last_hours = 0xff ;
  colon_seconds = 0x00 ;

  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    if (Wait_For_Next_Message (T_DISPLAY_DRIVER, DD_Msgq, (void *)&msg, sizeof(msg), sizeof(msg), Display_Driver_Thread_Str) < (int)(sizeof(msg)))
      continue ;

    if (Verbose & _V_DISPLAY)
      Print_Log (T_DISPLAY_DRIVER, "%s received message: Type=%hu, H=%hu, M=%hu, S=%hu, dots=0x%02x\n",
		 Display_Driver_Thread_Str, msg.type, msg.hours, msg.minutes, msg.seconds, msg.dots) ;
    //------------------------------------------------------------------
    // If a display test is running, don't do the normal display update
    // The DISP_TIME messages arrive at exactly one per second
    if (Display_Test && (msg.type == DISP_TIME))
    {
      if (Verbose & _V_DISPLAY)
	Print_Log (T_DISPLAY_DRIVER, "%s Display Test step=%d\n", Display_Driver_Thread_Str, Display_Test) ;

      // To simplify the commands below, blank everything and then selectively turn on the segments we want
      Write_Byte (0xff, ADDR_HOURS) ;
      Write_Byte (0xff, ADDR_MINUTES) ;
      Write_Byte (0xff, ADDR_SECONDS) ;
      Write_Byte (0x00, ADDR_LATCH) ; // Turn off all individual LEDs + decimal points

      if ((Display_Test >= 1) && (Display_Test <= 10))
	Write_Byte ( (((uint8_t)Display_Test - 1) << 4) | 0x0f, ADDR_HOURS) ;

      else if ((Display_Test >= 11) && (Display_Test <= 20))
	Write_Byte ( ((uint8_t)Display_Test - 11) | 0xf0, ADDR_HOURS) ;

      else if ((Display_Test >= 21) && (Display_Test <= 30))
	Write_Byte ( (((uint8_t)Display_Test - 21) << 4) | 0x0f, ADDR_MINUTES) ;

      else if ((Display_Test >= 31) && (Display_Test <= 40))
	Write_Byte ( ((uint8_t)Display_Test - 31) | 0xf0, ADDR_MINUTES) ;

      else if ((Display_Test >= 41) && (Display_Test <= 50))
	Write_Byte ( (((uint8_t)Display_Test - 41) << 4) | 0x0f, ADDR_SECONDS) ;

      else if ((Display_Test >= 51) && (Display_Test <= 60))
	Write_Byte ( ((uint8_t)Display_Test - 51) | 0xf0, ADDR_SECONDS) ;

      else if (Display_Test >= 61)
	Write_Byte (0x01 << (Display_Test - 61), ADDR_LATCH) ;

      Display_Test++ ;
    }
    //------------------------------------------------------------------
    // The DISP_TIME messages arrive at exactly one per second
    else if (Brightness_Test && (msg.type == DISP_TIME))
    {
      if (Verbose & _V_DISPLAY)
	Print_Log (T_DISPLAY_DRIVER, "%s Brightness Test\n", Display_Driver_Thread_Str) ;

      // To simplify the commands below, blank everything and then selectively turn on the segments we want
      Write_Byte (0x88, ADDR_HOURS) ;
      Write_Byte (0x88, ADDR_MINUTES) ;
      Write_Byte (0x88, ADDR_SECONDS) ;
      Write_Byte (0xff, ADDR_LATCH) ; // Turn on all individual LEDs + decimal points
      doing_brightness = 1 ;
    }
    //------------------------------------------------------------------
    set_dots = set_seconds = set_minutes = set_hours = dots = seconds = minutes = hours = 0 ;
    //------------------------------------------------------------------
    // Process the message according to its type
    // TIME messages are displayed if either of the display-lock modes are not active
    // The 'seconds' value of the time message is remembered for later, as this affects the colon during locked messages
    if (msg.type == DISP_TIME)
    {
      // Time messages display the time UNLESS in locked mode
      if (local_settings_mode == 0)
      {
	// The state of the colon will depend on the the hours value and the time format
	if (Time_Format_1224 == 24)
	  set_colon = COL_ON ;

	else if (msg.hours >= 12)
	{
	  set_colon = COL_PM ;

	  if (msg.hours > 12)
	    msg.hours -= 12 ;
	}

	else // msg.hours < 12
	{
	  set_colon = COL_AM ;

	  if (msg.hours == 0)
	    msg.hours = 12 ;
	}

	set_seconds = set_minutes = set_hours = 1 ;
	hours = Num_To_BCD (msg.hours, Leading_Zero_Blanking) ;
	minutes = Num_To_BCD (msg.minutes, 0) ;
	seconds = Num_To_BCD (msg.seconds, 0) ;
      }

      // The value of bit 0 in seconds affects the configuration of the colon
      colon_seconds = msg.seconds ;
      goto calculate_colon ;
    }
    //------------------------------------------------------------------
    // Process the message according to its type
    // TIME messages are displayed if either of the display-lock modes are not active
    // The 'seconds' value of the time message is remembered for later, as this affects the colon during locked messages
    else if (msg.type == DISP_NUMBERS)
    {
      set_colon = COL_ON ;
      set_seconds = set_minutes = set_hours = 1 ;
      hours = Num_To_BCD (msg.hours, Leading_Zero_Blanking) ;
      minutes = Num_To_BCD (msg.minutes, 0) ;
      seconds = Num_To_BCD (msg.seconds, 0) ;
      doing_numbers = 1 ; // lock display for one tick to stop it flickering on the second
      goto calculate_colon ;
    }
    //------------------------------------------------------------------
    else if (msg.type == DISP_RAW_NUMBERS)
    {
      set_seconds = set_minutes = set_hours = set_dots = 1 ;
      hours = msg.hours ;
      minutes = msg.minutes ;
      seconds = msg.seconds ;
      dots = msg.dots ;
      doing_numbers = 1 ; // lock display for one tick to stop it flickering on the second
    }
    //------------------------------------------------------------------
    // DOTS messages affect the decimal points associated with the digits.
    // The colon is not affected by DOTS messages
    else if (msg.type == DISP_DOTS)
    {
      // Dots messages set the state of the decimal points on the seven segment displays
      // They are independent of locked mode or not
      set_dots = 1 ;
      dots = msg.dots & 0x3f ;
      goto calculate_colon ;
    }
    //------------------------------------------------------------------
    // LOCKED_COLON, LOCKED_NO_COLON enter the first locked mode. The two options are to either display or not dispplay the colon
    // UNLOCKED_COLON is used to exit the first locked mode
    else if ((msg.type == DISP_LOCKED_COLON) || (msg.type == DISP_LOCKED_NO_COLON) || (msg.type == DISP_UNLOCKED_COLON))
    {
      local_settings_mode = (msg.type != DISP_UNLOCKED_COLON) ; // sets to either 0 or 1. Unlocked or locked

      if (local_settings_mode)
      {
	// The state of the colon will depend on the the hours value and the time format
	if (msg.type == DISP_LOCKED_NO_COLON)
	  set_colon = COL_OFF ;

	else if (Time_Format_1224 == 24)
	  set_colon = COL_ON ;

	else if ((msg.hours >= 100) && (msg.minutes >= 100))
	  set_colon = COL_OFF ;

	else if (msg.hours >= 12)
	{
	  set_colon = COL_PM ;

	  if (msg.hours > 12)
	    msg.hours -= 12 ;
	}

	else // msg.hours < 12
	{
	  set_colon = COL_AM ;

	  if (msg.hours == 0)
	    msg.hours = 12 ;
	}

	set_seconds = set_minutes = set_hours = 1 ;

	// The conditional statements in the Write_Byte statements below are to blank leading zeros when the number is NOT at time
	if (msg.hours < 100)
	  hours = Num_To_BCD (msg.hours, Leading_Zero_Blanking || ((msg.hours < 10) && (set_colon == COL_OFF))) ;
	else
	  hours = 0xff ;

	if (msg.minutes < 100)
	  minutes = Num_To_BCD (msg.minutes, ((msg.hours == 0) || (msg.hours >= 100)) && (msg.minutes < 10) && (set_colon == COL_OFF)) ;
	else
	  minutes = 0xff ;

	if (msg.seconds < 100)
	  seconds = Num_To_BCD (msg.seconds, 1) ;
	else
	  seconds = 0xff ;
      }

    calculate_colon:
      // At this point, we need to work out what to do with the colon, and then merge in the colon bits
      // The colon is set according to the colon mode
      if (Display_IP == 0)
      {
	set_dots = 1 ;

	if (set_colon == COL_ON)
	  dots |= TOP_COLON | BOT_COLON ;

	else if (set_colon == COL_PM)
	{
	  if (colon_seconds & 0x01)
	    dots |= TOP_COLON ;
	  else
	    dots |= TOP_COLON | BOT_COLON ;
	}

	else if (set_colon == COL_AM)
	{
	  if (colon_seconds & 0x01)
	    dots |= BOT_COLON ;
	  else
	    dots |= TOP_COLON | BOT_COLON ;
	}
      }
    }
    //------------------------------------------------------------------
    // Has the test now completed??? Set the variable to 0 when the test is done.
    if ((Display_Test > 69) || (doing_brightness && (Brightness_Test == 0)))
    {
      Display_Test = 0 ;
      doing_brightness = 0 ;
      last_hours = last_minutes = last_seconds = last_dots = 0xff ; // to force update of all digits
      set_dots = set_seconds = set_minutes = set_hours = 1 ;
    }

    if ((msg.type == DISP_TIME) && doing_numbers)
      doing_numbers = 0 ; // withhold updating the time display ONCE after a doing numbers message - to prevent flicker

    // Is the network down? Flash LEDs once every 12 seconds (5 times a minute) if that's the case
    else if ((msg.type == DISP_TIME) && (Display_Test == 0) && (Display_IP == 0) && (Network_Failure_Downcount > 0) && ((msg.seconds % 12) == 0))
    {
      // Write non BCD digits to the numbers to turn them off
      Write_Byte (last_hours = 0xff, ADDR_HOURS) ;
      Write_Byte (last_minutes = 0xff, ADDR_MINUTES) ;
      Write_Byte (last_seconds = 0xff, ADDR_SECONDS) ;

      // Write zero to the LEDs to turn them off
      Write_Byte (last_dots = 0x00, ADDR_LATCH) ;
    }

    // Finally, we can write the outputs that have been changed
    else if ((Display_Test == 0) && (Brightness_Test == 0))
    {
      if (set_hours && (last_hours != hours))
	Write_Byte (last_hours = hours, ADDR_HOURS) ;

      if (set_minutes && (last_minutes != minutes))
	Write_Byte (last_minutes = minutes, ADDR_MINUTES) ;

      if (set_seconds && (last_seconds != seconds))
	Write_Byte (last_seconds = seconds, ADDR_SECONDS) ;

      if (set_dots && (last_dots != dots))
	Write_Byte (last_dots = dots, ADDR_LATCH) ;
    }
  }

  Print_Log (T_DISPLAY_DRIVER, "%s unscheduled exit\n", Display_Driver_Thread_Str) ;

  // Tidy up before exiting
  mq_close (DD_Msgq) ;
  pthread_exit (NULL) ;
} // Display_Driver_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Clock_Thread
//
// A thread to keep track of the time and send time update messages to the LED displays

static const char Clock_Thread_Str[] = "Clock Thread" ;

void *
Clock_Thread (void *xarg)
{
  mqd_t			disp_driver_msgq ;
  TS			ts ;
  TM			tm ;
  DD_MSG_BUF		dd_msg ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_CLOCK)
    Print_Log (T_CLOCK, "Entering %s\n", Clock_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // The Display Driver Thread will already have started. We can open its message queue for writing
  disp_driver_msgq = Open_Msg_Queue_Tx (T_CLOCK, DD_Msgq_name, 0) ;
  Sleep_ns (THREAD_START_DELAY) ; // wait again before transmitting anything!

  if (Verbose & _V_CLOCK)
    Print_Log (T_CLOCK, "%s initialised\n", Clock_Thread_Str) ;

  // The clock / LED loop is here. This loop will circulate forever updating the LED display once per second,
  // as closely as possible to the transition from one second to the next.
  //
  // The accuracy of the clock will be determined entirely by the Linux operating system. The Linux system
  // clock is managed externally to this programme.
  // in system clock, and to periodically update time from a network time source.
  // The Linux operating system handles time zones and daylight savings transitions for us automatically.
  // Linux will handle leap seconds for us automatically.
  // I recommend an external daemon such as chrony to check for clock drift (because the clock on this
  // raspberry pi will not be stable as temperature changes etc) and to correct the linux time for clock drift.

  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // If the debug message gets printed, then this thread will be delayed a little (ie the time will be ever
    // so slightly inaccurate because of the printf delay). So we fetch time especially in this
    // debug conditional, even though we will immediately collect the same information just below here.
    if (Verbose & _V_CLOCK)
      Print_Log (T_CLOCK, "%s received tick\n", Clock_Thread_Str) ;

    // Read the time (in seconds and nanoseconds) since EPOCH time - this is a highly precise system call
    if (clock_gettime (CLOCK_REALTIME, &ts) < 0)
      Goodbye ("clock_gettime failed: %s\n", strerror(errno)) ;

    // Convert time into a human readable structure (Year/Month/Day HH:MM:SS) in the local timezone
    if (localtime_r (&ts.tv_sec, &tm) == NULL)
      Goodbye ("localtime_r failed\n") ;

    // Update the LEDs with the new time
    dd_msg.type = DISP_TIME ;
    dd_msg.hours = tm.tm_hour ;
    dd_msg.minutes = tm.tm_min ;
    dd_msg.seconds = tm.tm_sec ;
    dd_msg.dots = 0x00 ; // this is not used

    if (mq_send (disp_driver_msgq, (char *)&dd_msg, sizeof(dd_msg), 0) < 0)
      Goodbye ("Failed to send LED display msg: %s\n", strerror(errno)) ;

    // Once every five seconds, set global flag indicating that the dot flash can start.
    // This will synchronise the updating of the decimal points with seconds ticking over.
    if ((tm.tm_sec % 5) == 0)
      Dot_Flash_Counter = 1 ;

    // Delay enough to reach the next second transition
    // The actual delay could be just a tad longer due to Linux jitter + time to process message sending above
    // but the delay will cause this thread to be rescheduled very soon after the next real-world second ticksover
    Sleep_ns (NS_PER_SECOND - ts.tv_nsec) ;
  }

  Print_Log (T_CLOCK, "%s unscheduled exit\n", Clock_Thread_Str) ;

  // Tidy up before exiting
  mq_close (disp_driver_msgq) ;
  pthread_exit (NULL) ;
} // Clock_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// LED_Dimmer_Thread
//
// A thread to coordinate setting the LED Dimmer level (setting PWM level)

static const char LED_Dimmer_Thread_Str[] = "LED Dimmer Thread" ;

void *
LED_Dimmer_Thread (void *xarg)
{
  DIM_MSG_BUF		msg ;
  TV			Time ;
  uint16_t		dim_state ;
  uint8_t		alternate_pwm_freq = 0 ;
  int			error ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_DIMMER)
    Print_Log (T_LED_DIMMER, "Entering %s\n", LED_Dimmer_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // We are the recipient of Dim_Msgq. Open it now for reading before any other threads try to open it for writing.
  Dim_Msgq = Open_Msg_Queue_Rx (T_LED_DIMMER, Dim_Msgq_name, 5, sizeof(msg)) ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  if (Verbose & _V_DIMMER)
    Print_Log (T_LED_DIMMER, "%s initialised\n", LED_Dimmer_Thread_Str) ;

  dim_state = DIMMER_NORMAL ;

  // Messages to the dimmer thread may either be timeouts, or dim setting requests
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    if (Wait_For_Next_Message (T_LED_DIMMER, Dim_Msgq, (void *)&msg, sizeof(msg), sizeof(msg), LED_Dimmer_Thread_Str) < (int)(sizeof(msg)))
      continue ;

    if (Verbose & _V_DIMMER)
      Print_Log (T_LED_DIMMER, "%s received message: type=0x%02x, Duration=%hu, PWM_ratio=%.3f, alternate PWM freq=%hu\n",
		 LED_Dimmer_Thread_Str, msg.type, msg.seconds, msg.pwm_ratio, alternate_pwm_freq) ;

    // Pick apart the message to work out the dimmer setting
    switch (msg.type)
    {
      case DIM_UPDATE: // message telling us to set dimmer to whatever has been calculated
	// When a media stream is being downloaded off the Internet and decoded, the CPUs are heavily loaded, and
	// the software PWM does not work accurately at low PWM ratios and a higher PWM frequency. Therefore, when
	// that happens, we will change the PWM frequency so the low ratios work better. This may cause some flicker.
	if ((alternate_pwm_freq == 0) && (msg.pwm_ratio < LOW_PWM_RATIO) && (Mpv_Handle != NULL) && (MPV_State == MPV_PLAYING))
	{
	  // apply the alternate pwm frequency
	  alternate_pwm_freq = 1 ;

	  if ((error = gpioSetPWMfrequency (DIM_PWM, (strcasestr (Model_String, "Pi 4") != NULL) ? LOW_PI4_PWM_FREQ : LOW_PI3_PWM_FREQ)) < 0)
	    Goodbye ("Error changing SW PWM frequency: %d\n", error) ;
	}
	else if ((alternate_pwm_freq == 1) && ((msg.pwm_ratio > (LOW_PWM_RATIO+0.05)) || (Mpv_Handle == NULL) || (MPV_State != MPV_PLAYING)))
	{
	  // remove the alternate pwm frequency - there's a bit of hysteresis in the test above to stop it flapping
	  alternate_pwm_freq = 0 ;

	  if ((error = gpioSetPWMfrequency (DIM_PWM, PWM_FREQ)) < 0)
	    Goodbye ("Error changing SW PWM frequency: %d\n", error) ;
	}

	if (dim_state == DIMMER_NORMAL) // If not DIMMER_NORMAL, its DIMMER_FORCED
	  Set_PWM (msg.pwm_ratio) ;

	break ;

      case DIM_SET: // message telling us to set dimmer to a specific value for a specific time
	Set_PWM (msg.pwm_ratio) ;
      send_dimmer_message:
	dim_state = DIMMER_FORCED ;
	gettimeofday (&Time, NULL) ;
	Add_Time_ms (&Time, msg.seconds * 1000) ; // We are going to set a timeout to expire AFTER this time
	memmove (&Dimmer_Timeout, &Time, sizeof(Dimmer_Timeout)) ; // The timer is now running and we will expect a message
	break ;

      case DIM_BRIGHTNESS_TEST: // message telling us to run the brightness test
	Set_PWM (Min_PWM_Ratio) ;
	Brightness_Test = msg.seconds ; // Remember the time that has been requested
	goto send_dimmer_message ;

      case DIM_TIMEOUT_TO_BRIGHT: // message us telling us to jump to the bright phase of the brightness test
	Set_PWM (Max_PWM_Ratio) ;
	msg.seconds = Brightness_Test ; // Recall the time that has been requested
	Brightness_Test = 0xffff ; // signalling that the test is in the bright phase
	goto send_dimmer_message ;
	break ;

      case DIM_TIMEOUT_TO_NORMAL: // message telling us to set dimmer to the normal calculated value
	dim_state = DIMMER_NORMAL ;
	Brightness_Test = 0 ;	// Stop the brightness test (if it was running)
	Set_PWM ((Min_PWM_Ratio + Max_PWM_Ratio) / 2) ; // Temporarily in the middle. It will reset to the ambient when that gets recalculated
	break ;

      case DIM_STOP_CLOCK:	// Message telling us to kill the clock daemon and shut down
	Stop_Clock = 1 ;
	Threads_Active = _SHUTTING_DOWN ;
	break ;

      default: // this should not happen - there's a bug
	Goodbye ("Received unexpected dimmer message: %d\n", msg.type) ;
    }
  }

  Print_Log (T_LED_DIMMER, "%s unscheduled exit\n", LED_Dimmer_Thread_Str) ;

  // Tidy up before exiting
  mq_close (Dim_Msgq) ;
  pthread_exit (NULL) ;
} // LED_Dimmer_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// periodic_timer_callback + Timer_Thread
//
// This is like a timer interrupt handler. We will configure it so it is called once every 25ms
// as a continuous timer 'alarm' stream.

static const char Timer_Thread_Str[] = "Timer Thread" ;

void
Send_LED_Dot_Update_Message (const mqd_t Q, const uint8_t dots)
{
  DD_MSG_BUF		dd_msg ;

  // Update the LED display
  dd_msg.type = DISP_DOTS ;
  dd_msg.hours = dd_msg.minutes = dd_msg.seconds = 0xff ; // These are not used
  dd_msg.dots = dots ; // this is the bitmask of to use for the seven segment display dots

  if (mq_send (Q, (char *)&dd_msg, sizeof(dd_msg), 0) < 0)
    Goodbye ("Failed to send LED display msg: %s\n", strerror(errno)) ;
}

void
periodic_timer_callback (int signum)
{
  UNUSED (signum) ; // silence the compiler warning

  // This is a dummy signal handler. The work will be done in the thread that we created
  // that exclusively handles SIGALRM

  // Like a real interrupt service routine, you don't want to do much work here, just
  // emit signals or messages to the threads that will do the real work.
  // In fact, we will do absolutely NO WORK at all! At the same time as this callback is
  // triggered, the Timer_Thread will be woken up

} // periodic_timer_callback

void *
Timer_Thread (void *arg)
{
  sigset_t		signal_set;
  int			sig, n, setup_timeout, last_DST ;
  uint8_t		t_amp, t_mpv, t_net, t_button, t_alarm, t_media, t_ldr, t_peer, t_blue, dots ; // Note: these are 8 bit counters and can overflow if long timeouts used!
  uint8_t		clustered_clock_visible, start_delay ;
  mqd_t			dim_msgq ;
  mqd_t			media_msgq ;
  mqd_t			amp_msgq ;
  mqd_t			button_msgq ;
  mqd_t			disp_driver_msgq ;
  DIM_MSG_BUF		dim_msg ;
  MEDIA_MSG_BUF		media_msg ;
  AMP_MSG_BUF		amp_msg ;
  int			button_msg ;
  TV			Time ;
  TM			tm ;
  char			time_string [TIME_STRING_LEN] ;

  UNUSED (arg) ; // silence the compiler warning

  if (Verbose & (_V_TIMER | _V_ALARM))
    Print_Log (T_TIMER, "Entering %s\n", Timer_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // SIGALRM is the linux signal we will be using for the periodic timer.
  // We disabled SIGALRM for all threads by disabling it before initialising the threads. So it is currently
  // disabled here. We now need to re-enable SIGALRM in this thread, and this will be the only thread where
  // it is enabled.
  sigemptyset (&signal_set);
  sigaddset (&signal_set, SIGALRM);

  // Wait for all of the threads to have been launched
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Now we can open the dimmer and media message queues for writing
  dim_msgq = Open_Msg_Queue_Tx (T_TIMER, Dim_Msgq_name, 0) ;
  media_msgq = Open_Msg_Queue_Tx (T_TIMER, Media_Msgq_name, 0) ;
  amp_msgq = Open_Msg_Queue_Tx (T_TIMER, Amp_Msgq_name, 0) ;
  button_msgq = Open_Msg_Queue_Tx (T_TIMER, Button_Msgq_name, 0) ;
  disp_driver_msgq = Open_Msg_Queue_Tx (T_TIMER, DD_Msgq_name, 0) ;

  Sleep_ns (THREAD_START_DELAY) ; // wait again before transmitting anything!

  if (Verbose & (_V_TIMER | _V_ALARM))
    Print_Log (T_TIMER, "%s initialised\n", Timer_Thread_Str) ;

  t_amp = t_mpv = t_net = t_button = t_alarm = t_media = t_ldr = t_peer = t_blue = start_delay = 0 ;
  Moving_Average_Light_Level = 2048 ;
  Instantaneous_Light_Level = 0x7ff ; // 12 bits maximum value
  setup_timeout = 0 ;
  Audible_Clustered_Clock_Visible = -1 ;
  clustered_clock_visible = 0 ;

  // Initialise Current_Seconds_Count to NOW
  gettimeofday (&Time, NULL) ;
  Current_Seconds_Count = Time.tv_sec ;

  // Determine if daylight savings is in effect right now, and remember this in last_DST
  if (localtime_r (&Time.tv_sec, &tm) == NULL)
    Goodbye ("localtime_r failed\n") ;

  last_DST = tm.tm_isdst ;

  // The timer thread works as an infinte loop
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // sigwait returns the signal number. On error returns -1
    if (sigwait (&signal_set, &sig) < 0)
      Goodbye ("%s sigwait failed: %s\n", Timer_Thread_Str, strerror(errno)) ;

    if (sig != SIGALRM)
      Goodbye ("%s sigwait returned unexpected signal: %d\n", Timer_Thread_Str, sig) ;

    // Got SIGALRM - increment the various processing timer/counters. These counts are checked
    // below and when exceeded, the respective processes is called and the counter reset.
    t_amp++ ;
    t_mpv++ ;
    t_net++ ;
    t_button++ ;
    t_alarm++ ;
    t_media++ ;
    t_ldr++ ;
    t_peer++ ;
    t_blue++ ;
    // - - - - - - - - - - - - - - - -
    if (Seconds_To_Next_Alarm < 0)
    {
      // Incremented if no alarms set. Ignored otherwise. When the timeout expires, alarm file reloaded.
      // The purpose of this is to catch alarm changes introduced outside the programme and to
      // periodically reload the alarm config file to check if there are newly defined alarms
      setup_timeout++ ;

      if (setup_timeout >= SETUP_FILE_RELOAD_DELAY)
      {
	setup_timeout = 0 ;

	if (Verbose & _V_TIMER)
	  Print_Log (T_TIMER, "%s setup timeout expired\n", Timer_Thread_Str) ;

	// Read and interpret the setup and alarm file... (or reread the files - they may have changed)
	Read_Setup_And_Alarms (T_TIMER, 0) ;
      }
    }
    // - - - - - - - - - - - - - - - -
    // Read the time of day - will be used to determine when seconds tick over
    gettimeofday (&Time, NULL) ;

    // Sometimes the time jumps.... this can happen
    // (a) when the PI is first powered on and before the NTP daemon (crony) has synced time, the PI will be set to the
    //     time it last shut-down. When chrony syncs, there will be a leap forwards
    //
    // (b) there's a leap-second
    //
    // Note: for daylight savings changes, the epoch time doesn't jump. The daylight savings jump is calculated
    // relative to universal time, which epoch time already.

    // It is not clear to me whether the following code is required or not. It checks for the beginning
    // or end of daylight savings and if so, forces recalculation of the alarm tripping times.
    // I am unsure its required because all the alarms are calculated in Epoch time, but the calculations
    // depend on the operating system correctly determining the right epoch on either side of a daylight
    // savings switch. I cannot really wait around for half a year to find out, so I will just paranoid-add
    // the test below that determines when there's a change and forces an alarm recalc
    if (localtime_r (&Time.tv_sec, &tm) == NULL)
      Goodbye ("localtime_r failed\n") ;

    // In the test below, we are looking for a jump that exceeds 10 seconds in either direction (In general it will be a leap forwards)
    // We are also looking for a jump due to a change in daylight savings time
    if ( (last_DST != tm.tm_isdst) ||
	 (Time.tv_sec <= (Current_Seconds_Count - 10)) ||
	 (Time.tv_sec >= (Current_Seconds_Count + 10)) )
    {
      Print_Log (T_TIMER, "Step time change: %+d seconds\n", Time.tv_sec - Current_Seconds_Count) ;

      // Make the step adjustment
      Current_Seconds_Count = Time.tv_sec ;

      // Recalculate the time until the next alarm - the current down counter is incorrect
      pthread_mutex_lock (&Alarms_Mutex) ;
      Time_To_Next_Alarm (T_TIMER, 1) ;
      pthread_mutex_unlock (&Alarms_Mutex) ;
    }

    last_DST = tm.tm_isdst ; // Remember the daylight savings state for next time
    // - - - - - - - - - - - - - - - -
    if (t_ldr >= LDR_SCAN_FREQ) // The default LDR scan frequency is 4 times per second
    {
      t_ldr = 0 ;

      // Now is when we measure the ambient light level, and calculate the dimmer setting
      // The returned value (Instantaneous_Light_Level is arranged to be 0..FFF with 0 meaning dark, FFF meaning bright light)
      Instantaneous_Light_Level = Read_A_to_D () ;

      // We are running an exponential moving average calculation below.
      Moving_Average_Light_Level = ((1.0-MOV_AV_FRACT) * Instantaneous_Light_Level) + (MOV_AV_FRACT * Moving_Average_Light_Level) ;

      // Where does the ambient light measurement sit between the thresholds configured in setup??
      // Note: A smaller Instantaneous_Light_Level reading means less light!!! (it increases as it gets brighter)
      // When the ambient light level is higher, we want LEDs to be brighter
      if (Moving_Average_Light_Level <= Min_LDR_Threshold)
	dim_msg.pwm_ratio = Min_PWM_Ratio ;

      else if (Moving_Average_Light_Level >= Max_LDR_Threshold)
	dim_msg.pwm_ratio = Max_PWM_Ratio ;

      else
	dim_msg.pwm_ratio = Min_PWM_Ratio +
	  ((Max_PWM_Ratio - Min_PWM_Ratio) * (Moving_Average_Light_Level - Min_LDR_Threshold) / (double)(Max_LDR_Threshold - Min_LDR_Threshold)) ;

      dim_msg.type = DIM_UPDATE ;
      dim_msg.seconds = 0 ;

      if (Verbose & (_V_DIMMER | _V_TIMER))
	Print_Log (T_TIMER, "%s: A to D instantaneous=%" PRIu16 ", Moving Average=%.1f, PWM ratio=%.3f, time to next alarm=%ds\n",
		   Timer_Thread_Str, Instantaneous_Light_Level, Moving_Average_Light_Level, dim_msg.pwm_ratio, Seconds_To_Next_Alarm) ;

      if (mq_send (dim_msgq, (char *)&dim_msg, sizeof(dim_msg), 0) < 0)
	Goodbye ("%s failed to send PWM ratio msg: %s\n", Timer_Thread_Str, strerror(errno)) ;

      // If the dimmer is currently set to a temporary value, check whether that time has now passed
      if (Dimmer_Timeout.tv_sec != 0)
      {
	// Get the current system time
	if (Time_Difference_ms (&Dimmer_Timeout, &Time) >= 0)
	{
	  Dimmer_Timeout.tv_sec = 0 ; // stop the Dimmer Timeout from tripping again

	  if ((Brightness_Test == 0xffff) || (Brightness_Test == 0))
	    dim_msg.type = DIM_TIMEOUT_TO_NORMAL ;
	  else
	    dim_msg.type = DIM_TIMEOUT_TO_BRIGHT ;

	  dim_msg.seconds = 0 ;
	  dim_msg.pwm_ratio = 0 ;

	  // Signal to Dimmer_Thread that it should reset the PWM value to normal
	  if (mq_send (dim_msgq, (char *)&dim_msg, sizeof(dim_msg), 0) < 0)
	    Goodbye ("%s failed to send Dimmer Timeout msg: %s\n", Timer_Thread_Str, strerror(errno)) ;

	  if (Verbose & _V_TIMER)
	    Print_Log (T_TIMER, "%s Dimmer timeout expired\n", Timer_Thread_Str) ;
	}
      }
    }
    // - - - - - - - - - - - - - - - -
    // This section checks whether a new alarm has tripped
    if (t_alarm >= ALARM_SCAN_FREQ)
    {
      t_alarm = 0 ;

      // Check whether the vol offset needs to be adjusted - do this before checking for new alarm trip
      if (Time.tv_sec != Current_Seconds_Count)
      {
	// Once per minute, check whether to step the volume towards the target
	if ((Time.tv_sec % 60) == 0)
	{
	  if (Vol_Offset < Targ_Vol_Offset)
	    Vol_Offset++ ;
	  else if (Vol_Offset > Targ_Vol_Offset)
	    Vol_Offset-- ;
	}

	// Ensure only one thread can access the alarm variables at any one time
	// But also see that there are a number of different calls to pthread_mutex_unlock below.
	// Rather than wait until the end of the big if statement and unlocking in one place, I've
	// chosen to unlock as soon as the mutex is not required- this is because there could be
	// a significant delay because of the call to mq_send and we don't want to jam things.
	pthread_mutex_lock (&Alarms_Mutex) ;

	// Check whether an alarm has gone off if the seconds have advanced
	if (Seconds_To_Next_Alarm >= 0)
	{
	  Seconds_To_Next_Alarm -= Time.tv_sec - Current_Seconds_Count ;

	  if (Seconds_To_Next_Alarm <= 0)
	  {
	    // The alarm has just tripped
	    if (Verbose & (_V_TIMER | _V_ALARM))
	      Print_Log (T_TIMER, "%s Alarm tripped at %s\n", Timer_Thread_Str, Display_Time (time_string, &Time)) ;

	    // If the clock is a silent-clustered type, we now need to check if another clustered clock
	    // is accessible. If it is, we assume that the main clock is playing the alarm unless a main is not visible
	    if ((This_Clock_Type != 'Z') || (Audible_Clustered_Clock_Visible <= 0))
	    {
	      // Log a warning if we are forcing the alarm to play because there is no visible clustered clock
	      if ((This_Clock_Type == 'Z'))
		Print_Log (T_TIMER, "%s: Playing alarm despite this being a silent clock. Cannot see another visible clustered clock\n", Timer_Thread_Str) ;

	      // Are we chaining to the same streaming source??? If so, just adjust the volume
	      if (strcmp (Alarms[Next_Alarm_Index].url_path, Current_Alarm_Stream) == 0)
	      {
		// The next alarm is chaining onto the SAME current alarm. We only need to adjust volume and reset the alarm timer
		Seconds_Of_Play_Remaining = (Alarms[Next_Alarm_Index].duration * 60) + 1 ; // The +1 is to ensure smooth onward chaining
		Vol_Offset = Alarms[Next_Alarm_Index].init_vol_offset ;
		Targ_Vol_Offset = Alarms[Next_Alarm_Index].targ_vol_offset ;

		if (Verbose & (_V_TIMER | _V_ALARM))
		  Print_Log (T_TIMER, "%s Chaining next alarm: Seconds_Of_Play_Remaining=%ds, Vol_Offset=%hd\n", Timer_Thread_Str, Seconds_Of_Play_Remaining, Vol_Offset) ;

		// Ensure that Volume, Min_Volume, Vol_Offset and Targ_Vol_Offset are pegged within a sensible range
		Enforce_Volume_Limits (1) ;
	      }
	      else
	      {
		// Start the file or url stream by sending a message to the media player
		media_msg.seconds = (Alarms[Next_Alarm_Index].duration * 60) + 1 ; // The +1 is to ensure smooth onward chaining
		media_msg.index = -1 ;
		media_msg.volume = -1 ; // Don't change the absolute volume
		media_msg.init_vol_offset = Alarms[Next_Alarm_Index].init_vol_offset ; // apply this offset to absolute volume
		media_msg.targ_vol_offset = Alarms[Next_Alarm_Index].targ_vol_offset ; // apply this offset to absolute volume
		media_msg.responseQ = 0 ; // a dummy number that will cause alarm clock to crash if there's a bug. #0 will appear in debug log
		media_msg.type = MEDIA_ALARM_START ;

		// Is this alarm a playlist? If so, does it have a last-played record?
		if ((n = Is_Playlist(Alarms[Next_Alarm_Index].url_path, NULL, NULL)) > 0)
		  media_msg.index = n ;

		n = MEDIA_MSG_LEN + 1 + snprintf (media_msg.url_path, SETUP_FIELD_LEN, "%s", Alarms[Next_Alarm_Index].url_path) ; // copy the command
		strcpy (Current_Alarm_Stream, Alarms[Next_Alarm_Index].url_path) ; // Take note of the stream that is playing
		pthread_mutex_unlock (&Alarms_Mutex) ;

		// Send the media request to the media thread
		if (mq_send (media_msgq, (char *)&media_msg, n, 0) < 0)
		  Goodbye ("%s failed to send media msg: %s\n", Timer_Thread_Str, strerror(errno)) ;

		pthread_mutex_lock (&Alarms_Mutex) ;
	      }

	      // The previous alarm is history. We now need to recalculate when the next alarm will happen.
	      Time_To_Next_Alarm (T_TIMER, 0) ;
	    }

	    pthread_mutex_unlock (&Alarms_Mutex) ;

	    if (Verbose & (_V_TIMER | _V_ALARM))
	    {
	      Print_Log (T_TIMER, "%s After alarm tripped: Seconds_Of_Play_Remaining=%ds, Vol_Offset=%hd, Seconds_To_Next_Alarm=%d\n",
			 Timer_Thread_Str, Seconds_Of_Play_Remaining, Vol_Offset, Seconds_To_Next_Alarm) ;

	      if (Seconds_To_Next_Alarm > 0)
		Print_Log (T_TIMER, "%s: next stream='%s'\n", Timer_Thread_Str, Alarms[Next_Alarm_Index].url_path) ;
	    }
	  }
	  else
	    pthread_mutex_unlock (&Alarms_Mutex) ;
	}
	else
	  pthread_mutex_unlock (&Alarms_Mutex) ;

	// Remember the last epoch that was compared
	Current_Seconds_Count = Time.tv_sec ;
      }
    }
    // - - - - - - - - - - - - - - - -
    // We want to send MEDIA_TICK messages if either the radio or MPV are enabled.
    if ((t_media >= (uint8_t)MPV_SCAN_FREQ) &&
	((Mpv_Handle != NULL) || (Amp_State == A_AMP_ON_RADIO) || (Amp_State == A_AMP_ON_RADIO_PAUSE)))
    {
      t_media = 0 ;

      // Send the periodic media thread tick which it uses to monitor the child process
      media_msg.seconds = 0 ;
      media_msg.index = -1 ;
      media_msg.volume = -1 ; // Don't change the absolute volume
      media_msg.init_vol_offset = -200 ; // outside the range -MAX_VOL_OFFSET to +MAX_VOL_OFFSET
      media_msg.targ_vol_offset = -200 ; // outside the range -MAX_VOL_OFFSET to +MAX_VOL_OFFSET
      media_msg.responseQ = 1 ; // a dummy number that will cause alarm clock to crash if there's a bug. #1 will appear in debug log
      media_msg.type = MEDIA_TICK ;
      media_msg.url_path[0] = '\0' ;

      if (mq_send (media_msgq, (char *)&media_msg, MEDIA_MSG_LEN, 0) < 0)
	Goodbye ("%s failed to send media msg: %s\n", Timer_Thread_Str, strerror(errno)) ;
    }
    // - - - - - - - - - - - - - - - -
    if (t_amp >= AMP_SCAN_FREQ)
    {
      t_amp = 0 ;

      // Send tick messages to the amp manager if the amp is ON, or if there is at least one bluetooth client
      // The bluetooth client test is to bootstrap the amp manager... so that it turns the amp on if nothing else is on
      if ((Amp_State != A_AMP_OFF) || (Num_Bluetooth_Clients > 0))
      {
	// Send the periodic amp manager thread tick
	amp_msg.type		= A_AMP_TICK ;
	amp_msg.media_cmd	= MEDIA_NOTHING ;
	amp_msg.index		= -1 ;

	if (mq_send (amp_msgq, (char *)&amp_msg, AMP_MSG_LEN, 0) < 0)
	  Goodbye ("%s failed to send amp msg: %s\n", Timer_Thread_Str, strerror(errno)) ;
      }
    }
    // - - - - - - - - - - - - - - - -
    if (t_mpv >= MPV_CHECK_FREQ)
    {
      t_mpv = 0 ;

      if ((MPV_Kill_Timer >= 0) && (--MPV_Kill_Timer <= 0))
	Quit_MPV (T_TIMER, Timer_Thread_Str) ;
    }
    // - - - - - - - - - - - - - - - -
    if (t_net >= NET_CHECK_FREQ)
    {
      t_net = 0 ;

      if (Network_Failure_Downcount > 0)
      {
	Network_Failure_Downcount-- ;

	if (Network_Failure_Downcount == 0)
	  Goodbye ("%s Network has been down for %d hours: rebooting\n", Timer_Thread_Str, NET_FAILED_DOWNCOUNT/ONE_HOUR) ;
      }
    }
    // - - - - - - - - - - - - - - - -
    if (t_button >= BUTTON_SCAN_FREQ)
    {
      t_button = 0 ;
      button_msg = 0 ;

      if (mq_send (button_msgq, (char *)&button_msg, sizeof(button_msg), 0) < 0)
	Goodbye ("%s failed to send button msg: %s\n", Timer_Thread_Str, strerror(errno)) ;
    }
    // - - - - - - - - - - - - - - - -
    if (Dot_Flash_Counter)
    {
      if (Dot_Flash_Counter == 1)
      {
	dots = 0x00 ;

	// Flash the alarm indication LED
	if (Is_Alarm)
	{
	  // The decimal points for the seconds digits flash to indicate alarm active
	  dots |= 0x01 ;
	}

	// Seconds Tens dot indicates that a bluetooth device has connected
	if (Amp_State == A_AMP_ON_BLUETOOTH)
	  dots |= 0x02 ;

	// The decimal points for the hours and minutes digits flash to indicate pending alarms
	if ((Num_Alarms > 0) && (Seconds_To_Next_Alarm > 0))
	{
	  // Minute-Units: if an alarm is scheduled within the next 8H
	  if (Seconds_To_Next_Alarm < EIGHT_HOURS)
	    dots |= 0x04 ;

	  // Minute-Tens: if an alarm is scheduled between 8H and 16H
	  else if (Seconds_To_Next_Alarm < SIXTEEN_HOURS)
	    dots |= 0x08 ;

	  // Hours-Units: if an alarm is scheduled between 16H and 24H
	  else if (Seconds_To_Next_Alarm < ONE_DAY)
	    dots |= 0x10 ;

	  // Hours-Tens: if an alarm is scheduled between 1D and 7D
	  else if (Seconds_To_Next_Alarm < ONE_WEEK)
	    dots |= 0x20 ;
	}

	// If a dot has been set, send a message
	if (dots)
	  Send_LED_Dot_Update_Message (disp_driver_msgq, dots) ;
	else
	  Dot_Flash_Counter = 0 ; // Stop the timer until it is reset by the clock thread
      }

      if (Dot_Flash_Counter < DOT_FLASH_COUNT) // this is counter used to time lemgth of LED dot flash
	Dot_Flash_Counter++ ;

      else
      {
	// Turn all decimal points off
	Send_LED_Dot_Update_Message (disp_driver_msgq, 0x00) ;
	Dot_Flash_Counter = 0 ; // Stop the timer until it is reset by the clock thread
      }
    }
    // - - - - - - - - - - - - - - - -
    if (t_peer >= PEER_CHECK_FREQ)
    {
      t_peer = 0 ;
      clustered_clock_visible = 0 ; // Accumulate the count of visible clustered clocks in this variable

      // Scan the peers table looking for timed out peer entries
      for (n=0 ; n < NUM_PEERS ; n++)
      {
	// Time out a peer entry after it has gone silent
	if ((Peers[n].sin_addr.s_addr != 0) && ((Current_Seconds_Count - Peers[n].last_seen) > PEER_TIMEOUT))
	{
	  if (Verbose & _V_MULTICAST)
	    Print_Log (T_TIMER, "%s multicast peer timeout for '%s'\n", Timer_Thread_Str, Peers[n].name) ;

	  memset (&Peers[n], 0x00, sizeof(Peers[n])) ;
	}

	// This gives us a risk of PEER_TIMEOUT seconds in which an alarm may not trip correctly because we
	// have not yet declared a crashed clustered clock invisible
	if ((Peers[n].sin_addr.s_addr != 0) && ((Peers[n].type == 'A') || (Peers[n].type == 'C')))
	  clustered_clock_visible++ ;

	// Tick a slow counter to pace the early uptime
	if (start_delay	< 0xff)
	  start_delay++ ;
      }

      if ((Audible_Clustered_Clock_Visible != clustered_clock_visible) && (Verbose & (_V_TIMER | _V_ALARM | _V_MULTICAST)))
	Print_Log (T_TIMER, "%s Changed number of visible audible clocks from %hd to %hu\n",
		   Timer_Thread_Str, Audible_Clustered_Clock_Visible, clustered_clock_visible) ;

      // Why are we using a local variable and then setting a global variable? This is the ONLY place in the
      // code where Audible_Clustered_Clock_Visible is set. In the unlikely case where linux interrupts this thread
      // between clustered_clock_visible being set to 0 above and here, there will always be a consistent
      // value in the global variable Audible_Clustered_Clock_Visible. Normally you'd use a mutex to avoid this kind
      // of situation/problem, but setting the global variable is an atomic operation and so it is OK
      // to avoid the mutex in this case.
      Audible_Clustered_Clock_Visible = (int8_t)clustered_clock_visible ;
    }
    // - - - - - - - - - - - - - - - -
    if ((Bluetoothctl_PID > 0) && (t_blue >= BLUETOOTH_CHECK_FREQ))
    {
      t_blue = 0 ;

      // Check and decrement Check_Bluetooth_Devices
      if ((Check_Bluetooth_Devices > 1) && (--Check_Bluetooth_Devices == 1))
      {
	// Issue a dummy return press, so that Bluetooth_Rx_Thread gets tripped
	if (write (PARENT_WRITE_FD, "\n", 1) != 1)
	  Goodbye ("%s cannot write to bluetooth pipe: %s\n", Timer_Thread_Str, strerror(errno)) ;
      }

      if (Check_Bluetooth_Disconnect > 1)
	--Check_Bluetooth_Disconnect ;

      if ((start_delay == 0x04) && (Bluetooth_Init_Completed == 0) && (Verbose & _V_BLUETOOTH))
      {
	Print_Log (T_TIMER, "%s Bluetooth processes not yet completed initialisation\n", Timer_Thread_Str) ;
	start_delay++ ;
      }
    }
    // - - - - - - - - - - - - - - - -
  }

  Print_Log (T_TIMER, "%s unscheduled exit\n", Timer_Thread_Str) ;

  // Tidy up before exiting
  mq_close (dim_msgq) ;
  mq_close (media_msgq) ;
  mq_close (amp_msgq) ;
  mq_close (button_msgq) ;
  mq_close (disp_driver_msgq) ;
  pthread_exit (NULL) ;
} // Timer_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Setup_Thread
//
// A thread whose job is to load the setup, and reload setup when asked

void *
Setup_Thread (void *xarg)
{
  SETUP_MSG_BUF		setup_msg ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_SETUP)
    Print_Log (T_SETUP, "Entering %s\n", Setup_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // Read and interpret the setup and alarm files to initialise the clock
  Read_Setup_And_Alarms (T_SETUP, 1) ;

  // We are the recipient of Setup_Msgq. Open it now for reading before any other threads try to open it for writing.
  Setup_Msgq = Open_Msg_Queue_Rx (T_SETUP, Setup_Msgq_name, 5, sizeof(setup_msg)) ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  if (Verbose & _V_SETUP)
    Print_Log (T_SETUP, "%s initialised\n", Setup_Thread_Str) ;

  // Messages to the setup thread always trigger reloading of setup and alarm information.
  // The setup thread is lso used as a trigger for reading and returning the current LDR light level.
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    if (Wait_For_Next_Message (T_SETUP, Setup_Msgq, (void *)&setup_msg, sizeof(setup_msg), sizeof(setup_msg), Setup_Thread_Str) < (int)(sizeof(setup_msg)))
      continue ;

    if (Verbose & _V_SETUP)
      Print_Log (T_SETUP, "%s received message: type=%hu, vol=%hd, dur=%hd, Q=%u\n",
		 Setup_Thread_Str, setup_msg.type, setup_msg.volume, setup_msg.duration, (unsigned)setup_msg.responseQ) ;

    // If this is a S_READ_LDR_AND_STATUS message, then open the response message queue, send the value of Instantaneous_Light_Level,
    // and then close the queue again
    switch (setup_msg.type)
    {
      case S_RELOAD:
	if (Verbose)
	  Print_Log (T_SETUP, "%s reloading settings\n", Setup_Thread_Str) ;

	// Read and interpret the setup and alarm files... (or reread the files - they may have changed)
	Read_Setup_And_Alarms (T_SETUP, 1) ;
	break ;

      case S_DISPLAY_TEST:
	// Initiate the display test, then fall through to respond as if this was an LDR request
	Display_Test = 1 ; // Initiate a display test by the Display_Driver_Thread

	// fall through

      case S_READ_LDR_AND_STATUS:
	// Open the response queue in order to send the response. Each time we run this, there will be a different
	// response queue (different instance of alarm-clock that opened it), so we need to open it fresh each time.
	snprintf (Response_Msgq_name, TEMP_NAME_LEN, "/R%u", (unsigned)setup_msg.responseQ) ;

	if ((Response_Msgq = Open_Msg_Queue_Tx (T_SETUP, Response_Msgq_name, 1)) < 0)
	  Goodbye ("%s: Cannot open Response_Msg_Queue: %s\n", Setup_Thread_Str, strerror(errno)) ;

	// Now we can send the response
	// First two items are easy - the ambient light level and current volume can just be read from the global variables
	setup_msg.ambient = Instantaneous_Light_Level ;
	setup_msg.volume = Volume ;
	setup_msg.num_blue = Num_Bluetooth_Clients ;

	// The last item (playing duration) is more complicated because there are a couple of magic values we need to
	// check for.
	if (Seconds_Of_Play_Remaining >= 0) // Are we currently playing something (either media player or radio)
	  setup_msg.duration = (int16_t)Round_Nearest_Int (Seconds_Of_Play_Remaining / 60.0) ;
	else if ((Mpv_Handle != NULL) && (MPV_State != MPV_WAITING_FOR_EXIT))
	  setup_msg.duration = MAX_DURATION + 1 ; // this is a dummy value to say that nothing is playing, but MPV is still active
	else
	  setup_msg.duration = MAX_DURATION + 2 ; // this is a dummy value to say that nothing is playing, and we're off

	if (mq_send (Response_Msgq, (char *)&setup_msg, sizeof(setup_msg), 0) < 0)
	  Print_Log (T_SETUP, "%s: Couldn't send a read LDR message: %s\n", Setup_Thread_Str, strerror(errno)) ;

	else if (Verbose & _V_SETUP)
	  Print_Log (T_SETUP, "%s responded to Read_Ambient request: vol=%hd, ambient=%hu, dur=%hd, num_blue=%hu\n",
		     Setup_Thread_Str, setup_msg.volume, setup_msg.ambient, setup_msg.duration, setup_msg.num_blue) ;

	mq_close (Response_Msgq) ; // Note: this must be done by the receiver, not the transmitter!
	Response_Msgq_name[0] = '\0' ;
	Response_Msgq = 0 ;
	break ;

      case S_SET_VOL_DUR:
	if ((setup_msg.volume >= 0) && (setup_msg.volume <= MAX_VOLUME))
	{
	  Volume = setup_msg.volume ;
	  Enforce_Volume_Limits (0) ;
	}

	// Only change the duration if the amp is on (ie we are playing media)
	if ((Amp_State != A_AMP_OFF) && (Amp_State != A_AMP_ON_BLUETOOTH))
	{
	  // Set the media duration to the specified time
	  if ((setup_msg.duration > 0) && (setup_msg.duration <= MAX_DURATION))
	    Seconds_Of_Play_Remaining = setup_msg.duration * 60 ;

	  else if (setup_msg.duration == 0) // If set to zero, impose a three second countdown
	    Seconds_Of_Play_Remaining = 3 ;

	  continue ;
	}

	break ;

      case S_PAIR:
	if (Verbose & (_V_SETUP | _V_BLUETOOTH))
	  Print_Log (T_SETUP, "%s received initiate pairing message\n", Setup_Thread_Str) ;

	if ((Bluetoothctl_PID > 0) && (Bluetooth_Init_Completed > 0))
	{
	  Initiate_Bluetooth_Pairing = 1 ;

	  // Issue a dummy return press, so that Bluetooth_Rx_Thread gets tripped
	  if (write (PARENT_WRITE_FD, "\n", 1) != 1)
	    Goodbye ("%s cannot write to bluetooth pipe: %s\n", Setup_Thread_Str, strerror(errno)) ;
	}

	break ;

      case S_UNPAIR:
	if (Verbose & (_V_SETUP | _V_BLUETOOTH))
	  Print_Log (T_SETUP, "%s received unpair message\n", Setup_Thread_Str) ;

	if ((Bluetoothctl_PID > 0) && (Bluetooth_Init_Completed > 0))
	{
	  Initiate_Bluetooth_UnPairing = 1 ;

	  // Issue a dummy return press, so that Bluetooth_Rx_Thread gets tripped
	  if (write (PARENT_WRITE_FD, "\n", 1) != 1)
	    Goodbye ("%s cannot write to bluetooth pipe: %s\n", Setup_Thread_Str, strerror(errno)) ;
	}

	break ;

      default:
	Print_Log (T_SETUP, "%s: Unknown message type=%hu\n", Setup_Thread_Str, setup_msg.type) ;
	break ;
    }
  }

  Print_Log (T_SETUP, "%s unscheduled exit\n", Setup_Thread_Str) ;

  // Tidy up before exiting
  mq_close (Setup_Msgq) ;
  pthread_exit (NULL) ;
} // Setup_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Media_Player_Thread
//
// A thread whose job is to play media, and keep it playing.
// The thread also differentiates between the mpv media player and the external radio,
// choosing the right source to control.  Returns negative number if failed
//
// For additional information about using the libmpv API, try these links:
//  https://mpv.io/manual/master
//  https://github.com/mpv-player/mpv/blob/master/libmpv/client.h

static const char Media_Player_Thread_Str[]	= "Media Player Thread" ;

static const char *Stop_Cmd[]		= {"stop", NULL} ;
static const char *Next_Cmd[]		= {"playlist-next", NULL} ;
static const char *Prev_Cmd[]		= {"playlist-prev", NULL} ;
static const char *Restart_Cmd[]	= {"seek", "0", "absolute", NULL} ;
static const char *SeekFwd_Cmd[]	= {"seek", "5", NULL} ; // seek forwards 5 seconds
static const char *SeekBack_Cmd[]	= {"seek", "-5", NULL} ; // seek backwards 5 seconds
static const char *Clear_Cmd[]		= {"playlist-clear", NULL} ;
static const char *Shuffle_Cmd[]	= {"playlist-shuffle", NULL} ;

// Media_Command returns -1 on fatal error. Returns 0 on OK
int
Media_Command (const int who, const char *name, const uint8_t cmd, const int index, const char *str, const mqd_t amp_msgq, const unsigned responseQ)
{
  AMP_MSG_BUF		amp_msg ;
  MEDIA_RESPONSE_BUF	playlist_msg ;
  mpv_node		pl ;
  int			i, j, n ;
  int64_t		val ;

  switch (cmd)
  {
    case MEDIA_PAUSE:
      // Toggles the pause state
      if (((Amp_State == A_AMP_ON_RADIO) || (Amp_State == A_AMP_ON_RADIO_PAUSE)) && (amp_msgq != (mqd_t)0))
      {
	amp_msg.type		= A_AMP_ON_RADIO_PAUSE ;
	amp_msg.media_cmd	= MEDIA_NOTHING ;
	amp_msg.index		= -1 ;

	if ((Verbose & (_V_AMP | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	  Print_Log (who, "%s (cmd) sending AMP RADIO PAUSE message\n", name) ;

	if (mq_send (amp_msgq, (char *)&amp_msg, AMP_MSG_LEN, 0) < 0)
	  Goodbye ("Failed to send AMP msg: %s\n", strerror(errno)) ;

	return 0 ;
      }

      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT))
	return 0 ;

      // At this point, we know that MPV is running
      MPV_State = (MPV_State == MPV_PLAYING) ? MPV_PAUSED : MPV_PLAYING ;
      i = (MPV_State == MPV_PAUSED) ; // 0=NOT paused, 1=paused

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Pause=%d\n", name, i) ;

      if ((Mpv_Handle != NULL) && (mpv_set_property (Mpv_Handle, "pause", MPV_FORMAT_FLAG, &i) < 0))
      {
	Print_Log (who, "%s (cmd) set pause failed\n", name) ;
	return -1 ;
      }

      return 0 ;

    case MEDIA_STOP:
      // Turn off the amplifier (and radio if it is on)
      if (amp_msgq != (mqd_t)0)
      {
	amp_msg.type		= A_AMP_OFF ; // This turns off the radio as well (if it was on)
	amp_msg.media_cmd	= MEDIA_NOTHING ;
	amp_msg.index		= -1 ;

	if ((Verbose & (_V_AMP | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	  Print_Log (who, "%s (cmd) sending AMP OFF message\n", name) ;

	if (mq_send (amp_msgq, (char *)&amp_msg, AMP_MSG_LEN, 0) < 0)
	  Goodbye ("Failed to send AMP msg: %s\n", strerror(errno)) ;
      }

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Stop\n", name) ;

      Seconds_Of_Play_Remaining = Seconds_Of_Snooze_Remaining = -1 ;
      Is_Alarm = 0 ; // reset the alarm flag - we can find nothing to play for the alarm
      Fallback_State = F_OFF ;
      Current_Alarm_Stream[0] = '\0' ; // Forget the name of the media source that is playing

      if (Mpv_Handle != NULL)
      {
	if ((MPV_State == MPV_PLAYING) || (MPV_State == MPV_PAUSED))
	  MPV_State = MPV_STOPPED ;

	if (MPV_Kill_Timer < 0)
	  MPV_Kill_Timer = MPV_HOLDON_SECONDS ;

	return mpv_command (Mpv_Handle, Stop_Cmd) ;
      }

      return 0 ;

    case MEDIA_NEXT:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Next\n", name) ;

      // At this point, we know that MPV is running
      if (Mpv_Handle != NULL)
	i = mpv_command (Mpv_Handle, Next_Cmd) ;

      goto check_unpause ;

    case MEDIA_PREV:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Prev\n", name) ;

      // At this point, we know that MPV is running
      if (Mpv_Handle != NULL)
	i = mpv_command (Mpv_Handle, Prev_Cmd) ;

      goto check_unpause ;

    case MEDIA_RESTART:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Restart\n", name) ;

      // At this point, we know that MPV is running
      if (Mpv_Handle != NULL)
	i = mpv_command (Mpv_Handle, Restart_Cmd) ;

      goto check_unpause ;

    case MEDIA_SEEK_FWD:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Seek fwd\n", name) ;

      // At this point, we know that MPV is running
      if (Mpv_Handle != NULL)
	return mpv_command (Mpv_Handle, SeekFwd_Cmd) ;

      return 0 ;

    case MEDIA_SEEK_BACK:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Seek Back\n", name) ;

      // At this point, we know that MPV is running
      if (Mpv_Handle != NULL)
	i = mpv_command (Mpv_Handle, SeekBack_Cmd) ;

      goto check_unpause ;

    case MEDIA_SHUFFLE:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Shuffle\n", name) ;

      // At this point, we know that MPV is running
      if (Mpv_Handle != NULL)
	i = mpv_command (Mpv_Handle, Shuffle_Cmd) ;

      goto check_unpause ;

    case MEDIA_PLAYLIST_CLEAR:
      if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	Print_Log (who, "%s (cmd) Clear Playlist\n", name) ;

      Start_MPV (who, name) ;

      if (Mpv_Handle != NULL)
	return mpv_command (Mpv_Handle, Clear_Cmd) ;

      return 0 ;

    case MEDIA_PLAYLIST_JUMP:
      if ((MPV_State == MPV_STOPPED) || (MPV_State == MPV_WAITING_FOR_EXIT) || (Amp_State == A_AMP_ON_RADIO))
	return 0 ;

      // At this point, we know that MPV is running
      val = index ; // Need to set the index number into a 64 bit variable for the call

      if (val > 0)
      {
	if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	  Print_Log (who, "%s (cmd) Jump %d\n", name, index) ;

	if (Mpv_Handle != NULL)
	  i = mpv_set_property (Mpv_Handle, "playlist-pos-1", MPV_FORMAT_INT64, &val) ;

      check_unpause:
	if (i < 0)
	{
	  Print_Log (who, "%s (cmd) mpv command failed\n", name) ;
	  return -1 ;
	}
      }

      // If the media player is paused, we need to unpause it now
      if (MPV_State == MPV_PAUSED)
      {
	if (Seconds_Of_Play_Remaining <= 0)
	  Seconds_Of_Play_Remaining = Default_Media_Duration * 60 ;

	Seconds_Of_Snooze_Remaining = -1 ;
	i = 0 ; // 0 = unpaused

	if ((Verbose & (_V_MEDIA | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	  Print_Log (who, "%s (cmd) Pause=%d\n", name, i) ;

	if ((Mpv_Handle != NULL) && (mpv_set_property (Mpv_Handle, "pause", MPV_FORMAT_FLAG, &i) < 0))
	{
	  Print_Log (who, "%s (cmd) set pause failed\n", name) ;
	  return -1 ;
	}

	MPV_State = MPV_PLAYING ;
      }

      return 0 ;

    case MEDIA_ADD:		// Called by button manager or from the command line 'm' option
    case MEDIA_ALARM_START:	// Called by timer thread, or within media manager restarting a failed stream
      // Stop updating the index when new media added to the existing playlist, or a new stream is started.
      amp_msg.type	= (strcasecmp(str, "radio")) ? A_AMP_ON_MEDIA_OR_FILE : A_AMP_ON_RADIO ; // strcmp returns 0 if true, non-zero if false
      amp_msg.media_cmd	= cmd ;
      amp_msg.index	= index ;
      strcpy (amp_msg.url_path, str) ;
      Index_Filename[0] = '\0' ;
      Index_Filename_fileinfo.st_size = 0 ;

      if (Current_Alarm_Stream[0] == '\0')
	strcpy (Current_Alarm_Stream, str) ; // Take note of the stream that is playing

      // Turn on the amplifier if we have been given the amp message queue identity and amp state is different
      if (amp_msgq != (mqd_t)0)
      {
	if ((Verbose & (_V_AMP | _V_AMP_MEDIA)) || (Is_Alarm && (Verbose & _V_ALARM)))
	  Print_Log (who, "%s (cmd) sending AMP message=%d\n", name, amp_msg.type) ;

	// We know current amp state is different than we want, so send message and amp manager will record new state
	// and start the amp if required
	// The size of the message calculation assumes there may be unused bytes within the structure
	n = sizeof(amp_msg) - SETUP_FIELD_LEN + strlen(str) ;

	if (mq_send (amp_msgq, (char *)&amp_msg, n, 0) < 0)
	  Goodbye ("Failed to send AMP msg=%d: %s\n", amp_msg.type, strerror(errno)) ;
      }

      // At this point, we know that the amp is on and we have waited for volume to be adjusted to desired starting point
      // Do we need to stop / start the media player?
      if (amp_msg.type == A_AMP_ON_RADIO)
      {
	Quit_MPV (who, name) ;
	Current_Alarm_Stream[0] = '\0' ; // Forget the name of the media source that is playing
      }

      return 0 ;

    case MEDIA_PLAYLIST_REQUEST:
      // Open the response queue in order to send the responses. Each time we run this, there will be a different
      // response queue (different instance of alarm-clock that opened it), so we need to open it fresh each time.
      snprintf (Response_Msgq_name, TEMP_NAME_LEN, "/R%u", responseQ) ;

      if ((Response_Msgq = Open_Msg_Queue_Tx (who, Response_Msgq_name, 1)) < 0)
	Goodbye ("%s: Cannot open Response_Msg_Queue: %s\n", name, strerror(errno)) ;

      // Are there any bluetooth clients connected? Report these with an index number of zero
      if (Num_Bluetooth_Clients > 0)
      {
	for (i=0 ; (i < BLUETOOTH_DEV_RECS) && (Bluetooth_Devices[i].mac_address[0] != '\0') ; i++)
	{
	  n = sprintf (playlist_msg, "0,%s,%s", Bluetooth_Devices[i].mac_address, Bluetooth_Devices[i].device_name) ;

	  if (mq_send (Response_Msgq, (char *)&playlist_msg, n+1, 0) < 0)
	    Print_Log (who, "%s (cmd) Couldn't send playlist response message: %s\n", name, strerror(errno)) ;
	}
      }

      // There will be different behaviour if the radio is playing versus not
      // When the radio is playing, we will botch up a response that has the appearance of
      // having come from the media player
      if ((Amp_State == A_AMP_ON_RADIO) || (Amp_State == A_AMP_ON_RADIO_PAUSE))
      {
	n = sprintf (playlist_msg, "1,\"radio\",(%s)", (Amp_State == A_AMP_ON_RADIO) ? "playing" : "paused") ;

	// The string has been prepared. Send the string as a message
	if (mq_send (Response_Msgq, (char *)&playlist_msg, n+1, 0) < 0)
	  Print_Log (who, "%s (cmd) Couldn't send playlist response message: %s\n", name, strerror(errno)) ;

	else if (Verbose & _V_SETUP)
	  Print_Log (who, "%s (cmd) responded to Playlist request: '%s'\n", name, playlist_msg) ;
      }

      else if ((MPV_State != MPV_STOPPED) && (MPV_State != MPV_WAITING_FOR_EXIT))
      {
	// Consult the media player to obtain the current playlist
	if ((Mpv_Handle != NULL) && (mpv_get_property (Mpv_Handle, "playlist", MPV_FORMAT_NODE, &pl) < 0))
	{
	  Print_Log (who, "%s (cmd) mpv_get_property error\n", name) ;
	  goto _abort_responding ;
	}

	if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
	  Print_Log (who, "%s (cmd) Playlist request:\n", name) ;

	// We are expecting an array
	if (pl.format == MPV_FORMAT_NODE_ARRAY)
	{
	  // Iterate through the array to return the playlist items
	  for (i=0 ; i < pl.u.list->num ; i++)
	  {
	    // i iterates through the tracks in the playlist. It creates a numeric sequence of item numbers
	    // The numeric sequenice is *always* a positive integer (because zero has a special meaning)
	    n = sprintf (playlist_msg, "%d,", i+1) ;

	    // j iterates through the attributes for each track.
	    // Firstly find and display the filename
	    for (j=0 ; j < pl.u.list->values[i].u.list->num ; j++)
	    {
	      if (strcasecmp (pl.u.list->values[i].u.list->keys[j], "filename") == 0)
	      {
		// Enclose the name of the file in double quotes - to guard against embedded commas.
		// Also, the termination is ", rather than the quote on its own.
		n += sprintf (&playlist_msg[n], "\"%s\",", pl.u.list->values[i].u.list->values[j].u.string) ;
		break ;
	      }
	    }

	    // Secondly, identify if this is the current track (playing)
	    // But only do this if there is still duration in the playback
	    if (Seconds_Of_Play_Remaining >= 0)
	    {
	      for (j=0 ; j < pl.u.list->values[i].u.list->num ; j++)
		if ((strcasecmp (pl.u.list->values[i].u.list->keys[j], "current") == 0) ||
		    (strcasecmp (pl.u.list->values[i].u.list->keys[j], "playing") == 0))
		{
		  if (MPV_State == MPV_PAUSED)
		    n += sprintf (&playlist_msg[n], "(paused)") ;
		  else
		    n += sprintf (&playlist_msg[n], "(playing)") ;

		  break ;
		}
	    }

	    // The string has been prepared. Send the string as a message
	    if (mq_send (Response_Msgq, (char *)&playlist_msg, n+1, 0) < 0)
	      Print_Log (who, "%s (cmd) Couldn't send playlist response message: %s\n", name, strerror(errno)) ;

	    else if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
	      Print_Log (who, "%s  Playlist: '%s'\n", name, playlist_msg) ;
	  }
	}

	mpv_free_node_contents (&pl) ; // return the playlist structure to the memory pool
      }

      // In order to signal that there is nothing more, send the string 'end'
      if (mq_send (Response_Msgq, "end", 4, 0) < 0)
	Print_Log (who, "%s (cmd) Couldn't send playlist completed message: %s\n", name, strerror(errno)) ;

    _abort_responding:
      mq_close (Response_Msgq) ;
      Response_Msgq_name[0] = '\0' ;
      Response_Msgq = 0 ;
      return 0 ;

    default:
      break ;
  }

  // Unknown command - its an error
  Goodbye ("%s: Unknown media command %d\n", name, cmd) ;
  return -1 ;
} // Media_Command

void *
Media_Player_Thread (void *xarg)
{
  MEDIA_MSG_BUF		media_msg ;
  int			i, n ;
  TV			Time ;
  time_t		last_tick_time ;
  mqd_t			amp_msgq ;
  char			*peer_type ;
  char			ipv4_addr[INET_ADDRSTRLEN] ;
  char			shut_up_compiler_warning[SETUP_FIELD_LEN+1] ;
  MEDIA_RESPONSE_BUF	response ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
    Print_Log (T_MEDIA_PLAYER, "Entering %s\n", Media_Player_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  Is_Alarm = 0 ;
  Seconds_Of_Play_Remaining = Seconds_Of_Snooze_Remaining = -1 ;
  MPV_State = MPV_STOPPED ;
  MPV_Kill_Timer = -1 ;
  Fallback_State = F_OFF ;
  Vol_Offset = Targ_Vol_Offset = 0 ;

  // We are the recipient of Media_Msgq. Open it now for reading before any other threads try to open it for writing.
  Media_Msgq = Open_Msg_Queue_Rx (T_MEDIA_PLAYER, Media_Msgq_name, 5, sizeof(media_msg)) ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Now we can open the amp manager message queue for writing
  amp_msgq = Open_Msg_Queue_Tx (T_MEDIA_PLAYER, Amp_Msgq_name, 0) ;

  // initialise time
  gettimeofday (&Time, NULL) ;
  last_tick_time = Time.tv_sec ;

  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
    Print_Log (T_MEDIA_PLAYER, "%s initialised\n", Media_Player_Thread_Str) ;

  // Messages to the media thread contain the linux command text for launching a new media stream
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // This is an infinte loop - wait for messages and then do what the message requests.
    // Different messages to (a) start a new stream, (b) pause a stream, (c) stop a stream, (d) check on the status of a stream
    if ((n = Wait_For_Next_Message (T_MEDIA_PLAYER, Media_Msgq, (void *)&media_msg, MEDIA_MSG_LEN, sizeof(media_msg), Media_Player_Thread_Str)) < (int)MEDIA_MSG_LEN)
      continue ;

    gettimeofday (&Time, NULL) ;

    if ((Verbose & _V_MEDIA) || ((Verbose & (_V_AMP_MEDIA | _V_ALARM)) && (media_msg.type != MEDIA_TICK)))
    {
      Print_Log (T_MEDIA_PLAYER,
		 "%s RxMsg [type=%hu, index=%d, Dur=%d seconds, Vol=%" PRId16 ", Voffset=%" PRId16 "..%" PRId16 ", Q=%u, path='%s'] ret=%d\n"
		 "%s    Is_Alarm=%d, MPV_State=%hu, Seconds_Of_Play_Remaining=%d, Seconds_Of_Snooze_Remaining=%d, MPV_Kill_Timer=%d\n",
		 Media_Player_Thread_Str, media_msg.type, media_msg.index, media_msg.seconds, media_msg.volume,
		 media_msg.init_vol_offset, media_msg.targ_vol_offset, (unsigned)media_msg.responseQ, media_msg.url_path, n,
		 Media_Player_Thread_Str, Is_Alarm, MPV_State, Seconds_Of_Play_Remaining, Seconds_Of_Snooze_Remaining, MPV_Kill_Timer) ;
    }
    // - - - - - - - - - - - - - - - -
    // TICK messages are to get us to check on the status of the media player, and restart it if it has finished/aborted
    if (media_msg.type == MEDIA_TICK)
    {
      // The Seconds_Of_Play_Remaining and Seconds_Of_Snooze_Remaining are measured in seconds, but the tick period is less than one second
      // Only process the Seconds_Of_Play_Remaining and Seconds_Of_Snooze_Remaining when the seconds count ticks over
      if (Time.tv_sec != last_tick_time)
      {
	// Process SNOOZE (if the snooze timer is running)
	if (Seconds_Of_Snooze_Remaining >= 0)
	{
	  Seconds_Of_Snooze_Remaining -= Time.tv_sec - last_tick_time ;

	  if (Seconds_Of_Snooze_Remaining <= 0)
	  {
	    // The snooze timer has just expired
	    if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
	      Print_Log (T_MEDIA_PLAYER, "%s snooze finished\n", Media_Player_Thread_Str) ;

	    if ((Seconds_Of_Play_Remaining > 5) && (MPV_State == MPV_PAUSED))
	    {
	      // Restart the media stream if it still has time to run, otherwise kill it
	      // (Issuing a second pause will restart a paused stream)
	      if ((Amp_State != A_AMP_OFF) && (Amp_State != A_AMP_ON_BLUETOOTH) &&
		  (Media_Command (T_MEDIA_PLAYER, Media_Player_Thread_Str, MEDIA_PAUSE, -1, NULL, amp_msgq, media_msg.responseQ) < 0))
		goto terminate_media_stream ; // the unpause failed
	    }

	    Seconds_Of_Snooze_Remaining = -1 ;
	  }
	}

	// Process playing media
	if (Seconds_Of_Play_Remaining >= 0)
	{
	  Seconds_Of_Play_Remaining -= Time.tv_sec - last_tick_time ;

	  if (Seconds_Of_Play_Remaining <= 0)
	  {
	  terminate_media_stream:
	    if (Media_Command (T_MEDIA_PLAYER, Media_Player_Thread_Str, MEDIA_STOP, -1, NULL, amp_msgq, media_msg.responseQ) < 0)
	      Goodbye ("%s: Failed to stop media\n", Media_Player_Thread_Str) ;

	    if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
	      Print_Log (T_MEDIA_PLAYER, "%s media stopped\n", Media_Player_Thread_Str) ;
	  }
	}
      }

      // Remember the last epoch
      last_tick_time = Time.tv_sec ;
      continue ; // Its a tick message - nothing more to do except continue / wait for next message
    } // tick
    // - - - - - - - - - - - - - - - -
    if (media_msg.type == MEDIA_STOP)
      goto terminate_media_stream ;
    // - - - - - - - - - - - - - - - -
    // This group of commands only make sense if there is media playing. Otherwise, ignore
    if ((media_msg.type == MEDIA_PAUSE) || (media_msg.type == MEDIA_NEXT) || (media_msg.type == MEDIA_PREV) || (media_msg.type == MEDIA_RESTART) ||
	(media_msg.type == MEDIA_SEEK_FWD) || (media_msg.type == MEDIA_SEEK_BACK) || (media_msg.type == MEDIA_PLAYLIST_CLEAR) ||
	(media_msg.type == MEDIA_SHUFFLE) || (media_msg.type == MEDIA_PLAYLIST_JUMP) || (media_msg.type == MEDIA_PLAYLIST_REQUEST))
    {
      if (((Amp_State != A_AMP_OFF) && (Amp_State != A_AMP_ON_BLUETOOTH)) || (media_msg.type == MEDIA_PLAYLIST_REQUEST))
      {
	n = Media_Command (T_MEDIA_PLAYER, Media_Player_Thread_Str, media_msg.type, media_msg.index, NULL, amp_msgq, media_msg.responseQ) ;

	if ((n < 0) && ((media_msg.type == MEDIA_PAUSE) || (media_msg.type == MEDIA_RESTART)))
	  Goodbye ("%s: Failed to issue media command %d\n", Media_Player_Thread_Str, media_msg.type) ;

	// Keep track of snoozing times when pausing - toggle between snooze on/off
	if (media_msg.type == MEDIA_PAUSE)
	{
	  if (Seconds_Of_Snooze_Remaining < 0)
	    Seconds_Of_Snooze_Remaining = Default_Snooze_Duration ;

	  else
	    Seconds_Of_Snooze_Remaining = -1 ;

	  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
	    Print_Log (T_MEDIA_PLAYER, "%s Seconds_Of_Snooze_Remaining set to %d\n", Media_Player_Thread_Str, Seconds_Of_Snooze_Remaining) ;
	}

	else if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
	  Print_Log (T_MEDIA_PLAYER, "%s issued media command %d\n", Media_Player_Thread_Str, media_msg.type) ;
      }

      continue ;
    }
    // - - - - - - - - - - - - - - - -
    if (media_msg.type == MEDIA_PEERS_REQUEST)
    {
      // Open the response queue in order to send the responses. Each time we run this, there will be a different
      // response queue (different instance of alarm-clock that opened it), so we need to open it fresh each time.
      snprintf (Response_Msgq_name, TEMP_NAME_LEN, "/R%u", (unsigned)media_msg.responseQ) ;

      if ((Response_Msgq = Open_Msg_Queue_Tx (T_MEDIA_PLAYER, Response_Msgq_name, 1)) < 0)
	Goodbye ("%s: Cannot open Response_Msg_Queue: %s\n", Media_Player_Thread_Str, strerror(errno)) ;

      // Dump the contents of the peers table
      for (i=0 ; i < NUM_PEERS ; i++)
      {
	if (Peers[i].sin_addr.s_addr != 0)
	{
	  // Decode the IP address into text
	  inet_ntop (AF_INET, &Peers[i].sin_addr.s_addr, ipv4_addr, sizeof(ipv4_addr)) ;

	  if (Peers[i].type == 'A')
	    peer_type = "Audible clustered (can control but cannot be controlled by other clustered clocks)" ;

	  else if (Peers[i].type == 'C')
	    peer_type = "Audible clustered (can both control and be controlled by other clustered clocks)" ;

	  else if (Peers[i].type == 'S')
	    peer_type = "Stand alone clock" ;

	  else if (Peers[i].type == 'Z')
	    peer_type = "Silent clustered clock" ;

	  else
	    peer_type = "?" ;

	  // The following is to shut up an inexplicable compiler warning under the new Bookworm operating system. It is somehow
	  // counting the Peers[i].name string length to be 1102 bytes long??? (It is defined as 200 bytes)
	  strncpy (shut_up_compiler_warning, Peers[i].name, SETUP_FIELD_LEN) ;
	  shut_up_compiler_warning[SETUP_FIELD_LEN] = '\0' ;

	  // The long long int thing is because time_t could be either 32 or 64 bits, and there will be a compile error if its 64 and printed as 32!
	  n = sprintf (response, "%s,%s,%lld,%s", ipv4_addr, peer_type, (long long int)(Time.tv_sec - Peers[i].last_seen), shut_up_compiler_warning) ;

	  // The string has been prepared. Send the string as a message
	  if (mq_send (Response_Msgq, response, n+1, 0) < 0)
	    Print_Log (T_MEDIA_PLAYER, "%s: Couldn't send peers response message: %s\n", Media_Player_Thread_Str, strerror(errno)) ;
	}
      }

      // In order to signal that there is nothing more, send the string 'end'
      if (mq_send (Response_Msgq, "end", 4, 0) < 0)
	Print_Log (T_MEDIA_PLAYER, "%s: Couldn't send playlist completed message: %s\n", Media_Player_Thread_Str, strerror(errno)) ;

      mq_close (Response_Msgq) ;
      Response_Msgq_name[0] = '\0' ;
      Response_Msgq = 0 ;
      continue ;
    }
    // - - - - - - - - - - - - - - - -
    if ((media_msg.type == MEDIA_ADD) || (media_msg.type == MEDIA_ALARM_START))
    {
      // Make sure there is something to run
      if (media_msg.url_path[0] == '\0')
	continue ;

      Seconds_Of_Snooze_Remaining = -1 ;
      Is_Alarm = (media_msg.type == MEDIA_ALARM_START) ;

      // If volume level has been defined in the MEDIA_ALARM_START command, set the global Volume variable
      if ((media_msg.volume >= 0) && (media_msg.volume <= MAX_VOLUME))
	Volume = media_msg.volume ;

      // If volume offset has been defined in the MEDIA_ALARM_START command, set the global Volume_Offset variable
      if ((media_msg.init_vol_offset >= -MAX_VOL_OFFSET) && (media_msg.init_vol_offset <= MAX_VOL_OFFSET))
	Vol_Offset = media_msg.init_vol_offset ;

      if ((media_msg.targ_vol_offset >= -MAX_VOL_OFFSET) && (media_msg.targ_vol_offset <= MAX_VOL_OFFSET))
	Targ_Vol_Offset = media_msg.targ_vol_offset ;

      Enforce_Volume_Limits (media_msg.type == MEDIA_ALARM_START) ;

      if (Media_Command (T_MEDIA_PLAYER, Media_Player_Thread_Str, media_msg.type, media_msg.index, media_msg.url_path, amp_msgq, media_msg.responseQ) < 0)
	Goodbye ("%s: Failed to start media player '%s': %s\n", Media_Player_Thread_Str, media_msg.url_path, strerror(errno)) ;

      // Set the playing time to the the LARGER of the remaining time or the newly specified time
      // If this is not what the human wanted, they can issue a stop command to shut it up
      if (Seconds_Of_Play_Remaining < media_msg.seconds)
	Seconds_Of_Play_Remaining = media_msg.seconds ;

      // Is the next alarm going to chain? Add one second to the remaining play time to ensure a smooth chaining
      if ((Seconds_To_Next_Alarm > 0) && (Seconds_Of_Play_Remaining == (Alarms[Next_Alarm_Index].duration * 60)))
	Seconds_Of_Play_Remaining++ ;

      // Remember the last epoch
      last_tick_time = Time.tv_sec ;

      if (Verbose & (_V_MEDIA | _V_AMP_MEDIA | _V_ALARM))
	Print_Log (T_MEDIA_PLAYER, "%s send media command '%s'. Is_Alarm=%d, Index=%d, Vol=%d, Voffset=%d..%d, Seconds_Of_Play=%d\n",
		   Media_Player_Thread_Str, media_msg.url_path, Is_Alarm, media_msg.index, media_msg.volume,
		   media_msg.init_vol_offset, media_msg.targ_vol_offset, Seconds_Of_Play_Remaining) ;

      continue ;
    }
    // - - - - - - - - - - - - - - - -
    // Unknown message type?
    Goodbye ("%s: Unknown message type=%hu '%s': %s\n", Media_Player_Thread_Str, media_msg.type, media_msg.url_path) ;
  }

  Print_Log (T_MEDIA_PLAYER, "%s unscheduled exit\n", Media_Player_Thread_Str) ;

  // Tidy up before exiting
  mq_close (Media_Msgq) ;
  mq_close (amp_msgq) ;
  pthread_exit (NULL) ;
} // Media_Player_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Media_Manager_Thread
//
// Monitors events from the mpv slave and updates media states. Events include tracks starting and
// stopping, errors etc. The important thing we need is knowledge of when a track finishes or when
// a stream drops out.
static const char Media_Manager_Thread_Str[] = "Media Manager Thread" ;

void *
Media_Manager_Thread (void *xarg)
{
  mpv_event		*event ;
  mqd_t			amp_msgq ;
  AMP_MSG_BUF		amp_msg ;
  int			index ;
  int64_t		count, position ;
  FILE			*f ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
    Print_Log (T_MEDIA_MANAGER, "Entering %s\n", Media_Manager_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Now we can open the amp manager message queue for writing
  amp_msgq = Open_Msg_Queue_Tx (T_MEDIA_MANAGER, Amp_Msgq_name, 0) ;

  if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
    Print_Log (T_MEDIA_MANAGER, "%s initialised\n", Media_Manager_Thread_Str) ;

  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // If there is no mpv instance, wait a little and then try again
    if (Mpv_Handle == NULL)
    {
      Sleep_ns (0.5 * NS_PER_SECOND) ;
      continue ;
    }

    // The following call will wait for an event (up to 10000 seconds) and not return until the event occurs
    // Events should be happening routinely and frequently while mpv is running
    event = mpv_wait_event (Mpv_Handle, 10000) ;

    if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
      Print_Log (T_MEDIA_MANAGER, "%s received event: %d='%s'\n", Media_Manager_Thread_Str, event->event_id, mpv_event_name(event->event_id));

    switch (event->event_id)
    {
      case MPV_EVENT_SHUTDOWN:
	goto do_mpv_shutdown ;

      case MPV_EVENT_AUDIO_RECONFIG:
	// Are we expecting mpv to exit? If so, clean up the mess
	if (MPV_State == MPV_WAITING_FOR_EXIT)
	{
	do_mpv_shutdown:
	  if (Mpv_Handle != NULL)
	    mpv_destroy (Mpv_Handle) ;

	  Mpv_Handle = NULL ;
	  MPV_State = MPV_STOPPED ;
	  Print_Log (T_MEDIA_MANAGER, "%s: Media silent. mpv instance has been shut down.\n", Media_Manager_Thread_Str) ;
	}

	break ;

      case MPV_EVENT_IDLE:
	if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
	  Print_Log (T_MEDIA_MANAGER, "%s media stopped\n", Media_Manager_Thread_Str) ;

	if (MPV_State == MPV_WAITING_FOR_EXIT)
	  goto do_mpv_shutdown ;

	if (MPV_State != MPV_STOPPED)
	{
	  MPV_State = MPV_STOPPED ;
	  MPV_Kill_Timer = MPV_HOLDON_SECONDS ;
	}

	// The media has exited. If we are playing an alarm, we want to revert to something else.
	// Fallback_State holds the current fallback status. F_OFF means we have not yet fallen back to something.
	// Is there an alarm running and does it still have time to go?
	// We will try to fallback to the fallback file if the alarm still has time to go
	if (Is_Alarm)
	{
	  // If we were playing the original media (F_OFF), is there a fallback stream to attempt?
	  if ((Fallback_State == F_OFF) && (Fallback_Alarm_File[0] != '\0') && (strcasecmp(Fallback_Alarm_File, "radio") != 0))
	  {
	    // Is there a fallback alarm file index?
	    index = Is_Playlist (Fallback_Alarm_File, Index_Filename, &Index_Filename_fileinfo) ;
	    Fallback_State = F_MEDIA_OR_FILE ;

	    // We will try to play the fallback file/URL stream. Using MEDIA_ALARM_START will replace the currently playing stream
	    if (Media_Command (T_MEDIA_MANAGER, Media_Manager_Thread_Str, MEDIA_ALARM_START, index, Fallback_Alarm_File, amp_msgq, 0) < 0)
	    {
	      Print_Log (T_MEDIA_MANAGER, "%s: Failed to send fallback media command '%s': %s\n", Media_Manager_Thread_Str, Fallback_Alarm_File, strerror(errno)) ;
	      goto play_radio ;
	    }
	  }
	  else
	  {
	    // No fallback stream. Turn the radio on.
	  play_radio:
	    Media_Command (T_MEDIA_MANAGER, Media_Manager_Thread_Str, MEDIA_ALARM_START, -1, "radio", amp_msgq, 0) ;
	    Fallback_State = F_RADIO ;

	    if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
	      Print_Log (T_MEDIA_MANAGER, "%s: mpv exited (falling back to radio)'\n", Media_Manager_Thread_Str) ;
	  }
	}

	else if (Seconds_Of_Play_Remaining > 30) // If there is still playing time, restart from track 1
	{
	  // Its not an alarm, its the media player
	  // The current playlist or file finished, so initiate a repeat
	  amp_msg.type		= A_RESTART_LAST_MEDIA ;
	  amp_msg.media_cmd	= MEDIA_NOTHING ;
	  amp_msg.index		= 1 ;

	  if (Verbose & (_V_AMP | _V_AMP_MEDIA))
	    Print_Log (T_MEDIA_MANAGER, "%s sending AMP RESTART LAST MEDIA message\n", Media_Manager_Thread_Str) ;

	  if (mq_send (amp_msgq, (char *)&amp_msg, AMP_MSG_LEN, 0) < 0)
	    Goodbye ("%s failed to send AMP msg: %s\n", Media_Manager_Thread_Str, strerror(errno)) ;
	}

	break ;

      case MPV_EVENT_PLAYBACK_RESTART:
	if (Verbose & (_V_MEDIA | _V_AMP_MEDIA))
	  Print_Log (T_MEDIA_MANAGER, "%s media playing. Seconds of play remaining=%d\n", Media_Manager_Thread_Str, Seconds_Of_Play_Remaining) ;

	if (Mpv_Handle != NULL)
	{
	  if (Seconds_Of_Play_Remaining <= 0)
	  {
	    // This isn't expected to happen, but handle if it does
	    if (MPV_State == MPV_PLAYING)
	    {
	      if (Media_Command (T_MEDIA_MANAGER, Media_Manager_Thread_Str, MEDIA_PAUSE, -1, NULL, amp_msgq, 0) < 0)
		Goodbye ("%s: Failed to pause mpv: %s\n", Media_Manager_Thread_Str, strerror(errno)) ;
	    }

	    else if (MPV_State != MPV_PAUSED)
	    {
	      if (Media_Command (T_MEDIA_MANAGER, Media_Manager_Thread_Str, MEDIA_STOP, -1, NULL, amp_msgq, 0) < 0)
		Goodbye ("%s: Failed to stop mpv: %s\n", Media_Manager_Thread_Str, strerror(errno)) ;
	    }

	    if (MPV_Kill_Timer < 0)
	      MPV_Kill_Timer = MPV_HOLDON_SECONDS ;

	    break ;
	  }

	  MPV_State = MPV_PLAYING ;
	  MPV_Kill_Timer = -1 ; // Turn off the kill timer - we are playing and don't want to kill mpv yet!

	  // Is there an index file that needs to be updated???
	  // Read size of the current playlist and the new position
	  if ((Index_Filename[0] != '\0') &&
	      (mpv_get_property (Mpv_Handle, "playlist-count", MPV_FORMAT_INT64, &count) >= 0) &&
	      (mpv_get_property (Mpv_Handle, "playlist-pos-1", MPV_FORMAT_INT64, &position) >= 0) &&
	      (position <= count) &&
	      (f = fopen (Index_Filename, "w")))
	  {
	    // We have a new track which is being played. Write the NEXT track to the index file
	    // but if this is the last track in the playlist, change the next position to 1 (wrap around)
	    if (++position > count)
	      position = 1 ;

	    fprintf (f, "%" PRId64 "\n", position) ;
	    fclose (f) ;

	    // Set file ownership of the index file to be the same as the playlist
	    // (non-zero size field indicates valid ownership fields)
	    if (Index_Filename_fileinfo.st_size != 0)
	      chown (Index_Filename, Index_Filename_fileinfo.st_uid, Index_Filename_fileinfo.st_gid) ;

	    // In any case, ensure anybody can read and write the index file, so that it can be removed
	    chmod (Index_Filename, 0666) ;
	  }
	}

	break ;

      case MPV_EVENT_QUEUE_OVERFLOW:
	Goodbye ("%s mpv queue overflow\n", Media_Manager_Thread_Str) ;

      default:
	// Almost always, there will be no event (MPV_EVENT_NONE) which we will ignore
	// And we will also ignore other events
	break ;
    }
  }

  Print_Log (T_MEDIA_MANAGER, "%s unscheduled exit\n", Media_Manager_Thread_Str) ;
  mq_close (amp_msgq) ;
  pthread_exit (NULL) ;
} // Media_Manager_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Amp_Manager_Thread
//
// A thread whose job is to manage the state of the amplifier chip, and to manage changes to the volume level
// and to choose either the input to the amp: (a) raspberry pi media (the default) or (b) radio output
static const char Amp_Manager_Thread_Str[] = "Amp Manager Thread" ;

void *
Amp_Manager_Thread (void *xarg)
{
  AMP_MSG_BUF		amp_msg ;
  int			error, index ;
  uint8_t		current_volume, amp_pause, set_vol_lines_idle, restart_prevent_loop ;
  char			*Add_Cmd[4] ;
  char			url_path [SETUP_FIELD_LEN+1] ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & (_V_AMP | _V_AMP_MEDIA))
    Print_Log (T_AMP, "Entering %s\n", Amp_Manager_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  Amp_State = A_AMP_OFF ;
  Add_Cmd[0] = NULL ;
  index = -1 ;
  url_path[0] = '\0' ;
  current_volume = amp_pause = set_vol_lines_idle = restart_prevent_loop = 0 ;

  // We are the recipient of Amp_Msgq. Open it now for reading before any other threads try to open it for writing.
  Amp_Msgq = Open_Msg_Queue_Rx (T_AMP, Amp_Msgq_name, 5, sizeof(amp_msg)) ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Turn off the amplifier and radio relay - Low = OFF
  // Volume control lines are normally high, and get toggled low to activate the function.
  if (((error = gpioWrite (AMP_ENABLE, 0)) != 0) ||
      ((error = gpioWrite (RADIO_ENABLE, 0)) != 0) ||
      ((error = gpioWrite (VOLUP, 1)) != 0) ||
      ((error = gpioWrite (VOLDOWN, 1)) != 0))
    Goodbye ("%s Error setting amp, relay, volup or voldn outputs: %d\n", Amp_Manager_Thread_Str, error) ;

  if (Verbose & (_V_AMP | _V_AMP_MEDIA))
    Print_Log (T_AMP, "%s initialised\n", Amp_Manager_Thread_Str) ;

  // Messages to the media thread contain the linux command text for launching a new media stream
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // This is an infinte loop - wait for messages and then do what the message requests.
    // Different messages to (a) start a new stream, (b) pause a stream, (c) stop a stream, (d) check on the status of a stream
    if (Wait_For_Next_Message (T_AMP, Amp_Msgq, (void *)&amp_msg, AMP_MSG_LEN, sizeof(amp_msg), Amp_Manager_Thread_Str) < (int)AMP_MSG_LEN)
      continue ;

    if ((Verbose & _V_AMP) || ((Verbose & _V_AMP_MEDIA) && (amp_msg.type != A_AMP_TICK)))
      Print_Log (T_AMP, "%s RxMsg [type=%d, media_cmd=%d, index=%d]. Amp_State=%d, Volume=%d, Voffset=%d, current_volume=%d, set_vol_lines_idle=%d\n",
		 Amp_Manager_Thread_Str, amp_msg.type, amp_msg.media_cmd, amp_msg.index, Amp_State, Volume, Vol_Offset, current_volume, set_vol_lines_idle) ;

    switch (amp_msg.type)
    {
      case A_RESTART_LAST_MEDIA:
	// If the url_path points to a valid playlist file, then restart it
	if ((restart_prevent_loop == 0) && (url_path[0] != '\0') && (access (url_path, R_OK) == 0))
	{
	  amp_msg.index = 1 ;
	  restart_prevent_loop = 1 ;
	  goto restart_last_media ;
	}

	break ;

      case A_AMP_ON_MEDIA_OR_FILE:
	// In the case of this message, it may include the identity of a media stream that needs to be opened.
	// If this is the case, the time to open will be when the amp is on and the volume has reached the target
	// We need to remember the stream name and prepare the media command.
	// If the url path is missing but an index is specified, the message means jump to the index rather than
	// open the stream (the playlist will already be in mpv).
	strncpy (url_path, amp_msg.url_path, SETUP_FIELD_LEN+1) ; // url_path may be null in the case of restarting

      restart_last_media:
	Add_Cmd[0] = "loadfile" ; // open a file or a URL
	Add_Cmd[1] = url_path ;
	Add_Cmd[2] = Add_Cmd[3] = NULL ;
	index	   = amp_msg.index ;

	// If the command is MEDIA_ALARM_START, we want to clear the current playlist and immediately
	// start to play the new stream. We do this without the append-play option in the mpv_command call.
	// Without that option, the default behaviour is to replace the current media stream with the new one.
	if (amp_msg.media_cmd == MEDIA_ADD)
	  Add_Cmd[2] = "append-play" ;

	if (Verbose & (_V_AMP | _V_AMP_MEDIA))
	  Print_Log (T_AMP, "%s preparing to play '%s'\n", Amp_Manager_Thread_Str, url_path) ;

	if ((error = gpioWrite (RADIO_ENABLE, 0)) != 0)
	  Goodbye ("%s Error setting radio output off: %d\n", Amp_Manager_Thread_Str, error) ;

	// If we are starting something fresh, set the Index_Filename if this is a playlist
	// But if we are not starting fresh (ie amp is already on), then nix the index_filename
	if ((Amp_State == A_AMP_OFF) || (Amp_State == A_AMP_ON_BLUETOOTH))
	  index = Is_Playlist (url_path, Index_Filename, &Index_Filename_fileinfo) ;

	// If an index was supplied with this command, use that rather than the index discovered from the index file
	if (amp_msg.index > 0)
	  index = amp_msg.index ;

	goto amp_common ;

      case A_AMP_ON_RADIO:
	strcpy (url_path, "radio") ;
	Add_Cmd[0] = NULL ;
	index	   = -1 ;

      amp_common:
	if (((error = gpioWrite (AMP_ENABLE, 1) != 0) != 0) ||
	    ((error = gpioWrite (VOLUP, 1)) != 0) ||
	    ((error = gpioWrite (VOLDOWN, 1)) != 0))
	  Goodbye ("%s Error setting amp, volup or voldn outputs: %d\n", Amp_Manager_Thread_Str, error) ;

	// If we are turning the amp on and it was previoiusly off, we also need to start the volume state machine.
	// (Otherwise, the volume state machine is already running and the volume is at or approaching the target).
	// The PAM8407 amplifier has 32 volume levels numbered 1 to 32 in the data sheet. (1=loudest in the data sheet)
	// Upon power-on or coming out of shutdown mode, the chip defaults to 'level 9' in the datasheet terms.
	// I will have 32 levels numbered 0..31, with 31 being the loudest. The power-on level is therefore 32-9=23
	if ((Amp_State == A_AMP_OFF) || (Amp_State == A_AMP_ON_BLUETOOTH))
	{
	  current_volume = DEFAULT_VOLUME ; // PAM8307 default vol is 23 on a scale of 0..31
	  set_vol_lines_idle = 0 ;
	}

	// Reset the volume again (sigh)
	//if ((Amp_State != amp_msg.type) && (amp_msg.type != A_AMP_OFF))
	//  Set_Default_Volume () ;

	Amp_State = amp_msg.type ; // amplifier is now on. This remembers if its the media player or the radio
	amp_pause = 0 ;

	if (Verbose & (_V_AMP | _V_AMP_MEDIA))
	  Print_Log (T_AMP, "%s Amp On %d, Volume=%d, Voffset=%d, current_volume=%u\n", Amp_Manager_Thread_Str, amp_msg.type, Volume, Vol_Offset, current_volume) ;

	break ;

	// - - - - - - - - - - - - - - - -
	// Ignore amp off messages if there are connected bluetooth clients
      case A_AMP_OFF:
	if (Num_Bluetooth_Clients > 0)
	{
	  if (Verbose & (_V_AMP | _V_AMP_MEDIA))
	    Print_Log (T_AMP, "%s Amp Off ignored. %hu bluetooth clients\n", Amp_Manager_Thread_Str, Num_Bluetooth_Clients) ;

	  Seconds_Of_Play_Remaining = Seconds_Of_Snooze_Remaining = -1 ;
	  break ;
	}

	if (((error = gpioWrite (AMP_ENABLE, 0)) != 0) ||
	    ((error = gpioWrite (RADIO_ENABLE, 0)) != 0) ||
	    ((error = gpioWrite (VOLUP, 1)) != 0) ||
	    ((error = gpioWrite (VOLDOWN, 1)) != 0))
	  Goodbye ("%s Error setting amp, relay, volup or voldn outputs: %d\n", Amp_Manager_Thread_Str, error) ;

	Amp_State = A_AMP_OFF ;
	current_volume = amp_pause = set_vol_lines_idle = 0 ; // Amplifier is off / silent
	Seconds_Of_Play_Remaining = Seconds_Of_Snooze_Remaining = -1 ;

	if (Verbose & (_V_AMP | _V_AMP_MEDIA))
	  Print_Log (T_AMP, "%s Amp Off\n", Amp_Manager_Thread_Str) ;

	break ;

	// - - - - - - - - - - - - - - - -
      case A_AMP_ON_RADIO_PAUSE:
	if (Amp_State == A_AMP_ON_RADIO)
	{
	  Amp_State = A_AMP_ON_RADIO_PAUSE ;
	  amp_pause = 1 ;
	}
	else if (Amp_State == A_AMP_ON_RADIO_PAUSE)
	{
	  Amp_State = A_AMP_ON_RADIO ;
	  amp_pause = 0 ;

	  // Reset the volume again (sigh)
	  //Set_Default_Volume () ;
	}

	if (Verbose & (_V_AMP | _V_AMP_MEDIA))
	  Print_Log (T_AMP, "%s Amp radio pause=%d\n", Amp_Manager_Thread_Str, amp_pause) ;

	break ;
	// - - - - - - - - - - - - - - - -
      case A_AMP_TICK:
	// We may receive a tick because a bluetooth client has just connected. If that's the case, then enable the amp
	// even though nothing else is enabled
	if ((Amp_State == A_AMP_OFF) && (Num_Bluetooth_Clients > 0))
	{
	  if ((error = gpioWrite (AMP_ENABLE, 1)) != 0)
	    Goodbye ("%s Error setting amp on: %d\n", Amp_Manager_Thread_Str, error) ;

	  Amp_State = A_AMP_ON_BLUETOOTH ;
	  current_volume = DEFAULT_VOLUME ; // PAM8307 default vol is 23 on a scale of 0..31
	  set_vol_lines_idle = 0 ;
	  amp_pause = 0 ;

	  // Reset the volume again (sigh)
	  //Set_Default_Volume () ;

	  if (Verbose & (_V_AMP | _V_AMP_MEDIA | _V_BLUETOOTH))
	    Print_Log (T_AMP, "%s Amp On Bluetooth, Volume=%d, Voffset=%d, current_volume=%u\n", Amp_Manager_Thread_Str, Volume, Vol_Offset, current_volume) ;
	  break ;
	}

	if ((Amp_State == A_AMP_ON_BLUETOOTH) && (Num_Bluetooth_Clients == 0))
	{
	  if (((error = gpioWrite (AMP_ENABLE, 0)) != 0) ||
	      ((error = gpioWrite (VOLUP, 1)) != 0) ||
	      ((error = gpioWrite (VOLDOWN, 1)) != 0))
	    Goodbye ("%s Error setting amp off: %d\n", Amp_Manager_Thread_Str, error) ;

	  Amp_State = A_AMP_OFF ;
	  current_volume = amp_pause = set_vol_lines_idle = 0 ; // Amplifier is off / silent

	  if (Verbose & (_V_AMP | _V_AMP_MEDIA | _V_BLUETOOTH))
	    Print_Log (T_AMP, "%s Amp Off Bluetooth\n", Amp_Manager_Thread_Str) ;

	  break ;
	}

	if (Amp_State != A_AMP_OFF)
	{
	  // The set_vol_lines_idle variable is used to toggle volume control signals up and then down
	  if (set_vol_lines_idle == 1)
	  {
	    // set_vol_lines_idle is currently high ==> Turn Vol up and down lines off (set them high = off)
	    if ( ((error = gpioWrite (VOLDOWN, 1)) != 0) || ((error = gpioWrite (VOLUP, 1)) != 0) )
	      Goodbye ("%s Error setting volup or voldn outputs high: %d\n", Amp_Manager_Thread_Str, error) ;

	    set_vol_lines_idle = 0 ;
	  }

	  // the clauses below are only parsed if set_vol_lines_idle is 0, radio pause will cause volume to ramp down to zero
	  else if ( (amp_pause || (current_volume > (Volume + Vol_Offset))) &&
		    (current_volume > 0))
	  {
	    if ((error = gpioWrite (VOLDOWN, 0)) != 0)
	      Goodbye ("%s Error setting voldn output low: %d\n", Amp_Manager_Thread_Str, error) ;

	    current_volume-- ;
	    set_vol_lines_idle = 1 ;
	    goto squawk_volume ;
	  }

	  else if ((current_volume < (Volume + Vol_Offset)) && (current_volume < MAX_VOLUME) && (amp_pause == 0))
	  {

	    if ((error = gpioWrite (VOLUP, 0)) != 0)
	      Goodbye ("%s Error setting volup output low: %d\n", Amp_Manager_Thread_Str, error) ;

	    current_volume++ ;
	    set_vol_lines_idle = 1 ;

	  squawk_volume:
	    if (Verbose & (_V_AMP | _V_AMP_MEDIA | _V_BLUETOOTH))
	      Print_Log (T_AMP, "%s Amp Tick, Volume=%d, Voffset=%d, current_volume=%u\n", Amp_Manager_Thread_Str, Volume, Vol_Offset, current_volume) ;
	  }

	  // The amp is ON and volume is at the target level - do we need to start radio or digital media?
	  else if (strcmp (url_path, "radio") == 0)
	  {
	    if ((error = gpioWrite (RADIO_ENABLE, 1)) != 0)
	      Goodbye ("%s Error turning on radio/relay output: %d\n", Amp_Manager_Thread_Str, error) ;

	    // The radio is now turned on, we can nix the url_path
	    url_path[0] = '\0' ;
	    amp_pause = 0 ;
	  }

	  // Do we need to start digital media?
	  else if (Add_Cmd[0] != NULL)
	  {
	    // If MPV is not currently running, start it now (and stop the kill timer)
	    Start_MPV (T_AMP, Amp_Manager_Thread_Str) ;

	    if (Mpv_Handle != NULL) // This should always be non-null, but we want to prevent a crash if somehow it is null
	    {
	      // Stop/disconnect any bluetooth devices that are currently connected
	      if (Bluetoothctl_PID > 0)
	      {
		Disconnect_Bluetooth_Devices = 1 ;

		// Issue a dummy return press, so that Bluetooth_Rx_Thread gets tripped
		if (write (PARENT_WRITE_FD, "\n", 1) != 1)
		  Goodbye ("%s cannot write to bluetooth pipe: %s\n", Setup_Thread_Str, strerror(errno)) ;
	      }

	      // append or replace the current stream
	      if (mpv_command (Mpv_Handle, (const char **)Add_Cmd) >= 0)
	      {
		restart_prevent_loop = 0 ;

		if (Verbose & (_V_AMP | _V_AMP_MEDIA))
		  Print_Log (T_AMP, "%s sent media command '%s'. Is_Alarm=%d, Index=%d\n", Amp_Manager_Thread_Str, url_path, Is_Alarm, index) ;

		if (MPV_State == MPV_PAUSED)
		{
		  // Exit paused mode
		  error = 0 ;

		  if ((Mpv_Handle != NULL) && (mpv_set_property (Mpv_Handle, "pause", MPV_FORMAT_FLAG, &error) < 0))
		    Goodbye ("%s Set pause off failed\n", Amp_Manager_Thread_Str) ;

		  if (Verbose & (_V_AMP | _V_AMP_MEDIA))
		    Print_Log (T_AMP, "%s unpaused\n", Amp_Manager_Thread_Str) ;
		}

		MPV_State = MPV_PLAYING ;

		// If mpv started ok and an index has been supplied, has an index jump been requested?
		if (index > 0)
		{
		  // Try delaying a little (it was failing this call without a delay.)
		  // I have not experimented with length of the delay.
		  Sleep_ns (HALF_SECOND) ;

		  if (Media_Command (T_AMP, Amp_Manager_Thread_Str, MEDIA_PLAYLIST_JUMP, index, NULL, (mqd_t)0, 0) < 0)
		  {
		    Index_Filename[0] = '\0' ;
		    Index_Filename_fileinfo.st_size = 0 ;
		  }
		  else if (Verbose & (_V_AMP | _V_AMP_MEDIA))
		    Print_Log (T_AMP, "%s jumped to index %d\n", Amp_Manager_Thread_Str, index) ;
		}
	      }
	    }

	    Add_Cmd[0] = NULL ;
	    index = -1 ;
	  }
	}

	break ;

	// - - - - - - - - - - - - - - - -
      default:
	Goodbye ("%s: Unknown message type=%d\n", Amp_Manager_Thread, amp_msg.type) ;
	break ;
    }
  }

  Print_Log (T_AMP, "%s unscheduled exit\n", Amp_Manager_Thread_Str) ;

  // Tidy up before exiting
  mq_close (Amp_Msgq) ;
  pthread_exit (NULL) ;
} // Amp_Manager_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Rewrite_Setup_File
//
// Updates the setup file with the current Volume setting
void
Rewrite_Setup_File (const int who)
{
  FILE		*f, *o ;
  int		fd, vol, media ;
  char		*p ;
  char		temp_name[TEMP_NAME_LEN] ;
  char		line_buffer [LOG_STRING_LEN] ;

  // We have renamed a series of setup files. The
  if ((f = Open_Setup_For_Reading()) == NULL)
  {
    // We have a problem - there is no setup file!
    Print_Log (who, "Cannot open setup file '%s': %s\n", Setup_Filename, strerror(errno)) ;
    return ;
  }

  // The file is opened for reading, and the filename that was read is in setup_version
  // We now need to open a temporary file for writing
  strcpy (temp_name, Temp_Name) ;

  if ((fd = mkstemp (temp_name)) < 0)
  {
    fclose (f) ;
    Print_Log (who, "Cannot save default settings. Cannot open '%s' for writing: %s\n", temp_name, strerror(errno)) ;
    return ;
  }

  if ((o = fdopen (fd, "w")) == NULL)
  {
    fclose (f) ;
    close (fd) ;
    Print_Log (who, "Cannot save default settings. Failed to open '%s' for writing: %s\n", temp_name, strerror(errno)) ;
    return ;
  }

  // We now have two opened files. Copy from one to the other, updating volume and default_media_duration settings
  for (vol=media=1 ; fgets (line_buffer, LOG_STRING_LEN, f) != NULL ;)
  {
    // We have the next line in line_buffer - the termination (newline) is also in the buffer
    // Does this line commence with the string VOLUME? Replace the VOLUME setting if so
    for (p=line_buffer ; (*p != '\0') && (isspace (*p)) ; p++) ;

    if (strstr(p, "VOLUME") == p) // VOLUME must be at column 0
    {
      fprintf (o, "VOLUME=%d\n", Volume) ;
      vol = 0 ;
      continue ;
    }

    // Is this line containing DEFAULT_MEDIA_DURATION? Replace if so
    if (strstr(line_buffer, "DEFAULT_MEDIA_DURATION") != NULL)
    {
      fprintf (o, "DEFAULT_MEDIA_DURATION=%d\n", Default_Media_Duration) ;
      media = 0 ;
      continue ;
    }

    // Just copy the line we read to the output file - its not being changed
    if (fputs(line_buffer, o) == EOF)
    {
      Print_Log (who, "Writing to '%s' failed: %s\n", temp_name, strerror(errno)) ;
      close (fd) ;
      fclose (f) ;
      remove (temp_name) ;
      return ;
    }
  }

  // If we have not yet written out the values, then write them now
  if (vol)
    fprintf (o, "VOLUME=%d\n", Volume) ;

  if (media)
    fprintf (o, "DEFAULT_MEDIA_DURATION=%d\n", Default_Media_Duration) ;

  // Ensure the file we have just written is readable!
  if (fchmod(fd, S_IWUSR|S_IRUSR|S_IWGRP|S_IRGRP|S_IWOTH|S_IROTH) != 0)
    Print_Log (who, "Setting mode for '%s' failed: %s\n", Setup_Filename, strerror(errno)) ;

  fclose (o) ;
  fclose (f) ;
  Rename_Backups (who, temp_name, Setup_Filename) ;
} // Rewrite_Setup_File
///////////////////////////////////////////////////////////////////////////////////////////
// Read_Alarm_Details
// Reads the specified alarm details from the alarm file.

void
Read_Alarm_Details (const int who, const int index, int *hours, int *minutes, int *suspended, int *duration, int *init_vol_offset, int *targ_vol_offset, int *is_defined)
{
  FILE		*f ;
  int		file_size, line, valid_lines, n, offset, fd, rewrite_counter ;
  time_t	file_time ;
  char		*file_string ;
  char		format_str [SETUP_FIELD_LEN] ;
  char		susp_str [SETUP_FIELD_LEN] ;
  char		temp_name[TEMP_NAME_LEN] ;

  // Return default values if real values cannot be found
  *hours = Default_Alarm_Hour ;
  *minutes = Default_Alarm_Minute ;
  *suspended = 1 ;
  *duration = Default_Alarm_Duration ;
  *init_vol_offset = 0 ;
  *targ_vol_offset = 0 ;
  *is_defined = 0 ;

  if ((index < 1) || (index > NUM_ALARM_BUTTONS))
    return ;

  // Read the alarm file into a string. The for loop is just in case the read fails. If the read passes, the for loop will exit
  for (rewrite_counter=valid_lines=0 ; Read_Alarm_File_Into_String (who, &file_size, &file_string, NULL, &file_time) < 0 ; rewrite_counter++)
  {
    // Something went wrong. We didn't find the requested alarm for one of several reasons.
    if (rewrite_counter)
    {
      if (Verbose & (_V_SETUP | _V_ALARM))
	Print_Log (who, "Failed to reread alarm settings after file rewrite\n") ;

      if (file_string != NULL)
	free (file_string) ;

      return ;
    }

    // We need to rewrite the alarm file.
    // We will base it and any earlier alarms for which there is no data upon defaults.
    strcpy (temp_name, Temp_Name) ;

    if (Verbose & (_V_SETUP | _V_ALARM))
      Print_Log (who, "Rewriting new alarm file into %s\n", temp_name, strerror(errno)) ;

    // Open a temporary file for writing
    if ((fd = mkstemp (temp_name)) < 0)
    {
      Print_Log (who, "Cannot open '%s' for writing: %s\n", temp_name, strerror(errno)) ;

      if (file_string != NULL)
	free (file_string) ;

      return ;
    }

    if ((f = fdopen (fd, "w")) == NULL)
    {
      close (fd) ;
      Print_Log (who, "Failed to open '%s' for writing: %s\n", temp_name, strerror(errno)) ;

      if (file_string != NULL)
	free (file_string) ;

      return ;
    }

    // Output the file as we know it
    if ((file_string != NULL) && (file_string[0] != '\0'))
      fputs (file_string, f) ;

    else // Output the header line. This text should match equivalent in alarms.cgi in the web pages
      fputs (" Alarm Time,Suspended,Alarm Duration,Volume Offset,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday,Stream\n", f) ;

    // We now need to output default suspended alarms up to the requested index
    for (; valid_lines < index ; valid_lines++)
    {
      if (*init_vol_offset == *targ_vol_offset)
	fprintf (f, "%02d:%02d,_SUSPENDED_,%d,%d,Y,Y,Y,Y,Y,Y,Y,%s\n", *hours, *minutes, *duration, *init_vol_offset, Default_Stream_Or_File) ;
      else
	fprintf (f, "%02d:%02d,_SUSPENDED_,%d,%d..%d,Y,Y,Y,Y,Y,Y,Y,%s\n", *hours, *minutes, *duration, *init_vol_offset, *targ_vol_offset, Default_Stream_Or_File) ;
    }

    fclose (f) ;
    Rename_Backups (who, temp_name, Alarm_Filename) ; // create a backup of what's there before klobbering it with new file

    if (file_string != NULL)
      free (file_string) ;
  } // loop if we need to read the file again

  // At this point, we have read the contents of the alarm file into a string
  // Prepare the scanf format string: <HH:MM>, 'suspended/active', 'duration', 'volume offset'
  sprintf (format_str, " %%02d:%%02hd , %%%d[^ ,] , %%d , %%%d[0-9- .]", SETUP_FIELD_LEN, TIME_STRING_LEN) ;

  // Ignore the first line in the file, by setting offset to 0.
  // The first line contains the column headings. Next_Line() returns offset to the next line of text
  for (offset=0, line=1 ; (offset = Next_Line (file_string, file_size, offset, NULL)) > 0 ; line++, valid_lines++)
  {
    // We have the byte offset of the target line in 'offset'
    if ((n=sscanf(&file_string[offset], format_str, hours, minutes, &susp_str, duration, temp_name)) != 5)
    {
      // Couldn't read the target line. Something wrong!
      Print_Log (who, "read alarm details got %d fields from alarm definition line %d of '%s'\n", n, line+1, Alarm_Filename) ;
      break ;
    }

    if ((n=sscanf(temp_name, "%d %*[.] %d", init_vol_offset, targ_vol_offset)) == 0)
    {
      Print_Log (who, "read alarm details encountered volume offset syntax problem at line %d of '%s': '%s'\n", line+1, Alarm_Filename, temp_name) ;
      break ;
    }

    if (n == 1)
      *targ_vol_offset = *init_vol_offset ;

    // Perform checks on the values
    // The sscanf has already determined that the characters in the day fields are Y, N or a number
    if ((*hours < 0) || (*hours > 23) || (*minutes < 0) || (*minutes > 59) || (*duration <= 0) || (*duration > MAX_DURATION) ||
	(*init_vol_offset < -MAX_VOL_OFFSET) || (*init_vol_offset > MAX_VOL_OFFSET) ||
	(*targ_vol_offset < -MAX_VOL_OFFSET) || (*targ_vol_offset > MAX_VOL_OFFSET))
    {
      // Values are out of range. Something wrong!
      Print_Log (who, "read alarm details found out of range value(s) in alarm definition line %d of '%s'\n", line+1, Alarm_Filename) ;
      break ;
    }

    // Is this record suspended? Ignore if so
    if (strcasecmp(susp_str, "_ACTIVE_") == 0)
      *suspended = 0 ;

    else if (strcasecmp(susp_str, "_SUSPENDED_") == 0)
      *suspended = 1 ;

    else
    {
      // Cannot determine if active or suspended. Something wrong!
      Print_Log (who, "read alarm details failed to recognise if alarm is active or suspended at line %d of '%s'\n", line+1, Alarm_Filename) ;
      break ;
    }

    // Is this the line we are looking for?
    if (line >= index)
    {
      // Yes - this is the line we are looking for. Details are now in the variables and we can return.
      // We no longer need the buffered file - we can release the memory
      free (file_string) ;
      *is_defined = 1 ; // We read a valid alarm definition
      return ;
    }
  }

  // We failed to read the required alarm index. Set to defaults and use the defaults
  *hours = Default_Alarm_Hour ;
  *minutes = Default_Alarm_Minute ;
  *suspended = 1 ;
  *duration = Default_Alarm_Duration ;
  *init_vol_offset = 0 ;
  *targ_vol_offset = 0 ;

  if (file_string != NULL)
    free (file_string) ;
} // Read_Alarm_Details
///////////////////////////////////////////////////////////////////////////////////////////
// Rewrite_Alarm_File
// Used for writing updated information to an alarm setting
void
Rewrite_Alarm_File (const int who, const int index, int hours, int minutes, int suspended, int duration, int init_vol_offset, int targ_vol_offset)
{
  FILE		*o ;
  int		fd, line, n, offset, file_size ;
  char		eol_char, *sus_string, *file_string, *end_of_line ;
  char		format_str [SETUP_FIELD_LEN] ;
  char		remainder [LOG_STRING_LEN] ;
  char		temp_name[TEMP_NAME_LEN] ;

  if ((index < 1) || (index > NUM_ALARM_BUTTONS))
    return ;

  if (Read_Alarm_File_Into_String (who, &file_size, &file_string, NULL, NULL) < 0)
  {
    Print_Log (who, "Cannot rewrite alarm file\n") ;
    return ;
  }

  // Open a temporary file for writing
  strcpy (temp_name, Temp_Name) ;

  if ((fd = mkstemp (temp_name)) < 0)
  {
    Print_Log (who, "Cannot rewrite alarm file. Cannot open '%s' for writing: %s\n", temp_name, strerror(errno)) ;

    if (file_string != NULL)
      free (file_string) ;

    return ;
  }

  if ((o = fdopen (fd, "w")) == NULL)
  {
    Print_Log (who, "Cannot rewrite alarm file. Failed to fdopen '%s': %s\n", temp_name, strerror(errno)) ;
    remove (temp_name) ;

    if (file_string != NULL)
      free (file_string) ;

    return ;
  }

  if (Verbose & (_V_SETUP | _V_ALARM))
    Print_Log (who, "Updating alarm %d to %02d:%02d. Suspended=%d, Duration=%d, Vol offset=%d..%d\n",
	       index, hours, minutes, suspended, duration, init_vol_offset, targ_vol_offset) ;

  // Prepare the scanf format string: <HH:MM>, 'suspended/active', 'duration', 'volume offset'
  sprintf (format_str, " %%*02d:%%*02d , %%*[^ ,] , %%*d , %%*d, %%%d[^\n\t\r\f\v]", SETUP_FIELD_LEN) ;
  sus_string = (suspended) ? "_SUSPENDED_" : "_ACTIVE_" ;
  end_of_line = NULL ; // to shut up the compiler warning

  // Copy from previous alarm settings to the new temp file, rewriting the alarm that needs to be replaced.
  // Next_Line() returns offset to the beginning of the next line of text
  // end_of_line is pointer to the terminating newline
  for (offset=-1,line=0 ; (offset = Next_Line (file_string, file_size, offset, &end_of_line)) >= 0 ; line++)
  {
    // Is this the line we are looking for?
    if (line != index)
    {
      // This is not the line - just copy up until the next linefeed verbatim
      eol_char = *end_of_line ;
      *end_of_line = '\0' ;
      fprintf (o, "%s\n", &file_string[offset]) ;
      *end_of_line = eol_char ;

      // loop to read the next line until target found / end of file reached
      continue ;
    }

    // This is the line which needs to be rewritten. The line commences at the position 'offset'
    if ((n=sscanf(&file_string[offset], format_str, &remainder)) != 1)
    {
      // Couldn't read the target line. Something wrong!
      Print_Log (who, "rewrite alarm file failed to scan alarm %d of '%s'\n", index, Alarm_Filename) ;
      break ;
    }

    // We can now rewrite the line
    if (init_vol_offset == targ_vol_offset)
      fprintf (o, "%02d:%02d,%s,%d,%d,%s\n", hours, minutes, sus_string, duration, init_vol_offset, remainder) ;
    else
      fprintf (o, "%02d:%02d,%s,%d,%d..%d,%s\n", hours, minutes, sus_string, duration, init_vol_offset, targ_vol_offset, remainder) ;
  }

  if ((offset < 0) || (line <= index))
  {
    // If we have not yet written the new alarm to the file, add additional lines to the alarm file to
    // capture the new setting. This occurs when the existing file is too short for the alarm button that has been operated.
    for (; line <= index ; line++)
    {
      if (line == index)
      {
	if (init_vol_offset == targ_vol_offset)
	  fprintf (o, "%02d:%02d,%s,%d,%d,Y,Y,Y,Y,Y,Y,Y,\"%s\"\n", hours, minutes, sus_string, duration, init_vol_offset, Default_Stream_Or_File) ;
	else
	  fprintf (o, "%02d:%02d,%s,%d,%d..%d,Y,Y,Y,Y,Y,Y,Y,\"%s\"\n", hours, minutes, sus_string, duration, init_vol_offset, targ_vol_offset, Default_Stream_Or_File) ;
      }
      else // Write a suspended line containing defaults
	fprintf (o, "%02d:%02d,_SUSPENDED_,%d,0,Y,Y,Y,Y,Y,Y,Y,\"%s\"\n", Default_Alarm_Hour, Default_Alarm_Minute, Default_Alarm_Duration, Default_Stream_Or_File) ;
    }
  }

  // The temporary file has been written. We can close it and then rename it
  fclose (o) ;
  Rename_Backups (who, temp_name, Alarm_Filename) ; // create a backup of what's there before klobbering it with new file
  Print_Log (who, "Rewrote alarm file. Alarm lines=%d\n", line) ;

  if (file_string != NULL)
    free (file_string) ;
} // Rewrite_Alarm_File
///////////////////////////////////////////////////////////////////////////////////////////
// Send_Network_Packet()
//
// Call this routine to send a packet over the network. This centralised routine has
// a special purpose - if the sendto() call fails (to send the packet), then a timer gets
// started which will eventually) cause the clock to be rebooted. If the network is restored
// before the counter expires - all good. The most common cause for a send failure is
// that the wifi access point has gone down. In this case, we don't want the clock immediately
// rebooting. On the other hand, the OS may have locked up. In this case we want it to reboot.

void
Send_Network_Packet (const int who, const char *msg, const int n)
{
  if (sendto (Tx_Socket, msg, n, MSG_CONFIRM, (struct sockaddr *)&Tx_Address, sizeof(Tx_Address)) != n)
  {
    if (Network_Failure_Downcount <= 0)
    {
      // Start the network failure downcount
      Network_Failure_Downcount = NET_FAILED_DOWNCOUNT ;
      Print_Log (who, "sendto() failed to send %d byte network message (%s). Starting Network Failure Timer\n", n, strerror(errno)) ;
    }
  }
  else if (Network_Failure_Downcount > 0)
  {
    // Stop the network failure downcount
    Network_Failure_Downcount = 0 ;
    Print_Log (who, "Network connection has been restored.\n") ;
  }
} // Send_Network_Packet
///////////////////////////////////////////////////////////////////////////////////////////
// Read_Buttons()
//
// The reading is returned as a bitmap in bits 0..8
// The hardware returns 0 if pressed and 1 if button opened.
// Read_Buttons will invert the state of the button. 1=pressed and 0=opened

uint16_t
Read_Buttons (const int who)
{
  int			pb1, pb2, pb3, pb4, pb5, pb6, pb7, pb8, pb9, n ;
  char			button_msg[PACKET_SIZE] ;

  // If we are a clustered clock that is being remotely controlled, then return the remotely nominated button map
  // (and if so, the main clock doesn't scan its own physical buttons at all)
  if ( ((This_Clock_Type == 'C') || (This_Clock_Type == 'Z')) &&
       (Button_Downcount > 0) && (Button_Control_Token != 0))
  {
    // This downcount ensures that any remotely controlled buttons revert if the host goes offline
    if (--Button_Downcount == 0)
      Button_Control_Token = 0 ;

    return Remote_Buttons ; // This variable is set in the Multicast_Rx_Thread based on the remote system's button messages
  }

  pb1 = pb2 = pb3 = pb4 = pb5 = pb6 = pb7 = pb8 = pb9 = 0 ;

  if (((pb1 = gpioRead (TOG1a)) == PI_BAD_GPIO) ||
      ((pb2 = gpioRead (TOG1b)) == PI_BAD_GPIO) ||
      ((pb3 = gpioRead (PB1)) == PI_BAD_GPIO) ||
      ((pb4 = gpioRead (PB2)) == PI_BAD_GPIO) ||
      ((pb5 = gpioRead (PB3)) == PI_BAD_GPIO) ||
      ((pb6 = gpioRead (TOG2a)) == PI_BAD_GPIO) ||
      ((pb7 = gpioRead (TOG2b)) == PI_BAD_GPIO) ||
      ((pb8 = gpioRead (TOG3a)) == PI_BAD_GPIO) ||
      ((pb9 = gpioRead (TOG3b)) == PI_BAD_GPIO))
    Goodbye ("Error reading buttons: 1=%d 2=%d 3=%d 4=%d 5=%d 6=%d 7=%d 8=%d 9=%d\n", pb1, pb2, pb3, pb4, pb5, pb6, pb7, pb8, pb9) ;

  // Remember, reading 0 means button pressed. Reading 1 means button not pressed.
  // Convert the individual buttons into a bit mask
  pb1 ^= 0x01 ; // Invert the state of bit 0

  if (pb2 == 0)
    pb1 |= 0x02 ;

  if (pb3 == 0)
    pb1 |= 0x04 ;

  if (pb4 == 0)
    pb1 |= 0x08 ;

  if (pb5 == 0)
    pb1 |= 0x10 ;

  if (pb6 == 0)
    pb1 |= 0x20 ;

  if (pb7 == 0)
    pb1 |= 0x40 ;

  if (pb8 == 0)
    pb1 |= 0x80 ;

  if (pb9 == 0)
    pb1 |= 0x100 ;

  // If we are a clustered clock, send UDP messages with button presses
  if ((This_Clock_Type == 'A') || (This_Clock_Type == 'C') || (This_Clock_Type == 'Z'))
  {
    if (pb1 != 0)
    {
      if (Button_Control_Token == 0) // Commencing a remote control mode?
	// Choose a non-zero token between 1 and RAND_MAX
	for (; Button_Control_Token == 0 ; Button_Control_Token=(uint32_t)rand()) ;

      n = snprintf (button_msg, PACKET_SIZE, "B,%c,%u,%03x", This_Clock_Type, Button_Control_Token, pb1) ; // Button press
      goto send_button_message ;
    }
    else if (Button_Control_Token != 0) // Exiting remote control mode?
    {
      // Send our final gasp indicating buttons released
      n = snprintf (button_msg, PACKET_SIZE, "B,%c,%u,0", This_Clock_Type, Button_Control_Token) ;
      Button_Control_Token = 0 ;

    send_button_message:
      if (Verbose & _V_BUTTON)
	Print_Log (who, "Sending '%c' UDP Button message: %03hx\n", This_Clock_Type, pb1) ;

      Send_Network_Packet (who, button_msg, n) ;
    }
  }

  return (uint16_t)pb1 ;
} // Read_Buttons
///////////////////////////////////////////////////////////////////////////////////////////
// Button_Manager_Thread
//
// A thread whose job is to manage and respond to the push buttons. It is a surprisingly small amount of code given the apparent
// complexity of the various button press combinations + repeating and holding.
//
// There are THREE 'action' buttons which are used either on their own or in conjunction with other buttons. The 'action' buttons
// are used to CHANGE a setting and are B_PLUS, B_MINUS and B_STOP_TOGGLE.
//
// Action Buttons:
//   B_PLUS		up or +
//   B_MINUS		down or -
//   B_STOP_TOGGLE	Snooze / Stop / Toggle
//
// The remaining SIX buttons are 'information / selection' buttons. Used on their own or in combination with each other, these
// will display information about an alarm time, or duration etc, but will NOT CHANGE the current setting or value.
//
// Information / Selection  Buttons:
//   B_DURATION		Duration
//   B_MEDIA		Media functions
//   B_ALARM1		Alarm 1
//   B_ALARM2		Alarm 2
//   B_ALARM3		Alarm 3
//   B_ALARM4		Alarm 4
//
// To CHANGE a value, you optionally press a combination of the 'selection' buttons, THEN press and optionally hold one
// of the 'action' buttons.
//
// Any alarm button on its own will display the time for the associated alarm. The time will flash if the alarm is suspended.
//   Pressing B_ALARMn + B_STOP_TOGGLE will toggle it between suspended and active.
//   Pressing B_ALARMn + B_DURATION will display the duration of the alarm.
//   Pressing B_ALARMn + B_MEDIA will display the volume offset for the alarm.
//   Pressing B_ALARMn + (B_PLUS or B_MINUS) will increment or decrement the alarm time. Press and hold +/- for auto-repeat.
//   Pressing B_ALARMn + B_DURATION + (B_PLUS or B_MINUS) will increment or decrement the duration of the alarm time.
//   Pressing B_ALARMn + B_MEDIA + (B_PLUS or B_MINUS) will increment or decrement the volume offset.
//
// If no media playing
//   Pressing B_DURATION on its own will display the DEFAULT_MEDIA_DURATION
//   Releasing B_DURATION will then commence playing the default media stream/file.
//   Pressing B_DURATION + (B_PLUS or B_MINUS) will increment or decrement the DEFAULT_MEDIA_DURATION and save the change. (+/- will auto repeat)
//
// If media is currently playing
//   Pressing (B_PLUS or B_MINUS) on its own will increment or decrement the volume.
//   Pressing B_STOP_TOGGLE on its own will toggle snooze on and off. A long press will cancel the alarm or stop the stream.
//
//   Pressing B_DURATION on its own will display the minutes of snooze remaining (if snoozing) or minutes of play remaining
//   Pressing B_DURATION + (B_PLUS or B_MINUS) will increment or decrement the remaining playing time. (+/- will auto repeat)
//
//   Pressing B_DURATION + B_STOP_TOGGLE will display the minutes of snooze remaining (or flash 0 if snooze not active)
//   Pressing B_DURATION + B_STOP_TOGGLE + (B_PLUS or B_MINUS) will increment or decrement the remaining snooze time. (+/- will auto repeat)
//
// If B_MEDIA button is pressed (without an alarm button)
//   Pressing B_MEDIA + (B_PLUS or B_MINUS) will skip back or forwards within the current track.
//   Pressing B_MEDIA + (Long pressing B_PLUS or B_MINUS) will skip to next track or restart the current track.
//   Pressing B_MEDIA + (Very long pressing B_MINUS) will skip to previous track.
//   Pressing B_MEDIA + B_STOP_TOGGLE will initiate the default media stream. Additional presses jump to fallback and then radio.
//
// If the B_ALARM_1 and B_ALARM_3 buttons OR B_ALARM_2 and B_ALARM_4 buttons are pressed together, the clock will enter bluetooth pairing
// mode for the timeout that has been configured in the bluez configuration. As well as initiating bluetooth pairing, the clock will ALSO
// display the time remaining until the next alarm.
//
// If the B_ALARM_1 and B_ALARM_4 buttons OR B_ALARM_2 and B_ALARM_3 buttons are pressed together, the current IP address will be displayed
// one number at a time.

static const char Button_Manager_Thread_Str[] = "Button Manager Thread" ;

void
display_alarm_time (mqd_t disp_driver_msgq, uint8_t flash_on, uint16_t H, uint16_t M, uint16_t S, uint8_t D)
{
  DD_MSG_BUF		dd_msg ;
  uint32_t		secs ;

  // Pre-blank each pair of digits by setting to a number >99 but <=255
  dd_msg.hours = dd_msg.minutes = dd_msg.seconds = 0xff ; // Default is blank all digits
  dd_msg.dots = D ;

  // When minutes==ffff, we are exiting the alternate display mode - set all parameters to 0xff and send disp driver message
  if (M == 0xffff)
    dd_msg.type = DISP_UNLOCKED_COLON ;

  else if (M == 0xfffe)
  {
    // Seconds until the next alarm is in (Hours<<16 |  Seconds) (a 32 bit number)
    dd_msg.type = DISP_NUMBERS ;
    secs = (H << 16) | S ;

    if (secs == 0xffffffff)
      dd_msg.seconds = 0x00 ; // No alarm has been set - just display 00 in seconds digits

    else if (secs >= (100 * 3600)) // the longest time we can display is 99H 59M 59S
    {
      dd_msg.hours = 99 ;
      dd_msg.minutes = 59 ;
      dd_msg.seconds = 59 ;
    }
    else
    {
      dd_msg.hours = secs / 3600 ; // number of hours until the next alarm
      secs -= (dd_msg.hours) * 3600 ;

      dd_msg.minutes = secs / 60 ; // number of minutes until the next alarm
      secs -= (dd_msg.minutes) * 60 ;

      dd_msg.seconds = secs ;
    }
  }

  else if (H > 23)
  {
    // We are displaying a duration which is in the M variable
    dd_msg.type = DISP_LOCKED_NO_COLON ;

    if (flash_on && (M <= 9999))
    {
      if (M <= 99)
      {
	dd_msg.hours = 0xff ; // Blank the hours digits if the number is two digits or smaller
	dd_msg.minutes = M ;
      }
      else
      {
	dd_msg.hours = M / 100 ;
	dd_msg.minutes = M % 100 ;
      }
    }

    dd_msg.seconds = S ; // this is the alarm number
  }

  else
  {
    // We are displaying a time. Blank the colon if the LEDs are disabled. Otherwise enable the colon
    if (flash_on)
    {
      dd_msg.type = DISP_LOCKED_COLON ;
      dd_msg.hours = H ;
      dd_msg.minutes = M ;
    }
    else
      dd_msg.type = DISP_LOCKED_NO_COLON ;

    dd_msg.seconds = S ; // this is the alarm number
  }

  // The initial message type will set HMS only. The value in dd_msg.dots is ignored unless its a DISP_DOTS message.
  // We will send a follow up message if the dots are being changed.
  if (mq_send (disp_driver_msgq, (char *)&dd_msg, sizeof(dd_msg), 0) < 0)
    Goodbye ("%s failed to send display msg: %s\n", Button_Manager_Thread_Str, strerror(errno)) ;

  if (D != 0)
  {
    // DISP_DOTS messages can only affect bits 0-5, Bits 6 and 7 are masked out by the display driver.
    dd_msg.type = DISP_DOTS ;

    if (mq_send (disp_driver_msgq, (char *)&dd_msg, sizeof(dd_msg), 0) < 0)
      Goodbye ("%s failed to send display msg: %s\n", Button_Manager_Thread_Str, strerror(errno)) ;
  }
} // display_alarm_time
///////////////////////////////////////////////////////////////////////////////////////////
int
initiate_media_playing (int *dev_index, MEDIA_MSG_BUF *media_msg)
{
  int			media_msg_len ;

  // Start playing media stream. Rotate between Default_Stream_Or_File[], Fallback_Alarm_File[] and 'radio'
  media_msg->type = MEDIA_ADD ;
  media_msg_len = 0 ;

  // there are three potential options for dev_index: 0..2
  if ((*dev_index == 0) && (media_msg->url_path[0] == '\0'))
  {
    if (Default_Stream_Or_File[0] != '\0')
      media_msg_len = MEDIA_MSG_LEN + 1 + snprintf (media_msg->url_path, SETUP_FIELD_LEN, "%s", Default_Stream_Or_File) ; // copy file path or URL

    *dev_index = 1 ; // rotate the next device to the next one up
  }

  if ((*dev_index == 1) && (media_msg->url_path[0] == '\0'))
  {
    if (Fallback_Alarm_File[0] != '\0')
      media_msg_len = MEDIA_MSG_LEN + 1 + snprintf (media_msg->url_path, SETUP_FIELD_LEN, "%s", Fallback_Alarm_File) ; // copy file path or URL

    *dev_index = 2 ; // rotate the next device to the next one up
  }

  if ((*dev_index >= 2) && (media_msg->url_path[0] == '\0'))
  {
    media_msg_len = MEDIA_MSG_LEN + 1 + snprintf (media_msg->url_path, SETUP_FIELD_LEN, "radio") ; // copy command to enable the radio
    *dev_index = 0 ; // rotate the next device back to the beginning
  }

  if (Verbose & _V_BUTTON)
    Print_Log (T_BUTTON, "%s: MEDIA NEXT DEVICE=%d\n", Button_Manager_Thread_Str, *dev_index) ;

  return media_msg_len ;
}
///////////////////////////////////////////////////////////////////////////////////////////
void *
Button_Manager_Thread (void *xarg)
{
  uint16_t		this_buttons, last_buttons, new_buttons, changed_buttons, held, n, m ;
  uint8_t		dots, flash, launch_media, number ;
  int			msg, setup_file_timer, repeat_count, setup_changed, media_msg_len, dev_index, is_defined ;
  int			alarm_index, alarm_suspended, alarm_hours, alarm_minutes, alarm_duration, init_vol_offset, targ_vol_offset, increment ;
  DD_MSG_BUF		dd_msg ;
  MEDIA_MSG_BUF		media_msg ;
  mqd_t			media_msgq ;
  mqd_t			disp_driver_msgq ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_BUTTON)
    Print_Log (T_BUTTON, "Entering %s\n", Button_Manager_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  // We are the recipient of Button_Msgq. Open it now for reading before any other threads try to open it for writing.
  Button_Msgq = Open_Msg_Queue_Rx (T_BUTTON, Button_Msgq_name, 5, sizeof(msg)) ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  media_msgq = Open_Msg_Queue_Tx (T_BUTTON, Media_Msgq_name, 0) ;
  disp_driver_msgq = Open_Msg_Queue_Tx (T_BUTTON, DD_Msgq_name, 0) ;
  last_buttons = held = 0 ;
  launch_media = 0 ;
  setup_file_timer = -1 ;
  Display_IP = 0 ;
  dev_index = repeat_count = setup_changed = alarm_index = alarm_suspended = alarm_hours = alarm_minutes = alarm_duration = init_vol_offset = targ_vol_offset = is_defined = 0 ;

  if (Verbose & _V_BUTTON)
    Print_Log (T_BUTTON, "%s initialised\n", Button_Manager_Thread_Str) ;

  // Messages to the media thread contain the linux command text for launching a new media stream
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // This is an infinte loop - wait for messages and then do what the message requests.
    // Different messages to (a) start a new stream, (b) pause a stream, (c) stop a stream, (d) check on the status of a stream
    if (Wait_For_Next_Message (T_BUTTON, Button_Msgq, (void *)&msg, sizeof(msg), sizeof(msg), Button_Manager_Thread_Str) != sizeof(msg))
      continue ;

    // we are only interested in bit 0 of flash. When flashing, 1 means blank the display. 0 means illuminate the display
    flash = (uint8_t)(repeat_count / BUTTON_REPEAT_HALF_WAIT) & 0x01 ;
    // - - - - - - - - - - - - - - - -
    // setup_file_timer is a downcounter that delays rewriting the setup file. This lets the user make multiple changes
    // over a short timeframe without rewriting the setup file each individual change.
    if ((setup_file_timer >= 0) && (setup_file_timer-- == 0))
    {
      // The setup file timer has expired, and the value is now -1 (timer off)
      // Amend setup file using the current volume level as the default volume level

      if (Verbose & _V_BUTTON)
	Print_Log (T_BUTTON, "%s: setup_file_timer expired\n", Button_Manager_Thread_Str) ;

      Rewrite_Setup_File (T_BUTTON) ;
    }
    // - - - - - - - - - - - - - - - -
    // Display_IP is non-zero if we are in the process of displaying an IP address
    // We will decrement it once per button scan, which nominally happens 10 times per second.
    // Each number in the four different IPv4 address numbers will be displayed for 4*BUTTON_SCAN_FREQ
    if (Display_IP > 0)
    {
      Display_IP-- ;

      // Initialise digits to blank by prefilling with a number >99 but <=255
      dd_msg.hours = dd_msg.minutes = dd_msg.seconds = 0xff ; // Default is blank all digits
      dd_msg.dots = 0x00 ; // Default is all dots off
      dd_msg.type = DISP_RAW_NUMBERS ; // We are going to ask the display driver to display a set of precalculated BCD

      // Which of the four IP digits are we doing?
      m = 3 - (Display_IP / (BUTTON_SCAN_FREQ * 4)) ; // m will be one of 0, 1, 2, 3
      number = ((uint8_t *)&My_IPv4_sin_addr)[m] ;

      // There are three possibilities... we are either going to display 1, 2 or 3 numbers
      if (number > 99)
      {
	// three digits
	dd_msg.hours = Num_To_BCD (number / 10, 0) ;
	dd_msg.minutes = Num_To_BCD ((number % 10) * 10, 0) | 0x0f ;
      }
      else if (number > 9)
      {
	// two digits
	dd_msg.hours = Num_To_BCD (number, 0) ;
      }
      else
      {
	// Single digit
	dd_msg.hours = Num_To_BCD (number * 10, 0) | 0x0f ;
      }

      dd_msg.dots = 0x20 >> m ; // Turn on the dot corresponding to the position in the address

      // The initial message type will set HMS only. The value in dd_msg.dots is ignored unless its a DISP_DOTS message.
      // We will send a follow up message if the dots are being changed.
      if (mq_send (disp_driver_msgq, (char *)&dd_msg, sizeof(dd_msg), 0) < 0)
	Goodbye ("%s failed to send display msg: %s\n", Button_Manager_Thread_Str, strerror(errno)) ;
    }
    // - - - - - - - - - - - - - - - -
    // Read the buttons - ignore (ie loop and wait for another message) if nothing pressed at the moment
    if ((this_buttons = Read_Buttons(T_BUTTON)) == 0)
    {
      // No buttons pressed
      if (last_buttons != 0)
      {
	display_alarm_time (disp_driver_msgq, 0, 0, 0xffff, 0, 0) ; // Exit display locked mode when buttons released

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s All buttons released. prev repeat_count=%d\n", Button_Manager_Thread_Str, repeat_count) ;
      }

      last_buttons = 0 ;
      held = 0 ;
      repeat_count = 0 ;

      // If we are editing an alarm, we can cease editing now that buttons are released
      if ((alarm_index >= 1) && (alarm_index <= NUM_ALARM_BUTTONS))
      {
	// Do we need to save the alarm details that have been edited? Only save if change was made
	if (setup_changed)
	{
	  Rewrite_Alarm_File (T_BUTTON, alarm_index, alarm_hours, alarm_minutes, alarm_suspended, alarm_duration, init_vol_offset, targ_vol_offset) ;

	  // Read and interpret the setup and alarm file. This will recalculate the file hash
	  Read_Setup_And_Alarms (T_BUTTON, 1) ;
	}

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s Leaving alarm edit. index=%d, time=%02d:%02d, duration=%d, vol_offset=%d..%d, suspended=%d, setup_changed=%d\n",
		     Button_Manager_Thread_Str, alarm_index, alarm_hours, alarm_minutes, alarm_duration, init_vol_offset, targ_vol_offset, alarm_suspended, setup_changed) ;

	// and exit editing mode
	alarm_index = 0 ;
	setup_changed = 0;
      }

      // All buttons have been released. If we are launching media, time to do that now that buttons are released.
      if (launch_media)
      {
	launch_media = 0 ;

	// Don't proceed if something turned on the amp after deciding to do it via button press
	if ((Amp_State == A_AMP_OFF) || (Amp_State == A_AMP_ON_BLUETOOTH))
	{
	  // Issue the command to play media
	  media_msg.type = MEDIA_ADD ;
	  media_msg.index = media_msg.volume = -1 ;
	  media_msg.init_vol_offset = -200 ;
	  media_msg.targ_vol_offset = -200 ;
	  media_msg.seconds = Default_Media_Duration * 60 ;
	  media_msg.responseQ = 2 ; // a dummy number that will cause alarm clock to crash if there's a bug. #2 will appear in debug log
	  media_msg.url_path[0] = '\0' ;
	  dev_index = 0 ; // Attempt to play the default stream or file, then fallback to either the fallback or radio
	  media_msg_len = initiate_media_playing (&dev_index, &media_msg) ;

	  // If media_msg.type is not MEDIA_NOTHING, there's a message waiting to be sent
	  if (mq_send (media_msgq, (char *)&media_msg, media_msg_len, 0) < 0)
	    Goodbye ("%s failed to send media msg %hu: %s\n", Button_Manager_Thread_Str, media_msg.type, strerror(errno)) ;
	}
      }

      continue ;
    }
    // - - - - - - - - - - - - - - - -
    // To reach here, at least one button is pressed.
    // If the pressed button configuration *has not changed*, increment the repeat timer, otherwise reset it.
    // The longer the *same* button combination is pressed, the higher the repeat counter increments.
    if (last_buttons == this_buttons)
      repeat_count++ ;
    else
      repeat_count = 0 ;

    // Now work out what is freshly pressed and what is going to be 'repressed' with a repeat
    //
    // this_buttons: bitmask of the buttons that are currently pressed down
    // last_buttons: bitmask of the previous value of 'this_buttons'
    // changed_buttons: bitmask of the buttons that have changed
    // new_buttons: bitmask of the FRESHLY PRESSED (or repeated) buttons (ie changed_buttons that are FRESHLY PRESSED)
    //
    // 1. What has changed since last time and is currently pressed (ie it has been pressed since last time)?
    //    (last_buttons ^ this_buttons) is what has changed. Anding with this_buttons keeps only those STILL pressed.
    changed_buttons = (last_buttons ^ this_buttons) ;
    new_buttons = changed_buttons & this_buttons ;

    // 2. What has been pressed for the duration of the repeat timer? (only repeat the + and - buttons, and the sleep/stop/toggle button)
    if ((this_buttons & (B_PLUS | B_MINUS | B_STOP_TOGGLE)) && (repeat_count >= BUTTON_REPEAT_WAIT))
      new_buttons |= this_buttons ; // make it appear as if plus or minus has been pressed again

    // At this point, we have a new press (or a repeat that has been activated)
    if ((Verbose & _V_BUTTON) && changed_buttons)
      Print_Log (T_BUTTON, "%s Last=0x%03x, This=0x%03x, Chng=0x%03x, New=0x%03x, Rpt_Cnt=%d, MPV=%hu, Vol=%d, Voffset=%d, Setup_Timer=%d, Alm_Ind=%d, PlayTime=%d, SnoozeTime=%d\n",
		 Button_Manager_Thread_Str, last_buttons, this_buttons, changed_buttons, new_buttons, repeat_count, MPV_State,
		 Volume, Vol_Offset, setup_file_timer, alarm_index, Seconds_Of_Play_Remaining, Seconds_Of_Snooze_Remaining) ;

    // Remember the last combination of buttons that have been pressed for the next iteration through the loop
    last_buttons = this_buttons ;
    // - - - - - - - - - - - - - - - -
    // Check for simultaneous pressing of two alarm buttons (entering Bluetooth pairing mode)
    if (((this_buttons & B_ALARM1) && (this_buttons & B_ALARM3)) || ((this_buttons & B_ALARM2) && (this_buttons & B_ALARM4)))
    {
      // Pairing combination has been pressed. Is the bluetooth daemon running?
      if ((Bluetoothctl_PID > 0) && (Bluetooth_Init_Completed > 0) && ((this_buttons & B_REMOTE) == 0) && this_buttons)
      {
	// We only want to enter pairing mode ONCE - so only if a button is newly pressed
	if (new_buttons & (B_ALARM1 | B_ALARM2 | B_ALARM3 | B_ALARM4))
	{
	  Initiate_Bluetooth_Pairing = 1 ;

	  // Issue a dummy return press, so that Bluetooth_Rx_Thread gets tripped
	  if (write (PARENT_WRITE_FD, "\n", 1) != 1)
	    Goodbye ("%s cannot write to bluetooth pipe: %s\n", Button_Manager_Thread_Str, strerror(errno)) ;
	}
      }

      // The following will be tested each time through the loop
      if (Seconds_To_Next_Alarm > 0)
      {
	n = ((uint32_t)Seconds_To_Next_Alarm & 0x0fff0000) >> 16 ;
	m = (uint32_t)Seconds_To_Next_Alarm & 0x0000ffff ;
	display_alarm_time (disp_driver_msgq, 0, n, 0xfffe, m, 0x00) ;
      }
      else
	display_alarm_time (disp_driver_msgq, 0, 0xffff, 0xfffe, 0xffff, 0x2c) ;

      continue ;
    }
    // - - - - - - - - - - - - - - - -
    // Check for simultaneous pressing of two alarm buttons (display IP address)
    if (((this_buttons & B_ALARM1) && (this_buttons & B_ALARM4)) || ((this_buttons & B_ALARM2) && (this_buttons & B_ALARM3)))
    {
      // The alternate dual button press combination has been used. Display the IP address

      // We only want to reload the IP address ONCE - so only if a button is newly pressed
      // (Reload the hostname and IP address just in case DHCP has reissued a new address)
      if ((new_buttons & (B_ALARM1 | B_ALARM2 | B_ALARM3 | B_ALARM4)) || (Display_IP == 0))
      {
	if (determine_hostname_and_ip_address (T_BUTTON) != EXIT_SUCCESS)
	  break ; // ABORT! There was an error!

	// Enter the IP address display mode
	Display_IP = BUTTON_SCAN_FREQ * 4 * 4 ; // four numbers, for 4* button scan frequency
      }

      continue ;
    }
    // - - - - - - - - - - - - - - - -
    // Are we currently editing an alarm? If not, do check to see if we should enter edit mode
    // Don't allow remote editing of alarms
    if (((this_buttons & B_REMOTE) == 0) && ((alarm_index < 1) || (alarm_index > NUM_ALARM_BUTTONS)))
    {
      // Check if we need to enter edit mode by checking for one of the alarm buttons
      alarm_index = 0 ;

      if (this_buttons & B_ALARM1)
	alarm_index = 1 ;

      else if (this_buttons & B_ALARM2)
	alarm_index = 2 ;

      else if (this_buttons & B_ALARM3)
	alarm_index = 3 ;

      else if (this_buttons & B_ALARM4)
	alarm_index = 4 ;

      // Did we enter edit mode? If so, we need to read the alarm details from the alarm file
      if (alarm_index > 0)
      {
	Read_Alarm_Details (T_BUTTON, alarm_index, &alarm_hours, &alarm_minutes, &alarm_suspended, &alarm_duration, &init_vol_offset, &targ_vol_offset, &is_defined) ;

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s Start ALARM EDIT. index=%d, time=%02d:%02d, duration=%d, vol offset=%d..%d, suspended=%d, is_defined=%d\n",
		     Button_Manager_Thread_Str, alarm_index, alarm_hours, alarm_minutes, alarm_duration, init_vol_offset, targ_vol_offset, alarm_suspended, is_defined) ;
      }
    }
    // - - - - - - - - - - - - - - - -
    // Are we currently editing an alarm??? Alarm index will be a positive integer when editing
    if (alarm_index)
    {
      // If this alarm has not been set yet, then force blank the digits.
      // Only start to display something when an as-yet undefined alarm has been changed from its default.
      if ((setup_changed == 0) && (is_defined == 0))
	flash = 0 ;

      // One of the alarm buttons is pressed. Is the B_DURATION also pressed? This determines what gets displayed
      if (this_buttons & B_DURATION)
	display_alarm_time (disp_driver_msgq, (alarm_suspended) ? flash : 1, 0xff, alarm_duration, alarm_index, 0) ; // Q, flash, H, M, S, D

      else if (this_buttons & B_MEDIA)
      {
	// Because the 74HCT4511 can only display number 0..9 or blank the display, we have no way of illuminating a '-' sign
	// To show negative volume, we will flash all the decimal points.
	// The display_driver_thread will only look at bits 0..5 and ignore bits 6 and 7. We set bit 7 to ensure that the dots can be set off.
	dots = ((init_vol_offset >= 0) || (flash == 0)) ? 0x80 : 0xff ;
	display_alarm_time (disp_driver_msgq, (alarm_suspended) ? flash : 1, 0xff, abs(init_vol_offset), alarm_index, dots) ;
      }

      // display the alarm time - the B_DURATION button isn't pressed
      else
	display_alarm_time (disp_driver_msgq, (alarm_suspended) ? flash : 1, alarm_hours, alarm_minutes, alarm_index, 0) ;
    }

    else if ((this_buttons & B_DURATION) && (this_buttons & B_STOP_TOGGLE))
    {
      // Flash if snooze not active at the moment
      if (Seconds_Of_Snooze_Remaining < 2)
	display_alarm_time (disp_driver_msgq, flash, 0xff, 0, 0xff, 0) ;
      else
	display_alarm_time (disp_driver_msgq, 1, 0xff, Seconds_Of_Snooze_Remaining / 60, 0xff, 0) ;
    }

    else if ((this_buttons & B_DURATION) && (Seconds_Of_Play_Remaining > 0))
      display_alarm_time (disp_driver_msgq, 1, 0xff, Seconds_Of_Play_Remaining / 60, 0xff, 0) ;

    else if (this_buttons & B_DURATION) // Flash as playing is not active at the moment
      display_alarm_time (disp_driver_msgq, flash, 0xff, Default_Media_Duration, 0xff, 0) ;
    // - - - - - - - - - - - - - - - -
    // If no buttons pressed, continue without processing
    if (new_buttons == 0)
      continue ;

    increment = 0 ;

    // If the plus or minus buttons are pressed, calculate an increment based on how long they have been pressed
    if (new_buttons & (B_PLUS | B_MINUS))
    {
      // By how much should we increment (or decrement) the time?
      // If this is a repeat, the increment will be greater than 1, otherwise 1
      // The quantum of the increment increases the longer the + or - is held down
      if (repeat_count > (BUTTON_REPEAT_WAIT*4))
	increment = 10 ; // increment by tens after holding for the longest

      else if (repeat_count > (BUTTON_REPEAT_WAIT*2))
	increment = 2 ; // increment by twos after holding for a little

      else
	increment = 1 ; // unit increment initially

      // Change the sign of the increment if the minus button is pressed
      if (new_buttons & B_MINUS)
	increment = -increment ;

      if (Verbose & _V_BUTTON)
	Print_Log (T_BUTTON, "%s: Repeat=%d, Increment=%d\n", Button_Manager_Thread_Str, repeat_count, increment) ;
    }
    // - - - - - - - - - - - - - - - -
    // If in alarm edit mode, do we need to change hours/minutes/duration?
    if (alarm_index && (new_buttons & (B_PLUS | B_MINUS | B_STOP_TOGGLE)))
    {
      setup_changed = 1 ; // flag that we have made an alarm change of some kind

      // Are we enabling / disabling the alarm?
      if (this_buttons & B_STOP_TOGGLE)
      {
	alarm_suspended ^= 1 ; // toggle between true and false

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s: ALARM #%d suspended=%d\n", Button_Manager_Thread_Str, alarm_index, alarm_suspended) ;
      }

      // Are we incrementing alarm duration?
      else if (this_buttons & B_DURATION)
      {
	alarm_duration += increment ; // doing alarm duration

	if (alarm_duration < 1)
	  alarm_duration = 1 ;
	else if (alarm_duration > MAX_DURATION)
	  alarm_duration = MAX_DURATION ;

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s: ALARM #%d duration=%d\n", Button_Manager_Thread_Str, alarm_index, alarm_duration) ;
      }

      // Are we incrementing volume offset?
      else if (this_buttons & B_MEDIA)
      {
	init_vol_offset += increment ; // doing volume offset
	targ_vol_offset += increment ;

	if (init_vol_offset < -MAX_VOL_OFFSET)
	  init_vol_offset = -MAX_VOL_OFFSET ;
	else if (init_vol_offset > MAX_VOL_OFFSET)
	  init_vol_offset = MAX_VOL_OFFSET ;

	if (targ_vol_offset < -MAX_VOL_OFFSET)
	  targ_vol_offset = -MAX_VOL_OFFSET ;
	else if (targ_vol_offset > MAX_VOL_OFFSET)
	  targ_vol_offset = MAX_VOL_OFFSET ;

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s: ALARM #%d vol offset=%d..%d\n", Button_Manager_Thread_Str, alarm_index, init_vol_offset, targ_vol_offset) ;
      }

      else // we are incrementing hours+minutes
      {
	alarm_minutes += increment ;

	// Check for wrap around the hour, and wrap around midnight
	if (alarm_minutes < 0)
	{
	  alarm_minutes += 60 ;

	  if (--alarm_hours < 0)
	    alarm_hours += 24 ;
	}
	else if (alarm_minutes >= 60)
	{
	  alarm_minutes -= 60 ;

	  if (++alarm_hours >= 24)
	    alarm_hours -= 24 ;
	}

	// To make it look nice, when incrementing quickly, make the minutes end in a '0'
	if ((increment < -2) || (increment > 2))
	  alarm_minutes -= alarm_minutes % 10 ;

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s: ALARM #%d set time=%02d:%02d\n", Button_Manager_Thread_Str, alarm_index, alarm_hours, alarm_minutes) ;
      }

      continue ;
    }
    // - - - - - - - - - - - - - - - -
    // This section covers the B_DURATION button when pressed on its own
    // Is Snooze counting down?
    if ((this_buttons & B_DURATION) && (this_buttons & B_STOP_TOGGLE) && (Seconds_Of_Snooze_Remaining > 0))
    {
      Seconds_Of_Snooze_Remaining += increment * 60 ;

      // Check for too low or too high and clamp the time
      if (Seconds_Of_Snooze_Remaining < 60)
	Seconds_Of_Snooze_Remaining = 60 ;

      else if (Seconds_Of_Snooze_Remaining > ONE_HOUR)
	Seconds_Of_Snooze_Remaining = ONE_HOUR ;

      if (Verbose & _V_BUTTON)
	Print_Log (T_BUTTON, "%s: SNOOZE DURATION/TOGGLE. Seconds remaining=%d\n", Button_Manager_Thread_Str, Seconds_Of_Snooze_Remaining) ;

      continue ;
    }

    // Is play counting down? (This needs to come after the section above!)
    if ((this_buttons & B_DURATION) && (Seconds_Of_Play_Remaining > 0))
    {
      Seconds_Of_Play_Remaining += increment * 60 ;

      // Check for too low or too high and clamp the time
      if (Seconds_Of_Play_Remaining < 5)
	Seconds_Of_Play_Remaining = 5 ;

      else if (Seconds_Of_Play_Remaining > (MAX_DURATION*60))
	Seconds_Of_Play_Remaining = MAX_DURATION * 60 ;

      if (Verbose & _V_BUTTON)
	Print_Log (T_BUTTON, "%s: DURATION. Seconds play remaining=%d\n", Button_Manager_Thread_Str, Seconds_Of_Play_Remaining) ;

      continue ;
    }

    // B_DURATION has been pressed on its own (This needs to come after the sections above!)
    if (((Amp_State == A_AMP_OFF) || (Amp_State == A_AMP_ON_BLUETOOTH)) && (this_buttons & B_DURATION))
    {
      Default_Media_Duration += increment ;

      // Check for too low or too high and clamp the time
      if (Default_Media_Duration < 1)
	Default_Media_Duration = 1 ;

      else if (Default_Media_Duration > MAX_DURATION)
	Default_Media_Duration = MAX_DURATION ;

      if (Verbose & _V_BUTTON)
	Print_Log (T_BUTTON, "%s: DEFAULT_MEDIA_DURATION=%d\n", Button_Manager_Thread_Str, Default_Media_Duration) ;

      // We are going to rewrite the setup file, after the requisite delay
      setup_file_timer = SETUP_FILE_TIMER_DELAY ;
      launch_media = 1 ; // This will cause media to be launched when the button is released
      continue ;
    }
    // - - - - - - - - - - - - - - - -
    // Check for media commands. These are only checked if media is playing (the amp is on)
    if ((Amp_State != A_AMP_OFF) && (Amp_State != A_AMP_ON_BLUETOOTH) && (this_buttons & (B_MEDIA | B_PLUS | B_MINUS | B_STOP_TOGGLE)))
    {
      media_msg.type = MEDIA_NOTHING ;
      media_msg.index = media_msg.volume = -1 ;
      media_msg.init_vol_offset = -200 ;
      media_msg.targ_vol_offset = -200 ;
      media_msg.seconds = 0 ;
      media_msg.url_path[0] = '\0' ;
      media_msg.responseQ = 3 ; // a dummy number that will cause alarm clock to crash if there's a bug. #3 will appear in debug log
      media_msg_len = MEDIA_MSG_LEN ;

      // B_MEDIA works in conjunction with the up/down buttons to enable 'seek-fwd' + 'seek-back' functions (and if held down: 'next' + 'prev')
      // Is the SHIFT button pressed in conjunction with a fresh arrival of up or down??
      if ((this_buttons & B_MEDIA) && (new_buttons & (B_PLUS | B_MINUS)))
      {
	// Short press is seek forwards
	// Longer press is next track
	if (new_buttons & B_PLUS) // Up
	{
	  if (repeat_count == 0)
	    media_msg.type = MEDIA_SEEK_FWD ;

	  else if (repeat_count >= BUTTON_REPEAT_WAIT)
	  {
	    media_msg.type = MEDIA_NEXT ;
	    repeat_count = 0 ; // Restart the long repeat wait - it will increment to 1 before we check again
	  }

	  held = 0 ;

	  if (Verbose & _V_BUTTON)
	    Print_Log (T_BUTTON, "%s: MEDIA PLUS. Seek forwards or next\n", Button_Manager_Thread_Str) ;
	}

	else if (new_buttons & B_MINUS) // Down
	{
	  // Short press is seek backwards
	  // Longer press is restart the track
	  // Even longer press is prev track
	  if (repeat_count == 0)
	    media_msg.type = MEDIA_SEEK_BACK ;

	  else if ((repeat_count >= BUTTON_REPEAT_WAIT) && (held == 0))
	  {
	    media_msg.type = MEDIA_RESTART ;
	    repeat_count = 0 ; // Restart the long repeat wait - it will increment to 1 before we check again
	    held = 1 ;
	  }

	  else if (repeat_count >= BUTTON_REPEAT_WAIT)
	  {
	    media_msg.type = MEDIA_PREV ;
	    repeat_count = 0 ; // Restart the long repeat wait - it will increment to 1 before we check again
	  }

	  if (Verbose & _V_BUTTON)
	    Print_Log (T_BUTTON, "%s: MEDIA MINUS. Seek backwards or restart or prev track\n", Button_Manager_Thread_Str) ;
	}
      }

      else if ((this_buttons & B_MEDIA) && (new_buttons & B_STOP_TOGGLE)) // B_MEDIA + B_STOP_TOGGLE
      {
	if (repeat_count == 0)
	  media_msg_len = initiate_media_playing (&dev_index, &media_msg) ;
      }

      else if (new_buttons & B_STOP_TOGGLE) // B_STOP_TOGGLE on its own
      {
	// Send a snooze or stop message to the media thread depending in whether this is a short or long press
	// The type of message depends on how long we have pushed the button.
	// Short press will send a snooze message. Long press will send a stop message.
	if ((repeat_count == 0) && ((Second_Snooze_Ignore == 0) || (Seconds_Of_Snooze_Remaining < 0)))
	  media_msg.type = MEDIA_PAUSE ; // the mpv media player toggles the state of pause with each pause instruction
	else if (repeat_count >= (BUTTON_REPEAT_WAIT * 4))
	  media_msg.type = MEDIA_STOP ;
	else
	  continue ; // Ignore this repeat

	held = 0 ;

	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s: Media PAUSE or STOP. Repeat Count=%d. Message=%d\n", Button_Manager_Thread_Str, repeat_count, media_msg.type) ;
      }

      // If media_msg.type is not MEDIA_NOTHING, there's a message waiting to be sent
      if (media_msg.type != MEDIA_NOTHING)
      {
	if (mq_send (media_msgq, (char *)&media_msg, media_msg_len, 0) < 0)
	  Goodbye ("%s failed to send media msg %hu: %s\n", Button_Manager_Thread_Str, media_msg.type, strerror(errno)) ;

	continue ;
      }
    }
    // - - - - - - - - - - - - - - - -
    // Check for media commands. These are only checked if media is playing (the amp is on)
    if ((Amp_State != A_AMP_OFF) && (new_buttons & (B_PLUS | B_MINUS)))
    {
      if ((new_buttons & B_PLUS) && ((Volume + Vol_Offset) < MAX_VOLUME))
      {
	Volume++ ;
	Enforce_Volume_Limits (0) ;
	setup_file_timer = SETUP_FILE_TIMER_DELAY ;
	held = 0 ;
	goto disp_volume ;
      }

      else if ((new_buttons & B_MINUS) && ((Volume + Vol_Offset) > 0))
      {
	Volume-- ;
	Enforce_Volume_Limits (0) ;
	setup_file_timer = SETUP_FILE_TIMER_DELAY ;
	held = 0 ;

      disp_volume:
	if (Verbose & _V_BUTTON)
	  Print_Log (T_BUTTON, "%s: Volume=%d, Voffset=%d\n", Button_Manager_Thread_Str, Volume, Vol_Offset) ;
      }
    }
    // else media not playing and must have pressed a button that only does something when media is playing. Just ignore it.
  }

  Print_Log (T_BUTTON, "%s unscheduled exit\n", Button_Manager_Thread_Str) ;

  // Tidy up before exiting
  mq_close (Button_Msgq) ;
  mq_close (disp_driver_msgq) ;
  mq_close (media_msgq) ;
  pthread_exit (NULL) ;
} // Button_Manager_Thread
///////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////
// Multicast_Tx_Thread
//
// This thread generates periodic multicast packets for the purpose of other alarm clocks
// to learn that this clock exists

static const char Multicast_Tx_Thread_Str[] = "Multicast Tx Thread" ;

void *
Multicast_Tx_Thread (void *xarg)
{
  int			n ;
  char			hello[PACKET_SIZE] ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_MULTICAST)
    Print_Log (T_MULTICAST_TX, "Entering %s\n", Multicast_Tx_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  if (Verbose & _V_MULTICAST)
    Print_Log (T_MULTICAST_TX, "%s: I am [%s / %s] and am multicasting to group %s:%d\n",
	       Multicast_Tx_Thread_Str, My_Name, My_IPv4_Address, Multicast_Group, UDP_PORT) ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    // Hello message - sent AFTER the alarm file has been read
    if (Alarm_File_Time >= 0)
    {
      n = snprintf (hello, PACKET_SIZE, "H,%c,%x,%" PRId64 ",%s,%s", This_Clock_Type, Alarm_File_Hash, (int64_t)Alarm_File_Time, My_Name, My_IPv4_Address) ;

      if (Verbose & _V_MULTICAST)
	Print_Log (T_MULTICAST_TX, "%s sending %d byte hello message: '%s'\n", Multicast_Tx_Thread_Str, n, hello) ;

      Send_Network_Packet (T_MULTICAST_TX, hello, n) ;
    }

    sleep (HELLO_PERIOD) ; // send a 'hello I am here' message every HELLO_PERIOD seconds
  }

  Print_Log (T_MULTICAST_TX, "%s unscheduled exit\n", Multicast_Tx_Thread_Str) ;
  pthread_exit (NULL) ;
} // Multicast_Tx_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// Multicast_Rx_Thread
//
// Messages are in CSV format.
// [0]='H' if message is a hello message.
//     [2]='S' if other system is configured as stand alone, or 'C' for audible clustered, or 'Z' for silent clustered
//     [4] contains the hash of the alarm file
//     Fourth column contains the Epoch timestamp of the alarm file (seconds since 1st Jan 1970)
//     Fifth column contains system name of the other system
//     Sixth column contains IP address of the other system
//
// [0]='B' if message is a button press message. [2] contains 'R' or 'Z'
//     Third column contains non-zero 32-bit unsigned int random token that remains the same while buttons are pressed
//     Fourth column contains the 16-bit unsigned bitmask representing buttons
//
// [0]='A' if message is a Alarm file request.
// [0]='U' if message is a Alarm file update.
//
static const char Multicast_Rx_Thread_Str[] = "Multicast Rx Thread" ;

void *
Multicast_Rx_Thread (void *xarg)
{
  int			n, i, j, file_size ;
  mqd_t			disp_driver_msgq ;
  uint32_t		token, alarm_file_token, hash ;
  uint16_t		buttons ;
  socklen_t		src_addr_len ;
  struct sockaddr_in	src_addr ;
  int64_t		big_timestamp ;
  time_t		file_time ;
  UT			times ;
  TM			*time_info ;
  char			*file_string ;
  char			recv_data[PACKET_SIZE] ;
  char			ipv4_src[INET_ADDRSTRLEN] ;
  char			peer_name[SETUP_FIELD_LEN] ;
  char			temp_str[SETUP_FIELD_LEN+1] ;
  int			fd ;
  char			temp_name[TEMP_NAME_LEN] ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_MULTICAST)
    Print_Log (T_MULTICAST_RX, "Entering %s\n", Multicast_Rx_Thread_Str) ;

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  memset (Peers, 0x00, sizeof(Peers)) ; // blank the peers table
  src_addr_len = sizeof (src_addr) ;
  Button_Control_Token = 0 ; // 0 means blank/reset
  Remote_Buttons = 0x000 ;
  Button_Downcount = 0 ;
  alarm_file_token = 0 ;

  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // The Display Driver Thread will already have started. We can open its message queue for writing
  disp_driver_msgq = Open_Msg_Queue_Tx (T_MULTICAST_RX, DD_Msgq_name, 0) ;

  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
  ignore_this_message:
    // Receive a message from any source address....
    if ((n = recvfrom (Rx_Socket, recv_data, PACKET_SIZE, MSG_WAITALL, (struct sockaddr *)&src_addr, &src_addr_len)) < 0)
    {
      Print_Log (T_MULTICAST_RX, "%s: recvfrom failed: %s\n", Multicast_Rx_Thread_Str, strerror(errno)) ;
      break ;
    }

    // Ignore messages from ourself
    if (src_addr.sin_addr.s_addr == My_IPv4_sin_addr)
    {
      if (Verbose & _V_MULTICAST)
	Print_Log (T_MULTICAST_RX, "%s ignoring %d byte msg from self\n", Multicast_Rx_Thread_Str, n) ;

      continue ;
    }

    if (n >= PACKET_SIZE) // This is probably paranoia
      n = PACKET_SIZE-1 ;

    recv_data[n]= '\0' ; // terminate with a null. The message over the wire does not have the null terminator

    // Decode the actual IP address - don't believe what's in the message itself
    inet_ntop (AF_INET, &src_addr.sin_addr, ipv4_src, sizeof(ipv4_src)) ;

    if (Verbose & _V_MULTICAST)
      Print_Log (T_MULTICAST_RX, "%s received %d bytes '%s' from %s\n", Multicast_Rx_Thread_Str, n, recv_data, ipv4_src) ;
    //------------------------------------------------------------------
    // Record receipt of this packet by finding the record in the peers table, and updating it.
    // It is unlikely that a packet arrives from 0.0.0.0, but we will check anyway
    i = NUM_PEERS ;
    j = -1 ;

    // Determine whether this peer is already in the Peers[] table
    // The loop below will return its index in i (if it exists) or a blank entry index in j (if it exists)
    if (src_addr.sin_addr.s_addr != 0)
    {
      for (i=0 ; i < NUM_PEERS ; i++)
      {
	if (Peers[i].sin_addr.s_addr == src_addr.sin_addr.s_addr)
	{
	  // Ignore this record if the peer type does not match what's in the message
	  if (Peers[i].type != recv_data[2])
	  {
	    // if this is a hello message, update the peers table - the peer has changed their type
	    if (recv_data[0] == 'H')
	    {
	      if (Verbose & _V_MULTICAST)
		Print_Log (T_MULTICAST_RX, "%s Peer %s has changed type from '%c' to '%c'\n", Multicast_Rx_Thread_Str, ipv4_src, Peers[i].type, recv_data[2]) ;

	      j = i ;
	      goto update_peer_table ;
	    }
	    else if (Verbose & _V_MULTICAST)
	      Print_Log (T_MULTICAST_RX, "%s mismatched clock_type received '%s' from %s\n", Multicast_Rx_Thread_Str, recv_data, ipv4_src) ;

	    goto ignore_this_message ;
	  }

	  // We found an entry - record the time. The entry index is in 'i'
	  Peers[i].last_seen = Current_Seconds_Count ;
	  break ;
	}

	// If the entry is blank, use j to remember the first blank entry in the table
	else if ((j < 0) && (Peers[i].sin_addr.s_addr == 0))
	  j = i ;
      }
    }

    // Did we find a new entry and is there space in the table to record it?
    if ((i >= NUM_PEERS) && (j >= 0))
    {
      // This is new - create an entry in the table
    update_peer_table:
      Peers[j].last_seen = Current_Seconds_Count ;
      Peers[j].sin_addr.s_addr = src_addr.sin_addr.s_addr ;
      Peers[j].type = recv_data[2] ;
      i = j ;
    }
    //------------------------------------------------------------------
    // Identify the type of receive packet and process
    // Hello message. The packet has already been logged - we just need to read two of the four fields (and ignore the last)
    // Hello messages can arrive from any type of clock
    if (recv_data[0] == 'H')
    {
      // Extract the peer's system name. It is almost certainly the same as previously
      sprintf (temp_str, " %%x , %%" SCNd64 " , %%%d[^ ,\n\t\r\f\v]", SETUP_FIELD_LEN) ;

      if ((j = sscanf (&recv_data[4], temp_str, &hash, &big_timestamp, peer_name)) != 3)
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s cannot read %s hash, time and peer name: '%s'\n", Multicast_Rx_Thread_Str, ipv4_src, recv_data) ;

	continue ;
      }

      // if i is not less than NUM_PEERS, there is no room left in the Peers table
      // has the system name changed???
      if ((i < NUM_PEERS) && (strcmp (peer_name, Peers[i].name) != 0))
      {
	if ((Verbose & _V_MULTICAST) && (Peers[i].name[0] != '\0'))
	  Print_Log (T_MULTICAST_RX, "%s Peer %s has changed name from '%s' to '%s'\n", Multicast_Rx_Thread_Str, ipv4_src, Peers[i].name, peer_name) ;

	strcpy (Peers[i].name, peer_name) ;
      }

      // This cast is to get around the F*&^%G difference between 64 bit and 32 bit versions of time_t
      file_time = (time_t)big_timestamp ;

      // If we are a clustered clock and receive a hello message with a more recent + different alarm file
      // then request a new alarm file using an 'A' msg.
      if (((This_Clock_Type == 'A') || (This_Clock_Type == 'C') || (This_Clock_Type == 'Z')) &&
	  (Alarm_File_Hash != hash) && (Alarm_File_Time < file_time) && (Alarm_File_Time >= 0))
      {
	if ((Verbose & (_V_MULTICAST | _V_SETUP)) &&
	    ((time_info = localtime (&file_time)) != NULL) &&
	    (strftime (temp_str, sizeof(temp_str), Time_Format_Str, time_info) > 0) )
	  Print_Log (T_MULTICAST_RX, "%s Peer %s alarm hash=0x%x %s, local hash=0x%x, file time difference=%" PRId64 "\n",
		     Multicast_Rx_Thread_Str, ipv4_src, hash, temp_str, Alarm_File_Hash, (int64_t)(file_time - Alarm_File_Time)) ;

	// Initiate reload / copy of the alarm file
	for (alarm_file_token=0 ; alarm_file_token == 0 ; alarm_file_token=(uint32_t)rand()) ; // ensure token is non-zero
	i = snprintf (temp_str, PACKET_SIZE, "A,%c,%u", This_Clock_Type, alarm_file_token) ; // Alarm request message
	Print_Log (T_MULTICAST_RX, "%s sending alarm file retrieval message: '%s'\n", Multicast_Rx_Thread_Str, temp_str) ;
	Send_Network_Packet (T_MULTICAST_RX, temp_str, i) ;
      }

      continue ;
    }
    //------------------------------------------------------------------
    // Button press message has been received
    // Button messages are only sent by clustered clocks. We will ignore if we are an autonomous clustered clock
    if ((recv_data[0] == 'B') &&
	((recv_data[2] == 'A') || (recv_data[2] == 'C') || (recv_data[2] == 'Z')) &&
	((This_Clock_Type == 'C') || (This_Clock_Type == 'Z')))
    {
      if ((sscanf(&recv_data[4], "%u,%hx", &token, &buttons) != 2) || ((Button_Control_Token != 0) && (Button_Control_Token != token)))
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s button press message '%s' from %s doesn't match our token\n",
		     Multicast_Rx_Thread_Str, recv_data, ipv4_src, Button_Control_Token) ;

	continue ;
      }

      // If the buttons have been released, we can exit the remote control mode
      if ((Remote_Buttons = buttons) == 0x000)
	Button_Control_Token = 0 ;

      else // remember the token identity
      {
	Remote_Buttons |= B_REMOTE ; // Remember the button presses in this global variable, the B_REMOTE bit is a flag to show origin
	Button_Control_Token = token ; // This is only necessary the first time, but on second and subsequent, will set the same value
	Button_Downcount = BUTTON_SCAN_FREQ ; // The scan frequency is 10. Allow up to 10 messages to be lost before aborting
      }

      continue ;
    }
    //------------------------------------------------------------------
    // Alarm file request message
    // Alarm file request messages are only sent by clustered clocks
    if ((recv_data[0] == 'A') &&
	((recv_data[2] == 'A') || (recv_data[2] == 'C') || (recv_data[2] == 'Z')) &&
	(This_Clock_Type != 'S'))
    {
      if (sscanf(&recv_data[4], "%u", &token) != 1)
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s alarm request '%s' from %s cannot be parsed\n", Multicast_Rx_Thread_Str, recv_data, ipv4_src) ;

	continue ;
      }

      // We are being asked to send a copy of the alarm file. Read the file into a string and send that as a packet
      if (Read_Alarm_File_Into_String (T_MULTICAST_RX, &file_size, &file_string, NULL, &file_time) < 0)
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s cannot respond to alarm request\n", Multicast_Rx_Thread_Str) ;

	continue ;
      }

      // Pre-prepare the response string. It needs to commence with the right header...
      // We need to respond with the token the remote sent to us, so that they can then match the response
      i = sprintf (recv_data, "U,%c,%u,%" PRId64 ",", This_Clock_Type, token, (int64_t)file_time) ;

      // If the string is too long, we cannot respond either (or it makes no sense to respond because they will not
      // be listening with a packet buffer large enough to hold the file
      if (file_size > (PACKET_SIZE-i-1))
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s cannot respond to alarm request. Alarm file is %d bytes\n", Multicast_Rx_Thread_Str, file_size) ;
      }
      else
      {
	// We are set to respond, Need to append the file to the packet and determine the new packet size
	i += sprintf (&recv_data[i], "%s", file_string) ;

	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_TX, "%s sending %d byte alarm file in response to request\n", Multicast_Rx_Thread_Str, i) ;

	Send_Network_Packet (T_MULTICAST_TX, recv_data, i) ;
      }

      if (file_string != NULL)
	free (file_string) ;

      continue ;
    }
    //------------------------------------------------------------------
    // Alarm file update message
    // Alarm file update messages are only sent by clustered clocks
    if ((recv_data[0] == 'U') &&
	((recv_data[2] == 'A') || (recv_data[2] == 'C') || (recv_data[2] == 'Z')) &&
	(This_Clock_Type != 'S'))
    {
      if (sscanf(&recv_data[4], "%u , %" SCNd64, &token, &big_timestamp) != 2)
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s alarm response from %s cannot be parsed\n", Multicast_Rx_Thread_Str, ipv4_src) ;

	continue ;
      }
      // - - - - - -
      else if (token != alarm_file_token)
      {
	if (Verbose & _V_MULTICAST)
	  Print_Log (T_MULTICAST_RX, "%s token %s:%d doesn't match expected value=%d\n",
		     Multicast_Rx_Thread_Str, ipv4_src, token, alarm_file_token) ;

	continue ;
      }
      // - - - - - -
      else if (big_timestamp <= Alarm_File_Time)
      {
	// The timestamp for the new alarm file should be *after* the current local version
	file_time = (time_t)big_timestamp ;
	time_info = localtime (&file_time) ;
	strftime (temp_name, sizeof(temp_name), Time_Format_Str, time_info) ;
	time_info = localtime (&Alarm_File_Time) ;
	strftime (temp_str, sizeof(temp_str), Time_Format_Str, time_info) ;

	Print_Log (T_MULTICAST_RX, "%s alarm file from %s @ %s not after local version @ %s\n",
		   Multicast_Rx_Thread_Str, ipv4_src, temp_name, temp_str) ;

	continue ;
      }
      // - - - - - -
      // The tokens match and the new time is later - write the alarm file data to the alarm file
      // Skip over the token
      for (file_string=&recv_data[4] ; (*file_string != '\0') && (*file_string != ',') ; file_string++) ;

      // Skip over comma after token
      if (*file_string != '\0')
	file_string++ ;

      // Skip over the file time
      for (; (*file_string != '\0') && (*file_string != ',') ; file_string++) ;

      // Skip over comma following file time
      if (*file_string != '\0')
	file_string++ ;

      strcpy (temp_name, Temp_Name) ; // create a fresh temporary name template

      // We now need to determine the number of remaining characters. Use pointer arithmetic to directly subtract the
      // number of charcters between file_string and the beginning of the string
      if ((n -= file_string - recv_data) <= 0)
	Print_Log (T_MULTICAST_RX, "%s alarm file received from %s appears to be empty\n", Multicast_Rx_Thread_Str, ipv4_src) ;

      // Open a temporary file for writing
      else if ((fd = mkstemp (temp_name)) < 0)
	Print_Log (T_MULTICAST_RX, "%s cannot open '%s' for writing: %s\n", Multicast_Rx_Thread_Str, temp_name, strerror(errno)) ;

      else if ((write (fd, file_string, n)) != n)
      {
	Print_Log (T_MULTICAST_RX, "%s failed to write %d byte alarm file '%s': %s\n", Multicast_Rx_Thread_Str, n, temp_name, strerror(errno)) ;
	close (fd) ;
	remove (temp_name) ;
      }
      else
      {
	// The temporary file has been written. We can close it and then rename it
	close (fd) ;
	Rename_Backups (T_MULTICAST_RX, temp_name, Alarm_Filename) ; // create a backup of what's there before klobbering it with new file

	// Set the file date and time to be the same as the version we just downloaded
	times.actime = times.modtime = (time_t)big_timestamp ;

	if (utime (Alarm_Filename, &times) != 0)
	  Goodbye ("%s failed to set times on '%s': %s\n", Multicast_Rx_Thread_Str, Alarm_Filename, strerror(errno)) ;

	file_time = (time_t)big_timestamp ;
	time_info = localtime (&file_time) ;
	strftime (temp_str, sizeof(temp_str), Time_Format_Str, time_info) ;
	Print_Log (T_MULTICAST_RX, "%s Received %d byte alarm file (timestamp=%s) to replace current alarms\n", Multicast_Rx_Thread_Str, n, temp_str) ;

	// We need to update the local hash based on the latest alarm file
	Read_Setup_And_Alarms (T_MULTICAST_RX, 1) ;
      }

      // Pollute the token so it cannot be reused
      alarm_file_token = 0 ;
      continue ;
    }
  }

  Print_Log (T_MULTICAST_RX, "%s unscheduled exit\n", Multicast_Rx_Thread_Str) ;

  // Tidy up before exiting
  mq_close (disp_driver_msgq) ;
  pthread_exit (NULL) ;
} // Multicast_Rx_Thread
///////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////
// Bluetooth code starts here
///////////////////////////////////////////////////////////////////////////////////////////
// skip_to_string parses the incoming character stream from the bluetoothctl process
// and discards characters until either the prompt (# ) or one of the target strings are
// reached. The target strings are optional.
//   Returns 0 if reason for return is a prompt (# )
//   Returns 1 if reason for return is string1
//   Returns 2 if reason for return is string2
//   Returns 3 if reason for return is string3
//
// The line of text up until the characters that triggered the return will be stored in
// This_Question and the full line including characters after in This_Line.

static char		*Last_Point = NULL ;
static const char	Bluetooth_Manager_Thread_Str[] = "Bluetooth Manager Thread" ;

int
skip_to_string (const char *string1, const char *string2, const char *string3, const int do_full_line1, const int do_full_line2, const int do_full_line3)
{
  char		*p ;
  const char	*ret_string ;
  int		next_index, len, ret_val ;

  // Loop waiting for text
  // Pull the text out of a circular buffer of lines that has been prepared by the Bluetooth_Rx_Thread
  for (; (Threads_Active < _SHUTTING_DOWN) ;)
  {
    next_index = RB_extract + 1 ;

    if (next_index >= MAX_LINES)
      next_index = 0 ;

    // Last_Point points to any residual characters following the previously seen character
    if (Last_Point == NULL)
    {
      if (Read_Lines[RB_extract] == NULL)
      {
	// There are no characters to search - delay and try again soon
	Sleep_ns (QUARTER_SECOND) ;
	continue ;
      }

      // Read_Lines[RB_extract] is not NULL at this point
      Last_Point = Read_Lines[RB_extract] ;
    }

    // Is it the end of the line? This is flagged by end of string.
    if (*Last_Point == '\0')
    {
      if (Read_Lines[next_index] == NULL)
      {
	// There are no characters to search - delay and try again soon
	Sleep_ns (QUARTER_SECOND) ;
	continue ;
      }

      // The next index is not NULL at this point - we must be at the end of the line
      goto no_match_do_free ;
    }

    // At this point, we have a line of text pointed to by Last_Point.
    // Does it contain the one of the strings we are searching for?
    if ((p = strcasestr (Last_Point, "# ")) != NULL) // This is a prompt
    {
      // We have found a prompt. update Last_Point to point immediately after the prompt
      p += 2 ;
      Last_Point = p ;

      return 0 ;
    }

    if ((string1 != NULL) && ((p = strcasestr (Last_Point, string1)) != NULL))
    {
      // We have found string1 - are we looking for a complete line?
      if (do_full_line1 && (RB_extract == RB_insert))
      {
	Sleep_ns (HUNDRED_MS) ;
	continue ; // wait until the RB_extract line is NOT the line currently being populated
      }

      ret_string = string1 ;
      ret_val = 1 ;

      goto clean_exit ;
    }

    if ((string2 != NULL) && ((p = strcasestr (Last_Point, string2)) != NULL))
    {
      // We have found string2 - are we looking for a complete line?
      if (do_full_line2 && (RB_extract == RB_insert))
      {
	Sleep_ns (HUNDRED_MS) ;
	continue ; // wait until the RB_extract line is NOT the line currently being populated
      }

      ret_string = string2 ;
      ret_val = 2 ;

      goto clean_exit ;
    }

    if ((string3 != NULL) && ((p = strcasestr (Last_Point, string3)) != NULL))
    {
      // We have found string3 - are we looking for a complete line?
      if (do_full_line3 && (RB_extract == RB_insert))
      {
	Sleep_ns (HUNDRED_MS) ;
	continue ; // wait until the RB_extract line is NOT the line currently being populated
      }

      ret_string = string3 ;
      ret_val = 3 ;

    clean_exit:
      // Update Last_Point to point immediately after the string
      len = strlen (ret_string) ;
      p += len ;
      Last_Point = p ; // this points to the next character to read
      strcpy (This_Line, Read_Lines[RB_extract]) ; // Full line of text stored here.
      strcpy (This_Question, Read_Lines[RB_extract]) ; // this will become a partial line of text

      // We are going to return with the This_Question string containing the full line of text up until the string
      // because it will be useful to determine if we have seen this before or not
      if ((p = strcasestr (This_Question, ret_string)) != NULL)
      {
	p += len ;
	*p = '\0' ;
      }

      return ret_val ;
    }

    // At this point, there is no match - need to wait for more characters
    if (Read_Lines[next_index] != NULL)
    {
    no_match_do_free:
      free (Read_Lines[RB_extract]) ; // Return the text buffer to the OS
      Read_Lines[RB_extract] = NULL ; // Null out the pointer so we know we didn't overrun
      RB_extract = next_index ; // Adjust pointer to the next element in circular buffer
      Last_Point = NULL ; // We have not yet seen first character of next line - set to null
      continue ; // Don't wait before processing more characters
    }

    Sleep_ns (QUARTER_SECOND) ; // No characters received. Just wait around for more characters
  }

  return 0 ;
} // skip_to_string
///////////////////////////////////////////////////////////////////////////////////////////
// orchestrate_bluetoothctl_commands orchestrates sending bluetooth commands to bluetoothctl
#define MAX_FIELD		(150)
#define NUM_MAC_ADDRESSES	(30)

void
orchestrate_bluetoothctl_commands (const EXPECT *e)
{
  int		n ;
  char		command[MAX_FIELD] ;

  for (; (Threads_Active < _SHUTTING_DOWN) && ((e->target_string != NULL) || (e->command != NULL)) ; e++)
  {
    // mimic having received target_string
    // in case a target wasn't specified and skipping to prompt was specified
    n = 1 ;

    // If we are expecting a string or prompt, wait for those in sequence
    if (e->target_string != NULL)
      for (; (n = skip_to_string (e->target_string, NULL, NULL, 0, 0, 0)) == 0 ;) ; // 0 means prompt was found

    if ((n != 0) && (e->find_prompt))
      skip_to_string (NULL, NULL, NULL, 0, 0, 0) ;

    // The strings have been received. Send the command
    if (e->command != NULL)
    {
      n = sprintf (command, "%s\n", e->command) ;

      if (write (PARENT_WRITE_FD, command, n) < 0)
	Goodbye ("Cannot write '%s' to bluetooth pipe: %s\n", e->command, strerror(errno)) ;

      // wait for the command to be echoed back at us
      for (; skip_to_string (e->command, NULL, NULL, 0, 0, 0) == 0 ;) ; // 0 means prompt was found and we should keep looking
    }
  }
} // orchestrate_bluetoothctl_commands
///////////////////////////////////////////////////////////////////////////////////////////
// read_response reads WHOLE LINES until the target string is reached, discarding the
// beginning and ending partial lines.
//
// It will read characters up until the end of the line, or prompt '# '

int
read_response (char *full_line, const int len)
{
  int		next_index, n ;
  char		*p ;

  // Does Last_Point point part way into a line? If so, discard the remaining
  // characters and wait for the next whole line.
  for (; Last_Point != NULL ;)
  {
    next_index = RB_extract + 1 ;

    if (next_index >= MAX_LINES)
      next_index = 0 ;

    // Does the prompt string exist part way along this line?
    if ((p = strcasestr (Last_Point, "# ")) != NULL)
    {
      p += 2 ; // the length of the prompt search string
      Last_Point = p ;
      return 0 ;
    }

    // We don't have a prompt string, and we still have a partial line. Wait until the
    // next line commences so we can safely discard the partial line
    if (Read_Lines[next_index] != NULL)
    {
      // discard the remaining characters in this line. Point to the next line
      free (Read_Lines[RB_extract]) ;
      Read_Lines[RB_extract] = NULL ;
      RB_extract = next_index ;
      Last_Point = NULL ;
      break ;
    }

    // Hang around for a bit waiting
    Sleep_ns (QUARTER_SECOND) ;
  }

  // At this point, Last_Point is NULL and we are scanning from the beginning of a line
  // Loop waiting for whole lines of text
  for (;;)
  {
    next_index = RB_extract + 1 ;

    if (next_index >= MAX_LINES)
      next_index = 0 ;

    // Last_Point points to any residual characters past last searched string
    if (Last_Point == NULL)
    {
      if (Read_Lines[RB_extract] == NULL)
      {
	// There are no characters to read - delay and try again soon
	Sleep_ns (QUARTER_SECOND) ;
	continue ;
      }

      // Read_Lines[RB_extract] is definitely not NULL at this point
      Last_Point = Read_Lines[RB_extract] ;
    }

    // At this point, Last_Point points to the beginning of the line we are scanning
    // Does the prompt string exist part way along this line?
    if ((p = strcasestr (Last_Point, "# ")) != NULL)
    {
      p += 2 ; // the length of the prompt search string
      Last_Point = p ;
      return 0 ;
    }

    // No prompt. Do we have a complete line? We only know when there's a new line present
    if (Read_Lines[next_index] != NULL)
      break ;

    // There are no characters to search - delay and try again soon
    Sleep_ns (QUARTER_SECOND) ;
  }

  // We have a complete line of text pointed to by RB_extract at this point.
  // We are sure the prompt is not in this complete line
  if ((n = strlen (Last_Point)) >= len)
  {
    if (Verbose & _V_BLUETOOTH)
      Print_Log (T_BLUETOOTH_MAN, "%s Truncating response. %d chars available and string is %d chars\n", Bluetooth_Manager_Thread_Str, len, n) ;

    n = len - 1 ;
    Last_Point[n] = '\0' ;
  }

  memmove (full_line, Last_Point, n+1) ;
  free (Read_Lines[RB_extract]) ;
  Read_Lines[RB_extract] = NULL ;
  RB_extract = next_index ;
  Last_Point = NULL ;
  return n ; // number of characters
} // read_response
///////////////////////////////////////////////////////////////////////////////////////////
// build_paired_list creates a list of mac addresses that have been previously paired
int
build_paired_list (char	*paired_devices_array[])
{
  int		n, i ;
  char		format[MAX_FIELD], field1[MAX_FIELD] ;

  // The 'paired-devices' command is assumed to have already been issued.
  // It will cause a list of the previously paired devices to be displayed
  // Pick out the MAC addresses and build a list using malloced memory.
  // If no devices have been paired, the output will be blank and it will return to a prompt
  // MAC address is the second field
  sprintf (format, "%%*[^ ] %%%d[^ ]", MAX_FIELD) ; // prepare this string used for sscanf

  // Create a list of MAC addresses
  for (n=0 ; (Threads_Active < _SHUTTING_DOWN) && (n < NUM_MAC_ADDRESSES) && (skip_to_string ("Device", NULL, NULL, 1, 0, 0) == 1) ;)
  {
    // read MAC address from the device line
    // The string length must be 17 characters or the MAC address is invalid
    if ((sscanf (This_Line, format, field1) == 1) || (strlen(field1) != 17))
    {
      // Verify that the mac address is valid and in format xx:xx:xx:xx:xx:xx
      for (i=0 ; i < 17 ; i++)
      {
	if (((i + 1) % 3) != 0) // test characters except 2, 5, 8, 11, 14
	{
	  if (isxdigit(field1[i]) == 0)
	    break ;
	}
	else if (field1[i] != ':') // test characters 2, 5, 8, 11, 14
	  break ;
      }

      // The characters must form a valid MAC address
      if (i != 17)
	continue ;

      // We must not already have registered the same MAC address
      for (i=0 ; i < n ; i++)
      {
	if (strcasecmp(paired_devices_array[i], field1) == 0)
	  break ;
      }

      if (i < n)
	continue ;

      // Finally - we know we have a unique mac address
      paired_devices_array[n] = malloc (MAC_ADDRESS_LEN) ;
      memmove (paired_devices_array[n], field1, MAC_ADDRESS_LEN) ;
      n++ ;
    }
  }

  return n ;
} // build_paired_list
///////////////////////////////////////////////////////////////////////////////////////////
// build_connected_list creates a list of previously paired devices that are currently connected
void
build_connected_list (char *paired_devices_array[])
{
  int		n, i, j, c, ret_reason, connected ;
  char		format[MAX_FIELD], field1[MAX_FIELD] ;
  char		buffer [MAX_LINE] ;

  // Fetch a list of paired device mac addresses
  n = build_paired_list (paired_devices_array) ;

  if (Verbose & _V_BLUETOOTH)
    Print_Log (T_BLUETOOTH_MAN, "%s rebuilding connected devices list from %d paired devices\n", Bluetooth_Manager_Thread_Str, n) ;

  // Now we will count the number of connected clients
  memset (Bluetooth_Devices, 0x00, sizeof(Bluetooth_Devices)) ;
  sprintf (format, "%%*[^ ] %%%d[^\n]", MAX_FIELD) ; // prepare this string used for sscanf (\n won't be found, but will read to end of string)

  // we have a list of previously known mac addresses - now need to check if any of them are currently connected
  for (i=j=0 ; (Threads_Active < _SHUTTING_DOWN) && (i < n) && (j < BLUETOOTH_DEV_RECS) ; i++)
  {
    c = snprintf (buffer, sizeof(buffer), "info %s\n", paired_devices_array[i]) ;

    if (write (PARENT_WRITE_FD, buffer, c) < 0)
      Goodbye ("Cannot write info to bluetooth pipe: %s\n", strerror(errno)) ;

    connected = 0 ;

    while ((ret_reason = skip_to_string ("Name", "Connected", "Trusted", 1, 1, 1)) != 0)
    {
      // The connection status is in the second field
      if (sscanf (This_Line, format, field1) == 1) // read connected status from the connected line
      {
        // Create a connection record for this device
        memmove (Bluetooth_Devices[j].mac_address, paired_devices_array[i], MAC_ADDRESS_LEN) ;

        if (ret_reason == 1)
        {
	  // The Name
          memmove (Bluetooth_Devices[j].device_name, field1, DEVICE_NAME_LEN) ;
          Bluetooth_Devices[j].device_name[DEVICE_NAME_LEN-1] = '\0' ; // Ensure the string is null terminated (if its a long name)
        }

        else if (ret_reason == 2) // Connected
          connected = (*field1 == 'y') || (*field1 == 'Y') ;

        else // Trusted
          Bluetooth_Devices[j].trusted = (*field1 == 'y') || (*field1 == 'Y') ;
      }
    }

    // If this device is connected, then keep the record
    if (connected)
    {
      j++ ;
      Num_Bluetooth_Clients = (uint16_t)j ; // Setting to non-zero will trigger amp turn-on
    }

    free (paired_devices_array[i]) ;
  }

  // Setting this variable to zero will trigger amp turn-off.
  Num_Bluetooth_Clients = (uint16_t)j ;

  // Terminate the list of connected devices - the count is in j
  if (j < BLUETOOTH_DEV_RECS)
    Bluetooth_Devices[j].mac_address[0] = '\0' ;

  if (Verbose & _V_BLUETOOTH)
  {
    Print_Log (T_BLUETOOTH_MAN, "%s %hu connected bluetooth devices\n", Bluetooth_Manager_Thread_Str, Num_Bluetooth_Clients) ;

    for (n=0 ; (n < BLUETOOTH_DEV_RECS) && (Bluetooth_Devices[n].mac_address[0] != '\0') ; n++)
      Print_Log (T_BLUETOOTH_MAN, "%s * %s (%s) trusted=%d\n", Bluetooth_Manager_Thread_Str,
		 Bluetooth_Devices[n].mac_address, Bluetooth_Devices[n].device_name, Bluetooth_Devices[n].trusted) ;
  }
} // build_connected_list
///////////////////////////////////////////////////////////////////////////////////////////
// The job of Bluetooth_Manager_Thread is to parse the incoming stream in an orderly manner
// and to issue commands in an orderly manner. The thread also initialises the bluetoothctl task

void
issue_bluetooth_disconnect (char *mac_address)
{
  int		c ;
  char		buffer [MAX_LINE] ;

  c = snprintf (buffer, sizeof(buffer), "disconnect %s\n", mac_address) ;

  if (write (PARENT_WRITE_FD, buffer, c) < 0)
    Goodbye ("Cannot write disconnect to bluetooth pipe: %s\n", strerror(errno)) ;

  // Wait until the device has been removed, or the error that it is not available is displayed (error shouldn't happen)
  skip_to_string ("successful", NULL, NULL, 1, 0, 0) ;
}

void *
Bluetooth_Manager_Thread (void *xarg)
{
  int		n, i, c, ret_reason, seen_transport, new_connection ;
  ST		file_info ;
  char		*p ;
  char		buffer [MAX_LINE] ;
  char		prev_question [MAX_LINE] ;
  char		mac_address [MAC_ADDRESS_LEN] ;
  char		*mac_addresses [NUM_MAC_ADDRESSES] ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_BLUETOOTH)
    Print_Log (T_BLUETOOTH_MAN, "Entering %s\n", Bluetooth_Manager_Thread_Str) ;

  // Does bluetoothctl exist??? If not, don't start this thread
  if ((Bluetoothctl_PID <= 0) || (stat ("/usr/bin/bluetoothctl", &file_info) != 0))
  {
    Print_Log (T_BLUETOOTH_MAN, "%s: bluetoothctl does not exist. Not supporting bluetooth.\n", Bluetooth_Manager_Thread_Str) ;
    pthread_exit (NULL) ;
  }

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  memset (prev_question, 0x00, sizeof(prev_question)) ;
  Check_Bluetooth_Devices = Check_Bluetooth_Disconnect = 0 ;
  seen_transport = new_connection = 0 ;

  // Wait a short time for all of the threads to have been launched (and all message queues to have been defined)
  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  // ALSO wait for Bluetoothctl_Devices_Cmd && Bluetoothctl_Devices_String to be set
  for (; (Threads_Active == _INITIALISING) || (Bluetoothctl_Devices_Cmd == NULL) || (Bluetoothctl_Devices_String == NULL) ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Power cycle the bluetooth interface and then issue devices command
  orchestrate_bluetoothctl_commands (&Bluetooth_Initialisation[0]) ;
  Check_Bluetooth_Devices = 2 ; // Trigger loading of device list

  // Normal running happens below
  for (; Threads_Active < _SHUTTING_DOWN ;)
  {
    if ((ret_reason = skip_to_string ("(yes/no): ", "Connected: ", "Transport", 0, 1, 0)) == 1)
    {
      // non-zero return code from 'skip_to_string' means yes/no question was found
      // Unfortunately, the same question can be repeated multiple times. We only want to respond ONCE.
      // Luckily, the questions are numbered but it means we need to keep a copy of the numbered question
      if (strcmp (This_Question, prev_question) == 0)
	continue ; // ignore if the same

      strncpy (prev_question, This_Question, sizeof(prev_question)) ;
      Check_Bluetooth_Devices = 12 ; // Check for new devices 3 seconds after sending last yes/no

      if (write (PARENT_WRITE_FD, "yes\n", 4) < 0)
	Goodbye ("Cannot write yes to bluetooth pipe: %s\n", strerror(errno)) ;

      // When we respond with 'yes' messages are going to spew back and forth quickly for a few hundred ms
      // So wait for half a second to let the Rx manager sort out all the return / erase from bluetoothctl
      Sleep_ns (HALF_SECOND) ;
    }

    else if (ret_reason == 2) // Connected:
    {
      // Check for changed devices 3.5 seconds after receiving "Connected: yes/no"
      Check_Bluetooth_Devices = 14 ;
      i = strlen (This_Question) ;

      // connected rather than disconnected
      if ((i > MAC_ADDRESS_LEN) && ((p = strcasestr (This_Line, " Connected: yes")) != NULL))
      {
	// We're going to check the mac address length
	*p = '\0' ;
	p -= MAC_ADDRESS_LEN - 1 ;
	mac_address[0] = '\0' ;

	if ((p[2] == ':') && (p[5] == ':') && (p[8] == ':') && (p[11] == ':') && (p[14] == ':'))
	{
	  if (new_connection == 0)
	    new_connection = 1 ; // a new bluetooth device is attempting to connect. This starts the state machine

	  seen_transport = 0 ; // look for a 'transport' message after a new connection attempt
	  memmove (mac_address, p, MAC_ADDRESS_LEN) ;

	  if (Verbose & _V_BLUETOOTH)
	    Print_Log (T_BLUETOOTH_MAN, "%s Connection attempt %d from '%s'\n", Bluetooth_Manager_Thread_Str, new_connection, mac_address) ;
	}
      }
    }

    else if (ret_reason == 3) // Transport
    {
      // flag that the audio transport has connected (or is in process of connecting)
      if (new_connection > 0)
      {
	seen_transport++ ;

	if (Verbose & _V_BLUETOOTH)
	  Print_Log (T_BLUETOOTH_MAN, "%s Audio transport message seen %d times\n", Bluetooth_Manager_Thread_Str, seen_transport) ;
      }
    }

    else if (Check_Bluetooth_Devices == 1) // ret_reason=0 (we are at a prompt)
    {
      // If this is a new connection - verify the audio has connected (ie transport message) max 3 attempts
      // Only do this on Bookworm (or later) which is OS 12 or later
      if ((OS_Version >= 12) && (new_connection > 0) && (new_connection < 3) && (seen_transport == 0) && (mac_address[0] != '\0'))
      {
	if (Verbose & _V_BLUETOOTH)
	  Print_Log (T_BLUETOOTH_MAN, "%s connection from '%s' missed the audio transport!\n", Bluetooth_Manager_Thread_Str, mac_address) ;

	// audio has not connected  - disconnect and try again up to a maximum of 3 times
	issue_bluetooth_disconnect (mac_address) ;

	c = snprintf (buffer, sizeof(buffer), "connect %s\n", mac_address) ;

	if (write (PARENT_WRITE_FD, buffer, c) < 0)
	  Goodbye ("Cannot write connect to bluetooth pipe: %s\n", strerror(errno)) ;

	new_connection++ ;
	Check_Bluetooth_Devices = 24 ; // Loop again to check - but give it longer this time - 6 seconds
	continue ;
      }

      if ((new_connection >= 3) && (mac_address[0] != '\0'))
      {
	// Although we didn't connect, the bluetooth daemon (bluetoothctl) is still trying to connect with the device.
	// issue a disconnect now so that it ceases attempting to connect.
	issue_bluetooth_disconnect (mac_address) ;

	if (Verbose & _V_BLUETOOTH)
	  Print_Log (T_BLUETOOTH_MAN, "%s connection from '%s' missed the audio transport too many times!\n", Bluetooth_Manager_Thread_Str, mac_address) ;
      }

      new_connection = seen_transport = 0 ; // Stop this state machine
      mac_address[0] = '\0' ;

      // issue a paired-devices command to get a list of
      if (write (PARENT_WRITE_FD, Bluetoothctl_Devices_Cmd, strlen(Bluetoothctl_Devices_Cmd)) < 0)
	Goodbye ("Cannot write to cmd bluetooth pipe: %s\n", strerror(errno)) ;

      // wait for the command to be echoed back at us
      for (; skip_to_string (Bluetoothctl_Devices_String, NULL, NULL, 0, 0, 0) == 0 ;) ; // 0 means prompt was found, and we should keep looking
      build_connected_list (mac_addresses) ; // also count number of bluetooth devices connected and initiate amp turn-on

      // Now trust any devices that we have paired with that are not already trusted
      // This will make it easier for them to reconnect later
      for (n=0 ; (Threads_Active < _SHUTTING_DOWN) && (n < BLUETOOTH_DEV_RECS) && (Bluetooth_Devices[n].mac_address[0] != '\0') ; n++)
      {
	if (Bluetooth_Devices[n].trusted == 0)
	{
	  c = snprintf (buffer, sizeof(buffer), "trust %s\n", Bluetooth_Devices[n].mac_address) ;

	  if (write (PARENT_WRITE_FD, buffer, c) < 0)
	    Goodbye ("Cannot write trust to bluetooth pipe: %s\n", strerror(errno)) ;

	  for (; skip_to_string ("trust ", NULL, NULL, 0, 0, 0) == 0 ;) ; // 0 means prompt was found, and we should keep looking
	  for (; skip_to_string ("Changing ", NULL, NULL, 0, 0, 0) == 0 ;) ; // 0 means prompt was found, and we should keep looking
	}
      }

      // Flag that we have completed bluetooth initialisation
      Check_Bluetooth_Devices = 0 ;
      Bluetooth_Init_Completed = 1 ;
    }

    // The pairing and unpairing needs to be done through the same bluetoothctl instance as all the other work
    // I tried doing this from within the apache environment directly, and it just confuses bluetoothctl
    else if (Initiate_Bluetooth_Pairing)
    {
      if (Verbose & _V_BLUETOOTH)
	Print_Log (T_BLUETOOTH_MAN, "%s Initiating Pairing\n", Bluetooth_Manager_Thread_Str) ;

      orchestrate_bluetoothctl_commands (&Bluetooth_Pairing[0]) ;
      Initiate_Bluetooth_Pairing = 0 ;
    }

    else if (Initiate_Bluetooth_UnPairing)
    {
      if (Verbose & _V_BLUETOOTH)
	Print_Log (T_BLUETOOTH_MAN, "%s Initiating UnPairing\n", Bluetooth_Manager_Thread_Str) ;

      if (write (PARENT_WRITE_FD, Bluetoothctl_Devices_Cmd, strlen(Bluetoothctl_Devices_Cmd)) < 0)
	Goodbye ("Cannot write unpair to bluetooth pipe: %s\n", strerror(errno)) ;

      // wait for the command to be echoed back at us
      for (; skip_to_string (Bluetoothctl_Devices_String, NULL, NULL, 0, 0, 0) == 0 ;) ; // 0 means prompt was found, and we should keep looking

      // Fetch a list of paired device mac addresses - then remove each one of them
      n = build_paired_list (mac_addresses) ;

      for (i=0 ; (Threads_Active < _SHUTTING_DOWN) && (i < n) ; i++)
      {
	c = snprintf (buffer, sizeof(buffer), "remove %s\n", mac_addresses[i]) ;

	if (write (PARENT_WRITE_FD, buffer, c) < 0)
	  Goodbye ("Cannot write remove to bluetooth pipe: %s\n", strerror(errno)) ;

	// Wait until the device has been removed, or the error that it is not available is displayed (error shouldn't happen)
	for (; skip_to_string ("has been removed", "not available", NULL, 1, 1, 0) == 0 ;) ;

	if (Verbose & _V_BLUETOOTH)
	  Print_Log (T_BLUETOOTH_MAN, "%s Removed %s\n", Bluetooth_Manager_Thread_Str, mac_addresses[i]) ;

	// Free the malloced memory
	free (mac_addresses[i]) ;
      }

      Initiate_Bluetooth_UnPairing = 0 ;
    }

    else if (Disconnect_Bluetooth_Devices)
    {
      for (n=0 ; (Threads_Active < _SHUTTING_DOWN) && (n < Num_Bluetooth_Clients) && (Bluetooth_Devices[n].mac_address[0] != '\0') ; n++)
	issue_bluetooth_disconnect (Bluetooth_Devices[n].mac_address) ;

      if (n > 0)
      {
	if (Verbose & _V_BLUETOOTH)
	  Print_Log (T_BLUETOOTH_MAN, "%s Disconnected %d bluetooth devices\n", Bluetooth_Manager_Thread_Str, n) ;

	Check_Bluetooth_Devices = 4 ; // Check for changed devices in 1 second
      }

      Disconnect_Bluetooth_Devices = 0 ;
    }
  }

  Print_Log (T_BLUETOOTH_MAN, "%s unscheduled exit\n", Bluetooth_Manager_Thread_Str) ;
  Threads_Active = _SHUTTING_DOWN ; // aborting
  pthread_exit (NULL) ;
} // Bluetooth_Manager_Thread
///////////////////////////////////////////////////////////////////////////////////////////
// The job of Bluetooth_Rx_Thread is to read characters into a buffer, malloc some memory
// and then store the line that has just been read.
//
// We have to do this because a pipe doesn't contain a lot of buffer/storage, and will not
// block the sending process if it outputs more characters than are able to be processed.
// So we need a thread to consume and buffer the characters as they arrive from the pipe ASAP.
static const char Bluetooth_Rx_Thread_Str[] = "Bluetooth Rx Thread" ;

void *
Bluetooth_Rx_Thread (void *xarg)
{
  int                   c, n ;
  ST			file_info ;
  char			*buffer ;

  UNUSED (xarg) ; // silence the compiler warning

  if (Verbose & _V_BLUETOOTH)
    Print_Log (T_BLUETOOTH_RX, "Entering %s\n", Bluetooth_Rx_Thread_Str) ;

  // Is the bluetooth process running?
  // Does bluetoothctl exist??? If not, don't start this thread
  if ((Bluetoothctl_PID <= 0) || (stat ("/usr/bin/bluetoothctl", &file_info) != 0))
  {
    Print_Log (T_BLUETOOTH_RX, "%s: bluetoothctl does not exist. Not supporting bluetooth.\n", Bluetooth_Rx_Thread_Str) ;
    pthread_exit (NULL) ;
  }

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  for (n=0 ; n < MAX_LINES ; n++)
    Read_Lines[n] = NULL ;

  RB_insert = RB_extract = 0 ;

  // Wait a short time for all of the threads to have been launched (and all message queues to have been defined)
  // Wait for all of the threads to have been launched - specifically so message queues will be defined
  for (; Threads_Active == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Loop reading characters as lines of text, then storing the lines in
  // a circular buffer
  for (; Threads_Active < _SHUTTING_DOWN ; )
  {
    // Fetch a fresh memory buffer and erase it
    buffer = malloc (MAX_LINE+1) ;
    Read_Lines [RB_insert] = buffer ;
    memset (buffer, 0x00, MAX_LINE+1) ;

    for (n=0 ; (n < MAX_LINE) && (Threads_Active < _SHUTTING_DOWN) ; n++)
    {
      // Read one line of characters into buffer
      if ((c = read (PARENT_READ_FD, &buffer[n], 1)) < 0)
	Goodbye ("%s error reading: %s\n", Bluetooth_Rx_Thread_Str, strerror(errno)) ;

      else if (c == 0)
	Goodbye ("%s pipe closed: %s\n", Bluetooth_Rx_Thread_Str, strerror(errno)) ;

      // Did we just read a return? If the consumption process is not on the same line,
      // we can delete all characters and re-read this line
      if ((buffer[n] == '\r') && (RB_extract != RB_insert))
      {
	memset (buffer, 0x00, MAX_LINE+1) ;
	n = -1 ; // because for loop will increment back to 0
	continue ;
      }

      // Otherwise, if this is a return or newline, break out of the accumulation loop
      if ((buffer[n] == '\n') || (buffer[n] == '\r'))
	break ;
    }

    // Remove the trailing return or newline
    buffer[n] = '\0' ;

    // Is the line too long?
    if (n >= MAX_LINE)
    {
      Print_Log (T_BLUETOOTH_RX, "%s Read line too long for allocated buffer!\n", Bluetooth_Rx_Thread_Str) ;
      break ; // this break will cause a reboot
    }

    // The line is not blank and not too long
    if (Verbose & _V_BLUETOOTH)
      Print_Log (T_BLUETOOTH_RX, "%s: %s\n", Bluetooth_Rx_Thread_Str, buffer) ;

    if (++RB_insert >= MAX_LINES)
      RB_insert = 0 ;

    // Has the circular buffer overrun?
    if (Read_Lines [RB_insert] != NULL)
    {
      Print_Log (T_BLUETOOTH_RX, "%s Read buffer overrun!\n", Bluetooth_Rx_Thread_Str) ;

      // Count the number of entries in the queues so we can spit out an error
      for (n=c=0 ; n < MAX_LINES ; n++)
	if (Read_Lines[n] != NULL)
	  c++ ;

      Print_Log (T_TIMER, "%s RB_insert=%d, RB_extract=%d, c=%d, MAX=%d\n", Timer_Thread_Str, RB_insert, RB_extract, c, MAX_LINES) ;
      break ; // this break will cause a reboot
    }
  }

  Print_Log (T_BLUETOOTH_RX, "%s unscheduled exit\n", Bluetooth_Rx_Thread_Str) ;
  Threads_Active = _SHUTTING_DOWN ; // aborting
  pthread_exit (NULL) ;
} // Bluetooth_Rx_Thread
///////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////
// This structure defines the command line argument options

static const struct option long_options[] =
{
  // char *name         name of the long option
  // int has_arg        0=none, 1=required, 2=optional argument
  // int *flag          NULL:return value = val, ptr:return 0 and fill ptr with value
  // int val            value to return or load into *flag (???)

  {"ambient",		no_argument,		0,	'a'},
  {"bluetooth",		required_argument,	0,	'b'},
  {"command",		required_argument,	0,	'c'},
  {"daemonise",		no_argument,		0,	'd'},
  {"jump",		required_argument,	0,	'j'},
  {"help",		no_argument,		0,	'h'},
  {"license",		no_argument,		0,	'l'},
  {"log",		required_argument,	0,	'L'},
  {"media",		required_argument,	0,	'm'},
  {"PWM",		required_argument,	0,	'p'},
  {"query",		no_argument,		0,	'q'},
  {"reload",		no_argument,		0,	'r'},
  {"time",		required_argument,	0,	't'},
  {"volume",		required_argument,	0,	'v'},
  {"stop",		no_argument,		0,	'x'},
  {"brightness-test",	no_argument,		0,	'y'},
  {"display-test",	no_argument,		0,	'z'},

  {"verbose",		no_argument,		0,	'V'},
  {"vamp",		no_argument,		0,	'A'},
  {"vbutton",		no_argument,		0,	'B'},
  {"vclock",		no_argument,		0,	'C'},
  {"vdisplay",		no_argument,		0,	'D'},
  {"vmedia",		no_argument,		0,	'M'},
  {"vsetup",		no_argument,		0,	'S'},
  {"vtimer",		no_argument,		0,	'T'},
  {"vbluetooth",	no_argument,		0,	'U'},
  {"vmulticast",	no_argument,		0,	'W'},
  {"vdimmer",		no_argument,		0,	'X'},
  {"valarm",		no_argument,		0,	'Y'},
  {"vampmedia",		no_argument,		0,	'Z'},

  {NULL,                no_argument,            NULL,   0}
} ;

// opstring: single character equivalents of long_options.
// If character followed by ':', an argument is required
// If character followed by '::', an argument is optional
// If first character of opstring is ':', the return value distinguishes invalid and missing
// option by returning '?' or ':'
// leading '-' returns 0x01 for params not starting with '-' (useful for optional arguments)

static const char short_options[] = "-:ab:c:dj:hlL:m:p:qrt:v:xyzVABCDMSTUWXYZ" ;
///////////////////////////////////////////////////////////////////////////////////////////
int
main (int argc, char **argv)
{
  int				index, c, error, ret, cmd_count, pair ;
  uint8_t			cmd_type ;
  char				string [FILENAME_SIZE] ;
  char				path[SETUP_FIELD_LEN+1] ;
  mqd_t				Q = -1 ;
  SETUP_MSG_BUF			setup_msg ;
  DIM_MSG_BUF			dim_msg ;
  MEDIA_MSG_BUF			media_msg ;
  MEDIA_RESPONSE_BUF		playlist_msg ;
  SUBTASK			*subtask, **thread_param ;
  struct sigaction		sa ;
  struct itimerval		timer ;
  sigset_t			signal_set;
  void				*pret ;
  struct ip_mreq		mreq ;
  struct timespec		ts ;

  // Divine the time zone from system settings, and pre-configure the time routines
  tzset() ;
  path[0] = '\0' ;
  cmd_count = pair = 0 ;
  cmd_type = 0xff ;
  //------------------------------------------------------------------------------------------------------------------
  // Ensure the umask is what we are expecting
  //  files created with owner+group to have read/write but no execute, and world has read only
  umask (S_IXUSR | S_IXGRP | S_IXOTH | S_IWOTH) ;
  //------------------------------------------------------------------------------------------------------------------
  // This section reads the command-line arguments
  //
  // getopt_long (int argc, char * const argv[], const char *optstring, const struct option *longopts, int *longindex)
  // return values
  //  '0': set a flag according to the long option table
  //  '1': optarg points at a plain command line argument
  //  '?': invalid option
  //  ':': missing option argument
  //  'x': option character x
  //  -1 : end of options
  //
  // optind is defined as a global. it represents the parameter number that's been scanned last
  // optopt is defined as a global. It represents the character that caused the error

  // Command line parameters may be entered in any order. Some parameters MAY affect the way others are interpreted.
  // To deal with this, we will read the command line parameters in TWO passes. We need to read any parameter
  // that could affect another in the first pass, and we need to read any parameter that COULD be affected in
  // the second pass. There is only one parameter that is important in this regard, and it is '-v' (verbose).
  // and others that belong to that set. But we will read other parameters in the first pass anyway.

  for (error=0,index=-1 ; (c = getopt_long (argc, argv, short_options, long_options, &index)) != -1 ; index=-1)
  {
    // fprintf (stderr, "argv[%2d], argc=%d: longopt_name='%s', c='%c', optopt=%d, optarg='%s'\n", optind, argc, (index < 0) ? "-used short-" : long_options[index].name, c, optopt, optarg) ;

    if (optarg)
    {
      if (*optarg == '-')
      {
	// There's a missing argument. Change the constants so the right Error is displayed
	optopt = c ;
	c = ':' ;

	fprintf (stderr, "Argument missing after '-'\n\n") ;
	error++ ;
	continue ;
      }
    }

    switch (c)
    {
      case 'a': // Read ambient light level
	Read_Ambient = 1 ;
	break ;

      case 'b': // Initiate bluetooth pairing or unpairing
	if (pair > 0)
	{
	  fprintf (stderr, "Can't issue more than one -b command at the same time\n\n") ;
	  error++ ;
	}

	else if (strcasecmp (optarg, "pair") == 0)
	  pair = S_PAIR ;

	else if (strcasecmp (optarg, "unpair") == 0)
	  pair = S_UNPAIR ;

	else
	{
	  fprintf (stderr, "Unrecognised -b command '%s'\n\n", optarg) ;
	  error++ ;
	}

	break ;

      case 'c':
	if (strcasecmp (optarg, "playlist") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_PLAYLIST_REQUEST ;
	}

	else if (strcasecmp (optarg, "clocks") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_PEERS_REQUEST ;
	}

	else if (strcasecmp (optarg, "pause") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_PAUSE ;
	}

	else if (strcasecmp (optarg, "stop") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_STOP ;
	}

	else if (strcasecmp (optarg, "next") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_NEXT ;
	}

	else if (strcasecmp (optarg, "prev") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_PREV ;
	}

	else if (strcasecmp (optarg, "restart") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_RESTART ;
	}

	else if (strcasecmp (optarg, "seek-fwd") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_SEEK_FWD ;
	}

	else if (strcasecmp (optarg, "seek-back") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_SEEK_BACK ;
	}

	else if (strcasecmp (optarg, "clear") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_PLAYLIST_CLEAR ;
	}

	else if (strcasecmp (optarg, "shuffle") == 0)
	{
	  cmd_count++ ;
	  cmd_type = MEDIA_SHUFFLE ;
	}

	else
	{
	  fprintf (stderr, "Unrecognised -c command '%s'\n\n", optarg) ;
	  error++ ;
	}

	break ;

      case 'd': // daemonise
	Daemonise = 1 ;
	break ;

      case 'j': // jump to track number
	if ((sscanf (optarg, "%d", &Track_Num) != 1) || (Track_Num <= 0) || (Track_Num > 100000))
	{
	  fprintf (stderr, "Track number must be a positive integer\n\n") ;
	  error++ ;
	  Track_Num = -1 ;
	  break ;
	}

	cmd_type = MEDIA_PLAYLIST_JUMP ;
	break ;

      case 'h': // display the help message
	error++ ;
	break ;

      case 'l': // show license
	printf ("-------------------------------------------------------------------------------\n") ;
	printf ("Digital Alarm Clock project (version " VERSION ")\n\n") ;
	printf ("Developed by Stefan Keller-Tuberg\n") ;
	printf ("Copyright (C) 2023,2024 Stefan Keller-Tuberg\n") ;
	printf ("skt@keller-tuberg.homeip.net\n") ;
	printf ("\n") ;
	printf ("The Digital Alarm Clock project is free software: you can redistribute it and/or\n") ;
	printf ("modify it under the terms of the GNU General Public License as published by the\n") ;
	printf ("Free Software Foundation, either version 3 of the License, or (at your option)\n") ;
	printf ("any later version.\n") ;
	printf ("\n") ;
	printf ("The Digital Alarm Clock project is distributed in the hope that it will be\n") ;
	printf ("useful, but WITHOUT ANY WARRANTY; without even the implied warranty of\n") ;
	printf ("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n") ;
	printf ("See the GNU General Public License for more details.\n") ;
	printf ("\n") ;
	printf ("You should have received a copy of the GNU General Public License along with\n") ;
	printf ("the Digital Alarm Clock project. If not, see <https://www.gnu.org/licenses/>.\n") ;
	printf ("-------------------------------------------------------------------------------\n") ;
	return EXIT_SUCCESS ;

      case 'L': // Log messages to file
	// Open the log file for appending (it will open a fresh file if it doesn't already exist)
	if ((Log_File = fopen (optarg, "a")) == NULL)
	{
	  fprintf (stderr, "Cannot open log file '%s': %s\n\n", optarg, strerror(errno)) ;
	  error++ ;
	}

	break ;

      case 'm': // add media to the queue
	// read a string - which will be assumed to be URL or path to media file/folder
	strncpy (path, optarg, SETUP_FIELD_LEN) ;
	cmd_count++ ;
	cmd_type = MEDIA_ADD ;
	break ;

      case 'p': // Trial PWM
	// read a double value and ensure its in the expected range
	if ((sscanf (optarg, "%lf", &Trial_PWM) != 1) || (Trial_PWM < 0.0) || (Trial_PWM > 1.0))
	{
	  fprintf (stderr, "PWM ratio must be between 0.0 and 1.0\n\n") ;
	  error++ ;
	  Trial_PWM = -1.0 ;
	}

	break ;

      case 'q': // query the version
	printf (VERSION "\n") ;
	return EXIT_SUCCESS ;

      case 'r': // Reload configuration files
	Reload = 1 ;
	break ;

      case 't': // Specify a time (used in conjunction with 'm', or to change the playing duration)
	if ((sscanf (optarg, "%d", &Media_Time) != 1) || (Media_Time < 0) || (Media_Time > MAX_DURATION))
	{
	  fprintf (stderr, "Media time is specifed in minutes, and must be between 0 minutes (stop current stream) and three hours (%d)\n\n", MAX_DURATION) ;
	  error++ ;
	}

	break ;

      case 'v': // Set volume
	if ((sscanf (optarg, "%" SCNd16, &Set_Volume) != 1) || (Set_Volume < 0) || (Set_Volume > MAX_VOLUME))
	{
	  fprintf (stderr, "Volume must be between 0 (quietest) and %d (maximum)\n\n", MAX_VOLUME) ;
	  error++ ;
	  Set_Volume = -1 ;
	}

	break ;

      case 'x': // Stop Clock
	Stop_Clock = 1 ;
	break ;

      case 'y': // Brightness Test
	Brightness_Test = 1 ;
	break ;

      case 'z': // Display test
	Display_Test = 1 ;
	break ;

      case 'A': // Enable amp manager thread debugging
	Cmdline_Verbose |= _V_AMP ;
	break ;

      case 'B': // Enable buttonmanager thread debugging
	Cmdline_Verbose |= _V_BUTTON ;
	break ;

      case 'C': // Enable clock thread debugging
	Cmdline_Verbose |= _V_CLOCK ;
	break ;

      case 'D': // Enable display driver thread debugging
	Cmdline_Verbose |= _V_DISPLAY ;
	break ;

      case 'M': // Enable media player thread debugging
	Cmdline_Verbose |= _V_MEDIA ;
	break ;

      case 'S': // Enable setup thread debugging
	Cmdline_Verbose |= _V_SETUP ;
	break ;

      case 'T': // Enable timer thread debugging
	Cmdline_Verbose |= _V_TIMER ;
	break ;

      case 'V': // Enable all debugging options
	Cmdline_Verbose = _V_DISPLAY | _V_CLOCK	| _V_DIMMER | _V_TIMER | _V_SETUP | _V_MEDIA | _V_AMP | _V_BUTTON | _V_MULTICAST | _V_BLUETOOTH ;
	break ;

      case 'X': // Enable display dimmer thread debugging
	Cmdline_Verbose |= _V_DIMMER ;
	break ;

      case 'W': // Enable multicast debugging
	Cmdline_Verbose |= _V_MULTICAST ;
	break ;

      case 'U': // Enable bluetooth debugging
	Cmdline_Verbose |= _V_BLUETOOTH ;
	break ;

      case 'Y':
	Cmdline_Verbose |= _V_ALARM ;
	break ;

      case 'Z':
	Cmdline_Verbose |= _V_AMP_MEDIA ;
	break ;

      case '?':
	// This means there's an unknown option
	fprintf (stderr, "Unknown command line option: '%c'\n\n", optopt) ;
	error++ ;
	break ;

      default: // for any other parameter, skip and process during the second pass
	break ;
    }
  }
  //------------------------------------------------------------------------------------------------------------------
  if ((Media_Time == -1) && (path[0] != '\0'))
  {
    fprintf (stderr, "Media Time must be specified (using -t) when adding media (using -m)\n\n") ;
    error++ ;
  }

  c = 0 ;

  if (Cmdline_Verbose)
    c++ ;

  if (Daemonise)
    c++ ;

  if (c > 1)
  {
    fprintf (stderr, "Can't use -d (daemonise) and any debugging options at the same time\n\n") ;
    error++ ;
  }

  if (cmd_count > 1)
  {
    fprintf (stderr, "Can't issue more than one '-c' + '-m' command at the same time\n\n") ;
    error++ ;
  }

  if (error == 0)
  {
    c = 0 ;

    if (Track_Num > 0)
      c++ ;

    if (Trial_PWM >= 0.0)
      c++ ;

    if (Reload)
      c++ ;

    if (pair)
      c++ ;

    if (Read_Ambient)
      c++ ;

    if (Brightness_Test)
      c++ ;

    if (Display_Test)
      c++ ;

    if (Stop_Clock)
      c++ ;

    if ((Set_Volume >= 0) || (Media_Time >= 0))
      c++ ;

    if (c > 1)
    {
      fprintf (stderr, "Can't use more than one of -a, -j, -p, -r, -v, -x, -y or -z at the same time\n\n") ;
      error++ ;
    }
  }
  //------------------------------------------------------------------------------------------------------------------
  if (error)
  {
    strncpy (string, argv[0], FILENAME_SIZE-1) ;
    string[FILENAME_SIZE-1] = '\0' ;

    fprintf (stderr, "Usage: %s {-abcdjhlLmpqrtvxyzVABCDMSTWXYZ}\n", basename(string)) ;

    fprintf (stderr, "-a\t\tDisplay ambient light measurement, current volume, remaining playing time (in minutes),\n") ;
    fprintf (stderr, "\t\tand the number of connected bluetooth devices.\n") ;
    fprintf (stderr, "-b [cmd]\tSend bluetooth command. cmd must be one of 'pair', 'unpair'.\n") ;
    fprintf (stderr, "-c [cmd]\tSend command to the media player.\n") ;
    fprintf (stderr, "\t\tcmd must be one of 'stop', 'pause', 'next', 'prev', 'restart', 'seek-fwd', 'seek-back',\n") ;
    fprintf (stderr, "\t\t'clear', 'playlist', 'shuffle', 'clocks'.\n") ;
    fprintf (stderr, "-d\t\tDaemonise this process (run in the background and don't die when shell closes).\n") ;
    fprintf (stderr, "-j [NUM]\tJump to the specified track number in currently playing playlist.\n") ;
    fprintf (stderr, "-h\t\tDisplay this usage information.\n") ;
    fprintf (stderr, "-l\t\tDisplay copyleft information.\n") ;
    fprintf (stderr, "-L [filename]\tSave logging output to the specified filename.\n") ;
    fprintf (stderr, "-m 'media'\tAppend media path/URL to the current playlist. Quote strings containing special\n") ;
    fprintf (stderr, "\t\tcharacters. Note: -m must be used in conjunction with -t\n") ;
    fprintf (stderr, "-p [fraction]\tTemporarily set PWM ratio to fraction between 0.0 and 1.0.\n") ;
    fprintf (stderr, "-q\t\tQuery the software version.\n") ;
    fprintf (stderr, "-r\t\tReload configuration from '%s'\n", Setup_Filename) ;
    fprintf (stderr, "\t\tand alarms from '%s'.\n", Alarm_Filename) ;
    fprintf (stderr, "-t [duration]\tSpecify a time duration (in minutes) to play media stream specified with -m.\n") ;
    fprintf (stderr, "-v [volume]\tSet volume to specified level. 0=silent, 31=maximum.\n") ;
    fprintf (stderr, "-x\t\tStop the alarm-clock daemon.\n") ;
    fprintf (stderr, "-y\t\tInitiate a brightness test to check for uniform segment brightness.\n") ;
    fprintf (stderr, "-z\t\tInitiate a display test to check for broken LED segments.\n") ;

    fprintf (stderr, "\n-V\tBe verbose. Enable ALL the debug and trace information listed below.\n") ;
    fprintf (stderr, "-A\tDisplay debug information from the amp manager thread.\n") ;
    fprintf (stderr, "-B\tDisplay debug information from the button manager thread.\n") ;
    fprintf (stderr, "-C\tDisplay debug information from the clock thread.\n") ;
    fprintf (stderr, "-D\tDisplay debug information from the display driver thread.\n") ;
    fprintf (stderr, "-M\tDisplay debug information from the media player thread.\n") ;
    fprintf (stderr, "-S\tDisplay debug information from the setup manager thread.\n") ;
    fprintf (stderr, "-T\tDisplay debug information from the system timer thread.\n") ;
    fprintf (stderr, "-U\tDisplay debug information from the bluetooth threads.\n") ;
    fprintf (stderr, "-W\tDisplay debug information from the multicast threads.\n") ;
    fprintf (stderr, "-X\tDisplay debug information from the display dimmer thread.\n") ;
    fprintf (stderr, "-Y\tDisplay debug information about alarm setting and tripping.\n") ;
    fprintf (stderr, "-Z\tDisplay reduced debug information from the the media and amp manager threads.\n") ;

    fprintf (stderr, "\n(This is version " VERSION ")\n") ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------------------------------------------------------------------
  if ((Trial_PWM >= 0.0) || Brightness_Test || Stop_Clock)
  {
    if (Cmdline_Verbose)
      Print_Log (T_NULL, "Trial PWM=%.3f, Brightness_Test=%hu, Stop_Clock=%hu\n", Trial_PWM, Brightness_Test, Stop_Clock) ;

    // We are going to tentatively try to open a queue to a daemon (is it there?)
    if ((Q = Open_Msg_Queue_Tx (T_NULL, Dim_Msgq_name, 1)) < 0)
    {
      Print_Log (T_NULL, "Trial PWM: Is a daemon actually running?\n") ;
      goto invoke_reboot ;
    }

    // There's a queue there - we can send the message
    if (Trial_PWM >= 0.0)
    {
      dim_msg.type = DIM_SET  ;		// Command to temporarily change the dimmer level
      dim_msg.seconds = 7 ;		// Try the alternate dimmer value for 7 seconds
    }
    else if (Brightness_Test)
    {
      dim_msg.type = DIM_BRIGHTNESS_TEST ; // Command to run brightness test
      dim_msg.seconds = 7 ;		// Try the brightness test for 7 seconds dim, then 7 seconds bright
    }

    else
    {
      dim_msg.type = DIM_STOP_CLOCK ;	// Command to stop the daemon
      dim_msg.seconds = 0 ;
    }

    dim_msg.pwm_ratio = Trial_PWM ; // Set this ratio temporarily (not used for the brightness test messages)

    if (mq_send (Q, (char *)&dim_msg, sizeof(dim_msg), 0) < 0)
    {
      Print_Log (T_NULL, "Couldn't send a PWM trial message: %s\n", strerror(errno)) ;
      mq_close (Q) ;
      return EXIT_FAILURE ;
    }

    // Tidy up before exiting
    mq_close (Q) ;
    return EXIT_SUCCESS ;
  }
  //------------------------------------------------------------------------------------------------------------------
  if (Reload)
  {
    if (Cmdline_Verbose)
      Print_Log (T_NULL, "Reload configuration\n") ;

    // We are going to tentatively try to open a queue to a daemon (is it there?)
    if ((Q = Open_Msg_Queue_Tx (T_NULL, Setup_Msgq_name, 1)) < 0)
    {
      Print_Log (T_NULL, "Reload Config: Is a daemon actually running?\n") ;
      mq_close (Q) ;
      goto invoke_reboot ;
    }

    // There's a queue there - we can send the message to play the media file
    setup_msg.type = S_RELOAD ;
    setup_msg.volume = 0 ;
    setup_msg.ambient = 0 ;
    setup_msg.duration = 0 ;
    setup_msg.num_blue = 0 ;

    if (mq_send (Q, (char *)&setup_msg, sizeof(setup_msg), 0) < 0)
    {
      Print_Log (T_NULL, "Couldn't send a reload message: %s\n", strerror(errno)) ;
      mq_close (Q) ;
      return EXIT_FAILURE ;
    }

    // Tidy up before exiting
    mq_close (Q) ;
    return EXIT_SUCCESS ;
  }
  //------------------------------------------------------------------------------------------------------------------
  if (pair)
  {
    if (Cmdline_Verbose)
      Print_Log (T_NULL, "Initiate bluetooth pairing/unpairing\n") ;

    // We are going to tentatively try to open a queue to a daemon (is it there?)
    if ((Q = Open_Msg_Queue_Tx (T_NULL, Setup_Msgq_name, 1)) < 0)
    {
      Print_Log (T_NULL, "Reload Config: Is a daemon actually running?\n") ;
      mq_close (Q) ;
      goto invoke_reboot ;
    }

    // There's a queue there - we can send the message to play the media file
    setup_msg.type = pair ;
    setup_msg.volume = 0 ;
    setup_msg.ambient = 0 ;
    setup_msg.duration = 0 ;
    setup_msg.num_blue = 0 ;

    if (mq_send (Q, (char *)&setup_msg, sizeof(setup_msg), 0) < 0)
    {
      Print_Log (T_NULL, "Couldn't send pairing/unpairing message: %s\n", strerror(errno)) ;
      mq_close (Q) ;
      return EXIT_FAILURE ;
    }

    // Tidy up before exiting
    mq_close (Q) ;
    return EXIT_SUCCESS ;
  }
  //------------------------------------------------------------------------------------------------------------------
  // If you're a pthreads newb, the following may confuse you without explanation. Why am I sending a message with
  // the purpose of reading the GLOBAL variable 'Instantaneous_Light_Level', or why not just call Read_A_to_D directly?
  // The first answer is that although Linux may be running this instance of alarm_clock out of the same memory that
  // is running the daemon, the global variables in this instance will be DIFFERENT than the global variables used
  // by the daemon: the two instances of alarm_clock cannot communicate with each other via memory. i.e. there are two
  // unique instances of the global variables: one used by the daemon and the other used by this instance. This is why
  // we are using message queues here: one queue to send the message to the daemon, and a second queue to receive the
  // response. Why can't we just call Read_A_to_D? In general, this would work 99.99% of the time. But there's no way
  // to set up a mutex between the daemon and this instance of alarm-clock to enforce unique access, so there is a
  // very slim chance that there may be trouble with simultaneous access to the SPI. The only way to avoid this is
  // to ensure that only ONE thread accesses the A to D SPI peripheral, and that thread is in the daemon. So we will
  // instead send a message to the daemon to read its own version of the global variable, and then return that value
  // via a separate message queue.
  if (Read_Ambient || Display_Test)
  {
    // In order to read the LDR value, we need to establish a temporary response queue, then send a message to the
    // setup process, then wait for the response.
    // Firstly, open the response queue so its sitting waiting already.
    setup_msg.responseQ = getpid() ; // get this process's PID to use as a unique message queue name
    snprintf (Response_Msgq_name, TEMP_NAME_LEN, "/R%u", (unsigned)setup_msg.responseQ) ;
    Response_Msgq = Open_Msg_Queue_Rx (T_NULL, Response_Msgq_name, 5, sizeof(setup_msg)) ;

    // Secondly, open the transmit queue to send the request to read the LDR
    if ((Q = Open_Msg_Queue_Tx (T_NULL, Setup_Msgq_name, 1)) < 0)
    {
      Print_Log (T_NULL, "Read Ambient: Is a daemon actually running?\n") ;
      goto _abort_ldr ;
    }

    // Now we can send the request message
    setup_msg.type = (Read_Ambient) ? S_READ_LDR_AND_STATUS : S_DISPLAY_TEST ;
    setup_msg.volume = 0xffff ;
    setup_msg.ambient = 0xffff ;
    setup_msg.duration = 0xffff ;
    setup_msg.num_blue = 0xffff ;

    if (mq_send (Q, (char *)&setup_msg, sizeof(setup_msg), 0) < 0)
    {
      Print_Log (T_NULL, "Couldn't send a read LDR message: %s\n", strerror(errno)) ;
      goto _abort_ldr1 ;
    }

    // Next: hang around and wait for the response
    for (; Threads_Active < _SHUTTING_DOWN ;)
    {
      if (Wait_For_Next_Message (T_NULL, Response_Msgq, (void *)&setup_msg, sizeof(setup_msg), sizeof(setup_msg), "main") >= (int)(sizeof(setup_msg)))
	break ;
    }

    // The value returned in the message is the LDR level. Ensure it is within a sensible range (12 bits)
    // The values MAX_DURATION+1 and MAX_DURATION+2 are magic numbers. Anything larger than this is an error
    if ((setup_msg.ambient > 0xfff) || (setup_msg.volume < -1) || (setup_msg.volume > 31) || (setup_msg.duration > (MAX_DURATION+2)) || (setup_msg.num_blue > BLUETOOTH_DEV_RECS))
    {
      // Something fishy. One of the values is out of the expected range
      Print_Log (T_NULL, "One or more expected values out of range: ambient=0x%hx, volume=%hd, duration=%hd, num_blue=%hu\n",
		 setup_msg.ambient, setup_msg.volume, setup_msg.duration, setup_msg.num_blue) ;
    _abort_ldr1:
      mq_close (Q) ;
    _abort_ldr:
      mq_unlink (Response_Msgq_name) ;
      mq_close (Response_Msgq) ;
      Response_Msgq_name[0] = '\0' ;
      Response_Msgq = 0 ;
      goto invoke_reboot ;
    }

    // We have the result and it appears to be valid. Output as an unadorned number.
    printf ("%hu,%hd,%hd,%hu\n", setup_msg.ambient, setup_msg.volume, setup_msg.duration, setup_msg.num_blue) ;

    // Tidy up before exiting
    mq_close (Q) ;
    mq_unlink (Response_Msgq_name) ; // This call causes the queue name to be removed from the system, but queue remains open
    mq_close (Response_Msgq) ; // This call closes the message queue descriptor and that should trigger removal from system
    Response_Msgq_name[0] = '\0' ;
    Response_Msgq = 0 ;
    return EXIT_SUCCESS ;
  }
  //------------------------------------------------------------------------------------------------------------------
  // The following section deals with the other options that send messages from this instance to the daemon
  if ((cmd_count == 1) || (Track_Num > 0))
  {
    if (Cmdline_Verbose)
    {
      if (cmd_count == 1)
	Print_Log (T_NULL, "Sending media command=%hu, media='%s', Duration=%d (minutes), Volume=%hd, Index=%d\n", cmd_type, path, Media_Time, Set_Volume, Track_Num) ;
      else
	Print_Log (T_NULL, "Sending Media Duration update=%d (minutes)\n", cmd_type, Media_Time) ;
    }

    // We are going to tentatively try to open a queue to a daemon (is it there?)
    if ((Q = Open_Msg_Queue_Tx (T_NULL, Media_Msgq_name, 1)) < 0)
    {
      Print_Log (T_NULL, "Media command: Is a daemon actually running?\n") ;
      goto invoke_reboot ;
    }

    media_msg.responseQ = 4 ; // a dummy number that will cause alarm clock to crash if there's a bug. #4 will appear in debug log

    // If the command is to fetch the playlist or clocks list, we need to open a response queue
    if ((cmd_type == MEDIA_PLAYLIST_REQUEST) || (cmd_type == MEDIA_PEERS_REQUEST))
    {
      // Open the response queue so its sitting waiting already.
      media_msg.responseQ = getpid() ; // get this process's PID to use as a unique message queue name
      snprintf (Response_Msgq_name, TEMP_NAME_LEN, "/R%u", (unsigned)media_msg.responseQ) ;
      Response_Msgq = Open_Msg_Queue_Rx (T_NULL, Response_Msgq_name, 5, sizeof(playlist_msg)) ;
    }

    // Send a message to the media player thread
    // An integer indicating the number of seconds to play the media stream - zero means infinity
    media_msg.seconds = (Media_Time > 0) ? Media_Time * 60 : 0 ;
    media_msg.type = cmd_type ;
    media_msg.index = Track_Num ;
    media_msg.volume = Set_Volume ;
    media_msg.init_vol_offset = 0 ;
    media_msg.targ_vol_offset = 0 ;
    c = MEDIA_MSG_LEN + 1 + snprintf (media_msg.url_path, SETUP_FIELD_LEN, "%s", path) ; // copy the path

    // Send the media request to the media thread
    if (mq_send (Q, (char *)&media_msg, c, 0) < 0)
    {
      Print_Log (T_NULL, "Couldn't send media msg: %s\n", strerror(errno)) ;
      goto _abort_playlist ;
    }

    // If this is a playlist command, hang around for the responses
    if ((cmd_type == MEDIA_PLAYLIST_REQUEST) || (cmd_type == MEDIA_PEERS_REQUEST))
    {
      for (; Threads_Active < _SHUTTING_DOWN ;)
      {
	ret = Wait_For_Next_Message (T_NULL, Response_Msgq, (void *)&playlist_msg, 4, sizeof(playlist_msg), "main") ;

	if ((ret < 4) || (ret > (SETUP_FIELD_LEN-1)))
	  goto _abort_playlist ;

	// Force a null termination
	playlist_msg[ret] = '\0' ;

	// Is this the terminating string?
	if (strcmp (playlist_msg, "end") == 0)
	  break ;

	// Display the playlist message as received. There are three CSV columns. The
	// second column is enclosed in double quotes
	printf ("%s\n", playlist_msg) ;
      }

    _abort_playlist:
      mq_unlink (Response_Msgq_name) ;
      mq_close (Response_Msgq) ;
      Response_Msgq_name[0] = '\0' ;
      Response_Msgq = 0 ;
    }

    // Tidy up before exiting
    if (Q >= 0)
      mq_close (Q) ;

    return EXIT_SUCCESS ;
  }
  //------------------------------------------------------------------------------------------------------------------
  // This section sends message to change the volume level
  if ((Set_Volume >= 0) || (Media_Time >= 0))
  {
    if (Cmdline_Verbose)
      Print_Log (T_NULL, "Set Volume to %hd\n", Set_Volume) ;

    // We are going to tentatively try to open a queue to a daemon (is it there?)
    if ((Q = Open_Msg_Queue_Tx (T_NULL, Setup_Msgq_name, 1)) < 0)
    {
      Print_Log (T_NULL, "Set Vol: Is a daemon actually running?\n") ;
      mq_close (Q) ;

    invoke_reboot: // reboot the Pi - hopefully this will get the alarm clock working once more
      // Wait 15 seconds to give the human a chance to abort
      for (c=15 ; c > 0 ; c--)
      {
	Print_Log (T_NULL, "\rSystem rebooting in %d seconds. <CTRL-C> to abort!     ", c) ;
	sleep (1) ;
      }

      Print_Log (T_NULL, "\rSystem rebooting NOW                     \n") ;
      system ("sudo shutdown -r now") ; // Reboot immediately, because we have already waited

      // Maybe there's no point in returning, but we will do so anyway
      return EXIT_FAILURE ;
    }

    // There's a queue there - we can send the message to set volume or duration
    setup_msg.type = S_SET_VOL_DUR ;
    setup_msg.volume = (int16_t)Set_Volume ;
    setup_msg.duration = (int16_t)Media_Time ;
    setup_msg.ambient = 0xffff ;
    setup_msg.num_blue = 0xffff ;
    setup_msg.responseQ = 4 ; // a dummy number that will cause alarm clock to crash if there's a bug. #4 will appear in debug log

    if (mq_send (Q, (char *)&setup_msg, sizeof(setup_msg), 0) < 0)
    {
      Print_Log (T_NULL, "Couldn't send set volume message: %s\n", strerror(errno)) ;
      mq_close (Q) ;
      return EXIT_FAILURE ;
    }

    // Tidy up before exiting
    mq_close (Q) ;
    return EXIT_SUCCESS ;
  }
  //------------------------------------------------------------------------------------------------------------------
  if (Cmdline_Verbose)
    Print_Log (T_NULL, "Commencing initialisation\n") ;
  //------------------------------------------------------------------------------------------------------------------
  // We can only run one instance of this programme at a time, because the GPIO pins need to be dedicated to one
  // and only one instance at a time.
  Abort_If_Another_Instance_Already_Running (0) ;
  //------------------------------------------------------------------------------------------------------------------
  // If we have been asked to daemonise, now is the time to do this
  if (Daemonise)
  {
    if (Cmdline_Verbose)
      Print_Log (T_NULL, "Launching Daemon\n") ;

    strncpy (string, argv[0], FILENAME_SIZE-1) ;
    string[FILENAME_SIZE-1] = '\0' ;

    if (Fork_Off_A_Daemon(string) == 0)
    {
      // I am the parent (ie not the daemon)
      // we can silently exit
      return EXIT_SUCCESS ;
    }

    // To reach here, I am the daemon child
  }

  // To reach here, I am either the original process (ie user didn't ask to be daemonised), or the daemon child
  //------------------------------------------------------------------------------------------------------------------
#ifdef	USE_GPIO_DAEMON
  // Checks to see if the gpiod library daemon is running, and if not, starts it
  if (Start_PiGpiod() != EXIT_SUCCESS)
    Goodbye ("Error starting pigpiod library\n") ;
#endif

  // Don't abort if the bluetooth process cannot start.... the Pi2 doesn't support it (for example)
  if (Start_Bluetooth() != EXIT_SUCCESS)
    Print_Log (T_NULL, "Error starting bluetoothctl server. Is bluetooth supported on this hardware?\n") ;

  //------------------------------------------------------------------------------------------------------------------
  // Reset the default volume (this is a kind of kludge because alsa volume is messed up in the initial bookworm release)
  Set_Default_Volume () ;	// note: this should be the only place this is called (as currently written)
  //------------------------------------------------------------------------------------------------------------------
  Dimmer_Timeout.tv_sec = 0 ;	// Setting tv_sec to zero stops the timer

  if (determine_hostname_and_ip_address (T_NULL) != EXIT_SUCCESS)
    return EXIT_FAILURE ;

  if (Cmdline_Verbose)
    Print_Log (T_NULL, "My networking details are [%s / %s]\n", My_Name, My_IPv4_Address) ;
  //------------------------------------------------------------------------------------------------------------------
  // This section is all about making sure the right signals go to the right places. In a multi-threaded process, the
  // default is for a signal to be handled by the first available thread that gets scheduled. For some signals (eg
  // timeouts), this is not OK. We want a specific timeout handler thread to handle timeouts. For other signals such
  // as exceptions, we want those to end up in a routine that cleanly releases all the resources and then exits as
  // gracefully as possible - for example, we want to shut down the pigpiod cleanly rather than barf-exiting.
  //
  // The assignment of signal handlers happens here.
  sigemptyset (&signal_set) ; // start with an empty set
  sigaddset (&signal_set, SIGPIPE) ; // then add the sigpipe error (we are going to ignore signal and catch with write failures)
  sigaddset (&signal_set, SIGCHLD) ; // add child exited error (which we are going to ignore as above)
  sigaddset (&signal_set, SIGWINCH) ; // add window changed signal (we are going to ignore this too)

  if (pthread_sigmask(SIG_BLOCK, &signal_set, NULL) != 0)
  {
    Print_Log (T_NULL, "Failed to block masked events: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Now create a generic signal handler that will be used for for most exceptions (i.e. the ones we want to trap)
  // The various signals are numbered from 1 to SIGRTMAX
  for (c=1 ; c <= SIGRTMAX ; c++)
  {
    // Keep SegFault going to the default handler, and skip over sigpipe that we just blocked
    if ((c == SIGSEGV) || (c == SIGPIPE))
      continue ;

    memset (&sa, 0, sizeof(sa)) ;
    sa.sa_handler = Release_Resources ; // point the signal handler to Release_Resources
    sigaction (c, &sa, NULL) ;
  }

  // Now set up the timer interrupt. It will be using the SIGALRM signal.
  // We need to block SIGALRM from all threads except the one thread that we designate to process those alarms
  // So block this alarm in all threads, then enable it again in just that one thread that will be the handler.
  sigemptyset (&signal_set) ; // start with an empty set
  sigaddset (&signal_set, SIGALRM); // add the SIGALRM to it - we are going to use this for the timer interrupt

  if (pthread_sigmask (SIG_BLOCK, &signal_set, NULL) != 0) // Block in all threads
  {
    Print_Log (T_NULL, "Failed to block SIGALRM events: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Install timer_handler as the signal handler for SIGALRM - used to retransmit packets for error avoidance
  // Basically this defines the 'interrupt service routine'. Its a formality we need to fulfil, but if you go
  // looking, this routine is actually empty and does nothing.
  memset (&sa, 0, sizeof (sa)) ;
  sa.sa_handler = &periodic_timer_callback ;
  sigaction (SIGALRM, &sa, NULL) ;

  // At this point, any newly created threads will inherit the prevailing signal mask configured above.
  //------------------------------------------------------------------------------------------------------------------
  // We will use one socket for receive and another for transmit of the multicast UDP packets
  if ((Rx_Socket = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    Print_Log (T_NULL, "Rx socket creation failed: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // Enable SO_REUSEADDR to allow multiple instances of this programme to receive copies of the multicast datagrams
  c = 1 ;

  if (setsockopt(Rx_Socket, SOL_SOCKET, SO_REUSEADDR, (char *)&c, sizeof(c)) < 0)
  {
    perror ("Setting SO_REUSEADDR error") ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }

  // Define the IP address for receiving
  memset (&Rx_Address, 0x00, sizeof(Rx_Address)) ;
  Rx_Address.sin_family		= AF_INET ;
  Rx_Address.sin_addr.s_addr	= htonl(INADDR_ANY) ;
  Rx_Address.sin_port		= htons(UDP_PORT) ;

  // Bind the socket with the port address specified above
  if (bind (Rx_Socket, (struct sockaddr *)&Rx_Address, sizeof(Rx_Address)) < 0)
  {
    Print_Log (T_NULL, "Socket bind failed: %s\n", strerror(errno)) ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }

  // use setsockopt() to request that the kernel join a multicast group
  mreq.imr_multiaddr.s_addr = inet_addr(Multicast_Group) ;
  mreq.imr_interface.s_addr = htonl(INADDR_ANY) ;

  if (setsockopt(Rx_Socket, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&mreq, sizeof(mreq)) < 0)
  {
    Print_Log (T_NULL, "setsockopt IP_ADD_MEMBERSHIP: %s\n", strerror(errno)) ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------
  if ((Tx_Socket = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    Print_Log (T_NULL, "Tx socket creation failed: %s\n", strerror(errno)) ;
    close (Rx_Socket) ;
    return EXIT_FAILURE ;
  }

  // Define the IP address for transmitting
  memset (&Tx_Address, 0x00, sizeof(Tx_Address)) ;
  Tx_Address.sin_family		= AF_INET ;
  Tx_Address.sin_addr.s_addr	= inet_addr(Multicast_Group) ;
  Tx_Address.sin_port		= htons(UDP_PORT) ;
  //------------------------------------------------------------------------------------------------------------------
  // Initialise the random number generator using time as the seed. Choosing nanoseconds
  // is a good way of getting a very random number - it will critically depend on the
  // precise time at this instant
  clock_gettime (CLOCK_REALTIME, &ts) ;
  srand ((unsigned)ts.tv_nsec) ; // use nanoseconds to seed the random number generator
  //------------------------------------------------------------------------------------------------------------------
  // The global initialisation now mostly done. Launch threads now so that they can initialise themselves.
  //
  // In this case, we are going to create a supervisor thread, then a child thread. Strictly speaking, this is not
  // necessary and we could just have created children. However if one of the children exits, it is impossible to
  // work out which one exited unless we have a specific supervisor for each child.
  Threads_Active = _INITIALISING ;

  if (Cmdline_Verbose)
    Print_Log (T_NULL, "Launching threads!\n") ;

  for (subtask=&Subtasks[0] ; (Threads_Active == _INITIALISING) && (subtask->task != NULL) ; subtask++)
  {
    // the funny dance with malloc() here is the way posix threads pass arguments. There is room to pass
    // one argument, cast as a pointer to void regardless of the actual type. It must be malloced memory
    // and the child must free the memory before it exits.
    thread_param = malloc (sizeof(SUBTASK *)) ; // thread_param is a pointer to a pointer
    *thread_param = subtask ; // a pointer to this subtask's structure definition

    // I'm going to do something tricky... I will create a supervisor 'parent' for each 'real' thread. The
    // supervisor has no role except waiting for its child to die. Hopefully that never happens. But if it
    // does, for example there's a bug in the child and it referenced a null pointer (etc), then the
    // supervisor will squawk and at least tell us which child died before Linux kills all the other threads.
    // This gives us a hint about where to look for a problem....
    // It will be each instance of the supervisor that creates the 'real' children threads. If you refer to
    // its code (in common.c) you'll see its not complicated.
    if ((ret = pthread_create (&subtask->supervisor_id, NULL, Supervisor_Thread, (void *)thread_param)) != 0)
    {
      Print_Log (T_NULL, "Can't create %s thread (%d): %s\n", subtask->name, ret, strerror(errno)) ;
      return EXIT_FAILURE ;
    }
  }

  if (Cmdline_Verbose)
    Print_Log (T_NULL, "Threads launched!\n") ;
  //------------------------------------------------------------------------------------------------------------------
  // It is now safe to initialise the GPIOs
  Initialise_GPIO () ;

  // The threads will stall before fully starting, waiting on Threads_Active changing to _RUNNING.
  // NOTE: In each of the threads that opens one of the message queues in the loop condition below, it is important
  // that the call that opens the message queue (and sets the id to something other than 0) is the LAST thing that
  // happens before the (Threads_Active == _INITIALISING) delay loop in those threads. This ensures that other important
  // init variables are properly initialised before the message queue ID changes. Remember that the threads can run in
  // ANY sequence and the sequence is undefined. I copped a bunch of random / infrequent segfaults because I wasn't
  // careful!
  do
    Sleep_ns (THREAD_START_DELAY) ;
  while ((DD_Msgq == 0) || (Dim_Msgq == 0) || (Setup_Msgq == 0) || (Media_Msgq == 0) || (Amp_Msgq == 0) || (Button_Msgq == 0)) ;

  Threads_Active = _RUNNING ; // The system is now initialised and can commence running
  //------------------------------------------------------------------------------------------------------------------
  // The threads are now all running. Our last task is to commence the periodic timer 'interrupts'
  // They are a recurring timer alarm that calls the periodic_timer_callback
  timer.it_value.tv_sec = 0 ;
  timer.it_value.tv_usec = PERIODIC_INTERRUPT ; // first interrupt 25ms from now

  timer.it_interval.tv_sec = 0 ;
  timer.it_interval.tv_usec = PERIODIC_INTERRUPT ; // subsequent interrupts every 25ms from the first

  // Start this recurring timer
  setitimer (ITIMER_REAL, &timer, NULL) ;
  //------------------------------------------------------------------------------------------------------------------
  if (Cmdline_Verbose == 0)
    Print_Log (T_NULL, "Run with -h for help, or -V for debugging messages!\n") ;
  //------------------------------------------------------------------------------------------------------------------
  // Wait around for the supervisor threads to finish
  // Note: this normally doesn't happen - everything runs until CTRL-C is hit, and that gets intercepted by the
  // signal catcher and directed to Release_Resources. (CTRL-C == SIGINT)
  for (subtask=&Subtasks[0] ; subtask->task != NULL ; subtask++)
  {
    if (subtask->supervisor_id == 0)
      continue ;

    if (((c = pthread_join (subtask->supervisor_id, &pret)) != 0) || (pret != (void *)0))
      Print_Log (T_NULL, "%s thread join returned %d/%p\n", subtask->name, c, pret) ;

    else // something happened - we will just forget the niceties and get out of here all together
      break ;
  }
  //------------------------------------------------------------------------------------------------------------------
  return EXIT_SUCCESS ;
}
