////////////////////////////////////////////////////////////////////////////////////////
// This is a programme that demonstrates very basic launching of linux threads
// and shows how you can have more threads running than actual CPU cores
///////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/time.h>
#include <errno.h>
///////////////////////////////////////////////////////////////////////////////////////////
// This child subroutine is deliberately POORLY written for a multi-threaded operating
// system. It is going to DELIBERATELY waste time using a spin loop - which is really bad
// practice (you should call an OS delay, so the OS can put the thread to sleep and let
// something else run without wasting CPU time).
//
// But we are going to launch these badly delayed threads to try to consumer MORE than
// 100% of all CPU cores, and see how the operating system behaves. If it is a good OS,
// it will divide time more or less evenly between these CPU hoggers, and they will all
// run a little slower than normal because they cannot consume 100% of a CPU, they will
// share the CPU.

#define MAX_LOOPS	(5000)

void *
child_subroutine (void *arg)
{
  int			*i, loop, total_loops ;
  volatile uint32_t	del ;

  // read the parameter that was passed
  i = (int *)arg ;

  for (total_loops=0 ; total_loops < MAX_LOOPS ;)
  {
    loop = (rand() % 1000) + 1 ; // Pick a random number between 1 and 1000
    fprintf (stderr, "CHILD %4d: total loops waited=%4d, waiting for additional %4d loops\n", *i, total_loops, loop) ;
    total_loops += loop ;

    for (; loop ; loop--)
      for (del=0x2ffff ; del ; del--) ; // Waste time! This just burns CPU cycles (calibrated for a raspberry pi)
  }

  fprintf (stderr, "CHILD %4d: exited\n", *i) ;
  free (arg) ; // return the arg memory to the OS
  pthread_exit (NULL) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
int
main (int argc, char **argv)
{
  int		N, i, *p ;
  time_t	t ;
  struct timeval start_time, end_time ;
  double	duration ;

  if (argc != 2)
    goto show_usage ;

  // Read the value
  if ((N = atoi(argv[1])) <= 0)
  {
    printf ("The number of child threads must be a positive integer.\n\n") ;

  show_usage:
    printf ("Usage: %s [number of children]\n", argv[0]) ;
    printf ("Launch the specified number of time-wasting threads. Watch how the OS shares time.\n\n") ;
    printf ("Hint: determine the numbe of CPU cores on this system by executing 'cat /proc/cpuinfo'.\n") ;
    printf ("Then see how this programme works with less than or equal to that number of children.\n") ;
    printf ("Then see what happens when you exceed the number of CPU cores.\n") ;
    return EXIT_FAILURE ;
  }

  printf ("This demo programme will create %d different threads called CHILD, and each of those\n", N) ;
  printf ("will waste a random amount of time in an OS unfriendly way.\n") ;
  printf ("There will be more children than there are CPU cores, so no one child will be able to\n") ;
  printf ("consumer ALL of a CPU at the same time. If you launch more children than CPU cores,\n") ;
  printf ("you will see that the OS shares what time is available fairly between each child.\n\n") ;
  printf ("---------------------\n") ;

  pthread_t	thread [N];

  // Seed the random number generator using system time. We will also remember that time so we know
  // how long this programme ran
  srand ((unsigned) time (&t)) ;

  // Remember the exact time we started
  gettimeofday (&start_time, NULL) ;

  // Launch N threads, mallocing some memory so we can pass a parameter that gives each
  // child a unique number
  for (i=0 ; i < N ; i++)
  {
    p = malloc (sizeof(int)) ; // allocate memory for the parameter and initialise it
    *p = i ;

    // This is the Linux call to start a thread
    pthread_create (&thread[i], NULL, child_subroutine, (void *)p) ;
    fprintf (stderr, "MAIN   %3d: thread=%lx\n", i, thread[i]) ;
  }

  // Note: all the memory we allocated for the parameters is no longer ours! It belongs to the
  // threads, so we should not access it any more from here.

  // After the thread(s) have launched, we need to wait for them to finish.
  // The joining need not actually happen in the same order that the threads finish.
  // As implemented here, we wait for them to finish in a particular sequence, but the
  // actual ordering of the pthread_exit calls in real time might be different.
  for (i=0 ; i < N ; i++)
  {
    pthread_join (thread[i], NULL) ;
    fprintf (stderr, "MAIN   %3d thread returned\n", i) ;
  }

  // Remember the exact time we stopped
  gettimeofday (&end_time, NULL) ;

  // Determine the elapsed duration (in ms)
  duration = (end_time.tv_sec - start_time.tv_sec) * 1000.0 ;		// sec to ms
  duration += (end_time.tv_usec - start_time.tv_usec) / 1000.0 ;	// us to ms

  printf ("---------------------\nRan for %.3f seconds\n", duration / 1000.0) ;
}
