////////////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <stdnoreturn.h>
#include <pigpio.h>
#include <math.h>
///////////////////////////////////////////////////////////////////////////////////////////
// Define the GPIO pin functions here. Note that the GPIO names differ from the Raspberry Pi pin numbers!

#define IN_GPIO_a		(20)	// GPIO20 (pin 38 of raspberry pi expansion connector / minus function)
#define IN_GPIO_b		(21)	// GPIO21 (pin 40 of raspberry pi expansion connector / plus function)

#define OUT_GPIO_a		( 4)	// GPIO4  (pin  7 of raspberry pi expansion connector / D0)
#define OUT_GPIO_b		(18)	// GPIO18 (pin 12 of raspberry pi expansion connector / D1)

#define PWM_GPIO			(22)	// GPIO22 (pin 15 of raspberry pi expansion connector)

#define NUM_WAVE_IDS		(20)	// test 20 tones - slightly more than one musical octave
#define BOTTOM_NOTE		(300)	// Hz - I just chose this randomly - its not the frequency of a
					// note a musician would play if they were in tune!

int	Wave [NUM_WAVE_IDS] ;
///////////////////////////////////////////////////////////////////////////////////////////
double
Round_Nearest_Places (double number, int places)
{
  double		round_from ;
  long long int		rounded ;

  if (places < 0)
    places = 0 ;

  round_from = number * pow (10, places) ;
  rounded = (long long int)((round_from < 0) ? round_from - 0.5 : round_from + 0.5) ;

  // Precaution to stop spurious decimal places due to rounding error
  if (places == 0)
    return rounded ;

  return rounded / pow (10, places) ;
}

int
Round_Nearest_Int (const double number)
{
  int			x ;

  if (number > 0)
  {
    // Positive number
    x = number ;

    if ((number - x) >= 0.5)
      return x + 1 ;

    return x ;
  }

  // Negative number - we have trouble with rounding errors
  // Use the Round_Nearest_Places function to avoid this
  x = -number ; // x is positive equivalent integer of the negative number

  // The rounding issue happens with (-number-x)
  if (Round_Nearest_Places(-number - x, 1) >= 0.5)
    return -x - 1 ;

  return -x ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Goodbye is a general abort point.
// It will display an optional error message similarly to printf, close GPIO resources, and kill threads.

noreturn void
Goodbye (const char *fmt, ...)
{
  va_list               arglist ;

  // makes use of C's variable argument list macros - the dot dot dot above
  va_start (arglist, fmt) ;
  vfprintf (stderr, fmt, arglist) ; // a special version of fprintf using the variable argument list
  va_end (arglist) ;

  gpioTerminate() ; // unlink from the pigpio library
  exit (EXIT_FAILURE) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// init_gpio_pwm
//
// We are going to use pigpio's DMA capability to generate PWM sine waves. Its a clever
// implementation that requires some software to set it up, but then when you push the
// button to start, it will run forever without any further software intervention.
//
// The DMA engine has a feature where you can chain a series of commands together. The
// engine will execute the first command, then wait for a number of microseconds
// specified in that command step, and jump to the next command step and repeat. At the
// end of the list of commands, it jumps back to the beginning and starts over, without
// software needing to intervene. We use this capability to generate a PWM sine wave
// by setting up a sequence of steps to drive the desired GPIO pin high, then wait, then
// drive the pin low, then wait again (a possibly different time length), then drive the
// pin high again, low again, for a sequence defined as NUM_PWM_CYCLES. If we choose
// a constant time interval for each PAIR of commands, and vary the time high and time
// low in each subsequent pair of commands, then we have generated time-varying PWM.
// By carefully choosing the time high and time low and varying this from cycle to
// cycle, we can generate arbitrary waveform shapes, including sine waves. Of course,
// the actual wave shape is square/PWM, but if you put that through an (external) low
// pass filter, you'll get an approximation of sine wave. The larger the number of cycles
// in the pattern, the more accurate the sine wave. pigpiod has a time granularity
// of 1us, and so to generate audio-band sine waves, there's a limit to the number
// of cycles that we can use. I've set to 50 cycles per millisecond.

#define NUM_PWM_CYCLES		(50)

int
init_gpio_pwm (const int gpio, const int freq)
{
  double	micros_per_cycle, s ;
  int		i, cycle_time, high, low, num_pulses, wave_id ;
  gpioPulse_t	GpioPulses[NUM_PWM_CYCLES * 2] ;

  micros_per_cycle = 1000000.0 * (1.0 / freq) / NUM_PWM_CYCLES ;
  cycle_time = Round_Nearest_Int (micros_per_cycle) ;

  for (i=0 ; i < NUM_PWM_CYCLES ; i++)
  {
    // Determine length of high and low pulses based on sin of the angle
    s = 0.95 * sin ((0.001 + ((double)i / NUM_PWM_CYCLES)) * 2.0 * M_PI) ;

    high = Round_Nearest_Int ((micros_per_cycle / 2.0) + (s * micros_per_cycle / 2.0)) ;
    low = cycle_time - high ; // Make the low time complementary - always adds to cycle_time

    // Ensure that the high and the low can never be zero length
    if (high == 0)
    {
      high = 1 ;
      low-- ;
    }

    else if (low == 0)
    {
      low = 1 ;
      high-- ;
    }

    // The PWM is set in pairs. Firstly, set the pin ON time, then set the OFF time
    GpioPulses[i<<1].gpioOn = (1 << gpio) ; // 'gpio' is the GPIO name, not the pin number!
    GpioPulses[i<<1].gpioOff = 0 ;
    GpioPulses[i<<1].usDelay = high ;

    GpioPulses[(i<<1)+1].gpioOn = 0 ;
    GpioPulses[(i<<1)+1].gpioOff = (1 << gpio) ; // 'gpio' is the GPIO name, not the pin number!
    GpioPulses[(i<<1)+1].usDelay = low ;

    //printf ("%2d: %2dus high, %2dus low\n", i, high, low) ;
  }

  gpioWaveAddNew () ;

  if ((num_pulses = gpioWaveAddGeneric (NUM_PWM_CYCLES * 2, GpioPulses)) == PI_TOO_MANY_PULSES)
    Goodbye ("gpioWaveAddGeneric failed\n") ;

  printf ("  Nominal freq=%4dHz. Num samples in one cycle=%d, nominal time per sample=%4.2fus, actual time per sample=%dus, actual freq=%4.0fHz\n",
	  freq, num_pulses, micros_per_cycle, cycle_time, 1000000.0/cycle_time/NUM_PWM_CYCLES) ;

  if ((wave_id = gpioWaveCreate ()) < 0)
    Goodbye ("gpioWaveCreate failed: %d\n", wave_id) ;

  return wave_id ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Initialise the pigpio interface/library
//
// Configure pins to be used for GPIO as either inputs or outputs, and we will set pullup/down off.
// Note: the pin to be used for PWM does not require initialisation in the same way.
//
// Some terse help is available at the linux command line using 'man pigpio'

// I am just going to randomly choose a couple of GPIO pins and use them as inputs (for no particualr
// reason, just to see how it works.)
//
// I am also going to randomly choose a couple of GPIO pins as outputs and write some square
// waves to them, again for no reason - just to see how it works.
//
// There is also a PWM pin that will carry a pseudo sine wave - this is not initialised in the code
// below, but separately after this is initialisation is called.

static uint8_t GPIO_inputs[]  = {IN_GPIO_a, IN_GPIO_b, 0} ;
static uint8_t GPIO_outputs[] = {OUT_GPIO_a, OUT_GPIO_b, 0} ;

void
Initialise_PiGPIO (void)
{
  int		i, error, this_freq, end_freq ;
  double	note_ratio ;

  if (gpioInitialise() == PI_INIT_FAILED)
    Goodbye ("gpioInitialise failed\n") ;

  printf ("Setting GPIO inputs:") ;

  for (i=0 ; GPIO_inputs[i] > 0 ; i++)
  {
    if ((error = gpioSetMode (GPIO_inputs[i], PI_INPUT)) != 0)
      Goodbye ("\nError defining GPIO pin %d: %d\n", GPIO_inputs[i], error) ;

    if ((error = gpioSetPullUpDown (GPIO_inputs[i], PI_PUD_OFF)) != 0)
      Goodbye ("\nError setting pullup/down input pin %d: %d\n", GPIO_inputs[i], error) ;

    printf (" %d", GPIO_inputs[i]) ;
  }

  printf ("\nSetting GPIO outputs:") ;

  for (i=0 ; GPIO_outputs[i] > 0 ; i++)
  {
    if ((error = gpioSetMode (GPIO_outputs[i], PI_OUTPUT)) != 0)
      Goodbye ("\nError defining GPIO pin %d: %d\n", GPIO_outputs[i], error) ;

    if ((error = gpioSetPullUpDown (GPIO_outputs[i], PI_PUD_OFF)) != 0)
      Goodbye ("\nError setting pullup/down output pin %d: %d\n", GPIO_outputs[i], error) ;

    if ((error = gpioWrite (GPIO_outputs[i], 1)) != 0)
      Goodbye ("\nError setting output pin %d to 1: %d\n", GPIO_outputs[i], error) ;

    printf (" %d", GPIO_outputs[i]) ;
  }

  gpioWaveClear () ;

  // Next - define the memory structure that will hold the pseudo sine wave. It will be a sequence
  // of PWM settings. If one cycle of a sine way takes NUM_WAVE_IDS steps, then we are going to
  // define a PWM ratio at each of those steps that mimic the amplitude of that sine wave at each
  // of those points in time. Then to generate the sine wave, we will set up a DMA process in the
  // GPIO daemon that reads those settings in an infinite loop, repeating the same cycle over and
  // over.
  //
  // To look at the sine wave, you'll need an oscilloscope or an audio amplifier. Filter the output
  // of the GPIO pin using (say) a 1k series resistor driving a 0.1 microfarad capacitor to ground
  // (ie a low-pass filter with a 3dB corner frequency of 1.6kHz). Look at or listen to the waveform
  // at the junction between the resistor and capacitor.
  //
  // Note: the relationship between the frequencies of notes in a scale is as follows:
  //  * A one octave interval is the precise doubling of the frequency
  //  * There are 12 notes in a musical scale.
  // Therefore, the relationship between frequencies of adjacent notes is a factor of (1/12) * sqrt(2)

  note_ratio = pow(2, 1/12.0) ; // this is how to work out the frequency of the next note in the scale
  end_freq = Round_Nearest_Int (BOTTOM_NOTE * pow(note_ratio, NUM_WAVE_IDS-1)) ;
  printf ("\nDefining a series of sine waves starting at %d Hz and nominally finishing at %d Hz\n\n", BOTTOM_NOTE, end_freq) ;

  // Choose an ascending sequence of 'notes' (not precisely the same frequencies as an official scale,
  // but it will still sound like a tuned ascending sequence of notes)
  for (i=0 ; i < NUM_WAVE_IDS ; i++)
  {
    this_freq = Round_Nearest_Int(BOTTOM_NOTE * pow(note_ratio, i)) ;
    Wave[i] = init_gpio_pwm (PWM_GPIO, this_freq) ;
  }
} // Initialise_PiGPIO
///////////////////////////////////////////////////////////////////////////////////////////
int
main ()
{
  int		i, j, k, ret ;
  int		in0, in1 ;

  // We are going to set up and do THREE different unrelated things.
  // a) generate a pseudo sine wave out one GPIO pin
  // b) generate square waves of different frequencies out two other GPIO pins
  // c) read the state of two other GPIO pins

  // Firstly, initialise our interface to the GPIO daemon
  Initialise_PiGPIO () ;

  // Repeat the loop through every note, ten times
  printf ("\nOutputting tones - ten times through a loop of %d notes\n", NUM_WAVE_IDS) ;
  in0 = in1 = 0 ; // shut up the compiler warning

  // Loop through each 'scale' ten times
  // We will play 'audio' out of
  for (i=k=0 ; i < 10 ; i++)
  {
    // Loop through each note in our 'scale'
    for (j=0 ; j < NUM_WAVE_IDS ; j++)
    {
      // Start sending the next tone
      if ((ret = gpioWaveTxSend (Wave[j], PI_WAVE_MODE_REPEAT)) < 1)
	Goodbye ("gpioWaveTxSend failed: %d\n", ret) ;

      // Transmit the tone for 1 second
      sleep (1) ;

      // Stop sending that tone.
      if ((ret = gpioWaveTxStop()) != 0)
	Goodbye ("gpioWaveTxSend failed: %d\n", ret) ;

      // Read the GPIO inputs
      if (((in0 = gpioRead (IN_GPIO_a)) == PI_BAD_GPIO) ||
	  ((in1 = gpioRead (IN_GPIO_b)) == PI_BAD_GPIO))
	Goodbye ("Error reading inputs: %d %d\n", in0, in1) ;

      printf ("Input_0=%d, input_1=%d\n", in0, in1) ;

      // Write to the GPIO outputs. We will just write different square waves to those pins
      if (((ret = gpioWrite (OUT_GPIO_a, (k & 1) != 0)) != 0) ||
	  ((ret = gpioWrite (OUT_GPIO_b, (k & 2) != 0)) != 0))
	Goodbye ("\nError setting output pins: %d\n", ret) ;

      k++ ;
    }
  }

  printf ("Finished outputting notes\n") ;
  //------------------------------------------------------------------------------------------------------------------
  return EXIT_SUCCESS ;
}
