////////////////////////////////////////////////////////////////////////////////////////
// This is a programme that demonstrates the use of pipes to communicate between
// linux processes
///////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////////
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <errno.h>
#include <inttypes.h>
#include <fcntl.h>

#define PARENT_READ_PIPE	(0)	// The parent will read from this pipe (and child will write)
#define PARENT_WRITE_PIPE	(1)	// The parent will write into this pipe (and child will read)
#define NUM_PIPES		(2)

#define READ_FD			(0)	// read file desriptor offset used by pipe call
#define WRITE_FD		(1)	// write file descriptor offset used by pipe call
#define NUM_FDS			(2)

int	pipes [NUM_PIPES][NUM_FDS] ;

#define PARENT_READ_FD		(pipes[PARENT_READ_PIPE][READ_FD])
#define PARENT_WRITE_FD		(pipes[PARENT_WRITE_PIPE][WRITE_FD])
#define CHILD_READ_FD		(pipes[PARENT_WRITE_PIPE][READ_FD])
#define CHILD_WRITE_FD		(pipes[PARENT_READ_PIPE][WRITE_FD])
///////////////////////////////////////////////////////////////////////////////////////////
int
main (int argc, char **argv)
{
  int	ret, n ;
  char	*s, buffer[1000] ;
  FILE	*f, *g ;

  if (argc == 1)
  {
    fprintf (stderr, "Usage: %s [strings] [to] [send] [via] [pipe]\n", argv[0]) ;
    return EXIT_FAILURE ;
  }

  // The pipe function creates a pipe and puts the file descriptors for the reading and writing ends of the
  // pipe (respectively) into file_descriptor[0] and file_descriptor[1].
  // create two pipes
  if ((pipe(pipes[PARENT_READ_PIPE]) < 0) || (pipe(pipes[PARENT_WRITE_PIPE]) < 0))
  {
    printf ("Cannot create pipes: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------------
  // fork to create a child process
  if ((ret = fork()) == 0)
  {
    // I am the child
    // duplicate the pipes created by the parent and use these for stdin and out
    dup2 (CHILD_READ_FD, STDIN_FILENO) ;
    dup2 (CHILD_WRITE_FD, STDOUT_FILENO) ;

    // close the file descriptors that are not required by the child
    close (CHILD_READ_FD) ;
    close (CHILD_WRITE_FD) ;
    close (PARENT_READ_FD) ;
    close (PARENT_WRITE_FD) ;

    // Read what the parent has to send to us
    if (((f = fdopen (STDIN_FILENO, "r")) == NULL) || ((g = fdopen (STDOUT_FILENO, "w")) == NULL))
    {
      // Note: the error below won't print - because we couldn't open STDOUT!
      printf ("Cannot fdopen: %s\n", strerror(errno)) ;
      exit (EXIT_FAILURE) ;
    }

    fprintf (g, "I am the child\n") ;

    // Read from stdin and write to stdout until the end of the stream
    while (fgets(buffer, sizeof(buffer), f) != NULL)
    {
      // The line is in buffer - need to remove the newline from the end of it
      for (s=buffer ; *s && (*s != '\n') ; s++) ;
      *s = '\0' ;

      fprintf (g, "Child received: '%s'\n", buffer) ;
    }

    fclose (f) ;
    fclose (g) ;
    exit (EXIT_SUCCESS) ;
  }
  else if (ret < 0)
  {
    printf ("Cannot fork: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------------
  // I am the parent - close the file descriptors not required by the parent (ie the child ends of the pipes)
  close (CHILD_READ_FD) ;
  close (CHILD_WRITE_FD) ;

  printf ("I am the parent\n") ;

  // Open the parent ends of the pipe for reading and writing
  if (((f = fdopen (PARENT_WRITE_FD, "w")) == NULL) || ((g = fdopen (PARENT_READ_FD, "r")) == NULL))
  {
    printf ("Cannot fdopen: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // write the command line arguments into the outgoing pipe
  for (n=0 ; n < argc ; n++)
  {
    printf ("Parent writing: '%s'\n", argv[n]) ;

    if ((ret = fprintf (f, "%s\n", argv[n])) <= 0)
    {
      fprintf (stderr, "  failed (ret=%d): %s\n", ret, strerror(errno)) ;
      return EXIT_FAILURE ;
    }
  }

  // We have written our shit. Now we can close the pipe - this will close the file at the other end
  fclose (f) ;
  // close (PARENT_WRITE_FD) ; Don't need to close the file descriptor if we fclose the handle

  printf ("------------------------------\nWhat appears below is received from the child process...\n\n") ;

  // The child should have reflected our strings back at us. Read and display those strings.
  while (fgets(buffer, sizeof(buffer), g) != NULL)
  {
    // The line is in buffer - need to remove the newline from the end of it
    for (s=buffer ; *s && (*s != '\n') ; s++) ;
    *s = '\0' ;

    printf ("Parent received: \"%s\"\n", buffer) ;
  }

  fclose (g) ;
  // close (PARENT_READ_FD) ; Don't need to close the file descriptor if we fclose the handle
  return EXIT_SUCCESS ;
}
