////////////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <stdnoreturn.h>
#include <pigpio.h>
#include <math.h>
#include <string.h>

typedef struct timespec		TS ;
#define NS_PER_SECOND		(1000000000)
///////////////////////////////////////////////////////////////////////////////////////////
// Define the GPIO pin functions here. Note that the GPIO names differ from the Raspberry Pi pin numbers!

#define IN_GPIO_a		(20)	// GPIO20 (pin 38 of raspberry pi expansion connector / minus function)
#define IN_GPIO_b		(21)	// GPIO21 (pin 40 of raspberry pi expansion connector / plus function)

#define OUT_GPIO_a		( 4)	// GPIO4  (pin  7 of raspberry pi expansion connector / D0)
#define OUT_GPIO_b		(18)	// GPIO18 (pin 12 of raspberry pi expansion connector / D1)
#define OUT_GPIO_c		(17)	// GPIO17 (pin 11 of raspberry pi expansion connector / D2)
#define OUT_GPIO_d		(27)	// GPIO27 (pin 13 of raspberry pi expansion connector / D3)
#define OUT_GPIO_e		(23)	// GPIO23 (pin 16 of raspberry pi expansion connector / D4)
#define OUT_GPIO_f		(22)	// GPIO22 (pin 15 of raspberry pi expansion connector / D5)
#define OUT_GPIO_g		(24)	// GPIO24 (pin 18 of raspberry pi expansion connector / D6)
#define OUT_GPIO_h		(25)	// GPIO25 (pin 22 of raspberry pi expansion connector / D7)
#define OUT_GPIO_i		( 6)	// GPIO6  (pin 31 of raspberry pi expansion connector / A0)
#define OUT_GPIO_j		( 5)	// GPIO5  (pin 29 of raspberry pi expansion connector / A1)
///////////////////////////////////////////////////////////////////////////////////////////
// Linux-specific code starts here. This stuff can appear intimidating if you are not so
// familiar with Linux C coding. Don't worry too much if so. The programme would run ok even
// without this Linux-specific stuff, but this is all about dotting 'I's and crossing 'T's.
///////////////////////////////////////////////////////////////////////////////////////////
// Goodbye is a general abort point.
// It will display an optional error message similarly to printf, close GPIO resources, and kill threads.

noreturn void
Goodbye (const char *fmt, ...)
{
  va_list               arglist ;

  // makes use of C's variable argument list macros - the dot dot dot above
  va_start (arglist, fmt) ;
  vfprintf (stderr, fmt, arglist) ; // a special version of fprintf using the variable argument list
  va_end (arglist) ;

  gpioTerminate() ; // unlink from the pigpio library
  exit (EXIT_FAILURE) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Initialise the pigpio interface/library,
// Configure pins to be used for GPIO as either inputs or outputs, and we will set pullup/down off.
// Note: the pin to be used for PWM does not require initialisation in the same way.
//
// Some terse help is available at the linux command line using 'man pigpio'

static uint8_t GPIO_inputs[]  = {IN_GPIO_a, IN_GPIO_b, 0xff} ;
static uint8_t GPIO_outputs[] = {OUT_GPIO_a, OUT_GPIO_b, OUT_GPIO_c, OUT_GPIO_d, OUT_GPIO_e, OUT_GPIO_f, OUT_GPIO_g, OUT_GPIO_h, OUT_GPIO_i, OUT_GPIO_j, 0xff} ;

void
Initialise_PiGPIO (void)
{
  int		g, error ;

  if (gpioInitialise() == PI_INIT_FAILED)
    Goodbye ("gpioInitialise failed: try running 'sudo systemctl stop alarm-clock', then 'sudo systemctl stop pigpiod'\n") ;

  printf ("Setting GPIO inputs:") ;

  for (g=0 ; GPIO_inputs[g] != 0xff ; g++)
  {
    if ((error = gpioSetMode (GPIO_inputs[g], PI_INPUT)) != 0)
      Goodbye ("\nError defining GPIO input %d: %d\n", GPIO_inputs[g], error) ;

    if ((error = gpioSetPullUpDown (GPIO_inputs[g], PI_PUD_OFF)) != 0)
      Goodbye ("\nError setting pullup/down input pin %d: %d\n", GPIO_inputs[g], error) ;

    printf (" %d", GPIO_inputs[g]) ;
  }

  printf ("\nSetting GPIO outputs:") ;

  for (g=0 ; GPIO_outputs[g] != 0xff ; g++)
  {
    if ((error = gpioSetMode (GPIO_outputs[g], PI_OUTPUT)) != 0)
      Goodbye ("\nError defining GPIO output %d: %d\n", GPIO_outputs[g], error) ;

    if ((error = gpioSetPullUpDown (GPIO_outputs[g], PI_PUD_OFF)) != 0)
      Goodbye ("\nError setting pullup/down output pin %d: %d\n", GPIO_outputs[g], error) ;

    if ((error = gpioWrite (GPIO_outputs[g], 1)) != 0)
      Goodbye ("\nError setting output pin %d to 1: %d\n", GPIO_outputs[g], error) ;

    printf (" %d", GPIO_outputs[g]) ;
  }
} // Initialise_PiGPIO
///////////////////////////////////////////////////////////////////////////////////////////
// Sleep_ns is a wrapper to make calls to nanosleep easier
// It is set up as a delay call with ns granularity
//
// The maximum sleep time is ~4 seconds, as the number of nanoseconds is a 32 bit unsigned int

void
Sleep_ns (const uint32_t ns)
{
  TS    ts ;

  if (ns == 0)
    return ;

  ts.tv_sec = ns / NS_PER_SECOND ;
  ts.tv_nsec = ((int)ns - (ts.tv_sec * NS_PER_SECOND)) ;        // nanoseconds (fractional seconds)

  // nanosleep should return 0 if it slept correctly for the full duration
  // However it can be interrupted by a signal, and if so, we need to process
  // the signal and reinitiate the remainder of the delay.
  for (; nanosleep(&ts, &ts) != 0 ;)
  {
    // if interrupted by a signal, errno will be EINTR, and we just try again
    if (errno != EINTR)
      fprintf (stderr, "Problem with Sleep_ns (0x%08x ==> %d.%d): %s\n", ns, (int)ts.tv_sec, (int)ts.tv_nsec, strerror(errno)) ;
  }
} // Sleep_ns
///////////////////////////////////////////////////////////////////////////////////////////
int
main ()
{
  int		i, ret ;
  int		in0, in1 ;

  printf ("If this fails, try running 'sudo systemctl stop alarm-clock', then 'sudo systemctl stop pigpiod'\n") ;
  Initialise_PiGPIO () ;

  printf ("\n---------------------------------------------------------\n") ;
  printf ("Square waves being written to GPIO outputs\n") ;
  in0 = in1 = 0 ; // shut up the compiler warning

  // Loop forever
  for (i=0 ; ; i++)
  {
    // Read the GPIO inputs
    if (((in0 = gpioRead (IN_GPIO_a)) == PI_BAD_GPIO) ||
	((in1 = gpioRead (IN_GPIO_b)) == PI_BAD_GPIO))
      Goodbye ("Error reading inputs: %d %d\n", in0, in1) ;

    printf ("%5d: GPIO%d=%d, GPIO%d=%d\n", i, IN_GPIO_a, in0, IN_GPIO_b, in1) ;

    // Write to the GPIO outputs. We will just write different square waves to those pins
    if (((ret = gpioWrite (OUT_GPIO_a, (i & 1) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_b, (i & 2) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_c, (i & 4) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_d, (i & 8) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_e, (i & 16) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_f, (i & 32) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_g, (i & 64) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_h, (i & 128) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_i, (i & 256) != 0)) != 0) ||
	((ret = gpioWrite (OUT_GPIO_j, (i & 512) != 0)) != 0))
      Goodbye ("\nError setting GPIO outputs: err=%d\n", ret) ;

    // Sleep for 100ms
    Sleep_ns (NS_PER_SECOND / 10) ;
  }

  // Not reached
  //------------------------------------------------------------------------------------------------------------------
  return EXIT_SUCCESS ;
}
