#!/bin/bash
#                           Digital Alarm Clock
#                Copyright (C) 2023 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
####################################################################################################################
THIS_SCRIPT="/cgi-bin/alarms.cgi"

# Load common functions and data
INCLUDE_COMMON="common.sh"

if test -f "${INCLUDE_COMMON}" ; then
  . ${INCLUDE_COMMON}
fi

Make_Temp "$0"
Read_HTML_Parameters
Process_Inputs
Emit_HTML_Headers noborder
Do_Start_Body

echo "  <div class='background'>"

[[ -n "${STREAM}" ]] && DEFAULT_STREAM_OR_FILE="${STREAM}"
[[ "${INIT_VOLUME_ADJUST}" =~ ${IsNumber} ]] || INIT_VOLUME_ADJUST=0
[[ "${TARG_VOLUME_ADJUST}" =~ ${IsNumber} ]] || TARG_VOLUME_ADJUST=0
####################################################################################################################
# Emit alarm column calculates the value to include in a particular alarm column for a once-only alarm
# $1 - the name of the day (mon, tue, wed, thu, fri, sat, sun)
# $2 - Y or N (whether the alarm is configured for that day)
# $3 - time of the alarm in hours/minutes in 24H format (HH:MM)
Write_Alarm_Column()
{
  # Get the no-alarm case out of the way first
  if [[ "$2" == 'N' ]] ; then
    echo -n ",N" >> "${TMP}"
    return
  fi

  # There's a once-only alarm to output.
  DATE_TODAY=$(date +%Y-%m-%d)
  DATE_ALARM=$(date --date "$1" +%Y-%m-%d)

  # We need to dance around a bit. Does the alarm happen on same day of week as today? If so, is the time
  # before or after NOW?
  if [[ "${DATE_TODAY}" == "${DATE_ALARM}" ]] ; then
    TIME_NOW=$(date "+%H:%M")

    if [[ "${TIME_NOW}" < "$3" ]] ; then
      # The alarm is yet to occur today
      EPOCH_ALARM=$(date --date="$1 $3" +%s)
    else
      # The alarm time has already passed today - look at next week instead
      EPOCH_ALARM=$(date --date="next $1 $3" +%s)
    fi
  else
    EPOCH_ALARM=$(date --date="next $1 $3" +%s)
  fi

  # Finally, output the epoch of the alarm
  echo -n ",${EPOCH_ALARM}" >> "${TMP}"
}
####################################################################################################################
# Check whether a new stream has been specified
if [[ -n "${stop_browsing}" ]] ; then
  browse=''
  NEW_REFERENCE="$(Percent_Decode "${stop_browsing}")"
  [[ -n "${NEW_REFERENCE}" ]] && DEFAULT_STREAM_OR_FILE="${NEW_REFERENCE}"
fi

[[ -z "${DEFAULT_STREAM_OR_FILE}" ]] && DEFAULT_STREAM_OR_FILE="${MEDIA_DIR}"
[[ -f "${DEFAULT_STREAM_OR_FILE}" ]] && DEFAULT_BROWSE_DIR="$(dirname "${DEFAULT_STREAM_OR_FILE}")"
[[ -d "${DEFAULT_STREAM_OR_FILE}" ]] && DEFAULT_BROWSE_DIR="${DEFAULT_STREAM_OR_FILE}"
####################################################################################################################
# Process the alarm edit command
if [[ "${alarm_copy}" =~ ${IsNumber} ]] ; then
  # The value of alarm_copy is a number referring to the line within the alarm file that we have been asked to edit
  NUM_LINES=$(wc -l "${ALARM_FILE}" | sed 's/[[:space:]].*$//g')

  if [[ ${alarm_copy} -gt 1 ]] && [[ ${alarm_copy} -le ${NUM_LINES} ]] ; then
    ALARM_COPY_LINE=$(sed "${alarm_copy}q;d" "${ALARM_FILE}")
    IFS=, read -r DEFAULT_TIME SUSPENDED DEFAULT_ALARM_DURATION VOLADJ MON TUE WED THU FRI SAT SUN DEFAULT_STREAM_OR_FILE <<< "${ALARM_COPY_LINE}"

    # Need to check for numbers in the days of the week variables - indicating ONCE
    ONCE='N'

    if [[ "${MON}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      MON='Y'
    fi

    if [[ "${TUE}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      TUE='Y'
    fi

    if [[ "${WED}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      WED='Y'
    fi

    if [[ "${THU}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      THU='Y'
    fi

    if [[ "${FRI}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      FRI='Y'
    fi

    if [[ "${SAT}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      SAT='Y'
    fi

    if [[ "${SUN}" =~ ${IsNumber} ]] ; then
      ONCE='Y'
      SUN='Y'
    fi

    # The volume string may be a single signed integer, or a range between two signed integers separated by dots
    if [[ "${VOLADJ}" == *"."* ]] ; then
      IVOL="${VOLADJ%%.*}"
      TVOL="${VOLADJ##*.}"

      if [[ "${IVOL}" =~ ${IsNumber} ]] && [[ "${IVOL}" -ge ${MIN_VOL_ADJ} ]] && [[ "${IVOL}" -le ${MAX_VOL_ADJ} ]] ; then
	INIT_VOLUME_ADJUST=${IVOL}
      fi

      if [[ "${TVOL}" =~ ${IsNumber} ]] && [[ "${TVOL}" -ge ${MIN_VOL_ADJ} ]] && [[ "${TVOL}" -le ${MAX_VOL_ADJ} ]] ; then
	TARG_VOLUME_ADJUST=${TVOL}
      fi

    else
      # Just a single number, apparently. Can we read it?
      if [[ "${VOLADJ}" =~ ${IsNumber} ]] && [[ "${VOLADJ}" -ge ${MIN_VOL_ADJ} ]] && [[ "${VOLADJ}" -le ${MAX_VOL_ADJ} ]] ; then
	INIT_VOLUME_ADJUST=${VOLADJ}
	TARG_VOLUME_ADJUST=${VOLADJ}
      fi
    fi
  fi
fi
####################################################################################################################
# If nothing has been selected or initialised yet, Initialise to not being selected
[[ -z "${MON}" ]] && MON='Y'
[[ -z "${TUE}" ]] && TUE='Y'
[[ -z "${WED}" ]] && WED='Y'
[[ -z "${THU}" ]] && THU='Y'
[[ -z "${FRI}" ]] && FRI='Y'
[[ -z "${SAT}" ]] && SAT='Y'
[[ -z "${SUN}" ]] && SUN='Y'
[[ -z "${ONCE}" ]] && ONCE='N'
####################################################################################################################
# Check for new alarm definitions - but only process if the sha1 checksum of the current file matches previous
# This is to prevent errors when refreshing web pages (and issuing command for a seocnd time)
if [[ "${PREV_SHA1}" == "${CURRENT_SHA1}" ]] && [[ -n "${DEFAULT_TIME}" ]] ; then
  if [[ "${command}" == "new_alarm" ]] || [[ "${command}" == "overwrite_alarm" ]] ; then
    # Check that at least one day has been specified
    if [[ "${MON}" != 'Y' ]] && [[ "${TUE}" != 'Y' ]] && [[ "${WED}" != 'Y' ]] && [[ "${THU}" != 'Y' ]] &&
	   [[ "${FRI}" != 'Y' ]] && [[ "${SAT}" != 'Y' ]] && [[ "${SUN}" != 'Y' ]] ; then
      ALM_ERR="No days specified"
      EXPLANATION="At least one day must be specified."

    # Valid input received for a new alarm
    elif [[ -n "${MERROR}" ]] ; then
      ALM_ERR="${MERROR}"
      EXPLANATION="Specify a valid stream or file name or folder name"

    # Check that the supplied stream is non-blank
    elif [[ -z "${DEFAULT_STREAM_OR_FILE}" ]] ; then
      ALM_ERR="Stream not specified"
      EXPLANATION="Specify an internet stream or browse for a file or folder containing playable media."

    # Does the time already exist in the alarm file
    elif [[ -r "${ALARM_FILE}" ]] && [[ "${command}" == "new_alarm" ]] && grep -q "${DEFAULT_TIME}" "${ALARM_FILE}" ; then
      echo "   <form action='${THIS_SCRIPT}' method='get'>"
      echo '    <fieldset>'
      Include_Defaults
      echo "     <legend><b>Alarm already exists at ${DEFAULT_TIME}</b></legend>"
      echo "     <button type='submit' class='red' name='command' value='overwrite_alarm'>Overwrite the existing alarm at ${DEFAULT_TIME}</button>"
      echo '    </fieldset>'
      echo '   </form>'

    # If we are not in filesystem browsing mode, then create a new alarm with the provided input
    elif [[ -z "${browse}" ]] && [[ -n "${DEFAULT_ALARM_DURATION}" ]] && [[ ${DEFAULT_ALARM_DURATION} -ge 1 ]] && [[ ${DEFAULT_ALARM_DURATION} -le 180 ]] ; then
      STREAM="${DEFAULT_STREAM_OR_FILE}"

      if [[ -d "${DEFAULT_STREAM_OR_FILE}" ]] ; then
	Ingest_Media_Definition "${DEFAULT_STREAM_OR_FILE}"

	if [[ -z "${MERROR}" ]] ; then
	  STREAM="${MEDIA_CMD}"
	else
	  STREAM=""
	fi
      fi

      if [[ -n "${STREAM}" ]] ; then
	# The following line must be the same as that in common.sh
	# NOTE: The following line starts with a SPACE, so it sorts to be the first line. Don't remove the space!!!
	echo " Alarm Time,Suspended,Alarm Duration,Volume Adjust,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday,Stream" > "${TMP}"

	if [[ "${INIT_VOLUME_ADJUST}" == "${TARG_VOLUME_ADJUST}" ]] ; then
	  echo -n "${DEFAULT_TIME},_ACTIVE_,${DEFAULT_ALARM_DURATION},${INIT_VOLUME_ADJUST}" >> "${TMP}"
	else
	  echo -n "${DEFAULT_TIME},_ACTIVE_,${DEFAULT_ALARM_DURATION},${INIT_VOLUME_ADJUST}..${TARG_VOLUME_ADJUST}" >> "${TMP}"
	fi

	# Format is slightly different for once-only and for repetative alarms
	if [[ "${ONCE}" == 'Y' ]] ; then

	  # Check each day of the week, and either output the epoch time of the next one-off alarm, or 'N'
	  Write_Alarm_Column "mon" "${MON}" "${DEFAULT_TIME}"
	  Write_Alarm_Column "tue" "${TUE}" "${DEFAULT_TIME}"
	  Write_Alarm_Column "wed" "${WED}" "${DEFAULT_TIME}"
	  Write_Alarm_Column "thu" "${THU}" "${DEFAULT_TIME}"
	  Write_Alarm_Column "fri" "${FRI}" "${DEFAULT_TIME}"
	  Write_Alarm_Column "sat" "${SAT}" "${DEFAULT_TIME}"
	  Write_Alarm_Column "sun" "${SUN}" "${DEFAULT_TIME}"

	  # Finish the line with the stream and emit a newline this time
	  echo ",${STREAM}" >> "${TMP}"

	else
	  echo ",${MON},${TUE},${WED},${THU},${FRI},${SAT},${SUN},${STREAM}" >> "${TMP}"
	fi

	# Read the existing alarm file, removing the first line (which is a comment) and ignoring alarms at the same time as the new one
	[[ -f "${ALARM_FILE}" ]] && tail -n +2 "${ALARM_FILE}" 2> /dev/null | grep -v "${DEFAULT_TIME}" 2> /dev/null >> "${TMP}"

	if [[ -f "${TMP}" ]] && [[ -s "${TMP}" ]] ; then
	  # Do we want to back up the current alarm file??? 720 minutes is 12 hours
	  if test $(find "${ALARM_FILE}" -mmin +720) ; then
	    # The alarm file is older than 12 hours - create a backup before we overwrite it
	    dir=$(dirname "${ALARM_FILE}")
	    filename=$(basename "${ALARM_FILE##*/}")
	    extension="${filename##*.}"
	    filename="${filename%.*}"

	    # We are using a different backup numbering scheme than the C code alarmN.csv here, alarm.csv.N in the C code
	    for ((i=9 ; i >= 1 ; i--)) ; do
	      prev=$(( i - 1 ))
	      mv "${dir}/${filename}${prev}.${extension}" "${dir}/${filename}${i}.${extension}" 2> /dev/null
	    done

	    # move the current alarm file into the next temporary alarm filename
	    mv "${ALARM_FILE}" "${dir}/${filename}0.${extension}" 2> /dev/null
	  fi

	  sort "${TMP}" > "${ALARM_FILE}"
	  rm -f "${TMP}"

	  # The SHA1 hash for the file has changed.... update the hash
	  CURRENT_SHA1=$(shasum "${ALARM_FILE}" | awk '{print $1}')

	  # Issue the command to reload the new settings
	  $EXE -r > /dev/null 2>&1
	fi
      fi
    fi
  fi
fi
####################################################################################################################
if [[ -n "${ALM_ERR}" ]] ; then
  echo '   <fieldset>'
  echo '    <legend><b>Error!</b></legend>'
  echo "    <p style='color:red;font-size:25px;'>${ALM_ERR}!</p>"
  [[ -n "${EXPLANATION}" ]] && echo "    <p>${EXPLANATION}</p>"
  echo '   </fieldset>'
  echo '   <br>'
fi
####################################################################################################################
# Check if filesystem browsing is initiated or stopped
if [[ "${command}" == "commence_browsing" ]] ; then
  BROWSE_DIR="${DEFAULT_BROWSE_DIR}"
  browse='B'
fi

if [[ "${command}" == "cease_browsing" ]] ; then
  browse=''
fi
####################################################################################################################
# Check for alarm deletions or suspensions
if [[ "${command}" == "delete_alarm" ]] || [[ "${command}" == "suspend_alarm" ]] || [[ "${command}" == "reinstate_alarm" ]] ; then
  if [[ "${PREV_SHA1}" == "${CURRENT_SHA1}" ]] && [[ -f "${ALARM_FILE}" ]] && [[ -s "${ALARM_FILE}" ]] ; then
    NUM_LINES=$(wc -l "${ALARM_FILE}" | sed 's/[[:space:]].*$//g')
    NUM_CHANGED=0

    # We need to loop and try each potential line number. Delete the lines in reverse order, so that when there is
    # more than one alarm to delete, the line numbers don't get messed up. Also, we are not interested in the
    # first line of the file because that line holds the column headings
    for ((i=${NUM_LINES} ; i > 1 ; i--)) ; do
      VAR="Sel${i}"
      [[ -z "${!VAR}" ]] && continue ;

      # We have a line number to delete or suspend. Issue the command
      if [[ "${command}" == "delete_alarm" ]] ; then
	sed "${i}d" "${ALARM_FILE}" > "${TMP}"

      elif [[ "${command}" == "suspend_alarm" ]] ; then
	sed "${i}s/_ACTIVE_/_SUSPENDED_/" "${ALARM_FILE}" > "${TMP}"

      elif [[ "${command}" == "reinstate_alarm" ]] ; then
	sed "${i}s/_SUSPENDED_/_ACTIVE_/" "${ALARM_FILE}" > "${TMP}"
      fi

      # the following dance is necessary because apache is configured to be unable to move files
      # so we cannot move the temp file to the alarm file, we need to overwrite it
      [[ $? -eq 0 ]] && cat "${TMP}" > "${ALARM_FILE}"
      rm -f "${TMP}"

      NUM_CHANGED=$((NUM_CHANGED+1))
    done

    if [[ ${NUM_CHANGED} -eq 0 ]] ; then
      echo '   <fieldset>'
      echo '    <legend><b>Error!</b></legend>'
      echo '    <p style="color:red;font-size:25px;">No current alarms deleted or suspended!</p>'
      echo '    <p>Try ticking at least one alarm box before trying to delete or suspend alarms again.</p>'
      echo '   </fieldset>'
    else
      # The SHA1 hash for the file has changed.... update the hash
      CURRENT_SHA1=$(shasum "${ALARM_FILE}" | awk '{print $1}')

      # Issue the command to reload the new settings
      $EXE -r > /dev/null 2>&1
    fi
  fi
fi
####################################################################################################################
# If we are NOT in filesystem browsing mode, display the existing alarms, with option to delete
[[ -z "${browse}" ]] && Emit_HTML_Alarm_Table Y
####################################################################################################################
# input form that offers option to create a new alarm
# This form is more complicated than above, as it has one state for fetching the initial data, and a separate
# state that reveals a filesystem browser (for finding media files to play as alarm sounds/playlists)
if [[ -z "${browse}" ]] ; then
  echo "   <form action='${THIS_SCRIPT}' method='get'>"
  echo '    <fieldset>'

  if [[ -n "${ALARM_COPY_LINE}" ]] ; then
    (( alarm_copy-- ))
    echo "     <legend><b>Copy of alarm #${alarm_copy} values</b></legend>"
  else
    echo '     <legend><b>Create a new alarm</b></legend>'
  fi

  Include_Defaults
  echo '     New alarm time:'
  echo "     <input type='time' name='default_time' value='${DEFAULT_TIME}' required>"

  echo '     <hr>'
  Do_Slider default_alarm_duration "Alarm duration" 1 180 ${DEFAULT_ALARM_DURATION} 'N' "minutes"

  echo '     <hr>'
  echo "     <div class='tip'>"
  echo '      Day or days of the week for the new alarm:<br>'
  echo "      <span class='text'>Highlight 'Alarm' to enable the alarm on the associated day.<br><br>Highlight 'No Alarm' to disable the alarm on the associated day.</span>"
  echo '     </div>'

  echo "     <table class='noborder'>"
  Do_Choice T MON Monday "Alarm" "No Alarm"
  Do_Choice T TUE Tuesday "Alarm" "No Alarm"
  Do_Choice T WED Wedesday "Alarm" "No Alarm"
  Do_Choice T THU Thursday "Alarm" "No Alarm"
  Do_Choice T FRI Friday "Alarm" "No Alarm"
  Do_Choice T SAT Saturday "Alarm" "No Alarm"
  Do_Choice T SUN Sunday "Alarm" "No Alarm"
  echo "     </table>"
  echo "     <br>"

  echo "     <table class='noborder'>"
  Do_Choice T ONCE "Alarm type" "One-off" "Recurring"
  echo "     </table>"

  echo '     <hr>'
  echo "     <div class='tip'>"
  Do_Slider init_volume_adjust "Volume Adjustment" ${MIN_VOL_ADJ} ${MAX_VOL_ADJ} ${INIT_VOLUME_ADJUST} 'IVA' "added to the system-wide volume setting when this alarm first trips"
  Do_Slider targ_volume_adjust "Target Adjustment" ${MIN_VOL_ADJ} ${MAX_VOL_ADJ} ${TARG_VOLUME_ADJUST} 'TVA' "if different, volume gradually adjusts to this target as the alarm is playing"
  echo "      <span class='text'>These sliders may be used either (a) to gradually increase or decrease new alarm's volume, or (b) to equalise the volume of chained media streams or files that have different volume levels. If not required, leave both sliders set to 0.<br><br>The 'Volume Adjustment' is added to the current volume level when the alarm first trips. If the 'Target Adjustment' differs, volume will step once every minute while the alarm is playing until the target volume is reached.<br><br>If the alarm is a playlist, the adjustment will apply to every track in the playlist, even if the tracks in the playlist have not had their volumes equalised (ie for best results, run a volume equaliser over all the files in the playlist beforehand).</span>"
  echo '     </div>'

  echo '     <hr>'
  echo "     Accept the following defaults <i>OR</i> choose <i>one</i> of the following options, <i>THEN</i> click 'Create new alarm':<br>"
  echo "      <ol type='a'>"
  echo "       <li><button type='submit' class='blue' name='command' value='commence_browsing'>Browse for a file, folder or playlist</button></li><br>"

  echo "       <div class='tip'>"
  echo '        <li>Type a streaming URL <i>OR</i> a path to a media file, folder or playlist <i>OR</i> specify radio:<br>'

  EXTRA=""
  [[ -f "${DEFAULT_STREAM_OR_FILE}" ]] && [[ $(file "${DEFAULT_STREAM_OR_FILE}" | sed 's/^.*:[[:space:]]\+//g;s/[[:space:]].*//g') == "ASCII" ]] && EXTRA="($(wc -l < ${DEFAULT_STREAM_OR_FILE}) items)"
  Do_Text_Input 'stream' "${DEFAULT_STREAM_OR_FILE}" 'Enter a streaming URL or a path to a file, folder or playlist' "100" "${EXTRA}"

  [[ -n "${DEFAULT_STREAM_OR_FILE}" ]] && DEFAULT_STREAM_OR_FILE_VALUE="value='${DEFAULT_STREAM_OR_FILE}'"
  Insert_Examples
  echo "         <span class='text'>Enter a media streaming URL to stream live from the internet. Some examples of media streaming URLs can be found using the provided streaming search site link.<br><br>Enter the full filesystem path to a file or folder to play local media. Use the browse function to interactively find the right filesystem path.<br><br>Set this field to 'radio' (without the quotes) to default to turning on the radio.<br><br>If the specified path is a directory, a playlist file will be automatically generated and saved into the ${PLAYLIST_DIR} folder. Note that if the characters in the directory or file names contain spaces or special characters, the media files may not be recognised. You can run the fix-media-tree script from an ssh session to remove those characters.</span>"
  echo '        </li>'
  echo '       </div>'
  echo '      </ol><hr>'
  echo "     <div class='tip'>"
  echo "      <button type='submit' class='green' name='command' value='new_alarm'>Create new alarm</button>"
  echo "      <span class='text'>A new alarm will be created based on the information currently showing in this form</span>"
  echo '     </div>'

  echo '    </fieldset>'
  echo '   </form>'

  # Button to abort the alarm configuration menu
  echo "   <form action='/index.html'>"
  echo '    <fieldset>'
  echo "     <div class='tip'>"
  echo "      <button type='submit' class='red' >Return to main page</button>"
  echo "      <span class='text'>Clicking here will cause any unsaved changes in this form to be discarded and lost</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '   </form>'

else
  echo "   <form action='${THIS_SCRIPT}' method='get'>"
  Include_Defaults
  Include_FileSystem_Browser

  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Cease browsing for playable media</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='cease_browsing'>Cease browsing for a file, folder or playlist</button> (making no selection)<br>"
  echo '    </fieldset>'
  echo '   </form>'
fi

  echo '  </div>'
####################################################################################################################
Insert_Debug_Section
####################################################################################################################
echo ' </body>'
echo '</html>'

[[ -f "${TMP}" ]] && rm -f "${TMP}"
exit 0
