#!/bin/bash
#                           Digital Alarm Clock
#                Copyright (C) 2023 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
####################################################################################################################
THIS_SCRIPT="/cgi-bin/mediaplayer.cgi"
MEDIA_SCRIPT="/cgi-bin/media_change.cgi"

# Load common functions and data
INCLUDE_COMMON="common.sh"

if test -f "${INCLUDE_COMMON}" ; then
  . ${INCLUDE_COMMON}
fi

Make_Temp "$0"
Read_HTML_Parameters
Process_Inputs
Emit_HTML_Headers icons,noborder
Do_Start_Body

echo "  <div class='background'>"

[[ -n "${STREAM}" ]] && DEFAULT_STREAM_OR_FILE="${STREAM}"

if [[ -z "${PATH_TO_PLAYLIST_FILE}" ]] ; then
  PATH_TO_PLAYLIST_FILE="/media/current_playlist.m3u"
  USERNAME=$(cd /home ; ls 2> /dev/null | head -n 1)

  if [[ -n "${USERNAME}" ]] && [[ -d "/home/${USERNAME}" ]] && [[ -w "/home/${USERNAME}" ]] ; then
    PATH_TO_PLAYLIST_FILE="/home/${USERNAME}/current_playlist.m3u"
  fi
fi
####################################################################################################################
# Process and commands and directives

if [[ "${command}" == "commence_stream" ]] ; then
  browse='B'
  mode='M'
  command=''

  [[ -f "${DEFAULT_STREAM_OR_FILE}" ]] && BROWSE_DIR="$(dirname "${DEFAULT_STREAM_OR_FILE}")"
  [[ -d "${DEFAULT_STREAM_OR_FILE}" ]] && BROWSE_DIR="${DEFAULT_STREAM_OR_FILE}"
fi

if [[ "${command}" == "playlist_filename" ]] ; then
  browse='F'
  mode='P'
  command=''

  if [[ -f "${DEFAULT_STREAM_OR_FILE}" ]] ; then
    BROWSE_DIR="$(dirname "${PATH_TO_PLAYLIST_FILE}")"
  elif [[ -d "${PATH_TO_PLAYLIST_FILE}" ]] ; then
    BROWSE_DIR="${PATH_TO_PLAYLIST_FILE}"
  else
    BROWSE_DIR="$(dirname "${PATH_TO_PLAYLIST_FILE}")"
  fi
fi

if [[ "${command}" == "cease_browsing" ]] ; then
  browse=''
  command=''
fi

if [[ -n "${stop_browsing}" ]] ; then
  browse=''

  if [[ "${mode}" == 'M' ]] ; then
    Ingest_Media_Definition "${stop_browsing}"

    if [[ -z "${MERROR}" ]] ; then
      if [[ -f "${PLAYLIST}" ]] ; then
	DEFAULT_STREAM_OR_FILE="${PLAYLIST}"
      else
	DEFAULT_STREAM_OR_FILE="${NEW_REFERENCE}"
      fi

      LAUNCH_CMD="${MEDIA_CMD}"
    else
      STREAM_ERROR="${MERROR}"
      EXPLANATION="File must be a valid media file, with no spaces in filename or path."
    fi

  elif [[ "${mode}" == 'P' ]] ; then
    DECODED_PLAYLIST="$(Percent_Decode "${stop_browsing}")"
  fi

  mode=''
fi

[[ -n "${path_to_playlist_file}" ]] && DECODED_PLAYLIST="$(Percent_Decode "${path_to_playlist_file}")"

if [[ -n "${DECODED_PLAYLIST}" ]] ; then
  # Does DECODED_PLAYLIST point to an existing directory that is writeable?
  if [[ -d "${DECODED_PLAYLIST}" ]] && [[ -w "${DECODED_PLAYLIST}" ]] ; then
    PAtH_TO_PLAYLIST_FILE="${DECODED_PLAYLIST}/current_playlist.m3u"

  # Does DECODED_PLAYLIST point to an existing file??
  elif [[ -f "${DECODED_PLAYLIST}" ]] ; then
    PATH_TO_PLAYLIST_FILE="${DECODED_PLAYLIST}"

  # Does the file represented in DECODED_PLAYLIST exist in a writeable directory?
  else
    PLAYLIST_SAVE_DIR=$(dirname "${DECODED_PLAYLIST}")

    # If PLAYLIST_SAVE_DIR is a directory into which we can write, accept DECODED_PLAYLIST as the playlist filename
    if [[ -d "${PLAYLIST_SAVE_DIR}" ]] && [[ -w "${PLAYLIST_SAVE_DIR}" ]] ; then
	PATH_TO_PLAYLIST_FILE="${DECODED_PLAYLIST}"
    else
      STREAM_ERROR="${DECODED_PLAYLIST} is not a playlist in an accessible directory"
      EXPLANATION="File must be accessible"
    fi
  fi
fi
###################################################################################################################
if [[ "${command}" == "add_playlist" ]] ; then
  if [[ -f "${LAUNCH_CMD}" ]] ; then
    $EXE -m "${LAUNCH_CMD}" -t ${DEFAULT_MEDIA_DURATION} -v ${VOLUME} > /dev/null 2>&1
  else
    $EXE -m "${DEFAULT_STREAM_OR_FILE}" -t ${DEFAULT_MEDIA_DURATION} -v ${VOLUME} > /dev/null 2>&1
  fi

  # Delay a little to allow the media player to process the request. The delay needs to be longer for older Pis
  # On a Pi4, 0.75 seconds is OK. For a Pi3, 1 second appears to work ok.
  sleep 1
fi

if [[ "${command}" == "clear_playlist" ]] ; then
  $EXE -c clear > /dev/null 2>&1
fi

if [[ "${command}" == "prev_track" ]] ; then
  $EXE -c prev > /dev/null 2>&1
fi

if [[ "${command}" == "restart_track" ]] ; then
  $EXE -c restart > /dev/null 2>&1
fi

if [[ "${command}" == "seek_back" ]] ; then
  $EXE -c seek-back > /dev/null 2>&1
fi

if [[ "${command}" == "pause" ]] ; then
  $EXE -c pause > /dev/null 2>&1
fi

if [[ "${command}" == "seek_forwards" ]] ; then
  $EXE -c seek-fwd > /dev/null 2>&1
fi

if [[ "${command}" == "next_track" ]] ; then
  $EXE -c next > /dev/null 2>&1
fi

if [[ "${command}" == "stop" ]] ; then
  $EXE -c stop > /dev/null 2>&1
fi

if [[ "${command}" == "shuffle" ]] ; then
  $EXE -c shuffle > /dev/null 2>&1
fi
####################################################################################################################
if [[ "${command}" == "save_playlist" ]] && [[ -n "${PATH_TO_PLAYLIST_FILE}" ]] ; then
  # Read the playlist into an array
  readarray -t PlayList < <( ${EXE} -c playlist )

  if [[ ${#PlayList[@]} -gt 0 ]] ; then
    # Is the specified name a directory or a file?
    [[ -d "${PATH_TO_PLAYLIST_FILE}" ]] && PATH_TO_PLAYLIST_FILE="${PATH_TO_PLAYLIST_FILE}/current_playlist.m3u"

    # Remove existing playlist file of same name (if it exists)
    [[ -f "${PATH_TO_PLAYLIST_FILE}" ]] && rm -f "${PATH_TO_PLAYLIST_FILE}"

    # Save the items in the playlist to the file
    for ((i=0 ; i < ${#PlayList[@]} ; i++)) ; do
      echo ${PlayList[$i]} | sed 's/.*,"//;s/",.*//' >> "${PATH_TO_PLAYLIST_FILE}"
    done
  fi
fi
####################################################################################################################
[[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${DEFAULT_BROWSE_DIR}"
[[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${MEDIA_DIR}"
####################################################################################################################
if [[ -n "${STREAM_ERROR}" ]] ; then
  echo '   <fieldset>'
  echo '    <b>Error!</b>'
  echo "    <p style='color:red;font-size:25px;'>${STREAM_ERROR}!</p>"
  [[ -n "${EXPLANATION}" ]] && echo "    <p>${EXPLANATION}</p>"
  echo '   </fieldset>'
fi
###################################################################################################################
# Read ambient light, volume level and remaining playing time
CURRENT_READINGS=$($EXE -a)
IS_PLAYLIST=""

# Set VOLUME and DEFAULT_MEDIA_DURATION based on the values that the clock is currently using
if [[ -n "${CURRENT_READINGS}" ]] ; then
  CURRENT_VOLUME=$(echo $CURRENT_READINGS | cut -d, -f2)
  CURRENT_DURATION=$(echo $CURRENT_READINGS | cut -d, -f3)

  if [[ "${CURRENT_VOLUME}" -ge 0 ]] && [[ "${CURRENT_VOLUME}" -le 31 ]] ; then
    VOLUME="${CURRENT_VOLUME}"
  fi

  if [[ "${CURRENT_DURATION}" -ge 0 ]] && [[ "${CURRENT_DURATION}" -le 180 ]] ; then
    [[ "${CURRENT_DURATION}" -eq 0 ]] && CURRENT_DURATION='1'
    DEFAULT_MEDIA_DURATION="${CURRENT_DURATION}"
    IS_PLAYLIST="Y"
  fi
fi
#--------------------------------------------------------
echo "   <form action='${THIS_SCRIPT}' method='get'>"
Include_Defaults

if [[ "${browse}" == 'B' ]] ; then
  Include_FileSystem_Browser
  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Browsing for default folder</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='choose_default_streaming_folder'>Choose ${BROWSE_DIR} as the default streaming folder</button><br>"
  echo '    </fieldset>'

  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Cease browsing for playable media</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='cease_browsing'>Cease browsing for a file, folder or playlist</button> (making no selection)<br>"
  echo '    </fieldset>'
  echo '   </form>'

elif [[ "${browse}" == 'F' ]] ; then
  Include_FileSystem_Browser
  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Browsing for a playlist filename</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='choose_default_streaming_folder'>Choose ${BROWSE_DIR} as the default streaming folder</button><br>"
  echo '    </fieldset>'

  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Cease browsing for a playlist filename</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='cease_browsing'>Cease browsing for a file, folder or playlist</button> (making no selection)<br>"
  echo '    </fieldset>'
  echo '   </form>'

else
  # input form that lists existing alarms, and offers the option to delete them.
  echo '    <fieldset>'
  echo '     <legend><b>Media Location</b></legend>'
  echo '     Either'
  echo '     <ol type="a">'

  echo "      <div class='tip'>"
  echo '       <li>'
  echo "        Type a streaming URL <i>OR</i> a path to a media file, folder or playlist <i>OR</i> specify radio:<br>"
  Do_Text_Input 'stream' "${DEFAULT_STREAM_OR_FILE}" 'Enter a streaming URL or a path to a file, folder or playlist' "120"
  Insert_Examples
  echo "      <span class='text'>Enter a media streaming URL to stream live from the internet. Some examples of media streaming URLs can be found using the provided streaming search site link.<br><br>Enter the full filesystem path to a file or folder to play local media. Use the browse function to interactively find the right filesystem path.<br><br>Set this field to 'radio' (without the quotes) to play the radio.<br><br>If the specified path is a directory, a playlist file will be automatically generated and saved into the ${PLAYLIST_DIR} folder. Note that if the characters in the directory or file names contain spaces or special characters, the media files may not be recognised. You can run the fix-media-tree script from an ssh session to remove those characters.</span>"
  echo '       </li>'
  echo '      </div>'

  echo '      <li>'
  echo "       <button class='blue' type='submit' name='command' value='commence_stream'>Browse filesystem to find playable media, a folder or playlist</button><br>"
  echo '      </li>'
  echo '     </ol>'

  echo "     <i>THEN</i> click 'Play/Add to Playlist'<br>"

  echo "     <div class='tip'>"
  echo "      <button class='green' type='submit' name='command' value='add_playlist'>Play / Add to Playlist<br><i class='fa fa-play-circle fa-2x' style='padding-top: 7px'></i> <i class='fa fa-plus-square fa-2x' style='padding-top: 7px'></i></button>"
  echo "      <span class='text'>If nothing is playing, adding media to the playlist will start it playing.<br><br>The playlist is managed by the 'mpv' application. If a playlist has already been defined (and is visible below), newly added media will be appended without disrupting the current track or stream. If adding a playlist file (i.e. a filename ending in .pls, .m3u, .m3u8, etc), then 'mpv' will expand the contents when the first track from it starts playing.</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '    <br>'

  echo '    <fieldset>'
  echo "     <legend><b>Duration</b></legend>"
  echo "     <div class='tip'>"
  Do_Slider default_media_duration "Set playing time from now" 0 180 ${DEFAULT_MEDIA_DURATION} 'D' "minutes"
  echo "      <span class='text'>The duration can be set before adding the initial media, or it can be adjusted whilst the media is playing.</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '    <br>'

  echo '    <fieldset>'
  echo "     <legend><b>Volume</b></legend>"
  echo "     <div class='tip'>"
  Do_Slider volume "Set volume" 0 31 ${VOLUME} 'V'
  echo "      <span class='text'>The volume can be set before adding the initial media, or it can be adjusted whilst the media is playing.</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '    <br>'

  echo '    <fieldset>'
  echo "     <button class='green' type='submit' name='command' value='prev_track'>Previous Track<br><i class='fa fa-fast-backward fa-lg' style='padding-top: 7px'></i></button>"
  echo "     <button class='green' type='submit' name='command' value='restart_track'>Restart Track<br><i class='fa fa-step-backward fa-lg' style='padding-top: 7px'></i></button>"
  echo "     <button class='green' type='submit' name='command' value='seek_back'>Seek Backwards<br><i class='fa fa-backward fa-lg' style='padding-top: 7px'></i></button>"
  echo "     <button class='blue' type='submit' name='command' value='pause'>Pause<br><i class='fa fa-pause fa-lg' style='padding-top: 7px'></i></button>"

  echo "     <div class='tip'>"
  echo "      <button class='red' type='submit' name='command' value='stop'>Stop<br><i class='fa fa-stop fa-lg' style='padding-top: 7px'></i></button>"
  echo "      <span class='text'>Pressing 'Stop' will clear the current playlist and stop the media player.<br><br>To retain the current playlist, use 'pause' instead of 'stop'.</span>"
  echo '     </div>'

  echo "     <button class='green' type='submit' name='command' value='seek_forwards'>Seek Forwards<br><i class='fa fa-forward fa-lg' style='padding-top: 7px'></i></button>"
  echo "     <button class='green' type='submit' name='command' value='next_track'>Next Track<br><i class='fa fa-fast-forward fa-lg' style='padding-top: 7px'></i></button>"

  echo '    </fieldset>'
  echo '   </form>'

  if [[ -n "${IS_PLAYLIST}" ]] ; then
    echo "   <form action='${THIS_SCRIPT}' method='get'>"
    Include_Defaults

    echo '    <fieldset>'
    echo "     <div class='tip'>"
    echo "      <button class='red' type='submit' name='command' value='clear_playlist'>Clear Playlist<br><i class='fa fa-stop-circle fa-2x' style='padding-top: 7px'></i></button>"
    echo "      <span class='text'>Clearing the playlist will not interrupt the currently playing track.<br><br>Use 'pause' or 'stop' to stop the audio.</span>"
    echo '     </div>'

    echo "     <div class='tip'>"
    echo "      <button class='blue' type='submit' name='command' value='shuffle'>Shuffle<br><i class='fa fa-random fa-2x' style='padding-top: 7px'></i></button>"
    echo "      <span class='text'>Pressing 'Shuffle' will randomise the current playlist. The currently playing track will be shuffled into a new position without disrupting play.</span>"
    echo '     </div>'
    echo '    </fieldset>'
    echo '    <br>'

    echo '    <fieldset>'
    echo '     <legend><b>Save The Current Playlist</b></legend>'
    echo "     <div class='tip'>"
    Do_Text_Input 'path_to_playlist_file' "${PATH_TO_PLAYLIST_FILE}" 'Enter the full path to a playlist filenamwe into which to save the current playlist' "120"
    echo "      <button class='blue' type='submit' name='command' value='playlist_filename'>Browse filesystem to choose a filename into which to save the current playlist</button><br>"
    echo "      <button class='green' type='submit' name='command' value='save_playlist'>Save the current playlist into the nominated file</button>"
    echo "      <span class='text'>Enter the full linux pathname to a playlist file, or choose the browsing option to find the name of a playlist file you wish to overwrite, and then click the 'Save the current playlist' button to save the displayed playlist into that filename.</span>"
    echo '     </div>'
    echo '    </fieldset>'
    echo '   </form>'
  fi

  echo "   <form action='/index.html'>"
  echo '    <fieldset>'
  echo "     <button class='red' type='submit'>Return to main page</button>"
  echo '    </fieldset>'
  echo '   </form>'

  if [[ -n "${IS_PLAYLIST}" ]] ; then
    Insert_Current_Playlist
  fi
fi

echo '  </div>'
Insert_Debug_Section
####################################################################################################################
echo ' </body>'
echo '</html>'

[[ -f "${TMP}" ]] && rm -f "${TMP}"
exit 0
