#!/bin/bash
#                           Digital Alarm Clock
#                Copyright (C) 2023,2024 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
####################################################################################################################
# Enable the DEBUG flag by uncommenting the following. (The values may also be overwritten by the contents of setup file)
#DEBUG="Y"
ETC_DIR="/etc/alarm-clock"
SETUP_FILE="${ETC_DIR}/setup.conf"
ALARM_FILE="${ETC_DIR}/alarms.csv"
PLAYLIST_DIR="${ETC_DIR}/playlists"
FALLBACK_PLAYLIST="${PLAYLIST_DIR}/fallback-playlist.m3u"
EXE="alarm-clock"
SLIDER_CHANGE="/cgi-bin/slider_change.cgi"
MEDIA_DIR="/media"
IsNumber='^[+-]*[0-9]+$'
MIN_VOL_ADJ="-24"
MAX_VOL_ADJ="24"

# If the HTTP_HOST is not an IP address, set the domain name
[[ "${HTTP_HOST}" == *"."* ]] && DOMAIN_NAME=".${HTTP_HOST#*.}"

if [[ "${HTTP_HOST}" =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]] ; then
  OIFS=$IFS
  IFS='.'

  # Read the IP address into an array - check if the numbers are all less than 255
  ip=( $HTTP_HOST )
  IFS=$OIFS

  [[ ${ip[0]} -le 255 && ${ip[1]} -le 255 && ${ip[2]} -le 255 && ${ip[3]} -le 255 ]] && DOMAIN_NAME=""
fi

# If there is an existing configuration file - load its contents now
# The format is similar to a bash script - including comments. Load as if it was a bash script by including the file
if [[ -f "${SETUP_FILE}" ]] ; then
  . ${SETUP_FILE}
fi
####################################################################################################################
# Make a template for a temporary file, in case we need it
Make_Temp()
{
  BASE=$(basename "$1")
  TMP=$(mktemp /tmp/${BASE}.XXXXXX)
}
####################################################################################################################
# Function to propagate bash variable states through web page refreshes, using hidden inputs.
#
# The difficulty with web pages is that its a little clumsy to propagate state from one rendition of a page to the
# next (eg when pressing a button, you want to remember a bunch of other stuff that has already been input)
# I've kludged this using hidden inputs based around the shell variables that hold the values I want to propagate.
#
# Capitalised variables are shell variables that already have a state. Lower case variables are new inputs.
#
# DEFAULT_1224: 12 or 24
# DEFAULT_HOUR_ZERO: 'blank' or 'nonblank' (display leading zero for hours)
# DEFAULT_SNOOZE_PRESS: 'ignore' or 'restart' (second snooze press ignored, or restarts the media)
# DEFAULT_TIME: "07:00" the default time that appears on the new alarm page, in 24H format
# DEFAULT_ALARM_DURATION: 1 to 180. The default alarm duration that appears on the new alarm page.
# VOLUME: 0 to 31 - the current volume setting
# INIT_VOLUME_ADJUST: MIN_VOL_ADJ to MAX_VOL_ADJ - the current volume adjustment setting
# TARG_VOLUME_ADJUST: MIN_VOL_ADJ to MAX_VOL_ADJ - the current volume adjustment setting
# FALLBACK_ALARM_FILE: path to a file or folder that will be played if an alarm stream fails
# DEFAULT_STREAM_OR_FILE: web stream or path to file or folder that initially populates new alarm stream option
# DEFAULT_BROWSE_DIR: path that will be the default starting point when browsing the filesystem
# NEW_HOSTNAME: a new name to replace the current hostname
# LAUNCH_CMD: linux command to play the selected media (will be auto-filled if not manually entered)
# SHUFFLE: 'Y' if shuffle directories / playlists has been set
# BROWSE_DIR: the directory currently being browsed
# CLOCK_TYPE: 'A' if an autonomous clustered clock, 'C' if a non-autonomous clusted clock, 'Z' if a silent-clustered, 'S' if stand-alone

Include_Defaults()
{
  [[ -n "${DEFAULT_1224}" ]] &&		echo "    <input type='hidden' name='DEFAULT_1224' value='${DEFAULT_1224}'>"
  [[ -n "${DEFAULT_HOUR_ZERO}" ]] &&	echo "    <input type='hidden' name='DEFAULT_HOUR_ZERO' value='${DEFAULT_HOUR_ZERO}'>"
  [[ -n "${DEFAULT_SNOOZE_PRESS}" ]] &&	echo "    <input type='hidden' name='DEFAULT_SNOOZE_PRESS' value='${DEFAULT_SNOOZE_PRESS}'>"
  [[ -n "${DEFAULT_TIME}" ]] &&		echo "    <input type='hidden' name='DEFAULT_TIME' value='${DEFAULT_TIME}'>"
  [[ -n "${DEFAULT_ALARM_DURATION}" ]] && echo "    <input type='hidden' name='DEFAULT_ALARM_DURATION' value='${DEFAULT_ALARM_DURATION}'>"
  [[ -n "${DEFAULT_SNOOZE_DURATION}" ]] && echo "    <input type='hidden' name='DEFAULT_SNOOZE_DURATION' value='${DEFAULT_SNOOZE_DURATION}'>"
  [[ -n "${DEFAULT_MEDIA_DURATION}" ]] && echo "    <input type='hidden' name='DEFAULT_MEDIA_DURATION' value='${DEFAULT_MEDIA_DURATION}'>"
  [[ -n "${CURRENT_DURATION}" ]] &&	echo "    <input type='hidden' name='CURRENT_DURATION' value='${CURRENT_DURATION}'>"

  [[ -n "${VOLUME}" ]] &&		echo "    <input type='hidden' name='VOLUME' value='${VOLUME}'>"
  [[ -n "${MIN_VOLUME}" ]] &&		echo "    <input type='hidden' name='MIN_VOLUME' value='${MIN_VOLUME}'>"
  [[ -n "${CURRENT_VOLUME}" ]] &&	echo "    <input type='hidden' name='CURRENT_VOLUME' value='${CURRENT_VOLUME}'>"
  [[ -n "${INIT_VOLUME_ADJUST}" ]] &&	echo "    <input type='hidden' name='INIT_VOLUME_ADJUST' value='${INIT_VOLUME_ADJUST}'>"
  [[ -n "${TARG_VOLUME_ADJUST}" ]] &&	echo "    <input type='hidden' name='TARG_VOLUME_ADJUST' value='${TARG_VOLUME_ADJUST}'>"
  [[ -n "${FALLBACK_ALARM_FILE}" ]] &&	echo "    <input type='hidden' name='FALLBACK_ALARM_FILE' value='${FALLBACK_ALARM_FILE}'>"
  [[ -n "${DEFAULT_STREAM_OR_FILE}" ]] && echo "    <input type='hidden' name='DEFAULT_STREAM_OR_FILE' value='${DEFAULT_STREAM_OR_FILE}'>"
  [[ -n "${DEFAULT_BROWSE_DIR}" ]] &&	echo "    <input type='hidden' name='DEFAULT_BROWSE_DIR' value='${DEFAULT_BROWSE_DIR}'>"
  [[ -n "${LAUNCH_CMD}" ]] &&		echo "    <input type='hidden' name='LAUNCH_CMD' value='${LAUNCH_CMD}'>"
  [[ -n "${BROWSE_DIR}" ]] &&		echo "    <input type='hidden' name='BROWSE_DIR' value='${BROWSE_DIR}'>"
  [[ -n "${PATH_TO_PLAYLIST_FILE}" ]] && echo "    <input type='hidden' name='PATH_TO_PLAYLIST_FILE' value'${PATH_TO_PLAYLIST_FILE}'>"

  [[ -n "${DEFAULT_MIN_LED}" ]] &&	echo "    <input type='hidden' name='DEFAULT_MIN_LED' value='${DEFAULT_MIN_LED}'>"
  [[ -n "${DEFAULT_MAX_LED}" ]] &&	echo "    <input type='hidden' name='DEFAULT_MAX_LED' value='${DEFAULT_MAX_LED}'>"
  [[ -n "${DEFAULT_MIN_AMBIENT}" ]] &&	echo "    <input type='hidden' name='DEFAULT_MIN_AMBIENT' value='${DEFAULT_MIN_AMBIENT}'>"
  [[ -n "${DEFAULT_MAX_AMBIENT}" ]] &&	echo "    <input type='hidden' name='DEFAULT_MAX_AMBIENT' value='${DEFAULT_MAX_AMBIENT}'>"

  [[ -n "${CLOCK_TYPE}" ]] &&		echo "    <input type='hidden' name='CLOCK_TYPE' value='${CLOCK_TYPE}'>"

  [[ -n "${browse}" ]] &&		echo "    <input type='hidden' name='browse' value='${browse}'>"
  [[ -n "${mode}" ]] &&			echo "    <input type='hidden' name='mode' value='${mode}'>"
  [[ -n "${MON}" ]] &&			echo "    <input type='hidden' name='MON' value='${MON}'>"
  [[ -n "${TUE}" ]] &&			echo "    <input type='hidden' name='TUE' value='${TUE}'>"
  [[ -n "${WED}" ]] &&			echo "    <input type='hidden' name='WED' value='${WED}'>"
  [[ -n "${THU}" ]] &&			echo "    <input type='hidden' name='THU' value='${THU}'>"
  [[ -n "${FRI}" ]] &&			echo "    <input type='hidden' name='FRI' value='${FRI}'>"
  [[ -n "${SAT}" ]] &&			echo "    <input type='hidden' name='SAT' value='${SAT}'>"
  [[ -n "${SUN}" ]] &&			echo "    <input type='hidden' name='SUN' value='${SUN}'>"
  [[ -n "${ONCE}" ]] && 		echo "    <input type='hidden' name='ONCE' value='${ONCE}'>"
  [[ -n "${SHUFFLE}" ]] && 		echo "    <input type='hidden' name='SHUFFLE' value='${SHUFFLE}'>"
  [[ -n "${STREAM}" ]] && 		echo "    <input type='hidden' name='STREAM' value='${STREAM}'>"

  [[ -n "${CURRENT_SHA1}" ]] &&		echo "    <input type='hidden' name='PREV_SHA1' value='${CURRENT_SHA1}'>"

  [[ -n "${NEW_HOSTNAME}" ]] &&		echo "    <input type='hidden' name='NEW_HOSTNAME' value='${NEW_HOSTNAME}'>"
}
####################################################################################################################
# Read_HTML_Parameters
#
# When starting a new script instance, the hidden variables (and new inputs) specified above are interpreted by the
# shell in this call to Read_HTML_Parameters. The variables CONTENT_LENGTH and CONTENT_TYPE will be set by apache
# when calling the cgi. These shell variables contain the POST data that the browser inserted into the URL

Read_HTML_Parameters()
{
  # make POST and GET stings available as bash variables
  if [[ ! -z "${CONTENT_LENGTH}" ]] && [[ "${CONTENT_LENGTH}" -gt 0 ]] && [[ "${CONTENT_TYPE}" != "multipart/form-data" ]] ; then
    read -n $CONTENT_LENGTH POST_STRING <&0
    eval `echo "${POST_STRING//;}"|tr '&' ';'`
  fi

  eval `echo "${QUERY_STRING//;}"|tr '&' ';'`
}
####################################################################################################################
# Percent_Decode
#
# When the browser inserts strings into the POST data, special characters (and spaces) are hex encoded. This
# one liner function converts these HTTP percent-encoded special characters back into real text.

Percent_Decode()
{
  : "${*//+/ }"; echo -e "${_//%/\\x}"
}
####################################################################################################################
# Function to verify if a file or stream contains audio, and return the mplayer command to play it
Check_Valid_Stream()
{
  MEDIA_CMD=""
  MERROR=""

  if [[ -z "${1}" ]] ; then
    MERROR="Empty stream string"
    return 1
  fi

  if [[ "${1}" == *[%^*+\;\|\\,\!\$\&\"\']* ]] ; then
    MERROR="Stream name contains commas or quotes or other special characters"
    return 1
  fi

  # Determine the file extension and then convert to uppercase
  EXT="${1##*.}"
  EXT="${EXT^^}"

  if [[ "${EXT}" == 'M3U' ]] || [[ "${EXT}" == 'M3U8' ]] || [[ "${EXT}" == 'PL' ]] || [[ "${EXT}" == 'PLS' ]] || [[ "${EXT}" == 'ASX' ]] || [[ "${EXT}" == 'PLA' ]] || [[ "${EXT}" == 'PLAY' ]] || [[ "${EXT}" == 'PLAYLIST' ]] || [[ "${EXT}" == 'TXT' ]] ; then
    # Its a playlist
    # is it a local file? see if we can access at least one of the media files in the playlist
    if [[ "${1:0:1}" == "/" ]] ; then
      FOUND=""

      while IFS= read -r LINE ; do
	EXT="${LINE##*.}"
	EXT="${EXT^^}"

	# Ensure this file has an extension that is a known media file extension
	[[ "${EXT}" != "MP3" ]] && [[ "${EXT}" != "M4A" ]] && [[ "${EXT}" != "MP4" ]] && [[ "${EXT}" != "AAC" ]] && [[ "${EXT}" != "FLAC" ]] && [[ "${EXT}" != "WAV" ]] && [[ "${EXT}" != "OGG" ]] && [[ "${EXT}" != "WMA" ]] && continue

	if [[ -f "${LINE}" ]] ; then
	  FOUND="Y"
	  break
	fi
      done < "${1}"

      if [[ -z "${FOUND}" ]] ; then
	MERROR="Unable to find valid media files in playlist '${1}'"
	return 1
      fi

    elif [[ "${1:0:3}" != "http" ]] ; then
      MERROR="Invalid URL: '${1}'"
      return 1
    fi

    MEDIA_CMD="${1}"

  elif [[ "${1:0:1}" == "/" ]] && [[ -f "${1}" ]] ; then
      # Ensure this file has an extension that is a known media file extension
      if [[ "${EXT}" != "MP3" ]] && [[ "${EXT}" != "M4A" ]] && [[ "${EXT}" != "MP4" ]] && [[ "${EXT}" != "AAC" ]] && [[ "${EXT}" != "FLAC" ]] && [[ "${EXT}" != "WAV" ]] && [[ "${EXT}" != "OGG" ]] && [[ "${EXT}" != "WMA" ]] ; then
	MERROR="Invalid media file: '${1}'"
	return 1
      fi

      MEDIA_CMD="${1}"

  elif [[ "${1^^}" == "RADIO" ]] ; then
    MEDIA_CMD="radio"

  else
    MERROR="'${1}' is not an absolute path to a local media file"
    return 1
  fi

  [[ "${MEDIA_CMD}" == */.m3u ]] && MEDIA_CMD="${MEDIA_CMD%.m3u}blank.m3u" # If the filename is just .m3u, then change to blank.m3u
}
####################################################################################################################
# Create_Playlist
# Function to generate a playlist file in M3U format
# $1: directory under which playlist will be created
# $2: path to temporary file which will contain the playlist
# $3: full path to where the temporary file should be moved
Create_Playlist()
{
  SORT_OPTION=''
  [[ "${SHUFFLE^}" == 'Y' ]] && SORT_OPTION='-R'

  find "${1}" -type f \
    \( \
    -iname "*.mp3" -o \
    -iname "*.m4a" -o \
    -iname "*.mp4" -o \
    -iname "*.aac" -o \
    -iname "*.flac" -o \
    -iname "*.wav" -o \
    -iname "*.ogg" -o \
    -iname "*.wma" \
    \) 2> /dev/null | sort ${SORT_OPTION} 2> /dev/null > "${2}"

  MEDIA_CMD=""

  # Verify that there is at least one file in TMP, and that it is a media file
  if [[ -s "${2}" ]] ; then
    Check_Valid_Stream $(head -n1 ${TMP})

    if [[ -n "${MERROR}" ]] ; then
      rm -f "${TMP}" 2> /dev/null
      EXPLANATION="${MERROR}"
      [[ -z "${EXPLANATION}" ]] && EXPLANATION="Either no media files found, or corrupt files found."
      MERROR="Problem generating playlist of audio files under ${1}"

    elif [[ -n "${3}" ]] ; then
      rm -f "${3}" 2> /dev/null
      mv "${2}" "${3}" 2> /dev/null
      chmod 664 "${3}" 2> /dev/null
      MEDIA_CMD="${3}"
      [[ "${MEDIA_CMD}" == */.m3u ]] && MEDIA_CMD="${MEDIA_CMD%.m3u}blank.m3u" # If the filename is just .m3u, then change to blank.m3u
    fi
  fi
}
####################################################################################################################
# Ingest_Media_Definition
#
# Function to determine if input is a valid media file, or media folder. If it is a valid folder, create a playlist.
Ingest_Media_Definition()
{
  MERROR=""
  PLAYLIST=""

  if [[ -n "${1}" ]] ; then
    NEW_REFERENCE="$(Percent_Decode "${1}")"
    MEDIA_CMD="${NEW_REFERENCE}"

    if [[ -f "${NEW_REFERENCE}" ]] ; then
      Check_Valid_Stream "${NEW_REFERENCE}"

      if [[ -n "${MERROR}" ]] ; then
	EXPLANATION="${MERROR}"
	MERROR="File must be a valid media file, with no spaces in filename or path."
	MEDIA_CMD=""
      fi

    elif [[ -d "${NEW_REFERENCE}" ]] ; then
      # Its a folder - get a list of all music and create a playlist
      mkdir -p "${PLAYLIST_DIR}" 2> /dev/null
      chmod 755 "${PLAYLIST_DIR}" 2> /dev/null
      PLAYLIST="${NEW_REFERENCE}.m3u" # The full pathname to the playlist file
      [[ "${PLAYLIST}" == */.m3u ]] && PLAYLIST="${PLAYLIST%.m3u}blank.m3u" # If the filename is just .m3u, then change to blank.m3u
      PLAYLIST="${PLAYLIST//\//_}" # Changes slashes in the path into underscores
      PLAYLIST=$(echo ${PLAYLIST:1} | sed 's/[^[:alnum:]._]//g') # Removes all other special characters + spaces + first character
      Create_Playlist "${NEW_REFERENCE}" "${TMP}" "${PLAYLIST_DIR}/${PLAYLIST}"
      [[ -f "${PLAYLIST_DIR}/${PLAYLIST}" ]] && NEW_REFERENCE="${PLAYLIST_DIR}/${PLAYLIST}"
    fi

  else
    MERROR="Cannot ingest media under '${1}'"
    EXPLANATION="Must be a valid media file or folder containing media files, with no spaces in filenames or path."
    MEDIA_CMD=""
    PLAYLIST=""
  fi
}
####################################################################################################################
# Process_Inputs
#
# This function orchestrates the percent-decode of any string variables that were passed by the browser, as well
# as converting any new inputs to the associate shell variables.

Process_Inputs()
{
  # Calculate the SHA1 checksum of the current alarm file
  [[ -z "${PREV_SHA1}" ]] && PREV_SHA1="${CURRENT_SHA1}"
  [[ -f "${ALARM_FILE}" ]] && CURRENT_SHA1=$(shasum "${ALARM_FILE}" | awk '{print $1}')

  # Percent decode shell variables that may contain special characters
  [[ -n "${DEFAULT_BROWSE_DIR}" ]] &&	DEFAULT_BROWSE_DIR="$(Percent_Decode "${DEFAULT_BROWSE_DIR}")"
  [[ -n "${NEW_HOSTNAME}" ]] &&		NEW_HOSTNAME="$(Percent_Decode "${NEW_HOSTNAME}")"
  [[ -n "${FALLBACK_ALARM_FILE}" ]] &&	FALLBACK_ALARM_FILE="$(Percent_Decode "${FALLBACK_ALARM_FILE}")"
  [[ -n "${DEFAULT_STREAM_OR_FILE}" ]] && DEFAULT_STREAM_OR_FILE="$(Percent_Decode "${DEFAULT_STREAM_OR_FILE}")"
  [[ -n "${STREAM}" ]] &&		STREAM="$(Percent_Decode "${STREAM}")"
  [[ -n "${LAUNCH_CMD}" ]] &&		LAUNCH_CMD="$(Percent_Decode "${LAUNCH_CMD}")"
  [[ -n "${BROWSE_DIR}" ]] &&		BROWSE_DIR="$(Percent_Decode "${BROWSE_DIR}")"
  [[ -n "${DEFAULT_TIME}" ]] &&		DEFAULT_TIME="$(Percent_Decode "${DEFAULT_TIME}")"
  [[ -n "${PATH_TO_PLAYLIST_FILE}" ]] && PATH_TO_PLAYLIST_FILE="$(Percent_Decode "${PATH_TO_PLAYLIST_FILE}")"
  #------------------------------------------------------
  # Read recently input params and overwrite the defaults (if possible)
  if [[ -n "${default_1224}" ]] ; then
    [[ ${default_1224} -eq 12 ]] && DEFAULT_1224=12
    [[ ${default_1224} -eq 24 ]] && DEFAULT_1224=24
  fi
  #------------------------------------------------------
  # Read recently input clock type
  if [[ -n "${clock_type}" ]] ; then
    [[ ${clock_type:0:1} == "A" ]] && CLOCK_TYPE="AClustered"
    [[ ${clock_type:0:1} == "S" ]] && CLOCK_TYPE="Standalone"
    [[ ${clock_type:0:1} == "C" ]] && CLOCK_TYPE="Clustered"
    [[ ${clock_type:0:1} == "Z" ]] && CLOCK_TYPE="ZClustered"
  fi

  if [[ ${CLOCK_TYPE:0:1} == "A" ]] ; then
    CLOCK_TYPE_STRING="Audible Clustered <small>(can control other clustered clocks, but cannot be controlled by other clustered clocks)</small>"

  elif [[ ${CLOCK_TYPE:0:1} == "C" ]] ; then
    CLOCK_TYPE_STRING="Audible Clustered <small>(can be controlled by other clustered clocks)</small>"

  elif [[ ${CLOCK_TYPE:0:1} == "S" ]] ; then
    CLOCK_TYPE_STRING="Stand Alone"

  elif [[ ${CLOCK_TYPE:0:1} == "Z" ]] ; then
    CLOCK_TYPE_STRING="Silent Clustered"

  else
    CLOCK_TYPE_STRING="Stand Alone"
  fi
  #------------------------------------------------------
  # Make string comparison non-case sensitive using double comma to convert to lowercase
  if [[ -n "${default_hour_zero}" ]] ; then
    DEFAULT_HOUR_ZERO=""
    [[ "${default_hour_zero,,}" == "nonblank" ]] && DEFAULT_HOUR_ZERO="nonblank"
    [[ "${default_hour_zero,,}" == "blank" ]] && DEFAULT_HOUR_ZERO="blank"
  fi

  if [[ -n "${default_snooze_press}" ]] ; then
    DEFAULT_SNOOZE_PRESS=""
    [[ "${default_snooze_press,,}" == "ignore" ]] && DEFAULT_SNOOZE_PRESS="ignore"
    [[ "${default_snooze_press,,}" == "restart" ]] && DEFAULT_SNOOZE_PRESS="restart"
  fi
  #------------------------------------------------------
  if [[ -n "${default_time}" ]] ; then
    # The time will be decoded into 24 hour format! It will look something like "15:47" and will be a 5 character string
    DECODED_TIME="$(Percent_Decode "${default_time}")"
    NEW_TIME="${DECODED_TIME}"

    # Remove leading zeros
    DN=(${NEW_TIME//:/ })
    DN0=$(echo ${DN[0]} | bc)
    DN1=$(echo ${DN[1]} | bc)
  fi

  DT=(${DEFAULT_TIME//:/ })
  DT0=$(echo ${DT[0]} | bc)
  DT1=$(echo ${DT[1]} | bc)

  if [[ -n "${NEW_TIME}" ]] && [[ $DN0 -ge 0 ]] && [[ $DN0 -le 23 ]] && [[ $DN1 -ge 0 ]] && [[ $DN1 -le 59 ]] ; then
    DEFAULT_TIME="${NEW_TIME}"

  elif [[ -z "${DEFAULT_TIME}" ]] || [[ $DT0 -lt 0 ]] || [[ $DT0 -gt 23 ]] || [[ $DT1 -lt 0 ]] || [[ $DT1 -gt 59 ]] ; then
    DEFAULT_TIME="07:00"
  fi
  #------------------------------------------------------
  if [[ -n "${default_alarm_duration}" ]] && [[ ${default_alarm_duration} -ge 1 ]] && [[ ${default_alarm_duration} -le 180 ]] ; then
    DEFAULT_ALARM_DURATION="${default_alarm_duration}"
  elif [[ -z "${DEFAULT_ALARM_DURATION}" ]] || [[ ${DEFAULT_ALARM_DURATION} -lt 1 ]] || [[ ${DEFAULT_ALARM_DURATION} -gt 180 ]] ; then
    DEFAULT_ALARM_DURATION="120"
  fi
  #------------------------------------------------------
  if [[ -n "${default_snooze_duration}" ]] && [[ ${default_snooze_duration} -ge 1 ]] && [[ ${default_snooze_duration} -le 60 ]] ; then
    DEFAULT_SNOOZE_DURATION="${default_snooze_duration}"
  elif [[ -z "${DEFAULT_SNOOZE_DURATION}" ]] || [[ ${DEFAULT_SNOOZE_DURATION} -lt 1 ]] || [[ ${DEFAULT_SNOOZE_DURATION} -gt 60 ]] ; then
    DEFAULT_SNOOZE_DURATION="10"
  fi
  #------------------------------------------------------
  if [[ -n "${default_media_duration}" ]] && [[ ${default_media_duration} -ge 1 ]] && [[ ${default_media_duration} -le 180 ]] ; then
    DEFAULT_MEDIA_DURATION="${default_media_duration}"
  elif [[ -z "${DEFAULT_MEDIA_DURATION}" ]] || [[ ${DEFAULT_MEDIA_DURATION} -lt 1 ]] || [[ ${DEFAULT_MEDIA_DURATION} -gt 180 ]] ; then
    DEFAULT_MEDIA_DURATION="60"
  fi
  #------------------------------------------------------
  if [[ -n "${volume}" ]] && [[ ${volume} -ge 0 ]] && [[ ${volume} -le 31 ]] ; then
    VOLUME="${volume}"
  elif [[ -z "${VOLUME}" ]] || [[ ${VOLUME} -lt 0 ]] || [[ ${VOLUME} -gt 31 ]] ; then
    VOLUME="12"
  fi

  if [[ -n "${min_volume}" ]] && [[ ${min_volume} -ge 0 ]] && [[ ${min_volume} -le 15 ]] ; then
    MIN_VOLUME="${min_volume}"
  elif [[ -z "${MIN_VOLUME}" ]] || [[ ${MIN_VOLUME} -lt 0 ]] || [[ ${MIN_VOLUME} -gt 15 ]] ; then
    MIN_VOLUME="0"
  fi
  #------------------------------------------------------
  if [[ -n "${init_volume_adjust}" ]] && [[ ${init_volume_adjust} -ge ${MIN_VOL_ADJ} ]] && [[ ${init_volume_adjust} -le ${MAX_VOL_ADJ} ]] ; then
    INIT_VOLUME_ADJUST="${init_volume_adjust}"
  elif [[ -z "${INIT_VOLUME_ADJUST}" ]] || [[ ${INIT_VOLUME_ADJUST} -lt ${MIN_VOL_ADJ} ]] || [[ ${INIT_VOLUME_ADJUST} -gt ${MAX_VOL_ADJ} ]] ; then
    INIT_VOLUME_ADJUST="0"
  fi

  if [[ -n "${targ_volume_adjust}" ]] && [[ ${targ_volume_adjust} -ge ${MIN_VOL_ADJ} ]] && [[ ${targ_volume_adjust} -le ${MAX_VOL_ADJ} ]] ; then
    TARG_VOLUME_ADJUST="${targ_volume_adjust}"
  elif [[ -z "${TARG_VOLUME_ADJUST}" ]] || [[ ${TARG_VOLUME_ADJUST} -lt ${MIN_VOL_ADJ} ]] || [[ ${TARG_VOLUME_ADJUST} -gt ${MAX_VOL_ADJ} ]] ; then
    TARG_VOLUME_ADJUST="0"
  fi
  #------------------------------------------------------
  # Check for new stream definitions
  if [[ -n "${fallback_alarm_file}" ]] ; then
    Ingest_Media_Definition "${fallback_alarm_file}"
    [[ -z "${MERROR}" ]] && FALLBACK_ALARM_FILE="${NEW_REFERENCE}"
  fi
  #------------------------------------------------------
  if [[ -n "${launch_cmd}" ]] ; then
    NEW_LAUNCH_CMD="$(Percent_Decode "${launch_cmd}")"
    [[ -n "${NEW_LAUNCH_CMD}" ]] && LAUNCH_CMD="${NEW_LAUNCH_CMD}"
  fi
  #------------------------------------------------------
  if [[ -n "${stream}" ]] ; then
    Ingest_Media_Definition "${stream}"
    [[ -z "${MERROR}" ]] && STREAM="${NEW_REFERENCE}"
  fi
  #------------------------------------------------------
  [[ -n "${default_min_led}" ]] && [[ ${default_min_led} -ge 0 ]] && [[ ${default_min_led} -le 255 ]] && DEFAULT_MIN_LED=${default_min_led}
  [[ -z "${DEFAULT_MIN_LED}" ]] || [[ ${DEFAULT_MIN_LED} -lt 0 ]] || [[ ${DEFAULT_MIN_LED} -gt 255 ]] && DEFAULT_MIN_LED=8

  [[ -n "${default_max_led}" ]] && [[ ${default_max_led} -ge 0 ]] && [[ ${default_max_led} -le 255 ]] && DEFAULT_MAX_LED=${default_max_led}
  [[ -z "${DEFAULT_MAX_LED}" ]] || [[ ${DEFAULT_MAX_LED} -lt 0 ]] || [[ ${DEFAULT_MAX_LED} -gt 255 ]] && DEFAULT_MAX_LED=255
  #------------------------------------------------------
  [[ -n "${default_min_ambient}" ]] && [[ ${default_min_ambient} -ge 0 ]] && [[ ${default_min_ambient} -le 4095 ]] && DEFAULT_MIN_AMBIENT=${default_min_ambient}
  [[ -z "${DEFAULT_MIN_AMBIENT}" ]] || [[ ${DEFAULT_MIN_AMBIENT} -lt 0 ]] || [[ ${DEFAULT_MIN_AMBIENT} -gt 4095 ]] && DEFAULT_MIN_AMBIENT=300

  [[ -n "${default_max_ambient}" ]] && [[ ${default_max_ambient} -ge 0 ]] && [[ ${default_max_ambient} -le 4095 ]] && DEFAULT_MAX_AMBIENT=${default_max_ambient}
  [[ -z "${DEFAULT_MAX_AMBIENT}" ]] || [[ ${DEFAULT_MAX_AMBIENT} -lt 0 ]] || [[ ${DEFAULT_MAX_AMBIENT} -gt 4095 ]] && DEFAULT_MAX_AMBIENT=1000

  # Ensure min is <= max
  [[ ${DEFAULT_MIN_LED} -gt ${DEFAULT_MAX_LED} ]] && DEFAULT_MIN_LED="${DEFAULT_MAX_LED}"
  [[ ${DEFAULT_MIN_AMBIENT} -gt ${DEFAULT_MAX_AMBIENT} ]] && DEFAULT_MIN_AMBIENT="${DEFAULT_MAX_AMBIENT}"
  #------------------------------------------------------
  # Ensure the browsing directories are really directories
  [[ ! -d "${DEFAULT_BROWSE_DIR}" ]] && DEFAULT_BROWSE_DIR="${MEDIA_DIR}"
  [[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${DEFAULT_BROWSE_DIR}"
}
####################################################################################################################
# Function to output html option choice
# $1 T means table, anything else means no table
# $2 name of bash variable
# $3 description string
# left $4 button contents string
# right $5 button contents string
Do_Choice()
{
  CHECKED=""
  NOT_CHECKED=""

  if [[ "${!2}" == 'Y' ]] ; then
    CHECKED="checked='Y'"
  else
    NOT_CHECKED="checked='Y'"
  fi

  [[ "$1" == 'T' ]] && echo '     <tr>'
  [[ "$1" == 'T' ]] && echo '      <td>'
  [[ "$1" == 'T' ]] && echo "       ${3}"
  [[ "$1" == 'T' ]] && echo '      </td>'
  [[ "$1" == 'T' ]] && echo '      <td>'
  echo "       <div id='sel'>"
  [[ "$1" != 'T' ]] && echo "       &emsp;${3}"
  echo "        <label><input type='radio' name='${2}' value='Y' ${CHECKED}><span>${4}</span></label>"
  echo "        <label><input type='radio' name='${2}' value='N' ${NOT_CHECKED}><span>${5}</span></label>"
  echo "       </div>"
  [[ "$1" == 'T' ]] && echo '      </td>'
  [[ "$1" == 'T' ]] && echo '     </tr>'
}
####################################################################################################################
# Function to output html slider and number box
# $1=variable name
# $2=string to describe the slider
# $3=minimum value
# $4=maximum value
# $5=current value
# $6='B' to set LED brightness on change. 'V' to set volume. 'D' to set duration.
#    'IVA' or 'TVA' to set volume adjust. If anything else inc not set, don't do anything
# $7=units (optional)
Do_Slider()
{
  echo -n "        <input type=range class=slider id='range_${1}' name='${1}' value='${5}' min='${3}' max='${4}' oninput='change_slider_${1}(this.value)'"

  # call slider_change.cgi to send the button command
  if [[ "${6}" == 'B' ]] || [[ "${6}" == 'V' ]] || [[ "${6}" == 'D' ]] ; then
    echo " onmouseup='action_${1}(this.value)'/>"
    SLIDER_FUNCTION="action_${1}"
  else
    echo "/>"
    SLIDER_FUNCTION="do_nothing"
  fi

  echo "        <label for='${1}'>${2}:</label>"

  echo -n "        <input type='number' id='${1}' min='${3}' max='${4}' value='${5}' onchange='change_number_${1}(this.value)'"


  if [[ "${6}" == 'B' ]] || [[ "${6}" == 'V' ]] || [[ "${6}" == 'D' ]] ; then
    echo -n " onmouseup='action_${1}(this.value)' onkeyup='action_${1}(this.value)' />"
  else
    echo -n "/>"
  fi

  [[ -n "${7}" ]] && echo -n " (${7})"
  echo "<br>"

  echo '        <script>'
  echo "         var input_${1} = document.getElementById('${1}');"
  echo "         input_${1}.addEventListener('keypress', function(event)"
  echo "         {"
  echo '           if (event.key === "Enter")'
  echo "           {"
  echo "             event.preventDefault();"
  echo "             document.getElementById('${SLIDER_FUNCTION}').click();"
  echo "           }"
  echo "         } );"

  echo "         function show_${1}(newValue)"
  echo '         {'
  echo "          document.getElementById('range_${1}').innerHTML= newValue;"
  echo "          input_${1}.innerHTML= newValue;"
  echo '         }'

  echo "         function change_number_${1}(val)"
  echo '         {'
  echo "          document.getElementById('range_${1}').value = isNaN(parseInt(val, 10)) ? 0 : parseInt(val, 10);"
  echo "          show_${1}(val);"
  echo '         }'

  echo "         function change_slider_${1}(val)"
  echo '         {'
  echo "          input_${1}.value = isNaN(parseInt(val, 10)) ? 0 : parseInt(val, 10);"
  echo "          show_${1}(val);"
  echo '         }'

  # The following calls to SLIDER_CHANGE are a kludge to enable a call to the alarm-clock programme upon mouseup
  if [[ "${6}" == 'B' ]] ; then
    echo "         function action_${1}(val)"
    echo '         {'
    echo '          var x = Math.round(1000 * val / 255)/1000;'
    echo "          fetch('${SLIDER_CHANGE}?bright=' + x);"
    echo '         }'
  fi

  if [[ "${6}" == 'V' ]] || [[ "${6}" == 'D' ]] ; then
    echo "         function action_${1}(val)"
    echo '         {'
    echo "          fetch('${SLIDER_CHANGE}?${1}=' + val);"
    #echo '          setTimeout("location.reload();", 1);'
    echo '         }'
  fi

  if [[ "${6}" == 'IVA' ]] || [[ "${6}" == 'TVA' ]] ; then
    echo "         function action_${1}(val)"
    echo '         {'
    echo "          fetch('${SLIDER_CHANGE}?${1}=' + val);"
    echo '         }'
  fi

  echo '        </script>'
}
####################################################################################################################
# Function to set up an input line and intercept pressing of the Enter button
# $1 name of the input variable
# $2 default value for the input variable
# $3 placeholder
# $4 size of the input buffer
# $5 optional extra text to add after the input box
# $6 optional text to add as a tool-tip

Do_Start_Body()
{
  echo ' <body>'
  echo '  <button id="do_nothing" hidden onclick="javascript:void(0)"></button>'
}

Do_Text_Input()
{
  DEFAULT_INPUT_VALUE=""
  [[ -n "${2}" ]] && DEFAULT_INPUT_VALUE="value='${2}'"

  DEFAULT_PLACEHOLDER_VALUE=""
  [[ -n "${3}" ]] && DEFAULT_PLACEHOLDER_VALUE="placeholder='${3}'"

  DEFAULT_SIZE_VALUE=""
  [[ -n "${4}" ]] && DEFAULT_SIZE_VALUE="size=${4}"

  echo "        <input type=text name='${1}' id='${1}' ${DEFAULT_INPUT_VALUE} ${DEFAULT_PLACEHOLDER_VALUE} ${DEFAULT_SIZE_VALUE}>${5}<br>"

  if [[ -n "${6}" ]] ; then
    echo "        <span class=text>${6}</span>"
  fi

  echo "        <script>"
  echo "         var input_${1} = document.getElementById('${1}');"
  echo "         input_${1}.addEventListener('keypress', function(event)"
  echo "         {"
  echo '           if (event.key === "Enter")'
  echo "           {"
  echo "             event.preventDefault();"
  echo '             document.getElementById("do_nothing").click();'
  echo "           }"
  echo "         } );"
  echo "        </script>"
}
####################################################################################################################
# Emit headers here, so style of the clock web pages is consistent
Emit_HTML_Headers()
{
  echo 'Content-type: text/html'
  echo
  echo '<html>'
  echo ' <head>'

  if [[ "${1}" == *"refresh"* ]] ; then
    echo '  <meta http-equiv="refresh" content="30">'
    REFRESH_ACTIVE="Y"
  else
    echo '  <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">'
    REFRESH_ACTIVE="N"
  fi

  echo "  <title>${HOSTNAME}Raspberry Pi Alarm Clock on '${HOSTNAME}'</title>"

  [[ "${1}" == *"icons"* ]] && echo "  <link rel='stylesheet' href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css'>"

  echo '  <style>'

  echo '   .background'
  echo '   {'
  echo '     background-color: #f7f7e1; /* yellowish */'
  echo '   }'

  echo '   .blue'
  echo '   {'
  echo '     border: none;'
  echo '     color: white;'
  echo '     padding: 10px 15px;'
  echo '     text-align: center;'
  echo '     text-decoration: none;'
  echo '     display: inline-block;'
  echo '     font-size: 16px;'
  echo '     margin: 4px 2px;'
  echo '     cursor: pointer;'
  echo '     background-color: #0000ff;'
  echo '   }'

  echo '   .red'
  echo '   {'
  echo '     border: none;'
  echo '     color: white;'
  echo '     padding: 10px 15px;'
  echo '     text-align: center;'
  echo '     text-decoration: none;'
  echo '     display: inline-block;'
  echo '     font-size: 16px;'
  echo '     margin: 4px 2px;'
  echo '     cursor: pointer;'
  echo '     background-color: #f44336;'
  echo '   }'

  echo '   .green'
  echo '   {'
  echo '     border: none;'
  echo '     color: white;'
  echo '     padding: 10px 15px;'
  echo '     text-align: center;'
  echo '     text-decoration: none;'
  echo '     display: inline-block;'
  echo '     font-size: 16px;'
  echo '     margin: 4px 2px;'
  echo '     cursor: pointer;'
  echo '     background-color: #228b22;'
  echo '   }'

  echo '   .linkBlue'
  echo '   {'
  echo '     display: inline;'
  echo '     padding: 0;'
  echo '     border: 0;'
  echo '     background-color: transparent;'
  echo '     font-size: 1em;'
  echo '     color: #0000ff;'
  echo '     text-decoration: underline;'
  echo '   }'

  echo '   .linkRed'
  echo '   {'
  echo '     display: inline;'
  echo '     padding: 0;'
  echo '     border: 0;'
  echo '     background-color: transparent;'
  echo '     font-size: 1em;'
  echo '     color: #f44336;'
  echo '     text-decoration: underline;'
  echo '   }'

  echo '   .linkGreen'
  echo '   {'
  echo '     display: inline;'
  echo '     padding: 0;'
  echo '     border: 0;'
  echo '     background-color: transparent;'
  echo '     font-size: 1em;'
  echo '     color: #228b22;'
  echo '     text-decoration: underline;'
  echo '   }'

  echo '   .centre'
  echo '   {'
  echo '     display: block;'
  echo '     margin-left: auto;'
  echo '     margin-right: auto;'
  echo '     margin-left: auto;'
  echo '   }'

  echo '   .tip'
  echo '   {'
  echo '     position: relative;'
  echo '     display: inline-block;'
  echo '   }'

  echo '   .tip .text'
  echo '   {'
  echo '     visibility: hidden;'
  echo '     width: 550px;'
  echo '     font-family: arial, sans-serif;'
  echo '     background-color: #f2e394;'
  echo '     color: #000;'
  echo '     text-align: center;'
  echo '     border-radius: 20px;'
  echo '     padding: 15px 15px 15px 15px;'
  echo '     position: absolute;'
  echo '     z-index: 1;'
  echo '   }'

  echo '   .tip:hover .text'
  echo '   {'
  echo '     visibility: visible;'
  echo '     transition-property:visibility;'
  echo '     transition-delay:2.5s;'
  echo '   }'

  echo '   .playlist'
  echo '   {'
  echo '     width: 100%;'
  echo '     height: 400;'
  echo '   }'

  echo '   .slider'
  echo '   {'
  echo '     width: 300px;'
  echo '   }'

  echo '   #sel label'
  echo '   {'
  echo '     float:left;'
  echo '     width:60px;'
  echo '     margin:3px;'
  echo '     border-radius:4px;'
  echo '     border:1px solid #D0D0D0;'
  echo '     overflow:auto;'
  echo '   }'

  echo '   #sel label span'
  echo '   {'
  echo '     text-align:center;'
  echo '     font-size: 12px;'
  echo '     font-family: arial, sans-serif;'
  echo '     padding:1px 0px;'
  echo '     display:block;'
  echo '   }'

  echo '   #sel label input'
  echo '   {'
  echo '     position:absolute;'
  echo '     top:-20px;'
  echo '   }'

  echo '   #sel input:checked + span'
  echo '   {'
  echo '     background-color:#00bfff;'
  echo '     color:#000000;'
  echo '   }'

  echo '   table'
  echo '   {'
  echo '     font-family: arial, sans-serif;'
  echo '     border-collapse: collapse;'
  echo '   }'

  if [[ "${1}" == *"noborder"* ]] ; then
    echo '   table.striped'
    echo '   {'
#    echo '     width: 100%;'
    echo '     background-color: #f8f8f0;'
    echo '   }'

    echo '   table.noborder tr, table.noborder td, table.noborder th'
    echo '   {'
    echo '     border: none;'
    echo '     padding: 3px 10px 3px 10px;' # Top Right Bottom Left
    echo '   }'
  fi

  echo '   table.striped td, table.striped th'
  echo '   {'
  echo '     border: 1px solid #000000; /* black */'
  echo '     text-align: left;'
  echo '     padding: 8px;'
  echo '   }'

  echo '   table.striped tr:nth-child(even)'
  echo '   {'
  echo '     background-color: #dddddd; /*grey */'
  echo '   }'

  echo '   fieldset'
  echo '   {'
  echo '     border: 3px solid #477;'
  echo '   }'


  echo '  </style>'
  echo ' </head>'
}
####################################################################################################################
Insert_Examples()
{
  echo "     <small>Click <a href='/cgi-bin/streaming_radio.cgi' target='_blank' rel='noopener noreferrer'>here</a> to open a tab with links to streaming search sites, or search the net for streaming URLs.</small><br><br>"
}
####################################################################################################################
# Display the current alarms in a tabular format
# In passing, we will remove any expired alarm entries after the table has been written (so they won't appear next time)
Emit_HTML_Alarm_Table()
{
  [[ -n "${1}" ]] && echo "   <form action='${THIS_SCRIPT}' method='get'>"
  echo '    <fieldset>'
  echo '     <legend><b>Current Alarm Settings</b></legend>'

  if [[ -r "${ALARM_FILE}" ]] && [[ -f "${ALARM_FILE}" ]] && [[ -s "${ALARM_FILE}" ]]; then
    echo "     Today is <b>$(date +%A)</b>. The time is <b>$(date +"%H:%M")</b>.<br><br>"
    [[ -n "${1}" ]] && echo "     <div class='tip'>"

    LINE_NUMBER=0
    NUM_NORM=0
    NUM_SUSP=0
    CONTAINS_ALARMS='N'

    echo "     <table class='striped'>"
    echo '      <tr>'
    echo '       <th>Time</th>'
    echo '       <th>Duration</th>'
    echo '       <th>Alarm Days</th>'
    echo '       <th>Alarm File or Stream</th>'
    echo '       <th>Vol Adj</th>'
    echo '      </tr>'

    # Set up for determining if any one-off alarm has already occurred
    NOW=$(date +%s)

    while IFS=, read -r ALM_TIME SUSPENDED ALM_DUR VOLADJ Mon Tue Wed Thu Fri Sat Sun Stream ; do
      LINE_NUMBER=$((LINE_NUMBER+1))

      # Ignore the first line in the file because it contains column headings
      if [[ ${LINE_NUMBER} -eq 1 ]] ; then
	# The following line must be the same as that in alarms.cgi
        # NOTE: The following line starts with a SPACE, so it sorts to be the first line. Don't remove the space!!!
	echo " Alarm Time,Suspended,Alarm Duration,Volume Adjust,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday,Stream" > "${TMP}"

      else
	# The format of the day strings will tell us if this is a one-off alarm or a recurring alarm
	# Test all days to see if we find a field which is not 'Y' or 'N'
	# One off alarms will have a number in ONE and ONLY ONE day field
	IsNum='N'
	InPast='Y'
	[[ "${SUSPENDED}" != "_SUSPENDED_" ]] && SUSPENDED="_ACTIVE_"

	if [[ "${Mon}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Mon} -gt ${NOW} ]] && InPast='N'
	fi

	if [[ "${Tue}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Tue} -gt ${NOW} ]] && InPast='N'
	fi

	if [[ "${Wed}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Wed} -gt ${NOW} ]] && InPast='N'
	fi

	if [[ "${Thu}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Thu} -gt ${NOW} ]] && InPast='N'
	fi

	if [[ "${Fri}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Fri} -gt ${NOW} ]] && InPast='N'
	fi

	if [[ "${Sat}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Sat} -gt ${NOW} ]] && InPast='N'
	fi

	if [[ "${Sun}" =~ ${IsNumber} ]] ; then
	  IsNum='Y'
	  [[ ${Sun} -gt ${NOW} ]] && InPast='N'
	fi

	# The volume string may be a single signed integer, or a range between two signed integers separated by dots
	if [[ "${VOLADJ}" == *"."* ]] ; then
	  IVOL="${VOLADJ%%.*}"
	  TVOL="${VOLADJ##*.}"

	  if ! [[ "${IVOL}" =~ ${IsNumber} ]] || [[ "${IVOL}" -lt ${MIN_VOL_ADJ} ]] || [[ "${IVOL}" -gt ${MAX_VOL_ADJ} ]] ; then
	    IVOL="0"
	  fi

	  if ! [[ "${TVOL}" =~ ${IsNumber} ]] || [[ "${TVOL}" -lt ${MIN_VOL_ADJ} ]] || [[ "${TVOL}" -gt ${MAX_VOL_ADJ} ]] ; then
	    TVOL="0"
	  fi

	else
	  # Just a single number, apparently. Can we read it?
	  if ! [[ "${VOLADJ}" =~ ${IsNumber} ]] || [[ "${VOLADJ}" -lt ${MIN_VOL_ADJ} ]] || [[ "${VOLADJ}" -gt ${MAX_VOL_ADJ} ]] ; then
	    IVOL="0"
	    TVOL="0"
	  else
	    IVOL="${VOLADJ}"
	    TVOL="${VOLADJ}"
	  fi
	fi

	echo '      <tr>'
	echo -n '       <td>'

	if [[ -n "${1}" ]] ; then
	  SELNAME="Sel${LINE_NUMBER}"
	  echo -n "<input type=\"checkbox\" name=\"${SELNAME}\" value='Y'>"
	fi

	echo -n "${ALM_TIME}"

	if [[ "${SUSPENDED}" == "_SUSPENDED_" ]] ; then
	  echo -n ' <b>Suspended</b>'
	  NUM_SUSP=$((NUM_SUSP+1))

	elif [[ "${IsNum}" == 'Y' ]] && [[ "${InPast}" == 'N' ]] ; then
	  echo -n ' <b>Once</b>'
	  NUM_NORM=$((NUM_NORM+1))

	elif [[ "${IsNum}" == 'Y' ]] && [[ "${InPast}" == 'Y' ]] ; then
	  echo -n ' Expired'

	else
	  NUM_NORM=$((NUM_NORM+1))
	fi

	echo '</td>'

	echo "       <td>${ALM_DUR} minutes</td>"
	echo '       <td>'

	# If the field 'Y', or if its a number after the NOW epoch, there's an active alarm on that day
	[[ "${Mon^}" == 'Y' ]] || [[ "${Mon}" =~ ${IsNumber} && ${Mon} -gt ${NOW} ]] && echo -n 'Mon '
	[[ "${Tue^}" == 'Y' ]] || [[ "${Tue}" =~ ${IsNumber} && ${Tue} -gt ${NOW} ]] && echo -n 'Tue '
	[[ "${Wed^}" == 'Y' ]] || [[ "${Wed}" =~ ${IsNumber} && ${Wed} -gt ${NOW} ]] && echo -n 'Wed '
	[[ "${Thu^}" == 'Y' ]] || [[ "${Thu}" =~ ${IsNumber} && ${Thu} -gt ${NOW} ]] && echo -n 'Thu '
	[[ "${Fri^}" == 'Y' ]] || [[ "${Fri}" =~ ${IsNumber} && ${Fri} -gt ${NOW} ]] && echo -n 'Fri '
	[[ "${Sat^}" == 'Y' ]] || [[ "${Sat}" =~ ${IsNumber} && ${Sat} -gt ${NOW} ]] && echo -n 'Sat '
	[[ "${Sun^}" == 'Y' ]] || [[ "${Sun}" =~ ${IsNumber} && ${Sun} -gt ${NOW} ]] && echo -n 'Sun'
	echo  '</td>'

	echo -n '       <td>'
	[[ "${Stream}" ]] && echo -n "${Stream}"
	echo  '</td>'

	# We will mess a little with the way we display this number. Display a dash for zero, and always display sign"
	echo -n '       <td>'

	if [[ "${IVOL}" == "${TVOL}" ]] ; then
	  if [[ ${IVOL} -eq 0 ]] ; then
	    echo '---</td>'
	  else
	    printf "%+d</td>\n" $IVOL
	  fi
	else
	  printf "%+d to %+d</td>\n" $IVOL $TVOL
	fi

	# Emit the edit alarm option
	if [[ -n "${1}" ]] ; then
	  echo -n '       <td>'
	  echo -n "<button type='submit' class='linkRed' name='alarm_copy' value='${LINE_NUMBER}'>Copy</button><br>"
	  echo  '</td>'
	fi

	echo '      </tr>'

	if [[ "${IsNum}" == 'N' ]] || [[ "${InPast}" == 'N' ]] ; then
	  if [[ "${IVOL}" == "${TVOL}" ]] ; then
	    echo "${ALM_TIME},${SUSPENDED},${ALM_DUR},${IVOL},${Mon},${Tue},${Wed},${Thu},${Fri},${Sat},${Sun},${Stream}" >> "${TMP}"
	  else
	    echo "${ALM_TIME},${SUSPENDED},${ALM_DUR},${IVOL}..${TVOL},${Mon},${Tue},${Wed},${Thu},${Fri},${Sat},${Sun},${Stream}" >> "${TMP}"
	  fi

	  CONTAINS_ALARMS='Y'
	fi
      fi
    done < "${ALARM_FILE}"

    echo '     </table>'

    if [[ -n "${1}" ]] && [[ ${NUM_NORM} -ge 1 || ${NUM_SUSP} -ge 1 ]] ; then
      echo "      <span class='text'>Select one or more tick-boxes and choose one of the buttons below to remove, suspend or reinstate an alarm.<br><br>Click on 'copy' to copy the associated alarm details into the 'create a new alarm' section below.</span>"
      echo "     </div>"
      echo "     <br>"

      Include_Defaults
      echo "     <div class='tip'>"
      echo "      <button type='submit' class='red' name='command' value='delete_alarm'>Delete Selected Alarms</button>"
      echo "      <span class='text'>Clicking here will remove all information about alarms that have their tick-boxes selected. To reinstate a deleted alarm, all information about the alarm will need to be recreated from scratch.</span>"
      echo '     </div>'

      if [[ ${NUM_NORM} -ge 1 ]] ; then
	echo "     <div class='tip'>"
	echo "      <button type='submit' class='blue' name='command' value='suspend_alarm'>Suspend Selected Alarms</button>"
	echo "      <span class='text'>Clicking here will suspend the alarms that have their tick-boxes selected. Suspended alarms will not function but will remain in the alarm list and can be easily reinstated.</span>"
	echo '     </div>'
      fi

      if [[ ${NUM_SUSP} -ge 1 ]] ; then
	echo "     <div class='tip'>"
	echo "      <button type='submit' class='green' name='command' value='reinstate_alarm'>Reinstate Selected Alarms</button>"
	echo "      <span class='text'>Clicking here will reinstate the suspended alarms that have their tick-boxes selected. The alarms will function once more.</span>"
	echo '     </div>'
      fi
    fi

    # the following dance is necessary because apache is configured to be unable to move files
    # so we cannot move the temp file to the alarm file, we need to overwrite it
    if [[ "${CONTAINS_ALARMS}" != 'N' ]] ; then
      cmp -s "${TMP}" "${ALARM_FILE}" 2> /dev/null
      [[ $? != 0 ]] && cat "${TMP}" > "${ALARM_FILE}"
    fi

    rm -f "${TMP}" 2> /dev/null

  else
    echo '     <p style="color:red;font-size:15px;">No alarms have been set!</p>'
  fi

  echo '    </fieldset>'
  [[ -n "${1}" ]] && echo '   </form>'
}
####################################################################################################################
# FileSystem browser
#
# browse is set to F=file, D=directory, B=Both

Include_FileSystem_Browser()
{
  if [[ -n "${browse}" ]] ; then
    # Read the directory path into array DIR_LEVELS
    cd "${BROWSE_DIR}"
    saveIFS=$IFS
    IFS='/'
    DIR_LEVELS=(${BROWSE_DIR})
    IFS=$saveIFS

    # Display links to all preceding directories up to the root
    LEN=${#DIR_LEVELS[@]}
    MSG=""

    # Output directory links in the current directory (to change into these directories and continue browsing)
    REF="${BROWSE_DIR}"
    [[ "${REF}" != "/" ]] && REF="${REF}/"

    for I in * ; do
      if [[ -d "${I}" ]] && [[ -x "${I}" ]] ; then
	if [[ -z "${MSG}" ]] ; then
	  MSG='Y'
	  echo '    <fieldset>'
	  echo "     <legend><b>Navigating '${BROWSE_DIR}'</b></legend>"
	  echo '     <p>Click on red links to ascend or descend the directory tree:</p>'
	fi

	echo "     <button type='submit' class='linkRed' name='BROWSE_DIR' value='${REF}${I}'>${REF}${I}</button><br>"
      fi
    done

    # Output directory links above the current folder, up to the root
    REF="${BROWSE_DIR}"

    for (( i=${LEN}-2 ; i >= 0 ; i-- )) ; do
      REF="${REF%"/${DIR_LEVELS[$i+1]}"}"
      [[ -z "${REF}" ]] && REF="/"

      if [[ -z "${MSG}" ]] ; then
	MSG='Y'
	echo '    <fieldset>'
	echo "     <legend><b>Navigating '${BROWSE_DIR}'</b></legend>"
	echo '     <p>Click on red links to ascend or descend the directory tree:</p>'
      fi

      echo "     <button type='submit' class='linkRed' name='BROWSE_DIR' value='${REF}'>${REF}</button><br>"
    done
    [[ "${MSG}" == 'Y' ]] && echo '    </fieldset>'

    if [[ "${browse}" == 'F' ]] || [[ "${browse}" == 'B' ]] ; then
      MSG=""
      REF="${BROWSE_DIR}"
      [[ "${REF}" != "/" ]] && REF="${REF}/"

      for I in * ; do
	if [[ -r "${I}" ]] && [[ -f "${I}" ]] && [[ -s "${I}" ]] ; then
	  if [[ -z "${MSG}" ]] ; then
	    MSG='Y'
	    echo '    <br>'
	    echo '    <fieldset>'
	    echo "     <legend><b>Files in '${BROWSE_DIR}'</b></legend>"
	    echo '     <p>Click on blue links to select that file:</p>'
	  fi

	  echo "     <button type='submit' class='linkBlue' name='stop_browsing' value='${REF}${I}'>${I}</button><br>"
	fi
      done

      [[ "${MSG}" == 'Y' ]] && echo '    </fieldset>'
    fi

    if [[ "${browse}" == 'D' ]] || [[ "${browse}" == 'B' ]] ; then
      MSG=""
      REF="${BROWSE_DIR}"
      [[ "${REF}" != "/" ]] && REF="${REF}/"

      for I in * ; do
	if [[ -r "${I}" ]] && [[ -d "${I}" ]] ; then
	  if [[ -z "${MSG}" ]] ; then
	    MSG='Y'
	    echo '    <br>'
	    echo '    <fieldset>'
	    echo "     <legend><b>Folders in '${BROWSE_DIR}'</b></legend>"
	    echo '     <p>Click on green links to select that folder:</p>'
	  fi

	  echo "     <button type='submit' class='linkGreen' name='stop_browsing' value='${REF}${I}'>${I}</button><br>"
	fi
      done

      [[ "${MSG}" == 'Y' ]] && echo '    </fieldset>'
    fi
  fi
}
####################################################################################################################
# Insert_Current_Playlist - inserts a section on the web page listing the current playlist contents
Insert_Current_Playlist()
{
  echo "    <fieldset>"
  echo '     <legend><b>Current Playlist and Bluetooth Device List</b></legend>'
  echo "     <div class='playlist'>"

  echo "      <iframe id='playlist' src='/cgi-bin/playlist.cgi' width='100%' height='100%' frameborder='0'>"
  echo '       <p>Your browser does not support iframes.</p>'
  echo '      </iframe>'
  echo '     </div>'
  echo '    </fieldset>'

  echo '    <script>'
  echo '     window.setInterval("reloadIFrame();", 30000);' # Refresh the playlist every 30 seconds

  echo '     function reloadIFrame()'
  echo '     {'
  echo "       var frameHolder=document.getElementById('playlist');"
  echo '       frameHolder.src="/cgi-bin/playlist.cgi"'
  echo '     }'

  echo '    </script>'
}
####################################################################################################################
Insert_Debug_Section()
{
  if [[ -n "${DEBUG}" ]] ; then
    # Get our own IP address
    HN=$(which hostname)
    IP_ADDRESS=$($HN -I)

    # See if we can find a valid IP address in the list of addresses that were returned
    IP=$(which ip)

    for i in $IP_ADDRESS none ; do
      $IP route get ${i} > /dev/null 2>&1
      [[ ${?} -eq 0 ]] && break
    done

    echo '   <div>'
    echo "    <textarea style='width:100%;height:80%;resize:none' disabled>"
    echo "POST_STRING='$POST_STRING'"
    echo "QUERY_STRING='$QUERY_STRING'"
    echo "HTTP_HOST='$HTTP_HOST'"
    echo "DOMAIN_NAME='$DOMAIN_NAME'"
    echo

    [[ -n "${command}" ]] &&		echo "command=${command}"
    [[ -n "${commence_browsing}" ]] &&	echo "commence_browsing=${commence_browsing}"
    [[ -n "${commence_fallback}" ]] &&	echo "commence_fallback=${commence_fallback}"
    [[ -n "${choose_default_folder}" ]] && echo "choose_default_folder=${choose_default_folder}"
    [[ -n "${stop_browsing}" ]] &&	echo "stop_browsing=${stop_browsing}"
    [[ -n "${reboot_confirm}" ]] &&	echo "reboot_confirm=${reboot_confirm}"
    [[ -n "${alarm_copy}" ]] &&		echo "alarm_copy=${alarm_copy}"
    [[ -n "${A0}" ]] &&			echo "A0=${A0}"
    [[ -n "${A1}" ]] &&			echo "A1=${A1}"
    [[ -n "${A2}" ]] &&			echo "A2=${A2}"
    [[ -n "${A3}" ]] &&			echo "A3=${A3}"
    [[ -n "${A4}" ]] &&			echo "A4=${A4}"
    [[ -n "${A5}" ]] &&			echo "A5=${A5}"
    echo

    [[ -n "${DEFAULT_1224}" ]] &&	echo "DEFAULT_1224: '${DEFAULT_1224}'"
    [[ -n "${DEFAULT_HOUR_ZERO}" ]] &&	echo "DEFAULT_HOUR_ZERO: '${DEFAULT_HOUR_ZERO}'"
    [[ -n "${DEFAULT_SNOOZE_PRESS}" ]] && echo "DEFAULT_SNOOZE_PRESS: '${DEFAULT_SNOOZE_PRESS}'"
    [[ -n "${DEFAULT_TIME}" ]] &&	echo "DEFAULT_TIME: '${DEFAULT_TIME}'"
    [[ -n "${DEFAULT_ALARM_DURATION}" ]] && echo "DEFAULT_ALARM_DURATION: '${DEFAULT_ALARM_DURATION}'"
    [[ -n "${DEFAULT_SNOOZE_DURATION}" ]] && echo "DEFAULT_SNOOZE_DURATION: '${DEFAULT_SNOOZE_DURATION}'"
    [[ -n "${DEFAULT_MEDIA_DURATION}" ]] && echo "DEFAULT_MEDIA_DURATION: '${DEFAULT_MEDIA_DURATION}'"
    [[ -n "${CURRENT_DURATION}" ]] &&	echo "CURRENT_DURATION: '${CURRENT_DURATION}'"
    [[ -n "${VOLUME}" ]] &&		echo "VOLUME: '${VOLUME}'"
    [[ -n "${CURRENT_VOLUME}" ]] &&	echo "CURRENT_VOLUME: '${CURRENT_VOLUME}'"
    [[ -n "${INIT_VOLUME_ADJUST}" ]] &&	echo "INIT VOLUME ADJUST: '${INIT_VOLUME_ADJUST}'"
    [[ -n "${TARG_VOLUME_ADJUST}" ]] &&	echo "TARGET VOLUME ADJUST: '${TARG_VOLUME_ADJUST}'"
    [[ -n "${FALLBACK_ALARM_FILE}" ]] && echo "FALLBACK_ALARM_FILE: '${FALLBACK_ALARM_FILE}'"
    [[ -n "${DEFAULT_STREAM_OR_FILE}" ]] && echo "DEFAULT_STREAM_OR_FILE: '${DEFAULT_STREAM_OR_FILE}'"
    [[ -n "${DEFAULT_BROWSE_DIR}" ]] &&	echo "DEFAULT_BROWSE_DIR: '${DEFAULT_BROWSE_DIR}'"
    [[ -n "${NEW_HOSTNAME}" ]] &&	echo "NEW_HOSTNAME: '${NEW_HOSTNAME}'"
    [[ -n "${LAUNCH_CMD}" ]] &&		echo "LAUNCH_CMD: '${LAUNCH_CMD}'"
    [[ -n "${SHUFFLE}" ]] && 		echo "SHUFFLE: '${SHUFFLE}'"
    [[ -n "${BROWSE_DIR}" ]] &&		echo "BROWSE_DIR: '${BROWSE_DIR}'"
    echo
    [[ -n "${default_1224}" ]] &&	echo "default_1224: ${default_1224}"
    [[ -n "${default_hour_zero}" ]] &&	echo "default_hour_zero=${default_hour_zero}"
    [[ -n "${default_snooze_press}" ]] && echo "default_snooze_press=${default_snooze_press}"
    [[ -n "${default_time}" ]] &&	echo "default_time=${default_time}"
    [[ -n "${default_alarm_duration}" ]] && echo "default_alarm_duration: ${default_alarm_duration}"
    [[ -n "${default_snooze_duration}" ]] && echo "default_snooze_duration: ${default_snooze_duration}"
    [[ -n "${volume}" ]] &&		echo "volume: ${volume}"
    [[ -n "${init_volume_adjust}" ]] &&	echo "init volume adjust: ${init_volume_adjust}"
    [[ -n "${targ_volume_adjust}" ]] &&	echo "target volume adjust: ${targ_volume_adjust}"

    [[ -n "${fallback_alarm_file}" ]] && echo "fallback_alarm_file: ${fallback_alarm_file}"
    [[ -n "${stream}" ]] &&		echo "stream: ${stream}"
    [[ -n "${launch_cmd}" ]] &&		echo "launch_cmd: ${launch_cmd}"
    [[ -n "${MEDIA_CMD}" ]] &&		echo "MEDIA_CMD: ${MEDIA_CMD}"
    [[ -n "${MERROR}" ]] &&	 	echo "MERROR: ${MERROR}"
    echo

    [[ -n "${default_min_led}" ]] &&	echo "default_min_led: ${default_min_led}"
    [[ -n "${default_max_led}" ]] &&	echo "default_max_led: ${default_max_led}"
    [[ -n "${default_min_ambient}" ]] && echo "default_min_ambient: ${default_min_ambient}"
    [[ -n "${default_max_ambient}" ]] && echo "default_max_ambient: ${default_max_ambient}"
    [[ -n "${browse}" ]] &&		echo "browse: ${browse}"
    [[ -n "${mode}" ]] &&		echo "mode: ${mode}"
    echo

    [[ -n "${MON}" ]] && 		echo "MON: ${MON}"
    [[ -n "${TUE}" ]] && 		echo "TUE: ${TUE}"
    [[ -n "${WED}" ]] && 		echo "WED: ${WED}"
    [[ -n "${THU}" ]] && 		echo "THU: ${THU}"
    [[ -n "${FRI}" ]] && 		echo "FRI: ${FRI}"
    [[ -n "${SAT}" ]] && 		echo "SAT: ${SAT}"
    [[ -n "${SUN}" ]] && 		echo "SUN: ${SUN}"
    [[ -n "${ONCE}" ]] && 		echo "ONCE: ${ONCE}"
    [[ -n "${STREAM}" ]] && 		echo "STREAM: ${STREAM}"
    [[ -n "${NEW_VALUE}" ]] &&		echo "NEW_VALUE: '${NEW_VALUE}'"
    [[ -n "${DEFAULT_MIN_LED}" ]] &&	echo "DEFAULT_MIN_LED: ${DEFAULT_MIN_LED}"
    [[ -n "${DEFAULT_MAX_LED}" ]] &&	echo "DEFAULT_MAX_LED: ${DEFAULT_MAX_LED}"
    [[ -n "${DEFAULT_MIN_AMBIENT}" ]] && echo "DEFAULT_MIN_AMBIENT: ${DEFAULT_MIN_AMBIENT}"
    [[ -n "${DEFAULT_MAX_AMBIENT}" ]] && echo "DEFAULT_MAX_AMBIENT: ${DEFAULT_MAX_AMBIENT}"
    [[ -n "${CLOCK_TYPE}" ]] &&		echo "CLOCK_TYPE: ${CLOCK_TYPE}"
    echo

    if [[ -z "$IP_ADDRESS" ]] || [[ "$i" == "none" ]] ; then
      echo '   Cannot determine local IP address<br>'

    else
      echo "IP_ADDRESS=${IP_ADDRESS}"
    fi

    echo '    </textarea>'
    echo '   </div>'
  fi
}
