#! /bin/bash
###############################################################################
#                           Digital Alarm Clock
#                Copyright (C) 2023 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Telegraph Station Emulator.  If not, see <http://www.gnu.org/licenses/>.
###############################################################################
# Before running this script, prepare a Raspberry Pi SDMMC card using the
# Raspberry Pi Imager programme. Choose either a 32 bit or 64 bit Raspberry Pi
# OS based on the target model you are going to use. The Digital Alarm Clock
# would normally run headless (ie no HDMI monitor attached) so choose the Lite
# version of the OS image. If for whatever reason you would like your alarm
# clock to have a GUI, then choose the Full version of the OS.
#
# All Raspbery Pi images will initialise the wired and wireless (if HW has
# wireless) networks upon boot and use DHCP to obtain an IP address. Ensure
# that your network has a working DHCP server! (All broadband home routers
# do this by default so its those people that have set up static addresses
# that I am talking to here! Remember to enable DHCP!).
#
# The first boot of the fresh image may take a minute or two. If you have
# chosen a headless (Lite) image, the only means you will have to determine
# it has booted and is ready will be to ping its IP address and wait for
# a response. Be patient!
#
# When the Pi starts to respond, you can SSH to the Alarm Clock and log in.
# Hopefully you copied  the tarball that contained this script onto the Pi's
# /boot partition before you booted it?
#
# To run this script:
# 1. Log into the pi via ssh (or keyboard/mouse/monitor for a non-headless Pi)
# 2. Issue command to extract the source code. Two versions of the command are
#    provided:
#       For the Bookworm release of the OS
#       tar zxf /boot/firmware/alarm-clock_vN.tgz (where N is the version of the software)
#
#       For the Bullseye release of the OS
#       tar zxf /boot/alarm-clock_vN.tgz (where N is the version of the software)
#
# 3. Finally run the installer:
#       sudo ./Install_Clock.sh
#
# Answer the questions, wait for the script, reboot and you're done!
# Have fun!
###############################################################################
check_if_root()
{
  # Make sure only root can run our script
  if [[ $EUID -ne 0 ]] ; then
    echo "This script must be run as root." 1>&2
    echo "For example, run the script as follows:" 1>&2
    echo "   sudo $0" 1>&2
    exit 1
  fi
}

installation_failed()
{
  echo
  echo "The command failed for some reason. Look carefully at the error above."
  echo
  echo "If this is the first time the problem has happened and it looks like a temporary problem,"
  echo "such as a download failure, try running the script again."
  echo
  echo "If the problem says that it 'could not get lock', this is also likely to be a temporary"
  echo "problem due to the automatic upgrade process. Wait a moment and re-run the installation"
  echo "script."
  exit 1
}
###############################################################################
# The main script sequence commences here
check_if_root

# Determine what version of OS we are running on
NUM_BITS=$(getconf LONG_BIT)
SUPPORT_64=$(lscpu | grep "CPU op-mode")
OS_RELEASE_FILE="/etc/os-release"

# The tr -d '\0' command is to strip off the null termination as bash unavoidably throws a warning otherwise
MODEL=$(tr -d '\0' < /proc/device-tree/model)

if [[ -f "${OS_RELEASE_FILE}" ]] ; then
  . "${OS_RELEASE_FILE}"
  echo "----------------------------------------------------------------------"
  echo "Running on Raspberry Pi ${VERSION_CODENAME^}: ${NUM_BITS} bits"
  echo

  if [[ "${VERSION_ID}" -lt 12 ]] && [[ "${NUM_BITS}" == "32" ]] ; then
    echo "This version of the OS contains the Y2038 clock bug."
    echo "Even with this bug, the clock will work fine until January 2038."
    echo "Therefore plan to upgrade the operating system before then!"

    if [[ "${SUPPORT_64}" == *"64"* ]] ; then
      echo "Your hardware supports a 64 bit operating system."
      echo "The 64 bit version of Bullseye Lite does not have the Y2038 bug and is recommended."
    fi

    echo
    read -p "Hit <return> to continue installation using this ${VERSION_CODENAME^} OS, or CTRL-C to abort: " DUMMY

  elif [[ "${VERSION_ID}" -lt 11 ]] ; then
    echo "This version of the OS is not supported by the clock."
    echo "Upgrade the OS to the Bookworm (the latest) or Bullseye (the previous) version."
    echo

    if [[ "${SUPPORT_64}" == *"64"* ]] ; then
      echo "Your hardware supports a 64 bit OS. Consider upgrading to Bookworm Lite 64 bit."
    else
      echo "Your hardware supports a 32 bit OS. Consider upgrading to Bookworm Lite 32 bit."
    fi

    echo
    read -p "Hit <return> to continue installation using this ${VERSION_CODENAME^} OS, or CTRL-C to abort: " DUMMY

  elif [[ ${MODEL} == *"Pi 2"* ]] || [[ ${MODEL} == *"Pi 3"* ]] || [[ ${MODEL} == *"Pi Zero W"* ]] ; then

    if [[ "${NUM_BITS}" == "64" ]] ; then
      echo "This is a 64 bit version of the OS, but the media player and other software modules"
      echo "will not work properly in this version. The installation will run, but you won't be"
      echo "happy! Consider downloading and installing a 32 bit version of the OS!!!!"

      echo
      read -p "Hit <return> to continue installation using this ${VERSION_CODENAME^} OS, or CTRL-C to abort: " DUMMY
    fi
  fi
fi
#------------------------------------------------------------------------------
# This section gets information about username and password
X=$(who am i)
USERNAME=${X%%[[:space:]]*}
CURRENT_USERNAME="${USERNAME}"

if [[ "${USERNAME}" == "pi" ]] ; then
  echo "----------------------------------------------------------------------"
  NEW_USER=""
  echo "You are currently logged in as the pi user."
  echo
  echo "Enter an alternate username at the prompt below if you would like to create"
  echo "an account with a different name. Otherwise, just press enter."
  echo
  read -p 'Alternate username: ' NEW_USER

  if [[ -n "${NEW_USER}" ]] ; then
    # egrep returns 0 if the string exists
    egrep "^${NEW_USER}" /etc/passwd > /dev/null 2>&1

    if [[ $? -eq 0 ]] ; then
      echo "${NEW_USER} already exists!"
      exit 1
    fi

    USERNAME="${NEW_USER}"
  else
    echo "You have chosen to continue installing using the 'pi' account."
  fi
fi

echo
echo "----------------------------------------------------------------------"
echo "Enter the password for '${USERNAME}'"

while true ; do
  echo
  read -sp 'Password: ' PASS1
  echo
  read -sp 'Enter the password again: ' PASS2
  echo
  [[ "${PASS1}" == "${PASS2}" ]] && break
  echo "The passwords do not match! Try again."
done

PASSWORD="${PASS1}"
CRYPT_PASSWORD=$(perl -e 'print crypt($ARGV[0], "password")' ${PASSWORD})

echo
echo "'${USERNAME}' will be used for the samba (windows network fileshare) username using the same password"
echo "----------------------------------------------------------------------"
echo

echo "Do you want to enable a firewall that blocks all connections originating from a foreign subnet?"

while true ; do
  read -p 'Y/y/N/n: ' FIREWALL
  ( [[ "${FIREWALL^}" ==  'Y' ]] || [[ "${FIREWALL^}" ==  'N' ]] ) && break
  echo "Y or N (y or n) only."
done

if [[ "${FIREWALL^}" ==  'Y' ]] ; then
  echo "(If you would like to disable the firewall later, rerun this installation script and answer 'N')"
else
  rm -f /usr/local/bin/iptables-firewall.sh > /dev/null 2>&1
fi

echo
echo "Do you want to *attempt* to disable the Pi's power and activity LEDs?"

while true ; do
  read -p 'Y/y/N/n: ' LEDS
  ( [[ "${LEDS^}" ==  'Y' ]] || [[ "${LEDS^}" ==  'N' ]] ) && break
  echo "Y or N (y or n) only."
done

if [[ "${LEDS^}" ==  'Y' ]] ; then
  echo "(If you would like to enable the LEDs later, rerun this installation script and answer 'N')"
fi

echo
echo "----------------------------------------------------------------------"
echo "As this installation script runs, there should be no errors."
echo "If you spot an error or the script stops, there's a problem."
echo
echo "An occasional problem is that a download of one of the installs will stall"
echo "and then fail. If this happens to you, the script will stop and tell"
echo "you, suggesting you rerun the script. Do exactly that. The stuff that"
echo "has already been downloaded will not be downloaded a second time."
echo
echo "Some unavoidable warnings will be preceded by comments that the warning"
echo "may be ignored. These are OK."
echo
echo "Some warnings and comments are displayed when new packages are installed and"
echo "updates are applied: These are OK. The script will stop if these fail."
echo

for i in {10..01} ; do
  echo -en "Clock installation Script commencing in $i seconds    \r"
  sleep 1
done

echo -en "                                           \r"
echo "----------------------------------------------------------------------"
echo "The system will now auto-upgrade, auto-install the required packages, and auto-configure itself."
echo "The process may take a while. It will return to the prompt when completed and ask you to reboot."
echo
echo
echo
#------------------------------------------------------------------------------
# This section downloads and installs all recent raspberry pi package and security updates
echo "----------------------------------------------------------------------"
# First - apply a kludge fix to work around bug in the new bookworm installer
# The bug is that when running apt-get update, bookworm is throwing the following warning:
# Key is stored in legacy trusted.gpg keyring (/etc/apt/trusted.gpg), see the DEPRECATION section in apt-key(8) for details
#
# The work around kind-of describes the bug if you think about it. You'd think they would have fixed this before release!
if [[ "${VERSION_CODENAME^^}" == "BOOKWORM" ]] ; then
  if [[ -f /etc/apt/trusted.gpg ]] && [[ ! -f /etc/apt/trusted.gpg.d/trusted.gpg ]] ; then
    cp /etc/apt/trusted.gpg /etc/apt/trusted.gpg.d/trusted.gpg
  fi
fi

apt-get update
[ $? != 0 ] && installation_failed

echo "----------------------------------------------------------------------"
apt-get -y upgrade
[ $? != 0 ] && installation_failed

echo "----------------------------------------------------------------------"
apt-get -y install --no-install-recommends \
	samba samba-common-bin dos2unix raspberrypi-kernel-headers pigpio pigpiod make apache2 chrony \
	mlocate pmount bc mpv libmpv-dev unattended-upgrades apt-listchanges ipset iptables ssl-cert \
	pulseaudio pulseaudio-module-bluetooth pipewire pipewire-media-session apt-config-auto-update

[ $? != 0 ] && installation_failed

echo
echo "----------------------------------------------------------------------"
echo "Expanding partition to use all of available SD card size"
raspi-config nonint do_expand_rootfs > /dev/null 2>&1
#------------------------------------------------------------------------------
# Configure some apache/http server defaults so CGI will run.
a2enmod cgi
a2dismod -f deflate
a2enmod rewrite
chmod 755 /usr/lib/cgi-bin
chown www-data:www-data /usr/lib/cgi-bin
#------------------------------------------------------------------------------
# This section creates a new user account if it was requested right at the beginning
if [[ "${USERNAME}" != "${CURRENT_USERNAME}" ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Creating user ${USERNAME}"

  useradd -m -p "${CRYPT_PASSWORD}" "${USERNAME}"

  if [[ $? -eq 0 ]] ; then
    echo "An account has been created for '${USERNAME}'"
  else
    echo
    echo "Failed to create account named '${USERNAME}'"
    exit 1
  fi
fi

# Double paranoid check - is the home account where we expect?
if [[ ! -d "/home/${USERNAME}" ]] ; then
  echo
  echo "Cannot find home directory for ${USERNAME}"
  exit 1
fi

# Make the directory writeable so that the web server can access the directory tree to find media
chmod 755 "/home/${USERNAME}"

# Now create a samba user with the same password (including a samba user for the pi account, if that's what was specified)
echo
echo
echo "--------------------------------------------------------------------------"
echo "Creating samba user and password for ${USERNAME}"

echo -e "${PASSWORD}\n${PASSWORD}" | smbpasswd -a -s "${USERNAME}"

# Add the required groups for the new user
usermod "${USERNAME}" -a -G pi,adm,dialout,cdrom,sudo,audio,video,plugdev,games,users,input,netdev,spi,i2c,gpio,bluetooth 2> /dev/null
usermod -a -G sudo "${USERNAME}"

echo
echo
echo "--------------------------------------------------------------------------"
echo "Allowing ${USERNAME} to run sudo without supplying a password"

# Allow the new user to run sudo without a password
chmod 666 /etc/sudoers.d/010_pi-nopasswd
echo "${USERNAME} ALL=(ALL) NOPASSWD: ALL" > /etc/sudoers.d/010_pi-nopasswd

# Make the following change so that the web interface allows reboot of the clock from the wifi settings menu
usermod -a -G sudo www-data
[ $? != 0 ] && installation_failed

echo "www-data ALL=(ALL) NOPASSWD: ALL" >> /etc/sudoers.d/010_pi-nopasswd

if [[ "${USERNAME}" != "${CURRENT_USERNAME}" ]] ; then
  usermod "${CURRENT_USERNAME}" -a -G pi,adm,dialout,cdrom,sudo,audio,video,plugdev,games,users,input,netdev,spi,i2c,gpio,bluetooth
  usermod -a -G sudo "${CURRENT_USERNAME}"
  echo "${CURRENT_USERNAME} ALL=(ALL) NOPASSWD: ALL" >> /etc/sudoers.d/010_pi-nopasswd
fi

chmod 440 /etc/sudoers.d/010_pi-nopasswd
#------------------------------------------------------------------------------
# This section enables colour prompts for the bash shell which include the directory path
# Note: There's a funny thing with the variable DOLLAR. If I just type the dollar sign, then
# the bash shell running this script will interpret it here, not write the dollar sign to
# the file!
DOLLAR="\$"

# egrep returns 0 if the string exists
egrep "NORMFMT" /etc/skel/.bashrc > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Modifying command line prompts"

  cat << EOF > tmp.txt
###################################################
esc="\033["
USER=${DOLLAR}(whoami)

if [ "${DOLLAR}{USER}" = 'root' ] ; then
    USERFMT="${DOLLAR}{esc}36;41m"
    DIRFMT="${DOLLAR}{esc}30;46m"
    NORMFMT="${DOLLAR}{esc}0m"
else
    USERFMT="${DOLLAR}{esc}37;44m"
    DIRFMT="${DOLLAR}{esc}30;46m"
    NORMFMT="${DOLLAR}{esc}0m"
fi

PS1="${DOLLAR}{USERFMT}${DOLLAR}{USER}:${DOLLAR}{DIRFMT}\w${DOLLAR}{NORMFMT}\n%"
PS2="${DOLLAR}{USERFMT}Continue>${DOLLAR}{NORMFMT} "

alias ll='ls -l --color=auto' 2>/dev/null
alias l.='ls -d .* --color=auto' 2>/dev/null
alias ls='ls --color=auto' 2>/dev/null
EOF

  cat tmp.txt >> /etc/skel/.bashrc
  cat tmp.txt >> /etc/bash.bashrc
  cat tmp.txt >> ${HOME}/.bashrc  # because we are sudo, $HOME is set to /root, not the logged in user
  [[ "${USERNAME}" == "${CURRENT_USERNAME}" ]] && cat tmp.txt >> /home/${USERNAME}/.bashrc
  rm -f tmp.txt
fi
#------------------------------------------------------------------------------
# This section modifies fstab so that users can write a new file into to the /boot partition
# (to make wifi settings changes)
# Default setting is that only root can write a file into the /boot partition
echo
echo
echo "--------------------------------------------------------------------------"
echo "Modifying /boot permissions in /etc/fstab"

sed -i 's/\(^.*boot[[:space:]]\+vfat[[:space:]]\+\)defaults/\1dmask=000,fmask=111/g' /etc/fstab
#------------------------------------------------------------------------------
# This section installs firewall rules that blocks incoming traffic that originates on a different subnet
# Note: if your subnet changes after installation, then these rules will block access!

if [[ "${FIREWALL^^}" ==  'Y' ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Creating firewall rules..."

  cat <<EOF > /usr/local/bin/iptables-firewall.sh
#!/bin/bash
#
# This script sets up an iptables firewall which allows incomming connections from the local network,
# but blocks incoming connections from the elsewhere (eg the Internet).
#
# iptables configuration is not user friendly. It is easier to set up rules using ufw (apt-get install ufw)
# and then display the iptables rules that it has created, and then carefully add those rules or modify those
# rules below.
#
# To display the currently configured rules, in iptables syntax, use the following command-line command
# iptables-save

# Clear and flush all existing IPv4 and IPv6iptables rules
iptables -F
iptables -X

# Determine the local subnet. If we cannot determine the subnet, then exit with no firewall rules applied
SUBNET=${DOLLAR}( ip route | awk 'NR==2{print ${DOLLAR}1}' )
[[ -z "${DOLLAR}{SUBNET}" ]] && exit

# Create a set comprising the local LAN
ipset create LocalLan hash:net
ipset add LocalLan ${DOLLAR}{SUBNET}

# DROP all incoming and outcoming network traffic - a general deny ANY rule
# Dropping network traffic will trick the source the host does not exist
# (as opposed to REJECT, which will send back a reject, but then they know we exist)
iptables -P INPUT DROP
iptables -P FORWARD DROP
iptables -P OUTPUT DROP

# traffic to and from the loopback interface is allowed, but only if the source is the loopback interface
iptables -A INPUT -i lo -j ACCEPT
iptables -A OUTPUT -o lo -j ACCEPT
iptables -A INPUT -s 127.0.0.0/8 ! -i lo -j DROP

# drop traffic which is INVALID or non-conforming packets such as packets with malformed headers
iptables -A INPUT -m conntrack --ctstate INVALID -j DROP

# Allow traffic that is established or related to an already established outgoing connection
iptables -A INPUT -m conntrack --ctstate RELATED,ESTABLISHED -j ACCEPT
iptables -A FORWARD -m conntrack --ctstate RELATED,ESTABLISHED -j ACCEPT
iptables -A OUTPUT -m conntrack --ctstate RELATED,ESTABLISHED -j ACCEPT

# Allow various ICMP messages including ping (type 8) errors (type 3) and TTL/traceroute (11 and 12)
iptables -A INPUT -p icmp -m icmp --icmp-type 3  -j ACCEPT
iptables -A INPUT -p icmp -m icmp --icmp-type 8 -j ACCEPT
iptables -A INPUT -p icmp -m icmp --icmp-type 11 -j ACCEPT
iptables -A INPUT -p icmp -m icmp --icmp-type 12 -j ACCEPT

# Allow DHCP
iptables -A INPUT -p udp -m udp --sport 67 --dport 68 -j ACCEPT

# Allow packets originating on this Raspberry Pi to be received on this Raspberry Pi
iptables -A INPUT -m addrtype --src-type LOCAL -j ACCEPT

# Allow packets originating on a local LAN subnet
iptables -A INPUT -m set --match-set LocalLan src -j ACCEPT

# Allow *all* multicast packets arriving at this Raspberry Pi
iptables -A INPUT -m pkttype --pkt-type multicast -j ACCEPT

# Allow *all* outbound traffic from the Pi
# (An alternate strategy is to allow all established outbound, and then permit individual protocols)
iptables -A OUTPUT -j ACCEPT

# Drop all other inbound traffic - these rules will catch traffic from a foreign subnet and drop it
iptables -A INPUT -j DROP
iptables -A FORWARD -j DROP

# I am going to assume that you don't have any IPv6 routes from the internet into your home network
# Otherwise, you should block those also
EOF

  chmod 755 /usr/local/bin/iptables-firewall.sh
fi
#------------------------------------------------------------------------------
# This section creates a startup script that modifies the permissions on
# /etc/wpa_supplicant/wpa_supplicant.conf to make it readable by the apache daemon.
# This opens a small security hole - configured wifi passwords will be visible. If this is a problem
# for you, disable (and the wifi configuration web page will not work)
echo
echo
echo "--------------------------------------------------------------------------"
echo "Modifying /etc/wpa_supplicant/wpa_supplicant.conf permissions"

cat <<EOF > /etc/rc.local
#!/bin/bash
#
# rc.local

# make the /etc/wpa_supplicant/wpa_supplicant.conf file accessible from the web server
chmod 644 /etc/wpa_supplicant/wpa_supplicant.conf
EOF

# If we are implementing a firewall, add the iptables commands here
[[ "${FIREWALL^^}" ==  'Y' ]] && echo "/usr/local/bin/iptables-firewall.sh" >> /etc/rc.local
#------------------------------------------------------------------------------
# This section modifies the apache systemd config file
# 32 bit version of the raspbian installer configure apache to write temporary
# files to a different directory. The 64 bit installer does not. This is bullshit!!!!
echo
echo
echo "--------------------------------------------------------------------------"
echo "Modifying /lib/systemd/system/apache2.service"

sed -i 's/^PrivateTmp/#PrivateTmp/g' /lib/systemd/system/apache2.service
sed -i 's/^PrivateTmp/#PrivateTmp/g' /lib/systemd/system/apache2@.service
#------------------------------------------------------------------------------
# This section installs configuration and scripts to automount and dismount USB devices
# when you plug them in and remove them. Take care removing them - ensure they are not still
# being written to! The scripts come from here https://github.com/avanc/mopidy-vintage/wiki/Automount-USB-sticks
# and here https://krystof.io/auto-mounting-a-usb-drive-in-a-raspberry-pi

if [[ ! -f /etc/udev/rules.d/usbstick.rules ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Creating an automounter for USB thumb drive devices"

  cat <<EOF > /etc/udev/rules.d/usbstick.rules
ACTION=="add", KERNEL=="sd[a-z][0-9]", TAG+="systemd", ENV{SYSTEMD_WANTS}="usbstick-handler@%k"
EOF

  cat <<EOF > /lib/systemd/system/usbstick-handler@.service
[Unit]
Description=Mount USB sticks
BindsTo=dev-%i.device
After=dev-%i.device

[Service]
Type=oneshot
RemainAfterExit=yes
ExecStart=/usr/local/bin/automount %I
ExecStop=/usr/bin/pumount /dev/%I
EOF

  cat <<EOF > /usr/local/bin/automount
#!/bin/bash

# Grab the partition name - this will become the mountpoint under /media
PART=${DOLLAR}{1}
FS_LABEL=${DOLLAR}(lsblk -o name,label | grep ${DOLLAR}{PART} | awk '{print ${DOLLAR}2}')

if [[ -z "${DOLLAR}{FS_LABEL}" ]] ; then
  /usr/bin/pmount --umask 000 --dmask 000 --noatime -w --sync /dev/${DOLLAR}{PART} /media/${DOLLAR}{PART}
else
  /usr/bin/pmount --umask 000 --dmask 000 --noatime -w --sync /dev/${DOLLAR}{PART} /media/${DOLLAR}{FS_LABEL}_${DOLLAR}{PART}
fi
EOF

  chmod 755 /usr/local/bin/automount
fi
#------------------------------------------------------------------------------
# This section configures the samba server so that the raspberry pi will appear on
# your local network as a fileshare, and you can then log in and write to it
# using the raspberry pi user you created.

# egrep returns 0 if the string exists
egrep "PiFileSystem" /etc/samba/smb.conf > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Enabling a samba network fileserver on this Pi"

  cat << EOF >> /etc/samba/smb.conf
###################################################
# The following enables access throughout the entire Pi filesystem
[PiFileSystem]
  path = /
  writeable = Yes
  read only = No
  printable = No
  public = Yes
  guest ok = Yes
  create mask = 0777 # This is ANDed with the existing permissions (for Files)
  force create mode = 0644 # This is ORed after mask applied (rw-r--r--) (for Files)
  directory mask = 0777 # This is ANDed with the existing permissions (for Dirs)
  force directory mode = 0755 # This is ORed after mask applied (rw-r--r--) (for Dirs)

# The following overwrites default read-only home directory setting
[homes]
  read only = No
  create mask = 0777 # This is ANDed with the existing permissions
  force create mode = 0644 # This is ORed after mask applied (rw-r--r--)
  directory mask = 0777 # This is ANDed with the existing permissions (for Dirs)
  force directory mode = 0755 # This is ORed after mask applied (rw-r--r--) (for Dirs)

# The following overwrites the default log file size setting
[global]
  max log size = 10
EOF

  # Make the media directory writeable so you can add media and have RW access
  chmod 777 /media
  systemctl enable smbd
  systemctl restart smbd
  echo "Samba server accessible at '$(hostname).local' or '$(hostname)' (depending on your OS)"
  echo "You can log into the samba server using the account '${USERNAME}' and the passowrd you specified before."
fi
#------------------------------------------------------------------------------
# This section enables the raspberry pi built-in watchdog timer and turns off the
# bright power and activity LEDs
#
# http://0pointer.de/blog/projects/watchdog.html
# https://diode.io/raspberry%20pi/running-forever-with-the-raspberry-pi-hardware-watchdog-20202/
# https://www.domoticz.com/wiki/Setting_up_the_raspberry_pi_watchdog

# egrep returns 0 if the string exists
egrep "RebootWatchdogSec=5min" /etc/systemd/system.conf > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Enabling the hardware watchdog timer"
  sed -i 's/^.*RuntimeWatchdogSec=.*$/RuntimeWatchdogSec=10s/g' /etc/systemd/system.conf
  sed -i 's/^.*RebootWatchdogSec=.*$/RebootWatchdogSec=5min/g' /etc/systemd/system.conf
  sed -i 's/^.*DefaultTimeoutStopSec=.*$/DefaultTimeoutStopSec=30s/g' /etc/systemd/system.conf
fi

CONFIG_FILE="/boot/firmware/config.txt"
[[ ! -f "${CONFIG_FILE}" ]] && CONFIG_FILE="/boot/config.txt"

CONFIG_BACKUP="/home/${USERNAME}/config.txt.orig"
[[ ! -f "${CONFIG_BACKUP}" ]] && cp "${CONFIG_FILE}" "${CONFIG_BACKUP}"

# Apparently this is only required for Pi2 and older
# egrep returns 0 if the string exists
egrep "Enable hardware watchdog" "${CONFIG_FILE}" > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Enabling HW watchdog in ${CONFIG_FILE}"

  cat << EOF >> "${CONFIG_FILE}"

# Enable hardware watchdog
dtparam=watchdog=on
EOF

  # Disable stuff that will interfere with the clock - yes disabled is 1, not 0 (WTF?)
  raspi-config nonint do_camera 1
  raspi-config nonint do_spi 1
  raspi-config nonint do_i2c 1
  raspi-config nonint do_onewire 1

  if [[ "${VERSION_ID}" -lt 12 ]] ; then
      #raspi-config has changed under bookworm. The old do_serial command doesn't work non-interactively now
    raspi-config nonint do_serial 1
  else
    # hardware and console are disabled separately under bookworm
    raspi-config nonint do_serial_hw 1
    raspi-config nonint do_serial_cons 1
  fi
fi

if [[ "${LEDS^^}" ==  'Y' ]] ; then
  # egrep returns 0 if the string exists
  egrep "ALM_LED" "${CONFIG_FILE}" > /dev/null 2>&1

  if [[ $? -ne 0 ]] ; then
    echo
    echo
    echo "--------------------------------------------------------------------------"
    echo "Attempting to disable the power and activity LEDs in ${CONFIG_FILE}"
    # Bother - there is a slightly different LED disable command required for each Pi model (sigh)
    # And even greater bother, Raspberry Pi firmware tree had bugs at some stages, so the following
    # may not work for you. If the not working model is NOT a pi4, then try the config for the Pi4
    #
    # https://lindevs.com/turn-off-built-in-leds-on-raspberry-pi
    # https://github.com/seamusdemora/PiFormulae/blob/master/LEDControlForRPi.md
    #
    # To discover the devices that can be triggered
    #  sudo find /sys/devices -name '*trigger*'
    # ...then cat those files to see the possible values they can take on. The currently selected
    # one is in brackets
    if [[ ${MODEL} == *"Pi 3 Model B Plus"* ]] || [[ ${MODEL} == *"Pi 4"* ]] ; then

	cat << EOF >> "${CONFIG_FILE}"
# ALM_LED Turn off ACT LED (the green one next to PWR that flashes with SDMMC access)
dtparam=act_led_trigger=none # ALM_LED
dtparam=act_led_activelow=off # ALM_LED
# ALM_LED Turn off PWR LED (the bright red one that's on all the time)
dtparam=pwr_led_trigger=default-on # ALM_LED
dtparam=pwr_led_activelow=off # ALM_LED
EOF

    elif [[ ${MODEL} == *"Pi 2"* ]] || [[ ${MODEL} == *"Pi 3"* ]] ; then

	cat << EOF >> "${CONFIG_FILE}"
# ALM_LED Turn off ACT LED (the green one next to PWR that flashes with SDMMC access)
dtparam=act_led_trigger=none # ALM_LED
dtparam=act_led_activelow=off # ALM_LED
# ALM_LED Turn off PWR LED (the bright red one that's on all the time)
dtparam=pwr_led_trigger=none # ALM_LED
dtparam=pwr_led_activelow=off # ALM_LED
EOF

    elif [[ ${MODEL} == *"Pi 5"* ]] ; then
      # Ignore the new Pi 5: It was announced just before article went to press
      # and hardware is not yet available.
      echo "Cannot yet disable LEDs on Pi5. Hardware was not available at time of publication."
      echo "If you would like to help, please work out the required changes to ${CONFIG_FILE}"
      echo "and let me know so I can include these in the script."
      echo
      echo "**********************************************************************************"
      echo "IMPORTANT: At the time of going to press, the pigpiod library does not yet support"
      echo "the Pi5. The Pi5 has different GPIO hardware than earlier Pis and pigpiod will"
      echo "certainly require changes to support the Pi5."
      echo "Refer to https://github.com/joan2937/pigpio/issues/586"
      echo "Refer to https://github.com/joan2937/pigpio/issues"
      echo "**********************************************************************************"
      read -p 'Hit <return> to continue: ' DUMMY

    elif [[ ${MODEL} == *"Pi Zero 2"* ]] ; then

	cat << EOF >> "${CONFIG_FILE}"
# ALM_LED Turn off ACT LED (the green one)
dtparam=act_led_trigger=none # ALM_LED
dtparam=act_led_activelow=on # ALM_LED
EOF

    elif [[ ${MODEL} == *"Pi Zero W"* ]] ; then

	cat << EOF >> "${CONFIG_FILE}"
# ALM_LED Turn off ACT LED (the green one)
dtparam=act_led_trigger=none # ALM_LED
EOF

    elif [[ ${MODEL} == *"Pi Model"* ]] ; then
      # Ignore these early models - there is nothing we can do
      echo "Cannot disable LEDs on early model Pis"

    else
      echo "This Pi model was unknown when this script was written. Not disabling LEDs."
      echo "We recommend you search the Internet for a solution, or look for an update of this software."

    fi
  fi

else
  # The human has asked for the LEDs not to be enabled
  # egrep returns 0 if the string exists
  egrep "ALM_LED" "${CONFIG_FILE}" > /dev/null 2>&1

  if [[ $? -eq 0 ]] ; then
    echo
    echo
    echo "--------------------------------------------------------------------------"
    echo "Attempting to re-enable the power and activity LEDs in ${CONFIG_FILE}"

    # Remove lines that contain ALM_LED
    sed -i '/ALM_LED/d' "${CONFIG_FILE}"
  fi
fi

egrep "audio_pwm_mode" "${CONFIG_FILE}" > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Setting highest quality audio in ${CONFIG_FILE}"

  cat << EOF >> "${CONFIG_FILE}"
# Set highest audio quality
audio_pwm_mode=2
disable_audio_dither=1
EOF
fi
#------------------------------------------------------------------------------
# This section sets up the unattended upgrade process and sets the schedule
#
# It can be tested using
#    sudo unattended-upgrades --dry-run --debug -v
UPDATE_FILE="/etc/apt/apt.conf.d/50unattended-upgrades"
UPDATE_FILE_BACKUP="/etc/apt/apt.conf.d/50unattended-upgrades.orig"
UPDATE_MOD_MARKER="# changes for alarm-clock"

[[ ! -f "${UPDATE_FILE_BACKUP}" ]] && cp "${UPDATE_FILE}" "${UPDATE_FILE_BACKUP}"

# egrep returns 0 if the string exists
egrep "${UPDATE_MOD_MARKER}" "${UPDATE_FILE}" > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Configuring automatic security and timezone / daylight savings information updates"

  # Set the time at which the automatic upgrade takes place
  mkdir -p /etc/systemd/system/apt-daily-upgrade.timer.d

  cat << EOF > /etc/systemd/system/apt-daily-upgrade.timer.d/override.conf
[Timer]
OnCalendar=
OnCalendar=03:15
RandomizedDelaySec=0
EOF

  cat << EOF > /etc/apt/apt.conf
// This file comes from /usr/share/doc/apt/examples/apt.conf

APT
{
  // Options for apt-get
  Get
  {
     Assume-Yes "true";
     // Force-Yes "true";
     Download-Only "false";
     Fix-Broken "true";
  };

  Ignore-Hold "true";
};
EOF

  # To customise the following, run this command on the Pi to learn what origins are currently being used
  #      sudo apt-cache policy | grep 'o=[^,]*.a=[^,]*'
  # The output of that command shows 'o=***' (origin) and 'a=***' (archive)
  sed -i 's/^.*::Automatic-Reboot[[:space:]].*$/Unattended-Upgrade::Automatic-Reboot "true";/g' "${UPDATE_FILE}"
  sed -i 's/^.*::Automatic-Reboot-Time[[:space:]].*$/Unattended-Upgrade::Automatic-Reboot-Time "03:30";/g' "${UPDATE_FILE}"
  sed -i 's/^.*::Remove-Unused-Kernel-Packages[[:space:]].*/Unattended-Upgrade::Remove-Unused-Kernel-Packages "true";/g' "${UPDATE_FILE}"
  sed -i 's/^.*::Remove-Unused-Dependencies[[:space:]].*/Unattended-Upgrade::Remove-Unused-Dependencies "true";/g' "${UPDATE_FILE}"

  sed -i 's/^.*origin=Debian,codename=${distro_codename}-updates.*$/        "origin=Debian,codename=${distro_codename}-updates";/g' "${UPDATE_FILE}"
  sed -i 's/^.*origin=Debian,codename=${distro_codename}-proposed-updates.*$/        "origin=Raspbian,codename=${distro_codename},label=Raspbian";/g' "${UPDATE_FILE}"

  sed -i '/origin=Raspbian,codename=${distro_codename},label=Raspbian/a  XXXXXXXXXX/g' "${UPDATE_FILE}"
  sed -i 's/^.*XXXXXXXXXX.*$/        "origin=Raspberry Pi Foundation,codename=${distro_codename},label=Raspberry Pi Foundation";/g' "${UPDATE_FILE}"

  # The current release of Debian changes the 'Suite value' from 'stable' to 'oldstable'
  sed -i 's/^.*"o=Debian,a=stable-updates".*/        "o=Debian,a=oldstable-updates";/g' "${UPDATE_FILE}"
  sed -i 's/^.*"o=Debian,a=proposed-updates".*/        "o=Debian,a=oldstable-security";/g' "${UPDATE_FILE}"
  sed -i 's/^.*"o=Debian,a=stable".*/        "o=Debian,a=oldstable";/g' "${UPDATE_FILE}"

  echo unattended-upgrades unattended-upgrades/enable_auto_updates boolean true | debconf-set-selections
  [ $? != 0 ] && installation_failed

  # Set to auto-update once every three days (the default is daily)
  sed -i 's/^.*::Unattended-Upgrade[[:space:]].*$/APT::Periodic::Unattended-Upgrade "3";/g' /etc/apt/apt.conf.d/20auto-upgrades
  sed -i 's/^.*::Update-Package-Lists[[:space:]].*$/APT::Periodic::Update-Package-Lists "3";/g' /etc/apt/apt.conf.d/20auto-upgrades

  echo >> "${UPDATE_FILE}"
  echo "${UPDATE_MOD_MARKER}" >> "${UPDATE_FILE}"
fi
#------------------------------------------------------------------------------
# This section modifies the logrotate config to rotate logs daily
echo
echo
echo "--------------------------------------------------------------------------"
echo "Modifying /etc/logrotate.conf"

sed -i 's/weekly/daily/g' /etc/logrotate.conf
#------------------------------------------------------------------------------
# This section sets some raspberry pi configurations so that log files don't bloat out too big
# and fill the flash card

# egrep returns 0 if the string exists
egrep "RuntimeMaxUse=20M" /etc/systemd/journald.conf > /dev/null 2>&1

if [[ $? -ne 0 ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Reducing the size of system logfiles"

  echo "RuntimeMaxUse=20M" >> /etc/systemd/journald.conf
  echo "SystemMaxUse=50M" >> /etc/systemd/journald.conf
  echo "MAILTO=\"\"" >> /etc/crontab
  echo ACTION==\"add\", KERNEL==\"gpio\", MODE=\"0666\" > /etc/udev/rules.d/local.rules
fi
#------------------------------------------------------------------------------
# This section modifies bluetooth configuration to allow the Pi to be permanently discoverable
# Default setting is discoverable and pairable for three minutes only. The idle timeoout is
# generally not enabled, but we will set to three hours (180 minutes)
#
# Some of the detail came from here: https://mathieu-requillart.medium.com/my-ultimate-guide-to-the-raspberry-pi-audio-server-i-wanted-bluetooth-64c347ee0d22

echo
echo
echo "--------------------------------------------------------------------------"
echo "Modifying bluetooth configuration"

if [[ -z "$(grep "Class = 0x200414" /etc/bluetooth/main.conf)" ]] ; then
  # The following two timeouts are expressed in seconds
  sed -i 's/^#DiscoverableTimeout.*/DiscoverableTimeout = 180/g'  /etc/bluetooth/main.conf
  sed -i 's/^#PairableTimeout.*/PairableTimeout = 180/g'  /etc/bluetooth/main.conf
  sed -i 's/^#FastConnectable.*/FastConnectable = true/g'  /etc/bluetooth/main.conf
  sed -i 's/^#JustWorksRepairing.*/JustWorksRepairing = always/g'  /etc/bluetooth/main.conf
  sed -i 's/^#Cache.*/Cache = yes/g'  /etc/bluetooth/main.conf
  sed -i 's/^#Class.*/Class = 0x200414 #Present as an audio loudspeaker/g'  /etc/bluetooth/main.conf
  sed -i 's/^#ReconnectAttempts.*/ReconnectAttempts = 7/g'  /etc/bluetooth/main.conf
  sed -i 's/^#ReconnectIntervals.*/ReconnectIntervals=1,2,4,8,16,32,64/g'  /etc/bluetooth/main.conf

  # The following timeout is expressed in minutes
  sed -i 's/^#IdleTimeout.*/IdleTimeout=180/g'  /etc/bluetooth/input.conf

  # Modify the /etc/systemd/system/bluetooth.target.wants/bluetooth.service
  sed -i '/ExecStart=./ s/$/ --noplugin=sap/' /lib/systemd/system/bluetooth.service
  sed -i 's/^.*Restart=.*/Restart=always/g' /lib/systemd/system/bluetooth.service
fi

# Pulse audio is used for Bullseye, but not Bookworm
PULSEAUDIO_SERVICE="/lib/systemd/system/pulseaudio.service"
SET_CLOCK_AUDIO="/usr/local/bin/set-clock-audio.sh"
SET_CLOCK_AUDIO_SERVICE="/lib/systemd/system/set-clock-audio.service"

if [[ ! -f "${PULSEAUDIO_SERVICE}" ]] ; then
  cat << EOF > "${PULSEAUDIO_SERVICE}"
[Unit]
Description=Pulseaudio sound server
After=avahi-daemon.service network.target

[Service]
ExecStart=/usr/bin/pulseaudio --system --disallow-exit --disable-shm --daemonize=no
ExecReload=/bin/kill -HUP ${DOLLAR}MAINPID

[Install]
WantedBy=multi-user.target
EOF

  cat << EOF >> /etc/pulse/system.pa

.ifexists module-bluetooth-policy.so
load-module module-bluetooth-policy
.endif

.ifexists module-bluetooth-discover.so
load-module module-bluetooth-discover
.endif
EOF

  usermod root -a -G pulse,pulse-access,audio 2> /dev/null
  systemctl enable --now pulseaudio.service
  ln -s "${PULSEAUDIO_SERVICE}" /etc/systemd/system/pulseaudio.service
fi

if [[ ! -f "${SET_CLOCK_AUDIO}" ]] ; then
  cat <<EOF > "${SET_CLOCK_AUDIO}"
#!/bin/bash
#
# This script sets up the alarm-clock audio output for your specific situation.
# The script is run when the clock boots, and also when the alarm-clock programme
# runs. This paranoid repetitive behaviour is to work around bugs in the original
# (maybe early) release of Bookworm.

# You can edit this file to suit your specific needs.
# e.g. add command to set the default output device here if you're using an audio
# USB dongle rather than the analogue audio output (or HDMI audio on PiZero2W)
# /usr/bin/pactl set-default-sink "device-name"

# Set the default volume level to 86 percent max
/usr/bin/pactl set-sink-volume "@DEFAULT_SINK@" "86%"
EOF

  chmod 755 "${SET_CLOCK_AUDIO}"
fi

if [[ ! -f "${SET_CLOCK_AUDIO_SERVICE}" ]] ; then
  cat << EOF > "${SET_CLOCK_AUDIO_SERVICE}"
[Unit]
Description=(Re)set system audio for the alarm-clock
After=runlevel3.target runlevel4.target runlevel5.target

[Service]
User=${USERNAME}
ExecStart=${SET_CLOCK_AUDIO}
Type=oneshot
RemainAfterExit=true

[Install]
WantedBy=multi-user.target
EOF

  usermod "${USERNAME}" -a -G pulse,pulse-access,audio 2> /dev/null
  systemctl enable set-clock-audio.service
  ln -s "${SET_CLOCK_AUDIO_SERVICE}" /etc/systemd/system/set-clock-audio.service
fi
#------------------------------------------------------------------------------
# This section installs the clock software that (hopefully) is sitting in a tarball in the boot partition
echo
echo
echo "--------------------------------------------------------------------------"
echo "Enabling chrony time service"
cat << EOF > /etc/chrony/sources.d/Australia.sources
# Try to use a local gateway NTP server (if it exists)
server _gateway iburst

# Use public servers from the pool.ntp.org project.
# pool.ntp.org should return IP addresses for servers in or close to your country
pool pool.ntp.org iburst
EOF

systemctl enable --now chrony
[ $? != 0 ] && installation_failed

# To check chrony status, use the following commands
# chronyc tracking
# chronyc -N sources
#------------------------------------------------------------------------------
# Create some default setup files. Creating these isn't strictly necessary, but they
# will serve as templates if somebody wants to manually edit them before proceeding.
CONF_DIR="/etc/alarm-clock"

if [[ ! -d "${CONF_DIR}" ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Creating a pro-forma ${CONF_DIR} directory"
  mkdir -p "${CONF_DIR}" "${CONF_DIR}/playlists"
  chmod 775 "${CONF_DIR}" "${CONF_DIR}/playlists"
  chown www-data:www-data -R "${CONF_DIR}"
fi

if [[ ! -f "${CONF_DIR}/setup.conf" ]] || [[ ! -s "${CONF_DIR}/setup.conf" ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Creating a pro-forma ${CONF_DIR}/setup.conf"

  cat << EOF > "${CONF_DIR}/setup.conf"
# Raspberry Pi Alarm Clock setup file
#
# The LOG_FILE setting must be a path to a writable filename (eg use /var/log/alarm-clock.log)
# The VERBOSE setting is expressed in '0x----' hex as a bitmask (eg. 0x200) - see alarm-clock.c
#
# The following provide a guide for determining the VERBOSE bitmask:
#  DISPLAY   = 0x0001
#  CLOCK     = 0x0002
#  DIMMER    = 0x0004
#  TIMER     = 0x0008
#  SETUP     = 0x0010
#  MEDIA     = 0x0020
#  AMP	     = 0x0040
#  BUTTON    = 0x0080
#  AMP_MEDIA = 0x0100
#  ALARM     = 0x0200
#  MULTICAST = 0x0400
#  BLUETOOTH = 0x0800
#
# If used, it is recommended that VERBOSE and LOG_FILE are defined before any other variables.
#VERBOSE=0x0b90   # for example, 0xb90 sets the BLUETOOTH, ALARM, AMP_MEDIA, BUTTON and SETUP bits
#LOG_FILE="/var/log/alarm-clock.log"

DEFAULT_1224="24"
DEFAULT_HOUR_ZERO="blank"
DEFAULT_SNOOZE_PRESS="restart"
MIN_VOLUME="11"
VOLUME="12"
DEFAULT_TIME="07:00"
SHUFFLE="N"
DEFAULT_ALARM_DURATION=120
DEFAULT_MEDIA_DURATION=60
DEFAULT_SNOOZE_DURATION=10
DEFAULT_BROWSE_DIR="/home/${USERNAME}"
FALLBACK_ALARM_FILE="/home/${USERNAME}"
DEFAULT_STREAM_OR_FILE="/home/${USERNAME}"
DEFAULT_MIN_LED=8
DEFAULT_MAX_LED=150
DEFAULT_MIN_AMBIENT=5
DEFAULT_MAX_AMBIENT=1000
CLOCK_TYPE="Standalone"
EOF
fi

if [[ ! -f "${CONF_DIR}/alarms.csv" ]] || [[ ! -s "${CONF_DIR}/alarms.csv" ]] ; then
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "Creating a pro-forma ${CONF_DIR}/alarms.csv"

  # Create a dummy alarm in the suspended state
  cat << EOF > "${CONF_DIR}/alarms.csv"
 Alarm Time,Suspended,Alarm Duration,Volume Adjust,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday,Stream
06:30,_SUSPENDED_,20,-15..0,Y,Y,Y,Y,Y,N,N,http://www.abc.net.au/res/streaming/audio/aac/classic_fm.pls
08:00,_SUSPENDED_,30,-4,N,N,N,N,N,Y,Y,http://www.abc.net.au/res/streaming/audio/aac/abc_jazz.pls
09:00,_SUSPENDED_,60,0,N,N,N,N,N,Y,Y,radio
EOF

  # Force the date to be really old so it won't overwrite any clustered clock settings
  touch --date="1 Jan 2010 01:02:03" "${CONF_DIR}/alarms.csv"
fi

chown -R www-data:www-data "${CONF_DIR}"
chmod 664 ${CONF_DIR}/*
#------------------------------------------------------------------------------
# Check if another local customisation exists, and execute if it does
# (Filenames of the form Install_Local.sh, Install_Local0.sh, Install_Local1.sh...)
if compgen -G "Install_Local*.sh" > /dev/null ; then
  for i in Install_Local*.sh ; do
    echo
    echo
    echo
    echo "Running local installation script $i"
    source $i
  done
fi
#------------------------------------------------------------------------------
# Compile the clock software and install it
echo
echo
echo "--------------------------------------------------------------------------"
echo "Compiling and installing the alarm clock software"

# Change into the home account of the newly created user
cd "/home/${USERNAME}"

if [[ ! -d alarm-clock ]] ; then
  echo
  echo "Cannot find the source code in directory /home/${USERNAME}/alarm-clock"
  exit 1
fi

chown -R "${USERNAME}:${USERNAME}" alarm-clock
cd alarm-clock
THIS_DIR=$(pwd)

# First step is compile the reboot-seconds utility
cd reboot-seconds
make
[ $? != 0 ] && installation_failed

make install

# Second step is to determine what model raspberry pi this is, and whether that is compatible
cd ${THIS_DIR}

# Now test for an early model PI. If so, we will overwrite the source code - it will not run!
if [[ ${MODEL} == *"Pi Model"* ]] ; then
  echo
  echo
  echo "The alarm-clock software is not supported on this model of Raspberry Pi"
  echo "but the web interface will simulate the look and feel of the clock as it would run"
  echo "on a later model of Raspberry Pi"
  mv alarm-clock.c alarm-clock.c.wont_run_on_this_model_pi

  cat << EOF > alarm-clock.c
#include <stdio.h>
#include <stdlib.h>

int
main ()
{
  printf ("Not supported on this model Pi\n") ;
  exit (EXIT_SUCCESS) ;
}
EOF
fi

# Third step is to compile the source code. It should compile without errors
make
[ $? != 0 ] && installation_failed

make install-no-reboot
rm -f *.o 2>/dev/null
#------------------------------------------------------------------------------
# This section sets up the raspberry pi to auto-run the clock upon boot

# For some reason, the installation of pigpiod on 32 bit operating systems is different
# than on 64 bit operating systems. There is a hard configured config for pigpiod on 32
# bit OS which breaks the config we installed above. Blow it away if it exists...
rm -rf /etc/systemd/system/pigpiod.service.d 2> /dev/null

# The F*****NG configuration of pigpiod is different on 32 bit and 64 bit systems
# pigpiod must be run with -t0 on a 64 bit system, and -t1 on a 32 bit system.
# The 32 bit behaviour contradicts what is documented on the Pigpiod web documentation ;-(
ARCH=$(uname -m)

if [[ "${ARCH}" == "aarch64" ]] ; then
  T_COMMAND="-t0"
elif [[ "${ARCH}" == "armv7l" ]] ; then
  T_COMMAND="-t1"
elif [[ "${ARCH}" == "armv6l" ]] ; then
  T_COMMAND="-t1"
else
  echo
  echo
  echo "--------------------------------------------------------------------------"
  echo "uname -m returns '${ARCH}' which is an unexpected value. Time to debug..."
  echo
  echo "I have not seen armv8 on my Pi3, but I believe this might happen if running 64 bits."
  echo "I am not sure whether pigpiod requires -t0 or -t1 in this case. You'll need to experiment."
  echo
  echo "I also believe armv6 is a possibility on a Pi2 or earlier? I would guess use same settings as for armv6l?"
  echo
  echo "You may need to edit $0 at around line 1160 and sort this out."
  exit 1
fi

echo
echo
echo "--------------------------------------------------------------------------"
echo "Creating the systemd configuration to auto-run the alarm-clock"

# We are going to overwrite the systemd configuration file for pigpiod
# because we are going to change the launch options
cat << EOF > /lib/systemd/system/pigpiod.service
[Unit]
Description=Daemon required to control GPIO pins via pigpio

[Service]
Type=forking
ExecStart=/usr/bin/pigpiod -l -s10 ${T_COMMAND} -x0xFFFFFFF
Restart=always
ExecStop=/bin/systemctl kill pigpiod

[Install]
WantedBy=multi-user.target
EOF

# Configure the system configuration to auto-run the alarm-clock at next reboot
# We won't use the daemon feature built in to the executable, we will use
# systemd's daemonisation feature
cat << EOF > /lib/systemd/system/alarm-clock.service
[Unit]
Description=Raspberry Pi Clock
After=network-online.target
After=pigpiod.service

[Service]
Type=simple
ExecStart=/usr/local/bin/alarm-clock
Restart=always
ExecReload=/usr/local/bin/alarm-clock -r
ExecStop=/usr/local/bin/alarm-clock -x ; sleep 0.5 ; /bin/systemctl kill alarm-clock

[Install]
WantedBy=multi-user.target
EOF

chmod 644 /lib/systemd/system/alarm-clock.service /lib/systemd/system/pigpiod.service
systemctl daemon-reload
systemctl enable alarm-clock.service pigpiod.service
#------------------------------------------------------------------------------
echo
echo
echo "--------------------------------------------------------------------------"
echo "Adding a periodic job to check that the alarm-clock daemon is running (to force reboot if not)"

crontab -l | grep alarm-clock > /dev/null 2>&1

if [ $? != 0 ] ; then
  # If we reach here, there is no entry for the alarm-clock in crontab
  # Read the previous contents of the crontab into an array
  unset EXISTING_CRON
  readarray -t EXISTING_CRON <<< "$(crontab -l 2> /dev/null)"

  # Append the new crontab entry at the end of the crontab array.
  # Arbitrarily, we will choose to run the command once every 30 minutes.
  # The command calls alarm-clock to do a dummy read of the ambient light level.
  # When alarm-clock is called, it tries and open a message queue for communicating with the
  # alarm-clock daemon. If the message queue doesn't exist, alarm-clock will force a Raspberry Pi reboot.
  EXISTING_CRON+=('*/30 * * * * /usr/local/bin/alarm-clock -a')

  # replace the existing crontab with the appended version
  for i in "${EXISTING_CRON[@]}" ; do
    echo "${i}"
  done | crontab -
fi
#------------------------------------------------------------------------------
#if [[ "${VERSION_ID}" -ge 12 ]] ; then
#  egrep "dns=dnsmasq" /etc/NetworkManager/NetworkManager.conf > /dev/null 2>&1

#  if [ $? != 0 ] ; then
#    echo
#    echo
#    echo "--------------------------------------------------------------------------"
#    echo "Configuring a local wi-fi access point"

    # Get the name of this clock (which we will use as the access point name)
#    SYSNAME=$(head -n 1 /etc/hostname)
#    [[ -z "${SYSNAME}" ]] && SYSNAME="Clock"

    #RANDOM_PASSWORD=$(openssl rand -base64 6)
#    RANDOM_PASSWORD="recovery"
    #[[ -z "${RANDOM_PASSWORD}" ]] && RANDOM_PASSWORD="${SYSNAME}.recovery"

#    AP_IP="172.16.31.1"

#    apt-get -y install --no-install-recommends \
#	    dnsmasq

#    [ $? != 0 ] && installation_failed

#    sed -i '/\[main\]/a dns=dnsmasq' /etc/NetworkManager/NetworkManager.conf
#    [ $? != 0 ] && installation_failed

    # Disable dnsmasq from auto-running to avoid conflicts with the instance started by NetworkManager
#    systemctl disable dnsmasq
#    systemctl stop dnsmasq

    # Configure the access point (first removing the same connection name if it exists already)
#    nmcli con delete "${SYSNAME}-AP" > /dev/null 2>&1

#    nmcli con add type wifi ifname wlan0 mode ap con-name "${SYSNAME}-AP" ssid "${SYSNAME}" autoconnect false
    #nmcli con modify "${SYSNAME}-AP" 802-11-wireless.cloned-mac-address 00:12:34:56:78:9a
#    nmcli con modify "${SYSNAME}-AP" wifi-sec.key-mgmt wpa-psk
#    nmcli con modify "${SYSNAME}-AP" wifi-sec.proto rsn
#    nmcli con modify "${SYSNAME}-AP" wifi-sec.group ccmp
#    nmcli con modify "${SYSNAME}-AP" wifi-sec.pairwise ccmp
#    nmcli con modify "${SYSNAME}-AP" wifi-sec.psk "${RANDOM_PASSWORD}"
#    nmcli con modify "${SYSNAME}-AP" ipv4.method shared ipv4.address ${AP_IP}/24
#    nmcli con modify "${SYSNAME}-AP" ipv6.method disabled
    #nmcli con up "${SYSNAME}-AP"

    # Use 'nmcli con show' to display which wi-fi connection is active

#    echo
#    echo
#    echo "REMEMBER THE FOLLOWING DETAILS!!!"
#    echo "(These are the clock's internal access point which you can use if you lose the ability to connect"
#    echo "using your home wi-fi settings that you provided when configuring the SD card)"
#    echo
#    nmcli dev wifi show-password
#    echo
#    echo "The clock's IP address on this access point is ${AP_IP}, and the password is shown above."
#    echo "Write these details down now!"
#    echo
#    echo "To recover if the clock loses wi-fi connectivity to your home wi-fi network (eg you change your home"
#    echo "router or you change your home wi-fi password):"
#    echo
#    echo "1. connect to the clock's wifi access point, using the details above."
#    echo "2. open http://${AP_IP} to reach the clock's internal web pages"
#    echo "3. Reconfigure the clock's wi-fi settings to include your new home router!"
#    echo
#    echo
#  fi
#fi
#------------------------------------------------------------------------------
# This section creates a script to fix file and directory permissions so they are visible
echo
echo
echo "--------------------------------------------------------------------------"
echo "Creating fix-media-tree utility to fix access permissions and remove spaces"
echo "from media filenames in a directory tree of media files"

cat << EOF > /usr/local/bin/fix-media-tree
#!/bin/bash -noprofile
##############################################################
check_if_root()
{
  # Make sure only root can run our script
  if [[ ${DOLLAR}EUID -ne 0 ]] ; then
    echo "This script must be run as root." 1>&2
    echo "For example, run the script as follows:" 1>&2
    echo "   sudo ${DOLLAR}0" 1>&2
    exit 1
  fi
}
##############################################################
usage()
{
  echo "${DOLLAR}(basename ${DOLLAR}{1}) <directory or /path/to/directory>"
  echo
  echo "Changes the file permissions to 644 (-rw-r--r--)"
  echo "and directory permissions to 755 (-rwxr-xr-x)"
  echo "and it will also remove spaces from file and directory names."
}
##############################################################
# Remove spaces from directories and filenames
remove_spaces()
{
  [[ ! -d "${DOLLAR}{1}" ]] && return
  cd "${DOLLAR}{1}" || return

  # rename the sub-direcrtories and recurse
  for i in * ; do
    [[ ! -d "${DOLLAR}{i}" ]] && continue

    MOD_NAME="${DOLLAR}(echo ${DOLLAR}{i} | sed 's/[[:space:]]\+/_/g' | sed 's/[^[:alnum:]._]//g')"

    if [[ "${DOLLAR}{i}" != "${DOLLAR}{MOD_NAME}" ]] ; then
      echo "Renaming ${DOLLAR}{1}/${DOLLAR}{i} --> ${DOLLAR}{MOD_NAME}"
      mv "${DOLLAR}{i}" "${DOLLAR}{MOD_NAME}"
    fi

    remove_spaces "${DOLLAR}{i}"
  done

  # rename the files in this directory
  for i in * ; do
    [[ ! -f "${DOLLAR}{i}" ]] && continue

    MOD_NAME="${DOLLAR}(echo ${DOLLAR}{i} | sed 's/[[:space:]]\+/_/g' | sed 's/[^[:alnum:]._]//g')"

    if [[ "${DOLLAR}{i}" != "${DOLLAR}{MOD_NAME}" ]] ; then
      echo "Renaming ${DOLLAR}{1}/${DOLLAR}{i} --> ${DOLLAR}{MOD_NAME}"
      mv "${DOLLAR}{i}" "${DOLLAR}{MOD_NAME}"
    fi
  done
}
##############################################################
check_if_root

if [[ ${DOLLAR}# != 1 ]] ; then
  usage "${DOLLAR}{0}"
  exit 1
fi

if [[ ! -d "${DOLLAR}{1}" ]] ; then
  echo "'${DOLLAR}{1}' is not a directory"
  echo
  usage "${DOLLAR}{0}"
  exit 1
fi

# Change file and directory permissions
find "${DOLLAR}{1}" -type f -name '*' -print0 2>/dev/null | xargs -0 chmod 0644
find "${DOLLAR}{1}" -type d -name '*' -print0 2>/dev/null | xargs -0 chmod 0755
##############################################################
# Does the passed directory contain spaces?
MOD_NAME="${DOLLAR}(echo ${DOLLAR}{1} | sed 's/[[:space:]]\+/_/g')"

if [[ "${DOLLAR}{1}" != "${DOLLAR}{MOD_NAME}" ]] ; then
  echo "Renaming ${DOLLAR}{1} --> ${DOLLAR}{MOD_NAME}"
  mv "${DOLLAR}{1}" "${DOLLAR}{MOD_NAME}"
fi

remove_spaces "${DOLLAR}{1}"
EOF

chmod 755 /usr/local/bin/fix-media-tree

#cat << EOF > "/home/${USERNAME}/Fix_file_visibility_and_access.txt"
#After copying media files onto the Raspberry Pi from a Windows environment, you
#may find that the web server will not be able to find or browse the new files and
#directories, even though they are visible on the network and when you log in via
#ssh to the Pi.
#
#If so, the issue is possibly file and directory permissions having been set too
#restrictive. Because the web server runs as is own user, it will be unable to
#access files and directories for which it has not been granted permission.
#
#I have included a script to descend a directory tree and make the files and
#sub-directories readable from the web server. The script will also change any
#spaces within the filenames into underscore characters.
#
#To use the script, ssh onto the Pi and issue the command
#
#  fix-media-tree <Directory>
#
#where <Directory> is the Linux path to the name of the directory that cannot be
#accessed or contains media that cannot be accessed.
#EOF
#------------------------------------------------------------------------------
# This section creates a script to temporarily stop Linux managing the clock

echo
echo
echo "--------------------------------------------------------------------------"
echo "Creating stop-alarm-clock utility to stop Linux auto-managing the alarm clock"
echo "programme, so that you can run alarm-clock from the command line"

cat << EOF > /usr/local/bin/stop-alarm-clock
#!/bin/bash -noprofile
##############################################################
sudo systemctl stop alarm-clock
sudo systemctl stop pigpiod
sudo pkill -9 alarm-clock
sudo pkill -9 pigpiod

echo "**************************************************************"
echo "Commands have been issued to stop the alarm-clock software."
echo
echo "If you have a web page opened to the alarm-clock while it is"
echo "stopped, the crash detection algorithm is liable to reboot the"
echo "Pi to restart the clock."
echo
echo "To avoid this, either run alarm-clock on the command line"
echo "specifying any options you would like, such as:"
echo "  sudo alarm-clock -V"
echo
echo "or close all open web pages to the clock NOW!"
echo "**************************************************************"
EOF

chmod 755 /usr/local/bin/stop-alarm-clock
#------------------------------------------------------------------------------
# The installation is complete. It is now time to reboot the raspberry pi
echo
echo
echo "--------------------------------------------------------------------------"
echo "The installation script has ended. Were there any errors above??? There should have been none."
echo
echo "If there were no errors, it is time to reboot the pi. When it comes back up, you can log in using '${USERNAME}'"
echo "If there were errors, you'll need to work out what is wrong and maybe start from scratch."
echo
read -p 'Hit ENTER to reboot the raspberry pi or CTRL-C to break back to the prompt' DUMMY
echo
echo
echo "**************************************************************************"
echo "Go and make a cup of tea. Allow a few minutes before opening a web page to"
echo "your new clock so that the new OS can initialise itself comfortably before"
echo "you connect to it for the first time."
echo "**************************************************************************"
echo
echo

shutdown -r now
