
 
; Mic the Mouse
; has MEMs microphone and makes mouse sounds when environment is quiet
; stops making sound when a noise is heard 

#include <pic16f15214.inc>
;#include <xc.inc>   

;configuration bits
CONFIG FEXTOSC=OFF
CONFIG RSTOSC=HFINTOSC_32MHZ
CONFIG CLKOUTEN=OFF
CONFIG VDDAR=HI
CONFIG MCLRE=INTMCLR
CONFIG PWRTS=PWRT_OFF
CONFIG WDTE=01; watchdog enabled in software
CONFIG BOREN=OFF
CONFIG BORV=HI
CONFIG PPS1WAY=OFF
CONFIG STVREN=OFF
CONFIG BBSIZE=BB64K
CONFIG BBEN=OFF
CONFIG SAFEN=OFF
CONFIG WRTAPP=OFF
CONFIG WRTB=OFF
CONFIG WRTC=OFF
CONFIG WRTSAF=OFF
CONFIG LVP=OFF
CONFIG CP=OFF
    
;define bits
;STATUS flags    
#define C 0
#define Z 2
; FLASH read and write flags
#define NVMREGS 6    
#define RD 0  
#define WR 1
#define FREE 4
#define LWLO 5
#define WREN 2 
;Interrupts
#define GIE 7 
#define TMR0IE 5
#define TMR0IF 5  
    
; PSECT = program sections      

PSECT   Storage_Vec,global,class=CODE,delta=2
Storage_Vec:
    
; sound level threshold stored in Flash at FF0h
DW              03F02h		; A/D level comparison threshold for sound detection currently set at 03h (03F ms bits unused). 
    ; The ls bits value must be between 2 and 11. (03F)02 and (03F)11)

; RAM 
LEVEL       	equ	020h	; MEMS sound level threshold  
STORE1			equ	021h    ; delay counter	
STORE2			equ	022h 	; delay counter
STORE3			equ	023h	; delay counter
ON_FLG			equ	024h	; mouse sounds 'on' flag
TIMER1			equ	025h	; run timer
TIMER2			equ	026h	; run timer
CHECK_COUNT		equ 027h	; counter for MEMS checking
STO1	    	equ	028h	; AARGB1 storage
RNDM_COUNT		equ 029h	; counter for random value lookup 
PRE_SCALE       equ 02Ah    ; timer0 prescaler for timeout vary for required quietness period
WDT_TOUT        equ 02Bh    ; watchdog timer timeout
CHANCES         equ 02Ch    ; number of transient noises allowed
FLASH			equ	02Dh	; LED flash
QUIET_TIMER		equ	02Eh	; timer for quieting
QUIET_TIM0		equ	02Fh	; multiplier
QUIET			equ	030h	; quiet flag
TIME			equ	031h	; multiply burst period timer
TIME1			equ	032h	; store for TIME
TIME2			equ	033h	; store for time ending	
AD_RUNS         equ 034h    ; number of A/D runs in sound gaps
MOUSE_COUNTSms  equ 035h    ; mouse noises duration if no noises ms byte
MOUSE_COUNTSls  equ 036h    ; mouse noises duration if no noises ls byte  
TMR0_OVER       equ 037h    ; timer0 overflow flag store
VALUE			equ	038h    ; internal delay value	
PWM				equ	039h    ; pwm drive 	
WIDTH			equ	03Ah	; working pwm duty
WIDTH2			equ	03Bh	; working PWM duty
DENOM			equ	03Ch	; divider denominator temporary store
BURST_SS1		equ	03Dh	; burst length of first
BURST_SS2		equ	03Eh	; burst length of second
BURST_SS3		equ	03Fh	; burst length of third
BURST_SS4		equ	040h	; burst length of fourth
FREQ_SS1		equ	041h	; frequency of first tone 
FREQ_SS2		equ	042h	; frequency of second tone
FREQ_SS3		equ	043h	; frequency of second tone
FREQ_SS4		equ	044h	; frequency of second tone
PWM_X1			equ	045h	; PWM duty value of first tone
PWM_X2			equ	046h	; PWM duty value of second tone
PWM_X3			equ	047h	; PWM duty value of third tone
PWM_X4			equ	048h	; PWM duty value of fourth tone
         
; Calculation registers for multiply and divide routines
REMB1      		equ 063h	; remainder ls byte
REMB0       	equ 064h    ; most significant (ms) byte of remainder
TEMPD  		 	equ 065h    ; temporary
BARGB0			equ	066h	; multiplicand or denominator ms byte
BARGB1			equ	067h	; 
AARGB0			equ	068h	; multiplicand ms byte
AARGB1			equ	069h	; 
AARGB2			equ	06Ah	; 
AARGB3			equ	06Bh	; 
TEMPB0			equ	06Ch	; temp files
TEMPB1			equ	06Dh	; temp files
TEMPB2			equ	06Eh	; temp files
LOOPCOUNT		equ	06Fh	; loop counter 
        
; All banks RAM        
SOUND           equ 070h    ; A/D MEMS microphone sound level ms byte
SOUNDLS         equ 071h    ; A/D MEMS microphone sound level ls byte           
FREQ_SS         equ 072h    ; generator routine frequency
BURST_SS        equ 073h    ; generator routine burts length
PWM_X           equ 074h    ; generator pwm value for volume
SOFT            equ 075h    ; soft start and soft end flags 
BURST           equ 076h    ; Burst period
SLEEP_COUNT     equ 077h    ; number of wdt timeouts

; ******************************************************************
           
     
; For MPLAB X IDE Add the line below in the Production/Set project configuration/Customise/pic-as Global Options/Additional Options:
; -Wa,-a -Wl,-pPor_Vec=0h,-pStorage_Vec=FF0h

; PSECT = program sections      
; define Power on reset 
PSECT   Por_Vec,global,class=CODE,delta=2
; Power-On-Reset entry point
Por_Vec:
  	goto	_START	 
    

 ; **********************************************************
; lookup tables
; Timer variations for sound gaps  	
RANDOM:
	movf	RNDM_COUNT,w
    brw
	retlw	88	; 0
	retlw	55
	retlw	84
	retlw	75
	retlw	35
	retlw	103
	retlw	32
	retlw	8
	retlw	88
	retlw	60
	retlw	33
	retlw	80
	retlw	23
	retlw	35
	retlw	44
	retlw	123
	retlw	50
	retlw	1
	retlw	104
	retlw	103
	retlw	112
	retlw	36
	retlw	99
	retlw	7
	retlw	44
	retlw	126	; 25
	retlw	90
	retlw	88
	retlw	67
	retlw	45
	retlw	84
	retlw	76
	retlw	118
	retlw	5
	retlw	117
	retlw	88
	retlw	100
	retlw	51
	retlw	49
	retlw	45
	retlw	57	; 40
	retlw	111
	retlw	17
	retlw	34
	retlw	13	; 44
	retlw	95
	retlw	59
	retlw	119
	retlw	54
	retlw	41
	retlw	59
	retlw	50
	retlw	5
	retlw	16
	retlw	94
	retlw	98
	retlw	72
	retlw	51
	retlw	13
	retlw	124
	retlw	126	; 60
	retlw	36
	retlw	98
	retlw	20
	retlw	15
	retlw	68
	retlw	53
	retlw	35
	retlw	89	; 68
	retlw	101
	retlw	63
	retlw	91
	retlw	41
	retlw	25
	retlw	77
	retlw	1	; 75
	retlw	76
	retlw	40
	retlw	64
	retlw	88
	retlw	61
	retlw	29
	retlw	27
	retlw	77
	retlw	25
	retlw	67
	retlw	24
	retlw	3
	retlw	10
	retlw	34
	retlw	118	; 90
	retlw	117
	retlw	43
	retlw	35
	retlw	61
	retlw	12
	retlw	2
	retlw	48
	retlw	37
	retlw	94
	retlw	70	; 100
	retlw	123
	retlw	47
	retlw	32
	retlw	84
	retlw	73
	retlw	32
	retlw	1
	retlw	50
	retlw	89
	retlw	61
	retlw	39
	retlw	80
	retlw	65
	retlw	87
	retlw	54	; 115
	retlw	85
	retlw	117
	retlw	67
	retlw	30
	retlw	43	; 120
	retlw	88
	retlw	8
	retlw	53
	retlw	30
	retlw	83
	retlw	80
	retlw	50	; 127
	retlw	12
     
; timer1 prescale value for period required of quietness before Mouse sounds
TIMER0_PRESCALE:  ;  2m to 16m
   	movf	PRE_SCALE,w
    brw
    retlw   01001001B       ;0, /512 for ~8.8m timeout 8m
	retlw   01000111B       ;1, /128 for ~2.2m timeout 2m (first time run at power up)
    retlw   01001010B       ;2, /1024 for ~16.16m timeout 16m
    retlw   01001000B       ;3, /256 for ~4.4m timeout 4m
    retlw   01001000B       ;4, /256 for ~4.4m timeout 4m

WDT_PRESCALE:    
; watchdog timeout 4 seconds to 4 minutes. with SLEEP_COUNT, sleep times are repeated 30 times to provide between about 2m and 2hours
    movf    WDT_TOUT,w
    brw
    retlw   00100001B       ;0; 64s, 1 minute. All of these are increased by 30 with the SLEEP_COUNT at 30.
    retlw   00011001B       ;1; 4s (first time run at power up)
    retlw   00100101B       ;2; 256s, 4 minutes  
    retlw   00011011B       ;3; 8s
    retlw   00011101B       ;4; 16s
    retlw   00100011B       ;5; 128s, 2 minutes
    retlw   00011111B       ;6; 32s, 1/2 minute
    retlw   00011001B       ;7; 4s
   
; ***********************************************************
_START:
   
; set oscillator calibration
    BANKSEL OSCFRQ
	movlw	00000010B	; oscillator calibration value 4MHz
	movwf	OSCFRQ
   
    BANKSEL PORTA
; start up delay 
	call	DELAYms
; set inputs/outputs
	clrf	LATA		; outputs low
	movlw	00001000B	; pullups off except RA3
    BANKSEL WPUA
	movwf	WPUA

    movlw	00101000B	; outputs/inputs 
    BANKSEL TRISA
	movwf	TRISA		; port data direction register
  
    movlw	00011000B	; (wdt prescaler/ for 4s timeout) wdt enabled at sleep
    BANKSEL WDTCON
	movwf	WDTCON
 
    BANKSEL ANSELA
	clrf	ANSELA		; digital I/O
    bsf     ANSELA,5    ; AN5 as analog input 
   
    BANKSEL ADCON0
    movlw   00010101B
    movwf   ADCON0      ; AN5 
    movlw   01010000B   ; left justified, fosc/16 vref to vdd    
    movwf   ADCON1
    clrf    ADACT       ; Auto-conversion disabled
       
    BANKSEL T0CON0		; timer0 control 16-bit timer
	movlw	10011111B
	movwf	T0CON0		; enabled 1:16  prescaler
	movlw	01000111B
	movwf	T0CON1		; /128 for 134s (2.2minutes) between overflow.(with Fosc/4 : 4MHz oscillator)
; timer0 timeout flag used to determine if no noise over timer period
    BANKSEL PORTA
  
; initial conditions
    clrf    RNDM_COUNT  ; initial position in random value lookup
    clrf    PRE_SCALE   ; timer0 prescaler initially value in lookup
    clrf    WDT_TOUT    ; initial position in wdt period lookup 
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

 ; read threshold level
    call    READ            ; get value from Flash storage
    movwf   LEVEL           ; threshold level
    
; RESETS    
    BANKSEL PCON0           ; test reset sources
    movf    PCON0,w

    BANKSEL PORTA
    movwf   TEMPB0
    btfss   TEMPB0,4        ; watchdog timer reset?
    goto    MOUSE_RUNNING   ; watchdog timeout so run mouse
     
; check S2 pressed or just a power up switch-on via S1
    movlw   30
    call    DEL_SET         ; 100ms delay
    btfss   PORTA,3         ; check if S2 is pressed
    goto    LEVEL_NOT_CHNG
; Flash LED momentarily at power up  
    bsf     LATA,2          ; LED on
    movlw   5               ; ~20ms flash
    call    DEL_SET         ; delay.
    bcf     LATA,2
    goto    MOUSE_RUN
    
; switch pressed so change LEVEL; LEVEL is threshold level for sound detection
LEVEL_CHNG:
    clrwdt  ; reset to 16s timeout
    incf    LEVEL,f
LEVEL_NOT_CHNG:    
    movf    LEVEL,w
    sublw   11              ; maximum of 11
    btfsc   STATUS,C        
    goto    NOT_OVER
    movlw   2               ; minimum of 2
    movwf   LEVEL
NOT_OVER:  
; store in Flash
    movf    LEVEL,w
    call    FLASH_WRITE     ; Store value with w register containing LEVEL value 

; flash LEDs to indicate selection 
NO_SW:     
    movf    LEVEL,w
    movwf   TEMPB0          ; working register
; value between 2 and 11 but show threshold as a 1 to 10 value (as number of LED flashes) 
; take away 1
    movlw   1
    subwf   TEMPB0,f
    
; AN5 input set as a low output   
    movlw	00001000B       ; outputs/inputs set AN5 a low output
    BANKSEL TRISA
	movwf	TRISA           ; port data direction register
    BANKSEL PORTA
; MEMS microphone on   
    bsf     LATA,4          ; MEMS on
  
SHOW_AGAIN:
    bsf     LATA,2          ; LED on
    movlw   20              ; 80ms
    call    DEL_SET         ; display level with blinks of LED
    bcf     LATA,2          ; LED off
    movlw   140             ; 500ms
    call    DEL_SET         ; display level with blinks of LED
    
    decfsz  TEMPB0,f
    goto    SHOW_AGAIN
    
; set up watchdog for 16s for testing sensitivity  
; if it times out then start normal mouse run    
    movlw	00011101B       ; wdt prescaler for 16s timeout
    BANKSEL WDTCON
	movwf	WDTCON
    BANKSEL PORTA    
    
 ; check MEMS for sound
CK_MEMS1: 
   
	bcf     LATA,2          ; LED off
    call    AD_RUN          ; get microphone output    
	movf    SOUND,w         ; microphone sound
    subwf   LEVEL,w         ; if over 
    btfss   STATUS,C        ; Carry = 0 if sound >
	goto	LIGHT           ; when microphone output is high then noises
	bcf     LATA,2
    btfss   PORTA,3         ; if S2 closed increase LEVEL
    goto    LEVEL_CHNG
    goto    CK_MEMS1    
    
LIGHT:; LED shows when sound detected 
    bsf     LATA,2          ; LED on
    movlw   1               ; 4ms flash
    call    DEL_SET         ; delay. Indicate sound detection with LED
    btfss   PORTA,3         ; if S2 closed increase LEVEL
    goto    LEVEL_CHNG
    goto    CK_MEMS1
 ; end of showing LEVEL and testing Level value
 ; runs mouse after wdt timeout
 
 ;.............................................................................
 
MOUSE_RUNNING: ; run Mouse sounds on and off
  
; stop watchdog timer
    BANKSEL WDTCON
	movlw	00011000B       ; wdt off. Prescaler/ for 4s timeout)
    movwf	WDTCON
   
    BANKSEL PCON0           ; set wdt reset source flag
    bsf     PCON0,4
    BANKSEL PORTA
    goto    ALL_QUIET       ; mouse sounds
 ;.............................................................................   

MOUSE_RUN: ; run Mouse sounds on and off
  
; stop watchdog timer
    BANKSEL WDTCON
	movlw	00011000B       ; wdt off. Prescaler/ for 4s timeout)
    movwf	WDTCON
   
    BANKSEL PCON0           ; set wdt reset source flag
    bsf     PCON0,4
    BANKSEL PORTA
 ; wait for quiet    
;.........................................................................
    
FIND_QUIET:
    incf    PRE_SCALE,f
    movf    PRE_SCALE,w
    xorlw   4               ; if 4 reset to 0
    btfsc   STATUS,Z
    clrf    PRE_SCALE
    call    TIMER0_PRESCALE ; returns with timer prescaler for between 2.2 minutes to 16 minutes timeout
    BANKSEL T0CON1 	
	movwf	T0CON1          ; 2m to 16m   
    clrf    TMR0H           ; ms byte of timer cleared
    clrf    TMR0L
    BANKSEL	PIE0
	bcf		PIR0,TMR0IF     ; clear overflow flag for TMR0 
	BANKSEL PORTA 
; timer timeout flag used to determine if no noise over timer period
  
; Timer0 cleared. if timer0 times out (overflows) then noise was silent for whole period
       
 ; switch on microphone
 ; AN5 input set as a low output   
    movlw	00001000B       ; outputs/inputs set AN5 a low output
    BANKSEL TRISA
	movwf	TRISA           ; port data direction register
    BANKSEL PORTA
; MEMS microphone on   
    bsf     LATA,4          ; MEMS on

 ; 500ms on-time before reading  
    movlw   140
    call    DEL_SET         ; 500ms delay for microphone output to settle
    movlw   10
    movwf   CHANCES         ; ten noise transients allowed
NOISES_RUN: 
    call    AD_RUN          ; get microphone output (1ms period)    
	movf    SOUND,w         ; microphone sound level
    subwf   LEVEL,w         ; if over 
    btfss   STATUS,C        ; Carry = 0 if sound >
	goto	NOISE1          ; when microphone output is high then noises
; no noises. Must be no noise for timer 0 period
 
; Check if overflow flag set
    BANKSEL	PIE0
	movf	PIR0,w            ; get TMR0IF flag  
	BANKSEL PORTA 
    movwf   TMR0_OVER         ; timer0 overflow flag store
    btfss   TMR0_OVER,5       ; TMR0IF position in PIR0  
    goto    NOISES_RUN
    goto    ALL_QUIET         ; if overflow, then no noise during timer0 period  
NOISE1:
    movf    CHANCES,w
    btfsc   STATUS,Z          ; if zero do not decrease chances 
    goto    SHUT_DOWN         ; noises occurred after chances have ended   
    decfsz  CHANCES,f
    goto    NOISES_RUN
    goto    SHUT_DOWN

;..............................................................................

SHUT_DOWN:    
  
;  sleep counts
    movlw   30              ; 
    movwf   SLEEP_COUNT     ; number of wdt timeouts for overall sleep period
 
; MEMS microphone and LED off
    bcf     LATA,2
    bcf     LATA,4
    incf    WDT_TOUT,f
    movf    WDT_TOUT,w      ; value for lookup table
    xorlw   7               ; if 7 reset to 0
    btfsc   STATUS,Z
    clrf    WDT_TOUT
    call    WDT_PRESCALE    ; w returns with wdt prescaler value for between 4s to 256s timeout
    BANKSEL WDTCON
	movwf	WDTCON
    BANKSEL PORTA
SNOOZE: 
; flash LED each    
    bsf     LATA,2          ; LED on
    movlw   1
    call    DEL_SET    ;    ; 4ms
    bcf     LATA,2          ; LED off
    
	sleep					; stop operations. wdt cleared on sleep and wakeup
    nop
; repeat sleep until SLEEP_COUNT is zero
    decfsz  SLEEP_COUNT,f
    goto    SNOOZE
    
; end of snooze 4s x 30 (2m) to 4m x 30 (2hr)
    
; awakes with watchdog timeout
    BANKSEL WDTCON
    movlw	00011000B       ; wdt off (prescaler timer set when wdt on again)
    movwf	WDTCON
    BANKSEL PCON0           ; set wdt reset source flag
    bsf     PCON0,4
	BANKSEL PORTA
 
    goto    FIND_QUIET      ; check for quietness
    
;.........................................................................
    
ALL_QUIET:; ready to make Mouse sounds
; switch on microphone
; AN5 input set as a low output   
    movlw	00001000B       ; outputs/inputs set AN5 a low output
    BANKSEL TRISA
	movwf	TRISA           ; port data direction register
    BANKSEL PORTA
; MEMS microphone on   
    bsf     LATA,4          ; MEMS on
  
 ; 500ms on-time before reading  
    movlw   140
    call    DEL_SET         ; 500ms delay for microphone output to settle
    movlw   5
    movwf   CHANCES         ; noise detect chances counter (allows for x noise detects to be ignored)
    
; load mouse sounds duration   
    incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
    andlw   00000011B       ; maximum of 4. Limits mouse sounds run period
    movwf   MOUSE_COUNTSms  ; mouse noises duration if no noises ms byte
    incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call (ms bit is always clear in lookup table)
    movwf   MOUSE_COUNTSls  ; mouse noises duration ls byte
; adjust values so ls has all 8-bits and only have 1 bit in ms byte to reduce overal maximum and include ms bit in ls byte
; if bit 1 in MOUSE_COUNTSms is set then set bit 7 in MOUSE_COUNTSls
    btfsc   MOUSE_COUNTSms,1
    bsf     MOUSE_COUNTSls,7 ; utilise bit 7 in ls byte
; clear bit 1 in MOUSE_COUNTSms reducing maximum value of count
    bcf     MOUSE_COUNTSms,1   
; get random frequency values for four sequences
FREQUENCY_SETUP:
 	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	FREQ_SS1		; store initially
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	FREQ_SS2
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	FREQ_SS3
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	FREQ_SS4

; calculate as (6 x random store (FREQ_SS1))/~127 then add 5 for a 5-11 range
	movf	FREQ_SS1,w
	call	RUN_FREQ
	movwf	FREQ_SS1		; processed FREQ_SS1 frequency value
; calculate as (6 x random store (FREQ_SS2))/127 then add 5 for a 5-11 range
	movf	FREQ_SS2,w
	call	RUN_FREQ
	movwf	FREQ_SS2		; processed FREQ_SS2 frequency value
; calculate as (6 x random store (FREQ_SS3))/127 then add 5 for a 5-11 range
	movf	FREQ_SS3,w
	call	RUN_FREQ
	movwf	FREQ_SS3		; processed FREQ_SS3 frequency value
; calculate as (6 x random store (FREQ_SS4))/127 then add 5 for a 5-11 range
	movf	FREQ_SS4,w
	call	RUN_FREQ
	movwf	FREQ_SS4		; processed FREQ_SS4 frequency value
	goto	BURST_SETUP
RUN_FREQ:	; common code sequence
	movwf	AARGB1			; ms byte is cleared in the calculate routine
	movlw	8
	movwf	BARGB1			; for multiply by 6 (ms byte is cleared in the calculate routine)
	movlw	100
	movwf	DENOM			; denominator for divider		
	call	CALCULATE
	addlw	5               ; add the 5
	return

; ...............
BURST_SETUP:
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	BURST_SS1		; store initially
	movwf	BURST_SS2
	movwf	BURST_SS3
	movwf	BURST_SS4
; calculate as (BURST_SS1 x 25/ FREQ_SS1)
	movf	FREQ_SS1,w
	movwf	DENOM
	movf	BURST_SS1,w		; burst
	call	RUN_BURST
	movwf	BURST_SS1		; processed BURST_SS1/ frequency value
; calculate as (BURST_SS2 x 25/ FREQ_SS2)
	movf	FREQ_SS2,w		; denominator for divider
	movwf	DENOM
	movf	BURST_SS2,w		; burst
	call	RUN_BURST
	movwf	BURST_SS2		; processed BURST_SS2/ frequency value
; calculate as (BURST_SS3 x 25/ FREQ_SS3)
	movf	FREQ_SS3,w		; denominator for divider
	movwf	DENOM
	movf	BURST_SS3,w		; burst
	call	RUN_BURST
	movwf	BURST_SS3		; processed BURST_SS3/ frequency value
; calculate as (BURST_SS4 x 25/ FREQ_SS4)
	movf	FREQ_SS4,w		; denominator for divider
	movwf	DENOM
	movf	BURST_SS4,w		; burst
	call	RUN_BURST
	movwf	BURST_SS4		; processed BURST_SS4/ frequency value
	goto	PWM_SETUP

RUN_BURST:	; common code sequence
	movwf	BARGB1			; 
	movlw	10           	; was D25 
	movwf	AARGB1			; 
	call	CALCULATE		; 
	return
;.............
PWM_SETUP:
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	PWM_X1			; store initially
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	PWM_X2
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	PWM_X3
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
	movwf	PWM_X4
; calculate as (FREQ_SS1 x PWM_X1)/255 then add 1 for a 1-50 range
	movf	PWM_X1,w
	movwf	BARGB1	
	movf	FREQ_SS1,w
	call	RUN_PWMX
	movwf	PWM_X1		; processed PWM_X1 frequency value
    incf    PWM_X1,f    ; +1
; calculate as (FREQ_SS2 x PWM_X2)/255 then add 1 for a 1-50 range
	movf	PWM_X2,w
	movwf	BARGB1	
	movf	FREQ_SS2,w
	call	RUN_PWMX
	movwf	PWM_X2		; processed PWM_X2 frequency value
    incf    PWM_X2,f    ; +1
; calculate as (FREQ_SS1 x PWM_X3)/255 then add 1 for a 1-50 range
	movf	PWM_X3,w
	movwf	BARGB1	
	movf	FREQ_SS3,w
	call	RUN_PWMX
	movwf	PWM_X3		; processed PWM_X3 frequency value
    incf    PWM_X3,f    ; +1
; calculate as (FREQ_SS4 x PWM_X4)/255 then add 1 for a 1-50 range
	movf	PWM_X4,w
	movwf	BARGB1	
	movf	FREQ_SS4,w
	call	RUN_PWMX
	movwf	PWM_X4		; processed PWM_X4 frequency value
    incf    PWM_X4,f    ; +1
	goto	RUN_BURSTS

RUN_PWMX:	; common code sequence
	movwf	AARGB1			; ms byte is cleared in the calculate routine
	movlw	55              ; was 255
	movwf	DENOM			; denominator for divider		
	call	CALCULATE
	return

RUN_BURSTS:

;1 frequency
	movf	FREQ_SS1,w
	movwf	FREQ_SS
; burst length
	movf	BURST_SS1,w
	movwf	BURST_SS
; volume (duty cycle)
	movf	PWM_X1,w
	movwf	PWM_X		; sets volume PWM duty cycle
; soft start/end
	clrf	SOFT		; no soft start and soft end
	bsf		SOFT,0		; soft start when set
	call	GENERATOR_LOOP

;2 frequency
	movf	FREQ_SS2,w
	movwf	FREQ_SS
; burst length
	movf	BURST_SS2,w
	movwf	BURST_SS
; volume (duty cycle)
	movf	PWM_X2,w
	movwf	PWM_X		; sets volume PWM duty cycle
; soft start/end
	clrf	SOFT		; no soft start and soft end
	call	GENERATOR_LOOP

;3 frequency
	movf	FREQ_SS3,w
	movwf	FREQ_SS
; burst length
	movf	BURST_SS3,w
	movwf	BURST_SS
; volume (duty cycle)
	movf	PWM_X3,w
	movwf	PWM_X		; sets volume PWM duty cycle
; soft start/end
	clrf	SOFT		; no soft start and soft end
	call	GENERATOR_LOOP

;4 frequency
	movf	FREQ_SS4,w
	movwf	FREQ_SS
; burst length
	movf	BURST_SS4,w
	movwf	BURST_SS
; volume (duty cycle)
	movf	PWM_X4,w
	movwf	PWM_X		; sets volume PWM duty cycle
; soft start/end
	clrf	SOFT		; no soft start and soft end
	bsf		SOFT,1		; soft end when set
	call	GENERATOR_LOOP

GAP: ; Mouse sounds end if Mouse count reduces to 0 or there are noises detected
; run for so many Mouse sounds before sleep with MOUSE_COUNTSms,ls counter decrement to 0
    decfsz  MOUSE_COUNTSls,f    ; decrease to 0  mouse noises duration 
    goto    CONT_GAP
    decfsz  MOUSE_COUNTSms,f    ; when ls byte is 0 then decrease ms byte
    goto    CONT_GAP
    goto    SHUT_DOWN           ; when MS byte is zero end mouse sounds
    
CONT_GAP:    

; check for noises within sound gap. Any noise: stop and sleep
; if so go to SHUT_DOWN 
   
; wait a period first so piezo noises are off and MEMS microphone is not responding to generated sound
; flash LED for a bit
    bsf     LATA,2          ; LED on
    movlw   1
    call    DEL_SET    ;    ; 4ms
    bcf     LATA,2          ; LED off
; then delay for MEMS supply to settle    
    movlw   8
    call    DEL_SET    ;    ; 32ms
    
    movlw   16
    movwf   AD_RUNS         ; 16ms;  number of 1ms A/D runs in sound gaps used as a timer
 AD1:
    call    AD_RUN          ; 1ms to get microphone output 
	movf    SOUND,w         ; microphone sound
    subwf   LEVEL,w         ; if over 
    btfss   STATUS,C        ; Carry = 0 if sound >
	goto	SHUT_DOWNX 
DEC_AD1:
    decfsz  AD_RUNS,f
    goto    AD1 
   
; random delay    
	incf	RNDM_COUNT,f
	bcf		RNDM_COUNT,7	; 0-127 max
	call	RANDOM			; random number call
    andlw   00001111B       ; set maximum limit of 16
    movwf   AD_RUNS         ; number of A/D checks
AD2:
    call    AD_RUN          ; 1ms to get microphone output 
	movf    SOUND,w         ; microphone sound
    subwf   LEVEL,w         ; if over 
    btfss   STATUS,C        ; Carry = 0 if sound >
	goto	SHUT_DOWNY  
 DEC_AD2:
    decfsz  AD_RUNS,f
    goto    AD2 
	goto	FREQUENCY_SETUP ; keep up loop of sound generation

SHUT_DOWNX:
    decfsz  CHANCES,f         ; number of noise transients allowed reduced
    goto    DEC_AD1           ; has chances of avoiding shutdown with counter  
    goto    SHUT_DOWN         ; shut down  
SHUT_DOWNY:
    decfsz  CHANCES,f         ; number of noise transients allowed reduced
    goto    DEC_AD2           ; has chances of avoiding shutdown with counter  
    goto    SHUT_DOWN         ; shut down  
    
;[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[    

CALCULATE:
;
; load AARGBx and BARGBx values
	clrf	AARGB0		; ms bytes clear
	clrf	BARGB0
	call	FXM1616U	; multiply 

; shift answer for divide numerator
	movf	AARGB2,w	; second ls byte of result
	movwf	STO1
	movf	AARGB3,w
	movwf	AARGB2
	movf	AARGB1,w
	movwf	AARGB0
	movf	STO1,w
	movwf	AARGB1
; load denominator for divide
	movf	DENOM,w
	movwf	BARGB1
	clrf	BARGB0

	call	FXD2416U	; divide 
	movf	AARGB2,w
	btfsc	STATUS,Z	; if zero use 1
	incf	AARGB2,w
	return

;[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[	
GENERATOR_LOOP: 

; if soft start then start PWM at 1
	movf	PWM_X,w		; use PWM_X if no soft start
	btfsc	SOFT,0
	movlw	1
	movwf	PWM			; ramp up from 1 to PWM_X
GENERATOR_CONT:; continue
	movf	PWM,w
	movwf	WIDTH
	movwf	WIDTH2

	movf	FREQ_SS,w		; frequency value 
	movwf	STORE1

    call    ONE_ZERO        ; set bit 1, clear bit 0 
 
DEC5:

; on for PWM cycles	
	movf	WIDTH,w
	btfsc	STATUS,Z		; when zero clear PORTA
 	call    OFF_ONE_ZERO    ; clear bit 0 and 1
	decf	WIDTH,f			; decrement width
	decfsz	STORE1,f
	goto	DEC5
	movf	FREQ_SS,w		;  frequency value
	movwf	STORE1
    call    ZERO_ONE        ; set bit 0, clear bit 1

DEC6:
	
; on for PWM cycles	
	movf	WIDTH2,w
	btfsc	STATUS,Z		; when zero clear GPIO
	call    OFF_ONE_ZERO    ; clear bit 0 and 1
 	decf	WIDTH2,f		; decrement width
	decfsz	STORE1,f
	goto	DEC6
	btfss	SOFT,0			; if bit 0 is set then increase PWM for soft start
	goto	LOOPz
	movf	PWM,w
	subwf	PWM_X,w
	btfsc	STATUS,C
	incf	PWM,f			; only increase if PWM < PWM_X (final duty cycle)
LOOPz:
	decfsz	BURST_SS,f		; kHz burst length
	goto	GENERATOR_CONT 

; GENERATOR_END ; pwm ramp down
; if soft end flag is set, then reduce PWM duty to 0
	btfss	SOFT,1			; end with soft reduction of PWM duty cycle if bit 1 is set
	return
GENERATOR_CONT_DOWN: ;continue with PWM soft ending
	movf	PWM,w			; use last PWM value at start and decrease over time
	movwf	WIDTH
	movwf	WIDTH2

	movf	FREQ_SS,w		; frequency value 
	movwf	STORE1

    call    ONE_ZERO        ; set bit 1 and clear bit 0

DEC1_DN:

; on for PWM cycles	
	movf	WIDTH,w
	btfsc	STATUS,Z		; when zero clear GPIO
	call    OFF_ONE_ZERO    ; clear bit 0 and 1
	decf	WIDTH,f			; decrement width
	decfsz	STORE1,f
	goto	DEC1_DN
	movf	FREQ_SS,w		;  frequency value
	movwf	STORE1
    call    ZERO_ONE        ; set bit 0, clear bit 1

DEC2_DN:
	
; on for PWM cycles	
	movf	WIDTH2,w
	btfsc	STATUS,Z		; when zero clear GPIO
	call    OFF_ONE_ZERO    ; clear bit 0 and 1
	decf	WIDTH2,f		; decrement width
	decfsz	STORE1,f
	goto	DEC2_DN
; soft ending  
	movf	PWM,w
	btfsc	STATUS,Z
	return
	sublw	1
	btfsc	STATUS,C
	return
	decfsz	PWM,f			; only decrease if PWM > 1 
	goto	GENERATOR_CONT_DOWN 
	return

OFF_ONE_ZERO: ;Both bit0 and 1 off
    movf    LATA,w    
    andlw	11111100B		; clear PORTA,0  PORTA,1
	movwf	LATA
    return     
    
ONE_ZERO:                   ; set bit 1, clear bit 0    
    movf    LATA,w
	andlw	11111110B		; clear PORTA,0
	iorlw	00000010B		; set PORTA,1
	movwf	LATA
    return    

ZERO_ONE:                   ; set bit 0, clear bit 1
	movf	LATA,w
	andlw	11111101B		; clear PORTA,1
	iorlw	00000001B		; set PORTA,0
	movwf	LATA
    return
    
 ;]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]

; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U:       CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416A:     RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA:       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LA: 		RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8:       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L8:        RLF             AARGB1, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416B:     RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB:       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LB:        RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16:      ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L16:        RLF             AARGB2, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416C:     RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC:       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LC: 		RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L:			RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U:        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   8
                MOVWF   LOOPCOUNT
LOOPUM1616A:     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B:     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0
BLUM1616NAP:    BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP:    BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616:    RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA:     RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  8
                MOVWF  LOOPCOUNT
BLOOPUM1616:    RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA:     RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN

  
; *********************************************
; subroutines
    
; A/D of MEMs microphone
AD_RUN:
    movlw	00101000B	; outputs/inputs set AN5 an input for A/D
    BANKSEL TRISA
	movwf	TRISA		; port data direction register
    BANKSEL ADCON0
    bsf     ADCON0,0    ; ADC ON
    call    DELAY_FLASH ; 500us Acquisition delay
    BANKSEL ADCON0
    bsf     ADCON0,1    ; Start conversion GO set
    btfsc   ADCON0,1    ; Is conversion done?
    goto    $-1         ; if no, test again
    BANKSEL ADRESH
    movf    ADRESL,w
    movwf   SOUNDLS     ; lower byte
    movf    ADRESH,W    ; Read upper byte
    movwf   SOUND       ; Store in register "SOUND"
    BANKSEL PORTA  
        
; process for more sensitivity (ie using 9th bit)ms bit will be lost but added in
; to maintain correct detection threshold    
; if ms bit of SOUND is set, set again after left shift so it will remain as a high value
; we are not after the A/D value so much as just whether the level is over a threshold value
; called 'LEVEL' which is between 4 and 13. LED display shows 1 to 10
; move in 9th bit
    lslf    SOUNDLS,w
    rlf     SOUND,f
    btfsc   STATUS,C
    bsf     SOUND,7
    return
    
DEL_SET: ; Delay  
ACK2:
    movwf	FLASH
ACK:
   	call	DELAY_FLASH
	decfsz	FLASH,f         ; 
	goto	ACK
 	return

; delay loop 

DELAYms: ; 10ms for D23, 435us/1
	movlw	23          ; delay value
DELAYX:
 	movwf	STORE1		; STORE1 is number of loops value
LOOP8:	
	movlw	117
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9:
	
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8
	return

DELAY_FLASH: ; 5ms
    BANKSEL PORTA
  	movlw	100         ; delay value
HOLD:    
	movwf	STORE1		; STORE1 is number of loops value
LOOP10:	
	movlw	11
	movwf	STORE2		; STORE2 is internal loop value	
LOOP11:
	
	decfsz	STORE2,f
	goto	LOOP11
	decfsz	STORE1,f
	goto	LOOP10
	return
    
DELAYZ: ; requires timer value in w
    movwf	STORE1		; STORE1 is number of loops value
	
    incf	RNDM_COUNT,w
    andlw	00111111B	; 64 max
	movwf	RNDM_COUNT	; 
	call	RANDOM	
 	movwf	VALUE	
LOOP8C:	
	movf	VALUE,w		; average D'117'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9C:
	decfsz	STORE2,f
	goto	LOOP9C
	decfsz	STORE1,f
	goto	LOOP8C
	return
    
; read data memory
READ:
; This code block will read 1 word of program
; data will be returned in w for ls byte

	BANKSEL NVMADRL 		; Select Bank for NVMCON registers
    movlw   0F0H    
	movwf 	NVMADRL 		; Store LSB of address
	movlw 	0FH 			; PROG_ADDR_HI ;
	movwf 	NVMADRH 		; Store MSB of address
	bcf 	NVMCON1,NVMREGS ; Do not select Configuration Space
	bsf 	NVMCON1,RD      ; Initiate read
    movf 	NVMDATL,W 		; Get LSB of word
	BANKSEL PORTA           ; bank 0
	return

FLASH_WRITE:

; Initially, erase the row address
    BANKSEL NVMADRL
	movlw	0F0H
	movwf 	NVMADRL 		; Load lower 8 bits of erase address boundary
	movlw	0FH
	movwf	NVMADRH 		; Load upper 6 bits of erase address boundary
	bcf 	NVMCON1,NVMREGS ; Choose PFM memory area
	bsf		NVMCON1,FREE    ; Specify an erase operation
	call	UNLOCK_SEQ
	bcf		NVMCON1,WREN    ; Disable writes
    
; write routine 
	movlw	0FH
	movwf	NVMADRH 		; Load initial flash address
	movlw	0F0H
	movwf	NVMADRL
	movlw	000H 			; Load initial low byte data address
	movwf	FSR0L		
	movlw	20H             ; high byte address
	movwf	FSR0H
	bcf		NVMCON1,NVMREGS ; Set Program Flash Memory as write location
	bsf		NVMCON1,WREN    ; Enable writes
	bsf		NVMCON1,LWLO    ; Load only write latches
LOOP:
	moviw	FSR0++
	movwf	NVMDATL 		; Load first data byte
	movlw   03FH            ; 
    movwf	NVMDATH 		; Load second data byte
	movf	NVMADRL,W
	xorlw	1H              ; Check if lower bits of address are 00000
	andlw	1H              ; and if on last of 32 addresses
	btfsc	STATUS,Z 		; Last of 32 words?
	goto	START_WRITE 	; If so, go write latches into memory
	call	UNLOCK_SEQ 		; If not, go load latch
	incf	NVMADRL,F 		; Increment address
	goto	LOOP
START_WRITE:
	bcf		NVMCON1,LWLO    ; Latch writes complete, now write memory
 	call	UNLOCK_SEQ 		; Perform required unlock sequence
	bcf		NVMCON1,WREN    ; Disable writes
	BANKSEL PORTA           ; bank 0
	return

UNLOCK_SEQ:
    bsf		NVMCON1,WREN    ; enable writes
	movlw	55H
	movwf	NVMCON2 		; Begin unlock sequence
	movlw	0AAH
	movwf	NVMCON2
	bsf		NVMCON1,WR
	return


	end _START
