'-------------------------------------------------------------------------------
' ATtiny45 controlled AD9834 2m/146MHz DDS FM Signal Generator
' Author: Andrew Woodfield
' Date: February 2023
' File Name: AD9834_2mFMosc_Vxx
' Forked from: AD9834_Test12.bas
'
'-------------------------------------------------------------------------------
' Program Description:

' An ATtiny45 clocked by the internal 16MHz RC/PLL clock is used to control an
' AD9834 DDS (slightly) overclocked by an 80MHz external oscillator. The DDS
' alias or "Super-Nyquist" outputs (an Analog Devices term) around 146MHz are
' bandpass filtered to give outputs on spot frequencies across the 2m VHF band.
'
' The AD9834 powers up at 146.0MHz (Alias of 14.0MHz). This frequency can be
' changed in 0.5MHz steps from 144.0 (16.0MHz) to 148.0 MHz (12.0 MHz) using
' one of three pushbuttons. (Note: Reverse tuning with the selected alias!)
'
' A Memory pushbutton allows sequential selection of one of four 2m band
' frequencies which may be programmed in the ATtiny85's EEPROM by the user.
' .
' FM is produced using the AD9834's phase register. The third pushbutton turns
' the 1kHz FM test tone modulation on and off. FM is generated from the 8kHz
' sampled audio on the ADC0 input (PB5). The 1kHz sinewave is produced by a
' single-pole RC LPF which filters a software-generated PWM square wave output
' on PB0 whose shape minimises 5th and 7th harmonics.
'
' Interrupts are used intensively for all functions including a pin-level
' interrupt to detect the three pushbuttons via PB3 and ADC3, the generation
' of the 1kHz modulation tone, and the 8kHz ADC sampling for FM modulation.
'
' ---------------------------------------------------------------------------------------------------------------
' Fuse settings
'
' LOCK Bit Byte : 0ffh (No locks)
' Fuse EXTd Byte: 0ffh (No self-programming enabled)
'
' Fuse HIGH Byte: 05fh
'   RSTDISBL 0 Ext Reset DISABLED!!
'       DWEN 1 DW not enabled
'      SPIEN 0 SPI programming enabled
'      WDTON 1 Watchdog timer off
'     EESAVE 1 EEPROM not preserved in erase
'  BODLEVEL2 1 BOD disabled
'  BODLEVEL1 1
'  BODLEVEL0 1
'
' Fuse LOW Byte:  0e1h
'     CKDIV8 1 NOT divided by 8
'      CKOUT 1 Not enabled
'       SUT1 1 Slow rising power
'       SUT0 0
'     CKSEL3 0 16MHz internal RC clock with PLL multiplier
'     CKSEL2 0
'     CKSEL1 0
'     CKSEL0 1
'
'---------------------------------------------------------------------------------------------------------------
' Compiler Directives (some fundamental hardware issues)

$regfile = "attiny45.dat"
$crystal = 16000000

'---------------------------------------------------------------------------------------------------------------
' Hardware Setups
' set up direction of all ports

Config Portb.4 = Output                                     'SS ----> SCK (Slave)
Config Portb.2 = Output                                     'USCK ----> SCK (Slave)
Config Portb.1 = Output                                     'DO   ----> SDI (Slave)
Config Portb.0 = Output                                     '***** was Input   DI   ----> SDO (Slave) (Not used)
                                                            'Note! PB0 produces 1kHz squarewave as an output
Config portb.3 = input                                      'Freq and Mod pushbuttons

config pinb.5 = input                                       'adc0 for audio input

Set Portb.4                                                 'SS idles high
Set Portb.2                                                 'SCLK idles/starts high

' Hardware Aliases:

'none asigned

'------------------------------------------------------------------
' Declare Variables

Dim Frq As Long                                             'DDS data for stored frequency
Dim W1 as word at frq overlay                               'mirror variables for dds shift/write
Dim W2 as word at frq + 2 overlay                           'upper 16 bits

Dim regbuffer as word                                       'buffer for setting up send data with register address
dim RB1 as byte at regbuffer overlay                        'ls byte
dim RB2 as byte at regbuffer + 1 overlay                    'ms byte

Dim Frequency As Dword                                      'actual dds frequency including IF and multiplier
Dim DDSfreq as dword                                        'buffer for frequency value sent to DDS update routine
dim freqkeyused as bit                                      'flag to show Frequency key was used (0=not used 1=used)

Dim update as bit                                           'flags dds update
dim modflag as bit                                          'starts and stops modulation test

dim sddstemp as single                                      'single SIGNED 32 bit integer for dds value calc
dim Vaudio as word                                          'adc0 sampled 10-bit value for modulation
dim Vtemp as word                                           'temp 16 bit memory for modulation calculations
dim Vbutton as word                                         'adc3 sampled 10 bit value for pushbuttons
dim pwmstate as byte                                        '10-state counter for 1kHz PWM waveform

dim memorynumber as byte                                    'Points to one of four stored frequencies in EEPROM (Range: 1-4)
dim memkeyused as bit                                       'flag used to show Memory button was used (0=not used 1=used)

dim dummy1 as eram word                                     'reserve first two bytes in EEPROM (Not used)
dim eFREQ(4) as eram Dword                                  'reserve space for frequency data for channel 1 to 4 (unsigned 32-bit numbers)


' Initialise Variables
' ...Done in code below...

' Declare Constants
const ddsratio = 3.3554432                                  '2^28/80x10^6 for 80MHz clock
const stepsize = 500000                                     '500kHz steps

' Declare subroutines
Declare Sub CalcSendFreq                                    'convert and send freq data to DDS
Declare Sub Sendbuffer                                      'send 16 bit buffer data to DDS
'
'------------------------------------------------------------------
' Program starts here
'
Frequency = 14000000                                        'initialise frequency  (Generates desired alias at 146.0 MHz)
freqkeyused = 0                                             'freq key has not been pressed yet
update = 0                                                  'clear update flag
modflag = 0                                                 'modulation is off
memorynumber = 1                                            'first memory selected
memkeyused = 0                                              'memory button not used yet

'initialise ADC
admux = &b00000000                                          '10 bit result, adc0 selected, Vcc is reference
adcsra = &b11000101                                         '32 prescale - if fclk=16MHz then adc clk = 500kHz and adc 13-clk sample takes 26uS
'Note: adc has started...

'initialise the AD9834 and output the defined Frequency

regbuffer = &h2100                                          'configure and reset ad9834
sendbuffer

DDSfreq = frequency                                         'copy frequency for dds update routine
CalcSendFreq                                                'get startup dds value and send it to dds

regbuffer = &h2000                                          'turn on ad9834 output
sendbuffer

'get ready for modulation

Config Timer0 = Timer , Prescale = 8                        'Timer starts automatically so...
Stop Timer0
timer0 = &H06                                               'load value for 125uS interrupt time (8000 samples per second)
On Ovf0 modupdate                                           'define where interrupt goes
Enable Timer0                                               'enable the timer0 interrupt

'and set timer 1 for cycle-variant 1kHz PWM output on PB0
Config Timer1 = Timer , Prescale = 32                       'Timer 1 starts NOW!!!
timer1 = &HE2                                               'load value for first 61uS interrupt time
pwmstate = 0                                                'pwm 10-state counter
On Ovf1 tonetoggle                                          'define where interrupt goes
Enable Timer1                                               'enable the timer1 interrupt


Pcmsk = &B00001000                                          'enable pin change interrupt pcint0 for pinb.3  (pushbuttons)
On Pcint0 read_user_buttons                                 'vector for pin change
Enable Pcint0                                               'enable pin change interrupt


Enable Interrupts                                           'allow interrupts to occur
                                                              'but timer 0 is NOT running yet

Do

'NOTHING!

Loop


modupdate:                                                  'interrupt routine for frequency modulation

   timer0 = &H06                                            'reload value for 125uS interrupt time (8000 samples per second)
   Vaudio = adc                                             'read and save current adc register for audio level
   if Vaudio < 141 then Vaudio = 140                        'test for zero input
   Vaudio = Vaudio - 140                                    'oscillator swings from 0.7 - 4.36V
   Vtemp = Vaudio                                           'and save another copy
   admux = &b00000000                                       'select adc0
   adcsra = &b11000101                                      'and restart next sample
   shift Vaudio , left                                      '2*Vadc
   Vaudio = Vaudio + Vtemp                                  '3*Vadc
   Vaudio = Vaudio or &hC000                                'set upper two bits for PHASE0 register write
   regbuffer = Vaudio                                       'copy shifted ADC value to buffer for write

   sendbuffer                                               'write to AD9834 phase register 0

Return



tonetoggle:                                                 'interrupt routine for 1kHz square wave
   incr pwmstate                                            'point to next state
   if pwmstate = 10 then pwmstate = 0                       'and reset for 10 state generation
   timer1 = lookup(pwmstate , tonetime)                     'reload value for next interrupt time
   toggle portb.0                                           'just change output
Return

tonetime:
Data &He2 , &Hf4 , &H5c , &Hf4 , &He2 , &He2 , &Hf4 , &H5c , &Hf4 , &He2


read_user_buttons:                                          'pin interrupt routine for Freq select and Mod on/off buttons

   'here since pin change interrupt was detected. ADC3=0 if Mod button pressed. ADC3=287 if Freq button pressed

   disable interrupts

   Vbutton = getadc(3)
   Vbutton = getadc(3)                                      'read input twice to be sure we are reading adc3

   'restart read of ADC0 for modupdate
   admux = &b00000000                                       '10 bit result, adc0 selected, Vcc is reference
   adcsra = &b11000101                                      '32 prescale - if fclk=16MHz then adc clk = 500kHz and adc 13-clk sample takes 26uS

   if vbutton < 90 then                                     'mod button pressed so toggle mod on / off flag
      stop timer0                                           'just wait while we sort this out
      toggle modflag                                        'downkey toggles modulation on and off (via flag)
      if modflag = 1 then
         start timer0                                       'start modulation
      else                                                  ' modflag = 0 so...
         stop timer0                                        'stop the modulation
      end if
   end if


   if vbutton > 140 and vbutton < 180 then                  'memory button pressed so load next memory channel and flag freq change
      if modflag = 1 then stop timer0                       'stop modulation during frequency updating if it is running
      update = 1                                            'flag update of dds is required
      if freqkeyused = 0 then
         incr memorynumber                                  'select the next memory
         if memorynumber = 5 then memorynumber = 1          'rollover as required
        else
         freqkeyused = 0                                    'reset flag for next time and retain selected memory
      end if
      DDSfreq = eFREQ(memorynumber)                         'get EEPROM frequency for dds frequency update
      memkeyused = 1                                        'show memory channels are being used
      if modflag = 1 then start timer0                      'start modulation during frequency updating if it is running
   end if


   if vbutton > 250 and vbutton < 320 then                  'freq button pressed so calculate and flag freq change
      if modflag = 1 then stop timer0                       'stop modulation during frequency updating if it is running
      update = 1                                            'flag update of dds is required
      if memkeyused = 0 then
         Frequency = frequency - Stepsize
         if frequency < 12000000 then frequency = 16000000  'roll around 16 to 12MHz
        else                                                'DON'T shift frequency on this occasion
         memkeyused = 0                                     'reset flag for next time
      end if
      DDSfreq = frequency                                   'value for dds update routine
      freqkeyused = 1                                       'show frequency steps are being used
      if modflag = 1 then start timer0                      'start modulation during frequency updating if it is running
   end if


   If update = 1 Then                                       'flag indicates update of carrier frequency is required
      if modflag = 1 then stop timer0                       'make sure we are not interrupted
      CalcSendFreq                                          'get new dds value and send it to dds
      if modflag = 1 then start timer0                      'restart timer if required
      update = 0                                            'reset the flag for next time
   End If

   while pinb.3 = 0                                         'if the pin reads as a logic 0 then wait for user to release the button
      waitms 10
   wend

   enable interrupts

Return




'**********************************************************************
'   Subroutines
'**********************************************************************
'
'**********************************************************************
'    Calc DDS data and send to DDS
'    Subroutine takes approx 64uS with 16MHz uP clock
'**********************************************************************
Sub CalcSendFreq

sddstemp = DDSfreq * ddsratio                               'calc signed dds value
frq = sddstemp                                              'and convert to unsigned integer for dds

   'FRQ now holds 28 bit DDS data which must be bit-shifted for two-bit register address integration

Shift frq , left , 2                                        'shift data for register addr integration

regbuffer = w1                                              'get lower 16 bits (actually 14 bits) of FRQ
Shift regbuffer , right , 2                                 'restore 14 bits into correct place
regbuffer = regbuffer or &h4000                             'set upper two bits for FREQ0 register write (01)

sendbuffer                                                  'and send the 16 bits

regbuffer = w2                                              'get upper 16 bits (actually 14 bits) of FRQ
regbuffer = regbuffer or &h4000                             'set upper two bits for FREQ0 register write

sendbuffer                                                  'send 16 bits - no additional shift required

End Sub

'**********************************************************************
'   Send 16 bits of data in regbuffer to DDS via SPI
'   Subroutine takes approx 4uS with 16MHz uP clock
'**********************************************************************
'sclk and fsync start high
Sub Sendbuffer

        set portB.2                                         'SCLK idle state
        r16 = RB2                                           'get byte
$ASM
        cbi portb, 4                                        'SS on

        'send the byte using fast SPI transfer

        out USIDR , r16                                     'load SPI data register with byte to send
        LDI R16,&b00010001                                  'USIWM0=1, USICS0=0. USITC=1 (3 wire mode, no clock, toggle the clock)
        LDI R17,&b00010011                                  'USIWM0=1, USICS0=0. USITC=1, USICLK=1 (As above & strobe the data reg)
        out USICR , r16                                     'clock out MSB
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16                                     'clock out LSB
        out USICR , r17
        in  r16,USIDR                                       'read incoming data (but don't care)

$END ASM

        r16 = RB1                                           'get next byte
$ASM
        'send the byte using fast SPI transfer

        out USIDR , r16                                     'load SPI data register with byte to send
        LDI R16,&b00010001                                  'USIWM0=1, USICS0=0. USITC=1 (3 wire mode, no clock, toggle the clock)
        LDI R17,&b00010011                                  'USIWM0=1, USICS0=0. USITC=1, USICLK=1 (As above & strobe the data reg)
        out USICR , r16                                     'clock out MSB
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16
        out USICR , r17
        out USICR , r16                                     'clock out LSB
        out USICR , r17
        in  r16,USIDR                                       'read incoming data (but don't care)

        sbi portb, 4                                        'SS off

$END ASM

End Sub

'*************************************************************************