
//#define _DEBUG/////////////!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

typedef struct { float upper; float lower; float gain; } adiv;
typedef struct { float minval; float maxval; signed long delaysec; } alimit;
typedef struct { unsigned char state; signed long delaysec; } dlimit;

// Phone number to send alerts to:
char          SMS_Destination[32]     = "+61425256551";
// PIN number for the SIM card, if used:
char          SMS_PIN[5]              = "0000";
// SMS message centre number for your carrier:
char          SMS_MSG_CENTRE_NUM[32]  = "+61415011501";
// SMS command password, must be contained in a received SMS for any commands to work
char          SMS_PASSWORD[]          = "simon says";
// Send an SMS once a day (ie, 24 hours x 60 minutes x 60 seconds, 0 = off):
unsigned long SMS_send_interval       = 60*60*24;
// Never send an SMS more than once every half hour:
unsigned long SMS_min_send_interval   = 30*60;
// Check for SMS reception every half hour (0 = off)
unsigned long SMS_recv_check_interval = 60;//30*60;/////////////////////////////////////////////////////
// How long to suppress messages for after a STOP command is received (default eight hours)
unsigned long SMS_suppress_time       = 8*60*60;

alimit alimits[] = { /* analog input configuration */
 { /* min voltage */  5.5, /* max voltage */  6.9, /* delay (sec) */ 30/*15*60*/ }, // battery voltage
 { /* min voltage */  0.0, /* max voltage */ 16.0, /* delay (sec) */ 15*60 }, // input A1
 { /* min voltage */  0.0, /* max voltage */ 16.0, /* delay (sec) */ 15*60 }, // input A2
 { /* min voltage */  0.0, /* max voltage */ 16.0, /* delay (sec) */ 15*60 }, // input A3
 { /* min voltage */  0.0, /* max voltage */ 16.0, /* delay (sec) */ 15*60 }, // input A4
 { /* min voltage */  0.0, /* max voltage */ 16.0, /* delay (sec) */ 15*60 }  // input A5
};

dlimit dlimits[] = {
 { /* expected state */ 1, /* delay (sec) */ 15*60 }, // input D1
 { /* expected state */ 1, /* delay (sec) */ 15*60 }, // input D2
 { /* expected state */ 1, /* delay (sec) */ 15*60 }, // input D3
 { /* expected state */ 1, /* delay (sec) */ 15*60 }, // input D4
 { /* expected state */ 1, /* delay (sec) */ 15*60 }  // input D5
};

adiv adivs[] = { /* analog input dividers */
 { /* upper resistor (kOhms) */ 10.0, /* lower resistor (kOhms) */ 10.0, /* gain */ 1.0 }, // battery voltage divider
 { /* upper resistor (kOhms) */ 22.0, /* lower resistor (kOhms) */ 10.0, /* gain */ 1.0 }, // input A1 divider
 { /* upper resistor (kOhms) */ 22.0, /* lower resistor (kOhms) */ 10.0, /* gain */ 1.0 }, // input A2 divider
 { /* upper resistor (kOhms) */ 22.0, /* lower resistor (kOhms) */ 10.0, /* gain */ 1.0 }, // input A3 divider
 { /* upper resistor (kOhms) */ 22.0, /* lower resistor (kOhms) */ 10.0, /* gain */ 1.0 }, // input A4 divider
 { /* upper resistor (kOhms) */ 22.0, /* lower resistor (kOhms) */ 10.0, /* gain */ 1.0 }  // input A5 divider
};

float Low_Battery_Level = 5.0; // do not flash LED or send SMS with battery below this voltage

#define LED_PIN        13
#define INHIBIT_PIN    12
#define GPRS_POWER_PIN 9
#define MAX_GSM_TIME   60 // allow up to one minute to acquire network
#define MSG_WAIT_TIME  60 // wait this long after network acquired before checking for messages

//////////////////////////////////////////////////////////////////////////////////////////////////////

#include <avr/sleep.h>
#include <avr/power.h>
#include <avr/wdt.h>

// Latest analog input values
unsigned short ADCvals[6];
// Accumulator to keep track of time passing despite changing Watchdog Timer frequency.
unsigned short WatchDogAccum;
// Flag to indicate when watchdog timer has gone off and time has passed (value = how many seconds).
volatile unsigned char WatchDogFired;
// Used to scan through ADC/digital inputs.
unsigned char ADCinput;
// Used to keep track of how many times inputs have been sampled for averaging.
unsigned char ADCcounter;
// Bitwise state of digital input pins.
unsigned char DigitalState;
// How long has passed since last update (copied from WatchDogFired, so that it isn't overwritten during a long operation).
unsigned char TimePassed;
// Set when the LED is to flash, keeps track of how many half-flashes remain.
unsigned char LEDFlashes;
// State machine variable for GPRS board.
unsigned char GPRSState;
// Battery low flag.
unsigned char BatteryIsLow;
// How long each input has been outside its normal range in seconds.
unsigned long AlertDuration[6+5];
// Value of above variable last time it was reset
unsigned long LastAlertDuration[6+5];
// Used to prevent rapid-fire SMS sending (value in seconds).
unsigned long TimeSinceLastSMS;
// Starts at zero, accumulates time, reset if/when periodic status update is sent.
unsigned long Uptime;
// Set if a suppress message is received, counts down to zero and then normal sending resumes.
unsigned long SuppressTime;
// How long since the GPRS receiver has been powered up.
unsigned long SMSCheckTime;
// 1 = send message when GPRS module ready, 0 = just checking for incoming messages
unsigned char GPRSSendMessage;


// Watchdog timer interrupt handler. Wakes micro up periodically to perform tasks.
// We keep track of how long has passed since the last interrupt, which depends on the divider setting.
ISR(WDT_vect) {
  // Set WatchDogAccum to how many seconds have passed since last Watchdog Timer interrupt.
  WatchDogAccum += ((unsigned short)1)<<( (WDTCSR&7) | ((WDTCSR>>2)&8) );
  WatchDogFired += WatchDogAccum >> 6;
  WatchDogAccum &= 0x3F;
}

// This function puts the processor to sleep, to be woken up by the watchdog interrupt.
// It returns once the interrupt completes.
void GoToSleep(void) {
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);
  sleep_enable();
  sleep_mode();
  sleep_disable();
  power_all_enable();
}

// This is a copy of the SoftwareSerial class.
// We're including it in thie file because the built-in one has too
// small a buffer (just 64 bytes) and we don't want the user to have
// to modify system libraries to fix it. This class uses a bigger buffer.
// We have enough RAM for 256 which should do the job.
#define _SSBB_MAX_RX_BUFF 256
class SoftwareSerialBigBuf : public Stream
{
private:
  // per object data
  uint8_t _receivePin;
  uint8_t _receiveBitMask;
  volatile uint8_t *_receivePortRegister;
  uint8_t _transmitBitMask;
  volatile uint8_t *_transmitPortRegister;

  uint16_t _rx_delay_centering;
  uint16_t _rx_delay_intrabit;
  uint16_t _rx_delay_stopbit;
  uint16_t _tx_delay;

  uint16_t _buffer_overflow:1;
  uint16_t _inverse_logic:1;

  // static data
  static char _receive_buffer[_SSBB_MAX_RX_BUFF]; 
  static volatile uint8_t _receive_buffer_tail;
  static volatile uint8_t _receive_buffer_head;
  static SoftwareSerialBigBuf *active_object;

  // private methods
  void recv();
  uint8_t rx_pin_read();
  void tx_pin_write(uint8_t pin_state);
  void setTX(uint8_t transmitPin);
  void setRX(uint8_t receivePin);

  // private static method for timing
  static inline void tunedDelay(uint16_t delay);

public:
  // public methods
  SoftwareSerialBigBuf(uint8_t receivePin, uint8_t transmitPin, bool inverse_logic = false);
  ~SoftwareSerialBigBuf();
  void begin(long speed);
  bool listen();
  void end();
  bool isListening() { return this == active_object; }
  bool overflow() { bool ret = _buffer_overflow; _buffer_overflow = false; return ret; }
  int peek();

  virtual size_t write(uint8_t byte);
  virtual int read();
  virtual int available();
  virtual void flush();
  
  using Print::write;

  // public only for easy access by interrupt handlers
  static inline void handle_interrupt();
};




// Serial port for communicating with the GPRS module.
SoftwareSerialBigBuf GPRS(7, 8);

// This function runs when the micro is powered up.
void setup()
{
#ifdef _DEBUG
  Serial.begin(19200);
  Serial.println("Initialising...");
#endif
  // Set up the Watchdog Timer (WDT)
  MCUSR &= ~(1<<WDRF);                        // Clear watchdog reset flag.
  WDTCSR |= (1<<WDCE) | (1<<WDE);             // Enable watchdog and let us change the configuration.
  WDTCSR = (1<<WDP2) | (1<<WDP1) | _BV(WDIE); // Set it to generate an interrupt once per second.
  
  TimeSinceLastSMS = SMS_min_send_interval;
}

// This is used when flashing the LED, to get faster wake-ups.
void FastWDT() {
  MCUSR &= ~(1<<WDRF);                        // Clear watchdog reset flag.
  WDTCSR |= (1<<WDCE) | (1<<WDE);             // Enable watchdog and let us change the configuration.
  WDTCSR = (1<<WDP1) | (1<<WDP0) | _BV(WDIE); // Set it to generate an interrupt eight times per second.
}

// This is used when finished flashing the LED, to go back to the normal 1Hz interrupts.
void SlowWDT() {
  MCUSR &= ~(1<<WDRF);                        // Clear watchdog reset flag.
  WDTCSR |= (1<<WDCE) | (1<<WDE);             // Enable watchdog and let us change the configuration.
  WDTCSR = (1<<WDP2) | (1<<WDP1) | _BV(WDIE); // Set it to generate an interrupt once per second.
}

// Converts a number of seconds into a human-readable string, written into dest (eg, "3h15m").
// The pointer returned is at the end of this string.
char* printDuration(char* dest, unsigned long duration) {
  if( duration < 60 )
    return dest + sprintf(dest, "%ds", duration);
  if( duration < 3600 )
    return dest + sprintf(dest, "%dm%ds", (int)(duration / 60), (int)(duration % 60));
  duration /= 60;
  if( duration < 1440 )
    return dest + sprintf(dest, "%dh%dm", (int)(duration / 60), (int)(duration % 60));
  duration /= 60;
  return dest + sprintf(dest, "%dd%dh", (int)(duration / 24), (int)(duration % 24));
}

// Convert an ADC result to a voltage, based on the circuitry between the voltage source and that input.
float ADCvalToVoltage(unsigned char ADCinput) {
  return ADCvals[ADCinput] * 5 * ((adivs[ADCinput].lower ? adivs[ADCinput].upper / adivs[ADCinput].lower : 0) + 1) / (1024 * adivs[ADCinput].gain) + (ADCinput == 0 ? 0.75 : 0);
}

// Reads a series of characters from the software serial port until it gets a newline (which is not kept).
// maxms defines the number of milliseconds to wait, if this runs out or the buffer overflows, it returns 0.
unsigned short ReadGPRSData(char* buf, unsigned char maxbytes, unsigned short maxms, unsigned char ignoreblankline) {
  // Many AT command responses start with a blank line;
  // this makes it easy to skip it.
  if( ignoreblankline ) {
    maxms = ReadGPRSData(buf, maxbytes, maxms, 0);
    if( !maxms || buf[0] != '\0' )
      return 0;
  }

#ifdef _DEBUG
  char* origbuf = buf;
  Serial.print("R:");
#endif
  while(1) {
    if( GPRS.available() ) {
      buf[0] = GPRS.read();
      // ignore carriage returns
      if( buf[0] == '\r' )
        continue;
      // fail if we've run out of space
      if( --maxbytes == 0 ) {
#ifdef _DEBUG
        buf[0] = '\0';
        Serial.print("<overflow> (");
        Serial.print(origbuf);
        Serial.println(")");
#endif
        return 0;
      }
      // if we get a newline, we're finished
      if( buf[0] == '\n' ) {
        buf[0] = '\0';
#ifdef _DEBUG
        Serial.println(origbuf);
#endif
        return maxms;
      }
      ++buf;
    } else {
      // no data so wait 1ms and if this happens too many times, fail with a timeout
      if( maxms-- == 0 ) {
#ifdef _DEBUG
        Serial.println("<timeout>");
        if( buf != origbuf ) {
          buf[0] = '\0';
          Serial.print("(got \"");
          Serial.print(origbuf);
          Serial.println("\")");
        }
#endif
        return 0;
      }
      delay(1);
    }
  }
}

// Same as above but used to read a fixed number of characters when they are not terminated
// with a newline. Buffer must be at least maxbytes+1 large.
unsigned short ReadGPRSDataNoNL(char* buf, unsigned char maxbytes, unsigned short maxms, unsigned char ignoreblankline) {
  // Many AT command responses start with a blank line;
  // this makes it easy to skip it.
  if( ignoreblankline ) {
    maxms = ReadGPRSData(buf, maxbytes, maxms, 0);
    if( !maxms || buf[0] != '\0' )
      return 0;
  }

#ifdef _DEBUG
  char* origbuf = buf;
  Serial.print("Rn:");
#endif
  while(1) {
    if( GPRS.available() ) {
      buf[0] = GPRS.read();
      // if we get a carriage return or newline, that's a failure as we are not expecting them
      if( buf[0] == '\r' || buf[0] == '\n' ) {
#ifdef _DEBUG
        Serial.println("<newline>");
#endif
        return 0;
      }
      ++buf;
      // if we've run out of space, we're finished
      if( --maxbytes == 0 ) {
        buf[0] = '\0';
#ifdef _DEBUG
        Serial.println(origbuf);
#endif
        return 1;
      }
    } else {
      // no data so wait 1ms and if this happens too many times, fail with a timeout
      if( maxms-- == 0 ) {
#ifdef _DEBUG
        Serial.println("<timeout>");
        if( buf != origbuf ) {
          buf[0] = '\0';
          Serial.print("(got \"");
          Serial.print(origbuf);
          Serial.println("\")");
        }
#endif
        return 0;
      }
      delay(1);
    }
  }
}

// Get the serial port working after the GPRS module is powered up each time.
// Checks that it can get a response to an AT command before returning 1.
unsigned char SetupGPRS() {
  unsigned char tries = 3;
  char buf[16];

  GPRS.begin(19200);
  do {
    // clear any buffered incoming serial data
    while( GPRS.read() != -1 )
      ;
    // send an AT and check that we get the expected response
    GPRS.println("ATE0");
#ifdef _DEBUG
    Serial.println("S:ATE0");
#endif
    if( ReadGPRSData(buf, sizeof(buf), 100, 0) && !strcmp(buf, "ATE0") && ReadGPRSData(buf, sizeof(buf), 100, 1) && !strcmp(buf, "OK") )
      return 1;
    delay(10);
    // if not, try again a couple of times after a short delay
  } while( --tries );
  // didn't get a response so fail
  return 0;
}

// Reads the content of the oldest SMS stored in the SIM900 module.
unsigned char ReadSMS(char* buf, unsigned char maxbytes) {
  char temp[16], msgnum[8];
  char* ptr, * comma;

  // Send command to retrieve first stored SMS.
#ifdef _DEBUG
  Serial.println("S:AT+CMGL=\"ALL\"");
  Serial.flush();
#endif
  GPRS.println("AT+CMGL=\"ALL\"");
  // Check that we get a sensible header response (but we don't care what's in it).
  // Note that if there are no messages waiting, we'll get a CMS error 321 (invalid index)
  if( !ReadGPRSData(buf, maxbytes, 2000, 1) || strncmp(buf, "+CMGL: ", 7) )
    return 0;
  // store the message number
  comma = strstr(buf+7, ",");
  if( comma && comma-(buf+7) < 8 ) {
    strncpy(msgnum, buf+7, comma-(buf+7));
    msgnum[comma-(buf+7)] = '\0';
  } else {
    msgnum[0] = '1';
    msgnum[1] = '\0';
  }
  // Now read the message which appears on the next line(s), terminated with a blank line.
  // Separate the results with a newline in the response buffer.
  ptr = buf;
  while( 1 ) {
    if( maxbytes == 0 || !ReadGPRSData(ptr, maxbytes, 1000, 0) )
      return 0;
    if( ptr[0] == '\0' ) {
      ++ptr;
      --maxbytes;
      break;
    }
    while( ptr[0] != '\0' ) {
      ++ptr;
      --maxbytes;
    }
    if( maxbytes == 0 )
      return 0;
    ptr[0] = '\n';
    ++ptr;
    --maxbytes;
  }
#ifdef _DEBUG
  Serial.println("Got message OK.");
#endif
  // This should be followed by an OK, but it's possible there are other messages
  // so we will need to continue reading until we get an OK.
  if( maxbytes > 3 ) {
    while(1) {
      if( !ReadGPRSData(ptr, maxbytes, 1000, 0) )
        return 0;
      if( !strcmp(ptr, "OK") )
        break;
    }
  } else {
    if( !ReadGPRSData(temp, sizeof(temp), 100, 1) || strcmp(temp, "OK") )
      return 0;
  }
#ifdef _DEBUG
  Serial.print("S:AT+CMGD=");
  Serial.println(msgnum);
#endif
  GPRS.print("AT+CMGD=");
  GPRS.println(msgnum);
  if( !ReadGPRSData(temp, sizeof(temp), 2500, 1) || strcmp(temp, "OK") )
    return 0;
  return 1;
}

// Check to see if the SIM card needs a PIN set. If so, set it.
unsigned char CheckAndSetPIN() {
  char buf2[32];

  // First we must initialise the serial port.
  if( !SetupGPRS() )
    return 0;

  // Check to see whether the SIM needs a PIN.
  GPRS.println("AT+CPIN?");
#ifdef _DEBUG
  Serial.println("S:AT+CPIN?");
#endif
  if( !ReadGPRSData(buf2, 32, 2500, 1) )
    return 0;
  // If this is the response, it does...
  if( !strcmp(buf2, "+CPIN: SIM PIN") ) {
    if( !ReadGPRSData(buf2, 32, 100, 1) || strcmp(buf2, "OK") )
      return 0;

    // So send it and check that we get an "OK" in response.
#ifdef _DEBUG
    Serial.print("S:AT+CPIN=\"");
    Serial.print(SMS_PIN);
    Serial.println("\"");
#endif
    GPRS.print("AT+CPIN=\"");
    GPRS.print(SMS_PIN);
    GPRS.println("\"");
    if( !ReadGPRSData(buf2, 32, 2500, 1) || strcmp(buf2, "OK") )
      return 0;

    // Now check that the status has changed to "READY".
#ifdef _DEBUG
    Serial.println("S:AT+CPIN?");
#endif
    GPRS.println("AT+CPIN?");
    if( !ReadGPRSData(buf2, 32, 2500, 1) || strcmp(buf2, "+CPIN: READY") || !ReadGPRSData(buf2, 32, 100, 1) || strcmp(buf2, "OK") )
      return 0;
  } else if( !strcmp(buf2, "+CPIN: READY") ) {
    if( !ReadGPRSData(buf2, 32, 100, 1) || strcmp(buf2, "OK") )
      return 0;
  } else {
    // Unexpected response.
    return 0;
  }
  return 1;
}

// Function is self-explanatory (hopefully), returns 1 for success
unsigned char DisableMessageNotifications() {
  char buf2[32];

#ifdef _DEBUG
  Serial.println("S:AT+CNMI=0,0");
#endif
  GPRS.println("AT+CNMI=0,0");
  return ReadGPRSData(buf2, 32, 2500, 1) && !strcmp(buf2, "OK");
}

// Function is self-explanatory (hopefully), returns 1 for success
unsigned char SetMessageCentreNumber() {
  char buf2[32];

#ifdef _DEBUG
  Serial.println("S:AT+CSCA=\"+61415011501\"");
#endif
  GPRS.println("AT+CSCA=\"+61415011501\"");
  return ReadGPRSData(buf2, 32, 2000, 1) && !strcmp(buf2, "OK");
}

// Function is self-explanatory (hopefully), returns 1 for success
unsigned char SetTextMessageMode() {
  char buf2[33];

#ifdef _DEBUG
  Serial.println("S:AT+CMGF=1");
#endif
  GPRS.println("AT+CMGF=1");
  return ReadGPRSData(buf2, 32, 2000, 1) && !strcmp(buf2, "OK");
}

// Checks whether the SIM900 has connected to the network yet.
unsigned char IsConnectedToMobileNetwork() {
  char buf2[64], buf3[16];

#ifdef _DEBUG
  Serial.println("S:AT+CREG?");
#endif
  GPRS.println("AT+CREG?");
  if( ReadGPRSData(buf2, 64, 200, 1) && !strncmp(buf2, "+CREG: ", 7) && ReadGPRSData(buf3, 16, 200, 1) && !strcmp(buf3, "OK") ) {
    // typical response is +CREG: 2,1,”009A”,”2C3” with the "1" indicating connection to network
    char* comma = strstr(buf2+7, ",");
    if( comma ) {
      ++comma;
      while( *comma == ' ' )
        ++comma;
      return *comma == '1';
    } else {
      return 0;
    }
  } else {
    return 0;
  }
}

// Form a message containing the battery and input status, then if ActuallySend=1, send it.
// Otherwise simply dump it to the debug serial console.
unsigned char SendSMSAlert(unsigned char ActuallySend) {
  char buf[255];
  char* ptr = buf;
  float voltage = ADCvalToVoltage(0);
  // Start with the battery voltage.
  ptr += sprintf(ptr, "Batt: %d.%02dV", (int)voltage, (int)(voltage*100)%100);
  // If it's outside the allowed range, indicate for how long this has been the case.
  if( AlertDuration[0] > 0 ) {
    ptr += sprintf(ptr, "!");
    ptr = printDuration(ptr, AlertDuration[0]);
  } else {
    ptr += sprintf(ptr, " [OK]");
  }
  // Repeat same procedure as above for the five analog inputs.
  ptr += sprintf(ptr, " Analog:");
  for( ADCinput = 1; ADCinput <= 5; ++ADCinput ) {
     voltage = ADCvalToVoltage(ADCinput);
    ptr += sprintf(ptr, " %d.%02dV", (int)voltage, (int)(voltage*100)%100);
    if( AlertDuration[ADCinput] > 0 ) {
      ptr += sprintf(ptr, "!");
      ptr = printDuration(ptr, AlertDuration[ADCinput]);
    } else if( alimits[ADCinput].delaysec < 0 && LastAlertDuration[ADCinput] && LastAlertDuration[ADCinput] < -alimits[ADCinput].delaysec ) {
      ptr += sprintf(ptr, "!");
      ptr = printDuration(ptr, LastAlertDuration[ADCinput]);
    } else {
      ptr += sprintf(ptr, " [OK]");
    }
  }
  // Repeat same procedure as above for the five digital inputs, only these show 0/1 rather than a voltage.
  ptr += sprintf(ptr, " Digital:");
  for( ADCinput = 0; ADCinput <= 4; ++ADCinput ) {
    ptr += sprintf(ptr, " %d", (int)((DigitalState>>ADCinput)&1));
    if( AlertDuration[6+ADCinput] > 0 ) {
      ptr += sprintf(ptr, "!");
      ptr = printDuration(ptr, AlertDuration[6+ADCinput]);
    } else if( dlimits[ADCinput].delaysec < 0 && LastAlertDuration[ADCinput+6] && LastAlertDuration[ADCinput+6] < -dlimits[ADCinput].delaysec ) {
      ptr += sprintf(ptr, "!");
      ptr = printDuration(ptr, LastAlertDuration[6+ADCinput]);
    } else {
      ptr += sprintf(ptr, " [OK]");
    }
  }
  // Make sure we haven't exceeded the 160 character SMS limit (should not happen).
  if( ptr > buf + 160 )
    ptr = buf + 160;
  ptr[0] = '\0';
  if( ActuallySend ) {
    char buf2[32];
    char failed = 0;

    // Now to actually send the SMS.
    // Start with the destination phone number.
#ifdef _DEBUG
    Serial.print("S:AT+CMGS=\"");
    Serial.print(SMS_Destination);
    Serial.println("\"");
#endif
    GPRS.print("AT+CMGS=\"");
    GPRS.print(SMS_Destination);
    GPRS.println("\"");
    // We expect a message prompt in response
    if( ReadGPRSDataNoNL(buf2, 2, 5000, 1) && !strcmp(buf2, "> ") ) {
      unsigned char len;

      // Now send it.
#define MAX_SEND_CHUNK 200
      len = ptr - buf;
      ptr = buf;
      while(len) {
#ifdef _DEBUG
        Serial.print("Ss:");
        Serial.write((const uint8_t*)ptr, len > MAX_SEND_CHUNK ? MAX_SEND_CHUNK : len);
        Serial.write('\r');
        Serial.write('\n');
#endif
        GPRS.write((const uint8_t*)ptr, len > MAX_SEND_CHUNK ? MAX_SEND_CHUNK : len);
        GPRS.write('\r');
        GPRS.write('\n');
        ptr += len > MAX_SEND_CHUNK ? MAX_SEND_CHUNK : len;
        len -= len > MAX_SEND_CHUNK ? MAX_SEND_CHUNK : len;
        if( !ReadGPRSDataNoNL(buf2, 2, 2500, 1) || strcmp(buf2, "> ") ) {
          failed = 1;
          break;
        }
      }
      if( !failed ) {
        // send end-of-message tag
#ifdef _DEBUG
        Serial.println("S:\\032\\r\\n");
#endif
        GPRS.println("\032\r");
        // Check that we got a valid response.
        if( ReadGPRSData(buf2, 32, 5000, 1) && !strncmp(buf2, "+CMGS: ", 7) && ReadGPRSData(buf2, 32, 100, 1) && !strcmp(buf2, "OK") ) {
          TimeSinceLastSMS = 0;
          // only send alerts for too-short event periods once
          memset(LastAlertDuration, 0, sizeof(LastAlertDuration));
        } else {
          failed = 1;
        }
      } else {
        failed = 1;
      }
    } else {
      failed = 1;
    }
    // Return 1 for success.
    return !failed;
  } else {
    Serial.println(buf);
    Serial.flush();
    return 1;
  }
}

// Check the status of switch S2. We don't leave the pull-up enabled to save power.
unsigned char CheckInhibitSwitch() {
  unsigned char inhibited;
  digitalWrite(INHIBIT_PIN, HIGH); // Enable pull-up for switch S2.
  inhibited = digitalRead(INHIBIT_PIN);
  digitalWrite(INHIBIT_PIN, LOW);
  return inhibited;
}

void FlashLED() {
  digitalWrite(LED_PIN, HIGH);
  LEDFlashes = 47;
  FastWDT();
}

// This function samples each of the five analog inputs 16 times and averages the results.
// It then converts the value into a voltage, checks it against the limits and if outside,
// increases the alert duration for that input. Otherwise it resets the alert duration.
// It then calculates whether there is an overall alert situation.
unsigned char readADC() {
  unsigned char alert = 0;

  // Turn on LED during this procedure to flash it, but not if the battery is flat.
  if( !BatteryIsLow && !LEDFlashes )
    digitalWrite(LED_PIN, HIGH);
  for( ADCinput = 0; ADCinput <= 5; ++ADCinput ) {
    float voltage;
    ADCvals[ADCinput] = 0;
    // Sample the selected input 16 times and sum the results.
    for( ADCcounter = 0; ADCcounter < 16; ++ADCcounter )
      ADCvals[ADCinput] += analogRead(ADCinput);
    // Calculate average.
    ADCvals[ADCinput] /= 16;
    // Convert sample integer to floating-point voltage.
    voltage = ADCvalToVoltage(ADCinput);
    // Input 0 is the battery monitor.
    if( ADCinput == 0 )
      BatteryIsLow = (voltage < Low_Battery_Level);
    // Check if value is within or outside limits.
    if( alimits[ADCinput].minval < alimits[ADCinput].maxval ) {
      if( voltage < alimits[ADCinput].minval || voltage > alimits[ADCinput].maxval ) {
        AlertDuration[ADCinput] += TimePassed;
      } else {
        if( AlertDuration[ADCinput] > 0 )
          LastAlertDuration[ADCinput] = AlertDuration[ADCinput];
        AlertDuration[ADCinput] = 0;
      }
    } else if( alimits[ADCinput].minval > alimits[ADCinput].maxval ) {
      if( voltage >= alimits[ADCinput].minval && voltage <= alimits[ADCinput].maxval ) {
        AlertDuration[ADCinput] += TimePassed;
      } else {
        if( AlertDuration[ADCinput] > 0 )
          LastAlertDuration[ADCinput] = AlertDuration[ADCinput];
        AlertDuration[ADCinput] = 0;
      }
    }

    if( alimits[ADCinput].delaysec < 0 ) {
      if( LastAlertDuration[ADCinput] && LastAlertDuration[ADCinput] < -alimits[ADCinput].delaysec )
        alert = 1;
    } else {
      if( AlertDuration[ADCinput] >= alimits[ADCinput].delaysec )
        alert = 1;
    }
  }
  // Now scan the digital inputs, enabling each pull-up just before checking its state.
  DigitalState = 0;
  for( ADCinput = 0; ADCinput < 5; ++ADCinput ) {
    digitalWrite(2+ADCinput, HIGH);
    if( digitalRead(2+ADCinput) )
      DigitalState |= 1<<ADCinput;
    digitalWrite(2+ADCinput, LOW);
    // Check if this input state should generate an alert.
    if( ((DigitalState>>ADCinput) & 1) != dlimits[ADCinput].state ) {
      AlertDuration[ADCinput+6] += TimePassed;
    } else {
      if( AlertDuration[ADCinput] > 0 )
        LastAlertDuration[ADCinput] = AlertDuration[ADCinput];
      AlertDuration[ADCinput+6] = 0;
    }

    if( alimits[ADCinput].delaysec < 0 ) {
      if( LastAlertDuration[ADCinput+6] && LastAlertDuration[ADCinput+6] < -dlimits[ADCinput].delaysec )
        alert = 1;
    } else {
      if( AlertDuration[ADCinput+6] >= dlimits[ADCinput].delaysec )
        alert = 1;
    }
  }

  // If an alert is being generated, and we haven't sent a message lately, and the battery is OK
  // then send a message.
  if( !GPRSState && alert && TimeSinceLastSMS >= SMS_min_send_interval && !BatteryIsLow ) {
    unsigned char inhibited = SuppressTime || CheckInhibitSwitch();
#ifdef _DEBUG
    Serial.print("inhibited? ");
    Serial.println(inhibited);
#endif
    // But not if S2 is set to inhibit message sensing or a suppress message has been received.
    if( inhibited ) {
      // If inhibited, flash the LED rapidly once per minute while the alert condition remains.
      if( TimeSinceLastSMS > 60 )
        TimeSinceLastSMS -= 60;
      else
        TimeSinceLastSMS = 0;
      FlashLED();
    } else {
      // We're going to try sending a message so power up the GPRS receiver.
#ifdef _DEBUG
      Serial.println("GPRS power pin high");
#endif
      digitalWrite(GPRS_POWER_PIN, HIGH);
      // This lets us keep track of the GPRS state since we need to wait a while for it to power
      // up before actually sending the message.
      GPRSState = 1;
      GPRSSendMessage = 1;
      LEDFlashes = 0;
    }
  }
#ifdef _DEBUG
  Serial.print("LEDFlashes: ");
  Serial.print(LEDFlashes);
  Serial.print(", TimeSinceLastSMS: ");
  Serial.print(TimeSinceLastSMS);
  Serial.print(", Uptime: ");
  Serial.print(Uptime);
  Serial.print(", SMSCheckTime: ");
  Serial.println(SMSCheckTime);
#endif
  // Turn off the LED now, the micro is going back to sleep.
  if( !LEDFlashes && !GPRSState )
    digitalWrite(LED_PIN, LOW);
  return alert;
}

// This function checks to see if any messags have been received and if so acts on them.
// It is normally called after the GPRS module has been powered up for some time, to
// give it an opportunity to receive the message.
void CheckForSMSes() {
  char buf[255];
  char* temp;
  unsigned char MaxMSGS = 4; // don't process more than this in one go (to prevent endless loops)
  unsigned char MsgInfo;

  while( MaxMSGS-- && (MsgInfo = ReadSMS(buf, sizeof(buf))) ) {
    // If we have received an SMS, check to see if it contains the password.
#ifdef _DEBUG
    Serial.print("RM: ");
    Serial.println(buf);
#endif
    if( strstr(buf, SMS_PASSWORD) ) {
      // If so, check each of the commands in turn.
      // These ones control whether the unit is suppressing outgoing messages.
      if( strstr(buf, "suppress") ) {
        SuppressTime = SMS_suppress_time;
#ifdef _DEBUG
        Serial.print("suppressed for: ");
        Serial.println(SuppressTime);
#endif
      } else if( strstr(buf, "resume") ) {
        SuppressTime = 0;
#ifdef _DEBUG
        Serial.println("resume");
#endif
      }
      // This command prompts us to immediately send a status update.
      if( strstr(buf, "status") && GPRSState >= MAX_GSM_TIME+11 ) {
        // If the GPRS module is already powered up (likely), send a message
#ifdef _DEBUG
        Serial.println("status update requested");
#endif
        GPRSState = MAX_GSM_TIME+11;
        GPRSSendMessage = 1;
      }
      // Reset simply re-starts the timing of the periodic update message.
      if( strstr(buf, "reset") ) {
        Uptime = 0;
#ifdef _DEBUG
        Serial.println("Uptime reset");
#endif
      }

      // Redirect message changes the outgoing phone number.
      temp = strstr(buf, "redirect ");
      if( temp ) {
        char* temp2;
        // the number should be immediately after the redirect command
        temp += 9;
        // look to see if it's followed by a space
        temp2 = strstr(temp, " ");
        // If not, then the end of the message terminates the phone number
        if( !temp2 )
          temp2 = temp + strlen(temp);
        // Check if it's valid and if so, update the destination number.
        if( temp2 > temp && temp[0] == '+' && temp2 - temp < 32 ) {
          strncpy(SMS_Destination, temp, temp2-temp);
          SMS_Destination[temp2-temp] = '\0';

#ifdef _DEBUG
          Serial.print("Redirecting to: ");
          Serial.println(SMS_Destination);
#endif
        }
      }
    }
  }

  // Reset timing so the unit will wait a while before powering the module back up
  // to check for more SMS messages.
  SMSCheckTime = 0;
}

// This is the main loop of the program.
void loop() {
  // Turn off the LED if not busy with the GPRS module.
  if( !GPRSState )
    digitalWrite(LED_PIN, LOW);
  // Ensure that we check the status of the inputs immediately.
  WatchDogFired = 1;
  while(1) {
    // If something has indicated that the LED should be flashed, toggle that output pin.
    // Any code that sets LEDFlashes > 0 also increases the Watchdog Timer rate to 4Hz for
    // a 2Hz flash rate.
    if( LEDFlashes ) {
      digitalWrite(LED_PIN, !digitalRead(LED_PIN));
      // If enough flashes have occured, go back to the normal 1Hz Watchdog Timer rate.
      if( --LEDFlashes == 0 )
        SlowWDT();
    }
    // If at least one second has passed, scan the inputs.
    if( WatchDogFired ) {
      TimePassed = WatchDogFired;
      WatchDogFired = 0;

      // This counter limits SMS sending rate.
      TimeSinceLastSMS += TimePassed;
      // This counter controls status update sending rate.
      Uptime += TimePassed;
      // This counter controls how often the GPRS is powered up to check for incoming messages.
      SMSCheckTime += TimePassed;
      // This counter controls how long the unit stays "suppressed" (temporarily
      // not sending any outgoing messages).
      if( SuppressTime > TimePassed )
        SuppressTime -= TimePassed;
      else
        SuppressTime = 0;

      // If the GPRS unit is active, we need to take some sort of action
      if( GPRSState ) {
        ++GPRSState;
#ifdef _DEBUG
        Serial.print("GPRSState: ");
        Serial.println(GPRSState);
#endif
        // This turns off the power pin pulse one second after it starts,
        // as required to power up the module.
        if( GPRSState == 2 ) {
#ifdef _DEBUG
          Serial.println("GPRS power pin low");
#endif
          digitalWrite(GPRS_POWER_PIN, LOW);
        } else if( GPRSState == 6 ) {
          if( !CheckAndSetPIN() ) {
            // Failed, try again in five minutes.
            if( TimeSinceLastSMS > 300 )
              TimeSinceLastSMS -= 300;
            else
              TimeSinceLastSMS = 0;
            // Rapid flash LED to indicate failure.
            FlashLED();
            GPRSState = MAX_GSM_TIME+MSG_WAIT_TIME+8;
          }
        } else if( GPRSState == 8 ) {
          // We then wait an extra couple of seconds for the unit to power up
          // before setting the PIN (if necessary).
          if( !DisableMessageNotifications() ) {
            // Failed, try again in five minutes.
            if( TimeSinceLastSMS > 300 )
              TimeSinceLastSMS -= 300;
            else
              TimeSinceLastSMS = 0;
            // Rapid flash LED to indicate failure.
            FlashLED();
            GPRSState = MAX_GSM_TIME+MSG_WAIT_TIME+8;
          }
        } else if( GPRSState > 10 && GPRSState < MAX_GSM_TIME+7 ) {
          // We leave up to about 30 seconds between powering up/setting the PIN
          // and sending the message. But if the network comes up sooner, we set
          // up the unit and send the message immediately.
          if( IsConnectedToMobileNetwork() )
            GPRSState = MAX_GSM_TIME+7;
        } else if( GPRSState == MAX_GSM_TIME+8 ) {
          if( !SetMessageCentreNumber() ) {
            // Failed, try again in five minutes.
            if( TimeSinceLastSMS > 300 )
              TimeSinceLastSMS -= 300;
            else
              TimeSinceLastSMS = 0;
            // Rapid flash LED to indicate failure.
            FlashLED();
            GPRSState = MAX_GSM_TIME+MSG_WAIT_TIME+8;
          }
        } else if( GPRSState == MAX_GSM_TIME+10 ) {
          if( !SetTextMessageMode() ) {
            // Failed, try again in five minutes.
            if( TimeSinceLastSMS > 300 )
              TimeSinceLastSMS -= 300;
            else
              TimeSinceLastSMS = 0;
            // Rapid flash LED to indicate failure.
            FlashLED();
            GPRSState = MAX_GSM_TIME+MSG_WAIT_TIME+8;
          }
        } else if( GPRSState == MAX_GSM_TIME+12 ) {
          // Now send SMS, if an alert condition exists, otherwise
          // we'll check for incoming messages later.
          if( GPRSSendMessage ) {
            if( !SendSMSAlert(1) ) {
              // Failed, try again in five minutes.
              if( TimeSinceLastSMS > 300 )
                TimeSinceLastSMS -= 300;
              else
                TimeSinceLastSMS = 0;
              // Rapid flash LED to indicate failure.
              FlashLED();
              GPRSState = MAX_GSM_TIME+MSG_WAIT_TIME+8;
//          } else {
//            GPRSState = MAX_GSM_TIME+MSG_WAIT_TIME-2;
            }
          }
        } else if( GPRSState == MAX_GSM_TIME+MSG_WAIT_TIME+8 ) {
          // 10 seconds after sending the SMS or 40 seconds after powering the module
          // up specifically to check for incoming messages, receive any pending
          // incoming messages.
          CheckForSMSes();
        } else if( GPRSState == MAX_GSM_TIME+MSG_WAIT_TIME+9 ) {
          // Once any pending messages have been received, power down the GPRS module.
#ifdef _DEBUG
          Serial.println("GPRS power pin high");
#endif
          digitalWrite(GPRS_POWER_PIN, HIGH);
        } else if( GPRSState == MAX_GSM_TIME+MSG_WAIT_TIME+10 ) {
          // Finished powering down the module, reset the power pin and go back to idle mode.
#ifdef _DEBUG
          Serial.println("GPRS power pin low");
#endif
          digitalWrite(GPRS_POWER_PIN, LOW);
          GPRSState = 0;
        }
      }
      // Now scan all the inputs and update the alarm state.
      // This function will trigger a message if an alert occurs.
      readADC();
      // If no mesage was triggered and the unit is set up to provide periodic status updates,
      // and the battery is not flat, check to see if it's time to send one.
      if( !GPRSState && SMS_send_interval && Uptime >= SMS_send_interval && !BatteryIsLow ) {
        // If so, check to see whether message sending is inhibited.
        unsigned char inhibited = SuppressTime || CheckInhibitSwitch();
#ifdef _DEBUG
        Serial.print("inhibited? ");
        Serial.println(inhibited);
#endif
        if( inhibited ) {
          // If it is, just flash the LED.
          FlashLED();
        } else {
          // Otherwise, power up the GPRS module in preparation for sending a mesage.
#ifdef _DEBUG
          Serial.println("GPRS power pin high");
#endif
          digitalWrite(GPRS_POWER_PIN, HIGH);
          // This eventually triggers the message sending routine.
          GPRSState = 1;
          GPRSSendMessage = 1;
        }
        // Re-set the delay to the next status message.
        Uptime -= SMS_send_interval;
      }
      // If the unit is not sending any messages and it hasn't powered up the GPRS module for a while,
      // check to see whether it's time to power it up in order to check for incoming messages.
      if( !GPRSState && SMS_recv_check_interval && SMSCheckTime >= SMS_recv_check_interval && !BatteryIsLow ) {
        // If so and not inhibited, power it up.
        unsigned char inhibited = SuppressTime || CheckInhibitSwitch();
        if( !inhibited ) {
          // Turn GPRS power on.
          digitalWrite(GPRS_POWER_PIN, HIGH);
          // Make a note to check incoming messages in about 60 seconds, then power it down.
          GPRSState = 1;
          GPRSSendMessage = 0;
          // If this fails, we don't want it trying constantly so reset the counter.
          SMSCheckTime = 0;
        }
      }
    }
    // Once the inputs have been scanned and any message sending/receiving actions have been taken,
    // the micro goes back to sleep for 0.25-1 second before doing the above all over again.
#ifdef _DEBUG
    SendSMSAlert(0);
#endif
    GoToSleep();
  }
}



























/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////// Everything below here is just a copy of SoftwareSerial but with a bigger buffer //////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define _DEBUG 0
#define _DEBUG_PIN1 11
#define _DEBUG_PIN2 13

#include <avr/interrupt.h>
#include <avr/pgmspace.h>  
#include <Arduino.h>
//
// Lookup table
//
typedef struct _DELAY_TABLE
{
  long baud;
  unsigned short rx_delay_centering;
  unsigned short rx_delay_intrabit;
  unsigned short rx_delay_stopbit;
  unsigned short tx_delay;
} DELAY_TABLE;

#if F_CPU == 16000000

static const DELAY_TABLE PROGMEM table[] = 
{
  //  baud    rxcenter   rxintra    rxstop    tx
  { 115200,   1,         17,        17,       12,    },
  { 57600,    10,        37,        37,       33,    },
  { 38400,    25,        57,        57,       54,    },
  { 31250,    31,        70,        70,       68,    },
  { 28800,    34,        77,        77,       74,    },
  { 19200,    54,        117,       117,      114,   },
  { 14400,    74,        156,       156,      153,   },
  { 9600,     114,       236,       236,      233,   },
  { 4800,     233,       474,       474,      471,   },
  { 2400,     471,       950,       950,      947,   },
  { 1200,     947,       1902,      1902,     1899,  },
  { 600,      1902,      3804,      3804,     3800,  },
  { 300,      3804,      7617,      7617,     7614,  },
};

const int XMIT_START_ADJUSTMENT = 5;

#elif F_CPU == 8000000

static const DELAY_TABLE table[] PROGMEM = 
{
  //  baud    rxcenter    rxintra    rxstop  tx
  { 115200,   1,          5,         5,      3,      },
  { 57600,    1,          15,        15,     13,     },
  { 38400,    2,          25,        26,     23,     },
  { 31250,    7,          32,        33,     29,     },
  { 28800,    11,         35,        35,     32,     },
  { 19200,    20,         55,        55,     52,     },
  { 14400,    30,         75,        75,     72,     },
  { 9600,     50,         114,       114,    112,    },
  { 4800,     110,        233,       233,    230,    },
  { 2400,     229,        472,       472,    469,    },
  { 1200,     467,        948,       948,    945,    },
  { 600,      948,        1895,      1895,   1890,   },
  { 300,      1895,       3805,      3805,   3802,   },
};

const int XMIT_START_ADJUSTMENT = 4;

#elif F_CPU == 20000000

// 20MHz support courtesy of the good people at macegr.com.
// Thanks, Garrett!

static const DELAY_TABLE PROGMEM table[] =
{
  //  baud    rxcenter    rxintra    rxstop  tx
  { 115200,   3,          21,        21,     18,     },
  { 57600,    20,         43,        43,     41,     },
  { 38400,    37,         73,        73,     70,     },
  { 31250,    45,         89,        89,     88,     },
  { 28800,    46,         98,        98,     95,     },
  { 19200,    71,         148,       148,    145,    },
  { 14400,    96,         197,       197,    194,    },
  { 9600,     146,        297,       297,    294,    },
  { 4800,     296,        595,       595,    592,    },
  { 2400,     592,        1189,      1189,   1186,   },
  { 1200,     1187,       2379,      2379,   2376,   },
  { 600,      2379,       4759,      4759,   4755,   },
  { 300,      4759,       9523,      9523,   9520,   },
};

const int XMIT_START_ADJUSTMENT = 6;

#else

#error This version of SoftwareSerial supports only 20, 16 and 8MHz processors

#endif

//
// Statics
//
SoftwareSerialBigBuf *SoftwareSerialBigBuf::active_object = 0;
char SoftwareSerialBigBuf::_receive_buffer[_SSBB_MAX_RX_BUFF]; 
volatile uint8_t SoftwareSerialBigBuf::_receive_buffer_tail = 0;
volatile uint8_t SoftwareSerialBigBuf::_receive_buffer_head = 0;

//
// Debugging
//
// This function generates a brief pulse
// for debugging or measuring on an oscilloscope.
inline void DebugPulse(uint8_t pin, uint8_t count)
{
#if _DEBUG
  volatile uint8_t *pport = portOutputRegister(digitalPinToPort(pin));

  uint8_t val = *pport;
  while (count--)
  {
    *pport = val | digitalPinToBitMask(pin);
    *pport = val;
  }
#endif
}

//
// Private methods
//

/* static */ 
inline void SoftwareSerialBigBuf::tunedDelay(uint16_t delay) { 
  uint8_t tmp=0;

  asm volatile("sbiw    %0, 0x01 \n\t"
    "ldi %1, 0xFF \n\t"
    "cpi %A0, 0xFF \n\t"
    "cpc %B0, %1 \n\t"
    "brne .-10 \n\t"
    : "+r" (delay), "+a" (tmp)
    : "0" (delay)
    );
}

// This function sets the current object as the "listening"
// one and returns true if it replaces another 
bool SoftwareSerialBigBuf::listen()
{
  if (active_object != this)
  {
    _buffer_overflow = false;
    uint8_t oldSREG = SREG;
    cli();
    _receive_buffer_head = _receive_buffer_tail = 0;
    active_object = this;
    SREG = oldSREG;
    return true;
  }

  return false;
}

//
// The receive routine called by the interrupt handler
//
void SoftwareSerialBigBuf::recv()
{

#if GCC_VERSION < 40302
// Work-around for avr-gcc 4.3.0 OSX version bug
// Preserve the registers that the compiler misses
// (courtesy of Arduino forum user *etracer*)
  asm volatile(
    "push r18 \n\t"
    "push r19 \n\t"
    "push r20 \n\t"
    "push r21 \n\t"
    "push r22 \n\t"
    "push r23 \n\t"
    "push r26 \n\t"
    "push r27 \n\t"
    ::);
#endif  

  uint8_t d = 0;

  // If RX line is high, then we don't see any start bit
  // so interrupt is probably not for us
  if (_inverse_logic ? rx_pin_read() : !rx_pin_read())
  {
    // Wait approximately 1/2 of a bit width to "center" the sample
    tunedDelay(_rx_delay_centering);
    DebugPulse(_DEBUG_PIN2, 1);

    // Read each of the 8 bits
    for (uint8_t i=0x1; i; i <<= 1)
    {
      tunedDelay(_rx_delay_intrabit);
      DebugPulse(_DEBUG_PIN2, 1);
      uint8_t noti = ~i;
      if (rx_pin_read())
        d |= i;
      else // else clause added to ensure function timing is ~balanced
        d &= noti;
    }

    // skip the stop bit
    tunedDelay(_rx_delay_stopbit);
    DebugPulse(_DEBUG_PIN2, 1);

    if (_inverse_logic)
      d = ~d;

    // if buffer full, set the overflow flag and return
    if ((_receive_buffer_tail + 1) % _SSBB_MAX_RX_BUFF != _receive_buffer_head) 
    {
      // save new data in buffer: tail points to where byte goes
      _receive_buffer[_receive_buffer_tail] = d; // save new byte
      _receive_buffer_tail = (_receive_buffer_tail + 1) % _SSBB_MAX_RX_BUFF;
    } 
    else 
    {
#if _DEBUG // for scope: pulse pin as overflow indictator
      DebugPulse(_DEBUG_PIN1, 1);
#endif
      _buffer_overflow = true;
    }
  }

#if GCC_VERSION < 40302
// Work-around for avr-gcc 4.3.0 OSX version bug
// Restore the registers that the compiler misses
  asm volatile(
    "pop r27 \n\t"
    "pop r26 \n\t"
    "pop r23 \n\t"
    "pop r22 \n\t"
    "pop r21 \n\t"
    "pop r20 \n\t"
    "pop r19 \n\t"
    "pop r18 \n\t"
    ::);
#endif
}

void SoftwareSerialBigBuf::tx_pin_write(uint8_t pin_state)
{
  if (pin_state == LOW)
    *_transmitPortRegister &= ~_transmitBitMask;
  else
    *_transmitPortRegister |= _transmitBitMask;
}

uint8_t SoftwareSerialBigBuf::rx_pin_read()
{
  return *_receivePortRegister & _receiveBitMask;
}

//
// Interrupt handling
//

/* static */
inline void SoftwareSerialBigBuf::handle_interrupt()
{
  if (active_object)
  {
    active_object->recv();
  }
}

#if defined(PCINT0_vect)
ISR(PCINT0_vect)
{
  SoftwareSerialBigBuf::handle_interrupt();
}
#endif

#if defined(PCINT1_vect)
ISR(PCINT1_vect)
{
  SoftwareSerialBigBuf::handle_interrupt();
}
#endif

#if defined(PCINT2_vect)
ISR(PCINT2_vect)
{
  SoftwareSerialBigBuf::handle_interrupt();
}
#endif

#if defined(PCINT3_vect)
ISR(PCINT3_vect)
{
  SoftwareSerialBigBuf::handle_interrupt();
}
#endif

//
// Constructor
//
SoftwareSerialBigBuf::SoftwareSerialBigBuf(uint8_t receivePin, uint8_t transmitPin, bool inverse_logic /* = false */) : 
  _rx_delay_centering(0),
  _rx_delay_intrabit(0),
  _rx_delay_stopbit(0),
  _tx_delay(0),
  _buffer_overflow(false),
  _inverse_logic(inverse_logic)
{
  setTX(transmitPin);
  setRX(receivePin);
}

//
// Destructor
//
SoftwareSerialBigBuf::~SoftwareSerialBigBuf()
{
  end();
}

void SoftwareSerialBigBuf::setTX(uint8_t tx)
{
  pinMode(tx, OUTPUT);
  digitalWrite(tx, HIGH);
  _transmitBitMask = digitalPinToBitMask(tx);
  uint8_t port = digitalPinToPort(tx);
  _transmitPortRegister = portOutputRegister(port);
}

void SoftwareSerialBigBuf::setRX(uint8_t rx)
{
  pinMode(rx, INPUT);
  if (!_inverse_logic)
    digitalWrite(rx, HIGH);  // pullup for normal logic!
  _receivePin = rx;
  _receiveBitMask = digitalPinToBitMask(rx);
  uint8_t port = digitalPinToPort(rx);
  _receivePortRegister = portInputRegister(port);
}

//
// Public methods
//

void SoftwareSerialBigBuf::begin(long speed)
{
  _rx_delay_centering = _rx_delay_intrabit = _rx_delay_stopbit = _tx_delay = 0;

  for (unsigned i=0; i<sizeof(table)/sizeof(table[0]); ++i)
  {
    long baud = pgm_read_dword(&table[i].baud);
    if (baud == speed)
    {
      _rx_delay_centering = pgm_read_word(&table[i].rx_delay_centering);
      _rx_delay_intrabit = pgm_read_word(&table[i].rx_delay_intrabit);
      _rx_delay_stopbit = pgm_read_word(&table[i].rx_delay_stopbit);
      _tx_delay = pgm_read_word(&table[i].tx_delay);
      break;
    }
  }

  // Set up RX interrupts, but only if we have a valid RX baud rate
  if (_rx_delay_stopbit)
  {
    if (digitalPinToPCICR(_receivePin))
    {
      *digitalPinToPCICR(_receivePin) |= _BV(digitalPinToPCICRbit(_receivePin));
      *digitalPinToPCMSK(_receivePin) |= _BV(digitalPinToPCMSKbit(_receivePin));
    }
    tunedDelay(_tx_delay); // if we were low this establishes the end
  }

#if _DEBUG
  pinMode(_DEBUG_PIN1, OUTPUT);
  pinMode(_DEBUG_PIN2, OUTPUT);
#endif

  listen();
}

void SoftwareSerialBigBuf::end()
{
  if (digitalPinToPCMSK(_receivePin))
    *digitalPinToPCMSK(_receivePin) &= ~_BV(digitalPinToPCMSKbit(_receivePin));
}


// Read data from buffer
int SoftwareSerialBigBuf::read()
{
  if (!isListening())
    return -1;

  // Empty buffer?
  if (_receive_buffer_head == _receive_buffer_tail)
    return -1;

  // Read from "head"
  uint8_t d = _receive_buffer[_receive_buffer_head]; // grab next byte
  _receive_buffer_head = (_receive_buffer_head + 1) % _SSBB_MAX_RX_BUFF;
  return d;
}

int SoftwareSerialBigBuf::available()
{
  if (!isListening())
    return 0;

  return (_receive_buffer_tail + _SSBB_MAX_RX_BUFF - _receive_buffer_head) % _SSBB_MAX_RX_BUFF;
}

size_t SoftwareSerialBigBuf::write(uint8_t b)
{
  if (_tx_delay == 0) {
    setWriteError();
    return 0;
  }

  uint8_t oldSREG = SREG;
  cli();  // turn off interrupts for a clean txmit

  // Write the start bit
  tx_pin_write(_inverse_logic ? HIGH : LOW);
  tunedDelay(_tx_delay + XMIT_START_ADJUSTMENT);

  // Write each of the 8 bits
  if (_inverse_logic)
  {
    for (byte mask = 0x01; mask; mask <<= 1)
    {
      if (b & mask) // choose bit
        tx_pin_write(LOW); // send 1
      else
        tx_pin_write(HIGH); // send 0
    
      tunedDelay(_tx_delay);
    }

    tx_pin_write(LOW); // restore pin to natural state
  }
  else
  {
    for (byte mask = 0x01; mask; mask <<= 1)
    {
      if (b & mask) // choose bit
        tx_pin_write(HIGH); // send 1
      else
        tx_pin_write(LOW); // send 0
    
      tunedDelay(_tx_delay);
    }

    tx_pin_write(HIGH); // restore pin to natural state
  }

  SREG = oldSREG; // turn interrupts back on
  tunedDelay(_tx_delay);
  
  return 1;
}

void SoftwareSerialBigBuf::flush()
{
  if (!isListening())
    return;

  uint8_t oldSREG = SREG;
  cli();
  _receive_buffer_head = _receive_buffer_tail = 0;
  SREG = oldSREG;
}

int SoftwareSerialBigBuf::peek()
{
  if (!isListening())
    return -1;

  // Empty buffer?
  if (_receive_buffer_head == _receive_buffer_tail)
    return -1;

  // Read from "head"
  return _receive_buffer[_receive_buffer_head];
}

