
/* Arduino_pH_meter_sketch.ino
   A sketch for an Arduino based liquid pH meter using the
   Gravity pH sensor module and one of its sensor probes,
   together with a 16x2 LCD module with a serial I2C 'piggyback'
   interface.
      
   Written by Jim Rowe (Silicon Chip)
   Version 1.0, updated 11/01/2023 at 10.30 am
 
   NOTES:
   1. This project is designed to use an LCD module with a
   serial interface based on a PCF8574T serial to parallel converter.
   These have an I2C address in the range from 20-27h, with the highest
   address corresponding to all three links on the piggyback PCB being OPEN.
   However some modules have a PCF8574AT converter chip instead,
   and these have a different I2C address range: from 38-3Fh.
   If you have one of these latter modules, the only change that
   should need to be made is to change the address in line 31 below,
   from 0x27 to 0x3F.
   2. The value of GnIC1a shown below assumes that op-amp IC1a has a gain of
   exactly 3.0 - i.e., that the 20k and 10k resistors in the feedback divider
   have values very close to these values. So if you measure the actual gain
   of the op-amp in your module and it is different from 3.0, please change
   the value of GnIC1a to the measured gain.
   3. Similarly, the value of Senslope shown below is that for an 'ideal' pH
   sensor with a slope of -59.16mV per pH increment. With a real sensor,
   the value of Senslope may need to be changed to achieve calibration.
 */
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
LiquidCrystal_I2C lcd(0x27,16,2);   // set the LCD address to 0x27,
                                    // with 2 x 16 character lines
// ====================================================================
// declaration of global variables

int i;      // counter index
unsigned long int TenRdgs = 0;  // for addition of 10 readings
float AvgVal = 0.0;  // calculated average of the readings
float Vavg = 0.0;   // average reading in volts
float GnIC1a = 3.000;  // gain of IC1a. (change if measured different)
float Senslope = 0.05916; // slope of sensor output in V/pH increment
                      // (may need changing to suit sensor used)
float pHincrem = (GnIC1a * Senslope);  
float pHval = 0.0;  // the equivalent pH level
// ====================================================================
// setup function begins here
// ====================================================================
void setup()
{
  lcd.begin();
  Serial.begin(9600);   // starts up serial comms to PC for testing   
  lcd.backlight();    // turn on LCD's LED backlight
  lcd.setCursor(0,0);     // and show opening screen
  lcd.print("  Silicon Chip  ");
  lcd.setCursor(0,1);
  lcd.print("Liquid pH Meter ");

  Serial.println("Silicon Chip pH Meter");  // for testing
  delay(2000);            // then pause for 2 seconds

} // end of setup function

 // ===================================================================
 // main loop() function begins here
 // ===================================================================
void loop()
{
  lcd.clear();                // first clear LCD screen
  TenRdgs = 0;      // then reset TenRdgs
  for(i = 0; i<10; i++)
    {
      TenRdgs = TenRdgs + analogRead(A0); // add new reading
      delay (30);   // then wait for 30ms before continuing
    }

  AvgVal = TenRdgs/10.00;   // find the average of the 10 readings
  Vavg = 5.0 * (AvgVal/1023); // convert to a voltage reading
  pHval =7.0 - ((Vavg - 2.50)/pHincrem);  // then to pH value
  if(pHval < 0 ) pHval = 0.0;   // prevents negative value
  if(pHval > 14) pHval = 14.0;  // or values above 14
  
  lcd.setCursor(0,0);     // set LCD cursor to first line
  lcd.print("pH = " + String(pHval)); // and print pH value
  lcd.setCursor(0,1);     // then move to second line
  lcd.print("Vaverage = " + String(Vavg));
  
  Serial.println("Vaverage = " + String(Vavg) + " so pH = " + String(pHval));
  delay(2000);
}      // end of main loop

// =====================================================================
//    end of code
