/*
  **************************************
  *        COFFEE GRINDER TIMER        *
  **************************************

  Modified & Rewritten by Flavio Spedalieri (FRS), March 2020.

  V2.82 (Release 1) - 1B_128x64OLED (BC4)
  ---------------------------------------
  Features:  
    Two preset timers
    Manual Grind / Purge
    Configurable Display options; LCD or OLED options
    Two button or Single button for Preset 1 / 2 selection
    Encoder adjustment for on-the-fly time offset & programming
    LED Visual Indicator

  /* --------- Firmware Version, Date & Build Codes ------- */
String(fwVersion) = "2.82 FRS"; // 
String(fwDate) = "21.05.2020";
// Build Codes
String(bc1) = "2B_1602LCD (BC1)";     // 2-Button (P1, P2) 16x2 character LCD display
String(bc2) = "1B_1602LCD (BC2)";     // 1-Button (P1) 16x2 character LCD display
String(bc3) = "2B_128x64OLED (BC3)";  // 2-Button (P1, P2) 128x64 graphic OLED display
String(bc4) = "1B_128x64OLED (BC4)";  // 1-Button (P1) 128x64 graphic OLED display (Default Configuration)
String(bc5) = "1B_NW1602OLED (BC5)";  // 1-Button (P1) 16x2 character OLED display (Newhaven NHD-0216AW-IB3)
String(bc6) = "1B_DS12864OLED (BC6)"; // 1-Button (P1) 128x64 Digole DS12864OLED-2W OLED
String bc = bc4;
/* ----------------------------------------- */

#include <Encoder.h>     // http://www.pjrc.com/teensy/td_libs_Encoder.html
#include <EEPROM.h>      // EEPROM library
#include <LEDFader.h>    // https://github.com/jgillick/arduino-LEDFader

/*------------------ Display Setup & Communications -------------------------
 *
 * Program default to use SSD1306 0.96" 128x64 Monochrome OLED. 
 * If using SPI communications, set SSR to pin 12.
 *
 * To use other displays below, modifications to DISPLAY & INIT files required.
 * LCD communications: Parallel / IC2.
 */

//#define LCD_Display
#define GRAPHIC_OLED
//#define CHARACTER_OLED
//#define _Digole_Serial_I2C_

/*--------- 16x2 Backlit LCD Display ----------*/
#if defined (LCD_Display)
#include <LiquidCrystal_PCF8574.h>  // https://github.com/mathertel/LiquidCrystal_PCF8574
LiquidCrystal_PCF8574 lcd(0x27);    // I2C: SDA (Serial Data) Pin A4, SCL (Serial Clock) Pin A5
//LiquidCrystal_PCF8574 lcd(/* Address */0x27,/* RS */13,/* EN */11,/* D4 */5,/* D5 */4,/* D6 */3,/* D7 */2); // LCD Parallel Data Pins.
#endif

/*---- 128 x 64 Graphical Mono OLED Display ---- */
#if defined (GRAPHIC_OLED)
#include <U8g2lib.h>  // U8g2 monochrome graphics library
#include "SSD1306_Splash.h";
 
/*---------- SSD1306 128 x 64 OLED (I2C) ------------------- */
U8G2_SSD1306_128X64_NONAME_2_HW_I2C u8g2(U8G2_R0, /* reset=*/ U8X8_PIN_NONE);   // Page Buffer: 2 pages, HARDWARE I2C: SDA (Data) Pin A4, SCL (Clock) Pin A5
//U8G2_SSD1306_128X64_NONAME_F_HW_I2C u8g2(U8G2_R0, /* reset=*/ U8X8_PIN_NONE); // Full Page Buffer (Requires 1Kb RAM. Not enough memory on UNO)

/*---------- SH1106 128 x 64 OLED (SPI) ------------------- */
//U8G2_SH1106_128X64_NONAME_2_4W_HW_SPI u8g2(U8G2_R0, /* cs=*/ 14, /* dc=*/ 15, /* reset=*/ 7);
//U8G2_SH1106_128X64_NONAME_2_4W_SW_SPI u8g2(U8G2_R0, /* clock=*/ 13, /* data=*/ 11, /* cs=*/ 14, /* dc=*/ 15, /* reset=*/ 7);

#endif

/*---------- Newhaven NHD-0216AW-IB3 16x2 OLED ------------------- */
#if defined (CHARACTER_OLED)
/* Modification to the 'I2cCharDisplay.cpp' required to allow the Newhaven display to work;
    Function: "void I2cCharDisplay::oledBegin()", extended function section
      sendCommand(0xDA); // Set SEG Pins Hardware Configuration:
      sendCommand(0x00); // Enable SEG Left, Seq SEG pin config ** CHANGE PARAMETER FROM 0x10 to 0x00 *
*/
#include "I2cCharDisplay.h" // US2066 driver library https://www.dcity.org/portfolio/i2c-display-library/
#include "Wire.h"
#define OLEDADDRESS    0x3c                     // i2c address for the oled display
I2cCharDisplay oled(OLED_TYPE, OLEDADDRESS, 2); // create an oled object for a 2 line display
#endif


//--------I2C setup
#if defined(_Digole_Serial_I2C_)
#include <DigoleSerial.h>
#include "DS12864_Splash.h";
#include <Wire.h>
DigoleSerialDisp mydisp(&Wire, '\x27'); //I2C: A4: SDA (data line), A5: SCL (clock line).
#endif

/*-- END DISPLAY CONFIGURATION SECTION -- */

/*------- Idle Timer & OLED display power save mode --------------- */
//uint32_t IdleTime = 60;         // Time in seconds before setting system in idle
uint32_t screenSaverTimer = 5;  // Time in minutes before turning off display.

/*----------------------- Encoder Setup -------------------------------*/
/* Determines correct encoder behaviour; 1 physical detent to increment 
 * time by +/- 0.1 sec
 * Tested encoders: 2 for passive, 4 for active (5V). 
 */
int encoderScaler = 2;

/*--------------button pins -----------------------*/
Encoder myEnc(2, 3);    // Pins for encoder ** Added FRS March 2020 **
const int select = 4;   // Encoder button, short press enters preset program mode. Long press displays firmware version.
const int preset1 = 5;  // Preset 1 button
const int preset2 = 6;  // Preset 2 button. (Not used when configured for single preset button).
const int active = 7;   // Grinder run (timed).
const int manual = 8;   // Manual run button.
const int led1 = 9;     // LED 1 indicator (PWM) - Main Run Indicator & Status
const int led2 = 10;    // LED 2 indicator (PWM) - White LED, Select
const int led3 = 11;    // LED 3 indicator (PWM) - Red LED, Manual Purge
const int ssr1 = 12;    // specifies pin number to connect to external Solid-State Relay.

/*-----------------------------------------------------------------------------*/

/*-------- Initialise Program Variables & States ----------*/
int presetOneVal = 0;
int offsetState = 0;
int presetTwoVal = 0;
int offset2State = 0;
int grinderProg;
int selectState = 0;
int oldSelectState = 0;
int selectPreset = 0;
int startMsg = 0;
int startMsgState = 0;
int sec;
int msec;
//int usec; // FRS 17.5.2020: Depreciated
int x = 0;

/* Depreciated 11.5.2020, now handled by 'timerState'
int ps1State = 0;
int ps2State = 0;
int activeState = 0;
int manualState = 0;
*/

/* -- Main Timer Program State. -- 
 * 0 = idle
 * 1 = Preset 1 
 * 2 = Preset 2
 * 3 = Run
 * 4 = Manual
 * 5 = Offset Idle
 * 6 = Sleep
 */
int timerState = 1;

/* -------- Initialise Button Variables ----------*/
#define debounce 20

//Encoder Pin Button
int encBtn = 0;
int encBtnState = 0;
int encBtnLongPress = 2000;
int encBtnLastState = 0;

// Preset 1 Button 
int ps1Btn = 0;
int ps1BtnState = 0;
int ps1BtnLastState = 0;
int ps1BtnLongPress = 2000;

// Preset 2 Button
int ps2Btn = 0;
//int ps2BtnState = 0;
//int ps2BtnLastState = 0;
//int ps2BtnLongPress = 2000;

// Run Button
int activeBtn = 0;

// Manual Button
int manualBtn = 0;



/* ------ Initialise Main Display States. ------ 
 *  0; Default Preset 1 Display
 *  1; Preset 2 Display
 */
int displayState = 0;
int lastDisplayState = 0;
/*--------------------------*/ 

/*-------- Timing Variables used with millis -------------*/
uint32_t newTime;
uint32_t oldTime;
uint32_t ps1Time;
uint32_t ps1TimeLength;
uint32_t runTime1;
uint32_t ps2Time;
uint32_t ps2TimeLength;
uint32_t runTime2;
uint32_t grinderRunTime1 = 0;
uint32_t grinderRunTime2 = 0;
uint32_t grinderIdle = 0;
uint32_t countDown;
// Button Timers
uint32_t encBtnTimer = 0;
uint32_t ps1BtnTimer = 0;
//uint32_t ps2BtnTimer = 0;

/* -------- Initialise LED Variables ----------- */
bool idleLedState = true;
// LED Fading variables
uint32_t fadeTime = 3000;
const int maxLight = 255;
const int minLight = 128;
const int dirUp = 1;
const int dirDn = -1;
LEDFader led;  // Run Button LED
LEDFader ledW; // White LED (P1 Button)
LEDFader ledR; // Red LED (Manual Button)
int direction = dirUp;
/*-----------------------------------------------*/


void setup() {

  pinMode(preset1, INPUT_PULLUP);
  pinMode(preset2, INPUT_PULLUP); 
  pinMode(select, INPUT_PULLUP);
  pinMode(active, INPUT_PULLUP);
  pinMode(manual, INPUT_PULLUP);
  pinMode(ssr1, OUTPUT);
  digitalWrite(ssr1, LOW);

  Serial.begin(115200);


  // Additional initialisation parameters
  Init();
}

void loop() {

  buttonCheck();           // Button functions.
  newTime = millis();      // Main timer for countdown.
  presetFunctions();       // run the main code (functions for presets and grinder motor on).
  timerOffset();           // ** FRS 19.3.2020 - (Added Feature) reads encoder for on-the-fly time override. 
  led.update();            // ** FRS 3.4.2020 - (Added Feature) LED Fader.
  idleLED();               //    Idle mode "Breathing LED"
  sleepMode();             // ** FRS 27.4.2020 - (Added Feature) Turn off display after 5 min. 

}
