// ESR meter based on SC Mar2004
// Arduino code by Steve Matthysen
// 16x4 I2C LCD connected - tested OK with 20x4
//======================================================================
// Range  iESRrange ohms    mA    Divisor
// High       2     10-99   0.5     1
// Med        1     01-9.9  5.0     10
// Low        0     0-0.99  50      100
//  Good result: 10<=Count<=100
// Change to next higher ohm range(less mA)iESRrange++: Count>100
// Change to next lower ohm range (more mA)iESRrange--:  Count<10
//======================================================================

#include <EEPROM.h>
#include <Wire.h>
#include <LiquidCrystal_PCF8574.h>    //The LCD library using I2C to 8bit port expander
#include <avr/io.h>
#include <avr/interrupt.h>  // ISR interrupt service routine

LiquidCrystal_PCF8574 lcd(0x27);  //default address is 0x27, could be 0x20-0x27

//ESR defines - note range selection is active low
#define RAMPDISABLE 3   //Disable ramping charge on Ref Capacitor
#define RANGEL 10       //Lo Range 0 - 0.99ohm via Q3
#define RANGEM 11       //Med Range 1 - 9.9 ohm via Q2
#define RANGEH 12       //Hi Range 10 - 99 ohm via Q1
#define DISCHARGE 13     //Discharge DUT Was 9
#define ESRPULSE 6      //AIN0 Comparator +input
#define ESRRAMP 7       //AIN1 Comparator -input
#define ESRZERO 4       //Zero input for ESR
#define ESRMODE 2       //ESR mode selected = 1

//ESR global vars
volatile boolean iflag = false;   //Set when comparator goes high
boolean iESRsetupDone;            //Set once ESR mode setup has been completed
boolean ESRerror;                 //Set if no count returned
byte saveADCSRA;                  //Holds previous value of ADCSRA reg when switching between LC and ESR
int iESRcount;                    //Number of pulses from ESR reading
int iESRrange;                    //0 = Low, 1 = Med, 2 = High
float ESRzeroVal;
float ESRohms;
unsigned long TimeStart;
unsigned long TimePassed;
String ESRstr;

//Interrupt routine for ESR...
ISR(ANALOG_COMP_vect)
{
if (ACSR & (1<<ACO)) // ACO is set?
  {
  iflag = true;
  }
} //ESR Done

void setup() {
  pinMode(ESRMODE, INPUT);               //Setup ESR selection input on D2
  pinMode(RANGEH,OUTPUT);
  pinMode(RANGEM,OUTPUT);
  pinMode(RANGEL,OUTPUT);
  pinMode(DISCHARGE, OUTPUT);
  pinMode(ESRPULSE, INPUT);
  pinMode(ESRRAMP, INPUT);
  cli();                            //Global interrupt disable
  ADCSRB = 0;                       // AMUX disabled
  DIDR1 = (1 << AIN1D) | (1 << AIN0D);      // Disable digital inputs at AIN0 and AIN1
  ACSR =                            //Enable comparator and interrupt on rising edge
    (0 << ACD) |                    //Analog Comparator: Enabled
    (0 << ACBG) |                   //Disable Band Gap input and use AIN0 as +ve input,
    (0 << ACO)|                     //Analog Comparator Output: OFF
    (1 << ACI)|                     //Analog Comparator Interrupt Flag: Clear Pending Interrupt by setting the bit
    (1 << ACIE) |                   //ACOMP Interrupt enabled
    (0 << ACIC)|                    //Analog Comparator Input Capture: Disabled
    (1 << ACIS1) | (1 << ACIS0);    //Comparator Interrupt on rising edge
  sei();                            //Global interrupt enable
  Wire.begin();                     //I2C control library
  lcd.begin(20, 4);                 //Change to (16,4) for 16-character 4-line LCDs
  lcd.clear();
  lcd.setBacklight(255);
  Serial.begin(115200); 
}

void loop(){
  static bool ESRmodeSelected = 1;            //ESR-LC mode selection  ESR = 1  LC = 0
  bool zeroPressed;
  //ESRmodeSelected = digitalRead(ESRMODE);   //Determine if in ESR or LC mode
  if (ESRmodeSelected) {
    if (!iESRsetupDone){esrSetup();}          //Run once whilst in ESR mode
    iESRcount = getESRcount();                //Take an ESR reading
    getESRohms(true);
    //Process and display ESR result
    //When outputing the result to the LCD, if ESRohms > 99.99 then display Over Range
    //LCD ouput...
    lcd.clear();
    lcd.setCursor(0,0);
    lcd.print(F("ESR: "));
    if(!ESRerror){
      if(ESRohms<200){
        lcd.print(ESRohms,(2-iESRrange));         //Print result: dec for low=2, med=1,high=0
        lcd.print(F(" ohms"));
        }else{
        lcd.print(F("Over range!"));
        }
      }else{
      lcd.print(F("Read Error!"));
    }
    lcd.setCursor(0,2);                       //3rd line
    lcd.print(F("Range: "));
    lcd.print(ESRstr);
    lcd.setCursor(0,3);                       //4th line
    lcd.print(F("Leads: "));
    lcd.print(ESRzeroVal,2);
    lcd.print(F(" ohms"));
    //check if ESR zero button pressed else wait 1 sec...
    zeroPressed = false;
    TimeStart = millis();
    TimePassed = 0;
    while ((TimePassed < 1500)&&(!zeroPressed)){
      zeroPressed = !(digitalRead(ESRZERO));
      TimePassed = (millis()-TimeStart);
    }
    if(zeroPressed) {ESRzero();}
  }
}


void esrSetup(){
  pinMode(ESRZERO, INPUT);
  pinMode(RAMPDISABLE,OUTPUT);
  //Note - ADCSRA is used by LC meter so preserve and restore reg settings when switching between LC and ESR
  saveADCSRA=ADCSRA;
  esrStopState();                         //Set outputs to stop mode
  lcd.clear();
  lcd.setCursor(0,0);
  EEPROM.get(25,ESRzeroVal);              //Get any saved value for the ESR meter leads resistance
  if(isnan(ESRzeroVal)){ESRzeroVal = 2;}  //Zero value never been setup - set high to force zeroing
  //Serial.println(ESRzeroVal);
  if((ESRzeroVal<0)||(ESRzeroVal > 1)) {
    ESRzero();                            //execute zero process
  }else{
    //Serial.println(F("ESR Setup done"));
    lcd.print(F("ESR Mode"));
    lcd.setCursor(0,1);                   //Line 3
    lcd.print(F("Silicon Chip"));
    lcd.setCursor(0,3);
    lcd.print(F("Zero value:"));
    lcd.print(ESRzeroVal,2);
  }
  iESRsetupDone = true;                   //ESR setup completed - don't repeat if still in ESR mode
  delay(2000);
}


int getESRcount(){
  //Start off with High range - lowest charging current using Q1 for 10 - 100 ohms
  //If count<10 then switch to Q2 for range 1 - 9.9 ohms
  //If still <10 then switch to Q3 for range 0.1 - 1 ohm.
  //If count > 100 then change to next highest range.
  bool rangeOK;
  int rangePin;
  bool countDone;
  int count;
  int rangeChanges = 0;    //Number of times range was changed
  int lastRange;
  TimeStart = millis();
  TimePassed = 0;
  ESRerror = false;
  rangeOK = false;
  iESRrange = 2;  //High range
  while(!rangeOK) {
    //Serial.print(F("ESR Range:"));
    //Serial.println(iESRrange);
    count = 0;
    countDone = false;
    rangePin = (iESRrange+10);
    digitalWrite(DISCHARGE,HIGH);             //Discharge capacitor under test
    digitalWrite(RAMPDISABLE,LOW);            //Enable Ramping input
    while((!countDone)&&(TimePassed <4000)){  //Count pulses for current range - max 4 secs to prevent infinite loop
      count++;                                //Increment count
      delayMicroseconds(480);                 //Wait for 480uS whilst ramp increases (originally 492uS)
      ACSR |= B00010000;                      //Clear comparator interrupt bit if set
      iflag = false;                          //ensure interrupt flag is cleared
      digitalWrite(DISCHARGE,LOW);            //Remove capacitor discharge signal to enable pulse to occur
      digitalWrite(rangePin,LOW);             //Start short charge pulse of capacitor being tested
      delayMicroseconds(25);                  //Charge capacitor for 25uS (better stability than using 8uS).
      digitalWrite(rangePin,HIGH);            //Stop the charging pulse
      digitalWrite(DISCHARGE,HIGH);           //Discharge capacitor under test
      if(!iflag){                             //Comparator output stayed low - ramp voltage > charge pulse
        digitalWrite(RAMPDISABLE,HIGH);       //Disable Ramping input
        countDone = true;
      }else{
        Serial.print(F("Count:"));            //Use the program memory to hold the text data
        Serial.println(count);                //Print the count value
      }
      TimePassed = (millis()-TimeStart);
     }
    //Check if range is ok
    lastRange = iESRrange;                    //remember the current range
    if ((count>=10)&&(count<=100)) {          //Count is between 10 and 100 so good range.  Stop - we have a result.
      rangeOK = true;} else {
        if(count<10){                         //Count < 10 so try a lower range - if already on lowest then stop and accept result.
          if(iESRrange>0) {iESRrange--;} else {rangeOK = true;}
        } else {                              //count is > 100 lets try next higher range
              if (iESRrange<2) {iESRrange++;} else {rangeOK = true;}    //Go to next higher range - if at highest then accept the result.
      }
    }
    if (iESRrange != lastRange) {rangeChanges++;} //Range changed so increment the counter.
    if (rangeChanges>4) {rangeOK = true;}         //Range selection is flipping between two ranges - accept current range and result
  }
  //output to serial monitor
  Serial.println();                       //Move to new line
  Serial.print(F("Count:"));              //Use the program memory to hold the text data
  Serial.print(count);                    //Print the count value
  Serial.print("\t");                     //Print a tab
  Serial.print(F("ESR Range:"));
  Serial.print(iESRrange);
  Serial.print("\t");
  Serial.print(F("Range changes:"));
  Serial.println(rangeChanges);
  iESRrange = lastRange;                  //Reset iESRrange to the range associated with the count value
  esrStopState();
  if(TimePassed >=4000){ESRerror = true;}
  return count;
}

void getESRohms(bool lessZero){
  switch(iESRrange){
      case 0:
        ESRohms = (float)iESRcount/100;
        ESRstr = "Low";
        break;
       case 1:
        ESRohms = (float)iESRcount/10;
        ESRstr = "Med";
        break;
       case 2:
        ESRohms = (float)iESRcount;
        ESRstr = "High";
        break;  
    }
    if(lessZero) {ESRohms = ESRohms - ESRzeroVal;}
}

void esrStopState(){
    digitalWrite(RAMPDISABLE,HIGH);
    digitalWrite(RANGEH,HIGH);
    digitalWrite(RANGEM,HIGH);
    digitalWrite(RANGEL,HIGH);
    digitalWrite(DISCHARGE,HIGH);
}


void ESRzero() {
  bool zeroDone = false;
  lcd.clear();
  lcd.setCursor(0,0);
  lcd.print(F("Short test leads"));
  lcd.setCursor(0,1);                   //Line 2
  lcd.print(F("and press ZERO.."));
  delay(1000);
  while(!zeroDone){
    iESRcount = getESRcount();
    getESRohms(false);
    if(!(digitalRead(ESRZERO))){        //Zero pressed
      lcd.clear();
      lcd.setCursor(0,0);
      if((ESRohms < 1)&&(iESRrange == 0)){
        EEPROM.put(25,ESRohms);            //use 1 instead of 25 if not integrating with LC meter
        ESRzeroVal = ESRohms;
        zeroDone = true;
        lcd.print(F("Zeroing process"));
        lcd.setCursor(0,1);                   //Line 2
        lcd.print(F("   completed."));
      }else{                                  //invalid zero result...
        lcd.print(F("Invalid reading"));
        lcd.setCursor(0,1);                   //Line 2
        lcd.print(F("or bad leads."));
        lcd.setCursor(0,3);                   //Line 4
        lcd.print(F("   ABORTING.."));
      }
      delay(2000);
    }else{
      if(iESRcount < 100){
        lcd.setCursor(4,3);
        lcd.print(F("      "));
        lcd.setCursor(4,3);
        lcd.print(ESRohms,2);
        }else{
          lcd.setCursor(4,3);
          lcd.print(F("O----O"));
        }
      delay(500);
    }
  }
}
