; File DVM.ASM
; Uses assembly code for PIC16F84 microcontroller
; A 3-digit voltmeter with 12V or 24V operation
; Displays voltage from about 5V to 23.1V when powered from 12V battery
; Displays from about 12V to 46.2V when powered from 24V battery
; 100mV display resolution in 12V mode
; 200mV display resolution in 24V mode
; Remote voltage sensing for direct battery monitoring if required.
; Minimum hold voltage display (can display the minimum voltage that
; has been read since power has been on). This checks battery voltage
; during starting. 	
; A-D converter for voltage measurement uses PWM (pulse width modulation)
; which is converted to DC. This is compared in a comparator against divided 
; battery voltage. PWM sequence is initially set at 50% duty cycle (2.5V) and 
; the comparator checks if divided battery voltage is lower or higher than this. 
; It then adds or subtracts a 25% duty cycle (1.25V) and checks against the
; divided battery voltage again. This process continues for 8 cycles taking
; or adding smaller amounts of voltage (2.5v, 1.25V, 0.625V, .3125V,
; .15625V, .078125V, .0390625V and .01953125V) to obtain an 8-bit A-D
; conversion. The A-D converter is a PWM sucessive approximation type.
; The PWM output is at 1953Hz and is maintained for around 65.5ms at the
; fixed rate before changing to the next voltage in the Sucessive
; Approximation sequence. This allows the simple RC filter to settle so
; the output from the comparator is accurate. A-D can operate from 1.9 to
; 23.1V in 12V mode. (24V mode is simply the result multiplied by 2 with
; input attenuator dividing by an extra factor of 2)
;
; Processor pin allocations are as follows:
; RA0 Output disp1 driving transistor for common anode display
; RA1 Output disp2
; RA2 Output disp3
; RA3 Output for PWM to comparator 
; RA4 Input from switch

; RB0 Input from comparator
; RB1 c segment drive for seven segment LED display
; RB2 d segment drive
; RB3 e segment drive
; RB4 f segment drive
; RB5 a segment drive
; RB6 b segment drive
; RB7 g segment drive

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

EEPROM1		equ	H'00'	; non-volatile storage for 12/24V

; RAM

DISP1		equ	H'0C'	; working storage for Display1 
DISP2		equ	H'0D'	; working storage for Display2 
DISP3		equ	H'0E'	; working storage for Display3 
STATUS_TMP 	equ 	H'0F'	; temp storage for status during interrupt
W_TMP		equ	H'10'	; temporary storage for w during interrupt
FLAG_1		equ	H'11'	; bit 0 is multiplex or PWM flag, bit1 is for interrupt count
				; bit 2 is 12/24V select, bit 6 is EEPROM write repeat
PWM_CNT		equ	H'12'	; counter for PWM output
LOW_TME		equ	H'13'	; PWM low time
BIN_0		equ	H'14'	; binary value msd
BIN_1		equ	H'15'	; binary value 
BCD_0		equ	H'16'	; binary value space for x 2 in 24V range
BCD_1		equ	H'17'	; display value MS
BCD_2		equ	H'18'	; display value LS
TEMP		equ	H'19'	; temporary register
CNT_16		equ	H'1A'	; counter for BCD conversion
TEMP_2		equ	H'1B'	; temp storage during EEPROM write
LOW_1		equ	H'1C'	; lowest value of voltage reading
TEMP_1		equ	H'1D'	; temp store during 12/24V selection
EXTN		equ	H'1E'	; delay extension value

; preprogram EEPROM DATA
	
	ORG     2100
	DE	0x00		; EEPROM1 with 00 default value (12V)

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data. 

SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'10111100'	; 7-segment code 1
	retlw 	B'00010010'	; 7-segment code 2
	retlw 	B'00011000'	; 7-segment code 3
	retlw 	B'00101100'	; 7-segment code 4
	retlw 	B'01001000'	; 7-segment code 5
	retlw 	B'01000000'	; 7-segment code 6
	retlw 	B'10011100'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00001000'	; 7-segment code 9


;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3 in sequence plus 
; uses internal timer to initiate display update
; produces PWM output with duty set by LOW_TME register
; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  

; PWM output routine

PWM_MPX	movf	LOW_TME,w	; PWM low time - w
	btfsc	FLAG_1,0	; check bit 0
	goto	LOW_OUT
	bsf	FLAG_1,0	; clear if set
	nop			; output cycle time adjust
	bcf	PORTA,3		; RA3 low
	sublw	0x02		; PWM low time
	nop			; align timer with cycles of instruction
	addwf	TMR0,f		; 4MHz so low time is inverse of 4MHz/4/2/256  
	bcf	INTCON,T0IF	; clear TMRO interrupt flag
	goto	LITCHK		; multiplex
LOW_OUT	nop			; set equal time between high out and low out
	bsf	PORTA,3		; RA3 high
	addlw	0x02		; align timer
	addwf	TMR0,f		; 4MHz so high time is inverse of 4MHz/4/2/256  
	bcf	FLAG_1,0	; set if clear
	bcf	INTCON,T0IF
	goto	RECLAIM

; multiplex display routine

LITCHK	bsf	FLAG_1,1	; set flag for interrupt count
	btfss	PORTA,0		; skip if display 1 not lit
	goto	LIT1		; display 1 lit
	btfss	PORTA,1		; skip if display 2 not lit
	goto	LIT2		; display 2 lit
	btfss	PORTA,2		; skip if display 3 not lit				
	goto	LIT3		; display 3 lit
	movlw	B'11111110'	; seven segments off
	movwf	PORTB
	bcf	PORTA,2		; no displays lit so set disp3 (used if error in output)
	goto 	LITCHK		; goto check which is lit now
LIT1	bsf	PORTA,0		; disp1 off
	movf	DISP2,w		; disp2 to w
	movwf	PORTB		; seven segment value to portB
	bcf	PORTA,1		; disp2 powered
	goto	RECLAIM		; end of multiplex
LIT2	bsf	PORTA,1		; disp2 off
	movf	DISP3,w		; disp3 to w
	movwf	PORTB		; portB has seven-segment code 
	bcf	PORTA,2		; disp3 powered
	goto	RECLAIM		; end of multiplex
LIT3	bsf	PORTA,2		; disp3 off
	movf	DISP1,w		; disp1 to w
	movwf 	PORTB
	bcf	PORTA,0		; disp1 powered

; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN	movlw	B'11111110'	; code for 7-segment display off
	movwf 	DISP1		; display values
	movwf	DISP2		; initial display value
	movwf	DISP3		; initial display value
	movlw	0xFF
	movwf	LOW_1		; set lowest voltage reading to maximum first		
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'10000000'	; w = 10000000 binary
	movwf	OPTION_REG	; TMRO prescaler, PORTB pullups disabled
	movlw   B'00010000'	; w = 10000 binary (RA0-RA2, RA3 outputs, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00000111'
	movwf	PORTA		; portA RA0 high, RA1,RA2 outputs high, RA3 low
	
	movlw	EEPROM1		; address for EEPROM1
	call	EEREAD
	movwf	FLAG_1		; store in flag, bit 2 selects 12/24V
 
; check for pressed switch
	
	btfsc	PORTA,4		; is switch pressed
	goto	NO_OPEN		; switch open so no change
	btfss	FLAG_1,2	; test for 12/24V flag	
	goto	SET_24		; set for 24V
	bcf	FLAG_1,2	; set 12V
	movlw	B'11100010'	; L on display for 12V
	movwf	TEMP_1
	goto	MPX_DIS
SET_24	bsf	FLAG_1,2	; set 24V
	movlw	B'00100100'	; H on display for 24V
	movwf	TEMP_1
MPX_DIS bsf	PORTA,0		; display off
	movlw	0xFF
	movwf	PORTB		; clear all segments
	call	DELMOR		; more delay time (multiplex time)
	call	DELMOR		; more delay time 
	btfsc	PORTA,4		; wait for switch to open
	goto	MODESTO		; store setting when switch open

; multiplex display until switch opens

	bcf	PORTA,0		; RA0 low so display lit
	movf	TEMP_1,w
	movwf	PORTB		; 
	call	DELMOR		; (delay more) more delay time (multiplex)
	goto 	MPX_DIS

; write new mode option to EEPROM1

MODESTO	movlw	EEPROM1		; address for EEPROM1
	movwf	EEADR		; address for write
	movf	FLAG_1,w	; flag stored
	call	EWRITE		; write to EEPROM

; interrupt enable and program now runs

NO_OPEN	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	bsf	INTCON,GIE	; set global interrupt enable for above

; delay for input voltage to settle

	movlw	0x80		; delay extension
	movwf	EXTN
WAIT	call	DELMOR		; delay
	decfsz	EXTN		; out from delay when zero
	goto	WAIT
	
; Successive Approximation for A-D converter
	
NEWVAL	clrf	LOW_TME
	bsf	LOW_TME,7	; bit 7 set
	call	SAR
	btfss 	PORTB,0		; comparator out. if RB0 high then value below 128 (high bit 7)
	bcf	LOW_TME,7	; bit 7 cleared
	bsf	LOW_TME,6	; next LS bit test
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 6)
	bcf	LOW_TME,6
	bsf	LOW_TME,5
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 5)
	bcf	LOW_TME,5
	bsf	LOW_TME,4
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 4)
	bcf	LOW_TME,4
	bsf	LOW_TME,3
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 3)
	bcf	LOW_TME,3
	bsf	LOW_TME,2
	call 	SAR
	btfss	PORTB,0		; if RB0 high then value below (high bit 2)
	bcf	LOW_TME,2
	bsf	LOW_TME,1
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 1)
	bcf	LOW_TME,1
	bsf	LOW_TME,0
	call 	SAR
	btfss	PORTB,0		; bit 0 test
	bcf	LOW_TME,0

; check if switch pressed. If so then load LOW_1 to BIN_0 then to NO_LOW

	btfsc	PORTA,4		; is switch pressed
	goto	REAL_V		; real time voltage
	movf	LOW_1,w		; low value 
	movwf	BIN_0		; low_1 to bin_0
	goto	NO_LOW		; low value for display
REAL_V	movf	LOW_TME,w	; store 255 minus lowtime value into binary0 register
	sublw	D'255'		; get reverse value ie high output time
	movwf	BIN_0		; binary value of voltage
	subwf	LOW_1,w		; subtract BIN_0 from LOW_1 (lowest voltage)
	btfss	STATUS,c	; if set then equal or smaller
	goto	NO_LOW
	movf	BIN_0,w
	movwf	LOW_1		; low_1 has new low voltage value

; check if overrange or under range

NO_LOW	movf	BIN_0,w		; binary value to w
	sublw	D'18'		; check if less than 1.9V
	btfsc	STATUS,c	; if c is 0 then larger
	goto	SMALLR		; smaller so load LO into disp
	movf	BIN_0,w
	sublw	D'231'		; check if more than 23.1
	btfsc	STATUS,c	; more than so load OL into disp	
	goto	INRNGE		; value in range	
LRGER	movlw	B'10000000'	; "O" as in Overload
	movwf	DISP3	
	movlw	B'11100010'	; "L" as in overLoad
	movwf	DISP2
	movlw	B'11111110'	; blank display
	movwf	DISP1
	goto	NEWVAL
SMALLR	movlw	B'11100010'	; "L" into display as in Low
	movwf	DISP3	
	movlw	B'10000000'	; "O" as in LO
	movwf	DISP2
	movlw	B'11111110'	; blank display
	movwf	DISP1
	goto	NEWVAL
INRNGE	btfss	FLAG_1,2	; bit 2 is 24 or 12V range setting
	goto	TWELV
	rlf	BIN_0,f		; 24 volt so multiply by 2
	bcf	BIN_0,0		; clear 0 bit to prevent carry to bit 0 
	clrf	BIN_1		; MS binary value
	btfsc	STATUS,c	; check if carry set
	bsf	BIN_1,0		; set bit 0 if carry set
TWELV	call	BCD		; convert binary number to BCD
	movf	BCD_1,w		; BCD1 value to w
	xorlw	0x00		; compare with 0
	btfss	STATUS,z	; skip if zero
	goto 	SHOW		; show digit on disp3
	movlw	B'11111110'	; 7-segment display off
	goto	BLNK		; value to blank display
SHOW	movf	BCD_1,w	
	call 	SVNSEG		; convert to 7-segment value for display
BLNK	movwf	DISP3		; transfer BCD MSD to DISPlay 3
	movf	BCD_2,w		; contains packed BCD value
	andlw	0x0F		; extract LS bits
	call	SVNSEG
	movwf	DISP1		; place in display 1
	swapf	BCD_2,w		; swap uper and lower 4-bits
	andlw	0x0F		; extract bits
	call	SVNSEG
	movwf	DISP2		; place in middle digit register
	goto 	NEWVAL		; read input voltage again

; ******************************************************************************
; Subroutines

; delay period for multiplex rate

DELMOR	movlw 	D'255'		; delay period
	movwf	TEMP_2
SMALER	decfsz	TEMP_2,f	; reduce temp_2
	goto	SMALER		; temp_2 smaller by 1
	return			; end delay

; Subroutine for PWM cycle period for successive approximation A-D converter
	
SAR	movlw	0x80		; number of interrupts between PWM changes
	movwf	PWM_CNT
CNT_AGN	bcf	FLAG_1,1	; bit set in interrupt at multiplex rate
CNT_NOW	btfss	FLAG_1,1	; look at flag
	goto	CNT_NOW		; wait till flag set at multiplex interrupt 
	decfsz 	PWM_CNT,f	; reduce this value 
	goto	CNT_AGN		; cycle 
	return

; Subroutine to convert from 16-bit binary to 3-digit BCD (packed)
; Binary value is in BIN0 & BIN1. 
; Result in BCD is in BCD0, BCD1 & BCD2.  
; BCD0 is MSB, BCD2 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'16'
	movwf	CNT_16		; 16 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
	clrf	BCD_2
	

LOOPBCD	rlf	BIN_0,f
	rlf	BIN_1,f		; LSB shift left binary registers
	rlf	BCD_2,f		; LSB shift left BCD registers
	rlf	BCD_1,f
	rlf	BCD_0,f

	decfsz	CNT_16,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_2		; BCD LSB address
	movwf	FSR		; pointer for BCD2
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_1
	movwf	FSR
	call 	ADJBCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-2)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_RD	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_RD		; wait for low RD (read RD)	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	bcf	FLAG_1,6	; EEPROM write repeat flag
	movwf	TEMP_2		; store w in temporary storage
EWRIT	movf	TEMP_2,w	; place value in w (used for read data sequence) 
	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	
; read EEPROM DATA and check if written correctly
	
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_AGN	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_AGN		; wait for low RD	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	subwf	EEDATA,w	; compare read value with value stored
	btfsc	STATUS,z	; skip if not the same
	return			; value correctly written 
	btfsc	FLAG_1,6	; write repeat bit, skip if clear and rewrite
	return
	bsf	FLAG_1,6	; set repeat flag rewrite once only 
	goto 	EWRIT		; rewrite as not written correctly
	
	end	
