; Tachometer program
; Displays rpm from 000rpm to 9900rpm, bargraph shows rpm as pseudo analog display.
; The 10 LEDs are programmed to switch on at predetermined rpm. LEDs 8-10 light together
; showing red line. Output for limiter is activated at red line.
; The programming of LEDs 1-8 is automatic once redline value is entered for rpm8
; Thus if 4000rpm is entered for LED 8 then LED1 will be set at 500, LED2 at 1000,
; LED3 at 1500, LED4 at 2000, LED5 at 2500, LED6 at 3000, LED7 at 3500 and LED8-10 at 4000.
; The LED switch points can alternatively be set for any other value
; Dot or Bar graph display can be selected by pressing the Mode switch at power up
; Tacho is set in rpm mode at power up. The Mode switch selects between calibrate, RPM1-RPM8
; and the hysteresis setting for the bar display
; RPM values can be changed with up or down switches. Calibrate and Hysteresis can only be 
; set with the up switch  

; CALIBRATE TABLE:
; pulses per revolution, update time in seconds and calibration value for interrupt counter 
; where interrupt occurs every 500us. Display shows CAL from 1-12.
 
; cyl/4-stroke  cyl/2-stroke    P/rev   Count 	Calibration
;					period   
; 1				0.5	1.2	2400-1
; 2		1		1	0.6	1200-1
; 3				1.5	0.4	800-1
; 4		2		2	0.3	600-1
; 5				2.5	0.24	480-1
; 6		3		3	0.2	400-1
; 7				3.5	0.1714	342-1
; 8		4		4	0.150	300-1
; 9				4.5	0.0666	133-1
; 10		5		5	0.060	120-1
; 11				5.5	0.0545	109-1
; 12		6		6	0.050	100-1						 

; Processor pin allocations are as follows:

; RA0 Output common for bargraph LEDs driving transistor 
; RA1 Output disp 4
; RA2 Output disp 3
; RA3 Output for rev limit
; RA4 Input from switches

; RB0 Input from ignition coil pulses (interrupt)
; RB1 c segment drive for seven segment LED display
; RB2 d segment drive ..
; RB3 e segment drive
; RB4 f segment drive
; RB5 a segment drive
; RB6 b segment drive
; RB7 g segment drive

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

; EEPROM DATA

EEPROM0		equ	H'0F'	; non-volatile storage for limiter LED and output
EEPROM1		equ	H'10'	; non-volatile storage for change1 LED alarm 
EEPROM2		equ	H'11'	; non-volatile storage for change2 LED alarm
EEPROM3		equ	H'12'	; non-volatile storage for change3 LED alarm
EEPROM4		equ	H'13'	; non-volatile storage for change4 LED alarm 
EEPROM5		equ	H'14'	; non-volatile storage for change5 LED alarm
EEPROM6		equ	H'15'	; non-volatile storage for change6 LED alarm
EEPROM7		equ	H'16'	; non-volatile storage for change7 LED alarm 
EEPROM8		equ	H'17'	; non-volatile storage for display cal value
EEPROM9		equ	H'18'	; non-volatile storage of LS byte of calibration number
EEPROMA		equ	H'19'	; non-volatile storage of MS byte of calibration number
EEPROMB		equ	H'1A'	; non-volatile storage for dot bar selection
EEPROMC		equ	H'1B'	; non-volatile storage for hysteresis on graph and limiter

; RAM

DISP1		equ	H'0C'	; working storage for Display1 value (DISP3 on circuit) 
DISP2		equ	H'0D'	; working storage for Display2 value (DISP4 on circuit)
DISP3		equ	H'0E'	; working storage for Display3 value (LEDS 1-7 on circuit)

STATUS_TMP 	equ 	H'10'	; temp storage for status during interrupt
W_TMP		equ	H'11'	; temporary storage for w during interrupt
VALUE_1		equ 	H'12'	; delay value storage			
VALUE_2		equ	H'13'	; delay value storage
FLAG_1		equ	H'14'	; bit0 is timecount period update flag,bit1 for switch opening,
				; bit4 is display update flag, bit6 EEPROM write repeat 
FLAG_2		equ	H'15'	; bit0 is dot bar selector
CNT_8		equ	H'16'	; BCD conversion counter 
TEMP		equ	H'17'	; temporary storage during BCD conversion
TEMP_1		equ	H'18'	; temporary working storage
TEMP_2		equ	H'19'	; temp storage
TIME_CNT1	equ	H'1A'	; counter for pulse count period
TIME_CNT2	equ	H'1B'	; counter for pulse count period
TIME_CNT3	equ	H'1C'	; working lsb calibration number stored in EEPROM9
TIME_CNT4	equ	H'1D'	; working msd calibration number stored in EEPROMA
PULSE_CNT	equ	H'1E'   ; counter for tacho pulses  
TACHO_0		equ	H'1F'	; latched value for tachometer obtained from PULSE_CNT
MODE_1		equ	H'20'	; mode indicator
AUT_SET		equ	H'21'	; automatic setting for graph display (storage of difference)
HYSTER		equ	H'22'	; working hysteresis value
BCD_0		equ	H'23'	; bcd value after conversion
BCD_1		equ	H'24'	; bcd value
BIN_0		equ	H'25'	; binary value for BCD conversion
RPM_0		equ	H'26'	; rev limit 0 for limiter output (last 3-LEDS LED8-10)
RPM_1		equ	H'27'	; rev limit 1 (LED1)
RPM_2		equ	H'28'	; rev limit 2
RPM_3		equ	H'29'	; rev limit 3
RPM_4		equ	H'2A'	; rev limit 4
RPM_5		equ	H'2B'	; rev limit 5
RPM_6		equ	H'2C'	; rev limit 6
RPM_7		equ	H'2D'	; rev limit 7 (LED7)	
CALVAL		equ	H'2E'	; current calibration value for display
TIME_CNT5	equ	H'2F'	; counter for update count period
TIME_CNT6	equ	H'30'	; counter for update count period
TACHO_1		equ	H'31'	; temporary tachometer storage

; preprogram EEPROM DATA (00-3F from 2100-213F)
	
	ORG     210F			; start at 0F
	DE	D'40'			; limiter initial preset 4000rpm
	DE	D'00', D'00', D'00'	; preset EEPROM1-EEPROM3 0 rpm
	DE	D'00', D'00', D'00'	; preset EEPROM4-EEPROM6 0 rpm
	DE	D'00'			; preset EEPROM7 0 rpm
	DE	D'04'			; set EEPROM8 cal cyl/4-stroke to 4
	DE	0x57, 0x02		; set initial calibration EEPROM9 & EEPROMA @ 0.3seconds
	DE	0x00, 0x01		; clear EEPROMB for flag_2 dot/bar, EEPROMC hysteresis =1

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data.  
 
SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'10111100'	; 7-segment code 1
	retlw 	B'00010010'	; 7-segment code 2
	retlw 	B'00011000'	; 7-segment code 3
	retlw 	B'00101100'	; 7-segment code 4
	retlw 	B'01001000'	; 7-segment code 5
	retlw 	B'01000000'	; 7-segment code 6
	retlw 	B'10011100'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00001000'	; 7-segment code 9
	retlw	B'01111110'	; code for a "-" overrange
	retlw	B'00100100'	; code for "H" (hysteresis)
	
; subroutine to convert from display calibration numbers (1-12 etc) to actual 
; required count period calibration numbers.

CALBRT	addwf	PCL,f		; add value of display calibration to program counter
	nop			; align with initial 1 in display
	retlw	0x5F		; 2400-1	
	retlw	0x09
	retlw	0xAF		; 1200-1
	retlw	0x04
	retlw	0x1F		; 800-1
	retlw	0x03
	retlw	0x57		; 600-1
	retlw	0x02
	retlw	0xDF		; 480-1	
	retlw	0x01
	retlw	0x8F		; 400-1
	retlw	0x01
	retlw	0x55		; 342-1
	retlw	0x01
	retlw	0x2B		; 300-1
	retlw	0x01
	retlw	D'132'		; 133-1
	retlw	0x00
	retlw	D'119'		; 120-1
	retlw	0x00
	retlw	D'108'		; 109-1
	retlw	0x00
	retlw	D'99'		; 100-1
	retlw	0x00

;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3, in sequence plus 
; blanking leading zeros on display for Disp2  
; uses internal timer to initiate display update
; also updates time period counters which set period of tacho pulse counting
; checks tachometer reading against change settings
; tachometer signal update

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  

; which interrupt (INT or TMRO)

	btfsc	INTCON,T0IF	; TMRO overflow interrupt flag 
	goto	LIGHT		; TMRO overflow so multiplex display
SENSOR	btfsc	INTCON,INTF	; INT interrupt flag set then goto tacho
	goto	TACHO		; tachometer signal update 
	goto	RECLAIM		; end of interrupt reclaim w and status

; display update 

LIGHT	bcf	INTCON,T0IF	; clear interrupt flag
	movf	TMR0,w		; timer value	
				; freq is 4MHz/4/2/(256(-8+2)) = 2kHz or 0.5ms period
	addlw	0x08		; add to timer register and takes 2 cycles to start counting
	movwf	TMR0		; place in timer register
 	bsf	FLAG_1,0	; multiplex flag indicator
	btfss	PORTA,1		; skip if display 1 not lit
	goto	LT1		; display 1 lit
	btfss	PORTA,2		; skip if display 2 not lit
	goto	LT2		; display 2 lit
	btfss	PORTA,0		; skip if display 3 not lit				
	goto	LT3		; display 3 lit
	movlw	B'11111110'	; seven segments off
	movwf	PORTB
	bcf	PORTA,0		; no displays lit so set disp3 (used if error in output)
	goto 	LIGHT		; goto check which is lit now
LT1	bsf	PORTA,1		; disp1 off
	movf	DISP2,w		; look at display2
	xorlw	0x00		; compare disp2 with 0
	btfss	STATUS,Z	; skip if 0
	goto	NOBLNK1		; do not blank disp2
	movlw	B'11111110'	; 7-segment display off
	goto	BLANK1		; blank disp2 when disp2 and disp3 = 0
NOBLNK1	movf	DISP2,w		; disp2 to w
	call 	SVNSEG		; goto subroutine to get seven segment code 
BLANK1	movwf	PORTB		; seven segment value to portB
	bcf	PORTA,2		; disp2 powered
	goto	TIMECNT		; end of multiplex
LT2	bsf	PORTA,2		; disp2 off
	movf	DISP3,w		; disp3 to w
	movwf	PORTB		; portB lights led display
	bcf	PORTA,0		; disp3 powered
	goto	TIMECNT		; end of multiplex
LT3	bsf	PORTA,0		; disp3 off
	movf	DISP1,w		; DISP1 to w
	call	SVNSEG
	movwf 	PORTB
	bcf	PORTA,1		; DISP1 powered
	
; update time period counters for tacho pulse counter

TIMECNT	movf	TIME_CNT4,w	; MS Byte of calibration value to w
	xorwf	TIME_CNT2,w	; compare MS Byte counter with calibration
	btfss	STATUS,z	; skip if equal
	goto	INC_CT1		; 
	movf	TIME_CNT3,w	; LS Byte of calibration value
	xorwf	TIME_CNT1,w	; compare LS byte counter with calibration
	btfss	STATUS,z	; skip if equal
	goto	INC_CT1		; increment time counters
	clrf	TIME_CNT1	; clear timer counter 1
	clrf	TIME_CNT2	; and timer 2
	movf	PULSE_CNT,w	; look at pulse counter value
	movwf	TACHO_1		; tachometer value latched pulse counter value
	clrf	PULSE_CNT	; clear pulse counter ready for new RPM update
	bsf	FLAG_1,0	; set time period count update flag 
	goto	SENSOR
INC_CT1 incfsz	TIME_CNT1,f	; increase counter 1 skip if overflowed
	goto 	INC_CT2		; increment second counters
	incf	TIME_CNT2,f	; increase counter 2 when counter 1 overflows
	
; second counter for slower update times when calibration is >2p/rev
	
INC_CT2	movf	TIME_CNT6,w	; MS Byte of counter w
	xorlw	0x02		; compare MS Byte counter with 0.3 second update 
	btfss	STATUS,z	; skip if equal
	goto	INC_CT3		; 
	movf	TIME_CNT5,w	; LS Byte of counter
	xorlw	0x57		; compare LS byte counter with calibration
	btfss	STATUS,z	; skip if equal
	goto	INC_CT3		; increment time counters
	clrf	TIME_CNT5	; clear timer counter 1
	clrf	TIME_CNT6	; and timer 2
	bsf	FLAG_1,4	; set display update flag to signal update display reading
	movf	TACHO_1,w	; tacho value
	movwf	TACHO_0		; tacho value to operational tacho value every 0.3s
	goto	RPM_LED		; update 
INC_CT3	incfsz	TIME_CNT5,f	; increase counter 1 skip if overflowed
	goto 	RPM_LED		; 
	incf	TIME_CNT6,f	; increase counter 2 when counter 1 overflows
	
; check rpm against change LED settings
	
RPM_LED	movf	MODE_1,f	; mode (tacho, cal, rpm_1, rpm_2, rpm_3, etc)
	btfss	STATUS,z	; if zero bit set then tacho mode
	goto	SENSOR		; do not change if mode not tacho mode
	movf	RPM_1,w		; RPM1 value to w 
	subwf	TACHO_0,w	; subtract RPM1 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM1_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_1,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,2		; RPM1 change LED off
CHK_R2	movf	RPM_2,w		; RPM2 value
	subwf	TACHO_0,w	; subtract RPM2 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM2_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_2,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,4		; RPM2 LED off
CHK_R3	movf	RPM_3,w		; RPM3 value
	subwf	TACHO_0,w	; subtract RPM3 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM3_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_3,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,5		; RPM3 LED off
CHK_R4	movf	RPM_4,w		; RPM4 value to w 
	subwf	TACHO_0,w	; subtract RPM1 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM4_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_4,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,6		; RPM4 change LED off
CHK_R5	movf	RPM_5,w		; RPM5 value to w 
	subwf	TACHO_0,w	; subtract RPM5 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM5_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_5,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,7		; RPM5 change LED off
CHK_R6	movf	RPM_6,w		; RPM6 value to w 
	subwf	TACHO_0,w	; subtract RPM6 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM6_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_6,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,3		; RPM6 change LED off
CHK_R7	movf	RPM_7,w		; RPM7 value to w 
	subwf	TACHO_0,w	; subtract RPM7 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM7_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_7,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	DISP3,1		; RPM7 change LED off
CHK_R8	movf	RPM_0,w		; RPM0 value to w 
	subwf	TACHO_0,w	; subtract RPM0 from latched pulse counter value	
	btfsc	STATUS,c	; if positive or zero (c=1) then set change LED	
	goto	RPM0_ON
	movf	HYSTER,w	; hysteresis value
	subwf	RPM_0,w		; add hysteresis
	subwf	TACHO_0,w
	btfss	STATUS,c	
	bsf	PORTA,3		; limiter off
	goto	SENSOR
	
; dot mode turns off all LEDs below the highest lit LED (FLAG_2,0)

RPM1_ON	bcf	DISP3,2		; RPM1 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R2
	bsf	DISP3,2		; LED1 off
	btfss	DISP3,4		; is LED 2 on
	goto	CHK_R2
	btfss	DISP3,5		; is LED 3 on
	goto	CHK_R2
	btfss	DISP3,6		; is LED 4 on
	goto	CHK_R2
	btfss	DISP3,7		; is LED 5 on
	goto	CHK_R2
	btfss	DISP3,3		; is LED 6 on
	goto	CHK_R2
	btfss	DISP3,1		; is LED 7 on
	goto	CHK_R2
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R2
	bcf	DISP3,2		; RPM1 LED on
	goto	CHK_R2		; check rpm2 
RPM2_ON	bcf	DISP3,4		; RPM2 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R3		; check rpm3
	bsf	DISP3,4		; LED off
	btfss	DISP3,5		; is LED 3 on
	goto	CHK_R3
	btfss	DISP3,6		; is LED 4 on
	goto	CHK_R3
	btfss	DISP3,7		; is LED 5 on
	goto	CHK_R3
	btfss	DISP3,3		; is LED 6 on
	goto	CHK_R3
	btfss	DISP3,1		; is LED 7 on
	goto	CHK_R3
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R3
	movlw	B'00000100'
	iorwf	DISP3,f		; all LEDs off below LED2
	bcf	DISP3,4
	goto	CHK_R3
RPM3_ON	bcf	DISP3,5		; RPM3 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R4		; check rpm4
	bsf	DISP3,5		; LED off
	btfss	DISP3,6		; is LED 4 on
	goto	CHK_R4
	btfss	DISP3,7		; is LED 5 on
	goto	CHK_R4
	btfss	DISP3,3		; is LED 6 on
	goto	CHK_R4
	btfss	DISP3,1		; is LED 7 on
	goto	CHK_R4
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R4
	movlw	B'00010100'
	iorwf	DISP3,f		; all LEDs off below LED3
	bcf	DISP3,5
	goto	CHK_R4		; check rpm4
RPM4_ON	bcf	DISP3,6		; RPM4 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R5		; check rpm5
	bsf	DISP3,6		; LED off
	btfss	DISP3,7		; is LED 5 on
	goto	CHK_R5
	btfss	DISP3,3		; is LED 6 on
	goto	CHK_R5
	btfss	DISP3,1		; is LED 7 on
	goto	CHK_R5
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R5
	movlw	B'00110100'
	iorwf	DISP3,f		; all LEDs off below LED4
	bcf	DISP3,6
	goto	CHK_R5		; check rpm5 
RPM5_ON	bcf	DISP3,7		; RPM5 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R6		; check rpm6
	bsf	DISP3,7		; LED off
	btfss	DISP3,3		; is LED 6 on
	goto	CHK_R6
	btfss	DISP3,1		; is LED 7 on
	goto	CHK_R6
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R6
	movlw	B'01110100'
	iorwf	DISP3,f		; all LEDs off below LED5
	bcf	DISP3,7
	goto	CHK_R6		; check rpm6
RPM6_ON	bcf	DISP3,3		; RPM6 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R7		; check rpm7
	bsf	DISP3,3		; LED off
	btfss	DISP3,1		; is LED 7 on
	goto	CHK_R7
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R7
	movlw	B'11110100'
	iorwf	DISP3,f		; all LEDs off below LED6
	bcf	DISP3,3
	goto	CHK_R7		; check rpm7
RPM7_ON	bcf	DISP3,1		; RPM7 LED on
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	CHK_R8		; check rpm0 
	bsf	DISP3,1		; LED off
	btfss	PORTA,3		; is LED 8 on
	goto	CHK_R8
	movlw	B'11111100'
	iorwf	DISP3,f		; all LEDs off below LED7
	bcf	DISP3,1
	goto	CHK_R8
RPM0_ON	bcf	PORTA,3		; limiter signal
	btfss	FLAG_2,0	; dot flag set or bar not set
	goto	SENSOR		;
	movlw	0xFF
	movwf	DISP3		; all LEDs off
	goto	SENSOR	

; tachometer update

TACHO	bcf	INTCON,INTF	; clear INT flag
	incfsz	PULSE_CNT,f	; increase pulse counter value for each speed sensor input
	goto	RECLAIM
	movlw	0xFF		; max value
	movwf	PULSE_CNT	; place max count in pulse counter do not allow overrange	
	
; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN	clrf	FLAG_1		; clear flags
	clrf	TIME_CNT1	; clear time period counter for tacho pulse count update
	clrf	TIME_CNT2	; clear time period counter
	clrf	TIME_CNT5	; clear time period counter for tacho display update
	clrf	TIME_CNT6	; clear time period counter
	clrf	PULSE_CNT	; clear tacho pulse counter
	clrf	DISP1		; clear display registers
	clrf	DISP2
	clrf	MODE_1		; set mode to rpm display
	movlw	0xFF		; 
	movwf	DISP3		; clear rpm LEDs
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'10000000'	; w = 10000000 binary
	movwf	OPTION_REG	; TMRO prescaler no division, PORTB pullups disabled
	movlw   B'00010000'	; w = 10000 binary (RA0-RA3 outputs, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00001111'
	movwf	PORTA		; portA RA0,RA1,RA2,RA3 outputs high 
	
; get EEPROMB and place in FLAG_2
	
	movlw	EEPROMB		; address for EEPROMB
	call	EEREAD		; read EEPROM storage
	movwf	FLAG_2		; EEPROM to FLAG_2	

; check for pressed Mode switch
	
CHK_MDE	bsf	PORTA,1		; RA1 high so disp2 off
	bsf	PORTA,2		; RA2 high so disp3 off
	nop
	nop
	bcf	PORTA,0		; clear RA0 to check if mode switch pressed
	nop
	nop
	btfsc	PORTA,4		; is mode switch pressed
	goto	NO_OPEN		; switch open so no change
	btfss	FLAG_2,0	; test for bit 0	
	goto	CLR_BAR		; clear bar
	bcf	FLAG_2,0	; set bar
	movlw	B'01100000'	; 'b' on display for bar
	movwf	TEMP_1
	goto	MODEOPN
CLR_BAR	bsf	FLAG_2,0	; set dot/bar flag
	movlw	B'00110000'	; 'd' on display for dot
	movwf	TEMP_1
MODEOPN	bsf	PORTA,1		; display off
	movlw	0xFF
	movwf	PORTB		; display segments off
	bcf	PORTA,0		; clear RA0 to check if mode switch pressed
	call	DELMO		; more delay time (multiplex time)
	call	DELMO		; more delay time 
	btfsc	PORTA,4		; wait for switch to open
	goto	MODESTO		; store repeat setting when switch open

; multiplex display until switch opens

	bsf	PORTA,0		; RA0 high
	nop
	nop
	nop
	bcf	PORTA,1		; RA1 low so display 1 lit
	movf	TEMP_1,w
	movwf	PORTB		; 
	call	DELMO		; more delay time (multiplex)
	goto 	MODEOPN

; write new mode option to EEPROMB

MODESTO	movlw	EEPROMB		; address for EEPROMB
	movwf	EEADR		; address for write
	movf	FLAG_2,w	; flag stored
	call	EWRITE		; write to EEPROM

; get memory in EEPROM0 to EEPROM7 for RPM setting, place in RPM_0 to RPM_7
	
NO_OPEN	movlw	EEPROM0		; address for EEPROM0
	call	EEREAD		; read EEPROM storage
	movwf	RPM_0		; EEPROM0 to RPM_0
	movlw	EEPROM1		; address for EEPROM1
	call	EEREAD		; read EEPROM storage
	movwf	RPM_1		; EEPROM1 to RPM_1
	movlw	EEPROM2		; address for EEPROM2
	call	EEREAD
	movwf	RPM_2		; EEPROM2 to RPM_2
	movlw	EEPROM3		; address for EEPROM3
	call	EEREAD
	movwf	RPM_3		; EEPROM3 to RPM_3
	movlw	EEPROM4		; address for EEPROM4
	call	EEREAD		; read EEPROM storage
	movwf	RPM_4		; EEPROM4 to RPM_4
	movlw	EEPROM5		; address for EEPROM5
	call	EEREAD
	movwf	RPM_5		; EEPROM5 to RPM_5
	movlw	EEPROM6		; address for EEPROM6
	call	EEREAD
	movwf	RPM_6		; EEPROM6 to RPM_6
	movlw	EEPROM7		; address for EEPROM7
	call	EEREAD		; read EEPROM storage
	movwf	RPM_7		; EEPROM7 to RPM_7

; get calval from EEPROM8
	
	movlw	EEPROM8		; calibrate indicator cyl/4-stroke
	call 	EEREAD
	movwf	CALVAL		; calibration display value 

; get EEPROM9 and EEPROMA calibration values and place in TIME_CNT3 and TIME_CNT4

	movlw	EEPROM9		; address for EEPROM9
	call	EEREAD
	movwf	TIME_CNT3	; store in LS byte calibration register
	movlw	EEPROMA		; address for EEPROMA
	call	EEREAD
	movwf	TIME_CNT4	; store in MS byte calibration register

; get hysteresis setting for graph display 
	
	movlw	EEPROMC		; address for EEPROMC
	call	EEREAD
	movwf	HYSTER		; hysteresis register
	
; allow interrupts
	
	bsf	INTCON,INTE	; set interrupt enable for RB0
	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	bsf	INTCON,GIE	; set global interrupt enable for above
	
; place TACHO_0 reading into display

TO_DISP	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	movf	MODE_1,f	; mode (tacho, cal, rpm_1, rpm_2, rpm_3, etc)
	btfss	STATUS,z	; if zero bit set then tachometer mode
	goto	SW_CHK
	btfss	FLAG_1,0	; time count period update flag if set then continue
	goto	SW_CHK		; not updated yet so out
	bcf	FLAG_1,0	; clear update flag

; to slow display update if cal val > 4	

	movf	CALVAL,w	; calibration value
	sublw	0x04		; if greater than 4
	btfsc	STATUS,c	; update if c=0
	goto	UP_DTE		; cal smaller than 5 so update every time 
	btfss	FLAG_1,4	; is bit 4 set
	goto	SW_CHK		; only update display if bit 4 is set via counters 5 & 6	
UP_DTE	bcf	FLAG_1,4	; clear flag
	movf	TACHO_0,w	; tachometer value
	call	INST		; install 

; check if a switch is pressed
	
SW_CHK	bcf	FLAG_1,0	; clear multiplex update flag
MULTIP	btfss	FLAG_1,0	; check if multiplex update occured
	goto	MULTIP		; wait for new multiplex update for display
	btfsc	PORTA,4		; if a switch is pressed, then bit 4 will be low
	goto	TO_DISP		; no switch pressed so look again
	bcf	INTCON,T0IE	; clear interrupt to prevent incorrect switch result
	movlw	B'11111110'	; all seven segments off
	movwf	PORTB		; display off
	movf	PORTA,w		; look at RA0-RA2
	btfsc	PORTA,4		; if a switch is pressed, then bit 4 will be low
	goto	TO_DISP		; no switch pressed return
	btfsc	PORTA,0		; is it the mode switch	
	goto	D_U_CHK
	goto 	MODE
D_U_CHK	btfsc	PORTA,1		; is it the Down switch
	goto	U_CHK
	goto	DOWN
U_CHK	btfsc	PORTA,2		; is it the UP switch
	goto	TO_DISP		; no switch reading
	goto 	UP
MODE	bcf	INTCON,INTE	; clear interrupt enable for RB0
	incf	MODE_1,f	; cycle through modes
	movf	MODE_1,w
	sublw	0x0A		; if negative then clear mode_1
	btfsc	STATUS,c	; c=0 if negative (mode_1 > 10)
	goto	MOD_TST
CLR_MD	bsf	INTCON,INTE	; reallow interrupt enable for RB0
	clrf	MODE_1		; mode_1 clear so back to rpm mode
	clrf	DISP1		; display cleared
	clrf	DISP2
	movlw	0xFF
	movwf	DISP3		; rpm leds off
	goto	DELTME		; wait for closed switch, rpm mode
MOD_TST	movlw	0x01		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	CAL_MD		; calibrate mode
	movlw	0x02		; rpm1 mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM1_MD		; rpm1 mode
	movlw	0x03		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM2_MD		; rpm2 mode
	movlw	0x04		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM3_MD		; rpm3 mode
	movlw	0x05		; 
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM4_MD		; rpm4 mode
	movlw	0x06		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM5_MD		; rpm5 mode
	movlw	0x07		; 
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM6_MD		; rpm6 mode
	movlw	0x08		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM7_MD		; rpm7 mode
	movlw	0x09		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM0_MD		; rpm0 mode
	movlw	0x0A		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	HYST_MD		; hysteresis mode
	goto	CLR_MD

; rpmx mode, get EEPROMx value, call BIN to BCD routine place in display. Light LEDx at RBx.
; subroutine to place rpmx value to display, rpm leds cleared)

RENEW	call	EEREAD		; read EEPROM result in w
	movwf	BIN_0		; w to binary0 for BCD conversion
REN_O	call	INST		; install BCD values in display registers
	movlw	0xFF
	movwf	DISP3		; clear all RPM LEDs
	bsf	PORTA,3		; clear limit LED
	return
	
RPM0_MD	movlw	EEPROM0		; eeprom0 address
	call	RENEW
	bcf	PORTA,3		; light limit LED
	goto	DELTME		; wait for switch to open
RPM1_MD	movlw	EEPROM1		; eeprom1 address
	call	RENEW
	bcf	DISP3,2		; light LED1
	goto	DELTME		; wait for switch to open
RPM2_MD	movlw	EEPROM2		; eeprom2 address
	call	RENEW
	bcf	DISP3,4		; light LED2
	goto	DELTME		; wait for switch to open
RPM3_MD	movlw	EEPROM3		; eeprom3 address
	call	RENEW
	bcf	DISP3,5		; light LED3
	goto	DELTME		; wait for switch to open
RPM4_MD	movlw	EEPROM4 	; eeprom4 address
	call	RENEW
	bcf	DISP3,6 	; light LED4
	goto	DELTME		; wait for switch to open
RPM5_MD	movlw	EEPROM5		; eeprom5 address
	call	RENEW
	bcf	DISP3,7		; light LED5
	goto	DELTME		; wait for switch to open
RPM6_MD	movlw	EEPROM6		; eeprom6 address
	call	RENEW
	bcf	DISP3,3		; light LED6
	goto	DELTME		; wait for switch to open
RPM7_MD	movlw	EEPROM7		; eeprom7 address
	call	RENEW
	bcf	DISP3,1		; light LED7
	goto	DELTME		; wait for switch to open
CAL_MD	movlw	EEPROM8		; calibrate indicator pulses per rpm
	call 	EEREAD
	movwf	BIN_0		; for binary to BCD convert
	movwf	CALVAL		; calibration display value used in count up/down
	call	REN_O		; renew other mode
	goto	DELTME
HYST_MD movlw	EEPROMC		; hysteresis eeprom
	call 	EEREAD
	movwf	BIN_0		; for binary to BCD convert
	movwf	HYSTER		; hysteresis register
	call	REN_O
	movlw	0x0B		; value to show a "H" after 7-segment decode
	movwf	DISP2
	bsf	PORTA,3		; clear limit LED
	goto	DELTME 
UP	movlw	0x01		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	CAL_UP		; calibrate mode
	movlw	0x02		; rpm1 mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM1_UP		; rpm1 mode
	movlw	0x03		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM2_UP		; rpm2 mode
	movlw	0x04		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM3_UP
	movlw	0x05		; 
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM4_UP		; rpm4 mode
	movlw	0x06		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM5_UP		; rpm5 mode
	movlw	0x07		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM6_UP
	movlw	0x08		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM7_UP
	movlw	0x09		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM0_UP
	movlw	0x0A		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	HYST_UP
	goto	DELTME		; rpm mode so do not change
CAL_UP	movf	CALVAL,w	; calibration display value to w register
	xorlw  	D'12'		; compare with decimal'12'
	btfsc	STATUS,z	; if z=1 then calval=12
	goto	SET_LO		; do not increase past 12
	incf	CALVAL,w	; increment calval
	goto	STO_CAL
SET_LO	movlw	0x01		; low cal value
		
; store in EEPROM8 and W, call INST routine, change the calibration EEPROM9 & EEPROMA
; place in TIME_CNT3 and TIME_CNT4 
	
STO_CAL	clrf	TIME_CNT1	; clear time period counter
	clrf	TIME_CNT2
	clrf	PULSE_CNT
	movwf	CALVAL		; new calvalue
	movlw	EEPROM8		; EEPROM8 address
	movwf	EEADR		; address to write to
	movf	CALVAL,w	; new value of calibration
	call 	EWRITE		; eeprom write sequence	
	movlw	EEPROMA		; cal EEPROMA MSD
	movwf	EEADR
	rlf	CALVAL,w	; get cal value x2
	andlw	B'00011110'	; erase clear bit c
	call	CALBRT		; returns with calibration number for MSD (EEPROMA)
	movwf	TIME_CNT4	; working cal number MSD	
	call	EWRITE		; write new cal value to EEPROMA
	movlw	EEPROM9		; cal EEPROM9 LSD
	movwf	EEADR
	rlf	CALVAL,w	; get cal value x2
	andlw	B'00011110'	; erase c bit
	movwf	TEMP
	movlw	0x01
	subwf	TEMP,w		; (cal value x 2) -1
	call	CALBRT		; returns with calibration number for LSD (EEPROM9)
	movwf	TIME_CNT3	; working cal number MSD	
	call	EWRITE		; write new cal val to EEPROM9
	movf	CALVAL,w	; get cal value again
	call	INST		; install into display
	goto	DELTME
HYST_UP	incf	HYSTER,f	; increment hysteresis by 1
	movf	HYSTER,w	; hyster to w
	xorlw  	D'10'		; compare with decimal'10'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	HYSTER		; do not increase past 9
	movf	HYSTER,w
	call 	INST		; rewrite display	
	movlw	EEPROMC		; hysteresis storage address
	movwf	EEADR
	movf	HYSTER,w	; hysteresis to w
	call	EWRITE
	goto	DELTME
RPM0_UP	incf	RPM_0,f		; increment rpm0 limit by 1
	movf	RPM_0,w		; rpm0 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_0		; do not increase past 99
STO_0	movf	RPM_0,w
	call 	INST		; rewrite display	
	movlw	EEPROM0		; rpm1 storage address
	movwf	EEADR
	movf	RPM_0,w		; new rpm0 limit to w
	call	EWRITE
	call	AUTO_ST
	goto	DELTME
RPM1_UP	incf	RPM_1,f		; increment rpm1 limit by 1
	movf	RPM_1,w		; rpm1 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_1		; do not increase past 99
STO_1	movf	RPM_1,w
	call 	INST		; rewrite display	
	movlw	EEPROM1		; rpm1 storage address
	movwf	EEADR
	movf	RPM_1,w		; new rpm1 limit to w
	call	EWRITE
	goto	DELTME
RPM2_UP	incf	RPM_2,f		; increment rpm2 limit by 1
	movf	RPM_2,w		; rpm2 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_2		; do not increase past 99
STO_2	movf	RPM_2,w		; rpm2 to w
	call 	INST		; rewrite display	
	movlw	EEPROM2		; rpm2 storage address
	movwf	EEADR
	movf	RPM_2,w		; new rpm2 limit to w
	call	EWRITE
	goto	DELTME
RPM3_UP	incf	RPM_3,f		; increment rpm3 limit by 1
	movf	RPM_3,w		; rpm3 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_3		; do not increase past 99
STO_3	movf	RPM_3,w		; rpm3 to w
	call 	INST		; rewrite display	
	movlw	EEPROM3		; rpm3 storage address
	movwf	EEADR
	movf	RPM_3,w		; new rpm3 limit to w
	call	EWRITE
	goto	DELTME
RPM4_UP	incf	RPM_4,f		; increment rpm4 limit by 1
	movf	RPM_4,w		; rpm4 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_4		; do not increase past 99
STO_4	movf	RPM_4,w		; rpm4 to w
	call 	INST		; rewrite display	
	movlw	EEPROM4		; rpm4 storage address
	movwf	EEADR
	movf	RPM_4,w		; new rpm4 limit to w
	call	EWRITE
	goto	DELTME
RPM5_UP	incf	RPM_5,f		; increment rpm5 limit by 1
	movf	RPM_5,w		; rpm5 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_5		; do not increase past 99
STO_5	movf	RPM_5,w		; rpm5 to w
	call 	INST		; rewrite display	
	movlw	EEPROM5		; rpm5 storage address
	movwf	EEADR
	movf	RPM_5,w		; new rpm5 limit to w
	call	EWRITE
	goto	DELTME
RPM6_UP	incf	RPM_6,f		; increment rpm6 limit by 1
	movf	RPM_6,w		; rpm6to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_6		; do not increase past 99
STO_6	movf	RPM_6,w		; rpm6to w
	call 	INST		; rewrite display	
	movlw	EEPROM6		; rpm6 storage address
	movwf	EEADR
	movf	RPM_6,w		; new rpm6 limit to w
	call	EWRITE
	goto	DELTME
RPM7_UP	incf	RPM_7,f		; increment rpm7 limit by 1
	movf	RPM_7,w		; rpm7 to w
	xorlw  	D'100'		; compare with decimal'100'
	btfsc	STATUS,z	; if z=1 then overrange
	clrf	RPM_7		; do not increase past 99
STO_7	movf	RPM_7,w		; rpm7 to w
	call 	INST		; rewrite display	
	movlw	EEPROM7		; rpm7 storage address
	movwf	EEADR
	movf	RPM_7,w		; new rpm7 limit to w
	call	EWRITE
	goto	DELTME
DOWN	movlw	0x09		; rpm0 mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM0_DN		; rpm0 mode
	movlw	0x02		; rpm1 mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM1_DN		; rpm1 mode
	movlw	0x03		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM2_DN		; rpm2 mode
	movlw	0x04		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM3_DN
	movlw	0x05		; 
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM4_DN		; rpm4 mode
	movlw	0x06		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM5_DN		; rpm5 mode
	movlw	0x07		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM6_DN
	movlw	0x08		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	RPM7_DN
	goto	DELTME		; no down on cal, tacho or hysteresis modes
RPM0_DN	movf	RPM_0,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_0
	decf	RPM_0,f		; reduce by 1
	goto	STO_0
RPM1_DN	movf	RPM_1,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_1
	decf	RPM_1,f		; reduce by 1
	goto	STO_1
RPM2_DN	movf	RPM_2,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_2
	decf	RPM_2,f		; reduce by 1
	goto	STO_2
RPM3_DN	movf	RPM_3,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_3
	decf	RPM_3,f		; reduce by 1
	goto	STO_3
RPM4_DN	movf	RPM_4,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_4
	decf	RPM_4,f		; reduce by 1
	goto	STO_4
RPM5_DN	movf	RPM_5,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_5
	decf	RPM_5,f		; reduce by 1
	goto	STO_5
RPM6_DN	movf	RPM_6,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_6
	decf	RPM_6,f		; reduce by 1
	goto	STO_6
RPM7_DN	movf	RPM_7,w
	btfsc	STATUS,z	; stop below 00
	movlw	D'100'		; max of 99
	movwf	RPM_7
	decf	RPM_7,f		; reduce by 1
	goto	STO_7

; delay time for switch debounce period

DELTME	bsf	INTCON,T0IE	; allow interrupt
	movlw	D'50'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_1	movwf	VALUE_2		; VALUE_2 = w
LP_2	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1	
		
; wait for switch to open (must test for closed switch when RA0, RA1 and RA2 are low
; because switch will appear open at RA4 when switch pull down via RA0-RA2 is high) 	

CLOSED	bcf	FLAG_1,1	; clear flag that indicates switch closed
	call	ON		; check for closed switch
	call	ON		; do at least four times to return to original switch closure
	call	ON		;
	call	ON 		; 
	call	ON 		; do again just in case missed one interrupt during call
	btfsc	FLAG_1,1	; was switch closed 	
	goto 	CLOSED		; yes go to check again
	goto 	TO_DISP		; switch open so out

;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

; SUBROUTINES

; Automatic setting of lower 6 LEDs (divide RPM7 by 8 then sucessively 
; take this value away from RPM0 and place in RPM7, RPM6 etc. Also place in EEPROM0, 7 etc.)

AUTO_ST	movf	RPM_0,w		; get RPM0 value
	movwf	AUT_SET		; store
	rrf	AUT_SET,f	; divide by 2
	rrf	AUT_SET,f	; divide by 2
	rrf	AUT_SET,w	; divide by 2 for total of divide by 8
	andlw	0x0F		; extract LSB
	movwf	AUT_SET		; store difference value
	subwf	RPM_0,w
	movwf	RPM_7
	movlw	EEPROM7		; rpm7 storage address
	movwf	EEADR
	movf	RPM_7,w		; new rpm7 limit to w
	call	EWRITE
	movf	AUT_SET,w	; difference value
	subwf	RPM_7,w
	movwf	RPM_6
	movlw	EEPROM6		; rpm6 storage address
	movwf	EEADR
	movf	RPM_6,w		; new rpm6 limit to w
	call	EWRITE
	movf	AUT_SET,w	; difference value
	subwf	RPM_6,w
	movwf	RPM_5
	movlw	EEPROM5		; rpm5 storage address
	movwf	EEADR
	movf	RPM_5,w		; new rpm5 limit to w
	call	EWRITE
	movf	AUT_SET,w	; difference value
	subwf	RPM_5,w
	movwf	RPM_4
	movlw	EEPROM4		; rpm4 storage address
	movwf	EEADR
	movf	RPM_4,w		; new rpm4 limit to w
	call	EWRITE
	movf	AUT_SET,w	; difference value
	subwf	RPM_4,w
	movwf	RPM_3
	movlw	EEPROM3		; rpm3 storage address
	movwf	EEADR
	movf	RPM_3,w		; new rpm3 limit to w
	call	EWRITE
	movf	AUT_SET,w	; difference value
	subwf	RPM_3,w
	movwf	RPM_2
	movlw	EEPROM2		; rpm2 storage address
	movwf	EEADR
	movf	RPM_2,w		; new rpm2 limit to w
	call	EWRITE
	movf	AUT_SET,w	; difference value
	subwf	RPM_2,w
	movwf	RPM_1
	movlw	EEPROM1		; rpm1 storage address
	movwf	EEADR
	movf	RPM_1,w		; new rpm1 limit to w
	call	EWRITE
	return

; delay period for multiplex rate

DELMO	movlw 	D'255'		; delay period
	movwf	TEMP_2
SMALR	decfsz	TEMP_2,f	; reduce temp_2
	goto	SMALR		; temp_2 smaller by 1
	return			; end delay

; subroutine to place binary value into display registers after BCD conversion
	
INST	movwf	BIN_0		; place in binary 0 register
	sublw	D'99'		; if over 9900rpm then load dashes
	btfss	STATUS,c 
	goto	DASH		; if over 99 then display is dashes
	call	BCD		; convert binary number to BCD
	movf	BCD_1,w		; contains packed BCD value
	andlw	0x0F		; extract LS bits
	movwf	DISP1		; place in display 1
	movlw	0x0A		; check which mode
	xorwf	MODE_1,w	; cal mode
	btfsc	STATUS,z	; zero if the same
	goto	HYST_H	
	swapf	BCD_1,w		; swap upper and lower 4-bits
	andlw	0x0F		; extract bits
	movwf	DISP2		; place in middle digit register
	return 
DASH	movlw	0x0A		; value which gives a dash after seven segment decode
	movwf	DISP1
	movwf	DISP2
	return
HYST_H	movlw	0x0B		; place "H" in display (0B=H after 7-segment decode)
	movwf	DISP2
	return

; subroutine to wait for switch to reopen
	
ON	bcf	INTCON,T0IE	; prevent multiplex update
	bcf	FLAG_1,0	; clear flag that indicates update interrupt for display
RE_CHK	btfsc	PORTA,4		; check if switch closed
	goto	OPEN
	bsf	FLAG_1,1	; switch closed so set flag
	bsf	INTCON,T0IE	; allow multiplex interrupt
	goto	RE_CHK 		; closed so check till open 
OPEN	bcf	FLAG_1,0	; clear multiplex update flag wait for next multiplex update
	bsf 	INTCON,T0IE	; allow multiplex update
FLG_TST	btfss	FLAG_1,0	; is flag set indicating a new interrupt for multiplexing
	goto 	FLG_TST		; wait 
	return

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_RD	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_RD		; wait for low RD (read RD)	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	bcf	FLAG_1,6	; EEPROM write repeat flag
	movwf	TEMP_2		; store w in temporary storage
EWRIT	movf	TEMP_2,w	; place value in w (used for read data sequence) 
	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	
; read EEPROM DATA and check if written correctly
	
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_AGN	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_AGN		; wait for low RD	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	subwf	EEDATA,w	; compare read value with value stored
	btfsc	STATUS,z	; skip if not the same
	return			; value correctly written 
	btfsc	FLAG_1,6	; write repeat bit, skip if clear and rewrite
	return
	bsf	FLAG_1,6	; set repeat flag rewrite once only 
	goto 	EWRIT		; rewrite as not written correctly

; Subroutine to convert from 8-bit binary to 2-digit BCD (packed)
; Binary value is in BIN0  
; Result in BCD is in BCD0 & BCD1.  
; BCD0 is MSB, BCD1 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'8'
	movwf	CNT_8		; 8 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
LOOPBCD	rlf	BIN_0,f		; shift left binary registers
	rlf	BCD_1,f		; MSB shift left
	rlf	BCD_0,f		; LSB shift left BCD registers
	decfsz	CNT_8,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_1		; BCD LSB address
	movwf	FSR		; pointer for BCD1
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-1)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w in temp
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

	end
