; Rain Gauge program RAIN.ASM
; Tallys rain sensor signal at RB0 input and stores data into separate days	
; current time and todays reading stored every 10 minutes into EEPROM. Empty
; time stored in EEPROM at time it is set. (clock and empty time are 24-hour 
; clocks with 10-minute resolution) 
; Battery backup included but if this is not used or data lost due to battery failure will 
; restore at blackout return
; Stores up to 61 days of rainfall (current day (today) and 60 days previously)
; These readings can be recalled.
; The previous days readings include a valid data indicator for initial first 60 days logging
; A Reset clears valid record counter and todays rainfall data 
; A Hold allows stoping of any rainfall counting and is useful when cleaning the rain sensor
; When setting the clock it resets 10-minute counter back to zero 

;
; Processor pin allocations are as follows:

; RA0 Output driving LED bar graph
; RA1 Output disp 3
; RA2 Output disp 1
; RA3 Output disp 2
; RA4 Input from switches

; RB0 Input from rain gauge (interrupt)
; RB1 e segment drive set empty time
; RB2 d segment drive, hold/reset
; RB3 c segment drive, not used
; RB4 b segment drive, today
; RB5 a segment drive, previous days
; RB6 f segment drive, invalid data
; RB7 g segment drive, set time

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

; EEPROM DATA


EEPROM1		equ	H'00'	; non-volatile storage for today updated every 10 minutes  
EEPROM2		equ	H'01'	; non-volatile storage for today -1
EEPROM3		equ	H'02'	; non-volatile storage for today -2
EEPROM4		equ	H'03'	; non-volatile storage for today -3
EEPROM5		equ	H'04'	; non-volatile storage for today -4	
EEPROM6		equ	H'05'	; non-volatile storage for today -5
EEPROM7		equ	H'06'	; non-volatile storage for today -6 		
EEPROM8		equ	H'07'	; non-volatile storage for today -7 
EEPROM9		equ	H'08'	; non-volatile storage for today -8 
EEPROM10	equ	H'09'	; non-volatile storage for today -9
EEPROM11	equ	H'0A'	; non-volatile storage for today -10
EEPROM12	equ	H'0B'	; non-volatile storage for today -11
EEPROM13	equ	H'0C'	; non-volatile storage for today -12
EEPROM14	equ	H'0D'	; non-volatile storage for today -13
EEPROM15	equ	H'0E'	; non-volatile storage for today -14
EEPROM16	equ	H'0F'	; non-volatile storage for today -15
EEPROM17	equ	H'10'	; non-volatile storage for today -16
EEPROM18	equ	H'11'	; non-volatile storage for today -17
EEPROM19	equ	H'12'	; non-volatile storage for today -18  
EEPROM20	equ	H'13'	; non-volatile storage for today -19  
EEPROM21	equ	H'14'	; non-volatile storage for today -20
EEPROM22	equ	H'15'	; non-volatile storage for today -21 
EEPROM23	equ	H'16'	; non-volatile storage for today -22
EEPROM24	equ	H'17'	; non-volatile storage for today -23
EEPROM25	equ	H'18'	; non-volatile storage for today -24
EEPROM26	equ	H'19'	; non-volatile storage for today -25 
EEPROM27	equ	H'1A'	; non-volatile storage for today -26
EEPROM28	equ	H'1B'	; non-volatile storage for today -27
EEPROM29	equ	H'1C'	; non-volatile storage for today -28
EEPROM30	equ	H'1D'	; non-volatile storage for today -29
EEPROM31	equ	H'1E'	; non-volatile storage for today -30
EEPROM32	equ	H'1F'	; non-volatile storage for today -31 
EEPROM33	equ	H'20'	; non-volatile storage for today -32
EEPROM34	equ	H'21'	; non-volatile storage for today -33
EEPROM35	equ	H'22'	; non-volatile storage for today -34 
EEPROM36	equ	H'23'	; non-volatile storage for today -35
EEPROM37	equ	H'24'	; non-volatile storage for today -36
EEPROM38	equ	H'25'	; non-volatile storage for today -37
EEPROM39	equ	H'26'	; non-volatile storage for today -38 
EEPROM40	equ	H'27'	; non-volatile storage for today -39
EEPROM41	equ	H'28'	; non-volatile storage for today -40
EEPROM42	equ	H'29'	; non-volatile storage for today -41  
EEPROM43	equ	H'2A'	; non-volatile storage for today -42 
EEPROM44	equ	H'2B'	; non-volatile storage for today -43
EEPROM45	equ	H'2C'	; non-volatile storage for today -44
EEPROM46	equ	H'2D'	; non-volatile storage for today -45
EEPROM47	equ	H'2E'	; non-volatile storage for today -46
EEPROM48	equ	H'2F'	; non-volatile storage for today -47
EEPROM49	equ	H'30'	; non-volatile storage for today -48 
EEPROM50	equ	H'31'	; non-volatile storage for today -49
EEPROM51	equ	H'32'	; non-volatile storage for today -50
EEPROM52	equ	H'33'	; non-volatile storage for today -51
EEPROM53	equ	H'34'	; non-volatile storage for today -52
EEPROM54	equ	H'35'	; non-volatile storage for today -53
EEPROM55	equ	H'36'	; non-volatile storage for today -54
EEPROM56	equ	H'37'	; non-volatile storage for today -55
EEPROM57	equ	H'38'	; non-volatile storage for today -56
EEPROM58	equ	H'39'	; non-volatile storage for today -57 
EEPROM59	equ	H'3A'	; non-volatile storage for today -58 
EEPROM60	equ	H'3B'	; non-volatile storage for today -59 
EEPROM61	equ	H'3C'	; non-volatile storage for today -60
EEPROMET	equ	H'3D'	; empty time (initial 24:00)
EEPROMDT	equ	H'3E'	; valid eeprom data counter 0-61 (initial=0)
EEPROMTM	equ	H'3F'	; time storage (initial 12:00)

; RAM

DISP1		equ	H'0C'	; working storage for Display1 value 
DISP2		equ	H'0D'	; working storage for Display2 value 
DISP3		equ	H'0E'	; working storage for Display3 value 
DISP4		equ	H'0F'	; working storage for Display4 bargraph
STATUS_TMP 	equ 	H'10'	; temp storage for status during interrupt
W_TMP		equ	H'11'	; temporary storage for w during interrupt
VALUE_1		equ 	H'12'	; delay value storage			
VALUE_2		equ	H'13'	; delay value storage
FG_CHNG		equ	H'14'	; this flag changed to H'2B' to see if this enables it to operate on 
				; 'A' versions of Chip.
FLAG_2		equ	H'15'	; not used
CNT_8		equ	H'16'	; BCD conversion counter 
TEMP		equ	H'17'	; temporary storage during BCD conversion
TEMP_1		equ	H'18'	; temporary working storage
TEMP_2		equ	H'19'	; temp storage
TIME_CNT1	equ	H'1A'	; clock 
TIME_CNT2	equ	H'1B'	; empty time
TIME_CNT3	equ	H'1C'	; 10 minute counter LSD
TIME_CNT4	equ	H'1D'	; 10 minute counter MSD
MODE_1		equ	H'20'	; mode indicator (rainfall display, set and reset)
MODE_2		equ	H'21'	; mode indicator (today, yesterday,previous days)
RAIN_1		equ	H'22'   ; todays rain reading
VALID		equ	H'23'	; temp storage of valid rain reading
PULSE_CNT	equ	H'24'	; rain gauge pulse counter 
BCD_0		equ	H'25'	; bcd MS value after conversion
BCD_1		equ	H'26'	; bcd LS value
BIN_0		equ	H'27'	; binary value for BCD conversion
DAY		equ	H'28'	; day counter used when displaying previous day readings
PRESC		equ	H'29'	; prescaler divide by 16 for multiplexing rate to clock counter
ALTERNAT	equ	H'2A'	; alternate display counter
FLAG_1		equ	H'2B'	; bit0 new day, bit1 10-minute upadate, bit2-multiplex indicator
				; bit3-closed switch indicator, bit4 is new day recorded
				; bit5 is pulse count update, bit6 is for EEPROM write sequence

; preprogram EEPROM DATA (00-3F from 2100-213F)
	
	ORG     2100		; start at 00
	
	DE	D'00', D'01', D'02', D'03', D'04', D'05', D'06', D'07', D'08'	
	DE	D'09', D'10', D'11', D'12', D'13', D'14', D'15', D'16', D'17'
	DE	D'18', D'19', D'20', D'21', D'22', D'23', D'24', D'25', D'26'	
	DE	D'27', D'28', D'29', D'30', D'31', D'32', D'33', D'34', D'35'
	DE	D'36', D'37', D'38', D'39', D'40', D'41', D'42', D'43', D'44'	
	DE	D'45', D'46', D'47', D'48', D'49', D'50', D'51', D'52', D'53'
	DE	D'54', D'55', D'56', D'57', D'58', D'59', D'60' 	
	DE	D'240', D'01', D'120' ; 240 for empty time,01 for valid data,120 for 12:00 am time

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data.  
; RB7-RB1----gfabcde
 
SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'11100110'	; 7-segment code 1
	retlw 	B'01001000'	; 7-segment code 2
	retlw 	B'01000011'	; 7-segment code 3
	retlw 	B'00100110'	; 7-segment code 4
	retlw 	B'00010010'	; 7-segment code 5
	retlw 	B'00010000'	; 7-segment code 6
	retlw 	B'11000110'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00000010'	; 7-segment code 9
	retlw	B'01111110'	; 7-segment code "-"
	retlw	B'10001100'	; 7-segment code "R"
	retlw 	B'00011000'	; 7-segment code "E"
	retlw 	B'00010010'	; 7-segment code "S"
;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3, Disp4 in sequence plus 
; blanking leading zeros on display for Disp2 and Disp3 
; uses internal timer to initiate display update
; also updates time and
; rain gauge signal update

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf	STATUS,RP0	; select memory bank 0

; which interrupt (INT or TMRO)

	btfsc	INTCON,T0IF	; TMRO overflow interrupt flag 
	goto	LIGHT		; TMRO overflow so multiplex display
RAIN_T  btfsc	INTCON,INTF	; INT interrupt flag set then goto rain
	goto	RAIN		; rain gauge signal update 
	goto	RECLAIM		; end of interrupt reclaim w and status

; display update 

LIGHT	bsf	FLAG_1,2	; multiplex flag	
	bcf	INTCON,T0IF	; clear interrupt flag
				; freq is 3.2768MHz/4/256/(2) = 1600Hz
	btfss	DISP4,5		; test for previous day reading LED 
	goto	ALT_DIS		; alternate display function	
MULT_N	btfss	PORTA,2		; skip if display 1 not lit
	goto	LT1		; display 1 lit
	btfss	PORTA,3		; skip if display 2 not lit
	goto	LT2		; display 2 lit
	btfss	PORTA,1		; skip if display 3 not lit				
	goto	LT3		; display 3 lit
	btfss	PORTA,0		; skip if display 4 not lit
	goto	LT4		; display 4 lit
	movlw	B'11111110'	; seven segments off
	movwf	PORTB
	bcf	PORTA,0		; no displays lit so set disp 4 (used if error in output)
	goto 	LIGHT		; goto check which is lit now
LT1	bsf	PORTA,2		; disp1 off
	movf	DISP2,w		; look at display2
	btfss	STATUS,z	; skip if 0
	goto	NOBLNK1		; do not blank disp2
	movf	DISP3,f		; display 3
	btfss	STATUS,z	; skip if 0
	goto	NOBLNK1		; do not blank disp2
	movlw	B'11111110'	; 7-segment display off
	goto	BLANK1		; blank disp2 when disp2 and disp3 = 0
NOBLNK1	call 	SVNSEG		; goto subroutine to get seven segment code 
BLANK1	movwf	PORTB		; seven segment value to portB
	bcf	PORTA,3		; disp2 powered
	goto	TIMECNT		; end of multiplex
LT2	bsf	PORTA,3		; disp2 off
	movf	DISP3,w		; disp3 to w
	btfss	STATUS,z	; check if 0
	goto	NOBLNK2
	movlw	B'11111110'	; display off
	goto	BLANK2
NOBLNK2	call	SVNSEG		; seven segment code
BLANK2	movwf	PORTB		; portB lights display
	bcf	PORTA,1		; disp3 powered
	goto	TIMECNT		; end of multiplex
LT3	bsf	PORTA,1		; disp3 off
	movf	DISP4,w		; disp4 to w
	movwf 	PORTB
	bcf	PORTA,0		; disp4 powered
	goto	TIMECNT		; end of multiplex
LT4	bsf	PORTA,0		; display 4 off
	movf	DISP1,w		; display 1 to w
	call	SVNSEG		; seven segment code
	movwf	PORTB
	bcf	PORTA,2		; display 1 powered
	goto	TIMECNT

	
ALT_DIS	btfsc	ALTERNAT,6	; alternate counter 
	goto	MULT_N		; multiplex normally
	movf	DAY,w		; day value
	movwf	BIN_0		; into binary register
	call	BCD		; convert to BCD
	btfss	PORTA,2		; skip if display 1 not lit
	goto	LT1_ALT		; display 1 lit
	btfss	PORTA,3		; skip if display 2 not lit
	goto	LT2_ALT		; display 2 lit
	btfss	PORTA,1		; skip if display 3 not lit				
	goto	LT3_ALT		; display 3 lit
	btfss	PORTA,0		; skip if display 4 not lit
	goto	LT4_ALT		; display 4 lit

LT1_ALT	bsf	PORTA,2		; disp1 off
	swapf	BCD_1,w
	andlw	0x0F		; blank msd
	btfss	STATUS,z	; skip if 0
	goto	NOBL1		; do not blank disp2
	movlw	B'11111110'	; 7-segment display off
	goto	BLNK1		; blank disp2 when disp2 =0
NOBL1	call 	SVNSEG		; goto subroutine to get seven segment code 
BLNK1	movwf	PORTB		; seven segment value to portB
	bcf	PORTA,3		; disp2 powered
	goto	TIMECNT		; end of multiplex
LT2_ALT	bsf	PORTA,3		; disp2 off
	movlw	B'01111110'	; 7-segment code "-"
	movwf	PORTB
	bcf	PORTA,1		; disp3 powered
	goto	TIMECNT		; end of multiplex
LT3_ALT	bsf	PORTA,1		; disp3 off
	movf	DISP4,w		; disp4 to w
	movwf 	PORTB
	bcf	PORTA,0		; disp4 powered
	goto	TIMECNT		; end of multiplex
LT4_ALT	bsf	PORTA,0		; display 4 off
	movf	BCD_1,w		; bcd value to w
	andlw	0x0F		; remove ms digits
	call	SVNSEG		; seven segment code
	movwf	PORTB
	bcf	PORTA,2		; display 1 powered
	
; update time period counters for clock

TIMECNT	incf	PRESC,f		; increment prescaler
	btfss	PRESC,4		; if bit 4 high then / by 16
	goto	RAIN_T		; not counted yet
	incf	ALTERNAT,f	; alternate display counter
	clrf	PRESC		; prescaler is 0 ready to count again up to 16
	movf	TIME_CNT4,w	; MS byte of 10 minute counter
	xorlw	0xEA		; cf 10 minutes (100Hz/10 minutes = 60000 so MS bit is H'EA')
	btfss	STATUS,z	; skip if equal
	goto	NOT10		; not 10 minutes 
	movf	TIME_CNT3,w	; LS Byte of 10 minute counter
	xorlw	0x60	  	; compare LS byte of 10 minute counter with 60
	btfss	STATUS,z	; skip if equal
	goto	NOT10		; increment time counter
	clrf	TIME_CNT4
	clrf	TIME_CNT3
	bsf	FLAG_1,1	; indicator to renew data in EEPROM every 10 minutes
	bcf	FLAG_1,4	; new day done flag
	goto	RAIN_T
	
NOT10   incfsz	TIME_CNT3,f	; increase counter 3 skip when zero
	goto 	CF_MTY		; compare empty time with clock
	incf	TIME_CNT4,f	; increase counter 4 when counter 3 overflows
	
; check for empty time
	
CF_MTY	bcf	FLAG_1,0	; clear empty flag
	movf	TIME_CNT1,w	; clock
	xorwf	TIME_CNT2,w	; is empty time = clock
	btfss	STATUS,z	; if zero then reset pulse_cnt
	goto	RAIN_T
	bsf	FLAG_1,0	; indicates emptying time ready
	goto	RAIN_T

; rain gauge signal update

RAIN	bcf	INTCON,INTF	; clear INT flagspeed sensor input
	btfsc	DISP4,2		; if reset/hold LED on then do not increment pulse counter
	incfsz	PULSE_CNT,f	; increase pulse counter value for each rain gauge signal
	goto	REFR
	movlw	0xFF		; max value
	movwf	PULSE_CNT	; place max count in pulse counter do not allow overrange	
REFR	btfsc	DISP4,4		; check the Today LED
	goto	RECLAIM
	bsf	FLAG_1,5	; todays reading update flag

; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  	
; RESET		
; Set ports A & B

MAIN	clrf	FLAG_1		; clear flags
	bsf	FLAG_1,4	; set empty update flag preventing update on reset
	clrf	PRESC		; prescaler counter
	clrf	TIME_CNT3	; clear 10 minute time period counter LS byte
	clrf	TIME_CNT4	; clear 10 minute time period counter MS byte
	clrf	PULSE_CNT	; clear rain gauge pulse counter
	clrf	DISP1		; clear display registers
	clrf	DISP2
	clrf	DISP3
	clrf	DAY		; day counter
	clrf	MODE_1		; set mode to rain display
	clrf	MODE_2		; set to todays rain
	movlw	0xFF		; 
	movwf	DISP4		; clear LEDs
	bcf	DISP4,4		; set today LED indicator
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'11000000'	; w = 10000000 binary 
	movwf	OPTION_REG	; TMRO prescaler 2 division, PORTB pullups disabled
	movlw   B'00010000'	; w = 10000 binary (RA0-RA3 outputs, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00001111'
	movwf	PORTA		; portA RA0,RA1,RA2,RA3 outputs high 
	
; read data values
	movlw	EEPROMTM	; time
	call 	EEREAD
	movwf	TIME_CNT1	; clock ram
	movlw	EEPROMET	; empty time
	call	EEREAD
	movwf	TIME_CNT2	; empty time ram
	movlw	EEPROM1		; current rain value
	call 	EEREAD
	movwf	PULSE_CNT	; pulse counter
	call  	INST		; current rain mm count to display
	movlw	EEPROMDT	; valid eeprom data
	call	EEREAD		; valid count
	movwf	VALID

; allow interrupts
	bsf	INTCON,INTE	; set interrupt enable for RB0
	bsf	INTCON,GIE	; set global interrupt enable for above
	
; check for update flags	
	
TO_DO	bsf	INTCON,T0IE	; set interrupt
	btfss	FLAG_1,1	; 10 minute time flag if set then continue
	goto	EMPTY		; not updated yet so out
	incf	TIME_CNT1,f	; increment clock
	bcf	FLAG_1,1	; clear flag
	movf	TIME_CNT1,w	; look at value
	movwf	BIN_0
	call 	BCD		; convert to BCD
	movf	BCD_1,w
	andlw	B'00001111'	; remove bits 7-4
	sublw	0x05		; take from 5
	btfsc	STATUS,c	; if c=o then w is 6 or more
	goto	STORE
	movlw	D'04'		; if a 6 then add 4 to increment hours
	addwf	TIME_CNT1,f	; add 4 to clock
	movf	TIME_CNT1,w	;
	sublw	D'245'		; when 24:50 goes to 25:00
	btfsc	STATUS,c	; check if reached 1:00		
	goto	STORE
	movlw	D'10'		; 1:00 time
	movwf	TIME_CNT1	; new time
STORE	btfsc	DISP4,7		; is set time LED on
	goto	STO_EEP
	movf	TIME_CNT1,w	; time 
	call 	INST		; place in display
	
STO_EEP	movlw	EEPROMTM	; EEPROM time address
	movwf	EEADR		; address to write to
	movf	TIME_CNT1,w	; current time
	call 	EWRITE		; eeprom write sequence
	movlw	EEPROM1		; EEPROM current rain tally address
	movwf	EEADR		; address to write to
	movf	PULSE_CNT,w	; new value of rain count
	call 	EWRITE		; eeprom write sequence
			
EMPTY	btfss	FLAG_1,0	; is empty flag set (end of day for reading)
	goto	UP_RDG
	btfsc	FLAG_1,4	; updated flag
	goto	UP_RDG
	bcf	FLAG_1,0
	movlw	EEPROM1		; EEPROM current rain tally address
	movwf	EEADR		; address to write to
	movf	PULSE_CNT,w	; new value of rain count
	call 	EWRITE		; eeprom write sequence
	movf	PULSE_CNT,w	; rain gauge reading
	movwf	RAIN_1		; place in storage for the days rain reading
	clrf	PULSE_CNT	; return counter to zero new days counting
	btfsc	DISP4,4		; is today LED on
	goto 	INC_V
	movf	PULSE_CNT,w	; update pulse count to display
	call	INST
INC_V	incf	VALID,w
	sublw	D'60'		; compare with maximum count 60
	btfsc	STATUS,c	; if c is 0 then stop at 60
	goto	EEP_STO
	movlw	D'60'
	movwf	VALID		; valid has 60
EEP_STO	incf	VALID,f		; EEADR + 1
	movlw	EEPROMDT	; valid data count
	movwf	EEADR
	movf	VALID,w		; store valid count in eeprom each day
	call	EWRITE
	movlw	EEPROM61	; max EEPROM for today-60 
	movwf	EEADR
NEXTD	decfsz	EEADR,f		; address of earlier day
	goto	CYCLE
	movf	RAIN_1,w	; rain1 value store in eeprom 2 (01)
	incf	EEADR,f		; eeprom 2
	call	EWRITE
	bsf	FLAG_1,4	; days updated flag set so stops repeated update
	goto	SW_CHK
CYCLE	movf	EEADR,w	
	call 	EEREAD		; rainfall data of earlier day
	incf	EEADR,f		; address of later day
	call 	EWRITE		; new value to EEPROM
	decf	EEADR,f
	goto 	NEXTD		; next day

UP_RDG	btfss	FLAG_1,5	; check if new pulse count value
	goto	SW_CHK
	btfsc	DISP4,4		; is today LED on
	goto 	SW_CHK
	bcf	FLAG_1,5
	movf	PULSE_CNT,w	; update pulse count to display
	call	INST	
	
; check if a switch is pressed
	
SW_CHK	bcf	FLAG_1,2	; clear multiplex update flag
MULTIP	btfss	FLAG_1,2	; check if multiplex update occured
	goto	MULTIP		; wait for new multiplex update for display
	btfsc	PORTA,4		; if a switch is pressed, then bit 4 will be low
	goto	TO_DO		; no switch pressed so look again
	bcf	INTCON,T0IE	; clear interrupt to prevent incorrect switch result
	movlw	B'11111110'	; all seven segments off
	movwf	PORTB		; display off
	movf	PORTA,w		; look at RA0-RA2
	btfsc	PORTA,4		; if a switch is pressed, then bit 4 will be low
	goto	TO_DO		; no switch pressed return
	btfsc	PORTA,2		; is it the mode switch	
	goto	D_U_CHK
	goto 	MODE
D_U_CHK	btfsc	PORTA,1		; is it the Down switch
	goto	U_CHK
	goto	DOWN
U_CHK	btfsc	PORTA,3		; is it the UP switch
	goto	TO_DO		; no switch reading
	goto 	UP
MODE	incf	MODE_1,f	; cycle through modes
	movf	MODE_1,w
	sublw	0x03		; if negative then clear mode_1
	btfsc	STATUS,c	; c=0 if negative (mode_1 > 3)
	goto	MOD_TST
CLR_MD	clrf	MODE_1		; mode_1 clear so back to rainfall mode
MOD_TST	movf	MODE_1,w	; rain display mode
	btfsc	STATUS,z	; zero if rain display mode
	goto	RAIN_DS 	; rain display mode
	movlw	0x01		; set time mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	SET_TME		; set time mode
	movlw	0x02		; check which mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	SET_MTY		; set empty mode
	movlw	0x03		; check which mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	RST_HLD		; reset hold mode
	goto	CLR_MD

; set to todays rain

RAIN_DS	movf	PULSE_CNT,w	; todays rain counter
	call 	INST
	movlw	0xFF		; LEDs off
	movwf	DISP4
	bcf	DISP4,4		; todays indicator LED on
	clrf	DAY	
	goto	DELTME		; delay and switch debounce
SET_TME	movf	TIME_CNT1,w	; time ram
	call 	INST
	movlw	0xFF
	movwf	DISP4		; LEDs off
	bcf	DISP4,7		; set time indicator
	goto	DELTME
SET_MTY	movf	TIME_CNT2,w	; empty time ram
	call	INST
	movlw	0xFF
	movwf	DISP4		; LEDs off
	bcf	DISP4,1		; set empty time indicator
	goto	DELTME
RST_HLD	movlw	0xFF		; 
	movwf	DISP4		; indicator LEDs off
	bcf	DISP4,2		; reset hold LED on
	movlw	0x0A		; dash on display after seven segment decode
	movwf	DISP2		; dash on display 2
	movwf	DISP3		; & display 3
	incf	DISP3,w		; 0A + 1, w=0B
	movwf	DISP1		; an R in display after seven segment decode 	
	goto	DELTME

; if mode is 0 then reduce day value till 0 

DOWN	movf	MODE_1,w	; mode
	btfsc	STATUS,z	; skip if not zero
	goto	DAY_DN		; day down
	movlw	0x01		; set time mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	TME_DN		; set time mode
	movlw	0x02		; check which mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	MTY_DN		; set empty mode
	goto	DELTME
DAY_DN	clrf	ALTERNAT	; alternate display counter
	movf	DAY,w		; which day 
	sublw	D'59'		; check if at maximum	
	btfsc	STATUS,c	; check if 0
	incf	DAY,f		; do not increase if already 60
	movlw	EEPROM1		; todays eeprom store
	movwf	EEADR
	movf	DAY,w
	addwf	EEADR,f		; EEPROM1 address + DAY = required day address
	call	EEREAD
	call	INST		; value to display
	movlw	0xFF
	movwf	DISP4		; indicator LEDs off
	movf	DAY,w
	btfsc	STATUS,z	; skip if not 0
	goto	DLTE
	bcf	DISP4,5		; set previous day LED
	movf	DAY,w
	subwf	VALID,w
	btfss	STATUS,c	; if day is equal or less than valid skip
	bcf	DISP4,6		; light not valid LED
	btfsc	STATUS,z
	bcf	DISP4,6
	goto	DELTME
DLTE	bcf	DISP4,4		; light today LED 	
	goto	DELTME
TME_DN	bcf	FLAG_1,4	; updated flag
	clrf	TIME_CNT3	; time counter
	clrf	TIME_CNT4	; time counter
	movf	TIME_CNT1,w	;
	xorlw	D'10'		; when 1:00 goes to 24:50
	btfss	STATUS,z	; check if reached 1:00		
	goto	TM_CHK
	movlw	D'245'		; 24:50 time
	movwf	TIME_CNT1	; new time
	goto	TM_STO
TM_CHK	movf	TIME_CNT1,w	; look at value
	movwf	BIN_0
	call	BCD
	movf	BCD_1,w
	andlw	B'00001111'	; remove bits 7-4
	btfss	STATUS,z	; if z=1 then w is 0
	goto	DEC_TM
	movlw	D'04'		; 
	subwf	TIME_CNT1,f	; subtract 5 from clock
DEC_TM	decf	TIME_CNT1,f	; reduce time 
	goto	TM_STO		; store
	
MTY_DN	bcf	FLAG_1,4	; updated flag
	movf	TIME_CNT2,w	;
	xorlw	D'10'		; when 1:00 goes to 24:50
	btfss	STATUS,z	; check if reached 1:00		
	goto	MT_CHK
	movlw	D'245'		; 24:50 time
	movwf	TIME_CNT2	; new time
	goto	T_STR
MT_CHK	movf	TIME_CNT2,w	; look at value
	movwf	BIN_0
	call	BCD
	movf	BCD_1,w
	andlw	B'00001111'	; remove bits 7-4
	btfss	STATUS,z	; if z=1 then w is 0
	goto	DEC_MT
	movlw	D'04'		; 
	subwf	TIME_CNT2,f	; subtract 5 from clock
DEC_MT	decf	TIME_CNT2,f	; reduce time 
	goto	T_STR		; store
	goto	DELTME

UP	movf	MODE_1,w	; mode
	btfsc	STATUS,z	; skip if not zero
	goto	DAY_UP		; day up
	movlw	0x01		; set time mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	TME_UP		; set time mode
	movlw	0x02		; check which mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	MTY_UP		; set empty mode
	movlw	0x03
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z
	goto	RESET		; reset if up switch pressed on Hold/Reset mode 
	goto	DELTME
DAY_UP	clrf	ALTERNAT	; clear alternate counter
	movf	DAY,f		; which day 
	btfss	STATUS,z	; check if 0
	decf	DAY,f		; do not decrease if already 0
	movlw	EEPROM1		; todays eeprom store
	movwf	EEADR
	movf	DAY,w
	addwf	EEADR,f		; EEPROM1 address + DAY = required day address
	call	EEREAD
	call	INST		; value to display
	movlw	0xFF
	movwf	DISP4		; indicator LEDs off
	movf	DAY,w
	btfsc	STATUS,z	; skip if not 0
	goto	TDY
	bcf	DISP4,5		; set previous days LED
	subwf	VALID,w		; take day value from valid counter
	btfss	STATUS,c	; if day is equal or less than valid skip
	bcf	DISP4,6		; light not valid LED
 	btfsc	STATUS,z
	bcf	DISP4,6
	goto	DELTME
TDY	bcf	DISP4,4		; today LED
	movf	PULSE_CNT,w
	call	INST		; install pulse count value when displaying today 
	goto	DELTME
TME_UP	bcf	FLAG_1,4	; updated flag
	clrf	TIME_CNT3	; time counter
	clrf	TIME_CNT4	; time counter
	incf	TIME_CNT1,f	; increment clock
	movf	TIME_CNT1,w	; look at value
	movwf	BIN_0
	call	BCD
	movf	BCD_1,w
	andlw	B'00001111'	; remove bits 7-4
	sublw	0x05		; take from 5
	btfsc	STATUS,c	; if c=o then w is 6 or more
	goto	TM_STO
	movlw	D'04'		; if a 6 then add 4 to increment hours
	addwf	TIME_CNT1,f	; add 4 to clock
	movf	TIME_CNT1,w	;
	sublw	D'245'		; when 24:50 goes to 25:00
	btfsc	STATUS,c	; check if reached 1:00		
	goto	TM_STO
	movlw	D'10'		; 1:00 time
	movwf	TIME_CNT1	; new time
TM_STO	movlw	EEPROMTM	; EEPROM time address
	movwf	EEADR		; address to write to
	movf	TIME_CNT1,w	; current time
	call 	EWRITE		; eeprom write sequence
	movf	TIME_CNT1,w
	call 	INST
	goto	DELTME
MTY_UP	bcf	FLAG_1,4	; updated flag
	incf	TIME_CNT2,f	; increment empty time
	movf	TIME_CNT2,w	; look at value
	movwf	BIN_0
	call	BCD
	movf	BCD_1,w
	andlw	B'00001111'	; remove bits 7-4
	sublw	0x05		; take from 5
	btfsc	STATUS,c	; if c=o then w is 6 or more
	goto	T_STR
	movlw	D'04'		; if a 6 then add 4 to increment hours
	addwf	TIME_CNT2,f	; add 4 to clock
	movf	TIME_CNT2,w	;
	sublw	D'245'		; when 24:50 goes to 25:00
	btfsc	STATUS,c	; check if reached 1:00		
	goto	T_STR
	movlw	D'10'		; 1:00 time
	movwf	TIME_CNT2	; new time
T_STR	movlw	EEPROMET	; EEPROM empty time address
	movwf	EEADR		; address to write to
	movf	TIME_CNT2,w	; current time
	call 	EWRITE		; eeprom write sequence
	movf	TIME_CNT2,w
	call 	INST
	goto	DELTME
RESET	movlw	0x01		; 1
	movwf	VALID		; valid data counter
	clrf	PULSE_CNT	; rain gauge counter 
	clrf	RAIN_1		; todays rain
	clrf	DAY		; which day reading 
	movlw	EEPROMDT	; valid data storage
	movwf	EEADR		; address
	movlw	0x01		; clear EEPROMDT
	call 	EWRITE		;
	movlw	EEPROM1		; todays reading
	movwf	EEADR
	movlw	0x00
	call	EWRITE		; clear todays reading 
	movlw	0x0B		; R on display after seven segment decode
	movwf	DISP3		; R on display 3
	incf	DISP3,w		; 0B + 1 = 0C
	movwf	DISP2		; E on display 2
	incf	DISP2,w		; 0C + 1, w=0D
	movwf	DISP1		; an S in display after seven segment decode 
	goto	DELTME	

; delay time for switch debounce period

DELTME	bsf	INTCON,T0IE	; allow interrupt
	movlw	D'50'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_1	movwf	VALUE_2		; VALUE_2 = w
LP_2	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1	
		
; wait for switch to open (must test for closed switch when RA0, RA1 and RA2 are low
; because switch will appear open at RA4 when switch pull down via RA0-RA2 is high) 	

CLOSED	bcf	FLAG_1,3	; clear flag that indicates switch closed
	call	ON		; check for closed switch
	call	ON		; do at least four times to return to original switch closure
	call	ON		;
	call	ON 		; 
	call	ON 		; do again just in case missed one interrupt during call
	call	ON 		; do again just in case missed one interrupt during call
	btfsc	FLAG_1,3	; was switch closed 	
	goto 	CLOSED		; yes go to check again
	goto 	TO_DO		; switch open so out

;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

; SUBROUTINES

; subroutine to place binary value into display registers after BCD conversion

INST	movwf	BIN_0		; place in binary 0 register
	sublw	D'254'		; if over 255mm then load dashes
	btfss	STATUS,c 
	goto	DASH		; if over 255 then display is dashes
	call	BCD		; convert binary number to BCD
	movf	BCD_0,w		; ms byte for display
	movwf	DISP3
	movf	BCD_1,w		; contains packed BCD value
	andlw	0x0F		; extract LS bits
	movwf	DISP1
	swapf	BCD_1,w		; swap upper and lower 4-bits
	andlw	0x0F		; extract bits
	movwf	DISP2		; place in middle digit register
	return 
DASH	movlw	0x0A		; value which gives a dash after seven segment decode
	movwf	DISP1
	movwf	DISP2
	movwf	DISP3
	return

; subroutine to wait for switch to reopen
	
ON	bcf	INTCON,T0IE	; prevent multiplex update
	bcf	FLAG_1,2	; clear flag that indicates update interrupt for display
RE_CHK	btfsc	PORTA,4		; check if switch closed
	goto	OPEN
	bsf	FLAG_1,3	; switch closed so set flag
	bsf	INTCON,T0IE	; allow multiplex interrupt
	goto	RE_CHK 		; closed so check till open 
OPEN	bcf	FLAG_1,2	; clear multiplex update flag wait for next multiplex update
	bsf 	INTCON,T0IE	; allow multiplex update
FLG_TST	btfss	FLAG_1,2	; is flag set indicating a new interrupt for multiplexing
	goto 	FLG_TST		; wait 
	return

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_RD	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_RD		; wait for low RD (read RD)	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	bcf	FLAG_1,6	; EEPROM write repeat flag
	movwf	TEMP_2		; store w in temporary storage
EWRIT	movf	TEMP_2,w	; place value in w (used for read data sequence) 
	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	
; read EEPROM DATA and check if written correctly
	
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_AGN	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_AGN		; wait for low RD	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	subwf	EEDATA,w	; compare read value with value stored
	btfsc	STATUS,z	; skip if not the same
	goto	DELY		; value correctly written 
	btfsc	FLAG_1,6	; write repeat bit, skip if clear and rewrite
	goto	DELY
	bsf	FLAG_1,6	; set repeat flag rewrite once only 
	goto 	EWRIT		; rewrite as not written correctly
DELY	movlw	D'50'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_X	movwf	VALUE_2		; VALUE_2 = w
LP_Y	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_Y
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_X	
	return

; Subroutine to convert from 8-bit binary to 2-digit BCD (packed)
; Binary value is in BIN0  
; Result in BCD is in BCD0 & BCD1.  
; BCD0 is MSB, BCD1 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'8'
	movwf	CNT_8		; 8 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
LOOPBCD	rlf	BIN_0,f		; shift left binary registers
	rlf	BCD_1,f		; MSB shift left

	rlf	BCD_0,f		; LSB shift left BCD registers
	decfsz	CNT_8,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_1		; BCD LSB address
	movwf	FSR		; pointer for BCD1
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-1)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w in temp
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

	end
