/******** PC CONTROLLED LCD AND THERMOSTAT **********************************

  Written using Borland Turbo C/C++ for DOS, Version 3.0

  PIN DEFINITIONS
  ------------------------------------------------
  PRINTER PORT             LCD            DS1620
  D0 (2)                   D0 (7)           -
  D1 (3)                   D1 (8)           -
  D2 (4)                   D2 (9)           -
  D3 (5)                   D3 (10)          -
  D4 (6)                   D4 (11)          -
  D5 (7)                   D5 (12)          -
  D6 (8)                   D6 (13)          -
  D7 (9)                   D7 (14)          -
  STROBE (1)               E  (6)         CLK (2)
  AUTO   (14)              RS (4)         DQ  (1)
  INIT   (16)                -            RST (3) (ACTIVE LOW)
  ACK    (10)                -            DQ  (1)

  PORT BIT DEFINITIONS
  ----------------------------
  PORT         DATA BUS TO LCD
  PORT+1       BIT 6 - ACK
  PORT+2       BIT 0 - STROBE
               BIT 1 - AUTO
               BIT 2 - INIT

****************************************************************************/

#include <conio.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>

#define LPT1 0x378
#define LPT2 0x278

#define ON 1
#define OFF 0

#define CLS 0x01        //LCD Clear Display command

/******** FUNCTION PROTOTYPES **********************************************/

void PortInit(void);                    //parallel port initialization
void LCDInit(void);                     //LCD initialization
void LCDCmd(unsigned char data);        //write to LCD command register
void LCDData(unsigned char data);       //write to LCD data register
void LCDString(unsigned char *string);  //write string to LCD
void LCDClear(void);                    //clear LCD and home cursor
void LCDHome(void);                     //home cursor on LCD
void LCDLine2(void);                    //move cursor to line 2, position 1
void LCDRSHigh(void);
void LCDRSLow(void);
void LCDEnableHigh(void);
void LCDEnableLow(void);


/******** GLOBAL DECLARATIONS **********************************************/

unsigned int port = LPT1;               //set printer port base address


/******** PROGRAM START HERE ***********************************************/

int main (void)
{
  char choice,                //menu choice
       textstring[45],        //holds lcd text string to display
       newtext[45];           //holds new text string entered from keyboard

  unsigned char scrollcmd=0;  //LCD scrolling command
                              //0x18 = scroll left
                              //0x1c = scroll right
                              //0    = no scrolling

  int scrolling=OFF,          //scrolling on/off control
      scrollrate=200;         //LCD scrolling rate in milliseconds
                              //not allowed to be less than 10mS
  float newscrollrate;        //holds user entered scroll rate


  delay(250);                 //wait for power up stabilization
  PortInit();                 //initialize parallel port pins
  LCDInit();                  //initialize LCD

  strcpy(textstring,"PC controlled LCD demonstration");
  LCDString(textstring);

  do
    {
      clrscr();
      printf("PC CONTROLLED LCD DEMONSTRATION\n"
             "-------------------------------\n"
             "1. Enter LCD display text\n"
             "2. Scroll text left\n"
             "3. Scroll text right\n"
             "4. Stop scrolling\n"
             "5. Set scroll rate\n"
             "0. Quit\n\n"
             "Choice: ");

      while (!kbhit())        //if no key pressed then scroll display
        {
          if (scrolling)
            {
              LCDCmd(scrollcmd);
              delay(scrollrate);
            }
        };

      choice=getch();
      switch (choice)
        {
          case '1' :
                     printf("\nEnter new text (blank line to quit)\n"
                            "---------|---------|---------|---------|\n");

                     newtext[0]=41;    //set maximum number of input chars
                     cgets(newtext);
                     if (newtext[1]==0) break;
                     strcpy(textstring,newtext+2);
                     LCDClear();
                     LCDString(textstring);
                     break;

          case '2' :
                     scrollcmd=0x18;
                     scrolling=ON;
                     break;

          case '3' :
                     scrollcmd=0x1c;
                     scrolling=ON;
                     break;

          case '4' :
                     LCDClear();
                     LCDString(textstring);
                     scrolling=OFF;
                     break;

          case '5' :
                     printf("\nCurrent scroll rate = %d mS",scrollrate);

                     while(1)
                       {
                         printf("\nEnter scrolling rate (10-1000mS, 0 to quit) ");
                         scanf("%f",&newscrollrate);
                         if (newscrollrate==0) break;
                         if (newscrollrate>=10 && newscrollrate<=1000) break;
                       };

                     if (newscrollrate) scrollrate=(int)newscrollrate;

                     break;

          case '0' : LCDClear();
                     break;
        }
    }
  while (choice!='0');

  return 0;
}

/******** FUNCTIONS HERE ***************************************************/

/* Initialize parallel port pins */

void PortInit(void)
{
  outportb(port+2,0x03);     //SLCTIN high = DS1620 CLK high
                             //INIT low = DS1620 reset low
                             //AUTO low = LCD RS low
                             //STROBE low = LCD Enable low
  delay(10);
}

/* Initialize LCD - 8 bit interface, 2 lines, 5x7 dots */

void LCDInit(void)
{
  LCDCmd(0x38);              //8 bit interface, 2 lines, 5x7 dots
  LCDCmd(0x38);              //do it 3 times
  LCDCmd(0x38);
  LCDCmd(0x0c);              //display on, cursor off, cursor not blinking
  LCDCmd(0x06);              //move cursor with data write
  LCDClear();                //clear display and home cursor
}

/* Write 8-bit data to LCD command register */

void LCDCmd(unsigned char data)
{
  outportb(port,data);
  delay(1);
  LCDRSLow();
  delay(1);
  LCDEnableHigh();
  delay(1);
  LCDEnableLow();
  delay(1);
}

/* Write 8-bit data to LCD data register */

void LCDData(unsigned char data)
{
  outportb(port,data);
  delay(1);
  LCDRSHigh();
  delay(1);
  LCDEnableHigh();
  delay(1);
  LCDEnableLow();
  delay(1);
}

/* Write string to LCD */

void LCDString(unsigned char *string)
{
  int i;

  for (i=0; i<strlen(string); i++) LCDData(string[i]);
}

/* Clear LCD and home cursor */

void LCDClear(void)
{
  LCDCmd(0x01);                 //LCD Clear Display command
  delay(2);                     //wait for command to execute
}

/* Home cursor on LCD */

void LCDHome(void)
{
  LCDCmd(0x02);                 //LCD Home Cursor command
  delay(2);                     //wait for command to execute
}

/* Move cursor to line 2, position 1 */

void LCDLine2(void)
{
  LCDCmd(0x40|0x80);        //set LCD address to 40h (line 2, position 1)
  delay(1);
}

/* LCD RS input high */

void LCDRSHigh(void)
{
  outportb(port+2,inportb(port+2)&0xfd);
}

/* LCD RS input low */

void LCDRSLow(void)
{
  outportb(port+2,inportb(port+2)|0x02);
}

/* LCD ENABLE input high */

void LCDEnableHigh(void)
{
  outportb(port+2,inportb(port+2)&0xfe);
}

/* LCD ENABLE input low */

void LCDEnableLow(void)
{
  outportb(port+2,inportb(port+2)|0x01);
}

