/******** PC CONTROLLED LCD AND THERMOSTAT DEMONSTRATION ********************

  Written using Borland Turbo C/C++ for DOS, Version 3.0

  PIN DEFINITIONS
  ------------------------------------------------
  PRINTER PORT             LCD            DS1620
  D0 (2)                   D0 (7)           -
  D1 (3)                   D1 (8)           -
  D2 (4)                   D2 (9)           -
  D3 (5)                   D3 (10)          -
  D4 (6)                   D4 (11)          -
  D5 (7)                   D5 (12)          -
  D6 (8)                   D6 (13)          -
  D7 (9)                   D7 (14)          -
  STROBE (1)               E  (6)         CLK (2)
  AUTO   (14)              RS (4)         DQ  (1)
  INIT   (16)                -            RST (3) (ACTIVE LOW)
  ACK    (10)                -            DQ  (1)

  PORT BIT DEFINITIONS
  ----------------------------
  PORT         DATA BUS TO LCD
  PORT+1       BIT 6 - ACK
  PORT+2       BIT 0 - STROBE
               BIT 1 - AUTO
               BIT 2 - INIT


/******** DS1620 COMMAND DEFINITIONS ***************************************/

#define READTEMP 0xaa                //Read Temperature
#define READCOUNTER 0xa0             //Read Counter register
#define READSLOPE 0xa9               //Read Slope counter register
#define STARTCONVERT 0xee            //Start temperature conversion
#define STOPCONVERT 0x22             //Stop temperature conversion

#define WRITETH  0x01                //Write to TH register
#define WRITETL  0x02                //Write to TL register
#define READTH   0xa1                //Read TH register
#define READTL   0xa2                //Read TL register
#define WRITECONFIG 0x0c             //Write to Configuration register
#define READCONFIG 0xac              //Read Configuration register


/******** PARALLEL PORT ADDRESS DEFINITIONS ********************************/

#define LPT1 0x378
#define LPT2 0x278
#define LPT3 0x3BC      //port address if using monochrome video card


/******** CONSTANT DECLARATIONS ********************************************/

#define CELSIUS    0
#define FAHRENHEIT 1
#define PLUS       0
#define MINUS      1


/******** LIBRARY FILE INCLUDES ********************************************/

#include <conio.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>


/******** FUNCTION PROTOTYPES **********************************************/

void PortInit(void);                    //parallel port initialization
void LCDInit(void);                     //LCD initialization
void LCDCmd(unsigned char data);        //write to LCD command register
void LCDData(unsigned char data);       //write to LCD data register
void LCDString(unsigned char *string);  //write string to LCD
void LCDClear(void);                    //clear LCD and home cursor
void LCDHome(void);                     //home cursor on LCD
void LCDLine2(void);                    //move cursor to line 2, position 1
void LCDRSHigh(void);
void LCDRSLow(void);
void LCDEnableHigh(void);
void LCDEnableLow(void);

void DS1620Config(int data);             //DS1620 configuration
void DS1620WriteCommand(unsigned char command); //Write command to DS1620
void DS1620WriteData(int data);          //Write data to DS1620
int  DS1620Read(void);                   //Read data from DS1620
void DS1620ResetHigh(void);
void DS1620ResetLow(void);
void DS1620ClkHigh(void);
void DS1620ClkLow(void);
void DS1620DataHigh(void);
void DS1620DataLow(void);

/******** GLOBAL DECLARATIONS **********************************************/

unsigned int port = LPT1;     //set printer port base address



/******** PROGRAM START HERE ***********************************************/

int main (void)
{
  char choice,                //menu choice
       tempstring[10];        //used to convert temperature values to string

  int temp,tl,th,             //variables to hold temperature, TL & TH values
      oldtemp,oldth,oldtl,    //holds previous temperature & setpoint values
      scale=CELSIUS;          //temperature scale (celsius or fahrenheit)
                              //default to celsius

  float temp1,tl1,th1;        //decimal temperature values

  delay(250);                 //wait for power up stabilization
  PortInit();                 //initialize parallel port pins
  LCDInit();                  //initialize LCD
  DS1620Config(0x0002);       //configure DS1620 - continuous conversion

  DS1620WriteCommand(STARTCONVERT);  //start conversions
  delay(1000);                       //wait 1 second before reading

  oldtemp=-1;                 //preset old temperature & setpoint values
  oldth=-1;                   //to force initial display
  oldtl=-1;

  do
    {
      clrscr();
      printf("PC CONTROLLED LCD & THERMOSTAT DEMONSTRATION\n"
             "--------------------------------------------\n"
             "1. Set High Temperature point\n"
             "2. Set Low Temperature point\n"
             "3. Celsius scale\n"
             "4. Fahrenheit scale\n"
             "0. Quit\n\n"
             "Choice: ");

      while (!kbhit())        //if no key pressed then display temperature
                              //and high/low setpoints on LCD
        {
          DS1620WriteCommand(READTEMP);
          temp=DS1620Read();

          if (temp&0x100)     //if temperature is negative
            {
              temp=~temp+1;   //two's complement
              temp=temp&0xff; //keep least significant 8 bits only
              temp=temp*-1;   //convert to negative number
            }

          DS1620WriteCommand(READTH);       //get high temperature setpoint
          th=DS1620Read();

          if (th&0x100)
            {
              th=~th+1;
              th=th&0xff;
              th=th*-1;
            }

          DS1620WriteCommand(READTL);       //get low temperature setpoint
          tl=DS1620Read();

          if (tl&0x100)
            {
              tl=~tl+1;
              tl=tl&0xff;
              tl=tl*-1;
            }

          temp1=temp/2.0;                   //convert readings to decimal
          th1=th/2.0;
          tl1=tl/2.0;

          if (scale==FAHRENHEIT)
            {
              temp1=(temp1*9/5)+32;         //convert to degrees F
              th1=(th1*9/5)+32;
              tl1=(tl1*9/5)+32;
            }

          if (temp!=oldtemp)                //if temperature has changed
            {
              oldtemp=temp;
              LCDLine2();                        //move to second line of LCD
              sprintf(tempstring,"%1.1f",temp1); //convert temp to string
              LCDString(tempstring);             //and display

            }

          if ((th!=oldth)||(tl!=oldtl))     //if setpoints have changed
            {                               //update display
              oldth=th;
              oldtl=tl;

              LCDClear();
              LCDString("Temp    TH=");
              sprintf(tempstring,"%1.1f",th1);   //convert high setpoint to string
              LCDString(tempstring);             //and display

              LCDLine2();                        //move to second line of LCD
              sprintf(tempstring,"%1.1f",temp1); //convert temp to string
              LCDString(tempstring);             //and display

              if (strlen(tempstring)==4) LCDString("    ");
              else LCDString("   ");

              LCDString("TL=");
              sprintf(tempstring,"%1.1f",tl1);   //convert low setpoint to string
              LCDString(tempstring);             //and display
            }
        };

      choice=getch();                       //key pressed, get character

      switch (choice)
        {
          case '1' : printf("\nHigh setpoint (in degrees ");

                     if (scale==CELSIUS) putch('C'); else putch('F');

                     printf(") ");
                     scanf("%f",&th1);

                     if (scale==FAHRENHEIT)
                       {
                         th1=(th1-32)*5/9;
                       }

                     if ((th1<-55.0) || (th1>125.0))
                       {
                         printf("\nError: Out of Range\a");
                         delay(1000);
                         break;
                       }

                     th1=th1*2;
                     th=(int)th1;

                     DS1620WriteCommand(WRITETH);
                     DS1620WriteData(th);
                     oldth=-1;                    //force another read
                     break;

          case '2' : printf("\nLow setpoint (in degrees C)? ");

                     if (scale==CELSIUS) putch('C'); else putch('F');

                     printf(") ");
                     scanf("%f",&tl1);

                     if (scale==FAHRENHEIT)
                       {
                         tl1=(tl1-32)*5/9;
                       }

                     if ((tl1<-55.0) || (tl1>125.0))
                       {
                         printf("\nError: Out of Range\a");
                         delay(1000);
                         break;
                       }

                     tl1=tl1*2;
                     tl=(int)tl1;

                     DS1620WriteCommand(WRITETL);
                     DS1620WriteData(tl);
                     oldtl=-1;
                     break;

          case '3' : scale=CELSIUS;               //celsius scale
                     oldtemp=0;                   //force display update
                     oldth=-1;
                     break;

          case '4' : scale=FAHRENHEIT;            //fahrenheit scale
                     oldtemp=0;                   //force display update
                     oldth=-1;
                     break;

          case '0' : break;
        }
    }
  while (choice!='0');

  LCDLine2();
  LCDString("---- ");

  return 0;
}

/******** FUNCTIONS HERE ***************************************************/

/* Initialize parallel port pins */

void PortInit(void)
{
  outportb(port+2,0x03);     //SLCTIN high = DS1620 CLK high
                             //INIT low = DS1620 reset low
                             //AUTO low = LCD RS low
                             //STROBE low = LCD Enable low
  delay(10);
}

/* Initialize LCD - 8 bit interface, 2 lines, 5x7 dots */

void LCDInit(void)
{
  LCDCmd(0x38);              //8 bit interface, 2 lines, 5x7 dots
  LCDCmd(0x38);              //do it 3 times
  LCDCmd(0x38);
  LCDCmd(0x0c);              //display on, cursor off and not blinking
  LCDCmd(0x06);              //move cursor with data write
  LCDClear();                //clear display and home cursor
}

/* Write 8-bit data to LCD command register */

void LCDCmd(unsigned char data)
{
  outportb(port,data);
  delay(1);
  LCDRSLow();
  delay(1);
  LCDEnableHigh();
  delay(1);
  LCDEnableLow();
  delay(1);
}

/* Write 8-bit data to LCD data register */

void LCDData(unsigned char data)
{
  outportb(port,data);
  delay(1);
  LCDRSHigh();
  delay(1);
  LCDEnableHigh();
  delay(1);
  LCDEnableLow();
  delay(1);
}

/* Write string to LCD */

void LCDString(unsigned char *string)
{
  int i;

  for (i=0; i<strlen(string); i++) LCDData(string[i]);
}

/* Clear LCD and home cursor */

void LCDClear(void)
{
  LCDCmd(0x01);                 //LCD Clear Display command
  delay(2);                     //wait for command to execute
}

/* Home cursor on LCD */

void LCDHome(void)
{
  LCDCmd(0x02);                 //LCD Home Cursor command
  delay(2);                     //wait for command to execute
}

/* Move cursor to line 2, position 1 */

void LCDLine2(void)
{
  LCDCmd(0x40|0x80);        //set LCD address to 40h (line 2, position 1)
  delay(1);
}

/* LCD RS input high */

void LCDRSHigh(void)
{
  outportb(port+2,inportb(port+2)&0xfd);
}

/* LCD RS input low */

void LCDRSLow(void)
{
  outportb(port+2,inportb(port+2)|0x02);
}

/* LCD ENABLE input high */

void LCDEnableHigh(void)
{
  outportb(port+2,inportb(port+2)&0xfe);
}

/* LCD ENABLE input low */

void LCDEnableLow(void)
{
  outportb(port+2,inportb(port+2)|0x01);
}

/* Configure DS1620 chip */

void DS1620Config(int data)
{
  DS1620WriteCommand(WRITECONFIG);
  DS1620WriteData(data);          //write configuration
  delay(10);                      //wait for EE write to finish
}

/* Write 8-bit command to DS1620 */

void DS1620WriteCommand(unsigned char command)
{
  int i;

  DS1620ResetHigh();
  delay(1);

  for (i=0; i<8; i++)         //write 8-bit command byte to chip
    {
      DS1620ClkLow();
      if (command&0x01) DS1620DataHigh();
      else DS1620DataLow();
      delay(1);
      DS1620ClkHigh();
      delay(1);
      command=command>>1;     //ready to write next bit
    }
}

/* Write 9-bit data to DS1620
   Chip ignores bit 8 if not needed */

void DS1620WriteData(int data)
{
  int i;

  for (i=0; i<9; i++)         //write data to chip
    {
      DS1620ClkLow();
      if (data&0x01) DS1620DataHigh();
      else DS1620DataLow();
      delay(1);
      DS1620ClkHigh();
      delay(1);
      data=data>>1;             //ready to write next bit
    }

  DS1620ResetLow();
  delay(5);                     //wait 5mS for EEPROM write to finish
}

/* Read 9 bits of data from DS1620 chip */

int DS1620Read(void)
{
  int i;
  unsigned int data,temp;

  DS1620DataHigh();           //setup to read from data pin
  delay(1);
  data=0;                     //initialize data byte

  for (i=0; i<9; i++)
    {
      DS1620ClkLow();
      delay(1);
      if (inportb(port+1)&0x40) data=data+pow(2,i);  //if bit is high,
                                                     //add its value to total
      DS1620ClkHigh();
      delay(1);
    }

  DS1620ResetLow();
  delay(1);
  return data;
}

/* Set DS1620 RESET input high */

void DS1620ResetHigh(void)
{
  outportb(port+2,inportb(port+2)|0x04);
}

/* Set DS1620 RESET input low */

void DS1620ResetLow(void)
{
  outportb(port+2,inportb(port+2)&0xfb);
}

/* Set DS1620 CLK input high */

void DS1620ClkHigh(void)
{
  outportb(port+2,inportb(port+2)&0xf7);
}

/* Set DS1620 CLK input low */

void DS1620ClkLow(void)
{
  outportb(port+2,inportb(port+2)|0x08);
}

/* Set DS1620 DATA input high */

void DS1620DataHigh(void)
{
  outportb(port+2,inportb(port+2)&0xfd);
}

/* Set DS1620 DATA input low */

void DS1620DataLow(void)
{
  outportb(port+2,inportb(port+2)|0x02);
}