;==============================================================================
;	Program: morseclk.asm
;==============================================================================
;	Programmer: Leon Williams
;==============================================================================
;	Date: 20 September 2000
;==============================================================================
;	Target Processor: PIC 16F84/04P
;==============================================================================
;	Compiler: MPLAB Version 3.4
;==============================================================================
;	Release version: 1.0
;==============================================================================
;	Revisions: NIL
;==============================================================================
;	Program Description:
; A clock using a PIC16F84/04P that announces time via a piezo transducer
; in morse code. XTAL controlled by a 3.2768MHz crystal, divided by 4 to give 
; an internal clock of 819.2KHz (instruction cycle 1.22uS), and then divided 
; by 64 in the prescaler to give 12,800Hz. This is the clock for Timer0 which 
; simply counts down from 255 to 0 and repeats. Each time it reaches 0 it 
; interupts the PIC at a rate of 50Hz. During the interupt routine the various
; registers that adjust the time are updated.
; The main routine monitors these registers and uses them to announce the time.
; The hardware has 2 buttons connected to Port B, which are monitored, while the
; piezeo transducer is connected to Port A. The first button is the TIME button,
; which is pressed to announce the current time. The other button SET is 
; normally not acive. However, when both buttons are pressed at the same time,
; program mode is entered. During program mode the TIME button steps through
; the functions and loops back around. The SET button steps through the options 
; for that function. When both buttons are pressed at the same time, 
; program mode is exited and the clock returns to the main loop.
; *NOTE: SET is a compiler reserved term, so SSET has been used instead.
; 
; All the functions and options are announced in Morse code. 
; The Functions and available options are:
; 'X' = what you hear in morse
;
; Function			Options
;-----------------------------------------------------------------------------
; CHIME	 | 'ON' (announce time on hour), 'OFF' ( do not announce time on hour)
; MODE	 | '1' 12 hour time, '2' 24 hour time
; AM/PM	 | 'AM' AM , 'PM' PM
; SPEED	 | 'F' fast morse speed, 'S' slow morse speed
; HOUR	 | 'H' for hours, then a dot for each press to increment from 0
; MINUTES | 'M' for minutes, then a dot for each press to increment from 0	
;	   *NOTE: the current time settings will not be changed unless the SET
;	    	 button is pressed, after 'H' or 'M' is announced.
;
;
;=========================================================================

	list	p=16f84		; processor type
	radix	hex		; HEX is default language
	errorlevel	-302	; remove warning messages about bank 1/0

;--------------------------------------------------------------------------
;	CONFIGURATION BITS

	__CONFIG 	_CP_OFF & _WDT_OFF & _PWRTE_ON & _XT_OSC

; code protection=off, watch dog timer=off,
; powerup timer =on, oscillator=low speed crystal

;-------------------------------------------------------------------------
;			******Note must be in UPPER CASE*******

;	EXTERNAL EQUATES	

	#include 	<p16f84.inc>

;	INTERNAL EQUATES

ST_TEMP	equ	0x0C	; temporary register to store STATUS during interupt
W_TEMP	equ	0x0D	; temporary register to store W reg during interupt

INTCNT	equ	0x0E	; counter that counts interupts (50Hz), used for time update	
HOURS	equ	0x0F	; counts hours (1-12 or 0-24)
MINUTES	equ	0x10	; counts minutes (0-60)
SECONDS	equ	0x11	; counts seconds (0-60)

MYFLAGS	equ	0x12	; various global flags used by routines
MODE	equ	0	; 1 = 24 hour time, 0 = 12 hour time
CHIME	equ	1	; 1 = chime on hour, 0 = no chime on hour
AMPM	equ	2	; 1 = AM, 0 = PM
HOURUP	equ	3	; 1 = at hour, 0 = not at hour (set by int, cleared by main)
FIRST	equ	4	; 1 = done first time, = 0 NOT done first time (used in time setting)
SPEED	equ	5	; 1 = fast morse, 0 = slow morse
;		6
;		7

POINTER	equ	0x13	; GP pointer
DLYCNTR	equ	0x14	; counter inside interupt loop that is used by
			; the main routine to set delays. Is decremented by
			; interupt routine each 20mS. Is tested in delay for zero

NUMBER	equ	0x15	; holds number to be sent

MYCODE	equ	0x16	; register used in table reads

WORD1	equ	0x17	; set bits in these registers 
AM	equ	0	; before calling sendword, to indicate which word to send	
PM	equ	1
OK	equ	2
OFF	equ	3
ON	equ	4
PGM	equ	5
HR	equ	6
MN	equ	7

WORD2	equ	0x18
SLOW	equ	0
FAST	equ	1

CNTR10	equ	0x19	; counter used to hold 10s of hours or minutes
CNTR1	equ	0x1A	; counter used to hold 1s of hours or minutes

BUTTONS	equ	0x1B	; flags set for each button type by whatbutn routine
NONE	equ	0	; set if NO buttons pressed
TIME	equ	1	; set if only TIME  pressed
SSET	equ	2	; set if only SSET pressed
BOTH	equ	3	; set if BOTH buttons pressed

DOTSIZE	equ	0x1C	; holds the value set in program for slow or fast morse
DASHSIZE	equ	0x1D	; power on default is slow (MYFLAGS SPEED = 0)
CHSPSIZE	equ	0x1E
WDSPSIZE	equ	0x1F	

DOT	equ	b'00000001'	; table return values
DASH	equ	b'00000010'
CHRSPACE	equ	b'00000100'
WRDSPACE	equ	b'00001000'
DONE	equ	b'10000000'

PIEZO	equ	2	; port A output to sound buzzer 1=sound, 0=no sound
GPCNTR	equ	0x20	; general purpose counter, can be destroyed by any routine

;***************************MAIN START**********************************************

	org	0x000
	goto	main		; skip over interupt vector and go to start address
	org	0x004		; interrupt service
	goto	intserv

;----------------------------------------------------------------------------------
; mainloop routine scans  to see if buttons are pressed - active low.
; First checks if TIME button pressed, if it is then check for SSET button down as well
; If both are down then enter program mode. If only TIME down announce the time.
; If TIME button is not down check if hour is up, that is, if the minutes are at 60.
; If it is an hour then check to see if chime allowed. If it is then chime hourly time,
; otherwise just loop around again.


main	bcf	STATUS,RP0	; ensure we are in bank 0
	call	initpic		; initialise all registers, and turn off interupts

	bcf	PORTA,PIEZO	; ensure piezo is off
	bcf	INTCON,T0IF	; clear timer 0 interupt flag
	bsf	INTCON,GIE	; enable global interupts 
	bsf	INTCON,T0IE	; enable timer 0 interupt


mainretn	bsf	WORD1,OK		; sound letters OK at power up or after 
	call	sendword		; returning from program

mainloop	btfss	MYFLAGS,HOURUP	; test for hour ready
	goto	mainbutn		; no hour up yet, see if a button pressed
	btfsc	MYFLAGS,CHIME	; hour is ready, do we chime on hour?	
	call	chime		; yes, so chime hour time
	bcf	MYFLAGS,HOURUP	; clear hour flag if chime or not
	goto	mainloop		; loop around and wait
	
mainbutn	call	whatbutn		; see if a button is pressed
	btfsc	BUTTONS,TIME	; see if time button pressed on its own, 
	goto	maintime		; yes time on its own pressed, send time
	btfss	BUTTONS,BOTH	; time not pressed, see if both buttons pressed
	goto	mainloop		; not time or both, do nothing, just loop around
	call	program		; both buttons pressed, do program
	goto	mainretn

maintime	call	sendtime		; time pressed, but not set so just sound time
	goto	mainloop		; and return



;***************************MAIN END********************************************



;***************************CALLS***********************************************

;----------------------------START SENDWORD------------------------------------------
; This routine send the word in morse code, pointed to by calling routine.
; Relevant bit in WORD1 or WORD2 must be set first

sendword	clrf	POINTER		; reset table offset pointer

	movlw	d'1'		; synchronise morse start with start of next inturupt
sendwsyn	movf	DLYCNTR,f	; load DLYCNTR with 1 and wait to be zero, then 
	btfss	STATUS,Z		; can start sending dots or dashes
	goto	sendwsyn		; not zero yet, loop around

sendwagn	movf	POINTER,w	; load offset pointer in w reg
	
	btfsc	WORD1,AM
	call	_AM
	btfsc	WORD1,PM
	call	_PM
	btfsc	WORD1,OK
	call	_OK
	btfsc	WORD1,OFF
	call	_OFF
	btfsc	WORD1,ON
	call	_ON
	btfsc	WORD1,PGM
	call	_PGM
	btfsc	WORD1,HR
	call	_HR
	btfsc	WORD1,MN
	call	_MN

	btfsc	WORD2,SLOW
	call	_S
	btfsc	WORD2,FAST
	call	_F		

	movwf	MYCODE		; store returned data from table	
sendwend	btfsc	MYCODE,7		; bit 7=1 if done, else keep going
	goto	sendwfin		; found done, so go home

	btfsc	MYCODE,0		; not done yet, is it a dot?
	call	senddot
	btfsc	MYCODE,1		; is it a dash?
	call	senddash
	btfsc	MYCODE,2		; is it a character space?
	call	sndchrsp
	btfsc	MYCODE,3		; is it a word space?
	call	sndwdsp

	incf	POINTER,f	; point to next data
	goto	sendwagn		; around again	

sendwfin	clrf	WORD1
	clrf	WORD2
	return

;			**WORD TABLES**

_AM	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DOT
	retlw	DASH
	retlw	CHRSPACE
	retlw	DASH
	retlw	DASH
	retlw	DONE		;table terminator

_PM	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DOT
	retlw	DASH
	retlw	DASH
	retlw	DOT
	retlw	CHRSPACE
	retlw	DASH
	retlw	DASH
	retlw	DONE		;table terminator

_OK	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	CHRSPACE
	retlw	DASH
	retlw	DOT
	retlw	DASH
	retlw	DONE		;table terminator

_ON	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	CHRSPACE
	retlw	DASH
	retlw	DOT
	retlw	DONE		;table terminator

_OFF	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	CHRSPACE
	retlw	DOT
	retlw	DOT
	retlw	DASH
	retlw	DOT
	retlw	CHRSPACE
	retlw	DOT
	retlw	DOT
	retlw	DASH
	retlw	DOT
	retlw	DONE		;table terminator

_PGM	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DOT
	retlw	DASH
	retlw	DASH
	retlw	DOT
	retlw	CHRSPACE
	retlw	DASH
	retlw	DASH
	retlw	DOT
	retlw	CHRSPACE
	retlw	DASH
	retlw	DASH
	retlw	DONE		;table terminator

_HR	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DONE		;table terminator

_MN	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DASH
	retlw	DASH
	retlw	DONE		;table terminator

_S	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DONE		;table terminator

_F	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DOT
	retlw	DOT
	retlw	DASH
	retlw	DOT
	retlw	DONE		;table terminator


;----------------------------END SENDWORD---------------------------------------------

;--------------------------------START SENDNUM---------------------------------------
; This routine sends the number in NUMBER entered by calling routine.
; Does not use same scheme as sendword, because the morse numbers all have 5 dots or
; dashes. So can simply count down the table 5 times the number.

sendnum	clrf	POINTER		; reset table offset pointer
	movlw	d'5'		; counter of data bits sent (all numbers have 5)
	movwf 	GPCNTR

	movlw	d'1'		; synchronise morse start with start of next inturupt
sendnsyn	movf	DLYCNTR,f	; load DLYCNTR with 1 and wait to be zero, then 
	btfss	STATUS,Z		; can start sending dots or dashes
	goto	sendnsyn		; not zero yet, loop around
	
	movf	NUMBER,f		; test if number = 0
	btfsc	STATUS,Z		; z set if zero
	goto	sendnlp		; number is zero, start at top of table

sendndwn	movlw	d'5'		; number not zero, 
	addwf	POINTER,f	; add 5 x number to pointer to go down table
	decfsz	NUMBER,f		; decrement number
	goto	sendndwn		; not done yet, point further down table
	
sendnlp	movf	POINTER,w
	call	sendget		; go get the dot or dash or space from table
	movwf	MYCODE		; put w (value from table) into mycode reg

	btfsc	MYCODE,0		; is it a dot?
	call	senddot
	btfsc	MYCODE,1		; is it a dash?
	call	senddash
	btfsc	MYCODE,2		; is it a character space?
	call	sndchrsp
	btfsc	MYCODE,3		; is it a word space?
	call	sndwdsp	
 	decfsz	GPCNTR,f	 	; see if done 5 bits of the number yet
	goto	sendnagn		; not yet
	goto	sendnfin		; finished go home

sendnagn	incf	POINTER,f	; point to next data
	goto	sendnlp		; around again	

sendnfin	clrf	NUMBER		; clear pointers
	return

;			**NUMBER TABLE**

sendget	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DASH		; 0
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 1
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 2
	retlw	DOT
	retlw	DASH
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 3
	retlw	DOT
	retlw	DOT
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 4
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DASH

	retlw	DOT		; 5
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 6
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 7
	retlw	DASH
	retlw	DOT
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 8
	retlw	DASH
	retlw	DASH
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 9
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	DOT


;---------------------------END SENDNUM------------------------------------------------

;---------------------------START WHAT BUTTONS----------------------------------------------
; Buttons connected to PORT B. Pins held normally HIGH by internal pullups. 
; When button pressed pins go LOW.
; Firstly checks to see if a button is pressed. If is then delays to allow buttons
; to settle and then check again. If still a button pressed, does check for which
; one, sets BUTTONS flag and then returns. PortB returned values for buttons are:
; TIME = RB1 00000010
; SSET = RB2 00000100
; BOTH = 00000000
; NONE = 00000110

whatbutn	clrf	BUTTONS		; clear any previous flags set
	movf	PORTB,w		; get input value from portb where buttons are
	andlw	b'00000110'	; mask off unwanted bits, leave only bits 1 and 2
	movwf	CNTR10		; store result here temporary
	movlw	b'00000110'	; check for no buttons	
	subwf	CNTR10,w		; counter not destroyed
	btfsc	STATUS,Z		; z flag set if input same as w
	goto	whatnone

	movlw	d'5'		; button(s) pressed, 100mS debounce delay
	movwf	DLYCNTR		; and then check again
	call	delay

	movf	PORTB,w		; get input value from portb where buttons are
	andlw	b'00000110'	; mask off unwanted bits, leave only bits 1 and 2
	movwf	CNTR10		; store result here temporary
	movlw	b'00000110'	; check for no buttons	
	subwf	CNTR10,w		; counter not destroyed
	btfsc	STATUS,Z		; z flag set if input same as w
	goto	whatnone

	movlw	b'00000000'	; check for both buttons (active low)
	subwf	CNTR10,w		; counter not destroyed
	btfsc	STATUS,Z		; z flag set if input same as w
	goto	whatboth

	movlw	b'00000100'	; check for time button (active low)	
	subwf	CNTR10,w		; counter not destroyed
	btfsc	STATUS,Z		; z flag set if input same as w
	goto	whattime

whatsset	bsf	BUTTONS,SSET	; not none or both or time, must be sset pressed
	return			; do not wait for set button to be raised
				; so that it repeats automatically
	
whatnone	bsf	BUTTONS,NONE
	goto	whatdone
whatboth	bsf	BUTTONS,BOTH
	goto	whatdone
whattime	bsf	BUTTONS,TIME
	goto	whatdone

whatdone	movf	PORTB,w		; get input value from portb where buttons are
	andlw	b'00000110'	; mask off unwanted bits, leave only bits 1 and 2
	movwf	CNTR10		; store result here temporary
	movlw	b'00000110'	; check for no buttons	
	subwf	CNTR10,w		; counter10 not destroyed
	btfss	STATUS,Z		; z flag set if input same as w
	goto	whatdone		; not all raised yet, loop till are

	movlw	d'5'		; all button(s) raised, 100mS debounce delay
	movwf	DLYCNTR		; 
	call	delay
	return

;---------------------------END WHAT BUTTONS----------------------------------------


;--------------------------START PROGRAM-----------------------------------------------
; This routine lets the user select the various options. 
; Gets here if both SSET and TIME buttons are pressed together.
; TIME button steps through functions, while SSET button steps through options 
; for that function.
; Loops around functions until both buttons are again pressed together, 
; then routine returns to mainloop and sends OK

program	bsf	WORD1,PGM	; send program message
	call	sendword

progwait	call	whatbutn		; wait for time button or both to be pressed
	btfsc	BUTTONS,TIME
	goto	proch		; time button pressed, proceed
	btfsc	BUTTONS,BOTH
	goto	proend		; both pressed, clean up and return
	goto	progwait		; not time or both, loop around
;------------------------------------
proch	btfsc	MYFLAGS,CHIME	; look at flags
	goto	prochson		; chime is currently ON 
prochsof	bsf	WORD1,OFF	; send chime off
	call	sendword
	goto	prochlp		; done chime sending
prochson	bsf	WORD1,ON		; send chime on
	call	sendword

prochlp	call	whatbutn		; go and see if time pressed on its own
	btfsc	BUTTONS,TIME
	goto	promo		; is time button go onto 12/24 time setting
	btfsc	BUTTONS,SSET	; see if set is pressed
	goto	prochswp		; set pressed on its own, swap chime bit
	btfsc	BUTTONS,BOTH	; see if both buttons pressed	
	goto	proend		; both buttons pressed go home
	goto	prochlp		; no buttons pressed, just wait for a press!

prochswp	btfss	MYFLAGS,CHIME	; set pressed look at flags
	goto	prochon		; chime is currently OFF, go turn on
prochoff	bcf	MYFLAGS,CHIME	; currently ON, turn off chime
	goto	proch		; send again and wait 
prochon	bsf	MYFLAGS,CHIME	; turn on chime
	goto	proch		; send again and wait 

;-------------------------------------
promo	btfsc	MYFLAGS,MODE	; look at flags 1=24 0=12
	goto	promos24		; 24hour
promos12	movlw	d'1'
	movwf	NUMBER
	call	sendnum
	goto	promolp		; done sending

promos24	movlw	d'2'
	movwf	NUMBER
	call	sendnum

promolp	call	whatbutn		; go and see if time pressed on its own
	btfsc	BUTTONS,TIME
	goto	proap		; is time button go onto AM/PM setting
	btfsc	BUTTONS,SSET	; see if set is pressed
	goto	promoswp		; set pressed on its own, swap mode bit
	btfsc	BUTTONS,BOTH	; see if both buttons pressed	
	goto	proend		; both buttons pressed go home
	goto	promolp		; no buttons pressed, just wait for a press!

promoswp	btfss	MYFLAGS,MODE	; set pressed, look at flags
	goto	promo24		; mode is currently 12
	bcf	MYFLAGS,MODE	; currently 24, make 12
	goto	promo		; send again and wait 
promo24	bsf	MYFLAGS,MODE	; make 24
	goto	promo		; send again and wait 

;--------------------------
proap	btfsc	MYFLAGS,MODE	; look at flags, skip AMPM if 24 hour time
	goto	pros		; is 24 hour, so do not allow AMPM, go to speed

proapagn	btfsc	MYFLAGS,AMPM	; look at flags 1=AM 0=PM
	goto	proapsam		; AM
proapspm	bsf	WORD1,PM
	call	sendword		; send PM
	goto	proaplp		

proapsam	bsf	WORD1,AM
	call	sendword		; send AM

proaplp	call	whatbutn		; go and see if time pressed on its own
	btfsc	BUTTONS,TIME
	goto	pros		; is time button go onto speed setting
	btfsc	BUTTONS,SSET	; see if set is pressed
	goto	proapswp		; set pressed on its own, swap ampm bit
	btfsc	BUTTONS,BOTH	; see if both buttons pressed	
	goto	proend		; both buttons pressed go home
	goto	proaplp		; no buttons pressed, just wait for a press!

proapswp	btfss	MYFLAGS,AMPM	; set pressed look at flags
	goto	proapam		; currently PM
	bcf	MYFLAGS,AMPM	; currently AM, make PM
	goto	proap		; send again and wait 
proapam	bsf	MYFLAGS,AMPM	; make AM
	goto	proapagn		; send again and wait 

;--------------------------
pros	btfsc	MYFLAGS,SPEED	; look at flags 1=fast 0=slow
	goto	prossndf		; fast
prossnds	bsf	WORD2,SLOW
	call	sendword		; send slow
	goto	proslp		

prossndf	bsf	WORD2,FAST
	call	sendword		; send fast

proslp	call	whatbutn		; go and see if time pressed on its own
	btfsc	BUTTONS,TIME
	goto	protih		; is time button go do hours
	btfsc	BUTTONS,SSET	; see if set is pressed
	goto	prosswp		; set pressed on its own, swap ampm bit
	btfsc	BUTTONS,BOTH	; see if both buttons pressed	
	goto	proend		; both buttons pressed go home
	goto	proslp		; no buttons pressed, just wait for a press!

prosswp	btfss	MYFLAGS,SPEED	; set pressed, look at flags
	goto	prosetf		; currently slow
	bcf	MYFLAGS,SPEED	; currently fast, make slow
	movlw	d'6'		; load with slow speed values
	movwf	DOTSIZE		; 
	movlw	d'18'
	movwf	DASHSIZE	
	movlw	d'18'
	movwf	CHSPSIZE	
	movlw	d'36'
	movwf	WDSPSIZE	
	goto	pros		; send again and wait 
prosetf	bsf	MYFLAGS,SPEED	; make fast
	movlw	d'3'		; load with fast speed values
	movwf	DOTSIZE		; 
	movlw	d'9'
	movwf	DASHSIZE	
	movlw	d'9'
	movwf	CHSPSIZE	
	movlw	d'18'
	movwf	WDSPSIZE	
	goto	pros		; send again and wait 

;---------------------------------

protih	bsf	WORD1,HR		; send H for hour
	call	sendword
	bcf	MYFLAGS,FIRST	; allow reset of hours only if making a change

protihlp	call	whatbutn		; go and see if time pressed on its own
	btfsc	BUTTONS,TIME
	goto	protim		; is time button go onto minutes setting
	btfsc	BUTTONS,SSET	; see if set is pressed
	goto	protihnx		; set pressed on its own, set next hour
	btfsc	BUTTONS,BOTH	; see if both buttons pressed	
	goto	proend		; both buttons pressed go home
	goto	protihlp		; no buttons pressed, just wait for a press!

protihnx	call	senddot		; sound dot to show button pressed
	btfsc	MYFLAGS,FIRST	; see if this is the first time here
	goto	protihol		; not first time, so allow increment without zero
	clrf	HOURS		; first time so reset hours
	bsf	MYFLAGS,FIRST	; stop reset next time
protihol	incf	HOURS,f		; next hours
	btfsc	MYFLAGS,MODE	; now we check if over 12 or 24
	goto	protih2		; is 24 hour check if over 24
protih1	movlw	d'13'		; it is 12 hour
	subwf	HOURS,w
	btfss	STATUS,C
	goto	protihlp		; not 13 or over, loop around
	clrf	HOURS		; over 12 so reset hours
	goto	protihlp

protih2	movlw	d'24'		; it is 24 hour
	subwf	HOURS,w
	btfss	STATUS,C
	goto	protihlp		; not 24 or over, loop around
	clrf	HOURS		; over 23 so reset hours
	goto	protihlp
	
;--------------------------------
protim	bsf	WORD1,MN		; send M for minutes
	call	sendword
	bcf	MYFLAGS,FIRST	; allow reset of minutes only if making a change

protimlp	call	whatbutn		; go and see if time pressed on its own
	btfsc	BUTTONS,TIME
	goto	proch		; loop around again to top
	btfsc	BUTTONS,SSET	; see if set is pressed
	goto	protimnx		; set pressed on its own, set next hour
	btfsc	BUTTONS,BOTH	; see if both buttons pressed	
	goto	proend		; both buttons pressed go home
	goto	protimlp		; no buttons pressed, just wait for a press!

protimnx	call	senddot		; sound dot to show button pressed
	btfsc	MYFLAGS,FIRST	; see if this is the first time here
	goto	protimol		; not first time, so allow increment without zero
	clrf	MINUTES		; first time so reset minutes
	bsf	MYFLAGS,FIRST	; stop reset of minutes next time around
protimol	incf	MINUTES,f	; next minutes
	movlw	d'60'		; re load seconds down counter with 60
	movwf	SECONDS

	movlw	d'60'		; see if minutes over 60
	subwf	MINUTES,w
	btfss	STATUS,C
	goto	protimlp		; not 60 or over, loop around
	clrf	MINUTES		; over 60 so reset minutes
	goto	protimlp

;-----------------------------------------
proend	call	whatbutn		; wait for all buttons to be raised
	btfss	BUTTONS,NONE
	goto	proend
	return			; both buttons raised, done program

;----------------------------END PROGRAM----------------------------------------------


;------------------------START SENDTIME-------------------------------------------
; This routine sends the time in morse code when called, either by pressing the 
; TIME button or on the hour if CHIME is set.
; Takes the hours and minutes values which are binary and subtracts 10 from the value
; until cannot subtract any more. The result is two BCD numbers (1s and 10s). 
; These numbers are used to locate the dots and dashes in the sendnumber routine.
; Tests for 12 or 24 hour time. If it is 12 hour time sends correct AM/PM message.

sendtime	movlw	d'10'
	movwf	DLYCNTR
	call 	delay		; 200mS  delay
	clrf	CNTR10		; clear gp counter
	movf	HOURS,w		; get a copy of hours and put in counter 
	movwf	CNTR1
sendtia	movlw	d'10'		; see how many 10's are in hours
	subwf	CNTR1,w		; hours - 10 (w)
	btfss	STATUS,C		; if hours less than ten, no carry from sub  
	goto	sendtib		; C is 0, hours less than 10
	movwf	CNTR1		; store new hours
	incf	CNTR10,f		; hours >10, increment 10's of hours by one
	goto	sendtia		; try again (subtract another 10)

sendtib	movf	CNTR10,f		; test for - is there any 10s of hours
	btfsc	STATUS,Z		; Z flag set if there are no 10s of hours
	goto	sendtih		; Z flag is set, no 10s to send, just send 1s	
	movf	CNTR10,w		; Z flag is clear, so send 10s of hours
	movwf	NUMBER		
	call	sendnum
	call	sndchrsp

sendtih	movf	CNTR1,w		; send 1s of hours
	movwf	NUMBER
	call	sendnum

	movlw	d'25'
	movwf	DLYCNTR
	call 	delay		; 500mS pause between hours and minutes
			
	clrf	CNTR10		; clear gp counter
	movf	MINUTES,w	; get a copy of minutes and put in counter 
	movwf	CNTR1
sendtic	movlw	d'10'		; see how many 10's are in minutes
	subwf	CNTR1,w		; minutes - 10 (w) 
	btfss	STATUS,C		; if minutes less than ten, no carry, 
	goto	sendtid		; C is 0, minutes less than 10
	movwf	CNTR1		; store new minutes value
	incf	CNTR10,f		; increment 10's of minutes by one
	goto	sendtic		; try again (subtract another 10)

sendtid	movf	CNTR10,w		; send 10s of minutes
	movwf	NUMBER
	call	sendnum
	call	sndchrsp
	movf	CNTR1,w		; send 1s of minutes
	movwf	NUMBER
	call	sendnum

	call	sndwdsp

	btfsc	MYFLAGS,MODE	; see if 12 or 24 hour time
	return			; is 24 hour time so finished
	btfss	MYFLAGS,AMPM	; is 12 hour time so see if am or pm
	goto	sendtip		; is PM
	bsf	WORD1,AM
	call	sendword
	return
sendtip	bsf	WORD1,PM
	call	sendword
	return

;----------------------------END SENDTIME----------------------------------------

;-----------------------------START CHIME-----------------------------------------
; If a hour is up, that is the minutes have turned 60, 
; and chime is set, announce the time.

chime	call	sendtime

	return

;------------------------------END CHIME------------------------------------------


;-----------------------------START SENDDOT-------------------------------------------
; This routine sends(piezo ON) a dot (of dotsize) and then a space of 1 dot

senddot	movf	DOTSIZE,w	; get the current dot size, put in DLYCNTR
	movwf	DLYCNTR		; DLYCNTR decremented by int routine
	bsf	PORTA,PIEZO	; piezo output high	
senddota	movf	DLYCNTR,f	; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z		; z flag set if zero
	goto	senddota		; not zero yet, loop around
	bcf	PORTA,PIEZO	; dot finished, take output low

	movf	DOTSIZE,w	; wait one dot time, as a inter dot/dash delay
	movwf	DLYCNTR		; DLYCNTR decremented by int routine
senddotb	movf	DLYCNTR,f	; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z		; z flag set if zero
	goto	senddotb		; not zero yet, loop around

	return
;------------------------------END SEND DOT--------------------------------------------

;-----------------------------START SENDDASH-------------------------------------------

senddash	movf	DASHSIZE,w	; get dash size and put in DLYCNTR
	movwf	DLYCNTR		; DLYCNTR decremented by int routine
	bsf	PORTA,PIEZO	; piezo output high	
senddsha	movf	DLYCNTR,f	; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z		; z flag set if zero
	goto	senddsha		; not zero yet, loop around
	bcf	PORTA,PIEZO	; dash finished, take output low

	movf	DOTSIZE,w	; wait one dot time, as a inter dot/dash delay
	movwf	DLYCNTR		; DLYCNTR decremented by int routine
senddshb	movf	DLYCNTR,f	; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z		; z flag set if zero
	goto	senddshb		; not zero yet, loop around

	return
;------------------------------END SEND DASH--------------------------------------------

;-----------------------------START SNDCHRSP-------------------------------------------
; Space of no sound between each group of dots and dashes that form a character

sndchrsp	movf	CHSPSIZE,w	; 1 char space 
	movwf	DLYCNTR		; DLYCNTR decremented by int routine
sndchra	movf	DLYCNTR,f	; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z		; z flag set if zero
	goto	sndchra		; not zero yet, loop around
	bcf	PORTA,PIEZO	; character space finished, take output low

	return
;------------------------------END SNDCHRSP--------------------------------------------

;-----------------------------START SNDWDSP-------------------------------------------
; Space between words or groups of characters

sndwdsp	movf	WDSPSIZE,w	; 1 word space 
	movwf	DLYCNTR		; DLYCNTR decremented by int routine
sndwda	movf	DLYCNTR,f	; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z		; z flag set if zero
	goto	sndwda		; not zero yet, loop around
	bcf	PORTA,PIEZO	; character space finished, take output low

	return
;------------------------------END SNDWDSP--------------------------------------------


;------------------------------START DELAY------------------------------------------
; delay = value in DLYCNTR x 20mS. DLYCNTR must be loaded before calling this routine
; DLYCNTR is (unusually) copied to itself. This sets Zero flag if reg is zero.

delay	movf	DLYCNTR,f; check if counter=0. DLYCNTR is decremented by interupt 
	btfss	STATUS,Z	; z flag set if zero
	goto	delay	; not zero yet, loop around		
 	return
;-------------------------------END DELAY-------------------------------------------


;--------------------------INTERUPT START-------------------------------------------
; this interrupt routine is entered when timer 0 overflows at a rate of 50Hz
; updates seconds, minutes and hours counters
; checks for MODE and increments hours either 1-12 or 0-24, and sets AM PM settings
; also sets an hour flag for main routine each 60 minutes
; also decrements DLYCNTR each 20mS  for main routine delays

intserv	movwf	W_TEMP		; save w reg
	movf	STATUS,w		; move status reg into w reg
	bcf	STATUS,RP0	; ensure file register set to bank 0
	movwf	ST_TEMP		; save status reg
	bcf	INTCON,GIE	; disable global interupts 
	bcf	INTCON,T0IF	; clear timer 0 interupt flag
	
	decf	DLYCNTR,f	; used by external delay routines  	

intcnt	decfsz	INTCNT,f		; decrement interupt counter (counts down from 50)
	goto	intdone		; interupt counter not zero yet, do nothing
	movlw	d'50'		; re load interupt counter with 50
	movwf	INTCNT
intschk	decfsz	SECONDS,f	; done 50 interupts, now decrement seconds counter
	goto	intdone		; not counted down 60 seconds yet, finished

	movlw	d'60'		; re load seconds down counter with 60
	movwf	SECONDS

	incf	MINUTES,f	; done 60 seconds, now increment minutes
intmchk	movlw	d'60'		; check if minutes reached 60
	subwf	MINUTES,w	; minutes - 60, result in w, minutes not destroyed
	btfss	STATUS,Z		; z = 1 if equal (minutes = 60)
	goto	intdone		; not 60 minutes yet, finished

inthchk	clrf	MINUTES		; done 60 minutes, reset minutes to 0
	bsf	MYFLAGS,HOURUP	; set hour flag
	btfsc	MYFLAGS,MODE	; check if 12 or 24 hour time
	goto	int24hr		; is set to 24 hour time
int12hr	incf	HOURS,f		; 12 hour time, done 60 minutes now increment hours
	movlw	d'12'		; check if at 11:59 + 1 minute to change AM/PM
	subwf	HOURS,w		; hours -12
	btfss	STATUS,Z		; yes at 12 so update am/pm
	goto	int1pm		; not up to 11:59 +1 minute yet, check for 1pm
	btfss	MYFLAGS,AMPM	; set correct am pm
	goto	intsetpm
	bcf	MYFLAGS,AMPM	; was set to 1 = AM, now set to 0 = PM
	goto	intdone
intsetpm	bsf	MYFLAGS,AMPM	; was set to 0 = PM, now set to 1 = AM
	goto	intdone

int1pm	movlw	d'13'		; check if at 12:59 + 1 minute
	subwf	HOURS,w		; hours -13
	btfss	STATUS,Z
	goto	intdone		; not up to 12:59 +1 minute yet, finished
	movlw	d'1'		; gone past 12:59, set hours to 1
	movwf	HOURS
	goto	intdone

int24hr	incf	HOURS,f		; 24 hour time, done 60 minutes now increment hours
	movlw	d'24'		; check if at 23:59 + 1 minute
	subwf	HOURS,w		; hours - 24
	btfss	STATUS,Z
	goto	intdone		; not up to 23: 59 +1 minute yet
	movlw	d'0'		; gone past 23:59, set hours to 0
	movwf	HOURS

intdone	bsf	INTCON,GIE	; re-enable global interupts (re-enables timer 0 int)	
	bcf	STATUS,RP0
	movf	ST_TEMP,w	; get status
	movwf 	STATUS		; restore status
	swapf	W_TEMP,f	
	swapf	W_TEMP,w		; restore w reg

	retfie			; return to main program

;--------------------------INTERUPT END--------------------------------------------


;--------------------------START INIT----------------------------------------------
; initialisation routine, setup defaults for PIC hardware and defined registers
;
; PORT A outputs
; RA2=output PIEZO, rest outputs - not used
;
; PORT B mixed
; RB1=input TIME, RB2=input SSET, rest outputs - not used
;

initpic	bcf	INTCON,GIE	;disable global interupts
	bcf	STATUS,RP0	;go to register bank 0
	clrf	STATUS
	movlw	b'00000000'	; disable all interupts
	movwf	INTCON		; 

	clrf	PORTA		; clear port latches a,b 
	clrf	PORTB

	bsf	STATUS,RP0	; go to register bank 1
	movlw	b'00000101'	; portb pullups
	movwf	OPTION_REG	; Tmr0 int clk,  prescaler 64

	movlw	b'00000000'	; 
	movwf	TRISA		; set port A outputs
	movlw	b'00000110'	; 
	movwf	TRISB		; set port B inputs

	bcf	STATUS,RP0	; go back to register bank 0

	movlw	d'50'		; pre load interupt down counter with 50
	movwf	INTCNT
	movlw	d'60'		; pre load seconds down counter with 60
	movwf	SECONDS

	movlw	d'6'		; pre-load speed registers with slow values
	movwf	DOTSIZE		; 
	movlw	d'18'
	movwf	DASHSIZE	
	movlw	d'18'
	movwf	CHSPSIZE	
	movlw	d'36'
	movwf	WDSPSIZE	

	clrf	MINUTES		; clear registers
	clrf	HOURS
	clrf	MYFLAGS
	clrf	DLYCNTR
	clrf	WORD1
	clrf	WORD2
	clrf	NUMBER
	clrf	BUTTONS

	return			; finished

;-----------------------------END INIT------------------------------------------



	end			; PHEW! no more
