
;==============================================================================
;	Program: dialup.asm
;
;==============================================================================
;	Programmer: Leon Williams
;==============================================================================
;	Date: March 2001
;==============================================================================
;	Target Processor: PIC 16F84/04P
;==============================================================================
;	Compiler: MPLAB Version 3.4
;==============================================================================
;	Release version: 1.0
;==============================================================================
;	Revisions: NIL 
;==============================================================================
;	Program Description:
; A program based on a PIC16F84, to answer an incoming telephone call, and 
; respond to DTMF digits received via an external microphone and amp.
; Coupled to a MC145436 DTMF decoder, which also supplies the external clock 
; for the PIC. This clock is much slower than normal with a frequency
; of around 447KHz. Because of this, consideration has to been given to instruction
; cycle times when real time routines are used.
; The device is intruder protected by a 6 digit password, and a timer is employed
; to release the call if a digit is not entered within 3 minutes. 
; There is a diagnostic routine to test the operation of the
; DTMF detector and a routine to 'train' the system to the particluar ring cadence
; on the connected telephone line. Also there is the ability to recover from
; a forgotten password by restoring a default password. An EEPROM stores the
; password and ring detector cadence limit values in case of a power outage.
; There are two outputs to drive relays and two opto coupled inputs.
; An output drives a solenoid to operate the talk button on a hands free telephone.
; These techniques avoids the requirement to directly couple to the telephone line.

;==============================================================================

	list	p=16f84		; processor type
	radix	hex		; HEX is default language
	errorlevel	-302	; remove warning messages about bank 1/0

;--------------------------------------------------------------------------
;	CONFIGURATION BITS

	__CONFIG 	_CP_OFF & _WDT_OFF & _PWRTE_ON & _XT_OSC

; code protection=off, watch dog timer=off,
; powerup timer =on, oscillator=low speed crystal

;-------------------------------------------------------------------------
;	EXTERNAL FILES	
			
	#include 	<p16f84.inc>

;	INTERNAL EQUATES

;-----------------REGISTERS------------------------------------------------------

ST_TEMP	equ	0x0C	; temporary register to store STATUS during interupt
W_TEMP	equ	0x0D	; temporary register to store W reg during interupt

MYFLAGS	equ	0x0E	; various global flags used by routines
RINGOK	equ	0	; 1= detected valid ring bursts
TIMEOUT	equ	1	; 1= did not decode a DTMF digit within 3 seconds
IDLE	equ	2	; 1= did not decode a DTMF digit within 60 seconds
BADPASS	equ	3	; 1= a bad password digit entered from user
;	equ	4	; 
;	equ	5	;
;	equ	6	;
;	equ	7	;

; Global variables

POINTER	equ	0x0F	; GP pointer
COUNT1	equ	0x10	; GP counter 1
COUNT2	equ	0x11	; GP counter 2
COUNT3	equ	0x12	; GP counter 3
DTMFDATA	equ	0x13	; holds received dtmf digit

PASS1	equ	0x14	; password RAM locations loaded from EEPROM
PASS2	equ	0x15
PASS3	equ	0x16
PASS4	equ	0x17
PASS5	equ	0x18
PASS6	equ	0x19

RING1L	equ	0x20	; ring time stored values loaded from EEPROM
RING1H	equ	0x21	; L is the lower limit allowed
RING2L	equ	0x22	; H is the upper limit allowed
RING2H	equ	0x23
RING3L	equ	0x24
RING3H	equ	0x25
RING4L	equ	0x26	
RING4H	equ	0x27
RING5L	equ	0x28
RING5H	equ	0x29
RING6L	equ	0x2A
RING6H	equ	0x2B

TONEFREQ	equ	0x30	; holds half cycle value for tone frequency in 8.94uS steps
TONEDUR	equ	0x31	; holds value for tone beep length
HASHCNT	equ	0x32	; counts hash digits received, to check for ending call
DEFCNT	equ	0x33	; default counter
PASSDGT	equ	0x34	; number of password digits
W4CNTR1	equ	0x35	; used by routines as a wait counter
W4CNTR2	equ	0x36	; used by routines as a wait counter
LOADCNT	equ	0x37	; used in EEPROM loading routines
PASSNUM	equ	0x38	; holds number of password digits
DIALTRY	equ	0x39	; holds number of tries to get dial tone
SOLTIME	equ	0x3A	; holds time solenoid operated
DLYCNTR	equ	0x3B	; counter used by delay routine only
HALFCYCL equ	0x3C	; counter used only in beep routine

;-----------------DEFINES------------------------------------------------------------
; Used in tone routines. Cycle = 8.94uS.
; Counter value = ( ( 1 / wanted freq(Hz) ) / 2 ) / 8.94uS
; Routine takes 3 cycles to count down and check each half cycle, so value used is the
; calculated value divided by 3.

_500Hz	equ	d'37'
_800Hz	equ	d'23'
_1000Hz	equ	d'18'
_1200Hz	equ	d'15'
_2100Hz	equ	d'9'

; used in tone routines, for length of burst. Timer 0 clock = 256 x 8.94uS = 2.288mS.
; timer 0 reg value = 255 - ( wanted period(mS) / 2.288mS ). N.B. Timer 0 counts up!

_500mS	equ	d'36'	
_200mS	equ	d'167'	 
_100mS	equ	d'211'	

ATTEMPTS	equ	d'2'	; number of password attempts allowed
PASSDIGS	equ	d'6'	; number of password digits in a string

;PORT A
D0	equ	0	; DTMF decoder data inputs
D1	equ	1	
D2	equ	2
D3	equ	3
DV	equ	4	; data vaid input. 1 = valid dtmf digit decoded and ready

;PORT B
RING	equ	0	; follows ring cadence, goes low when ring burst is active
PRGM	equ	1	; input tested at pwr up. 1=norm, 0=program pass/ring
IN1	equ	2	; input 1 opto-coupler. 1= off, 0=on
IN2	equ	3	; input 2 opto-coupler. 1= off, 0=on
LOOP	equ	4	; output to turn on solenoid to loop line. 1=loop, 0=idle
SPKR	equ	5	; output to drive speaker. Pin is toggled at audio rate.
OUT1	equ	6	; output 1 drives relay via transistor. 1= on, 0=off
OUT2	equ	7	; output 2 drives relay via transistor. 1= on, 0=off

;***********************************************************************************
;*************************** MAIN START ********************************************
;***********************************************************************************
; Gets here at power up

start	org	0x000
	goto	main		; skip over interupt vector and go to start address
	org	0x004		; interrupt service
	goto	intserv

main	bcf	STATUS,RP0	; ensure we are in bank 0
	call	initpic		; initialise all registers, and turn off interupts

	movlw	d'50'		; 1 second delay to allow hardware to settle
	movwf	DLYCNTR
	call	delay
	call	tonepwr		; announce power up OK 
;------------------------------------------------------------------------------------------
; now test program switch to see if program pins shorted (input low)
; NOT shorted - continue as per normal power up
; SHORTED for less than 5 seconds - DTMF testing
; SHORTED for greater than 5 seconds, but less than 10 seconds - RING training
; SHORTED for greater than 10 seconds - program default password and RING detection hi/lo values 

	clrf	COUNT1
	btfsc	PORTB,PRGM	; test input	
	goto	mainnorm		; not low - normal power up, do not do any program 

mainprgm	movlw	d'50'		; program is low, see how many seconds it is low
	movwf	DLYCNTR
	call	delay		; 1 second delay (50x20mS = 1S)
	incf	COUNT1,f		; add one more second to timer
	btfss	PORTB,PRGM	; test input again	
	goto	mainprgm		; still low, loop again

	movlw	d'10'		; test for > 10 seconds
	subwf	COUNT1,w		; count1 - w
	btfsc	STATUS,C		; c set if > 10
	goto	Default		; program pins shorted > 10 seconds,program default password 		
	movlw	d'5'		; test for > 5 seconds
	subwf	COUNT1,w		; count1 - w
	btfsc	STATUS,C		; c set if > 5
	goto	Train		; shorted < 10, > 5, do RING training
	goto	DTMFtest		; shorted < 5, do DTMF testing

mainnorm	call	loadpass		; get password from EEPROM and put in RAM
	call	loadring		; get ring detect values from EEPROM and put in RAM
	movlw	d'50'		; 1 second delay (50x20mS = 1S)
	movwf	DLYCNTR
	call	delay		; 
	call	home		; ensure on-hook state after normal power up

mainloop	btfsc	PORTB,RING	; see if ring active
	goto	mainloop		; no ring, just wait		
	call	ringchk		; got a ring burst edge, go see if it is valid ring
	btfss	MYFLAGS,RINGOK	; flag set if was valid ring
	goto	mainloop		; not valid ring

;------------------------------------------------------------------------------------
; Gets here after valid ring detected. Loops line and sends answer tone.

answer	bcf	MYFLAGS,RING	; clear flag
	call	solenoid		; loop line, by pulsing phone talk button
	call	toneansw		; send 20 beep tone to indicate answer to caller

;-----------------------------------------------------------------------------------	
; Now wait for a password to be entered. If any digit is not received within 3 seconds
; then the routine exits and goes on-hook to terminate the call.
; If 6 digits are received within the timeout period, but the password is wrong,
; a second attempt is allowed. If this fails the call is terminated.

password	movlw 	ATTEMPTS
	movwf	COUNT1		; number of password attempts

passnew	bcf	MYFLAGS,BADPASS	; clear bad flag
	bcf	MYFLAGS,TIMEOUT	; clear timeout flag
	movlw	PASS1		; password lookup RAM start
	movwf	FSR
	movlw	PASSDIGS		; number of password digits
	movwf	PASSNUM

passagn	call	getdtmf		; wait for a digit, 3Sec timeout. Digit in DTMFDATA
	btfsc	MYFLAGS,TIMEOUT	; see if timed out
	goto	passfin		; timeout, so return
	movf	INDF,w		; get digit from RAM
	subwf	DTMFDATA,w	; see if same as received digit	
	btfss	STATUS,Z		; z flag set if same
	bsf	MYFLAGS,BADPASS	; not same, set got bad password digit flag
	incf	FSR,f		; point to next RAM location
	decfsz	PASSNUM,f	; see if done 6 digits
	goto	passagn		; not yet try one more
	movlw	d'25'		; done all  6, wait 500mS 
	movwf	DLYCNTR
	call	delay
	
passtest	btfsc	MYFLAGS,BADPASS	; did we get a bad password digit this time?
	goto	passmore		; got a bad digit, any more goes?
	call	tonegdps		; send tone to announce good password
	goto	command		; go get command dtmf digits

passmore	call	toneerr		; bad digit so send error tone
	decfsz	COUNT1,f		; see if got one more chance
	goto	passnew		; yes got one more chance at password

passfin	call	tonebye
	call	solenoid		; terminate call
	goto	mainloop		; go back and wait for next call

;--------------------------------------------------------------------------------
; Gets here if a valid password has been decoded. Also gets here after responding
; to a command, and waits for the next command. 
; If no DTMF digit decoded within the timeout period of 180 seconds
; the call is terminated.....no second chances. Safety protection against holding
; the line permanently off-hook.

command	clrf	HASHCNT		; reset hash counter
commnext	call	waitdtmf		; wait for 3 min for a command
	btfsc	MYFLAGS,IDLE	; see if been idle 3 min 
	goto	commfin		; idle exceeded, end call

;----------------------------------------------------------------------------------
; Tests DTMF digit received, starting with a delay between receiving
; digit and decoding command, to give the user time between hearing the transmitted 
; DTMF digits and the tone response.
; HASHCNT counts #s and if 2 # are received in a row then the call is ended. 
; If only 1 # or a # followed by any other digit, just reset HASHCNT and ignore.

commhash	movf	DTMFDATA,w
	sublw	0x0C		; if # key 
	btfss	STATUS,Z
	goto	comm1		; not hash this time

	incf	HASHCNT,f	; got hash this time
	btfsc	HASHCNT,1	; bit 1 will be high if 2 # received in a row
	goto	commfin		; got two in a row, end call
	goto	commnext		; only got one #, wait for next dtmf digit

comm1	movlw	d'40'		; 40x20mS = 800mS
	movwf	DLYCNTR		; to allow user time after enter digit to listen to response
	call	delay

	clrf	HASHCNT		; this digit not #, clear hash counter
	movf	DTMFDATA,w
	sublw	0x01		; if 1 key 
	btfsc	STATUS,Z
	goto	turnon1		; 

comm2	movf	DTMFDATA,w
	sublw	0x02		; if 2 key 
	btfsc	STATUS,Z
	goto	turnoff1		; 

comm3	movf	DTMFDATA,w
	sublw	0x03		; if 3 key 
	btfsc	STATUS,Z
	goto	turnon2		; 

comm4	movf	DTMFDATA,w
	sublw	0x04		; if 4 key 
	btfsc	STATUS,Z
	goto	turnoff2		; 

comm5	movf	DTMFDATA,w
	sublw	0x05		; if 5 key 
	btfsc	STATUS,Z
	goto	testin1		; 

comm6	movf	DTMFDATA,w
	sublw	0x06		; if 6 key 
	btfsc	STATUS,Z
	goto	testin2		; 

commstar	movf	DTMFDATA,w
	sublw	0x0B		; if * key 
	btfsc	STATUS,Z
	goto	newpass		; 

	call	toneerr		; no digit match - send error tone
	goto	commnext		; and try again

;----------------------------------------------------------------------------------
; Perform the command 

turnon1	bsf	PORTB,OUT1	;
	call	toneone
	call	toneon
	goto	commnext

turnoff1	bcf	PORTB,OUT1
	call	toneone
	call	toneoff
	goto	commnext

turnon2	bsf	PORTB,OUT2
	call	tonetwo
	call	toneon
	goto	commnext

turnoff2	bcf	PORTB,OUT2
	call	tonetwo
	call	toneoff
	goto	commnext

testin1	call	toneone
	btfsc	PORTB,IN1	; active low
	goto	testin1l
testin1h	call	toneon	
	goto	commnext
testin1l	call	toneoff
	goto	commnext

testin2	call	tonetwo
	btfsc	PORTB,IN2	; active low 
	goto	testin2l
testin2h	call	toneon	
	goto	commnext
testin2l	call	toneoff
	goto	commnext

newpass	call	setpass		; go and allow user to input new password
	goto	commnext

commfin	call	tonebye		; send user ack to command and warn going off-hook
	call	solenoid		; terminate call
	goto	mainloop		; OK, go back to wait for call


;------------------------------------------------------------------------------
; this routine sets the default password 'to 123456' and the default ring time
; values in the EEPROM

Default	movlw	d'1'		; first default password digit
	movwf	EEDATA		
	movlw	d'0'		; first EEPROM address counter
	movwf	EEADR		
	movlw	d'6'		; number of password elements
	movwf	DEFCNT

Defpsdo	call	eewrite		; write to EEPROM
	incf	EEDATA,f		; next password digit
	incf	EEADR,f		; next password address
	decfsz	DEFCNT,f		; check to see if done all digits
	goto	Defpsdo		; not yet

defrng	clrf	POINTER		; table offset pointer
	movlw	0x10		; first EEPROM address counter
	movwf	EEADR		
	movlw	d'12'		; number of ring values
	movwf	DEFCNT

Defrngdo	movf	POINTER,w	; get table pointer
	call	defrgtbl		; get value from table
	movwf	EEDATA		; ready for write
	call	eewrite		; write to EEPROM	
	incf	POINTER,f	; point to next value in table
	incf	EEADR,f		; next EEPROM address
	decfsz	DEFCNT,f		; check to see if done all values
	goto	Defrngdo		; not yet
	call	tonepass		; announce defaults written

Defrngwt	goto	Defrngwt		; wait till power off

defrgtbl	addwf	PCL,f		; add offset in w reg to program counter
	retlw	d'30'		; ring on L - cycle 1 (30x20mS=600mS)
	retlw	d'50'		; H (50x20mS=1000mS)
	retlw	d'0'		; ring off L (0x20mS=0mS)
	retlw	d'15'		; H (15x20mS=300mS)
	retlw	d'0'		; ring off L
	retlw	d'15'		; H	
	retlw	d'30'		; ring on L - cycle 2	
	retlw	d'50'		; H
	retlw	d'0'		; ring off L
	retlw	d'15'		; H
	retlw	d'0'		; ring off L
	retlw	d'15'		; H

;----------------------------------------------------------------------------------
; Diagnostic routine to sample incoming Ring signal cadence and store results in EEPROM
; Ring is sampled for 6 seconds, and each second count is converted to 2 counts indicating
; the upper(H) and lower(L) limits. Counts are incremented each 20mS, so maximum count is
; 1sec/20mS = 50. Typical Ring cadence is 400mS on, 200mS off, 400mS on and 2000mS off.
; This means that the typical on time count is 800/20 = 40. Due to circuit rise and fall
; times and possible exchange variations, the main Ring Detect routine is given a range between
; L and H to allow a valid ring cadence to be detected. This complex system is used so that
; random loud sounds do not falsely trigger the Ring detection operation.
; Routine waits for power off after storing 6 seconds of high and low limits.

Train	call	tonetwo		; signal that reached ring training
	movlw	d'25'		; 500mS delay
	movwf	DLYCNTR
	call	delay

Trainwt	btfsc	PORTB,PRGM	; wait now till pins shorted to start
	goto	Trainwt		; not yet shorted
	call	tonetwo		

Trainsth	btfss	PORTB,RING	; look for a ring leading edge (high to low)
	goto	Trainsth		; is low (ring is active) so wait some more
Trainstl	btfsc	PORTB,RING	; ring now high (silence) wait for low	
	goto	Trainstl		; still silence

	clrf	COUNT1		; OK got edge, we can start, clear good counter
	movlw	d'50'		; 50 x 20mS = 1 second
	movwf	COUNT2		; counts 1 second periods

Trainlp	movlw	d'1'		; 20mS delay
	movwf	DLYCNTR
	call	delay
	decfsz	COUNT2,f		; have we done 1 second
	goto	Traintst		; not yet, see what state ring is
	movlw	d'20'		; test against 20, 20x20mS=400mS (ideally=800mS)
	subwf	COUNT1,w		; count1 - w(20), result in w, so count1 not lost
	btfss	STATUS,C		; c set if count1 > 20
	goto	Trainsth		; no good, count1 < 20, try again
	goto	Transtrt		; good go and store 6 seconds of ring
	
Traintst btfsc	PORTB,RING	; test ring
	goto	Trainlp		; not active (is high)
	incf	COUNT1,f		; ring active, add to good count
	goto	Trainlp
;------------------------------------------------------------------------
; gets here after detection of at least 400mS of ring during a 1 second period
; ring should be now in 2 second silence period
; now wait till ring is active (low) again, and then store 6 seconds

Transtrt	movlw	d'6'		; 6 seconds to test ring
	movwf	COUNT3
	movlw	d'25'		; 500mS delay to avoid false edges as ring dies
	movwf	DLYCNTR
	call	delay
Tranfind	btfsc	PORTB,RING	; look for a ring leading edge (low)
	goto	Tranfind
	movlw	RING1L		; got leadibg edge, point to start of temp storage
	movwf	FSR		; indirect pointer

Tranagn	clrf	COUNT1		; clear good counter
	movlw	d'50'		; 50 x 20mS = 1 second
	movwf	COUNT2

Tranloop	movlw	d'1'		; 20mS delay
	movwf	DLYCNTR
	call	delay
	decfsz	COUNT2,f		; have we done 1 second
	goto	Trantest		; not yet see what state ring is

	movf	COUNT1,w		; done 1 second, get good count
	movwf	INDF		; store in RAM
	incf	FSR,f		; next RAM position
	decfsz	COUNT3,f		; see if done 6 seconds
	goto	Tranagn		; not done 6 seconds yet
	goto	Transtor		; yes done all 6, go and store in EEPROM

Trantest btfsc	PORTB,RING	; test ring
	goto	Tranloop		; not active (is high)
	incf	COUNT1,f		; ring active, add to good count
	goto	Tranloop

;-----------------------------------------------------------------------------------
; now convert 6 values temporarily in RAM to High and Low limits and store in EEPROM
		
Transtor	movlw	RING1L		; start of RAM
	movwf	FSR		; indirect register
	movlw	d'6'		; number of ring values to be converted from RAM into EEPROM
	movwf	LOADCNT
	movlw	0x10	
	movwf	EEADR		; first eeprom address = 0x10 = dec 16

TranstrL	movf	INDF,w		; get value
	movwf	COUNT1		; store original value here
	movwf	COUNT2		; here for dividing

; check if value is very low (less than 10), and so write default low limit values
; this is because if value is 0000 0011 for example, when we rotate to divide by 4
; the value will become 0000 0000. When we add and subtract the divided value,
; both the high and low limits will be the same and equal to the original value.
; this would mean that if the ring count was not exactly 0000 0011 it would fail.

	movlw	d'10'		; 
	subwf	COUNT2,w		; count2 - w
	btfss	STATUS,C		; C flag set if COUNT2 > W
	goto	Translow		; yes value is very low

	rrf	COUNT2,f		; divide ring counter value by 4, to get 25%
	rrf	COUNT2,f	
	movf	COUNT2,w		; get divided value
	andlw	0x3f		; clear top 2 bits
	subwf	COUNT1,w		; subtract divided value from original value (75%)
	movwf	EEDATA
	call	eewrite		; write low limit value

	incf	EEADR,f		; next EEPROM position
	movf	COUNT2,w		; get divided value
	andlw	0x3f		; clear top 2 bits
	addwf	COUNT1,w		; add divided value to original value (125%)
	movwf	EEDATA
	call	eewrite		; write high limit value
	goto	Transnxt

Translow	movlw	d'0'		; low limit
	movwf	EEDATA
	call	eewrite
	incf	EEADR,f		; next EEPROM position
	movlw	d'15'		; high limit
	movwf	EEDATA
	call	eewrite

Transnxt	incf	EEADR,f		; next EEPROM address
	incf	FSR,f		; next ram position
	decfsz	LOADCNT,f	; see if done all values
	goto	TranstrL		; not yet	

Transend	call	toneone		; signal done storing
	goto	Transend		; wait till power off

;-------------------------------------------------------------------------------------
; diagnostic routine to test incoming DTMF digits.
; Sounds number of beeps equal to DTMF digit. 
; digit 1 = 1 beep etc. Note. 0 = 10, # = 11 and * = 12 beeps

DTMFtest	call	toneone		; indicate at dtmf test
DTMFtlp	btfss	PORTA,DV		; see if Data valid is high (digit ready)
	goto	DTMFtlp		; no, try again

DTMFtyes	btfsc	PORTA,DV		; wait for data valid to go low again
	goto	DTMFtyes		; still high, wait
	movf	PORTA,w		; data valid low, get DTMF digit
	andlw	0x0F		; mask off top 4 bits
	movwf	DTMFDATA	
	movlw	d'15'		; 300mS delay between key release and beeps
	movwf	DLYCNTR
	call	delay
DTMFtnxt	call	tonebeep
	movlw	d'10'		; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay
	decfsz	DTMFDATA,f
	goto	DTMFtnxt		; send next beep	
	goto	DTMFtlp		; finished this digit, go get next one


;*********************************************************************************
;*************************** MAIN END ********************************************
;*********************************************************************************


;*************************** CALLS ***********************************************


;------------------------------START RINGCHK---------------------------------------
; Gets here after the leading edge of a burst of negative going ring is detected. 
; Burst is = ON(LOW) 400mS : OFF(HIGH) 200mS :  ON(LOW) 400mS : OFF(HIGH) 2000mS.
; Tests 6 x 1 second counts with those stored in RAM.
; Any time that a 1 second count is not within the lower and upper limits
; in RAM it returns MYFLAGS RINGOK clear.
; If after 6 seconds all the counts are within the limits, it returns MYFLAGS RINGOK set

ringchk	movlw	RING1L		; put start of RAM stored limit values into FSR
	movwf	FSR
	movlw	d'6'		; length of test
	movwf	COUNT3		; 6 seconds 
ringagn	movlw	d'50'		; each test period
	movwf	COUNT2		; 50 x 20mS = 1 second
	clrf	COUNT1		; reset good ring cntr (20mS samples when ring is active low)
ringloop	movlw	d'1'		; 20mS delay
	movwf	DLYCNTR
	call	delay
	decfsz	COUNT2,f		; see if done this second
	goto	ringtest		; not yet so test ring input again
	goto	ringcomp		; finished this second, go compare count with stored values

ringtest	btfsc	PORTB,RING	; test ring
	goto	ringloop		; not ring, loop around
	incf	COUNT1,f		; add 1 to good sample ring count
	goto	ringloop		; back again

ringcomp	movf	INDF,w		; get LOW limit value from RAM
	subwf	COUNT1,w		; count1 - W
	btfss	STATUS,C		; C set if count1 > W (good)
	goto	ringbad		; c is clear (RAM(W) > COUNT1), COUNT1 is too low
	incf	FSR,f		; point to next RAM value (HIGH limit value)

	movf	INDF,w		; get HIGH limit value from RAM
	subwf	COUNT1,w		; high this time, count1 - W
	btfsc	STATUS,C		; C clear if count1 < W (good)
	goto	ringbad		; c is set (RAM(W) < COUNT1), COUNT1 is too high
	incf	FSR,f		; point to next RAM value (LOW) for next 1 second test

ringend	decfsz	COUNT3,f		; see if done all 6 seconds
	goto	ringagn
	bsf	MYFLAGS,RINGOK	; got all 6 good comparisons, go home good ring
	return

ringbad	bcf	MYFLAGS,RINGOK	; ring not detected, go home with flag cleared
	return	

;-------------------------------END RINGCHK-----------------------------------------


;------------------------------START TONEPWR--------------------------------------
; Generates a single tone burst of 1000Hz for 500mS at power up after initialisation.

tonepwr 	movlw	_500mS	; 500mS beep
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
	call	beep
	return

;------------------------------END TONEPWR-----------------------------------------

;------------------------------START TONEGDPS--------------------------------------
; Generates a 3 step rising tone to indicate password accepted.

tonegdps	movlw	_200mS	; 200mS beep
	movwf	TONEDUR
	movlw	_800Hz	; 800Hz
	movwf	TONEFREQ
	call	beep
	movlw	d'10'	; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay

	movlw	_200mS	; 200mS beep
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
	call	beep
	movlw	d'10'	; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay

	movlw	_200mS	; 200mS beep
	movwf	TONEDUR
	movlw	_1200Hz	; 1200Hz
	movwf	TONEFREQ
	call	beep
	return

;------------------------------END TONEGDPS-----------------------------------------


;------------------------------START TONEBYE--------------------------------------
; Generates a 3 step falling tone to indicate ending call.

tonebye	movlw	_500mS	; 500mS beep
	movwf	TONEDUR
	movlw	_2100Hz	; 2100Hz
	movwf	TONEFREQ
	call	beep
	movlw	d'10'	; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay

	movlw	_500mS	; 500mS beep
	movwf	TONEDUR
	movlw	_1200Hz	; 1200Hz
	movwf	TONEFREQ
	call	beep
	movlw	d'10'	; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay

	movlw	_500mS	; 500mS beep
	movwf	TONEDUR
	movlw	_800Hz	; 800Hz
	movwf	TONEFREQ
	call	beep
	movlw	d'10'	; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay

	movlw	_500mS	; 500mS beep
	movwf	TONEDUR
	movlw	_500Hz	; 500Hz
	movwf	TONEFREQ
	call	beep
	return

;------------------------------END TONEBYE-----------------------------------------


;------------------------------START TONEON--------------------------------------
; Generates 2 short rising frequency beeps to indicate ON.

toneon	movlw	_200mS	; 200mS beep
	movwf	TONEDUR
	movlw	_1200Hz	; 1200Hz
	movwf	TONEFREQ
	call	beep

	movlw	d'5'	; 100mS delay between each beep
	movwf	DLYCNTR
	call	delay
	
	movlw	_200mS	; 200mS beep
	movwf	TONEDUR
	movlw	_2100Hz	; 2100Hz
	movwf	TONEFREQ
	call	beep

	return

;------------------------------END TONEON-----------------------------------------

;------------------------------START TONEBEEP--------------------------------------
; Generates 1 short beep.

tonebeep	movlw	_100mS	; 100mS beep
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
	call	beep

	return

;------------------------------END TONEONE-----------------------------------------

;------------------------------START TONEONE--------------------------------------
; Generates 1 short beep to indicate ONE.

toneone	movlw	_100mS	; 100mS beep
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
	call	beep

	movlw	d'25'	; 500mS delay 
	movwf	DLYCNTR
	call	delay


	return

;------------------------------END TONEONE-----------------------------------------

;------------------------------START TONETWO--------------------------------------
; Generates 2 short beeps to indicate TWO.

tonetwo	movlw	_100mS	; 100mS beep
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
	call	beep

	movlw	d'5'	; 100mS delay between each beep
	movwf	DLYCNTR
	call	delay
	
	movlw	_100mS	; 100mS beep
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
	call	beep

	movlw	d'25'	; 500mS delay 
	movwf	DLYCNTR
	call	delay

	return

;------------------------------END TONETWO-----------------------------------------

;------------------------------START TONEOFF--------------------------------------
; Generates low frequency beep to indicate OFF

toneoff	movlw	_500mS	; 500mS beep
	movwf	TONEDUR
	movlw	_800Hz	; 800Hz
	movwf	TONEFREQ
	call	beep

	return

;------------------------------END TONEOFF-----------------------------------------

;------------------------------START TONEPASS--------------------------------------
; Generates a series of 6 beeps of 1000Hz when the default password is programmed.

tonepass	movlw	d'6'	; 6 beeps
	movwf	COUNT1
	movlw	_200mS	; 200mS long
	movwf	TONEDUR
	movlw	_1000Hz	; 1000Hz
	movwf	TONEFREQ
tonepslp	call	beep
	movlw	d'5'	; 100mS delay between each beep
	movwf	DLYCNTR
	call	delay
	decfsz	COUNT1,f	; check if done 6
	goto	tonepslp	; not yet

	return

;------------------------------END TONEPASS-----------------------------------------

;------------------------------START TONEERR--------------------------------------
; Generates a burst of 4 500Hz pulses to signal that the key was not recognised.

toneerr	movlw 	d'4'
	movwf	COUNT1
	movlw	_500mS	; 500mS long
	movwf	TONEDUR
	movlw	_500Hz	; 500Hz
	movwf	TONEFREQ
toneerlp	call	beep
	movlw	d'10'	; 200mS delay between each beep
	movwf	DLYCNTR
	call	delay
	decfsz	COUNT1,f
	goto	toneerlp

	return

;------------------------------END TONEERR----------------------------------------

;------------------------------START TONEANSW--------------------------------------
; Generates a series of 2100Hz pulses when ring is detected 
; and the line is looped (off-hook)

toneansw	movlw	d'20'	; 20 beeps
	movwf	COUNT1
	movlw	_100mS	; 100mS long
	movwf	TONEDUR
	movlw	_2100Hz	; 2100Hz
	movwf	TONEFREQ

tonealp	call	beep
	movlw	d'5'	; 100mS delay between each beep
	movwf	DLYCNTR
	call	delay
	decfsz	COUNT1,f	; check if done all
	goto	tonealp	; not yet

	return
;------------------------------END TONEANSW-----------------------------------------


;------------------------------START BEEP--------------------------------------
; Generates a burst of tone out of PORT B SPKR, by toggling port B pin.
; Instruction cycle = 8.94uS. During each output half cycle the value in TONEFREQ 
; is decremented each instruction cycle. When it reaches zero, the next half cycle 
; is started. At the end of a full cycle the value in TONEDUR is decremented.
; When it reaches zero the routine ends and the output pin is returned to zero.
; TONEFREQ = tone half cycle period/8.94uS. Must be loaded before calling this routine.
; Example for 1000Hz: TONEFREQ = ( (1/1000Hz)/2 ) / 8.94uS = 56 cycles.
; Routine takes 3 cycles to count down and check each half cycle, so value used is the
; calculated value divided by 3.
; Timer 0 maximum count is 585mS. TONEDUR must be pre loaded first with period value.
; Counter clocks up from 00 to FF and flag T0IF is set when overflows from FF to 00.

beep	bsf	STATUS,RP0	; go to register bank 1
	movlw	b'10000111'	; no portb pullups
	movwf	OPTION_REG	; Tmr0 int clk,  prescaler 256
	bcf	STATUS,RP0	; go back to register bank 0

	movf	TONEDUR,w	; load timer 0 with tone duration
	movwf	TMR0
	bcf	INTCON,T0IF	; clear timer 0 flag

beeploop	bcf	PORTB,SPKR	; output low
	movf	TONEFREQ,w	; get tone half cycle value 
	movwf	HALFCYCL
beeplow	decfsz	HALFCYCL,f	; see if done down count of low half cycle
	goto	beeplow		; not done yet

	bsf	PORTB,SPKR	; output high
	movf	TONEFREQ,w	; get tone half cycle value 
	movwf	HALFCYCL
beephigh	decfsz	HALFCYCL,f	; see if done down count of high half cycle
	goto	beephigh		; not done yet

	btfss	INTCON,T0IF	; see if timer 0 counted down to zero 
	goto	beeploop		; timer 0 not counted down yet, do next cycle

 	bcf	PORTB,SPKR	; done, so set output pin low

	return

;------------------------------END BEEP-----------------------------------------

;-------------------------------START GETDTMF-------------------------------------
; waits for 3 seconds for a DTMF digit. If one is decoded within 3 seconds returns
; digit in lower nibble of DTMFDATA. If none found within 3 seconds returns with
; TIMEOUT set in MYFLAGS

getdtmf	movlw	d'150'		; wait for maximum 150x20mS = 3S
	movwf	W4CNTR1

getdno	btfsc	PORTA,DV		; see if Data valid is high (digit ready)
	goto	getdyes		; yes got digit
	movlw	d'1'		; no digit yet, wait another 20mS
	movwf	DLYCNTR
	call	delay
	decfsz	W4CNTR1,f	; see if timeout yet
	goto	getdno		; no timeout yet, try again
	bsf	MYFLAGS,TIMEOUT	; timed out return
	return	

getdyes	movf	PORTA,w		; get DTMF digit
	andlw	0x0F		; mask off top 4 bits
	movwf	DTMFDATA	

getdend	btfsc	PORTA,DV		; wait for data valid to go low
	goto	getdend		; still high wait again
	bcf	MYFLAGS,TIMEOUT	; ensure timed out flag is clear
	return
;-------------------------------END GETDTMF------------------------------------


;-------------------------------START WAITDTMF-------------------------------------
; waits for 3 minutes for a DTMF digit. 
; If one is decoded within 3 minutes returns digit in lower nibble of DTMFDATA. 
; If none found within 3 minutes, returns with IDLE set in MYFLAGS
; Each minute a reminder tone is sent

waitdtmf	movlw	d'50'		; 50 x 20mS = 1sec
	movwf	W4CNTR1
	movlw	d'179'		; 180 x 1S = 3 min (179 wait secs + 1 second to do 2 beeps)
	movwf	W4CNTR2

waitdno	btfsc	PORTA,DV		; see if Data valid is high (digit ready)
	goto	waitdyes		; yes got digit
	movlw	d'1'		; no digit yet, wait another 20mS
	movwf	DLYCNTR
	call	delay
	decfsz	W4CNTR1,f	; see if 1 second timeout yet
	goto	waitdno		; no timeout yet, try again
	movlw	d'50'		; 1 second timeout, reload 1 second counter
	movwf	W4CNTR1	
	decfsz	W4CNTR2,f	; see if 180 second timeout
	goto	waitd120		; not 180 second timeout yet, see if time to send reminder beep
	bsf	MYFLAGS,IDLE	; idle time exceeded, return
	return	

waitd120	movlw	d'120'
	subwf	W4CNTR2,w	; W4CNTR2 - w
	btfss	STATUS,Z	
	goto	waitd60		; not 120 second time yet, see if 60	
	call	tonepwr		; 120 seconds up, send beep
	goto	waitdno		; loop again
waitd60	movlw	d'60'
	subwf	W4CNTR2,w	; W4CNTR2 - w
	btfss	STATUS,Z	
	goto	waitdno		; not 60 second time yet, loop again
	call	tonepwr		; 60 seconds up, send beep
	goto	waitdno	

waitdyes	movf	PORTA,w		; get DTMF digit
	andlw	0x0F		; mask off top 4 bits
	movwf	DTMFDATA	

waitdend	btfsc	PORTA,DV		; wait for data valid to go low
	goto	waitdend		; still high wait again
	bcf	MYFLAGS,IDLE	; ensure idle out flag is clear
	return
;-------------------------------END WAITDTMF------------------------------------


;------------------------------START HOME---------------------------------------
; The talk button is pressed by energising the solenoid and then listening
; for dialtone, which indicates off-hook. 
; When off-hook state detected, then pulses phone talk button again to go on-hook and
; returns.
; Dialtone is indicated by continuous low level on RING input, and must be 
; at least 2 seconds continous to be considered good.
; If dail tone is not found after 4 x 5 second spaced attempts, it waits for 10 minutes 
; and tries again.

home	movlw	d'4'		; number of times allowed to try and get dial tone
	movwf	DIALTRY		; in quick succession
homenext	call	solenoid	

;-------------------------------------------------------------------------------
; now listen for dial tone, permanent low on ring input

homelstn	movlw	d'100'		; wait 2 second for dial tone to activate
	movwf	DLYCNTR
	call	delay
	movlw	d'100'		; sample input for 2 seconds (100x20mS)
	movwf	W4CNTR1
	clrf	W4CNTR2		; counts good samples of low ring line
hometst	btfsc	PORTB,RING	; is ring input low?	
	goto	homechk		; this time ring is high, test if done 2 seconds
	incf	W4CNTR2,f	; ring still low, add to good count
homechk	movlw	d'1'		; 20mS delay
	movwf	DLYCNTR
	call	delay
	decfsz	W4CNTR1,f	; have we done 2 seconds yet?
	goto	hometst		; not done 2 seconds yet
	movf	W4CNTR2,w	; done 2 seconds was it low for a long time?
	sublw	d'80'		; test count value, good > 80 out of 100
	btfss	STATUS,C		; c flag set if COUNT2 < 80 (bad)
	goto	homedone		; c is clear, COUNT2 > 80 (good)

	movlw	d'250'		; didn't get dial tone 
	movwf	DLYCNTR		; wait 5 seconds and try again
	call	delay
	decfsz	DIALTRY,f	; have we done all 4 quick attempts
	goto	homenext		; not yet, keep going
	goto	homewait

homedone	call	solenoid		; found dial tone, now operate again to go on-hook
	return	

;------------------------------------------------------------------------------------
; had 4 attempts spaced by 5 seconds and no dial tone found. Maybe line fault or
; power off to phone. Try again after 10 minutes.

homewait	movlw	d'250'		; yes done 4 attempts, wait 10 minutes, try again
	movwf	W4CNTR1		; 250x20mS = 5S
	movlw	d'120'		; 120x5S = 600S = 10min
	movwf	W4CNTR2

homewlp	movlw	d'1'		; 20mS delay
	movwf	DLYCNTR
	call	delay
	decfsz	W4CNTR1,f	; check 5 second counter
	goto	homewlp		; not done 5 seconds yet
	movlw	d'250'		; finished 5 seconds, reload counter
	movwf	W4CNTR1		
	decfsz	W4CNTR2,f	; check if done 10 minutes
	goto	homewlp		; not yet
	goto	home		; finished 10 minutes, try for dial tone again
	
;-------------------------------END HOME-----------------------------------------


;------------------------------START SOLENOID---------------------------------------
; Operates the telephone talk button by energising the solenoid for 1 second. 

solenoid	movlw	d'50'		; 1 second (50x20mS)
	movwf	SOLTIME
	bsf	PORTB,LOOP	; turn on solenoid
	
solewait	movlw	d'1'		; wait 20mS 
	movwf	DLYCNTR		
	call	delay
	decfsz	SOLTIME,f	; see if done 1 second
	goto	solewait		; no not yet
	bcf	PORTB,LOOP	; de-energise solenoid
	return			; finished

;-------------------------------END SOLENOID----------------------------------------


;-----------------------------START SETPASS--------------------------------------
; Lets user enter new passwword
; If a digit is not received within 3 seconds, the process is aborted an error tone
; is sent and the old password is restored. 
; If all six digits are received OK, a good password tone is sent and the new
; password is written to EEPROM.

setpass	call	tonepass		; announce ready to change password
	bcf	MYFLAGS,TIMEOUT	; clear timeout flag
	movlw	PASS1		; password lookup RAM start
	movwf	FSR
	movlw	PASSDIGS		; number of password digits
	movwf	PASSNUM

setpsnxt	call	getdtmf		; wait for a digit, 3 sec timeout. Digit in DTMFDATA
	btfsc	MYFLAGS,TIMEOUT	; see if timed out
	goto	setpsbad		; timeout do not store
	movf	DTMFDATA,w	; get digit
	movwf	INDF		; put digit into RAM
	incf	FSR,f		; point to next RAM location
	decfsz	PASSNUM,f	; see if done 6 digits
	goto	setpsnxt		; not yet try one more
	movlw	d'25'		; done all  6, wait 500mS 
	movwf	DLYCNTR
	call	delay

	movlw	PASS1		; password lookup RAM start
	movwf	FSR	
	movlw	d'0'		; first EEPROM address 
	movwf	EEADR		
	movlw	PASSDIGS		; number of password elements
	movwf	PASSNUM

setpswr	movf	INDF,w		; get digit from RAM
	movwf	EEDATA		; put into EEPROM data reg
	call	eewrite		; write to EEPROM
	incf	EEADR,f		; next EEPROM password address
	incf	FSR,f		; next RAM address
	decfsz	PASSNUM,f	; check to see if done 6 digits
	goto	setpswr		; not yet
	call	tonepass		; sound good password stored
	return

setpsbad	call	toneerr		; send error tone to user
	call	loadpass		; restore old password
	return
	
;------------------------------END SETPASS-----------------------------------------


;------------------------------START LOADRING--------------------------------------
; This routine loads the 12 RING time values from EEPROM starting at address 0x10
; and writing them to RAM starting at location RING1. This is done to allow quick
; password checking without having to read from the slow EEPROM.

loadring	movlw	RING1L	; start of RAM
	movwf	FSR	; indirect register
	movlw	d'12'	; number of ring values
	movwf	LOADCNT
	movlw	0x10
	movwf	EEADR	; first eeprom address = 16 (10 HEX)

loadrglp call	eeread	; digit goes into EEDATA
	movf	EEDATA,w
	movwf	INDF	; write to RAM address pointed to by FSR
	incf	EEADR,f	; next EEPROM address
	incf	FSR,f	; next RAM address
	decfsz	LOADCNT,f; check if done all 6 values
	goto	loadrglp ; not yet

	return

;-------------------------------END LOADRING---------------------------------------

;------------------------------START LOADPASS--------------------------------------
; This routine loads the 6 password digits from EEPROM starting at address 0
; and writing them to RAM starting at location PASS1L. This is done to allow quick
; password checking without having to read from the slow EEPROM.

loadpass	movlw	PASS1	; start of RAM
	movwf	FSR	; indirect register
	movlw	d'6'	; number of password digits
	movwf	LOADCNT
	clrf	EEADR	; first eeprom address = 0

loadloop call	eeread	; digit goes into EEDATA
	movf	EEDATA,w
	movwf	INDF	; write to RAM address pointed to by FSR
	incf	EEADR,f	; next EEPROM address
	incf	FSR,f	; next RAM address
	decfsz	LOADCNT,f; check if done all 6 digits
	goto	loadloop ; not yet

	return

;-------------------------------END LOADPASS---------------------------------------


;------------------------------START EEWRITE---------------------------------------
; Writes single byte to EEPROM.
; data must be in EEDATA, address of EEPROM must be in EEADR

eewrite	bsf	STATUS,RP0	; bank 1
	clrf	EECON1		; reset all bits to start
	bsf	EECON1,WREN	; enable writes to eeprom
	movlw	0x55		; protection code to prevent false writes
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf	EECON1,WR	; write data

eewait	btfss	EECON1,EEIF	; flag = 1 when done
	goto	eewait

	bcf	STATUS,RP0	; back to bank 0
	return

;-------------------------------END EEWRITE-----------------------------------------


;------------------------------START EEREAD---------------------------------------
; Reads a single byte from EEPROM.
; data goes into EEDATA. Address of EEPROM must be in EEADR first

eeread	bsf	STATUS,RP0	; bank 1
	clrf	EECON1		; clear all bits first
	bsf	EECON1,RD	; read from eeprom@eeadr into eedata
	bcf	STATUS,RP0	; back to bank 0
	
	return

;-------------------------------END EEREAD-----------------------------------------


;------------------------------START DELAY------------------------------------------
; delay = value in DLYCNTR x 20mS. DLYCNTR must be loaded before calling this routine
; PIC external clock = 447KHz. Internal cycle = 447KHz/4 = 111.8KHz = 8.94uS.
; Tmer 0 pre-scaler = 16. 20mS = 8.94uS x 16 x 140. 139 to take into account overhead.
; Timer 0 counts up and is loaded with 255 - 139 (116) each time it overflows 
; from FF to 00.

delay	bsf	STATUS,RP0	; go to register bank 1
	movlw	b'10000011'	; no portb pullups
	movwf	OPTION_REG	; Tmr0 int clk,  prescaler 16
	bcf	STATUS,RP0	; go back to register bank 0

delaylp1	bcf	INTCON,T0IF	; clear timer 0 flag
	movlw	d'116'		; pre-load timer 0
	movwf	TMR0
delaylp2	btfss	INTCON,T0IF	; see if timer 0 counted down to zero
	goto	delaylp2		; timer 0 not counted down yet
	decfsz	DLYCNTR,f	; timer 0 done, see if done the n x 20mS delay
	goto	delaylp1		; not yet, loop around		
 	return			; finished n x 20mS

;-------------------------------END DELAY-------------------------------------------


;--------------------------INTERUPT START-------------------------------------------
; this interrupt routine is not used, but code included just in case something goes 
; wrong

intserv	movwf	W_TEMP		; save w reg
	movf	STATUS,w		; move status reg into w reg
	bcf	STATUS,RP0	; ensure file register set to bank 0
	movwf	ST_TEMP		; save status reg
	bcf	INTCON,GIE	; disable global interupts 

	movf	ST_TEMP,w	; get status
	movwf 	STATUS		; restore status
	swapf	W_TEMP,f	
	swapf	W_TEMP,w		; restore w reg

	retfie			; return to main program

;--------------------------INTERUPT END--------------------------------------------


;--------------------------START INITPIC----------------------------------------------
; initialisation routine, setup defaults for PIC hardware and defined registers
;

initpic	bcf	INTCON,GIE	;disable global interupts
	bcf	STATUS,RP0	;go to register bank 0
	clrf	STATUS
	movlw	b'00000000'	; disable all interupts
	movwf	INTCON		; 

	bsf	STATUS,RP0	; go to register bank 1
	movlw	b'10000011'	; no portb pullups
	movwf	OPTION_REG	; Tmr0 int clk,  prescaler 16

	movlw	b'00011111'	; 
	movwf	TRISA		; set port A I/O
	movlw	b'00001111'	; 
	movwf	TRISB		; set port B I/O

	clrf	EECON1
	bcf	STATUS,RP0	; go back to register bank 0

	clrf 	PORTB
	clrf	PORTA
	clrf	MYFLAGS

	return			; finished

;-----------------------------END INITPIC------------------------------------------

	end			; PHEW! no more
