
// Chip: PIC16F18146
// This code does not require a pro license to compile; -O2 optimisation is good enough.

/*
 * At start-up, A/B change GPS baud rate
 * A short press: cycle colour scheme
 * B short press: change second 'hand'
 * A long press: time zone offset mode
 * B long press: brightness adjustment mode
 * In time zone offset mode:
 * * A short press: -15 minutes
 * * B short press: +15 minutes
 * * A long press: return to clock
 * * B long press: toggle 1 hour offset
 * In brightness adjustment mode:
 * * A short press: dimmer
 * * B short press: brighter
 * * A long press: return to clock
 * * B long press: adjust LDR sensitivity
 */

//#define DEBUGGING // Disables the initial LED test mode to make debugging easier.
//#define INTERRUPT_BASED_PWM // Enables the alternative (original) PWM method. It works but flickers at very low brightnesses due to unpredictable interrupt delays.

#include "config.h"
#include "uart.h"
#include "gpsserial.h"
#include <xc.h>
#include <string.h>

typedef struct { unsigned char A, B, C, D; } port;
typedef struct { unsigned char hour, min, sec, subsec; } time;
typedef enum { black = 0, red = 1, blue = 2, mauve = 3, green = 4, yellow = 5, cyan = 6, white = 7 } colour;
typedef enum { matching_subsec = 0, white_subsec = 1, no_subsec = 2, no_sec = 3 } second_hand_config;
typedef enum { CA, CC } RGB_LED_type;
typedef enum { ShowTime, AdjustTZ, AdjustLDRMaxBrightness, AdjustLDRMinBrightness } mode;
typedef struct { colour hour, min, sec, subsec; } scheme;

RGB_LED_type LED_type;
unsigned char cur_state = 0, update_state = 0, which_used = 0;
port anode_states[2][8];
port cathode_states[2][8];
unsigned char brightness = 128, b1down = 0, b2down = 0, b1wasdown, b2wasdown, have_am_pm = 0;
unsigned short button_press_counter, button_release_counter, maxb1press, maxb2press, b1shortpress, b1longpress, b2shortpress, b2longpress;
static const unsigned short baud_rates[] = { BRG_4800, BRG_9600, BRG_19200, BRG_38400, BRG_57600, BRG_115200 };
mode cur_mode = ShowTime;
__eeprom unsigned char which_colour_scheme = 0, which_baud_rate = 5;
__eeprom second_hand_config second_hand_cfg = matching_subsec;
__eeprom signed short tzoffset = 10*60;
__eeprom unsigned char dst = 0;
__eeprom unsigned char ldr_min_brightness = 4; // 0 to 59
__eeprom unsigned char ldr_max_brightness = 30; // 0 to 59
extern unsigned char gps_fix, gps_numsats, gps_detected, gps_fixgood, gps_datetimelatlongood;

#define PIN0 (1<<0)
#define PIN1 (1<<1)
#define PIN2 (1<<2)
#define PIN3 (1<<3)
#define PIN4 (1<<4)
#define PIN5 (1<<5)
#define PIN6 (1<<6)
#define PIN7 (1<<7)

#define LED_AM 60
#define LED_PM 61

#define DEBOUNCE_SHORT 4
#define DEBOUNCE_LONG 30

// Figures out which anode and cathode to drive to light an individual LED numbered 0-59 (around the face) or 60-61 (AM/PM).
void get_led_mask(unsigned char LEDnum, colour LEDc, port* pAnode, port* pCathode) {
  if( LED_type == CC ) {
      port* temp = pAnode;
      pAnode = pCathode;
      pCathode = temp;
  }
  pAnode->A = 0;
  pAnode->B = 0;
  pAnode->C = 0;
  pCathode->A = 0;
  pCathode->B = 0;
  pCathode->C = 0;
  if( LEDnum == LED_AM ) {
    pAnode->C = PIN6;
    pCathode->C = PIN4;
  } else if( LEDnum == LED_PM ) {
    pAnode->C = PIN6;
    pCathode->C = PIN5;
  } else {
    unsigned char a = LEDnum>>2;
    unsigned char c;

    if( a < 8 )
      pAnode->C = (unsigned char)(1<<a);
    else if( a < 12 )
      pAnode->B = (unsigned char)(1<<(a-4));
    else
      pAnode->A = (unsigned char)(1<<(a-12));

    for( c = 0; c < 3; ++c ) {
      if( LEDc & 1 ) {
        unsigned char b = (unsigned char)((LEDnum&3) + ((LEDnum&3)<<1) + c + a + 1);

        while( b > 14 )
          b -= 15;

        if( b < 8 )
          pCathode->C |= (unsigned char)(1<<b);
        else if( b < 12 )
          pCathode->B |= (unsigned char)(1<<(b-4));
        else
          pCathode->A |= (unsigned char)(1<<(b-12));
      }
      LEDc = LEDc >> 1;
    }
  }
}

// Called every 1/61st of a second. Debug code to make clock work without a time source is commented out.
void advance_time(time* t) {
    if( ++t->subsec == 61 ) {
        t->subsec = 0;
/*
        if( ++t->sec == 60 ) {
            t->sec = 0;
            if( ++t->min == 60 ) {
                t->min = 0;
                if( ++t->hour == 13 )
                    t->hour = 1;
            }
        }
*/
    }
}

// Figures out the combinations of anodes & cathodes to drive in sequence to display a particular time on the clock give the colour scheme and settings.
void get_led_masks_from_time(port* pAnodes, port* pCathodes, time* pTim, scheme* pSch, second_hand_config cfg) {
  unsigned char hour_hand_pos = pTim->hour, subsec_hand_pos = pTim->sec + pTim->subsec;

  while( hour_hand_pos > 11 )
    hour_hand_pos -= 12;
  while( subsec_hand_pos >= 60 )
    subsec_hand_pos -= 60;
  hour_hand_pos = (unsigned char)((hour_hand_pos<<2) + hour_hand_pos + pTim->min / 12);

  if( (cur_mode == AdjustTZ && pTim->subsec >= 30) ) {
      memset(pAnodes, 0, 5*sizeof(*pAnodes));
      memset(pCathodes, 0, 5*sizeof(*pCathodes));
      return;
  }
  if( cur_mode == AdjustTZ )
      cfg = no_sec;

  get_led_mask(hour_hand_pos, pSch->hour, &pAnodes[0], &pCathodes[0]);
  get_led_mask(pTim->min, pSch->min, &pAnodes[1], &pCathodes[1]);
  switch(cfg) {
      case matching_subsec:
          get_led_mask(pTim->sec, pSch->sec, &pAnodes[2], &pCathodes[2]);
          get_led_mask(subsec_hand_pos, pSch->sec, &pAnodes[3], &pCathodes[3]);
          break;
      case white_subsec:
          get_led_mask(pTim->sec, pSch->sec, &pAnodes[2], &pCathodes[2]);
          get_led_mask(subsec_hand_pos, white, &pAnodes[3], &pCathodes[3]);
          break;
      case no_subsec:
          get_led_mask(pTim->sec, pSch->sec, &pAnodes[2], &pCathodes[2]);
      case no_sec:
          pAnodes[3].A = 0;
          pAnodes[3].B = 0;
          pAnodes[3].C = 0;
          pCathodes[3].A = 0;
          pCathodes[3].B = 0;
          pCathodes[3].C = 0;
          break;
  }
  if( cfg == no_sec ) {
    pAnodes[2].A = 0;
    pAnodes[2].B = 0;
    pAnodes[2].C = 0;
    pCathodes[2].A = 0;
    pCathodes[2].B = 0;
    pCathodes[2].C = 0;
  }
  get_led_mask(pTim->hour >= 12 ? LED_PM : LED_AM, white, &pAnodes[4], &pCathodes[4]);
//  get_led_mask(pTim->hour > 12 ? LED_PM : LED_AM, red, &A, &K);
}

// Initialises the ADC peripheral.
void initADC(void){
    ADCON0=0;           //reset ADC, single ended
    ADCON1=0;
    ADCON2=0;           //legacy mode
    ADCON3=0;
    ADPRE=0;
    ADACQ=0;
    ADCAP=0;            //extra capacitance
    ADCON0bits.CS=1;    //ADCRC
    ADCON0bits.ADFM0=1; //right justify
    ADREF = 0;          //VDD
    ADCON0bits.ADON=1;  //turn on, ready
}

// Reads an analog input.
unsigned short getADC(unsigned char c){
    ADPCH=c;
    ADCON0bits.GO=1;
    while(ADCON0bits.GO){}    
    return ADRES;
}

// Figure out which ADC channel corresponds to a given RAx/RBx/RCx pin.
unsigned char ADC_channel_for_port(port* p) {
    unsigned char ret = 58, val = 0;
    if( p->A ) {
        ret = 0;
        val = p->A;
    } else if( p->B ) {
        ret = 8;
        val = p->B;
    } else if( p->C ) {
        ret = 16;
        val = p->C;
    }
    while( val > 1 ) {
        ++ret;
        val >>= 1;
    }
    return ret;
}

// Uses the weak pull-up to sense if a given LED is present in a given configuration (CA/CC).
unsigned char have_LED(unsigned char LEDnum, colour LEDc, RGB_LED_type type) {
    unsigned char ret = 0;
    unsigned short ADCval;
    port A, C;
    get_led_mask(LEDnum, LEDc, &A, &C);
    if( type == CA ) {
        WPUA = A.A;
        WPUB = A.B;
        WPUC = A.C;
        LATA = 0;
        LATB = 0;
        LATC = 0;
        TRISA = ~C.A;
        TRISB = ~C.B;
        TRISC = ~C.C;
        ANSELA |= A.A;
        ANSELB |= A.B;
        ANSELC |= A.C;
        ADCval = getADC(ADC_channel_for_port(&A));
        ANSELA &= ~A.A;
        ANSELB &= ~A.B;
        ANSELC &= ~A.C;
        ret = (ADCval < 2000 && ADCval > 512);
    } else {
        WPUA = C.A;
        WPUB = C.B;
        WPUC = C.C;
        LATA = 0;
        LATB = 0;
        LATC = 0;
        TRISA = ~A.A;
        TRISB = ~A.B;
        TRISC = ~A.C;
        ANSELA |= C.A;
        ANSELB |= C.B;
        ANSELC |= C.C;
        ADCval = getADC(ADC_channel_for_port(&C));
        ANSELA &= ~C.A;
        ANSELB &= ~C.B;
        ANSELC &= ~C.C;
        ret = (ADCval < 2000 && ADCval > 512);
    }
    TRISA = 0xFF;
    TRISB = 0xFF;
    TRISC = 0xFF;
    WPUA = PIN4;
    WPUB = 0;
    WPUC = 0;
    return ret;
}

// Determines the RGB colour scheme for a setting number between 0 and 5
void get_colour_scheme(scheme* ps, unsigned char which) {
    switch(which>>1) {
        case 0:
            ps->sec = red;
            break;
        case 1:
            ps->sec = green;
            break;
        case 2:
        default:
            ps->sec = blue;
            break;
    }
    switch(ps->sec) {
        case red:
            if( which&1 ) {
                ps->min = blue;
                ps->hour = green;
            } else {
                ps->min = green;
                ps->hour = blue;
            }
            break;
        case green:
            if( which&1 ) {
                ps->min = blue;
                ps->hour = red;
            } else {
                ps->min = red;
                ps->hour = blue;
            }
            break;
        case blue:
            if( which&1 ) {
                ps->min = green;
                ps->hour = red;
            } else {
                ps->min = red;
                ps->hour = green;
            }
            break;
        default:
            break;
    }
    ps->subsec = white;
}

// Adds the time zone offset to the UTC time to get local time
void addTzOffset(time* pNow, time* pLocal) {
    signed short offset = tzoffset;
    signed short hour, min;
    if( dst )
        offset += 60;
    
    pLocal->sec = pNow->sec;
    pLocal->subsec = pNow->subsec;
    hour = (signed short)pNow->hour + offset/60;
    min = (signed short)pNow->min + offset%60;
    while( min > 60 ) {
        min -= 60;
        ++hour;
    }
    while( min < 0 ) {
        min += 60;
        --hour;
    }
    while( hour >= 24 )
        hour -= 24;
    while( hour < 0 )
        hour += 24;
    pLocal->hour = (unsigned char)hour;
    pLocal->min = (unsigned char)min;
}

// Applies a weak pull-up to RA5 to verify if it's pulled low by a button press or the LDR resistance is just really high.
unsigned char is_RA5_really_low() {
    unsigned short adc;
    WPUAbits.WPUA5 = 1;
    adc = getADC(5);// RA5
    WPUAbits.WPUA5 = 0;
    return adc < 384;
}

// Sets PWM to a particular RAx/RBx/RCx pin. The switch-based version may be faster but will only work with one pin at a time.
#ifndef INTERRUPT_BASED_PWM
port last_pwm_output = { 0, 0, 0, 0 };
void set_PPS(port* p, unsigned char function) {
#ifdef PPS_USE_SWITCH
    switch(p->A) {
        case 0x01:
            RA0PPS = function;
            break;
        case 0x02:
            RA1PPS = function;
            break;
        case 0x04:
            RA2PPS = function;
            break;
        case 0x20:
            RA4PPS = function;
            break;
    }
    switch(p->B) {
        case 0x10:
            RB4PPS = function;
            break;
        case 0x20:
            RB5PPS = function;
            break;
        case 0x40:
            RB6PPS = function;
            break;
        case 0x80:
            RB7PPS = function;
            break;
    }
    switch(p->C) {
        case 0x01:
            RC0PPS = function;
            break;
        case 0x02:
            RC1PPS = function;
            break;
        case 0x04:
            RC2PPS = function;
            break;
        case 0x08:
            RC3PPS = function;
            break;
        case 0x10:
            RC4PPS = function;
            break;
        case 0x20:
            RC5PPS = function;
            break;
        case 0x40:
            RC6PPS = function;
            break;
        case 0x80:
            RC7PPS = function;
            break;
    }
#else
  if( p->A ) {
    if( p->A & 0x0F ) {
      if( p->A & 0x03 ) {
        if( p->A & 0x01 )
          RA0PPS = function;
        if( p->A & 0x02 )
          RA1PPS = function;
      }
      if( p->A & 0x04 ) {
        RA2PPS = function;
      }
    }
    if( p->A & 0x30 ) {
      if( p->A & 0x10 )
        RA4PPS = function;
      if( p->A & 0x20 )
        RA5PPS = function;
    }
  }
  else if( p->B ) {
    if( p->B & 0x30 ) {
      if( p->B & 0x10 )
        RB4PPS = function;
      if( p->B & 0x20 )
        RB5PPS = function;
    }
    else if( p->B & 0xC0 ) {
      if( p->B & 0x40 )
        RB6PPS = function;
      if( p->B & 0x80 )
        RB7PPS = function;
    }
  }
  else if( p->C ) {
    if( p->C & 0x0F ) {
      if( p->C & 0x03 ) {
        if( p->C & 0x01 )
          RC0PPS = function;
        if( p->C & 0x02 )
          RC1PPS = function;
      }
      if( p->C & 0x0C ) {
        if( p->C & 0x04 )
          RC2PPS = function;
        if( p->C & 0x08 )
          RC3PPS = function;
      }
    }
    if( p->C & 0xF0 ) {
      if( p->C & 0x30 ) {
        if( p->C & 0x10 )
          RC4PPS = function;
        if( p->C & 0x20 )
          RC5PPS = function;
      }
      if( p->C & 0xC0 ) {
        if( p->C & 0x40 )
          RC6PPS = function;
        if( p->C & 0x80 )
          RC7PPS = function;
      }
    }
  }
#endif
}
// Disables the current PWM pin and sets the new one specified to be the PWM output.
void select_PWM_output(port* p) {
    if( p->A != last_pwm_output.A || p->B != last_pwm_output.B || p->C != last_pwm_output.C ) {
        set_PPS(&last_pwm_output, 0); // 0 = LATxy
        last_pwm_output = *p;
        set_PPS(&last_pwm_output, 11); // 11 = PWM1S1P1_OUT
    }
}
#endif

void main(void) {
    time now = { 12, 5, 0 }, local;
    scheme sch;
    unsigned char i, bootup = 0, bootup_delay = 0, brightness_chaser = 0, ldr_max, valid_time = 0;
    unsigned short ldr, last_ldr = 4095;

    get_colour_scheme(&sch, which_colour_scheme);

    // RA5 is the only real analog input
    ANSELA = PIN5;
    ANSELB = 0;
    ANSELC = 0;
    initADC();

#ifdef DEBUGGING
    LED_type = CA;
    bootup = 62;
#else
    // Pressing a button at power-up can override the CA/CC LED sensing in case it is wrong.
    ldr = getADC(5);// LDR is on RA5
    if( ldr > 4096-32 )
        LED_type = CA;
    else if( ldr < 384 && is_RA5_really_low() )
        LED_type = CC;
    else if( have_LED(0, red, CC) )
        LED_type = CC;
    else
        LED_type = CA;
#endif
    have_am_pm = have_LED(LED_AM, white, LED_type) || have_LED(LED_PM, white, LED_type);

    // Set up timers & PWM
#ifdef INTERRUPT_BASED_PWM
    T0CON1 = 0b01000011; // FOSC/4, prescaler = 4
#else
    T0CON1 = 0b01000100; // FOSC/4, prescaler = 8
#endif
    T0CON0 = 0b10000000; // on, 8-bit timer, no postscaler
    TMR0H = 0xFF;
    PIR0bits.TMR0IF = 0;
    while( !PIR0bits.TMR0IF )
        ;
    PIR0bits.TMR0IF = 0;
    PIE0bits.TMR0IE = 1;
    INTCONbits.GIE = 1;
  
    T1CON = 0b00010011;
    T1CLK = 1;
    PIR1bits.TMR1IF = 0;
#ifdef INTERRUPT_BASED_PWM
    T2CLKCON = 1; // FOSC/4
    T2CON = 0b10100000; // on, prescaler = 4
    PIR2bits.TMR2IF = 0;
    PIE2bits.TMR2IE = 1;
#else
    PWM1CLK = 2; // FOSC
    PWM1PRH = 0;
    PWM1PRL = 255;
    PWM1CON = 0b1000000; // enabled
    PWM1S1CFG = 0;
    PWM1S1P1H = 0;
    PWM1S1P1L = 255;
    PWMLOAD = 1;
    PWMEN = 1;
#endif
    INTCONbits.PEIE = 1;

    // Set up serial port.
    WPUA = PIN4;
    uartInit(baud_rates[which_baud_rate]);

    while(1) {
        if( bootup < 62 ) {
            // This is the initial LED test mode that lights each one white in turn at full brightness.
            get_led_mask(bootup, white, &anode_states[which_used^1][0], &cathode_states[which_used^1][0]);
            get_led_mask(bootup, white, &anode_states[which_used^1][1], &cathode_states[which_used^1][1]);
            get_led_mask(bootup, white, &anode_states[which_used^1][2], &cathode_states[which_used^1][2]);
            get_led_mask(bootup, white, &anode_states[which_used^1][3], &cathode_states[which_used^1][3]);
            get_led_mask(bootup, white, &anode_states[which_used^1][4], &cathode_states[which_used^1][4]);
            if( ++bootup_delay == 20 ) {
                bootup_delay = 0;
                ++bootup;
            }
#ifdef INTERRUPT_BASED_PWM
            T2PR = 255;
#else
            PWM1S1P1L = 255;
            PWMLOAD = 1;
#endif
        } else if( cur_mode == AdjustLDRMinBrightness || cur_mode == AdjustLDRMaxBrightness ) {
            // When in LDR adjustment mode, we change the clock display to a chaser so you can see the settings.
            unsigned char min, max;
            min = ldr_min_brightness;
            max = ldr_max_brightness;
            if( brightness_chaser < min || brightness_chaser > max )
                brightness_chaser = min;
            else
                ++brightness_chaser;
              get_led_mask(brightness_chaser, cur_mode == AdjustLDRMaxBrightness ? red : blue, &anode_states[which_used^1][0], &cathode_states[which_used^1][0]);
              get_led_mask(brightness_chaser, cur_mode == AdjustLDRMaxBrightness ? red : blue, &anode_states[which_used^1][1], &cathode_states[which_used^1][1]);
              get_led_mask(brightness_chaser, cur_mode == AdjustLDRMaxBrightness ? red : blue, &anode_states[which_used^1][2], &cathode_states[which_used^1][2]);
              memset(&anode_states[which_used^1][3], 0, sizeof(anode_states[which_used^1][3]));
              memset(&cathode_states[which_used^1][3], 0, sizeof(cathode_states[which_used^1][3]));
              memset(&anode_states[which_used^1][4], 0, sizeof(anode_states[which_used^1][4]));
              memset(&cathode_states[which_used^1][4], 0, sizeof(cathode_states[which_used^1][4]));
        } else {
            //When we are getting time from the GPS module, add the timezone offset and then update the clock face.
            if( valid_time ) {
                addTzOffset(&now, &local);
                get_led_masks_from_time(anode_states[which_used^1], cathode_states[which_used^1], &local, &sch, second_hand_cfg);
                --valid_time;
            } else {
                // If the GPS time is not valid, display the chaser and baud rate setting instead.
                unsigned char baud_pos = (which_baud_rate + 1) * 5;
                if( gps_detected )
                    --gps_detected;

                get_led_mask(brightness_chaser, gps_detected ? green : red, &anode_states[which_used^1][0], &cathode_states[which_used^1][0]);
                get_led_mask(brightness_chaser, gps_detected ? green : red, &anode_states[which_used^1][1], &cathode_states[which_used^1][1]);
                get_led_mask(brightness_chaser, gps_detected ? green : red, &anode_states[which_used^1][2], &cathode_states[which_used^1][2]);
                get_led_mask(baud_pos, blue, &anode_states[which_used^1][3], &cathode_states[which_used^1][3]);
                get_led_mask(baud_pos, blue, &anode_states[which_used^1][4], &cathode_states[which_used^1][4]);

                if( ++brightness_chaser >= 60 )
                    brightness_chaser = 0;
            }
        }
        // This calculates the mask for which outputs need to be driven to light the given LEDs.
        for( i = 0; i < 5; ++i ) {
            cathode_states[which_used^1][i].A = ~(anode_states[which_used^1][i].A|cathode_states[which_used^1][i].A);
            cathode_states[which_used^1][i].B = ~(anode_states[which_used^1][i].B|cathode_states[which_used^1][i].B);
            cathode_states[which_used^1][i].C = ~(anode_states[which_used^1][i].C|cathode_states[which_used^1][i].C);
        }
        // It's double buffered so the ISR doesn't access the data while it's being changed. Switch to the new data buffer now.
        which_used ^= 1;

        // Read and parse serial data until the next tick when we need to update the clock face.
        while( !PIR1bits.TMR1IF ) {
            if( uartAvailable() ) {
                rtccTime t;
                rtccDate d;
                if( serial_read(4, &t, &d) ) {
                    if( now.hour != t.hour || now.min != t.min || now.sec != t.sec ) {
                        now.hour = t.hour;
                        now.min = t.min;
                        now.sec = t.sec;
                        now.subsec = 0;
                    }
                    valid_time = 255;
                }
            }
        }
        PIR1bits.TMR1IF = 0;

        // Check the LDR reading, which also indicates if a button is pressed.
        ldr = getADC(5);// LDR is on RA5
        b1down = ldr > 4096-32;
        b2down = ldr < 384 && is_RA5_really_low();

        // During the LED test, unless a button is pressed, skip the rest.
        if( bootup < 62 ) {
            if( b1down )
                bootup = 62;
            continue;
        }

        // Advance the time and update the display brightness based on the LDR reading.
        advance_time(&now);
        if( !b1down && !b2down )
            last_ldr = last_ldr - (last_ldr >> 5) + (ldr>>5);
        ldr = last_ldr;
        if( ldr > 2048+1024 )
            ldr -= 2048+1024;
        else
            ldr = 0;
        ldr = ldr_min_brightness * 255 / 59 + (ldr_max_brightness - ldr_min_brightness) * ldr / 90;
        ldr_max = ldr_max_brightness >= 30 ? 255 : (ldr_max_brightness * 255 / 30);
        if( ldr > ldr_max )
            ldr = ldr_max;
        brightness = (unsigned char)ldr;
        if( cur_mode == AdjustLDRMaxBrightness )
            brightness = ldr_max;
#ifdef INTERRUPT_BASED_PWM
        if( brightness < 2 )
            brightness = 2;
        T2PR = brightness;
#else
        if( brightness < 1 )
            brightness = 1;
        PWM1S1P1L = brightness;
        PWMLOAD = 1;
#endif
        
        // Button handling logic for settings etc.
        if( b1down && button_release_counter == 20 ) {
            if( !b1wasdown ) {
                b1wasdown = 1;
                button_press_counter = 0;
            } else {
                if( ++button_press_counter > maxb1press )
                    maxb1press = button_press_counter;
            }
        } else if( b2down && button_release_counter == 20 ) {
            if( !b2wasdown ) {
                b2wasdown = 1;
                button_press_counter = 0;
            } else {
                if( ++button_press_counter > maxb2press )
                    maxb2press = button_press_counter;
            }
        } else {
            if( b1wasdown ) {
                b1wasdown = 0;
                button_release_counter = 0;
            }
            if( b2wasdown ) {
                b2wasdown = 0;
                button_release_counter = 0;
            }
            if( button_release_counter == DEBOUNCE_SHORT ) {
                if( maxb1press > DEBOUNCE_SHORT ) {
                    if( maxb1press > DEBOUNCE_LONG ) {
                        ++b1longpress;
                    } else {
                        ++b1shortpress;
                    }
                }
                maxb1press = 0;
                if( maxb2press > DEBOUNCE_SHORT ) {
                    if( maxb2press > DEBOUNCE_LONG ) {
                        ++b2longpress;
                    } else {
                        ++b2shortpress;
                    }
                }
                maxb2press = 0;
                ++button_release_counter;
            } else if( button_release_counter < 20 ) {
                ++button_release_counter;
            }
        }

        if( !valid_time ) {
            if( b1shortpress ) {
                --b1shortpress;
                if( which_baud_rate > 0 ) {
                    --which_baud_rate;
                    uartInit(baud_rates[which_baud_rate]);
                }
            }
            if( b2shortpress ) {
                --b2shortpress;
                if( which_baud_rate < 5 ) {
                    ++which_baud_rate;
                    uartInit(baud_rates[which_baud_rate]);
                }
            }
        } else switch(cur_mode) {
            case ShowTime:
                if( b1shortpress ) {
                    --b1shortpress;
                    which_colour_scheme = (which_colour_scheme+1) & 7;
                    get_colour_scheme(&sch, which_colour_scheme);
                }
                if( b1longpress ) {
                    cur_mode = AdjustTZ;
                    --b1longpress;
                }
                if( b2shortpress ) {
                    --b2shortpress;
                    if( second_hand_cfg == no_sec )
                        second_hand_cfg = matching_subsec;
                    else
                        ++second_hand_cfg;
                }
                if( b2longpress ) {
                    cur_mode = AdjustLDRMaxBrightness;
                    --b2longpress;
                }
                break;
            case AdjustTZ:
                if( b1shortpress ) {
                    if( tzoffset > -14*60 )
                        tzoffset = tzoffset - 15;
                    --b1shortpress;
                }
                if( b1longpress ) {
                    cur_mode = ShowTime;
                    --b1longpress;
                }
                if( b2shortpress ) {
                    if( tzoffset < 14*60 )
                        tzoffset = tzoffset + 15;
                    --b2shortpress;
                }
                if( b2longpress ) {
                    dst = !dst;
                    --b2longpress;
                }
                break;
            case AdjustLDRMaxBrightness:
                if( b1shortpress ) {
                    if( ldr_max_brightness > ldr_min_brightness )
                        --ldr_max_brightness;
                    --b1shortpress;
                }
                if( b1longpress ) {
                    cur_mode = ShowTime;
                    --b1longpress;
                }
                if( b2shortpress ) {
                    if( ldr_max_brightness < 59 )
                        ++ldr_max_brightness;
                    --b2shortpress;
                }
                if( b2longpress ) {
                    cur_mode = AdjustLDRMinBrightness;
                    --b2longpress;
                }
                break;
            case AdjustLDRMinBrightness:
                if( b1shortpress ) {
                    if( ldr_min_brightness > 0 )
                        --ldr_min_brightness;
                    --b1shortpress;
                }
                if( b1longpress ) {
                    cur_mode = ShowTime;
                    --b1longpress;
                }
                if( b2shortpress ) {
                    if( ldr_min_brightness < ldr_max_brightness )
                        ++ldr_min_brightness;
                    --b2shortpress;
                }
                if( b2longpress ) {
                    cur_mode = AdjustLDRMaxBrightness;
                    --b2longpress;
                }
                break;
        }
    }
}

// With the default hardware-based PWM, the ISR just has to do two things:
// 1) Multiplex the LEDs, cycling through the 5-6 possible states.
// 2) Receive incoming serial data.
void __interrupt() isr(){
#ifdef INTERRUPT_BASED_PWM
    if( PIR2bits.TMR2IF ) {
        LATA = 0;
        LATB = 0;
        LATC = 0;
        PIR2bits.TMR2IF = 0;
    } else
#endif
    if( PIR0bits.TMR0IF ) {
        port* p1;
        port* p2;
        if( ++cur_state == (4 + have_am_pm) )
            cur_state = 0;
        TRISA = 0xFF;
        TRISB = 0xFF;
        TRISC = 0xFF;
        p1 = &anode_states[which_used][cur_state];
        LATA = p1->A;
        LATB = p1->B;
        LATC = p1->C;
#ifndef INTERRUPT_BASED_PWM
        select_PWM_output(p1);
#endif
        p2 = &cathode_states[which_used][cur_state];
        TRISA = p2->A;
        TRISB = p2->B;
        TRISC = p2->C;
#ifdef INTERRUPT_BASED_PWM
        TMR2 = 0;
#endif
        PIR0bits.TMR0IF = 0;
#ifdef INTERRUPT_BASED_PWM
        PIR2bits.TMR2IF = 0;
#endif
    }
    uartISR();
}
