
/*
  device: PIC18F45K80 (TQFP-44)
  pin mapping:
	19 RA0/AN0 : LCD COM
	20 RA1/AN1 : LCD 1G
	21 RA2/AN2 : INA282 OUT
	22 RA3/AN3 : OPA2376 OUT
	24 RA5/AN4 : LCD 1F
	31 RA6     : LCD 2A
	30 RA7     : LCD 2F
	 8 RB0     : LCD DP2
	 9 RB1     : LCD 2C
	10 RB2     : LCD 2D
	11 RB3     : LCD 2E
	14 RB4     : LCD DP1
	15 RB5     : LCD 1C
	16 RB6     : LCD 1D / PGC
	17 RB7     : LCD 1E / PGD
	32 RC0     : LCD 2B
	35 RC1     : LCD 3G
	36 RC2     : LCD 3F
	37 RC3     : LCD 3A
	42 RC4     : LCD 4B
	43 RC5     : LCD 4C
	44 RC6     : LCD 4D
	 1 RC7     : LCD 4E
	38 RD0     : LCD 3B
	39 RD1     : LCD 4G
	40 RD2     : LCD 4F
	41 RD3     : LCD 4A
	 2 RD4     : LCD DP3
	 3 RD5     : LCD 3C
	 4 RD6     : LCD 3D
	 5 RD7     : LCD 3E
	25 RE0/AN5 : LCD 1A
	26 RE1/AN6 : LCD 1B
	27 RE2/AN7 : LCD 2G
	18 RE3/MCLR: S1 / PGR
*/

#include "p18F45K80.h"
#include <string.h>

#pragma config RETEN 		= ON		//VREG Sleep Enable bit
#pragma config INTOSCSEL 	= LOW		//LF-INTOSC Low-power Enable bit
#pragma config SOSCSEL 		= DIG		//SOSC Power Selection and mode Configuration bits
#pragma config XINST 		= OFF		//Extended Instruction Set
#pragma config FOSC 		= INTIO2	//Oscillator
#pragma config PLLCFG 		= OFF		//PLL x4 Enable bit
#pragma config FCMEN 		= ON		//Fail-Safe Clock Monitor
#pragma config IESO 		= OFF		//Internal External Oscillator Switch Over Mode
#pragma config PWRTEN 		= OFF		//Power Up Timer
#pragma config BOREN 		= OFF		//Brown Out Detect
#pragma config BORV 		= 3			//Brown-out Reset Voltage bits
#pragma config BORPWR 		= HIGH		//BORMV Power level
#pragma config WDTEN 		= OFF		//Watchdog Timer
#pragma config WDTPS 		= 16384		//Watchdog Postscaler
#pragma config CANMX 		= PORTB		//ECAN Mux bit
//#pragma config MSSPMSK 	= MSK7		//MSSP address masking
#pragma config MCLRE 		= OFF		//Master Clear Enable
#pragma config STVREN 		= ON		//Stack Overflow Reset
//#pragma config BBSIZ0 		= BB2K		//Boot Block Size
#pragma config CP0 			= OFF		//Code Protect 00800-03FFF
#pragma config CP1 			= OFF		//Code Protect 04000-07FFF
#pragma config CP2 			= OFF		//Code Protect 08000-0BFFF
#pragma config CP3 			= OFF		//Code Protect 0C000-0FFFF
#pragma config CPB 			= OFF		//Code Protect Boot
#pragma config CPD 			= OFF		//Data EE Read Protect
#pragma config WRT0 		= OFF		//Table Write Protect 00800-03FFF
#pragma config WRT1 		= OFF		//Table Write Protect 04000-07FFF
#pragma config WRT2 		= OFF		//Table Write Protect 08000-0BFFF
#pragma config WRT3 		= OFF		//Table Write Protect 0C000-0FFFF
#pragma config WRTC 		= OFF		//Config. Write Protect
#pragma config WRTB 		= OFF		//Table Write Protect Boot
#pragma config WRTD 		= OFF		//Data EE Write Protect

void high_isr(void);
#pragma code high_vector=0x08
void interrupt_at_high_vector(void)
{
  _asm GOTO high_isr _endasm
}
#pragma code

void low_isr(void);
#pragma code low_vector=0x18
void interrupt_at_low_vector(void)
{
  _asm GOTO low_isr _endasm
}
#pragma code

unsigned char lcd_state[5];
unsigned short ms_timer;

#pragma interrupt high_isr
void high_isr(void) {
  if( PIR3bits.CCP2IF ) {
    LATA = lcd_state[0];
    LATB = lcd_state[1];
    LATC = lcd_state[2];
    LATD = lcd_state[3];
    LATE = lcd_state[4];
    PIR3bits.CCP2IF = 0;
    ++ms_timer;
  } else if( PIR4bits.CCP3IF ) {
    LATA = 0xFF;
    LATB = 0xFF;
    LATC = 0xFF;
    LATD = 0xFF;
    LATE = 0xFF;
    PIR4bits.CCP3IF = 0;
    ++ms_timer;
  } else if( PIR4bits.CCP4IF ) {
    LATA = ~lcd_state[0];
    LATB = ~lcd_state[1];
    LATC = ~lcd_state[2];
    LATD = ~lcd_state[3];
    LATE = ~lcd_state[4];
    PIR4bits.CCP4IF = 0;
    ++ms_timer;
  } else if( PIR4bits.CCP5IF ) {
    LATA = 0;
    LATB = 0;
    LATC = 0;
    LATD = 0;
    LATE = 0;
    PIR4bits.CCP5IF = 0;
    ++ms_timer;
  }
}

#pragma interrupt low_isr
void low_isr(void) {
}

#define LCD_A  (1   )
#define LCD_B  (1<<1)
#define LCD_C  (1<<2)
#define LCD_D  (1<<3)
#define LCD_E  (1<<4)
#define LCD_F  (1<<5)
#define LCD_G  (1<<6)
#define LCD_DP (1<<7)

char lcd_digits[4], dp;
static const unsigned char digits[10] = {  LCD_A | LCD_B | LCD_C | LCD_D | LCD_E | LCD_F,         // 0
                                                   LCD_B | LCD_C,                                 // 1
                                           LCD_A | LCD_B |         LCD_D | LCD_E |         LCD_G, // 2
                                           LCD_A | LCD_B | LCD_C | LCD_D |                 LCD_G, // 3
                                                   LCD_B | LCD_C |                 LCD_F | LCD_G, // 4
                                           LCD_A |         LCD_C | LCD_D |         LCD_F | LCD_G, // 5
                                           LCD_A |         LCD_C | LCD_D | LCD_E | LCD_F | LCD_G, // 6
                                           LCD_A | LCD_B | LCD_C,                                 // 7
                                           LCD_A | LCD_B | LCD_C | LCD_D | LCD_E | LCD_F | LCD_G, // 8
                                           LCD_A | LCD_B | LCD_C | LCD_D |         LCD_F | LCD_G  // 9
};

static const unsigned char letters[26] = { LCD_A | LCD_B | LCD_C |         LCD_E | LCD_F | LCD_G, // A
                                                           LCD_C | LCD_D | LCD_E | LCD_F | LCD_G, // b
                                           LCD_A |                 LCD_D | LCD_E | LCD_F,         // C
                                                   LCD_B | LCD_C | LCD_D | LCD_E |         LCD_G, // d
                                           LCD_A |                 LCD_D | LCD_E | LCD_F | LCD_G, // E
                                           LCD_A |                         LCD_E | LCD_F | LCD_G, // F
                                           LCD_A | LCD_B | LCD_C | LCD_D |         LCD_F | LCD_G, // g
                                                   LCD_B | LCD_C |         LCD_E | LCD_F | LCD_G, // H
                                                                           LCD_E | LCD_F,         // I
                                                   LCD_B | LCD_C | LCD_D,                         // J
                                                                           LCD_E | LCD_F | LCD_G, // k (poor)
                                                                   LCD_D | LCD_E | LCD_F,         // L
                                           LCD_A |         LCD_C |         LCD_E |         LCD_G, // m (poor)
                                           LCD_A | LCD_B | LCD_C |         LCD_E | LCD_F,         // N
                                           LCD_A | LCD_B | LCD_C | LCD_D | LCD_E | LCD_F,         // O
                                           LCD_A | LCD_B |                 LCD_E | LCD_F | LCD_G, // P
                                           LCD_A | LCD_B | LCD_C |                 LCD_F | LCD_G, // q
                                                                           LCD_E |         LCD_G, // r
                                           LCD_A |         LCD_C | LCD_D |         LCD_F | LCD_G, // S
                                           LCD_A | LCD_B | LCD_C,                                 // t (poor)
                                                   LCD_B | LCD_C | LCD_D | LCD_E | LCD_F,         // U
                                                   LCD_B | LCD_C | LCD_D | LCD_E,                 // V (poor)
                                           LCD_A |         LCD_C | LCD_D | LCD_E,                 // w (poor)
                                                   LCD_B | LCD_C |         LCD_E | LCD_F | LCD_G, // x (poor)
                                                   LCD_B | LCD_C | LCD_D |         LCD_F | LCD_G, // Y
                                           LCD_A | LCD_B |         LCD_D | LCD_E |         LCD_G  // Z
};


unsigned char ReadEEPROMByte(const rom unsigned char* address);
void WriteEEPROMByte(rom unsigned char* address, unsigned char data);
unsigned long ReadEEPROMLong(const rom unsigned long* address);
void WriteEEPROMLong(rom unsigned long* address, unsigned long data);

#pragma romdata dataEEPROM=0xF00000
rom unsigned long eeCalibration[3];
rom signed char eeVoltCal;
rom unsigned char eeFlip;
rom unsigned char eeMode[127];
#pragma udata

static unsigned char lcd_flip_digit(unsigned char data) {
  return ((data&0x07)<<3)|((data&0x38)>>3)|(data&0xC0);
}

static void lcd_update_state(unsigned char flip) {
  unsigned char new_lcd_state[5];
  unsigned char segments[4];
  unsigned char i, ldp;

  for( i = 0; i < 4; ++i ) {
    if( lcd_digits[i] >= '0' && lcd_digits[i] <= '9' )
      segments[i] =  digits[lcd_digits[i]-'0'];
    else if( lcd_digits[i] >= 'A' && lcd_digits[i] <= 'Z' )
      segments[i] =  letters[lcd_digits[i]-'A'];
    else if( lcd_digits[i] >= 'a' && lcd_digits[i] <= 'z' )
      segments[i] =  letters[lcd_digits[i]-'a'];
    else if( lcd_digits[i] >= '-' )
      segments[i] =  LCD_G;
    else
      segments[i] =  0;
  }

  ldp = dp;
  if( flip ) {
	unsigned char temp = segments[0];
    segments[0] = lcd_flip_digit(segments[3]);
    segments[3] = lcd_flip_digit(temp);
	temp = segments[1];
    segments[1] = lcd_flip_digit(segments[2]);
    segments[2] = lcd_flip_digit(temp);
    if( ldp == 1 )
      ldp = 3;
    else if( ldp == 3 )
      ldp = 1;
  }

  new_lcd_state[0] = ((segments[0]&LCD_G)?2:0)|((segments[0]&LCD_F)?32:0)|((segments[1]&LCD_A)?64:0)|((segments[1]&LCD_F)?128:0);
  new_lcd_state[1] = ((segments[1]>>1)&0x0E)|((segments[0]<<3)&0xE0)|(ldp==2?0x01:0)|(ldp==1?0x10:0);
  new_lcd_state[2] = ((segments[1]&LCD_B)?1:0)|((segments[2]&LCD_G)?2:0)|((segments[2]&LCD_F)?4:0)|((segments[2]&LCD_A)?8:0)|((segments[3]<<3)&0xF0);
  new_lcd_state[3] = ((segments[2]&LCD_B)?1:0)|((segments[3]&LCD_G)?2:0)|((segments[3]&LCD_F)?4:0)|((segments[3]&LCD_A)?8:0)|((segments[2]<<3)&0xE0)|(ldp==3?0x10:0);
  new_lcd_state[4] = (segments[0]&3)|((segments[1]&LCD_G)?4:0);

  INTCONbits.GIEH = 0;
  memcpy(lcd_state, (const void*)new_lcd_state, 5);
  INTCONbits.GIEH = 1;
}

#define LCD_PERIOD 5000
unsigned char lcd_contrast = 255;

static void lcd_update_contrast(void) {
  unsigned short contrast_period1 = ((unsigned long)(LCD_PERIOD) * (unsigned long)lcd_contrast)>>10;
  unsigned short contrast_period2 = contrast_period1+(LCD_PERIOD)/2;
  CCPR2H = contrast_period1>>8;
  CCPR2L = contrast_period1;
  CCPR4H = contrast_period2>>8;
  CCPR4L = contrast_period2;
}

unsigned char debounce = 0, buttonpress = 0, buttonunpress = 0;

unsigned long get_adc_reading(unsigned char input, unsigned short num_avgs) {
  unsigned long avg = 0;

  ADCON0 = 1|(input<<2); // ADC on, use input AN2

  do {
	if( !(num_avgs&31) ) {
	  debounce = (debounce<<1)|!PORTEbits.RE3;
      if( debounce == 0xFF ) {
        buttonpress = 1;
      } else if( debounce == 0 ) {
        buttonunpress = 1;
      }
    }   
    PIR1bits.ADIF = 0;
    ADCON0bits.GO = 1;
    while( !PIR1bits.ADIF )
      ;
    avg += (((unsigned short)(ADRESH&15))<<8) | ADRESL;
  } while( --num_avgs );

  return avg;
}

unsigned short get_adc_volts_reading(unsigned short num_avgs) {
  unsigned long temp;
  ADCON1 = (0<<4); // VSS is negative reference, VDD is positive reference
  temp = get_adc_reading(31, num_avgs);
  ADCON1 = (3<<4); // VSS is negative reference, 4.095V is positive reference
  return 419430UL * num_avgs / temp;
}

signed long adc_reading_to_nominal_5V(signed long reading, signed short volts, unsigned short scalefactor) {
	// offset is (approximately) volts / 13
	return reading - ((unsigned long)volts * (unsigned long)scalefactor + 7) / 13;
}

signed long adc_reading_to_actual_Vin(signed long reading, signed short volts, unsigned short scalefactor) {
	// offset is (approximately) volts / 13
	return reading + ((unsigned long)volts * (unsigned long)scalefactor + 7) / 13;
}

#define NUM_AVGS 2048
#define NUM_CAL_AVGS 8
#define UAMP_AVGS 8


void IncrementModeInEEPROM(void) {
  unsigned char i;
  unsigned char first = ReadEEPROMByte(&eeMode[0]);
  for( i = 1; i < 127; ++i ) {
    unsigned char val = ReadEEPROMByte(&eeMode[i]);
    if( val != first ) {
      WriteEEPROMByte(&eeMode[i], (val+1)%3);
      return;
    }
  }
  WriteEEPROMByte(&eeMode[0], (first+1)%3);
}

unsigned short absdiff(unsigned short a, unsigned short b) {
  if( a > b )
    return a - b;
  else
    return b - a;
}

void main(void) {
  unsigned short volts;
  unsigned long baseline2, baseline3, voltCalRatio;
  unsigned long fortieths_of_microamp_hours = 0;
  unsigned char i, button = 0, slow_debounce = 0;
  unsigned char flip = 0, flash = 0, mode;
  signed char voltcal = 0;
  ms_timer = 0;

  T1CONbits.T1CKPS = 3; // 16MHz / 8 = 2MHz

  ANCON0bits.ANSEL2 = 1;
  ADCON2 = (1<<7)|(3<<3)|2; // use A/D RC oscillator, right justified result
  ADCON1 = (3<<4); // VSS is negative reference, 4.095V is positive reference
  ADCON0 = 1|(2<<2); // ADC on, use input AN2

  CCPR3H = ((LCD_PERIOD)/2)>>8;
  CCPR3L = ((LCD_PERIOD)/2);
  CCPR5H = ( LCD_PERIOD   )>>8;
  CCPR5L = ( LCD_PERIOD    );
  lcd_update_contrast();

  lcd_digits[0] = ' ';
  lcd_digits[1] = ' ';
  lcd_digits[2] = ' ';
  lcd_digits[3] = ' ';
  dp = 0;
  lcd_update_state(flip);

  PIR3bits.CCP2IF = 0;
  PIE3bits.CCP2IE = 1;
  PIR4bits.CCP3IF = 0;
  PIE4bits.CCP3IE = 1;
  PIR4bits.CCP4IF = 0;
  PIE4bits.CCP4IE = 1;
  PIR4bits.CCP5IF = 0;
  PIE4bits.CCP5IE = 1;
  CCP2CONbits.CCP2M = 10;
  CCP3CONbits.CCP3M = 10;
  CCP4CONbits.CCP4M = 10;
  CCP5CONbits.CCP5M = 11;
  T1CONbits.TMR1ON = 1;

  ANCON0bits.ANSEL0 = 0;
  TRISAbits.TRISA0 = 0;
  LATAbits.LATA0 = 0;
  INTCONbits.GIEH = 1;
  INTCONbits.PEIE = 1;

  TRISA = 0x1c;
  TRISB = 0;
  TRISC = 0;
  TRISD = 0;
  TRISE = 0xf8;

  baseline2 = ReadEEPROMLong(&eeCalibration[0]);
  baseline3 = ReadEEPROMLong(&eeCalibration[1]);
  voltCalRatio = ReadEEPROMLong(&eeCalibration[2]);
  flip = ReadEEPROMByte(&eeFlip);
  if( !PORTEbits.RE3 ) {
    flip = !flip;
    WriteEEPROMByte(&eeFlip, flip);
    button = 1;
    buttonunpress = 0;
    debounce = 0xff;
  }
  if( !baseline2 ) {
    mode = 4;
    voltcal = 0;
  } else {
    mode = 0;
    for( i = 0; i < 127; ++i )
      mode += ReadEEPROMByte(&eeMode[i]);
    mode %= 4;
  }

  while(1) {
	signed short uamps, mamps, volts;
    unsigned short temp_ms_timer;
	unsigned long ltemp;

	volts = get_adc_volts_reading(256);
    if( mode != 4 )
      volts = (volts * voltCalRatio + 32768) >> 16;
/*
    uamps = (((signed long)get_adc_reading(3, NUM_AVGS) - (signed long)adc_reading_to_actual_Vin(baseline3, volts, NUM_AVGS)) + (NUM_AVGS/8)) / (NUM_AVGS/4);
    mamps = (((signed long)get_adc_reading(2, NUM_AVGS) - (signed long)adc_reading_to_actual_Vin(baseline2, volts, NUM_AVGS)) + (NUM_AVGS/8)) / (NUM_AVGS/4);
    uamps = (((signed long)uamps * voltCalRatio + 32768) >> 16);
    mamps = (((signed long)mamps * voltCalRatio + 32768) >> 16);
*/
    uamps = ((((signed long)get_adc_reading(3, NUM_AVGS) - (signed long)adc_reading_to_actual_Vin(baseline3, volts, NUM_AVGS)) + (NUM_AVGS/8)) * (voltCalRatio / (NUM_AVGS/4)) + 32768)>>16;
    mamps = ((((signed long)get_adc_reading(2, NUM_AVGS) - (signed long)adc_reading_to_actual_Vin(baseline2, volts, NUM_AVGS)) + (NUM_AVGS/8)) * (voltCalRatio / (NUM_AVGS/4)) + 32768)>>16;

    // ms_timer actually increments at 200Hz
	temp_ms_timer = ms_timer;
    ms_timer = 0;
    // mamps = 10000 corresponds to 1A, uamps = 1000 corresponds to 1mA, volts = 100 corresponds to 1V
    if( uamps > 9999 ) {
      // 200mA / 5V gives 1W for 1/3600Wh per second
      // 2000 x 500 = 1E6 so we want mamps x volts x ms_timer / (200 * 3600 / 40) to give uamp hours per second, giving a division factor of 18000
      if( mamps > 0 )
        fortieths_of_microamp_hours += (unsigned long)mamps * (unsigned long)volts * (unsigned long)temp_ms_timer / 18000UL;
    } else {
      if( uamps > 0 )
        fortieths_of_microamp_hours += (unsigned long)uamps * (unsigned long)volts * (unsigned long)temp_ms_timer / 1800000UL;
    }
    // wrap around at exactly 100Wh
    if( fortieths_of_microamp_hours >= 4000000000UL )
      fortieths_of_microamp_hours -= 4000000000UL;

	if( mode == 0 ) {
		signed short result;
		unsigned char c, amps = 0;
		static signed short uamp_readings[UAMP_AVGS], num_uamp_readings = 0, i;
		if( uamps > 9999 ) {
	      if( mamps < 0 ) {
	        result = 0;
	        dp = 1;
		  } else if( mamps > 9999 ) {
			mamps /= 10;
	        dp = 1;
			amps = 1;
		  } else {
	        dp = 3;
	      }
		  result = mamps;
          if( mamps )
            num_uamp_readings = 0;
	    } else {
          dp = 1;
          if( num_uamp_readings == UAMP_AVGS ) {
            memmove(uamp_readings+0, (const void*)(uamp_readings+1), sizeof(uamp_readings)-sizeof(*uamp_readings));
            uamp_readings[UAMP_AVGS-1] = uamps;
          } else {
            uamp_readings[num_uamp_readings++] = uamps;
          }
          result = 0;
          for( i = 0; i < num_uamp_readings; ++i )
            result += uamp_readings[i];
          result = (result + (num_uamp_readings>>1)) / num_uamp_readings;
          if( result < 0 )
            result = 0;
          if( uamps < 0 )
            uamps = 0;
          if( absdiff(result, uamps) > 9 ) {
            result = uamps;
            uamp_readings[0] = uamps;
            num_uamp_readings = 1;
          }
	    }
		lcd_digits[3] = '0'+(result%10);
		result /= 10;
		lcd_digits[2] = result || dp                 ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[1] = result || dp == 1 || dp == 3 ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[0] = result || dp == 1            ? '0'+(result) : ' ';
        if( amps ) {
          if( mamps > 15000 ) {
            lcd_digits[0] = ' ';
            lcd_digits[1] = 'o';
            lcd_digits[2] = 'L';
            lcd_digits[3] = ' ';
          } else {
            lcd_digits[3] = 'A';
          }
        }
	} else if( mode == 1 || mode == 4 ) {
		unsigned short result = volts;
        if( mode == 4 )
          result += voltcal;
		dp = 2;
		lcd_digits[3] = '0'+(result%10);
		result /= 10;
		lcd_digits[2] = result || dp                 ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[1] = result || dp == 1 || dp == 3 ? '0'+(result%10) : ' ';
		lcd_digits[0] = mode == 4 ? 'C' : 'b';
        if( mode == 4 && (++flash)&1 ) {
//          lcd_digits[0] = lcd_digits[1] = lcd_digits[2] = lcd_digits[3] = ' ';
          dp = 0;
        }
	} else if( mode == 2 ) {
		unsigned long result;
		char prefix;
		if( uamps > 9999 ) {
	      if( mamps < 0 ) {
	        result = 0;
		  } else {
            // compensate for I^2R losses in 50m# shunt
            result = ((unsigned long)mamps * (unsigned long)volts) - ( (unsigned long)mamps * (unsigned long)mamps / (1000*20) );
		  }
	    } else if( uamps < 0 ) {
	      result = 0;
	    } else {
          static signed short uamp_readings[UAMP_AVGS], num_uamp_readings = 0, i;

          if( num_uamp_readings == UAMP_AVGS ) {
            memmove(uamp_readings+0, (const void*)(uamp_readings+1), sizeof(uamp_readings)-sizeof(*uamp_readings));
            uamp_readings[UAMP_AVGS-1] = uamps;
          } else {
            uamp_readings[num_uamp_readings++] = uamps;
          }
          result = 0;
          for( i = 0; i < num_uamp_readings; ++i )
            result += uamp_readings[i];
          result = (result + (num_uamp_readings>>1)) / num_uamp_readings;
          if( result < 0 )
            result = 0;
          if( uamps < 0 )
            uamps = 0;
          if( absdiff(result, uamps) > 9 ) {
            result = uamps;
            uamp_readings[0] = uamps;
            num_uamp_readings = 1;
          }

	      result = result * volts / 100;
	    }
		prefix = 'P';
		if( result < 10000UL ) {
			result /= 10;
			dp = 2;
			prefix = 'L';
		} else if( result < 1000000UL ) {
			result /= 1000;
			dp = 1;
		} else if( result < 10000000UL ) {
			result /= 10000;
			dp = 2;
		} else {
			result /= 100000;
			dp = 3;
		}
		lcd_digits[3] = '0'+(result%10);
		result /= 10;
		lcd_digits[2] = result || dp                 ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[1] = result || dp == 1 || dp == 3 || prefix == 'L' ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[0] = prefix;
	} else if( mode == 3 ) {
        unsigned long result;
        unsigned char prefix;
        if( fortieths_of_microamp_hours < 4000000UL ) {
          result = fortieths_of_microamp_hours / 40UL;
          prefix = 'F';
        } else {
          result = fortieths_of_microamp_hours / 40000UL;
          prefix = 'E';
        }
        dp = 1;
        while( result >= 1000UL ) {
          result /= 10UL;
          if( ++dp > 3 )
            dp = 0;
        }
		lcd_digits[3] = '0'+(result%10);
		result /= 10;
		lcd_digits[2] = result || dp                 ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[1] = result || dp == 1 || dp == 3 || prefix == 'L' ? '0'+(result%10) : ' ';
		result /= 10;
		lcd_digits[0] = prefix;
	}

    lcd_update_state(flip);

    if( buttonpress && !button ) {
		button = 1;
        if( mode != 4 ) {
		  if( ++mode == 4 )
            mode = 0;
          IncrementModeInEEPROM();
        }      
	} else if( buttonunpress && button ) {
		button = 0;
        if( mode == 4 ) {
          if( voltcal <= 0 )
            voltcal = -voltcal + 1;
          else
            voltcal = -voltcal;
          if( voltcal == 51 )
            voltcal = 0;
        }
	}
    buttonpress = buttonunpress = 0;
	slow_debounce = (slow_debounce<<1)|(debounce==0xFF);
	if( slow_debounce == 0xFF ) {
		slow_debounce = 0;
        if( mode == 4 ) {
          voltCalRatio = ((volts + voltcal) * 0x10000UL + (volts/2)) / volts;
          WriteEEPROMLong(&eeCalibration[2], voltCalRatio);

          lcd_digits[0] = 'C';
          lcd_digits[1] = 'A';
          lcd_digits[2] = 'L';
          lcd_digits[3] = 'I';
          dp = 0;
          lcd_update_state(flip);

          volts = (get_adc_volts_reading(NUM_AVGS) * voltCalRatio + 32768) >> 16;
          baseline2 = (adc_reading_to_nominal_5V(get_adc_reading(2, NUM_AVGS*NUM_CAL_AVGS), volts, NUM_AVGS*NUM_CAL_AVGS)+(NUM_CAL_AVGS/2))/NUM_CAL_AVGS;
          baseline3 = (adc_reading_to_nominal_5V(get_adc_reading(3, NUM_AVGS*NUM_CAL_AVGS), volts, NUM_AVGS*NUM_CAL_AVGS)+(NUM_CAL_AVGS/2))/NUM_CAL_AVGS;
          WriteEEPROMLong(&eeCalibration[0], baseline2);
          WriteEEPROMLong(&eeCalibration[1], baseline3);
          fortieths_of_microamp_hours = 0;

          mode = 0;
		} else {
          mode = 4;
          voltcal = 0;
        }      
	}
  }
}


unsigned char ReadEEPROMByte(const rom unsigned char* Address) {
  EEADRH = (unsigned char)(((unsigned int)Address)>>8); // Load the high byte of the EEPROM address
  EEADR = (unsigned char)(unsigned int)Address; // Load the low byte of the EEPROM address
  EECON1bits.EEPGD = 0;
  EECON1bits.CFGS = 0;
  EECON1bits.RD = 1; // Do the read
  return EEDATA; // Return with the data
}

void WriteEEPROMByte(rom unsigned char* Address, unsigned char Data) {
  unsigned char GIE_Status; // Variable to save Global Interrupt Enable bit

  EEADRH = (unsigned char)(((unsigned int)Address)>>8); // Load the high byte of the EEPROM address
  EEADR = (unsigned char)(unsigned int)Address; // Load the low byte of the EEPROM address
  EEDATA = Data; // Load the EEPROM data
  EECON1bits.EEPGD = 0;
  EECON1bits.CFGS = 0;
  EECON1bits.WREN = 1; // Enable EEPROM writes
  GIE_Status = INTCONbits.GIE; // Save the Global Interrupt Enable bit
  INTCONbits.GIE = 0; // Disable global interrupts
  EECON2 = 0x55; // Required sequence to start the write cycle
  EECON2 = 0xAA; // Required sequence to start the write cycle
  EECON1bits.WR = 1; // Required sequence to start the write cycle
  INTCONbits.GIE = GIE_Status; // Restore the Global Interrupt Enable bit
  EECON1bits.WREN = 0; // Disable EEPROM writes
  while (EECON1bits.WR); // Wait for the write cycle to complete
}

typedef union {
  unsigned long l;
  unsigned char c[4];
} cl;

unsigned long ReadEEPROMLong(const rom unsigned long* Address) {
  cl temp;
  const rom unsigned char* addr = (const rom unsigned char*)Address;
  temp.c[0] = ReadEEPROMByte(addr++);
  temp.c[1] = ReadEEPROMByte(addr++);
  temp.c[2] = ReadEEPROMByte(addr++);
  temp.c[3] = ReadEEPROMByte(addr);
  return temp.l;
}

void WriteEEPROMLong(rom unsigned long* Address, unsigned long Data) {
  cl temp = { Data }; 
  rom unsigned char* addr = (rom unsigned char*)Address;
  WriteEEPROMByte(addr++, temp.c[0]);
  WriteEEPROMByte(addr++, temp.c[1]);
  WriteEEPROMByte(addr++, temp.c[2]);
  WriteEEPROMByte(addr,   temp.c[3]);
}
