; Heart rate monitor program HEART.ASM
; 
;
; Processor pin allocations are as follows:

; RA0 Output driving LED bar graph
; RA1 Output disp 3
; RA2 Output disp 1
; RA3 Output disp 2
; RA4 Input from switches

; RB0 Input from heart pulse sensor (interrupt)
; RB1 e segment drive, 
; RB2 d segment drive, 
; RB3 c segment drive, piezo drive
; RB4 b segment drive, 
; RB5 a segment drive, 
; RB6 f segment drive, 
; RB7 g segment drive, 

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

; EEPROM DATA

EEPROM1		equ	H'00'	; timer store
EEPROM2		equ	H'01'	; heart beat minimum
EEPROM3		equ	H'02'	; heart beat maximum

; RAM

DISP1		equ	H'0C'	; working storage for Display1 value 
DISP2		equ	H'0D'	; working storage for Display2 value 
DISP3		equ	H'0E'	; working storage for Display3 value 
DISP4		equ	H'0F'	; working storage for Display4 bargraph
STATUS_TMP 	equ 	H'10'	; temp storage for status during interrupt
W_TMP		equ	H'11'	; temporary storage for w during interrupt
VALUE_1		equ 	H'12'	; delay value storage			
VALUE_2		equ	H'13'	; delay value storage
FLAG_1		equ	H'14'	; bit 0 error flag, bit 1 1-second update
				; bit2 multiplex indicator
				; bit3 closed switch indicator, 
				; bit5 is pulse count update, 
FLAG_2		equ	H'15'	; overrange flag bit 0, alarm bit1, timer run bit2, force stop 3
CNT_8		equ	H'16'	; BCD conversion counter 
TEMP		equ	H'17'	; temporary storage during BCD conversion
TEMP_1		equ	H'18'	; temporary working storage
TEMP_2		equ	H'19'	; temp storage
HEART_1		equ	H'1A'   ; current heart rate reading
TIME_CNT1	equ	H'1B'	; timer seconds
TIME_CNT2	equ	H'1C'	; timer minutes
TIME_CNT3	equ	H'1D'	; time out alarm 
HBMIN		equ	H'1E'	; minimum heart beat rate
HBMAX		equ	H'1F'	; maximum heart beat rate
DISP		equ	H'20'	; display (heart, timer mins or sec)  
MODE_1		equ	H'21'	; mode indicator 
MODE_2		equ	H'22'	; mode indicator 

VALID		equ	H'23'	; temp storage 
PULSE_CNT	equ	H'24'	; pulse counter 
BCD_0		equ	H'25'	; bcd MS value after conversion
BCD_1		equ	H'26'	; bcd LS value
BIN_0		equ	H'27'	; binary value for BCD conversion
UPDATE1		equ	H'28'	; count up for 1 second counter
UPDATE2		equ	H'29'	; count up for 1 second counter
HBEATA		equ	H'2A'	; heart beat counter
HBEATB		equ	H'2B'	; heart beat counter
HBEATC		equ	H'2C'	; heart beat storage value
HBEATD		equ	H'2E'	; heart beat storage value
ACCaLO		equ	H'2F'	; divide routine denominator
ACCaHI		equ	H'30'	; divide routine denominator
ACCbLO		equ	H'31'	; divide routine numerator
ACCbHI		equ	H'32'	; divide routine numerator
ACCcLO		equ	H'33'	; divide routine remainder
ACCcHI		equ	H'34'	; divide routine remainder
ACCdLO		equ	H'35'	; divide routine ACCbLO storage
ACCdHI		equ	H'36'	; divide routine ACCbHI storage
ALARM_T		equ	H'37'	; alarm time duration
DIV_FLG		equ	H'38'	; heart beat counter, bit 0, bit 1 count bit
SCAN_TM		equ	H'39'	; scan timer for set LEDs during timing
ALM_FLS		equ	H'3A'	; bit 7 is on/off flag for alarm source indicator set LEDs
MODE_S		equ	H'3B'	; mode_2 storage for flashing set LEDs during alarm
REC_0		equ	H'3C'	; -0 minutes recovery heart beat rate
REC_1		equ	H'3D'	; -1 minutes recovery heart beat rate
REC_2		equ	H'3E'	; -2 minutes recovery heart beat rate
REC_3		equ	H'3F'	; -3 minutes recovery heart beat rate
REC_4		equ	H'40'	; -4 minutes recovery heart beat rate
REC_5		equ	H'41'	; -5 minutes recovery heart beat rate
DELOP		equ	H'42'	; delay loop
REC_FLG		equ	H'43'	; recovery flag bit 0 to stop Err after 5 mins, Bit 1 allows alarm every minute 
MIN_CNT		equ	H'44'	; recovery minute counter
FLAG_3		equ	H'45'	; initialisation has occurred for error. bit 0

; preprogram EEPROM DATA (00-3F from 2100-213F)
	
	ORG     2100		; start at 00
	
	DE	D'30', D'130', D'160' ; timer 30minutes,heart beat min 130 heart beat max 160

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data.  
; RB7-RB1----gfabcde
 
SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'11100110'	; 7-segment code 1
	retlw 	B'01001000'	; 7-segment code 2
	retlw 	B'01000011'	; 7-segment code 3
	retlw 	B'00100110'	; 7-segment code 4
	retlw 	B'00010010'	; 7-segment code 5
	retlw 	B'00010000'	; 7-segment code 6
	retlw 	B'11000110'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00000010'	; 7-segment code 9
	retlw	B'01111110'	; 7-segment code "-"
	retlw	B'10001100'	; 7-segment code "R"
	retlw 	B'00011000'	; 7-segment code "E"
	retlw 	B'00010010'	; 7-segment code "S"
	retlw	B'10011000'	; 7-segment code "C"

;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3, Disp4 in sequence plus 
; blanking leading zeros on display for Disp2 and Disp3 
; uses internal timer to initiate display update
; also updates heart rate

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf	STATUS,RP0	; select memory bank 0

; which interrupt (INT or TMRO)

	btfsc	INTCON,T0IF	; TMRO overflow interrupt flag 
	goto	LIGHT		; TMRO overflow so multiplex display
HEART_T btfsc	INTCON,INTF	; INT interrupt flag set then goto heart
	goto	HEART		; heart beat signal update 
	goto	RECLAIM		; end of interrupt reclaim w and status


LIGHT	bsf	FLAG_1,2	; multiplex flag	
							
; update time period counters for clock

	movf	TMR0,w		; timer value	
				; freq is 4MHz/4/4/(256(-8+2))=1kHz
	nop
	nop			; synchronise counter read with divide by 4 ie add two cycles
	addlw	0x8		; add to timer register and takes 2 cycles to start counting
	movwf	TMR0		; place in timer (the two cycles plus 2-cycle delay = 4cycles)
	
	bcf	INTCON,T0IF	; clear TMRO interrupt flag
	incfsz	UPDATE1,f	; increase counter 1 skip if overflowed
	goto 	CHK_UPD		; goto check update time
	incf	UPDATE2,f	; increase counter 2 when counter 1 overflows
 	
CHK_UPD	movf	UPDATE2,w	; MS Byte of update counter to w
	xorlw	0x03		; 03 compare MS Byte counter with (1000 decimal for 1 second update)
	btfss	STATUS,z	; skip if equal
	goto	HB_CNTR		; not, so continue and multiplex display
	movf	UPDATE1,w	; LS Byte of update counter
	xorlw	0xE8		; 000-3E8=1000 compare LS byte counter with (1000) Using E8 as 
				; we increment update1 after clearing so miss 1 count
	btfss	STATUS,z	; skip if equal
	goto	HB_CNTR		; not 1 second yet
	clrf	UPDATE1		; clear timer counter 1
	clrf	UPDATE2		; and timer 2
	incf	UPDATE1,f	; increase by 1 so if decremented in adjust routine will
				; not go to FF
	bsf	FLAG_1,1	; set bit 1 for 1 second flag


; heart beat counter

HB_CNTR	btfss	DIV_FLG,1
	goto	DIV_NO		; count every second time
	bcf	DIV_FLG,1
	incfsz	HBEATA,f	; increase heart beat counter
	goto	TST_ERR		; test error
	incf	HBEATB,f	; increase when HBEATA overflowed
TST_ERR	movlw	H'09'		; compare with >2304 or H 900 for pulse time longer than 4.6s
	subwf	HBEATB,w	; subtract w from f
	btfss	STATUS,c	; if carry is 1 then HBEATB is larger 
	goto	MPX_DS1
	bsf	FLAG_1,0	; set error flag
	goto	MPX_DSP
DIV_NO	bsf	DIV_FLG,1
	goto	MPX_DSP

; display update

MPX_DS1	bcf	FLAG_1,0
	bcf	FLAG_3,0	; clear error initialisation flag

MPX_DSP	btfss	FLAG_1,0	; if error flag clear continue
	goto 	MPX_CNT
	movf	MODE_1,w
	btfss	STATUS,z
	goto	MPX_CNT
	btfsc	REC_FLG,0	; after 5- minutes recovery flag if set show no error
	goto	MPX_CNT
	btfsc	FLAG_3,0	; has error initialisation occurred
	goto	CLR_BT		; if so then bypass initilisation
	clrf	DISP		; set at HB display mode
	bcf	DISP4,5		; heart beat on display
	bsf	DISP4,6		; timer off if error
	btfsc	REC_FLG,1	; is recovery flag set
	goto	CLR_BT		; clear beat, show error but not stop timer	
	bsf	FLAG_2,1	; alarm flag
	bcf	FLAG_2,2	; clear timer run flag (clear to stop timer)
	clrf	UPDATE1		; clear timer counter 1
	clrf	UPDATE2		; and timer 2
	btfss	FLAG_2,4	; was alarm already set
	movlw	0x00		; value to show HB min set LED
	movwf	MODE_2
	movwf	MODE_S
	bsf	FLAG_3,0	; set already error initialisation flag
CLR_BT	clrf	HBEATC
	clrf	HBEATD		; heart beat values cleared if error
	clrf	HEART_1
	movlw	0x0B		; seven segment value for r (as in err for error)
	movwf	DISP2
	movwf	DISP1
	movlw	0x0C		; seven segment value for e as in error
	movwf	DISP3
		
MPX_CNT	btfss	PORTA,2		; skip if display 1 not lit
	goto	LT1		; display 1 lit
	btfss	PORTA,3		; skip if display 2 not lit
	goto	LT2		; display 2 lit
	btfss	PORTA,1		; skip if display 3 not lit				
	goto	LT3		; display 3 lit
	btfss	PORTA,0		; skip if display 4 not lit
	goto	LT4		; display 4 lit
	
LT1	bsf	PORTA,2		; disp1 off
	movf	DISP2,w		; look at display2
	btfss	STATUS,z	; skip if 0
	goto	NOBLNK1		; do not blank disp2
	movf	DISP3,f		; display 3
	btfss	STATUS,z	; skip if 0
	goto	NOBLNK1		; do not blank disp2
	movlw	B'11111110'	; 7-segment display off
	goto	BLANK1		; blank disp2 when disp2 and disp3 = 0
NOBLNK1	call 	SVNSEG		; goto subroutine to get seven segment code 
BLANK1	movwf	PORTB		; seven segment value to portB
	bcf	PORTA,3		; disp2 powered
	goto	HEART_T		; end of multiplex
LT2	bsf	PORTA,3		; disp2 off
	movf	DISP3,w		; disp3 to w
	btfss	STATUS,z	; check if 0
	goto	NOBLNK2
	movlw	B'11111110'	; display off
	goto	BLANK2
NOBLNK2	call	SVNSEG		; seven segment code
BLANK2	movwf	PORTB		; portB lights display
	bcf	PORTA,1		; disp3 powered
	goto	HEART_T		; end of multiplex
LT3	bsf	PORTA,1		; disp3 off
	movf	DISP4,w		; disp4 to w
	movwf 	PORTB

; alarm driver

	incfsz	ALARM_T,f	; alarm timer
	goto 	CHK_2
	decf	ALARM_T,f	; stays at FF unless cleared by alarm
CHK_2	btfsc	REC_FLG,1	; recovery flag set
	goto	ZRO_ALM
	movf	TIME_CNT2,w	; timer
	btfsc	STATUS,z	; is it zero then alarm
	goto	ZRO_ALM
	btfsc	FLAG_2,3	; force stop
	goto	CLR_ALM
ZRO_ALM	movlw	0xFF		; alarm duration time
	subwf	ALARM_T,w	; alarm timer
	btfsc	STATUS,c	; check if negative
	goto	CLR_ALM
ALMCONT	btfsc	REC_FLG,1
	goto	ALM_BY		; bypass alarm flag	
	btfss	FLAG_2,1	; alarm flag
	goto	CLR_ALM		; alarm signal off
ALM_BY	btfsc	ALARM_T,6	; test bit
	goto	CLR_ALM
	bcf	PORTB,3		; clear (set alarm)
	goto	HBLED
CLR_ALM	bsf	PORTB,3		; set (clear alarm)

; heart beat LED flasher 

HBLED	btfss	PORTB,0		; check for heart beat input high or low
	goto 	SET_LED	
	bcf	PORTB,4		; set so clear
	goto	CLR_LED
SET_LED	bsf	PORTB,4
CLR_LED	bcf	PORTA,0		; disp4 powered

; timer LED flasher

	incf	SCAN_TM,f	; timer scan rate for set LEDs
	btfss	FLAG_2,3	; force stop flag set?
	goto	NXT_LED		; LEDs off if stopped
	movf	MODE_1,w
	btfss	STATUS,z	; if zero then leds off
	goto	HEART_T
	bsf	PORTB,2		; timer set LED off
	bsf	PORTB,1		; heart beat max LED off
	bsf	PORTB,7		; heart beat min LED off
	goto	HEART_T
NXT_LED	movf	MODE_1,w
	btfss	STATUS,z	; if zero then leds on
	goto	HEART_T

; show alarm cause (LED)

	btfss	FLAG_2,1	; alarm flag
	goto	NO_ALM
	btfss	SCAN_TM,5	; check for high bit
	goto	MOD_LED
	clrf	SCAN_TM		; scan rate counter
CHK_7	btfss	ALM_FLS,7	; (bit 7 on/off flash)
	goto	SET_7
	bcf	ALM_FLS,7
	movf	MODE_2,w
	movwf	MODE_S		; store MODE_2
	movlw	0x03		; clears all set LEDs
	movwf	MODE_2
	goto	MOD_LED
SET_7	bsf	ALM_FLS,7
	movf	MODE_S,w
	movwf	MODE_2
	goto	MOD_LED		; MODE_2 LED display

NO_ALM	btfss	SCAN_TM,5	; check for bit high
	goto	MOD_LED
	incf	MODE_2,f	; increase for display set LEDs
	clrf	SCAN_TM
MOD_LED	btfsc	MODE_2,0	; is bit 0 high
	goto	ONE_SET		; bit 0 set
	btfsc	MODE_2,1	; check bit 1
	goto	TWO_SET
	bcf	PORTB,7		; hb min LED on
	goto	HEART_T	
ONE_SET	btfsc	MODE_2,1	; check bit 1 
	goto	X_SET		; both 0 & 1 bits set
	bcf	PORTB,1		; hb max LED
	goto	HEART_T
TWO_SET	bcf	PORTB,2		; timer set LED on
	goto	HEART_T
X_SET	bsf	PORTB,7		; HB min LED off
	bsf	PORTB,2		; timer LED off
	bsf	PORTB,1		; HB max LED off
	goto	HEART_T

LT4	bsf	PORTA,0		; display 4 off
	movf	DISP1,w		; display 1 to w
	call	SVNSEG		; seven segment code
	movwf	PORTB
	bcf	PORTA,2		; display 1 powered
	goto	HEART_T

; heart beat signal update

HEART	bcf	INTCON,INTF	; clear INT flag sensor input
	btfss	DIV_FLG,0
	goto	NO_CNT
	bcf	DIV_FLG,0
	movf	HBEATA,w	; heart beat counter
	movwf	HBEATC		; heart beat store
	movf	HBEATB,w	; heart beat counter
	movwf	HBEATD		; heart beat store
	bcf	FLAG_1,0	; clear error flag
	bsf	FLAG_1,5	; heart beat update flag
	clrf	HBEATA
	clrf	HBEATB		; clear counters at every heart beat
	goto	RECLAIM
NO_CNT	bsf	DIV_FLG,0

; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  	
; RESET		
; Set ports A & B

MAIN	movlw	0xFF
	movwf	ALARM_T		; alarm timer
	bsf	ALM_FLS,7	; alarm set LED flash flag
	clrf	MODE_S		; mode_2 storage
	clrf	DIV_FLG		; division flags
	clrf	FLAG_1		; clear flags
	clrf	FLAG_2
	clrf	REC_FLG		; recovery flag
	bsf	FLAG_2,3	; set forced stop flag
	clrf	HBEATA		; clear heart beat counter
	clrf	HBEATB		; clear heart beat counter
	clrf	TIME_CNT1	; seconds counter
	clrf	DISP1		; clear display registers
	clrf	DISP2
	clrf	DISP3
	clrf	DISP		; heart beat display
	clrf	MODE_1		; set mode to heart beat/timer display
	clrf	MODE_2		; LED set storage during alarm
	clrf	HEART_1		; heart beat
	clrf	FLAG_3		; initialisation flag for error bit 0
	movlw	0xFF		; 
	movwf	DISP4		; clear LEDs
	bcf	DISP4,5		; set heart LED indicator
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'11000001'	; w = 11000001 binary 
	movwf	OPTION_REG	; TMRO prescaler /4 division, PORTB pullups disabled
	movlw   B'00010000'	; w = 10000 binary (RA0-RA3 outputs, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00001111'
	movwf	PORTA		; portA RA0,RA1,RA2,RA3 outputs high 
	
; read data values
	movlw	EEPROM1		; alarm time
	call 	EEREAD
	movwf	TIME_CNT3	; alarm time ram
	movwf	TIME_CNT2	; timer
	clrf	TIME_CNT1	; seconds timer
	movlw	EEPROM2		; heart beat min
	call	EEREAD
	movwf	HBMIN		; min heart rate ram
	movlw	EEPROM3		; heart beat max
	call 	EEREAD
	movwf	HBMAX		; max heart rate ram

; allow interrupts
	bsf	INTCON,INTE	; set interrupt enable for RB0
	bsf	INTCON,GIE	; set global interrupt enable for above
	
; check for update flags
	
	
TO_DO	bsf	INTCON,T0IE	; set interrupt
	btfss	FLAG_1,1	; 1 sec flag if set then continue
	goto	BEAT		; not updated yet so out
	btfsc	FLAG_2,3	; force stop
	goto	BEAT
	btfss	FLAG_2,2	; timer run flag
	goto	BEAT
	bcf	FLAG_1,1	; clear 1 sec flag
	movlw	0x01		; 1 minute left
	subwf	TIME_CNT2,w	; test if minutes are 1
	btfsc	STATUS,z
	goto	SECS		; minutes down to 1 so show seconds
	movf	TIME_CNT1,w	; test if zero
	movlw	D'60'		; ready to load 60 if was zero
	btfsc	STATUS,z	; if zero then load with 59
	movwf	TIME_CNT1	
	decfsz	TIME_CNT1,f	; decrement clock (60 to 59)
	goto	MOR_MIN
	decf	TIME_CNT2,f	; decrease minutes counter
	movlw	D'60'
	movwf	TIME_CNT1	; set to 60	
MOR_MIN	btfsc	DISP4,6		; is minute timer LED on
	goto	BEAT
	bsf	DISP4,5		; heart beat LED off if on	
	movf	TIME_CNT2,w	; minutes time 
	call 	INST		; place in display
	goto	BEAT
L_HI	movlw	D'60'		; seconds
	movwf	TIME_CNT1
SECS	movf	TIME_CNT1,w	; check if already 0
	btfsc	STATUS,z	; if 0 then load 59
	goto	L_HI
	decfsz	TIME_CNT1,f	; decrease seconds
	goto	SEC_SHW
	bsf	FLAG_2,1	; alarm flag
	bcf	FLAG_2,2	; timer run flag
	bsf	FLAG_2,3	; force stop
	clrf	ALARM_T		; timer for alarm
	clrf	TIME_CNT2	; zero minutes
	goto	RECOVRY		; recovery routine

SEC_SHW	btfsc	DISP4,6		; is timer LED on
	goto	BEAT
	bsf	DISP4,5		; heart beat LED off if on
	movf	TIME_CNT1,w	; seconds time
	call	INST		; place in display
	
BEAT	btfss	FLAG_1,5	; is heart beat flag set
	goto	SW_CHK		; update heart beat only when flag set
	bcf	FLAG_1,5	; clear heart beat flag
	movf	HBEATD,w
	movwf	ACCaHI		; for divide
	btfsc	STATUS,z	; if zero then ok not overrange
	goto	OVER		; otherwise load --- for overrange in display
	bcf	FLAG_2,0	; clear flag for overrange
	movf	HBEATC,w	; heart beat counter
	movwf	ACCaLO		; for divide
	
	call	D_DIV		; divide routine
	movf	ACCbLO,w	; result
	movwf	HEART_1
	incfsz	HEART_1,w	; increase by 1 (if 255 then becomes 0)
	goto	MIN_MAX
	clrf	HEART_1		; make a zero

; check heart rate with HB min and HB max

MIN_MAX	bcf	FLAG_2,4	; already set alarm flag
	btfsc	FLAG_2,1	; is alarm flag already set
	bsf	FLAG_2,4	; alarm already set flag
	bcf	FLAG_2,1	; clear alarm flag
	movf	HEART_1,w	; heart beat 
	subwf	HBMAX,w		; subtract heart beat from max setting
	btfsc	STATUS,c	; if c is 0 then larger so alarm
	goto	MIN_CHK
	bsf	FLAG_2,1	; set alarm flag
	btfss	FLAG_2,4	; was alarm already set
	clrf	ALARM_T		; alarm timer
	movlw	0x01		; value to show HB max set LED
	movwf	MODE_2
	movwf	MODE_S
	goto	B_DSP
MIN_CHK	movf	HBMIN,w
	subwf	HEART_1,w	; subtract min setting from heart beat
	btfsc	STATUS,c	; if c is 0 then smaller so alarm
	goto    S_ALM
	bsf	FLAG_2,1	; alarm flag
	bcf	FLAG_2,2	; clear timer run flag (clear to stop timer)
	clrf	UPDATE1		; clear timer counter 1
	clrf	UPDATE2		; and timer 2
	btfss	FLAG_2,4	; was alarm already set
	clrf	ALARM_T		; alarm time
	movlw	0x00		; value to show HB min set LED
	movwf	MODE_2
	movwf	MODE_S
 	goto	B_DSP
S_ALM	bsf	FLAG_2,2	; clear timer run flag (set to run)		
 
; heart beat to display

B_DSP	btfsc	DISP4,5		; is heart beat on display
	goto	SW_CHK
	bsf	DISP4,6		; turn off timer LED if on
	movf	HEART_1,w	; get value
	call 	INST		; update heart beat value to display
	goto	SW_CHK
OVER	bsf	FLAG_2,0	; flag for overrange on heart beat
	bcf	DISP4,5		; set HB LED
	bsf	DISP4,6		; timer LED off
	call	INST		; overrange so show HB overrange --- dashes
	clrf	HEART_1
	clrf	ALARM_T		; alarm time
	movlw	0x03		; no LEDs on SET displays
	movwf	MODE_2
	clrf	DISP		; return to HB display mode

; check if a switch is pressed
	
SW_CHK	bcf	FLAG_1,2	; clear multiplex update flag
MULTIP	btfss	FLAG_1,2	; check if multiplex update occured
	goto	MULTIP		; wait for new multiplex update for display
	btfsc	PORTA,4		; if a switch is pressed, then bit 4 will be low
	goto	TO_DO		; no switch pressed so look again
	bcf	INTCON,T0IE	; clear interrupt to prevent incorrect switch result
	movlw	B'11111110'	; all seven segments off
	movwf	PORTB		; display off
	movf	PORTA,w		; look at RA0-RA2
	btfsc	PORTA,4		; if a switch is pressed, then bit 4 will be low
	goto	TO_DO		; no switch pressed return
	btfsc	PORTA,2		; is it the mode switch	
	goto	D_U_CHK
	goto 	MODE
D_U_CHK	btfsc	PORTA,1		; is it the Down switch
	goto	U_CHK
	goto	DOWN
U_CHK	btfsc	PORTA,3		; is it the UP switch
	goto	TO_DO		; no switch reading
	goto 	UP
MODE	incf	MODE_1,f	; cycle through modes
	movf	MODE_1,w
	sublw	0x03		; if negative then clear mode_1
	btfsc	STATUS,c	; c=0 if negative (mode_1 > 3)
	goto	MOD_TST
CLR_MD	clrf	MODE_1		; mode_1 clear so back to heart beat mode
MOD_TST	movf	MODE_1,w	; heart beat display mode
	btfsc	STATUS,z	; zero if heart beat display mode
	goto	HART_TM 	; heart/timer display mode
	movlw	0x03		; set timer mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	SET_TMR		; set timer mode
	movlw	0x01		; check which mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	SET_MIN		; set min HB mode
	movlw	0x02		; check which mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	SET_MAX		; set max HB mode
	goto	CLR_MD

; show heartbeat or timer display

HART_TM	clrf	DISP		; clear disp for HB 	
	movf	HEART_1,w	; heart beat
	call 	INST
	movlw	0xFF		; LEDs off
	movwf	DISP4
	bcf	DISP4,5		; Heart Beat LED on
	goto	DELTME		; delay and switch debounce
SET_TMR	movf	TIME_CNT3,w	; timer ram
	call 	INST
	movlw	0xFF
	movwf	DISP4		; LEDs off
	bcf	DISP4,2		; set time indicator
	goto	DELTME
SET_MIN	movf	HBMIN,w		; min heart beat
	call	INST
	movlw	0xFF
	movwf	DISP4		; LEDs off
	bcf	DISP4,7		; set HBMIN indicator
	goto	DELTME
SET_MAX	movf	HBMAX,w		; max heart beat
	call	INST
	movlw	0xFF
	movwf	DISP4		; LEDs off
	bcf	DISP4,1		; set HBMAX indicator	
	goto	DELTME

; if mode is 0 then reduce disp value till 0 

DOWN	movf	MODE_1,w	; mode
	btfsc	STATUS,z	; skip if not zero
	goto	HB_TIME		; heart beat or timer
	movlw	0x03		; set time mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	TME_DN		; set time mode
	movlw	0x02		; check which mode
	xorwf	MODE_1,w	; mode
	btfss	STATUS,z	; zero if the same
	goto	HB_MIND		; heart beat min
	goto	HB_MAXD		; heart beat max
	
HB_TIME	movf	DISP,w		; which display 
	btfsc	STATUS,z	; check if 0
	goto	INC_DSP
	movlw	0xFF
	movwf	DISP		; set at FF so when inc will be 00
INC_DSP incfsz	DISP,f		; do not increase if already 1
	goto	TME_DIS		; heart beat
HB_DISP	movf	HEART_1,w	; heart beat
	call	INST		; value to display
	movlw	0xFF
	movwf	DISP4		; indicator LEDs off
	bcf	DISP4,5		; set heart beat LED
	goto	DELTME	
TME_DIS	movf	TIME_CNT2,w	; minutes timer
	call	INST
	movlw	0xFF
	movwf	DISP4
	bcf	DISP4,6
	goto	DELTME

TME_DN	decf	TIME_CNT3,f	; time counter
	clrf	TIME_CNT1	; clear seconds
	movf	TIME_CNT3,w	; transfer to timer
	movwf	TIME_CNT2
	movf	TIME_CNT3,w
	call	INST
	movlw	EEPROM1
	movwf	EEADR		; address to write to
	movf	TIME_CNT3,w	; current time
	call 	EWRITE		; eeprom write sequence
	goto	DELTME		; store in EEPROM and return

HB_MIND	decf	HBMIN,f		; heart beat minimum
	movf	HBMIN,w
	call	INST
	movlw	EEPROM2
	movwf	EEADR		; address to write to
	movf	HBMIN,w		; 
	call 	EWRITE		; eeprom write sequence
	goto	DELTME		; store in EEPROM and return

HB_MAXD	decf	HBMAX,f		; heart beat maximum
	movf	HBMAX,w
	call	INST
	movlw	EEPROM3
	movwf	EEADR		; address to write to
	movf	HBMAX,w		; 
	call 	EWRITE		; eeprom write sequence
	goto	DELTME		; store in EEPROM and return

UP	movf	MODE_1,w	; mode
	btfsc	STATUS,z	; skip if not zero
	goto	START		; heart beat or timer
	movlw	0x03		; set time mode
	xorwf	MODE_1,w	; mode
	btfsc	STATUS,z	; zero if the same
	goto	TME_UP		; set time mode
	movlw	0x02		; check which mode
	xorwf	MODE_1,w	; mode
	btfss	STATUS,z	; zero if the same
	goto	HB_MINU		; heart beat min
	goto	HB_MAXU		; heart beat max
	
START	btfss	FLAG_2,3	; check forced stop flag
	goto	ST_STOP
	bsf	FLAG_2,2	; set timer run flag
	bcf	FLAG_2,3	; force stop flag
	clrf	UPDATE1		; clear timer counter 1
	clrf	UPDATE2		; and timer 2
	movf	TIME_CNT2,f	; check if zero
	btfss	STATUS,z	; if zero then load preset	
	goto	DELTME
	movf	TIME_CNT3,w
	movwf	TIME_CNT2
	clrf	TIME_CNT1
	goto	DELTME
ST_STOP	bcf	FLAG_2,2	; timer stop
	bsf	FLAG_2,3	; force stop flag
	goto	DELTME


TME_UP	incf	TIME_CNT3,f	; time counter
	clrf	TIME_CNT1	; clear seconds
	movf	TIME_CNT3,w	; transfer to timer
	movwf	TIME_CNT2
	movf	TIME_CNT3,w
	call	INST
	movlw	EEPROM1
	movwf	EEADR		; address to write to
	movf	TIME_CNT3,w	; current time
	call 	EWRITE		; eeprom write sequence
	goto	DELTME		; store in EEPROM and return

HB_MINU	incf	HBMIN,f		; heart beat minimum
	movf	HBMIN,w
	call	INST
	movlw	EEPROM2
	movwf	EEADR		; address to write to
	movf	HBMIN,w		; 
	call 	EWRITE		; eeprom write sequence
	goto	DELTME		; store in EEPROM and return

HB_MAXU	incf	HBMAX,f		; heart beat maximum
	movf	HBMAX,w
	call	INST
	movlw	EEPROM3
	movwf	EEADR		; address to write to
	movf	HBMAX,w		; 
	call 	EWRITE		; eeprom write sequence
	goto	DELTME		; store in EEPROM and return


; if mode is 0 then reset timer and ms counters and start counting
; timer set, HBmin and HB max increase by 1 


; delay time for switch debounce period

DELTME	bsf	INTCON,T0IE	; allow interrupt
	movlw	D'50'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_1	movwf	VALUE_2		; VALUE_2 = w
LP_2	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1	
		
; wait for switch to open (must test for closed switch when RA0, RA1 and RA2 are low
; because switch will appear open at RA4 when switch pull down via RA0-RA2 is high) 	

CLOSED	bcf	FLAG_1,3	; clear flag that indicates switch closed
	call	ON		; check for closed switch
	call	ON		; do at least four times to return to original switch closure
	call	ON		;
	call	ON 		; 
	call	ON 		; do again just in case missed one interrupt during call
	call	ON 		; do again just in case missed one interrupt during call
	btfsc	FLAG_1,3	; was switch closed 	
	goto 	CLOSED		; yes go to check again
	goto 	TO_DO		; switch open so out

; recovery routine

RECOVRY	clrf	MIN_CNT		; minute counter
	clrf	REC_1		; recovery HB after 1 minute
	clrf	REC_2		; 2 minutes
	clrf	REC_3		; heart beat after 3 minutes
	clrf	REC_4		; 4 minutes
	clrf	REC_5		; heart beat recovery after 5 minutes
	bsf	REC_FLG,1	; set recovery flag to allow alarm
	bcf	FLAG_2,0	; clear overrange flag	
	movf	HEART_1,w	; heart beat value at timer end
	movwf	REC_0		; -0 mins HB rate
ROL_DIS	movlw	0xFF		; clear display 
	movwf	DISP4
	goto	FIRST5

RECVR	movlw	0xFF		; clear display 
	movwf	DISP4
	movlw	0x0B		; value for an 'r' on display
	movwf	DISP3
	movlw	0x0C		; value for an 'E' on display
	movwf	DISP2
	movlw	0x0E		; value for a 'C' on display
	movwf	DISP1
	call	DELAY		; display REC for a short time
	return

; first 5-mins show rEC and heart beat
	
FIRST5	btfsc	REC_FLG,0	; if clear the 5-mins have passed
	goto	AFT_5		; show the heart beat rates recorded
	call	RECVR		; REC on display
	movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movf	MIN_CNT,w	; minute counter
	movwf	DISP1		; display -x minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	call	RECVR
	call	HB_CALC		; calculate heart rate and place in HEART_1 
 	movf	HEART_1,w	; current heart rate
	call 	INST		; place in display
	bcf	DISP4,5		; heart beat rate LED on
	bsf	DISP4,6		; timer led off
	call	DELAY
	call	HB_CALC		; calculate heart rate and place in HEART_1 
 	movf	HEART_1,w	; current heart rate
	call 	INST		; place in display
	call	DELAY
	goto	FIRST5		; show REC again then time REC then heart beat

; after 5-mins show heart beat rates at every minute
	
AFT_5	call	RECVR		; show REC
	movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movlw	0x00		; zero
	movwf	DISP1		; display -0 minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	movf	REC_0,w		; heart beat at zero minutes	
	call	INST		; value to display
	bsf	DISP4,6		; timer LED off
	bcf	DISP4,5		; HB LED on
	call 	DELAY
ONE_MIN movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movlw	0x01		; 
	movwf	DISP1		; display -1 minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	bsf	DISP4,6		; timer LED off
	bcf	DISP4,5		; HB LED on
	movf	REC_1,w		; heart beat at 1 minutes	
	call	INST		; value to display
	call 	DELAY
TWO_MIN movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movlw	0x02		; 
	movwf	DISP1		; display -2 minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	bsf	DISP4,6		; timer LED off
	bcf	DISP4,5		; HB LED on
	movf	REC_2,w		; heart beat at 2 minutes	
	call	INST		; value to display
	call 	DELAY
THR_MIN movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movlw	0x03		; 
	movwf	DISP1		; display -3 minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	bsf	DISP4,6		; timer LED off
	bcf	DISP4,5		; HB LED on
	movf	REC_3,w		; heart beat at 3 minutes	
	call	INST		; value to display
	call 	DELAY
FOR_MIN movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movlw	0x04		; 
	movwf	DISP1		; display -4 minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	movf	REC_4,w		; heart beat at 4 minutes	
	call	INST		; value to display
	bsf	DISP4,6		; timer LED off
	bcf	DISP4,5		; HB LED on
	call 	DELAY
FIV_MIN movlw	0x0A		; "-"
	movwf	DISP2		; show a minus	
	clrf	DISP3		; show a zero (blanked)
	movlw	0x05		; 
	movwf	DISP1		; display -5 minutes recovery
	bsf	DISP4,5		; HB LED off
	bcf	DISP4,6		; timer LED on
	call	DELAY
	movf	REC_5,w		; heart beat at zero minutes	
	call	INST		; value to display
	bsf	DISP4,6		; timer LED off
	bcf	DISP4,5		; HB LED on
	call 	DELAY
	bsf	DISP4,5		; HB LED off
	bsf	DISP4,6		; Timer LED off
	goto	ROL_DIS		; roll display (REC, -0,HB -1,HB, -3,HB, -5,HB)

;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

; SUBROUTINES
	
; heart beat calculate subroutine

HB_CALC	movf	HBEATD,w	; heart beat counter
	movwf	ACCaHI		; for divide
	btfsc	STATUS,z	; if zero then ok not overrange
	goto	OVER_R		; otherwise load 0 display
	movf	HBEATC,w	; heart beat counter
	movwf	ACCaLO		; for divide
	call	D_DIV		; divide routine
	movf	ACCbLO,w	; result
	movwf	HEART_1
	incfsz	HEART_1,w	; increase by 1 (if 255 then becomes 0)
	return
	clrf	HEART_1		; make a zero
	return

OVER_R	clrf	HEART_1
	return

; timer (update minutes and place heart beat HEART_1 values (call HB_CALC) into REC_X memory)

TIMER	btfss	FLAG_1,1	; 1 sec flag if set then continue
	return			; not updated yet so out
	btfsc	REC_FLG,0	; stop on recovery flag (5-minutes are up)
	bcf	REC_FLG,1	; stops alarm after 5-mins		
	bcf	FLAG_1,1	; clear 1 sec flag
	incf	TIME_CNT1,f	; increase seconds
	movlw	D'60'		; 
	subwf	TIME_CNT1,w	; check if 60
	btfss	STATUS,c	; if zero then clear
	return			; not minutes yet
	clrf	TIME_CNT1	
	incfsz	TIME_CNT2,f	; next minute
	goto	CNT_MIN
	decf	TIME_CNT2,f	; if zero go back to 255 (stops returning to a 1 minute a second time)
CNT_MIN	movf	TIME_CNT2,w	; 
	xorlw	0x01		; 1 minute
	btfsc	STATUS,z
	goto	ONE_CNT
	movf	TIME_CNT2,w
	xorlw	0x02		; 2 minutes
	btfsc	STATUS,z
	goto	TWO_CNT
	movf	TIME_CNT2,w
	xorlw	0x03		; 3 minutes
	btfsc	STATUS,z
	goto	THR_CNT
	movf	TIME_CNT2,w
	xorlw	0x04		; 4 minutes
	btfsc	STATUS,z
	goto	FOR_CNT
	movf	TIME_CNT2,w
	xorlw	0x05		; 5 minutes
	btfsc	STATUS,z
	goto	FIV_CNT
	return

ONE_CNT	call	HB_CALC
	movf	HEART_1,w
	movwf	REC_1
	incf	MIN_CNT,f
	clrf	ALARM_T		; alarm time
	return
TWO_CNT	call	HB_CALC
	movf	HEART_1,w
	movwf	REC_2
	incf	MIN_CNT,f
	clrf	ALARM_T		; alarm time
	return
THR_CNT	call	HB_CALC
	movf	HEART_1,w
	movwf	REC_3
	incf	MIN_CNT,f
	clrf	ALARM_T		; alarm time
	return
FOR_CNT	call	HB_CALC
	movf	HEART_1,w
	movwf	REC_4
	incf	MIN_CNT,f
	clrf	ALARM_T		; alarm time
	return
FIV_CNT	call	HB_CALC
	movf	HEART_1,w
	movwf	REC_5
	incf	MIN_CNT,f
	clrf	ALARM_T		; alarm time
	bsf	REC_FLG,0	; set recovery flag (stops err on display after 5-mins)
	return


; Delay

DEL_OP	decfsz	DELOP,f
	goto	DEL_1
	return
DELAY	movlw	0x05		
	movwf	DELOP		; number of delay loops
DEL_1	movlw	0xFF		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	0xFF		; set delay period value 2 
LOOP_1	movwf	VALUE_2		; VALUE_2 = w
LOOP_2	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LOOP_2
	call	TIMER
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LOOP_1	
	bcf	FLAG_1,3	; clear flag that indicates switch closed
	call	ON		; check for closed switch
	call	ON		; do at least four times to return to original switch closure
	call	ON		;
	call	ON 		; 
	call	ON 		; do again just in case missed one interrupt during call
	btfss	FLAG_1,3	; was switch closed 	
	goto	DEL_OP
	bcf	INTCON,INTE	; clear interrupts
	bcf	INTCON,GIE
	bcf	INTCON,T0IE
	movlw	0xFF
	movwf	PORTA		; turn off displays
	goto 	MAIN		; switch closed so start at beginning of program
	

; subroutine to place binary value into display registers after BCD conversion

INST	movwf	BIN_0		; place in binary 0 register
	btfss	FLAG_2,0	; is flag set (overrange)
	goto	BCD_GO
	movf	MODE_1,f
	btfss	STATUS,z	; if zero then dash
	goto	BCD_GO
	btfss	DISP4,5 	; is it the heart beat display
	goto	DASH		; if over 255 then display is dashes
	movf	TIME_CNT2,w
	movwf	BIN_0
BCD_GO	call	BCD		; convert binary number to BCD
	movf	BCD_0,w		; ms byte for display
	movwf	DISP3
	movf	BCD_1,w		; contains packed BCD value
	andlw	0x0F		; extract LS bits
	movwf	DISP1
	swapf	BCD_1,w		; swap upper and lower 4-bits
	andlw	0x0F		; extract bits
	movwf	DISP2		; place in middle digit register
	return 
DASH	movlw	D'255'		; max reading
	movwf	HEART_1
	movlw	0x0A		; value which gives a dash after seven segment decode
	movwf	DISP1
	movwf	DISP2
	movwf	DISP3
	return

; subroutine to wait for switch to reopen
	
ON	bcf	INTCON,T0IE	; prevent multiplex update
	bcf	FLAG_1,2	; clear flag that indicates update interrupt for display
RE_CHK	btfsc	PORTA,4		; check if switch closed
	goto	OPEN
	bsf	FLAG_1,3	; switch closed so set flag
	bsf	INTCON,T0IE	; allow multiplex interrupt
	goto	RE_CHK 		; closed so check till open 
OPEN	bcf	FLAG_1,2	; clear multiplex update flag wait for next multiplex update
	bsf 	INTCON,T0IE	; allow multiplex update
FLG_TST	btfss	FLAG_1,2	; is flag set indicating a new interrupt for multiplexing
	goto 	FLG_TST		; wait 
	return

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_RD	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_RD		; wait for low RD (read RD)	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	bcf	FLAG_1,6	; EEPROM write repeat flag
	movwf	TEMP_2		; store w in temporary storage
EWRIT	movf	TEMP_2,w	; place value in w (used for read data sequence) 
	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	
; read EEPROM DATA and check if written correctly
	
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_AGN	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_AGN		; wait for low RD	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	subwf	EEDATA,w	; compare read value with value stored
	btfsc	STATUS,z	; skip if not the same
	return			; value correctly written 
	btfsc	FLAG_1,6	; write repeat bit, skip if clear and rewrite
	return
	bsf	FLAG_1,6	; set repeat flag rewrite once only 
	goto 	EWRIT		; rewrite as not written correctly

; Subroutine to convert from 8-bit binary to 2-digit BCD (packed)
; Binary value is in BIN0  
; Result in BCD is in BCD0 & BCD1.  
; BCD0 is MSB, BCD1 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'8'
	movwf	CNT_8		; 8 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
LOOPBCD	rlf	BIN_0,f		; shift left binary registers
	rlf	BCD_1,f		; MSB shift left

	rlf	BCD_0,f		; LSB shift left BCD registers
	decfsz	CNT_8,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_1		; BCD LSB address
	movwf	FSR		; pointer for BCD1
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-1)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w in temp
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

; divide routine denominator in ACCaHI & ACCaLO. Numerator in ACCbHI & ACCbLO. 
; Result in ACCbHI & ACCbLO. Remainder in ACCcHI & ACCcLO.

D_DIV	movlw	D'16'		; 16 shifts
	movwf	TEMP
	movlw	0xEA		; 0xEA place 60000 (H EA60) in numerator 	
	movwf	ACCdHI
	movlw	0x60		; 0x60
	movwf	ACCdLO
	clrf	ACCbLO
	clrf	ACCbHI
	clrf	ACCcHI		; remainder ms byte
	clrf	ACCcLO		; remainder ls byte
DLOOP	bcf	STATUS,c	; clear carry
	rlf	ACCdLO,f	; ACCbLO store
	rlf	ACCdHI,f	; ACCbHI store
	rlf	ACCcLO,f	; numerator
	rlf	ACCcHI,f	
	movf	ACCaHI,w	; denominator HI
	subwf	ACCcHI,w	; is denom > remainder
	btfss	STATUS,z
	goto	NOCHK
	movf	ACCaLO,w	; denominator
	subwf	ACCcLO,w	; remainder (checking LSB)
NOCHK	btfss	STATUS,c
	goto	NOGO
	movf	ACCaLO,w	; sub numerator from denominator
	subwf	ACCcLO,f
	btfss	STATUS,c
	decf	ACCcHI,f
	movf	ACCaHI,w
	subwf	ACCcHI,f
	bsf	STATUS,c	; for a 1 into result
NOGO	rlf	ACCbLO,f
	rlf	ACCbHI,f
	decfsz	TEMP,f
	goto	DLOOP
	return

	end
