;**************************************************************************
;**************************************************************************
;*
;*
;*                 IR Remote Receiver & LCD Display Firmware
;* 
;*                           Version 1.0  13/08/01
;*
;*                           Created by P.B.Smith
;*                   (C)2001 Silicon Chip Publications P/L
;*                           All Rights Reserved.
;*
;*                          www.siliconchip.com.au
;*
;*               +-------------------------------------------+
;*               | This source code is released as FREEWARE  |
;*               | and as such is NOT in the public domain.  |
;*               +-------------------------------------------+
;*
;**************************************************************************
;**************************************************************************

.nolist
.include "2313ndef.inc"		;AVR definitions (non-standard!)
.list

;**************************************************************************
;*
;* REGISTERS
;*
;**************************************************************************

.def	rc5_state	=R2	;RC5 state
.def	rc5_bit_cnt	=R3	;RC5 bit count
.def	rc5_system	=R4	;RC5 received system bits
.def	rc5_command	=R5	;RC5 received command bits
.def	rc5_repeat	=R6	;RC5 key repeat count
.def	led_on_cnt	=R8	;LED on timer
.def	rx_buff_head	=R9	;pointer to receive buffer head
.def	rx_buff_tail	=R10	;pointer to receive buffer tail
.def	tx_buff_head	=R11	;pointer to transmit buffer head
.def	tx_buff_tail	=R12	;pointer to transmit buffer tail
.def	esc_code	=R13	;terminal ESC code
.def	esc_seq_cnt	=R14	;terminal ESC sequence count
.def	status		=R15	;SREG save (used by tmr0_irq)

.def	A		=R16	;scratch
.def	B		=R17	;scratch
.def	flags		=R18	;various flags (see bit vars below)
.def	tick_low	=R19	;low counter (ticks every 64us)
.def	tick_high	=R20	;high counter (ticks every 16.384ms)
.def	loop_cnt1	=R21	;loop counters
.def	loop_cnt2	=R22
.def	rc5_cnt		=R23	;RC5 timer
.def	lcd_addr_cntr	=R24	;LCD address counter image
.def	C		=R25	;scratch (used by rx_irq)
.def	XL		=R26
.def	D		=R27	;scratch (used by tx_irq)
.def	YL		=R28
.def	E		=R29	;scratch (used by tmr0_irq)
.def	ZL		=R30
.def	ZH		=R31

; Register-bound bit variables

.equ	CMD_DATA	=0	;LCD command/data (0=command, 1=data)
.equ	LINE_WRAP	=1	;line wrap (0=off, 1=on)
.equ	AUTO_SCROLL	=2	;auto-scroll at EOS (0=off, 1=on)
.equ	ESC_FLAG	=3	;terminal escape sequence (0=idle, 1=active)
.equ	DNLD_ERROR	=4	;eeprom download error (0=OK, 1=err)
.equ	RX_BAUD		=5	;new baud param received (0=no, 1=yes)
.equ	RX_PWRDLY	=6	;new power-on delay param received (0=no, 1=yes)
.equ	RC5_DATA_RDY	=7	;RC5 receive data ready


;**************************************************************************
;*
;* HARDWARE SPECIFIC
;*
;**************************************************************************

; Port B bits

.equ	LCD_RS		=7	;LCD register select port bit
.equ	LCD_RW		=6	;LCD read/write port bit
.equ	LCD_E		=4	;LCD enable (strobe) port bit
.equ	LCD_DB4		=3	;LCD data bit 4
.equ	LCD_DB5		=2	;LCD data bit 5
.equ	LCD_DB6		=1	;LCD data bit 6
.equ	LCD_DB7		=0	;LCD data bit 7

; Port D bits (except the UART lines):
 
.equ	LED_BIT		=6	;LED
.equ	SW_BIT		=6	;push-button switch
.equ	CTS_BIT		=3	;RS232 handshaking (CTS) line
.equ	IRR_BIT		=2	;IR receiver input


;**************************************************************************
;*
;* CONSTANTS
;*
;**************************************************************************

; ASCII codes used by LCD terminal routines

.equ	NUL		=$00	;ignored
.equ	SOH		=$01	;^A  home cursor
.equ	EOT		=$04	;^D  hide cursor
.equ	ENQ		=$05	;^E  show underline cursor
.equ	ACK		=$06	;^F  show blinking block cursor
.equ	BEL		=$07	;^G  sound buzzer (not implemented)
.equ	BS		=$08	;^H  backspace cursor
.equ	HT		=$09	;^I  horizontal tab
.equ	LF		=$0A	;^J  move cursor down one line
.equ	VT		=$0B	;^K  move cursor up one line
.equ	FF		=$0C	;^L  clear screen
.equ	CR		=$0D	;^M  move cursor to first position of line
.equ	SO		=$0E	;^N  backlight on (not implemented)
.equ	SI		=$0F	;^O  backlight off (not implemented)
.equ	DLE		=$10	;^P  position cursor
.equ	DC3		=$13	;^S  auto-scroll on
.equ	DC4		=$14	;^T  auto-scroll off
.equ	ESC		=$1B	;^[  escape

; Serial i/o constants

.equ	BAUD_2400	=103	;values for **4MHz crystal**
.equ	BAUD_4800	=51
.equ	BAUD_9600	=25
.equ	BAUD_19200	=12

.equ	RX_BUFF_SIZE	=64	;rx buffer size
.equ	TX_BUFF_SIZE	=16	;tx buffer size
.equ	RX_HIGH_WATER	=58	;drop CTS (XOFF) at this mark
.equ	RX_LOW_WATER	=42	;raise CTS (XON) at this mark


;**************************************************************************
;*
;* MACROS
;*
;**************************************************************************

.macro MESSAGE
	ldi	ZH,high(@0*2)	;get pointer to string
	ldi	ZL,low(@0*2)
	rcall	display_string	;write message to LCD
.endmacro

	.cseg
	.org	$0

;**************************************************************************
;*
;* INTERRUPT VECTORS
;*
;**************************************************************************
        
	rjmp	reset		;reset
	rjmp	bad_irq		;IRQ0
	rjmp	bad_irq		;IRQ1
	rjmp	bad_irq		;timer 1 capture
	rjmp    bad_irq		;timer 1 compare
	rjmp	bad_irq 	;timer 1 overflow
	rjmp	tmr0_irq	;timer 0 overflow
	rjmp	rx_irq		;UART rx complete
	rjmp	tx_irq		;UART data register empty
	rjmp	bad_irq		;UART tx complete
	rjmp	bad_irq		;analog comparator

bad_irq:
	reti
	

;**************************************************************************
;*
;* RESET ENTRY POINT
;*
;**************************************************************************
	
reset:	cli				;disable all interrupts
	ldi	A,low(RAMEND)		;set stack top
	out	SPL,A

	sbi	ACSR,ACD		;power down comparator
		
	clr	flags			;init variables
	sts	rc5_prev_cmd,flags
	sts	rc5_prev_sys,flags
	out	UCR,flags		;disable UART (might be soft reset)
	clr	rc5_state
	clr	rc5_repeat
	
	ldi	A,0b00001010		;port D all inputs except CTS & TXD
	out	DDRD,A
	ldi	A,0b01111100		;port D inputs pulled up
	out	PORTD,A

	ldi	A,1			;CT0 clocked at CK
	out	TCCR0,A
	ldi	A,1<<TOIE0		;enable CT0 overflow interrupt
	out	TIMSK,A
	
	sei				;enable interrupts
	
	rcall	init_lcd		;initialise LCD
	rcall	init_cg			;program custom LCD characters
	
;Read power-up message from EEPROM and display it

	ldi	A,start_msg
	rcall	eeread			;get start msg display parameter
	tst	B
	brne	res2			;skip message display
	
	ldi	C,eemsg
res1:	mov	A,C			;eeprom address pointer
	rcall	eeread			;get a character
	tst	B
	breq	res2			;null terminates
	rcall	term_write		;display it
	inc	C
	rjmp	res1
	
; Read switch and enter 'setup' mode if pressed

res2:	rcall	get_switch		;test for switch depressed
	brcs	res3			;skip if not

	rcall	setup_mode		;enter 'setup' mode if pressed		
	rjmp	res5			;skip power-up delay

; Read power-on delay parameter from EEPROM and wait accordingly.
; This can be used to prevent 'garbage' from being received on the serial
; line (and thus displayed on the LCD) while the host computer is booting.

res3:	ldi	A,pwr_delay
	rcall	eeread			;get power-up delay value
	tst	B
	breq	res5			;none programmed
	
res4:	ldi	A,61			;for 1 sec
	rcall	delay_long
	dec	B
	brne	res4			;delay for B seconds

res5:	rcall	init_uart		;initialise serial i/o
	
; Power-on initialisation complete


;**************************************************************************
;*
;* MAIN 
;*
;* Everything except LCD i/o is interrupt driven. Main provides the
;* terminal interface between received serial data and the LCD, and
;* feeds received IR codes to the serial transmit buffer. 
;*
;**************************************************************************

main:	rcall	get_char		;get char from receive buffer
	brcs	mn1			;skip if none received
	
	rcall	term_write		;send to terminal
	
mn1:	sbrs	flags,RC5_DATA_RDY	;skip if RC5 receive ready
	rjmp	main			;loop
	
	rcall	tx_ir_code		;transmit RC5 codes
	rjmp	main			;loop


;**************************************************************************
;*
;* SETUP MODE
;*
;* Invoked by pressing S1 during power up.
;*
;**************************************************************************

setup_mode:
	MESSAGE	msg1			;display firmware rev message
	rcall	wait_switch		;wait for 4 secs or switch press 

	MESSAGE	msg2			;display 'set baud rate' message
	rcall	wait_switch		;wait 4 secs for switch press
	brcc	stm4			;go if pressed

	MESSAGE	msg6			;display 'download eeprom' message
	rcall	wait_switch		;wait 4 secs for switch press
	brcc	stm1			;go if pressed
	rjmp	stm13			; else exit
	
stm1:	MESSAGE	msg7			;display 'waiting for data' message
	rcall	download_eeprom
	brcs	stm3			;skip if download failed

	MESSAGE	msg8			;display 'download ok'
stm2:	rcall	get_switch
	brcs	stm2			;loop until switch pressed	
	rjmp	reset			; then do a soft reset

stm3:	MESSAGE	msg9			;display 'download failed'
	rjmp	stm2

stm4:	ldi	A,baud_rate
	rcall	eeread			;get current rate
	mov	C,B
	mov	D,B
	rjmp	stm7			;go display current

stm5:	rcall	wait_switch		;wait 4 secs for switch press
	brcc	stm6			;skip if pressed
	rjmp	stm12			;timeout, go update eeprom

; Key pressed, display next option

stm6:	inc	C
stm7:	cpi	C,0
	breq	stm8
	cpi	C,1
	breq	stm9
	cpi	C,2
	breq	stm10
	clr	C			;reset to first option

stm8:	ldi	ZH,high(msg3*2)		;9600	
	ldi	ZL,low(msg3*2)
	rjmp	stm11
	
stm9:	ldi	ZH,high(msg4*2)		;4800
	ldi	ZL,low(msg4*2)
	rjmp	stm11
	
stm10:	ldi	ZH,high(msg5*2)		;2400
	ldi	ZL,low(msg5*2)

stm11:	rcall	display_string
	rjmp	stm5	

; Update baud_rate value in EEPROM

stm12:	cp	D,C
	breq	stm13			;skip write if no change
	
	ldi	A,baud_rate		;eeprom address
	mov	B,C			;byte to write
	rcall	eewrite			;update the code

stm13:	ldi	B,FF			;(form feed)
	rcall	term_write		;clear display
	ret


;**************************************************************************
;*
;* WAIT FOR SWITCH PRESS
;*
;**************************************************************************

wait_switch:
	clr	tick_high
wsw1:	rcall	get_switch
	brcc	wsw2			;exit if pressed
	
	cpi	tick_high,255
	brne	wsw1			;wait 4 secs only
	sec
wsw2:	ret

	
;**************************************************************************
;*
;* GET SWITCH (S1) STATE
;*
;**************************************************************************

get_switch:
	sec
	sbic	PIND,SW_BIT		;test for switch depressed 
	rjmp	gsw2
	
gsw1:	sbis	PIND,SW_BIT		;wait until released
	rjmp	gsw1	

	ldi	A,4
	rcall	delay_long		;debounce (wait > 20ms)
	clc
gsw2:	ret	


;**************************************************************************
;*
;* DOWNLOAD EEPROM
;*
;*
;* Format used is INTELLEC 8 / MDS (INTEL HEX).
;*
;*
;*                        DATA RECORD
;*                        -----------
;*  
;*               : BC AAAA TT HHHHHHHH CC YYZZ
;* 
;*   :      = Start Character
;*   BC     = Byte Count (the hex number of bytes in the record)
;*   AAAA   = 16-bit address of first data byte (in hex notation)
;*   TT     = Record Type (00)
;*   HH     = One data byte (in hex notation. Repeated up to 255 times)
;*   CC     = Checksum. Negation (two's complement) of all
;*            preceding bytes in record except Start Character
;*   YYZZ   = ASCII carriage return & line feed characters
;*            (note : this field optional by definition)
;*  
;*  
;*                    END-OF-FILE RECORD
;*                    ------------------
;*  
;*                      : BC AAAA TT
;*  
;*   :      = Start Character
;*   BC     = Byte Count (00 for end-of-file)
;*   AAAA   = Address (Normally entry address. Set to 0000 if not used)
;*   TT     = Record Type (01)
;*  
;**************************************************************************

download_eeprom:
	rcall	init_uart		;init serial i/o
dle1:	rcall	get_char		;get a char
	brcs	dle1

	cpi	B,$3A			;':'
	brne	dle1			;wait for start character

	clr	R1			;init checksum
	rcall	get_byte		;get byte count
	brcs	dle8			;abort if conversion error
	mov	ZL,B
	rcall	get_byte		;ignore high byte start address
	rcall	get_byte		;get low byte start address
	brcs	dle8			;abort if conversion error
	cpi	B,128
	brsh	dle8			;abort if address exceeds size

	mov	ZH,B
	add	B,ZL			;start + length
	cpi	B,129			;will byte count exceed size?
	brsh	dle8			;abort if so

dle3:	rcall	get_byte		;get record type
	brcs	dle8			;abort if conversion error
	tst	B
	breq	dle4			;skip if data record

	ldi	tick_high,224
	rjmp	dle10			;should be EOF record, go clean up
	
dle4:	rcall	get_byte		;get data byte
	brcs	dle8			;abort if conversion error
	
	cpi	ZH,65			;baud rate byte?
	brne	dle5			;skip if not
	sts	baud_save,B		;save for now...
	sbr	flags,1<<RX_BAUD
	rjmp	dle7

dle5:	cpi	ZH,66			;power-on delay byte?
	brne	dle6
	sts	pwrdly_save,B		;save for now...
	sbr	flags,1<<RX_PWRDLY		
	rjmp	dle7

dle6:	mov	A,ZH
	rcall	eewrite			;write the byte to eeprom
dle7:	inc	ZH
	dec	ZL
	brne	dle4			;loop to get complete record

	rcall	get_byte		;get checksum byte
	tst	R1			;checksum should be 00!
	breq	dle1			;loop to do next record if OK

dle8:	sbr	flags,1<<DNLD_ERROR	;flag the error and abort...

; Flush any further incoming data.

dle9:	clr	tick_high
dle10:	rcall	get_char
	cpi	tick_high,255
	brlo	dle10			;wait a bit...
	
	sec
	sbrc	flags,DNLD_ERROR	;skip if no errors detected
	rjmp	dle13			; else quit
	
; Data received successfully. Check if the two critical config
; bytes (saved earlier) were downloaded. If so, program them now!

	sbrs	flags,RX_BAUD
	rjmp	dle11
	
	ldi	A,65
	lds	B,baud_save
	rcall	eewrite			;write baud rate parameter

dle11:	sbrs	flags,RX_PWRDLY
	rjmp	dle12

	ldi	A,66
	lds	B,pwrdly_save
	rcall	eewrite			;write power-on delay parameter	

dle12:	clc
dle13:	ret
	 

;**************************************************************************
;*
;* RECEIVE TWO ASCII CHARACTERS AND CONVERT TO HEX BYTE
;*
;* Entry: R1 = running checksum
;* Exit : B  = hex byte
;*
;**************************************************************************

get_byte:
	rcall	get_char		;get a char
	brcs	get_byte
	rcall	asc_hex			;convert to hex
	swap	B
	mov	R0,B			;save as high nibble

gtb1:	rcall	get_char		;get next char
	brcs	gtb1
	rcall	asc_hex			;convert to hex
	or	B,R0			;merge high + low nibbles
	add	R1,B			;keep checksum
	clc
	ret


;**************************************************************************
;*
;* CONVERT ASCII TO HEX
;*
;* Entry: B = ASCII character
;* Exit : B = hex nibble
;*
;**************************************************************************

asc_hex:
	subi	B,$30			;'0'
	brcs	ahx1			;bad HEX if < 0
	cpi	B,10
	brcs	ahx2			;done if 0-9
	andi	B,$1F			;to upper case
	subi	B,7
	cpi	B,$0A
	brcs	ahx1			;bad hex if < ASCII 'A'
	cpi	B,$10
	brcs	ahx2			;bad hex if > ASCII 'F'
ahx1:	pop	B			;dump caller
	pop	B
	sec
ahx2:	ret
		

;**************************************************************************
;*
;* DISPLAY ASCII STRING
;*
;**************************************************************************

display_string:
	lpm
	tst	R0
	breq	dsp1			;$00 terminates
	mov	B,R0
	push	ZL			;ouch!
	push	ZH
	rcall	term_write
	pop	ZH
	pop	ZL
	adiw	ZL,1
	rjmp	display_string
dsp1:	ret


;**************************************************************************
;*
;* MESSAGE STRINGS
;*
;**************************************************************************

msg1:
.db	FF,"Firmware rev 1.0",0
msg2:
.db	FF,"Set baud rate?",CR,LF,0
msg3:
.db	CR,"=> 9600",0
msg4:
.db	CR,"=> 4800",0
msg5:
.db	CR,"=> 2400",0
msg6:
.db	FF,"Download EEPROM?",0
msg7:
.db	FF,"Waiting for data",0
msg8:
.db	FF,"Download OK!",0
msg9:
.db	FF,"Download failed!",0

	
;**************************************************************************
;**************************************************************************
;*
;*                        LCD TERMINAL OUTPUT ROUTINES
;*
;**************************************************************************
;**************************************************************************

term_write:
	bst	flags,ESC_FLAG		;ESC sequence in progress?
	brbc	TFLAG,trm2		;skip if not

	tst	esc_code		;got code yet?
	brne	trm1			;yes
	mov	esc_code,B		;no, so this is it

trm1:	mov	A,esc_code		;get the code
	rjmp	trm3			;go execute handler
	
trm2:	tst	B			;null?
	breq	trm6			;ignore if it is
	
	cpi	B,$20			;control code?
	mov	A,B
	brsh	trm4			;skip if not

trm3:	rcall	func_exec		;execute control/escape handler
	rjmp	trm6
	
trm4:	cpi	B,128			;special char?
	brlo	trm5			;skip if not
	
	cpi	B,136
	brsh	trm5			;Kanji/Greek char
	
	subi	B,128			;custom char, remove offset
trm5:	rcall	display_char
trm6:	ret


;**************************************************************************
;*
;* DISPLAY CHARACTER
;*
;* Entry: B = ASCII character
;*
;**************************************************************************

display_char:
	rcall	write_lcd_data		;display the character
	cpi	lcd_addr_cntr,16	;first line?
	brlo	dpy1			;yes, skip next bit...

; Characters written to the bottom line are also written
; to a buffer (line_buff) in memory.

	mov	A,lcd_addr_cntr
	andi	A,$0F			;keep offset in line
	ldi	ZL,line_buff
	add	ZL,A			
	st	Z,B			;save char in line buffer
	
dpy1:	cpi	lcd_addr_cntr,15	;end of first line?
	brne	dpy2			;no

	inc	lcd_addr_cntr		;yes, bump counter and...
	rcall	set_curs		;...wrap cursor to next line
	rjmp	dpy5

; If at the last char position on the bottom line, and if auto-scroll
; is enabled, then scroll up and clear the last line. Otherwise, just
; wrap cursor back to start of top line.

dpy2:	cpi	lcd_addr_cntr,31	;end of second line?
	brne	dpy4			;skip if not...
		
	sbrc	flags,AUTO_SCROLL	;skip if auto-scroll off
	rjmp	dpy3			
	
	rcall	home_curs		;wrap cursor back to top line
	rjmp	dpy5			
	
dpy3:	rcall	scroll_up		;scroll up a line
	ldi	lcd_addr_cntr,16
	rcall	set_curs		;return cursor to line start
	rjmp	dpy5
	
dpy4:	inc	lcd_addr_cntr		;update AC image
dpy5:	ret


;**************************************************************************
;*
;* ESCAPE
;*
;**************************************************************************

escape:
	sbr	flags,1<<ESC_FLAG	;flag sequence start
	clr	esc_code
	clr	esc_seq_cnt		;init sequence counter
	ret


;**************************************************************************
;*
;* HOME CURSOR
;*
;**************************************************************************

home_curs:
	clr	lcd_addr_cntr
	rcall	set_curs
	ret


;**************************************************************************
;*
;* SET CURSOR POSITION
;*
;* In order to be compatible with 'SEE' LCD displays, ^P provides cursor
;* positioning. The byte after ^P is the required cursor position (+64).
;* This routine handles ^P as though it were an escape sequence, storing
;* "P" as the escape code.
;*
;**************************************************************************

pos_curs:
	sbr	flags,1<<ESC_FLAG	;flag sequence start
	ldi	A,$50			;"P"
	mov	esc_code,A		;save as escape code
	clr	esc_seq_cnt		;init sequence counter
	ret

; Enters here when position byte received

pos_curs_here:
	cbr	flags,1<<ESC_FLAG	;sequence complete
	subi	B,64
	brcs	pcr1			;ignore if invalid (<64)
	cpi	B,32
	brsh	pcr1			;ignore if exceeds display size
	mov	lcd_addr_cntr,B
	rcall	set_curs
pcr1:	ret	


;**************************************************************************
;*
;* SET CURSOR POSITION
;*
;* The LCD cursor is synchronised with the position in lcd_addr_cntr.
;*
;**************************************************************************

set_curs:
	mov	B,lcd_addr_cntr
	cpi	B,16			;second line?
	brlo	scr1			;skip if not
	subi	B,-48			;adjust
scr1:	ori	B,0b10000000		;include command bit
	rcall	write_lcd_cmd
	ret	


;**************************************************************************
;*
;* CLEAR DISPLAY
;*
;**************************************************************************

clr_display:
	ldi	B,0b00000001
	rcall	write_lcd_cmd		;write the clear instruction
	clr	lcd_addr_cntr		;zero address counter image

clr_line_buff:
	ldi	B,$20
	ldi	ZL,line_buff
	ldi	loop_cnt2,16
clb1:	st	Z+,B			;clear (last) line buffer too
	dec	loop_cnt2
	brne	clb1	
	ret


;**************************************************************************
;*
;* HIDE CURSOR
;*
;**************************************************************************

hide_curs:
	ldi	B,0b00001100
	rcall	write_lcd_cmd
	ret

	
;**************************************************************************
;*
;* SHOW UNDERLINE CURSOR
;*
;**************************************************************************

line_curs:
	ldi	B,0b00001110
	rcall	write_lcd_cmd
	ret


;**************************************************************************
;*
;* SHOW BLOCK CURSOR
;*
;**************************************************************************
		
block_curs:
	ldi	B,0b00001101
	rcall	write_lcd_cmd
	ret


;**************************************************************************
;*
;* BACKSPACE
;*
;**************************************************************************

back_space:
	tst	lcd_addr_cntr
	breq	bsp1			;ignore if at home on first line
	dec	lcd_addr_cntr		;back up cursor
	rcall	set_curs
	ldi	B,$20
	rcall	display_char		;erase new position
	dec	lcd_addr_cntr
	rcall	set_curs		;back up again
bsp1:	ret	


;**************************************************************************
;*
;* LINEFEED
;*
;**************************************************************************

line_feed:
	cpi	lcd_addr_cntr,16	;which line?
	brsh	lfd1			;go if second
	subi	lcd_addr_cntr,-16	;...move cursor down a line
	rcall	set_curs	
	rjmp	lfd2

lfd1:	push	lcd_addr_cntr		;save current cursor position
	rcall	scroll_up		;scroll display up
	pop	lcd_addr_cntr		;restore cursor
	rcall	set_curs
lfd2:	ret


;**************************************************************************
;*
;* CARRIAGE RETURN
;*
;**************************************************************************

curs_return:
	andi	lcd_addr_cntr,$30
	rcall	set_curs
	ret


;**************************************************************************
;*
;* SCROLL DISPLAY UP
;*
;* Note that instead of reading the last line from DD ram, we just write
;* out line_buff. This cuts scrolling time in half (important with an
;* incoming 9600bps stream).
;*
;**************************************************************************

scroll_up:
	rcall	home_curs
	ldi	loop_cnt2,16
	ldi	ZL,line_buff
	
scu2:	ld	B,Z+
	rcall	write_lcd_data		;overwrite first line with last
	dec	loop_cnt2
	brne	scu2	

; Clear last line & line_buff. To save time, don't
; call display_char (write the data directly).

	ldi	lcd_addr_cntr,16
	rcall	set_curs		;cursor to start of last line
	
	ldi	loop_cnt2,16		;clear the last line
	ldi	B,$20

scu3:	rcall	write_lcd_data
	dec	loop_cnt2
	brne	scu3
	rcall	clr_line_buff		;clear line buffer	
	ret


;**************************************************************************
;*
;* HORIZONTAL TAB
;*
;**************************************************************************

horiz_tab:
	mov	A,lcd_addr_cntr		;get current position
	neg	A			;calc distance to next stop
	andi	A,3		
	brne	htb1
	ldi	A,4			;stops are at 4-byte intervals	
htb1:	mov	R0,A
htb2:	ldi	B,$20			;space out to next stop
	rcall	display_char
	dec	R0
	brne	htb2
	ret				
		

;**************************************************************************
;*
;* VERTICAL TAB
;*
;**************************************************************************

vert_tab:
	cpi	lcd_addr_cntr,16	;which line?
	brlo	vrt1			;ignore if first
	subi	lcd_addr_cntr,16	;move cursor up a line
	rcall	set_curs
vrt1:	ret


;**************************************************************************
;*
;* NOT IMPLEMENTED (YET)!
;*
;**************************************************************************

not_yet:
	ret


;**************************************************************************
;*
;* AUTO-SCROLL ON/OFF
;*
;**************************************************************************

scroll_on:
	sbr	flags,1<<AUTO_SCROLL
	ret


scroll_off:
	cbr	flags,1<<AUTO_SCROLL
	ret


;**************************************************************************
;*
;* DEFINE CUSTOM CHARACTER
;*
;**************************************************************************

define_char:
	tst	esc_seq_cnt		;first byte ("D") in sequence?
	breq	dch1			;yes, just exit
	
	mov	A,esc_seq_cnt
	dec	A			;second byte?
	brne	dch2			;no, must be bit pattern

	subi	B,$30			;yes, convert to hex
	cpi	B,8			;check valid symbol number
	brsh	dch3			;abort if not 0-7

	lsl	B			;x8 to get CG RAM address
	lsl	B
	lsl	B
	ori	B,0b01000000		;include 'CG RAM' bit
	rcall	write_lcd_cmd		;set address counter
	rcall	write_lcd_cmd		;do it again

dch1:	inc	esc_seq_cnt
	rjmp	dch4

dch2:	rcall	write_lcd_data		;write one byte of bit pattern
	mov	A,esc_seq_cnt
	cpi	A,9			;done all 8 bytes?
	brne	dch1			;exit if not
	
	rcall	set_curs		;reset AC for DD RAM access
dch3:	cbr	flags,1<<ESC_FLAG	;seq complete, clear flag
dch4:	ret


;**************************************************************************
;*
;* EXEC CONTROL/ESCAPE CODE FUNCTION
;*
;* Search the lookup table for a match with the received control/escape
;* code. If found, execute the defined handler. 
;*
;**************************************************************************

func_exec:
	ldi	ZH,high(ctrl_table*2)	;lookup table
	ldi	ZL,low(ctrl_table*2)
fex1:	lpm				;get code
	adiw	ZL,2
	tst	R0			;test for table end
	brne	fex2
	cbr	flags,1<<ESC_FLAG	;clear possible ESC
	rjmp	fex4			;exit, no match

fex2:	cp	R0,A			;match?
	breq	fex3			;yes
	adiw	ZL,2			;no, move to next entry
	rjmp	fex1	
	
fex3:	lpm				;get handler address...				
	adiw	ZL,1
	push	R0			;...push it onto stack
	lpm
	push	R0			;do second byte
fex4:	ret				;now execute


;**************************************************************************
;*
;* LOOKUP TABLE FOR CONTROL/ESCAPE FUNCTIONS
;*
;* Each entry is four bytes long. The first byte is the control or escape
;* code, second byte is always zero, and third/fourth bytes are the
;* address of the associated handler routine.
;*
;* The table is terminated with a null.
;*
;**************************************************************************

ctrl_table:
	.db	SOH
	.dw	home_curs
	.db	EOT
	.dw	hide_curs
	.db	ENQ
	.dw	line_curs
	.db	ACK
	.dw	block_curs
	.db	BEL
	.dw	not_yet
	.db	BS
	.dw	back_space
	.db	HT
	.dw	horiz_tab
	.db	LF
	.dw	line_feed
	.db	VT
	.dw	vert_tab
	.db	FF
	.dw	clr_display
	.db	CR
	.dw	curs_return
	.db	SO
	.dw	not_yet
	.db	SI
	.dw	not_yet
	.db	DLE
	.dw	pos_curs
	.db	DC3
	.dw	scroll_on
	.db	DC4
	.dw	scroll_off
	.db	ESC
	.dw	escape
	.db	"D"
	.dw	define_char
	.db	"P"
	.dw	pos_curs_here
	.db	0			;table end


;**************************************************************************
;*
;* INITIALISE UART
;*
;**************************************************************************

init_uart:
	clr	A
	out	UCR,A			;disable UART
	sts	io_err_cnt,A		;clear async error counter

; Get baud rate parameter from EEPROM and determine divisor

	ldi	A,baud_rate
	rcall	eeread
	tst	B
	breq	iut1			;0=9600
	ldi	A,BAUD_4800
	dec	B
	breq	iut2			;1=4800
	ldi	A,BAUD_2400
	dec	B
	breq	iut2			;2=2400
iut1:	ldi	A,BAUD_9600
iut2:	out	UBRR,A			;set rate

; Init receive & transmit ring buffers

	ldi	A,rx_buff_start
	mov	rx_buff_head,A		;head = tail = rx buffer empty
	mov	rx_buff_tail,A
	ldi	A,tx_buff_start
	mov	tx_buff_head,A		;head = tail = tx buffer empty
	mov	tx_buff_tail,A

	in	A,UDR			;flush rx data register
	ldi	A,0b10011000		;enable rx, tx & rx ints
	out	UCR,A	
	cbi	PORTD,CTS_BIT		;assert CTS
	ret


;**************************************************************************
;*
;* GET CHARACTER FROM RECEIVE BUFFER
;*
;**************************************************************************

get_char:
	cbi	UCR,RXCIE		;disable rx interrputs
	
	cp	rx_buff_head,rx_buff_tail
	sec
	breq	gch4			;exit if buffer empty
	
	mov	XL,rx_buff_tail
	ld	B,X+			;get a char
	cpi	XL,rx_buff_end+1	;test for end of physical buffer
	brne	gch1			;skip if not
	
	ldi	XL,rx_buff_start	;wrap pointer
gch1:	mov	rx_buff_tail,XL
	sbis	PORTD,CTS_BIT		;skip if CTS deasserted
	rjmp	gch3
		
	mov	XL,rx_buff_head
	sub	XL,rx_buff_tail		;calc number of bytes in buffer
	brpl	gch2

	neg	XL
	ldi	A,rx_buff_size
	sub	A,XL
	mov	XL,A
	
gch2:	cpi	XL,rx_low_water		;low water mark?
	brne	gch3			;skip if not
	cbi	PORTD,CTS_BIT		; else assert CTS

gch3:	clc
gch4:	sbi	UCR,RXCIE		;re-enable rx interrputs
	ret
	

;**************************************************************************
;*
;* PUT CHARACTER IN TRANSMIT BUFFER
;*
;**************************************************************************

put_char:
	cbi	UCR,UDRIE		;disable tx interrupts

	mov	YL,tx_buff_head
	sub	YL,tx_buff_tail		;check for space first
	brpl	pch2

	com	YL
	brne	pch3			;go if not full

pch1:	lds	A,io_err_cnt		
	inc	A			;bump async error counter
	sts	io_err_cnt,A
	sec
	rjmp	pch5			;exit
		
pch2:	cpi	YL,tx_buff_size-1
	breq	pch1			;exit if buffer full

pch3:	mov	YL,tx_buff_head
	st	Y+,B			;write char to buffer
	cpi	YL,tx_buff_end+1	;test for end of physical buffer
	brne	pch4			;skip if not

	ldi	YL,tx_buff_start	;wrap pointer
pch4:	mov	tx_buff_head,YL
	clc

pch5:	sbi	UCR,UDRIE		;re-enable tx interrupts
	ret 	
	

;**************************************************************************
;*
;* UART RX COMPLETE INTERRUPT HANDLER
;*
;**************************************************************************

rx_irq:
	in	C,SREG			;save state
	push	C
	
	in	C,USR			;get UART status
	andi	C,(1<<FE)+(1<<OR)	;test for framing or overrun errs.
	in	C,UDR			;get the received byte	
	sei				;enable further interrupts
	brne	riq3			;go if error(s)

riq1:	mov	XL,rx_buff_head
	sub	XL,rx_buff_tail		;calc number of bytes in buffer
	brpl	riq2
	
	neg	XL
	ldi	C,rx_buff_size
	sub	C,XL
	mov	XL,C

riq2:	cpi	XL,rx_buff_size-1
	brne	riq4			;skip if buffer not full

riq3:	lds	C,io_err_cnt		;buffer overflow or async error...
	inc	C			;...bump error counter
	sts	io_err_cnt,C
	rjmp	riq7			;and exit
	
riq4:	cpi	XL,rx_high_water	;reached high water mark?
	brne	riq5			;skip if not

	sbi	PORTD,CTS_BIT		;negate CTS (RS232 flow control)
riq5:	mov	XL,rx_buff_head
	in	C,UDR			;get the received byte again		
	st	X+,C			;buffer it
	cpi	XL,rx_buff_end+1	;test for end of physical buffer
	brne	riq6			;skip if not
	
	ldi	XL,rx_buff_start	;wrap pointer
riq6:	mov	rx_buff_head,XL

riq7:	pop	C
	out	SREG,C			;restore state
	reti	


;**************************************************************************
;*
;* UART TX DATA REG EMPTY INTERRUPT HANDLER
;*
;**************************************************************************

tx_irq:
	in	D,SREG			;save state
	push	D
	
	cp	tx_buff_head,tx_buff_tail
	brne	tiq1			;skip if buffer not empty

	cbi	UCR,UDRIE		;disable further tx ints for now
	rjmp	tiq3
	
tiq1:	mov	YL,tx_buff_tail
	ld	D,Y+			;get a byte from the buffer
	out	UDR,D			;write it to the tx register
	cpi	YL,tx_buff_end+1	;test for end of physical buffer
	brne	tiq2			;skip if not
	
	ldi	YL,tx_buff_start	;wrap pointer
tiq2:	mov	tx_buff_tail,YL	

tiq3:	pop	D
	out	SREG,D			;restore state
	reti


;**************************************************************************
;*
;* INIT CUSTOM LCD CHARACTERS
;*
;* Initialise the 8 user-definable CG slots. CG bit patterns occupy the
;* the first 64 bytes of eeprom. 
;*
;**************************************************************************

init_cg:
	ldi	B,0b01000000
	rcall	write_lcd_cmd		;zero CGRAM address counter
	rcall	write_lcd_cmd		;do it again

	ldi	C,8*8			;chars to do * bytes/char
	ldi	ZL,symbols		;start of patterns in EEPROM
	
icg1:	mov	A,ZL
	rcall	eeread			;read a symbol pattern
	rcall	write_lcd_data		;write it to CGRAM
	inc	ZL
	dec	C
	brne	icg1			;loop to write all

	rcall	set_curs		;reset AC for DD RAM access
	ret


;**************************************************************************
;*
;* INITIALISE THE LCD MODULE
;*
;**************************************************************************

init_lcd:
	ldi	A,7
	rcall	delay_long		;wait about 100ms for LCD power-up

	ldi	A,0b11011111		;all LCD bits to outputs
	out	DDRB,A
	ldi	A,0b00100000		;RS, R/W & E bits low 
	out	PORTB,A

; The next sequence of instruction writes forces the
; LCD controller to perform a reset.

	ldi	B,0b00110000
	rcall	write_lcd_nibble	;function set: 8-bit interface
	
	ldi	A,65
	rcall	delay_short		;wait at least 4.1ms
	
	ldi	B,0b00110000
	rcall	write_lcd_nibble	;function set: 8-bit interface
	
	ldi	A,3
	rcall	delay_short		;wait at least 100us
	
	ldi	B,0b00110000
	rcall	write_lcd_nibble	;function set: 8-bit interface
	
	ldi	A,2
	rcall	delay_short		;wait at least 39us

; Reset done. Now set for 4-bit interface and
; complete the initialisation.

	cbr	flags,1<<CMD_DATA	;flag command write
	
	ldi	B,0b00100000		;function set: 4-bit interface
	rcall	write_lcd_nibble
	
	ldi	A,2			;wait at least 39us
	rcall	delay_short

	ldi	B,0b00100000		;function set: 4-bit interface
	rcall	write_lcd_nibble

	ldi	B,0b10000000		;function set: 2-line, 5x8 font
	rcall	write_lcd_nibble

	ldi	B,0b00001100		;display on, cursor off, blink off
	rcall	write_lcd_cmd

	ldi	B,0b00000001		;display clear
	rcall	write_lcd_cmd
	
	ldi	B,0b00000110		;increment mode, entire shift off
	rcall	write_lcd_cmd

	clr	lcd_addr_cntr
	rcall	clr_line_buff
	ret


;**************************************************************************
;*
;* READ/WRITE BYTE FROM/TO LCD MODULE
;*
;* Entry: B = data (writes)
;* Exit : B = data (reads)
;*
;**************************************************************************

write_lcd_cmd:
	cbr	flags,1<<CMD_DATA	;flag command write
	rcall	write_lcd
	ret
	

write_lcd_data:
	sbr	flags,1<<CMD_DATA	;flag data write
	rcall	write_lcd
	ret


; Write a data/control byte to the LCD module.

write_lcd:
	rcall	wait_lcd_rdy
	ldi	A,0b11011111
	out	DDRB,A			;all LCD bits to outputs
	ldi	A,0b00100000		;RS, R/W & E bits low 
	out	PORTB,A

; The hardware only connects the upper LCD data bus bits
; (DB4-DB7), so transfers are done in two 4-bit nibbles.

	rcall	write_lcd_nibble	;write high nibble
	rcall	write_lcd_nibble	; and low bibble
	rcall	portb_idle
	ret	


; Write nibble to LCD module.

write_lcd_nibble:
	swap	B
	mov	R1,B
	ldi	loop_cnt1,4		;number of bits to do

; Reverse bit order to match hardware.
; (PB0->DB7, PB1->DB6, etc)

wln1:	ror	R1			;shift bit to carry
	rol	A			; and back
	dec	loop_cnt1
	brne	wln1			;do complete nibble

	andi	A,0b00001111 		;discard upper nibble
	sbrc	flags,CMD_DATA		;skip if instruction write
	sbr	A,1<<LCD_RS		;set RS bit for data write
	out	PORTB,A			;write data & control
	nop
	sbi	PORTB,LCD_E		;strobe the E bit
	nop
	cbi	PORTB,LCD_E
	ret
	

read_lcd_data:
	rcall	wait_lcd_rdy		;wait for LCD ready
	ldi	A,0b11010000
	out	DDRB,A			;control bits out, data bits in
	ldi	A,0b11100000		;RS & R/W high, E low 
	out	PORTB,A
	clr	B
	rcall	read_lcd_nibble
	rcall	read_lcd_nibble
	rcall	portb_idle
	ret		


; Read a nibble from the LCD module

read_lcd_nibble:
	sbi	PORTB,LCD_E		;strobe the E bit
	nop
	in	A,PINB			;read data nibble
	cbi	PORTB,LCD_E
	clr	R1
	ldi	loop_cnt1,4		;number of bits to do

; Reverse bit order to match hardware.

rln1:	ror	A			;shift bit to carry
	rol	R1			; and back
	dec	loop_cnt1
	brne	rln1			;do complete nibble

	swap	B
	or	B,R1
	ret


;**************************************************************************
;*
;* WAIT FOR LCD READY
;*
;* A timeout on READY is included (1.6ms) so that even if the display is
;* not attached, RC5 receive still functions (and debugging faulty
;* hardware is easier, too!)
;*
;**************************************************************************

wait_lcd_rdy:
	ldi	A,0b11010000
	out	DDRB,A			;LCD data bits in, control out
	ldi	A,0b01100000		;RS & E bits low, R/W high 
	out	PORTB,A
	clr	tick_low		;clear timeout counter

wlr1:	sbi	PORTB,LCD_E		;E bit high	
	nop
	in	A,PINB			;read the LCD status register
	cbi	PORTB,LCD_E		;E bit low
	nop
	sbi	PORTB,LCD_E		;do second 4-bit transfer	
	nop
	cbi	PORTB,LCD_E
	sbrs	A,LCD_DB7		;skip if busy flag set
	rjmp	wlr2
	
	cpi	tick_low,25		;1.6ms timeout count
	brne	wlr1			;keep trying until timeout

; BUSY flag now clear. A further 1.5(1/fOSC) delay is needed for some
; controller variants before the next command.

wlr2:	ldi	A,7
wlr3:	dec	a
	brne	wlr3			;delay 21us **4MHz crystal**
	ret


;**************************************************************************
;*
;* PUT PORT B IN 'IDLE' STATE
;*
;* This is done to minimize the possibility of damage to port B if the
;* user plugs in an ISP dongle while power is applied.
;*
;**************************************************************************

portb_idle:
	ldi	A,0b00010000		;LCD E bit output, all else input
	out	DDRB,A
	ldi	A,0b11100000		;LCD E bit low, pull up RS & R/W
	out	PORTB,A
	ret


;**************************************************************************
;*
;* EEPROM RANDOM READ/WRITE ROUTINES
;*
;* Entry: A = address, B = data (writes)
;* Exit : B = data (reads)
;*
;**************************************************************************

eeread:
	sbic	EECR,EEWE
	rjmp	eeread			;loop until last cycle complete

	out	EEAR,A			;write address
	out	EEDR,B			;write data
	sbi	EECR,EERE		;set read strobe
	in	B,EEDR			;get data
	ret				;(read takes 4 clk cycles)
	
; Writes typically take 2.5ms @ 5V

eewrite:
	sbic	EECR,EEWE
	rjmp	eewrite			;loop until last cycle complete

	out	EEAR,A			;write address
	out	EEDR,B			;write data
	cli				;interrupts off for the next bit...
	sbi	EECR,EEMWE		;set master write enable
	sbi	EECR,EEWE		;set write strobe
	sei
	ret


;**************************************************************************
;*
;* TIMED DELAYS
;*
;**************************************************************************

delay_short:
	clr	tick_low		;clear counter
dls1:	cp	tick_low,A
	brlo	dls1			;A * 64us
	ret

delay_long:
	clr	tick_high		;clear counter
dll1:	cp	tick_high,A
	brlo	dll1			;A * 16.384ms
	ret


;**************************************************************************
;*
;* TRANSMIT IR CODE
;*
;* Record format is as follows:
;*
;* Byte 1 : start record (FE)
;* Byte 2 : system code
;* Byte 3 : command code
;* Byte 4 : checksum (two's complement) of bytes 2 & 3
;*
;**************************************************************************

tx_ir_code:
	cbr	flags,1<<RC5_DATA_RDY	;clear on pickup

; Remote will send same code on key press and key release. Next bit
; removes the (redundant) release. If the key if held down, we lose
; the first repeat, but get if back on release!

	lds	A,rc5_prev_cmd
	cp	rc5_command,A		;same as last command code?
	brne	tir1			;skip if not
	
	lds	A,rc5_prev_sys
	cp	rc5_system,A		;same as last system code?
	brne	tir1			;skip if not
	
	inc	rc5_repeat
	mov	A,rc5_repeat
	cpi	A,1			;key release?
	breq	tir3			;ignore if it is
	rjmp	tir2
	
tir1:	clr	rc5_repeat

; Send the IR record

tir2:	ldi	B,$FE			;start record
	rcall	put_char
	clr	R1			;init checksum
	mov	B,rc5_system		;IR system code
	add	R1,B
	rcall	put_char
	mov	B,rc5_command		;IR command code
	add	R1,B
	rcall	put_char
	mov	B,R1			;checksum
	neg	B			;two's comp it
	rcall	put_char
	sts	rc5_prev_cmd,rc5_command
	sts	rc5_prev_sys,rc5_system
tir3:	ret


;**************************************************************************
;*
;* TIMER/COUNTER 0 OVERFLOW INTERRUPT HANDLER
;*
;* Tick interval is 64us (1/fOSC x 256). Two main functions are performed
;* each interrupt, as follows:
;*
;* The 16-bit timer in tick_high & tick_low is updated.
;*
;* The RC5 receive "state machine" is executed. 
;*
;**************************************************************************

tmr0_irq:
	in	status,SREG		;save state
	push	ZH			;save used global
	push	ZL
	
	inc	rc5_cnt
	inc	tick_low		;bump low byte counter
	brne	tmi1			;skip if no overflow
	inc	tick_high		; else bump high byte

tmi1:	dec	led_on_cnt		;LED on down-count
	brne	run_state
	cbi	DDRD,LED_BIT		;count expired, switch off
	sbi	PORTD,LED_BIT
	

; Start execution in the correct state.

run_state:
	ldi	ZH,high(rc5_jmp_table)
	ldi	ZL,low(rc5_jmp_table)
	add	ZL,rc5_state
	clr	E
	adc	ZH,E
	ijmp
		
rc5_jmp_table:
	rjmp	rc5_state_0
	rjmp	rc5_state_1
	rjmp	rc5_state_2
	rjmp	rc5_state_3
	rjmp	rc5_state_4
	rjmp	rc5_state_5
	rjmp	rc5_state_6
	rjmp	rc5_state_7
	
; Wait for line high

rc5_state_0:
	sbis	PIND,IRR_BIT		;wait for line high
	rjmp	rc5_exit
		
	clr	rc5_cnt			;init counter
	inc	rc5_state		;move to state 1
	rjmp	rc5_exit	

; Establish line is idle (high) for 5ms before
; waiting for a start bit.

rc5_state_1:
	sbis	PIND,IRR_BIT		;skip if line high
	rjmp	s1_0			; else reset
	
	cpi	rc5_cnt,78		;idle high for 5ms?
	brlo	rc5_exit		;keep waiting if not

	inc	rc5_state		;move to state 2
	rjmp	rc5_exit

s1_0:	clr	rc5_state		;reset machine
	rjmp	rc5_exit
	
; Wait (forever) for start bit

rc5_state_2:
	sbic	PIND,IRR_BIT		;skip if start bit detected
	rjmp	rc5_exit

s2_0:	clr	rc5_cnt			;init counter
	inc	rc5_state		;move to state 3
	rjmp	rc5_exit

; Wait for rising edge of first start bit

rc5_state_3:
	sbis	PIND,IRR_BIT		;skip if line returned high
	rjmp	s3_0			; else go check for timeout

	clr	rc5_cnt			;init counter
	inc	rc5_state		;move to state 4
	rjmp	rc5_exit

s3_0:	cpi	rc5_cnt,31		;more than 2ms?
	brlo	rc5_exit		;nope
	
	clr	rc5_state		;yes, reset machine	
	rjmp	rc5_exit

; Wait for trailing edge of second start bit to synchronise timing

rc5_state_4:
	sbic	PIND,IRR_BIT		;skip when line goes low
	rjmp	s4_0			; else go check for timeout
	
	ldi	E,12			;init bit count
	mov	rc5_bit_cnt,E
	clr	rc5_system		;zero the bit store
	clr	rc5_command	
	clr	rc5_cnt			;init counter
	inc	rc5_state		;move to state 5
	rjmp	rc5_exit

s4_0:	cpi	rc5_cnt,31		;more than 2ms?
	brlo	rc5_exit		;nope
	clr	rc5_state		;yes, reset machine

; Restore state and exit ISR. The exit's placed here
; so as to be in relative branch range of all.

rc5_exit:
	pop	ZL
	pop	ZH
	out	SREG,status		;restore state
	reti

; Sample the line at 1/4 bit time and store result.

rc5_state_5:
	cpi	rc5_cnt,21		;3/4 bit time from last edge (1.34ms)?
	brlo	rc5_exit		;exit if not

	inc	rc5_state		;assume will be low, next state = 6
	sbic	PIND,IRR_BIT		;skip if sampled low
	rjmp	s5_1
	
	clc				;to shift in a low bit
s5_0:	rol	rc5_command
	rol	rc5_system
	clr	rc5_cnt			;init counter
	rjmp	rc5_exit

s5_1:	inc	rc5_state		;sampled high, next state = 7
	sec
	rjmp	s5_0			;go shift in the high bit

; Bit was a low, so wait for rising edge to syncronize timing

rc5_state_6:
	sbis	PIND,IRR_BIT		;skip if sampled high
	rjmp	s6_2
	
s6_0:	dec	rc5_bit_cnt		;done all bits?
	brne	s6_1			;not yet...
	
	mov	E,rc5_command		;get all the bits...
	rol	E
	rol	rc5_system		;...in the right places
	rol	E
	rol	rc5_system

	bst	rc5_system,5		;move toggle bit...
	bld	rc5_command,7		;to command byte MSB
	
	clt
	bld	rc5_system,5
	bld	rc5_command,6		;clear unused bits
	
	cbi	PORTD,LED_BIT		;switch on LED
	sbi	DDRD,LED_BIT
	ldi	E,7
	mov	led_on_cnt,E		;115ms down-count
	
	sbr	flags,1<<RC5_DATA_RDY	;flag for foreground task
	clr	rc5_state		;reset machine
	rjmp	rc5_exit
	
s6_1:	clr	rc5_cnt			;more to do, init counter
	ldi	E,5
	mov	rc5_state,E		;next state
	rjmp	rc5_exit
		
s6_2:	cpi	rc5_cnt,35		;5/4 bit time from edge (2.2ms)?
	brlo	rc5_exit
	
	clr	rc5_state		;timeout waiting for middle edge
	rjmp	rc5_exit	

; Bit was a high, so wait for falling edge to syncronize timing

rc5_state_7:
	sbis	PIND,IRR_BIT
	rjmp	s6_0
	rjmp	s6_2
	

.db	"(c)2001 SC"			;copyright string

;**************************************************************************
;*
;* DEFINE USED SRAM
;*
;**************************************************************************
	
	.dseg

rx_buff_start:	.byte	rx_buff_size-1	;serial i/o receive buffer
rx_buff_end:	.byte	1

tx_buff_start:	.byte	tx_buff_size-1	;serial i/o transmit buffer
tx_buff_end:	.byte	1

line_buff:	.byte	16		;LCD output line buffer
rc5_prev_cmd:	.byte	1		;RC5 command code save
rc5_prev_sys:	.byte	1		;RC5 system code save		
baud_save:	.byte	1		;baud rate param save
pwrdly_save:	.byte	1		;power-on delay param save
io_err_cnt:	.byte	1		;serial i/o error counter

; Reserve at least 20 bytes of stack space at top of SRAM!!


;**************************************************************************
;*
;* DEFINE USED EEPROM
;*
;**************************************************************************

	.eseg

; Custom LCD symbol bit patterns. Each pattern occupies 8 bytes,
; with space for 8 in total. Patterns below are for MP3 player.

	.org	0
symbols:	
.db	16,24,28,30,28,24,16,0		;"Play" graphic
.db	27,27,27,27,27,27,27,0		;"Pause" graphic
.db	31,31,31,31,31,31,31,0		;"Stop" graphic
.db	28,16,24,23,29,7,1,0		;"Eq" graphic
.db	28,16,28,4,28,0,0,0		;"Shuffle" graphic
.db	0,0,0,7,5,6,5,0			;"Repeat" graphic
.db	28,16,28,7,29,6,5,0		;"Shuffle" + "Repeat" graphic
.db	0				;spare slot

; Configuration

	.org	64			;must start here
baud_rate:
.db	0				;0=9600, 1=4800, 2=2400
pwr_delay:
.db	0				;power-on delay in seconds
start_msg:
.db	0				;enable/disable startup message
;					; (0=enabled)

.byte	5				;reserved
	
; Startup message (32 characters max)

	.org	72			;must start here
eemsg:
.db	"  SILICON CHIP  ",0		;terminate with zero
