; File TEMP.ASM
; Uses assembly code for PIC16F84 microcontroller
;
; Processor pin allocations are as follows:
; RA2 Output disp1 driving transistor for common anode display
; RA3 Output disp2
; RA1 Output CMOS switch temperature selection
; RA0 Output for PWM to comparator 
; RA4 Input from switch

; RB0 Input from comparator
; RB1 c segment drive for seven segment LED display
; RB2 d segment drive
; RB3 e segment drive
; RB4 f segment drive
; RB5 a segment drive
; RB6 b segment drive
; RB7 g segment drive

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

EEPROM1		equ	H'00'	; non-volatile storage for 0 deg C cal value internal sensor
EEPROM2		equ	H'01'	; 0 deg C value ls bits internal sensor
EEPROM3		equ	H'02'	; external sensor cal
EEPROM4		equ	H'03'	; external cal value
EEPROM5		equ	H'04'	; internal/external temp

; RAM

DISP1		equ	H'0C'	; HEX value test for Display1 
DISP2		equ	H'0D'	; HEX value test for Display2 
DISP3		equ	H'0E'	; HEX value test for Display3 
STATUS_TMP 	equ 	H'0F'	; temp storage for status during interrupt
W_TMP		equ	H'10'	; temporary storage for w during interrupt
FLAG_1		equ	H'11'	; bit 0 is multiplex or PWM flag, bit1 is for interrupt count
				; bit 6 is EEPROM write repeat
PWM_CNT		equ	H'12'	; counter for PWM output
LOW_TME		equ	H'13'	; PWM low time
BIN_0		equ	H'14'	; binary value lsd
BIN_1		equ	H'15'	; binary value 
BCD_0		equ	H'16'	; BCD MS
BCD_1		equ	H'17'	; display value MS
BCD_2		equ	H'18'	; display value LS
TEMP		equ	H'19'	; temporary register
CNT_16		equ	H'1A'	; counter for BCD conversion
TEMP_2		equ	H'1B'	; temp storage during EEPROM write
CAL_FLG		equ	H'1C'	; calibrate flag
TEMP_1		equ	H'1D'	; temp store 
EXTN		equ	H'1E'	; delay extension value
LOW_TM1		equ	H'1F'	; PWM least significant bits
LOW_STO		equ	H'20'	; storage of LOW_TME
FLAG3		equ	H'21'	; bit 0 flag for internal/external sensor, bit 1 for switch pressed
				; bit 2 for EEPROM copy
PWM_C1		equ	H'22'	; timer for PWM
CNT_PWM		equ	H'23'	; counter for PWM
STO_W		equ	H'24'	; w store
REFLO		equ	H'2D'	; reference voltage at 0 degrees C internal sensor
REFHI		equ	H'2E'	; reference voltage internal sensor
REFLE		equ	H'2F'	; ref voltage for external sensor
REFHE		equ	H'30'	; ref voltage for external sensor
; math routines
SIGN            equ     0x31    ; save location for sign 
TEMPB1          equ     0x32
TEMPB0          equ     0x33
TEMPD           equ     0x33    ; temporary storage
AARGB3		equ	0x34
AARGB2          equ     0x35
AARGB1          equ     0x36
AARGB0          equ     0x37
AARG            equ     0x37    ; most significant byte of argument A
BARGB1          equ     0x38
BARGB0          equ     0x39
BARG            equ     0x39    ; most significant byte of argument B
REMB1           equ     0x3A
REMB0           equ     0x3B    ; most significant byte of remainder
LOOPCOUNT       equ     0x3C    ; loop counter
STO1		equ	0x3D	; AARGB1 storage
STO2		equ	0x3E	; AARGB2 storage
; display value storage
ONEINT		equ	H'3F'	; display value for internal temperature DISP1
TWOINT		equ	H'40'	; display value for internal temperature DISP2
THRINT		equ	H'41'	; display value for internal temperature DISP3
ONEEXT		equ	H'42'	; display value for external temp DISP1
TWOEXT		equ	H'43'	; DISP2 ext
THREXT		equ	H'44'	; DISP3 ext
ONETMP		equ	H'45'	; temp display value
TWOTMP		equ	H'46'	; temp display value
THRTMP		equ	H'47'	; temp display value
DLY_CNT		equ	H'48'  	; delay counter for internal/external switching

; preprogram EEPROM DATA
	
	ORG     2100
	DE	0x8B		; EEPROMs with 0 deg C default value 
	DE	0x0C		; nominal 2.73V @ 0 deg C
	DE	0x8B		; external sensor default cal value 
	DE	0x0C
	DE	0x01		; set for internal temperature display

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data. 

SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'10111100'	; 7-segment code 1
	retlw 	B'00010010'	; 7-segment code 2
	retlw 	B'00011000'	; 7-segment code 3
	retlw 	B'00101100'	; 7-segment code 4
	retlw 	B'01001000'	; 7-segment code 5
	retlw 	B'01000000'	; 7-segment code 6
	retlw 	B'10011100'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00001000'	; 7-segment code 9
	retlw	B'00000100'	; 7-segment code for A test purposes only
	retlw	B'01100000'	; 7-segment code for b test purposes only
	retlw	B'11000010'	; 7-segment code for c test purposes only
	retlw	B'00110000'	; 7-segment code for d test purposes only
	retlw	B'01000010'	; 7-segment code for E test purposes only
	retlw	B'01000110'	; 7-segment code for F test purposes only


;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3 in sequence plus 
; uses internal timer to initiate display update
; produces PWM output with duty set by LOW_TME register
; start interrupt by saving w and status registers before altered by interrupt routine      

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf	STATUS,RP0	; select memory bank 0
	
; PWM output routine

PWM_MPX	btfsc	FLAG_1,0	; check bit 0
	goto	LOW_OUT
	bsf	FLAG_1,0	; clear if set
	nop			; output cycle time adjust
	bsf	PORTA,0		; RA0 low
	incf	CNT_PWM,f	; next pwm count
	movf	CNT_PWM,w
	andlw	0x0F		; extract ls bits, bits 0 to bits 4 
	movwf	CNT_PWM
	movf	LOW_TM1,w
	subwf	CNT_PWM,w
	movf	LOW_TME,w	; get lowtime value
	btfss	STATUS,c	; if c is 0 then add 1 to interrupt counter
	addlw	0x01		; add 1

ADD	sublw	0x02		; PWM low time
	nop			; align timer with cycles of instruction
	addwf	TMR0,f		; 4MHz so low time is inverse of 4MHz/4/2/256  
	bcf	INTCON,T0IF	; clear TMRO interrupt flag
	goto	RECLAIM		; 
LOW_OUT	bcf	FLAG_1,0	; set if clear
				; set equal time between high out and low out
	bcf	PORTA,0		; RA0 high
	movf	LOW_TM1,w
	subwf	CNT_PWM,w
	movf	LOW_TME,w	; get lowtime value
	btfss	STATUS,c	; if c is 0 then add 1 to interrupt counter
	goto	ADD2
	addlw	0x01		; add 1
	goto 	ADDL
ADD2	addlw	0x02		; align timer
ADDL	addwf	TMR0,f		; 4MHz so high time is inverse of 4MHz/4/2/256  
	bcf	INTCON,T0IF

; multiplex display routine
	
LITCHK	bsf	PORTA,4		; set high 
	bsf	FLAG_1,1	; set flag for interrupt count
	movlw	0xFE		; all segments off
	movwf	PORTB 		; blanking
			
	btfss	PORTA,2		; skip if display 1 not lit
	goto	LIT1		; display 1 lit
	btfss	PORTA,3		; skip if display 2 not lit
	goto	LIT2		; display 2 lit

; if switch closed display off
	
	btfss	PORTA,4		; if RA4 low then switch closed
	goto	CHK_FLG		; check switch flag and no display
	bcf	PORTA,2		; disp1 powered				
	btfss	FLAG3,0		; if flag3,0 high then internal sensor
	goto	MOVE1
	movf	ONEINT,w	; disp1 internal value to w use disp1 for hex testing
	bsf	PORTA,4		; external LED off	
	goto	MOV1
MOVE1	movf	ONEEXT,w	; display1 external use disp1 for hex testing
	bcf	PORTA,4		; external LED lit
MOV1	movwf 	PORTB
	goto	RECLAIM		; end of multiplex

LIT1	bcf	PORTA,3		; disp2 powered
	nop			; time to settle low
	bsf	PORTA,2		; disp1 off
	btfss	PORTA,4		; if RA4 low then switch closed
	goto	CHK_FLG		; no display
	btfss	FLAG3,0
	goto	MOVE2
	movf	TWOINT,w	; display 2 internal use DISP2 for HEX testing 
	goto	MOV2
MOVE2	movf	TWOEXT,w	; display2 external use DISP2 for hex testing
MOV2	movwf	PORTB		; seven segment value to portB
	goto	DLY_LP		; end of multiplex

; switch closed check flag3,1 if clear then set and load DLY_CNT with 50 plus toggle flag3,0
; bypass if flag3,1 already set

CHK_FLG	btfsc	FLAG3,1		; switch closed flag
	goto	DLY_LP
	bsf	FLAG3,1		; set closed switch flag (cleared after delay)
	bsf	FLAG3,2		; EEPROM copy flag
	movlw	0x50		; delay for switch debounce
	movwf	DLY_CNT		; 

	btfsc	FLAG3,0		; if set clear and if clear set
	goto	CLR_F3
	bsf	FLAG3,0
	goto	RECLAIM
CLR_F3	bcf	FLAG3,0
	goto 	RECLAIM

LIT2	bsf	PORTA,3		; disp2 off
	btfss	PORTA,4		; if RA4 low then switch closed
	goto	CHK_FLG		; check switch flags and no display
	btfss	FLAG3,0
	goto	MOVE3
	movf	THRINT,w	; display 3 internal use disp3 for hex testing
	goto	MOV3
MOVE3	movf	THREXT,w	; display3 external use disp3 for hex testing 
MOV3	movwf	PORTB		; seven segment value to portB	
		
DLY_LP	btfsc	PORTA,3		; is RA3 low wait till low 
	goto	RECLAIM
	decfsz	DLY_CNT,f	; delay counter
	goto	RECLAIM
	movlw	0xFF
	movwf	DLY_CNT		; end of delay time
	btfsc	PORTA,4		; is switch closed
	bcf	FLAG3,1		; switch pressed flag off when switch opens
	
; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN	movlw	B'01111110'	; code for 7-segment display dashes
	movwf 	DISP1		; display values
	movwf	DISP2		; initial display value
	movwf	DISP3		; initial display value
	movwf	ONEINT		; display values for internal temperature
	movwf	TWOINT
	movwf	THRINT
	movwf	ONEEXT		; display values for external temperature
	movwf	TWOEXT
	movwf	THREXT
	clrf	CNT_PWM		; counter during pwm
	clrf	LOW_TME
	clrf	LOW_TM1
	clrf	CAL_FLG
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'10000000'	; w = 10000000 binary
	movwf	OPTION_REG	; TMRO prescaler /2, PORTB pullups disabled
	movlw   B'00000000'	; (RAx outputs, RA4 output)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00011100'
	movwf	PORTA		; portA RA0, RA1 low. RA2,RA3, RA4 high

	movlw	EEPROM1		; address for EEPROM1
	call	EEREAD
	movwf	REFHI		; reference voltage at 0 deg C internal sensor 
	movlw	EEPROM2
	call	EEREAD
	movwf	REFLO	
	movlw	EEPROM3		; address for EEPROM3
	call	EEREAD
	movwf	REFHE		; reference voltage at 0 deg C external sensor
	movlw	EEPROM4
	call	EEREAD
	movwf	REFLE
	
	movlw	EEPROM5		; storage of internal/external sensor setting
	call	EEREAD
	movwf	FLAG3		; internal or external sensor flag
	btfss	FLAG3,0		; if set then internal display value
	bsf	PORTA,1		; set so will read external temperature first
	bcf	FLAG3,1		; switch closed flag
	bcf	FLAG3,2		; EEPROM copy flag

; check for calibrate

	btfsc	PORTA,4		; is cal switch pressed
	goto	NO_OPEN		; not pressed	
	bsf	CAL_FLG,0	; set cal flag
	bcf	PORTA,1		; set low first for CAL
	movlw	B'11000010'	; C for CAL
	movwf	THRINT
	movwf	THREXT
	movlw	B'00000100'	; A
	movwf	TWOINT
	movwf	TWOEXT
	movlw	B'11100010'	; L
	movwf	ONEINT
	movwf	ONEEXT
	
; interrupt enable and program now runs

NO_OPEN	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	bsf	INTCON,GIE	; set global interrupt enable for above

	
; Successive Approximation for A-D converter
	
NEWVAL	btfsc	PORTA,1		; if low then set high so select internal and external... 
	goto	EXT_S		; ...sensors alternatively
	bsf	PORTA,1
	goto	INT_S
EXT_S	bcf	PORTA,1
INT_S	clrf	LOW_TME		; ms bits make low first
	clrf	LOW_TM1		; ls bits
	
	bsf	LOW_TME,7	; bit 7 set
	movlw	0x08
	call	SAR
	btfss 	PORTB,0		; comparator out. if RB0 high then value below 
	bcf	LOW_TME,7	; bit 7 cleared
	bsf	LOW_TME,6	; next LS bit test
	movlw	0x08
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 6)
	bcf	LOW_TME,6
	bsf	LOW_TME,5
	movlw	0x07
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 5)
	bcf	LOW_TME,5
	bsf	LOW_TME,4
	movlw	0x06
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 4)
	bcf	LOW_TME,4
	bsf	LOW_TME,3
	movlw	0x06
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 3)
	bcf	LOW_TME,3
	bsf	LOW_TME,2
	movlw	0x05
	call 	SAR
	btfss	PORTB,0		; if RB0 high then value below (high bit 2)
	bcf	LOW_TME,2
	bsf	LOW_TME,1
	movlw	0x05
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit 1)
	bcf	LOW_TME,1
	bsf	LOW_TME,0
	movlw	0x05
	call 	SAR
	btfss	PORTB,0		; bit 0 test
	bcf	LOW_TME,0

; extra ls bits to add in 4-bit resolution to above 8-bits
	
	bsf	LOW_TM1,3	; next LS bit test
	movlw	0x04
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit) 
	bcf	LOW_TM1,3
	bsf	LOW_TM1,2	; next LS bit test
	movlw	0x04
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit)
	bcf	LOW_TM1,2
	bsf	LOW_TM1,1
	movlw	0x04
	call 	SAR
	btfss 	PORTB,0		; if RB0 high then value below (high bit) 
	bcf	LOW_TM1,1
;	bsf	LOW_TM1,0	; next LS bit test (for 12-bit A/D) not used
;	movlw	0x04
;	call 	SAR
;	btfss	PORTB,0
;	bcf	LOW_TM1,0
	

CALC	movf	LOW_TME,w	; store 255 minus lowtime value into binary0 register
	sublw	D'255'		; get reverse value ie high output time
	movwf	LOW_TME		; for multiply
	movf	LOW_TM1,w	; ls bits
	sublw	0xFF		; get reverse value ie high output time
	andlw	0x0F		; extract bits
	movwf	LOW_TM1		;

; check if in cal mode

	btfss	CAL_FLG,0	; if set then store new cal
	goto	C_SGN

; write new calibration to EEPROM

MODESTO	btfss	PORTA,1		; internal or external sensor
	goto	EXT_CAL
	movlw	EEPROM1		; address for EEPROM1
	movwf	EEADR		; address for write
	movf	LOW_TME,w
	movwf	REFHI
	call	EWRITE		; write to EEPROM
	movlw	EEPROM2		; address for EEPROM2
	movwf	EEADR		; address for write
	movf	LOW_TM1,w
	movwf	REFLO
	call	EWRITE		; write to EEPROM
	goto	C_SGN
EXT_CAL movlw	EEPROM3		; address for EEPROM3
	movwf	EEADR		; address for write
	movf	LOW_TME,w
	movwf	REFHE
	call	EWRITE		; write to EEPROM
	movlw	EEPROM4		; address for EEPROM4
	movwf	EEADR		; address for write
	movf	LOW_TM1,w
	movwf	REFLE
	call	EWRITE		; write to EEPROM
	bcf	CAL_FLG,0	; end of cal
	

; multiply 273

C_SGN	clrf	SIGN		; set minus sign to plus, disp2 blanking off
	movlw	0x01		; 111Hexadecimal is 273Decimal 
	movwf	AARGB0
	movlw	0x11
	movwf	AARGB1

; by Vout(t)
 
	swapf	LOW_TME,w	; ms digits
	andlw	0x0F		; extract ms 4-digits

; (*) to show hex values of A-D conversion on display. Test puposes only.

;	call	svnseg		; * show Hex *
;	movwf	THRINT		; *
;	movwf	THREXT		; *
	movwf	BARGB0		; ms byte
	
	swapf	LOW_TME,w	; ls digits
	andlw	0xF0		; extract ls 4-digits
;	movf	low_tme,w	; * for hex display *
;	andlw	0x0F		; *
;    	call	svnseg		; * show Hex *
;	movwf	TWOINT		; *
;	movwf	TWOEXT		; *
	movwf	BARGB1

	movf	LOW_TM1,w	; ls bits
;	call	svnseg		; *	
;	movwf	ONEINT		; * show hex *
;	movwf	ONEEXT		; *
	addwf	BARGB1,f
;	goto	newval		; * bypass math * 

	call    FXM1616U	; multiply. Result is in AARGB

; move across by 8-bits ie aargb1 to aargb0, aargb2 to aargb1 aargb3 to 2 clear 3

	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2

; check if internal or external sensor reading

	btfss	PORTA,1		; if low then external sensor
	goto	LD_EXT

; internal

	swapf	REFHI,w		; ms digits
	andlw	0x0F		; extract ms 4-digits
	movwf	BARGB0		; ms byte
	swapf	REFHI,w		; ls digits
	andlw	0xF0		; extract ls 4-digits
	movwf	BARGB1
	movf	REFLO,w		; ls bits
	addwf	BARGB1,f
	goto	DIV_CON		; continue with divide	

; external	

LD_EXT	swapf	REFHE,w		; ms digits
	andlw	0x0F		; extract ms 4-digits
	movwf	BARGB0		; ms byte
	swapf	REFHE,w		; ls digits
	andlw	0xF0		; extract ls 4-digits
	movwf	BARGB1
	movf	REFLE,w		; ls bits
	addwf	BARGB1,f
	
DIV_CON	call	FXD2416U	; divide
	movf	AARGB1,w
	movwf	STO1
	movf	AARGb2,w
	movwf	STO2
	call	D_SUB		; subtract
	btfss	AARGB1,7	; if bit 7 set then redo subtraction and add (-) minus sign
	goto	CONV
	
	movf	STO1,w
	movwf	BARGB0
	movf	STO2,w
	movwf	BARGB1
	movlw	0x01
	movwf	AARGB1
	movlw	0x11
	movwf	AARGB2
	call	D_RSUB		; subtract in reverse set minus sign
	bsf	SIGN,0 

CONV	movf	AARGB1,w
	movwf	BIN_1
	movf	AARGB2,w
	movwf	BIN_0
	call	BCD		; convert to BCD values
	movf	BCD_1,w		; ms value to w
	andlw	0x0F		; compare with 0
	btfss	STATUS,z	; skip if zero
	goto 	SHOW		; show digit on disp3
	movlw	B'11111110'	; 7-segment display off
	bsf	SIGN,1		; blanking for disp2
	goto	BLNK		; value to blank display
SHOW	movf	BCD_1,w
	andlw	0x0F		; extract ms bits	
	call 	SVNSEG		; convert to 7-segment value for display
BLNK	movwf	THRTMP		; transfer BCD MSD to DISPlay 3
	btfsc	SIGN,0		; if sign set then drive - sign
	bcf	THRTMP,7	
	movf	BCD_2,w		; least sig byte
	andlw	0x0F		; extract LS bits
	call	SVNSEG
	movwf	ONETMP		; place in display 1
	swapf	BCD_2,w
	andlw	0x0F		; extract bits
	btfss	STATUS,z	; if zero then maybe blank 
	goto	NOBNK_2
	btfss	SIGN,1		; is disp3 blanked
	goto	NOBNK_2
	movlw	B'11111110'	; 7-segment display off
	goto	BLNK_2
NOBNK_2	call	SVNSEG
BLNK_2	movwf	TWOTMP		; place in display register
	
; test if internal or external temperature reading

	btfsc	PORTA,1		; if high then internal
	goto 	INTSNS		; internal sensor
	movf	ONETMP,w	; move temp values to external display values	
	movwf	ONEEXT
	movf	TWOTMP,w
	movwf	TWOEXT
	movf	THRTMP,w
	movwf	THREXT
	goto	NEWVAL		; read input voltage
INTSNS	movf	ONETMP,w	; move temp values to internal display values
	movwf	ONEINT
	movf	TWOTMP,w
	movwf	TWOINT
	movf	THRTMP,w
	movwf	THRINT
	goto 	NEWVAL		; read input voltage again

; ******************************************************************************
; Subroutines


; Subroutine for PWM cycle period for successive approximation A-D converter 
; pwm_cnt must be multiple of 16

SAR	clrf	CNT_PWM		; ls bits counter
		
DEL1	movwf	PWM_C1
SAR_AGN	movlw	0xE0		; number of interrupts between PWM changes
	movwf	PWM_CNT
CNT_AGN	bcf	FLAG_1,1	; bit set in interrupt at multiplex rate
CNT_NOW	btfss	FLAG_1,1	; look at flag
	goto	CNT_NOW		; wait till flag set at multiplex interrupt 
	decfsz 	PWM_CNT,f	; reduce this value 
	goto	CNT_AGN		; cycle
	decfsz	PWM_C1,f
	goto	SAR_AGN	 	; loop again

; if FLAG3,2 set write to EEPROM

WAIT_LO	btfss	FLAG3,2		; EEPROM copy flag
	return
	movlw	EEPROM5		; address for internal/external EEPROM5 storage
	movwf	EEADR		; address for write
	movf	FLAG3,w
	call	EWRITE		; write to EEPROM
	bcf	FLAG3,2		; stop EEPROM copy
	return

; Subroutine to convert from 16-bit binary to 3-digit BCD (packed)
; Binary value is in BIN0 & BIN1. BIN0 is low byte BIN1 is high byte
; Result in BCD is in BCD0, BCD1 & BCD2.  
; BCD0 is MSB, BCD2 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'16'
	movwf	CNT_16		; 16 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
	clrf	BCD_2
	

LOOPBCD	rlf	BIN_0,f
	rlf	BIN_1,f		; LSB shift left binary registers
	rlf	BCD_2,f		; LSB shift left BCD registers
	rlf	BCD_1,f
	rlf	BCD_0,f

	decfsz	CNT_16,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_2		; BCD LSB address
	movwf	FSR		; pointer for BCD2
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_1
	movwf	FSR
	call 	ADJBCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-2)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_RD	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_RD		; wait for low RD (read RD)	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	bcf	FLAG_1,6	; EEPROM write repeat flag
	movwf	TEMP_2		; store w in temporary storage
EWRIT	movf	TEMP_2,w	; place value in w (used for read data sequence) 
	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	
; read EEPROM DATA and check if written correctly
	
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_AGN	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_AGN		; wait for low RD	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	subwf	EEDATA,w	; compare read value with value stored
	btfsc	STATUS,z	; skip if not the same
	return			; value correctly written 
	btfsc	FLAG_1,6	; write repeat bit, skip if clear and rewrite
	return
	bsf	FLAG_1,6	; set repeat flag rewrite once only 
	goto 	EWRIT		; rewrite as not written correctly
	
; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LA 	RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LC 	RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,c
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L		RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,c
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,c
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,c
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,c
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,c
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,c
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,c
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,c
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN

; subroutine subtract AARGB1 AARGB2 - BARGB0 BARGB1 = AARGB
	
D_SUB	movlw	0x01		; 0111H is 273
	movwf	BARGB0 
	movlw	0x11
	movwf	BARGB1 
D_RSUB	call	NEG_A		; complement of A

	movf	BARGB1,w 
	addwf	AARGB2,f 	; add lsb
	btfsc	STATUS,c	; add carry
	incf	AARGB1,f 
	movf	BARGB0,w 
	addwf	AARGB1,f 
	return

NEG_A	comf	BARGB1,f 
	incf	BARGB1,f 
	btfsc	STATUS,z
	decf	BARGB0,f 
	comf	BARGB0,f 
	return	

	
	
	end	
