
; High power lamp dimmer using phase control
; also includes auto dimming


; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _HS_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations
; EEPROM DATA


EEPROM1		equ	H'00'	; non-volatile storage for dim level 
EEPROM2		equ	H'01'	; non-volatile storage for flash level
EEPROM3		equ	H'02'	; non-volatile storage for min level
EEPROM4		equ	H'03'	; non-volatile storage for A-auto rate
EEPROM5		equ	H'04'	; non-volatile storage for B-auto rate 	


; RAM

W_TMP		equ	H'0C'	; temporary store for w in interrupt
STATUS_TMP	equ	H'0D'	; temporary store of status in interrupt  
BRIGHT_C	equ	H'0E'	; current lamp brightness 
BRIGHT_S	equ	H'0F'	; brightness setting for lookup table 
BRIGHT_V	equ	H'10'	; stored brightness
MPX_CNT		equ	H'11'	; display multiplex counter 
STOR1		equ	H'12'	; storage
POSITION	equ	H'13'	; position counter within half sine wave starting from zero crossing
RATE		equ	H'14'	; rate execution counter
FLAG		equ	H'15'	; bit 0, off flag, 1 on, 2 up, 3 down 
COUNTZ_Z	equ	H'16'	; counter from zero crossing to zero crossing
COUNTZ		equ	H'17'	; counter storage of z-z counter
SW_FLG		equ	H'18'	; switch indicator flags
SW_FLG2		equ	H'19'	; switch indicator flags
FLAG_1		equ	H'1A'	; repeat flag during EEPROM write
TEMP_2		equ	H'1B'	; storage for offset
TEMP_3		equ	H'1C'	; lev_con (level conversion) storage of dim value
FLAG_2		equ	H'1D'	; auto dim flag
FLAG_ON		equ	H'1E'	; 0, -. 1,up/dn. 2,
				; 3,  4, stop dimming 
DIVIDE		equ	H'1F'	; divider of timer counter
TIMER		equ	H'20'	; temp timer value
FLAG_Z		equ	H'21'	; zero crossing flag
PULSE_V		equ	H'22'	; pulse value 
PULSE_O		equ	H'23'	; operational pulse value
DIVID_O		equ	H'24'	; operational divide value
TEMP		equ	H'25'	; temporary store
SPAC_E		equ	H'26'	; counter to break up interrupt routine to run over several IRs 
TIME_D		equ	H'27'	; auto dim timer
TIM_FLG		equ	H'28'	; flag to stop time delay till ready
FLG_OFF 	equ	H'29'	; gate drive off flag
GAP_T		equ	H'2A'	; timer for delay between LEDs being driven
DEL_STO		equ	H'2B'	; delay storage
EP_FLG		equ	H'2C'	; flag to indicate EEPROM storage in process
FLSH_LEV	equ	H'2D'	; flash level
DIM_LEV		equ	H'2E'	; dimming level
AUTO_A		equ	H'2F'	; auto dim rate A
AUTO_B		equ	H'30'	; auto dim rate B
MIN_LEV		equ	H'31'	; minimum dim level
FLAG3		equ	H'32'	; flag for resetting 4017
FLASH		equ	H'33'	; flash timer for current brightness

; preprogram EEPROM DATA (00-3F from 2100-213F)
	
	ORG     2100		; start at 00
	
	DE	D'38', D'22', D'154', D'01', D'05'	

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT

; lookup table for lamp brightness versus phase and convert to power curve for brightness

BRIGH_T movwf	TEMP_2	; temporary store
	movlw	D'180'	; 
	subwf	TEMP_2,w
	btfsc	STATUS,c; if c is zero then negative
	retlw	D'241'	; return 241 as greater than table
	
	movlw	D'38'	; 
	subwf	TEMP_2,f
	btfss	STATUS,c; if c is zero then negative
	retlw	D'8'	; return 8 as less than or equal to 0

	movf	TEMP_2,w
	addwf	PCL,f	; add value to program counter
	
	retlw	D'12'	; 39
	retlw	D'16'	; 40 
	retlw	D'20'	; 41 
	retlw	D'24'	; 42 
	retlw	D'28'	; 43 
	retlw	D'32'	; 44 
	retlw	D'36'	; 45 
	retlw	D'40'	; 46 
	retlw	D'44'	; 47 
	retlw	D'48'	; 48 
	retlw	D'52'	; 49 
	retlw	D'56'	; 50 
	retlw	D'60'	; 51 
	retlw	D'64'	; 52 
	retlw	D'68'	; 53 
	retlw	D'71'	; 54 
	retlw	D'74'	; 55 	
	retlw	D'77'	; 56 
	retlw	D'80'	; 57 
	retlw	D'83'	; 58 
	retlw	D'86'	; 59 
	retlw	D'89'	; 60 
	retlw	D'92'	; 61 
	retlw	D'95'	; 62
	retlw	D'98'	; 63 
	retlw	D'100'	; 64 
	retlw	D'102'	; 65 	
	retlw	D'104'	; 66 
	retlw	D'106'	; 67 
	retlw	D'108'	; 68 
	retlw	D'110'	; 69 
	retlw	D'112'	; 70
	retlw	D'114'	; 71 
	retlw	D'116'	; 72 
	retlw	D'118'	; 73 
	retlw	D'120'	; 74 
	retlw	D'122'	; 75 
	retlw	D'124'	; 76 
	retlw	D'126'	; 77 	
	retlw	D'128'	; 78 
	retlw	D'130'	; 79 
	retlw	D'132'	; 80 
	retlw	D'134'	; 81 
	retlw	D'136'	; 82 
	retlw	D'138'	; 83 
	retlw	D'140'	; 84 
	retlw	D'142'	; 85 
	retlw	D'144'	; 86 
	retlw	D'146'	; 87 
	retlw	D'148'	; 88 
	retlw	D'150'	; 89 	
	retlw	D'151'	; 90 
	retlw	D'152'	; 91 
	retlw	D'153'	; 92 
	retlw	D'154'	; 93 
	retlw	D'155'	; 94
	retlw	D'156'	; 95 
	retlw	D'157'	; 96 
	retlw	D'158'	; 97 
	retlw	D'159'	; 98 
	retlw	D'160'	; 99
	retlw	D'161'	; 100 
	retlw	D'162'	; 101 	
	retlw	D'163'	; 102 
	retlw	D'164'	; 103 
	retlw	D'165'	; 104 
	retlw	D'166'	; 105 
	retlw	D'167'	; 106
	retlw	D'168'	; 107 
	retlw	D'169'	; 108 
	retlw	D'170'	; 109 
	retlw	D'171'	; 110 
	retlw	D'172'	; 111 
	retlw	D'173'	; 112 
	retlw	D'174'	; 113 	
	retlw	D'175'	; 114 
	retlw	D'176'	; 115 
	retlw	D'177'	; 116 
	retlw	D'178'	; 117 
	retlw	D'179'	; 118
	retlw	D'180'	; 119 
	retlw	D'181'	; 120 
	retlw	D'182'	; 121 
	retlw	D'183'	; 122 
	retlw	D'184'	; 123 
	retlw	D'185'	; 124 
	retlw	D'186'	; 125 	
	retlw	D'187'	; 126 
	retlw	D'188'	; 127 
	retlw	D'189'	; 128 
	retlw	D'190'	; 129 
	retlw	D'191'	; 130
     	retlw	D'192'	; 131 
	retlw	D'193'	; 132 
	retlw	D'194'	; 133 	
	retlw	D'195'	; 134 
	retlw	D'196'	; 135 
	retlw	D'197'	; 136 
	retlw	D'198'	; 137 
	retlw	D'199'	; 138
	retlw	D'200'	; 139 
	retlw	D'201'	; 140 
	retlw	D'202'	; 141 
	retlw	D'203'	; 142 
	retlw	D'204'	; 143 
	retlw	D'205'	; 144 
	retlw	D'206'	; 145 	
	retlw	D'207'	; 146 
	retlw	D'208'	; 147 
	retlw	D'209'	; 148 
	retlw	D'210'	; 149 
	retlw	D'211'	; 150
	retlw	D'212'	; 151
	retlw	D'213'	; 152 
	retlw	D'214'	; 153   
	retlw	D'215'	; 154 
	retlw	D'216'	; 155 	
	retlw	D'217'	; 156 
	retlw	D'218'	; 157 
	retlw	D'219'	; 158 
	retlw	D'220'	; 159 
	retlw	D'221'	; 160
	retlw	D'222'	; 161 
	retlw	D'223'	; 162 
	retlw	D'224'	; 163 
	retlw	D'225'	; 164 	
	retlw	D'226'	; 165 
	retlw	D'227'	; 166 
	retlw	D'228'	; 167 
	retlw	D'229'	; 168 
	retlw	D'230'	; 169
	retlw	D'231'	; 170
	retlw	D'232'	; 171
	retlw	D'233'	; 172 
	retlw	D'234'	; 173 
	retlw	D'235'	; 174 
	retlw	D'236'	; 175 	
	retlw	D'237'	; 176 
	retlw	D'238'	; 177 
	retlw	D'239'	; 178 
	retlw	D'240'	; 179
	retlw	D'241'	; 180 
	retlw	D'242'	; 181
	 

; convert DIM_LEV (0-39 on LED display) to BRIGHT_S value

LEV_CON addwf	PCL,f	; add value to program counter
	retlw	D'153'	; zero on display (DIM_LEV)
	retlw	D'150'	; 1 on display
	retlw	D'147'	; 2 on display
	retlw	D'144'	; 3 on display
	retlw	D'141'	; 4 on display
	retlw	D'138'	; 5 on display
	retlw	D'135'	; 6 on display
	retlw	D'132'	; 7 on display
	retlw	D'129'	; 8 on display
	retlw	D'126'	; 9 on display
	retlw	D'123'	; 10 on display
	retlw	D'120'	; 11 on display
	retlw	D'117'	; 12 on display
	retlw	D'114'	; 13 on display
	retlw	D'111'	; 14 on display
	retlw	D'108'	; 15 on display
	retlw	D'105'	; 16 on display
	retlw	D'102'	; 17 on display
	retlw	D'99'	; 18 on display
	retlw	D'96'	; 19 on display
	retlw	D'93'	; 20 on display
	retlw	D'90'	; 21 on display
	retlw	D'87'	; 22 on display
	retlw	D'84'	; 23 on display
	retlw	D'81'	; 24 on display
	retlw	D'78'	; 25 on display
	retlw	D'75'	; 26 on display
	retlw	D'72'	; 27 on display
	retlw	D'69'	; 28 on display
	retlw	D'66'	; 29 on display
	retlw	D'63'	; 30 on display
	retlw	D'60'	; 31 on display
	retlw	D'57'	; 32 on display
	retlw	D'54'	; 33 on display
	retlw	D'51'	; 34 on display
	retlw	D'48'	; 35 on display
	retlw	D'45'	; 36 on display
	retlw	D'42'	; 37 on display
	retlw	D'39'	; 38 on display
	retlw	D'36'	; 39 on display

; switch values

SWVAL	addwf	PCL,f	; add value to program counter
	retlw	B'00000001' 	; zero switch (set)
	retlw	B'00000010'	; 1 switch (set dimming level up)
	retlw	B'00000100'	; 2 switch (dim down)
	retlw	B'00001000'	; 3 switch (flash off)
	retlw	B'00010000'	; 4 switch (flash on)
	retlw	B'00100000'	; 5 switch (set dimming level down)
	retlw	B'01000000'	; 6 switch (dim up)
	retlw	B'10000000'	; 7 switch (nil)

; ......................................................................
; subroutines

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	bcf	STATUS,RP0	; select bank 0
	return

; check 0-7 switches on 4017 counter outputs (from interrupt)

CK_07	movlw	0x00		; page 0 PCLATH address
	movwf	PCLATH
	btfss	PORTA,0		; if high then a switch is pressed
	goto	NO_SW		; no switch pressed
	movf	MPX_CNT,w	; multiplex count value (0-7)
	call	SWVAL		; get switch value
	iorwf	SW_FLG2,f	; set flag 	

; check when dimming up or down

	btfss	FLAG_2,0	; dimming up in progress
	goto	CK_DDN		; check dimming down
	btfsc	SW_FLG2,2	; is down switch pressed
R_SET	goto	T_SET		; set switch reversed flag if down switch pressed
	goto	RECLAIM		; CK_FLS
CK_DDN 	btfss	FLAG_2,1	; dimming down in progress
	goto	RECLAIM
	btfss	SW_FLG2,6	; is up switch pressed	
	goto	RECLAIM
	
T_SET	bsf	FLAG_2,7
	clrf	TIME_D
	goto	RECLAIM

NO_SW	movf	MPX_CNT,w	; if off then turn off switch flag	
	call 	SWVAL
	movwf	TEMP
	comf	TEMP,w		; complementary value from SWVAL lookup table
	andwf	SW_FLG2,f	; clear flag
	goto	RECLAIM


; ............................................................................	

; start at next page address > 0100H

; LED display values

LDVAL	addwf	PCL,f	; add value to program counter

	retlw	B'01111111'	; 1
	retlw	B'00111111'	; 2
	retlw	B'10111111'	; 3
	retlw	B'10011111'	; 4
	retlw	B'11011111'	; 5
	retlw	B'11001111'	; 6
	retlw	B'11101111'	; 7
	retlw	B'11100111'	; 8
	retlw	B'11110111'	; 9
	retlw	B'11110111'	; 10
	
	retlw	B'11110111'	; 11
	retlw	B'11100111'	; 12
	retlw	B'11101111'	; 13
	retlw	B'11001111'	; 14
	retlw	B'11011111'	; 15
	retlw	B'10011111'	; 16
	retlw	B'10111111'	; 17
	retlw	B'00111111'	; 18
	retlw	B'01111111'	; 19
	retlw	B'01111111'	; 20
	
	retlw	B'01111111'	; 21
	retlw	B'00111111'	; 22
	retlw	B'10111111'	; 23
	retlw	B'10011111'	; 24
	retlw	B'11011111'	; 25
	retlw	B'11001111'	; 26
	retlw	B'11101111'	; 27
	retlw	B'11100111'	; 28
	retlw	B'11110111'	; 29
	retlw	B'11110111'	; 30

	retlw	B'11110111'	; 31
	retlw	B'11100111'	; 32
	retlw	B'11101111'	; 33
	retlw	B'11001111'	; 34
	retlw	B'11011111'	; 35
	retlw	B'10011111'	; 36
	retlw	B'10111111'	; 37
	retlw	B'00111111'	; 38
	retlw	B'01111111'	; 39


; convert from counter to display value
	

CURRENT	movwf	TEMP_2	; temporary store
	movlw	D'180'	; 
	subwf	TEMP_2,w
	btfsc	STATUS,c; if c is zero then negative
	retlw	D'0'	; return 0 as greater than table

	movlw	D'38'	; 
	subwf	TEMP_2,f
	btfss	STATUS,c; if c is zero then negative
	retlw	D'38'	; return 38 as less than or equal to 0

	movf	TEMP_2,w
	addwf	PCL,f	; add value to program counter
	retlw	D'38'	; 39
	retlw	D'38'	; 40 
	retlw	D'38'	; 41 
	retlw	D'37'	; 42 
	retlw	D'37'	; 43 
	retlw	D'37'	; 44 
	retlw	D'36'	; 45 
	retlw	D'36'	; 46 
	retlw	D'36'	; 47 
	retlw	D'35'	; 48 
	retlw	D'35'	; 49 
	retlw	D'35'	; 50 
	retlw	D'34'	; 51 
	retlw	D'34'	; 52 
	retlw	D'34'	; 53 
	retlw	D'33'	; 54 
	retlw	D'33'	; 55 	
	retlw	D'33'	; 56 
	retlw	D'32'	; 57 
	retlw	D'32'	; 58 
	retlw	D'32'	; 59 
	retlw	D'31'	; 60 
	retlw	D'31'	; 61 
	retlw	D'31'	; 62
	retlw	D'30'	; 63 
	retlw	D'30'	; 64 
	retlw	D'30'	; 65 	
	retlw	D'29'	; 66 
	retlw	D'29'	; 67 
	retlw	D'29'	; 68 
	retlw	D'28'	; 69 
	retlw	D'28'	; 70
	retlw	D'28'	; 71 
	retlw	D'27'	; 72 
	retlw	D'27'	; 73 
	retlw	D'27'	; 74 
	retlw	D'26'	; 75 
	retlw	D'26'	; 76 
	retlw	D'26'	; 77 	
	retlw	D'25'	; 78 
	retlw	D'25'	; 79 
	retlw	D'25'	; 80 
	retlw	D'24'	; 81 
	retlw	D'24'	; 82 
	retlw	D'24'	; 83 
	retlw	D'23'	; 84 
	retlw	D'23'	; 85 
	retlw	D'23'	; 86 
	retlw	D'22'	; 87 
	retlw	D'22'	; 88 
	retlw	D'22'	; 89 	
	retlw	D'21'	; 90 
	retlw	D'21'	; 91 
	retlw	D'21'	; 92 
	retlw	D'20'	; 93 
	retlw	D'20'	; 94
	retlw	D'20'	; 95 
	retlw	D'19'	; 96 
	retlw	D'19'	; 97 
	retlw	D'19'	; 98 
	retlw	D'18'	; 99
	retlw	D'18'	; 100 
	retlw	D'18'	; 101 	
	retlw	D'17'	; 102 
	retlw	D'17'	; 103 
	retlw	D'17'	; 104 
	retlw	D'16'	; 105 
	retlw	D'16'	; 106
	retlw	D'16'	; 107 
	retlw	D'15'	; 108 
	retlw	D'15'	; 109 
	retlw	D'15'	; 110 
	retlw	D'14'	; 111 
	retlw	D'14'	; 112 
	retlw	D'14'	; 113 	
	retlw	D'13'	; 114 
	retlw	D'13'	; 115 
	retlw	D'13'	; 116 
	retlw	D'12'	; 117 
	retlw	D'12'	; 118
	retlw	D'12'	; 119 
	retlw	D'11'	; 120 
	retlw	D'11'	; 121 
	retlw	D'11'	; 122 
	retlw	D'10'	; 123 
	retlw	D'10'	; 124 
	retlw	D'10'	; 125 	
	retlw	D'9'	; 126 
	retlw	D'9'	; 127 
	retlw	D'9'	; 128 
	retlw	D'8'	; 129 
	retlw	D'8'	; 130
     	retlw	D'8'	; 131 
	retlw	D'7'	; 132 
	retlw	D'7'	; 133 	
	retlw	D'7'	; 134 
	retlw	D'6'	; 135 
	retlw	D'6'	; 136 
	retlw	D'6'	; 137 
	retlw	D'5'	; 138
	retlw	D'5'	; 139 
	retlw	D'5'	; 140 
	retlw	D'4'	; 141 
	retlw	D'4'	; 142 
	retlw	D'4'	; 143 
	retlw	D'3'	; 144 
	retlw	D'3'	; 145 	
	retlw	D'3'	; 146 
	retlw	D'2'	; 147 
	retlw	D'2'	; 148 
	retlw	D'2'	; 149 
	retlw	D'1'	; 150
	retlw	D'1'	; 151
	retlw	D'1'	; 152 
	retlw	D'0'	; 153 
	retlw	D'0'	; 154 
	retlw	D'0'	; 155 	
	retlw	D'0'	; 156 
	retlw	D'0'	; 157 
	retlw	D'0'	; 158 
	retlw	D'0'	; 159 
	retlw	D'0'	; 160
	retlw	D'0'	; 161 
	retlw	D'0'	; 162 
	retlw	D'0'	; 163 
	retlw	D'0'	; 164 	
	retlw	D'0'	; 165 
	retlw	D'0'	; 166 
	retlw	D'0'	; 167 
	retlw	D'0'	; 168 
	retlw	D'0'	; 169
	retlw	D'0'	; 170
	retlw	D'0'	; 171
	retlw	D'0'	; 172 
	retlw	D'0'	; 173 
	retlw	D'0'	; 174 
	retlw	D'0'	; 175 	
	retlw	D'0'	; 176 
	retlw	D'0'	; 177 
	retlw	D'0'	; 178 
	retlw	D'0'	; 180
	retlw	D'0'	; 181 

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf	STATUS,RP0	; select memory bank 0

; check which interrupt	

	btfsc	INTCON,INTF	; INT interrupt flag set then goto zero
	goto	ZERO		; zero crossing signal update
	btfsc	INTCON,T0IF	; TMRO overflow interrupt flag then goto counter
	goto	COUNTER		; TMRO overflow so increase counter 
	goto	RECLAIM		; end of interrupt reclaim w and status

; POSITION counter timer 250 counts are 10ms 

COUNTER	incf	POSITION,f	; position in half wave starting from zero crossing point

; adjust interrupt rate with counter

	movf	DIVIDE,w	; divide value
	movwf	DIVID_O		; operating divide value
	movf	PULSE_O,f	; pulse value number of interrupt speedups
	btfsc	STATUS,z	; if zero leave
	goto	NO_CHNG
	incf	DIVID_O,f
	decf	PULSE_O,f	; freq is 20MHz/4/4/50. =25kHz or 40us period
NO_CHNG	movf	TMR0,w		; each change of DIVID_O speeds interrupt time to 39.2us	
	addwf	DIVID_O,w	; add to timer register and takes 2 cycles to start counting
	movwf	TMR0	
	bcf	INTCON,T0IF	; clear TMRO interrupt flag
 
; increase zero crossing counter
	
	incfsz	COUNTZ_Z,f	; increase the zero crossing to zero crossing counter
	goto	FIRST
	bsf	FLAG_Z,0	; if zero set flag 
FIRST	btfsc	FLAG_Z,0	; if flag set then second half of sine wave
	goto	POSN
	movf	COUNTZ_Z,w
	xorwf	COUNTZ,w	; check if same (countZ = 250)
	btfsc	STATUS,z	; if zero then half way through zero-zero crossing point
	goto	ZERO_H		; go to half way between zero crossings
	goto	NOT_PK

; check if POSITION counter = BRIGHT_C counter then trigger @ 40us gate drive each

POSN	movf	COUNTZ_Z,w
	xorlw	D'120'		; check if around max peak of sine wave 
	btfss	STATUS,z	; if zero then check RB0
	goto	NOT_PK
	btfss	PORTB,0		; if low then power is off
	bsf	FLG_OFF,0	; gate pulses off flag

NOT_PK	btfsc	FLG_OFF,0	; if set then gate drive off
	goto	G_DV_O		; gate drive off
	movf	POSITION,w	; look at position in mains waveform
	xorwf	BRIGHT_C,w	; current brightness. If zero then drive Triac gate
	btfsc	STATUS,z	; if zero bit high drive gate (is off, on at next timer interrupt)
	goto	FLG_TST
	incf	BRIGHT_C,w	; drive gate for second time so total of 80us
	xorwf	POSITION,w
	btfss	STATUS,z
	goto	G_DV_O
FLG_TST	movlw	B'11100111'	; gate drive low (RA3/4) (on) (LED driven via RA2) 
	andwf	PORTA,f		; gate drive

; drive lamp LED from minimum level value up

N_LEDS	btfsc	FLG_OFF,0	; if set then LED off
	goto	SPACE_1		; LED drive kept off
	movlw	0x00		; page 0 PCLATH address
	movwf	PCLATH
	movf	MIN_LEV,w	; minimum brightness (preheat)
	call	BRIGH_T		; lookup table
	sublw	D'255'
	subwf	POSITION,w	; if beyond min brightness then do not light LED
	btfss	STATUS,c	; if c is 0 then counted beyond 0
	goto	SPACE_1		; more interrupt space
	xorwf	BRIGHT_C,w	; current brightness. If zero then drive LED
	btfsc	STATUS,z	; if zero bit high drive LED
	bcf	PORTA,2		; light LED
	goto	SPACE_1

G_DV_O	movlw	B'00011000'	; gate drive high (off)
	iorwf	PORTA,f
	goto	N_LEDS

SPACE_1	incf	SPAC_E,f	; increase SPACE
	btfss	SPAC_E,0	; check bit 0, set; check switches
	goto	MTPX		; when bit 0 is clear
	
; check  switches on portB 3-7 outputs
	
	movf	MPX_CNT,w	; look at multiplex count value 
	sublw	D'7'		; check if 7
	btfsc	STATUS,c	; if c =0 then 8 or more
	goto	CK_07		; check switches 0-7

	movlw	B'11111000'	; LED displays off
	iorwf	PORTB,f		; 
	clrf	SW_FLG		; clear switch flag

	bcf	PORTB,7		; check flash/dim level switch
	btfss	PORTA,1		; if low then switch closed
	bsf	SW_FLG,7	; set flag if switch closed
	bsf	PORTB,7
	
	bcf	PORTB,6		; check auto/manual switch
	btfss	PORTA,1		; if low then switch closed
	bsf	SW_FLG,6	; set flag if switch closed
	bsf	PORTB,6

	bcf	PORTB,5		; check rate up switch
	btfss	PORTA,1		; if low then switch closed
	bsf	SW_FLG,5	; set flag if switch closed
	bsf	PORTB,5

	bcf	PORTB,4		; check a/b rate switch
	btfss	PORTA,1		; if low then switch closed
	bsf	SW_FLG,4	; set flag if switch closed
	bsf	PORTB,4
	
	bcf	PORTB,3		; check rate down switch
	btfss	PORTA,1		; if low then switch closed
	bsf	SW_FLG,3	; set flag if switch closed
	bsf	PORTB,3
	
	bsf	FLAG3,0		; set reset 4017 flag
	
	goto	RECLAIM		; multiplex on next interrupt after this switch check

; drive LED bar displays
	
MTPX	movlw	B'11111000'	; LED displays off
	iorwf	PORTB,f		;
	btfsc	EP_FLG,0	; is eeprom storage in progress
	goto	RECLAIM		; bypass display drive	
	btfss	FLAG3,0		; is reset 4017 flag set
	goto	IN_MP		; increase MPX counter
	clrf	MPX_CNT		; clear multiplex counter
	bsf	PORTB,1		; reset 4017 counter
	nop			; time for reset
	bcf	PORTB,1		; release reset on 4017 counter	
	clrf	FLAG3		; reset 4017 counter flag off
	goto	NO_CK		; bypass clocking 4017

IN_MP	incf	MPX_CNT,f	; next multiplex count
	movlw	D'08'
	movwf	GAP_T		; gap timer between driving next leds
T_CYC	decfsz	GAP_T,f
	goto	T_CYC	
	bsf	PORTB,2		; clock 4017 next count

NO_CK	btfss	SPAC_E,7	; if bit 7 set drive setting display
	goto	SET_DSP
	btfss	FLASH,3		; display flasher
	goto	RECLAIM
	movlw	0x01
	movwf	PCLATH
	movf	BRIGHT_S,w	; brightness
	call	CURRENT		; convert to display value
	movwf	TEMP		; store
	movlw	0x02
	movwf	PCLATH
	movf	MPX_CNT,w	
	addwf	PCL,f		; add MPX_CNT value to program counter
	goto	RECLAIM
	goto	RECLAIM
	goto	DIM_PRD		; dimming down in process acknowledge 
	goto	CK_3		; THREE brightness 
	goto	CK_4		; FOUR brightness
	goto	DIM_PRU		; dimming up in progress
	goto	CK_6		; SIX brightness
	goto	CK_7		; SEVEN brightness
	
SET_DSP	movlw	0x02		; page 2 PCLATH address
	movwf	PCLATH
	movf	MPX_CNT,w	
	addwf	PCL,f		; add MPX_CNT value to program counter
	goto	NIL_Z		; 4017 output is zero
	goto	ONE
	goto	TWO
	goto	THREE
	goto	FOUR
	goto	FIVE
	goto	SIX
;	goto	SEVEN		; next line

SEVEN	btfss	SW_FLG,7	; dim or flash level display
	goto	DM_LEV7
	movf	FLSH_LEV,w	; flash level value
CK_R7	movwf	TEMP
CK_7	movf	TEMP,w
	movlw	D'09'
	subwf	TEMP,w		; next decade
	btfss	STATUS,c	; if c is 1 then > = 9 	
	goto	RECLAIM
	movf	TEMP,w
	sublw	D'19'		
	btfss	STATUS,c
	goto	RECLAIM
	goto	DV_LD
DM_LEV7	movf	DIM_LEV,w
	goto	CK_R7	

; indicate dimming operation

DIM_PRD	btfss	FLAG_2,1	; dimming down flag	
	goto 	RECLAIM		; flag not set so out
DIM_IND	clrf	TEMP
	goto	DV_LD		; drive LED	
DIM_PRU	btfss	FLAG_2,0	; dimming up flag
	goto	RECLAIM
	goto	DIM_IND		; dimming indication

SIX	btfss	SW_FLG,7	; dim or flash level display
	goto	DM_LEV6
	movf	FLSH_LEV,w	; flash level value
CK_S6	movwf	TEMP
CK_6	movf	TEMP,w
	sublw	D'09'
	btfss	STATUS,c	; if c is 1 then < = 9 	
	goto	RECLAIM
DV_LD	movlw	0x01		; page 1 PCLATH address
	movwf	PCLATH
	movf	TEMP,w
	call	LDVAL		; get LED code
	andwf	PORTB,f		; drive LEDs	
	goto	RECLAIM
DM_LEV6 movf	DIM_LEV,w	; dim level value
	goto	CK_S6	

THREE	btfss	SW_FLG,7	; dim or flash level display
	goto	DM_LEV3
	movf	FLSH_LEV,w	; flash level value
CK_L	movwf	TEMP
CK_3	movf	TEMP,w
	movlw	D'19'
	subwf	TEMP,w		; next decade
	btfss	STATUS,c	; if c is 1 then < = 19 	
	goto	RECLAIM
	movf	TEMP,w
	sublw	D'29'		
	btfss	STATUS,c
	goto	RECLAIM
	goto	DV_LD
DM_LEV3	movf	DIM_LEV,w
	goto	CK_L

FOUR	btfss	SW_FLG,7	; dim of flash level display
	goto	DM_LEV4
	movf	FLSH_LEV,w	; flash level value
CK_R4	movwf	TEMP
CK_4	movf	TEMP,w
	movlw	D'29'
	subwf	TEMP,w		; next decade
	btfss	STATUS,c	; if c is 1 then  =>29 	
	goto	RECLAIM
	movf	TEMP,w
	sublw	D'38'		
	btfss	STATUS,c
	goto	RECLAIM
	goto	DV_LD
DM_LEV4	movf	DIM_LEV,w
	goto	CK_R4

; Autodim display

TWO	btfsc	SW_FLG,4	; AUTO level display
	goto	AUT_B2
	movf	AUTO_A,w	; A flash level value
	goto	CK_S6
AUT_B2  movf	AUTO_B,w	; B level value
	goto	CK_S6	

NIL_Z	btfsc	SW_FLG,4	; Auto level display
	goto	AUT_B0
	movf	AUTO_A,w	; flash level value
	goto	CK_R7
AUT_B0	movf	AUTO_B,w
	goto	CK_R7

ONE	btfsc	SW_FLG,4	; A or B rate display
	goto	AUTO_B1
	movf	AUTO_A,w	; A value
	goto	CK_L
AUTO_B1	movf	AUTO_B,w
	goto	CK_L

FIVE	btfsc	SW_FLG,4	; A or B rate display
	goto	AUTO_B5
	movf	AUTO_A,w	; A value
	goto	CK_R4
AUTO_B5	movf	AUTO_B,w
	goto	CK_R4

; zero crossing 

ZERO	movf	TMR0,w		; timer value
	movwf	TIMER		; temp storage for TMR0	
	bcf	INTCON,INTF	; clear INT flag

; reset timer counter
; freq is 20MHz/4/4/50. [50 =(256(-208-2)] =25kHz or 40us period 

	movf	DIVIDE,w
	movwf	TMR0		; set timer
	nop			; make sure timer reset before clearing flag 
;	nop
;	nop
	movf	COUNTZ_Z,w
	sublw	D'243'

	btfss	STATUS,z	; if zero check timer status
	goto	FST_SLO

; check timer value to within acceptable range (ie. in middle of count between 205 and 255)
	
	movf	TIMER,w		; look at timer
	sublw	D'235'		; is it larger than 235
	btfsc	STATUS,c	; if c is 0 then >235 if 1 then <235 so speed up
	goto	FASTER
	movf	TIMER,w
	sublw	D'250'
	btfsc	STATUS,c	; if c is 1 then <= 250
	goto	RESET		; in lock so leave values as is
	goto	SLOWER
FST_SLO	btfss	STATUS,c	; if 0 then count is larger so slow down
	goto	SLOWER
FASTER	incf	PULSE_V,w
	sublw	D'250'
	btfsc	STATUS,c	; if c is 0 then do not increase
	goto	INC_PLS
	clrf	PULSE_V
	incf	DIVIDE,f	; increase divide as PULSE_V has exhausted its range
	goto	RESET		
INC_PLS	incf	PULSE_V,f
	goto 	RESET
SLOWER	movf	PULSE_V,w
	btfss	STATUS,z
	goto	DEC_PLS
	decf	DIVIDE,f	; zero PULSE_V so reduce Divide value 
	movlw	D'250'
	movwf	PULSE_V
	goto	RESET
DEC_PLS	decf	PULSE_V,f
	

; zero crossing reset counter (also compensates for shift due to input filtering)


RESET	bcf	FLAG_Z,0	; clear zero crossing sine flag (plus or minus half sine wave)
	clrf	COUNTZ_Z	; counter from zero crossing to zero crossing
	bcf	INTCON,T0IF	; clear TMRO interrupt flag
ZERO_H	bsf	PORTA,2		; LED off
	movf	PULSE_V,w
	movwf	PULSE_O		; transfer value to operating counter
	movlw	D'00'		; an offset compensates for filter phase change 
	movwf	POSITION	; reset position counter any offset is to counteract shift in zero
 				; voltage crossing detector due to filtering
	movlw	0x00		; page 0 PCLATH address
	movwf	PCLATH
	movf	BRIGHT_S,w	; update brightness
	call	BRIGH_T		; lookup table	
	movwf	BRIGHT_C	; update every zero crossing only (10ms to prevent erratic 
				; brightness during dimming up) 
	btfsc	TIM_FLG,1	; if time delay decrement flag set bypass (cleared on time delay)
	goto	TIM_BY
	movf	TIME_D,w	; time delay
	btfss	STATUS,z	; check if zero
	decf	TIME_D,f	; decrease
TIM_BY	incf	FLASH,f		; flasher timer

; end of interrupt reclaim w and status 

RECLAIM	movlw	0x00		; page 0 PCLATH address
	movwf	PCLATH
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register

	bcf	PORTB,2		; clock 4017 counter low
	retfie			; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN	bcf	TIM_FLG,1	; time delay decrement flag clear
	clrf	PULSE_V		; pulse value
	clrf	PULSE_O		; operating pulse value
	clrf	MPX_CNT		; multiplex counter
	clrf	FLAG_2		; auto dim flags
	bsf	FLG_OFF,0	; gate drive off at reset	
	clrf	FLAG_ON		; dimming flags
	clrf	FLAG3		; resetting 4017 flag
	clrf	SW_FLG2		; switch flags
	clrf	SW_FLG		; switch flags

	movlw	D'208'		; initial divider for timer counter
	movwf	DIVIDE		; freq is 20MHz/4/4/50. [50 =(256(-208-2)] =25kHz or 40us period 
	
	movlw	D'250'		; zero crossing
	movwf	COUNTZ		; counter for zero crossing - zero crossing length
	clrf	COUNTZ_Z	; clear countZ-Z zero crossing counter
	bcf	EP_FLG,0	; eeprom storage in process flag (stops display going erratic) 
	
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; (RB0 input, RB1- RB7 outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'10000001'	; 
	movwf	OPTION_REG	; TMRO prescaler is 4, PORTB pullups disabled
	movlw   B'00000011'	; 1's are inputs (RA0,1) RA2-RA4 outputs 
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111111'	; 1 for RB (reset)
	movwf	PORTB		; portB outputs high
	movlw	B'00011100'	; Triac gate trigger off, LED off
	movwf	PORTA		; portA values set
	bcf	PORTB,1		; release reset on 4017 counter
	
; recall stored values

	movlw	EEPROM1		; Dim level
	call 	EEREAD
	movwf	DIM_LEV
	movlw	EEPROM2		; Flash level
	call 	EEREAD
	movwf	FLSH_LEV
	movlw	EEPROM3		; Min level
	call 	EEREAD
	movwf	MIN_LEV
	movwf	BRIGHT_S	; current brightness
	movwf	BRIGHT_V	; stored brightness
	movlw	EEPROM4		; Auto dim A rate
	call 	EEREAD
	movwf	AUTO_A
	movlw	EEPROM5		; Auto dim B rate
	call 	EEREAD
	movwf	AUTO_B

; interrupt enable 

	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	bsf	INTCON,INTE	; set interrupt enable for RB0
	bsf	INTCON,GIE	; set global interrupt enable for above


; check operation switches first. Up dim, down dim, flash on and flash off 

; switch function

SW_CK	btfsc	SW_FLG2,4	; check flash on flag
	goto	FLS_ON		; 
	btfsc	SW_FLG2,3	; check flash off switch
	goto	FLS_OFF		; 

	btfsc	FLAG_2,0	; auto dim /up
	goto	DIM_UP
	btfsc	FLAG_2,1	; auto dim /down
	goto	DIM_DN

	btfsc	SW_FLG2,6	; check dim up switch
	goto	DIM_UP
	btfsc	SW_FLG2,2	; check dim down switch
	goto	DIM_DN

	btfsc	SW_FLG2,0	; write to EEPROM
	goto 	EEPROM		;  stores EEPROM values
	btfsc	SW_FLG2,5	; set dim level down
	goto	DMS_DN		;  Dim and Flash levels
	btfsc	SW_FLG2,1	; set dim level up
	goto	DMS_UP		;  Dim and flash levels
SW_CK9	btfsc	SW_FLG,5	; set rate up switch
	goto	RATE_UP		;  A and B rates
SW_CKA	btfsc	SW_FLG,3	; set rate down switch
	goto	RATE_DN		;  A and B rates
	movf	BRIGHT_V,w	; if no switches pressed revert to old brightness setting
	movwf	BRIGHT_S	; current brightness
	clrf	FLAG_ON		; min set run flag
	goto	SW_CK

EEPROM
	bsf	EP_FLG,0	; eeprom storage in process flag
	btfsc	FLAG_ON,2	; is eeprom already programmed flag set
	goto	ESCPE	
	bsf	FLAG_ON,2	; eeprom already programmed
	movlw	EEPROM1		; DIM_LEV storage
	movwf	EEADR
	movf	DIM_LEV,w
	call	EWRITE		; write to EEPROM
	incf	EEADR,f		; flash level
	movf	FLSH_LEV,w
	call	EWRITE		; write to EEPROM
	incf	EEADR,f		; min level storage (preheat)
	movf	MIN_LEV,w	; minimum level
	call 	EWRITE		; store value
	incf	EEADR,f		; A dim rate storage
	movf	AUTO_A,w
	call	EWRITE		; write to EEPROM
	incf	EEADR,f		; B dim rate storage
	movf	AUTO_B,w
	call	EWRITE		; write to EEPROM
ESCPE	movlw	D'100'		; 1s delay
	bcf	EP_FLG,0	; eeprom storage in process flag 
	goto	TIM1		; delay

MIN_SET	btfsc	FLAG_ON,1	; min set run flag
	goto	NO_MIN
	bsf	FLAG_ON,1
	movlw	D'180'
	movwf	BRIGHT_S
NO_MIN	decfsz	BRIGHT_S,f	; current brightness
	goto	SET_MIN
	movlw	D'180'
	movwf	BRIGHT_S
SET_MIN	movf	BRIGHT_S,w
	movwf	MIN_LEV		; minimum level
	
	goto	TIM0

DMS_DN	btfsc	SW_FLG,7	; dim or flash
	goto	FLSH_DN
	movf	DIM_LEV,w	; dim level
	btfss	STATUS,z	; check if zero
	decf	DIM_LEV,f
	
TIM0	movlw	D'25'		; sets delay at 250ms
TIM1	movwf	TIME_D		; delay timer (decremented on zero crossing)
TIM_DEL	movf	TIME_D,w
	bcf	TIM_FLG,1
	btfsc	SW_FLG2,4	; check flash on flag
	clrf	TIME_D		; 
	btfsc	SW_FLG2,3	; check flash off switch
	clrf	TIME_D	
	btfss	STATUS,z	; when zero out
	goto	TIM_DEL
	goto	SW_CK

DMS_UP 	btfsc	SW_FLG,7	; dim or flash
	goto	FLSH_UP
	movf	DIM_LEV,w	; dim level
	sublw	D'37'		; check if at count of 38
	btfsc	STATUS,c	; if c is 0 then at max do not increase
	incf	DIM_LEV,f	
	goto	TIM0

FLSH_DN	movf	FLSH_LEV,w	; flash level
	btfss	STATUS,z	; check if zero
	decf	FLSH_LEV,f
	goto	TIM0

FLSH_UP	movf	FLSH_LEV,w	; flash level
	sublw	D'37'		; check if at count of 38
	btfsc	STATUS,c	; if c is 0 then at max do not increase
	incf	FLSH_LEV,f	
	goto	TIM0

RATE_DN	btfsc	SW_FLG,4	; A or B dim rate
	goto	RATEBD		; B rate
	movf	AUTO_A,w	; A rate
	btfss	STATUS,z	; check if zero
	decf	AUTO_A,f	
	goto	TIM0

RATE_UP	btfsc	SW_FLG,4	; A or B dim rate
	goto	RATEBU		; B rate
	movf	AUTO_A,w	; dimming rate
	sublw	D'37'		; check if at count of 38
	btfsc	STATUS,c	; if c is 0 then at max do not increase
	incf	AUTO_A,f	
	goto	TIM0

RATEBD	movf	AUTO_B,w	; B rate
	btfss	STATUS,z	; check if zero
	decf	AUTO_B,f	
	goto	TIM0

RATEBU	movf	AUTO_B,w	; dimming rate
	sublw	D'37'		; check if at count of 38
	btfsc	STATUS,c	; if c is 0 then at max do not increase
	incf	AUTO_B,f	
	goto	TIM0

FLS_ON	clrf	FLAG_2		; dimming stop
	bcf	FLG_OFF,0	; gate drive flag off
	movlw	0x00		; page 0 PCLATH address
	movwf	PCLATH
	movf	FLSH_LEV,w	; flash level
	call	LEV_CON		; level conversion
 	movwf	BRIGHT_S 	; current brightness setting
	goto	SW_CK
	
FLS_OFF	clrf	FLAG_2		; dimming stop
	bcf	FLG_OFF,0	; gate drive flag off
	movf	MIN_LEV,w	; minimum level
	movwf	BRIGHT_S	; current brightness setting
	goto	SW_CK

; check auto or manual dimming (set flag for dimming in auto) and dimming rate

DIM_UP	btfss	SW_FLG2,0	; write to EEPROM switch
	goto	DIM_CON
	btfsc	SW_FLG2,1	; set up switch
	goto	MIN_SET		; Minimum_SET if Flash on, Dim up and Dim set switches closed
DIM_CON	bcf	FLG_OFF,0	; off flag
	btfsc	FLAG_2,7	; has dimming switch reversed
	goto	T_REV		; stop dimming
	bcf	FLAG_ON,3	; clear dim down stop flag
	btfsc	FLAG_ON,4	; is dimming stop flag set
	goto	CLR_UP
	btfss	SW_FLG,6	; check if auto dim mode
	bsf	FLAG_2,0	; set auto dim flag (dim-up)
	btfsc	SW_FLG,6
	clrf	FLAG_2		; clear auto dim flags if manual
	btfss	SW_FLG,4	; A or B dimming rate
	goto	A_DIM
	movf	AUTO_B,w	; B auto rate
A_LOAD	bsf	TIM_FLG,1	; stop Time delay decrement
	movwf	TIME_D		; time delay between each change
	movwf	DEL_STO		; storage and operating value
	movlw	0x00		; page 0 PCLATH address
	movwf	PCLATH
QUICK	movf	DIM_LEV,w	; display dimming level
	call	LEV_CON		; convert to brightness value
	movwf	TEMP_3
	decf	BRIGHT_S,w	; increase brightness setting 
	subwf	TEMP_3,w	; dim level value after conversion in LEV_CON
	btfsc	STATUS,c	; if c is 1 then BRIGHT_S at min value
	goto	MAX_B		; max brightness
	decf	BRIGHT_S,f
	movf	BRIGHT_S,w
	movwf	BRIGHT_V	; stored brightness
	movf	DEL_STO,w	; if zero bypass
	btfsc	STATUS,z
	goto	TIM_DEL
	decfsz	DEL_STO,f
	goto	DEC_TM
	goto	QUICK		; speed up dimming
DEC_TM	decf	TIME_D,f
	goto	TIM_DEL
MAX_B	movf	BRIGHT_S,w
	movwf	BRIGHT_V
	bsf	FLAG_ON,4	; end dimming flag
CLR_UP	bcf	FLAG_ON,3
	clrf	FLAG_2		; clear auto dim flag
	goto	TIM0		; time delay
A_DIM	movf	AUTO_A,w	; A auto rate
	goto	A_LOAD

DIM_DN	bcf	FLG_OFF,0	; gate drive flag off
	btfss	FLAG_2,7	; has dimming switch reversed
	goto	DN_CON		; down continue
T_REV	clrf	FLAG_2		; clear dimming flag
	movlw	D'30'
	goto	TIM1	
DN_CON	bcf	FLAG_ON,4	; dim up stop flag clear
	btfsc	FLAG_ON,3	; is dimming stop flag set
	goto	CLR_DN
	btfss	SW_FLG,6	; check if auto dim mode
	bsf	FLAG_2,1	; set auto dim flag (dim-down)
	btfsc	SW_FLG,6
	clrf	FLAG_2		; clear auto dim flags if manual
	btfss	SW_FLG,4	; A or B dimming rate
	goto	A_DIMD
	movf	AUTO_B,w	; B auto rate
A_LOD	bsf	TIM_FLG,1	; stop decrement of time flag
	movwf	TIME_D		; time delay between each change
	movwf	DEL_STO		; operating storage
QUIK_D	incf	BRIGHT_S,w	; current brightness 
	subwf	MIN_LEV,w	; min brightness  
	btfss	STATUS,c	; light is brighter than setting if c is 0
	goto	MIN_B		; do nothing to brightness
	incf	BRIGHT_S,f	; decrease brightness
	movf	BRIGHT_S,w
	movwf	BRIGHT_V	; stored brightness
	movf	DEL_STO,w	; if zero bypass
	btfsc	STATUS,z
	goto	TIM_DEL
	decfsz	DEL_STO,f
	goto	DEC_TIM
	goto	QUIK_D		; speed up dimming
DEC_TIM	decf	TIME_D,f
	goto	TIM_DEL

MIN_B	bsf	FLAG_ON,3	; set dim down stop flag
	movf	BRIGHT_S,w
	movwf	BRIGHT_V
CLR_DN	bcf	FLAG_ON,4
	clrf	FLAG_2		; clear auto dim flag
	goto	TIM0		; delay
A_DIMD	movf	AUTO_A,w
	goto	A_LOD

	end		 	
