;***************************************************************************
;*
;* IMPORTANT: This version runs only on the AT90S2313. In addition, the
;*            port B pins used for MOSI & MISO are swapped to suit the
;*            Silicon Chip programmer (see comments below).
;*
;***************************************************************************
;*
;* Title	: AVR ISP (Auto adr inc, 19200bps)
;* Version	: 2.3
;* Last updated	: August 8 2004
;* Target	: AT90S2313
;* File		: AVR910c.asm
;* Author(s)	: Ole Saether, Terje Frostad,
;*		  Ingar Fredriksen, Morten W. Lund
;*                Haakon Skar, Paal Kastnes
;*
;* DESCRIPTION
;*	The firmware on all programmers now support a unified protocol for 
;*	program and data memory programming. The host computer do not need
;*	to know if the programmer operates in serial or parallel mode.
;*
;*	The following commands are supported. All commands start with a
;*	single letter. The programmer returns 13d (carriage return) or the
;*	data read after the command is finished.
;*
;*                                     +-------------+------------+------+
;*  Commands                           | Host writes | Host reads |      |
;*  --------                           +-----+-------+------+-----+      |
;*                                     | ID  | data  | data |     | Note |
;* +-----------------------------------+-----+-------+------+-----+------+
;* | Enter programming mode            | 'P' |       |      | 13d |   1  |
;* | Report autoincrement address      | 'a' |       | 	    | 'Y' |      |
;* | Set address                       | 'A' | ah al |      | 13d |   2  |
;* | Write program memory, low byte    | 'c' |    dd |      | 13d |   3  |
;* | Write program memory, high byte   | 'C' |    dd |      | 13d |   3  |
;* | Issue Page Write                  | 'm' |       |      | 13d |      |
;* | Read program memory               | 'R' |       |dd(dd)|     |   4  |
;* | Write data memory                 | 'D' |    dd |      | 13d |      |
;* | Read data memory                  | 'd' |       |   dd |     |      |
;* | Chip erase                        | 'e' |       |      | 13d |      |
;* | Write lock bits                   | 'l' |    dd |      | 13d |      |
;* | Write fuse bits                   | 'f' |    dd |      | 13d |  11  |
;* | Read fuse and lock bits           | 'F' |       |   dd |     |  11  |
;* | Leave programming mode            | 'L' |       |      | 13d |   5  |
;* | Select device type                | 'T' |    dd |      | 13d |   6  |
;* | Read signature bytes              | 's' |       | 3*dd |     |      |
;* | Return supported device codes     | 't' |       | n*dd | 00d |   7  |
;* | Return software identifier        | 'S' |       | s[7] |     |   8  |
;* | Return sofware version            | 'V' |       |dd dd |     |   9  |
;* | Return hardware version           | 'v' |       |dd dd |     |   9  |
;* | Return programmer type            | 'p' |       |   dd |     |  10  |
;* | Set LED                           | 'x' |    dd |      | 13d |  12  |
;* | Clear LED                         | 'y' |    dd |      | 13d |  12  |
;* | Universial command                | ':' |  3*dd |   dd | 13d |      |
;* | New universal command	       | '.' |  4*dd |   dd | 13d |      |
;* | Special test command	       | 'Z' |  2*dd |   dd |     |      |
;* +-----------------------------------+-----+-------+------+-----+------+
;*
;* NOTE 1
;*	The Enter programming mode command MUST be sent one time prior to
;*	the other commands, with the exception of the 't', 'S', 'V', 'v'
;*	and 'T' commands. The 'T' command must be sent before this command
;*	(see note 6).
;*
;*	For programmers supporting both parallel and serial programming
;*	mode this command enters parallel programming mode. For programmers
;*	supporting only serial programming mode, this command enters serial
;*	programming mode.
;*
;* NOTE 2
;*	The ah and al are the high and low order bytes of the address. For
;*	parallel programmers this command issues the Load Address Low/High
;*	Byte command. For serial programmers the address byte is stored for
;*	use by the Read/Write commands.
;*
;* NOTE 3
;*	For parallel programmers this command issues the Program Flash
;*	command. For serial programmers this command iussues the Write
;*	Program Memory Command. For devices with byte-wide program memories
;*	only the low byte command should be used.
;*
;* NOTE 4
;*	The contents of the program memory at the address given by the 'A'
;*	command are written to the serial port in binary form. For byte
;*	wide memories one byte is written. For 16 bit memories two bytes
;*	are written,MSB first.
;*
;* NOTE 5
;*	This command must be executed after the programming is finished.
;*
;* NOTE 6
;*	The select device type command must be sent before the enter
;*	programming command
;*
;* NOTE 7
;*	The supported device codes are returned in binary form terminated
;*	by 0x00.
;*
;* NOTE 8
;*	This return a 7 character ASCII string identifying the programmer.
;*	For the development board it is "AVR DEV", for the parallel
;*	programmer it is "AVR PPR" and for the in-curcuit programmer it is
;*	"AVR ICP".
;*
;* NOTE 9
;*	The software/hardware version are returned as two ASCII numbers.
;*
;* NOTE 10
;*	This command should be used to identify the programmer type. The
;*	return value is 'S' for serial (or SPI) programmers or 'P' for
;*	parallel programmers.
;*
;* NOTE 11
;*	The write fuse bits command are available only on parallel
;*	programmers and only for AVR devices (device code < 0x80). The host
;*	should use the return programmer type command to determine the
;*	programmer type, do not use the  "AVR PPR" idenifier because other
;*	programmers may be available in the future.
;*
;* NOTE 12
;*	Currently only the AVR development board has LEDs. The other boards
;*	must implement this commands as NOPs.
;*
;* NOTE 13
;*      Devices using Page Mode Programming write one page of flash memory
;*      before issuing a Page Mode Write Pulse.
;*
;* HISTORY
;*	V2.3	04.08.08 (SC)		Added all devices supported by AVRProg v1.37
;*					(suggested by John Samperi - thanks John)
;*      V2.2c   03.06.07 (sjdavies)     Changed the 'x' and 'y' commands to accept
;*					a parameter ('flushing' bugfix)
;*	V2.2a	02.08.16 (SC)		Swapped MOSI & MISO port bits
;*      V2.2    02.04.10 (sjdavies)     Ported to AT90S2313 running at 4MHz
;*					(uses hardware UART & stack)
;*      V2.2    00.03.10 (pkastnes)     Added support for multiple new devices
;*	V2.1	98.10.26 (mlund)	New date marking.
;*					Removed support for AT90S1200C.
;*					Added support for AT90S4433A.
;*	V2.0	98.01.06 (mlund)	ATmega103 support
;*	V1.7	97.11.06 (mlund)	Universial command (':') implemented.
;*					Releases all pins when not in 
;*					programming mode.
;*	V1.6e	97.11.04 (mlund)	mega103 rev D support
;*	V1.6c	97.10.30 (mlund)	Auto incrementing / SPI sync
;*					also works for mega103.
;*      V1.6	97.09.09 (hskar)	Created Page Mode Version (mega103)
;*	V1.5	97.08.21 (mlund)	Modified / Bugfix / Major cleanup
;*	...	...			(no records)
;*	V?.?	97.03.15 (OS)		Created
;* 
;* 
;*		
;* Device Support List v2.2 (see below for devices added in v2.3)    
;* --------------------------------------------------------------
;* 
;* +-------+----------+------+-------+------+------+------+-------+
;* |Device |Signature | Code | Flash |EEProm| Lock | Fuse | PMode |
;* +-------+----------+------+-------+------+------+------+-------+
;* |tiny12 | 1E 90 05 | 0x55 |  R/W  | R/W  | R/W  | R/W  | Byte  |
;* |tiny15 | 1E 90 06 | 0x56 |  R/W  | R/W  | R/W  | R/W  | Byte  |
;* |       |          |      |       |      |      |      |       |
;* | S1200 | 1E 90 01 | 0x13 |  R/W  | R/W  |  W   | NA   | Byte  |
;* |       |          |      |       |      |      |      |       |
;* | S2313 | 1E 91 01 | 0x20 |  R/W  | R/W  |  W   | NA   | Byte  |
;* | S2323 | 1E 91 02 | 0x48 |  R/W  | R/W  | R/W  | R/W  | Byte  |    
;* | S2333 | 1E 91 05 | 0x34 |  R/W  | R/W  | R/W  | R/W  | Byte  |    
;* | S2343 | 1E 91 03 | 0x4C |  R/W  | R/W  | R/W  | R/W  | Byte  |    
;* |       |          |      |       |      |      |      |       |
;* | S4414 | 1E 92 01 | 0x28 |  R/W  | R/W  |  W   | NA   | Byte  |    
;* | S4433 | 1E 92 03 | 0x30 |  R/W  | R/W  | R/W  | R/W  | Byte  |    
;* | S4434 | 1E 92 02 | 0x6C |  R/W  | R/W  | R/W  | R/W  | Byte  |    
;* |       |          |      |       |      |      |      |       |
;* | S8515 | 1E 93 01 | 0x38 |  R/W  | R/W  |  W   | NA   | Byte  |    
;* | S8535 | 1E 93 03 | 0x68 |  R/W  | R/W  | R/W  | R/W  | Byte  |    
;* |       |          |      |       |      |      |      |       |
;* |mega83 | 1E 93 05 | 0x65 |  R/W  | R/W  | R/W  | R/W  | Page  |    
;* |mega103| 1E 97 01 | 0x41 |  R/W  | R/W  | R/W  | R/W  | Page  |    
;* |mega161| 1E 94 01 | 0x60 |  R/W  | R/W  | R/W  | R/W  | Page  |    
;* |mega163| 1E 94 02 | 0x64 |  R/W  | R/W  | R/W  | R/W  | Page  |    
;* +-------+----------+------+-------+------+------+------+-------+
;* 
;* LEGEND:
;* -------
;* Signature - Device Signature Byte
;* Code      - Unique device code used by AVRProg to identify device
;* Flash     - Flash size in bytes
;* EEProm    - EEProm size in bytes
;* Lock      - Lockbits
;* Fuse      - Fusebits
;* PMode     - Indicates if device uses byte or page programming mode
;* 
;* R/W - Read and Write Access
;* R   - Read Access Only
;* W   - Write Access Only
;* NA  - Not Accessible
;* 
;***************************************************************************

;**** includes ****

.nolist
.include "2313def.inc"
.list

;***************************************************************************
;*
;* CONSTANTS
;*	device codes
;*
;* DESCRIPTION
;*	The following device codes must be used by the host computer. Note
;*	that the device codes are arbitrary selected, they do not have any
;*	thing in common with the signature bytes stored in the device.
;*
;*	This list includes all devices supported by AVRProg v1.37. Not all
;*	devices have been tested with this programmmer.
;*
;***************************************************************************

.equ	S1200D	= 0x13		;90S1200
.equ	S2313A	= 0x20		;90S2313
.equ	S4414A	= 0x28		;90S4414
.equ	S4433A	= 0x30		;90S4433
.equ	S2333A	= 0x34		;90S2333
.equ	S8515A	= 0x38		;90S8515
.equ	m8515	= 0x3A		;mega8515
.equ	m8515_b = 0x3B 		;mega8515 boot
.equ	m103	= 0x41		;mega103
.equ	m603	= 0x42		;mega603
.equ	m128	= 0x43		;mega128
.equ	m128_b	= 0x44		;mega128 boot
.equ	S2323A	= 0x48		;90S2323
.equ	S2343A	= 0x4C		;90S2343
.equ	tn11	= 0x50		;tiny11
.equ	tn10	= 0x51		;tiny10
.equ	tn12	= 0x55		;tiny12
.equ	tn15	= 0x56		;tiny15
.equ	tn19	= 0x58		;tiny19
.equ	tn28	= 0x5C		;tiny28
.equ	tn26	= 0x5E		;tiny26
.equ	m161	= 0x60		;mega161
.equ	m161_b	= 0x61		;mega161 boot
.equ	m163	= 0x64		;mega163
.equ	m83	= 0x65		;mega83
.equ	m163_b	= 0x66		;mega163 boot
.equ	m83_b	= 0x67		;mega83 boot
.equ	S8535	= 0x68		;90S8535
.equ	S4434	= 0x6C		;90S4434
.equ	C8534	= 0x70		;90C8534
.equ	C8544	= 0x71		;90C8544
.equ	m32	= 0x72		;mega32
.equ	m32_b	= 0x73		;mega32 boot
.equ	m16	= 0x74		;mega16
.equ	m16_b	= 0x75		;meag16 boot
.equ	m8	= 0x76		;mega8
.equ	m8_b	= 0x77		;mega8 boot

.equ	C1051	= 0x80		;89C1051
.equ	C2051	= 0x81		;89C2051
.equ	S8252	= 0x86		;89S8252
.equ	S53	= 0x87		;89S53


;**** Revision Codes ****

.equ	SW_MAJOR = '2'		; Major SW revision number
.equ	SW_MINOR = '3'		; Minor SW revision number
.equ	HW_MAJOR = '1'		; Major HW revision number
.equ	HW_MINOR = '0'		; Minor HW revision number


;***************************************************************************
;*
;* MACROS
;*	Program Macros
;*
;* DESCRIPTION
;*	Change the following four macros if the RESET pin to the
;*	target moves and/or if the SCK/MISO/MOSO moves.
;*
;***************************************************************************

.equ	SRESET	= 4		;RESET = port bit 4

.macro	set_reset
	sbi	PORTB,SRESET
.endm

.macro	clr_reset
	cbi	PORTB,SRESET
.endm

.macro	ddrb_init
	ldi	temp1,(0xFF ^ (1<<MISO))
	out	DDRB,temp1	;'MISO' is input, the rest is output
.endm

.macro	ddrb_release
	ldi	temp1,(1<<SRESET)
	out	DDRB,temp1	;'SRESET' is output, the rest is input
.endm

.macro	pulse_sck
	sbi	PORTB,SCK
	ldi	temp2,6

m0:	dec	temp2
	brne	m0

	cbi	PORTB,SCK
	ldi	temp2,3

m1:	dec	temp2
	brne	m1
.endm


;*****************
;* SPI Constants *
;*****************

; use these for the Silicon Chip programmer:

.equ	MOSI    = 5		;MOSI = port bit 5 
.equ	MISO	= 6		;MISO = port bit 6
.equ	SCK	= 7		;SCK  = port bit 7

; use these for the unmodified AVR910 hardware:

;.equ	MOSI    = 6		;MOSI = port bit 6 
;.equ	MISO	= 5		;MISO = port bit 5
;.equ	SCK	= 7		;SCK  = port bit 7


;******************
;* UART Constants *
;******************

.equ	N = 12			;divisor for 19,200 bit rate (4MHz crystal)


;*****************************
;* Global Register Variables *
;*****************************

.def	device		= r16	;device code
.def	temp1		= r17
.def	temp2		= r18
.def	s_data		= r19	;SPI data
.def	u_data		= r20	;UART data
.def	addrl		= r21	;low order byte of address
.def	addrh		= r22	;high order byte of address
.def	bit_cnt		= r23	;bit count used by UART routine
.def	cmd		= r25	;serial programming command
.def	count		= r26	;timeout for "enter programming mode"
.def	param1		= r27
.def	cmd1		= r28
.def	cmd2		= r29
.def	cmd3		= r30
.def	rd_s_data	= r31


;*********************
;* Interrupt Vectors *
;*********************

.cseg
	rjmp	RESET		;reset vector


;***************************************************************************
;*
;* FUNCTION
;*	u_init
;*
;* DESCRIPTION
;*	Initialize UART.
;*
;***************************************************************************

u_init:	
	ldi	temp1,N		;set baud rate
	out	UBRR,temp1
	ldi	temp1,1<<TXEN|1<<RXEN ;initialize UART for TX and RX
	out	UCR,temp1
	in	temp1,UDR	;flush data register
	ret


;***************************************************************************
;*
;* FUNCTION
;*	putc
;*
;* DESCRIPTION
;*	Send a character on the UART.
;*
;***************************************************************************

putc:
	sbis	USR,UDRE	;test for TX register empty
	rjmp	putc		;loop until TX empty
	out	UDR,u_data	;send the byte
	ret


;***************************************************************************
;*
;* FUNCTION
;*	getc
;*
;* DESCRIPTION
;*	Wait for UART receive character.
;*
;***************************************************************************

getc:
	sbis	USR,RXC		;wait until a character has been received
	rjmp	getc

	in	temp1,USR	;get UART status
	in	u_data,UDR	;read byte from UART
	andi	temp1,(1<<FE)+(1<<OR) ;test for framing or overrun errors
	brne	getc		;ignore data if error(s)
	ret


;***************************************************************************
;*
;* FUNCTION
;*	delay
;*
;* DESCRIPTION
;*	 Make a small delay.
;*
;***************************************************************************

delay:	
	ldi	temp2,0xFF
dl:	dec	temp2
	brne	dl
	dec	temp1
	brne	delay
	ret


;***************************************************************************
;*
;* FUNCTION
;*	wrser
;*
;* DESCRIPTION
;*	 Write and read bytes on the SPI.
;*
;***************************************************************************

rdser:	
	clr	s_data

wrser:	
	ldi	temp1,8
	ldi	rd_s_data,0

wrs0:	rol	s_data
	brcc	wrs1
	
	sbi	PORTB,MOSI
	rjmp	wrs2

wrs1:	cbi	PORTB,MOSI
wrs2:	lsl	rd_s_data
	sbic	PINB,MISO
	ori	rd_s_data,1
	pulse_sck
	dec	temp1
	brne	wrs0

	mov	s_data,rd_s_data
	ret


;***************************************************************************
;*
;* FUNCTION
;*	spi_init (Enter programming mode)
;*
;* DESCRIPTION
;*	Initialize SPI interface.
;*
;***************************************************************************

spi_init:
	ddrb_init		;initialize port B
	cbi	portb,SCK	;clear SCK
	set_reset		;release RESET

	ldi	temp1,0xFF
	rcall	delay

	clr_reset		;RESET target

	ldi	temp1,0xFF
	rcall	delay

	ldi	s_data,0xAC
	rcall	wrser		;SPI write (byte 1)
	ldi	s_data,0x53
	rcall	wrser		;SPI write (byte 2)

; SPI Synchronization (fix!)

	cpi	device,0x20	;if ((device >= 0x20) && (device <= 0x7F))
	brlo	s2
	tst	device
	brmi	s2

s0b:	ldi	count,32

s1:	rcall	rdser		;SPI read (byte 3)
	cpi	s_data,0x53
	breq	s3

	ldi	s_data,0
	rcall	wrser		;SPI write (byte 4)
	pulse_sck
	ldi	s_data,0xAC
	rcall	wrser		;SPI write (byte 1)
	ldi	s_data,0x53
	rcall	wrser		;SPI write (byte 2)
	dec	count
	brne	s1
	rjmp	s3

s2:	ldi	s_data,0
	rcall	wrser		;SPI write (byte 3)
s3:	ldi	s_data,0
	rcall	wrser		;SPI write (byte 4)

	ldi	temp1,0x10
	rcall	delay
	ret


;***************************************************************************
;*
;* FUNCTION
;*	show_id
;*
;* DESCRIPTION
;*	Show our ID ("AVR ISP") on the serial line.
;*
;***************************************************************************

show_id:
	ldi	u_data,'A'
	rcall	putc
	ldi	u_data,'V'
	rcall	putc
	ldi	u_data,'R'
	rcall	putc
	ldi	u_data,0x20
	rcall	putc
	ldi	u_data,'I'
	rcall	putc
	ldi	u_data,'S'
	rcall	putc
	ldi	u_data,'P'
	rcall	putc
	ret


;***************************************************************************
;*
;* RESET
;*
;* DESCRIPTION
;*	Initialization
;*
;***************************************************************************

RESET:	
	ldi	temp1,low(RAMEND) ;set stack top
	out	SPL,temp1
	sbi	ACSR,ACD	;power down comparator

	ser	temp1
	out	PORTD,temp1
	set_reset		;release target 'RESET' line
	out	PORTB,temp1
	ddrb_release
	rcall	u_init		;initialize UART


;***************************************************************************
;*
;* PROGRAM
;*	waitcmd -> main
;*
;* DESCRIPTION
;*	Wait for and execute commands.
;*
;***************************************************************************

waitcmd:
	rcall	getc		;while (getc() == ESC) {};
	cpi	u_data,0x1b
	breq	waitcmd


;**** Device type ****

	cpi	u_data,'T'	;device type?
	brne	w0

	rcall	getc		;get device
	mov	device,u_data
	rjmp	put_ret


;**** Return software identifier ****

w0:	
	cpi	u_data,'S'	;return software identifier?
	brne	w1

	rcall	show_id
	rjmp	waitcmd


;**** Return software version ****

w1:	
	cpi	u_data,'V'	;return software version?
	brne	w2

	ldi	u_data,SW_MAJOR
	rcall	putc
	ldi	u_data,SW_MINOR
	rcall	putc
	rjmp	waitcmd


;**** Return Hardware Version ****

w2:	
	cpi	u_data,'v'	;return hardware version?
	brne	w3

	ldi	u_data,HW_MAJOR
	rcall	putc
	ldi	u_data,HW_MINOR
	rcall	putc
	rjmp	waitcmd


;**** Show supported devices ****

w3:	
	cpi	u_data,'t'	;show supported devices?
	breq	w31
	rjmp	w4

w31:	ldi	u_data,S1200D
	rcall	putc
	ldi	u_data,S2313A
	rcall	putc
	ldi	u_data,S4414A
	rcall	putc
	ldi	u_data,S4433A
	rcall	putc
	ldi	u_data,S2333A
	rcall	putc
	ldi	u_data,S8515A
	rcall	putc
	ldi	u_data,m8515
	rcall	putc
	ldi	u_data,m8515_b
	rcall	putc
	ldi	u_data,m103
	rcall	putc
	ldi	u_data,m603
	rcall	putc
	ldi	u_data,m128
	rcall	putc
	ldi	u_data,m128_b
	rcall	putc
	ldi	u_data,S2323A
	rcall	putc
	ldi	u_data,S2343A
	rcall	putc
	ldi	u_data,tn11
	rcall	putc
	ldi	u_data,tn10
	rcall	putc
	ldi	u_data,tn12
	rcall	putc
	ldi	u_data,tn15
	rcall	putc
	ldi	u_data,tn19
	rcall	putc
	ldi	u_data,tn28
	rcall	putc
	ldi	u_data,tn26
	rcall	putc
	ldi	u_data,m161
	rcall	putc
	ldi	u_data,m161_b
	rcall	putc
	ldi	u_data,m163
	rcall	putc
	ldi	u_data,m83
	rcall	putc
	ldi	u_data,m163_b
	rcall	putc
	ldi	u_data,m83_b
	rcall	putc
	ldi	u_data,S8535
	rcall	putc
	ldi	u_data,S4434
	rcall	putc
	ldi	u_data,C8534
	rcall	putc
	ldi	u_data,C8544
	rcall	putc
	ldi	u_data,m32
	rcall	putc
	ldi	u_data,m32_b
	rcall	putc
	ldi	u_data,m16
	rcall	putc
	ldi	u_data,m16_b
	rcall	putc
	ldi	u_data,m8
	rcall	putc
	ldi	u_data,m8_b
	rcall	putc

	ldi	u_data,C1051
	rcall	putc
	ldi	u_data,C2051
	rcall	putc
	ldi	u_data,S8252
	rcall	putc
	ldi	u_data,S53
	rcall	putc

	ldi	u_data,0x00	;end of device list
	rcall	putc
	rjmp	waitcmd


;**** Return programmer type ****

w4:	
	cpi	u_data,'p'	;return programmer type?
	brne	w51

	ldi	u_data,'S'	;for serial programmer
	rcall	putc
	rjmp	waitcmd


;**** Return autoincrement address support

w51:	
	cpi	u_data,'a'	;return address auto increment?
	brne	w5

	ldi	u_data,'Y'	;we support autoinc
	rcall	putc
	rjmp	waitcmd


;**** Set LED ****

w5:	
	cpi	u_data,'x'	;set LED? (ignored)
	brne	w6

	rcall   getc		;ignore data byte
	rjmp	put_ret


;**** Clear LED ****

w6:	
	cpi	u_data,'y'	;clear LED? (ignored)
	brne	w7

	rcall   getc		;ignore data byte
	rjmp	put_ret


;**** Enter programming mode ****

; We require that the device code be selected
; before any of the other commands

w7:
	cpi	device,S1200D
	breq	w71
	cpi	device,S2313A
	breq	w71
	cpi	device,S4414A
	breq	w71
	cpi	device,S4433A
	breq	w71
	cpi	device,S2333A
	breq	w71
	cpi	device,S8515A
	breq	w71
	cpi	device,m8515
	breq	w71
	cpi	device,m8515_b
	breq	w71
	cpi	device,m103
	breq	w71
	cpi	device,m603
	breq	w71
	cpi	device,m128
	breq	w71
	cpi	device,m128_b
	breq	w71
	cpi	device,S2323A
	breq	w71
	cpi	device,S2343A
	breq	w71
	cpi	device,tn11
	breq	w71
	cpi	device,tn10
	breq	w71
	cpi	device,tn12
	breq	w71
	cpi	device,tn15
	breq	w71
	cpi	device,tn19
	breq	w71
	cpi	device,tn28
	breq	w71
	cpi	device,tn26
	brne	w72
w71:	rjmp	w73

w72:	cpi	device,m161
	breq	w73
	cpi	device,m161_b
	breq	w73
	cpi	device,m163
	breq	w73
	cpi	device,m83
	breq	w73
	cpi	device,m163_b
	breq	w73
	cpi	device,m83_b
	breq	w73
	cpi	device,S8535
	breq	w73
	cpi	device,S4434
	breq	w73
	cpi	device,C8534
	breq	w73
	cpi	device,C8544
	breq	w73
	cpi	device,m32
	breq	w73
	cpi	device,m32_b
	breq	w73
	cpi	device,m16
	breq	w73
	cpi	device,m16_b
	breq	w73
	cpi	device,m8
	breq	w73
	cpi	device,m8_b
	breq	w73

	cpi	device,C1051
	breq	w73
	cpi	device,C2051
	breq	w73
	cpi	device,S8252
	breq	w73
	cpi	device,S53
	breq	w73
	rjmp	put_err

w73:	cpi	u_data,'P'	;enter programming mode?
	brne	w8

	rcall	spi_init
	rjmp	put_ret


;**** Wait program memory (not implemented) ****

;* USAGE
;*	wait_pm(byte cmd, byte c_data);
;*
;*	cmd :		0x28 - wait for high byte written
;*			0x20 - wait for low byte written
;*	u_data :	current data written

;wait_pm:			; do
;				; {
;	mov	s_data,cmd	;	wrser(cmd);		// SPI write (byte 1)
;	rcall	wrser
;	mov	s_data,addrh	;	wrser(addrh);		// SPI write (byte 2)
;	rcall	wrser
;	mov	s_data,addrl	;	wrser(addrl);		// SPI write (byte 3)
;	rcall	wrser
;	rcall	rdser		;	s_data = rdser();	// SPI read (byte 4)
				; }
;	cp	s_data,u_data	; while(s_data != u_data);
;	brne	wait_pm
;	ret


;**** Write program memory, high byte ****

w8:	
	cpi	u_data,'C'	;write program memory (high byte)?
	brne	w9

	rcall	getc
w81:	
	ldi	s_data,0x48
	rcall	wrser		;SPI write (byte 1)
	mov	s_data,addrh
	rcall	wrser		;SPI write (byte 2)
	mov	s_data,addrl
	rcall	wrser		;SPI write (byte 3)
	mov	s_data,u_data
	rcall	wrser		;SPI write (byte 4)

	ldi	temp1,1		;auto-increment address
	clr	temp2
	add	addrl,temp1
	adc	addrh,temp2
	rjmp	writeFLASHdelay


;**** Write program memory, low byte ****

w9:	
	cpi	u_data,'c'	;write program memory (low byte)?
	brne	w12
	rcall	getc

	ldi	s_data,0x40
	rcall	wrser		;SPI write (byte 1)
	mov	s_data,addrh
w91:
	rcall	wrser		;SPI write (byte 2)
	mov	s_data,addrl
	rcall	wrser		;SPI write (byte 3)
	mov	s_data,u_data
	rcall	wrser		;SPI write (byte 4)

; Check for page-mode device

writeFLASHdelay:
	cpi	device,m8515
	breq	w92
	cpi	device,m8515_b
	breq	w92
	cpi	device,m103
	breq	w92
	cpi	device,m603
	breq	w92
	cpi	device,m128
	breq	w92
	cpi	device,m128_b
	breq	w92
	cpi	device,tn26
	breq	w92
	cpi	device,m161
	breq	w92
	cpi	device,m161_b
	breq	w92
	cpi	device,m163
	breq	w92
	cpi	device,m163_b
	breq	w92
	cpi	device,m83
	breq	w92
	cpi	device,m83_b
	breq	w92
	cpi	device,m32
	breq	w92
	cpi	device,m32_b
	breq	w92
	cpi	device,m16
	breq	w92
	cpi	device,m16_b
	breq	w92
	cpi	device,m8
	breq	w92
	cpi	device,m8_b
	breq	w92

	ldi	temp1,0x20	;24,585 cycles delay
	rcall	delay		;(no delay for page-mode devices)

w92:	rjmp	put_ret


;**** Read program memory ****

w12:	
	cpi	u_data,'R'	;read program memory?
	brne	w10

	ldi	s_data,0x28
	rcall	wrser		;SPI write (byte 1)
	mov	s_data,addrh
	rcall	wrser		;SPI write (byte 2) 
	mov	s_data,addrl
	rcall	wrser		;SPI write (byte 3)
	rcall	rdser
	mov	u_data,s_data
	rcall	putc		;Send data (byte 4)

	ldi	s_data,0x20
	rcall	wrser		;SPI write (byte 1)
	mov	s_data,addrh
	rcall	wrser		;SPI write (byte 2)
	mov	s_data,addrl
	rcall	wrser		;SPI write (byte 3)
	rcall	rdser
	mov	u_data,s_data
	rcall	putc		;Send data (byte 4)

	rjmp	readaddrinc


;**** Load address ****

w10:	
	cpi	u_data,'A'	;load address?
	brne	w11

	rcall	getc
	mov	addrh,u_data
	rcall	getc
	mov	addrl,u_data
	rjmp	put_ret


;**** Write data memory ****

w11:	
	cpi	u_data,'D'	;write data memory?
	brne	w13

	rcall	getc

	ldi	s_data,0xC0
	rcall	wrser
	mov	s_data,addrh
	rcall	wrser
	mov	s_data,addrl
	rcall	wrser
	mov	s_data,u_data
	rcall	wrser
	ldi	temp1,0x20
	rcall	delay

	ldi	temp1,1		;auto-increment address
	clr	temp2
	add	addrl,temp1
	adc	addrh,temp2

	rjmp	put_ret


;**** Read data memory ****

w13:	
	cpi	u_data,'d'	;read data memory?
	brne	w14

	ldi	s_data,0xA0
	rcall	wrser		;SPI write (byte 1)
	mov	s_data,addrh
	rcall	wrser		;SPI write (byte 2)
	mov	s_data,addrl
	rcall	wrser		;SPI write (byte 3)
	rcall	rdser		;Send data (byte 4)
	mov	u_data,s_data
	rcall	putc

readaddrinc:
	ldi	temp1,1		;auto-increment address
	clr	temp2
	add	addrl,temp1
	adc	addrh,temp2
	rjmp	waitcmd


;**** Leave programming mode ****

w14:	
	cpi	u_data,'L'	;leave programming mode?
	brne	w15

	ddrb_release
	set_reset		;release 'RESET'
	rjmp    put_ret


;**** Chip erase ****

w15:	
	cpi	u_data,'e'	;chip erase?
        brne	w16

	ldi	s_data,0xAC
	rcall	wrser
	ldi	s_data,0x80
	rcall	wrser
	ldi	s_data,4
	rcall	wrser
	ldi	s_data,0
	rcall	wrser
	ldi	temp1,0x30
	rcall	delay
	rjmp	put_ret


;**** Write lock bits ****

w16:	
	cpi	u_data,'l'	;write lock bits?
	brne	w17

	rcall	getc
	ldi	s_data,0xAC
	rcall	wrser
	mov	s_data,u_data
	andi	s_data,6
	ori	s_data,0xE0
	rcall	wrser
	ldi	s_data,0
	rcall	wrser
w162:	
	ldi	s_data,0
	rcall	wrser
	ldi	temp1,0x30
	rcall	delay
	rjmp	put_ret


;**** Read signature bytes ****

w17:	
	cpi	u_data,'s'	;read signature bytes?
	brne	w18

	ldi	param1,2
	rcall	w17call
	ldi	param1,1
	rcall	w17call
	ldi	param1,0
	rcall	w17call
	rjmp	waitcmd

w17call:
	ldi	s_data,0x30
	rcall	wrser
	ldi	s_data,0
	rcall	wrser
	mov	s_data,param1
	rcall	wrser
	rcall	rdser
	mov	u_data,s_data
	rcall	putc
	ret


;**** Write program memory page ****

w18:	
	cpi	u_data,'m'	;write program memory page?
	brne	w19

	ldi	s_data,0x4C
	rcall	wrser		;SPI write (byte 1)

	mov	s_data,addrh 
	rcall	wrser		;SPI write (byte 2)
	mov	s_data,addrl 
	rcall	wrser		;SPI write (byte 3)

	ldi	s_data,0
	rcall	wrser		;SPI write (byte 4)

	ldi	temp1,0xFF
	rcall	delay
	rjmp	put_ret


;**** Universal command ****

w19:	
	cpi	u_data,':'	;universal command?
	brne	w191

	rcall	getc
	mov	cmd1,u_data
	rcall	getc
	mov	cmd2,u_data
	rcall	getc
	mov	cmd3,u_data
	clr	u_data
	rcall	NewUniversal

	ldi	temp1,0xFF
	rcall	delay
	rjmp	put_ret


;**** New universal command ****

w191:	
	cpi	u_data,'.'	;new universal command?
	brne	w99

	rcall	getc
	mov	cmd1,u_data
	rcall	getc
	mov	cmd2,u_data
	rcall	getc
	mov	cmd3,u_data
	rcall	getc
	rcall	NewUniversal

	ldi	temp1,0xFF
	rcall	delay
	rjmp	put_ret

NewUniversal:
	mov	s_data,cmd1
	rcall	wrser
	mov	s_data,cmd2
	rcall	wrser
	mov	s_data,cmd3
	rcall	wrser
	mov	s_data,u_data
	rcall	wrser

	mov	u_data,rd_s_data

	rcall	putc
	ret
w99:


;**** Command error ****

put_err:
	ldi	u_data,'?'
	rcall	putc		;send <?>
	rjmp	waitcmd


;**** Reply command ****

put_ret:
	ldi	u_data,0x0D
	rcall	putc		;send <CR>
	rjmp	waitcmd


;**** End of file ****
