;**************************************************************************
;**************************************************************************
;*
;*
;*                          SuperCharger Firmware
;* 
;*                          Version 1.0  20/01/03
;*
;*                          Created by P.B.Smith
;*                  (C)2003 Silicon Chip Publications P/L
;*                          All Rights Reserved.
;*
;*                         www.siliconchip.com.au
;*
;*              +-------------------------------------------+
;*              | This source code is released as FREEWARE  |
;*              | and as such is NOT in the public domain.  |
;*              +-------------------------------------------+
;*
;************************************^*************************************


;**************************************************************************
;*
;* CHANGE RECORD
;*
;* Version 1.1  18/04/04  PBS
;*
;* Fixed bug in 0mV delta V termination
;* Increased 0mV delta V filter timer to 18/36mins
;* Made -6mV delta V termination more robust 
;* Improved LTC 'failsafe' error handling
;* Reduced top-off time to 1 hour for both NiMH & NiCd
;* Modified discharge-before-charge cell auto-detect
;* Modified progress bar scaling slightly
;* Increased max. {vin_v} measurement to 28V
;*
;**************************************************************************

.nolist
.include "2313ndef.inc"		;AVR definitions
.list

;**************************************************************************
;*
;* Register definitions.
;*
;**************************************************************************

.def	status		=R0	;SREG save (used by timer_int)
.def	tick		=R1	;ticks every 5ms
.def	seconds		=R2	;ticks every second
.def	minutes		=R3	;ticks every minute (0-239 mins)
.def	hours		=R4	;ticks every hour
.def	tick5m		=R5	;ticks every 5ms (slaved to tick1s)
.def	tick1s		=R6	;ticks every second
.def	gtimer		=R7	;general purpose 5ms timer
.def	ftimer		=R8	;led flash rate timer
.def	btimer		=R9	;switch debounce timer
.def	keynum		=R10	;switch number pressed/released
.def	iobyte		=R11	;front panel output (row/column) save
.def	divider		=R12	;LTC battery divider ratio
.def	adc_mux		=R13	;LTC mux select bits
.def	ltc_mode	=R14	;LTC mode
.def	batp_cnt	=R15	;LTC BATP error counter

.def	A		=R16	;scratch
.def	B		=R17	;scratch
.def	C		=R18	;scratch
.def	D		=R19	;scratch
.def	E		=R20	;scratch
.def	F		=R21	;scratch
.def	adc_lo		=R22	;adc low byte
.def	adc_hi		=R23	;adc high byte
.def	aflags		=R24	;various flags (see bit vars below)
.def	bflags		=R25	;various flags (see bit vars below)
.def	ltc_stat	=R26	;(XL) LTC status (fail-safe) byte
.def	lcnt		=R27	;(XH) scratch loop counter
.def	YL		=R28	;(YL) scratch/pointer
.def	debug		=R29	;(YH) debug register
.def	next_tick	=R30	;(ZL) next sample tick
.def	state		=R31	;(ZH) charge state	

; Register-bound bit variables

; aflags

.equ	dischg		=0	;discharge before charge (0=no, 1=yes)
.equ	c_rate		=1	;charge rate (0=rapid, 1=fast)
.equ	c_slow		=2	;charge rate (0=rapid/fast, 1=slow)  
.equ	c_type		=3	;cell type (0=NiMH, 1=NiCd) 
.equ	key_rdy		=4	;key available (0=no, 1=yes)
.equ	key_stat	=5	;key status (0=release, 1=press)	
.equ	key_wait	=6	;key debounce active (0=no, 1=yes)
.equ	ee_update	=7	;EEPROM update required (0=no, 1=yes)

; bflags

.equ	ltc_reset	=0	;LTC FS latch clear pending
.equ	ltc_batp	=1	;LTC BATP error occurred
.equ	ltc_mcv		=2	;LTC MCV error occurred
.equ	adc_reset	=3	;ADC system reset pending
.equ	adc_reget	=4	;ADC system was reset, resample
.equ	adc_reaut	=5	;ADC system range reset pending


;**************************************************************************
;*
;* Hardware specific.
;*
;**************************************************************************

; Port B bits
 
.equ	LTC_CLK		=7	;LTC 'CLK' & ISP 'CLK' 
.equ	LTC_DIN		=6	;LTC 'DIN' & ISP 'MISO'
.equ	LTC_DOUT	=5	;LTC 'DOUT' ISP 'MOSI'
.equ	LTC_CS		=4	;LTC 'CS' (chip select)
.equ	PWM		=3	;PWM (for current control) 
.equ	PIEZO		=2	;piezo buzzer 
.equ	CCSW		=1	;constant current switch
.equ	COL3		=0	;led matrix column 3 driver (active low)

; Port D bits

.equ	COL2		=6	;led matrix column 2 driver (active low)
.equ	COL1		=5	;led matrix column 1 driver (active low)
.equ	COL0		=4	;led matrix column 0 driver (active low)
.equ	ROW3		=3	;led matrix row 3, key matrix column 1
.equ	ROW2		=2	;led matrix row 2, key matrix column 0
.equ	ROW1		=1	;led matrix row 1, key matrix row 1
.equ	ROW0		=0	;led matrix row 0, key matrix row 0

;**************************************************************************
;*
;* Bit constants.
;*
;**************************************************************************

; LED control (see led_out)

.equ	LON		=7	;led on
.equ	LOFF		=6	;led off
.equ	LTOG		=5	;led flash

; LTC command bits, as aligned in memory

; byte 0

.equ	NULL		=7	;null bit (ignored)
.equ	START		=6	;start bit

; byte 1

.equ	MOD0		=7	;mode select
.equ	MOD1		=6	;mode select
.equ	SGL_DIFF	=5	;single-ended/differential conversion
.equ	MSBF		=4	;MSB first/LSB first
.equ	DS0		=3	;ADC data input select
.equ	DS1		=2
.equ	DS2		=1
.equ	DIV0		=0	;battery divider ratio select

; byte 2

.equ	DIV1		=7
.equ	DIV2		=6
.equ	DIV3		=5
.equ	PS		=4	;power shutdown
.equ	DR0		=3	;duty cycle ratio select
.equ	DR1		=2	
.equ	DR2		=1
.equ	FSCLR		=0	;fail-safe latch clear

; byte 3

.equ	TO0		=7	;timeout period select
.equ	TO1		=6
.equ	TO2		=5
.equ	VR0		=4	;charging loop ref voltage select
.equ	VR1		=3

; LTC status bits

.equ	BATP		=7	;battery present
.equ	BATR		=6	;battery reversed
.equ	FMCV		=5	;maximum cell voltage
.equ	FEDV		=4	;end discharge voltage
.equ	FHTF		=3	;high temperature fault
.equ	FLTF		=2	;low temperature fault
.equ	TOUT		=1	;timeout
.equ	FS		=0	;fail-safe occured

;**************************************************************************
;*
;* Byte constants.
;*
;**************************************************************************

.equ	NOM_CELL_V	=1400	;nominal cell voltage (mV) (upper limit)
.equ	KEY_TIME	=35/5	;key debounce time (5ms increments)

; Charge states

.equ	INIT		=0
.equ	SOFT		=1
.equ	SLOW		=2
.equ	RAPID		=3	;both rapid & fast, see {c_rate}
.equ	TOPOFF		=4
.equ	TRICKLE		=5

; Firmware-generated duty cycle values

.equ	C_1_RATE	=20	;20 in 20 = 100% (1C @ full hardware rate)
.equ	C_2_RATE	=20	;20 in 20 = 100% (0.5C @ 1/2 hardware rate)
.equ	C_3_RATE	=12	;12 in 20 = 60%  (0.3C @ 1/2 hardware rate)
.equ	C_10_RATE	=4	;4  in 20 = 20%  (0.1C @ 1/2 hardware rate)
.equ	C_40_RATE	=1	;1  in 20 = 5%   (0.025C @ 1/2 hardware rate)

; LTC modes

.equ	IDLE		=$00
.equ	CHARGE		=$40
.equ	DISCHARGE	=$80
.equ	GASGAUGE	=$C0

; LTC ADC mux select bits
 	
.equ	GAS_OUT		=$00	;gas gauge output
.equ	BAT_TEMP	=$08	;t_bat pin
.equ	AMB_TEMP	=$04	;t_amb pin
.equ	V_CELL		=$0C	;battery divider output
.equ	V_IN		=$02	;v_in pin (DC input / 10)


;**************************************************************************

	.cseg
	.org	$0

;**************************************************************************
;*
;* Hardware reset entry point.
;*
;**************************************************************************        

	cli				;disable interrupts
	ldi	A,low(RAMEND)		;set stack top
	out	SPL,A
	sbi	ACSR,ACD		;power down comparator
	rjmp	reset			;reset code continues...
	reti				;don't remove this


;**************************************************************************
;*
;* Timer/counter 0 overflow interrupt handler.
;*
;* Tasks:
;*
;*	- Updates the system clock in {hours}:{minutes}:{seconds}
;*	- Scans the keyswitch array
;*	- Multiplexes the LED array
;*
;*	Note:	The clock is *not* entirely accurate, due in part to the
;*	      	4MHz clock division as well as the time taken to perform
;*		certain tasks (some ticks are lost). No attempt is made
;*		to perform periodic adjustments as it's close enough.	
;*
;**************************************************************************

timer_int:
	in	status,SREG		;save state
	push	A
	push	B
	push	YL
	ldi	A,256-78		;78 * (250ns * 256) = 4.992ms 
	out 	TCNT0,A			;reload timer

; 5 millisecond tasks

	inc	tick			;tick...tick...tick
	inc	tick5m			;bump 5ms counter (slaved to tick1s)
	inc	ftimer			;and flash rate counter
	inc	btimer			;and switch debounce wait counter
	inc	gtimer			;and general-purpose timer

	mov	A,tick5m
	cpi	A,200			;200 * 4.992ms = 998.4ms expired?
	brlo	tmr1			;skip if not
	
	clr	tick5m
	inc	tick1s			;bump 1 second timer	

tmr1:	mov	A,tick
	cpi	A,100			;100 * 4.992ms = 499.2ms expired?
	breq	tmr3			;go if so

	cpi	A,200			;200 * 4.992ms = 998.4ms up?
	brlo	tmr4			;no, skip seconds tasks

; one second tasks

	clr	tick			;reset 5ms tick
	inc	seconds			;bump seconds counter
	
	mov	A,seconds
	cpi	A,60			;one minute expired?
	brne	tmr3			;nope
	
	clr	seconds			;yes, reset seconds counter
	inc	minutes			;bump minutes counter
	
	mov	A,minutes
	cpi	A,60
	breq	tmr2			;60 minutes expired
	cpi	A,60*2
	breq	tmr2			;120 minutes expired
	cpi	A,60*3
	breq	tmr2			;180 minutes expired
	cpi	A,60*4
	brne	tmr3			;skip if < 240 minutes

	clr	minutes			;reset minutes counter
tmr2:	inc	hours			;bump hours counter
	
; 1/2 second tasks

tmr3:	rcall	toggle_leds		;toggle selected led bits

; Scan keys & leds every 5ms (200Hz)

tmr4:	rcall	scan_key		;scan switch matrix
	rcall	scan_led		;scan led matrix

tmr5:	pop	YL			;restore state
	pop	B
	pop	A
	out 	SREG,status
	reti


;**************************************************************************
;*
;* Toggle selected led bits for 1/2 second flash rate.
;*
;**************************************************************************

toggle_leds:
	ldi	YL,lrows
	rcall	tld1
tld1:	ld	A,Y			;lrows
	ldd	B,Y+2			;flrows
	inc	tick			
	dec	tick			;seconds mark?
	brne	tld2			;no, must be 'on' cycle	
	com	B			;'off' cycle, invert mask
	and	A,B			;and switch off the flash bits
	rjmp	tld3
tld2:	eor	A,B			;switch on the flash bits
tld3:	st	Y+,A			;update row store
	ret


;**************************************************************************
;*
;* Scan front-panel key array.
;*
;* output: {key_rdy} is set if press or release detected
;*         {key_stat} is set if press, clear if release
;*         {keynum} = key value
;*
;* Front panel switches are connected in a 2 x 2 matrix - two columns
;* and two rows. Column 0 is driven low and both rows read, followed
;* by column 1. If any key is pressed, it is converted to a logical
;* key number (1-4) and returned in {keynum}.
;*
;* Key releases are also detected and returned, allowing the main
;* routines to determine how long a particular key is depressed.
;* Flag bit {key_rdy} indicates a key press or release has occured.
;*
;* A KEY_TIME debounce period is invoked on key press & release.
;*
;* This routine should be called directly before scan_led as it does
;* not restore the PORT D state.
;* 
;**************************************************************************

scan_key:
	sbi	PORTB,COL3		;ensure led column 3 driver is off
	sbrs 	aflags,key_wait		;skip if debounce period active
	rjmp	sk1			;scan next

	mov	A,btimer		;get debounce timer
	cpi	A,KEY_TIME		;waited long enough?
	brlo	sk7			;nope

	cbr	aflags,1<<key_wait	;yes, clear wait flag
		
sk1:	ldi	A,0b11111100		;switch rows to inputs	
	out	DDRD,A		

	ldi	A,0b11111011		;led column 2-0 drivers off, switch column 0 low
	out	PORTD,A

	ldi	A,5			;(min A=3)
sk2:	dec	A			;allow bus to settle [(0.75us * val) + 0.5us]
	brne	sk2	

	in	A,PIND			;read switch rows
	andi	A,0b00000011		;discard everything else
	lsl	A			;shift up out of the way
	lsl	A
	mov	B,A			;save result
	
	ldi	A,0b11110111		;column 1 low
	out	PORTD,A
	
	ldi	A,5			;(min A=3)
sk3:	dec	A			;allow bus to settle [(0.75us * val) + 0.5us]
	brne	sk3	

	in	A,PIND			;read front panel
	andi	A,0b00000011		;keep switch row bits
	or	B,A			;include with previous column 0 bits
	cpi	B,$0F			;any key pressed?
	brne	sk4			;yes

	sbrs	aflags,key_stat		;skip if waiting for release
	rjmp	sk7			; else done

	cbr	aflags,1<<key_stat	;flag released	
	rjmp	sk6			;and exit with status 	

sk4:	sbrc	aflags,key_rdy		;skip if last key collected
	rjmp	sk7			; mailbox full, exit

	sbrc	aflags,key_stat		;skip if not waiting for a release
	rjmp	sk7			; else discard this one

	sbr	aflags,1<<key_stat	;next action is a release
	clr	A
sk5:	inc	A			;convert key bit to logical number (1-4)
	lsr	B
	brcs	sk5
	
	mov	keynum,A
sk6:	clr	btimer			;start debounce timer
	sbr	aflags,1<<key_wait|1<<key_rdy ;mail key & flag debounce wait

sk7:	ldi	A,$FF
	out	PORTD,A			;all leds off
	out	DDRD,A			;all bits out
	ret


;**************************************************************************
;*
;* Multiplex LED array.
;* 
;**************************************************************************

; Determine next active column

scan_led:
	sbi	PORTB,COL3		;ensure led column 3 driver is off
	mov	A,iobyte
	sbr	A,0b00001000		;set internibble carry
	lsl	A			;shift enable bit to next column
	brcs	sl1			;skip if it didn't pop out
	cbr	A,0b00010000		;wrap to first column (0) if it did
sl1:	cbr	A,0b00001111		;clear unwanted (row) bits

; Unpack the row bits for the next active column

	cpi	A,0b11010000		;unpack which row pair?
	mov	iobyte,A
	brlo	sl2
	lds	A,lrows			;row bits for columns 0 & 1
	sbrc	iobyte,4		;0 or 1?
	swap	A			;get 
	rjmp	sl3

sl2:	lds	A,hrows			;row bits for columns 2 & 3
	sbrc	iobyte,6
	swap	A	

; Merge the row & column bits and write to front panel

sl3:	cbr	A,0b11110000		;clear unwanted nibble
	or	iobyte,A		;merge column & row bits
	out	PORTD,iobyte		;write the column driver & row bits
	sbrs	iobyte,7		;skip if column bit 3 should be off		
	cbi	PORTB,COL3		; else turn it on
	ret


;**************************************************************************
;*
;* Hardware reset, initialise everything.
;*
;**************************************************************************        	

reset:	clr	aflags			;init flags
	clr	bflags
	
; Init ports

	ldi	A,0b11101111		;led and key switch row/column bits
	mov	iobyte,A		;COL0 low (on) on first scan
	ldi	A,0b11111111
	out	PORTD,A			;all port D bits pulled up
	out	DDRD,A			;all port D bits to outputs

	ldi	A,0b00011111		;SPI bits in, all else out
	out	DDRB,A
	ldi	A,0b11110111		;SPI bits pulled up, other devices disabled
	out	PORTB,A

; Init Timer/Counter0 (for 4.992ms system timer)

	ldi 	A,0b00000100		;CT0 clocked at CK/256 (64us period)
	out 	TCCR0,A
	ldi	A,1<<TOIE0		;enable CT0 overflow interrupt
	out 	TIMSK,A

; Init Timer/Counter1 (for 8-bit, non-inverting PWM)

	ldi	A,0b00000001
	out	TCCR1B,A		;CT1 clocked at CK: 4MHz / 510 = 7.84kHz PWM
	sei				;enable interrupts

; Light all leds & sound a tone

	rcall	light_leds

;**************************************************************************
;*
;* Soft reset entry point.
;*
;**************************************************************************

soft1:	rcall	beep_once
	rcall	init_ltc		;'reset' LTC
soft2:	rcall	clear_leds		;all leds off

; Perform basic EEPROM data integrity test

	rcall	checksum		;calc. EEPROM checksum
	ldi	a,chksum
	rcall	ee_read			;get stored checksum
	cp	C,B			;match?
	breq	user_input		;yes, continue

	ldi	debug,3			;error code (4 beeps total)
	rcall	user_alert		;crash & burn
ee_err:	rjmp	ee_err


;**************************************************************************
;*
;* Front panel input. 
;*
;**************************************************************************

user_input:
	cbr	aflags,1<<c_slow|1<<c_type|1<<ee_update
	ldi	A,chemistry
	rcall	ee_read			;get last cell type
	tst	B			;NiMH?
	breq	ui2			;yes

	sbr	aflags,1<<c_type	;no, force NiCd
	ldi	B,1

ui2:	ldi	A,12
	add	A,B			;add base led number
	sbr	A,1<<LON	
	rcall	led_out			;light 'cell type' led	

ui3:	ldi	A,capacity
	rcall	ee_read			;get last cell capacity
	cpi	B,9			;check valid (0-8)
	brlo	ui4			;skip if OK
	clr	B			;else force to lowest rate
						
ui4:	mov	E,B			;save a copy
	mov	A,B
	sbr	A,1<<LON
	rcall	led_out			;light the appropriate capacity led

ui5:	ldi	A,9 | 1<<LON
	rcall	led_out			;light 'standby' led

ui6:	rcall	get_key			;wait for keypress
	brcc	ui6
	
	dec	A	
	brne	ui7			;go if not 'up' key

; 'Up' key

	cpi	E,8			;'up' key, already at max?
	breq	ui6			;ignore if so
	
	mov	A,E
	sbr	A,1<<LOFF
	rcall	led_out			;current led off
	inc	E			;step up
	mov	A,E
	sbr	A,1<<LON
	rjmp	ui10			;to turn on next highest led

ui7:	dec	A
	brne	ui9			;go if not 'cell type' key
	
; 'Cell type' key

	sbrc	aflags,c_type		;skip if NiMH  
	rjmp	ui8

	sbr	aflags,1<<c_type	;set for NiCd
	ldi	A,12 | 1<<LOFF
	rcall	led_out			;turn off 'NiMH' led
	ldi	A,13 | 1<<LON
	rjmp	ui10			;go turn on 'NiCd' led

ui8:	cbr	aflags,1<<c_type	;clear for NiMH
	ldi	A,13 | 1<<LOFF
	rcall	led_out			;turn off 'NiCd' led 
	ldi	A,12 | 1<<LON
	rjmp	ui10			;go turn on 'NiMH' led

ui9:	dec	A
	breq	ui11			;go if 'Go/Stop' key
	
; 'Down' key

	tst	E			;must be 'down' key
	breq	ui6			;ignore if already at min.			
	
	mov	A,E
	sbr	A,1<<LOFF
	rcall	led_out			;current led off
	dec	E			;step down
	mov	A,E
	sbr	A,1<<LON		;to turn on next lowest led

ui10:	rcall	led_out
	sbr	aflags,1<<ee_update	;flag EEPROM update required
	rjmp	ui6

; 'Go/Stop' key

ui11:	cbr	aflags,1<<c_rate	;rapid charge is the default
	ldi	A,11 | 1<<LON
	rcall	led_out			;light the 'rapid charge' led
	ldi	A,9 | 1<<LOFF
	rcall	led_out			;turn off the 'standby' led

ui12:	sbr	aflags,1<<dischg	;set discharge before charge (for now)
ui13:	clr	tick5m			;reset timers
	clr	tick1s
ui14:	mov	A,tick1s
	cpi	A,2
	brsh	ui20			;go if waited two secs

	rcall	get_key
	brcs	ui15			;go if press
	brtc	ui14			;loop if not release

ui15:	mov	B,A
	cpi	A,3			;was it the 'Go/Stop' key?
	brne	ui14			;ignore if not
ui16:	brtc	ui17			;skip if press

	cbr	aflags,1<<dischg	;release, clear discharge flag (for now)
	rjmp	ui13			;loop

ui17:	sbrc	aflags,c_rate		;skip if rapid charge
	rjmp	ui18			;must be fast charge

	sbr	aflags,1<<c_rate	;set for fast charge
	ldi	A,10 | 1<<LON
	rcall	led_out			;light the 'fast charge' led
	ldi	A,11 | 1<<LOFF	
	rjmp	ui19			;go turn off the 'rapid charge' led

ui18:	cbr	aflags,1<<c_rate	;clear for rapid charge
	ldi	A,11 | 1<<LON
	rcall	led_out			;light the 'rapid charge' led
	ldi	A,10 | 1<<LOFF
ui19:	rcall	led_out			;turn off the 'fast charge' led
	rjmp	ui12

ui20:	sts	c_index,E		;save capacity index
	mov	A,E
	sbr	A,1<<LOFF
	rcall	led_out			;turn off the capacity led

	sbrs	aflags,ee_update	;skip if EEPROM update required
	rjmp	scharge

	mov	B,E
	ldi	A,capacity
	rcall	ee_write		;update cell capacity index
	
	clr	B
	sbrc	aflags,c_type
	inc	B
	ldi	A,chemistry
	rcall	ee_write		;update cell type


;**************************************************************************
;*
;* Charge sequence.
;*
;* [DISCHARGE] -> INIT -> SOFT -> RAPID/FAST -> TOPUP -> TRICKLE
;*    -or-
;* [DISCHARGE] -> INIT -> SLOW -> TRICKLE
;*
;**************************************************************************

scharge:
	ldi	A,11 | 1<<LTOG
	sbrc	aflags,c_rate		;skip if rapid charge
	ldi	A,10 | 1<<LTOG		;must be fast charge
	rcall	led_out			;flash 'rapid charge' or 'fast charge'

	rcall	beep_once		;indicate cycle started
	sbrc	aflags,dischg
	rcall	beep_once		;beep (again) if discharge enabled

; Wait 3 seconds for Go/Stop key press (slow charge request)

	clr	tick5m			;reset timers
	clr	tick1s
sg1:	rcall	get_key
	brcc	sg2			;skip if no key pressed

	cpi	A,3			;was it the 'Go/Stop' key?
	brne	sg2			;ignore if not
	sbr	aflags,1<<c_slow	;else flag slow charge rate (0.1C)
	rcall	beep_once		;and acknowledge

	ldi	A,11 | 1<<LTOG		;flash 'rapid' *and* 'fast' leds
	sbrs	aflags,c_rate
	ldi	A,10 | 1<<LTOG
	rcall	led_out

sg2:	mov	A,tick1s
	cpi	A,3
	brlo	sg1			;loop for about 3 secs	

;**************************************************************************
;*
;* Prepare for discharge and/or charge.
;*
;* Determine cell count and detect open-circuit/reverse-connected packs.
;* 
;**************************************************************************

chg1:	ldi	state,INIT		;charge state
	cbi	PORTB,CCSW		;current source on (to detect open packs)
	rcall	auto_range		;establish initial divider ratio
	sbi	PORTB,CCSW		;current source off
	brts	chg4			;abort if unsuccessful

	sbr	bflags,1<<ltc_reset	;to reset LTC fail-safe latch
	rcall	set_idle		;do it
	mov	A,ltc_stat		;get return status
	andi	A,1<<BATR | 1<<FEDV	;mask error bits
	brne	chg9			;< 900mV and/or < 100mV, go check

	sbrs	aflags,dischg		;skip if discharge before charge
	rjmp	chg11			;else go do slow charge/soft-start

;**************************************************************************
;*
;* Discharge.
;*
;* The stack is discharged to 900mv/cell before charge.
;*
;* NOTE: For the correct discharge end voltage (900mv/cell), the number of
;* cells must be accurately determined. This is only possible if the cells
;* are all within the nominal range (1.2 - 1.4V) at start of discharge!
;*
;**************************************************************************

	sbr	bflags,1<<adc_reaut	;flag for auto-range after 10 secs
chg2:	ldi	A,DISCHARGE
	rcall	set_mode		;enable discharge, get status
	sbrc	ltc_stat,FEDV		;skip if > 900mV 
	rjmp	chg11			;< 900mV, discharge complete 

	clr	tick5m			;reset timers
	clr	tick1s
chg3:	rcall	go_stop			;restart if 'Go/Stop' key
	mov	A,tick1s
	cpi	A,10
	brlo	chg3			;wait about 10 secs

	sbrs	bflags,adc_reaut
	rjmp	chg2			;loop until vcell < 900mV

	cbr	bflags,1<<adc_reaut
	rcall	auto_range		;re-establish divider now
	brtc	chg2			;loop if OK

chg4:	ldi	debug,5
	rjmp	charge_fail		;abort if auto/overrange error

;**************************************************************************
;*
;* Init charge.
;*
;* If total stack voltage is < 900mV, then an initial charge (perhaps
;* better termed a 'cell restoration' charge) is applied. Current is about
;* 50mA DC at 100% duty cycle, as supplied by the constant current source.
;* Terminates when vbatt is > 950mV or after 3 hours have expired. 
;*
;**************************************************************************

chg9:	sbrs	ltc_stat,BATR		;skip if vcell < 100mV
	rjmp	chg10			;must be < 900mV but > 100mV...

; < 100mV, prompt the user to check for reversed cells...

	ldi	debug,3
	rcall	user_alert		;flash all leds & beep 3 times

chg10:	ldi	A,180			;timeout (minutes)
	ldi	B,C_1_RATE
	rcall	charge_cycle
	brcc	chg11
	rjmp	charge_fail		;failed if timeout or other error

chg11:	ldi	A,$18
	sts	ltc_tx3,A		;set LTC 5 min timeout (was disabled)
	ser	A
	sts	dot_bar,A		;reset front panel progress bar

	sbrc	aflags,c_slow		;skip to do soft-start
	rjmp	chg19			;else do slow charge

;**************************************************************************
;*
;* Soft-start charge.
;*
;* Applies 0.5C current at 60% duty cycle for 0.3C effective rate.
;* Terminates after 5 minutes.
;*
;**************************************************************************

chg12:	ldi	A,11 | 1<<LON
	sbrc	aflags,c_rate		;skip if rapid charge
	ldi	A,10 | 1<<LON		;must be fast charge
	rcall	led_out			;steady 'rapid charge' or 'fast charge'
	
	ldi	state,SOFT		;charge state
	ldi	A,5			;timeout (minutes)
	ldi	B,C_3_RATE		;duty cycle (0.3C)
	rcall	charge_cycle
	brtc	charge_fail		;failed if *not* timeout

;**************************************************************************
;*
;* Rapid/fast charge.
;*
;* Applies 1.5C / 1C / 0.5C current at 100% duty cycle.
;* Terminates at -delta V / zero delta V.
;*
;* Timeouts are at about 120% (NiMH) & 135% (NiCd) real capacity,
;* with 5% added to allow for set_mode latency.
;*
;* Nicads above 1200mAH can't be charged at the full 1.5C rate,
;* so timeouts must be adjusted accordingly, as follows:
;*
;*    1400mAH capacity @ 1800mA = 1.28C = 65 mins
;*    1600mAH capacity @ 1800mA = 1.12C = 75 mins
;*    1800mAH capacity @ 1800mA = 1C    = 84 mins
;*
;**************************************************************************

chg13:	ldi	state,RAPID

	sbrc	aflags,c_type		;skip if NiMH
	rjmp	chg14

	ldi	A,75			;rapid charge (1C) t/o for NiMH (125%)
	sbrc	aflags,c_rate
	ldi	A,150			;fast charge (0.5C) t/o for NiMH (125%)
	rjmp	chg16

chg14:	ldi	A,168			;fast charge (0.5C) t/o for NiCd (140%)
	sbrc	aflags,c_rate		;skip if not fast
	rjmp	chg16	

	lds	B,c_index		;get cell capacity index
	ldi	A,56			;rapid charge (1.5C) t/o for NiCd (140%)
	subi	B,6
	brlo	chg16			;200 - 1200mAH, use 1.5C timeout

chg15:	subi	A,-10			;add 10 mins
	dec	B
	brpl	chg15			;calc. for 1400, 1600 & 1800mAH
	dec	A

chg16:	ldi	B,C_1_RATE
	rcall	charge_cycle
	brts	chg17			;don't error on timeout
	brcs	charge_fail		;failed

chg17:	rcall	beep_once		;indicate rapid/fast charge end
	rcall	clear_leds		;clear progress bar

;**************************************************************************
;*
;* Top-off charge.
;*
;* Applies 0.5C current at 20% duty cycle for 0.1C effective rate.
;* Terminates after 1 hour.
;*
;**************************************************************************

chg18:	ldi	A,11 | 1<<LTOG
	sbrc	aflags,c_rate		;skip if rapid charge
	ldi	A,10 | 1<<LTOG
	rcall	led_out			;flash 'rapid' or 'fast'

	ldi	state,TOPOFF
	ldi	A,60			;timeout (minutes)
	rjmp	chg20

;**************************************************************************
;*
;* Slow charge.
;*
;* Applies 0.5C current at 20% duty cycle for 0.1C effective rate.
;* Terminates after 16 hours.
;*
;**************************************************************************

chg19:	ldi	state,SLOW		;charge state
	ldi	A,16			;timeout (hours) 
chg20:	ldi	B,C_10_RATE		;duty cycle (0.1C)
	rcall	charge_cycle
	brtc	charge_fail		;failed if *not* timeout
	rcall	clear_leds		;clear progress bar

;**************************************************************************
;*
;* Trickle charge.
;*
;* Applies 0.5C current at 5% duty cycle for 0.025C effective rate.
;* Terminates when 'Go/Stop' key pressed.
;*
;**************************************************************************

chg21:	ldi	A,9 | 1<<LTOG
	rcall	led_out			;flash 'standby' led
	
	ldi	state,TRICKLE
	ldi	B,C_40_RATE
	rcall	charge_cycle		;trickle charge until user abort
	
; Fault occured if returns here (probably disconnected batteries)

;**************************************************************************
;*
;* Charge failed or user aborted.
;*
;**************************************************************************

; Beep {debug} times and flash all leds.

charge_fail:
	rcall	init_ltc		;put LTC in idle mode
cf1:	rcall	user_alert		;indicate error condition
	rjmp	soft2			;restart


;**************************************************************************
;*
;* Restart if 'Go/Stop' key pressed
;*
;**************************************************************************

go_stop:
	rcall	get_key			;key press?
	brcc	ua2			;continue if not
	cpi	A,3			;'Go/Stop' key?
	brne	ua2			;ignore if not

ck1:	ldi	A,low(RAMEND)
	out	SPL,A			;reset stack
	rjmp	soft1			;restart


;**************************************************************************
;*
;* User alert
;*
;**************************************************************************

user_alert:
	rcall	flash_leds		;flash all leds
	mov	A,debug
	rcall	beep_m			;beep {debug} times
ua1:	rcall	get_key			;wait for any keypress
	brcc	ua1
	rcall	restore_leds		;restore led state
ua2:	ret


;**************************************************************************
;*
;* Charge cycle.
;*
;**************************************************************************

charge_cycle:
	sts	timeout,A		;save some stuff
	sts	on_time,B

	clr	seconds			;reset the clock
	clr	minutes
	clr	hours
	clr	batp_cnt		;reset error count
	sbr	bflags,1<<adc_reset|1<<adc_reaut ;to reset detection system

; Get hardware PWM value for desired charge current

cy1:	lds	A,c_index		;get cell capacity index
	mov	B,A
	lsl	B			;for 3-byte entries
	add	B,A
	cpi	state,RAPID
	brne	cy2			;skip if not rapid/fast charge

	sbrc	aflags,c_rate		;skip if rapid
	rjmp	cy2			;use fast (0.5C) rate
	
	inc	B			;bump for rapid (1C) rate
	sbrc	aflags,c_type		;skip if NiMH
	inc	B			;bump for NiCd (1.5C) rate
cy2:	ldi	A,pwm_rate		;pwm rate table
	add	A,B			;add index
	rcall	ee_read			;get pwm rate for this capacity
	sts	pwm_val,B		;save for later (see set_mode)

;**************************************************************************
;*
;* Main charge loop.
;*
;* All charge modes (except INIT) are driven at the rapid/fast rate
;* and pulse-width modulated under firmware control to give the correct
;* average charge current.
;*
;* The PWM period is 20 secs, so an {on_time} of 20 = 100% duty cycle.
;* 
;**************************************************************************

cy3:	clr	tick5m			;reset timers
	clr	tick1s

cy4:	cpi	state,INIT		;INIT state?
	brne	cy5			;skip if not
	cbi	PORTB,CCSW		;constant current source on
	rjmp	cy6
		
cy5:	ldi	A,CHARGE
	rcall	set_mode		;start charge

cy6:	mov	next_tick,tick1s
	inc	next_tick		;next sample second
cy7:	rcall	go_stop			;abort if 'Go/Stop' pressed
	
; Test for cycle timeout

	cpi	state,TRICKLE
	breq	cy8			;no timeout for trickle
		
	lds	A,timeout
	cp	minutes,A
	sbrc	aflags,c_slow
	cp	hours,A
	brlo	cy8
	
	ldi	debug,6
	rjmp	cx2			;abort if timeout

cy8:	cp	tick1s,next_tick	;time to sample?
	brlo	cy7			;loop until it is

	rcall	set_idle		;charge off
	rcall	acquire			;sample voltages

;**************************************************************************
;*
;* Do supply rail & battery voltage limit checks.
;*
;**************************************************************************

; Check for high or low DC rail

	cpi	adc_lo,low(28680)	;{adc_hi:adc_lo} = vin_v
	ldi	A,high(28680)		;
	cpc	adc_hi,A
	ldi	debug,9
	brsh	cy12			;>=28V, abort
	
	cpi	adc_lo,low(10500)
	ldi	A,high(10500)
	cpc	adc_hi,A		;check if < 10.5V (approx 11.5V input)
	ldi	debug,10
	brlo	cy12			;abort

; Check for flat or short-circuit stack
	
	ldi	YL,var_buf		;pointer to data array
	ldd	adc_lo,Y+vbat_v		;get current vbatt sample
	ldd	adc_hi,Y+vbat_v+1

	cpi	adc_lo,low(950)		;>= 950mV?
	ldi	A,high(950)
	cpc	adc_hi,A	
	brlo	cy9			;skip if < 950mV
	
	cpi	state,INIT		;>= 950mV, INIT state?
	brne	cy10			;nope, continue
	rjmp	cx1			;yes, so cycle complete

cy9:	ldi	debug,7			;< 950mV...
	cpi	state,2			;INIT or SOFT state?
	brsh	cy12			;abort if neither
	rjmp	cy16			;else skip remaining checks

cy10:	cpi	state,SOFT		;SOFT state?
	breq	cy16			;skip remaining checks if so

; Check for open circuit / high resistance stack or insufficient headroom.
; Min voltage headroom = 1.8V (BATP comparator) + 0.7V (VDD reg) = 2.5V

cy11:	sbrc	bflags,ltc_mcv		;skip if not MCV error
	rjmp	dt3			;else abort with error code 8

	sbrs	bflags,ltc_batp		;skip if BATP failsafe
	rjmp	cy13			;else continue

	inc	batp_cnt		;bump error count
	mov	A,batp_cnt
	cpi	A,3			;three consecutive errors?
	brne	cy15			;go reset detection system if not

	ldi	debug,11
cy12:	rjmp	cx3			;else abort

cy13:	clr	batp_cnt		;reset error count
	cpi	state,RAPID		;rapid/fast charge?
	brne	cy16			;skip terminating tests if not

	mov	A,minutes
	cpi	A,5			;charging for 5 mins?
	brlo	cy16			;skip tests if not

	sbrs	bflags,adc_reaut
	rjmp	cy14

; After 5 mins. of rapid/fast charge, the vbatt divider is reset
; to zero (1 cell). This will cause set_mode to (re)establish the
; correct ratio for the pack under charge.

	cbr	bflags,1<<adc_reaut
	clr	divider			;reset divider

cy14:	sbrs	bflags,adc_reset	;skip if reset request
	rjmp	cy17

; Termination detect system reset...

	cbr	bflags,1<<adc_reset	;clear the reset
cy15:	sbr	bflags,1<<adc_reget	;one cycle required for settling...
cy16:	rjmp	dt13			;skip tests

cy17:	sbrs	bflags,adc_reget
	rjmp	dt1

	cbr	bflags,1<<adc_reget

; Clear all detect variables (peak/previous voltages and 'filter' counters)

cy18:	ldi	YL,var_buf+2
	ldi	lcnt,8
	clr	A
cy19:	st	Y+,A
	dec	lcnt
	brne	cy19
	
;**************************************************************************
;*
;* Detect terminating conditions (rapid & fast charge modes only).
;*
;* The charge is terminated on one of three conditions:
;*
;*      * -6mV delta V (filter: 10 seconds)
;*      * 0mV (+/-3mV) delta V (filter: rapid=18 mins, fast=36 mins)
;*      * 120% (NiMH) or 135% (NiCd) returned capacity (timeout)
;*
;* Note: the 0mV delta V termination is intended only to limit damage that
;*	 occurs when attempting to charge cells that are already fully
;*       charged and therefore may not exhibit a negative voltage dip.
;*         
;**************************************************************************

; Check for radical voltage excursions

dt1:	ldi	YL,var_buf		;pointer to data array
	ldd	C,Y+vbat_ps		;get previous sample
	ldd	D,Y+vbat_ps+1
	
	mov	A,C
	or	A,D
	breq	dt4			;skip check if reget	

	sub	C,adc_lo		;previous - current
	sbc	D,adc_hi
	brsh	dt2			;current <= previous
	
	com	D			;current > previous, adjust
	neg	C
	brcs	dt2
	inc	D

dt2:	tst	D			;+/- 256mV or more vbatt change?
	breq	dt4			;skip if not

dt3:	ldi	debug,8			;error code to use
	rjmp	cx3			;abort

; Compute 6mV/cell value based on current v_batt divider

dt4:	clr	B
	mov	lcnt,divider		;v_batt divider
dt5:	subi	B,-6
	dec	lcnt
	brpl	dt5

; Compare current sample with peak sample thus far

	ldd	C,Y+vbat_pk		;get peak sample
	ldd	D,Y+vbat_pk+1

	sub	C,adc_lo		;peak - current
	sbc	D,adc_hi
	brsh	dt7			;current <= peak
	
	com	D			;current > peak, adjust
	neg	C
	brcs	dt6
	inc	D

dt6:	cp	C,B
	brlo	dt9			;must be +1 digit, do 0dV detect 

; Current sample is >= peak + 2 digits (+6mV)

	clr	A
	std	Y+zdv_cnt,A
	std	Y+zdv_cnt+1,A
	std	Y+ndv_cnt,A
	
	ldd	A,Y+pkv_cnt		;get 'filter' counter
	inc	A
	cpi	A,10			;10 secs (10 detects)?
	brlo	dt12

	std	Y+vbat_pk,adc_lo	;save current as new peak
	std	Y+vbat_pk+1,adc_hi
	rjmp	dt11			;done for now

; Current sample <= peak, check if -2 digits (-6mV)

dt7:	ldd	A,Y+ndv_cnt		;get 'filter' counter
	inc	A
	cp	C,B
	brsh	dt8
	
	clr	A			;reset count if not -2 digits
dt8:	std	Y+ndv_cnt,A
	cpi	A,10			;10 secs (10 detects)?
	breq	cx1			;terminate

; Current sample is +1 or less compared to peak

dt9:	ldd	C,Y+zdv_cnt		;get 16-bit 'filter' counter
	ldd	D,Y+zdv_cnt+1
	inc	C			;add one
	brne	dt10
	inc	D
dt10:	cpi	C,low(18*60)		;18 minute count (rapid)
	sbrc	aflags,c_rate		;skip if rapid charge
	cpi	C,low(36*60)		;36 minute count (fast)
	ldi	A,high(18*60)
	sbrc	aflags,c_rate
	ldi	A,high(36*60)	
	cpc	D,A
	brsh	cx1			;0dV terminate

	std	Y+zdv_cnt,C		;save the 0dV 'filter' counter
	std	Y+zdv_cnt+1,D
			
dt11:	clr	A
dt12:	std	Y+pkv_cnt,A		;update peak 'filter' counter
	std	Y+vbat_ps,adc_lo	;save current sample
	std	Y+vbat_ps+1,adc_hi

; Done detection stuff, update front panel.

dt13:	cpi	state,SLOW
	breq	dt14			;update if SLOW charge 

	cpi	state,RAPID
	brne	dt15			;no update for SOFT, TOPUP or TRICKLE
	
dt14:	rcall	prog_bar		;do the update

; 'Soft PWM' pulse width & period generation.

dt15:	ldi 	A,20
	cp	tick1s,A		;end of modulation period (20 secs)?
	brlo	dt16			;skip if not
	rjmp	cy3			;else loop to reset for next period	

dt16:	lds	A,on_time
	cp	tick1s,A		;end of on_time (pulse width)?
	brsh	dt17			;skip if expired 
	rjmp	cy4			;else loop to restart charge 

dt17:	rjmp	cy6			;loop without charge restart

;**************************************************************************
;*
;* Charge cycle exits here (except user abort). 
;*
;**************************************************************************

cx1:	clc				;success
	rjmp	cx5

cx2:	set				;timeout flag
	rjmp	cx4

cx3:	clt				;no timeout
cx4:	sec				;error flag
cx5:	ret


;**************************************************************************
;*
;* Generate front panel 'progress bar'.
;*
;* In charge mode, front panel cell capacity LEDs become elasped time
;* indicators, with full scale being approx. 105% (NiMH) and 115% (NiCd)
;* of selected capacity.
;*
;**************************************************************************

prog_bar:
	lds	A,dot_bar		;get last led lit
	inc	A			;first iteration?
	breq	pg1			;yes

	cpi	A,9
	brsh	pg4			;ignore if all on already

	lds	B,bar_time		;get minutes mark
	mov	C,minutes
	sbrc	aflags,c_slow
	mov	C,hours
	cp	C,B			;time for update?
	brlo	pg4			;skip if not
	
pg1:	sts	dot_bar,A
	sbr	A,1<<LON
	rcall	led_out			;light the next led

; Reset minutes mark

	mov	B,hours
	ldi	A,2
	sbrc	aflags,c_slow
	rjmp	pg3			;slow charge (0.1C = 2 hours/dot)
	
	mov	B,minutes
	sbrc	aflags,c_type		;skip if NiMH
	rjmp	pg2			;must be NiCd

	ldi	A,8			;rapid charge (1C) = 8 mins/dot
	sbrc	aflags,c_rate		;skip if rapid
	ldi	A,16			;fast charge (0.5C) = 16 mins/dot
	rjmp	pg3
	
pg2:	ldi	A,17			;fast charge (0.5C) = 17 mins/dot
	sbrc	aflags,c_rate		;skip if rapid
	rjmp	pg3

; NiCd @ 1.5C rate, handle as special case...

	lds	C,c_index		;get cell capacity index
	ldi	A,6			;1.5C = 6 mins/dot
	subi	C,6
	brlo	pg3			;go if 200 - 1200mAH (1.5C)			
	
	add	A,C			;adjust for 1400 - 1800mAH
	inc	A

pg3:	add	B,A			;time to light next led
	sts	bar_time,B		;save it		
pg4:	ret


;**************************************************************************
;*
;* Auto detect number of connected cells.
;*
;* output: {adc_hi:adc_lo} = last sample
;*         {divider} = updated vbatt divider ratio
;*
;* uses:   A,B,C,D,E,lcnt
;*
;**************************************************************************

auto_range:
	ldi	E,13			;cycle count + 1
	rjmp	aad5

aad1:	ldi	A,V_CELL
	rcall	read_adc		;read vbatt

	clr	B
	clr	C
	clr	lcnt

aad2:	subi	B,low(-NOM_CELL_V)	;add one cell voltage
	sbci	C,high(-NOM_CELL_V)

	cp	adc_lo,B		;sample < accumulated stack volts?
	cpc	adc_hi,C	
	brlo	aad3			;yes, exit loop			

	inc	lcnt			;no, bump cell count
	cpi	lcnt,7
	brlo	aad2			;scan for up to 7 cells (0-6)
	rjmp	aad6			;abort if exceeds max. (overrange)

; Determine if vbatt is within same range as last measurement.

aad3:	cp	divider,lcnt		;same as current divider?
	breq	aad7			;yes, just exit

aad4:	mov	divider,lcnt		;update divider
	rcall	set_idle		;write divider change to LTC

aad5:	ldi	A,30
	rcall	delay_m			;wait for VCELL settling time (150ms)

	dec	E			;bump cycle count
	brne	aad1			;try 12 times (arbitrary)

aad6:	set				;overrange error
	rjmp	aad9
	
aad7:	cpi	lcnt,6 
	brne	aad8
	dec	divider			;assume 6-cell stack, adjust down
aad8:	clt	
aad9:	ret


;**************************************************************************
;*
;* Read battery & DC rail voltages.
;*
;* output: {adc:hi:adc_lo} = rail voltage
;*         {vbat_v} = vbatt
;* 
;**************************************************************************

acquire:
	ldi	A,V_CELL
	rcall	read_adc		;read vbatt

	sts	var_buf+vbat_v,adc_lo	;save sample
	sts	var_buf+vbat_v+1,adc_hi

; Read and save DC rail voltage

aq4:	ldi	A,V_IN			;to sample rail voltage (vin)
;					;return in {adc:hi:adc_lo}

;**************************************************************************
;*
;* Read input voltage.
;*
;* input:  {A} = ADC channel
;* output: {adc_hi:adc_lo} = sample in millivolts
;*
;* uses:   A,B,C,D,lcnt 
;*
;**************************************************************************

read_adc:
	mov	adc_mux,A		;write ADC channel to command string
	mov	A,ltc_mode		;to preserve current mode
	rcall	set_mode		;do the read

	lds	A,ltc_rx1		;get result in {B:A}
	lds	B,ltc_rx0
	andi	B,3
	ldi	adc_lo,3		;multiplier
	rcall	mpy8_16			;convert to millivolts {B:A} * 3
	
	mov	A,adc_lo		;get result in {B:A} again
	mov	B,adc_hi
	mov	C,adc_mux		;channel select

	cpi	C,V_IN			;measuring v_in?
	ldi	adc_lo,10		;for fixed divide-by-10
	breq	rad1			;skip if v_in channel
	
	mov	adc_lo,divider		;assume v_cell, get multiplier
	inc	adc_lo			;zero-based, adjust
rad1:	rcall	mpy8_16			;scale {adc_hi:adc_lo} to mV
	ret		


;**************************************************************************
;*
;*  Initialise LTC and place in idle mode.
;*
;**************************************************************************

init_ltc:
	sbi	PORTB,CCSW		;ensure constant current source off
	ldi	YL,ltc_tx0		;pointer to LTC command/status store
	ldi	A,$40			;command byte 0 initial value
	st	Y+,A
	ldi	A,$3C			;             1	
	st	Y+,A
	ldi	A,$03			;             2
	st	Y+,A			
	ldi	A,$F8			;             3
	st	Y,A

	ldi	A,5			;initial vbatt divider
	mov	divider,A
	ldi	A,V_CELL		;initial adc channel
	mov	adc_mux,A
	clr	ltc_mode		;for IDLE mode


;**************************************************************************
;*
;* Put LTC in idle mode
;*
;**************************************************************************

set_idle:
	ldi	A,IDLE


;**************************************************************************
;*
;* Set LTC mode.
;*
;* input:  {A} = desired mode
;* output: {ltc_stat} = LTC status
;*
;* uses:   A,B,C,YL
;*
;* When entering charge mode, the current is ramped up gently by lowering
;* the D-A output over a defined period. This is necessary to prevent
;* false-triggering the LTC 'BATP' comparator.
;*
;* If an MCV error is returned on exiting charge mode, then the vbatt
;* divider is incremented and the mode set rerun. This dynamically adjusts
;* for rising cell voltage, making sure that the ADC is never overranged. 
;*
;**************************************************************************

set_mode:
	push	ltc_mode		;save current mode
	mov	ltc_mode,A		;update with new
	cpi	A,CHARGE		;entering charge mode? 
	brne	sm1			;skip PWM enable if not
		
; Set PWM (D-A) for lowest programmable current.

	ldi	A,230			;value for about 100mA (+1 bit)
	out	OCR1AL,A		;write it to the control register
	ldi	A,$81
	out	TCCR1A,A		;enable PWM output

	ldi	A,7
	rcall	delay_m			;wait for D-A settling time (35ms)

	sbr	bflags,1<<ltc_reset	;to reset fail-safe latch
sm1:	rcall	rw_ltc			;write the mode
	mov	B,ltc_mode
	cpi	B,CHARGE		;charge mode?
	pop	B			;B = previous mode
	brne	sm5			;skip if not charging

	sbrs	ltc_stat,FS		;skip if command failed
	rjmp	sm2

	sbr	bflags,1<<ltc_reset
	rcall	rw_ltc			;retry the charge command
	
; Charge mode entered, ramp up the current...

sm2:	lds	C,pwm_val		;desired PWM value
	ldi	B,230			;base PWM value			
sm3:	out	OCR1AL,B		;write it to the control register

	clr	A
sm4:	dec	A
	brne	sm4			;delay about 192uS...
	
	cp	C,B
	dec	B			;step size
	dec	B
	brlo	sm3			;loop until we've reached desired value	

	out	OCR1AL,C		;adjust for possible over-step
	rjmp	sm14			;and exit
	
sm5:	cpi	B,CHARGE		;did we just exit charge mode?
	brne	sm13			;nope

; Exited charge mode, check fail-prone latch for errors

sm6:	cbr	bflags,1<<ltc_mcv|1<<ltc_batp ;clear error flags initially

	mov	A,ltc_stat
	andi	A,1<<BATR|1<<FHTF|1<<FLTF ;sanity check...
	brne	sm13			;ignore these errors
	
sm7:	sbrc	ltc_stat,FMCV		;skip if not MCV ($A0/$A1)
	rjmp	sm10			;else go bump divider

	sbrc	ltc_stat,FEDV		;skip if not EDV ($90/$91)
	sbrs	bflags,adc_reaut	;continue if first 5 mins of charge
	rjmp	sm8			;else ignore
	
	dec	divider			;EDV error, reduce divider
	brpl	sm12

	inc	divider			;was already at min...
	rjmp	sm13			; ignore & exit
	
sm8:	sbrs	ltc_stat,BATP		;skip if BATP (battery present)
	rjmp	sm9			;no BATP (there's no pull-up on vbatt,
;					;so this actually means VDD is too low)

	sbrs	ltc_stat,FS		;assume BATP triggered FS (not latched!)
	rjmp	sm13			;checks complete, exit

sm9:	sbr	bflags,1<<ltc_batp	;flag BATP error (insufficient headroom)
	rjmp	sm13			; and quit

; Charge terminated with MCV (maximum cell voltage) error.
; Increase the divider by one point and rerun the mode set.

sm10:	ldi	A,6
	cp	divider,A		;allow 1-6 cells (0-5)
	brlo	sm11
	
	sbr	bflags,1>>ltc_mcv	;exceeded max. range
	rjmp	sm13			;quit

sm11:	inc	divider			;bump divider
sm12:	rcall	rw_ltc			;update LTC divider

	ldi	A,30
	rcall	delay_m			;wait for vcell settling time (150ms)

	sbr	bflags,1<<adc_reset	;reset flag for charge_cycle

; Disable PWM output

sm13:	cbi	PORTB,PWM		;ensure port pin is low
	clr	A
	out	TCCR1A,A		;disable & disconnect PWM
sm14:	ret


;**************************************************************************
;*
;* Build LTC command string & transmit it.
;*
;**************************************************************************

rw_ltc:

; Transfer MOD, DS & DIV0 bits to LTC command byte 1

	ldi	YL,ltc_tx1
	ld	A,Y
	andi	A,0b00110001		;exclude old
	or	A,ltc_mode		;include new MOD (mode) bits
	or	A,adc_mux		;include new DS (adc mux) bits

	mov	B,divider
	bst	B,0
	bld	A,DIV0			;include new DIV0 bit
	st	Y+,A

; Transfer DIV1 - DIV3 & FSCLR bits to LTC command byte 2

	ld	A,Y
	bst	B,1
	bld	A,DIV1
	bst	B,2
	bld	A,DIV2
	bst	B,3
	bld	A,DIV3

	bst	bflags,ltc_reset
	bld	A,FSCLR
	st	Y,A
	cbr	bflags,1<<ltc_reset	;clear LTC reset status
	
	
;**************************************************************************
;*
;* Write/read LTC command/adc/status word.
;*
;* Timing for **4MHz crystal**
;*
;* output: {A} = LTC status
;*
;* uses: A,B,C,YL
;*
;**************************************************************************

spi_io:
	cli				;interrupts off!
	ldi	A,0b11011111
	out	DDRB,A			;PORTB all bits out except LTC_DOUT
	cbi	PORTB,LTC_CLK		;CLK low for now
	
	ldi	B,6			;bytes to send + receive
	ldi	YL,ltc_tx0		;pointer to command/status store
	cbi	PORTB,LTC_CS		;enable LTC
	ldi	C,2			;only 2 bits in first byte
	rjmp	spi2

spi1:	ldi	C,8			;bits in this byte
spi2:	clr	A			;for null transmit
	cpi	B,3			;'real' tx and/or rx data?
	brlo	spi3			;skip if rx only, use null for tx
	
	ld	A,Y+			;fetch tx byte
spi3:	lsl	A			;pop the next bit out
	brcs	spi4			;go if high bit
	cbi	PORTB,LTC_DIN		;must be low bit
	rjmp	spi5
	
spi4:	sbi	PORTB,LTC_DIN		;make DIN high

spi5:	rjmp	spi6			;waste 2 clocks...400ns min. setup

spi6:	sbi	PORTB,LTC_CLK		;CLK high (strobe bit in)	

	rjmp	spi7			;waste 4 clocks...			
spi7:	rjmp	spi8			;...800ns min. CLK high (150ns hold)

spi8:	cbi	PORTB,LTC_CLK		;CLK low (strobe bit out)

	rjmp	spi9			;waste 3 clocks...
spi9:	nop

	sbic	PINB,LTC_DOUT		;skip if low bit
	inc	A			;else make high bit
	dec	C
	brne	spi3			;do entire byte

	cpi	B,4			;rx data yet?
	brsh	spi10			;skip if not

	st	Y+,A			;store received byte
spi10:	dec	B			;bytes to do
	brne	spi1			;loop to do complete 48-bit word
	
	mov	ltc_stat,A		;save status

	sbi	PORTB,LTC_CS		;disable LTC first
	ldi	B,0b00011111		;PORTB SPI bits in, all else out
	out	DDRB,B
	sbi	PORTB,LTC_DOUT		;pull up the SPI pins
	sbi	PORTB,LTC_DIN
	sbi	PORTB,LTC_CLK
	sei				;reenable interrupts
	ret
	

;**************************************************************************
;*
;* LED control.
;*
;* input: {A} = logical LED number (0-15) & on/off/flash parameter bits,
;*              organised as follows:
;*
;*           
;* bit no. 7 6 5 4 3 2 1 0
;*         | | |   | | | |
;*         | | |   +-+-+-+--- logical LED number (0-15)
;*         | | |
;*         | | +------------- 1=flash
;*         | |
;*         | +--------------- 1=off
;*         |   
;*         +----------------- 1=on     
;* 
;*
;* uses:  A,B,C,Z,lcnt
;*
;*
;*  +--------------------------------+
;*  |   ---            8  O    ---   | 
;*  |2 |S1 | O 13      7  O   |S2 | 1|
;*  |  |   | O 12      6  O   |   |  |
;*  |   ---            5  O    ---   |
;*  |                  4  O          |
;*  |   ---  O 11      3  O    ---   | 
;*  |3 |S4 | O 10      2  O   |S3 | 4|
;*  |  |   | O  9      1  O   |   |  |
;*  |   ---            0  O    ---   |
;*  +--------------------------------+
;*
;*  Front panel LED & switch numbering
;*
;*
;**************************************************************************

led_out:
	push	lcnt
	mov	C,A			;save parameter bits

	andi	A,$0F
	ldi	B,led_table
	add	A,B
	rcall	ee_read			;get physical bit number

	mov	lcnt,B			;becomes shift count
	cbr	lcnt,0b00001000		;clear high bit
	inc	lcnt			;one iteration at least
	clr	A
	sec				;bit to be shifted

lo1:	rol	A			;shift the bit...
	dec	lcnt
	brne	lo1			;...into position
						
	ldi	YL,lrows		;pointer to row patterns
	sbrc	B,3			;skip if low rows
	inc	YL			;high rows, bump pointer

	sbrs	C,LOFF			;skip if OFF request
	rjmp	lo2
	
	ld	B,Y			;get current pattern
	or	B,A			;turn bit OFF
	st	Y,B			;update store
	rjmp	lo3

lo2:	sbrs	C,LTOG			;skip if FLASH request
	rjmp	lo3
	
	inc	YL			;bump pointer to flrows/fhrows
	inc	YL
	ld	B,Y			;get current pattern
	or	B,A			;enable the bit
	rjmp	lo5			;go store

lo3:	com	A			;ON or OFF, invert mask
	sbrs	C,LON			;skip if ON
	rjmp	lo4			;done OFF, so go disable FLASH 
	
	ld	B,Y			;get current row pattern
	and	B,A			;turn new bit ON (active low)
	st	Y,B

lo4:	inc	YL			;bump pointer to flrows/fhrows
	inc	YL
	ld	B,Y
	and	B,A
lo5:	st	Y,B
	pop	lcnt
	ret


;**************************************************************************
;*
;* Flash all LEDs.
;*
;* Writes lows to all rows (to enable) and highs to all
;* flash row bits. Saves existing state for later restore.
;*  
;**************************************************************************

flash_leds:
	ldi	YL,lrows
	clr	A
	rcall	fl1
	dec	A
fl1:	ld	B,Y
	st	Y+,A	
	std	Y+3,B	
	ld	B,Y
	st	Y+,A
	std	Y+3,B
	ret


;**************************************************************************
;*
;* Restore LED state.
;*
;**************************************************************************

restore_leds:
	ldi	YL,lrows
	ldi	lcnt,4
rl1:	ldd	A,Y+4
	st	Y+,A
	dec	lcnt
	brne	rl1
	ret
	

;**************************************************************************
;*
;* Turn all LEDs on.
;*
;**************************************************************************

light_leds:
	ldi	YL,lrows
	clr	A
	st	Y+,A
	st	Y+,A
	rjmp	cl1
	

;**************************************************************************
;*
;* Turn all LEDs off.
;*
;**************************************************************************

clear_leds:
	ldi	YL,lrows
	ser	A
	st	Y+,A
	st	Y+,A
	clr	A
cl1:	st	Y+,A
	st	Y,A
	ret


;**************************************************************************
;*
;* Check if front panel key press/release occurred.
;*
;* output: {A} = logical key number, as follows:
;*
;* 	C flag set = key press
;*	C flag clear & T bit set = key release
;*	C flag clear & T bit clear = no press/release	
;*
;**************************************************************************

get_key:
	clt
	sbrs	aflags,key_rdy
	rjmp	gk1			;exit if no key press/release

	mov	A,keynum		;get the key number
	cbr	aflags,1<<key_rdy	;flag the pickup
	sec
	sbrc	aflags,key_stat		;skip if release
	rjmp	gk2			;go if press

	set
gk1:	clc
gk2:	ret
	

;**************************************************************************
;*
;* 'Beep' the piezo.
;*
;**************************************************************************

beep_once:
	ldi	A,1

beep_m:	push	A
	cbi	PORTB,PIEZO		;piezo on
	ldi	A,100/5			;100ms tone length
	rcall	delay_m
	sbi	PORTB,PIEZO		;piezo off
	ldi	A,400/5
	rcall	delay_m			;400 millisecs pause
	pop	A
	dec	A
	brne	beep_m			;beep 'A' times...
	ret


;**************************************************************************
;*
;* Timed delays.
;*
;* input:  {A} = delay in seconds or millisecs (5ms resolution)
;*
;**************************************************************************

delay_s:
	push	A
	ldi	A,1000/5
	rcall	delay_m			;delay one second
	pop	A
	dec	A
	brne	delay_s			;do 'em all
	ret

delay_m:
	clr	gtimer			;reset 5ms timer
dly1:	cp	gtimer,A		;waited long enough?
	brlo	dly1			;loop if not
	ret


;***************************************************************************
;*
;* 16*16-bit and 8*16-bit unsigned multiplication.
;*
;* input:  {adc_hi:adc_lo} = multiplier
;*         {B:A} = multiplicand
;*
;* output: {adc_hi:adc_lo} = result low word
;*         {D:C} = result high word
;*
;* uses:   A,B,C,D,lcnt
;*
;***************************************************************************

mpy8_16:
	clr	adc_hi

mpy16:	clr	C			;clear high bytes of result
	clr	D
	ldi	lcnt,16			;loop count
	lsr	adc_hi
	ror	adc_lo

mp16_1:	brcc	mp16_2			;if bit 0 of multiplier set...
	add	C,A			;...add multiplicand lo to res byte 2
	adc	D,B			;...add multiplicand hi to res byte 3
mp16_2:	ror	D			;shift result byte 3
	ror	C			;shift result byte 2
	ror	adc_hi			;spin result byte 1 and mult hi
	ror	adc_lo			;spin result byte 0 and mult lo
	dec	lcnt			;bump loop count
	brne	mp16_1			;do all 16 bits
	ret
	

;**************************************************************************
;*
;* Checksum EEPROM.
;*
;* output: {C} = 8-bit checksum
;*
;**************************************************************************

checksum:
	clr	C
	ldi	a,chksum+1
cks1:	rcall	ee_read
	add	C,B
	inc	A
	cpi	A,chk_end
	brlo	cks1
	ret


;**************************************************************************
;*
;* EEPROM random read/write.
;*
;* input:  {A} = address, B = data (writes)
;* output: {B} = data (reads)
;*
;**************************************************************************

ee_read:
	sbic	EECR,EEWE
	rjmp	ee_read			;loop until last cycle complete

	out	EEAR,A			;write address
	sbi	EECR,EERE		;set read strobe
	in	B,EEDR			;get data
	ret				;(read takes 4 clk cycles)
	
; Writes typically take 2.5ms @ 5V

ee_write:
	sbic	EECR,EEWE
	rjmp	ee_write		;loop until last cycle complete

	out	EEAR,A			;write address
	out	EEDR,B			;write data
	cli				;interrupts off for the next bit..
	sbi	EECR,EEMWE		;set master write enable
	sbi	EECR,EEWE		;set write strobe
	sei
	ret


;**************************************************************************
;*
;* Define used SRAM.
;*
;**************************************************************************
	
	 .dseg

; LTC command/adc/status word save

; Command bytes (to LTC)

ltc_tx0: .byte	1		;command byte 0
ltc_tx1: .byte	1		;command byte 1
ltc_tx2: .byte	1		;command byte 2
ltc_tx3: .byte	1		;command byte 3

; ADC & status bytes (from LTC)

ltc_rx0: .byte	1		;adc high
ltc_rx1: .byte	1		;adc low
ltc_rx2: .byte	1		;status

; Voltage & charge termination detect variables

var_buf: .byte	10		;data array

; Indices into var_buf array

.equ	 vbat_v	 = 0		;current vbatt voltage
.equ	 vbat_ps = 2		;previous vbatt voltage
.equ	 vbat_pk = 4		;peak vbatt voltage
.equ	 zdv_cnt = 6		;0dV detect filter count
.equ	 ndv_cnt = 8		;-dV detect filter count
.equ	 pkv_cnt = 9		;peak detect filter count

pwm_val: .byte	1		;OCR1 PWM value

bar_time:.byte	1		;front panel progress bar minutes mark
dot_bar: .byte	1		;front panel progress bar position
on_time: .byte	1		;on time per cycle (period) in seconds
timeout: .byte	1		;charge cycle timeout in minutes or hours
c_index: .byte  1		;cell capacity index

; LED row bits

lrows:   .byte	1		;led row pattern (low)
hrows:   .byte	1		;led row pattern (high)
flrows:  .byte	1		;led row flash pattern (low)
fhrows:  .byte	1		;led row flash pattern (high)

lrstate: .byte	1		;led state save
hrstate: .byte	1
flrstate:.byte	1
fhrstate:.byte	1


;**************************************************************************
;*
;* Define used EEPROM.
;*
;**************************************************************************

	.eseg
	.org	0

reserved:
.db	0			;reserved, don't use

chemistry:
.db	0			;cell type (0=NiMH, 1=NiCd)

capacity:
.db	0			;cell capacity index (0-8)

chksum:
.db	$E0			;8-bit checksum of following data

; Lookup table for logical to physical LED bit conversion.

led_table:
	.db	9,5		;0,1
	.db	1,10		;2,3
	.db	6,2		;4,5
	.db	11,7		;6,7
	.db	3,8		;8,9
	.db	12,13		;10,11
	.db	14,15		;12,13
	.db	0,4		;14,15 (not used)


; PWM (D-A) values for 0.5C, 1C & 1.5C charge currents

pwm_rate:
	.db	$E4,$D7,$CA	;100,200,300mA   (200mAH)
	.db	$D7,$BB,$A3	;200,400,600mA   (400mAH)
	.db	$CA,$A3,$7B	;300,600,900mA   (600mAH)
	.db	$BB,$88,$53	;400,800,1200mA  (800mAH)
	.db	$AA,$6E,$2A	;500,1000,1500mA (1000mAH)
	.db	$A3,$53,$02	;600,1200,1800mA (1200mAH)
	.db	$96,$38,$02	;700,1400,1800mA (1400mAH)
	.db	$88,$1D,$02	;800,1600,1800mA (1600mAH)
	.db	$7B,$02,$02	;900,1800,1800mA (1800mAH)

chk_end:			;marks end of 'protected' EEPROM
