#include "util.h"

unsigned int vcc=0;
unsigned int vbat=0;
unsigned int vaux=0;

unsigned int shuntI=0;  //measured pulse current in mA
unsigned int v0=0;      //initial battery V in mV
unsigned int v1=0;      //load battery V in mV

//array for results of scan
unsigned int tResults[TEST_COUNT][3];
unsigned char tDac [TEST_COUNT];
unsigned int intRes=0;  //calculcated .intResistance from most recent test

__eeprom unsigned char maxCurrent=TEST_CURRENT_MIN;  //maximum test current, in A for simplicity

volatile unsigned char tmOut=0;
volatile unsigned char tFlag=0;
volatile unsigned char tFree=0;
volatile unsigned char cdTm=0;

powerSource_t psu=BATTERY_OFF;

const char screenNames[][12]={
    "TEST BATT ",
    "CALIBRATE ",
    "INT RES.  ",
    "TIMER     ",
    "% DROP    ",
    "MAX I     ",
    "I SCALE   ",
    "VBAT CAL  ",
    "VAUX CAL  ",
    "VAUX MIN  ",
    "DEFAULTS  ",
    "???       "    
};


const settings_t backupSettings={DEFAULT_SETTINGS}; //in flash
//__eeprom settings_t savedSettings={DEFAULT_SETTINGS};
__eeprom settings_t cur={DEFAULT_SETTINGS};
screenMode_t screen=SCREEN_TEST;

unsigned int getShuntI(void){   //get shunt current in mA
    unsigned long a=0;
    unsigned char i;
    unsigned char adref=ADREF;  //store temp
    ADCON0bits.ADON=0;  //turn off
    ADREF=ADREF_FVR;        //switch reference
    ADCON0bits.ADON=1;  //turn on, ready
    __delay_us(REF_CHANGE_DELAY);
    getADC(VSHUNT); //dummy read
    for(i=0;i<I_ADC_OS;i++){
        a=a+getADC(VSHUNT);        
    }
    ADCON0bits.ADON=0;  //turn off
    ADREF=adref;    //restore reference ASAP
    ADCON0bits.ADON=1;  //turn on, ready
    return (unsigned int)((a*cur.iFullScale)/(4096UL*I_ADC_OS));
}

void getV(void){    //all the voltages, VCC, BATSENSE, AUXSENSE
    unsigned int a;
    char i;
    a=0;
    for(i=0;i<ADC_OS;i++){a=a+getVCC();}
    vcc=a/ADC_OS;
    vcc=getVCC();
    a=0;
    for(i=0;i<ADC_OS;i++){a=a+getADC(BATSENSE);}
    a=a/ADC_OS;
    a=adjust(a,vcc);    //scale to VCC;
    vbat=adjust(a,cur.vBatScale);  //scale by divider
    a=0;
    for(i=0;i<ADC_OS;i++){a=a+getADC(AUXSENSE);}
    a=a/ADC_OS;
    a=adjust(a,vcc);    //scale to VCC;
    vaux=adjust(a,cur.vAuxScale);  //scale by divider
}

scanResult_t runPulse(unsigned char dac){
    getV();
    if((vaux<cur.minVaux) && (vbat<cur.minVaux)){return TEST_VAUXBAT_LOW;}
    if(vbat>VBAT_MAX){return TEST_VOLTAGE_HIGH;}    
    DAC1DATL=dac;
    v0=vbat;
    latbits.TESTCON=1;
    delay(10);
    getV();
    v1=vbat;
    shuntI=getShuntI();
    latbits.TESTCON=0;    
    DAC1DATL=0xFF;
    if(1000*maxCurrent<shuntI){return TEST_CURRENT_HIGH;}
    if(((v0/100)*(100-cur.pcDrop))>v1){return TEST_VOLTAGE_LOW;}
    return TEST_OK;
}

unsigned char runShuntScan(void){
    unsigned char dac=0xFF;
    scanResult_t r;
    getV();
    shuntI=0;
    v0=0;
    v1=0;   //clear last
    if(vbat<BATTERY_MIN){return 0xFF;}  //fail
    while((dac>DAC_PROBE_MIN) && (shuntI<I_SHUNT_THRESH)){
        r=runPulse(dac);        
        if(r!=TEST_OK){return 0xFF;}    //fail
        dac--;
        delay(10);
    }
    if(dac<=DAC_PROBE_MIN){return 0xFF;}    //fail
    return dac;
}

void resetTimer(void){
    tmOut=cur.cdTime;
}

void testScreen(void){  //display on TEST BATT screen
    OLEDpage=2;                            
    OLEDcol=2;
    OLEDchararray("BATTERY: ",arial);
    OLEDscanshort(vbat);
    OLEDlzb(5);
    OLEDchar(OLEDbuf[5],arial);
    OLEDchar(OLEDbuf[6],arial);
    OLEDchar('.',arial);
    OLEDchar(OLEDbuf[7],arial);
    OLEDchar('V',arial);                                        
    OLEDpage=4;                            
    OLEDcol=2;
    OLEDchararray("MAX CURR:  ",arial);
    OLEDscanshort(maxCurrent);
    OLEDlzb(8);
    OLEDchar(OLEDbuf[8],arial);
    OLEDchar(OLEDbuf[9],arial);
    OLEDchar('A',arial);                                        
    OLEDpage=6;                            
    OLEDcol=2;//OLEDchararray("DN  UP  GO  SE",arial);
    OLEDchararray(ICON_DOWN_ARRAY ICON_BLANK_ARRAY ICON_UP_ARRAY ICON_BLANK_ARRAY ICON_RIGHT_ARRAY ICON_BLANK_ARRAY ICON_MODE_ARRAY,icons);
}

void waitForButtonOrT(unsigned int t){
    t=t/16;
    while((anyButton()==0) && (t>0) && (tFlag==0)){
        delay(16);
        t--;
    }
    tFlag=0;    //reset
}

void runTestAndDisplay(void){
    unsigned char dac=0xFF;
    unsigned int d=0;
    char b=0;
    scanResult_t r;
    char i,n;     //to count through tests
    unsigned int vi;
    OLEDclear(0);//refresh
    OLEDpage=0;                            
    OLEDcol=2;
    for(i=0;i<TEST_COUNT;i++){  //clear results
        tResults[i][0]=0;
        tResults[i][1]=0;
        tResults[i][2]=0;        
    }
    intRes=0;
    if(vbat<BATTERY_MIN){
        OLEDpage=1;                            
        OLEDchararray("  NO BATTERY",arial);    
        OLEDpage=5;                            
        OLEDcol=2;
        OLEDchararray("BATTERY: ",arial);
        OLEDscanshort(vbat);
        OLEDlzb(5);
        OLEDchar(OLEDbuf[5],arial);
        OLEDchar(OLEDbuf[6],arial);
        OLEDchar('.',arial);
        OLEDchar(OLEDbuf[7],arial);
        OLEDchar('V',arial);                                        
        delay(1000);
        return;
    }
    vi=(unsigned int)((((unsigned long)(vbat))*maxCurrent)/1000);   //mV=>V
    if(vi>SOA_LIMIT){ //test would exceed SOA
        OLEDpage=0;                            
        OLEDchararray("  SOA ERROR",arial);    
        OLEDpage=2;                            
        OLEDcol=2;
        OLEDscanshort(SOA_LIMIT);
        OLEDchararray("LIMIT=",arial);
        OLEDlzb(8);
        OLEDchararray(&OLEDbuf[5],arial);
        OLEDchar('W',arial);                                        
        OLEDpage=4;                            
        OLEDcol=2;
        OLEDchararray("   VI=",arial);
        OLEDscanshort(vi);
        OLEDlzb(8);
        OLEDchararray(&OLEDbuf[5],arial);
        OLEDchar('W',arial);  
        OLEDpage=6;
        OLEDcol=2;
        OLEDchararray("   TRY ",arial);
        OLEDscanshort((maxCurrent*SOA_LIMIT)/vi);
        OLEDlzb(8);
        OLEDchararray(&OLEDbuf[7],arial);
        OLEDchar('A',arial);  
        delay(1000);
        return;        
    }
    OLEDchararray("Running",arial);                        
    OLEDpage=2;                            
    OLEDcol=2;
    //estimate DAC setting
    if((cur.iOffset<DAC_PROBE_MIN)||(cur.iMax<cur.iOffset)){  //shouldn't happen, error
        OLEDchararray("SETTINGS ERROR",arial);
        delay(1000);
    }else{
        d=cur.iMax/cur.iOffset; //this is mA/DAC step
        d=(maxCurrent*1000)/d;  //DAC step offset
        if(d>cur.iOffset){d=cur.iOffset;}
        dac=cur.iOffset-(unsigned char)d;
        populateTestArray(dac);
        for(i=0;i<TEST_COUNT;i++) {
            OLEDpage=0;
            OLEDcol=80;
            //OLEDchar(i + '0', arial);
            r=runPulse(tDac[i]);  //results in v0,v1,shuntI
            tResults[i][0]=v0;
            tResults[i][1]=v1;
            tResults[i][2]=shuntI;
            delay(100);
            if(r!=TEST_OK){break;}
        }
        if(i){r=TEST_OK;}   //at least one valid result, so can be displayed
        if(i>(TEST_COUNT-1)){i=TEST_COUNT-1;}
        //cdTm=maxCurrent/2;
        cdTm=(unsigned char)(vi/20)+1;  //round up
        OLEDclear(0);//refresh
        if(r==TEST_OK){
            intRes=getResistance(tResults[i][0],tResults[i][1],tResults[i][2]); //this should not be adjusted, use raw            
            if(intRes==ERROR_R){intRes=0;}
            if(intRes>INT_RES_MAX){intRes=INT_RES_MAX;}
            if(intRes<INT_RES_MIN){intRes=INT_RES_MIN;}            
            showTestResult(i,i);  //show last            
            n=i;                
            resetTimer();
            while((b!=ENTER_BUTTON)&&(tmOut || (cur.autoPowerOff==0))){
                b=buttonPressAndHold();
                if(b){resetTimer();}
                checkForPowerOff();
                if(b==DOWN_BUTTON){
                    if(n){n--;}
                    showTestResult(n,i);
                }
                if(b==UP_BUTTON){
                    if(n<i){n++;}
                    showTestResult(n,i);
                }      
                if(b==ENTER_BUTTON){
                    while(anyButton()){}    //wait for release
                    delay(10);
                }
            }   //ENTER_BUTTON/timeout drops through
            waitForCooldown();
            resetTimer();
        }else{
            OLEDpage=2;                            
            OLEDcol=2;
            switch(r){
                case TEST_VOLTAGE_LOW : OLEDchararray("V too low",arial); break;
                case TEST_CURRENT_HIGH: OLEDchararray("I too high",arial); break;
                case TEST_VAUXBAT_LOW : OLEDchararray("AUX/BAT low",arial); break;
                case TEST_VOLTAGE_HIGH: OLEDchararray("V too high",arial); break;
                case TEST_VI_HIGH     : OLEDchararray("SOA exceeded",arial); break;
                default:                OLEDchararray("Other error",arial); break;
            }
            delay(1000);                
        }
    }
}

void populateTestArray(unsigned char dac){  //fill tDac with suitable values
    signed char i;
    char s;
    //for(i=0;i<TEST_COUNT;i++){tDac[i]=dac;} //eg trivial all the same
    //for(i=0;i<TEST_COUNT;i++){tDac[i]=dac+TEST_COUNT-i-1;} //eg nearby values
    //unsigned int d1=dac;
    //unsigned int d0=cur.iOffset;
    s=(cur.iOffset-dac)/TEST_COUNT; //what are approximate steps?
    if(s==0){s=1;}
    for(i=TEST_COUNT-1;i>=0;i--){
        tDac[i]=dac;
        dac=dac+s;
    }
}

void showTestResult(unsigned char n,unsigned char i){
    unsigned int r;
    //1st line
    OLEDpage=0;                            
    OLEDcol=2;
    OLEDchararray("OK #",arial);    
    OLEDchar(n+'0',arial);
    OLEDchar('/',arial);
    OLEDchar(i+'0',arial);
    //OLEDscanshort(tDac[n]); //raw DAC digital value
    OLEDscanshort(getVdac(tDac[n],dacref4)); //calculated voltage
    OLEDlzb(8);
    OLEDchararray(&OLEDbuf[5],arial);
    OLEDchararray("mV",arial);
    //2nd line
    OLEDpage=2;                            
    OLEDcol=2;
    OLEDscanshort(tResults[n][0]);
    OLEDlzb(5);
    OLEDchar(OLEDbuf[5],arial);
    OLEDchar(OLEDbuf[6],arial);
    OLEDchar('.',narrow);
    OLEDchar(OLEDbuf[7],arial);
    OLEDchar('V',arial);                                        
    OLEDcol=48;
    OLEDscanshort(tResults[n][1]);
    OLEDlzb(5);
    OLEDchar(OLEDbuf[5],arial);
    OLEDchar(OLEDbuf[6],arial);
    OLEDchar('.',narrow);
    OLEDchar(OLEDbuf[7],arial);
    OLEDchar('V',arial);                                        
    OLEDcol=96;
    OLEDscanshort(getPCdrop(tResults[n][0],tResults[n][1]));
    OLEDlzb(8);
    OLEDchararray(&OLEDbuf[8],arial);
    OLEDchar('%',arial);                                        
    //3rd line
    OLEDpage=4;                            
    OLEDcol=2;
    if(tResults[n][2]==0){  //no current measured
        OLEDchararray("NO CURRENT    ",arial);        
    }else{
        OLEDscanshort(tResults[n][2]);
        if(tResults[n][2]<1000){//mA
            OLEDlzb(8);
            OLEDchararray(&OLEDbuf[7],arial);        
            OLEDchar('m',arial);                                                    
        }else{      //show in Amps
            OLEDlzb(5);
            OLEDchar(OLEDbuf[5],arial);
            OLEDchar(OLEDbuf[6],arial);
            OLEDchar('.',arial);
            OLEDchar(OLEDbuf[7],arial);            
        }
        OLEDchararray("A  ",arial);  
        r=getResistance(tResults[n][0],tResults[n][1],tResults[n][2]);
        if(r==ERROR_R){
            OLEDchararray("HI R ",arial);  
        }else{
            r=adjustResistance(r);
            OLEDscanshort(r);
            if(r<1000){ //show in mOhm
                OLEDlzb(8);
                OLEDchararray(&OLEDbuf[7],arial);        
                OLEDchararray("m" FONT_STRING_OMEGA,arial);
                //OLEDchar('m',arial);                                        
                //OLEDchar(FONT_OMEGA,arial);                                                    
            }else{      //show in Ohm
                OLEDlzb(5);
                OLEDchar(OLEDbuf[5],arial);
                OLEDchar(OLEDbuf[6],arial);
                OLEDchar('.',arial);
                OLEDchar(OLEDbuf[7],arial);
                OLEDchar(FONT_OMEGA,arial);                                                    
            }        
        }
    }
    //4th line
    OLEDpage=6;                            
    OLEDcol=2;
    //OLEDchararray("DN  UP  BK    ",arial);
    OLEDchararray(ICON_DOWN_ARRAY ICON_BLANK_ARRAY ICON_UP_ARRAY ICON_BLANK_ARRAY ICON_LEFT_ARRAY ICON_BLANK_ARRAY ICON_BLANK_ARRAY,icons);

}

unsigned int getVdac(unsigned char dac, unsigned int ref){  //ie 8-bit=> 256<=>ref
    unsigned long a=((unsigned long)dac)*((unsigned long)ref);
    a=a/256;
    return (unsigned int)a;
}

unsigned char getPCdrop(unsigned int v0, unsigned int v1){  //calculate % drop from v0 to v1
    unsigned int d;
    unsigned char r;
    if(v1>v0){return 0;}    //no drop
    if(v0<BATTERY_MIN){return 0;}   //avoid /0
    d=v0-v1;
    r=(unsigned char)((((unsigned long)(d))*100)/v0);
    if(r>99){return 99;}else{return r;} //clamp for display purposes
}

unsigned int getResistance(unsigned int v0, unsigned int v1, unsigned int i){ //expects mV/mA, gives mOhm
    unsigned long r;
    if(v1>v0){return 0;}    //no drop
    r=v0-v1;
    r=1000UL*r;
    if(i<1){return ERROR_R;}  //fail   
    r=r/i;
    if(r>ERROR_R){r=ERROR_R;}
    return (unsigned int)r;
}

unsigned int adjustResistance(unsigned int r){  //remove offset
    if(r<cur.intResistance){return 0;}
    return r-cur.intResistance;
}

void checkForPowerOff(void){
    if(PRESSED(S_POWER) && (psu==BATTERY_AUX)){  //ignore if running from test battery
        waitForCooldown();
        turnAuxPowerOff();
    } //off    
}

void turnAuxPowerOff(void){
    OLEDclear(0);
    OLEDbrightness(0xFF);  
    OLEDpage=3;                            
    OLEDcol=50;
    OLEDchararray("OFF",arial);
    while(PRESSED(S_POWER)){}    //wait for release, otherwise it will switch back on
    delay(1000);
    latbits.POWERCON=0;
    delay(100);    //if there is an alt power source, program will continue to run
    RESET();        //so force reset    
}

void waitForCooldown(void){
    if(cdTm){
        OLEDclear(0);
        OLEDpage=1;                            
        OLEDcol=29;
        OLEDchararray("Cooldown",arial);
        while(cdTm){
            OLEDpage=5;                            
            OLEDcol=47;
            OLEDscanshort(cdTm);
            OLEDlzb(8);
            OLEDchar(OLEDbuf[7],arial);
            OLEDchar(OLEDbuf[8],arial);
            OLEDchar(OLEDbuf[9],arial);
            OLEDchar('s',arial);                    
            delay(100); 
        }
        OLEDclear(0);
    }
}