;==============================================================================
;	Program: alarm.asm
;==============================================================================
;	Programmer: Leon Williams
;==============================================================================
;	Date: September 2002
;==============================================================================
;	Target Processor: PIC 16F84/04P
;==============================================================================
;	Compiler: MPLAB Version 5.2
;==============================================================================
;	Release version: 1.0
;==============================================================================
;	Revisions: NIL
;==============================================================================
;	Program Description:
;
; A program to dial a telephone number using a modem when an input alarm 
; condition occurs.
;
;=========================================================================

	list	p=16f84		; processor type
	radix	hex		; HEX is default language
	errorlevel	-302	; remove warning messages about bank 1/0

;--------------------------------------------------------------------------
;	CONFIGURATION BITS

	__CONFIG 	_CP_OFF & _WDT_OFF & _PWRTE_ON & _XT_OSC

; code protection=off, watch dog timer=off,
; powerup timer =on, oscillator=low speed crystal

;-------------------------------------------------------------------------
;		****Note UPPER CASE*******	

#include 	<p16f84.inc>

;PORT A

#define	LED	PORTA,1	; LED output. 1 = OFF, 0 = ON

;PORT B

#define	RXD	PORTB,0	; Receive Data input pin. Mark = 1, Space = 0
#define	TXD	PORTB,1	; Transmit Data output pin. Mark = 1, Space = 0
#define	RESET	PORTB,2	; reset output, 0 = norm, 1 = reset active
#define	PROGRAM	PORTB,4	; program switch. 1 = norm, 0 = program mode
#define	STOP	PORTB,5	; input to stop alarm detect. 0 = no detect, 1 = detect
#define	MODE	PORTB,6	; 1 = norm alarm (N/O),  0 = invert alarm (N/C)
#define	ALARM	PORTB,7	; alarm input. 1 = NO alarm, 0 = Alarm active

;Strings
#define	PRINUM	STRINGS1,0
#define	SECNUM	STRINGS1,1
#define	USESEC	STRINGS1,2
#define	RETRY	STRINGS1,3
#define	ERROR	STRINGS1,4
#define	YES	STRINGS1,5
#define	NO	STRINGS1,6
#define	BYE	STRINGS1,7
#define	BANNER	STRINGS2,0
#define	SELECT	STRINGS2,1
#define	AUTOMODE	STRINGS2,2

;general flags 
#define	GOTCHAR	MYFLAGS,7	; 1 = got a char in RXBUFF, 0=norm
#define	NUM2DIAL	MYFLAGS,6	; 1 = dial primary number, 0 = secondary number
#define	CALLFAIL	MYFLAGS,5	; 1 = call attempts failed to get response, 0 = OK
#define	PRGMMODE	MYFLAGS,4	; 1 = in program mode, 0 = norm
#define	ALARMON	MYFLAGS,3	; 1 = Yes alarm , 0 = No alarm


;---------------------------------------------------------------------------------------------
;RAM definitions

W_TEMP	equ	0x0C	; interrupt routine temp register
ST_TEMP	equ	0x0D	; interrupt routine temp register
MYFLAGS	equ	0x0E	; holds various flags
COUNT1	equ	0x0F	; GP counter 1

COUNT2	equ	0x10	; GP counter 2
RBITCNT	equ	0x11	; bit counter used in interupt routine only
TEMP1	equ	0x12	; GP temporary register
POINTER	equ	0x13	; GP pointer
MENNUM	equ	0x14	; holds the menu item number to send
FLASHHI	equ	0x15	; holds high value of LED flash counter
FLASHLO	equ	0x16	; holds low value of LED flash counter
RINGCNT	equ	0x17	; counts ring bursts in alarm routine

TBITCNT	equ	0x18	; bit counter used in send char only
GAPSECCNT	equ	0x19	; gap between ring seconds counter
SEC45CNT	equ	0x1A	; counts 45 seconds in 1 second increments
SEC90CNT	equ	0x1B	; counts 90 seconds in 1 second increments
CALLCNT	equ	0x1C	; counter of calls
NUMofTRY	equ	0x1D	; number of tries in calling routine
RXTEMP	equ	0x1E	; temporary receive holding buffer
TXCNT	equ	0x1F	; counter used in sendchar only

RXBUFF	equ	0x20	; received char is placed here
TXBUFF	equ	0x21	; char to be transmitted is put here
DLYCNT1	equ	0x22	; counter1 used only in delay routines
DLYCNT2	equ	0x23	; counter2 used only in delay routines
DELAYN	equ	0x24	; holds value for delay routine
HALFCNT	equ	0x25	; counter used in half bit delay only
STRINGS1	equ	0x26	; holds 1st byte of flags for send string routine
STRINGS2	equ	0x27	; holds 2nd byte of flags for send string routine

;EEPROM locations

EEPRIADR	equ	d'0'	; primary number store start address in EEPROM - total 20
EESECADR	equ	d'20'	; secondary number store start address in EEPROM - total 20
EETRYADR	equ	d'40'	; number of retries address store in eeprom
EEUSEADR	equ	d'41'	; use secondary flag, bit0, 1= use secondary 0= not use sec
EEACTADR	equ	d'42'	; alarm active flag, 0x55= had alarm & not reset, 0x00=normal
EEAUTADR	equ	d'43'	; auto reset mode = 1, normal call in to reset mode = 0
; Equates

LSB	equ	0	; least sig bit
MSB	equ	7	; most sig bit
SP	equ	0x20	; space
CR	equ	0x0D	; carriage return
LF	equ	0x0A	; line return

;*******************************MAIN PROGRAM**********************************************

	org	0x000
	goto	powerup	; skip over interupt vector 
	org	0x004	; interupt service
	goto	intserv

powerup	bcf	STATUS,RP0	; ensure we are in bank 0
	call	initpic		; initialise all registers, and turn off interupts
	bsf	TXD		; set txd data pin into marking
	bsf	LED		; turn off LED to start
	bcf	GOTCHAR		; clear got a char flag
	bsf	INTCON,GIE	; enable interupts
	clrf	MYFLAGS

	goto	start

;-------------------------------START TABLES----------------------------------------------------
; Tables are placed in the first 256 bytes

PRItble	addwf	PCL,f		; add offset to program pointer
	retlw	CR
	retlw	LF
	retlw	'('
	retlw	'P'
	retlw	')'		
	retlw	'r'		
	retlw	'i'		
	retlw	'm'	
	retlw	'a'		
	retlw	'r'		
	retlw	'y'		
	retlw	SP
	retlw	'n'	
	retlw	'u'		
	retlw	'm'
	retlw	'b'		
	retlw	'e'		
	retlw	'r'		
	retlw	SP	
	retlw	'='		
	retlw	SP		
	retlw	0xFF		;table terminator

SECtble	addwf	PCL,f		; add offset to program pointer
	retlw	CR
	retlw	LF
	retlw	'('
	retlw	'S'
	retlw	')'		
	retlw	'e'		
	retlw	'c'		
	retlw	'o'	
	retlw	'n'		
	retlw	'd'
	retlw	'a'		
	retlw	'r'		
	retlw	'y'		
	retlw	SP
	retlw	'n'	
	retlw	'u'		
	retlw	'm'
	retlw	'b'		
	retlw	'e'		
	retlw	'r'		
	retlw	SP
	retlw	'='		
	retlw	SP		
	retlw	0xFF		;table terminator

USEtble	addwf	PCL,f		; add offset to program pointer
	retlw	CR
	retlw	LF
	retlw	'('
	retlw	'U'
	retlw	')'		
	retlw	's'		
	retlw	'e'		
	retlw	SP
	retlw	'S'		
	retlw	'e'		
	retlw	'c'
	retlw	'o'		
	retlw	'n'		
	retlw	'd'
	retlw	'a'		
	retlw	'r'		
	retlw	'y'		
	retlw	SP
	retlw	'='		
	retlw	SP		
	retlw	0xFF		;table terminator

RETtble	addwf	PCL,f		; add offset to program pointer
	retlw	CR
	retlw	LF
	retlw	'('
	retlw	'R'
	retlw	')'		
	retlw	'e'		
	retlw	't'		
	retlw	'r'	
	retlw	'i'		
	retlw	'e'		
	retlw	's'		
	retlw	SP		
	retlw	'='		
	retlw	SP		
	retlw	0xFF		;table terminator

ERRtble	addwf	PCL,f		; add offset to program pointer
	retlw	CR	
	retlw	LF
	retlw	0x07
	retlw	'*'	
	retlw	'E'		
	retlw	'R'	
	retlw	'R'		
	retlw	'O'		
	retlw	'R'	
	retlw	'*'		
	retlw	CR	
	retlw	LF
	retlw	0x07		; BELL
	retlw	0xFF		;table terminator

YEStble	addwf	PCL,f		; add offset to program pointer
	retlw	'Y'		
	retlw	'e'	
	retlw	's'	
	retlw	0xFF		;table terminator

NOtble	addwf	PCL,f		; add offset to program pointer
	retlw	'N'	
	retlw	'o'	
	retlw	0xFF		;table terminator

BYEtble	addwf	PCL,f		; add offset to program pointer
	retlw	CR	
	retlw	LF
	retlw	LF
	retlw	'B'
	retlw	'y'	
	retlw	'e'
	retlw	CR	
	retlw	LF
	retlw	0xFF		;table terminator

BANtble	addwf	PCL,f	; add offset to program pointer
	retlw	CR
	retlw	LF
	retlw	'V'
	retlw	SP
	retlw	'1'
	retlw	'.'		
	retlw	'0'
	retlw	CR
	retlw	LF		
	retlw	0xFF

SELtble	addwf	PCL,f		; add offset to program pointer
	retlw	'S'		
	retlw	'e'	
	retlw	'l'
	retlw	'e'		
	retlw	'c'	
	retlw	't'
	retlw	'.'
	retlw	'.'		
	retlw	0xFF		;table terminator

AUTtble	addwf	PCL,f		; add offset to program pointer
	retlw	CR
	retlw	LF
	retlw	'('
	retlw	'A'
	retlw	')'		
	retlw	'u'	
	retlw	't'
	retlw	'o'
	retlw	'm'	
	retlw	'a'
	retlw	't'		
	retlw	'i'	
	retlw	'c'
	retlw	SP
	retlw	'm'	
	retlw	'o'		
	retlw	'd'
	retlw	'e'		
	retlw	SP
	retlw	'='		
	retlw	SP		
	retlw	0xFF		;table terminator

ATAtble	addwf	PCL,f	; add offset to program pointer
	retlw	'A'
	retlw	'T'		
	retlw	'A'
	retlw	CR
	
ATDTtble	addwf	PCL,f	; add offset to program pointer
	retlw	'A'
	retlw	'T'		
	retlw	'D'
	retlw	'T'
	
;------------------------------END TABLES-------------------------------------------------------


;----------------------------START SENDSTRING---------------------------------------------------
; Sends the string pointed to by the bit set in STRINGS1 and STRINGS2
; if 0xFF found, then finished

SndStr	clrf	POINTER		; reset table offset pointer

SndSnxt	movf	POINTER,w		; load offset pointer
	btfsc	PRINUM
	call	PRItble		; get primary number char to send
	btfsc	SECNUM
	call	SECtble		; get secondary number char to send
	btfsc	USESEC
	call	USEtble		; get use sec char to send
	btfsc	RETRY
	call	RETtble		; get retry char to send
	btfsc	ERROR
	call	ERRtble		; get error char to send
	btfsc	YES
	call	YEStble		; get yes char to send
	btfsc	NO
	call	NOtble		; get no char to send
	btfsc	BYE
	call	BYEtble		; get bye char to send
	btfsc	BANNER
	call	BANtble		; get banner char to send
	btfsc	SELECT
	call	SELtble		; get select char to send
	btfsc	AUTOMODE
	call	AUTtble		; get select char to send
		
	movwf	TXBUFF		; get char 
	btfsc	TXBUFF,MSB	; do end check, bit 7 = 1 if FF found
	goto	SndSfin		; yes finished string
	call	sendchar		; not done yet, send this char
	incf	POINTER,f		; next char
	goto	SndSnxt	

SndSfin	clrf	STRINGS1		; clear table pointer
	clrf	STRINGS2
	return
	
;-------------------------------END SENDSTRING--------------------------------------------------


;----------------------------START ANSWER---------------------------------------------------------
; this routine sends the ATA sequence to the modem to make it go online and answer a call

answer	bcf	INTCON,GIE
	movlw	d'4'	; num char to send
	movwf	COUNT1
	clrf	POINTER

answloop	movf	POINTER,w	; get pointer
	call	ATAtble	; get char from table
	movwf	TXBUFF
	call	sendchar	; send the char via the TXD line
	movlw	d'1'	; 100ms delay between chars
	movwf	DELAYN
	call	Delay100
	incf	POINTER,f	; next char in table
	decfsz	COUNT1,f	; finished al?
	goto	answloop	; not yet
	bsf	INTCON,GIE
	bcf	INTCON,INTF
	return		; yes finished


;-------------------------------END ANSWER--------------------------------------------------------


;----------------------------START DIAL----------------------------------------------------------
; Sends ATDT then either the primary or secondary number depending on the state of the 
; NUM2DIAL flag.
; If it is a 1 then the primary is sent, if it is a 0 the secondary.
; A CR will be the last char stored in the eeprom during the programming process.
; The routine checks each char after sending, and if it was a CR it then exits.
; However if for some reason there is no CR, a check is made of the number of char sent. 
; If it is 20 before seeing a CR it exits. This is to avoid overunning eeprom reads.


dial	bcf	INTCON,GIE
	movlw	d'4'	; number of char to send
	movwf	COUNT1
	clrf	POINTER

dialloop	movf	POINTER,w	; get pointer
	call	ATDTtble	; get char from table
	movwf	TXBUFF
	call	sendchar	; send the char via the TXD line
	movlw	d'1'	; 100ms delay between chars
	movwf	DELAYN
	call	Delay100
	incf	POINTER,f	; next char in table
	decfsz	COUNT1,f	; finished all?
	goto	dialloop	; not yet
	goto	dialnum?

dialnum?	btfss	NUM2DIAL	; is it primary number to dial?
	goto	dialnsec	; no is secondary
	
dialnpri	movlw	EEPRIADR	; get primary number address in eeprom
	movwf	EEADR
	goto	dialler
dialnsec	movlw	EESECADR	; get secondary number address in eeprom
	movwf	EEADR

dialler	movlw	d'20'	; maxiumum number of digits allowed, if CR not found
	movwf	COUNT1

dialnxt	call	eeread	; get digit
	movf	EEDATA,w
	movwf	TXBUFF
	movwf	TEMP1	; store for later comparison of CR because TXBUFF destroyed
	call	sendchar	; send digit
	movlw	CR
	subwf	TEMP1,w
	btfsc	STATUS,Z
	goto	dialfin	; was CR all digits sent, go home

	decfsz	COUNT1,f	; done 20 yet
	goto	dialwait	; no
	goto	dialfin	; error, done 20 and no CR

dialwait	bsf	LED	; flash LED
	movlw	d'1'	; 100ms delay between chars
	movwf	DELAYN
	call	Delay100
	bcf	LED

	incf	EEADR,f	; point to next char in eeprom
	goto	dialnxt

dialfin	bsf	INTCON,GIE
	bcf	INTCON,INTF
	return		; was CR all digits sent, go home


;----------------------------END DIAL-------------------------------------------------------------



;----------------------------START START---------------------------------------------------------
; Gets here at power up.
; Firstly tests the program switch and the stop switch. 
; If the program switch is low, it does a debounce and if still low waits till it is
; high and then goes to program mode.
; If the stop switch is high (normal), ignores it and checks if alarm is still active.
; If the stop switch is low, it resets the alarm active flag.
; After these power up checks it looks for the modem by sending it AT and waiting 
; for an OK response.
; Once the modem is found, it hen goes into a loop, monitoring the alarm input, 
; the modem for a received ring and flashes the LED at about 0.5Hz.

start	clrf	STRINGS1		; clear both string pointers
	clrf	STRINGS2
	btfsc	PROGRAM		; see if program switch on (low)
	goto	startstop		; not program mode, check for stop switch
	movlw	d'5'		; is on, so do a validity check after 500mS 
	movwf	DELAYN
	call	Delay100
	btfsc	PROGRAM		; see if program switch still on 
	goto	startalm?		; no, so was invalid program switch, check if had alarm
startlp	btfss	PROGRAM		; still low after 500mS 
	goto	startlp		; wait till open
	bcf	LED		; switch now open, turn LED on in program mode
	goto	program		; go program values	

startstop btfsc	STOP		; check alarm detection at power up
	goto	startmdm		; switch high, so normal operation
	movlw	EEACTADR		; switch is low, clear alarm active flag in eeprom
	movwf	EEADR
	movlw	0x00		; 0x55 is alarm valid, 0x00 is no alarm
	movwf	EEDATA
	call	eewrite		

startmdm	call	atok		; go send AT to modem until get an OK response

startalm?	movlw	EEACTADR		; check alarm active flag in eeprom
	movwf	EEADR
	call	eeread
	movlw	0x55		; 0x55 is alarm valid, 0x00 is no alarm
	subwf	EEDATA,w
	btfsc	STATUS,Z
	goto	alarm		; alarm is still active, go do calls

;----------------------------------------------------------------------
; This is the standby loop. Scans the alarm input, the modem for a received ring
; and also flashes the LED.

scan	movlw	d'125'		; no active alarms or detection stopped
	movwf	FLASHHI
scanlo	movlw	d'200'		; 125 x 200 x loop time is approx = 500mS
	movwf	FLASHLO

scandtct	call	chkalarm		; detection allowed, go check for alarm
	btfss	ALARMON		; is there an alarm?
	goto	scanchar		; no alarm this time
	goto	alarm		; is alarm - go and make calls	

scanchar	btfsc	GOTCHAR		; no alarm, received a ring char on RXD?
	goto	scanring		; yes go see if char received is ring
	decfsz	FLASHLO,f		; no char, test to see if counters all zero
	goto	scandtct		; lo not zero yet
	decfsz	FLASHHI,f		; lo is zero check hi
	goto	scanlo		; hi ok, just lo is zero
	btfss	LED		; both lo and hi = zero, see if Led on or off
	goto	scanled
	bcf	LED		; is off so turn on
	goto	scan
scanled	bsf	LED		; is on so turn off
	goto	scan

scanring	bcf	GOTCHAR		; clear flag
	movlw	'2'		; ascii 2 is ring from modem (will be followed by CR)
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	scan		; not a 2 this time
	goto	RNA		; yes is ring this time


;-----------------------------END START-----------------------------------------------------------


;-----------------------------START CHKALARM----------------------------------------------------------
; this routine tests for alarm
; Sets flag ALARMON if active

chkalarm	bcf	ALARMON		; reset flag to start
	btfss	STOP		; detection stopped?
	return			; yes stopped detection, do nothing
	btfss	MODE		; detection allowed. normal or invert detect?
	goto	chkalinv		; is inverted detection
	btfsc	ALARM		; normal detect. is it alarm? i.e. PIC pin=0
	return			; no alarm, do nothing
	goto	chkalyes		; yes is alarm

chkalinv	btfss	ALARM		; inverted detect. is it alarm? i.e. PIC pin=1
	return			; no alarm, do nothing
			
chkalyes	bsf	ALARMON		; yes we have alarm, so set flag
	movlw	EEACTADR		; active alarm address in eeprom
	movwf	EEADR
	movlw	0x55		; 0x55 is alarm valid, 0x00 is no alarm
	movwf	EEDATA
	call	eewrite
	return

;-----------------------------END CHKALARM-----------------------------------------------------------


;-----------------------------START ATOK----------------------------------------------------------
; this routine sends AT CR to the modem and waits 2 seconds for an OK (digit 0)

atok	bcf	INTCON,GIE	; disable interupts while sending
	bcf	LED		; flash LED
	movlw	'A'		; A
	movwf	TXBUFF
	call	sendchar
	movlw	d'1'		; 100mS delay between chars
	movwf	DELAYN
	call 	Delay100	
	movlw	'T'		; T
	movwf	TXBUFF
	call	sendchar
	movlw	d'1'		; 100mS delay between chars
	movwf	DELAYN
	call 	Delay100
	movlw	CR		; CR
	movwf	TXBUFF
	call	sendchar

	bsf	INTCON,GIE	; enable interrupts again
	bcf	INTCON,INTF
	bcf	GOTCHAR		; clear flag
	bsf	LED		; end flash LED
	nop

	movlw	d'20'		; 2 second delay to wait for char
	movwf	DELAYN
	call	Delay100	

atoktst	btfsc	GOTCHAR		; did we get a char during the 2 second wait
	goto	atokchar		; yes got char, check if it is OK (0)
	goto	atok		; no, finished waiting, try again

atokchar	movlw	'0'		; got a char is it OK?
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	atok		; not a 0, try again
	
	movlw	d'3'		; yes was OK, 300mS second delay 
	movwf	DELAYN
	call	Delay100
	return			; go home


;----------------------------END ATOK-----------------------------------------------------------


;----------------------------START ALARM----------------------------------------------------------
; This routine, is accessed following detection of an alarm input.
; It dials the primary number the amount of times set in the retries register. 
; If use secondary is = Yes, then after the primary number has been called 
; the secondary number is dialled, the same amount of times.
; After each call the modem stays on line for a time set in the modem S register.
; While online it sends out a calling tone. 
;
; If automatic mode is YES, then after all the calls are made it resets the output
; and returns to scan mode. Calls are spaced 45 seconds apart, and all
; the calls as programmed in retries will be made.
;
; If automatic mode is NO, then after each call is made it waits 45 seconds for a ring char from the modem after dialling.
; If none is received within the 45 second period it redials unless all the retries 
; have been exhausted. If all calls have failed to get a response it goes into 
; a fail call mode.
; If ring is detected within a 45 second period, or while it is in failed call mode
; it waits for a total of 3 calls within 90 seconds before resetting the alarm.
; All incoming calls will be answered, until there are three within 90 seconds, 
; where it will reset the alarm and then return to scan mode. 
; Note must get 2 ring bursts to be valid call.


alarm	bcf	LED
	clrf	CALLCNT		; clear call counter
	bsf	NUM2DIAL		; select primary number to be first dialled
	bcf	CALLFAIL		; clear calls failed flag

alarmtry	movlw	EETRYADR		; get retry numbers from eeprom
	movwf	EEADR
	call	eeread
	movlw	0x30		; subtract 0x30 from ascii number to leave decimal
	subwf	EEDATA,w
	movwf	NUMofTRY		; store retries here for testing

almauto	movlw	EEAUTADR		; check if in auto mode
	movwf	EEADR
	call	eeread
	btfsc	EEDATA,LSB	; bit 0, 1 = yes, 0= no
	goto	autodial		; yes in auto mode, just make calls then reset

almdial	call	dial		; dial number
	movlw	d'45'		; wait max of 45 secs between calls for ring
	movwf	SEC45CNT
	clrf	RINGCNT		; clear ring burst counter ready for next call

almwait	btfsc	GOTCHAR		; test got a char from modem flag
	goto	almring		; got a char, go and  test for ring
	movlw	d'10'		; no char, 1 second delay
	movwf	DELAYN	
	call	Delay100
	decfsz	SEC45CNT,f	; done 45 seconds waiting?
	goto	almwait		; not yet go around again
		
almreturn	decfsz	NUMofTRY,f	; done all retries?
	goto	almdial		; not done all yet, so do next dial sequence
	btfss	NUM2DIAL		; done all the retries, which number dialling?
	goto	almfail		; was secondary, so failed both numbers

	movlw	EEUSEADR		; was primary, check if to use sec num
	movwf	EEADR
	call	eeread
	btfss	EEDATA,LSB	; bit 0, 1 = yes, 0= no
	goto	almfail		; only using primary number, and all failed	
	bcf	NUM2DIAL		; yes use sec number, clear flag to dial sec number
	goto	alarmtry		; and go around again to dial secondary number

almring	bcf	GOTCHAR		; got char within 45 sec after call, clear flag
	movlw	'2'		; see if it was ring			
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	almwait		; got a char but not ring, keep looping around
	incf	RINGCNT,f		; increment ring count
	btfss	RINGCNT,1		; test bit 1, will be set if count = 2
	goto	almwait		; not 2 rings yet	
	goto	alm3		; yes got 2 ring bursts, go check for 3 in 90 seconds	
	
	
	

;--------------------------------------------
; Will get here if all outgoing call attempts have failed to get a response
; Waits indefinitely for 2 ring bursts.
; Then goes and an checks for total of 3 incoming calls. 


almfail	bsf	CALLFAIL		; set failed flag
	bcf	GOTCHAR		; clear flag to be safe
	clrf	RINGCNT		; clear ring burst counter
almfwait	btfss	GOTCHAR		; test got a char from modem 
	goto	almfwait		; no char yet
	bcf	GOTCHAR		; clear flag
	movlw	'2'		; see if it was ring			
	subwf	RXBUFF,w
	btfss	STATUS,Z		; test for ring
	goto	almfwait		; wait again
	incf	RINGCNT,f		; increment ring count
	btfss	RINGCNT,1		; test bit 1, will be set if count = 2
	goto	almfwait		; not 2 rings yet	

;-------------------------------------------
; Gets here in two ways:
; 1. after a ring has been received during a 45 second wait period
; following a modem call out.
; 2. from failed call mode after ring has been found while waiting indefinitely. 
; Waits 90 seconds maximum for 3 seperate calls. During this period all
; incoming calls will be answered. 
; If get 3 calls in 90 seconds the alarm will be reset, 
; the alarm active flag cleared and will revert to scan mode.
;
; If not got 3 calls in 90 seconds then if still got calls to dial
; returns and does next one.
; If in failed call mode, goes back and waits indefinitely for a ring again.
; Due to some modems spuriously producing ring when going offline after 
; timing out, 2 ring bursts must be received in the wait for 3 calls routine
; before a call will be registered.


alm3	movlw	d'90'		; now see if get 3 rings within 90 seconds
	movwf	SEC90CNT		; counts 90 seconds timeout 

alm3answ	movlw	d'5'		; half second delay
	movwf	DELAYN	
	call	Delay100
	call	answer		; answer incoming ring
	clrf	RINGCNT		; clear ring burst counter ready for next call

	incf	CALLCNT,f		; increment incoming call counter
	movlw	d'3'		; test for 3 calls occured
	subwf	CALLCNT,w		; CALLCNT - w(3)
	btfsc	STATUS,Z		; Z set if CALLCNT = 3 
	goto	alm3yes		; yep got 3 calls, no more waiting

alm3wait	btfss	GOTCHAR		; not 3 calls yet, test got a char from modem 
	goto	alm3fin?		; no char, so test if already waited 90 seconds
	bcf	GOTCHAR		; got a char, clear flag
	movlw	'2'		; see if it was ring			
	subwf	RXBUFF,w
	btfss	STATUS,Z		; test for ring
	goto	alm3fin?		; not ring, check for end of 90 second timeout	
	incf	RINGCNT,f		; increment ring count
	btfss	RINGCNT,1		; test bit 1, will be set if count = 2
	goto	alm3fin?		; not 2 rings yet	
	goto	alm3answ		; yes got 2 ring bursts, answer call

alm3fin?	movlw	d'10'		; no ring, 1 second delay
	movwf	DELAYN	
	call	Delay100
	decfsz	SEC90CNT,f	; have we waited 90 seconds yet?
	goto	alm3wait		; not yet, go around again looking for ring
	clrf	CALLCNT		; yes waited 90 sec, clear call counter
	btfss	CALLFAIL		; not got 3 calls yet, are we in failed mode?
	goto	almreturn		; no, still calls to do, go back and dial
	goto	almfail		; go back wait for another ring
				
alm3yes	movlw	EEACTADR		; got 3 calls response, clear alarm active flag in eeprom
	movwf	EEADR
	movlw	0x00		; 0x55 is alarm valid, 0x00 is no alarm
	movwf	EEDATA
	call	eewrite
	
	bsf	RESET		; reset output for 1 second
	movlw	d'10'		; 1 second delay
	movwf	DELAYN	
	call	Delay100
	bcf	RESET

	goto	scan		; finshed go back to scan mode
;-------------------------------------------
; gets here after an alarm and automatic mode is set to YES
; dials number sna dthen resets, and wiats for alarm to clear before
; returning to scan mode

autodial	call	dial		; dial number
	movlw	d'250'		; 45 second delay
	movwf	DELAYN		; 25 + 20
	call	Delay100
	movlw	d'200'		
	movwf	DELAYN	
	call	Delay100
		
	decfsz	NUMofTRY,f	; done all retries?
	goto	autodial		; not yet, so do next dial sequence
	btfss	NUM2DIAL		; done all the retries, which number dialling?
	goto	autreset		; was secondary, all done now

	movlw	EEUSEADR		; was primary, check if to use sec num
	movwf	EEADR
	call	eeread
	btfss	EEDATA,LSB	; 1 = yes use sec, 0= no only prim
	goto	autreset		; only using primary number, all done	
	bcf	NUM2DIAL		; yes use sec number, clear flag to dial sec number
	goto	alarmtry		; load retries again to dial secondary number

autreset	movlw	EEACTADR		; clear alarm active flag 
	movwf	EEADR
	movlw	0x00		; 0x55 is alarm valid, 0x00 is no alarm
	movwf	EEDATA
	call	eewrite
	
	bsf	RESET		; reset output for 1 second
	movlw	d'10'		; 1 second delay
	movwf	DELAYN	
	call	Delay100
	bcf	RESET	

autwait	call	chkalarm		; go check for alarm input state
	btfsc	ALARMON		; is there still an alarm?
	goto	autwait		; yes, so wait till gone
	movlw	d'50'		; no alarm now, 5 second delay
	movwf	DELAYN		; to let things settle down
	call	Delay100		; before resuming scan mode
		
	goto	scan		; finshed go back to scan mode

;----------------------------END ALARM---------------------------------------------------------



;----------------------------START RING NO ALARM --------------------------------------------------------
; Gets here after getting an initial burst of ring while there is no alarm present.
; Counts bursts of ring during a 90 second maximum period, and determines the number
; of individual calls received.
; Main loop has a delay of 1 second checking for ring char each time.
; Each second the gap second counter is incremented and the timeout 90 second counter
; is decremented. The call count is continually checked after each ring, even if 
; the 90 second timer has not elapsed.
; The time between ring bursts in the same call is around 3 seconds. 
; If there is a gap larger than 6 seconds between rings, then a new call must 
; have occured and the call counter is incremented.
; If 3 separate calls are detected within 90 seconds then the last call is answered.
; This is used by a remote user to verify that the system is turned on and operating 
; correctly. The measurement accuracy of ring gaps is +- 1 second

RNA	bcf	LED		; turn on LED
	clrf	GAPSECCNT		; clear reset gap between ring seconds counter
	movlw	d'1'		; already had a call so make call counter = one
	movwf	CALLCNT		; call counter
	movlw	d'90'
	movwf	SEC90CNT		; counts 90 seconds timeout
	
RNAdel	movlw	d'10'		; 1 second delay
	movwf	DELAYN
	call	Delay100
	incf	GAPSECCNT,f	; one second done, increment the gap second counter
	btfss	GOTCHAR		; have we received a char from the modem?
	goto	RNAfin?		; no char, check if timed out
	bcf	GOTCHAR		; got a char, reset flag
	movlw	'2'		; test for ring (ascii 2)
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	RNAfin?		; no ring this time, check if timed out 
	
	movlw	d'6'		; yes is ring, test for > 6S between last and this ring
	subwf	GAPSECCNT,w	; GAPSECCNT - w(6)
	btfss	STATUS,C		; C clear if GAPSECCNT < 6, C set if GAPSECCNT = 5 or more
	goto	RNArstgp		; not > 6 seconds, so is ring from same call	
	incf	CALLCNT,f		; was > 6 seconds, is new call, incr call count		
RNArstgp	clrf	GAPSECCNT		; clear gap counter, wait for next ring burst
RNAcalls	movlw	d'3'		; test for 3 calls occured
	subwf	CALLCNT,w		; CALLCNT - w(3)
	btfsc	STATUS,Z		; C set if CALLCNT = 3 
	goto	RNAans		; got 3 calls - go answer call
		
RNAfin?	call	chkalarm		; check for alarm while waiting for ring
	btfsc	ALARMON		; is there an alarm?
	goto	alarm		; yes alarm happened while waiting - go call!!
	decfsz	SEC90CNT,f	; no alarm, not got 3 calls, finish 90 seconds?
	goto	RNAdel		; not done 90 seconds, loop again
	goto	scan		; waited for 90 seconds but did not get 3 calls, ignore rings

RNAans	call	answer		; got 3 calls within 90 seconds, answer call to show all OK.
	movlw	d'20'		; wait 2 seconds 
	movwf	DELAYN		; before going back to scan mode
	call	Delay100
	clrf	RXTEMP		; clear any left over received bytes
	clrf	RXBUFF
	goto	scan		; finished
	
;----------------------------END RING NO ALARM---------------------------------------------------------


;--------------------------INTERUPT START-------------------------------------------
; This interrupt service routine is used to receive characters on the RXD line
; and load the char into RXBUFF, and then set the GOTCHAR flag.
; Gets here on RB0 interrupt only. This is a negative edge trigger interrupt
; which occurs at the falling edge of a start bit.
; Data must be 2400bps, 8 data bits, no parity and 1 stop bit. LSB first bit received.
; RS232 Mark = -ve, Space = +ve, TTL out of MAX232 Mark = 1, Space = 0
; Even though interrupts are disabled through clearing GIE, the INTF flag is still 
; set by other negative transitions of the data as the bits arrive on the RB0 pin. 
; Therefore to stop interrupt re-entry when finished loading all the bits into RXBUFF, 
; INTF is reset again before exiting the interrupt service routine.
; The next interrupt will be the negative edge of the start bit of the next byte.
; As a further precaution the routine wits untill halfway through the stop bit
; before exiting, to avoid possible interrupts due to noisy falling edges.
;
; If we are in program mode, all chars are allowed and each is loaded into RXBUFF
; as they arrive and the flag GOTCHAR is set for every one.
;
; If we are not in program mode, then a filter is applied that only allows the GOTCHAR
; flag to be set for chars that signal modem activity. The modem returns 0 CR for OK,
; 2 CR for ring etc. 
; Each char is loaded into a temp register, and after a CR is received the previous
; char is then put back into the RXBUFF register, and the GOTCHAR flag is set.
; When in this mode, the GOTCHAR flag will not be set until a CR has been received.
;
; The timimg of the sampling for the receive UART function relies on the halfbit routine.
; While this routine provides a precise half bit timing delay at 2400bps, there will be 
; a gradual shift away from sampling at the centre of the bit as each bit is received 
; due to the extra time taken in the process of sampling, writing and rotating etc. 
; This amounts to about 30 uS by the time bit 8 is received. While this is not ideal
; no attempts to correct this have been employed, because each bit is 416uS wide, and
; a deviation of 30uS from the centre should not cause any sampling problems. 
; Also, the timimg is restarted for each character, and the error is not cumulative
; for long strings.


intserv	movwf	W_TEMP		; save w reg
	movf	STATUS,w		; move status reg into w reg
	bcf	STATUS,RP0	; ensure file register set to bank 0
	movwf	ST_TEMP		; save status reg
	bcf	INTCON,GIE	; disable global interupts while getting data bits
	bcf	INTCON,INTF	; clear RB0 interupt flag

	movlw	d'8'		; 8 data bits counter
	movwf	RBITCNT		
	call	halfbit		; wait till start bit finished		
	call	halfbit

intnxt	call	halfbit		; now wait till in middle of data bit
	btfsc	RXD		; test data bit, 
	goto	intis1		; bit is 1		
intis0	bcf	RXBUFF,MSB	; bit is 0 
	goto	intend?		; go test if done 8 bits	
intis1	bsf	RXBUFF,MSB

intend?	call	halfbit		; wait another half bit till data bit finished
	rrf	RXBUFF,f		; point to next bit position
	decfsz	RBITCNT,f		; see if done 8 bits
	goto	intnxt		; not done 8 bits yet
	rlf	RXBUFF,f		; done, get bits in correct place, correct for extra shift
	
	call	halfbit		; now wait till halfway into stop bit

intmode?	btfsc	PRGMMODE		; see if we are in program mode (program =1)
	goto	intok		; yes in program mode allow all chars

intchk	movlw 	CR		; not in program mode, see if a CR
	subwf	RXBUFF,w
	btfsc	STATUS,Z
	goto	intisCR		; yes is a CR, get previous char from store
	movf	RXBUFF,w		; not a CR, store received char in store
	movwf	RXTEMP
	bcf	GOTCHAR		; ensure flag is clear
	goto	intfin		; done
	
intisCR	movf	RXTEMP,w		; was a CR so get stored char
	movwf	RXBUFF
	goto	intok		; and go set flag		

intok	bsf	GOTCHAR		; set got a char flag for main program
	
intfin	movf	ST_TEMP,w		; get status
	movwf 	STATUS		; restore status
	swapf	W_TEMP,f	
	swapf	W_TEMP,w		; restore w reg

	bcf	INTCON,INTF	; clear RB0 interupt flag to ensure no unwanted re-entry

	retfie			; return to main program, also enables global interrupts

;--------------------------INTERUPT END--------------------------------------------


;----------------------------START PROGRAM---------------------------------------------------
; program values

program	bsf	PRGMMODE		; set program mode flag for interrupt routine
	call	sendmenu		; send menu with current values to PC

progwait	btfss	PROGRAM		; see if finished, switch will be low to terminate
	goto	progfin		; yes switch now low, wait for it to be released	
	btfss	GOTCHAR		; flag = 1 if received a char
	goto	progwait		; not got a char yet
	
	bcf	GOTCHAR		; got a char, clear flag
	call	echo		; return char to PC from RXBUFF
	movlw	CR		; get to new line on PC screen
	movwf	TXBUFF
	call	sendchar
	movlw	LF
	movwf	TXBUFF
	call	sendchar
;------------------------------

progtst1	movlw	'A'
	subwf	RXBUFF,w		; get received char and test with A
	btfss	STATUS,Z		; z flag set if same
	goto	progtst2		; not a 1
	bsf	AUTOMODE
	call	SndStr

progalp	btfss	GOTCHAR		; flag = 1 if got a char
	goto	progalp		; not yet
	bcf	GOTCHAR		; clear flag
	call	echo		; return char to PC

	movlw	'Y'		; test for an upper case Y
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	progan		; no Y so test for N
progayok	movlw	0x01
	movwf	EEDATA		; is a Y, set flag (bit 0)
	movlw	EEAUTADR		; and write to eeprom location
	movwf	EEADR
	call	eewrite
	goto	progacr		

progan	movlw	'N'		; test for an upper case N
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	progerr		; no Y or N, error
proganok	clrf	EEDATA		; is N, clear flag 
	movlw	EEAUTADR		; and write to eeprom location
	movwf	EEADR
	call	eewrite

progacr	btfss	GOTCHAR		; wait for CR, flag = 1 if got a char
	goto	progacr		; not yet
	bcf	GOTCHAR		; clear flag
	movlw	CR
	subwf	RXBUFF,w		; is it CR
	btfss	STATUS,Z		; CR if Z = 1
	goto	progacr		; no

	goto	program		; done automatic flag, go around again

;-----------------------------------------
	
progtst2	movlw	'P'
	subwf	RXBUFF,w		; get received char and test with P
	btfss	STATUS,Z		; z flag set if same
	goto	progtst3		; not 2
	bsf	PRINUM		; yes is 2, send pri num string
	call	SndStr

	movlw	d'20'		; counter of chars entered, max allowed 20
	movwf	COUNT1
	movlw	EEPRIADR		; pri num start address in eeprom
	movwf	EEADR		; put into adress pointer used by eewrite		
progplp	btfss	GOTCHAR		; flag = 1 if got char
	goto	progplp		; not found start bit yet
	bcf	GOTCHAR		; clear flag
	call	echo		; return char to PC

	movlw	0x30		; test received char for less than 0
	subwf	RXBUFF,w		; RXBUFF - w(0x30)
	btfss	STATUS,C		; C clear if RXBUFF < 0x30
	goto	progpcr?		; is less than 0x30 - error, check if CR

	movlw	0x3A		; test for larger than 9
	subwf	RXBUFF,w		; RXBUFF - w(0x3A)
	btfsc	STATUS,C		; C set if RXBUFF = > 0x3A
	goto	progperr		; is larger than 0x39 - error
	goto	progpok		; is between 0 and 9, write char 

progpcr?	movlw	CR		; not a number, test for carrige return 
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	progperr		; not  number and not a CR - error		 
	movf	RXBUFF,w		; yes got CR so refresh screen and wait
	movwf	EEDATA		; and write in eeprom location
	call	eewrite
	goto	program		; was a CR so end write and re-display

progpok	movf	RXBUFF,w		; it is a number between 0 and 9, get received char
	movwf	EEDATA		; and write in eeprom location
	call	eewrite

	incf	EEADR,f		; not CR so go around again
	decfsz	COUNT1,f		; done 20 chars?
	goto	progplp		; not yet

progperr	movlw	d'20'		; ERROR too many chars or illegal char 
	movwf	COUNT1		; clear out eeprom locations
	movlw	EEPRIADR		; pri num start address in eeprom
	movwf	EEADR		; put into address pointer used by eewrite
progpelp	movlw	SP		; write a space
	movwf	EEDATA
	call	eewrite
	incf	EEADR,f		; next eeprom location
	decfsz	COUNT1,f		; cleared 20 locations?
	goto	progpelp		; not yet
	decf	EEADR,f		; go back and write a CR in the last location
	movlw	CR		; as a string termination char	
	movwf	EEDATA
	call	eewrite	
	goto	progerr		; yes done 20, now send error message
		
;--------------------------------
progtst3	movlw	'S'
	subwf	RXBUFF,w		; get received char and test with S
	btfss	STATUS,Z		; z flag set if same
	goto	progtst4
	bsf	SECNUM
	call	SndStr

	movlw	d'20'		; counter of chars entered, max allowed 20
	movwf	COUNT1
	movlw	EESECADR		; sec num start address in eeprom
	movwf	EEADR		; put into adress pointer used by eewrite		
progslp	btfss	GOTCHAR		; flag = 1 if got a char
	goto	progslp		; not yet
	bcf	GOTCHAR		; clear flag
	call	echo		; return char to PC

	movlw	0x30		; test received char for less than 0
	subwf	RXBUFF,w		; RXBUFF - w(0x30)
	btfss	STATUS,C		; C clear if RXBUFF < 0x30
	goto	progscr?		; is less than 0x30 - error, check if CR

	movlw	0x3A		; test for larger than 9
	subwf	RXBUFF,w		; RXBUFF - w(0x3A)
	btfsc	STATUS,C		; C set if RXBUFF > 0x39
	goto	progserr		; is larger than 0x39 - error
	goto	progsok		; is between 0 and 9, write char 

progscr?	movlw	CR		; not a number, test for carrige return 
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	progserr		; not  number and not a CR - error		 
	movf	RXBUFF,w		; yes got CR so refresh screen and wait
	movwf	EEDATA		; and write in eeprom location
	call	eewrite
	goto	program		; was a CR so end write and re-display

progsok	movf	RXBUFF,w		; it is a number between 0 and 9, get received char
	movwf	EEDATA		; and write in eeprom location
	call	eewrite

	incf	EEADR,f		; not CR so go around again
	decfsz	COUNT1,f		; done 20 chars?
	goto	progslp		; not yet
	
progserr	movlw	d'20'		; ERROR condition too many chars, clear out eeprom locations
	movwf	COUNT1
	movlw	EESECADR		; sec num start address in eeprom
	movwf	EEADR		; put into adress pointer used by eewrite
progselp	movlw	SP		; write a space
	movwf	EEDATA
	call	eewrite
	incf	EEADR,f		; next eeprom location
	decfsz	COUNT1,f		; cleared 20 locations?
	goto	progselp		; not yet
	decf	EEADR,f		; go back and write a CR in the last location
	movlw	CR	
	movwf	EEDATA
	call	eewrite	
	goto	progerr		; yes, now send error message

	
;--------------------------------
progtst4	movlw	'U'
	subwf	RXBUFF,w		; get received char and test with U
	btfss	STATUS,Z		; z flag set if same
	goto	progtst5
	bsf	USESEC
	call	SndStr

progulp	btfss	GOTCHAR		; flag = 1 if got a char
	goto	progulp		; not yet
	bcf	GOTCHAR		; clear flag
	call	echo		; return char to PC

	movlw	'Y'		; test for an upper case Y
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	progun		; no Y so test for N
proguyok	movlw	0x01
	movwf	EEDATA		; is a Y, set flag (bit 0)
	movlw	EEUSEADR		; and write to eeprom location
	movwf	EEADR
	call	eewrite
	goto	progucr		

progun	movlw	'N'		; test for an upper case N
	subwf	RXBUFF,w
	btfss	STATUS,Z
	goto	progerr		; no Y or N, error
progunok	clrf	EEDATA		; is n, clear flag 
	movlw	EEUSEADR		; and write to eeprom location
	movwf	EEADR
	call	eewrite

progucr	btfss	GOTCHAR		; wait for CR, flag = 1 if got a char
	goto	progucr		; not yet
	bcf	GOTCHAR		; clear flag
	movlw	CR
	subwf	RXBUFF,w		; is it CR
	btfss	STATUS,Z		; CR if Z = 1
	goto	progucr		; no

	goto	program		; done use secondary flag, go around again

;----------------------------------
progtst5	movlw	'R'
	subwf	RXBUFF,w		; get received char and test with R
	btfss	STATUS,Z		; z flag set if same
	goto	progerr
	bsf	RETRY
	call	SndStr
	
progrlp	btfss	GOTCHAR		; flag = 1 if got a char
	goto	progrlp		; not yet
	bcf	GOTCHAR		; clear flag
	call	echo		; return char to PC

	movlw	0x31		; test received char for less than 1
	subwf	RXBUFF,w		; RXBUFF - w(0x31)
	btfss	STATUS,C		; C clear if RXBUFF < 0x31
	goto	progerr		; is less than 0x31 - error

	movlw	0x3A		; test for larger than 9
	subwf	RXBUFF,w		; RXBUFF - w(0x3A)
	btfsc	STATUS,C		; C set if RXBUFF = > 0x3A
	goto	progerr		; is larger than 0x39 - error

	movf	RXBUFF,w
	movwf	EEDATA		; store number
	movlw	EETRYADR		; write to eeprom location
	movwf	EEADR
	call	eewrite

progrcr	btfss	GOTCHAR		; wait for CR, flag = 1 if got a char
	goto	progrcr		; not yet
	bcf	GOTCHAR		; clear flag
	movlw	CR
	subwf	RXBUFF,w		; is it CR
	btfss	STATUS,Z		; CR if Z = 1
	goto	progrcr		; no

	goto	program		; done retries, go around again


;-----------------------------------
progerr	bsf	ERROR		; send error message
	call	SndStr
	movlw	d'5'		; 500mS delay
	movwf	DELAYN
	call	Delay100
	goto	program


progfin	movlw	d'5'		; switch closed, 500mS debounce delay
	movwf	DELAYN
	call	Delay100
progfinw	btfss	PROGRAM		; now wait for switch to be opened, stops hangups
	goto	progfinw		; no is still low
	bsf	BYE		; switch open again, send finished message
	call	SndStr

	movlw	EEACTADR		; reset alarm active flag in eeprom
	movwf	EEADR
	movlw	0x00		; 0x55 is alarm valid, 0x00 is no alarm
	movwf	EEDATA
	call	eewrite

	movlw	d'10'		; 1 second delay
	movwf	DELAYN
	call	Delay100
	
	bcf	PRGMMODE		; clear in program mode flag
	bsf	LED		; turn off LED 
	goto	startmdm		; and exit program mode

;----------------------------END PROGRAM----------------------------------------------------


;----------------------------START SEND MENU-----------------------------------------------
; this routine sends the menu with the current programmed values

sendmenu	bcf	INTCON,GIE	; turn off receive interrupt while sending
	movlw	CR		; get to new line on PC screen
	movwf	TXBUFF
	call	sendchar
	movlw	LF
	movwf	TXBUFF
	call	sendchar
	bsf	BANNER		; send opening banner
	call	SndStr

sendma	bsf	AUTOMODE
	call	SndStr
	
	movlw	EEAUTADR		; get flag to see if automatic reset mode
	movwf	EEADR
	call	eeread
	btfss	EEDATA,LSB	; bit 0: 1 = yes, 0= no
	goto	sendman
	bsf	YES
	call	SndStr
	goto	sendmp
sendman	bsf	NO
	call	SndStr

sendmp	bsf	PRINUM
	call	SndStr

	movlw	d'20'
	movwf	COUNT1		; counts number of eeprom chars read
	movlw	EEPRIADR		; first address in eeprom
	movwf	COUNT2
	
sendmplp	movf	COUNT2,w
	movwf	EEADR
	call	eeread
	movf	EEDATA,w
	movwf	TXBUFF		; store in transmit buffer
	movlw	CR		; check for end of numbers with CR at end
	subwf	TXBUFF,w
	btfsc	STATUS,Z		; z flag set if is CR
	goto	sendms		; yes is CR, so go send secondary number
	call	sendchar		; not end yet
	incf	COUNT2,f		; next eeprom address
	decfsz	COUNT1,f		; done all 20? Check for no CR in number storage range
	goto	sendmplp		; no CR yet and not done 20 numbers, go again

sendms	bsf	SECNUM
	call	SndStr

	movlw	d'20'
	movwf	COUNT1		; counts number of eeprom chars read
	movlw	EESECADR		; first address in eeprom
	movwf	COUNT2
	
sendmslp	movf	COUNT2,w
	movwf	EEADR
	call	eeread
	movf	EEDATA,w
	movwf	TXBUFF		; store in transmit buffer
	movlw	CR		; check for end of numbers with CR at end
	subwf	TXBUFF,w
	btfsc	STATUS,Z		; z flag set if is CR
	goto	sendmu		; yes is CR, so go send use sec
	call	sendchar		; not end yet
	incf	COUNT2,f		; next eeprom address
	decfsz	COUNT1,f		; done all 20? Check for no CR in number storage range
	goto	sendmslp		; no CR yet and not done 20 numbers, go again
	
sendmu	bsf	USESEC
	call	SndStr
	
	movlw	EEUSEADR		; get flag to see if using sec num
	movwf	EEADR
	call	eeread
	btfss	EEDATA,LSB	; bit 0: 1 = yes, 0= no
	goto	sendmun
	bsf	YES
	call	SndStr
	goto	sendmr
sendmun	bsf	NO
	call	SndStr

sendmr	bsf	RETRY
	call	SndStr

	movlw	EETRYADR		; get retries
	movwf	EEADR
	call	eeread
	movf	EEDATA,w
	movwf	TXBUFF		; store in transmit buffer
	call	sendchar
	
	movlw	CR		; get to new line on PC screen
	movwf	TXBUFF
	call	sendchar
	movlw	LF
	movwf	TXBUFF
	call	sendchar
	bsf	SELECT		; send select-
	call	SndStr

	bcf	INTCON,INTF	; clear RB0 interupt flag to ensure no unwanted re-entry
	bsf	INTCON,GIE	; re-enable interrupts

	return

;----------------------------END SEND MENU---------------------------------------------



;----------------------------START ECHO-----------------------------------------------
; this routine echos back the received char that is in RXBUFF from last char received

echo	movf	RXBUFF,w
	movwf	TXBUFF
	call	sendchar
	return

;----------------------------END ECHO---------------------------------------------


;--------------------------SENDCHAR----------------------------------------------
; this routine sends the char in TXBUFF via the TXD pin.
; 1 start bit, 8 data bits and 1 stop bit at 2400bps. LSB first
; TTL into MAX232 Mark = 1, Space = 0       RS232 Mark = -ve, Space = +ve, 
; start and stop bit timing is simply done in an external delay routine, however
; the dtat bit delay is done in a loop in this routine, that takes into consideration
; the delay in looping and testing bits etc. Using this closely defined loop timing
; means that the bit timing is very accurate.

sendchar	bcf	TXD		; start bit = space
	movlw	d'8'		; 8 data bits counter
	movwf	TBITCNT	

sendchst	call	halfbit		; 2 x halfbit =  full start bit length
	call	halfbit
	
sndchbit	btfss	TXBUFF,LSB	; now send 8 data bits
	goto	sendch0
sendch1	bsf	TXD		; bit is a 1
	goto	sendchdl
sendch0	bcf	TXD		; bit is a 0
	nop			; balance up 0 and 1 cycles

sendchdl	movlw	d'135'		; 1 bit length
	movwf	TXCNT		; bit delay total = 11(overhead)+(135x3[loop]) = 416uS
sendchwt	decfsz	TXCNT,f		; done delay
	goto	sendchwt		; not yet

	rrf	TXBUFF,f		; finished delay, point to next bit
	
	decfsz	TBITCNT,f		; see if sent 8 bits
	goto	sndchbit		; not yet

	bsf	TXD		; sent all dat bits, do a stop bit (mark)
	call	halfbit		; 2 x halfbit = time for stop bit
	call	halfbit
	
	return

;-------------------------END SENDCHAR--------------------------------------------


;--------------------------START HALFBIT---------------------------------------------
; this routine is a delay of one half a bit period at 2400 bps.
; At 2400 the bit time is 1/2400 = 416uS, half a bit becomes 208uS
; the routine takes into account the time taken to call this routine and return time.
; total delay = 2 call + 2 return + 4 load + (66x3)loops + 2 last loop = 208uS

halfbit	movlw	d'67'
	movwf	HALFCNT
	nop
	nop
	
halfbwt	decfsz	HALFCNT,f
	goto	halfbwt

	return
	
;---------------------------END HALFBIT-------------------------------------------	


;------------------------------START EEWRITE---------------------------------------
; Writes single byte to EEPROM.
; data must be in EEDATA, address of EEPROM must be in EEADR

eewrite	bsf	STATUS,RP0	; bank 1
	clrf	EECON1		; reset all bits to start
	bsf	EECON1,WREN	; enable writes to eeprom
	movlw	0x55		; protection code to prevent false writes
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf	EECON1,WR		; write data

eewrwait	btfss	EECON1,EEIF	; flag = 1 when done
	goto	eewrwait

	bcf	STATUS,RP0	; back to bank 0
	return

;-------------------------------END EEWRITE-----------------------------------------


;------------------------------START EEREAD---------------------------------------
; Reads a single byte from EEPROM.
; data goes into EEDATA. Address of EEPROM must be in EEADR first

eeread	bsf	STATUS,RP0	; bank 1
	clrf	EECON1		; clear all bits first
	bsf	EECON1,RD		; read from eeprom@eeadr into eedata
	bcf	STATUS,RP0	; back to bank 0
	
	return

;-------------------------------END EEREAD-----------------------------------------


;==========================START DELAY100===============================================
; Total delay = 100mS x DELAYN
; DELAYN must be loaded before calling this routine
;------------------------------------------------------------------------------------
Delay100	movlw	d'200'	; 200 x 500uS = 100mS
	movwf	DLYCNT1
Delay100A	movlw	d'165'	; 500uS loop
	movwf	DLYCNT2

Delay100B	decfsz	DLYCNT2,f	
	goto	Delay100B
	decfsz	DLYCNT1,f	; done loops1?
	goto	Delay100A	; not yet
	decfsz	DELAYN,f	; have we done all the 100mS loops yet
	goto	Delay100	; Not yet	
	return		; finished

;---------------------------END DELAY100-----------------------------------------------


;-----------------------------START INIT-----------------------------------------
; initialisation routine, setup defaults for PIC registers
;
;-------------------------------------------------------------------------------------

initpic	bcf	INTCON,GIE	;disable global interupts
	bcf	STATUS,RP0	;go to register bank 0
	clrf	STATUS
	movlw	b'00010000'	; enable only RB0 interupt
	movwf	INTCON		; but leave GIE clear here to start

	clrf	PORTA		; clear port latches a,b 
	clrf	PORTB

	bsf	STATUS,RP0	; go to register bank 1
	movlw	b'10000000'	; portb pullups disabled, falling edge RB0 int 
	movwf	OPTION_REG	; Tmr0 int clk,  prescaler 2

	movlw	b'00000000'	; 
	movwf	TRISA		; set port A
	movlw	b'11110001'	; 
	movwf	TRISB		; set port B 

	bcf	STATUS,RP0	; go back to register bank 0

	return			; finished

;-------------------------END INIT------------------------------------------

	end			; no more

