; File INSTRUMENT.ASM
; Uses assembly code for PIC16F84 microcontroller
;
; Processor pin allocations are as follows:
; RA2 Output disp3 driving Dp LED for alarm
; RA3 Output PWM signal for A/D converter
; RA1 Output common display driver DISP2
; RA0 Output common display driver DISP1
; DISP3 lit when both RA0, RA1 high
; RA4 Input from switches

; RB0 Input from comparator
; RB1 c segment drive for seven segment LED display
; RB2 d segment drive
; RB3 e segment drive
; RB4 f segment drive
; RB5 a segment drive
; RB6 b segment drive
; RB7 g segment drive

; CPU configuration
; 	
	list P=16F84
	#include "p16f84.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

EEPROM1		equ	H'00'	; A/D conversion value for reflo storage MS bits
EEPROM2		equ	H'01'	; A/D conversion value for reflo storage LS bits

EEPROM3		equ	H'02'	; A/D conversion value for refhi MS bits 
EEPROM4		equ	H'03'	; A/D conversion value for refhi LS bits 

EEPROM5		equ	H'04'	; display value for reflo storage MS bits
EEPROM6		equ	H'05'	; display value for reflo storage LS bits

EEPROM7		equ	H'06'	; display value for refhi MS bits 
EEPROM8		equ	H'07'	; display value for refhi LS bits 

EEPROM9		equ	H'08'	; display value for alarm MS bits 
EEPROMA		equ	H'09'	; display value for alarm LS bits 
EEPROMB		equ	H'0A'	; alarm sense (high or low) above threshold 

; RAM

DISP1		equ	H'0C'	; value for Display1 
DISP2		equ	H'0D'	; value for Display2 
DISP3		equ	H'0E'	; value for Display3 
STATUS_TMP 	equ 	H'0F'	; temp storage for status during interrupt
W_TMP		equ	H'10'	; temporary storage for w during interrupt
FLAG_1		equ	H'11'	; bit 0 is multiplex or PWM flag, bit1 is for interrupt count
				; bit 6 is EEPROM write repeat
PWM_CNT		equ	H'12'	; counter for PWM output
LOW_TME		equ	H'13'	; PWM low time
BIN_0		equ	H'14'	; binary value lsd
BIN_1		equ	H'15'	; binary value 
BCD_0		equ	H'16'	; BCD MS
BCD_1		equ	H'17'	; display value MS
BCD_2		equ	H'18'	; display value LS
TEMP		equ	H'19'	; temporary register
CNT_16		equ	H'1A'	; counter for BCD conversion
TEMP_2		equ	H'1B'	; temp storage during EEPROM write
CAL_FLG		equ	H'1C'	; calibrate flag
TEMP_1		equ	H'1D'	; temp store 
EXTN		equ	H'1E'	; delay extension value
LOW_TM1		equ	H'1F'	; PWM least significant bits
STORE		equ	H'20'	; storage of LOW_T0
FLAG3		equ	H'21'	; bit 1 for switch pressed
				; bit 2 for EEPROM copy
PWM_C1		equ	H'22'	; timer for PWM
CNT_PWM		equ	H'23'	; counter for PWM
STO_W		equ	H'24'	; w store
VALUE_1		equ	H'25'	; delay counter
VALUE_2		equ	H'26'	; delay counter
LOW_T1		equ	H'27'	; storage of A/D LS
LOW_T0		equ	H'28'	; storage of A/D MS
STOR1		equ	H'29'	; storage of LOW_T1

REFLO		equ	H'2D'	; reference voltage for lowest reading MS
REFLO1		equ	H'2E'	; reference voltage for lowest reading LS
REFHI		equ	H'2F'	; reference voltage for highest reading MS
REFHI1		equ	H'30'	; reference voltage for highest reading LS
REFLOD		equ	H'31'	; display value for lowest input from A/D MS
REFLOD1		equ	H'32'	; display vlaue for lowest input from A/D LS 
REFHID		equ	H'33'	; display value for highest input from A/D reading MS
REFHID1		equ	H'34'	; display value for highest input from A/D reading LS
REFLM		equ	H'35'	; display value for alarm LED MS
REFLM0		equ 	H'36'	; display value for alarm LED LS
SELECT		equ	H'37'	; option selection using resistors on RB3,4
STORAG1 	equ	H'38'	; storage for calculations
STORAG2	 	equ	H'39'	; storage for calculations
STORAG3		equ	H'3A'	; storage for calculations
STORAG4		equ	H'3B'	; storage for calculations
STORAG5		equ	H'3C'	; storage for calculations
STORAG6		equ	H'3D'	; storage for calculations
ASTORE		equ	H'3E'	; storage of switch settings

; math routines
SIGN            equ     0x41    ; save location for sign 
TEMPB1          equ     0x42
TEMPB0          equ     0x43
TEMPD           equ     0x44    ; temporary storage
AARGB3		equ	0x45
AARGB2          equ     0x46
AARGB1          equ     0x47
AARGB0          equ     0x48
AARG            equ     0x48    ; most significant byte of argument A
BARGB1          equ     0x49
BARGB0          equ     0x4A
BARG            equ     0x4A    ; most significant byte of argument B
REMB1           equ     0x4B
REMB0           equ     0x4C    ; most significant byte of remainder
LOOPCOUNT       equ     0x4D    ; loop counter
STO1		equ	0x4E	; AARGB1 storage
STO2		equ	0x4F	; AARGB2 storage


; preprogram EEPROM DATA
	
	ORG     2100
	DE	H'02'		; low value A/D
	DE	H'00' 
	DE	H'0C'		; high value A/D
	DE	H'00'		; 
	DE	H'00'		; 0 display value
	DE	H'00'		;
	DE	H'00'
	DE	H'64'		; 100 display max
	DE	H'00' 
	DE	H'32'		; alarm
	DE	H'00'		; alarm sense

; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data. 

SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'10111100'	; 7-segment code 1
	retlw 	B'00010010'	; 7-segment code 2
	retlw 	B'00011000'	; 7-segment code 3
	retlw 	B'00101100'	; 7-segment code 4
	retlw 	B'01001000'	; 7-segment code 5
	retlw 	B'01000000'	; 7-segment code 6
	retlw 	B'10011100'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00001000'	; 7-segment code 9
;	retlw	B'00000100'	; 7-segment code for A 
;	retlw	B'01100000'	; 7-segment code for b 
;	retlw	B'11000010'	; 7-segment code for c 
;	retlw	B'00110000'	; 7-segment code for d 
;	retlw	B'01000010'	; 7-segment code for E 
;	retlw	B'01000110'	; 7-segment code for F 

;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3 in sequence plus 
; uses internal timer to initiate display update
; produces PWM output with duty set by LOW_TME register
; start interrupt by saving w and status registers before altered by interrupt routine      

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf	STATUS,RP0	; select memory bank 0
	
; PWM output routine

PWM_MPX	btfsc	FLAG_1,0	; check bit 0
	goto	LOW_OUT
	bsf	FLAG_1,0	; clear if set
	nop			; output cycle time adjust
	bcf	PORTA,3		; RA3 low
	incf	CNT_PWM,f	; next pwm count
	movf	CNT_PWM,w
	andlw	0x0F		; extract ls bits, bits 0 to bits 4 
	movwf	CNT_PWM
	movf	LOW_TM1,w
	subwf	CNT_PWM,w
	movf	LOW_TME,w	; get lowtime value
	btfss	STATUS,c	; if c is 0 then add 1 to interrupt counter
	addlw	0x01		; add 1

ADD	sublw	0x02		; PWM low time
	nop			; align timer with cycles of instruction
	addwf	TMR0,f		; 4MHz so low time is inverse of 4MHz/4/2/256  
	bcf	INTCON,T0IF	; clear TMRO interrupt flag
	goto	RECLAIM		; 
LOW_OUT	bcf	FLAG_1,0	; set if clear
				; set equal time between high out and low out
	bsf	PORTA,3		; RA3 high
	movf	LOW_TM1,w
	subwf	CNT_PWM,w
	movf	LOW_TME,w	; get lowtime value
	btfss	STATUS,c	; if c is 0 then add 1 to interrupt counter
	goto	ADD2
	addlw	0x01		; add 1
	goto 	ADDL
ADD2	addlw	0x02		; align timer
ADDL	addwf	TMR0,f		; 4MHz so high time is inverse of 4MHz/4/2/256  
	bcf	INTCON,T0IF

; multiplex display routine
	
	bsf	FLAG_1,1	; set flag for interrupt count
	movlw	0xFE		; all segments off
	movwf	PORTB 		; blanking
	
	btfss	PORTA,0		; skip if display 1 not lit
	goto	LIT1		; display 1 lit
	btfss	PORTA,1		; skip if display 2 not lit
	goto	LIT2		; display 2 lit

	bcf	PORTA,0		; disp1 powered				
	movf	DISP1,w		; display1 
	nop			; time for display
	nop
	nop
	movwf 	PORTB
	goto	RECLAIM		; end of multiplex

LIT1	bcf	PORTA,1		; disp2 powered
	bsf	PORTA,0		; disp1 off
	movf	DISP2,w		; display2
	nop			; time to settle low 
	nop
	nop
	movwf 	PORTB
	goto	RECLAIM		; end of multiplex

LIT2	bsf	PORTA,1		; disp2 off
	movf	DISP3,w		; display3
	nop
	nop
	nop 
	movwf	PORTB		; seven segment value to portB	
		

; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN	movlw	B'01111110'	; code for 7-segment display dashes
	movwf 	DISP1		; display values
	movwf	DISP2		; initial display value
	movwf	DISP3		; initial display value
	clrf	CNT_PWM		; counter during pwm
	clrf	LOW_TME
	clrf	LOW_TM1
	clrf	LOW_T0
	clrf	LOW_T1
	clrf	CAL_FLG	
	clrf	SELECT		; mode
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'00000000'	; w = 00000000 binary
	movwf	OPTION_REG	; TMRO prescaler /2, PORTB pullups enabled
	movlw   B'00010000'	; (RAx outputs, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00000111'
	movwf	PORTA		; portA RA0,1,2 high

; get EEPROM stored values

; REFLO		; reference voltage for lowest reading MS
; REFLO1	; reference voltage for lowest reading LS
; REFHI		; reference voltage for highest reading MS
; REFHI1	; reference voltage for highest reading LS
; REFLOD	; display value for lowest input from A/D MS
; REFLOD1	; display value for lowest input from A/D LS 
; REFHID	; display value for highest input from A/D reading MS
; REFHID1	; display value for highest input from A/D reading LS
; REFLM		; display value for alarm LED MS
; REFLM0	; display value for alarm LED LS

	movlw	EEPROM1		; address for EEPROM1
	call	EEREAD
	movwf	REFLO		; reference voltage for reading (low Voltage MS) 
	movlw	EEPROM2
	call	EEREAD
	movwf	REFLO1		; reference voltage for reading (low voltage LS)
	movlw	EEPROM3		; address for EEPROM3
	call	EEREAD
	movwf	REFHI		; reference voltage for reading (high voltage MS)
	movlw	EEPROM4		; address for EEPROM4
	call	EEREAD
	movwf	REFHI1		; reference voltage for reading (high voltage LS)
	movlw	EEPROM5		; address for EEPROM5	
	call	EEREAD
	movwf	REFLOD		; display value for lowest input from A/D MS
	movlw	EEPROM6		; address for EEPROM6
	call	EEREAD
	movwf	REFLOD1		; display vlaue for lowest input from A/D LS 
	movlw	EEPROM7		; address for EEPROM7
	call	EEREAD
	movwf	REFHID		; display value for highest input from A/D reading MS
	movlw	EEPROM8		; address for EEPROM8
	call	EEREAD
	movwf	REFHID1		; display value for highest input from A/D reading LS
	movlw	EEPROM9		; address for EEPROM9
	call	EEREAD
	movwf	REFLM		; display value for alarm LED MS
	movlw	EEPROMA		; address for EEPROMA
	call	EEREAD
	movwf	REFLM0		; display value for alarm LED LS

; check for alarm sense

	btfsc	PORTA,4		; is mode switch pressed
	goto	NO_OPEN		; not pressed	
	movlw	EEPROMB		; address for alarm sense
	call	EEREAD
	movwf	TEMP
	btfss	TEMP,0		; check sense bit
	goto	SENS_P		; positive sense
	clrf	TEMP
	movlw	EEPROMB
	movwf	EEADR
	movf	TEMP,w
	movwf	EEDATA
	call	EWRITE
	movlw	B'11111110'	; blank for +
	movwf	DISP3
	movlw	B'00000100'	; A for Alarm
	movwf	DISP1
	movlw	B'11100010'	; L for aLarm
	movwf	DISP2
	goto	NO_OPEN
SENS_P	bsf	TEMP,0
	movlw	EEPROMB
	movwf	EEADR
	movf	TEMP,w
	movwf	EEDATA
	call	EWRITE
	movlw	B'01111110'	; minus sign
	movwf	DISP3
	movlw	B'00000100'	; A for Alarm
	movwf	DISP1
	movlw	B'11100010'	; L for aLarm
	movwf	DISP2

; interrupt enable and program now runs

NO_OPEN	btfss	PORTA,4		; is mode switch pressed
	goto	NO_OPEN		; pressed, wait until open	
	bsf	INTCON,T0IE	; set interrupt enable for TMR0 
	bsf	INTCON,GIE	; set global interrupt enable for above

; Successive Approximation for A-D converter
	
NEWVAL	movf	SELECT,w
	btfss	STATUS,z	; if zero normal mode
	call	DELTME		; delay between events
	movf	PORTA,w
	movwf	ASTORE		; portA storage
	btfss	ASTORE,4
	goto	SWITCH		; check switch
	
CKSEL	clrf	LOW_TME		; ms bits make low first
	clrf	LOW_TM1		; ls bits
	
	bsf	LOW_TME,7	; most significant bit 7 set
	movlw	0x08
	call	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; comparator out. if RB0 high then value below 
	bcf	LOW_TME,7	; bit 7 cleared
	bsf	LOW_TME,6	; next LS bit test
	movlw	0x08
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit 6)
	bcf	LOW_TME,6
	bsf	LOW_TME,5
	movlw	0x07
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit 5)
	bcf	LOW_TME,5
	bsf	LOW_TME,4
	movlw	0x06
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit 4)
	bcf	LOW_TME,4
	bsf	LOW_TME,3
	movlw	0x06
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit 3)
	bcf	LOW_TME,3
	bsf	LOW_TME,2
	movlw	0x05
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss	PORTB,0		; if RB0 high then value below (high bit 2)
	bcf	LOW_TME,2
	bsf	LOW_TME,1
	movlw	0x05
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit 1)
	bcf	LOW_TME,1
	bsf	LOW_TME,0
	movlw	0x05
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss	PORTB,0		; bit 0 test
	bcf	LOW_TME,0

; extra ls bits to add in 4-bit resolution to above 8-bits
	
	bsf	LOW_TM1,3	; 9 bit A-D 
	movlw	0x04
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH	
	btfss 	PORTB,0		; if RB0 high then value below (high bit) 
	bcf	LOW_TM1,3
	bsf	LOW_TM1,2	; next 10 bit A-D 
	movlw	0x04
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit)
	bcf	LOW_TM1,2
	bsf	LOW_TM1,1	; for 11-bit A/D 
	movlw	0x04
	call 	SAR
	btfss	ASTORE,4
	goto	SWITCH
	btfss 	PORTB,0		; if RB0 high then value below (high bit) 
	bcf	LOW_TM1,1
	bsf	LOW_TM1,0	; next LS bit test (for 12-bit A/D) 
;	movlw	0x04
;	call 	SAR
;	btfss	ASTORE,4
;	goto	SWITCH
;	btfss	PORTB,0
;	bcf	LOW_TM1,0

; process values	

	movf	LOW_TME,w	; store 255 minus lowtime value into binary0 register
	sublw	D'255'		; get reverse value ie high output time
	movwf	LOW_T0		; for multiply

	movf	LOW_TM1,w	; ls bits
	sublw	0xFF		; get reverse value ie high output time
	andlw	0x0F		; extract bits
	movwf	LOW_T1		;

	bcf	SIGN,0		; positive sign to begin
	swapf	LOW_T0,w
	andlw	0xF0		; ms bits
	addwf	LOW_T1,f	; combine low bytes into LOW_T1 
	swapf	LOW_T0,w
	andlw	0x0F		; shift ms bits to ls bits
	movwf	LOW_T0

; Check mode

CK_MD	movf	SELECT,w
	btfsc	STATUS,z	; if zero normal mode
	goto	CALC		; normal calculation mode
	btfsc	SELECT,2	; if set hex mode
	goto	A_D
	decfsz	SELECT,w	; if zero then cal low mode
	goto	HICALAM		; high cal or alarm mode

; low cal mode

	movf	REFLOD,w	; display value for lowest input from A/D MS
	movwf	BIN_1
	movf	REFLOD1,w	; display value for lowest input from A/D LS 
	movwf	BIN_0
	goto	DISP_BD		; show on display

HICALAM movf	SELECT,w	; mode
	xorlw	0x02		; hi cal 
	btfss	STATUS,z
	goto	ALARM_M		
	movf	REFHID,w	; display value for high input from A/D MS
	movwf	BIN_1
	movf	REFHID1,w	; display value for high input from A/D LS 
	movwf	BIN_0
	goto	DISP_BD		; show on display

ALARM_M movf	REFLM,w		; display value for alarm MS
	movwf	BIN_1
	movf	REFLM0,w	; display value for alarm LS 
	movwf	BIN_0
	goto	DISP_BD		; show on display

; A/D selections required so display A/D values
	
A_D	bcf	SIGN,0
	movf	LOW_T0,w
	movwf	STORE
	xorlw	0x0F
	btfsc	STATUS,z	; if zero then full scale
	goto	FULL
	rrf	STORE,f	        ; move MS byte right and to carry
	rrf	LOW_T1,w	; move ls byte right to divide by 2
	movwf	STOR1
	rrf	STORE,w
	andlw	B'00000011'	; delete extraneous carry bits in LOW_T0
	movwf	BIN_1		; binary MS byte
	rrf	STOR1,w
	movwf	BIN_0
	goto	DISP_BD
FULL	movlw	B'01000110'	; 7-segment code for F 
	movwf	DISP1
	movlw	B'10100000'	; 7-segment code U 
	movwf	DISP2
	movlw	B'11100010'	; 7-segment code for L
	movwf	DISP3
	goto	NEWVAL

; A/D hex values 
;	bcf	SIGN,0		; clear minus sign
;	movf	LOW_T0,w	; ms digits
;	andlw	0x0F		; extract ms 4-digits
;	call	SVNSEG		; convert to 7-segment 
;	movwf	DISP1		; ms byte
	
;	swapf	LOW_T1,w	; ls digits
;	andlw	0x0F		; extract ms 4-digits
;	call	SVNSEG
;	movwf	DISP2

;	movf	LOW_T1,w	; ls bits
;	andlw	0x0F
	
;	call	SVNSEG
;	movwf	DISP3
;	goto 	NEWVAL		; read input voltage again


; Calculation. Display shows. 

; (A/D current reading - A/D low calibration Value)x (req'd reading @ high cal - req'd reading @ low cal) + required reading at low cal 
;                                                    ---------------------------------------------- 
; 						    (A/D high cal value - A/D low cal value)	

; 1. subtract A/D low calibration value from A/D current reading


; current A/D reading

CALC	bcf	SIGN,0		; positive sign to begin
	movf	LOW_T1,w	; ls byte
	movwf	AARGB2
	movwf	STO2
	movf	LOW_T0,w
	movwf	AARGB1
	movwf	STO1

; A/D low calibration value

	movf	REFLO,w		; A/D reference voltage for reading (low voltage MS)
	movwf	BARGB0
	movf	REFLO1,w	; A/D reference voltage for reading (low voltage LS)
	movwf	BARGB1
	 
	call	D_SUB		; subtract
	btfss	AARGB1,7	; if bit 7 set then redo subtraction and add (-) minus sign
	goto	SECOND
	
	movf	STO1,w
	movwf	BARGB0
	movf	STO2,w
	movwf	BARGB1
	movf	REFLO,w		; reference voltage for reading (low Voltage MS)
	movwf	AARGB1
	movf	REFLO1,w	; reference voltage for reading (low voltage LS)
	movwf	AARGB2
	call	D_SUB		; subtract in reverse set minus sign
	bsf	SIGN,0 		; set minus sign

; 2. subtract required display reading at low calibration from required display reading at high calibration

SECOND
	
; store previous calculation

	movf	AARGB1,w	; MS answer
	movwf	STORAG1
	movf	AARGB2,w	; LS answer
	movwf	STORAG2

; check for zero

	btfss	STATUS,z	; is STORAG2 zero
	goto	CAL_SEC
	movf	STORAG1,w
	btfsc	STATUS,z	; is storage1 zero
	goto	ZRO_VAL

CAL_SEC	movf	REFLOD,w	; display value for lowest input from A/D MS
	movwf	BARGB0
	movf	REFLOD1,w	; display vlaue for lowest input from A/D LS
	movwf	BARGB1
	movf	REFHID,w	; display value for highest input from A/D reading MS
	movwf	AARGB1
	movf	REFHID1,w	; display value for highest input from A/D reading LS
	movwf	AARGB2
	
	call	D_SUB		; subtract
	btfss	AARGB1,7	; if bit 7 set then redo subtraction and add (-) minus sign
	goto	THIRD		; third calculation

; process sign

	btfss	SIGN,0		; if sign is already negative set as positive	
	goto	SETNEG
	bcf	SIGN,0		; set positive
	goto	CALCREV
SETNEG	bsf	SIGN,0		; set negative

CALCREV movf	REFLOD,w	; display value for lowest input from A/D MS
	movwf	AARGB1
	movf	REFLOD1,w	; display value for lowest input from A/D LS
	movwf	AARGB2
	movf	REFHID,w	; display value for highest input from A/D reading MS
	movwf	BARGB0
	movf	REFHID1,w	; display value for highest input from A/D reading LS
	movwf	BARGB1
	call	D_SUB		; subtract (recalculate as sign negative)

; 3. subtract A/D low cal value from A/D high cal value

THIRD	movf	AARGB1,w	; MS answer
	movwf	STORAG3		; store previous result
	movf	AARGB2,w	; LS answer
	movwf	STORAG4

; A/D high calibration value

	movf	REFHI,w		; A/D reference voltage for reading (high voltage MS)
	movwf	AARGB1
	movf	REFHI1,w	; A/D reference voltage for reading (high voltage LS)
	movwf	AARGB2
	
; A/D low calibration value

	movf	REFLO,w		; A/D reference voltage for reading (low voltage MS)
	movwf	BARGB0
	movf	REFLO1,w	; A/D reference voltage for reading (low voltage LS)
	movwf	BARGB1
	 
	call	D_SUB		; subtract

	movf	AARGB1,w	; MS answer
	movwf	STORAG5		; store previous result
	movf	AARGB2,w	; LS answer
	movwf	STORAG6

; multiply calculations 1 and 2

	movf	STORAG3,w	; retrieve calculations in two
	movwf	AARGB0
	movf	STORAG4,w
	movwf	AARGB1	
	
	movf	STORAG1,w	; retrieve calculations in 1 
	movwf	BARGB0
	movf	STORAG2,w
	movwf	BARGB1
	call    FXM1616U	; multiply. Result is in AARGB

; move across by 8-bits ie aargb1 to aargb0, aargb2 to aargb1 aargb3 to 2 clear 3

	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	clrf	AARGB3


	movf	STORAG6,w
	movwf	BARGB1
	movf	STORAG5,w
	movwf	BARGB0

; 4. divide 
	
	call	FXD2416U	; divide

; 5. add or subtract required reading at low calibration value
	
	movf	AARGB1,w
	movwf	BARGB0
	movwf	STO1
	movf	AARGB2,w
	movwf	BARGB1
	movwf	STO2
	movf	REFLOD1,w	; required reading at A/D low cal
	movwf	AARGB2
	movf	REFLOD,w
	movwf	AARGB1
	btfss	SIGN,0		; if sign set then subtract from reading at low cal
	goto 	ADD_RES		; add result
	call	D_SUB		; subtract
	bcf	SIGN,0		; clear sign
	btfss	AARGB1,7	; if bit 7 set then redo subtraction and add (-) minus sign
	goto	DISP_RD		; prepare values for display
	
	movf	STO1,w
	movwf	AARGB1
	movf	STO2,w
	movwf	AARGB2
	movf	REFLOD,w	; reference voltage for reading (low Voltage MS)
	movwf	BARGB0
	movf	REFLOD1,w	; reference voltage for reading (low voltage LS)
	movwf	BARGB1
	call	D_SUB		; subtract in reverse set minus sign
	bsf	SIGN,0 		; set minus sign	
	goto	DISP_RD

ADD_RES call	D_ADD
DISP_RD	movf	AARGB1,w	; ms of result
	movwf	BIN_1
	movf	AARGB2,w	; LS of result
	movwf	BIN_0
	goto	DISP_BD

ZRO_VAL	movf	REFLOD,w	; required reading at A/D low cal
	movwf	BIN_1
	movwf	AARGB1
	movf	REFLOD1,w
	movwf	BIN_0
	movwf	AARGB2
	clrf	SIGN		; positive

; display conversion to decimal from binary

DISP_BD	movf	SELECT,w
	btfss	STATUS,z
	bcf	SIGN,0		; clear sign if not normal mode
	bcf	SIGN,1		; clear blanking sign
	call	BCD		; convert to BCD values
	movf	BCD_1,w		; ms value to w
	andlw	0x0F		; compare with 0
	btfss	STATUS,z	; skip if zero
	goto 	SHOW		; show digit on disp3
	movlw	B'11111110'	; 7-segment display off
	bsf	SIGN,1
	goto	BLNK		; value to blank display
SHOW	movf	BCD_1,w
	andlw	0x0F		; extract ms bits	
	call 	SVNSEG		; convert to 7-segment value for display
BLNK	movwf	DISP1		; transfer BCD MSD to DISPlay 1
	btfsc	SIGN,0		; if sign set then drive - sign
	bcf	DISP1,7
	movf	BCD_2,w		; least sig byte
	andlw	0x0F		; extract LS bits
	call	SVNSEG
	movwf	DISP3		; place in display 3
	swapf	BCD_2,w
	andlw	0x0F		; extract bits
	btfss	STATUS,z	; if zero then maybe blank 
	goto	NOBNK_2
	btfss	SIGN,1		; is disp3 blanked
	goto	NOBNK_2
	movlw	B'11111110'	; 7-segment display off
	goto	BLNK_2
NOBNK_2	call	SVNSEG
BLNK_2	movwf	DISP2		; place in display register
	
; check alarm level
	movf	SELECT,w	; normal mode
	btfss	STATUS,z
	goto	NEWVAL		; bypass alarm check
	movf	REFLM,w
	subwf	AARGB1,f	; check for the same value
	btfsc	STATUS,z
	goto	CKLS
	btfss	STATUS,c	; if current value >
	goto	NO_ALM
	goto	ALM_ON		; 
CKLS	movf	REFLM0,w	; ls of alarm
	subwf	AARGB2,w	; ls of current value
	btfss	STATUS,c
	goto	NO_ALM

; check alarm sense selection

ALM_ON	movlw	EEPROMB		; alarm sense storage
	call	EEREAD
	movwf	TEMP
	btfss	TEMP,0
	goto	CLR_SNS	
	bsf	PORTA,2	
	goto	NEWVAL
CLR_SNS bcf	PORTA,2
	goto 	NEWVAL		; read input voltage again
NO_ALM	movlw	EEPROMB		; alarm sense storage
	call	EEREAD
	movwf	TEMP
	btfss	TEMP,0
	goto	SET_SNS	
	bcf	PORTA,2	
	goto	NEWVAL
SET_SNS	bsf	PORTA,2
	goto	NEWVAL

; pressed switch routines

SWITCH	btfsc 	ASTORE,4	; was switch pressed
	goto	NEWVAL
	btfss	ASTORE,0	; UP switch
	goto	UP
	btfss	ASTORE,1	; down switch
	goto	DOWN
MODE	movlw	0xFF
	movwf	DISP1		; display off
	movwf	DISP2
	movwf	DISP3
	call	DELTME		; delay between 
	incf	SELECT,w
	sublw	0x04
	btfss	STATUS,c
	goto	CLR_SEL
	incf	SELECT,f
	goto	CK_MD
CLR_SEL clrf	SELECT
	bsf	PORTA,2		; alarm LED off
	movlw	B'01111110'	; code for 7-segment display dashes
	movwf 	DISP1		; display values
	movwf	DISP2		; initial display value
	movwf	DISP3		; initial display value
	call	DELTME
	goto	CK_MD

DOWN
UP	movf	SELECT,w
	btfsc	STATUS,z	; if zero bypass as normal mode
	goto	CK_MD
	btfsc	SELECT,2	; if set bypass as hex mode
	goto	CK_MD
	decfsz	SELECT,w	; if zero then cal low mode
	goto	HI_ALM		; high cal or alarm mode
	
; write current A/D value to EEPROM1 and 2

LOW_CAL	movlw	EEPROM1		; address for EEPROM1
	movwf	EEADR		; data register
	movf	LOW_T0,w	; current A/D MS byte
	movwf	REFLO		; reference voltage for reading (low Voltage MS) 
	call	EWRITE
	movlw	EEPROM2
	movwf	EEADR		; data register
	movf	LOW_T1,w	; current A/D LS byte
	movwf	REFLO1		; reference voltage for reading (low Voltage LS) 
	call	EWRITE
	
; check up or down

	btfss	ASTORE,0	; UP switch
	goto	UP_LOD		; up low display value

; down so decrease display value REFLOD and REFLOD1	
	
	movf	REFLOD1,w	; low value reference LS
	btfss	STATUS,z	; if zero check REFLOD (MS) 
	goto	DECLOD		; not zero so decrease REDLOD1
	movf	REFLOD,w	; low value reference MS
	btfsc	STATUS,z	; both zero so keep at zero
	goto	EEPL		; store in EEPROM
	decf	REFLOD,f	; REFLOD1 was zero so decrease REFLOD
DECLOD	decf	REFLOD1,f
	goto	EEPL
	 	
; increment display value REFLOD and REFLOD1. Store in EEPROM

UP_LOD	incfsz	REFLOD1,f
	goto	CK_OVR		; check overrange
	incf	REFLOD,f
	goto	EEPL
	
CK_OVR	movf	REFLOD1,w
	xorlw	0xE8		; (check for 1+ 999 decimal)
	btfss	STATUS,z	; if zero check reflod
	goto	EEPL		; eeprom low storage
	movf	REFLOD,w
	xorlw	0x03		; check for 999 decimal
	btfsc	STATUS,z
	decf	REFLOD1,f	; stop at 999	
		
EEPL	movlw	EEPROM5		; address for EEPROM5
	movwf	EEADR
	movf	REFLOD,w	; display value for lowest input from A/D MS	
	call	EWRITE

	movlw	EEPROM6		; address for EEPROM6
	movwf	EEADR
	movf	REFLOD1,w	; display value for lowest input from A/D LS 
	call	EWRITE
	goto	CK_MD


HI_ALM	movf	SELECT,w	; mode
	xorlw	0x02		; hi cal 
	btfss	STATUS,z
	goto	ALM_M		; 

; high cal
HI_CAL	movlw	EEPROM3		; address for EEPROM3
	movwf	EEADR		; data register
	movf	LOW_T0,w	; current A/D MS byte
	movwf	REFHI		; reference voltage for reading (high Voltage MS) 
	call	EWRITE
	movlw	EEPROM4
	movwf	EEADR		; data register
	movf	LOW_T1,w	; current A/D LS byte
	movwf	REFHI1		; reference voltage for reading (high Voltage LS) 
	call	EWRITE

; check up or down

	btfss	ASTORE,0	; UP switch
	goto	UP_HID		; up high display value

; down so decrease display value REFHID and REFHID1	
	
	movf	REFHID1,w	; hi value reference LS
	btfss	STATUS,z	; if zero check REFHID (MS) 
	goto	DECHID		; not zero so decrease REFHID1
	movf	REFHID,w	; low value reference MS
	btfsc	STATUS,z	; both zero so keep at zero
	goto	EEPL		; store in EEPROM
	decf	REFHID,f	; REFHID1 was zero so decrease REFHID
DECHID	decf	REFHID1,f
	goto	EEPH
	
; increment display value REFHID and REFHID1. Store in EEPROM

UP_HID	incfsz	REFHID1,f
	goto	IS_OVR		; check overrange
	incf	REFHID,f
	goto	EEPH

IS_OVR	movf	REFHID1,w
	xorlw	0xE8		; (check for 1+ 999 decimal)
	btfss	STATUS,z	; if zero check reflod
	goto	EEPH		; eeprom high storage
	movf	REFHID,w
	xorlw	0x03		; check for 999 decimal
	btfsc	STATUS,z
	decf	REFHID1,f	; stop at 999
		
EEPH	movlw	EEPROM7		; address for EEPROM7
	movwf	EEADR
	movf	REFHID,w	; display value for high input from A/D MS	
	call	EWRITE

	movlw	EEPROM8		; address for EEPROM8
	movwf	EEADR
	movf	REFHID1,w	; display value for high input from A/D LS 
	call	EWRITE
	goto	CK_MD

; ALARM check up or down

ALM_M	btfss	ASTORE,0	; UP switch
	goto	UP_ALM		; up alarm display value

; down so decrease display value REFLM and REFLM0	
	
	movf	REFLM0,w	; alarm value reference LS
	btfss	STATUS,z	; if zero check REFLM (MS) 
	goto	DECALM		; not zero so decrease REFLM0
	movf	REFLM,w		; low value reference MS
	btfsc	STATUS,z	; both zero so keep at zero
	goto	EEPA		; store in EEPROM
	decf	REFLM,f		; REFLM0 was zero so decrease REFLM
DECALM	decf	REFLM0,f
	goto	EEPA
	
; increment display value REFLM and REFLM0. Store in EEPROM


UP_ALM	incfsz	REFLM0,f
	goto	LM_OVR		; check overrange
	incf	REFLM,f
	goto	EEPA

LM_OVR	movf	REFLM0,w
	xorlw	0xE8		; (check for 1+ 999 decimal)
	btfss	STATUS,z	; if zero check reflod
	goto	EEPH		; eeprom high storage
	movf	REFLM,w
	xorlw	0x03		; check for 999 decimal
	btfsc	STATUS,z
	decf	REFLM0,f	; stop at 999	
	
EEPA	movlw	EEPROM9		; address for EEPROM9
	movwf	EEADR
	movf	REFLM,w		; display value for high input from A/D MS	
	call	EWRITE

	movlw	EEPROMA		; address for EEPROMA
	movwf	EEADR
	movf	REFLM0,w	; display value for high input from A/D LS 
	call	EWRITE
	goto	CK_MD



; ******************************************************************************
; Subroutines

; delay

DELTME	movlw	D'255'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_1	movwf	VALUE_2		; VALUE_2 = w
LP_2	nop
	nop
	nop
	nop
	nop
	nop
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1	
	return

; Subroutine for PWM cycle period for successive approximation A-D converter 
; pwm_cnt must be multiple of 16

SAR	clrf	CNT_PWM		; ls bits counter
		
DEL1	movwf	PWM_C1
SAR_AGN	movlw	0xD0		; number of interrupts between PWM changes
	movwf	PWM_CNT
CNT_AGN	bcf	FLAG_1,1	; bit set in interrupt at multiplex rate
CNT_NOW	btfss	FLAG_1,1	; look at flag
	goto	CNT_NOW		; wait till flag set at multiplex interrupt 

	movf	PORTA,w
	movwf	ASTORE		; portA storage
	btfss	PORTA,4		; check if switch pressed
	return			; switch pressed return
	decfsz 	PWM_CNT,f	; reduce this value 
	goto	CNT_AGN		; cycle
	
	decfsz	PWM_C1,f
	goto	SAR_AGN	 	; loop again

	movf	SELECT,w	; mode
	btfsc	STATUS,z	; normal mode
	goto	OFF_ALM		; bypass flash alarm LED 
	btfsc	PORTA,2		; check alarm LED
	goto	ON_ALM
	bsf	PORTA,2		; alarm LED off
	goto	OFF_ALM	
ON_ALM	bcf	PORTA,2		; flash alarm LED
OFF_ALM	return

; Subroutine to convert from 16-bit binary to 3-digit BCD (packed)
; Binary value is in BIN0 & BIN1. BIN0 is low byte BIN1 is high byte
; Result in BCD is in BCD0, BCD1 & BCD2.  
; BCD0 is MSB, BCD2 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'16'
	movwf	CNT_16		; 16 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
	clrf	BCD_2
	

LOOPBCD	rlf	BIN_0,f
	rlf	BIN_1,f		; LSB shift left binary registers
	rlf	BCD_2,f		; LSB shift left BCD registers
	rlf	BCD_1,f
	rlf	BCD_0,f

	decfsz	CNT_16,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_2		; BCD LSB address
	movwf	FSR		; pointer for BCD2
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_1
	movwf	FSR
	call 	ADJBCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-2)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag
	bcf	STATUS,RP0	; bank 0 
	return			; value correctly written 
	
	
; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LA 	RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LC 	RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,c
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L		RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,c
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,c
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,c
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,c
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,c
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,c
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,c
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,c
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN

; subroutine subtract AARGB1 AARGB2 - BARGB0 BARGB1 = AARGB
	
D_SUB	call	NEG_A		; complement of A

D_ADD	movf	BARGB1,w 
	addwf	AARGB2,f 	; add lsb
	btfsc	STATUS,c	; add carry
	incf	AARGB1,f 
	movf	BARGB0,w 
	addwf	AARGB1,f 
	return

NEG_A	comf	BARGB1,f 
	incf	BARGB1,f 
	btfsc	STATUS,z
	decf	BARGB0,f 
	comf	BARGB0,f 
	return	

	
	
	end	
