
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
//#include <sys/ioctl.h>
#include "WAV.h"
#include "FIFO.h"
#include "TCPWAVStream.h"

inline void msleep(int msec) {
  usleep(msec*1000);
}

int WAVDataShrink(unsigned char* buf, int len, int blockAlign) {
  int i, j;
  for( i = 0; i <= len - blockAlign * 10; i += blockAlign ) {
    for( j = i; j < i + blockAlign * 10; ++j )
      if( buf[j] != 0 )
        break;
    if( j == i + blockAlign * 10 ) {
      memmove(buf + i, buf + i + blockAlign, len - i - blockAlign);
      len -= blockAlign;
      i += blockAlign * 8;
    }
  }
  return len;
}

int WAVDataPad(unsigned char* buf, int len, int blockAlign) {
  int i, j;
  for( i = 0; i <= len - blockAlign * 10; i += blockAlign ) {
    for( j = i; j < i + blockAlign * 10; ++j )
      if( buf[j] != 0 )
        break;
    if( j == i + blockAlign * 10 ) {
      memmove(buf + i + blockAlign, buf + i, len - i - blockAlign);
      len += blockAlign;
      i += blockAlign * 10;
    }
  }
  return len;
}

//#define _DEBUG
void show_status(int file_open, FIFO* pFIFO, TCPWAVStream* pStream, WAVHeader* pWAVHeader) {
  static int temp = 0;
  if( ++temp < 10 )
    return;
  temp = 0;
  printf("File %6s  TCP connection %7s Buffer %3.1f%% full  %6dkHz %2dbit %6s\r",
          file_open ? "open  " : "closed",
          pStream->connection_state == connected ? "open   " : pStream->connection_state == connecting ? "opening" : "closed ",
          FIFO_BytesStored(pFIFO)*100.0/pFIFO->size,
          file_open ? (int)pWAVHeader->fmtHeader.sampleRate : 0,
          file_open ? (int)pWAVHeader->fmtHeader.bitsPerSample : 0,
          file_open ? pWAVHeader->fmtHeader.numChannels == 1 ? "mono  " : pWAVHeader->fmtHeader.numChannels == 2 ? "stereo" : "      " : "      ");
  fflush(stdout);
}

#define FIFO_SIZE      1024*1024*2
#define TCP_BUFFER_SIZE 128*1024

int main(int argc, char** argv) {
  WAVHeader curwavheader;
  FIFO fifo;
  TCPWAVStream stream;
  int startup = 1;
  int len, ret;
  unsigned char buf[4096*2];

  if( argc != 3 ) {
    fprintf(stderr, "Usage: %s <WAV file name> <hostname|ip>[:<port>]\n", argv[0]);
    return -1;
  }

  FIFO_Init(&fifo, 1024*1024*2);
  ret = TCPWAVStream_Init(&stream, argv[2]);
  if( ret < 0 )
    return ret;
  memset(&curwavheader, 0, sizeof(curwavheader));

  while(1) {
    
    int fd = argv[1][0] == '-' && argv[1][1] == '\0' ? 0 : open(argv[1], O_RDONLY);
    if( fd >= 0 ) {
      WAVHeader wavheader;
      struct stat statinfo;

#ifdef _DEBUG
      fprintf(stderr, "File opened\n");
#endif

      len = read(fd, buf, sizeof(buf));
      if( WAVreadWAVHeader(buf, &wavheader, len) ) {
        size_t data_start = (const unsigned char*)wavheader.dataHeader.dataPtr - (const unsigned char*)&wavheader;
        if( fd != 0 ) {
          if( startup ) {
            int temp = lseek(fd, 0, SEEK_END);
            if( temp > data_start ) {
              data_start += (temp - data_start) / wavheader.fmtHeader.blockAlign * wavheader.fmtHeader.blockAlign;
#ifdef _DEBUG
              fprintf(stderr, "Seek to end (%u)\n", data_start);
#endif
            }            
          }
          lseek(fd, data_start, SEEK_SET);
          startup = 0;
        } else {
//          unsigned long mode = 1;
//          ioctl(fd, FIONBIO, &mode);
        }
        if( TCPWAVStream_IsConnected(&stream) && memcmp(&wavheader.fmtHeader, &curwavheader.fmtHeader, sizeof(wavheader.fmtHeader)) ) {
          if( stream.bytes_left < FIFO_BytesStored(&fifo) ) {
#ifdef _DEBUG
            fprintf(stderr, "Format change, disconnect.\n");
#endif
            TCPWAVStream_Disconnect(&stream);
          } else {
            stream.bytes_left = FIFO_BytesStored(&fifo); // force connection to close & send new WAV header (with updated format info) once the currently buffered data has been sent
          }
        }
        memcpy(&curwavheader, &wavheader, sizeof(curwavheader));

        while(1) {
          len = read(fd, buf, sizeof(buf));
#ifndef _DEBUG
          show_status(1, &fifo, &stream, &curwavheader);
#endif
          if( len > 0 ) {
            if( FIFO_BytesFree(&fifo) >= len ) {
              FIFO_Write(&fifo, buf, len);
#ifdef _DEBUG
              fputc('.', stderr);
              fflush(stderr);
#endif
            } else {
#ifdef _DEBUG
              fprintf(stderr, "Buffer overrun.\n");
#endif
            }
          } else if( fd > 0 ) {
            size_t close_pos = lseek(fd, 0, SEEK_CUR);
            int timeout = 1000/25;
            close(fd);
            fd = 0;

            while( stat(argv[1], &statinfo) == 0 && statinfo.st_size == close_pos && timeout-- ) {
#ifdef _DEBUG
              fputc('_', stderr);
              fflush(stderr);
#endif
              msleep(5);//25);
            }

            if( timeout <= 0 ) {
#ifdef _DEBUG
              fprintf(stderr, "File hasn't changed in a while, deleting.\n");
#endif
              if( unlink(argv[1]) == 0 )
                break;
              // otherwise the sound may just be paused in which case we should continue reading
            }

            if( statinfo.st_size < close_pos || (fd = open(argv[1], O_RDONLY)) == 0 || lseek(fd, close_pos, SEEK_SET) != close_pos ) {
#ifdef _DEBUG
              fprintf(stderr, "File truncated [%zu/%zu].\n", (size_t)statinfo.st_size, close_pos);
#endif
              break;
            }
          }
          if( TCPWAVStream_IsConnected(&stream) ) {
            if( FIFO_BytesStored(&fifo) >= 4096 ) {
              int i, sent;
              for( i = 0; i < 8 && TCPWAVStream_IsConnected(&stream) && FIFO_BytesStored(&fifo) >= 4096; ++i ) {
                len = FIFO_Read(&fifo, buf, 4096);
                if( FIFO_BytesStored(&fifo) >= fifo.size*3/4 ) {
                  len = WAVDataShrink(buf, len, curwavheader.fmtHeader.blockAlign);
#ifdef _DEBUG
                  if( len < 4096 )
                    fprintf(stderr, "Buffer filling, shrinking by %d bytes.\n", 4096 - len);
#endif
                } else if( FIFO_BytesStored(&fifo) < fifo.size/4 ) {
                  len = WAVDataPad(buf, len, curwavheader.fmtHeader.blockAlign);
#ifdef _DEBUG
                  if( len > 4096 )
                    fprintf(stderr, "Buffer filling, padding by %d bytes.\n", len - 4096);
#endif
                } else {
#ifdef _DEBUG
                  fputc('*', stderr);
                  fflush(stderr);
#endif
                }
//                fprintf(stderr, "%d bytes in FIFO\n", FIFO_BytesStored(&fifo));
                sent = TCPWAVStream_SendData(&stream, buf, len, 0);
                if( sent < len ) {
                  FIFO_Unread(&fifo, len - sent);
                  break;
                }
              }
            } else {
#ifdef _DEBUG
              fprintf(stderr, "Buffer underrun, disconnecting.\n");
#endif
              TCPWAVStream_Disconnect(&stream);
              FIFO_Empty(&fifo);
            }
          } else if( FIFO_BytesStored(&fifo) >= fifo.size/2 + TCP_BUFFER_SIZE * 2 + curwavheader.fmtHeader.bitsPerSample * curwavheader.fmtHeader.numChannels * curwavheader.fmtHeader.sampleRate / 8 / 10) {
#ifdef _DEBUG
            fprintf(stderr, "Buffer at least half full, connecting...\n");
#endif
            if( TCPWAVStream_Connect(&stream, 0, TCP_BUFFER_SIZE) ) {
#ifdef _DEBUG
              fprintf(stderr, "Connected.\n");
#endif
              curwavheader.riffHeader.riff.chunkSize = 0x7FFFFFFE;
              curwavheader.dataHeader.data.chunkSize = curwavheader.riffHeader.riff.chunkSize - sizeof(curwavheader.riffHeader) - sizeof(curwavheader.fmtHeader);
              len = WAVmakeWAVHeader(buf, &curwavheader, sizeof(buf));
              TCPWAVStream_SendHeader(&stream, buf, len, 0, curwavheader.riffHeader.riff.chunkSize);
            }
          }
        }
      }
      if( fd )
        close(fd);
    }
#ifdef _DEBUG
    fprintf(stderr, "Back out in main loop.\n");
#endif
#ifndef _DEBUG
    show_status(0, &fifo, &stream, &curwavheader);
#endif
    startup = 0;
    if( TCPWAVStream_IsConnected(&stream) ) {
      if( FIFO_BytesStored(&fifo) >= 4096 ) {
        int sent;

        len = FIFO_Read(&fifo, buf, 4096);
        if( FIFO_BytesStored(&fifo) >= fifo.size*3/4 ) {
          len = WAVDataShrink(buf, len, curwavheader.fmtHeader.blockAlign);
#ifdef _DEBUG
          if( len < 4096 )
            fprintf(stderr, "Buffer filling, shrinking by %d bytes.\n", 4096 - len);
#endif
        } else if( FIFO_BytesStored(&fifo) < fifo.size/4 ) {
          len = WAVDataPad(buf, len, curwavheader.fmtHeader.blockAlign);
#ifdef _DEBUG
          if( len > 4096 )
            fprintf(stderr, "Buffer filling, padding by %d bytes.\n", len - 4096);
#endif
        } else {
#ifdef _DEBUG
          fputc('*', stderr);
          fflush(stderr);
#endif
        }
        sent = TCPWAVStream_SendData(&stream, buf, len, 0);
        if( sent < len )
          FIFO_Unread(&fifo, len - sent);
      } else {
#ifdef _DEBUG
        fprintf(stderr, "Run out of data, disconnecting.\n");
#endif
        TCPWAVStream_Disconnect(&stream);
        FIFO_Empty(&fifo);
      }
    } else {
      msleep(50);
#ifdef _DEBUG
      fputc('_', stderr);
      fflush(stderr);
#endif
    }
  }

  FIFO_Deinit(&fifo);
  return 0;
}
