
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  WAV.c: WAV file read functions
  Written by Mauro Grassi and updated by Nicholas Vinen, 2009-2013
*/

#include "wav.h"
#define _CRT_SECURE_NO_WARNINGS
#include <string.h>

unsigned int WAVgetInt(unsigned char* inPtr, unsigned int* x)
{
	/* assumes little endian */
	unsigned int s;
	s=(unsigned int)(*inPtr)+((unsigned int)(*(inPtr+1))<<8);
	*x=s;
	return 2;
}

unsigned int WAVgetLong(unsigned char* inPtr, unsigned long* x)
{
	/* assumes little endian */
	unsigned long s;
	s=(unsigned long)(*inPtr)+((unsigned long)(*(inPtr+1))<<8)+((unsigned long)(*(inPtr+2))<<16)+((unsigned long)(*(inPtr+3))<<24);
	*x=s;
	return 4;
}

unsigned char WAVconvertToUpper(unsigned char in)
{
	if((in>='a')&&(in<='z'))return in-0x20; 
	return in;
}

unsigned char WAVconvertToLower(unsigned char in)
{
	if((in>='A')&&(in<='Z'))return in+0x20; 
	return in;
}

unsigned int WAVgetIDu(unsigned char* inPtr, unsigned long* x)
{
	/* assumes little endian */
	unsigned long s;
	s=(unsigned long)WAVconvertToUpper(*inPtr)+((unsigned long)WAVconvertToUpper(*(inPtr+1))<<8)+((unsigned long)WAVconvertToUpper(*(inPtr+2))<<16)+((unsigned long)WAVconvertToUpper(*(inPtr+3))<<24);
	*x=s;
	return 4;
}

unsigned int WAVgetIDl(unsigned char* inPtr, unsigned long* x)
{
	/* assumes little endian */
	unsigned long s;
	s=(unsigned long)WAVconvertToLower(*inPtr)+((unsigned long)WAVconvertToLower(*(inPtr+1))<<8)+((unsigned long)WAVconvertToLower(*(inPtr+2))<<16)+((unsigned long)WAVconvertToLower(*(inPtr+3))<<24);
	*x=s;
	return 4;
}

unsigned int WAVreadRiffChunk(unsigned char* inPtr, RiffHeader* riffHeader, unsigned int skipIt)
{
	/* read a Riff Header, or skip it */
	if(skipIt)
	{

	}
	else
	{
		inPtr+=WAVgetIDu((unsigned char*)inPtr, (unsigned long*)&riffHeader->riff.chunkID);
		inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&riffHeader->riff.chunkSize);
		inPtr+=WAVgetIDu((unsigned char*)inPtr, (unsigned long*)&riffHeader->WAVID);
	}
	return 12;
}

unsigned int WAVreadDataChunk(unsigned char* inPtr, DataHeader* dataHeader, unsigned int skipIt)
{
	/* read a Data Header, or skip it */
	if(skipIt)
	{
			
	}
	else
	{
		inPtr+=WAVgetIDl((unsigned char*)inPtr, (unsigned long*)&dataHeader->data.chunkID);
		inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&dataHeader->data.chunkSize);
		dataHeader->dataPtr=inPtr;
	}
	return 8;
}

unsigned int WAVreadFmtChunk(unsigned char* inPtr, FmtHeader* fmtHeader, unsigned int skipIt)
{
	/* read a FMT Header, or skip it */
	if(skipIt)
	{
			
	}
	else
	{
		inPtr+=WAVgetIDl((unsigned char*)inPtr, (unsigned long*)&fmtHeader->fmt.chunkID);
		inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&fmtHeader->fmt.chunkSize);
		inPtr+=WAVgetInt((unsigned char*)inPtr, (unsigned int*)&fmtHeader->audioFormat);
		inPtr+=WAVgetInt((unsigned char*)inPtr, (unsigned int*)&fmtHeader->numChannels);
		inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&fmtHeader->sampleRate);
		inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&fmtHeader->byteRate);
		inPtr+=WAVgetInt((unsigned char*)inPtr, (unsigned int*)&fmtHeader->blockAlign);
		inPtr+=WAVgetInt((unsigned char*)inPtr, (unsigned int*)&fmtHeader->bitsPerSample);		
	}
	return 24;
}

unsigned int WAVreadListChunk(unsigned char* inPtr, ListHeader* listHeader, unsigned int skipIt)
{
	unsigned long x, s;
	unsigned int u;
	unsigned int limit;
	unsigned int k;
	
	u=(unsigned int)inPtr;
	inPtr+=WAVgetIDl((unsigned char*)inPtr, (unsigned long*)&listHeader->list.chunkID);
	inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&listHeader->list.chunkSize);
	limit=(unsigned int)inPtr+listHeader->list.chunkSize;
	inPtr+=WAVgetIDl((unsigned char*)inPtr, (unsigned long*)&listHeader->INFOID);
	if((listHeader->INFOID==LIST_INFO_ID)&&(skipIt==0))
	{
			while((unsigned int)inPtr<limit)
			{
				inPtr+=WAVgetIDl((unsigned char*)inPtr, (unsigned long*)&x);
				inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&s);
				switch(x)
				{
					default:
						/* skip it */
						inPtr+=s;
						break;
					
					case ARTIST_INFO_ID:
						k=0;
						while(s>0)
						{
							if(k<(MAX_ARTIST_LENGTH-1))listHeader->trackArtist[k++]=*inPtr;
							else break;
						
							inPtr++;
							s--;	
						}
						inPtr+=s;
						listHeader->trackArtist[k]='\0';
						break;
						
					case TRACK_INFO_ID:
						k=0;
						while(s>0)
						{							
							if(k<(MAX_TITLE_LENGTH-1))listHeader->trackTitle[k++]=*inPtr;
							else break;
							inPtr++;
							s--;
						}
						inPtr+=s;
						listHeader->trackTitle[k]='\0';
						break;
				
					case DATE_INFO_ID:
						k=0;
						while(s>0)
						{							
							if(k<(MAX_DATE_LENGTH-1))listHeader->trackDate[k++]=*inPtr;
							else break;
							inPtr++;
							s--;
						}
						inPtr+=s;
						listHeader->trackDate[k]='\0';
						break;
				
					case COMMENT_INFO_ID:
						k=0;
						while(s>0)
						{
							if(k<(MAX_COMMENT_LENGTH-1))listHeader->trackComment[k++]=*inPtr;
							else break;
							inPtr++;
							s--;
						}
						inPtr+=s;
						listHeader->trackComment[k]='\0';
						break;
				}
			}
	}		
	else
	{
			inPtr+=listHeader->list.chunkSize;
	}
	return (int)inPtr-(int)u;
}

unsigned int WAVreadWAVChunk(unsigned char* inPtr, WAVHeader* header, unsigned int* isDataChunk, unsigned int skipIt)
{
	/* read conditional on the ID being correct at inPtr */
	unsigned long x, size;
	unsigned int u;
	
	*isDataChunk=0;
	u=(unsigned int)inPtr;
	WAVgetIDl((unsigned char*)inPtr, (unsigned long*)&x);
	WAVgetLong((unsigned char*)inPtr+4, (unsigned long*)&size);
		
	switch(x)
	{
		default:
			inPtr+=size;
			break;
			
		case RIFF_HEADER_ID:
			inPtr+=WAVreadRiffChunk(inPtr, (RiffHeader*)&header->riffHeader, 0);
			break;
					
		case DATA_HEADER_ID:
			inPtr+=WAVreadDataChunk(inPtr, (DataHeader*)&header->dataHeader, 0);
			*isDataChunk=1;
			break;
			
		case FMT_HEADER_ID:
		 	inPtr+=WAVreadFmtChunk(inPtr, (FmtHeader*)&header->fmtHeader, skipIt);
			break;
			
		case LIST_HEADER_ID:
			inPtr+=WAVreadListChunk(inPtr, (ListHeader*)&header->listHeader, skipIt);	
			break;
	}
	return (int)inPtr-(int)u;
}

unsigned int WAVreadWAVHeader(unsigned char* inPtr, WAVHeader* header, unsigned maxBytes)
{
	unsigned int isDataChunk;
	unsigned int u;
	unsigned int limit;
	
	isDataChunk=0;
	u=(unsigned int)inPtr;
	inPtr+=WAVgetIDu((unsigned char*)inPtr, (unsigned long*)&header->riffHeader.riff.chunkID);
	inPtr+=WAVgetLong((unsigned char*)inPtr, (unsigned long*)&header->riffHeader.riff.chunkSize);
	limit=(unsigned int)u+maxBytes;
	inPtr+=WAVgetIDu((unsigned char*)inPtr, (unsigned long*)&header->riffHeader.WAVID);
	if((header->riffHeader.riff.chunkID==RIFF_HEADER_ID)&&(header->riffHeader.WAVID==WAV_HEADER_ID))
	{
			while(((unsigned int)inPtr<limit)&&(isDataChunk==0))
			{
				inPtr+=WAVreadWAVChunk(inPtr, header, &isDataChunk, 0);
			}
			
			if(isDataChunk!=0)
			{
				return 1;
			}
			else return 0;
	}
	return 0;
}

void WAVinitHeader(WAVHeader* header)
{
	header->riffHeader.riff.chunkID=0;
	header->riffHeader.riff.chunkSize=0;
	header->riffHeader.WAVID=0;
	header->fmtHeader.fmt.chunkID=0;
	header->fmtHeader.fmt.chunkSize=0;
	header->fmtHeader.audioFormat=0;
	header->fmtHeader.numChannels=0;
	header->fmtHeader.sampleRate=0;
	header->fmtHeader.byteRate=0;
	header->fmtHeader.blockAlign=0;
	header->fmtHeader.bitsPerSample=0;	
	header->listHeader.list.chunkID=0;
	header->listHeader.list.chunkSize=0;
	header->listHeader.INFOID=0;
	strcpy((char*)header->listHeader.trackTitle, "Untitled Track");
	strcpy((char*)header->listHeader.trackArtist, "Unknown Artist");
	header->listHeader.trackDate[0]='\0';
	header->listHeader.trackComment[0]='\0';
	header->dataHeader.data.chunkID=0;
	header->dataHeader.data.chunkSize=0;
	header->dataHeader.dataPtr=0;
}

unsigned int WAVmakeWAVHeader(unsigned char* Ptr, const WAVHeader* pHeader, unsigned int maxBytes) {
	if( maxBytes < sizeof(pHeader->riffHeader)+sizeof(pHeader->fmtHeader)+sizeof(pHeader->dataHeader) )
		return 0;
	memcpy(Ptr, &pHeader->riffHeader, sizeof(pHeader->riffHeader));
	memcpy(Ptr+sizeof(pHeader->riffHeader), &pHeader->fmtHeader, sizeof(pHeader->fmtHeader));
	memcpy(Ptr+sizeof(pHeader->riffHeader)+sizeof(pHeader->fmtHeader), &pHeader->dataHeader, sizeof(pHeader->dataHeader));
	return sizeof(pHeader->riffHeader)+sizeof(pHeader->fmtHeader)+sizeof(pHeader->dataHeader);
}
