
#include <stddef.h>
#include <string.h>
#include <netdb.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <errno.h>
#include "TCPWAVStream.h"

int TCPWAVStream_Init(TCPWAVStream* pStream, const char* dest) {
  const char* colon = strchr(dest, ':');
  char* buf;
  unsigned short port = 44100;
  in_addr_t addr;

  memset(pStream, 0, sizeof(*pStream));
  if( colon ) {
    char* endptr = (char*)colon;
    port = strtoul(colon+1, &endptr, 10);
    if( endptr != colon + strlen(colon) ) {
      fprintf(stderr, "Invalid port number.\n");
      return -3;
    }
    buf = (char*)malloc(colon - dest + 1);
    strncpy(buf, dest, colon-dest);
    buf[colon-dest] = '\0';
  } else {
    buf = (char*) dest;
  }

  if( buf[0] >= '0' && buf[0] <= '9' ) {
    addr = inet_addr(buf);
    if( addr == INADDR_NONE ) {
      if( buf != dest )
        free(buf);
      fprintf(stderr, "Invalid destination address.\n");
      return -4;
    }
  } else {
    struct hostent* host = gethostbyname(buf);
    if( !host ) {
      if( buf != dest )
        free(buf);
      fprintf(stderr, "Invalid destination address.\n");
      return -4;
    }
    addr = *((in_addr_t*)host->h_addr_list[0]);
  }
  pStream->ip = addr;
  pStream->port = htons(port);
  if( buf != dest )
    free(buf);

  pStream->connection_state = disconnected;
  return 0;
}

int TCPWAVStream_Connect(TCPWAVStream* pStream, int block, int buffer_size) {
  struct sockaddr_in dest;

  if( !pStream->sock ) {
//    int len = sizeof(int);
    pStream->sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if( !pStream->sock )
      return 0;
    if( setsockopt(pStream->sock, SOL_SOCKET, SO_SNDBUF, &buffer_size, sizeof(buffer_size)) != 0 ) {
      close(pStream->sock);
      pStream->sock = 0;
      return 0;
    }
    pStream->block = 1;
  }

  memset(&dest, 0, sizeof(dest));
  dest.sin_family = AF_INET;
  dest.sin_addr.s_addr = pStream->ip;
  dest.sin_port = pStream->port;
  if( pStream->block != block ) {
    unsigned long mode = block ? 0 : 1;
    if( ioctl(pStream->sock, FIONBIO, &mode) != 0 ) {
      close(pStream->sock);
      pStream->sock = 0;
      pStream->connection_state = disconnected;
      return 0;
    }
    pStream->block = block;
  }
  if( connect(pStream->sock, (const struct sockaddr*)&dest, sizeof(dest)) < 0 ) {
    if( errno == EINPROGRESS || errno == EALREADY ) {
      pStream->connection_state = connecting;
      return 1;
    } else if( errno != EISCONN ) {
      close(pStream->sock);
      pStream->sock = 0;
      pStream->connection_state = disconnected;
      return 0;
    }
  }
  pStream->connection_state = connected;
  return 1;
}

void TCPWAVStream_Disconnect(TCPWAVStream* pStream) {
  if( pStream->sock )
    close(pStream->sock);
  pStream->sock = 0;
  pStream->connection_state = disconnected;
}

int TCPWAVStream_IsConnected(TCPWAVStream* pStream) {
  return pStream->connection_state == connected;
}

int TCPWAVStream_SendHeader(TCPWAVStream* pStream, const unsigned char* data, size_t len, int block, size_t max_data_len) {
  int ret;

  pStream->bytes_left = len;
  ret = TCPWAVStream_SendData(pStream, data, len, block);
  if( ret == len )
    pStream->bytes_left = max_data_len;
  return ret;
}

int TCPWAVStream_SendData(TCPWAVStream* pStream, const unsigned char* data, size_t len, int block) {
  if( pStream->connection_state == connected ) {
    ssize_t ret;
    int orig = len;

    if( len > pStream->bytes_left )
      len = pStream->bytes_left;
    if( len == 0 && orig > 0 ) {
      TCPWAVStream_Disconnect(pStream);
      return 0;
    }

    if( pStream->block != block ) {
      unsigned long mode = block ? 0 : 1;
      if( ioctl(pStream->sock, FIONBIO, &mode) != 0 ) {
        close(pStream->sock);
        pStream->sock = 0;
        pStream->connection_state = disconnected;
        return 0;
      }
      pStream->block = block;
    }
    ret = send(pStream->sock, data, len, MSG_NOSIGNAL);
    if( ret > 0 ) {
      pStream->bytes_left -= ret;
      if( pStream->bytes_left == 0 && orig > len )
        TCPWAVStream_Disconnect(pStream);
      return ret;
    } else {
      if( errno != EAGAIN && errno != EWOULDBLOCK ) {
        close(pStream->sock);
        pStream->sock = 0;
        pStream->connection_state = disconnected;
      }
      return 0;
    }
  } else {
    return 0;
  }
}
