#include "util.h"

//this is a check that fails to compile if struct is wrong size
//an error on this line probably means that the struct is larger than a flash page
char a[(sizeof(union flashpadded)==(FLASH_SECTOR_SIZE))?0:-1];

int NTPok=0;
datetime_t dt;	//for general usage
datetime_t tInit = {	//to set initial time/date
		.year  = 2022,	//overridden by value in flash settings
		.month = 01,
		.day   = 01,
		.dotw  = 6, 	// 0 is Sunday, so 6 is Saturday
		.hour  = 00,
		.min   = 00,
		.sec   = 00
};

//C SDK doesn't have much to convert between timestamps and RTC, so use this:
uint64_t NTPoffset=INIT_TIMESTAMP;	//add to time_us_64 to get timestamp,
char* s=NULL;						//for input scanning
char b[SCAN_LEN]="";
int p=0;
int menuState=0; 		
int subMenu=0;
char scanSSID[SCAN_COUNT][SSIDLEN];	//wifi scan results	
int scanRSSI[SCAN_COUNT];			//wifi scan results
int scanChan[SCAN_COUNT];			//wifi scan channel
int scanAuth[SCAN_COUNT];			//wifi scan auth type
uint64_t lastWiFiCheck=0;
uint64_t lastNTPfetch=0;
uint64_t nextIPAPIfetch=0;			//timeouts
int currentSSID=0;					//index into saveSSID[SAVE_COUNT][]
int wifiConnected=0;				//global for status LED display
int ledState=0;						//0-3, higher is better
const char hex[]="0123456789ABCDEF";//for writing out checksum
int failCount=0;

//http for fetching IPAPI
char httpbuf[HTTP_BUF_SIZE]="";
httpc_state_t *connection = NULL;
httpc_connection_t httpconset = {
		.use_proxy = 0,
		.headers_done_fn = headers_done_fn,
		.result_fn = result_fn
};

// saved data is part of a flashpadded struct, see layout/contents in util.h
union flashpadded cur;																//working version in RAM
const union flashpadded __attribute__ ((aligned (FLASH_SECTOR_SIZE))) flashimage={	//in flash
	.d.wifiCC="XX",
//add 'builtin' SSIDs here, eg:
//	.d.saveSSID={"SSID 1","SSID 2"},
//	.d.savePASS={"PASSWORD 1","PASSWORD 2"},
	.d.ipAPI="ip-api.com/line?fields=lat,lon",
	.d.lat=-33.85,	//float not mm.ssaaaa
	.d.lon=151.18,	//float not mm.ssaaaa	
	.d.baudrate=9600,
	.d.talker="GP",
	.d.NTPtimeout=NTP_REFRESH,
	.d.debugOn=0,
	.d.NTPurl={"pool.ntp.org"},
	.d.defYear=2022
};

void set_system_time(uint32_t sec,uint32_t frac){	//called by SNTP when it has a time update
	uint64_t newNTPoffset;
	lastNTPfetch=time_us_64();	//use this for precision
	newNTPoffset=1000000ULL*(sec)+frac-lastNTPfetch;
	printf("\r\n****\r\n NTP adjustment: %lld \r\n****\r\n",newNTPoffset-NTPoffset);
	NTPoffset=newNTPoffset;
    datetime_t dt;
	dt=getDateTime(sec);
    rtc_set_datetime(&dt);		//we don't actually use this, but it could be useful
	NTPok=1;					//set flag
}

datetime_t getDateTime(uint32_t sec){	//from timestamp
	time_t epoch = sec;
    datetime_t dt;
    struct tm *time = gmtime(&epoch);
	dt.year = (int16_t) (1900 + time->tm_year);
    dt.month = (int8_t) (time->tm_mon + 1);
    dt.day = (int8_t) time->tm_mday;
    dt.hour = (int8_t) time->tm_hour;
    dt.min = (int8_t) time->tm_min;
    dt.sec = (int8_t) time->tm_sec;
    dt.dotw = (int8_t) time->tm_wday;
	return dt;
}

char* scanSerial(void){
	int d;
	int f=1;
	if(p==0){b[p]=0;} //if data has been read out, reset
	while(f){
		d=getchar_timeout_us(0);
		if(d==PICO_ERROR_TIMEOUT){
			f=0;
		}else{
			lastWiFiCheck=time_us_64();			//reset timer on any keystroke to avoid printing when user typing
			if(d>=' '){
			  if(p<SCAN_LEN-2){
				b[p]=d;
				p++;
				b[p]=0; //null term
				putchar_raw(d);
			  }
			}else{
			  if(d==BACKSPACE){  //backspace
				if(p){
				  p--;
				  b[p]=0; //delete last
				  putchar_raw(BACKSPACE);    //back up
				  putchar_raw(' ');  		 //blank
				  putchar_raw(BACKSPACE);    //back up again
				}
			  }
			  if(d==13){printf("\r\n");p=0;return b;}
			}
		}
	}
  return 0;
}

void checkMenu(void){
	int i,j,newYear;
	float newLatLon;
	long newBaud;
	long newTimeout;
	int unlocked=0;		//to check that we have released other core
	s=scanSerial();	
	if(s){
		switch(menuState){
			case 0:
				if(s[0]==0){		//pressing Enter on blank line
					printMenu();
				}
				menuState=toupper(s[0]);
				//if(menuState=='~'){reset_usb_boot(0,0);}	//bootloader
				if(menuState<'0'){menuState=0;}
				if((menuState>'9')&&(menuState<'A')){menuState=0;}
				if(menuState>'J'){menuState=0;}
				if(menuState=='0'){  //reprint					
					menuState=0;
					printMenu();
				}else if(menuState=='1'){
				  doNetScan();        //similar to ScanNetworks sketch
				  menuState=0;
				  printMenu();        
				}else if(menuState=='2'){
					showSaved();
					menuState=0;
					printMenu();        
				}else if(menuState=='3'){
					j=s[1]-'0';
					if((j>=0)&&(j<SAVE_COUNT)){
						cur.d.saveSSID[j][0]=0;    //blank
						cur.d.savePASS[j][0]=0;    //blank
						printf("SSID %2 deleted.\r\n",j);
						showSaved();            
					}else{
						printf("Type 3 followed by saved network number.\r\n");
					}
					menuState=0;
					printMenu();        
				}else if(menuState=='4'){
					j=s[1]-'0';
					if((j>=0)&&(j<SCAN_COUNT)){
						for(i=0;i<SAVE_COUNT;i++){
							if(cur.d.saveSSID[i][0]==0){            //null
								strcpy(cur.d.saveSSID[i],scanSSID[j]);
								printf("%2d %-32s\r\n",i,cur.d.saveSSID[i]);
								subMenu=i;							//to put password in correct place
								i=SAVE_COUNT+1;     				//finish loop on success
							}
						}
						if(i==SAVE_COUNT){
							printf("No empty slots, please delete a slot.\r\n");
							showSaved();            
							menuState=0;
							printMenu();        
						}else{
							printf("Added OK\r\n");
							menuState='6';
							printf("Enter password.\r\n");
						}
					}else{
						printf("Type 4 followed by scanned network number.\r\n");
						menuState=0;
						printMenu();        				
					}
				}else if(menuState=='5'){
					j=-1;
					for(i=0;i<SAVE_COUNT;i++){
						if(cur.d.saveSSID[i][0]==0){    //blank
							j=i;
							i=SAVE_COUNT;		//jump out
						}
					}          
					if(j>=0){
						printf("Enter SSID name.\r\n");
						subMenu=j;
					}else{
						printf("No empty slots, please delete a slot.\r\n");
						menuState=0;
						printMenu();        
					}
				}else if(menuState=='6'){
					j=s[1]-'0';
					if((j>=0)&&(j<SAVE_COUNT)){
						if(cur.d.saveSSID[j][0]==0){
							printf("Enter SSID name for %2d first.\r\n",j);
							menuState=0;
							printMenu();        							
						}else{
							cur.d.savePASS[j][0]=0;    //blank
							printf("Enter password.\r\n");
							subMenu=j;
						}
					}else{
						printf("Type 6 followed by saved network number.\r\n");
						menuState=0;
						printMenu();        
					}
				}else if(menuState=='7'){
					printf("Testing networks.\r\n");						
					saveTest();
					menuState=0;
					printMenu();        
				}else if(menuState=='8'){
					printf("Saving to flash.\r\n");						
					sleep_ms(10);		//allow to be printed
					uint32_t x;
					//critical section
					if(multicore_lockout_start_timeout_us(10000000ULL)){	//10s
						x=save_and_disable_interrupts();
						flash_range_erase((uint32_t)&flashimage - (intptr_t)XIP_BASE, FLASH_SECTOR_SIZE);
						flash_range_program((uint32_t)&flashimage - (intptr_t)XIP_BASE, (uint8_t*)&cur, FLASH_SECTOR_SIZE);
						restore_interrupts(x);
						while(!unlocked){
							unlocked=multicore_lockout_end_timeout_us(10000);
						}
					//end critical section
						printf("Save done.\r\n");						
						sleep_ms(10);		//allow to be printed
					}else{
						printf("Save failed, could not lock other core. Please try again.\r\n");											
					}
					menuState=0;
					printMenu();        					
				}else if(menuState=='9'){
					printf("XX	World\r\n");
					printf("AU  Australia\r\n");
					printf("NZ  New Zealand\r\n");
					printf("Enter two-letter country code:\r\n");					
				}else if(menuState=='A'){
					printf("Enter IPAPI URL:\r\n");					
				}else if(menuState=='B'){
					printf("Enter latitude as decimal:\r\n");					
				}else if(menuState=='C'){
					printf("Enter longitude as decimal:\r\n");					
				}else if(menuState=='D'){
					printf("Enter baudrate:\r\n");										
				}else if(menuState=='E'){
					printf("Enter Talker (eg 'GP'):\r\n");										
				}else if(menuState=='F'){
					printf("Enter NTP validity timeout in minutes:\r\n");										
				}else if(menuState=='G'){
					printf("Enter NTP URL:\r\n");					
				}else if(menuState=='H'){
					printf("Enter default year:\r\n");					
				}else if(menuState=='I'){
					if(cur.d.debugOn){
						cur.d.debugOn=0;
					}else{
						cur.d.debugOn=1;
					}
					menuState=0;
					printMenu();        
				}else if(menuState=='J'){
					saveAndReset();
					menuState=0;
				}
				break;
			case '5':
				if(strlen(s)<33){
					strcpy(cur.d.saveSSID[subMenu],s);
					menuState='6';
					printf("SSID saved.\r\n");
					printf("Enter password.\r\n");
				}else{
					printf("Error.\r\n");
					menuState=0;
					printMenu();        
				}
				break;
			case '6':
				if(strlen(s)<33){
					strcpy(cur.d.savePASS[subMenu],s);
					menuState=0;
					printf("Password saved.\r\n");
				}else{
					printf("Error.\r\n");
				}
				menuState=0;
				printMenu();        
				break;
			case '9':
				cur.d.wifiCC[0]=s[0];
				cur.d.wifiCC[1]=s[1];
				cur.d.wifiCC[2]=0;				
				validateCC();
				printf("Country code set to %c%c.\r\n",cur.d.wifiCC[0],cur.d.wifiCC[1]);
				printf("Save to flash and reboot the Pico W Clayton's GPS to reload country code.");
				menuState=0;
				printMenu();        
				break;
			case 'A':
				if((strlen(s)<SCAN_LEN-2)){
					strcpy(cur.d.ipAPI,s);
					printf("Set to %s\r\n",cur.d.ipAPI);
				}else{
					printf("Error with URL (could be too long or too short).\r\n");
				}
				menuState=0;
				printMenu();        
				break;
			case 'B':
				newLatLon=atof(s);
				if((newLatLon!=0.0)&&(newLatLon>=-90)&&(newLatLon<=90)){	//check if valid
					cur.d.lat=newLatLon;
					printf("Set to %f\r\n",cur.d.lat);
				}else{
					printf("Error, invalid value.\r\n");					
				}
				menuState=0;
				printMenu();        
				break;
			case 'C':
				newLatLon=atof(s);
				if((newLatLon!=0.0)&&(newLatLon>=-180)&&(newLatLon<=180)){	//check if valid
					cur.d.lon=newLatLon;
					printf("Set to %f\r\n",cur.d.lon);
				}else{
					printf("Error, invalid value.\r\n");					
				}
				menuState=0;
				printMenu();        
				break;
			case 'D':
				newBaud=atol(s);
				if((newBaud>=300)&&(newBaud<=921600)){	//check if valid
					cur.d.baudrate=newBaud;
					printf("Set to %ld\r\n",cur.d.baudrate);
				}else{
					printf("Error, invalid value.\r\n");					
				}
				menuState=0;
				printMenu();        
				break;
			case 'E':
				cur.d.talker[0]=s[0];
				cur.d.talker[1]=s[1];
				cur.d.talker[2]=0;				
				printf("Talker set to %c%c.\r\n",cur.d.talker[0],cur.d.talker[1]);
				menuState=0;
				printMenu();        
				break;
			case 'F':
				newTimeout=atol(s);
				if((newTimeout>=60)&&(newTimeout<=50000)){	//check if valid, 50k is ~ 1month
					cur.d.NTPtimeout=newTimeout;
					printf("Set to %ld minutes.\r\n",cur.d.NTPtimeout);
				}else{
					printf("Error, invalid value.\r\n");					
				}
				menuState=0;
				printMenu();        
				break;
			case 'G':
				if((strlen(s)<SCAN_LEN-2)&&(strlen(s)>0)){
					strcpy(cur.d.NTPurl,s);
					printf("Set to %s\r\n",cur.d.NTPurl);
					sntp_setservername(0,cur.d.NTPurl);
				}else{
					printf("Error with URL (could be too long or too short).\r\n");
				}
				menuState=0;
				printMenu();        
				break;
			case 'H':
				newYear=atol(s);
				if((newYear>=1970)&&(newYear<=4095)){	//check if valid
					cur.d.defYear=newYear;
					printf("Set to %d\r\n",cur.d.defYear);
				}else{
					printf("Error, invalid value.\r\n");					
				}
				menuState=0;
				printMenu();        
				break;
			default:
				printf("Error.\r\n");
				menuState=0;
				printMenu();        
				break;		
		}
	}
}

void initAll(void){
	stdio_init_all();
	//cyw43_deinit(&cyw43_state);
	//cyw43_arch_deinit();	//this locks up in SDK 1.4.0 and 1.5.0
	cyw43_arch_init_with_country(CYW43_COUNTRY(cur.d.wifiCC[0],cur.d.wifiCC[1],0));
	cyw43_arch_enable_sta_mode();	
    rtc_init();    // Start the RTC
	sntp_setoperatingmode(SNTP_OPMODE_POLL);
	sntp_setservername(0,cur.d.NTPurl);
	sntp_init();
	tInit.year=cur.d.defYear;
    rtc_set_datetime(&tInit);	//reasonable default	
	NTPoffset=31557600000000ULL*(cur.d.defYear-1970);	//average number of us in a year
	if(savedTime>NTPoffset){		//use persistent value if appropriate
		NTPoffset=savedTime;
		printf("\r\nWatchdog Reboot detected.\r\n");  
	}
}

void printMenu(void){
	int wStat;
	ip_addr_t NTPIP;	
	dms temp;
	printf("\r\nWiFi Status: ");  
	wStat=cyw43_tcpip_link_status(&cyw43_state,CYW43_ITF_STA);
	printStat(wStat);
	printf("Menu:\r\n");
	printf("1 : Scan networks\r\n");
	printf("2 : Show saved\r\n");
	printf("3n: Delete SSID (n from saved list)\r\n");
	printf("4n: Set SSID (n from scan list)\r\n");
	printf("5 : Manual SSID\r\n");
	printf("6n: Set Password (n from saved list)\r\n");
	printf("7 : Test saved\r\n");
	printf("8 : Save to flash\r\n");
	printf("9 : Set Country Code (currently %c%c)\r\n",cur.d.wifiCC[0],cur.d.wifiCC[1]);
	printf("A : Set IPAPI URL (%s)\r\n",cur.d.ipAPI);
	temp=getDMS(cur.d.lat);
	printf("B : Set Latitude (%f = %d\xf8%d'%d\"%c)\r\n",cur.d.lat,temp.d,temp.m,temp.s,temp.lat);
	temp=getDMS(cur.d.lon);
	printf("C : Set Longitude (%f = %d\xf8%d'%d\"%c)\r\n",cur.d.lon,temp.d,temp.m,temp.s,temp.lon);
	printf("D : Set baudrate (%ld baud)\r\n",cur.d.baudrate);
	printf("E : Set Talker (currently %c%c)\r\n",cur.d.talker[0],cur.d.talker[1]);	
	printf("F : Set NTP validity timeout (%llu min)\r\n",cur.d.NTPtimeout);	
	printf("G : Set NTP server (%s [%s])\r\n",cur.d.NTPurl,ipaddr_ntoa(sntp_getserver(0)));
	printf("H : Set default year (%d)\r\n",cur.d.defYear);	
	if(cur.d.debugOn){
		printf("I : Turn debug off (currently on).\r\n");
	}else{
		printf("I : Turn debug on (currently off).\r\n");
	}
	printf("J : Reboot Clayton's Pico W GPS Time Source\r\n");	
}

void doNetScan(void){	
	int i;
	for(i=0;i<SCAN_COUNT;i++){
		scanSSID[i][0]=0;	//blank
		scanRSSI[i]=-100;	//invalid
		scanChan[i]=0;		//blank
		scanAuth[i]=0;		//blank
	}
	cyw43_wifi_scan_options_t scan_options = {0};	//not actually used
	cyw43_wifi_scan(&cyw43_state, &scan_options, NULL, scan_result);
	printf("Scanning\r\n");
	while(cyw43_wifi_scan_active(&cyw43_state)){}		//wait; the scan calls scan_result in the background
	printf("Scan complete\r\n");
	sortScanned();			//in order of RSSI
	showScan();
}

void sortScanned(void){			//in order of RSSI
	int i,j,c;
	char tempSSID[SSIDLEN];
	int tempInt;
	j=SCAN_COUNT;
	c=1;
	while(c && j){				//check that we don't get any swaps, should only need SCAN_COUNT cycles at most
		c=0;
		for(i=0;i<SCAN_COUNT-1;i++){
			if(scanRSSI[i]<scanRSSI[i+1]){		//out of order, swap
				strcpy(tempSSID,scanSSID[i]);
				strcpy(scanSSID[i],scanSSID[i+1]);
				strcpy(scanSSID[i+1],tempSSID);
				tempInt=scanRSSI[i];
				scanRSSI[i]=scanRSSI[i+1];
				scanRSSI[i+1]=tempInt;
				tempInt=scanChan[i];
				scanChan[i]=scanChan[i+1];
				scanChan[i+1]=tempInt;
				tempInt=scanAuth[i];
				scanAuth[i]=scanAuth[i+1];
				scanAuth[i+1]=tempInt;
				c++;
			}
		}
		if(j){j--;}				//avoid underflow
	}
}

void showScan(void){
	int i;
	printf("Scanned networks list:\r\n");
	printf(" n SSID                             RSSI Chan Auth\r\n");
	for(i=0;i<SCAN_COUNT;i++){
		if(scanRSSI[i]>-100){
			printf("%2d %-32s %4d %4d %s\r\n",i,scanSSID[i],scanRSSI[i],scanChan[i],(scanAuth[i]?"PASS":"OPEN"));
		}
	}
}

void showSaved(void){
	int i;
	printf("Saved networks list:\r\n");
	for(i=0;i<SAVE_COUNT;i++){
		if(cur.d.saveSSID[i][0]){						//don't show empty
			printf("%2d %-32s\r\n",i,cur.d.saveSSID[i]);
		}
	}
}

//this inpired by wifi-scan demo
static int scan_result(void *env, const cyw43_ev_scan_result_t *result) {
    int i,j;
	if(result){
		for(i=0;i<SCAN_COUNT;i++){
			if(strcmp(result->ssid,scanSSID[i])==0){return 0;}	//matches existing, so skip and return
			if(scanSSID[i][0]==0){								//empty available
				strcpy(scanSSID[i],result->ssid);
				scanRSSI[i]=result->rssi;
				scanChan[i]=result->channel;
				scanAuth[i]=result->auth_mode;
				return 0;										//result processed OK
			}
		}		
	}
    return 0;													//couldn't add to list
}

void saveTest(void){
	int i,j,k,n,wStat;
	k=0;n=0;
	for(i=0;i<SAVE_COUNT;i++){
		if(cur.d.saveSSID[i][0]){
			printf("%2d %-32s >",i,cur.d.saveSSID[i]);
			wStat=cyw43_wifi_leave(&cyw43_state,CYW43_ITF_STA);
			if(cur.d.savePASS[i][0]==0){		//no password => no security
				wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[i],NULL,CYW43_AUTH_OPEN);
				//wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[i],cur.d.savePASS[i],CYW43_AUTH_WPA2_AES_PSK);
			}else{
				wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[i],cur.d.savePASS[i],CYW43_AUTH_WPA2_AES_PSK);
			}
			for(j=0;j<50;j++){
				sleep_ms(300);
				wStat=cyw43_tcpip_link_status(&cyw43_state,CYW43_ITF_STA);
				if((wStat<CYW43_LINK_DOWN)||(wStat>CYW43_LINK_NOIP)){j=9999;}	//definitive result
			}
			printStat(wStat);
			if(wStat==CYW43_LINK_UP){n++;}		//count OK
			wStat=cyw43_wifi_leave(&cyw43_state,CYW43_ITF_STA);
			k++;
		}
	}	
	printf("%4d Networks tested, %2d OK.\r\n",k,n);
}

void checkWIFI(void){
	uint64_t now;
	uint32_t uSecond;
	uint32_t temp;
	static int lastStat=9999;			//unused value to force initial report
	int wStat;	
	now=time_us_64();
	wStat=cyw43_tcpip_link_status(&cyw43_state,CYW43_ITF_STA);
	wifiConnected=(wStat==CYW43_LINK_UP);
	if(((now-lastNTPfetch)/60000000ULL)>cur.d.NTPtimeout){
		if(NTPok){wStat=cyw43_wifi_leave(&cyw43_state,CYW43_ITF_STA);}	//try other networks if NTP not OK
		NTPok=0;
	}
	if(((now-lastNTPfetch)/60000000ULL)>(cur.d.NTPtimeout*2)){			//down for a long time
		saveAndReset();
	}
	ledState=(NTPok?2:0)+(wifiConnected?1:0);	
	if(ledState){
		uSecond=(now/100000ULL)%10;										//tenths of a second
		if(uSecond<(ledState*2)){										//flash when active in relation to ledState
			cyw43_arch_gpio_put(CYW43_WL_GPIO_LED_PIN, (uSecond&1));	//flash LED on
		}else{
			cyw43_arch_gpio_put(CYW43_WL_GPIO_LED_PIN, 0);				//LED off
		}
	}else{							
		cyw43_arch_gpio_put(CYW43_WL_GPIO_LED_PIN, 1);					//LED on solid
	}
	if((now-lastWiFiCheck)>(15000000ULL)){								//15s
		lastWiFiCheck=now;
		if(wStat!=lastStat){											//report any changes
			printStat(wStat);
			lastStat=wStat;
		}	
		temp=stampNow();
		dt=getDateTime(temp);		
		printf("Time is %02d:%02d:%02d on %02d/%02d/%04d. ",dt.hour,dt.min,dt.sec,dt.day,dt.month,dt.year);		
		if(NTPok){
			printf("NTP OK. Last updated %llu minutes ago.\r\n",(now-lastNTPfetch)/60000000ULL);
			failCount=0;	//as long as NTP OK, don't reset
		}else{
			printf("NO NTP.\r\n");
		}
		if(!cyw43_wifi_scan_active(&cyw43_state)){	//if scan not active
			if(wStat!=CYW43_LINK_UP){				//if not connected
				printStat(wStat);
				wStat=cyw43_wifi_leave(&cyw43_state,CYW43_ITF_STA);
				if(currentSSID>=SAVE_COUNT){currentSSID=0;}
				while((cur.d.saveSSID[currentSSID][0]==0)&&(currentSSID<SAVE_COUNT)){
					currentSSID++;		//skip to next, skip any blank until end of list
				}
				if(currentSSID<SAVE_COUNT){
					printf("Connecting to %2d %-32s\r\n",currentSSID,cur.d.saveSSID[currentSSID]);
					if(cur.d.savePASS[currentSSID][0]==0){		//no password => no security
						wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[currentSSID],NULL,CYW43_AUTH_OPEN);
						//wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[currentSSID],cur.d.savePASS[currentSSID],CYW43_AUTH_WPA2_AES_PSK);
					}else{
						wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[currentSSID],cur.d.savePASS[currentSSID],CYW43_AUTH_WPA2_AES_PSK);
					}
					currentSSID++;		//next
				}else{
					//commands to reinit WiFi seem to lock up, so just reboot
					currentSSID=0;
					failCount++;
					printf("Fail count: %2d/%2d\r\n",failCount,FAIL_LIMIT+1);
					if(failCount>FAIL_LIMIT){
						saveAndReset();
					}
					
				}				
			}			
		}
		if(nextIPAPIfetch<now){
			getIPAPI();
		}
	}
}

void printStat(int wStat){
	union ipAddress ipAdd={.ip=0};
	switch(wStat){
		case CYW43_LINK_DOWN:    printf("Link Down\r\n"); break;
		case CYW43_LINK_JOIN:    printf("Connected\r\n"); break;
		case CYW43_LINK_NOIP:    printf("Connected, no IP\r\n"); break;
		case CYW43_LINK_UP :
			ipAdd.ip=ip4_addr_get_u32(&(cyw43_state.dhcp_client.offered_ip_addr));
			printf("Connected with IP: %d.%d.%d.%d\r\n",ipAdd.oct[0],ipAdd.oct[1],ipAdd.oct[2],ipAdd.oct[3]);
			break;
		case CYW43_LINK_FAIL:    printf("Connect failed\r\n"); break;
		case CYW43_LINK_NONET:   printf("SSID not found\r\n"); break;
		case CYW43_LINK_BADAUTH: printf("Auth failed (password?)\r\n"); break;
		default:			     printf("Status unknown\r\n"); break;
	}
}

void initialConnect(void){		//run once on boot to quickly connect if available
	int i,j,wStat;
	doNetScan();
	for(i=0;i<SAVE_COUNT;i++){				//cycle through saved
		if(cur.d.saveSSID[i][0]){			//if not blank
			for(j=0;j<SCAN_COUNT;j++){		//to see if they are available
				if(strcmp(cur.d.saveSSID[i],scanSSID[j])==0){	//match => connect
					printf("Connecting to %2d %-32s\r\n",i,cur.d.saveSSID[i]);
					if(cur.d.savePASS[i][0]==0){		//no password => no security
						wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[i],NULL,CYW43_AUTH_OPEN);
						//wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[i],cur.d.savePASS[i],CYW43_AUTH_WPA2_AES_PSK);
					}else{
						wStat=cyw43_arch_wifi_connect_async(cur.d.saveSSID[i],cur.d.savePASS[i],CYW43_AUTH_WPA2_AES_PSK);
					}
					lastWiFiCheck=time_us_64();	//reset timer to give time to connect
					return;
				}
			}
		}
	}
	printf("No saved networks found.\r\n");
}

void validateCC(void){		//check and validate country code, sets to XX if invalid
	int ccOK=1;	//assume OK and if problems, reset
	cur.d.wifiCC[0]=toupper(cur.d.wifiCC[0]);	//fix lower case
	cur.d.wifiCC[1]=toupper(cur.d.wifiCC[1]);	//fix lower case	
	if(cur.d.wifiCC[0]<'A'){ccOK=0;}			//out of range
	if(cur.d.wifiCC[1]<'A'){ccOK=0;}
	if(cur.d.wifiCC[0]>'Z'){ccOK=0;}
	if(cur.d.wifiCC[1]>'Z'){ccOK=0;}
	if(!ccOK){
		cur.d.wifiCC[0]='X';					//reset
		cur.d.wifiCC[1]='X';
		cur.d.wifiCC[2]=0;
	}
}

void getIPAPI(void){		//initiate IPAPI request
	int i,pSplit;
	if(wifiConnected){
		nextIPAPIfetch=time_us_64()+1000000ULL*60ULL*IPAPI_REFRESH;
		i=strlen(cur.d.ipAPI)-1;
		if(i<0){
			printf("IPAPI disabled.\r\n");
			return;
		}else{
			printf("IPAPI start.\r\n");
		}
		pSplit=0;
		while(i){
			if(cur.d.ipAPI[i]=='/'){pSplit=i;}	//find first /
			i--;								//by scanning backwards
		}
		if(pSplit){
			cur.d.ipAPI[pSplit]=0;
			httpc_get_file_dns(cur.d.ipAPI,HTTP_DEFAULT_PORT,&cur.d.ipAPI[pSplit+1],&httpconset, recv_fn, NULL, &connection);
			cur.d.ipAPI[pSplit]='/';
		}else{
			httpc_get_file_dns(cur.d.ipAPI,HTTP_DEFAULT_PORT,"",&httpconset, recv_fn, NULL, &connection);
		}
	}else{
		printf("Skip IPAPI fetch, no WiFi.\r\n");
	}
}

//these are callbacks used by HTTP calls from httpc_get_file_dns
err_t headers_done_fn(httpc_state_t *connection, void *arg, struct pbuf *hdr, u16_t hdr_len, u32_t content_len){
    printf("Headers of %d bytes report %d bytes of content.\r\n",hdr_len,content_len);	
    return ERR_OK;
}

err_t recv_fn(void *arg, struct tcp_pcb *tpcb, struct pbuf *p, err_t err){
    printf("Received %d bytes.\r\n",p->len);	
    int i;
	int j;
	j=p->len;
	if(j>HTTP_BUF_SIZE-1){j=HTTP_BUF_SIZE-1;}
	for(i=0;i<j;i++){httpbuf[i]=((char*)(p->payload))[i];}
	httpbuf[j]=0;		//ensure null term
	return ERR_OK;
}

void result_fn(void *arg, httpc_result_t httpc_result, u32_t rx_content_len, u32_t srv_res, err_t err){
	printf("HTTP finished:%d\r\n",srv_res);	
	switch(httpc_result){
		case HTTPC_RESULT_OK: printf("OK\r\n"); break;
		case HTTPC_RESULT_ERR_UNKNOWN : printf("Unknown error\r\n"); break;
		case HTTPC_RESULT_ERR_CONNECT : printf("Connection to server failed\r\n"); break;
		case HTTPC_RESULT_ERR_HOSTNAME : printf("Failed to resolve hostname\r\n"); break;
		case HTTPC_RESULT_ERR_CLOSED  : printf("Connection unexpectedly closed\r\n"); break;
		case HTTPC_RESULT_ERR_TIMEOUT  : printf("Server timeout\r\n"); break;
		case HTTPC_RESULT_ERR_SVR_RESP  : printf("Server error\r\n"); break;
		case HTTPC_RESULT_ERR_MEM  : printf("Memory error\r\n"); break;
		case HTTPC_RESULT_LOCAL_ABORT  : printf("Local abort\r\n"); break;
		case HTTPC_RESULT_ERR_CONTENT_LEN  : printf("Content length mismatch\r\n"); break;
		default : printf("Invalid result\r\n"); break;
	}
	if((httpc_result==HTTPC_RESULT_OK)&&(srv_res=200)){
		setLatLon();
	}
}

void setLatLon(void){	//from httpbuf
	//data in httpbuf two decimal numbers separated by CR
	int i,pSplit;
	float newLat,newLon;
	i=strlen(httpbuf)-1;
	if(i<0){return;}
	pSplit=0;
	while(i){
		if(httpbuf[i]<' '){	    		//find 
			pSplit=i;
			httpbuf[i]=0;				//null
		}		
		i--;							//by scanning backwards
	}
	if(pSplit){
		newLat=atof(httpbuf);
		newLon=atof(&httpbuf[pSplit+1]);
		printf("Lat/Lon=%f,%f\r\n",newLat,newLon);
		if((newLat!=0.0)&&(newLat>=-90)&&(newLat<=90)){	//check if valid
			if((newLon!=0.0)&&(newLon>=-180)&&(newLon<=180)){
				cur.d.lat=newLat;
				cur.d.lon=newLon;
			}
		}
	}
}

dms getDMS(float c){
	dms res;
	long sec;
	sec=c*3600;	//integer number of seconds
	if(sec<0){
		res.lat='S';
		res.lon='W';
		sec=abs(sec);
	}else{
		res.lat='N';
		res.lon='E';
	}
	res.s=sec%60;
	res.m=(sec/60)%60;
	res.d=(sec/3600);
	return res;	
}

float getDegrees(dms c){		//convert DMS to float
	float res;
	res=c.s+c.m*60+c.d*3600;
	res=res/3600.0;
	if((c.lat=='S')||(c.lon=='W')){res=-res;}
	return res;
}

void addChecksum(char *s){	//assumes that there is room to add the two characters
	unsigned char checksum=0x00;
	int n=strlen(s)-1;
	int i;
	for(int i=1;i<n;i++){	//leading $ in NMEA is not included
		checksum=checksum^s[i];
	}
	s[n+1]=hex[(checksum>>4)&0xF];
	s[n+2]=hex[(checksum>>0)&0xF];
	s[n+3]=0;		//null terminate
}

uint32_t stampNow(void){		//get current timestamp
	return (NTPoffset+time_us_64())/1000000ULL;		
}

void saveAndReset(void){		//store an offset in persistent RAM and reboot
	savedTime=NTPoffset+time_us_64()+WATCHDOG_REBOOT_OFFSET;	//store time to save over reboot
	watchdog_enable(1, 1);										//software reboot as per SDK docs
	while(1);						
}