/*
  PIC16F15224

  LED1 -> RA0
  LED2 -> RA5
  LED3 -> RC5
  LED4 -> RC3
  LED5 -> RC1
  LED6 -> RA4
  LED7 -> RC4
  LED8 -> RA1

  VR1  -> RA2/ANA2 (speed/direction)
  VR2  -> RC2/ANC2 (brightness/beam width)

  Use Timer0 for main loop (1ms rate)
  Use Timer2 and two PWM modules for PWM
*/

#include "config.h"
#include <xc.h>

#define PIN0 (1<<0)
#define PIN1 (1<<1)
#define PIN2 (1<<2)
#define PIN3 (1<<3)
#define PIN4 (1<<4)
#define PIN5 (1<<5)

typedef struct { unsigned char A; unsigned char C; } ports;
static const ports LEDs[8] = { { PIN0, 0 }, { PIN5, 0 }, { 0, PIN5 }, { 0, PIN3 }, { 0, PIN1 }, { PIN4, 0 }, { 0, PIN4 }, { PIN1, 0 } };

void PinsInit() {
    // LED outputs: RA0, RA1, RA4, RA5, RC1, RC3, RC4, RC5 (initialise to high)
    // ADC inputs: RA2, RC2
    // unused (set as output): RC0
      LATA = 0b00110011;
      LATC = 0b00111010;
    ANSELA = 0b11001100;
    ANSELC = 0b11000101;
     TRISA = 0b00001100;
     TRISC = 0b00000100;
}

void Timer0Init() {
    // set up Timer0 for 4ms interval, setting T0IF
    PIR0 = 0;
    PIE0 = 0;
    TMR0H = 255;
    TMR0L = 0;
    T0CON1 = 0b01000101; // prescaler = 32
    T0CON0 = 0b11100000;
}

void Timer2Init() {
    T2CLKCON = 1; // Fosc/4
    T2HLT = 0b10100000;
    T2PR = 0xFF;
    T2CON = 0b10000000;
    PWM3CON = 0b10010000;
    PWM4CON = 0b10010000;
}

void ADCInit() {
    ADCON0=0;            //reset
    ADCON1= 0b10110000;  //1=right align, 011=ADCRC, 00=NA, 00=V+REF=VDD    
    ADACT=0;             //no auto trigger    
    ADCON0bits.ADON=1;   //turn on    
}

unsigned short ReadADC(unsigned char c){   //get result from selected channel
    ADCON0bits.CHS=c;
    ADCON0bits.ADGO=1;              //start a reading
    while(ADCON0bits.ADGO){}        //wait till done
    return ADRES;
}

void SetRxyPPS(unsigned char PWM1, unsigned char PWM2, unsigned char PWM3, unsigned char PWM4) {
    RA0PPS = (PWM1 == 1 || PWM3 == 1 ? 3 : (PWM2 == 1 || PWM4 == 1 ? 4 : 0));
    RA5PPS = (PWM1 == 2 || PWM3 == 2 ? 3 : (PWM2 == 2 || PWM4 == 2 ? 4 : 0));
    RC5PPS = (PWM1 == 3 || PWM3 == 3 ? 3 : (PWM2 == 3 || PWM4 == 3 ? 4 : 0));
    RC3PPS = (PWM1 == 4 || PWM3 == 4 ? 3 : (PWM2 == 4 || PWM4 == 4 ? 4 : 0));
    RC1PPS = (PWM1 == 5 || PWM3 == 5 ? 3 : (PWM2 == 5 || PWM4 == 5 ? 4 : 0));
    RA4PPS = (PWM1 == 6 || PWM3 == 6 ? 3 : (PWM2 == 6 || PWM4 == 6 ? 4 : 0));
    RC4PPS = (PWM1 == 7 || PWM3 == 7 ? 3 : (PWM2 == 7 || PWM4 == 7 ? 4 : 0));
    RA1PPS = (PWM1 == 8 || PWM3 == 8 ? 3 : (PWM2 == 8 || PWM4 == 8 ? 4 : 0));
}

void SetPWM1(unsigned char delay) {
    PWM3DCH = delay;
}

void SetPWM2(unsigned char delay) {
    PWM4DCH = delay;
}

void UpdateLEDs(unsigned char A, unsigned char C, unsigned char* PWM) {
    LATA = A;
    LATC = C;
    SetRxyPPS(PWM[0], PWM[1], PWM[4], PWM[5]);
    SetPWM1(PWM[2]);
    SetPWM2(PWM[3]);
}

unsigned char lights[8];
unsigned short dir = 512;
unsigned short spread = 384*32;
unsigned char dual_beam = 1;

void update_lights() {
  unsigned char l;
  unsigned short ldir = 0, spr = spread, dim = spread>>5;
  signed short diff, amt;
  if( spr < 256*32 )
      spr = 256*32;
  if( dim > 256 )
      dim = 256;
  for( l = 0; l < 8; ++l ) {
    diff = (signed short)(dir - ldir);
    if( diff < 0 )
      diff = -diff;
    if( diff >= 0 && diff < spr ) {
      if( spr > 255*32 && diff < spr-255*32 )
        lights[l] = 255*dim>>8;
      else
        lights[l] = (unsigned char)((((spr-(unsigned short)diff)>>5)*dim)>>8);
      lights[l] = (unsigned char)((((unsigned short)lights[l]) * ((unsigned short)lights[l])) / 255);
    } else {
      lights[l] = 0;
    }
    ldir += 8192;
  }
  if( dual_beam ) {
      unsigned char lt[8];
      for( l = 0; l < 8; ++l )
          lt[l] = lights[l];
      for( l = 0; l < 8; ++l )
          if( lt[(l+4)&7] + lights[l] > 255 )
              lights[l] = 255;
          else
              lights[l] = lt[(l+4)&7] + lights[l];
  }
}

void update_LEDs() {
  unsigned char A, C, i, PWM[6];
  A = 0xFF;
  C = 0xFF;
  PWM[0] = 0;
  PWM[1] = 0;
  PWM[2] = 0;
  PWM[3] = 0;
  PWM[4] = 0;
  PWM[5] = 0;
  for( i = 0; i < 8; ++i ) {
    if( lights[i] > 127 ) {
      A &= ~LEDs[i].A;
      C &= ~LEDs[i].C;
    }
    if( lights[i] > 0 && lights[i] < 255 ) {
        if( !PWM[0] ) {
            PWM[0] = i+1;
            PWM[2] = lights[i];
        } else if( !PWM[1] ) {
            PWM[1] = i+1;
            PWM[3] = lights[i];
        } else if( PWM[2] == lights[i] ) {
            PWM[4] = i+1;
        } else if( PWM[3] == lights[i] ) {
            PWM[5] = i+1;
        }
    }
  }
  UpdateLEDs(A, C, PWM);
}

unsigned short fix_speed(signed short speed) {
    if( speed < 0 )
        return -((speed/4)*(speed/4))/4;
    else
        return  ((speed/4)*(speed/4))/4;
}

void main(void) {
    unsigned short speed = 512, beam = 512;

    OSCFRQ = 3;//2; // increase clock speed to 8MHz
    
    PinsInit();
    Timer0Init();
    Timer2Init();
    ADCInit();

    while(1) {
        // wait for 4ms tick
        while( !PIR0bits.TMR0IF )
            ;
        PIR0bits.TMR0IF = 0;
        update_lights();
        update_LEDs();
        dir = dir + fix_speed(512-speed);

        speed = (speed*15 + ReadADC(2)) >> 4; // RA2
        beam = (beam*15 + ReadADC(18)) >> 4; // RC2
        if( beam < 512 ) {
            spread = beam * 64;
            dual_beam = 0;
        } else {
            spread = (1023 - beam) * 32;
            dual_beam = 1;
        }
    }
}
