

/*  NodeMCU, OLED 128x64 I2C
    Retrieves current date and time from NTP server
    Receives power data from SolarEdge inverter
    Measures heatsink and HWS temps
    Displays calculated available power, heatsink, HWS temps and HWS amps on OLED screen
    Turns on Opto/mosfet in proportion to the power available, rate limited to 1%/4sec
*/

#include <ESP8266WiFi.h>
#include <WifiUDP.h>
#include <NTPClient.h>
#include <TimeLib.h>
#include "ThingSpeak.h"
#include <ModbusIP_ESP8266.h> //https://github.com/emelianov/modbus-esp8266
#include <ArduinoOTA.h>
#include <Wire.h>
#include <U8g2lib.h> //OLED display
#include <OneWire.h>             //Used by DS18b20 temp sensor
#include <DallasTemperature.h>   //Used by DS18b20 temp sensor
#include<ADS1115_WE.h> 
#include <ESP8266Ping.h>

//ThingSpeak and Wifi login information.
const char* api_key = "XXXXXXXXXXXXXXXX"; //solar diverter channel 
unsigned long myChannelNumber = 123456;
char ssid[] = "YourSSID";
char pass[] = "YourPassw";
const char* Hostname = "SolarDiverter_House";

//WiFi static IP connection


IPAddress gateway(192, 168, 50, 1); // Change to suit your LAN
IPAddress local_IP(192, 168, 50, 183); // Change to suit your LAN. Choose an IP address that is not already used in your LAN
IPAddress subnet(255, 255, 255, 0);
IPAddress primaryDNS(8, 8, 8, 8); 
IPAddress secondaryDNS(8, 8, 4, 4); 
WiFiClient client;

//Modbus connection
uint16_t REG = 30056;   // Modbus reg for Sigenergy phase C pwr. Replace with your Modbus register
uint16_t port = 502; //Sigenergy Modbus port.  Replace with the Modbus port of your inverter. Note: most OEMs use 502!
IPAddress inverter(192, 168, 50, 239);  // Address of my Sigenergy inverter. IP address is made static on my LAN. Replace with yours
ModbusIP mb;  //ModbusIP object

//NTP and time
#define NTP_OFFSET   60 * 60 * 10      // In seconds. UTC + 10 hrs (Brisb)
#define NTP_INTERVAL 60 * 60 * 1000    // In miliseconds
#define NTP_ADDRESS  "au.pool.ntp.org"
WiFiUDP Udp;
NTPClient timeClient(Udp, NTP_ADDRESS, NTP_OFFSET, NTP_INTERVAL);
const char* Lmonth;
unsigned long local;



//Display
U8G2_SSD1309_128X64_NONAME2_F_SW_I2C u8g2(U8G2_R0, 5, 4 ); //No rotation, CLK and SDA pins resp.

//Temp sensors
#define ONE_WIRE_BUS 2   // DS18B20 on GPIO2
float waterTemp = 0;
float hsinkTemp = 0;
OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature DS18B20(&oneWire);

//LDR
#define ldrPin 10

//HWS current
#define I2C_ADDRESS 0x48 //ADS1115 current sensor address
ADS1115_WE adc = ADS1115_WE(I2C_ADDRESS);

//Opto & fan pins
#define Opto 14 
#define fanPin 12  
 
// power variables
int16_t pc; //opto duty cycle in percent. It is the available pwr as a % of max pwr drawn by HWS
int16_t prevpc;
float Watts = 0; // available pwr from inverter (-ve means importing)
float sumWatts = 0;
float avWatts = 0;
const int num_readings = 6; //number of modbus readings
float mVperAmp = 45.43;           // this is my empirically derived cal factor. Datasheet says 40
double Voltage = 0;
double VRMS = 0;
double AmpsRMS = 0;
double maxADCVolt = 4.85; //5V supply measured value
float sumAmps = 0;
float avAmps = 0;
float offset = 0.18; //offset of current sensor. Change to suit

unsigned long currentMillis = 0; 
unsigned long currentMillis2 = 0;

// PushingBox scenario DeviceId code and API
char devId [] = "v5XXXXXXXXXXXXXX"; // Change to yours
char serverName[] = "api.pushingbox.com";

//alarm 
int8_t alarmCount = 0;
bool alarmSent = 0;


void setup() {
 pinMode (ldrPin, INPUT);
 pinMode (Opto, OUTPUT);
 pinMode (fanPin, OUTPUT);
 
 Serial.begin (115200);
 delay (50);
 Serial.println("");
 otaInit();
 ArduinoOTA.handle();
 Wire.begin(4,5); //SDA and CLK resp.

  u8g2.begin(); 
  DS18B20.begin();
   
   // WiFi
    if (!WiFi.config(local_IP, gateway, subnet, primaryDNS, secondaryDNS)) { 
    Serial.println("STA Failed to configure");
    }
  WiFi.hostname(Hostname);
  delay(100);
  
  WiFi.begin(ssid, pass);

    //display 
  u8g2.setPowerSave(0); 
  u8g2.clearBuffer();
  u8g2.setFont(u8g2_font_helvB10_tr); // medium-large bold font
  u8g2.drawStr(10,20,"Connecting");
  u8g2.drawStr(25,40,"to WiFi...");
  u8g2.sendBuffer();
  
while((WiFi.status() != WL_CONNECTED)&& millis() <= 30000) {  //continue if no connection after 30s
    Serial.print(".");
    delay(500);
  }

  if(WiFi.status() != WL_CONNECTED){ 
    ESP.deepSleep(10*1000000, WAKE_RF_DEFAULT); //reboot after 10s
  }
  
  Serial.println(F("connected to local wifi...yeay!"));
  Serial.printf("Hostname: %s\n", WiFi.hostname().c_str());
  Serial.print(F("IP address is: ")); Serial.println(WiFi.localIP());
  Serial.print (F("RRSI: ")); Serial.println(WiFi.RSSI());

  u8g2.clearBuffer(); 
  u8g2.setFont(u8g2_font_helvB10_tr); // medium-large bold font
  u8g2.drawStr(10,20,"Connected to");
  u8g2.drawStr(0,40,ssid);

  IPAddress myip = WiFi.localIP();
  String fullip = String(myip[0]) + "." + myip[1] + "." + myip[2] + "." + myip[3];
  char buf[17];
  fullip.toCharArray(buf, 15);
  u8g2.drawStr(5,60,buf);
  u8g2.sendBuffer();
  
  ThingSpeak.begin(client);  // Initialize ThingSpeak
    
  //Log temps on each wake from sleep
  getTemps();
  sendThingspeakTemps ();

  //Current ADC
  if(!adc.init()){
    Serial.println("ADS1115 not connected!");
  }
  adc.setVoltageRange_mV(ADS1115_RANGE_4096); //measurement range
  adc.setCompareChannels(ADS1115_COMP_0_GND); //compare A0 to GND
  adc.setConvRate(ADS1115_860_SPS); //max 860samples/sec
  adc.setMeasureMode(ADS1115_CONTINUOUS); //continuous mode

  //PWM
  analogWriteRange(100); // % range
    
  //Modbus
  mb.client();
 }


void loop () {
 
  for (int i=1; i<=60; i++){ //do 60 loops of code before writing to data to ThingSpeak (~every 5-min)
  currentMillis = millis();
  Watts = 0;
  pc = 0;
  AmpsRMS = 0;
  
  ArduinoOTA.handle();
       
  //temp Sensors
  getTemps();

// In case of poor solar conditions
      checkTime();
  if (hour(local)==15 && minute(local)==30 && waterTemp<50){ //if water is still cold at 3:30pm, NOTE ALSO: Change these values to check calibration of current sensor (i.e. to turn heater on 100% now).
    u8g2.clearBuffer(); 
    u8g2.setFont(u8g2_font_chroma48medium8_8r);
    u8g2.drawStr(0,18,"Override");
    u8g2.drawStr(0,30," heating ");
    u8g2.sendBuffer();        
    Serial.println(F("Starting override heating"));
      for (int i=0; i<36; i++){ //36 cycles of 5-min = 180-min. Change to suit.
          analogWrite(Opto, 100); //supply 100% power to HWS
          if (hsinkTemp >= 35){ // test if heat sink needs active cooling
           digitalWrite(fanPin, HIGH);
             } else {
             digitalWrite(fanPin, LOW);
              }
          currentMillis = millis();
          while (millis()-currentMillis<5*60*1000){ //wait 5-min
             ArduinoOTA.handle();
             yield();
             }
 
        getTemps();

    // current flowing to HWS
    Voltage = getVPP(ADS1115_COMP_0_GND); //compare AIN0 to GND
    VRMS = (Voltage/2.0) *0.707;   //root 2 is 0.707
    AmpsRMS = ((VRMS * 1000)/mVperAmp-offset); 

    if (AmpsRMS <= 0.05){
      AmpsRMS = 0;
    }
          
   sendThingspeakTemps (); //record temps every 5-min for 2 hrs while heating
        }
  
      digitalWrite (Opto, LOW);
      u8g2.setFont(u8g2_font_chroma48medium8_8r);
      u8g2.drawStr(0,45,"End override");
      u8g2.drawStr(0,57,"heating ");
      u8g2.sendBuffer();
      currentMillis = millis();
            while (millis()-currentMillis<1000){ //delay 1s for display
              ArduinoOTA.handle();
              }
      u8g2.clearDisplay();
      ESP.deepSleep(10*60*1000000, WAKE_RF_DEFAULT); //10-min
  }

       //Check daylight
    byte ldr = digitalRead(ldrPin);

    if (ldr == 0){ // If it's dark, go to sleep 
      Serial.println("Getting dark - going to sleep now...."); 
      u8g2.setPowerSave(1); 
      digitalWrite (Opto, LOW);
      delay(10);
      ESP.deepSleep(10*60*1000000, WAKE_RF_DEFAULT); //10-min
      }

    //Heatsink cooling
     if (hsinkTemp >= 35){ // test if heat sink needs active cooling
     digitalWrite(fanPin, HIGH);
       } else {
       digitalWrite(fanPin, LOW);
        }
  
     getSolar();  //get data from inverter


  //Heating
  Serial.printf("Watts is: %0.02f \r\n", Watts); 
  if (Watts/1000 >= 0.5){ //Only start heating if >=0.5KW pwr available AND we are feeding into the grid
    pc= (Watts-500)*100/3840; // available power as a % of max power demand from HW element (3.84KW), less buffer of 500W
   
      if (pc - prevpc > 0){ 
        pc= prevpc + 2; //slowly increase power to HWS as more power becomes available
              if (pc>=100){
              pc=100;
              }   
      } else{
        pc = prevpc - 2; //slowly decrease power to HWS as solar excess dwindles
              if (pc<0){
              pc=0;
              } 
      } 
      analogWrite(Opto, pc);
      Serial.printf("DAC value for opto is: %i%% \r\n", pc); 
    } else {
    Serial.printf("Insufficient solar available (%.2f KW)\r\n", Watts/1000);
    pc = prevpc - 2;
    if (pc<0){
      pc=0;
      }
    analogWrite(Opto, pc);
    Serial.printf("DAC value for opto is: %i%% \r\n", pc);
     }  
  prevpc = pc;

    // current flowing to HWS
    Voltage = getVPP(ADS1115_COMP_0_GND); //compare AIN0 to GND
    VRMS = (Voltage/2.0) *0.707;   //root 2 is 0.707
    AmpsRMS = ((VRMS * 1000)/mVperAmp-offset); 

    if (AmpsRMS <= 0.05){
      AmpsRMS = 0;
    }
    
    AmpsRMS = AmpsRMS * pc/100;
    Serial.printf("AmpsRMS is: %.2f \r\n", AmpsRMS);
 
    //display 
    u8g2.clearBuffer(); 
    u8g2.setFont(u8g2_font_helvB10_tr);
    u8g2.drawStr(0,18,"Water: ");
    char waterTStr[6];
    dtostrf(waterTemp,4,2,waterTStr); //4 char, 2 dec places
    u8g2.drawStr(65,18,waterTStr);
    u8g2.setFont(u8g2_font_chroma48medium8_8r);
    u8g2.drawStr(105,18,"C");

    u8g2.setFont(u8g2_font_helvB10_tr);
    u8g2.drawStr(0,33,"H'sink: ");
    char sinkTStr[6];
    dtostrf(hsinkTemp,4,2,sinkTStr);
    u8g2.drawStr(65,33,sinkTStr);
    u8g2.setFont(u8g2_font_chroma48medium8_8r);
    u8g2.drawStr(105,33,"C");

    float KW = Watts/1000;
    u8g2.setFont(u8g2_font_helvB10_tr);
    u8g2.drawStr(0,48,"Exc pwr: ");
    char pwrStr[6];
    dtostrf(KW,4,2,pwrStr);
    u8g2.drawStr(65,48,pwrStr);
    u8g2.setFont(u8g2_font_chroma48medium8_8r);
    u8g2.drawStr(105,48,"KW");

    u8g2.setFont(u8g2_font_helvB10_tr);
    u8g2.drawStr(0,63,"HWS: ");
    char HWSStr[5];
    dtostrf(AmpsRMS,4,2,HWSStr);
    u8g2.drawStr(65,63,HWSStr);
    u8g2.setFont(u8g2_font_chroma48medium8_8r);
    u8g2.drawStr(105,63,"A");
    u8g2.sendBuffer();
    

    sumWatts = sumWatts + Watts;
    sumAmps = sumAmps + AmpsRMS;
  
    if (i==60){ //every 5-min (60 loops of 5s), check inverter online, av and send data 
      avWatts = sumWatts/60;
      avAmps = sumAmps/60;

  // check if solar inverter is online
  Serial.print(F("Pinging ip ")); Serial.println(inverter);
  Serial.print(F("alarmCount :")); Serial.println(alarmCount);
  Serial.print(F("alarmSent :")); Serial.println(alarmSent);

  if(Ping.ping(inverter)){
    Serial.println("Inverter is UP!");
      if (alarmCount > 0){ //reset alarmCount if inverter is UP
        alarmCount = 0;
        Serial.println("alarmCount reset to 0");  
        }
      }
   else // increment alarm Count
    {
      Serial.println("Error: Inverter can't be reached!");   
      if (alarmCount < 3){
      alarmCount++;
      }
      if ((alarmCount == 3) && (alarmSent == false)){ // only send alarm if the inverter can't be reached for >=15min
        Serial.println("alarmCount trigger of 3 reached!");
        delay (50);
        sendToPushingBox(devId);
        alarmSent = true; // this is reset on power-up after sleep
      }
     }//end increment alarmCount

// Send data to ThingSpeak
     sendThingspeakAll(); 
      sumWatts = 0;
      avWatts = 0;
      sumAmps = 0;
      avAmps = 0;
   } //end av and send data
   
  while(millis()- currentMillis<5000){ //loop delay 5s
  ArduinoOTA.handle(); // monitor for OTA updates while on hold
  yield();
      }
         
   Serial.print("Looptime after delay: ");
   Serial.println(millis()-currentMillis);
    } //end TS "for" loop
} //end program loop

// -------- Functions ----------
void getTemps() {
     //temp Sensors
  DS18B20.requestTemperatures();
  waterTemp = DS18B20.getTempCByIndex(1); // temperature reading of first temp sensor
  Serial.printf("Water temp is: %.2f \r\n", waterTemp); 
  while (waterTemp==-127.00){
  waterTemp = DS18B20.getTempCByIndex(1); // keep sampling temp until a proper reading is obtained
  }

  hsinkTemp = DS18B20.getTempCByIndex(0); // temperature reading of 2nd temp sensor
  Serial.printf("Heatsink temp is: %.2f \r\n", hsinkTemp);
  while (hsinkTemp==-127.00){
    hsinkTemp = DS18B20.getTempCByIndex(0); // keep sampling temp until a proper reading is obtained
  }
}

void sendThingspeakTemps (){
  float Amps = AmpsRMS; //convert to float for ThingSpeak
  ThingSpeak.setField(1, waterTemp);
  ThingSpeak.setField(2, hsinkTemp);
  ThingSpeak.setField(4, Amps);
  int x = ThingSpeak.writeFields(myChannelNumber, api_key);
  if(x == 200){
  Serial.println("Channel update successful.");
  delay(5);
}
  else{
    Serial.println("Problem updating channel. HTTP error code " + String(x));
  }
}

void sendThingspeakAll () {
  ThingSpeak.setField(1, waterTemp);
  ThingSpeak.setField(2, hsinkTemp);
  ThingSpeak.setField(3, avWatts/1000);
  ThingSpeak.setField(4, avAmps);
  
 // write to the ThingSpeak channel
  int x = ThingSpeak.writeFields(myChannelNumber, api_key);
  if(x == 200){
    Serial.println("Channel update successful.");
  }
  else{
    Serial.println("Problem updating channel. HTTP error code " + String(x));
  }
}

void checkTime() { //used to heat if the water is still cold after 8pm and reset alarm flag
  if (WiFi.status() == WL_CONNECTED) //Check WiFi connection status
  {  
    label:
    // update the NTP client and get the UNIX UTC timestamp 
    timeClient.update();
    local =  timeClient.getEpochTime();
    Serial.println(local); 
    if (year(local)==1970){ //test if invalid time is returned
      currentMillis2 = millis();
      while (currentMillis2 + 500> millis()){}; //delay 500ms
      goto label;
    }

      if (month(local)==1){Lmonth = "Jan";}
else if (month(local)==2){Lmonth = "Feb";}
else if (month(local)==3){Lmonth = "Mar";}
else if (month(local)==4){Lmonth = "Apr";}
else if (month(local)==5){Lmonth = "May";}
else if (month(local)==6){Lmonth = "Jun";}
else if (month(local)==7){Lmonth = "Jul";}
else if (month(local)==8){Lmonth = "Aug";}
else if (month(local)==9){Lmonth = "Sep";}
else if (month(local)==10){Lmonth = "Oct";}
else if (month(local)==11){Lmonth = "Nov";}
else if (month(local)==12){Lmonth = "Dec";}

    char timeString[32];
    sprintf(timeString, "Local time: %02d-%s-%02d %02d:%02d:%02d", year(local), Lmonth, day(local), hour(local), minute(local), second(local));
    Serial.println(timeString); 
    
  } //end WiFi.status 
    else // attempt to connect to wifi again if disconnected
  {
    WiFi.begin(ssid, pass);
      }
}

// Modbus Transaction callback
bool cb(Modbus::ResultCode event, uint16_t transactionId, void* data) { 
  if (event != Modbus::EX_SUCCESS)                  // If transaction got an error
    Serial.printf("Modbus result: %02X\n", event);  // Display Modbus error code
  if (event == Modbus::EX_TIMEOUT) {    // If Transaction timeout took place, response is E4. Transactions cancelled is E6
    mb.disconnect(inverter);              // Close connection to slave and
    mb.dropTransactions();              // Cancel all waiting transactions
  }
  return true;
}

void getSolar() {
 mb.connect(inverter, port);
 uint16_t res[2]={0,0};
    
    if (mb.isConnected(inverter)) {   // Check if connection to Modbus Slave is established
    mb.readHreg(inverter, REG, (uint16_t*)res,2,cb,247);  // Initiate Read from 2 consecutive registers from Modbus Slave ID 247
  } else {
    Serial.println("Modbus slave NOT connected");  
    mb.connect(inverter, port);           // Try to connect if no connection
    }
  mb.task();   // Common local Modbus task
  Watts = res[1];
    if (Watts>40000){ 
    Watts = Watts - 65535;
    }

   ArduinoOTA.handle();
   
   Serial.printf ("Phase C Real Power: %.02f KW (-ve is Export for Sigenergy)\r\n", Watts/1000);
   Watts = -Watts; // use only if -ve means export in your Modbus protocol
}

//Current sensor
float getVPP(ADS1115_MUX channel)
{
  float result;
  int readValue;                // value read from the sensor
  int maxValue = 0;             // store max value here
  int minValue = 4096;          // ADC resolution

    uint32_t start_time = millis();
   while((millis()-start_time) < 2000) //sample for 2s or ~100 wave cycles
   {
    adc.setCompareChannels(channel);
       readValue = adc.getResult_mV();
        maxValue = (readValue > maxValue) ? readValue : maxValue; //if readValue is new max, record new maxValue 
        minValue = (readValue < minValue) ? readValue : minValue; //if readValue is new min, record new minValue 
     }

    // Subtract min from max
   result = ((maxValue - minValue) * maxADCVolt)/4096.0; // This is the peak-to-peak voltage (Vpp)
      
   return result;
 }

// alarm function
 void sendToPushingBox(char devId[]){ 
  client.stop(); 
  Serial.println("connecting...");
  if(client.connect(serverName, 80)) {
    Serial.println("connected");
    Serial.println("sending request to PushingBox");
    client.print("GET /pushingbox?devid=");
    client.print(devId);
    client.println(" HTTP/1.1");
    client.print("Host: ");
    client.println(serverName);
    client.println("User-Agent: Arduino");
    client.println();
    delay (10000);
  } 
  else {
    Serial.println("connection failed");
  } 
}

void otaInit(){
  //OTA
  ArduinoOTA.setHostname(Hostname);
  ArduinoOTA.onStart([]() {
    String type;
    if (ArduinoOTA.getCommand() == U_FLASH) {
      type = "sketch";
    } else { // U_FS
      type = "filesystem";
    }

    Serial.println("Start updating " + type);
  });
  ArduinoOTA.onEnd([]() {
    Serial.println("\nEnd");
  });
  ArduinoOTA.onProgress([](unsigned int progress, unsigned int total) {
    Serial.printf("Progress: %u%%\r", (progress / (total / 100)));
  });
  ArduinoOTA.onError([](ota_error_t error) {
    Serial.printf("Error[%u]: ", error);
    if (error == OTA_AUTH_ERROR) {
      Serial.println("Auth Failed");
    } else if (error == OTA_BEGIN_ERROR) {
      Serial.println("Begin Failed");
    } else if (error == OTA_CONNECT_ERROR) {
      Serial.println("Connect Failed");
    } else if (error == OTA_RECEIVE_ERROR) {
      Serial.println("Receive Failed");
    } else if (error == OTA_END_ERROR) {
      Serial.println("End Failed");
    }
  });
  ArduinoOTA.begin();
}
