#include "irtx.h"
#include "util.h"
#include "io.h"

volatile unsigned char pulseCount=0;    //this is internal to ISR
volatile unsigned char seqPos=0;        //ISR
unsigned char pulseLengths[IR_SEQ_LEN];        //duration in ticks (depends on protocol)
unsigned char pulseOutput[IR_SEQ_LEN];         //output, makes Manchester easier to encode, allows lengths over 255
unsigned int pulseTotal=0;
volatile unsigned char seqLen=0;
volatile unsigned char irState=IR_IDLE;
volatile unsigned char irCmd=IR_STOP;
const unsigned char bitMask[8]={1,2,4,8,16,32,64,128}; //used in getBit
irCode_t activeCode={0,0,0};
union {        //access as bytes etc
    unsigned char b[4];
    unsigned long u;
} codeBytes;
char toggleBit=0;       //use global

//function to check validity of enum
#define X(a, b) case a:
char isValidIRtype(irTypes_t t){
    switch(t){
        ENUM_VALUES
            return 1;
            break;
        default:
            return 0;
            break;
    }
}
#undef X

void txPinInit(void){   //do this once
    unsigned char intcon;
    anselbits.IR_TX_PIN=0;
    trisbits.IR_TX_PIN=1;   //off, ready to be controlled
    latbits.IR_TX_PIN=0;    //assume active high    
    intcon=INTCON;
    INTCONbits.GIE=0; //Suspend interrupts
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 0; //clear PPSLOCKED bit
    pps.IR_TX_PIN=3;    //PWM3
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 1; //set PPSLOCKED bit       
    INTCON=intcon;    
}

void irTxInit(unsigned char per, unsigned char post){
    //PWM3 and TMR2
    T2CON=0;    //reset
    T2CLKCON=1;             //FOSC/4
    T2CONbits.T2CKPS=0;     //8MHz
    T2CONbits.T2OUTPS=post-1;    //postscale is n+1
    T2HLT=0;                //free running
    TMR2=0;
    PR2=per-1;            //custom period
    T2CONbits.TMR2ON=1;    
    PWM3CON=0;  //off, polarity normal
    PWM3DCL=0;
    PWM3DCH=per/3; //1/3 duty as suggested for most receivers
    PIR1bits.TMR2IF=0;
    PIE1bits.TMR2IE=1;
    INTCONbits.PEIE=1;
    PWM3CONbits.EN=1;   //on    
}

void irTxDeInit(void){  //shut down
    T2CON=0;    //reset
    PWM3CON=0;  //off, polarity normal    
    INTCONbits.PEIE=0;  //no interrupts
}

void tmr2init(void){
    //TIMER2
    T2CON=0;    //reset
    //T2CLKCON=5;             //500kHz MFINTOSC
    T2CLKCON=1;             //FOSC/4
    T2CONbits.T2CKPS=7;     //128 62500kHz, 16us
    T2CONbits.T2OUTPS=0;    //postscale is n+1
    T2HLT=0;                //free running
    TMR2=0;
    PR2=255;                //full period
    PIR1bits.TMR2IF=0;
    PIE1bits.TMR2IE=1;
    INTCONbits.PEIE=1;
    T2CONbits.TMR2ON=1;    
}

char getBit(unsigned char *d, unsigned char i){  //get bit, assuming LSB first
    char n,j;
    n=i>>3;
    j=i&7;
    if(d[n]&bitMask[j]){
        return 1;
    }else{
        return 0;
    }
}

unsigned char bitRev(unsigned char b){
    unsigned char r=0;
    if(b&128){r=r|  1;}
    if(b& 64){r=r|  2;}
    if(b& 32){r=r|  4;}
    if(b& 16){r=r|  8;}
    if(b&  8){r=r| 16;}
    if(b&  4){r=r| 32;}
    if(b&  2){r=r| 64;}
    if(b&  1){r=r|128;}
    return r;
}

unsigned char IRencode(irCode_t c){ //encode into pulseLengths etc
    seqLen=0;
    pulseTotal=0;
    activeCode=c;       //store for use
    switch(c.irType){
        case IR_NEC:encodeNEC();break;
        case IR_SIRC_12:
        case IR_SIRC_15:
        case IR_SIRC_20:
            encodeSIRC();break;
        case IR_RC5:encodeRC5();break;
        case IR_RC6:encodeRC6();break;
        default: break; //zeroed initially, returns fail
    }
    return seqLen;
}

void encodeRC6(void){    //from activeCode; protocol specific code
    signed char i;
    addPulse(RC6_START,1);
    addPulse(RC6_TOGGLE,0);
    manchesterBit(1,RC6_PULSE); //fixed 1 bit
    manchesterBit(0,RC6_PULSE);
    manchesterBit(0,RC6_PULSE);
    manchesterBit(0,RC6_PULSE); //3 mode bita
    if(toggleBit){
        manchesterBit(1,RC6_TOGGLE);
        toggleBit=0;
    }else{
        manchesterBit(0,RC6_TOGGLE);
        toggleBit=1;        
    }
    codeBytes.b[0]=(unsigned char)activeCode.data;  //8-bits
    codeBytes.b[1]=(unsigned char)activeCode.address; //8-bits
    for(i=15;i>=0;i--){ //reverse order MSB first
        manchesterBit(getBit(codeBytes.b,(unsigned char)i),RC6_PULSE);
    }
    padToInterval(RC6_INTERVAL);
}

void encodeRC5(void){    //from activeCode; protocol specific code
    signed char i;
    //starts with 2 1 bits
    manchesterBitInv(1,RC5_PULSE);
    manchesterBitInv(1,RC5_PULSE);
    if(toggleBit){
        manchesterBitInv(1,RC5_PULSE);
        toggleBit=0;
    }else{
        manchesterBitInv(0,RC5_PULSE);
        toggleBit=1;        
    }
    codeBytes.u=0;
    codeBytes.b[0]=activeCode.data&0x3F;  //6 bits only
    codeBytes.u=codeBytes.u|((activeCode.address&0x1F)<<6);  //5 bits of address
    for(i=10;i>=0;i--){ //reverse order MSB first
        manchesterBitInv(getBit(codeBytes.b,(unsigned char)i),RC5_PULSE);
    }
    padToInterval(RC5_INTERVAL);
}

void encodeSIRC(void){    //from activeCode; protocol specific code
    char i,j;
    addPulse(SIRC_START,1);
    addPulse(SIRC_SHORT,0);
    codeBytes.u=0;
    codeBytes.b[0]=activeCode.data&0x7F;  //7 bits only, same for all versions
    switch(activeCode.irType){
        case IR_SIRC_12:
            codeBytes.u=codeBytes.u|((activeCode.address&0x1F)<<7);  //5 bits of address
            j=12;
            break;
        case IR_SIRC_15:
            codeBytes.u=codeBytes.u|((activeCode.address&0xFF)<<7);  //8 bits of address
            j=15;
            break;
        case IR_SIRC_20:
            codeBytes.u=codeBytes.u|(((unsigned long)(activeCode.address&0x1FFF))<<7);  //13 bits of address
            j=20;
            break;
        default:    //handle as though 12-bit
            codeBytes.u=codeBytes.u|((activeCode.address&0x1F)<<7);  //5 bits of address
            j=12;
            break;
    }
    for(i=0;i<j;i++){
        if(getBit(codeBytes.b,i)){
            addPulse(SIRC_LONG,1);
        }else{
            addPulse(SIRC_SHORT,1);
        }                
        addPulse(SIRC_SHORT,0);
    }
    padToInterval(SIRC_INTERVAL);
}

void encodeNECrepeat(void){     //special repeat sequence
    seqLen=0;
    pulseTotal=0;
    addPulse(NEC_START_MARK,1);
    addPulse(NEC_REPEAT_SPACE,0);            
    addPulse(NEC_SHORT,1);
    padToInterval(NEC_INTERVAL);    
}

void encodeNEC(void){    //from activeCode; protocol specific code
    char i;
    addPulse(NEC_START_MARK,1);
    addPulse(NEC_START_SPACE,0);    
    codeBytes.b[0]=(unsigned char)activeCode.address;
    codeBytes.b[1]=(unsigned char)(activeCode.address^0xFF); //invert for checksum
    codeBytes.b[2]=(unsigned char)activeCode.data;
    codeBytes.b[3]=(unsigned char)(activeCode.data^0xFF);    //invert for checksum
    for(i=0;i<32;i++){
        addPulse(NEC_SHORT,1);
        if(getBit(codeBytes.b,i)){
            addPulse(NEC_LONG,0);    
        }else{
            addPulse(NEC_SHORT,0);    
        }                
    }        
    addPulse(NEC_SHORT,1);
    padToInterval(NEC_INTERVAL);
}

void padToInterval(unsigned int t){     //fill sequence with low until t
    while(pulseTotal<t){
        if((t-pulseTotal)<250){
            addPulse((unsigned char)(t-pulseTotal),0);    
        }else{
            addPulse(250,0);                
        }
    }    
}

void addPulse(unsigned char d, char b){ //duration, bit
    pulseTotal=pulseTotal+d;
    pulseLengths[seqLen]=d;
    pulseOutput[seqLen++]=b;    
}

void manchesterBit(char c, unsigned char t){  //encode bit c with length t
    if(c){
        addPulse(t,1);    
        addPulse(t,0);
    }else{
        addPulse(t,0);
        addPulse(t,1);    
    }    
}

void manchesterBitInv(char c, unsigned char t){  //encode bit c with length t (sense of RC5 is reverse to RC6)
    if(c){
        addPulse(t,0);    
        addPulse(t,1);
    }else{
        addPulse(t,1);
        addPulse(t,0);    
    }    
}

void __interrupt() isr(void){
    if(PIR1bits.TMR2IF && PIE1bits.TMR2IE){
        //trisbits.S3=0;  //instrumenting
        PIR1bits.TMR2IF=0;
        if(irState){    //running/repeating/starting
            pulseCount++;
            if(pulseCount>=pulseLengths[seqPos]){
                pulseCount=0;
                seqPos++;
                if(seqPos<seqLen){
                    if(pulseOutput[seqPos]){
                        trisbits.IR_TX_PIN=0;
                        setLED(batLED);
                    }else{
                        trisbits.IR_TX_PIN=1;
                        setLED(OFF);
                    }
                }else{  //finished first, top or set up repeat data
                    if(irCmd==IR_STOP){
                        irState=IR_IDLE;
                        seqPos=0;
                        pulseCount=0;
                    }else{
                        seqPos=0;
                        pulseCount=0;
                        if(pulseOutput[0]){
                            trisbits.IR_TX_PIN=0;
                            setLED(batLED);
                        }else{
                            trisbits.IR_TX_PIN=1;
                            setLED(OFF);
                        }
                        if(irState==IR_RUNNING){
                            //set up repeat
                            if(activeCode.irType==IR_NEC){encodeNECrepeat();}
                            irState=IR_REPEATING;   //ready to repeat                        

                        }                        
                    }
                }
            }
        }else{          //idle => start
            if(irCmd!=IR_STOP){
                irState=IR_RUNNING;
                seqPos=0;
                pulseCount=0;
                if(pulseOutput[0]){
                    trisbits.IR_TX_PIN=0;
                    setLED(batLED);
                }else{
                    trisbits.IR_TX_PIN=1;
                    setLED(OFF);
                }
            }            
        }        
        //trisbits.S3=1;  //instrumenting
    }   
}

