#include "st25dv.h"
//RW SMB (aka I2C) for Tag chips
char ioBuf[8]="";       //local use
unsigned int eeSize=0;
unsigned int ndefSize=0;
unsigned int payloadPtr=0;
unsigned int payloadSize=0;
unsigned int textPtr=0;
unsigned int textSize=0;
char ndefText[NDEF_TEXT_SIZE]="";
const char ndefHead[NDEF_HEAD_SIZE]={ //naive 4byte CC, short record NDEF text type
    0xE1,0x40,0x40,0x00,
    0x03,0x00,0xD1,0x01,    //#5 is NDEF size=text size+7
    0x00,0x54,0x02,0x65,    //#8 is payload size=text size+3
    0x6E
};

void SMstop(void){
    trisbits.ST25DV_SDA    =0; //ensure SDA low
    trisbits.ST25DV_SCL    =1; //SCL hi
    trisbits.ST25DV_SDA    =1; //then SDA hi    
}

void SMstart(void){
    trisbits.ST25DV_SDA    =0; //SDA low
    trisbits.ST25DV_SCL    =0; //SCL low    
}

void SMinit(void){
    anselbits.ST25DV_SDA   =0; //digital
    anselbits.ST25DV_SCL   =0; //digital
    trisbits.ST25DV_SDA    =1; //input= pulled up by bus
    trisbits.ST25DV_SCL    =1; //input= pulled up by bus
    wpubits.ST25DV_SDA     =1; //safety pullups
    wpubits.ST25DV_SCL     =1; //safety pullups
    latbits.ST25DV_SDA     =0; //low when driven
    latbits.ST25DV_SCL     =0; //low when driven
#ifdef ST25DV_PWR
    anselbits.ST25DV_PWR   =0; //digital
    trisbits.ST25DV_PWR    =0; //output
    latbits.ST25DV_PWR     =1; //high
    __delay_ms(1);
#endif
    SMstop();      //idle state    
}

void SMdeInit(void){        //for shutdown
#ifdef ST25DV_PWR
    anselbits.ST25DV_PWR   =0; //digital
    trisbits.ST25DV_PWR    =0; //output
    latbits.ST25DV_PWR     =0; //high
#endif
    latbits.ST25DV_SDA     =0; //low when driven
    latbits.ST25DV_SCL     =0; //low when driven    
    trisbits.ST25DV_SDA    =0; //SDA low
    trisbits.ST25DV_SCL    =0; //SCL low    
    wpubits.ST25DV_SDA     =0;
    wpubits.ST25DV_SCL     =0;
}

smByte_t SMbyte(const smByte_t b){
    smByte_t ret;
    ret.d=0;
    ret.a=0;
    char m=128;
    while(m){
        if(b.d&m){trisbits.ST25DV_SDA=1;}else{trisbits.ST25DV_SDA=0;}
        trisbits.ST25DV_SCL=1;    
        SMbitdelay();
        if(portbits.ST25DV_SDA){ret.d=ret.d|m;} //read bit here
        trisbits.ST25DV_SCL=0;
        m=m>>1;
    }
    if(b.a){trisbits.ST25DV_SDA=1;}else{trisbits.ST25DV_SDA=0;}
    trisbits.ST25DV_SCL=1;    
    SMbitdelay();
    if(portbits.ST25DV_SDA){ret.a=1;}
    trisbits.ST25DV_SCL=0;    
    return ret;
}

void SMbitdelay(void){ //adjust to suit processor speed, bus speed etc
    NOP();
    NOP();
    NOP();
    NOP();
    NOP();    
}

smByte_t ST25DVreadByte(unsigned int a){  //read a single byte at a
    smByte_t ret,in,out;
    ret.d=0;
    ret.a=ST25DV_NAK;
    SMstart();
    out.a=ST25DV_NAK;
    out.d=ST25DV_WRITE(ST25DV_USRMEM);  //I2C bus address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return ret;}
    out.d=(a>>8)&0xFF;  //high address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return ret;}
    out.d=(a>>0)&0xFF;  //low address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return ret;}    
    SMstop();
    SMstart();
    out.d=ST25DV_READ(ST25DV_USRMEM);  //I2C bus address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return ret;}
    out.d=0xFF;                //idle high for read
    in=SMbyte(out);    
    SMstop();
    ret.d=in.d;
    if(in.a==ST25DV_NAK){   //NAK expected for normal read
        ret.a=ST25DV_ACK;
    }else{
        ret.a=ST25DV_NAK;        
    }    
    return ret;    
}

char ST25DVreadBytes(unsigned int a, char* p, unsigned int n){  //read n bytes into p from address a, returns true on success
    smByte_t in,out;
    unsigned int i=0;
    SMstart();
    out.a=ST25DV_NAK;
    out.d=ST25DV_WRITE(ST25DV_USRMEM);  //I2C bus address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=(a>>8)&0xFF;  //high address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=(a>>0)&0xFF;  //low address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}    
    SMstop();
    SMstart();
    out.d=ST25DV_READ(ST25DV_USRMEM);  //I2C bus address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=0xFF;                //idle high for read
    out.a=ST25DV_ACK;              //send acks to allow reading to continue
    while(i<n){
        if(i==(n-1)){out.a=ST25DV_NAK;} //flag that this is last
        in=SMbyte(out);    
        p[i]=in.d;
        i++;
    }
    SMstop();
    return 1;    
}

char ST25DVscanNDEF(void){        //process NDEF and put data into globals
    unsigned int cPtr=0;    
    unsigned int hSize=0;   //CC and TLV header
    union {ndefRec_t bits; char byte;} ndefRec;
    unsigned int typeLen=0;
    unsigned int payLen=0;
    unsigned int idLen=0;
    unsigned int textOffset=0;
    char notDone=4;
    //reset all
    eeSize=0;
    ndefSize=0;
    payloadPtr=0;
    payloadSize=0;
    textPtr=0;
    textSize=0;
    if(ST25DVreadBytes(0,ioBuf,8)==0){return 0;}    //look for CC
    if(ioBuf[0]==0xE1){ //short CC
        cPtr=4;
        eeSize=ioBuf[2]*8;
    }else if(ioBuf[0]==0xE2){ //long CC
        cPtr=8;
        eeSize=ioBuf[6]*2048+ioBuf[7]*8;
    }else{
        return 0;           //unknown
    }
    if(ST25DVreadBytes(cPtr,ioBuf,4)==0){return 0;} //look for TLV header for NDEF message
    if(ioBuf[0]!=0x03){return 0;}           //NDEF TLV
    if(ioBuf[1]==0xFF){
        cPtr=cPtr+4;
        ndefSize=ioBuf[2]*256+ioBuf[3];        
    }else{
        cPtr=cPtr+2;
        ndefSize=ioBuf[1];
    }
    hSize=cPtr;     //size of header processed so far
    while(notDone){
        if(cPtr>(hSize+ndefSize)){return 0;}         //scanned 
        if(ST25DVreadBytes(cPtr,ioBuf,8)==0){return 0;} //look for NDEF record
        ndefRec.byte=ioBuf[0];      //record type
        if((ndefRec.bits.tnf==0)||(ndefRec.bits.tnf==7)){return 0;} //no text record found    
        typeLen=ioBuf[1];           //one byte
        cPtr++;                     //after record type
        cPtr++;                     //after type length
        if(ndefRec.bits.sr){
            payLen=ioBuf[2];
            cPtr++;                 //after payload length
            if(ndefRec.bits.il){
                idLen=ioBuf[3];
                cPtr++;             //after ID length
            }    
        }else{
            payLen=ioBuf[5]+256*ioBuf[4];   //only handle up to 16bits of length...
            cPtr=cPtr+4;            //after payload length
            if(ndefRec.bits.il){
                idLen=ioBuf[6];
                cPtr++;             //after ID length
            }    
        }    
        if((ndefRec.bits.tnf==1)&&(typeLen==1)){    //NFC type, might be text
            if(ST25DVreadBytes(cPtr,ioBuf,8)==0){return 0;} //look for type
            if(ioBuf[0]=='T'){
                payloadPtr=cPtr+typeLen;
                payloadSize=payLen;
                if((ioBuf[1]&0x80)==0){             //is UTF-8
                    textOffset=(ioBuf[1]&0x3F)+1;   //skip language encoding
                    textPtr=payloadPtr+textOffset;  //less text encoding header
                    textSize=payLen-textOffset;                
                    for(cPtr=0;cPtr<NDEF_TEXT_SIZE;cPtr++){ndefText[cPtr]=0;}   //blank
                    if(textSize>NDEF_TEXT_SIZE){textSize=NDEF_TEXT_SIZE;}       //cap
                    if(ST25DVreadBytes(textPtr,ndefText,textSize)==0){return 0;} //load text                    
                    return 1;
                }else{
                    return 0;
                }
            }else{
                cPtr=cPtr+typeLen+payLen+idLen; //a different type, skip to next
            }
        }else{
            cPtr=cPtr+typeLen+payLen+idLen; //a different type, skip to next
        }
        if(ndefRec.bits.me){return 0;}      //message end found without text
        if(notDone){notDone--;}             //bound number of loops 
    }
    return 0;   // no result???
}

char SMprobe(char a){   //check for ack/nak of address a on bus, true for ack
    smByte_t in,out;
    SMstart();
    out.a=ST25DV_NAK;
    out.d=ST25DV_WRITE(a);  //I2C bus address
    in=SMbyte(out);
    SMstop();
    if(in.a==ST25DV_NAK){
        return 0;
    }else{
        return 1;        
    }
}

char ST25DVwriteByte(unsigned int a,const char d){    //write a single byte, true for success
    smByte_t in,out;
    SMstart();
    out.a=ST25DV_NAK;
    out.d=ST25DV_WRITE(ST25DV_USRMEM);  //I2C bus address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=(a>>8)&0xFF;  //high address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=(a>>0)&0xFF;  //low address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}    
    out.d=d;            //data
    in=SMbyte(out);
    SMstop();
    if(in.a==ST25DV_NAK){
        return 0;
    }    
    //check for write completion
    __delay_ms(6);      //nominal programming time is 5ms
    return SMprobe(ST25DV_USRMEM);
}

char ST25DVwrite4Byte(unsigned int a,const char* d,char n){    //write n<=4 bytes (in a single page), true for success
    smByte_t in,out;
    char p=0;
    if(n<1){return 0;}                               //no point writing 0 bytes
    if(n>4){return 0;}                               //this can only handle 4
    if((a&0xFFFC)!=((a+n-1)&0xFFFC)){return 0;}      //not all in same page
    SMstart();
    out.a=ST25DV_NAK;
    out.d=ST25DV_WRITE(ST25DV_USRMEM);  //I2C bus address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=(a>>8)&0xFF;  //high address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}
    out.d=(a>>0)&0xFF;  //low address
    in=SMbyte(out);
    if(in.a==ST25DV_NAK){SMstop();return 0;}    
    while(p<n){
        out.d=d[p];            //data
        in=SMbyte(out);
        if(in.a==ST25DV_NAK){SMstop();return 0;}    
        p++;
    }
    SMstop();
    //check for write completion
    __delay_ms(6);      //nominal programming time is 5ms
    return SMprobe(ST25DV_USRMEM);    
}

char ST25DVwriteBytes(unsigned int a,const char* d,unsigned int n){    //uses SMwrite4Byte to write arbitrary amounts
    unsigned int p0,p1;
    if(n<1){return 0;}                               //no point writing 0 bytes
    p0=0;
    while(1){
        p1=p0|3;     //to end of current block
        if(p1>(n-1)){p1=n-1;}   //check for terminal block
        if(ST25DVwrite4Byte(a+p0,&d[p0],(char)(p1+1-p0))){
            if(p1==(n-1)){return 1;}            //all written
        }else{
            return 0;
        }
        p0=p1+1;        //next block
    }
    return 0;
}

int assembleTextNDEF(char* dst, int dstSize, char* src, int srcSize){  //assemble for writing
    unsigned int i;
    if(dstSize<(NDEF_HEAD_SIZE+srcSize+1)){return 0;}    //won't fit
    if(srcSize>MAX_NDEF_SIZE){return 0;}                 //limit of format
    for(i=0;i<NDEF_HEAD_SIZE;i++){dst[i]=ndefHead[i];}
    for(i=0;i<srcSize;i++){dst[i+NDEF_HEAD_SIZE]=src[i];}
    dst[NDEF_HEAD_SIZE+srcSize]=0xFE;    //NDEF terminator
    dst[5]=(srcSize+7)&0xFF;
    dst[8]=(srcSize+3)&0xFF;
    return NDEF_HEAD_SIZE+srcSize+1;    //return
}
