; minCOdtt.ASM: minimalistic CO gas detector with multi-level alarm.
; In brief : keep RA1 output low, for 60s => P-mosfet T1 on => Vcc(MQ7) = 5.0V   
; then keep RA1 output in PWM, for 90s => P-mosfet T1 in PWM => Vcc(MQ7) = 1.4V
; at the beginning of a new cycle (60s+90s), pick an analog measure thru AN0.
; according to user defined thresholds, trigger the piezo thru RA2 output. 

;controller	equ	10F320
controller	equ	10F322

	if	controller == 10F320
	List P = 10F320
#include <p10f320.inc> 			
	endif

	if	controller == 10F322
	List P = 10F322
#include <p10f322.inc> 			
	endif

	Errorlevel -302		; avoid warning #302: Register in operand not in bank0. 

    __config _FOSC_INTOSC & _MCLRE_OFF & _LVP_OFF & _WDTE_OFF & _BOREN_OFF

;----------------------------- define hardware --------------------------------
;internal 0.5 MHz

#define	 gateT1	  PORTA, 1		; RA1 output, drives gate of P-mosfet T1
#define	 PIEZO	  PORTA, 2		; RA2 output, drives piezo buzzer
#define	 BP1	  PORTA, 3		; RA3 input pushbutton, to test piezo buzzer

;----------------------------- define constants -------------------------------

;OPTION_REG: p.103
#define  OPTIONVAL	b'01000111'	; pullup ON, <5>=0:RA2 I/O, TMR0 prescale(1:256)
;#define	 OPTIONVAL	b'11000011'	; pullup OFF, <5>=0:RA2 I/O, TMR0  1:16

;INTCON: p.46
#define	 INTERMASK	b'00100000'	; <5>=1: enables TMR0 interrupts
;#define	INTERMASK	b'00110000'	; use TMR0 and GP2/INT interruption

;user defined thresholds
THRESHOLD_1	  EQU	.64			; trigger the level 1 alarm (1 bip, each 20s) 
THRESHOLD_2	  EQU	.128		; trigger the level 2 alarm (2 bips, each 10s) 
THRESHOLD_3	  EQU	.192		; trigger the level 3 alarm (3 bips, each 5s)

;----------------------------- define variables -------------------------------

	CBLOCK	0x40	; SRAM = [0x40...0x7F] = 64 bytes (PIC10F320/322)

;context
	w_tmp
	STATUS_tmp
;delay
	duree60s			; counter 60s to keep P-mosfet on
	duree90s			; counter 90s to keep P-mosfet in PWM mode (duty 28%)
	cpt1, cpt2			; 8bit counters for BipHi routine
;measure
	meas				; 8bit ADC value
;alarm
	cntAL1, cntAL2, cntAL3 ; alarm1, 2, 3 counters
;flags
	bnFlags	;<0>=1 => int.TMR0 has occured
			;<1>=1 => PWM enabled
			; ...
			;<7>=1 => ... 

	ENDC
;11 bytes used

;--------------------------------- define bits --------------------------------

#define	 bTMR0		bnFlags, 0 ;<0>=1 => int.TMR0 has occured
#define	 bPWM2		bnFlags, 1 ;<1>=1 => PWM enabled

;------------------------------------------------------------------------------
; macro to put a literal value (constant) into a register
;------------------------------------------------------------------------------
MOVLF	macro	literal, file	; literal -> file
	movlw	literal
	movwf	file
	endm

;-----------------------------------------------------------------------------
; macro to save context
;-----------------------------------------------------------------------------
SAVE_CONTEXT	macro
	movwf	w_tmp  			; save W register
	swapf	STATUS, w		; swap status, then put result into w
	movwf	STATUS_tmp		; save status swapped
	endm

;-----------------------------------------------------------------------------
; macro to restore context
;-----------------------------------------------------------------------------
RESTORE_CONTEXT	macro
	swapf	STATUS_tmp, w	; swap old status, then put result into w
	movwf   STATUS			; restore status
	swapf   w_tmp, f		; Invert nibbles in old W (without modifying Z)
	swapf   w_tmp, w  		; Invert again nibbles in W
							; (so, W is restored without modifying STATUS)
	endm


;//////////////////////////////////////////////////////////////////////////////
;	MAIN PROGRAM
;//////////////////////////////////////////////////////////////////////////////
	ORG		0x00			; reset vector
	goto	Init
	ORG		0X04			; interrupt vector
	goto	IntTMR0

;//////////////////////////////////////////////////////////////////////////////

#include "IF8B16B3.INC"

;------------------------------------------------------------------------------
; ADC 8-bit for a PIC10F322
;------------------------------------------------------------------------------
measAN0
;config ADCON
	movlw	b'00000001'	; <7:5>=000 => Fosc/2, <4:2>=000 => AN0, <0>=1 => ADC on
	movwf	ADCON
;32us delay before doing meas.
	goto	$+1			; 8us + 8us
	goto	$+1			; 8us + 8us
;start meas
	bsf		ADCON, GO_NOT_DONE	; initiate one A/D conversion
	btfsc	ADCON, GO_NOT_DONE	; A/D conversion complete ?
	goto	$-1			; no
;end meas
	movf	ADRES, W	; yes, then retreive measure
	movwf	meas		; store it in meas
	retlw	0

;-----------------------------------------------------------------------------
; trigger the level 1 alarm (1 bip, each 20s)
;-----------------------------------------------------------------------------
alarm1
;  (cntAL1 = 38) * 524.288 = 19.922 s

;if (cntAL1 >= .1) goto skipBip1 (do Bip only when cntAL1 = 0)
	SIregGEval8b	cntAL1, .1, skipBip1
	;movlw	.64
	call	BipHi
	
skipBip1
	incf	cntAL1, f
	SIregLTval8b	cntAL1, .38, endAlarm1
	clrf	cntAL1
	
endAlarm1
	retlw	0

;-----------------------------------------------------------------------------
; trigger the level 2 alarm (2 bips, each 10s)
;-----------------------------------------------------------------------------
alarm2
;  (cntAL2 = 19) * 524.288 = 9.961 s

;if (cntAL2 >= .2) goto skipBip2 (do Bip-Bip only when cntAL2 < 2)
	SIregGEval8b	cntAL2, .2, skipBip2
	;movlw	.64
	call	BipHi
	
skipBip2
	incf	cntAL2, f
	SIregLTval8b	cntAL2, .19, endAlarm2
	clrf	cntAL2
	
endAlarm2
	retlw	0

;-----------------------------------------------------------------------------
; trigger the level 3 alarm (3 bips, each 5s)
;-----------------------------------------------------------------------------
alarm3
;  (cntAL3 = 10) * 524.288 = 5.242 s

;if (cntAL3 >= .3) goto skipBip3 (do Bip-Bip-Bip only when cntAL3 < 3)
	SIregGEval8b	cntAL3, .3, skipBip3
	;movlw	.64
	call	BipHi
	
skipBip3
	incf	cntAL3, f
	SIregLTval8b	cntAL3, .10, endAlarm3
	clrf	cntAL3
	
endAlarm3
	retlw	0

;-----------------------------------------------------------------------------
; BipHi
;-----------------------------------------------------------------------------
BipHi
	;clrf  cycles
	movlw	.200
	movwf	cpt1
Hi
	;clrwdt			; clear watchdog counter (not the WDTCON prescale)
	bcf   PIEZO		; PIEZO LOW...
	;bsf	WPUA, WPUA3		; <3>=1 => weak pullup enabled for RA3 => PIEZO on
	movlw	.5
	movwf	cpt2
	decfsz cpt2, f
	goto   $-1		; repeat Hi tone during 24 * 5 = 120 us

	bsf   PIEZO		; PIEZO HIGH...
	;bcf	WPUA, WPUA3		; <3>=0 => weak pullup disabled for RA3 => PIEZO off
	movlw	.5
	movwf	cpt2
	decfsz cpt2, f
	goto   $-1		; repeat Hi tone during 24 * 5 = 120 us

	decfsz cpt1, f
	goto   Hi		; repeat Hi tone during # 0.32 * 200 = 64 ms
	retlw  0

;------------------------------------------------------------------------------
;test user thresholds
;------------------------------------------------------------------------------
limits
; if meas > CO Level Alarm1, then goto trigAlarm1
	SIregGEval8b	meas, THRESHOLD_1, trigAlarm1
	clrf	cntAL1			; clear alarm1 counter
	clrf	cntAL2			; clear alarm2 counter
	clrf	cntAL3			; clear alarm3 counter
	retlw  0
	
trigAlarm1
; if meas > CO Level Alarm2, then goto trigAlarm2
	SIregGEval8b	meas, THRESHOLD_2, trigAlarm2
	clrf	cntAL2			; clear alarm2 counter
	clrf	cntAL3			; clear alarm3 counter
	goto	alarm1			; trigger the level 1 alarm 

trigAlarm2
; if meas > CO Level Alarm3, then goto trigAlarm3
	SIregGEval8b	meas, THRESHOLD_3, trigAlarm3
	clrf	cntAL1			; clear alarm1 counter
	clrf	cntAL3			; clear alarm3 counter
	goto	alarm2			; trigger the level 2 alarm
	
trigAlarm3
	clrf	cntAL1			; clear alarm1 counter
	clrf	cntAL2			; clear alarm2 counter
	goto	alarm3			; trigger the level 3 alarm

;-----------------------------------------------------------------------------
; enable PWM
;-----------------------------------------------------------------------------
enablePWM

;step0:	disable Timer2
	bcf		T2CON, TMR2ON
;step1: disable PWM2 pin (RA1)
	bsf		TRISA, RA1
;step2: clear the PWM2CON register
	clrf	PWM2CON
;step3: set PWM period for 7bit resolution
	MOVLF	0x1F, PR2	; PWM period = (PR2+1)*4*TMR2prescale/Fosc
						; PWM period = 256us => PWM freq = 3.906 kHz 
;step4: clear PWM2DCH & PWM2DCL
	;clrf	PWM2DCH
	;clrf	PWM2DCL
	MOVLF	0xC0, PWM2DCL	; duty cycle = 28% of 128 = 35% = 0x23 = b'00100011'
	MOVLF	0x08, PWM2DCH	; so PWM2DCL<7:6> = 11  &  PWM2DCH = 00001000
;step5: cfg & start TMR2
	bcf	PIR1, TMR2IF
	movlw	b'00000000'	; <6:3>=0000 postcaler 1:1 ; <1:0>=00 prescaler is 1
	movwf	T2CON
	bsf		T2CON, TMR2ON	; start timer2
;step6: enable PWM output
	bcf		TRISA, RA1	; RA1 output, hence PWM pin enabled again
	btfss	PIR1, TMR2IF
	goto	$-1			; wait until timer2 overflows
;step7: set PWMxOE bit of PWMxCON
	bsf		PWM2CON, PWM2OE ; output to PWM2 pin is enabled
;step8: load PWMxCON
	;MOVLF	b'11100000', PWM2CON
	MOVLF	b'11110000', PWM2CON
	
	bsf		bPWM2		; PWM2 is enabled
	retlw  0

;-----------------------------------------------------------------------------
; disable PWM
;-----------------------------------------------------------------------------
disablePWM
	bcf		T2CON, TMR2ON
	clrf	PWM2CON

	bcf		bPWM2		; PWM2 is disabled
	retlw  0

;//////////////////////////////////////////////////////////////////////////////
Init
;set GPIOs
	MOVLF	b'11111001', TRISA	; RA1,RA2 outputs ; RA0,RA3 inputs
	MOVLF	b'00000100', PORTA	; RA1 low, RA2 high
	MOVLF	b'00000001', ANSELA	; use only RA0 as AN0
	MOVLF	b'00001010', WPUA	; RA1, RA3 pullup enabled (to lower Idd)

;DS41585A.PDF(p.28/210): The HFINTOSC consists of a primary & secondary clock. 
;The secondary clock starts first with rapid start-up time, but low accuracy. 
;The primary clock follows with slower start-up time and higher accuracy. 

;set freq. to 0.5 MHz, then wait until sec. clock ready and pri. clock stable
	;movlw	b'01010000'		; OSCCON<6:4>=101 => 4 MHz (Tcycle = 1us)
	;movlw	b'01010000'		; OSCCON<6:4>=100 => 2 MHz (Tcycle = 2us)
	;movlw	b'00110000'		; OSCCON<6:4>=011 => 1 MHz (Tcycle = 4us)
	movlw	b'00100000'		; OSCCON<6:4>=010 => 0.5MHz (Tcycle = 8us)
	;iorwf	OSCCON, F
	movwf	OSCCON
	btfss	OSCCON, HFIOFR	; HFIOFR=1 => secondary clock is ready
	goto	$-1
	btfss	OSCCON, HFIOFS	; HFIOFS=1 => primary clock is stable
	goto	$-1

;init OPTION_REG & INTCON
	movlw	OPTIONVAL		; <7>=0: pullup ON, <5>=0: RA2 I/O, ...
	movwf	OPTION_REG		; ... TMR0 prescale (1:256) => 524.288 ms overflow
	movlw	INTERMASK
	movwf	INTCON			; enable TMR0 interrupt
	
;init variables	
	clrf	meas			; clear meas
	clrf	duree60s		; clear duree60s
	clrf	duree90s		; clear duree90s
	clrf	cntAL1			; clear alarm1 counter
	clrf	cntAL2			; clear alarm2 counter
	clrf	cntAL3			; clear alarm3 counter
	bcf		bTMR0
	bcf		bPWM2

;splash
	call	BipHi

;PROVISOIRE
	;bsf	 CLKRCON, CLKROE	; output Fosc/4 thru RA2

;enable interrupts
	bcf		INTCON, TMR0IF	; clear TMR0 interrupt flag
	bsf		INTCON, GIE		; set general interrupt enable

;do infinite loop ===========================================================
loop
	btfss	bTMR0			; int.TMR0 has occured ? 
	goto	loop			; no

; overflow, each 524.288 ms
;  (duree60s = 115) * 524.288 = 60.293 s
;  (duree90s = 172) * 524.288 = 90.177 s

;test piezo
	btfss	BP1				; BP1 idle ?
	call	BipHi			; no, it has been pushed

;test user thresholds
	call limits				; test thresholds, and trigger alarm if necessary
	
;next tick
	bcf		bTMR0
	incf	duree60s, f	; duree60s++

;if (duree60s >= .115) goto pulse90s (we are over the 60s pulse)
	SIregGEval8b	duree60s, .115, pulse90s
	;bcf		gateT1			; gate of T1 low => T1 on
	goto	loop
	
pulse90s
	decf	duree60s, f
	incf	duree90s, f	; duree90s++
;if (duree90s >= .172) goto endT150 (we are over the 90s pulse)
	SIregGEval8b	duree90s, .172, endT150
	bsf		gateT1			; gate of T1 high => T1 off
	goto	loop

endT150
	bcf		gateT1			; gate of T1 low => T1 on
	clrf	duree60s		; clear duree60s
	clrf	duree90s		; clear duree90s

;pick meas.	
	CALL	measAN0
	goto	loop

;----------------------------- Interrupt routine -----------------------------
; overflow, each 524.288 ms
IntTMR0
	SAVE_CONTEXT

	bsf		bTMR0
	bcf		INTCON, TMR0IF	; clear TMR0 interrupt flag

	RESTORE_CONTEXT
	retfie  				; return from interrupt

;-----------------------------------------------------------------------------
	END
