/*  
 * USB CDC-ACM Demo
 *
 * This file may be used by anyone for any purpose and may be used as a
 * starting point making your own application using M-Stack.
 *
 * It is worth noting that M-Stack itself is not under the same license as
 * this file.
 *
 * M-Stack is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
 * of the Apache License, version 2.0 which apply to this file.  If you have
 * purchased a commercial license for this software from Signal 11 Software,
 * your commerical license superceeds the information in this header.
 *
 * Alan Ott
 * Signal 11 Software
 * 2014-05-12
 
 //modified to be basically functional for a 16F1455/no changes needed for 16F1459
 //see usb_descriptors.c for name/ID strings etc
 //config changed to use crystal
   
 */
#define _XTAL_FREQ 16000000
#include "usb.h"
#include <xc.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"

//project specific IO
#include "io.h"

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif

//rx data is processed directly from the USB buffer
#define TXBUFSIZE 44
char txbuf[TXBUFSIZE]="";
int txptr=0;

//see also #defines at end of usb_config.h
#define USE_LINE_CODING
#ifdef USE_LINE_CODING
const static struct cdc_line_coding line_coding =
//static struct cdc_line_coding line_coding =
{
	9600,
	CDC_CHAR_FORMAT_1_STOP_BIT,
	CDC_PARITY_NONE,
	8,
};

#endif //USE_LINE_CODING

#define ADCBUFSIZE 250  //240 => 2 full cycles at 50Hz
#define ATARGET 845     //target amplitude, defined by voltage reference and min/max
#define VAVTARGET 270   //target average amplitude
#define CV50HZ (111)     //estimated CV for 50Hz
#define CV60HZ (222)    //estimated CV for 60Hz
#define CV100HZ (444)   //estimated CV for 100Hz
#define CVAMP (59)      //estimated CV for amplitude setting
#define PER50HZ (1920)  //target period for 50Hz = 96000/50
#define PER60HZ (1600)  //target period for 60Hz = 96000/60
#define PER100HZ (960)  //target period for 100Hz = 96000/100
#define MODE50HZ (1)    
#define MODE60HZ (2)    
#define MODE100HZ (3)    
#define MODEMANUAL (4)    

unsigned int ctr=0;
unsigned char tmout=0;  //for debouncing
unsigned char a=CVAMP;  //CV for amplitude
unsigned int f=CV50HZ;  //CV for frequency
int adcbuf[ADCBUFSIZE];
unsigned char adcptr=0; // off till all is running
unsigned char adcstate=0;
int mp;                 //mid-point reference
signed char mode=MODE50HZ;
int ptarget=PER50HZ;    //target period
int pcurrent=0;         //actual period
unsigned int fsafe=0;   //calculated freq, safe of /0
const unsigned int pot[]={10000,1000,100,10,1}; //for displaying decimals powers of ten
int zc1,zc2;            //zero cross indexes
int min,max;            //min/max amplitude
int delt,stp,frac;      //used to determine fractional period
unsigned int vav;       //average voltage amplitude
char lock=0;            //is output locked on?
char lastlock=0;
char sinemode=1;        //sine on by default

void doInt(unsigned int n, char d){     //output an unsigned int to USB buffer
    char j=0;
    char s='0';
    char lzf=1;
    while(j<5){
        while(n>=pot[j]){n=n-pot[j];s++;}
        if(d==j){txbuf[txptr++]='.';}
        if((d-1)==j){lzf=0;}    //always show digit before dp
        if(s!='0'){lzf=0;}
        if(j==4){lzf=0;}    //always show last digit
        if(lzf){
            txbuf[txptr++]=' ';            
        }else{
            txbuf[txptr++]=s;            
        }
        s='0';
        j++;
    }
}

void loadMode(void){            //helper to load mode settings
    lock=0;     //assume lock lost
    if(mode==MODE50HZ){
        ptarget=PER50HZ;
        f=CV50HZ;
    }
    if(mode==MODE60HZ){
        ptarget=PER60HZ;
        f=CV60HZ;
    }
    if(mode==MODE100HZ){
        ptarget=PER100HZ;
        f=CV100HZ;        
    }
    if(sinemode){
        a=CVAMP;        
    }else{
        a=1;                
    }
}

void setShow(void){     //this uses a 0-767 range by dithering over three pots
    unsigned int ff=0;
    unsigned int r=f;
    if(f<1){f=1;}       //avoid wrap
    if(f>700){f=700;}   //avoid wrap and lengthy run of iterative divide
    if(a<1){a=1;}       //avoid wrap
    if(a>253){a=253;}   //avoid wrap
    while(r>2){ //unsigned divide, compact and fairly fast as f<768
        ff++;
        r=r-3;
    }
    setDP(0,ff);
    if(r<2){setDP(1,ff);}else{setDP(1,ff+1);}
    setDP(2,a);
    if(r<1){setDP(3,ff);}else{setDP(3,ff+1);}
}

void buttons(void){
    static char oldb=0;
    char b=0;
    if(SW1==0){b=b|1;}
    if(SW2==0){b=b|2;}
    if(tmout==0){
        if((b&1) && (!(oldb&1))){  //button one pressed
            mode=mode+1;
            if(mode>MODE100HZ){mode=MODE50HZ;}
            loadMode();
        }
        if((b&2) && (!(oldb&2))){  //button two pressed        
            if(sinemode){
                sinemode=0;
                lock=0;
            }else{
                sinemode=1;
                lock=0;    
            }
            loadMode();
        }
    }
    if(b != oldb){tmout=255;}   //~24ms
    oldb=b;
}

void showManual(void){  //output manual a/f settings to USB
    if(txptr==0){      //only if TX buffer empty
        txbuf[txptr++]='A';
        txbuf[txptr++]='=';
        doInt(a,6);                        
        txbuf[txptr++]=',';
        txbuf[txptr++]='F';
        txbuf[txptr++]='=';
        doInt(f,6);                        
        txbuf[txptr++]=13;
        txbuf[txptr++]=10;
    }     
}

void showOP(void){
    txbuf[txptr++]='V';
    txbuf[txptr++]='=';    
    doInt(((max-min)*1000UL)/ATARGET,2);
    txbuf[txptr++]='V';

    //for debugging only
    //doInt(vav,0);
    //doInt(max-min,0);
    //doInt(pcurrent,0);

    txbuf[txptr++]=',';
    txbuf[txptr++]='F';
    txbuf[txptr++]='=';    
    doInt(fsafe,3);
    txbuf[txptr++]='H';
    txbuf[txptr++]='z';
    txbuf[txptr++]=13;
    txbuf[txptr++]=10;        
}

int main(void){
    int i;
    hardware_init();

#ifdef MULTI_CLASS_DEVICE
	cdc_set_interface_list(cdc_interfaces, sizeof(cdc_interfaces));
#endif
	usb_init();
    ioinit();
    setShow();
    mp=getADC(MID_REF);
    mp=getADC(MID_REF);
    ADCON0bits.CHS=AC_OUT;  //leave it like this for sampling
    adcptr=1;               //run sampling
    while (1) {
        if((ptarget<pcurrent+2)&&(ptarget>pcurrent-3)){
            if(sinemode==0){lock=1;}        //only need period lock for non-sine
            if((ATARGET-3<max-min)&&(ATARGET+4>max-min)){
                if((VAVTARGET-2<vav)&&(VAVTARGET+3>vav)){
                    lock=1;
                }
            }
        }
        if((ptarget>pcurrent+15)||(ptarget<pcurrent-14)){
            lock=0;
        }
        if((ATARGET-25>max-min)||(ATARGET+24<max-min)){
            if(sinemode==1){lock=0;}    //only needed for sine mode
        }
        if((VAVTARGET-9>vav)||(VAVTARGET+10<vav)){
            if(sinemode==1){lock=0;}    //only needed for sine mode
        }
        buttons();
        LED1=0;
        LED2=0;
        LED3=0;
        if((ctr&4096) || lock){ //always on half the time to flash if not locked on
            if(sinemode || (ctr&512)){ //flicker if not sine
                if(mode==MODE50HZ){LED1=1;}
                if(mode==MODE60HZ){LED2=1;}
                if(mode==MODE100HZ){LED3=1;}
                if(mode==MODEMANUAL){LED1=1;LED2=1;LED3=1;}  //manual, only active on USB                                    
            }
        }
        /*
        //debugging loss of lock
        if(lastlock!=lock){
            if(txptr==0){
                txbuf[txptr++]=lock+'0';
                showOP();
            }
        }
        lastlock=lock;
        */
        if(adcstate==1){    //samples done
            for(i=1;i<ADCBUFSIZE;i++){adcbuf[i]=adcbuf[i]-mp;}//offset adjust
                zc1=0;zc2=0;
                min=0;max=0;
                delt=0;stp=0;frac=0;
                vav=0;
                for(i=1;i<ADCBUFSIZE-1;i++){                    
                    if(adcbuf[i]<min){min=adcbuf[i];}   //find min
                    if(adcbuf[i]>max){max=adcbuf[i];}   //find max
                    if((adcbuf[i]<0)&&(adcbuf[i+1]>=0)){  //find
                        if(zc1==0){zc1=i;}              //1st crossing
                        else if(zc2==0){zc2=i;}         //and 2nd crossing
                    }
                }
                if(zc1 && zc2){
                    delt=adcbuf[zc1]-adcbuf[zc2];   //increase between periods
                    stp=adcbuf[zc2+1]+adcbuf[zc1+1]-adcbuf[zc2]-adcbuf[zc1];    //total period (double)
                    if(stp){frac=(delt*32)/stp;}   //scale to 4 bits
                    for(i=zc1;i<zc2;i++){vav=vav+abs(adcbuf[i]);}
                    vav=vav/(zc2-zc1);
                }
                if(zc1 && zc2 && stp){
                    pcurrent=(zc2-zc1)*16+frac;
                    fsafe=9600000UL/pcurrent;
                    if(mode<MODEMANUAL){
                        if(pcurrent < ptarget){f--;setShow();}
                        if(pcurrent > ptarget){f++;setShow();}
                    }
                }
                if(mode<MODEMANUAL){
                    if(sinemode){
                        if(max-min < ATARGET){a--;setShow();}   //a works backwards
                        if(max-min > ATARGET){a++;setShow();}
                    }else{
                        a=1;setShow();      //max amplitude
                    }
                }
                mp=getADC(MID_REF);     //get ready for next
                mp=getADC(MID_REF);
                ADCON0bits.CHS=AC_OUT;  //leave it like this for sampling
                adcstate=0;
                adcptr=1;
            }
        if (!usb_is_configured()){  //alternate mode if not enumerated
        }else{                      //USB working
            if(txptr>TXBUFSIZE-2){txptr=TXBUFSIZE-2;}   //clip buffer  
        }
        //data is sent from txbuf, size is txptr
        if (usb_is_configured() &&
		    !usb_in_endpoint_halted(2) &&
		    !usb_in_endpoint_busy(2)) {
			int i,k; 
			unsigned char *buf = usb_get_in_buffer(2);
            k=txptr;
            if(k){
                if(k>32){k=32;}     //limit data sent            
                for (i = 0; i < k; i++) {buf[i] = txbuf[i];}
                for(i=0;i<txptr;i++){txbuf[i]=txbuf[i+k];}  //move buffer up
                txptr=txptr-k;
                usb_send_in_buffer(2, k);
            }
		}

		//data is received from buffer and directly processed
        if (usb_is_configured() &&
		    !usb_out_endpoint_halted(2) &&
		    usb_out_endpoint_has_data(2)) {
			const unsigned char *out_buf;
			size_t out_buf_len;
			out_buf_len = usb_get_out_buffer(2, &out_buf);
            for(i=0;i<out_buf_len;i++){
                char r=out_buf[i];
                if((r=='s')||(r=='S')){
                    sinemode=1;
                    lock=0;    
                    loadMode();                    
                }
                if((r=='r')||(r=='R')){
                    sinemode=0;
                    lock=0;    
                    loadMode();                    
                }
                if(r=='1'){mode=MODE50HZ;loadMode();}
                if(r=='2'){mode=MODE60HZ;loadMode();}
                if(r=='3'){mode=MODE100HZ;loadMode();}
                if(r=='m'){mode=MODEMANUAL;loadMode();}                
                if(r=='M'){mode=MODEMANUAL;loadMode();}                
                if(r==' '){
                    if(txptr==0){       //only if TX buffer empty
                        showManual();
                        showOP();
                    }
                }
                if(r=='-'){
                    f=f-1;
                    setShow();
                    showManual();
                }
                if(r=='='){
                    f=f+1;
                    setShow();
                    showManual();
                }
                if(r==','){
                    a=a-1;
                    setShow();
                    showManual();
                }
                if(r=='.'){
                    a=a+1;
                    setShow();
                    showManual();
                }
            }
            usb_arm_out_endpoint(2);
		}
    }
}

/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return -1;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
#ifdef USE_LINE_CODING
	//line_coding = *coding;
#endif //USE_LINE_CODING
	return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	/* This is where baud rate, data, stop, and parity bits are set. */
#ifdef USE_LINE_CODING
    *coding = line_coding;
#endif //USE_LINE_CODING

	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	//
    return 0;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
	return 0;
}


#ifdef _PIC14E
void interrupt isr()
{
	if(PIR1bits.TMR2IF && PIE1bits.TMR2IE){
        if(adcptr){
            adcbuf[adcptr]=ADRES;                           //save last
            ADCON0bits.ADGO=1;                              //next reading
            adcptr++;
            if(adcptr>=ADCBUFSIZE){adcstate=1;adcptr=0;}    //done
        }
        PIR1bits.TMR2IF=0;//clear 
        ctr++;       
        if(tmout){tmout--;}
    }    
    usb_service();
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif
