; Battery Charger uses LCD module 
; Measures voltage and current plus temperature to develop charging
; modes are: Bulk, absorb and float with equalisation as option.
; PWM output is filtered and controls charging

	list P=16F628A
	#include p16f628A.inc

	__config _INTOSC_OSC_NOCLKOUT & _WDT_OFF & _PWRTE_ON & _BOREN_ON & _MCLRE_ON & _LVP_OFF

; Define variables at memory locations


EEPROM1		equ	H'00'	; non-volatile storage for battery capacity 
EEPROM2		equ	H'01'	; non-volatile storage for battery type (Lead Acid, Gel_Cell, AGM, CALCIUM/LEAD, Specific 1 and 2) 
EEPROM3		equ	H'02'	; non-volatile storage for bulk(cyclic) battery voltage at 20 deg C 
EEPROM4		equ	H'03'	; non-volatile storage for float battery voltage at 20 deg C
EEPROM5		equ	H'04'	; non-volatile storage for mV/degrees C factor

EEPROM6		equ	H'05'	; non-volatile storage for bulk (cyclic) battery voltage at 20 deg C. Specific/1 battery
EEPROM7		equ	H'06'	; non-volatile storage for float battery voltage at 20 deg C. Specific/1 battery 
EEPROM8		equ	H'07'	; non-volatile storage for mV/degrees C factor. Specific/1 battery

EEPROM9		equ	H'08'	; non-volatile storage for bulk (cyclic) battery voltage at 20 deg C. Specific/2 battery
EEPROM10	equ	H'09'	; non-volatile storage for float battery voltage at 20 deg C. Specific/2 battery 
EEPROM11	equ	H'0A'	; non-volatile storage for mV/degrees C factor. Specific/2 battery

; Bank 0 RAM
; cleared to zero at reset
AD_CK		equ	H'20'	; clock counter for A/D conversion
AD_VAL		equ	H'21'	; A/D value
SENSOR		equ	H'22'	; sensor select counter
DEGC_D		equ	H'23'	; degrees C A/D value
VOLTAGE_D	equ	H'24'	; voltage A/D value
CURRENT_D	equ	H'25'	; current A/D value
HOURS1		equ	H'26'	; hours LS byte (hhhh.h)
UPDATE1		equ	H'27'	; seconds period counter
UPDATE2		equ	H'28'	; 6-minutes period LS byte
UPDATE3		equ	H'29'	; 6-minutes period MS byte
CH_MODE		equ	H'2A'	; charging mode (bulk, absorb, float)
EQUALISE	equ	H'2B'	; equalisation mode set 
MODE		equ	H'2C'	; display mode
PARAM		equ	H'2D'	; parameter mode value
POINT_TWO	equ	H'2E'	; display update counter
CONV_COMP	equ	H'2F'	; conversions complete flag
TMR_CK		equ	H'30'	; timer clock
SLOW		equ	H'31'	; slow V or I control rate flag
I_ADDM		equ	H'32'	; current addition ms byte
I_ADDL		equ	H'33'	; current addition ls byte
I_ADDC		equ	H'34'	; current addition counter
DIV_CNT		equ	H'35'	; divider counter
D_BOUNCE	equ	H'36'	; switch debounce timer
I_AVG		equ	H'37'	; average current
CYC_AD		equ	H'38'	; additional cycles for A/D conversion time
VOLTAGE_U	equ	H'39'	; uncorrected A/D voltage value
TEN_BIT		equ	H'3A'	; PWM 10-bit counter
	
; bytes above cleared to zero at reset

OUT1		equ	H'3B'	; MS ASCII value
OUT2		equ	H'3C'	; MID ASCII value 
OUT3		equ	H'3D'	; LS ASCII value
BIN_0		equ	H'3E'	; Binary value
BUS_TME		equ	H'3F'	; time to ensure LCD is not busy
D_STO		equ	H'40'	; data storage in LCD data transfer 
BCD_0		equ	H'41'	; BCD working register
BCD_1		equ	H'42'	; BCD working register
CNT_8		equ	H'43'	; count of 8 in Binary to BCD conversion
NEGTVE		equ	H'44'	; negative sign flag
TIME_I		equ	H'45'	; delay store value
STORE1		equ	H'46'	; storage 
STORE2		equ	H'47'	; storage 
AH_VALUE	equ	H'48'	; AH of battery
NEG_COMP	equ	H'49'	; sign for temperature compensation of voltage
BULK_V		equ	H'4A'	; Bulk charge Voltage (after temperature compensation) 
BATTERY		equ	H'4B'	; battery type
STORAGE		equ	H'4C'	; compensation value of battery voltage
FLOAT_V		equ	H'4D'	; float voltage after temp compensation
BULK_0		equ	H'4E'	; bulk charge voltage at 20 C
FLOAT_0		equ	H'4F'	; float charge voltage at 20 C
I_SET		equ	H'50'	; current setting
V_SET		equ	H'51'	; voltage setting 
TEMP		equ	H'52'	; data storage
MV_DEGC		equ	H'53'	; mV/degrees C factor
TEM_PARAM	equ	H'54'	; temporary storage for parameters
TEMP_DSP	equ	H'55'	; temporary display value
SLOW_T		equ	H'56'	; slow control timer
I_CORR		equ	H'57'	; current correction for voltage
STORE_CK	equ	H'58'	; storage of current required
REPEAT		equ	H'59'	; repeat operation counter
SPACE_BAR	equ	H'5A'	; number of spaces

; math routines
REMB0		equ	0x5B
REMB1		equ	0x5C
TEMP1		equ 0x5D

TEMPB1      equ 0x5E
TEMPB0      equ 0x5F
TEMPD		equ	0x60
AARGB3		equ	0x62
AARGB2      equ 0x63
AARGB1      equ 0x64
AARGB0      equ 0x65	; most significant byte of argument A
BARGB1      equ 0x66
BARGB0      equ 0x67	; most significant byte of argument B
LOOPCOUNT   equ 0x68   	; loop counter



; All Banks RAM

W_TMP		equ	H'70'	; storage of w before interrupt
STATUS_TMP	equ	H'71'	; status storage before interrupt

; preprogram EEPROM DATA 
	
	ORG     2100
	DE	D'08', 0x00, D'227', D'214', D'80', D'229', D'213', D'144', D'229', D'213', D'144'
; Ah number 8, battery 0 (Lead Acid), Bulk voltage 227/255 x 160 = 14.2, float V 214/255 x 160 = 13.4V, mv/deg C 80/4 =20mV/deg C
; Specific 1 and 2 are 229/255 x 160 =14.4 bulk, float at 213/225 x 160= 13.4 and 144/4=36mV/degC
; note lead acid specs must be the same when transferred at LEAD_ACID_T

; start at memory 0

	org	0
	goto	MAIN
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		
;*******************************************************************************************
; lookup tables for amp hour, 25%, 5% and 2% current
; AMP HOUR values selected from 0 to 15 and converted to AH or Current in table
	
AHOUR
	addwf	PCL,f		; add value to program counter
	retlw	D'4'		; 4AH
	retlw	D'8'		; 8AH
	retlw	D'12'		; 12AH
	retlw	D'16'		; 16AH
	retlw	D'20'		; 20AH
	retlw	D'24'		; 24AH
	retlw	D'30'		; 30AH	
	retlw	D'40'		; 40AH
	retlw	D'60'		; 60AH
	retlw	D'80'		; 80AH
	retlw	D'90'		; 90AH
	retlw	D'100'		; 100AH
	retlw	D'125'		; 125AH
	retlw	D'150'		; 150AH
	retlw	D'175'		; 175AH
	retlw	D'200'		; 200AH
	retlw	D'225'		; 225AH
	retlw	D'250'		; 250AH
	
TWO_PCNT				; current that is 2% of AH
	addwf	PCL,f		; add value to program counter
	retlw	D'1'		; 80mA
	retlw	D'2'		; 160mA
	retlw	D'3'		; 240mA
	retlw	D'4'		; 360mA
	retlw	D'4'		; 400mA
	retlw	D'5'		; 480mA
	retlw	D'6'		; 600mA
	retlw	D'8'		; 800A
	retlw	D'12'		; 1.2A
	retlw	D'16'		; 1.6A
	retlw	D'18'		; 1.8A
	retlw	D'20'		; 2.0A
	retlw	D'25'		; 2.5A
	retlw	D'30'		; 3.0A
	retlw	D'35'		; 3.5A
	retlw	D'40'		; 4.0A
	retlw	D'45'		; 4.5A
	retlw	D'50'		; 5.0A

FIVE_PCNT				; current that is 5% of AH
	addwf	PCL,f		; add value to program counter
	retlw	D'2'		; 200mA
	retlw	D'4'		; 400mA
	retlw	D'6'		; 600mA
	retlw	D'8'		; 800mA
	retlw	D'10'		; 1A
	retlw	D'12'		; 1.2A
	retlw	D'15'		; 1.5
	retlw	D'20'		; 2A
	retlw	D'30'		; 3A
	retlw	D'40'		; 4.0A
	retlw	D'45'		; 4.5A
	retlw	D'50'		; 5.0A
	retlw	D'63'		; 6.25A
	retlw	D'75'		; 7.5A
	retlw	D'88'		; 8.75A
	retlw	D'100'		; 10.0A
	retlw	D'113'		; 11.25A
	retlw	D'125'		; 12.5A

TWO5_PCNT				; current that is 25% of AH
	addwf	PCL,f		; add value to program counter
	retlw	D'10'		; 1A
	retlw	D'20'		; 2A
	retlw	D'30'		; 3A
	retlw	D'40'		; 4A
	retlw	D'50'		; 5A
	retlw	D'60'		; 6A
	retlw	D'75'		; 7.5A
	retlw	D'100'		; 10A
	retlw	D'150'		; 15A
	retlw	D'166'		; 16.6A 
	retlw	D'166'		; 16.6A(limit of charger)
	retlw	D'166'		; 16.6A
	retlw	D'166'		; 16.6A
	retlw	D'166'		; 16.6A
	retlw	D'166'		; 16.6A
	retlw	D'166'		; 16.6A
	retlw	D'166'		; 16.6A
	retlw	D'166'		; 16.6A
;******************************************************************************************* 

; INTERRUPT

; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,W	; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	bcf 	STATUS,RP0	; bank 0 
	bcf	    INTCON,T0IF	; clear TMRO interrupt flag
	movf	TMR0,W		; timer value	
; altered from 250 Hz and added timer clock	; freq is 4MHz/4/16/(256(-8+2))=250Hz

	addlw	0x8			; add to timer register and takes 2 cycles to start counting
	movwf	TMR0		; place in timer (the two cycles plus 2-cycle delay = 4cycles)

; A/D conversion

AD_CONV
	movf	AD_CK,W		; clock counter
	btfsc	STATUS,Z	; if zero start new cycle 	
	goto	NEW_CYC		; clock out new value
	bcf		PORTA,2		; clock low
	sublw	0x07		; clock to 8
	btfss	STATUS,C	; if c = 0 then clear AD_CK
	goto	EOC			; end of conversion
	bcf		STATUS,C	; clear carry
	btfsc	PORTA,4		; read data bit
	bsf		STATUS,C
	rlf		AD_VAL,f	; shift data in
	incf	AD_CK,f		; next data
	bsf		PORTA,2		; clock high
	goto	CK1_s 	
EOC
	bsf		PORTA,2		; clock high
	movf	SENSOR,W	; sensor select counter
	andlw	0x03		; extract bits 0 and 1
	btfsc	STATUS,Z	; if zero then Voltage reading ready
	goto	VOLTG
	btfss	SENSOR,1	; if bit 1 set either temperature or current reading
	goto	SELECT		; invalid setting
	btfss	SENSOR,0	; if bit 0 clear then temperature reading
	goto	TEMPR		; bit 0 set so current

	movf	AD_VAL,W	; A/D reading
	movwf	CURRENT_D	; Current digital A/D value

; average current calculation (average over 50 conversions)
	movf	I_ADDC,w	; average addition counter
	sublw	D'49'		; compare with 50 (use 1 less for comparison)
	btfss	STATUS,C	; if negative then counted up ready
	goto	I_DIV		; divide 
	incf	I_ADDC,f	; next count
	movf	AD_VAL,W	; 
	addwf	I_ADDL,f	; add to LS byte
	btfsc	STATUS,C	; if carry set increase the MS byte
	incf	I_ADDM,f	; add to ms byte
	goto	SELECT
I_DIV
	clrf	I_ADDC		; addition counter

; divide by 50, then clear I_ADDM and I_ADDL			

	clrf	AARGB0		; most sig byte
	movf	I_ADDM,W	; ms byte
	movwf	AARGB1		; 
	movf	I_ADDL,W	; ls byte
	movwf	AARGB2		; 

	movlw	D'50'		; divide by value
	movwf	BARGB1		; ls byte of divisor
	clrf	BARGB0		; ms byte of divisor
	
	call	FXD2416U	; divide 
	movf	AARGB2,w	; averaged value
	movwf	I_AVG
	clrf	I_ADDM		; ms byte of counted current
	clrf	I_ADDL		; ls byte of counted current
	goto	SELECT

VOLTG
; apply current correction (assumes total of 0.02 ohms through 0.005 ohm current sense and .015 ohm in the leads)
; eg 10A with .02 ohms drops 200mV. So require a 2 x 255/160 drop in voltage A/D value. This is 3.18
; at 10A current A/D is 100. Dividing this by 32 gives 3.12 (close enough). So take (current (A/D)/32) from voltage (A/D values) 

	movf	CURRENT_D,w	; current reading
	btfsc	SLOW,0		; if slow control use average current measurement rather than instantaneous value
	movf	I_AVG,w		; average current reading
	movwf	I_CORR		; store value
	movlw	0x05		; divide by 32
	movwf	DIV_CNT		; store
DIV_CONT
	bcf		STATUS,C	; clear carry
	rrf		I_CORR,f	; /2
 	decfsz	DIV_CNT,f
	goto	DIV_CONT	; division continued

	movf	AD_VAL,w	; current voltage value in A/D converter
	movwf	VOLTAGE_U	; uncorrected voltage (before current x leads correction)

	movf	I_CORR,w	; divided value
	subwf	AD_VAL,W	; A/D reading - current correction
	btfss	STATUS,C	; if c is 0 then negative so set at zero
	movlw	0x00
	movwf	VOLTAGE_D	; Voltage digital (A/D) value
	goto	SELECT

TEMPR
	movf	AD_VAL,W	; temperature A/D reading
	movwf	DEGC_D		; degrees C A/D value

; select next sensor
SELECT
	incf	SENSOR,f	; next sensor value
	btfsc	SENSOR,1	; if bit 1 clear and bit 0 set increase again
	goto	BY_INC
	btfss	SENSOR,0	; if bit 0 set increase SENSOR value to bypass 01 code
	goto	BY_INC	
	incf	SENSOR,f
	bsf		CONV_COMP,0	; set bit for conversions complete flag
BY_INC
	bcf		PORTA,0		; B input to 4051 clear
	bcf		PORTA,1		; C input to 4051 clear
	btfsc	SENSOR,0	; if bit 0 set, set RA0
	bsf		PORTA,0		; set RA0 high
	btfsc	SENSOR,1	; if bit 1 set, set RA1 
	bsf		PORTA,1		; set RA1 high
	
	clrf	AD_CK		; ready for new conversion
	bcf		PORTA,2		; clock low
	bsf		PORTA,3		; CS high to start conversion (high for at least 17us)
	movlw	D'08'		; delay 
	movwf	CYC_AD
DEL_CNV
	decfsz	CYC_AD,f	; keep clock low for 36 internal clocks of TLC548/9
	goto	DEL_CNV
	goto	CK1_s

NEW_CYC	
	bcf		PORTA,3		; CS; bring RA1 low (conversion started)
	clrf	AD_VAL		; A/D value
	nop
	nop					; time for data
	btfss	PORTA,4		; data; read bit 7 data
	goto	CK_HI
	bsf		AD_VAL,0	; set bit 0 if set (rotated left later for 8-positions)

CK_HI
	bsf		PORTA,2		; clock; high
	incf	AD_CK,f		; address clock
	
; check for 1 second period elapsed

CK1_s
ONE_SEC

; check	timer
	incf	TMR_CK,f	; timer clock
	movf	TMR_CK,w
	xorlw	0x02		; count to 2
	btfss	STATUS,Z	; if zero 1/250 s
	goto	RECLAIM1
	clrf	TMR_CK		; clear ready to count again 
	incf	POINT_TWO,f	; point 2 seconds counter (0.2s = count 50)
	incf	UPDATE1,f	; increase counter 
	movf	UPDATE1,W
	xorlw	D'250'		; compare counter with 250 for 1 second
	btfss	STATUS,Z	; skip if equal
	goto	RECLAIM		; not 1 second
	clrf	UPDATE1		; seconds period count 
	incfsz	UPDATE2,f	; seconds counter LS byte for 6 minutes
	goto	CK_360
	incf	UPDATE3,f	; seconds counter MS byte
						; ms byte of 168h or 360(Decimal) seconds = 6 minutes = 0.1hours 
CK_360
	btfss	UPDATE3,0	; if ls bit set (then 01) check LS byte
	goto	RECLAIM		; bypass hours update

; increase hours
	movf	UPDATE2,w
	xorlw	0x68		; 
	btfss	STATUS,Z	; if 68 then 6 minutes
	goto	RECLAIM		
	clrf	UPDATE2		; return to 0 minutes
	clrf	UPDATE3	
	incf	HOURS1,f	; xx.x hours (25.5 max)	
	
; end of interrupt reclaim w and status 

RECLAIM	
	movf	SLOW_T,W	; slow control timer
	btfss	STATUS,Z	; if zero do not decrease
	decf	SLOW_T,f	; decrease till zero
	movf	D_BOUNCE,W	; switch debounce timer
	btfss	STATUS,Z	; if zero do not decrease
	decf	D_BOUNCE,f	; decrease till zero

RECLAIM1
	swapf	STATUS_TMP,W; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,W		; swap bits and into w register
	retfie				; return from interrupt

; *********************************************************************************

; initialise ports

MAIN
	movlw	0x07		; comparators off
	movwf	CMCON		; I/O pins
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000101'	; port B outputs and inputs 
	movwf	TRISB		; port B data direction register
	movlw	B'01110000'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000010'	; TMR0 /8
	movwf	OPTION_REG	; port B pullups enabled
	bcf		STATUS,RP0	; memory bank 0
;	clear registers
	movlw	0x20		; start address
	movwf	FSR
NEXTZRO
	clrf	INDF		; indirect addressing clear register
	incf	FSR,F		; next address
	movf	FSR,W
	xorlw	0x3C		; last address +1
	btfss	STATUS,Z	; out when done
	goto 	NEXTZRO	
; registers cleared shown below
;	clear	AD_CK		; clock counter for A/D conversion
;	clear	AD_VAL		; A/D value
;	clear	SENSOR		; counter for sensor slection
;	clear	DEGC_D		; degrees C A/D value
;	clear	VOLTAGE_D	; voltage A/D value
;	clear	CURRENT_D	; current A/D value
;	clear	HOURS1		; elapsed time
;	clear 	UPDATE1		; timer counter
;	clear	UPDATE2		; 6 mins counter (ls)
;	clear	UPDATE3		; 6 minutes counter (ms)
;	clear	CH_MODE		; charging mode (bulk, absorb, float)
;	clear	EQUALISE	; equalisation mode set 
;	clear	MODE		; display mode
;	clear	PARAM		; parameters mode at 0
;	clear	POINT_TWO	; 0.2 seconds counter
;	clear	CONV_COMP	; clear conversions complete flag 
;	clear	TMR_CK		; timer clock
;	clear	SLOW		; slow rate of V and I control
;	clear	I_ADDM		; current addition ms byte
;	clear	I_ADDL		; current addition ls byte
;	clear	I_ADDC		; current addition counter
;	clear	D_BOUNCE	; switch debounce timer
;	clear	I_AVG		; average current value
;	clear 	VOLTAGE_U	; uncorrected A/D voltage ie before current x leads correction
; 	clear   AV_FLG		; average value ready flag
 
	clrf	PORTA
	bsf 	PORTA,3		; set CS for A/D converter	
	movlw	D'200'		; 
	movwf	SLOW_T		; slow control timer
	btfss	PORTB,0		; if RB0 set, set to mode = 1
	incf	MODE,f		; mode =1

; PWM setup

	clrf	CCPR1L		; ms byte of PWM
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'	; set PWM mode
	movwf	CCP1CON		; enable PWM operation
	
; set up initial conditions for display

	movlw	0x0F		; start up delay
	call	DELX
	clrf	PORTB		; RS, R/W, E low 
	call	INIT_LC	
	movlw	0x3F		; start up delay
	call	DELX
	call	INIT_LC
	call	DELAYms
	call	INIT_LC
	
; set 4-bit operation
	
	bcf		PORTB,4		; display command 4-bits
	bcf		PORTA,7		; register select low
	call	SET_CLEAR_B1; set and clear Enable (PortB,1)	
	call	DELAYms
	movlw	0x04		; 4 times
	movwf	REPEAT		; repeat counter
DSP_FUNCTN
	movlw	B'00101100'	; display function (4-bits, 2 lines, 5x10 dots)
	call	LOAD

	movlw	B'00001110'	; blinking off, cursor off
	call	LOAD
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	0x20
	call	DELX
	movlw	B'00000110'	; entry mode. cursor moves right, display not shifted
	call	LOAD
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	0x20
	call	DELX
	decfsz	REPEAT,f
	goto	DSP_FUNCTN	; repeat loading
	
; allow interrupts

SET_UP
	bsf		INTCON,T0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE	; set global interrupt enable 

; delay to ensure A/D values are all read (temp, current and Voltage)

	movlw	0x20		; start up delay
	call	DELX

; get EEPROM values
	movlw	EEPROM1		; AH value in EEPROM1
	call	EEREAD		; read value
	movwf	AH_VALUE	; battery AH value
	movlw	EEPROM2		; battery type EEPROM2
	call	EEREAD		; read value
	movwf	BATTERY		; battery type
	movlw	EEPROM3		; bulk charge V in EEPROM3
	call	EEREAD		; read value
	movwf	BULK_0		; battery V for bulk charge at 20 deg C
	movlw	EEPROM4		; float V in EEPROM4
	call	EEREAD		; read value
	movwf	FLOAT_0		; battery V for float at 20 deg C
	movlw	EEPROM5		; mV/degrees C EEPROM5
	call	EEREAD		; read value
	movwf	MV_DEGC

MORE					; return here after running each update

; if voltage is 16V then charger off
; if PWM is set to 5V (maximum), then charger off
; if current over 20A then charger off
	
; pwm
	incfsz	CCPR1L,w	; check PWM value	
	goto	I_CHKING	; check current
	goto	OFF_CHRGER	; charger off
; current
I_CHKING
	movlw	D'200'		; 20A
	subwf	I_AVG,w		; Average current - 200
	btfsc	STATUS,C	; if negative current ok (c zero)
	goto	OFF_CHRGER	; overcurrent
; voltage
V_CHKING
	incfsz	VOLTAGE_U,w	; A/D for voltage before voltage drop/ current correction
	goto	SWITCH		; not at maximim so continue
; battery high
; off except during equalise
	movlw	0x02
	xorwf	CH_MODE,W	; if in float mode clear Equalise
	btfsc	STATUS,Z	; 
	goto	SWITCH

	call	SET_TO_OFF	; clear pwm and set display
	call	SPACE3
	call	BATTERY_DSP	; battery on display
	movlw	A'?'
	call	DRV_LCD		; write ?
	call	SPACE4
	goto	OFF_CHG

; charger error
OFF_CHRGER
	call	SET_TO_OFF
	call	SPACE4
	movlw	A'<'
	call	DRV_LCD
	call	OFF_DSP		; write OFF>
	call	SPACE7

OFF_CHG
	goto	OFF_CHG	

SET_TO_OFF
	clrf	CCPR1L		; set PWM at zero (Charger off)
	movlw	B'00000001'	; display clear
	call	LOAD
	movlw	0x81		; address at first line 
 	call	LOAD
	movlw	0x20
	call	DELX
	return

; check switch pressing

SWITCH
; if parameter link out, set parameters for Specific1 and specific2 batteries
	btfsc	PORTB,2		; if set change parameters mode
	goto	CH_PARAM

	movf	D_BOUNCE,W	; debounce timer
	btfss	STATUS,Z	; if zero check switches
	goto	DRV_DISPLAY
; check switch lockouts (after charging started and in absorb, equalisation and float phases)
	movf	SLOW,w		; slow I and V control rate setting
	btfss	STATUS,Z	; if zero or clear then can use switches
	goto	DRV_DISPLAY	; bypass switch check

	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,7		; reset (start) switch
	movlw	0x02
	call	DELX
	btfss 	PORTA,6		; if set then reset switch pressed (start switch)
	goto	MOD_SW
	clrf	MODE		; set to zero and charger runs
	clrf	SLOW		; slow rate of V and I control flag
	
; mode switch
MOD_SW

	movf	CH_MODE,w	; charging mode
	btfss	STATUS,Z	; if bulk mode check mode switch (set switch)
	goto	DRV_DISPLAY	; bypass mode switch in Absorb and float modes
; check switches
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,6		; mode switch
	movlw	0x02
	call	DELX
	btfss 	PORTA,6		; if set then mode switch pressed
	goto	DRV_DISPLAY
	incf	MODE,f		; display mode
	movf	MODE,W		; check limit
	sublw	0x03		; if more than 2, clear
	movlw	0x01		; ready if required to set MODE to 1
	btfss	STATUS,C	; 
	movwf	MODE		; back to 1
	movlw	D'150'		; delay
	movwf	D_BOUNCE	; decreased to zero in interrupt
	movlw	D'80'
	movwf	POINT_TWO	; set 0.2s timer to 0.2seconds so display will update immediately
	
; drive display 
DRV_DISPLAY
; update at 0.2 seconds
	movf	POINT_TWO,w	; counter 
	sublw	D'80'		; 0.2s count
	btfsc	STATUS,C	; when negative update display
	goto	CHRG_CONTROL; charge control loop	
	clrf	POINT_TWO
; line 1
	movlw	0x80		; address at first line far left
 	call	LOAD

; check display mode
CK_MODES
	movf	MODE,W		; mode
	btfsc	STATUS,Z	; if zero charge mode
	goto	CH_MODES
	clrf	CCPR1L		; pwm off for other modes
	clrf	SLOW		; slow control of V or I flag
	clrf	I_ADDM		; ms byte of counted current
	clrf	I_ADDL		; ls byte of counted current
	xorlw	0x01		; AH mode
	btfsc	STATUS,Z
	goto	AH_MODE
	movf	MODE,W
	xorlw	0x02		; Battery type mode
	btfsc	STATUS,Z
	goto	BATT_TYPE
	goto	EQ_SEL		; equalise select

; change parameters mode
CH_PARAM
	clrf	CCPR1L		; pwm off
	clrf	SLOW		; slow control flag
	clrf	I_ADDM		; ms byte of counted current
	clrf	I_ADDL		; ls byte of counted current
	movf	D_BOUNCE,W	; debounce timer
	btfss	STATUS,Z	; if zero check switches
	goto	MORE
	movlw	0x02
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,6		; mode switch
	movlw	0x02
	call	DELX
	btfss 	PORTA,6		; if set then mode switch pressed
	goto	CK_PARAM_MODES
	incf	PARAM,f		; Parameter mode
	movf	PARAM,W		; check limit
	sublw	0x05		; if more than 5, clear
	btfss	STATUS,C	; 
	clrf	PARAM		; back to 0
	movlw	D'150'		; delay
	movwf	D_BOUNCE	; decreased to zero in interrupt

; check parameter modes
CK_PARAM_MODES
	movf	PARAM,w		; parameters mode 0,1,2 are bulk;float;mv/degC for Specific/1 battery (3,4,5 are for Specific/2)
	btfsc	STATUS,Z	; if zero Specific/1 Bulk Voltage
	goto	SP1_BULKV

	movf	PARAM,w		; parameter modes
	xorlw	0x01
	btfsc	STATUS,Z	; if zero Specific/1 Float Voltage
	goto	SP1_FLOATV

	movf	PARAM,w		; parameter modes
	xorlw	0x02
	btfsc	STATUS,Z	; if zero Specific/1 Compensation (mV/deg C)
	goto	SP1_COMP

	movf	PARAM,w		; parameter modes
	xorlw	0x03
	btfsc	STATUS,Z	; if zero Specific/2 BULK Voltage
	goto	SP2_BULKV

	movf	PARAM,w		; parameter modes
	xorlw	0x04
	btfsc	STATUS,Z	; if zero Specific/2 Float Voltage
	goto	SP2_FLOATV

	movf	PARAM,w		; parameter modes
	xorlw	0x05
	btfsc	STATUS,Z	; if zero Specific/2 Compensation (mV/deg C)
	goto	SP2_COMP

; Specific 1 bulk voltage
SP1_BULKV
; get specific/1 BULK voltage
	movlw	EEPROM6		; BULK charge V for Specific/1 in EEPROM6
	call	EEREAD		; read value
	movwf	TEM_PARAM	; battery V for BULK charge at 20 deg C storage
; check up and down switches	
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_BULK_SP	; increase bulk voltage
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_BULK_SP
	goto	SPI_BV_DSPLY; display
DN_BULK_SP
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	call	DECREASE
	goto	WRI_BULK_SP1
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; decrease or increase, then check if display will change 
; if not decrease or increase again. This gives consistent
; change in values on the display
DECREASE
	movf	TEM_PARAM,w
	call	CALCUL_ATE	; get value for display
	movwf	TEMP_DSP	; store
DOWN_TEM
	decf	TEM_PARAM,f ; decrease
	movf	TEM_PARAM,w
	call	CALCUL_ATE
	xorwf	TEMP_DSP,w	; previous display value
	btfss	STATUS,Z	; if the same decrease again
	return
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	goto	DOWN_TEM
	return
INCREASE
	movf	TEM_PARAM,w
	call	CALCUL_ATE	; get value for display
	movwf	TEMP_DSP	; store
UP_TEM
	incf	TEM_PARAM,f ; decrease
	movf	TEM_PARAM,w
	call	CALCUL_ATE
	xorwf	TEMP_DSP,w	; previous display value
	btfss	STATUS,Z	; if the same decrease again
	return
	incfsz	TEM_PARAM,w	; temporary value increase up to max
	goto	UP_TEM
	return	
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;	
UP_BULK_SP
	incf	TEM_PARAM,w	; temporary value increase up to max
	sublw	D'251'		; equivalent to 15.7V max setting ie 250/255 x 16.0 = 15.7V
	btfsc	STATUS,C	; if less than 250 continue with increase
	call	INCREASE	; increase if not passed FF
WRI_BULK_SP1
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE	
	movlw	EEPROM6		; write to EEPROM
	call	BULK_WRITE2 ; write to EEPROM6

; check battery selection. If SPECIFIC # 1 then load parameters
	movf	BATTERY,w	; battery type selection
	xorlw	0x04		; specific 1
	btfss	STATUS,Z
	goto	SPI_BV_DSPLY
	movf	TEM_PARAM,w	; bulk voltage threshold
	call	BULK_WRITE	; transfer to BULK_0 write to EEPROM3
 
; line 1
SPI_BV_DSPLY
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPEC_DISPLAY; writes SPECIFIC on display
	movlw	A'1'		; type 1
	call	DRV_LCD
	call	SPACE5
	
; line 2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD

; get specific/1 bulk voltage
	movf	TEM_PARAM,w	; battery V for bulk charge at 20 deg C storage
	call	VOLT_DSP_SP	; writes voltage on display xx.xV

	call	CYCLE_DSP	; write CYCLE on display 
	call	AT20DEGC
	goto	MORE
CYCLE_DSP
	movlw	A'C'
	call	DRV_LCD
	movlw	A'Y'
	call	DRV_LCD
	movlw	A'C'		; 
	call	DRV_LCD
	movlw	A'L'		; 
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
AT20DEGC
	call	SPACE1
	movlw	A'2'
	call	DRV_LCD
	movlw	A'0'		; 20 deg C
	call	DRV_LCD
	movlw	0xDF		; degrees symbol
	call	DRV_LCD
	movlw	A'C'		; 20 deg C
	call	DRV_LCD
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; specific 1 float voltage
SP1_FLOATV
; get specific/1 float voltage
	movlw	EEPROM7		; Float charge V for Specific/1 in EEPROM7
	call	EEREAD		; read value
	movwf	TEM_PARAM	; battery V for float charge at 20 deg C storage
; check up and down switches	
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_FLOAT_SP	; increase float voltage
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_FLOAT_SP
	goto	SPI_FV_DSPLY; display
DN_FLOAT_SP
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	call	DECREASE
	goto	WRI_FLOAT_SP1

UP_FLOAT_SP
	incf	TEM_PARAM,w	; temporary value increase up to max
	sublw	D'251'		; equivalent to 15.7V max setting ie 250/255 x 16.0 = 15.7V
	btfsc	STATUS,C	; if less than 250 continue with increase
	call	INCREASE
WRI_FLOAT_SP1
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE	
	movlw	EEPROM7		; write to EEPROM
	call	FLOAT_WRITE2 ; write to EEPROM7

; check battery selection. If SPECIFIC # 1 then load parameters
	movf	BATTERY,w	; battery type selection
	xorlw	0x04		; specific 1
	btfss	STATUS,Z
	goto	SPI_FV_DSPLY
	movf	TEM_PARAM,w	; bulk voltage threshold
	call	FLOAT_WRITE	; transfer to FLOAT_0 write to EEPROM4

; line 1
SPI_FV_DSPLY
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPEC_DISPLAY; writes SPECIFIC on display
	movlw	A'1'		; type 1
	call	DRV_LCD
	call	SPACE5
	
; line 2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD

; get specific/1 float voltage

	movf	TEM_PARAM,w	; battery V for float charge at 20 deg C storage
	call	VOLT_DSP_SP	; writes voltage on display xx.xV

	call	FLOAT_DIS_NSP; write float on display No space
	call	AT20DEGC	; write @20degC on display
	goto	MORE

; specific/1 compensation value (mV/deg C)
SP1_COMP
; get specific/1 compensation value
	movlw	EEPROM8		; compensation for Specific/1 in EEPROM8 (mV/deg C)
	call	EEREAD		; read value
	movwf	TEM_PARAM	; compensation value
; check up and down switches	
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_COMP_SP	; increase compensation
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_COMP_SP
	goto	SPI_CP_DSPLY; display
DN_COMP_SP
; decrease by factor of 4 to mimic display value which is divided by 4 ie (TEM_PARAM/4)
	call	DEC_4		; divide by 4
	goto	WRI_COMP_SP1
DEC_4
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	decf	TEM_PARAM,f
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	decf	TEM_PARAM,f
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	decf	TEM_PARAM,f
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	decf	TEM_PARAM,f
	return
		
UP_COMP_SP
; increase by factor of 4 to mimic display value which is divided by 4 ie (TEM_PARAM/4)
	
	call	INC_4		; increase by a factor of 4
	goto	WRI_COMP_SP1
INC_4
	incfsz	TEM_PARAM,w	; temporary value increase up to max
	incf	TEM_PARAM,f	; increase if not passed FF
	incfsz	TEM_PARAM,w	; temporary value increase up to max
	incf	TEM_PARAM,f	; increase if not passed FF
	incfsz	TEM_PARAM,w	; temporary value increase up to max
	incf	TEM_PARAM,f	; increase if not passed FF
	incfsz	TEM_PARAM,w	; temporary value increase up to max
	incf	TEM_PARAM,f	; increase if not passed FF
	return
WRI_COMP_SP1
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE	
	movlw	EEPROM8		; write to EEPROM
	call	T_CORR_WRITE2 ; write to EEPROM8

; check battery selection. If SPECIFIC # 1 then load parameters
	movf	BATTERY,w	; battery type selection
	xorlw	0x04		; specific 1
	btfss	STATUS,Z
	goto	SPI_CP_DSPLY
	movf	TEM_PARAM,w	; mV/degC
	call	T_CORR_WRITE; transfer to MV_DEGC write to EEPROM5

; line 1
SPI_CP_DSPLY
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPEC_DISPLAY; writes SPECIFIC on display
	movlw	A'1'		; type 1
	call	DRV_LCD
	call	SPACE5
		
; line 2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	movlw	A'-'		; -
	call	DRV_LCD
	call	SPEC_TEMP
	goto	MORE

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
SPEC_TEMP
; get specific/1 compensation value (divide by 4)

	movf	TEM_PARAM,w	; compensation temporary value
	bcf		STATUS,C	; clear carry
	rrf		TEM_PARAM,f
	bcf		STATUS,C	; clear carry
	rrf		TEM_PARAM,w

;convert to ASCII	
	movwf	BIN_0		; ls binary value
	call	BIN_ASCII
	movf	OUT2,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_COMP

	movf	OUT2,W		; not zero so display
	call	DRV_LCD
	goto	LS_COMPS
LS_COMP					; leading zero blanking or negative sign
	call	SPACE1		; space instead of 0

LS_COMPS
	movf	OUT3,W		; ls digit
	call	DRV_LCD

	movlw	A'm'		; mV
	call	DRV_LCD
	movlw	A'V'
	call	DRV_LCD
	movlw	A'/'		; /deg C
	call	DRV_LCD
	movlw	0xDF		; degrees symbol
	call	DRV_LCD
	movlw	A'C'
	call	DRV_LCD
	call	SPACE8
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Specific 2 bulk voltage
SP2_BULKV
; get specific/2 BULK voltage
	movlw	EEPROM9		; BULK charge V for Specific/2 in EEPROM9
	call	EEREAD		; read value
	movwf	TEM_PARAM	; battery V for BULK charge at 20 deg C storage
; check up and down switches	
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_BULK_SP2	; increase bulk voltage
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_BULK_SP2
	goto	SPI_BV_DSPL2; display
DN_BULK_SP2
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	call	DECREASE
	goto	WRI_BULK_SP2
UP_BULK_SP2
	incf	TEM_PARAM,w	; temporary value increase up to max
	sublw	D'251'		; equivalent to 15.7V max setting ie 250/255 x 16.0 = 15.7V
	btfsc	STATUS,C	; if less than 250 continue with increase
	call	INCREASE
WRI_BULK_SP2
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE	
	movlw	EEPROM9		; write to EEPROM
	call	BULK_WRITE2 ; write to EEPROM9

; check battery selection. If SPECIFIC # 2 then load parameters
	movf	BATTERY,w	; battery type selection
	xorlw	0x05		; specific 2
	btfss	STATUS,Z
	goto	SPI_BV_DSPL2
	movf	TEM_PARAM,w	; bulk voltage threshold
	call	BULK_WRITE	; transfer to BULK_0 write to EEPROM3

; line 1
SPI_BV_DSPL2
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPEC_DISPLAY; writes SPECIFIC on display
	movlw	A'2'		; type 2
	call	DRV_LCD
	call	SPACE5
	
; line 2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD

; get specific/1 bulk voltage
	movf	TEM_PARAM,w	; battery V for bulk charge at 20 deg C storage
	call	VOLT_DSP_SP	; writes voltage on display xx.xV

	call	CYCLE_DSP	; write CYCLE on display
	call	AT20DEGC
	goto	MORE

; specific 2 float voltage
SP2_FLOATV
; get specific/2 float voltage
	movlw	EEPROM10	; Float charge V for Specific/2 in EEPROM10
	call	EEREAD		; read value
	movwf	TEM_PARAM	; battery V for float charge at 20 deg C storage
; check up and down switches	
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_FLOAT_SP2; increase float voltage
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_FLOAT_SP2
	goto	SPI_FV_DSPL2; display
DN_FLOAT_SP2
	movf	TEM_PARAM,w	; temporary value
	btfss	STATUS,Z	; if zero do not decrease 
	call	DECREASE	
	goto	WRI_FLOAT_SP2
UP_FLOAT_SP2
	incf	TEM_PARAM,w	; temporary value increase up to max
	sublw	D'251'		; equivalent to 15.7V max setting ie 250/255 x 16.0 = 15.7V
	btfsc	STATUS,C	; if less than 250 continue with increase
	call	INCREASE
WRI_FLOAT_SP2
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE	
	movlw	EEPROM10	; write to EEPROM
	call	FLOAT_WRITE2; write to EEPROM10

; check battery selection. If SPECIFIC # 2 then load parameters
	movf	BATTERY,w	; battery type selection
	xorlw	0x05		; specific 2
	btfss	STATUS,Z
	goto	SPI_FV_DSPL2
	movf	TEM_PARAM,w	; FLOAT voltage threshold
	call	FLOAT_WRITE	; transfer to BULK_0 write to EEPROM4

; line 1
SPI_FV_DSPL2
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPEC_DISPLAY; writes SPECIFIC on display
	movlw	A'2'		; type 2
	call	DRV_LCD
	call	SPACE5
		
; line 2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD

; get specific/2 float voltage

	movf	TEM_PARAM,w	; battery V for float charge at 20 deg C storage
	call	VOLT_DSP_SP	; writes voltage on display xx.xV

	call	FLOAT_DIS_NSP; write float on display No space
	call	AT20DEGC	; write @20degC on display
	goto	MORE

; specific/2 compensation value (mV/deg C)
SP2_COMP
; get specific/2 compensation value
	movlw	EEPROM11	; compensation for Specific/2 in EEPROM11 (mV/deg C)
	call	EEREAD		; read value
	movwf	TEM_PARAM	; compensation value
; check up and down switches	
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_COMP_SP2	; increase compensation
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_COMP_SP2
	goto	SPI_CP_DSPL2; display
DN_COMP_SP2
; decrease by a factor of 4 to mimic display value which is divided by 4 ie (TEM_PARAM/4)
	call	DEC_4		; decrease ie divide by 4	
	goto	WRI_COMP_SP2
UP_COMP_SP2
; increase by a factor of 4 to mimic display value which is divided by 4 ie (TEM_PARAM/4)
	call	INC_4		; multiply by 4
WRI_COMP_SP2
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE	
	movlw	EEPROM11	; write to EEPROM
	call	T_CORR_WRITE2 ; write to EEPROM11

; check battery selection. If SPECIFIC # 2 then load parameters
	movf	BATTERY,w	; battery type selection
	xorlw	0x05		; specific 2
	btfss	STATUS,Z
	goto	SPI_CP_DSPL2
	movf	TEM_PARAM,w	; mv/degC
	call	T_CORR_WRITE; transfer to MV/DEGC write to EEPROM5

; line 1
SPI_CP_DSPL2
	movlw	0x80		; address at first line far left
 	call	LOAD
	call	SPEC_DISPLAY; writes SPECIFIC on display
	movlw	A'2'		; type 2
	call	DRV_LCD
	call	SPACE5
	
; line 2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	movlw	A'-'		; -
	call	DRV_LCD
	call	SPEC_TEMP
	goto	MORE

; battery type mode
BATT_TYPE
	call	BATTERY_DSP	; write battery on display
	goto	WRI_TYPE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; write Battery on display
BATTERY_DSP
	movlw	A'B'
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'T'		; 
	call	DRV_LCD
	movlw	A'T'		; 
	call	DRV_LCD
	movlw	A'E'		; write Battery
	call	DRV_LCD
	movlw	A'R'
	call	DRV_LCD
	movlw	A'Y'
	call	DRV_LCD
	call	SPACE1
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
WRI_TYPE
	movlw	A'T'		; write type
	call	DRV_LCD
	movlw	A'Y'
	call	DRV_LCD
	movlw	A'P'
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	call	SPACE4

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	movlw	A'<'
	call	DRV_LCD
	movf	BATTERY,w	; battery type number
	btfsc	STATUS,Z	; if zero then Lead Acid
	goto	LEAD_ACID_T
	movf	BATTERY,w
	xorlw	0x01		; 
	btfsc	STATUS,Z	; if zero then GEL_CELL
	goto	GEL_CELL_T
	movf	BATTERY,w
	xorlw	0x02		; 
	btfsc	STATUS,Z	; if zero then AGM
	goto	AGM_T
	movf	BATTERY,w
	xorlw	0x03		; 
	btfsc	STATUS,Z	; if zero then CALCIUM/LEAD
	goto	CALCIUM_T	
	movf	BATTERY,w
	xorlw	0x04		; 
	btfsc	STATUS,Z	; if zero then SPECIFIC1
	goto	SPECIFIC1_T
	movf	BATTERY,w
	xorlw	0x05		; 
	btfsc	STATUS,Z	; if zero then SPECIFIC2
	goto	SPECIFIC2_T
		
LEAD_ACID_T	
; transfer data to EEPROM and registers (also make these the same as the EEPROM 3,4 & 5 preset)
; bulk voltage
	movlw	D'227'		; bulk voltage at 20 deg c for Lead Acid battery (227/255 x 16.0= 14.2)
	call	BULK_WRITE
; float voltage
	movlw	D'214'		; float voltage at 20 deg c for lead Acid battery (214/255 x 16.0=13.4)
	call	FLOAT_WRITE
; mV/deg C
	movlw	D'80'		; mv/ deg c for lead Acid battery (80/4 = 20mV/deg C)
	call	T_CORR_WRITE
	goto	WRITE_LEADACID
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
BULK_WRITE
	movwf	BULK_0		; Bulk Voltage
	movwf	TEM_PARAM	; temporary
BULK_WRITE1
	movlw	EEPROM3		; write to EEPROM
BULK_WRITE2
FLOAT_WRITE2
T_CORR_WRITE2
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	TEM_PARAM,w
	call	EWRITE		; write to EEPROM 
	return
FLOAT_WRITE
	movwf	FLOAT_0		; float Voltage
	movwf	TEM_PARAM	; temporary
FLOAT_WRITE1
	movlw	EEPROM4		; write to EEPROM
	goto	FLOAT_WRITE2

T_CORR_WRITE
	movwf	MV_DEGC		; mv/deg C
	movwf	TEM_PARAM	; temporary
T_CORR_WRITE1
	movlw	EEPROM5		; write to EEPROM
	goto	T_CORR_WRITE2

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; write LEAD ACID
WRITE_LEADACID
	call	WR_LEAD
	goto	WR_ACID
WR_LEAD
	movlw	A'L'		; write LEAD
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'A'		; 
	call	DRV_LCD
	movlw	A'D'		; 
	call	DRV_LCD
	return
WR_ACID
	call	SPACE1
	movlw	A'A'		; write ACID
	call	DRV_LCD
	movlw	A'C'
	call	DRV_LCD
	movlw	A'I'
	call	DRV_LCD
	movlw	A'D'		; 
	call	DRV_LCD
EOLINE
	movlw	A'>'
	call	DRV_LCD
EOLINE1
	call	SPACE4
EOLINE2
	call	SPACE2
	goto	BATT_UP_DN

GEL_CELL_T
; transfer data to EEPROM and registers
; bulk voltage
	movlw	D'225'		; bulk voltage at 20 deg c for GEL_CELL battery (225/255 x 16.0= 14.1)
	call	BULK_WRITE
; float voltage
	movlw	D'212'		; float voltage at 20 deg c for GEL_CELL battery (212/255 x 16.0=13.3)
	call	FLOAT_WRITE
; mV/deg C
	movlw	D'100'		; mv/ deg c for GEL_CELL battery (100/4 = 25mV/deg C)
	call	T_CORR_WRITE
; write GEL_CELL
	movlw	A'G'
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'L'		; 
	call	DRV_LCD
	movlw	A'-'		; 
	call	DRV_LCD
	movlw	A'C'		; write GEL-CELL
	call	DRV_LCD
	movlw	A'E'
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	movlw	A'L'		; 
	call	DRV_LCD
	goto	EOLINE

AGM_T			
; transfer data to EEPROM and registers
; bulk voltage
	movlw	D'230'		; bulk voltage at 20 deg c for AGM battery (230/255 x 16.0= 14.4)
	call	BULK_WRITE
; float voltage
	movlw	D'212'		; float voltage at 20 deg c for AGM battery (212/255 x 16.0=13.3)
	call	FLOAT_WRITE
; mV/deg C
	movlw	D'144'		; mv/ deg c for AGM battery (144/4 = 36mV/deg C)
	call	T_CORR_WRITE

; write AGM
	movlw	A'A'
	call	DRV_LCD
	movlw	A'G'
	call	DRV_LCD
	movlw	A'M'		; 
	call	DRV_LCD
	movlw	A'>'		; 
	call	DRV_LCD
	call	SPACE6
	goto	EOLINE1

CALCIUM_T
; transfer data to EEPROM and registers
; bulk voltage
	movlw	D'239'		; bulk voltage at 20 deg c for CALCIUM/LEAD battery (239/255 x 16.0= 15.0V)
	call	BULK_WRITE
; float voltage
	movlw	D'220'  	; float voltage at 20 deg c for CALCIUM/LEAD battery (219/255 x 16.0=13.8)
	call	FLOAT_WRITE
; mV/deg C
	movlw	D'80'		; mv/ deg c for CALCIUM/LEAD battery (80/4 = 20mV/deg C)
	call	T_CORR_WRITE
	
	movlw	A'C'		; write CALCIUM/LEAD 
	call	DRV_LCD
	movlw	A'A'
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	movlw	A'C'		; 
	call	DRV_LCD
	movlw	A'I'
	call	DRV_LCD
	movlw	A'U'
	call	DRV_LCD
	movlw	A'M'		; 
	call	DRV_LCD
	movlw	A'/'		; 
	call	DRV_LCD
	call	WR_LEAD		; write LEAD
	movlw	A'>'
	call	DRV_LCD
	goto	EOLINE2

SPECIFIC1_T	
; transfer data to EEPROM and registers
; bulk voltage
	movlw	EEPROM6		; bulk voltage at 20 deg c for Specific/1 battery
	call	EEREAD		; read value
	call	BULK_WRITE
; float voltage
	movlw	EEPROM7		; float voltage at 20 deg c for Specific/1 battery
	call	EEREAD		; read value
	call	FLOAT_WRITE
; mV/deg C
	movlw	EEPROM8		; mv/ deg c for Specific/1 battery
	call	EEREAD		; read value
	call	T_CORR_WRITE

; write SPECIFIC1
	call	SPEC_DISPLAY
	goto	WR_1
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
SPEC_DISPLAY
	movlw	A'S'
	call	DRV_LCD
	movlw	A'P'
	call	DRV_LCD
	movlw	A'E'		; 
	call	DRV_LCD
	movlw	A'C'		; 
	call	DRV_LCD
	movlw	A'I'		; write SPECIFIC
	call	DRV_LCD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'I'
	call	DRV_LCD
	movlw	A'C'		; 
	call	DRV_LCD
	call	SPACE1
	movlw	A'#'		; 
	call	DRV_LCD
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
WR_1					; write 1
	movlw	A'1'
	call	DRV_LCD

	goto	EOLINE

SPECIFIC2_T	
; transfer data to EEPROM and registers
; bulk voltage
	movlw	EEPROM9		; bulk voltage at 20 deg c for Specific/2 battery
	call	EEREAD		; read value
	call	BULK_WRITE
; float voltage
	movlw	EEPROM10	; float voltage at 20 deg c for Specific/2 battery
	call	EEREAD		; read value
	call	FLOAT_WRITE
; mV/deg C
	movlw	EEPROM11	; mv/ deg c for Specific/2 battery
	call	EEREAD		; read value
	call	T_CORR_WRITE

; write SPECIFIC2
	call	SPEC_DISPLAY; write SPECIFIC on display	
	movlw	A'2'		; specific 2
	call	DRV_LCD
	goto	EOLINE

; is up or down switch pressed
BATT_UP_DN
	movf	D_BOUNCE,W	; debounce timer
	btfss	STATUS,Z	; if zero check switches
	goto	MORE
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_BATT		; Battery type up
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_BATT		; Battery type down
	goto	MORE
DN_BATT
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE
	movlw	D'80'
	movwf	POINT_TWO	; set 0.2s timer to 0.2seconds so display will update immediately
	movf	BATTERY,w	; battery value
	btfss	STATUS,Z	; if zero do not decrease but set at next battery
	goto	DECREASE_BATT
	movlw	D'05'		; next battery type
	movwf	BATTERY		; next battery
	goto	WRI_BATT
DECREASE_BATT
	decf	BATTERY,f
WRI_BATT	
	movlw	EEPROM2		; write to EEPROM
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	BATTERY,w
	call	EWRITE
	goto	MORE
UP_BATT
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE
	movlw	D'80'
	movwf	POINT_TWO	; set 0.2s timer to 0.2seconds so display will update immediately
	incf	BATTERY,w	; increase battery value
	sublw	D'5'		; stop at count of 5
	btfsc	STATUS,C
	goto	INCREASE_BATT
	clrf	BATTERY		; return to zero battery type position
	goto	WRI_BATT
INCREASE_BATT
	incf	BATTERY,f	; next battery value
	goto	WRI_BATT

EQ_SEL
; equalisation selection
	call	EQ_WRITE
	goto	EQ_LINE2

EQ_WRITE
	movlw	A'E'
	call	DRV_LCD
	movlw	A'Q'
	call	DRV_LCD
	movlw	A'U'		; 
	call	DRV_LCD
	movlw	A'A'		; 
	call	DRV_LCD
	movlw	A'L'		; write Equalisation 
	call	DRV_LCD
	movlw	A'I'
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD
	movlw	A'A'		;
	call	DRV_LCD
	movlw	A'T'
	call	DRV_LCD
	movlw	A'I'
	call	DRV_LCD
	movlw	A'O'
	call	DRV_LCD
	movlw	A'N'
	call	DRV_LCD
	call	SPACE4
	return

EQ_LINE2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	movlw	A'<'
	call	DRV_LCD
	btfss	EQUALISE,0	; if set equalisation on
	goto	EQ_OFF
	movlw	A'O'
	call	DRV_LCD
	movlw	A'N'
	call	DRV_LCD
	movlw	A'>'
	call	DRV_LCD	
	
SPC_12
	call	SPACE8
	call	SPACE4
	goto	TOGGLE_EQ
EQ_OFF
	call	OFF_DSP		; write OFF
	goto	SPC_12
OFF_DSP
	movlw	A'O'
	call	DRV_LCD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'F'
	call	DRV_LCD
	movlw	A'>'
	call	DRV_LCD
	return

TOGGLE_EQ
	
; is up switch pressed
	movf	D_BOUNCE,W	; debounce timer
	btfss	STATUS,Z	; if zero check switches
	goto	MORE
	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UPDN_EQ		; EQ up
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UPDN_EQ		; EQ down
	goto	MORE
UPDN_EQ
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE
	movlw	D'80'
	movwf	POINT_TWO	; set 0.2s timer to 0.2seconds so display will update immediately
	incf	EQUALISE,f	; toggle 0 bit
	goto	MORE

; amp hour mode
AH_MODE
	call	BATTERY_DSP ; write battery on display
	movlw	A'A'
	call	DRV_LCD
	movlw	A'M'		; write AMP_HOUR
	call	DRV_LCD
	movlw	A'P'
	call	DRV_LCD
	call	SPACE1
	movlw	A'H'
	call	DRV_LCD
	movlw	A'O'
	call	DRV_LCD
	movlw	A'U'
	call	DRV_LCD
	movlw	A'R'
	call	DRV_LCD

	movlw	B'11000000'	; line 2 on LCD
	call	LOAD
	movlw	A'<'
	call	DRV_LCD
	movf	AH_VALUE,W	; Ah capacity
	call	AHOUR		; get display AH value	
	movwf	BIN_0		; ls binary value
	call	BIN_ASCII

	movf	OUT1,W		; ms digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLKAH
	movf	OUT1,w		; not zero so display
	call	DRV_LCD	
	goto	MS_NOBLKAH
MS_BLKAH
	call	SPACE1		; space instead of zero
MIDS
	movf	OUT2,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLKAH
MS_NOBLKAH
	movf	OUT2,W		; not zero so display	
	call	DRV_LCD
	goto	LS_AH
LS_BLKAH
	call	SPACE1		; space instead of 0
LS_AH
	movf	OUT3,W		; ls digit
	call	DRV_LCD

	movlw	A'A'		; Ah
	call	DRV_LCD
	movlw	A'h'		; Ah
	call	DRV_LCD
	movlw	A'>'
	call	DRV_LCD
	call	SPACE8
	call	SPACE1
	movf	D_BOUNCE,W	; debounce timer
	btfss	STATUS,Z	; if zero check switches
	goto	MORE
; is up switch pressed

	call	UP_SWITCH	; determine if up switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	UP_AH		; amp hour up
	call	DOWN_SWITCH	; determine if down switch pressed
	btfsc	PORTA,6		; check switch closure
	goto	DN_AH		; amp hour down
	goto	MORE
DN_AH
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE
	movlw	D'80'
	movwf	POINT_TWO	; set 0.2s timer to 0.2seconds so display will update immediately
	movf	AH_VALUE,w	; amp hour value
	btfss	STATUS,Z	; if zero do not decrease
	decf	AH_VALUE,f
WRI_AH	
	movlw	EEPROM1		; write to EEPROM
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bcf		STATUS,RP0	; bank 0		
	movf	AH_VALUE,w
	call	EWRITE
	goto	MORE
UP_AH
	movlw	D'150'		; switch delay
	movwf	D_BOUNCE
	movlw	D'80'
	movwf	POINT_TWO	; set 0.2s timer to 0.2seconds so display will update immediately
	incf	AH_VALUE,w	; increase amp hour value
	sublw	D'17'		; stop at count of 17
	btfsc	STATUS,C
	incf	AH_VALUE,f	; next Ah value
	goto	WRI_AH

; check charging mode (if float), check equalise)

CH_MODES
	btfsc	PORTB,2		; if clear then can run
	goto	MORE
	movf	CH_MODE,W	; charging mode
	btfsc	STATUS,Z	; if zero then Bulk Charge mode
	goto	BULK_DSP
	decf	CH_MODE,W	; 
	btfsc	STATUS,Z	; if zero then ABSORB mode
	goto	ABSORB_DSP
FLOAT
	btfsc	EQUALISE,0	; if set write equalise
	goto	EQU_DSP
	call	SPACE5
	call	FLOAT_DIS_NSP	; write FLOAT
	call	SPACE6
	goto	EO_CHGE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
FLOAT_DIS_NSP
	movlw	A'F'
	call	DRV_LCD
	movlw	A'L'
	call	DRV_LCD
	movlw	A'O'		; 
	call	DRV_LCD
	movlw	A'A'		; 
	call	DRV_LCD
	movlw	A'T'		; write FLOAT
	call	DRV_LCD
	return

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
EO_CHGE
	call	SPACE2
	goto	LINE2
EQU_DSP
	call	SPACE2
	call	EQ_WRITE		; write equalisation on display
	goto	LINE2
BULK_DSP
	call	SPACE4
	call	BULK_DISPLAY	; write Bulk on display
	call	SPACE8
	goto	CHG_DSP
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
BULK_DISPLAY
	call	SPACE2	
BULK_DIS_NSP
	movlw	A'B'
	call	DRV_LCD
	movlw	A'U'
	call	DRV_LCD
	movlw	A'L'		; 
	call	DRV_LCD
	movlw	A'K'		; write BULK
	call	DRV_LCD
	call	SPACE1
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

CHG_DSP
	call	SPACE3
	goto	LINE2
ABSORB_DSP
	call	SPACE3
	movlw	A'A'
	call	DRV_LCD
	movlw	A'B'		; write ABSORPTION
	call	DRV_LCD
	movlw	A'S'
	call	DRV_LCD
	movlw	A'O'
	call	DRV_LCD
	movlw	A'R'
	call	DRV_LCD
	movlw	A'P'
	call	DRV_LCD
	movlw	A'T'
	call	DRV_LCD
	movlw	A'I'		; write ABSORPTION
	call	DRV_LCD
	movlw	A'O'
	call	DRV_LCD
	movlw	A'N'
	call	DRV_LCD
	call	SPACE4
			
LINE2
	movlw	B'11000000'	; line 2 on LCD
	call	LOAD

; write temperature
; temperature is A/D value x 510/255 -273

	clrf	AARGB0		; ms multiplier clear
	movf	DEGC_D,W	; A/D value of temperature
	movwf	AARGB1		; ls multiplier
	movlw	0x01		; 01FE is multiplier value (510)
	movwf	BARGB0		; ms byte
	movlw	0xFE		; ls
	movwf	BARGB1		; ls byte
	call	FXM1616U	; 16 x 16 multiply 

	call	SHIFT_AARGB	; shift results left by 1 byte

	movlw	0xFF		; divide by 255
	movwf	BARGB1		; ls byte of divisor
	clrf	BARGB0		; ms byte of divisor
	
	call	FXD2416U	; divide by 255

	movlw	0x01		; ms binary of 273
	movwf	BARGB0		; ms byte
	movlw	0x11		; ls binary of 273
	movwf	BARGB1		; ls byte
	call	SUBTRACT	; subtract 273 from result

; Check if negative
	movf	AARGB2,W	; subtracted value
	btfsc	NEGTVE,0 	; if set then negative
	comf	AARGB2,W	; complement value 	 

	movwf	BIN_0		; ls binary value
	call	BIN_ASCII

; if out1 not zero then write -- or LO depending whether >99 or <-9

	movf	OUT1,W		; ms digit
	xorlw	0x30		; check if zero
	btfss	STATUS,Z	; if zero continue
	goto	OVER_T		; over (or under) temperature
		
	movf	OUT2,W		; mid digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	LS_BLKT
MID_TEMP
	movf	OUT2,W		; not zero so display
	btfsc	NEGTVE,0	; if negative then beyond -9 so low temp	
	goto	OVER_T		; LO temperature
	call	DRV_LCD
	goto	LS_TEMP
LS_BLKT					; leading zero blanking or negative sign
	btfsc	NEGTVE,0	; if negative insert - sign
	goto	WRI_NEG
NO_NEG
	call	SPACE1		; space instead of 0
	goto	LS_TEMP
WRI_NEG
	movf	OUT3,W		; ls digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z	; if zero no negative sign
	goto	NO_NEG
	call	DASH		; write - (negative)
LS_TEMP
	movf	OUT3,W		; ls digit
	call	DRV_LCD
UNIT_T
	movlw	0xDF		; degrees symbol
	call	DRV_LCD
	movlw	A'C'		; Celcius
	call	DRV_LCD
	call	SPACE1
	goto	WRITE_VOLT

OVER_T
	btfsc	NEGTVE,0	; check if negative
	goto	LO_RDNG
	call	DASH_DASH	; write --
	goto	UNIT_T		; write degrees C
DASH_DASH	
	movlw	A'-'		; write -- 
	call	DRV_LCD
DASH
	movlw	A'-'		; write - 
	call	DRV_LCD
	return
	
LO_RDNG
	movlw	A'L'		; write LO 
	call	DRV_LCD
	movlw	A'O'		; write LO 
	call	DRV_LCD
	goto	UNIT_T		; write degrees C

; write Voltage
; voltage is A/D value x 160/255

WRITE_VOLT
	call	VOLT_DSP
	goto	WRI_CURRENT

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
VOLT_DSP
; write voltage to display
; check if overrange
	incfsz	VOLTAGE_U,W ; if at maximum write -.-
	goto	NOT_OVER_V
	call	DASH_DASH	; write --	
	movlw	A'.'		; decimal point
	call	DRV_LCD
	call	DASH		; write -
	goto	WRI_VOLT	; write V
NOT_OVER_V 	
	movf	VOLTAGE_D,W	; A/D value of voltage
VOLT_DSP_SP
	call	CALCUL_ATE
	goto	CONV_DISP
CALCUL_ATE
	clrf	AARGB0		; ms multiplier clear

	movwf	AARGB1		; ls multiplier
	clrf	BARGB0		; ms byte
	movlw	D'160'		; ls
	movwf	BARGB1		; ls byte
	call	FXM1616U	; 16 x 16 multiply
 
	call	SHIFT_AARGB	; shift results left by 1 byte

	movlw	0xFF		; divide by 255
	movwf	BARGB1		; ls byte of divisor
	clrf	BARGB0		; ms byte of divisor

	call	FXD2416U	; divide by 255

	movf	AARGB2,W	; divided value
	return
; convert to ASCII and write to display
CONV_DISP
	movwf	BIN_0		; ls binary value
	call	BIN_ASCII
	movf	OUT1,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLKV
	movf	OUT1,W		; not zero so display	
	call	DRV_LCD
	goto	MID_VOLT
MS_BLKV
	call	SPACE1		; space instead of 0
MID_VOLT
	movf	OUT2,W		; mid digit
	call	DRV_LCD
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	OUT3,W		; ls digit
	call	DRV_LCD
WRI_VOLT
	movlw	A'V'		; Volts
	call	DRV_LCD
	call	SPACE1
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; write Current
; curent is A/D value /10 ie add decimal point. eg 16.0A is 160 A/D
WRI_CURRENT
	movf	I_AVG,W		; average current

	movwf	BIN_0		; ls binary value
	call	BIN_ASCII
	movf	OUT1,W		; first digit
	xorlw	0x30		; check if zero
	btfsc	STATUS,Z
	goto	MS_BLKC
	movf	OUT1,W		; not zero so display	
	call	DRV_LCD
	goto	MID_CURR
MS_BLKC
	call	SPACE1		; space instead of 0
MID_CURR
	movf	OUT2,W		; mid digit
	call	DRV_LCD
	movlw	A'.'		; decimal point
	call	DRV_LCD
	movf	OUT3,W		; ls digit
	call	DRV_LCD
	movlw	A'A'		; Amps
	call	DRV_LCD

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Charging control loop

; check display mode (only charge when mode is 0)
CHRG_CONTROL
	movf	MODE,W		; mode
	btfss	STATUS,Z	; if zero then charge
	goto	NO_CHARGE
; check charging mode (if float), check equalise)

	movf	CH_MODE,W	; charging mode
	btfsc	STATUS,Z	; if zero then Bulk Charge mode
	goto	BULK_CONT
	decf	CH_MODE,W	; 
	btfsc	STATUS,Z	; if zero then ABSORB mode
	goto	ABSORB_CONT

	btfsc	EQUALISE,0	; if set write equalise
	goto	EQU_CONT	; after eq clear EQUALISE
	goto	FLOAT_CONT


BULK_CONT
; move PWM duty upwards until current reaches 25% of rated battery AH capacity.
; check voltage and stop at predetermined value (battery type and temperature compensation) 

; calculate required switch off voltage stored in BULK_V (Bulk Voltage)

	call	COMPENSATE	; calculate compensation value of voltage wrt temperature
	call	ADD_SUB		; add or subtract compensation value
	goto	V_RDY		; check voltage
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; compensation calculation (compensate from 0 degreesC upward)

; battery voltage required x10/160 x 255 gives voltage A/D value.eg 15V : 150/160 x 255 = 239
; Temp compensation is (A/D for temperature at 20 degrees C (146) - A/D value of current temperature) (x2) x mv/degrees C factor x 255/16
; (x2) because each ls bit of A/D is 2 degrees so mult mV/deg C by 2. x 255/16 because this is the conversion to volts
; eg 0.036mV/deg C x 2 and x 256/16 is 1.15 or 253/220  
COMPENSATE
	clrf	NEG_COMP	; negative compensation
; check if temperature sensor open circuit, assume 40 deg C if open
	movf	DEGC_D,W	; degrees C reading
	sublw	D'180' 		; is it high ie above about 3.5V
	btfsc	STATUS,C	; if negative assume 40 deg C
	goto	CHK_ZROC	; normal temperature compensation as per reading
	movlw	D'156'		; 40 deg C
	goto	COMP_BELOW0
; check if -10 degrees C do not compensate below this temperature	
CHK_ZROC
	movf	DEGC_D,W	; degrees reading from A/D converter
	sublw	D'131'		; -10 degrees ((273 -10) x 9.8mV)/5 x 255
	btfss	STATUS,C	; if negative then compensate if positive set at 0 degrees
	goto	COMP_ABOVE0
	movlw	D'131'		; -10 degrees C
	goto	COMP_BELOW0
COMP_ABOVE0
	movf	DEGC_D,w	; degrees reading from A/D converter
COMP_BELOW0
	sublw	D'146'		; A/D value at 20 degrees C, 68 deg F
	btfsc	STATUS,C	; if c is 0 then negative
	goto	NON_NEG
	bsf		NEG_COMP,0	; set if negative
	movwf	AARGB1
	comf	AARGB1,W

; multiply by mV/deg C factor
NON_NEG
	movwf	AARGB1		; ls multiplier
	clrf	AARGB0		; ms byte
	clrf	BARGB0		; ms byte
	movf	MV_DEGC,w	; ls
	movwf	BARGB1		; ls byte
	call	FXM1616U	; 16 x 16 multiply 

	call	SHIFT_AARGB	; shift results by 1 byte left

	movlw	D'124'		; divide by 124
	movwf	BARGB1		; ls byte of divisor
	clrf	BARGB0		; ms byte of divisor
	call	FXD2416U	; divide by 250
	movf	AARGB2,W	; divided value
	movwf	STORAGE
	return
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	
; add or subtract to BULK_0 for Bulk charge (BULK_0 dependent on battery type)
ADD_SUB
	movf	STORAGE,W	; obtain compensation value
	btfsc	NEG_COMP,0	; if negative take away
	goto	TAK_AWAY
	addwf	BULK_0,W	; bulk voltage at 20 degrees C (A/D voltage reading is Voltage required/160 x 255)
	btfss	STATUS,C	; if carry set then overrange so set at FF
	goto	VALUE_B	
	movlw	0xFF
VALUE_B
	movwf	BULK_V		; bulk Voltage
	return
TAK_AWAY
	subwf	BULK_0,W	; bulk V at 0 degrees C
	movwf	BULK_V		; bulk voltage
	return

; check if voltage level reached
V_RDY
; 0 volts to 310mV
	movlw	0x05		; low A/D voltage value
	subwf	VOLTAGE_D,W	; if voltage close to zero (battery not connected)
	btfss	STATUS,C	; if negative
	goto	NO_VOLTS

; check BULK_V. If > 10... If Voltage_D <10.5V apply 2% of AH till 10.5V reached
; if BULK_V < 10..... If VOLTAGE_D <5.2V apply 2% of AH till 5.2V reached

	movlw	D'160'		; 10V A/D voltage value
	subwf	BULK_V,w	; if greater or smaller
	btfss	STATUS,C
	goto	SIX_V	
; bulk_V > 10V so check VOLTAGE_D for < 10.5V
	movlw	D'165'		; 10.5V A/D voltage value
	subwf	VOLTAGE_D,W	; if voltage less than 10.5 (battery discharged)
	btfss	STATUS,C	; if negative
	goto	LOW_VOLTS
	goto	B_ULK_V
SIX_V
	movlw	D'83'		; 5.2V A/D voltage value
	subwf	VOLTAGE_D,W	; if voltage less than 5.2V (battery discharged)
	btfss	STATUS,C	; if negative
	goto	LOW_VOLTS
; bulk voltage
B_ULK_V
	movf	BULK_V,W	; bulk voltage
	subwf	VOLTAGE_D,W	; current voltage reading from A/D
	btfsc	STATUS,C	; if negative continue
	goto	NXT_MODE	; end of bulk charge	

; set required current for AH of battery

	movf	AH_VALUE,W	; Amp Hour value for battery (0 to 16)
	call	TWO5_PCNT	; get 25% current value 
	call	I_CONTROL	; sets current to battery
 	goto	MORE
	
NO_VOLTS
	clrf	CCPR1L		; no PWM drive if no battery
	clrf	SLOW		; slow control flag
	clrf	I_ADDM		; ms byte of counted current
	clrf	I_ADDL		; ls byte of counted current
	goto	MORE
LOW_VOLTS
	movf	AH_VALUE,W	; Amp Hour value
	call	TWO_PCNT	; apply 2%AH when battery low
	call	I_CONTROL	; sets current to battery
 	goto	MORE
; prepare for next mode
NXT_MODE
	incf	CH_MODE,f	; next charge Mode ABSORB
	clrf	UPDATE1		; seconds counter
	clrf	UPDATE2		; return to 0 minutes
	clrf	UPDATE3		; minutes counter
	clrf	HOURS1 		; xx.x hours (25.5h max)ready to start timer	
		
ABSORB_CONT				; absorption control, maintain bulk voltage with temperature correction 
	call	COMPENSATE	; calculate compensation value of voltage wrt temperature
	call	ADD_SUB		; add or subtract compensation value from bulk voltage @ 20C
	movf	BULK_V,W
	call	V_CONTROL	; maintain voltage
; Check if in slow control
	btfss	SLOW,0		; if bit zero set then slow control so check current
	goto	CK_TMRI		; check timer

; check if battery voltage the same as BULK_V (ie control is locked)
	movf	VOLTAGE_D,W	; corrected battery voltage measurement (correction wrt drop across leads)
	xorwf	BULK_V,W	; required bulk voltage (with temperature correction)
	btfss	STATUS,Z	; if zero check the 2% current
	goto	CK_TMRI		; check timer

; check if average current is zero
	movf	I_AVG,W		; average current
	btfsc	STATUS,Z	; if zero bypass checking 2% of Ah 
	goto	CK_TMRI		; check timer

; check current for 2% of AH
	movf	AH_VALUE,W	; battery AH value
	call	TWO_PCNT	; get 2% current
	subwf	I_AVG,W		; is averaged current less than 2%
	btfss	STATUS,C	
	goto	NXT_MODE1	; current dropped to 2% so float mode

; check timer
CK_TMRI
	movlw	D'26'		; 2.5 hours
	subwf	HOURS1,W
	btfss	STATUS,C
	goto	MORE
NXT_MODE1
	incf	CH_MODE,f	; next charge Mode Float
	clrf	UPDATE1		; seconds counter
	clrf	UPDATE2		; return to 0 minutes
	clrf	UPDATE3		; minutes counter
	clrf	HOURS1		; xx.x hours (25.5h max)ready to start timer

FLOAT_CONT				; float control
; maintain at set V	with temp compensation

	call	COMPENSATE	; calculate temperature compensation value
; add or subtract to FLOAT_0 for Float charge
FLOAT_CK
	movf	STORAGE,W	; obtain compensation value
	btfsc	NEG_COMP,0	; if negative take away
	goto	TAK_AWAY_F
	addwf	FLOAT_0,W	; float V at 20 degrees C (A/D voltage reading is req V/160 x 255)
	btfss	STATUS,C	; if carry set then overrange so set at FF
	goto	VALUE_F	
	movlw	0xFF
VALUE_F
	movwf	FLOAT_V		; Float Voltage
	goto	FLT_RDY		; voltage ready
TAK_AWAY_F
	subwf	FLOAT_0,W	; float V at 20 degrees C
	movwf	FLOAT_V		; Float voltage

FLT_RDY
	movf	FLOAT_V,W
	call	V_CONTROL	; maintain voltage
	goto	MORE
	
EQU_CONT
; if temperature >40 deg C no equalisation
	movf	DEGC_D,w	; temperature
	sublw	D'156'		; 40 deg C
	btfss	STATUS,C	; if negative then no equalisation
	goto	EQ_OUT	
; add 5% of AH capacity for 3 hours then clear EQUALISE to goto float
	movf	AH_VALUE,W	; Amp Hour value
	call	FIVE_PCNT	; apply 5%AH 
	call	I_CONTROL	; sets current to battery
; check timer
	movlw	D'31'		; 3 hours 
	subwf	HOURS1,W
	btfss	STATUS,C
	goto	MORE
EQ_OUT
	clrf	EQUALISE	; clear ready for float	
	clrf	CCPR1L		; current off at end of equalise
	clrf	SLOW		; slow control rate off
	goto	MORE	
; not charging
NO_CHARGE
	clrf	CCPR1L		; pwm off
	clrf	SLOW		; slow control flag
	clrf	I_ADDM		; ms byte of counted current
	clrf	I_ADDL		; ls byte of counted current
	goto	MORE
;***************************************************************************
; subroutines

; current control
I_CONTROL
	movwf	I_SET		; set current
; check if A/D conversions ready
	btfss	CONV_COMP,0	; if set A/D data ready
	return
; check counter
	btfss	UPDATE1,0	; control when 0 bit set
	return

; check slow control flag
	btfss	SLOW,0		; if flag set then slow control
	goto	FAST_I
	movf	SLOW_T,w	; check timer
	btfss	STATUS,Z	; if zero then slow control
	return				; wait till timer ended
	movlw	D'100'		; time = 100/250 seconds
	movwf	SLOW_T	

FAST_I
; check voltage and set pwm so that output of x 6.6 amplifier equals measured voltage
; this is a fast way to bring pwm up from 0 to close to battery voltage 
	bcf		STATUS,C	; carry cleared
	rrf		VOLTAGE_D,w	; divide by 2 (it happens that if the A/D value of voltage is divided by two
						; and placed in the PWM register the resulting output after amplification
						; in x 6.6 amplifier is equal to the measured voltage)
	subwf	CCPR1L,w	; is subtracted from CCPR1L (PWM register)
	btfsc	STATUS,C	; if negative load CCPR1L with divided by 2 A/D voltage value
	goto	SET_CURRENT
	rrf		VOLTAGE_D,w	; get /2 value again
	movwf	CCPR1L		; set pwm 
	call	DELAYms		
	return				; wait for current change  
; set current
SET_CURRENT	 
	movf	I_SET,w		; set current
	subwf	CURRENT_D,W
CURRENT_COMP
	btfsc	STATUS,C	; if carry is zero keep increasing PWM
	goto	BY_CCINC
CC_INC_CK
	incfsz	CCPR1L,W	; increase PWM drive check for zero (maximum)
	goto	CC_INC		; increase the CCPR1L register
	return				; bypass increasing CCPR1L register
CC_INC
	incf	TEN_BIT,f	; increase ten bit pwm
	swapf	TEN_BIT,w	; bits 0,1 to 4,5
	iorlw	B'00001100' ; keep pwm on at bits 2,3
	andlw	B'00111100'	; clear bits 0,1
	movwf	CCP1CON 	; 10-bit pwm
	movlw	B'00000011'	; maintain counter bits 0,1
	andwf	TEN_BIT,w	; 
	xorlw	B'00000000'	; if bits 0,1 are zero increase CCPR1L
	btfsc	STATUS,Z
	incf	CCPR1L,f	; increase 8-bit PWM
	return
BY_CCINC
	movf	I_SET,W		; set current
	xorwf	CURRENT_D,W
BY_INC_LP
	btfsc	STATUS,Z	; if zero keep current PWM
	goto	PWM_KEEP
CC_DEC
	bsf		SLOW,0		; set the slow rate of control when first begin to decrement (ie max I has been exceeded)
	decf	CCPR1L,W
	xorlw	0xFF		; if passed beyond zero bypass decrement
	btfsc	STATUS,Z
	return
	decf	TEN_BIT,f	; decrease ten bit pwm
	swapf	TEN_BIT,w	; bits 0,1 to 4,5
	iorlw	B'00001100' ; keep pwm on at bits 2,3
	andlw	B'00111100'	; clear bits 0,1
	movwf	CCP1CON 	; 10-bit pwm
	movlw	B'00000011'	; maintain counter bits 0,1
	andwf	TEN_BIT,w	; 
	xorlw	B'00000011'	; if bits 0,1 are both set decrease CCPR1L
	btfsc	STATUS,Z	
	decf	CCPR1L,f
	return
PWM_KEEP
	bsf		SLOW,0		; set the slow rate of control when zero (ie max I has been reached)
	return

; voltage control
V_CONTROL
	movwf	V_SET		; set voltage

; check if A/D conversions ready
	btfss	CONV_COMP,0	; if set A/D data ready
	return
; check counter
	btfss	UPDATE1,0	; control when set ie every second change in update1
	return
	bcf		CONV_COMP,0	; clear again

; check slow control flag
	btfss	SLOW,0		; if flag set then slow control
	goto	FAST_V
	movf	SLOW_T,w	; check timer
	btfss	STATUS,Z	; if zero then slow control
	return				; wait till timer ended
	movlw	D'100'		; time = 100/250 seconds
	movwf	SLOW_T	
FAST_V
; firstly check current. Must not go over 25% of AH even if voltage lower than required
	movf	AH_VALUE,W	; Ah setting
	call	TWO5_PCNT	; get current value setting (equivalent to A/D current reading), value in W register
	movwf	STORE_CK
	movf	CURRENT_D,W	; A/D current
	subwf	STORE_CK,W	; 
	btfss	STATUS,C 	; if current greater then decrease PWM
	goto	CC_DEC		; decrement PWM
; current ok so set voltage	 	
	movf	V_SET,w		; set voltage
	subwf	VOLTAGE_D,W
	btfss	STATUS,C	; if carry is zero keep increasing PWM
	goto	CC_INC_CK	; increase the CCPR1L register

; decrease or keep current pwm
; if current is zero do not decrease any further
	movf	CURRENT_D,W	; current
	btfsc	STATUS,Z	; if zero return
	return	
	movf	V_SET,W		; set voltage
	xorwf	VOLTAGE_D,W
	goto	BY_INC_LP


; shift results left by 1 byte

SHIFT_AARGB
	movf	AARGB1,W
	movwf	AARGB0		; shift result left by 1 byte
	movf	AARGB2,W
	movwf	AARGB1		; shift result left by 1 byte
	movf	AARGB3,W
	movwf	AARGB2		; shift result left by 1 byte
	return

; initialise display

INIT_LC
	movlw	B'00110000'	; initialise module
	movwf	PORTB
SET_CLEAR_B1
	nop
	bsf		PORTB,1		; enable high
	nop
	bcf		PORTB,1		; low
	return

; add space in display
SPACE8
	movlw	0x08
	goto	XTRA_SPACE
SPACE7
	movlw	0x07
	goto	XTRA_SPACE
SPACE6
	movlw	0x06
	goto	XTRA_SPACE
SPACE5
	movlw	0x05
	goto	XTRA_SPACE
SPACE4
	movlw	0x04
	goto	XTRA_SPACE
SPACE3
	movlw	0x03
	goto	XTRA_SPACE
SPACE2
	movlw	0x02
	goto	XTRA_SPACE
SPACE1
	movlw	0x01

XTRA_SPACE
	movwf	SPACE_BAR
ANOTHER_SPACE
	movlw	0x20		; space
	call	DRV_LCD
	decfsz	SPACE_BAR,f	; decrease number of spaces
	goto	ANOTHER_SPACE
	return

; delay timer

DELAY_TM
	movlw	0x20
DELX	
	movwf	TIME_I		; initial timer 
TIM_CON
	call	DELAYms
	decfsz	TIME_I,f
	goto	TIM_CON		; continue timer

; nominal 10ms delay loop

DELAYms
	movlw	0x10		; ~10ms delay with 4 MHz clock
	movwf	STORE1		; STORE1 is number of loops value
LOOP1
	movlw	0xA0
	movwf	STORE2		; STORE2 is internal loop value	
LOOP2
	decfsz	STORE2,f
	goto	LOOP2
	decfsz	STORE1,f
	goto	LOOP1		; decrease till STORE1 is zero
	return

; preload display commands (4-bit) 

LOAD
	call	LEV_UPPER	; upper bits	

	bcf		PORTA,7		; register select low
	
	call	LEV_LOWER	; get lower bits
	bcf		PORTA,7		; register select low
	goto	BUS_CK		; check busy flag
	
; driving the LCD module with display data

DRV_LCD	
	call	LEV_UPPER	; get upper bits
	bsf		PORTA,7		; register select
	
	call	LEV_LOWER	; get lower bits
	bsf		PORTA,7		; register select
	nop

; check busy flag

BUS_CK
	call	SET_CLEAR_B1; set and clear Enable (PortB,1)	
	clrf	BUS_TME		; delay at 255 counts
BUS_Y
	decfsz	BUS_TME,f
	goto	BUS_Y
	return

; upper bits

LEV_UPPER
	movwf	D_STO		; store data
	movf	PORTB,W		; get portB levels
	andlw	0x0F		; clear upper bits keep lower bits
	movwf	PORTB		; upper bits cleared
	movf	D_STO,W
	andlw	0xF0		; upper bits	
	iorwf	PORTB,f		; w to portb
	return
; lower bits

LEV_LOWER
	call	SET_CLEAR_B1; set and clear Enable (PortB,1)	
	movf	PORTB,W		; get portB levels
	andlw	0x0F		; clear upper bits keep lower bits
	movwf	PORTB		; upper bits cleared
	swapf	D_STO,W
	andlw	0xF0		; get lower bits
	iorwf	PORTB,f		; place display commands in portB
	return

; Down switch check
DOWN_SWITCH
	movlw	0x02
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,5		; down switch
DY_SW
	movlw	0x02
	call	DELX
	return

; Up switch check
UP_SWITCH
	movlw	0x0F
	andwf	PORTB,f		; set RB4-7 low
	bsf		PORTB,4		; up switch
	goto	DY_SW


; Subroutine to convert from 8-bit binary to 2-digit BCD (packed)
; Binary value is in BIN_0  
; Result in BCD is in BCD_0 & BCD_1.  
; BCD_0 is MSB, BCD_1 is LSB
; converts to unpacked ASCII in OUT1, OUT2, OUT3 (out1 is ms byte, out3 is ls byte)

BIN_ASCII
	bcf		STATUS,C	; clear carry bit
	movlw	D'8'
	movwf	CNT_8		; 8 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
LOOPBCD	
	rlf		BIN_0,f		; shift left binary registers
	rlf		BCD_1,f		; MSB shift left

	rlf		BCD_0,f		; LSB shift left BCD registers
	decfsz	CNT_8,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust

; completed decimal to BCD operation, convert to unpacked ASCII
	
	movf	BCD_1,W		; ls decimal
	andlw	0x0F		; ls
	addlw	0x30		; convert to ASCII
	movwf	OUT3
	swapf	BCD_1,W 	; mid decimal value
	andlw	0x0F
	addlw	0x30		; convert to ASCII
	movwf	OUT2
	movf	BCD_0,W
	addlw	0x30		; convert to ASCII
	movwf	OUT1		; ms decimal value

	return				; ASCII values returned in OUT1, OUT2, OUT3 

; subroutine decimal adjust

DECADJ
	movlw	BCD_1		; BCD LSB address
	movwf	FSR			; pointer for BCD1
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR			; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD
	movlw	0x03		; w has 03 
	addwf	INDF,W		; add 03 to BCDx register (x is 0-1)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,W		; add 30 to BCDx register
	movwf	TEMP		; store w in temp
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return				; end subroutine	

; subroutine to read EEPROM memory

EEREAD	
	bsf 	STATUS,RP0	; select memory bank 1
	movwf 	EEADR		; indirect special function register
	bsf		EECON1,RD	; read EEPROM
	movf	EEDATA,W	; EEPROM value in w
	bcf		STATUS,RP0	; select bank 0
	return

; subroutine to write to EEPROM

EWRITE
	bsf		STATUS,RP0	; select bank 1
	movwf	EEDATA		; data register
	bcf		INTCON,GIE	; disable interrupts

	bsf		EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bsf		INTCON,GIE	; enable interrupts
	bcf		EECON1,WREN	; clear WREN bit
WRITE	
	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0	; bank 0 
	return				; value written 


;**********************************************************************************************
  ; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LA 		RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LC 		RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	      		ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L			RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN
      



; subtract AARGB1 (ms) AARGB2 (ls) - BARGBO BARGB1 = AARGB1 AARGB2
SUBTRACT

; check if BARGB > AARGB if so set neg sign
	clrf	NEGTVE		; negative sign
	movf	AARGB1,W
	subwf	BARGB0,W
	btfss	STATUS,C	; check if = or >
	goto	NEG_SUB		; negative so can subtract
	btfss	STATUS,Z	; if equal check ls byte
	goto	NEGATE		; is > 
	movf	AARGB2,W
	subwf	BARGB1,W
	btfss	STATUS,C	; if = or >
	goto	NEG_SUB
NEGATE
	bsf		NEGTVE,0	; negative sign
	
NEG_SUB
	call	NEG_A		; complement of A
	movf	BARGB1,W 
	addwf	AARGB2,f 	; add lsb
	btfsc	STATUS,C	; add carry
	incf	AARGB1,f 
	movf	BARGB0,W 
	addwf	AARGB1,f 
	return

NEG_A
	comf	BARGB1,f 
	incf	BARGB1,f 
	btfsc	STATUS,Z
	decf	BARGB0,f 
	comf	BARGB0,f 
	return	                             
	end
